// =============================================================================
//
//      --- kvi_stat_systray.cpp ---
//
//   This is a plugin for KVIrc 2. It allows you to see
//   statistics of your IRC sessions - total words said etc.
//   More features are still to come.
//
//   (c) Krzysztof Godlewski <kristoff@poczta.wprost.pl>
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviStatSysTray"

#include <qcursor.h>
#include <qpainter.h>

#define __KVIRC_PLUGIN__

#include "kvi_locale.h"
#include "kvi_options.h"
#include "kvi_popupmenu.h"
#include "kvi_stat_chan.h"
#include "kvi_stat_controller.h"
#include "kvi_stat_systray.h"
#include "kvi_systray.h"

extern KviStatController *g_pStatPluginController;

static const QString g_statIcon("stat_dock.png");

KviStatSysTray::KviStatSysTray(KviSysTray *parent, KviFrame *frm, const char *tooltip)
	: KviSysTrayWidget(parent, tooltip ? tooltip : __tr("Your statistics"))
{
	m_pFrm    = frm;
	m_pParent = parent;
	rmbPopup  = new KviPopupMenu();
	rmbPopup->insertItem(_CHAR_2_QSTRING(__tr("Show StatWin")),        g_pStatPluginController, SLOT(slotShowStats()));
	rmbPopup->insertItem(_CHAR_2_QSTRING(__tr("Configure plugin")),    g_pStatPluginController, SLOT(slotShowConfig()));
	rmbPopup->insertItem(_CHAR_2_QSTRING(__tr("Reset stats")),         g_pStatPluginController, SLOT(slotReset()));
	rmbPopup->insertSeparator();
	rmbPopup->insertItem(_CHAR_2_QSTRING(__tr("Save stats file now")), g_pStatPluginController, SLOT(slotSaveStats()));
	rmbPopup->insertItem(_CHAR_2_QSTRING(__tr("Hide this widget")),    this,                    SLOT(slotHide()));
	setBackgroundMode(NoBackground);
	emit wantOptions();
	wantScrolling = false;
	textXPos = 50;
	startTimer(1000); // 1 lags a lot!

	g_pStatPluginController->registerStatTray(this);
}

KviStatSysTray::~KviStatSysTray()
{
	g_pStatPluginController->unregisterStatTray(this);
}

void KviStatSysTray::mousePressEvent(QMouseEvent *ev)
{
	switch( ev->button() ) {
		case QMouseEvent::RightButton: { // Pop up the menu
			rmbPopup->exec(QCursor::pos());
		} break;

		default:
			// Ignore anything but right mouse button
			break;
	}
}

void KviStatSysTray::timerEvent(QTimerEvent *ev)
{
	// Stopping and restarting the timer is a hard job.
	// Not better just to leave it running and maybe stop and restart it
	// when the textDelay has changed?
	killTimers();
	if( textDirectionLeft )
		textXPos -= textJump;
	else
		textXPos += textJump;
	repaint(false);
	startTimer(textDelay);
}

// This code is taken from kvi_systray.cpp, since I had NO idea how to
// repaint the widget correctly.
void KviStatSysTray::paintEvent(QPaintEvent *ev)
{
	if( !isVisible() ) return;

	bool temp = m_showWords;
	QPainter pa(m_pSysTray->getMemoryBuffer());

	paintDefaultBackground(&pa);

	KviStr s, sw, sj, sk, sb, st;
	const KviStatChan *pCurrentChan = g_pStatPluginController->currentChan();

	if( g_pStatPluginController->showConsoleStats() ) {
		 s.sprintf(__tr("Total statistics since %s"), g_pStatPluginController->startDate());
		m_showWords = true; // This must be true here, since displaying numOnIrc
							// relies on that; that is why there is a temp bool
		sw.sprintf(__tr("Sessions: %d"),              g_pStatPluginController->numOnIrc());
		sj.sprintf(__tr("Chans joins: %d"),           g_pStatPluginController->numJoins());
		sk.sprintf(__tr("Kicks: %d"),                 g_pStatPluginController->numKicks());
		sb.sprintf(__tr("Bans: %d"),                  g_pStatPluginController->numBans());
		st.sprintf(__tr("Topics: %d"),                g_pStatPluginController->numTopics());
	} else if( pCurrentChan ) {
		 s.sprintf(__tr("Stats for %s since %s"),    pCurrentChan->name(), m_startDate.latin1());
		sw.sprintf(__tr("Words: %d"),                pCurrentChan->words());
		sj.sprintf(__tr("Joins: %u"),                pCurrentChan->joins());
		sk.sprintf(__tr("Kicks: %u"),                pCurrentChan->kicks());
		sb.sprintf(__tr("Bans: %u"),                 pCurrentChan->bans());
		st.sprintf(__tr("Topics: %u"),               pCurrentChan->topics());
	} else {
		s.sprintf(__tr("You must join a channel to see the stats"));
		sw = sj = sk = sb = st = s;
	}

	pa.setPen(m_pSysTray->getForeground());

	int xpos;
	if( !scrollingWhen ) wantScrolling = false;
	if( wantScrolling )
		xpos = textXPos;
	else
		xpos = 4;
	int slen       = 0;  // Length of the strings
	int spacelen   = 0;  // Length of spaces between the strings
	int font_width = 6;  // Width of the font
	if( height() > 32 ) {
		// Big mode
		g_pOptions->m_fntSysTray.setPointSize(10);
		pa.setFont(g_pOptions->m_fntSysTray);
		pa.drawText(4, height() - 24, s.ptr(), s.len());
		g_pOptions->m_fntSysTray.setPointSize(12);
		pa.setFont(g_pOptions->m_fntSysTray);
		font_width = QFontMetrics(g_pOptions->m_fntSysTray).width("A");
		if( m_showWords ) {
			pa.drawText(xpos, height() - 8, sw.ptr(), sw.len());
			xpos += QFontMetrics(g_pOptions->m_fntSysTray).width(sw.ptr()) + 8;
			slen += sw.len();
			spacelen += 8;
		}
		if( m_showJoins ) {
			pa.drawText(xpos, height() - 8 , sj.ptr(), sj.len());
			xpos += QFontMetrics(g_pOptions->m_fntSysTray).width(sj.ptr()) + 8;
			slen += sj.len();
			spacelen += 8;
		}
		if( m_showKicks ) {
			pa.drawText(xpos, height() - 8, sk.ptr(), sk.len());
			xpos += QFontMetrics(g_pOptions->m_fntSysTray).width(sk.ptr()) + 8;
			slen += sk.len();
			spacelen += 8;
		}
		if( m_showBans ) {
			pa.drawText(xpos, height() - 8, sb.ptr(), sb.len());
			xpos += QFontMetrics(g_pOptions->m_fntSysTray).width(sb.ptr()) + 8;
			slen += sb.len();
			spacelen += 8;
		}
		if( m_showTopics ) {
			pa.drawText(xpos, height() - 8, st.ptr(), st.len());
			slen += st.len();
			xpos += QFontMetrics(g_pOptions->m_fntSysTray).width(st.ptr()) + 8;
			spacelen += 8;
		}
	} else { // Small mode
		g_pOptions->m_fntSysTray.setPointSize(12);
		pa.setFont(g_pOptions->m_fntSysTray);
		font_width = QFontMetrics(g_pOptions->m_fntSysTray).width("A");
		if( m_showWords ) {
			pa.drawText(xpos, height() - 6, sw.ptr(), sw.len());
			xpos += QFontMetrics(g_pOptions->m_fntSysTray).width(sw.ptr()) + 8;
			slen += sw.len();
			spacelen += 8;
		}
		if( m_showJoins ) {
			pa.drawText(xpos, height() - 6, sj.ptr(), sj.len());
			xpos += QFontMetrics(g_pOptions->m_fntSysTray).width(sj.ptr()) + 8;
			slen += sj.len();
			spacelen += 8;
		}
		if( m_showKicks ) {
			pa.drawText(xpos, height() - 6, sk.ptr(), sk.len());
			xpos += QFontMetrics(g_pOptions->m_fntSysTray).width(sk.ptr()) + 8;
			slen += sk.len();
			spacelen += 8;
		}
		if( m_showBans ) {
			pa.drawText(xpos, height() - 6, sb.ptr(), sb.len());
			xpos += QFontMetrics(g_pOptions->m_fntSysTray).width(sb.ptr()) + 8;
			slen += sb.len();
			spacelen += 8;
		}
		if( m_showTopics ) {
			pa.drawText(xpos, height() - 6, st.ptr(), st.len());
			slen += st.len();
			xpos += QFontMetrics(g_pOptions->m_fntSysTray).width(st.ptr()) + 8;
			spacelen += 8;
		}
	}

	paintDefaultFrame(&pa);
	bitBlt(this, 0, 0, m_pSysTray->getMemoryBuffer(), 0, 0, width(), height());

	m_showWords = temp;

	if( textDirectionLeft ) {
		if( textXPos + (font_width * slen) - spacelen - 60 <= 0 ) textXPos = width();
	} else if( textXPos >= width() )
		textXPos = - (font_width * slen) - spacelen + 60;

	if( scrollingWhen == 1 ) // 1 == ScrollingAlways
		wantScrolling = true;
}

QString KviStatSysTray::icon()
{
	return g_statIcon;
}

void KviStatSysTray::setOptions(bool sw, bool sj, bool sk, bool sb, bool st)
{
	m_showWords  = sw;
	m_showJoins  = sj;
	m_showKicks  = sk;
	m_showBans   = sb;
	m_showTopics = st;
}

void KviStatSysTray::setScrollingOptions(int when, int delay, int jump, bool dirLeft)
{
	scrollingWhen     = when;
	textJump          = jump;
	textDelay         = delay;
	textDirectionLeft = dirLeft;
}

void KviStatSysTray::setStartDate(const char *date)
{
	m_startDate = date;
}

void KviStatSysTray::slotHide()
{
	m_pSysTray->removeWidget(this, true);
}

#include "m_kvi_stat_systray.moc"
