// =============================================================================
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Andrea Parrella (anandrea@iname.com)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviUrlDialog"

#include <qcursor.h>

#ifdef QT_NO_ASCII_CAST
	#undef QT_NO_ASCII_CAST
#endif

#define __KVIRC_PLUGIN__

#include "kvi_iconloader.h"
#include "kvi_listview.h"
#include "kvi_locale.h"
#include "kvi_menubar.h"
#include "kvi_mdi_manager.h"
#include "kvi_popupmenu.h"
#include "kvi_process.h"
#include "kvi_url.h"
#include "kvi_url_configdialog.h"
#include "kvi_url_dialog.h"
#include "kvi_url_toolbar.h"
#include "kvi_winproperties.h"
#include "kvirc_plugin.h"

/**
 * KviUrlDialog.
 */
KviUrlDialog::KviUrlDialog(QPtrList<KviUrl> *pList, KviFrame *lpFrm)
	: KviWindow("!URL List", KVI_WND_TYPE_PLUGIN, lpFrm)
{
	m_pIcon    = 0;
	g_pList    = pList;
	m_pMenuBar = new KviMenuBar(this);
	m_pUrlList = new KviListView(this, "list");
	KviConfig cfg(g_configPath.ptr());

	KviPopupMenu *pop;

	pop = new KviPopupMenu(this);
	pop->insertItem(__tr("&Configure"),    this, SLOT(config()));
	pop->insertItem(__tr("&Help"),         this, SLOT(help()));
	pop->insertItem(__tr("Close &Window"), this, SLOT(close_slot()));
	m_pMenuBar->insertItem(__tr("&Plugin"), pop);

	pop = new KviPopupMenu(this);
	pop->insertItem(__tr("&Load"),  this, SLOT(loadList()));
	pop->insertItem(__tr("&Save"),  this, SLOT(saveList()));
	pop->insertItem(__tr("&Clear"), this, SLOT(clear()));
	m_pMenuBar->insertItem(__tr("&List"), pop);

	m_pUrlList->setShowSortIndicator(true);
	m_pUrlList->addColumn(QString(__tr("URL")));
	m_pUrlList->addColumn(QString(__tr("Window")));
	m_pUrlList->addColumn(QString(__tr("Count")));
	m_pUrlList->addColumn(QString(__tr("Timestamp")));

	cfg.setGroup("colsWidth");
	m_pUrlList->setColumnWidth(0, cfg.readIntEntry("Url",       170));
	m_pUrlList->setColumnWidth(1, cfg.readIntEntry("Window",    130));
	m_pUrlList->setColumnWidth(2, cfg.readIntEntry("Count",      70));
	m_pUrlList->setColumnWidth(3, cfg.readIntEntry("Timestamp",  70));

	connect(m_pUrlList, SIGNAL(doubleClicked(KviListViewItem *)), SLOT(dblclk_url(KviListViewItem *)));
	connect(m_pUrlList,
		SIGNAL(rightButtonPressed(KviListViewItem *, const QPoint &, int)),
		SLOT(popup(KviListViewItem *, const QPoint &, int))
	);

	m_pUrlList->setFocus();
}

void KviUrlDialog::config()
{
	if( !g_pConfigDialog )
		g_pConfigDialog = new KviUrlConfigDialog();
}

void KviUrlDialog::help()
{
	m_pFrm->requestHelpOn("doc_plugin_url.kvihelp");
}

void KviUrlDialog::saveList()
{
	saveUrlList();
}

void KviUrlDialog::loadList()
{
	loadUrlList();
}

void KviUrlDialog::clear()
{
	g_pList->clear();
	for(KviUrlDlgList *tmpitem = g_pUrlDlgList->first(); tmpitem; tmpitem = g_pUrlDlgList->next() ) {
		if( tmpitem->dlg )
			tmpitem->dlg->m_pUrlList->clear();
	}
}

void KviUrlDialog::saveProperties()
{
	KviWindowProperty p;
	p.rect           = externalGeometry();
	p.isDocked       = isAttached();
	p.splitWidth1    = 0;
	p.splitWidth2    = 0;
	p.timestamp      = 0;
	p.imagesVisible  = 0;
	KviWindow *w     = m_pFrm->activeWindow();
	p.isMaximized    = isAttached() && w ? w->isMaximized() : isMaximized();
	p.topSplitWidth1 = 0;
	p.topSplitWidth2 = 0;
	p.topSplitWidth3 = 0;
	g_pOptions->m_pWinPropertiesList->setProperty(caption(), &p);
}

void KviUrlDialog::close_slot()
{
	close();
}

void KviUrlDialog::remove()
{
	if( !m_pUrlList->currentItem() ) {
		kvirc_plugin_warning_box(__tr("Select a URL"));
		return;
	}
	for( KviUrl *tmp = g_pList->first(); tmp; tmp = g_pList->next() ) {
		if( tmp->url == KviStr(m_pUrlList->currentItem()->text(0)) ) {
			g_pList->find(tmp);
			g_pList->remove();
			m_pUrlList->takeItem(m_pUrlList->currentItem());
			return;
		}
	}
}

void KviUrlDialog::findtext()
{
	if( !m_pUrlList->currentItem() ) {
		kvirc_plugin_warning_box(__tr("Select a URL"));
		return;
	}
	for( KviUrl *tmp = g_pList->first(); tmp; tmp = g_pList->next() ) {
		if( tmp->url == KviStr(m_pUrlList->currentItem()->text(0)) ) {
			g_pList->find(tmp);
			KviStr ft = "findtext %";
			ft.replaceAll('%', tmp->url.ptr());
			KviWindow *wnd = m_pFrm->findWindow(tmp->window.ptr());
			if( wnd ) {
				if( kvirc_plugin_execute_command(wnd, ft.ptr()) ) {
					if( wnd->mdiParent() )
						m_pFrm->m_pMdi->setTopChild(wnd->mdiParent(), true);
				}
			} else
				kvirc_plugin_warning_box(__tr("Window not found"));
		}
	}
}

void KviUrlDialog::dblclk_url(KviListViewItem *item)
{
	KviStr tmp = g_pOptions->m_szBrowserCommandline;
	tmp.stripWhiteSpace();
	if( tmp.hasData() ) {
		tmp.replaceAll('%', item->text(0));
		tmp.stripWhiteSpace();
		if( tmp.hasData() ) {
			KviProcess *proc = new KviProcess();
			connect(proc, SIGNAL(processExited(KviProcess *)), this, SLOT(processExited(KviProcess *)));
			if( !proc->run(tmp.ptr()) ) {
				delete proc;
				g_pApp->warningBox(__tr("Could not execute %s"), tmp.ptr());
			}
		}
	}
}

void KviUrlDialog::processExited(KviProcess *proc)
{
	if( proc ) delete proc;
}

void KviUrlDialog::popup(KviListViewItem *item, const QPoint &p, int col)
{
	if( col == 0 ) {
		m_szUrl = item->text(0);
		KviPopupMenu p;
		p.insertItem(__tr("&Remove"),    this, SLOT(remove()));
		p.insertItem(__tr("&Find Text"), this, SLOT(findtext()));
		p.insertSeparator();
		m_pListPopup = new KviPopupMenu();
		int i = 0;
		for( KviWindow *w = m_pFrm->m_pWinList->first(); w; w = m_pFrm->m_pWinList->next() ) {
			if( (w->type() <= KVI_WND_TYPE_CHAT) || (w->type() == KVI_WND_TYPE_UWINDOW) ) {
				m_pListPopup->insertItem(w->caption(), i);
				m_pListPopup->connectItem(i, this, SLOT(sayToWin(int)));
				i++;
			}
		}
		p.insertItem(__tr("&Say to Window"), m_pListPopup);
		p.exec(QCursor::pos());
	}
}

void KviUrlDialog::sayToWin(int itemID)
{
	KviStr say = "say %1 %2";
	say.replaceAll("%1", m_pListPopup->text(itemID).latin1());
	say.replaceAll("%2", m_szUrl.ptr());
	KviWindow *wnd = m_pFrm->findWindow(m_pListPopup->text(itemID).latin1());
	if( wnd ) {
		if( !kvirc_plugin_execute_command(wnd, say.ptr()) ) {
			wnd->raise();
			wnd->setActiveWindow();
			wnd->setFocus();
		}
	} else
		kvirc_plugin_warning_box(__tr("Window not found"));
}

QPixmap *KviUrlDialog::myIconPtr()
{
	if( m_pIcon == 0 )
		m_pIcon = new QPixmap(KviIconLoader::loadIcon("icon_url.png"));
	return m_pIcon;
}

void KviUrlDialog::addUrl(QString url, QString window, QString count, QString timestamp)
{
	KviListViewItem *UrlItem = new KviListViewItem(m_pUrlList);

	UrlItem->setText(0, url);
	UrlItem->setText(1, window);
	UrlItem->setText(2, count);
	UrlItem->setText(3, timestamp);
}

void KviUrlDialog::resizeEvent(QResizeEvent *)
{
	int hght = m_pMenuBar->heightForWidth(width());
	m_pMenuBar->setGeometry(0, 0, width(), hght);
	m_pUrlList->setGeometry(0, hght, width(), height() - hght);
}

KviUrlDialog::~KviUrlDialog()
{
	KviConfig cfg(g_configPath.ptr());
	cfg.setGroup("ConfigDialog");
	if( cfg.readBoolEntry("SaveColumnWidthOnClose", false) ) {
		cfg.setGroup("ColsWidth");
		cfg.writeEntry("Url",       m_pUrlList->columnWidth(0));
		cfg.writeEntry("Window",    m_pUrlList->columnWidth(1));
		cfg.writeEntry("Count",     m_pUrlList->columnWidth(2));
		cfg.writeEntry("Timestamp", m_pUrlList->columnWidth(3));
	}

	delete m_pUrlList;
	m_pUrlList = 0;
	KviUrlDlgList *tmpitem = findFrame(m_pFrm);
	tmpitem->dlg = 0;
	if( m_pIcon ) {
		delete m_pIcon;
		m_pIcon = 0;
	}
}

#include "m_kvi_url_dialog.moc"
