// =============================================================================
//
//   Copyright (C) 1999-2000 Szymon Stefanek
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviXmms"

#define _KVI_XMMS_CPP_

#include <dlfcn.h>

#include <qpainter.h>
#include <qpixmap.h>

#define __KVIRC_PLUGIN__

#include "kvi_dockablewidget.h"
#include "kvi_process.h"
#include "kvi_xmms.h"
#include "kvi_xmms_icons.h"
#include "kvi_xmms_widget.h"
#include "kvirc_plugin.h"

int   (*libxmms_remote_play)(int)                    = 0;
int   (*libxmms_remote_stop)(int)                    = 0;
int   (*libxmms_remote_pause)(int)                   = 0;
bool  (*libxmms_remote_is_playing)(int)              = 0;
bool  (*libxmms_remote_is_paused)(int)               = 0;
bool  (*libxmms_remote_is_running)(int)              = 0;
char *(*libxmms_remote_get_playlist_file)(int, int)  = 0;
char *(*libxmms_remote_get_playlist_title)(int, int) = 0;
int   (*libxmms_remote_get_playlist_pos)(int)        = 0;
int   (*libxmms_remote_get_playlist_length)(int)     = 0;
int   (*libxmms_remote_get_playlist_time)(int, int)  = 0;
void  (*libxmms_remote_playlist_prev)(int)           = 0;
void  (*libxmms_remote_playlist_next)(int)           = 0;
void  (*libxmms_remote_eject)(int)                   = 0;
int   (*libxmms_remote_get_main_volume)(int)         = 0;
void  (*libxmms_remote_set_main_volume)(int, int)    = 0;
void  *libxmms_handle = 0;

bool xmms_plugin_command_xmms(KviPluginCommandStruct *cmd)
{
	if( cmd->params->count() > 1 ) {
		KviStr command = kvirc_plugin_param(cmd, 1);
		if( !libxmms_remote_is_running(0) ) {
			KviProcess proc;
			proc.run("xmms");
		}
		     if( kvi_strEqualCI(command.ptr(), "play")   ) libxmms_remote_play(0);
		else if( kvi_strEqualCI(command.ptr(), "stop")   ) libxmms_remote_stop(0);
		else if( kvi_strEqualCI(command.ptr(), "pause")  ) libxmms_remote_pause(0);
		else if( kvi_strEqualCI(command.ptr(), "prev")   ) libxmms_remote_playlist_prev(0);
		else if( kvi_strEqualCI(command.ptr(), "next")   ) libxmms_remote_playlist_next(0);
		else if( kvi_strEqualCI(command.ptr(), "eject")  ) libxmms_remote_eject(0);
		else if( kvi_strEqualCI(command.ptr(), "undock") ) kvirc_plugin_remove_all_docked_widgets(cmd->handle);
		else if( kvi_strEqualCI(command.ptr(), "dock") ) {
			KviXmmsWidget *w = new KviXmmsWidget(cmd->handle, cmd->frame->m_pStatusBar, cmd->frame);
			kvirc_plugin_add_docked_widget(cmd->handle, cmd->frame, w);
			w->show();
		} else if( kvi_strEqualCI(command.ptr(), "volume") ) {
			if( cmd->params->count() < 3 ) {
				cmd->error = KVI_ERROR_MissingParameter;
				cmd->errorstr = "Missing volume value";
				return false;
			}
			KviStr tmp = kvirc_plugin_param(cmd, 2);
			bool bOk = false;
			int vol = tmp.toInt(&bOk);
			if( !bOk || (vol < 0) ) {
				cmd->error = KVI_ERROR_InvalidParameter;
				cmd->errorstr = "An unsigned volume value is expected";
				return false;
			}
			libxmms_remote_set_main_volume(0, vol);
		} else {
			cmd->error = KVI_ERROR_InvalidParameter;
			cmd->errorstr = "Invalid xmms command";
			return false;
		}
	}
	return true;
}

bool xmms_plugin_function_xmmsIsRunning(KviPluginCommandStruct *cmd, KviStr *buffer)
{
	buffer->append(libxmms_remote_is_running(0) ? '1' : '0');
	return true;
}

bool xmms_plugin_function_xmmsIsPlaying(KviPluginCommandStruct *cmd, KviStr *buffer)
{
	buffer->append(libxmms_remote_is_playing(0) ? '1' : '0');
	return true;
}

bool xmms_plugin_function_xmmsIsPaused(KviPluginCommandStruct *cmd, KviStr *buffer)
{
	buffer->append(libxmms_remote_is_paused(0) ? '1' : '0');
	return true;
}

bool xmms_plugin_function_xmmsCurrentPlaylistPos(KviPluginCommandStruct *cmd, KviStr *buffer)
{
	buffer->setNum(libxmms_remote_get_playlist_pos(0));
	return true;
}

bool xmms_plugin_function_xmmsMainVolume(KviPluginCommandStruct *cmd, KviStr *buffer)
{
	buffer->setNum(libxmms_remote_get_main_volume(0));
	return true;
}

bool xmms_plugin_function_xmmsPlaylistLength(KviPluginCommandStruct *cmd, KviStr *buffer)
{
	buffer->setNum(libxmms_remote_get_playlist_length(0));
	return true;
}

bool xmms_plugin_function_xmmsPlaylistFileName(KviPluginCommandStruct *cmd, KviStr *buffer)
{
	if( cmd->params ) {
		if( cmd->params->count() > 1 ) {
			bool bOk = false;
			KviStr tmp =  kvirc_plugin_param(cmd, 1);
			int pos = tmp.toInt(&bOk);
			if( !bOk ) {
				cmd->error = KVI_ERROR_InvalidParameter;
				cmd->errorstr = "Playlist position as an unsigned number expected";
				return false;
			}
			char *f = libxmms_remote_get_playlist_file(0, pos);
			if( f ) buffer->append(f);
			return true;
		}
	}
	char *f = libxmms_remote_get_playlist_file(0, libxmms_remote_get_playlist_pos(0));
	if( f ) buffer->append(f);
	return true;
}

bool xmms_plugin_function_xmmsPlaylistTitle(KviPluginCommandStruct *cmd, KviStr *buffer)
{
	if( cmd->params ) {
		if( cmd->params->count() > 1 ) {
			bool bOk = false;
			KviStr tmp =  kvirc_plugin_param(cmd, 1);
			int pos = tmp.toInt(&bOk);
			if( !bOk ) {
				cmd->error = KVI_ERROR_InvalidParameter;
				cmd->errorstr = "Playlist position as an unsigned number expected";
				return false;
			}
			char *f = libxmms_remote_get_playlist_title(0, pos);
			if( f ) buffer->append(f);
			return true;
		}
	}
	char *f = libxmms_remote_get_playlist_title(0, libxmms_remote_get_playlist_pos(0));
	if( f ) buffer->append(f);
	return true;
}

bool xmms_plugin_function_xmmsPlaylistTime(KviPluginCommandStruct *cmd, KviStr *buffer)
{
	if( cmd->params ) {
		if( cmd->params->count() > 1 ) {
			bool bOk = false;
			KviStr tmp =  kvirc_plugin_param(cmd, 1);
			int pos = tmp.toInt(&bOk);
			if( !bOk ) {
				cmd->error = KVI_ERROR_InvalidParameter;
				cmd->errorstr = "Playlist position as an unsigned number expected";
				return false;
			}
			int t = libxmms_remote_get_playlist_time(0, pos);
			KviStr str(KviStr::Format, "%d", t);
			buffer->append(str.ptr());
			return true;
		}
	}
	int t = libxmms_remote_get_playlist_time(0, libxmms_remote_get_playlist_pos(0));
	KviStr stri(KviStr::Format, "%d", t);
	if( stri.ptr() ) buffer->append(stri.ptr());
	return true;
}

bool xmms_plugin_init(KviPluginCommandStruct *cmd)
{
	libxmms_handle = dlopen("libxmms.so", RTLD_GLOBAL | RTLD_NOW);
	if( !libxmms_handle ) {
		debug("[libkvixmms]: could not load libxmms: %s", dlerror());
		cmd->errorstr.sprintf("[libkvixmms]: could not load libxmms: %s", dlerror());
		return false;
	}
	libxmms_remote_play = (int (*)(int)) dlsym(libxmms_handle, "xmms_remote_play");
	if( !libxmms_remote_play ) {
		debug("[libkvixmms]: could not find symbol xmms_remote_play: %s", dlerror());
		cmd->errorstr.sprintf("[libkvixmms]: could not find symbol xmms_remote_play: %s", dlerror());
		dlclose(libxmms_handle);
		return false;
	}
	libxmms_remote_stop = (int (*)(int)) dlsym(libxmms_handle, "xmms_remote_stop");
	if( !libxmms_remote_stop ) {
		debug("[libkvixmms]: could not find symbol xmms_remote_stop: %s", dlerror());
		cmd->errorstr.sprintf("[libkvixmms]: could not find symbol xmms_remote_stop: %s", dlerror());
		dlclose(libxmms_handle);
		return false;
	}
	libxmms_remote_playlist_prev = (void (*)(int)) dlsym(libxmms_handle, "xmms_remote_playlist_prev");
	if( !libxmms_remote_playlist_prev ) {
		debug("[libkvixmms]: could not find symbol xmms_remote_playlist_prev: %s", dlerror());
		cmd->errorstr.sprintf("[libkvixmms]: could not find symbol xmms_remote_playlist_prev: %s", dlerror());
		dlclose(libxmms_handle);
		return false;
	}
	libxmms_remote_playlist_next = (void (*)(int)) dlsym(libxmms_handle, "xmms_remote_playlist_next");
	if( !libxmms_remote_playlist_next ) {
		debug("[libkvixmms]: could not find symbol xmms_remote_playlist_next: %s", dlerror());
		cmd->errorstr.sprintf("[libkvixmms]: could not find symbol xmms_remote_playlist_next: %s", dlerror());
		dlclose(libxmms_handle);
		return false;
	}
	libxmms_remote_pause = (int (*)(int)) dlsym(libxmms_handle, "xmms_remote_pause");
	if( !libxmms_remote_pause ) {
		debug("[libkvixmms]: could not find symbol xmms_remote_pause: %s", dlerror());
		cmd->errorstr.sprintf("[libkvixmms]: could not find symbol xmms_remote_pause: %s", dlerror());
		dlclose(libxmms_handle);
		return false;
	}
	libxmms_remote_eject = (void (*)(int)) dlsym(libxmms_handle, "xmms_remote_eject");
	if( !libxmms_remote_eject ) {
		debug("[libkvixmms]: could not find symbol xmms_remote_eject: %s", dlerror());
		cmd->errorstr.sprintf("[libkvixmms]: could not find symbol xmms_remote_eject: %s", dlerror());
		dlclose(libxmms_handle);
		return false;
	}
	libxmms_remote_is_playing = (bool (*)(int)) dlsym(libxmms_handle, "xmms_remote_is_playing");
	if( !libxmms_remote_is_playing ) {
		debug("[libkvixmms]: could not find symbol xmms_remote_playing: %s", dlerror());
		cmd->errorstr.sprintf("[libkvixmms]: could not find symbol xmms_remote_playing: %s", dlerror());
		dlclose(libxmms_handle);
		return false;
	}
	libxmms_remote_is_paused = (bool (*)(int)) dlsym(libxmms_handle, "xmms_remote_is_paused");
	if( !libxmms_remote_is_paused ) {
		debug("[libkvixmms]: could not find symbol xmms_remote_is_paused: %s", dlerror());
		cmd->errorstr.sprintf("[libkvixmms]: could not find symbol xmms_remote_is_paused: %s", dlerror());
		dlclose(libxmms_handle);
		return false;
	}
	libxmms_remote_is_running = (bool (*)(int)) dlsym(libxmms_handle, "xmms_remote_is_running");
	if( !libxmms_remote_is_running ) {
		debug("[libkvixmms]: could not find symbol xmms_remote_is_running: %s", dlerror());
		cmd->errorstr.sprintf("[libkvixmms]: could not find symbol xmms_remote_is_running: %s", dlerror());
		dlclose(libxmms_handle);
		return false;
	}
	libxmms_remote_get_playlist_file = (char * (*)(int, int)) dlsym(libxmms_handle, "xmms_remote_get_playlist_file");
	if( !libxmms_remote_get_playlist_file ) {
		debug("[libkvixmms]: could not find symbol xmms_remote_get_playlist_file: %s", dlerror());
		cmd->errorstr.sprintf("[libkvixmms]: could not find symbol xmms_remote_get_playlist_file: %s", dlerror());
		dlclose(libxmms_handle);
		return false;
	}
	libxmms_remote_get_playlist_time = (int (*)(int, int)) dlsym(libxmms_handle, "xmms_remote_get_playlist_time");
	if( !libxmms_remote_get_playlist_time ) {
		debug("[libkvixmms]: could not find symbol xmms_remote_get_playlist_time: %s", dlerror());
		cmd->errorstr.sprintf("[libkvixmms]: could not find symbol xmms_remote_get_playlist_time: %s", dlerror());
		dlclose(libxmms_handle);
		return false;
	}
	libxmms_remote_get_playlist_title = (char * (*)(int, int)) dlsym(libxmms_handle, "xmms_remote_get_playlist_title");
	if( !libxmms_remote_get_playlist_title ) {
		debug("[libkvixmms]: could not find symbol xmms_remote_get_playlist_title: %s", dlerror());
		cmd->errorstr.sprintf("[libkvixmms]: could not find symbol xmms_remote_get_playlist_title: %s", dlerror());
		dlclose(libxmms_handle);
		return false;
	}
	libxmms_remote_get_playlist_pos = (int (*)(int)) dlsym(libxmms_handle, "xmms_remote_get_playlist_pos");
	if( !libxmms_remote_get_playlist_pos ) {
		debug("[libkvixmms]: could not find symbol xmms_remote_get_playlist_pos: %s", dlerror());
		cmd->errorstr.sprintf("[libkvixmms]: could not find symbol xmms_remote_get_playlist_pos: %s", dlerror());
		dlclose(libxmms_handle);
		return false;
	}
	libxmms_remote_get_playlist_length = (int (*)(int)) dlsym(libxmms_handle, "xmms_remote_get_playlist_length");
	if( !libxmms_remote_get_playlist_length ) {
		debug("[libkvixmms]: could not find symbol xmms_remote_get_playlist_length: %s", dlerror());
		cmd->errorstr.sprintf("[libkvixmms]: could not find symbol xmms_remote_get_playlist_length: %s", dlerror());
		dlclose(libxmms_handle);
		return false;
	}
	libxmms_remote_get_main_volume = (int (*)(int)) dlsym(libxmms_handle, "xmms_remote_get_main_volume");
	if( !libxmms_remote_get_main_volume ) {
		debug("[libkvixmms]: could not find symbol xmms_remote_get_main_volume: %s", dlerror());
		cmd->errorstr.sprintf("[libkvixmms]: could not find symbol xmms_remote_get_main_volume: %s", dlerror());
		dlclose(libxmms_handle);
		return false;
	}
	libxmms_remote_set_main_volume = (void (*)(int, int)) dlsym(libxmms_handle, "xmms_remote_set_main_volume");
	if( !libxmms_remote_set_main_volume ) {
		debug("[libkvixmms]: could not find symbol xmms_remote_set_main_volume: %s", dlerror());
		cmd->errorstr.sprintf("[libkvixmms]: could not find symbol xmms_remote_set_main_volume: %s", dlerror());
		dlclose(libxmms_handle);
		return false;
	}
	g_pXmmsButtonsNormal  = new QPixmap(xmms_buttons_xpm);
	g_pXmmsButtonsActive  = new QPixmap(xmms_buttons_active_xpm);
	g_pXmmsButtonsPressed = new QPixmap(xmms_buttons_pressed_xpm);
	g_pXmmsLogo           = new QPixmap(xmms_logo_xpm);

	kvirc_plugin_register_command (cmd->handle, "XMMS",                   xmms_plugin_command_xmms);
	kvirc_plugin_register_function(cmd->handle, "xmmsIsRunning",          xmms_plugin_function_xmmsIsRunning);
	kvirc_plugin_register_function(cmd->handle, "xmmsIsPlaying",          xmms_plugin_function_xmmsIsPlaying);
	kvirc_plugin_register_function(cmd->handle, "xmmsIsPaused",           xmms_plugin_function_xmmsIsPaused);
	kvirc_plugin_register_function(cmd->handle, "xmmsCurrentPlaylistPos", xmms_plugin_function_xmmsCurrentPlaylistPos);
	kvirc_plugin_register_function(cmd->handle, "xmmsPlaylistFileName",   xmms_plugin_function_xmmsPlaylistFileName);
	kvirc_plugin_register_function(cmd->handle, "xmmsPlaylistTitle",      xmms_plugin_function_xmmsPlaylistTitle);
	kvirc_plugin_register_function(cmd->handle, "xmmsPlaylistTime",       xmms_plugin_function_xmmsPlaylistTime);
	kvirc_plugin_register_function(cmd->handle, "xmmsPlaylistLength",     xmms_plugin_function_xmmsPlaylistLength);
	kvirc_plugin_register_function(cmd->handle, "xmmsMainVolumne",        xmms_plugin_function_xmmsMainVolume);
	return true;
}

void xmms_plugin_cleanup()
{
	if( libxmms_handle ) {
		dlclose(libxmms_handle);
		libxmms_handle = 0;
	}
	delete g_pXmmsButtonsNormal;  g_pXmmsButtonsNormal  = 0;
	delete g_pXmmsButtonsActive;  g_pXmmsButtonsActive  = 0;
	delete g_pXmmsButtonsPressed; g_pXmmsButtonsPressed = 0;
	delete g_pXmmsLogo;           g_pXmmsLogo           = 0;
}

/**
	@document: doc_plugin_xmms.kvihelp
	@title: The XMMS controller plugin
		The XMMS controller plugin is an interface to the
		libxmms library provided with the xmms mp3 player.<br>
		It dynamically loads the libxmms library, so it must
		be found by the dynamic linker at plugin load time.<br>
		If you get any "shared libraries" error while loading this
		plugin, make sure that libxmms is on a path that is found
		int /etc/ld.so.conf.<br>
		If it is not, add that directory and then run ldconfig.<br>
		<docsubtitle>Exported commands</docsubtitle>
		/xmms &lt;option&gt; [parameter]<br>
		Options:<br>
		<b>play</b>   : Starts playing a song in xmms.<br>
		<b>stop</b>   : Acts like pressing the stop button in xmms.<br>
		<b>pause</b>  : Pauses or resumes playing.<br>
		<b>next</b>   : Plays the next song in the playlist.<br>
		<b>prev</b>   : Plays the previous song in the playlist.<br>
		<b>eject</b>  : Acts like pressing the "eject" button in xmms.<br>
		<b>dock</b>   : Runs a controller window that docks in the kvirc status bar.<br>
		<b>undock</b> : Terminates the controller window.<br>
		<b>volume</b> : Sets the main volume to [parameter].<br>
		All these commands start a new xmms session if xmms is not running.<br>
		<docsubtitle>Exported identifiers and functions</docsubtitle>
		<b>$xmmsIsRunning</b> : returns 1 if an xmms session is being run.<br>
		A special word is needed for this identifier.<br>
		It will not detect sessions that are running with a UID different
		from the one KVIrc is running.<br>
		<b>$xmmsIsPlaying</b> : returns 1 if xmms is playing a song, 0 if not.<br>
		<b>$xmmsIsPaused</b>  : returns 1 is xmms is in pause mode.<br>
		<b>$xmmsCurrentPlaylistPos</b> : returns the index of the song that is being played.<br>
		<b>$xmmsPlaylistFileName(&lt;playlist_position&gt;)</b> : returns the file name
		of the song that is at &lt;playlist_position&gt; index in the playlist.<br>
		<b>$xmmsPlaylistTime(&lt;playlist_position&gt;)</b> : returns the duration
		of the song that is at &lt;playlist_position&gt; index in the playlist (in milliseconds).<br>
		<b>$xmmsPlaylistTitle(&lt;playlist_position&gt;)</b> : returns the title
		of the song that is at &lt;playlist_position&gt; index in the playlist.<br>
		<b>$xmmsPlaylistLength</b> : returns the length of the current playlist.<br>
		<b>$xmmsMainVolume</b> : returns the main volume setting.<br>
*/
void xmms_plugin_help()
{
	kvirc_plugin_show_help_page("doc_plugin_xmms.kvihelp");
}

KviPlugin kvirc_plugin =
{
	"XMMS",
	"Remote control for the XMMS program",
	"0.1",
	"Szymon Stefanek <stefanek@tin.it>",
	"<html>"
		"Exports the <b>/XMMS</b> command, "
		"a lot of related identifiers, "
		"and a dockable controller."
		"</html>",
	xmms_plugin_init,
	xmms_plugin_cleanup,
	0,
	xmms_plugin_help
};
