//=============================================================================
//
//   File : kvi_qstring.cpp
//   Creation date : Mon Aug 04 2003 13:36:33 CEST by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2003 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//=============================================================================

//=============================================================================
//
// Helper functions for the QString class
//
//=============================================================================

#define __LIBKVILIB__

#include "kvi_qstring.h"
#include "kvi_string.h"
#include "kvi_malloc.h"

#include <ctype.h> // for tolower()
#include <stdio.h> // for sprintf()

namespace KviQString
{
	bool equalCSN(const QString &sz1,const QString &sz2,unsigned int len)
	{
		if(len == 0)return true; // assume equal
		const QChar * c1 = sz1.unicode();
		const QChar * c2 = sz2.unicode();
		unsigned int lmin = QMIN(sz1.length(),sz2.length());
		if(lmin < len)return false;
		const QChar * c1e = c1 + len;
		
		if(!c1 || !c2)return (c1 == c2);
		
		while(c1 < c1e)
		{
			if(c1->unicode() != c2->unicode())return false;
			c1++;
			c2++;
		}
		return (c1 == c1e);
	}

	bool equalCIN(const QString &sz1,const QString &sz2,unsigned int len)
	{
		if(len == 0)return true; // assume equal
		const QChar * c1 = sz1.unicode();
		const QChar * c2 = sz2.unicode();
		unsigned int lmin = QMIN(sz1.length(),sz2.length());
		if(lmin < len)return false;
		const QChar * c1e = c1 + len;

		if(!c1 || !c2)return (c1 == c2);

		while(c1 < c1e)
		{
			if(c1->lower().unicode() != c2->lower().unicode())return false;
			c1++;
			c2++;
		}
		return (c1 == c1e);
	}

	bool equalCSN(const QString &sz1,const char * sz2,unsigned int len)
	{
		if(len == 0)return true; // assume equal
		const QChar * c1 = sz1.unicode();
		if(sz1.length() < len)return false;
		const QChar * c1e = c1 + len;

		if(!sz2)return !c1;
		if(!c1)return !sz2;

		while((c1 < c1e) && (*sz2))
		{
			if(c1->unicode() != *sz2)return false;
			c1++;
			sz2++;
		}
		return (c1 == c1e);
	}

	bool equalCIN(const QString &sz1,const char * sz2,unsigned int len)
	{
		if(len == 0)return true; // assume equal
		const QChar * c1 = sz1.unicode();
		if(sz1.length() < len)return false;
		const QChar * c1e = c1 + len;

		if(!sz2)return !c1;
		if(!c1)return !(*sz2);

		while((c1 < c1e) && (*sz2))
		{
			if(c1->lower().unicode() != tolower(*sz2))return false;
			c1++;
			sz2++;
		}
		return (c1 == c1e);
	}

	bool equalCS(const QString &sz1,const QString &sz2)
	{
		if(sz1.length() != sz2.length())return false;

		const QChar * c1 = sz1.unicode();
		const QChar * c2 = sz2.unicode();
		const QChar * c1e = c1 + sz1.length();
		
		if(!c1 || !c2)return (c1 == c2);
		
		while(c1 < c1e)
		{
			if(c1->unicode() != c2->unicode())return false;
			c1++;
			c2++;
		}
		return (c1 == c1e);
	}

	bool equalCI(const QString &sz1,const QString &sz2)
	{
		if(sz1.length() != sz2.length())return false;

		const QChar * c1 = sz1.unicode();
		const QChar * c2 = sz2.unicode();
		const QChar * c1e = c1 + sz1.length();

		if(!c1 || !c2)return (c1 == c2);

		while(c1 < c1e)
		{
			if(c1->lower().unicode() != c2->lower().unicode())return false;
			c1++;
			c2++;
		}
		return (c1 == c1e);
	}

	bool equalCS(const QString &sz1,const char * sz2)
	{
		const QChar * c1 = sz1.unicode();
		const QChar * c1e = c1 + sz1.length();

		if(!c1)return !sz2;

		while((c1 < c1e) && (*sz2))
		{
			if(c1->unicode() != *sz2)return false;
			c1++;
			sz2++;
		}
		return ((c1 == c1e) && (*sz2 == '\0'));
	}

	bool equalCI(const QString &sz1,const char * sz2)
	{
		const QChar * c1 = sz1.unicode();
		const QChar * c1e = c1 + sz1.length();

		if(!c1)return !sz2;

		while((c1 < c1e) && (*sz2))
		{
			if(c1->lower().unicode() != tolower(*sz2))return false;
			c1++;
			sz2++;
		}
		return ((c1 == c1e) && (*sz2 == '\0'));
	}

	int cmpCS(const QString &sz1,const QString &sz2)
	{
		const QChar * c1 = sz1.unicode();
		const QChar * c2 = sz2.unicode();
		const QChar * c1e = c1 + sz1.length();
		const QChar * c2e = c2 + sz2.length();

		if(!c1 || !c2)return (c1 == c2);

		for(;;)
		{
			if(c1 >= c1e)
			{
				if(c2 < c2e)return /* 0 */ - (c2->unicode());
				return 0;
			}
			if(c2 >= c2e)return c1->unicode() /* - 0 */;

			int diff = c1->unicode() - c2->unicode();
			if(diff)return diff;

			c1++;
			c2++;
		}

		return 0; // never here
	}

	int cmpCI(const QString &sz1,const QString &sz2)
	{
		const QChar * c1 = sz1.unicode();
		const QChar * c2 = sz2.unicode();
		const QChar * c1e = c1 + sz1.length();
		const QChar * c2e = c2 + sz2.length();

		if(!c1 || !c2)return (c1 == c2);

		for(;;)
		{
			if(c1 >= c1e)
			{
				if(c2 < c2e)return /* 0 */ - (c2->lower().unicode());
				return 0;
			}
			if(c2 >= c2e)return c1->lower().unicode() /* - 0 */;

			int diff = c1->lower().unicode() - c2->lower().unicode();
			if(diff)return diff;

			c1++;
			c2++;
		}

		return 0; // never here
	}

	int cmpCIN(const QString &sz1,const QString &sz2,unsigned int len)
	{
		if(len == 0)return 0; // assume equal
		unsigned int l1 = QMIN(len,sz1.length());
		unsigned int l = QMIN(l1,sz2.length()); // FIXME: THIS IS NOT OK

		const QChar * c1 = sz1.unicode();
		const QChar * c2 = sz2.unicode();
		const QChar * c1e = c1 + l;

		if(!c1 || !c2)return (c1 == c2);
		int diff = 0;

		while((c1 < c1e) && !(diff = (c1->lower().unicode() - c2->lower().unicode())))
		{
			c1++;
			c2++;
		}

		return diff;
	}

	void ensureLastCharIs(QString &szString,const QChar &c)
	{
		if(!lastCharIs(szString,c))szString.append(c);
	}

	void stripRightWhiteSpace(QString &s)
	{
		int iRemove = 0;
		while(s.length() > 0)
		{
			if(s.at(s.length() - (iRemove + 1)).isSpace())iRemove++;
			else break;
		}
		if(iRemove > 0)s.remove(s.length() - iRemove,iRemove);
	}

	void detach(QString &sz)
	{
		sz.setLength(sz.length());
	}

	const QChar * nullTerminatedArray(const QString &sz)
	{
		//sz.setLength(sz.length()); // detach!
		return (const QChar *)sz.ucs2(); // MAY BE NULL!
	}
	
	void appendNumber(QString &s,double dReal)
	{
		char buffer[64];
		::sprintf(buffer,"%f",dReal);
		s.append(buffer);
	}
	
	void appendNumber(QString &s,int iInteger)
	{
		char buffer[32];
		::sprintf(buffer,"%d",iInteger);
		s.append(buffer);
	}
	
	void appendNumber(QString &s,unsigned int uInteger)
	{
		char buffer[32];
		::sprintf(buffer,"%u",uInteger);
		s.append(buffer);
	}

	void vsprintf(QString &s,const QString &szFmt,kvi_va_list list)
	{
#define MEMINCREMENT 32

		int reallen = 0;
		int allocsize = MEMINCREMENT;

		//s.setLength(allocsize);

		const QChar * fmt = nullTerminatedArray(szFmt);
		if(!fmt)
		{
			s = QString::null;
			return;
		}

		QChar * buffer = (QChar *)kvi_malloc(sizeof(QChar) * allocsize);
		//QChar * p = (QChar *)s.unicode();

		char *argString;
		long argValue;
		unsigned long argUValue;
	
		//9999999999999999999999999999999\0
		char numberBuffer[32]; //enough ? 10 is enough for 32bit unsigned int...
		char *pNumBuf;
		unsigned int tmp;

		QChar * p = buffer;

#define INCREMENT_MEM \
		{ \
			allocsize += MEMINCREMENT; \
			buffer = (QChar *)kvi_realloc(buffer,sizeof(QChar) * allocsize); \
			p = buffer + reallen; \
		}

#define INCREMENT_MEM_BY(numchars) \
		{ \
			allocsize += numchars + MEMINCREMENT; \
			buffer = (QChar *)kvi_realloc(buffer,sizeof(QChar) * allocsize); \
			p = buffer + reallen; \
		}


		for(; fmt->unicode() ; ++fmt)
		{
			if(reallen == allocsize)INCREMENT_MEM

			//copy up to a '%'
			if(*fmt != '%')
			{
				*p++ = *fmt;
				reallen++;
				continue;
			}

			++fmt; //skip this '%'
			switch(*fmt)
			{
				case 's': // char * string
				{
					argString = kvi_va_arg(list,char *);
					if(!argString)
					{
						argString = "[!NULL!]";
						argValue = 8;
					} else {
						argValue = (long)strlen(argString);
					}
					//check for space...
					if((allocsize - reallen) < argValue)INCREMENT_MEM_BY(argValue)

					while(*argString)*p++ = QChar(*argString++);
					reallen += argValue;
					continue;
				}
				case 'S': // KviStr * string
				{
					KviStr * str = kvi_va_arg(list,KviStr *);
					if(!str)continue;
					if((allocsize - reallen) < str->len())INCREMENT_MEM_BY(str->len())
					argString = str->ptr();
					while(*argString)*p++ = QChar(*argString++);
					reallen += str->len();
					continue;
				}
				case 'Q': // QString * string
				{
					QString * str = kvi_va_arg(list,QString *);
					if(!str)continue;
					if(str->isEmpty())continue;
					int len = str->length();
					if((allocsize - reallen) < len)INCREMENT_MEM_BY(len)
					const QChar * ch = str->unicode();
					while(len--)*p++ = *ch++;
					reallen += str->length();
					continue;
				}
				case 'c': //char
				{
					//
					// I'm not sure about this...
					// In the linux kernel source the
					// unsigned char is extracted from an integer type.
					// We assume that gcc stacks a char argument
					// as sizeof(int) bytes value.
					// Is this always true ?
					//
					*p++ = (char)kvi_va_arg(list,int);
					reallen++;
					continue;
				}
				case 'q': // QChar *
				{
					//
					// I'm not sure about this...
					// In the linux kernel source the
					// unsigned char is extracted from an integer type.
					// We assume that gcc stacks a char argument
					// as sizeof(int) bytes value.
					// Is this always true ?
					//
					*p++ = *((QChar *)kvi_va_arg(list,QChar *));
					reallen++;
					continue;
				}
				case 'd': //signed integer
				{
					argValue = kvi_va_arg(list,int);
					if(argValue < 0)
					{ //negative integer
						*p++ = '-';
						reallen++;
						argValue = -argValue; //need to have it positive
						// most negative integer exception (avoid completely senseless (non digit) responses)
						if(argValue < 0)argValue = 0;  //we get -0 here
					}
					//write the number in a temporary buffer
					pNumBuf = numberBuffer;
					do {
						tmp = argValue / 10;
						*pNumBuf++ = argValue - (tmp * 10) + '0';
					} while((argValue = tmp));
					//copy now....
					argUValue = pNumBuf - numberBuffer; //length of the number string
					if((allocsize - reallen) < (int)argUValue)INCREMENT_MEM_BY(argUValue)
					do { *p++ = QChar(*--pNumBuf); } while(pNumBuf != numberBuffer);
					reallen += argUValue;
					continue;
				}
				case 'u': //unsigned integer
				{
					argUValue = kvi_va_arg(list,unsigned int); //many implementations place int here
					//write the number in a temporary buffer
					pNumBuf = numberBuffer;
					do {
						tmp = argUValue / 10;
						*pNumBuf++ = argUValue - (tmp * 10) + '0';
					} while((argUValue = tmp));
					//copy now....
					argValue = pNumBuf - numberBuffer; //length of the number string
					if((allocsize - reallen) < argValue)INCREMENT_MEM_BY(argValue)
					do { *p++ = *--pNumBuf; } while(pNumBuf != numberBuffer);
					reallen += argValue;
					continue;
				}
				case 'h':
				case 'x': // hexadecimal unsigned integer
				{
					static char hexsmalldigits[]="0123456789abcdef";
					argUValue = kvi_va_arg(list,unsigned int); //many implementations place int here
					//write the number in a temporary buffer
					pNumBuf = numberBuffer;
					do {
						tmp = argUValue / 16;
						*pNumBuf++ = hexsmalldigits[argUValue - (tmp * 16)];
					} while((argUValue = tmp));
					//copy now....
					argValue = pNumBuf - numberBuffer; //length of the number string
					if((allocsize - reallen) < argValue)INCREMENT_MEM_BY(argValue)
					do { *p++ = *--pNumBuf; } while(pNumBuf != numberBuffer);
					reallen += argValue;
					continue;
				}
				case 'H':
				case 'X': // hexadecimal unsigned integer
				{
					static char hexbigdigits[]="0123456789ABCDEF";
					argUValue = kvi_va_arg(list,unsigned int); //many implementations place int here
					//write the number in a temporary buffer
					pNumBuf = numberBuffer;
					do {
						tmp = argUValue / 16;
						*pNumBuf++ = hexbigdigits[argUValue - (tmp * 16)];
					} while((argUValue = tmp));
					//copy now....
					argValue = pNumBuf - numberBuffer; //length of the number string
					if((allocsize - reallen) < argValue)INCREMENT_MEM_BY(argValue)
					do { *p++ = *--pNumBuf; } while(pNumBuf != numberBuffer);
					reallen += argValue;
					continue;
				}
				default: //a normal percent followed by some char
				{
					*p++ = '%';  //write it
					reallen++;
					if(*fmt)
					{
						if(reallen == allocsize)INCREMENT_MEM
						*p++ = *fmt;
						reallen++;
					}
					continue;
				}
			}
		}

		s.setUnicode(buffer,reallen);
		kvi_free(buffer);
		//s.squeeze();
	}


	QString & sprintf(QString &s,const QString &szFmt,...)
	{
		kvi_va_list list;
		kvi_va_start_by_reference(list,szFmt);
		//print...with max 256 chars
		KviQString::vsprintf(s,szFmt,list);
		kvi_va_end(list);
		return s;
	}

	void appendFormatted(QString &s,const QString &szFmt,...)
	{
		QString tmp;
		kvi_va_list list;
		kvi_va_start_by_reference(list,szFmt);
		//print...with max 256 chars
		KviQString::vsprintf(tmp,szFmt,list);
		kvi_va_end(list);
		s.append(tmp);
	}

	bool matchWildExpressionsCI(const QString &szM1,const QString &szM2)
	{
		//Matches two regular expressions containging wildcards (* and ?)
	
		//          s1
		//          m1
		// mask1 : *xor
		// mask2 : xorand*xor
		//         m2
		//          s2
	
		//                        s2
		//                       m2
		//                       | 
		// XorT!xor@111.111.111.11
		//
		// *!*@*.net
		//      |
		//      m1
		//      s1
		//
		const QChar * m1 = nullTerminatedArray(szM1);
		const QChar * m2 = nullTerminatedArray(szM2);


		if(!(m1 && m2 && (*m1)))return false;
		const QChar * savePos1 = 0;
		const QChar * savePos2 = m2;
		while(*m1)
		{
			//loop managed by m1 (initially first mask)
			if(*m1=='*')
			{
				//Found a wildcard in m1
				savePos1 = ++m1;            //move to the next char and save the position...this is our jolly
				if(!*savePos1)return true;  //last was a wildcard , matches everything ahead...
				savePos2 = m2+1;            //next return state for the second string
				continue;                   //and return
			}
			if(!(*m2))return false;         //m2 finished and we had something to match here!
			if(m1->lower()==m2->lower())
			{
				//chars matched
				m1++;                       //Go ahead in the two strings
				m2++;                       //
				if((!(*m1)) && *m2 && savePos1)
				{
					//m1 finished , but m2 not yet and we have a savePosition for m1 (there was a wildcard)...
					//retry matching the string following the * from the savePos2 (one char ahead last time)
					m1 = savePos1;          //back to char after wildcard
					m2 = savePos2;          //back to last savePos2
					savePos2++;             //next savePos2 will be next char
				}
			} else {
				if(*m2 == '*')
				{
					//A wlidcard in the second string
					//Invert the game : mask1 <-> mask2
					//mask2 now leads the game...
					savePos1 = m1;          //aux
					m1 = m2;                //...swap
					m2 = savePos1;          //...swap
					savePos1 = m1;          //sync save pos1
					savePos2 = m2 + 1;      //sync save pos2
					continue;               //...and again
				}
				// m1 != m2 , m1 != * , m2 != *
				if((*m1 == '?') || (*m2 == '?'))
				{
					m1++;
					m2++;
					if((!(*m1)) && *m2 && savePos1)
					{
						//m1 finished , but m2 not yet and we have a savePosition for m1 (there was a wildcard)...
						//retry matching the string following the * from the savePos2 (one char ahead last time)
						m1 = savePos1;          //back to char after wildcard
						m2 = savePos2;          //back to last savePos2
						savePos2++;             //next savePos2 will be next char
					}
				} else {
					if(savePos1)
					{
						//Have a jolly man...allow not matching...
						m1 = savePos1;          //go back to char after wildcard...need to rematch...
						m2 = savePos2;          //back to last savePos2
						savePos2++;             //and set next savePos2
					} else return false;        //No previous wildcards...not matched!
				}
			}
		}
		return (!(*m2));                     //m1 surely finished , so for the match , m2 must be finished too
	}


	bool matchStringCI(const QString &szExp,const QString &szStr)
	{
		//               a
		//               .
		// exp = a*x?mem*a
		// str = arexoxmexamemizazv
		//                         .
		//                         n
		const QChar * exp = nullTerminatedArray(szExp);
		const QChar * str = nullTerminatedArray(szStr);
		const QChar * afterWild = 0;
		const QChar * nextStrToCheck = 0;

		if(!(exp && str))return false;
	
		while(*exp)
		{
			if(*exp == '*')
			{
				// exp is a wildcard...
				afterWild = ++exp;
				nextStrToCheck = str + 1;
				if(!(*exp))return true; // and it's the last char in the string: matches everything ahead
				continue;
			}
	
			if(!(*str))return false; // str finished but we had something to match :(
	
			if(exp->lower() == str->lower())
			{
				// chars matched
				++exp;
				++str;
				if((!(*exp)) && *str)goto check_recovery;
				continue;
			}
	
			if(*exp == '?')
			{
				// any-char wildcard
				++exp;
				++str;
				continue;
			}
	
	check_recovery:
			// chars unmatched!!!
			if(afterWild)
			{
				// we had a wildcard in exp...
				// let's use this jolly then
				exp = afterWild;
				str = nextStrToCheck;
				nextStrToCheck++;
				// and try to compare now
				continue;
			}
	
			return false; // no match :(
		}
		return (!(*str));
	}

	void cutFromFirst(QString &s,const QChar &c,bool bIncluded)
	{
		int idx = s.find(c);
		if(idx == -1)return;
		s.truncate(bIncluded ? idx : idx + 1);
	}

	void cutFromLast(QString &s,const QChar &c,bool bIncluded)
	{
		int idx = s.findRev(c);
		if(idx == -1)return;
		s.truncate(bIncluded ? idx : idx + 1);
	}
	
	void cutToFirst(QString &s,const QChar &c,bool bIncluded,bool bClearIfNotFound)
	{
		int idx = s.find(c);
		if(idx == -1)
		{
			if(bClearIfNotFound)s = "";
			return;
		}
		s.remove(0,bIncluded ? idx + 1 : idx);
	}
	
	void cutToLast(QString &s,const QChar &c,bool bIncluded,bool bClearIfNotFound)
	{
		int idx = s.findRev(c);
		if(idx == -1)
		{
			if(bClearIfNotFound)s = "";
			return;
		}
		s.remove(0,bIncluded ? idx + 1 : idx);
	}

	void cutFromFirst(QString &s,const QString &c,bool bIncluded)
	{
		int idx = s.find(c);
		if(idx == -1)return;
		s.truncate(bIncluded ? idx : idx + c.length());
	}

	void cutFromLast(QString &s,const QString &c,bool bIncluded)
	{
		int idx = s.findRev(c);
		if(idx == -1)return;
		s.truncate(bIncluded ? idx : idx + c.length());
	}
	
	void cutToFirst(QString &s,const QString &c,bool bIncluded,bool bClearIfNotFound)
	{
		int idx = s.find(c);
		if(idx == -1)
		{
			if(bClearIfNotFound)s = "";
			return;
		}
		s.remove(0,bIncluded ? idx + c.length() : idx);
	}
	
	void cutToLast(QString &s,const QString &c,bool bIncluded,bool bClearIfNotFound)
	{
		int idx = s.findRev(c);
		if(idx == -1)
		{
			if(bClearIfNotFound)s = "";
			return;
		}
		s.remove(0,bIncluded ? idx + c.length() : idx);
	}

};
