#ifndef _KVI_REGUSERSDB_H_
#define _KVI_REGUSERSDB_H_
//=================================================================================================
//
//   File : kvi_regusersdb.h
//   Creation date : Sat Sep 09 2000 15:30:56 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2000-2004 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//=================================================================================================

//
// REGISTERED USERS
//     
//     Here we manage users resigered by mask and their (generic!) properties
//

#include "kvi_settings.h"
#include "kvi_heapobject.h"
#include "kvi_qstring.h"
#include "kvi_ircmask.h"
#include "kvi_debug.h"

#include "kvi_list.h"
#include <qdict.h>


class KviRegisteredUserDataBase;

//=================================================================================================
//
// KviRegisteredUser
//

class KVILIB_API KviRegisteredUser : public KviHeapObject
{
	friend class KviRegisteredUserDataBase;
public:
	KviRegisteredUser(const QString &name);
	~KviRegisteredUser();
private:
	QString                      m_szName;
	QDict<QString>             * m_pPropertyDict;   // owned properties
	KviPtrList<KviIrcMask>     * m_pMaskList;       // owned masks
protected:
	// mask ownership is transferred! (always!) returns false if the mask was already there
	bool addMask(KviIrcMask * mask);
	bool removeMask(KviIrcMask * mask);
	KviIrcMask * findMask(const KviIrcMask &mask);
public:
	const QString &name(){ return m_szName; };
	bool matches(const KviIrcMask &mask);
	bool matchesFixed(const KviIrcMask &mask);
	bool matchesFixed(const QString &nick,const QString &user,const QString &host);

	void setProperty(const QString &name,const QString &value);

	const QString & getProperty(const QString &name);       // returns 0 if the property is not there
	bool getProperty(const QString &name,QString &value); // returns false if the property is not there
	bool getBoolProperty(const QString &name);           // returns true if the property is there and is true
	// the propertyDict may be 0!
	QDict<QString> * propertyDict(){ return m_pPropertyDict; };
	// this is never zero (but may contain no masks)
	KviPtrList<KviIrcMask> * maskList(){ return m_pMaskList; };
};

//============================================================================================================
//
// KviRegisteredMask
//

class KVILIB_API KviRegisteredMask
{
private:
	KviRegisteredUser * m_pUser;               // pointer , not owned!
	KviIrcMask        * m_pMask;               // pointer , not owned!
	int                 m_iMaskNonWildChars;
public:
	KviRegisteredMask(KviRegisteredUser * u,KviIrcMask * m);
	~KviRegisteredMask(){};
public:
	int                 nonWildChars(){ return m_iMaskNonWildChars; };
	KviRegisteredUser * user(){ return m_pUser; };
	KviIrcMask        * mask(){ return m_pMask; };
};

typedef KviPtrList<KviRegisteredMask> KviRegisteredMaskList;

//=================================================================================================
//
// KviRegisteredUsersDb
//
//    Manages a set of KviRegisteredUser instances stored in the m_pUserDict dictionary
//    The users are identified by masks stored in m_pMaskDict and m_pWildMaskList
//    m_pMaskDict contains lists of non wild-nick KviRegisteredMask that point to users
//    m_pWildMaskList is a list of wild-nick KviRegisteredMask that point to users
//

class KVILIB_API KviRegisteredUserDataBase
{
public:
	KviRegisteredUserDataBase();
	~KviRegisteredUserDataBase();
private:
	QDict<KviRegisteredUser>     * m_pUserDict; // unique namespace, owns the objects, does not copy keys
	QDict<KviRegisteredMaskList> * m_pMaskDict; // owns the objects, copies the keys
	KviRegisteredMaskList        * m_pWildMaskList; // owns the objects
public:
	void copyFrom(KviRegisteredUserDataBase * db);
	KviRegisteredUser * addUser(const QString &name); // returns 0 if already there
	KviRegisteredUser * getUser(const QString &name); // returns existing or adds
	bool removeUser(const QString &name);
	KviRegisteredUser * findUserByName(const QString &name){ return m_pUserDict->find(name); };
	// mask must be allocated on the heap and the ownership is transferred!
	// returns non zero if there is already an user with this mask (returns the pointer to it!)
	KviRegisteredUser * addMask(KviRegisteredUser * u,KviIrcMask * mask);
	bool removeMaskByPointer(KviIrcMask * mask);
	bool removeMask(const KviIrcMask &mask);
	KviRegisteredUser * findMatchingUser(const QString &nick,const QString &user,const QString &host);
	KviRegisteredUser * findUserWithMask(const KviIrcMask &mask);
	KviRegisteredMask * findExactMask(const KviIrcMask &mask);
	KviRegisteredMask * findMatchingMask(const QString &nick,const QString &user,const QString &host);
	//Only used in few places (actually one) of the code, but lot of times;perfect for inlining...
	//bool isIgnoredUser(const char * nick,const char * user,const char * host);
	void load(const QString &filename);
	void save(const QString &filename);

	QDict<KviRegisteredUser> * userDict(){ return m_pUserDict; };
};


#endif //_KVI_REGUSERSDB_H_
