//=============================================================================
//
//   File : kvi_kvs_corefunctions_gl.cpp
//   Created on Fri 31 Oct 2003 01:52:04 by Szymon Stefanek
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 2003 Szymon Stefanek <pragma at kvirc dot net>
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//=============================================================================

#define __KVIRC__

#include "kvi_kvs_corefunctions.h"

#include "kvi_kvs_kernel.h"

#include "kvi_kvs_timermanager.h"

#include "kvi_locale.h"

namespace KviKvsCoreFunctions
{
	///////////////////////////////////////////////////////////////////////////////////////////////

	/*
		@doc: hash
		@type:
			function
		@title:
			$array
		@short:
			Explicitly creates a hash
		@syntax:
			$hash(<key>,<value>,<key>,<value>,....);
		@description:
			Returns a hash with the specified items. Each <value> is indexed by the preceeding <key>.
			This is just an explicit way of creating a hash with a defined set of items,
			useful for increasing readability. 
		@examples:
			[cmd]alias[/cmd](test){ [cmd]return[/cmd] $hash(1,X,2,Y,3,Z); };
			%x = $test();
			[cmd]foreach[/cmd](%y,%x)
			{
				[cmd]echo[/cmd] %y;
			}
			[cmd]foreach[/cmd](%y,[fnc]$keys[/fnc](%x))
			{
				[cmd]echo[/cmd] %y
			}
		@seealso:
			[fnc]$array[/fnc]
	*/

	_KVS_COREFUNCTION(hash)
	{
		KviKvsHash * a = new KviKvsHash();
		
		for(KviKvsVariant * key = pParams->first();key;key = pParams->next())
		{
			KviKvsVariant * val = pParams->next();
			if(!val)break;
			QString sz;
			key->asString(sz);
			a->set(sz,new KviKvsVariant(*val));
		}
		
		pRetBuffer->setHash(a);
		return true;
	}

	///////////////////////////////////////////////////////////////////////////////////////////////

	/*
		@doc: isempty
		@type:
			function
		@title:
			$isEmpty
		@short:
			Checks if a variable is set (empty or non empty)
		@syntax:
			$isEmpty(<data>)
		@description:
			Returns 0 if <data> is actually set to some non-empty value
			and 1 otherwise. Since KVIrc treats unset variables as empty
			ones then this function is the exact opposite of [fnc]$isSet[/fnc].
		@seealso:
			[fnc]$isEmpty[/fnc]
	*/

	_KVS_COREFUNCTION(isEmpty)
	{
		KviKvsVariant * v = pParams->first();
		if(!v)
		{
			c->error(__tr2qs("Missing data to evaluate"));
			return false;
		}
		pRetBuffer->setInteger(v->isEmpty() ? 1 : 0);
		return true;
	}

	///////////////////////////////////////////////////////////////////////////////////////////////

	/*
		@doc: isnumeric
		@type:
			function
		@title:
			$isNumeric
		@short:
			Finds whether a variable contains a rappresentation of a number
		@syntax:
			$isNumeric(<data>)
		@description:
			Returns 1 if the <data> is an integer or a real number, 0 otherwise.
	*/

	_KVS_COREFUNCTION(isNumeric)
	{
		KviKvsVariant * v = pParams->first();
		if(!v)
		{
			c->error(__tr2qs("Missing data to evaluate"));
			return false;
		}
		KviKvsNumber n;
		pRetBuffer->setInteger(v->asNumber(n) ? 1 : 0);
		return true;
	}

	///////////////////////////////////////////////////////////////////////////////////////////////

	/*
		@doc: isset
		@type:
			function
		@title:
			$isSet
		@short:
			Checks if a variable is set (non-empty)
		@syntax:
			$isSet(<data>)
		@description:
			Returns 1 if <data> is actually set to some non-empty value
			and 0 otherwise. If <data> is a variable , then this function
			simply checks if the variable is set. If <data> is a constant
			then this function checks if the constant is non empty. 
			Since KVIrc treats empty strings as "unset" values then
			this function could be also called "isNonEmpty" and it is
			the perfect opposite of [fnc]$isEmpty[/fnc]
		@seealso:
			[fnc]$isEmpty[/fnc]
	*/

	_KVS_COREFUNCTION(isSet)
	{
		KviKvsVariant * v = pParams->first();
		if(!v)
		{
			c->error(__tr2qs("Missing data to evaluate"));
			return false;
		}
		pRetBuffer->setInteger(v->isEmpty() ? 0 : 1);
		return true;
	}

	///////////////////////////////////////////////////////////////////////////////////////////////

	/*
		@doc: istimer
		@type:
			function
		@title:
			$isTimer
		@short:
			Checks for a timer existence
		@syntax:
			$isTimer(<name>)
		@description:
			Returns 1 if the timer named <name> is actually running, 0 otherwise
		@seealso:
			[cmd]timer[/cmd],[cmd]killtimer[/cmd]
	*/

	_KVS_COREFUNCTION(isTimer)
	{
		KviKvsVariant * vName = pParams->first();
		if(!vName || vName->isEmpty())
		{
			c->error(__tr2qs("Missing timer name"));
			return false;
		}
		QString szName;
		vName->asString(szName);
		pRetBuffer->setInteger(KviKvsTimerManager::instance()->timerExists(szName) ? 1 : 0);
		return true;
	}

	///////////////////////////////////////////////////////////////////////////////////////////////

	/*
		@doc: keys
		@type:
			function
		@title:
			$keys
		@short:
			Returns the array of keys of a hash
		@syntax:
			$keys(<hash>)
		@description:
			Returns an array with the keys of the <hash> parameter.
			<hash> must be obviously a hash (or eventually an empty variable
			that is treated as an empty hash).
		@seealso:
			[cmd]foreach[/cmd]
	*/

	_KVS_COREFUNCTION(keys)
	{
		KviKvsVariant * vHash = pParams->first();
		if(!vHash)
		{
			c->error(__tr2qs("Missing hash for key evaluation"));
			return false;
		}
		if(vHash->isEmpty())
		{
			// an empty anything is a valid empty hash too
			// there are no keys at all
			pRetBuffer->setArray(new KviKvsArray());
			return true;
		}
		if(!vHash->isHash())
		{
			QString szType;
			vHash->getTypeName(szType);
			c->error(__tr2qs("The argument evaluated to type '%Q' while a hash was expected"),&szType);
			return false;
		}
		KviKvsArray * a = new KviKvsArray();
		int idx = 0;
		KviKvsHashIterator it(*(vHash->hash()->dict()));
		while(it.current())
		{
			a->set(idx,new KviKvsVariant(it.currentKey()));
			idx++;
			++it;
		}
		pRetBuffer->setArray(a);
		return true;
	}

	///////////////////////////////////////////////////////////////////////////////////////////////

};

