//=============================================================================
//
//   File : kvi_kvs_popupmenu.cpp
//   Created on Wed 07 Jan 2004 05:02:57 by Szymon Stefanek
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 2004 Szymon Stefanek <pragma at kvirc dot net>
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//=============================================================================

#define __KVIRC__

#include "kvi_kvs_popupmenu.h"
#include "kvi_locale.h"
#include "kvi_iconmanager.h"
#include "kvi_out.h"
#include "kvi_app.h"
#include "kvi_kvs_popupmanager.h"
#include "kvi_console.h"
#include "kvi_config.h"
#include "kvi_cmdformatter.h"

// popup names

// rootname : the root popup
//       rootname.anonymousmenuX : child popups with no name
//       rootname.externalrootname : child popups copied from external menus
//       rootname.itemX : child items
//       rootname.separatorX : child separators
//       rootname.labelX : child labels


KviKvsPopupMenuItem::KviKvsPopupMenuItem(Type t,const QString &szItemName,const QString &szCondition)
{
	m_szItemName = szItemName;
	m_eType = t;
	if(szCondition.isEmpty())
	{
		// true by default
		m_pKvsCondition = 0;
	} else {
		QString szName = "condition callback for ";
		szName += szItemName;
		m_pKvsCondition = new KviKvsScript(szName,new QString(szCondition),KviKvsScript::Expression);
	}
}

KviKvsPopupMenuItem::~KviKvsPopupMenuItem()
{
	if(m_pKvsCondition)delete m_pKvsCondition;
}

void KviKvsPopupMenuItem::clear()
{
}

KviKvsScript * KviKvsPopupMenuItem::kvsIcon()
{
	return 0;
}

KviKvsScript * KviKvsPopupMenuItem::kvsText()
{
	return 0;
}

KviKvsScript * KviKvsPopupMenuItem::kvsCode()
{
	return 0;
}

bool KviKvsPopupMenuItem::evaluateCondition(KviKvsPopupMenuTopLevelData * pData)
{
	if(!m_pKvsCondition)return true;
	KviKvsVariant vRet;

	if(!m_pKvsCondition->run(pData->window(),
			pData->parameters(),
			&vRet,
			KviKvsScript::PreserveParams,
			pData->extendedRunTimeData()))
	{
		// broken condition
		pData->window()->output(KVI_OUT_PARSERWARNING,__tr2qs("Broken condition in menu setup: assuming false"));
		return false;
	}
	return vRet.asBoolean();
}


///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

KviKvsPopupMenuItemSeparator::KviKvsPopupMenuItemSeparator(const QString &szItemName,const QString &szCondition)
: KviKvsPopupMenuItem(KviKvsPopupMenuItem::Separator,szItemName,szCondition)
{
}

KviKvsPopupMenuItemSeparator::~KviKvsPopupMenuItemSeparator()
{
}

void KviKvsPopupMenuItemSeparator::fill(KviKvsPopupMenu * pMenu,KviKvsPopupMenuTopLevelData * pData,int iIdx)
{
	if(!evaluateCondition(pData))return;
	pMenu->insertSeparator();
}

KviKvsPopupMenuItem * KviKvsPopupMenuItemSeparator::clone() const
{
	return new KviKvsPopupMenuItemSeparator(m_szItemName,m_pKvsCondition->code());
}


/////////////////////////////////////////////////////////////////////////////////////////////////////////


KviKvsPopupMenuItemWithTextAndIcon::KviKvsPopupMenuItemWithTextAndIcon(KviKvsPopupMenuItem::Type t,const QString &szItemName,const QString &szText,const QString &szIcon,const QString &szCondition)
: KviKvsPopupMenuItem(t,szItemName,szCondition)
{
	QString szName = "text callback for ";
	szName += szItemName;
	m_pKvsText = new KviKvsScript(szName,new QString(szText),KviKvsScript::Parameter);

	if(szIcon.isEmpty())
	{
		m_pKvsIcon = 0;
	} else {
		szName = "icon callback for ";
		szName += szItemName;
		m_pKvsIcon = new KviKvsScript(szName,new QString(szIcon),KviKvsScript::Parameter);
	}
}

KviKvsPopupMenuItemWithTextAndIcon::~KviKvsPopupMenuItemWithTextAndIcon()
{
	delete m_pKvsText;
	if(m_pKvsIcon)delete m_pKvsIcon;
}


KviKvsScript * KviKvsPopupMenuItemWithTextAndIcon::kvsIcon()
{
	return m_pKvsIcon;
}

KviKvsScript * KviKvsPopupMenuItemWithTextAndIcon::kvsText()
{
	return m_pKvsText;
}

QPixmap * KviKvsPopupMenuItemWithTextAndIcon::evaluateIcon(KviKvsPopupMenuTopLevelData * pData)
{
	if(!m_pKvsIcon)return 0;

	KviKvsVariant vRet;
	if(!m_pKvsIcon->run(pData->window(),
			pData->parameters(),
			&vRet,
			KviKvsScript::PreserveParams,
			pData->extendedRunTimeData()))
	{
		// broken text
		pData->window()->output(KVI_OUT_PARSERWARNING,__tr2qs("Broken icon parameter: ignoring"));
		return 0;
	}

	QString szRet;
	vRet.asString(szRet);

	QPixmap * p = g_pIconManager->getImage(szRet);
	if(!p)pData->window()->output(KVI_OUT_PARSERWARNING,__tr2qs("Can't find the icon \"%Q\": ignoring"),&szRet);
	return p;
}

QString KviKvsPopupMenuItemWithTextAndIcon::evaluateText(KviKvsPopupMenuTopLevelData *pData)
{
	QString szRet;
	if(!m_pKvsText)return szRet;
	KviKvsVariant vRet;
	if(!m_pKvsText->run(pData->window(),
			pData->parameters(),
			&vRet,
			KviKvsScript::PreserveParams,
			pData->extendedRunTimeData()))
	{
		// broken text
		pData->window()->output(KVI_OUT_PARSERWARNING,__tr2qs("Broken text parameter: assuming empty string"));
		return szRet;
	}
	vRet.asString(szRet);
	return szRet;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////

KviKvsPopupMenuItemLabel::KviKvsPopupMenuItemLabel(const QString &szItemName,const QString &szText,const QString &szIcon,const QString &szCondition)
: KviKvsPopupMenuItemWithTextAndIcon(KviKvsPopupMenuItem::Label,szItemName,szText,szIcon,szCondition)
{
	m_pLabel = 0;
}

KviKvsPopupMenuItemLabel::~KviKvsPopupMenuItemLabel()
{
	if(m_pLabel)delete m_pLabel;
}

KviKvsPopupMenuItem * KviKvsPopupMenuItemLabel::clone() const
{
	return new KviKvsPopupMenuItemLabel(m_szItemName,m_pKvsText->code(),m_pKvsIcon ? m_pKvsIcon->code() : QString::null,m_pKvsCondition->code());
}

void KviKvsPopupMenuItemLabel::fill(KviKvsPopupMenu * pMenu,KviKvsPopupMenuTopLevelData * pData,int iIdx)
{
	if(!evaluateCondition(pData))return;
	QString szText = evaluateText(pData);
	QPixmap * pPix = evaluateIcon(pData);
	if(m_pLabel)delete m_pLabel;
	m_pLabel = new QLabel(szText,pMenu);
	pMenu->insertItem(m_pLabel);
	m_pLabel->setFrameStyle(QFrame::StyledPanel | QFrame::Raised);
	if(pPix)m_pLabel->setPixmap(*pPix);
}


////////////////////////////////////////////////////////////////////////////////////////////////////////////

KviKvsPopupMenuItemItem::KviKvsPopupMenuItemItem(const QString &szItemName,const QString &szCode,const QString &szText,const QString &szIcon,const QString &szCondition)
: KviKvsPopupMenuItemWithTextAndIcon(KviKvsPopupMenuItem::Item,szItemName,szText,szIcon,szCondition)
{
	QString szName = "click callback for ";
	szName += szItemName;
	m_pKvsCode = new KviKvsScript(szName,new QString(szCode));
}

KviKvsPopupMenuItemItem::~KviKvsPopupMenuItemItem()
{
	delete m_pKvsCode;
}

void KviKvsPopupMenuItemItem::fill(KviKvsPopupMenu * pMenu,KviKvsPopupMenuTopLevelData * pData,int iIdx)
{
	if(!evaluateCondition(pData))return;
	QString szText = evaluateText(pData);
	QPixmap * pPix = evaluateIcon(pData);
	int id;
	if(pPix)id = pMenu->insertItem(*pPix,szText);
	else id = pMenu->insertItem(szText);
	pMenu->setItemParameter(id,iIdx);
}

KviKvsPopupMenuItem * KviKvsPopupMenuItemItem::clone() const
{
	return new KviKvsPopupMenuItemItem(m_szItemName,m_pKvsCode->code(),m_pKvsText->code(),m_pKvsIcon ? m_pKvsIcon->code() : QString::null,m_pKvsCondition->code());
}

KviKvsScript * KviKvsPopupMenuItemItem::kvsCode()
{
	return m_pKvsCode;
}


/////////////////////////////////////////////////////////////////////////////////////////////////////////////

KviKvsPopupMenuItemMenu::KviKvsPopupMenuItemMenu(const QString &szItemName,KviKvsPopupMenu * pMenu,const QString &szText,const QString &szIcon,const QString &szCondition)
: KviKvsPopupMenuItemWithTextAndIcon(KviKvsPopupMenuItem::Menu,szItemName,szText,szIcon,szCondition)
{
	m_pMenu = pMenu;
}

KviKvsPopupMenuItemMenu::~KviKvsPopupMenuItemMenu()
{
	delete m_pMenu;
}

KviKvsPopupMenuItem * KviKvsPopupMenuItemMenu::clone() const
{
	KviKvsPopupMenu * copy = new KviKvsPopupMenu(m_pMenu->name());
	copy->copyFrom(m_pMenu);
	return new KviKvsPopupMenuItemMenu(m_szItemName,copy,m_pKvsText->code(),m_pKvsIcon ? m_pKvsIcon->code() : QString::null,m_pKvsCondition->code());
}


void KviKvsPopupMenuItemMenu::fill(KviKvsPopupMenu * pMenu,KviKvsPopupMenuTopLevelData * pData,int iIdx)
{
	if(!evaluateCondition(pData))return;
	QString szText = evaluateText(pData);
	QPixmap * pPix = evaluateIcon(pData);
	int id;
	if(pPix)id = pMenu->insertItem(*pPix,szText,m_pMenu);
	else id = pMenu->insertItem(szText,m_pMenu);
	pMenu->setItemParameter(id,iIdx);
}

void KviKvsPopupMenuItemMenu::clear()
{
	m_pMenu->clearMenuContents();
}



////////////////////////////////////////////////////////////////////////////////////////////////////////////


KviKvsPopupMenuItemExtMenu::KviKvsPopupMenuItemExtMenu(const QString &szItemName,const QString &szMenuName,const QString &szText,const QString &szIcon,const QString &szCondition)
: KviKvsPopupMenuItemWithTextAndIcon(KviKvsPopupMenuItem::ExtMenu,szItemName,szText,szIcon,szCondition)
{
	m_szMenuName = szMenuName;
	m_pMenu = 0;
}

KviKvsPopupMenuItemExtMenu::~KviKvsPopupMenuItemExtMenu()
{
	if(m_pMenu)delete m_pMenu;
}

void KviKvsPopupMenuItemExtMenu::clear()
{
	if(m_pMenu)
	{
		delete m_pMenu;
		m_pMenu = 0;
	}
}

KviKvsPopupMenuItem * KviKvsPopupMenuItemExtMenu::clone() const
{
	return new KviKvsPopupMenuItemExtMenu(m_szItemName,m_szMenuName,m_pKvsText->code(),m_pKvsIcon ? m_pKvsIcon->code() : QString::null,m_pKvsCondition->code());
}


void KviKvsPopupMenuItemExtMenu::fill(KviKvsPopupMenu * pMenu,KviKvsPopupMenuTopLevelData * pData,int iIdx)
{
	if(!evaluateCondition(pData))return;
	QString szText = evaluateText(pData);
	QPixmap * pPix = evaluateIcon(pData);

	KviKvsPopupMenu * source = KviKvsPopupManager::instance()->lookup(m_szMenuName);
	if(source)
	{
		if(source->isLocked())
		{
			pData->window()->output(KVI_OUT_PARSERWARNING,__tr2qs("Recursive definition detected for popup '%Q': ignoring"),&(pMenu->popupName()));
			return;
		}
		QString tmp;
		KviQString::sprintf(tmp,"%Q.%Q",&(pMenu->popupName()),&m_szMenuName);
		if(m_pMenu)delete m_pMenu;
		m_pMenu = new KviKvsPopupMenu(tmp);
		m_pMenu->copyFrom(source);
		m_pMenu->setParentPopup(pMenu);
		int id;
		if(pPix)id = pMenu->insertItem(*pPix,szText,m_pMenu);
		else id = pMenu->insertItem(szText,m_pMenu);
		pMenu->setItemParameter(id,iIdx);
	} else {
		pData->window()->output(KVI_OUT_PARSERWARNING,__tr2qs("Can't find the external popup '%Q'; ignoring"),&m_szMenuName);
	}
}






KviKvsPopupMenuTopLevelData::KviKvsPopupMenuTopLevelData(KviKvsVariantList * pParameters,KviWindow * pWindow)
{
	m_pExtendedRunTimeData = new KviKvsExtendedRunTimeData(new KviKvsHash());
	m_pParameters = pParameters;
	m_pWindow = pWindow;
}

KviKvsPopupMenuTopLevelData::~KviKvsPopupMenuTopLevelData()
{
	delete m_pExtendedRunTimeData;
	delete m_pParameters;
}







KviKvsPopupMenu::KviKvsPopupMenu(const QString &szName)
:QPopupMenu(0,szName)
{
	m_szName = szName;
	m_pItemList = new KviPtrList<KviKvsPopupMenuItem>;
	m_pItemList->setAutoDelete(true);
	m_pPrologues = new KviPtrList<KviKvsScript>;
	m_pPrologues->setAutoDelete(true);
	m_pEpilogues = new KviPtrList<KviKvsScript>;
	m_pEpilogues->setAutoDelete(true);
	m_pParentPopup = 0;
	m_pTopLevelData = 0;
	m_pTempTopLevelData = 0;
	m_bSetupDone = false;
	connect(this,SIGNAL(activated(int)),this,SLOT(itemClicked(int)));
	connect(this,SIGNAL(aboutToShow()),this,SLOT(setupMenuContents()));
}


KviKvsPopupMenu::~KviKvsPopupMenu()
{
	delete m_pItemList;
	delete m_pPrologues;
	delete m_pEpilogues;
	if(m_pTopLevelData)delete m_pTopLevelData;
	if(m_pTempTopLevelData)delete m_pTempTopLevelData;
}


void KviKvsPopupMenu::copyFrom(const KviKvsPopupMenu * src)
{
	doClear();

	for(KviKvsScript * se = src->m_pEpilogues->first();se;se = src->m_pEpilogues->next())
	{
		m_pEpilogues->append(new KviKvsScript(*se));
	}

	for(KviKvsScript * sp = src->m_pPrologues->first();sp;sp = src->m_pPrologues->next())
	{
		m_pPrologues->append(new KviKvsScript(*sp));
	}


	for(const KviKvsPopupMenuItem * it = src->m_pItemList->first();it;it = src->m_pItemList->next())
	{
		addItemInternal(it->clone());
	}
}

void KviKvsPopupMenu::addPrologue(const QString &szCode)
{
	// FIXME: translate this or not ?
	QString szName = "prologue for ";
	szName += m_szName;
	m_pPrologues->append(new KviKvsScript(szName,new QString(szCode)));
}

void KviKvsPopupMenu::addEpilogue(const QString &szCode)
{
	// FIXME: translate this or not ?
	QString szName = "epilogue for ";
	szName += m_szName;
	m_pEpilogues->append(new KviKvsScript(szName,new QString(szCode)));
}

KviKvsPopupMenuTopLevelData * KviKvsPopupMenu::topLevelData()
{
	if(parentPopup())return parentPopup()->topLevelData();
	return m_pTopLevelData;
}


bool KviKvsPopupMenu::isLocked()
{
	if(topLevelPopup()->isVisible())return true;
	KviKvsPopupMenuTopLevelData * d = topLevelData();
	return d ? d->isLocked() : false;
}


KviKvsPopupMenu * KviKvsPopupMenu::topLevelPopup()
{
	if(parentPopup())return parentPopup();
	return this;
}


KviKvsPopupMenu * KviKvsPopupMenu::addPopup(const QString &szText,const QString &szIcon,const QString &szCondition)
{
	QString szName;
	KviQString::sprintf(szName,"%Q.subpopup%d",&m_szName,m_pItemList->count());
	KviKvsPopupMenu * pNew = new KviKvsPopupMenu(szName);
	pNew->setParentPopup(this);
	addItemInternal(new KviKvsPopupMenuItemMenu(szName,pNew,szText,szIcon,szCondition));
	return pNew;
}

void KviKvsPopupMenu::addSeparator(const QString &szCondition)
{
	QString szName;
	KviQString::sprintf(szName,"%Q.separator%d",&m_szName,m_pItemList->count());
	addItemInternal(new KviKvsPopupMenuItemSeparator(szName,szCondition));
}

void KviKvsPopupMenu::addLabel(const QString &szText,const QString &szIcon,const QString &szCondition)
{
	QString szName;
	KviQString::sprintf(szName,"%Q.label%d",&m_szName,m_pItemList->count());
	addItemInternal(new KviKvsPopupMenuItemLabel(szName,szText,szIcon,szCondition));
}

void KviKvsPopupMenu::addItem(const QString &szCode,const QString &szText,const QString &szIcon,const QString &szCondition)
{
	QString szName;
	KviQString::sprintf(szName,"%Q.item%d",&m_szName,m_pItemList->count());
	addItemInternal(new KviKvsPopupMenuItemItem(szName,szCode,szText,szIcon,szCondition));

}

void KviKvsPopupMenu::addExtPopup(const QString &szName,const QString szText,const QString &szIcon,const QString &szCondition)
{
	QString szItemName;
	KviQString::sprintf(szItemName,"%Q.%Q",&m_szName,&szName);
	addItemInternal(new KviKvsPopupMenuItemExtMenu(szItemName,szName,szText,szIcon,szCondition));

}

void KviKvsPopupMenu::addItemInternal(KviKvsPopupMenuItem * it)
{
	if(isLocked())debug("Ooops... KviKvsPopupMenu is locked in ::addItem()");
	m_pItemList->append(it);
}

void KviKvsPopupMenu::doPopup(const QPoint & pnt,KviWindow * wnd,KviKvsVariantList * pParams)
{
	// This might be a compat problem later :(((((
	// it is also an ugly trick
	clearMenuContents();
	m_pTempTopLevelData = new KviKvsPopupMenuTopLevelData(pParams,wnd);
	QPopupMenu::popup(pnt);
}

void KviKvsPopupMenu::clearMenuContents()
{
	m_bSetupDone = false;

	clear();

	for(KviKvsPopupMenuItem * it = m_pItemList->first();it;it = m_pItemList->next())
	{
		it->clear();
	}

	if(m_pTopLevelData)
	{
		delete m_pTopLevelData;
		m_pTopLevelData = 0;
	}
	if(m_pTempTopLevelData)
	{
		delete m_pTempTopLevelData;
		m_pTempTopLevelData = 0;
	}
}

void KviKvsPopupMenu::doClear()
{
	clear();
	if(m_pTopLevelData)
	{
		delete m_pTopLevelData;
		m_pTopLevelData = 0;
	}
	if(m_pTempTopLevelData)
	{
		delete m_pTempTopLevelData;
		m_pTempTopLevelData = 0;
	}
	m_bSetupDone = false;
	m_pItemList->clear();

	m_pPrologues->clear();
	m_pEpilogues->clear();
}


void KviKvsPopupMenu::lock(bool bLock)
{
	KviKvsPopupMenuTopLevelData * d = topLevelData();
	if(!d)return;
	d->setLocked(bLock);
}


void KviKvsPopupMenu::setupMenuContents()
{
	// This might be a compat problem later :((((

	if(parentPopup() == 0)
	{
		if(m_pTempTopLevelData == 0)
		{
			// We have been called by a KviMenuBar!
			// m_bSetupDone is not valid here
			clearMenuContents();
			m_pTopLevelData = new KviKvsPopupMenuTopLevelData(new KviKvsVariantList(),g_pActiveWindow);
		} else {
			if(m_bSetupDone)return;
			// we have been called by doPopup
			// the menu contents have been already cleared
			if(m_pTopLevelData)debug("Ops.. something got messed in KviKvsPopupMenu activation system");
			// Swap the top level data from temporary to the permanent
			m_pTopLevelData = m_pTempTopLevelData;
			m_pTempTopLevelData = 0;
		}
	} else {
		if(m_bSetupDone)return;
	}

	m_bSetupDone = true;


	// HACK...this is to remove the separator inserted by Qt when popup() is called and the popup is empty
	clear();


	KviKvsPopupMenuTopLevelData * d = topLevelData();

	if(!d)
	{
		debug("Ops...menu contents changed behind my back!");
		return;
	}


	lock(true);


	if(!g_pApp->windowExists(d->window()))d->setWindow(g_pApp->activeConsole());

	executePrologues(d);

	// Fill this menu contents
	int idx = 0;
	for(KviKvsPopupMenuItem * it = m_pItemList->first();it;it = m_pItemList->next())
	{
		it->fill(this,d,idx);
		++idx;
	}

	executeEpilogues(d);

	lock(false);

}

void KviKvsPopupMenu::executePrologues(KviKvsPopupMenuTopLevelData * pData)
{
	for(KviKvsScript * s = m_pPrologues->first();s;s = m_pPrologues->next())
	{
		if(!s->run(pData->window(),
			pData->parameters(),
			0,
			KviKvsScript::PreserveParams,
			pData->extendedRunTimeData()))
		{
			pData->window()->output(KVI_OUT_PARSERWARNING,__tr2qs("Broken prologue in popup menu '%Q': ignoring"),&m_szName);
		}
	}
}

void KviKvsPopupMenu::executeEpilogues(KviKvsPopupMenuTopLevelData * pData)
{
	for(KviKvsScript * s = m_pEpilogues->first();s;s = m_pEpilogues->next())
	{
		if(!s->run(pData->window(),
			pData->parameters(),
			0,
			KviKvsScript::PreserveParams,
			pData->extendedRunTimeData()))
		{
			pData->window()->output(KVI_OUT_PARSERWARNING,__tr2qs("Broken epilogue in popup menu '%Q': ignoring"),&m_szName);
		}
	}
}


void KviKvsPopupMenu::itemClicked(int itemId)
{
	int param = itemParameter(itemId);
	KviKvsPopupMenuItem * it = m_pItemList->at(param);
	KviKvsPopupMenuTopLevelData * d = topLevelData();
	if(it && d)
	{
		if(it->isItem())
		{
			// rebind if window is lost
			if(!g_pApp->windowExists(d->window()))d->setWindow(g_pApp->activeConsole());

			// FIXME: we could avoid locking since scripts can be shared now!
			// see KviKvsTimerManager implementation
			lock(true);
			((KviKvsPopupMenuItemItem *)it)->kvsCode()->run(
					d->window(),
					d->parameters(),
					0,
					KviKvsScript::PreserveParams,
					d->extendedRunTimeData());
			// FIXME: should we print somethng if run() returns false ?
			lock(false);
			
		} else debug("oops....clicked something that is not an item at position %d",param);
		// FIXME: #warning "Maybe tell that the window has changed"
	} else debug("oops....no menu item at position %d",param);
	// UGLY Qt 3.0.0.... we can't clear menu contents here :(
//#if QT_VERSION < 300
//	topLevelPopup()->clearMenuContents();
//#endif
}


void KviKvsPopupMenu::load(const QString &prefix,KviConfig * cfg)
{
	doClear();

	int cnt;
	int idx;

	QString tmp = prefix;
	tmp.append("_PrologueCount");

	cnt = cfg->readIntEntry(tmp,0);

	if(cnt > 0)
	{
		for(idx = 0;idx < cnt;idx++)
		{
			KviQString::sprintf(tmp,"%Q_Prologue%d",&(prefix),idx);
			QString pr = cfg->readQStringEntry(tmp,"");
			if(!pr.isEmpty())addPrologue(pr);
		}
	} else {
		// Might be old version!
		KviQString::sprintf(tmp,"%Q_Prologue",&(prefix));
		QString pr = cfg->readQStringEntry(tmp,"");
		if(!pr.isEmpty())addPrologue(pr);
	}

	KviQString::sprintf(tmp,"%Q_EpilogueCount",&prefix);
	cnt = cfg->readIntEntry(tmp,0);

	if(cnt > 0)
	{
		for(idx = 0;idx < cnt;idx++)
		{
			KviQString::sprintf(tmp,"%Q_Epilogue%d",&prefix,idx);
			QString ep = cfg->readQStringEntry(tmp,"");
			if(!ep.isEmpty())addEpilogue(ep);
		}
	} else {
		// Might be old version!
		KviQString::sprintf(tmp,"%Q_Epilogue",&prefix);
		QString ep = cfg->readQStringEntry(tmp,"");
		if(!ep.isEmpty())addEpilogue(ep);
	}


	KviQString::sprintf(tmp,"%Q_Count",&prefix);

	cnt = cfg->readIntEntry(tmp,0);

	for(idx = 0;idx < cnt;idx++)
	{
		QString pre;
		KviQString::sprintf(pre,"%Q_%d",&prefix,idx);
		KviQString::sprintf(tmp,"%Q_Type",&pre);
		
		int type = cfg->readIntEntry(tmp,3);
		switch(type)
		{
			case 0: // separator
			{
				QString expr;
				KviQString::sprintf(tmp,"%Q_Expr",&pre);
				expr = cfg->readQStringEntry(tmp,"");
				addSeparator(expr);
			}
			break;
			case 1: // item
			{
				QString text,icon,code,expr;
				KviQString::sprintf(tmp,"%Q_Text",&pre);
				text = cfg->readQStringEntry(tmp,"Unnamed");
				KviQString::sprintf(tmp,"%Q_Icon",&pre);
				icon = cfg->readQStringEntry(tmp,"");
				KviQString::sprintf(tmp,"%Q_Code",&pre);
				code = cfg->readQStringEntry(tmp,"");
				KviQString::sprintf(tmp,"%Q_Expr",&pre);
				expr = cfg->readQStringEntry(tmp,"");
				addItem(code,text,icon,expr);
			}
			break;
			case 2: // menu
			{
				QString text,icon,expr;
				KviQString::sprintf(tmp,"%Q_Text",&pre);
				text = cfg->readQStringEntry(tmp,"Unnamed");
				KviQString::sprintf(tmp,"%Q_Icon",&pre);
				icon = cfg->readQStringEntry(tmp,"");
				KviQString::sprintf(tmp,"%Q_Expr",&pre);
				expr = cfg->readQStringEntry(tmp,"");
				KviKvsPopupMenu * pop = addPopup(text,icon,expr);
				pop->load(pre,cfg);
			}
			break;
			case 3: // label
			{
				QString text,icon,expr;
				KviQString::sprintf(tmp,"%Q_Text",&pre);
				text = cfg->readQStringEntry(tmp,"Unnamed");
				KviQString::sprintf(tmp,"%Q_Icon",&pre);
				icon = cfg->readQStringEntry(tmp,"");
				KviQString::sprintf(tmp,"%Q_Expr",&pre);
				expr = cfg->readQStringEntry(tmp,"");
				addLabel(text,icon,expr);
			}
			break;
			case 4: // extmenu
			{
				QString text,icon,code,expr;
				KviQString::sprintf(tmp,"%Q_Text",&pre);
				text = cfg->readQStringEntry(tmp,"Unnamed");
				KviQString::sprintf(tmp,"%Q_Icon",&pre);
				icon = cfg->readQStringEntry(tmp,"");
				KviQString::sprintf(tmp,"%Q_ExtName",&pre);
				code = cfg->readQStringEntry(tmp,"");
				KviQString::sprintf(tmp,"%Q_Expr",&pre);
				expr = cfg->readQStringEntry(tmp,"");
				addExtPopup(code,text,icon,expr);
			}
			break;
			default: // ignore
			break;
		}
	}

}

// FIXME: #warning "NOBODY EDITS THE POPUPS IN THE CONFIG!...A binary config would be faster and work better for sure here"

void KviKvsPopupMenu::save(const QString & prefix,KviConfig * cfg)
{
	int idx;

	KviKvsScript * s;
	QString tmp;

	KviQString::sprintf(tmp,"%Q_PrologueCount",&prefix);
	cfg->writeEntry(tmp,m_pPrologues->count());

	idx = 0;
	for(s = m_pPrologues->first();s;s = m_pPrologues->next())
	{
		KviQString::sprintf(tmp,"%Q_Prologue%d",&prefix,idx);
		cfg->writeEntry(tmp,s->code());
		idx++;
	}

	KviQString::sprintf(tmp,"%Q_EpilogueCount",&prefix);
	cfg->writeEntry(tmp,m_pEpilogues->count());

	idx = 0;
	for(s = m_pEpilogues->first();s;s = m_pEpilogues->next())
	{
		KviQString::sprintf(tmp,"%Q_Epilogue%d",&prefix,idx);
		cfg->writeEntry(tmp,s->code());
		idx++;
	}

	KviQString::sprintf(tmp,"%Q_Count",&prefix);
	cfg->writeEntry(tmp,m_pItemList->count());
	idx = 0;


	for(KviKvsPopupMenuItem * it = m_pItemList->first();it;it = m_pItemList->next())
	{
		QString pre;
		KviQString::sprintf(pre,"%Q_%d",&prefix,idx);
		KviQString::sprintf(tmp,"%Q_Type",&pre);
		int typeCode = 0;
		switch(it->type())
		{
			case KviKvsPopupMenuItem::Label:        typeCode = 3; break;
			case KviKvsPopupMenuItem::Separator:    typeCode = 0; break;
			case KviKvsPopupMenuItem::Menu:         typeCode = 2; break;
			case KviKvsPopupMenuItem::Item:         typeCode = 1; break;
			case KviKvsPopupMenuItem::ExtMenu:      typeCode = 4; break;
		}

		cfg->writeEntry(tmp,typeCode);

		s = it->kvsCondition();
		if(s)
		{
			KviQString::sprintf(tmp,"%Q_Expr",&pre);
			cfg->writeEntry(tmp,s->code());
		}

		s = it->kvsIcon();
		if(s)
		{
			KviQString::sprintf(tmp,"%Q_Icon",&pre);
			cfg->writeEntry(tmp,s->code());
		}
		
		s = it->kvsText();
		if(s)
		{
			KviQString::sprintf(tmp,"%Q_Text",&pre);
			cfg->writeEntry(tmp,s->code());
		}
		
		s = it->kvsCode();
		if(s)
		{
			KviQString::sprintf(tmp,"%Q_Code",&pre);
			cfg->writeEntry(tmp,s->code());
		}

		if(it->isMenu())
		{
			((KviKvsPopupMenuItemMenu *)it)->menu()->save(pre,cfg);
		} else if(it->isExtMenu())
		{
			KviQString::sprintf(tmp,"%Q_ExtName",&pre);
			cfg->writeEntry(tmp,((KviKvsPopupMenuItemExtMenu *)it)->extName());
		}

		++idx;
	}
}

void KviKvsPopupMenu::generateDefPopupCore(QString &buffer)
{

	QString tmp;

	buffer = "";

	KviKvsScript * s;
	
	for(s = m_pPrologues->first();s;s = m_pPrologues->next())
	{
		buffer.append("prologue\n");
		tmp = s->code();
		tmp.stripWhiteSpace();
		KviCommandFormatter::blockFromBuffer(tmp);
		buffer.append(tmp);
		buffer.append('\n');
	}

	for(KviKvsPopupMenuItem * it = m_pItemList->first();it;it = m_pItemList->next())
	{
		switch(it->type())
		{
			case KviKvsPopupMenuItem::Item:
				if(it->kvsIcon())KviQString::appendFormatted(buffer,"item(%Q,%Q)",&(it->kvsText()->code()),&(it->kvsIcon()->code()));
				else KviQString::appendFormatted(buffer,"item(%Q)",&(it->kvsText()->code()));
				if(it->kvsCondition())KviQString::appendFormatted(buffer," (%Q)",&(it->kvsCondition()->code()));
				buffer.append("\n");
				tmp = it->kvsCode()->code();
				KviCommandFormatter::blockFromBuffer(tmp);
				buffer.append(tmp);
				buffer.append("\n");
			break;
			case KviKvsPopupMenuItem::Menu:
				if(it->kvsIcon())KviQString::appendFormatted(buffer,"popup(%Q,%Q)",&(it->kvsText()->code()),&(it->kvsIcon()->code()));
				else KviQString::appendFormatted(buffer,"popup(%Q)",&(it->kvsText()->code()));
				if(it->kvsCondition())KviQString::appendFormatted(buffer," (%Q)",&(it->kvsCondition()->code()));
				buffer.append("\n");
				((KviKvsPopupMenuItemMenu *)it)->menu()->generateDefPopupCore(tmp);
				KviCommandFormatter::blockFromBuffer(tmp);
				buffer.append(tmp);
				buffer.append("\n");
			break;
			case KviKvsPopupMenuItem::Separator:
				if(it->kvsCondition())KviQString::appendFormatted(buffer,"separator(%Q)\n\n",&(it->kvsCondition()->code()));
				else buffer.append("separator\n\n");
			break;
			case KviKvsPopupMenuItem::Label:
				if(it->kvsIcon())KviQString::appendFormatted(buffer,"label(%Q,%Q)",&(it->kvsText()->code()),&(it->kvsIcon()->code()));
				else KviQString::appendFormatted(buffer,"label(%Q)",&(it->kvsText()->code()));
				if(it->kvsCondition())KviQString::appendFormatted(buffer," (%Q)",&(it->kvsCondition()->code()));
				buffer.append("\n\n");
			break;
			case KviKvsPopupMenuItem::ExtMenu:
				if(it->kvsIcon())KviQString::appendFormatted(buffer,"extpopup(%Q,%Q,%Q)",&(it->kvsText()->code()),&(((KviKvsPopupMenuItemExtMenu *)it)->extName()),&(it->kvsIcon()->code()));
				else KviQString::appendFormatted(buffer,"extpopup(%Q)",&(it->kvsText()->code()));
				if(it->kvsCondition())KviQString::appendFormatted(buffer," (%Q)",&(it->kvsCondition()->code()));
				buffer.append("\n\n");
			break;
		}
	}

	for(s = m_pEpilogues->first();s;s = m_pEpilogues->next())
	{
		buffer.append("epilogue\n");
		tmp = s->code();
		tmp.stripWhiteSpace();
		KviCommandFormatter::blockFromBuffer(tmp);
		buffer.append(tmp);
		buffer.append('\n');
	}

}

void KviKvsPopupMenu::generateDefPopup(QString &buffer)
{
	KviQString::sprintf(buffer,"defpopup(%s)\n",name());
	QString core;

	generateDefPopupCore(core);
	KviCommandFormatter::blockFromBuffer(core);
	buffer.append(core);
}






