//=============================================================================
//
//   File : kvi_kvs_script.cpp
//   Creation date : Thu 25 Sep 2003 05.12 CEST by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2003 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//=============================================================================

#define __KVIRC__

#include "kvi_kvs_script.h"
#include "kvi_kvs_parser.h"
#include "kvi_kvs_report.h"
#include "kvi_kvs_runtimecontext.h"
#include "kvi_kvs_treenode_instruction.h"
#include "kvi_kvs_variantlist.h"
#include "kvi_kvs_kernel.h"

#include "kvi_locale.h"
#include "kvi_out.h"


class KVIRC_API KviKvsScriptData
{
	friend class KviKvsScript;
protected:
	unsigned int                m_uRefs;     // Reference count for this structure

	QString                     m_szName;    // script context name
	QString                   * m_pszBuffer; // NEVER TOUCH THIS
	const QChar               * m_pBuffer;   // this points to m_szBuffer: use it to extract string data

	KviKvsScript::ScriptType    m_eType;     // the type of the code in m_pszBuffer

	KviKvsTreeNodeInstruction * m_pTree;     // syntax tree
	unsigned int                m_uLock;     // this is increased while the script is being executed
};



//#warning "THERE IS SOME MESS WITH m_szBuffer and m_pBuffer : with some script copying we may get errors with negative char indexes!"

KviKvsScript::KviKvsScript(const QString &szName,QString * pszBuffer,ScriptType eType)
{
	m_pData = new KviKvsScriptData;
	m_pData->m_uRefs = 1;
	m_pData->m_szName = szName;
	m_pData->m_eType = eType;
	m_pData->m_pszBuffer = pszBuffer;
	if(m_pData->m_pszBuffer->isNull())*(m_pData->m_pszBuffer) = "";
	//KviQString::detach(*(m_pData->m_pszBuffer));
	m_pData->m_pBuffer = KviQString::nullTerminatedArray(*(m_pData->m_pszBuffer)); // never 0
	m_pData->m_uLock = 0;
	m_pData->m_pTree = 0;
}

KviKvsScript::KviKvsScript(const QString &szName,QString * pszBuffer,KviKvsTreeNodeInstruction * pPreparsedTree,ScriptType eType)
{
	m_pData = new KviKvsScriptData;
	m_pData->m_uRefs = 1;
	m_pData->m_szName = szName;
	m_pData->m_pszBuffer = pszBuffer;
	m_pData->m_eType = eType;
	if(m_pData->m_pszBuffer->isNull())*(m_pData->m_pszBuffer) = "";
	//KviQString::detach(*(m_pData->m_pszBuffer));
	m_pData->m_pBuffer = KviQString::nullTerminatedArray(*(m_pData->m_pszBuffer)); // never 0
	m_pData->m_uLock = 0;
	m_pData->m_pTree = pPreparsedTree;
}

KviKvsScript::KviKvsScript(const KviKvsScript &src)
{
	m_pData = src.m_pData;
	m_pData->m_uRefs++;
}

KviKvsScript::~KviKvsScript()
{
	if(m_pData->m_uRefs < 2)
	{
		if(m_pData->m_uLock)debug("WARNING: Destroying a locked KviKvsScript");
		if(m_pData->m_pTree)delete m_pData->m_pTree;
		delete m_pData->m_pszBuffer;
		delete m_pData;
	} else {
		m_pData->m_uRefs--;
	}
}

void KviKvsScript::setName(const QString &szName)
{
	m_pData->m_szName = szName;
}

const QString & KviKvsScript::name() const
{
	return m_pData->m_szName;
}

const QString & KviKvsScript::code() const
{
	return *(m_pData->m_pszBuffer);
}

bool KviKvsScript::locked() const
{
	return m_pData->m_uLock > 0;
}

void KviKvsScript::dump(const char * prefix)
{
	if(m_pData->m_pTree)m_pData->m_pTree->dump(prefix);
	else debug("%s KviKvsScript : no tree to dump",prefix);
}

bool KviKvsScript::run(KviWindow * pWnd,KviKvsVariantList * pParams,KviKvsVariant * pRetVal,int iRunFlags,KviKvsExtendedRunTimeData * pExtData)
{
	if(!m_pData->m_pTree)
	{
		if(!parse(pWnd,iRunFlags))
		{
			if(pParams && !(iRunFlags & PreserveParams))delete pParams;
			return false;
		}
	}
	
	return execute(pWnd,pParams,pRetVal,iRunFlags,pExtData);
}

bool KviKvsScript::parse(KviWindow * pOutput,int iRunFlags)
{
	if(m_pData->m_pTree)
	{
		// there is already a tree
		// if we have more than one ref, detach!
		if(m_pData->m_uRefs > 1)
		{
			// mmmh.. more than one ref! .. detach
			m_pData->m_uRefs--;
			KviKvsScriptData * d = new KviKvsScriptData;
			d->m_uRefs = 1;
			d->m_eType = m_pData->m_eType;
			d->m_pszBuffer = new QString(*(m_pData->m_pszBuffer));
			if(d->m_pszBuffer->isNull())*(d->m_pszBuffer) = "";
			KviQString::detach(*(d->m_pszBuffer));
			d->m_pBuffer = KviQString::nullTerminatedArray(*(d->m_pszBuffer)); // never 0
			d->m_uLock = 0;
			m_pData = d;
		} else {
			// only a single ref: we're the owner of the tree
			if(m_pData->m_uLock)
			{
				// ops... someone is locked in THIS script object
				debug("WARNING: Trying to reparse a locked KviKvsScript!");
				return false;
			}
			if(m_pData->m_pTree)delete m_pData->m_pTree;
            m_pData->m_pTree = 0;
			delete m_pData->m_pszBuffer;
            m_pData->m_pszBuffer = 0;
		}
	} // else there is no tree at all, nobody can be locked inside

	KviKvsParser p;
	// parse never blocks
	switch(m_pData->m_eType)
	{
		case Expression:
			m_pData->m_pTree = p.parseAsExpression(m_pData->m_pBuffer,(iRunFlags & AssumeGlobals) ? KviKvsParser::AssumeGlobals : 0);
		break;
		case Parameter:
			m_pData->m_pTree = p.parseAsParameter(m_pData->m_pBuffer,(iRunFlags & AssumeGlobals) ? KviKvsParser::AssumeGlobals : 0);
		break;
		case InstructionList:
		default:
			m_pData->m_pTree = p.parse(m_pData->m_pBuffer,(iRunFlags & AssumeGlobals) ? KviKvsParser::AssumeGlobals : 0);
		break;
	}

	debug("\n\nDUMPING SCRIPT");
	dump("");
	debug("END OF SCRIPT DUMP\n\n");

	if(p.reportList() && pOutput)
		printReportList(pOutput,p.reportList(),false);
		
	return !p.error();
}


bool KviKvsScript::execute(KviWindow * pWnd,KviKvsVariantList * pParams,KviKvsVariant * pRetVal,int iRunFlags,KviKvsExtendedRunTimeData * pExtData)
{
	bool bDeleteParams = !(iRunFlags & PreserveParams);

	// do we have a parsed tree ?
	if(!m_pData->m_pTree)
	{
		if(pParams && bDeleteParams)delete pParams;
		// this is intended for developers only
		pWnd->outputNoFmt(KVI_OUT_PARSERERROR,"[developer error]: you must succesfully call KviKvsScript::parse() before KviKvsScript::execute()");
		return false;
	}
	// do we need to pass dummy params ?
	if(!pParams)
	{
		pParams = KviKvsKernel::instance()->emptyParameterList();
		bDeleteParams = false;
	}

	bool bDeleteRetVal = false;

	if(!pRetVal)
	{
		pRetVal = new KviKvsVariant();
		bDeleteRetVal = true;
	}

	KviKvsRunTimeContext ctx(this,pWnd,pParams,pRetVal,pExtData);

	bool bOk = true;
	
	// lock this script
	m_pData->m_uLock++;
	
	if(!m_pData->m_pTree->execute(&ctx))
	{
		if(ctx.error())bOk = false;
		// else just a halt, return or sth like that
	}
	
	// we can't block any longer: unlock
	m_pData->m_uLock--;
	
	if(ctx.reportList())printReportList(pWnd,ctx.reportList(),true);
	
	// don't forget to delete the params
	if(bDeleteParams)delete pParams;
	if(bDeleteRetVal)delete pRetVal;
    pParams = 0; pRetVal = 0;
	return bOk;
}

void KviKvsScript::printReportList(KviWindow * pOutput,KviPtrList<KviKvsReport> * pList,bool bRunTime)
{
//#warning "FIXME: print errors better (location... etc...)"

	if(pList->count() < 1)return;

	bool bGotError = false;
	KviKvsReport * r;
	for(r = pList->first();r;r = pList->next())
	{
		if(r->type() == KviKvsReport::Error)
		{
			bGotError = true;
			break;
		}
	}

	int out = bGotError ? KVI_OUT_PARSERERROR : KVI_OUT_PARSERWARNING;
	pOutput->output(out,__tr2qs("In script context '%Q':"),&(m_pData->m_szName));

	for(r = pList->first();r;r = pList->next())
	{
		QString szMsg = "[";

		if(r->type() == KviKvsReport::Warning)
		{
			//if(!(iRunFlags & PrintWarnings))continue;
			out = KVI_OUT_PARSERWARNING;
			if(bRunTime)szMsg += __tr2qs("runtime warning");
			else szMsg += __tr2qs("compilation warning");
		} else {
			//if(!(iRunFlags & PrintErrors))continue;
			out = KVI_OUT_PARSERERROR;
			if(bRunTime)szMsg += __tr2qs("runtime error");
			else szMsg += __tr2qs("compilation error");
		}
	
		szMsg += "] ";
		
		if(r->location())
		{
			int nLine = 1;
			int nChar = 1;
			const QChar * c = m_pData->m_pBuffer;
			while(*c && c < r->location())
			{
				if(*c == '\n')
				{
					nLine++;
					nChar = 1;
				} else {
					nChar++;
				}
				c++;
			}
			QString szLocation;
			szLocation.setNum(nLine);
			szMsg += szLocation;
			szMsg += ",";
			szLocation.setNum(nChar);
			szMsg += szLocation;
			szMsg += ": ";
		}
		szMsg += r->description();

		pOutput->outputNoFmt(out,szMsg);
	}
}
