//=============================================================================
//
//   File : kvi_kvs_treenode_expression.cpp
//   Creation date : Mon 06 Oct 2003 01.35 CEST by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2003 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//=============================================================================

#define __KVIRC__

#include "kvi_kvs_treenode_expression.h"
#include "kvi_locale.h"

#include <math.h>

KviKvsTreeNodeExpression::KviKvsTreeNodeExpression(const QChar * pLocation)
: KviKvsTreeNodeData(pLocation)
{
	m_pParentExpression = 0;
}

KviKvsTreeNodeExpression::~KviKvsTreeNodeExpression()
{
}

void KviKvsTreeNodeExpression::dump(const char * prefix)
{
	debug("%s Expression",prefix);
}

int KviKvsTreeNodeExpression::precedence()
{
	return PREC_MAXIMUM;
}

KviKvsTreeNodeExpression * KviKvsTreeNodeExpression::left()
{
	debug("KviKvsTreeNodeExpression::left() : should never end up here!");
	return 0;
}

KviKvsTreeNodeExpression * KviKvsTreeNodeExpression::right()
{
	debug("KviKvsTreeNodeExpression::right() : should never end up here!");
	return 0;
}

void KviKvsTreeNodeExpression::setLeft(KviKvsTreeNodeExpression *)
{
	debug("KviKvsTreeNodeExpression::setLeft() : should never end up here!");
}

void KviKvsTreeNodeExpression::setRight(KviKvsTreeNodeExpression *)
{
	debug("KviKvsTreeNodeExpression::setRight() : should never end up here!");
}

KviKvsTreeNodeExpression * KviKvsTreeNodeExpression::parentWithPrecedenceLowerThan(int iPrec)
{
	if(precedence() > iPrec)return this;
	if(!parentExpression())return 0;
	return parentExpression()->parentWithPrecedenceLowerThan(iPrec);
}

int KviKvsTreeNodeExpression::firstBinaryOperator()
{
	if(!left())return precedence();
	return left()->firstBinaryOperator();
}

///////////////////////////////////////////////////////////////////////////////


KviKvsTreeNodeExpressionVariableOperand::KviKvsTreeNodeExpressionVariableOperand(const QChar * pLocation,KviKvsTreeNodeData * pData)
: KviKvsTreeNodeExpression(pLocation)
{
	m_pData = pData;
	m_pData->setParent(this);
}

KviKvsTreeNodeExpressionVariableOperand::~KviKvsTreeNodeExpressionVariableOperand()
{
	delete m_pData;
}

void KviKvsTreeNodeExpressionVariableOperand::dump(const char * prefix)
{
	debug("%s ExpressionVariableOperand",prefix);
	QString tmp = prefix;
	tmp.append("  ");
	m_pData->dump(tmp.latin1());
}

bool KviKvsTreeNodeExpressionVariableOperand::evaluateReadOnly(KviKvsRunTimeContext * c,KviKvsVariant * pBuffer)
{
	return m_pData->evaluateReadOnly(c,pBuffer);
}


///////////////////////////////////////////////////////////////////////////////

KviKvsTreeNodeExpressionConstantOperand::KviKvsTreeNodeExpressionConstantOperand(const QChar * pLocation,KviKvsVariant * pConstant)
: KviKvsTreeNodeExpression(pLocation)
{
	m_pConstant = pConstant;
}

KviKvsTreeNodeExpressionConstantOperand::~KviKvsTreeNodeExpressionConstantOperand()
{
	delete m_pConstant;
}

void KviKvsTreeNodeExpressionConstantOperand::dump(const char * prefix)
{
	debug("%s ExpressionConstantOperand",prefix);
	QString tmp = prefix;
	tmp.append("  ");
	m_pConstant->dump(tmp.latin1());
}

bool KviKvsTreeNodeExpressionConstantOperand::evaluateReadOnly(KviKvsRunTimeContext * c,KviKvsVariant * pBuffer)
{
	pBuffer->copyFrom(m_pConstant);
	return true;
}

///////////////////////////////////////////////////////////////////////////////


KviKvsTreeNodeExpressionOperator::KviKvsTreeNodeExpressionOperator(const QChar * pLocation)
: KviKvsTreeNodeExpression(pLocation)
{
}

KviKvsTreeNodeExpressionOperator::~KviKvsTreeNodeExpressionOperator()
{
}

void KviKvsTreeNodeExpressionOperator::dump(const char * prefix)
{
	debug("%s ExpressionOperator",prefix);
}



///////////////////////////////////////////////////////////////////////////////


KviKvsTreeNodeExpressionUnaryOperator::KviKvsTreeNodeExpressionUnaryOperator(const QChar * pLocation,KviKvsTreeNodeExpression * pData)
: KviKvsTreeNodeExpressionOperator(pLocation)
{
	m_pData = pData;
	m_pData->setParent(this);
	m_pData->setParentExpression(this);
}

KviKvsTreeNodeExpressionUnaryOperator::~KviKvsTreeNodeExpressionUnaryOperator()
{
	delete m_pData;
}

void KviKvsTreeNodeExpressionUnaryOperator::dump(const char * prefix)
{
	debug("%s ExpressionUnaryOperator",prefix);
	QString tmp = prefix;
	tmp.append("  ");
	m_pData->dump(tmp.latin1());
}

bool KviKvsTreeNodeExpressionUnaryOperator::evaluateOperand(KviKvsRunTimeContext * c)
{
	KviKvsVariant v;
	if(!m_pData->evaluateReadOnly(c,&v))return false;

	if(!v.asNumber(m_nData))
	{
		c->error(location(),__tr2qs("Operand of unary operator didn't evaluate to a number"));
		return false;
	}

	return true;
}

///////////////////////////////////////////////////////////////////////////////


KviKvsTreeNodeExpressionUnaryOperatorNegate::KviKvsTreeNodeExpressionUnaryOperatorNegate(const QChar * pLocation,KviKvsTreeNodeExpression * pData)
: KviKvsTreeNodeExpressionUnaryOperator(pLocation,pData)
{
}

KviKvsTreeNodeExpressionUnaryOperatorNegate::~KviKvsTreeNodeExpressionUnaryOperatorNegate()
{
}

void KviKvsTreeNodeExpressionUnaryOperatorNegate::dump(const char * prefix)
{
	debug("%s ExpressionUnaryOperatorNegate",prefix);
	QString tmp = prefix;
	tmp.append("  ");
	m_pData->dump(tmp.latin1());
}

int KviKvsTreeNodeExpressionUnaryOperatorNegate::precedence()
{
	return PREC_OP_NEGATE;
}

bool KviKvsTreeNodeExpressionUnaryOperatorNegate::evaluateReadOnly(KviKvsRunTimeContext * c,KviKvsVariant * pBuffer)
{
	if(!evaluateOperand(c))return false;
	if(m_nData.isReal())pBuffer->setReal(-m_nData.real());
	else pBuffer->setInteger(-m_nData.integer());
	return true;
}


///////////////////////////////////////////////////////////////////////////////


KviKvsTreeNodeExpressionUnaryOperatorBitwiseNot::KviKvsTreeNodeExpressionUnaryOperatorBitwiseNot(const QChar * pLocation,KviKvsTreeNodeExpression * pData)
: KviKvsTreeNodeExpressionUnaryOperator(pLocation,pData)
{
}

KviKvsTreeNodeExpressionUnaryOperatorBitwiseNot::~KviKvsTreeNodeExpressionUnaryOperatorBitwiseNot()
{
}

void KviKvsTreeNodeExpressionUnaryOperatorBitwiseNot::dump(const char * prefix)
{
	debug("%s ExpressionUnaryOperatorBitwiseNot",prefix);
	QString tmp = prefix;
	tmp.append("  ");
	m_pData->dump(tmp.latin1());
}

int KviKvsTreeNodeExpressionUnaryOperatorBitwiseNot::precedence()
{
	return PREC_OP_BITWISENOT;
}

bool KviKvsTreeNodeExpressionUnaryOperatorBitwiseNot::evaluateReadOnly(KviKvsRunTimeContext * c,KviKvsVariant * pBuffer)
{
	if(!evaluateOperand(c))return false;
	if(m_nData.isReal())pBuffer->setInteger(~(int)(m_nData.real()));
	else pBuffer->setInteger(~m_nData.integer());
	return true;
}


///////////////////////////////////////////////////////////////////////////////


KviKvsTreeNodeExpressionUnaryOperatorLogicalNot::KviKvsTreeNodeExpressionUnaryOperatorLogicalNot(const QChar * pLocation,KviKvsTreeNodeExpression * pData)
: KviKvsTreeNodeExpressionUnaryOperator(pLocation,pData)
{
}

KviKvsTreeNodeExpressionUnaryOperatorLogicalNot::~KviKvsTreeNodeExpressionUnaryOperatorLogicalNot()
{
}

void KviKvsTreeNodeExpressionUnaryOperatorLogicalNot::dump(const char * prefix)
{
	debug("%s ExpressionUnaryOperatorLogicalNot",prefix);
	QString tmp = prefix;
	tmp.append("  ");
	m_pData->dump(tmp.latin1());
}

int KviKvsTreeNodeExpressionUnaryOperatorLogicalNot::precedence()
{
	return PREC_OP_LOGICALNOT;
}

bool KviKvsTreeNodeExpressionUnaryOperatorLogicalNot::evaluateReadOnly(KviKvsRunTimeContext * c,KviKvsVariant * pBuffer)
{
	KviKvsVariant v;
	if(!m_pData->evaluateReadOnly(c,&v))return false;
//#warning "FIXME: We could use setNothing() for false and setInteger(1) for true: this would save memory allocations for false conditions"
	pBuffer->setInteger(v.asBoolean() ? 0 : 1);
	return true;
}



///////////////////////////////////////////////////////////////////////////////


KviKvsTreeNodeExpressionBinaryOperator::KviKvsTreeNodeExpressionBinaryOperator(const QChar * pLocation)
: KviKvsTreeNodeExpressionOperator(pLocation)
{
	m_pLeft = 0;
	m_pRight = 0;
}

KviKvsTreeNodeExpressionBinaryOperator::~KviKvsTreeNodeExpressionBinaryOperator()
{
	if(m_pLeft)delete m_pLeft;
	if(m_pRight)delete m_pRight;
}

bool KviKvsTreeNodeExpressionBinaryOperator::evaluateOperands(KviKvsRunTimeContext * c)
{
	KviKvsVariant v1;
	if(!m_pLeft->evaluateReadOnly(c,&v1))return false;
	if(!v1.asNumber(m_nLeft))
	{
		c->error(location(),__tr2qs("Left operand didn't evaluate to a number"));
		return false;
	}
	KviKvsVariant v2;
	if(!m_pRight->evaluateReadOnly(c,&v2))return false;
	if(!v2.asNumber(m_nRight))
	{
		c->error(location(),__tr2qs("Right operand didn't evaluate to a number"));
		return false;
	}
	return true;
}

KviKvsTreeNodeExpression * KviKvsTreeNodeExpressionBinaryOperator::left()
{
	return m_pLeft;
}

KviKvsTreeNodeExpression * KviKvsTreeNodeExpressionBinaryOperator::right()
{
	return m_pRight;
}

int KviKvsTreeNodeExpressionBinaryOperator::firstBinaryOperator()
{
	return precedence();
}

void KviKvsTreeNodeExpressionBinaryOperator::setLeft(KviKvsTreeNodeExpression * pLeft)
{
	m_pLeft = pLeft;
	m_pLeft->setParent(this);
	m_pLeft->setParentExpression(this);
}

void KviKvsTreeNodeExpressionBinaryOperator::setRight(KviKvsTreeNodeExpression * pRight)
{
	m_pRight = pRight;
	m_pRight->setParent(this);
	m_pRight->setParentExpression(this);
}


void KviKvsTreeNodeExpressionBinaryOperator::dumpOperands(const char * prefix)
{
	QString tmp = prefix;
	tmp.append("  ");
	if(m_pLeft)m_pLeft->dump(tmp.latin1());
	if(m_pRight)m_pRight->dump(tmp.latin1());
}

void KviKvsTreeNodeExpressionBinaryOperator::dump(const char * prefix)
{
	debug("%s ExpressionBinaryOperator",prefix);
	dumpOperands(prefix);
}

////////////////////////////////////////////////////////////////////////////////


#define PREIMPLEMENT_BINARY_OPERATOR(__name,__stringname,__precedence) \
	__name::__name(const QChar * pLocation) \
	: KviKvsTreeNodeExpressionBinaryOperator(pLocation){} \
	__name::~__name(){} \
	void __name::dump(const char * prefix){ debug("%s " __stringname,prefix); dumpOperands(prefix); } \
	int __name::precedence(){ return __precedence; };

PREIMPLEMENT_BINARY_OPERATOR(KviKvsTreeNodeExpressionBinaryOperatorSum,"ExpressionBinaryOperatorSum",PREC_OP_SUM)

bool KviKvsTreeNodeExpressionBinaryOperatorSum::evaluateReadOnly(KviKvsRunTimeContext * c,KviKvsVariant * pBuffer)
{
	if(!evaluateOperands(c))return false;
	if(m_nLeft.isInteger())
	{
		if(m_nRight.isInteger())pBuffer->setInteger(m_nLeft.integer() + m_nRight.integer());
		else pBuffer->setReal(m_nLeft.integer() + m_nRight.real());
	} else {
		if(m_nRight.isInteger())pBuffer->setReal(m_nLeft.real() + m_nRight.integer());
		else pBuffer->setReal(m_nLeft.real() + m_nRight.real());
	}
	return true;
}

PREIMPLEMENT_BINARY_OPERATOR(KviKvsTreeNodeExpressionBinaryOperatorSubtraction,"ExpressionBinaryOperatorSubtraction",PREC_OP_SUBTRACTION)

bool KviKvsTreeNodeExpressionBinaryOperatorSubtraction::evaluateReadOnly(KviKvsRunTimeContext * c,KviKvsVariant * pBuffer)
{
	if(!evaluateOperands(c))return false;
	if(m_nLeft.isInteger())
	{
		if(m_nRight.isInteger())pBuffer->setInteger(m_nLeft.integer() - m_nRight.integer());
		else pBuffer->setReal(((double)(m_nLeft.integer())) - m_nRight.real());
	} else {
		if(m_nRight.isInteger())pBuffer->setReal(m_nLeft.real() - ((double)(m_nRight.integer())));
		else pBuffer->setReal(m_nLeft.real() - m_nRight.real());
	}
	return true;
}

PREIMPLEMENT_BINARY_OPERATOR(KviKvsTreeNodeExpressionBinaryOperatorMultiplication,"ExpressionBinaryOperatorMultiplication",PREC_OP_MULTIPLICATION)

bool KviKvsTreeNodeExpressionBinaryOperatorMultiplication::evaluateReadOnly(KviKvsRunTimeContext * c,KviKvsVariant * pBuffer)
{
	if(!evaluateOperands(c))return false;	
	if(m_nLeft.isInteger())
	{
		if(m_nRight.isInteger())pBuffer->setInteger(m_nLeft.integer() * m_nRight.integer());
		else pBuffer->setReal(m_nLeft.integer() * m_nRight.real());
	} else {
		if(m_nRight.isInteger())pBuffer->setReal(m_nLeft.real() * m_nRight.integer());
		else pBuffer->setReal(m_nLeft.real() * m_nRight.real());
	}
	return true;
}


PREIMPLEMENT_BINARY_OPERATOR(KviKvsTreeNodeExpressionBinaryOperatorDivision,"ExpressionBinaryOperatorDivision",PREC_OP_DIVISION)

bool KviKvsTreeNodeExpressionBinaryOperatorDivision::evaluateReadOnly(KviKvsRunTimeContext * c,KviKvsVariant * pBuffer)
{
	if(!evaluateOperands(c))return false;	

	if(m_nRight.isInteger())
	{
		if(m_nRight.integer() == 0)
		{
			c->error(location(),__tr2qs("Division by zero"));
			return false;
		}
		if(m_nLeft.isInteger())pBuffer->setInteger(m_nLeft.integer() / m_nRight.integer());
		else pBuffer->setReal(m_nLeft.real() / ((double)(m_nRight.integer())));
	} else {
		if(m_nRight.real() == 0.0)
		{
			c->error(location(),__tr2qs("Division by zero"));
			return false;
		}
		if(m_nLeft.isInteger())pBuffer->setReal(((double)(m_nLeft.integer())) / m_nRight.real());
		else pBuffer->setReal(m_nLeft.real() / m_nRight.real());
	}
	return true;
}


PREIMPLEMENT_BINARY_OPERATOR(KviKvsTreeNodeExpressionBinaryOperatorModulus,"ExpressionBinaryOperatorModulus",PREC_OP_MODULUS)

bool KviKvsTreeNodeExpressionBinaryOperatorModulus::evaluateReadOnly(KviKvsRunTimeContext * c,KviKvsVariant * pBuffer)
{
	if(!evaluateOperands(c))return false;	

	if(m_nRight.isInteger())
	{
		if(m_nRight.integer() == 0)
		{
			c->error(location(),__tr2qs("Division by zero"));
			return false;
		}
		if(m_nLeft.isInteger())pBuffer->setInteger(m_nLeft.integer() % m_nRight.integer());
		else pBuffer->setReal(fmod(m_nLeft.real(),((double)(m_nRight.integer()))));
	} else {
		if(m_nRight.real() == 0.0)
		{
			c->error(location(),__tr2qs("Division by zero"));
			return false;
		}
		if(m_nLeft.isInteger())pBuffer->setReal(fmod(((double)(m_nLeft.integer())),m_nRight.real()));
		else pBuffer->setReal(fmod(m_nLeft.real(),m_nRight.real()));
	}
	return true;
}

PREIMPLEMENT_BINARY_OPERATOR(KviKvsTreeNodeExpressionBinaryOperatorBitwiseAnd,"ExpressionBinaryOperatorBitwiseAnd",PREC_OP_BITWISEAND)

bool KviKvsTreeNodeExpressionBinaryOperatorBitwiseAnd::evaluateReadOnly(KviKvsRunTimeContext * c,KviKvsVariant * pBuffer)
{
	if(!evaluateOperands(c))return false;
	int iLeft = m_nLeft.isInteger() ? m_nLeft.integer() : (int)m_nLeft.real();
	int iRight = m_nRight.isInteger() ? m_nRight.integer() : (int)m_nRight.real();
	pBuffer->setInteger(iLeft & iRight);
	return true;
}

PREIMPLEMENT_BINARY_OPERATOR(KviKvsTreeNodeExpressionBinaryOperatorBitwiseOr,"ExpressionBinaryOperatorBitwiseOr",PREC_OP_BITWISEOR)

bool KviKvsTreeNodeExpressionBinaryOperatorBitwiseOr::evaluateReadOnly(KviKvsRunTimeContext * c,KviKvsVariant * pBuffer)
{
	if(!evaluateOperands(c))return false;
	int iLeft = m_nLeft.isInteger() ? m_nLeft.integer() : (int)m_nLeft.real();
	int iRight = m_nRight.isInteger() ? m_nRight.integer() : (int)m_nRight.real();
	pBuffer->setInteger(iLeft | iRight);
	return true;
}

PREIMPLEMENT_BINARY_OPERATOR(KviKvsTreeNodeExpressionBinaryOperatorBitwiseXor,"ExpressionBinaryOperatorBitwiseXor",PREC_OP_BITWISEXOR)

bool KviKvsTreeNodeExpressionBinaryOperatorBitwiseXor::evaluateReadOnly(KviKvsRunTimeContext * c,KviKvsVariant * pBuffer)
{
	if(!evaluateOperands(c))return false;
	int iLeft = m_nLeft.isInteger() ? m_nLeft.integer() : (int)m_nLeft.real();
	int iRight = m_nRight.isInteger() ? m_nRight.integer() : (int)m_nRight.real();
	pBuffer->setInteger(iLeft ^ iRight);
	return true;
}

PREIMPLEMENT_BINARY_OPERATOR(KviKvsTreeNodeExpressionBinaryOperatorShiftLeft,"ExpressionBinaryOperatorShiftLeft",PREC_OP_SHIFTLEFT)

bool KviKvsTreeNodeExpressionBinaryOperatorShiftLeft::evaluateReadOnly(KviKvsRunTimeContext * c,KviKvsVariant * pBuffer)
{
	if(!evaluateOperands(c))return false;
	int iLeft = m_nLeft.isInteger() ? m_nLeft.integer() : (int)(m_nLeft.real());
	int iRight = m_nRight.isInteger() ? m_nRight.integer() : (int)(m_nRight.real());
	pBuffer->setInteger(iLeft << iRight);
	return true;
}

PREIMPLEMENT_BINARY_OPERATOR(KviKvsTreeNodeExpressionBinaryOperatorShiftRight,"ExpressionBinaryOperatorShiftRight",PREC_OP_SHIFTRIGHT)

bool KviKvsTreeNodeExpressionBinaryOperatorShiftRight::evaluateReadOnly(KviKvsRunTimeContext * c,KviKvsVariant * pBuffer)
{
	if(!evaluateOperands(c))return false;
	int iLeft = m_nLeft.isInteger() ? m_nLeft.integer() : (int)(m_nLeft.real());
	int iRight = m_nRight.isInteger() ? m_nRight.integer() : (int)(m_nRight.real());
	pBuffer->setInteger(iLeft >> iRight);
	return true;
}

PREIMPLEMENT_BINARY_OPERATOR(KviKvsTreeNodeExpressionBinaryOperatorAnd,"ExpressionBinaryOperatorAnd",PREC_OP_AND)

bool KviKvsTreeNodeExpressionBinaryOperatorAnd::evaluateReadOnly(KviKvsRunTimeContext * c,KviKvsVariant * pBuffer)
{
	KviKvsVariant v1;
	KviKvsVariant v2;
	if(!m_pLeft->evaluateReadOnly(c,&v1))return false;
	//#warning "FIXME: We could use setNothing() as false: this would save memory allocations (and thus time)"
	if(!v1.asBoolean())
	{
		pBuffer->setInteger(0);
		return true;
	}
	if(!m_pRight->evaluateReadOnly(c,&v2))return false;
	pBuffer->setInteger(v2.asBoolean() ? 1 : 0);
	return true;
}


PREIMPLEMENT_BINARY_OPERATOR(KviKvsTreeNodeExpressionBinaryOperatorOr,"ExpressionBinaryOperatorOr",PREC_OP_OR)

bool KviKvsTreeNodeExpressionBinaryOperatorOr::evaluateReadOnly(KviKvsRunTimeContext * c,KviKvsVariant * pBuffer)
{
	KviKvsVariant v1;
	KviKvsVariant v2;
	if(!m_pLeft->evaluateReadOnly(c,&v1))return false;
	//#warning "FIXME: We could use setNothing() as false: this would save memory allocations (and thus time)"
	if(v1.asBoolean())
	{
		pBuffer->setInteger(1);
		return true;
	}
	if(!m_pRight->evaluateReadOnly(c,&v2))return false;
	pBuffer->setInteger(v2.asBoolean() ? 1 : 0);
	return true;
}


PREIMPLEMENT_BINARY_OPERATOR(KviKvsTreeNodeExpressionBinaryOperatorXor,"ExpressionBinaryOperatorXor",PREC_OP_XOR)

bool KviKvsTreeNodeExpressionBinaryOperatorXor::evaluateReadOnly(KviKvsRunTimeContext * c,KviKvsVariant * pBuffer)
{
	KviKvsVariant v1;
	KviKvsVariant v2;
	if(!m_pLeft->evaluateReadOnly(c,&v1))return false;
	if(!m_pRight->evaluateReadOnly(c,&v2))return false;
	//#warning "FIXME: We could use setNothing() as false: this would save memory allocations (and thus time)"
	if(v1.asBoolean())
		pBuffer->setInteger(v2.asBoolean() ? 0 : 1);
	else {
		if(v2.asBoolean())
			pBuffer->setInteger(v1.asBoolean() ? 0 : 1);
		else
			pBuffer->setInteger(0);
	}
	return true;
}

PREIMPLEMENT_BINARY_OPERATOR(KviKvsTreeNodeExpressionBinaryOperatorLowerThan,"ExpressionBinaryOperatorLowerThan",PREC_OP_LOWERTHAN)

bool KviKvsTreeNodeExpressionBinaryOperatorLowerThan::evaluateReadOnly(KviKvsRunTimeContext * c,KviKvsVariant * pBuffer)
{
	KviKvsVariant v1;
	KviKvsVariant v2;
	if(!m_pLeft->evaluateReadOnly(c,&v1))return false;
	if(!m_pRight->evaluateReadOnly(c,&v2))return false;

	bool bBothNumbers = true;
	
	if(!v1.asNumber(m_nLeft))bBothNumbers = false;
	else if(!v2.asNumber(m_nRight))bBothNumbers = false;

	if(bBothNumbers)
	{
		if(m_nLeft.isInteger())
		{
			if(m_nRight.isInteger())pBuffer->setInteger(m_nLeft.integer() < m_nRight.integer());
			else pBuffer->setInteger(((double)(m_nLeft.integer())) < m_nRight.real());
		} else {
			if(m_nRight.isInteger())pBuffer->setInteger(m_nLeft.real() < ((double)(m_nRight.integer())));
			else pBuffer->setInteger(m_nLeft.real() < m_nRight.real());
		}
	} else {
		// at least one is not a number
		QString sz1;
		QString sz2;
		v1.asString(sz1);
		v2.asString(sz2);
		pBuffer->setInteger(sz1 < sz2);
	}
	return true;
}

PREIMPLEMENT_BINARY_OPERATOR(KviKvsTreeNodeExpressionBinaryOperatorGreaterThan,"ExpressionBinaryOperatorGreaterThan",PREC_OP_GREATERTHAN)

bool KviKvsTreeNodeExpressionBinaryOperatorGreaterThan::evaluateReadOnly(KviKvsRunTimeContext * c,KviKvsVariant * pBuffer)
{
	KviKvsVariant v1;
	KviKvsVariant v2;
	if(!m_pLeft->evaluateReadOnly(c,&v1))return false;
	if(!m_pRight->evaluateReadOnly(c,&v2))return false;

	bool bBothNumbers = true;
	
	if(!v1.asNumber(m_nLeft))bBothNumbers = false;
	else if(!v2.asNumber(m_nRight))bBothNumbers = false;

	if(bBothNumbers)
	{
		if(m_nLeft.isInteger())
		{
			if(m_nRight.isInteger())pBuffer->setInteger(m_nLeft.integer() > m_nRight.integer());
			else pBuffer->setInteger(((double)(m_nLeft.integer())) > m_nRight.real());
		} else {
			if(m_nRight.isInteger())pBuffer->setInteger(m_nLeft.real() > ((double)(m_nRight.integer())));
			else pBuffer->setInteger(m_nLeft.real() > m_nRight.real());
		}
	} else {
		// at least one is not a number
		QString sz1;
		QString sz2;
		v1.asString(sz1);
		v2.asString(sz2);
		pBuffer->setInteger(sz1 > sz2);
	}
	return true;
}

PREIMPLEMENT_BINARY_OPERATOR(KviKvsTreeNodeExpressionBinaryOperatorLowerOrEqualTo,"ExpressionBinaryOperatorLowerOrEqualTo",PREC_OP_LOWEROREQUALTO)

bool KviKvsTreeNodeExpressionBinaryOperatorLowerOrEqualTo::evaluateReadOnly(KviKvsRunTimeContext * c,KviKvsVariant * pBuffer)
{
	KviKvsVariant v1;
	KviKvsVariant v2;
	if(!m_pLeft->evaluateReadOnly(c,&v1))return false;
	if(!m_pRight->evaluateReadOnly(c,&v2))return false;

	bool bBothNumbers = true;
	
	if(!v1.asNumber(m_nLeft))bBothNumbers = false;
	else if(!v2.asNumber(m_nRight))bBothNumbers = false;

	if(bBothNumbers)
	{
		if(m_nLeft.isInteger())
		{
			if(m_nRight.isInteger())pBuffer->setInteger(m_nLeft.integer() <= m_nRight.integer());
			else pBuffer->setInteger(((double)(m_nLeft.integer())) <= m_nRight.real());
		} else {
			if(m_nRight.isInteger())pBuffer->setInteger(m_nLeft.real() <= ((double)(m_nRight.integer())));
			else pBuffer->setInteger(m_nLeft.real() <= m_nRight.real());
		}
	} else {
		// at least one is not a number
		QString sz1;
		QString sz2;
		v1.asString(sz1);
		v2.asString(sz2);
		pBuffer->setInteger(sz1 <= sz2);
	}
	return true;
}

PREIMPLEMENT_BINARY_OPERATOR(KviKvsTreeNodeExpressionBinaryOperatorGreaterOrEqualTo,"ExpressionBinaryOperatorGreaterOrEqualTo",PREC_OP_GREATEROREQUALTO)

bool KviKvsTreeNodeExpressionBinaryOperatorGreaterOrEqualTo::evaluateReadOnly(KviKvsRunTimeContext * c,KviKvsVariant * pBuffer)
{
	KviKvsVariant v1;
	KviKvsVariant v2;
	if(!m_pLeft->evaluateReadOnly(c,&v1))return false;
	if(!m_pRight->evaluateReadOnly(c,&v2))return false;

	bool bBothNumbers = true;
	
	if(!v1.asNumber(m_nLeft))bBothNumbers = false;
	else if(!v2.asNumber(m_nRight))bBothNumbers = false;

	if(bBothNumbers)
	{
		if(m_nLeft.isInteger())
		{
			if(m_nRight.isInteger())pBuffer->setInteger(m_nLeft.integer() >= m_nRight.integer());
			else pBuffer->setInteger(((double)(m_nLeft.integer())) >= m_nRight.real());
		} else {
			if(m_nRight.isInteger())pBuffer->setInteger(m_nLeft.real() >= ((double)(m_nRight.integer())));
			else pBuffer->setInteger(m_nLeft.real() >= m_nRight.real());
		}
	} else {
		// at least one is not a number
		QString sz1;
		QString sz2;
		v1.asString(sz1);
		v2.asString(sz2);
		pBuffer->setInteger(sz1 >= sz2);
	}
	return true;
}

PREIMPLEMENT_BINARY_OPERATOR(KviKvsTreeNodeExpressionBinaryOperatorEqualTo,"ExpressionBinaryOperatorEqualTo",PREC_OP_EQUALTO)

bool KviKvsTreeNodeExpressionBinaryOperatorEqualTo::evaluateReadOnly(KviKvsRunTimeContext * c,KviKvsVariant * pBuffer)
{
	KviKvsVariant v1;
	KviKvsVariant v2;
	if(!m_pLeft->evaluateReadOnly(c,&v1))return false;
	if(!m_pRight->evaluateReadOnly(c,&v2))return false;

	bool bBothNumbers = true;
	
	if(!v1.asNumber(m_nLeft))bBothNumbers = false;
	else if(!v2.asNumber(m_nRight))bBothNumbers = false;

	if(bBothNumbers)
	{
		if(m_nLeft.isInteger())
		{
			if(m_nRight.isInteger())pBuffer->setInteger(m_nLeft.integer() == m_nRight.integer());
			else pBuffer->setInteger(((double)(m_nLeft.integer())) == m_nRight.real());
		} else {
			if(m_nRight.isInteger())pBuffer->setInteger(m_nLeft.real() == ((double)(m_nRight.integer())));
			else pBuffer->setInteger(m_nLeft.real() == m_nRight.real());
		}
	} else {
		// at least one is not a number
		QString sz1;
		QString sz2;
		v1.asString(sz1);
		v2.asString(sz2);
		pBuffer->setInteger(sz1 == sz2);
	}
	return true;
}


PREIMPLEMENT_BINARY_OPERATOR(KviKvsTreeNodeExpressionBinaryOperatorNotEqualTo,"ExpressionBinaryOperatorNotEqualTo",PREC_OP_NOTEQUALTO)

bool KviKvsTreeNodeExpressionBinaryOperatorNotEqualTo::evaluateReadOnly(KviKvsRunTimeContext * c,KviKvsVariant * pBuffer)
{
	KviKvsVariant v1;
	KviKvsVariant v2;
	if(!m_pLeft->evaluateReadOnly(c,&v1))return false;
	if(!m_pRight->evaluateReadOnly(c,&v2))return false;

	bool bBothNumbers = true;
	
	if(!v1.asNumber(m_nLeft))bBothNumbers = false;
	else if(!v2.asNumber(m_nRight))bBothNumbers = false;

	if(bBothNumbers)
	{
		if(m_nLeft.isInteger())
		{
			if(m_nRight.isInteger())pBuffer->setInteger(m_nLeft.integer() != m_nRight.integer());
			else pBuffer->setInteger(((double)(m_nLeft.integer())) != m_nRight.real());
		} else {
			if(m_nRight.isInteger())pBuffer->setInteger(m_nLeft.real() != ((double)(m_nRight.integer())));
			else pBuffer->setInteger(m_nLeft.real() != m_nRight.real());
		}
	} else {
		// at least one is not a number
		QString sz1;
		QString sz2;
		v1.asString(sz1);
		v2.asString(sz2);
		pBuffer->setInteger(sz1 != sz2);
	}
	return true;
}
