//=============================================================================
//
//   File : kvi_kvs_variant.cpp
//   Created on Tue 07 Oct 2003 04:01:19 by Szymon Stefanek
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 2003 Szymon Stefanek <pragma at kvirc dot net>
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//=============================================================================

#define __KVIRC__

#include "kvi_kvs_variant.h"



#ifdef SHARED_VARIANT


KviKvsVariant::KviKvsVariant()
{
	m_pData = 0;
}

KviKvsVariant::KviKvsVariant(QString * pString)
{
	m_pData = new KviKvsVariantData;
	m_pData->m_eType = KviKvsVariantData::String;
	m_pData->m_uRefs = 1;
	m_pData->m_u.pString = pString;
}

KviKvsVariant::KviKvsVariant(const QString &szString)
{
	m_pData = new KviKvsVariantData;
	m_pData->m_eType = KviKvsVariantData::String;
	m_pData->m_uRefs = 1;
	m_pData->m_u.pString = new QString(szString);
}

KviKvsVariant::KviKvsVariant(KviKvsArray * pArray)
{
	m_pData = new KviKvsVariantData;
	m_pData->m_eType = KviKvsVariantData::Array;
	m_pData->m_uRefs = 1;
	m_pData->m_u.pArray = pArray;
}

KviKvsVariant::KviKvsVariant(KviKvsHash * pHash)
{
	m_pData = new KviKvsVariantData;
	m_pData->m_eType = KviKvsVariantData::Hash;
	m_pData->m_uRefs = 1;
	m_pData->m_u.pHash = pHash;
}

KviKvsVariant::KviKvsVariant(double * pReal)
{
	m_pData = new KviKvsVariantData;
	m_pData->m_eType = KviKvsVariantData::Real;
	m_pData->m_uRefs = 1;
	m_pData->m_u.pReal = pReal;
}

KviKvsVariant::KviKvsVariant(double dReal)
{
	m_pData = new KviKvsVariantData;
	m_pData->m_eType = KviKvsVariantData::Real;
	m_pData->m_uRefs = 1;
	m_pData->m_u.pReal = new double;
	*(m_pData->m_u.pReal) = dReal;
}

KviKvsVariant::KviKvsVariant(int iInteger)
{
	m_pData = new KviKvsVariantData;
	m_pData->m_eType = KviKvsVariantData::Integer;
	m_pData->m_uRefs = 1;
	m_pData->m_u.iInteger = iInteger;
}

KviKvsVariant::KviKvsVariant(const KviKvsVariant &v)
{
	m_pData = v.m_pData;
	if(m_pData)m_pData->m_uRefs++;
}

#define DELETE_VARIANT_CONTENTS \
	switch(m_pData->m_eType) \
	{ \
		case KviKvsVariantData::Array: delete m_pData->m_u.pArray; break; \
		case KviKvsVariantData::Hash: delete m_pData->m_u.pHash; break; \
		case KviKvsVariantData::String: delete m_pData->m_u.pString; break; \
		case KviKvsVariantData::Real: delete m_pData->m_u.pReal; break; \
		default: /* make gcc happy */ break; \
	}

#define DETACH_CONTENTS \
	if(m_pData) \
	{ \
		if(m_pData->m_uRefs <= 1) \
		{ \
			DELETE_VARIANT_CONTENTS \
			delete m_pData; \
		} else { \
			m_pData->m_uRefs--; \
		} \
	}

#define RENEW_VARIANT_DATA \
	if(m_pData) \
	{ \
		if(m_pData->m_uRefs > 1) \
		{ \
			m_pData->m_uRefs--; \
			m_pData = new KviKvsVariantData; \
			m_pData->m_uRefs = 1; \
		} else { \
			DELETE_VARIANT_CONTENTS \
		} \
	} else { \
		m_pData = new KviKvsVariantData; \
		m_pData->m_uRefs = 1; \
	}

KviKvsVariant::~KviKvsVariant()
{
	DETACH_CONTENTS
}

void KviKvsVariant::setString(QString * pString)
{
	RENEW_VARIANT_DATA
	m_pData->m_eType = KviKvsVariantData::String;
	m_pData->m_u.pString = pString;
}

void KviKvsVariant::setString(const QString &szString)
{
	RENEW_VARIANT_DATA
	m_pData->m_eType = KviKvsVariantData::String;
	m_pData->m_u.pString = new QString(szString);
}

void KviKvsVariant::setReal(double dReal)
{
	RENEW_VARIANT_DATA
	m_pData->m_eType = KviKvsVariantData::Real;
	m_pData->m_u.pReal = new double;
	*(m_pData->m_u.pReal) = dReal;
}

void KviKvsVariant::setReal(double * pReal)
{
	RENEW_VARIANT_DATA
	m_pData->m_eType = KviKvsVariantData::Real;
	m_pData->m_u.pReal = pReal;
}

void KviKvsVariant::setInteger(int iInteger)
{
	RENEW_VARIANT_DATA
	m_pData->m_eType = KviKvsVariantData::Integer;
	m_pData->m_u.iInteger = iInteger;
}

void KviKvsVariant::setArray(KviKvsArray * pArray)
{
	RENEW_VARIANT_DATA
	m_pData->m_eType = KviKvsVariantData::Array;
	m_pData->m_u.pArray = pArray;
}

void KviKvsVariant::setHash(KviKvsHash * pHash)
{
	RENEW_VARIANT_DATA
	m_pData->m_eType = KviKvsVariantData::Hash;
	m_pData->m_u.pHash = pHash;
}

void KviKvsVariant::setNothing()
{
	if(m_pData)
	{
		if(m_pData->m_uRefs <= 1)
		{
			DELETE_VARIANT_CONTENTS
			delete m_pData;
            m_pData = 0;
		} else {
			m_pData->m_uRefs--;
		}
		m_pData = 0;
	}
}

bool KviKvsVariant::isEmpty() const
{
	if(!m_pData)return true;
	switch(m_pData->m_eType)
	{
		case KviKvsVariantData::String: return m_pData->m_u.pString->isEmpty(); break;
		case KviKvsVariantData::Array: return m_pData->m_u.pArray->isEmpty(); break;
		case KviKvsVariantData::Hash: return m_pData->m_u.pHash->isEmpty(); break;
		default: /* make gcc happy */ break;
	}
	return false;
}

bool KviKvsVariant::asBoolean() const
{
	if(!m_pData)return false;
	switch(m_pData->m_eType)
	{
		case KviKvsVariantData::String:
		{
			if(m_pData->m_u.pString->isEmpty())return false;
			// check integer or real values
			bool bOk;
			int iVal = m_pData->m_u.pString->toInt(&bOk);
			if(bOk)return iVal;
			double dVal = m_pData->m_u.pString->toDouble(&bOk);
			if(bOk)return (dVal != 0.0);
			// non number, non empty
			return true;
		}
		break;
		case KviKvsVariantData::Integer: return m_pData->m_u.iInteger; break;
		case KviKvsVariantData::Real: return *(m_pData->m_u.pReal) != 0.0; break;
		case KviKvsVariantData::Array: return !(m_pData->m_u.pArray->isEmpty()); break;
		case KviKvsVariantData::Hash: return !(m_pData->m_u.pHash->isEmpty()); break;
		default: /* make gcc happy */ break;
	}
	debug("WARNING: invalid variant type %d in KviKvsVariant::asBoolean()",m_pData->m_eType);
	return false;
}

bool KviKvsVariant::asNumber(KviKvsNumber &n) const
{
	if(!m_pData)return false;

	if(isInteger())
	{
		n.m_u.iInteger = m_pData->m_u.iInteger;
		n.m_type = KviKvsNumber::Integer;
		return true;
	}
	if(isReal())
	{
		n.m_u.dReal = *(m_pData->m_u.pReal);
		n.m_type = KviKvsNumber::Real;
		return true;
	}
	if(asInteger(n.m_u.iInteger))
	{
		n.m_type = KviKvsNumber::Integer;
		return true;
	}
	if(asReal(n.m_u.dReal))
	{
		n.m_type = KviKvsNumber::Real;
		return true;
	}
	return false;
}


bool KviKvsVariant::asNumberExt(KviKvsNumber &n) const
{
	if(!m_pData)
	{
		n.m_u.iInteger = 0;
		n.m_type = KviKvsNumber::Integer;
		return true;
	}

	if(isInteger())
	{
		n.m_u.iInteger = m_pData->m_u.iInteger;
		n.m_type = KviKvsNumber::Integer;
		return true;
	}
	if(isReal())
	{
		n.m_u.dReal = *(m_pData->m_u.pReal);
		n.m_type = KviKvsNumber::Real;
		return true;
	}
	if(asInteger(n.m_u.iInteger))
	{
		n.m_type = KviKvsNumber::Integer;
		return true;
	}
	if(asReal(n.m_u.dReal))
	{
		n.m_type = KviKvsNumber::Real;
		return true;
	}
	if(asIntegerExt(n.m_u.iInteger))
	{
		n.m_type = KviKvsNumber::Integer;
		return true;
	}
	return false;
}


bool KviKvsVariant::asInteger(int &iVal) const
{
	if(!m_pData)return false;
	switch(m_pData->m_eType)
	{
		case KviKvsVariantData::Integer:
			iVal = m_pData->m_u.iInteger;
			return true;
		break;
		case KviKvsVariantData::String:
		{
			bool bOk;
			iVal = m_pData->m_u.pString->toInt(&bOk);
			return bOk;
		}
		break;
		case KviKvsVariantData::Real:
			// FIXME: this truncates the value!
			iVal = (int)*(m_pData->m_u.pReal);
			return true;
		break;
		default: /* make gcc happy */
		break;
	}
	return false;
}

bool KviKvsVariant::asIntegerExt(int &iVal) const
{
	if(!m_pData)return false; // return 0 ?
	switch(m_pData->m_eType)
	{
		case KviKvsVariantData::Integer:
			iVal = m_pData->m_u.iInteger;
			return true;
		break;
		case KviKvsVariantData::String:
		{
			bool bOk;
			iVal = m_pData->m_u.pString->toInt(&bOk);
			if(bOk)return true;
			iVal = m_pData->m_u.pString->length();
			return true;
		}
		break;
		case KviKvsVariantData::Real:
			// FIXME: this truncates the value!
			iVal = (int)*(m_pData->m_u.pReal);
			return true;
		break;
		case KviKvsVariantData::Array:
			iVal = m_pData->m_u.pArray->size();
			return true;
		break;
		case KviKvsVariantData::Hash:
			iVal = m_pData->m_u.pHash->size();
			return true;
		break;
		default: /* make gcc happy */
		break;
	}
	return false;
}


bool KviKvsVariant::asReal(double & dVal) const
{
	if(!m_pData)return false;
	switch(m_pData->m_eType)
	{
		case KviKvsVariantData::Integer:
			dVal = m_pData->m_u.iInteger;
			return true;
		break;
		case KviKvsVariantData::String:
		{
			bool bOk;
			dVal = m_pData->m_u.pString->toDouble(&bOk);
			return bOk;
		}
		break;
		case KviKvsVariantData::Real:
			dVal = *(m_pData->m_u.pReal);
			return true;
		break;
		default: /* by default we make gcc happy */ break;
	}
	return false;
}

void KviKvsVariant::asString(QString &szBuffer) const
{
	if(!m_pData)
	{
		szBuffer = QString::null;
		return;
	}
	switch(m_pData->m_eType)
	{
		case KviKvsVariantData::String: szBuffer = *(m_pData->m_u.pString); break;
		case KviKvsVariantData::Array: szBuffer = QString::null; m_pData->m_u.pArray->appendAsString(szBuffer); break;
		case KviKvsVariantData::Hash: szBuffer = QString::null; m_pData->m_u.pHash->appendAsString(szBuffer); break;
		case KviKvsVariantData::Integer:  szBuffer.setNum(m_pData->m_u.iInteger); break;
		case KviKvsVariantData::Real: szBuffer.setNum(*(m_pData->m_u.pReal)); break;
		default: /* make gcc happy */ break;
	}
}

void KviKvsVariant::appendAsString(QString &szBuffer) const
{
	if(!m_pData)return;
	switch(m_pData->m_eType)
	{
		case KviKvsVariantData::String: szBuffer.append(*(m_pData->m_u.pString)); break;
		case KviKvsVariantData::Array: m_pData->m_u.pArray->appendAsString(szBuffer); break;
		case KviKvsVariantData::Hash: m_pData->m_u.pHash->appendAsString(szBuffer); break;
		case KviKvsVariantData::Integer:  KviQString::appendNumber(szBuffer,m_pData->m_u.iInteger); break;
		case KviKvsVariantData::Real: KviQString::appendNumber(szBuffer,*(m_pData->m_u.pReal)); break;
		default: /* make gcc happy */ break;
	}
}

void KviKvsVariant::dump(const char * prefix) const
{
	if(!m_pData)
	{
		debug("%s Nothing",prefix);
		return;
	}
	switch(m_pData->m_eType)
	{
		case KviKvsVariantData::String: debug("%s String(%s)",prefix,m_pData->m_u.pString->latin1()); break;
		case KviKvsVariantData::Array: debug("%s Array",prefix); break;
		case KviKvsVariantData::Hash: debug("%s Hash",prefix); break;
		case KviKvsVariantData::Integer:  debug("%s Integer(%d)",prefix,m_pData->m_u.iInteger); break;
		case KviKvsVariantData::Real: debug("%s Real(%f)",prefix,*(m_pData->m_u.pReal)); break;
		default: /* make gcc happy */ break;
	}
}

void KviKvsVariant::copyFrom(const KviKvsVariant * v)
{
	DETACH_CONTENTS
	m_pData = v->m_pData;
	if(m_pData)m_pData->m_uRefs++;
}

void KviKvsVariant::copyFrom(const KviKvsVariant &v)
{
	DETACH_CONTENTS
	m_pData = v.m_pData;
	if(m_pData)m_pData->m_uRefs++;

}

void KviKvsVariant::takeFrom(KviKvsVariant * v)
{
	DETACH_CONTENTS
	m_pData = v->m_pData;
	v->m_pData = 0;
}

void KviKvsVariant::takeFrom(KviKvsVariant &v)
{
	DETACH_CONTENTS
	m_pData = v.m_pData;
	v.m_pData = 0;
}

void KviKvsVariant::getTypeName(QString &szBuffer) const
{
	if(!m_pData)
	{
		szBuffer = "nothing";
		return;
	}
	switch(m_pData->m_eType)
	{
		case KviKvsVariantData::String: szBuffer = "string"; break;
		case KviKvsVariantData::Hash: szBuffer = "hash"; break;
		case KviKvsVariantData::Array: szBuffer = "array"; break;
		case KviKvsVariantData::Real: szBuffer = "real"; break;
		case KviKvsVariantData::Integer: szBuffer = "integer"; break;
		default: szBuffer = "internal_error"; break;
	}
}


#else //!SHARED_VARIANT

KviKvsVariant::KviKvsVariant()
{
	m_type = Nothing;
}

KviKvsVariant::KviKvsVariant(QString * pString)
{
	m_type = String;
	m_u.pString = pString;
}

KviKvsVariant::KviKvsVariant(const QString &szString)
{
	m_type = String;
	m_u.pString = new QString(szString);
}

KviKvsVariant::KviKvsVariant(KviKvsArray * pArray)
{
	m_type = Array;
	m_u.pArray = pArray;
}

KviKvsVariant::KviKvsVariant(KviKvsHash * pHash)
{
	m_type = Hash;
	m_u.pHash = pHash;
}

KviKvsVariant::KviKvsVariant(double * pReal)
{
	m_type = Real;
	m_u.pReal = pReal;
}

KviKvsVariant::KviKvsVariant(double dReal)
{
	m_type = Real;
	m_u.pReal = new double;
	*(m_u.pReal) = dReal;
}

KviKvsVariant::KviKvsVariant(int iInteger)
{
	m_type = Integer;
	m_u.iInteger = iInteger;
}

KviKvsVariant::KviKvsVariant(const KviKvsVariant &v)
{
	m_type = v.m_type;
	switch(m_type)
	{
		case String: m_u.pString = new QString(*(v.m_u.pString)); break;
		case Array: m_u.pArray = new KviKvsArray(*(v.m_u.pArray)); break;
		case Hash: m_u.pHash = new KviKvsHash(*(v.m_u.pHash)); break;
		case Integer: m_u.iInteger = v.m_u.iInteger; break;
		case Real: m_u.pReal = new double; *(m_u.pReal) = *(v.m_u.pReal); break;
		//case Nothing: do nothing :); break;
		default: /* make gcc happy */ break;
	}
}

#define DELETE_CONTENTS \
	switch(m_type) \
	{ \
		case Array: delete m_u.pArray; break; \
		case Hash: delete m_u.pHash; break; \
		case String: delete m_u.pString; break; \
		case Real: delete m_u.pReal; break; \
		default: /* make gcc happy */ break; \
	}

KviKvsVariant::~KviKvsVariant()
{
	DELETE_CONTENTS
}

void KviKvsVariant::setString(QString * pString)
{
	DELETE_CONTENTS

	m_type = String;
	m_u.pString = pString;
}

void KviKvsVariant::setString(const QString &szString)
{
	DELETE_CONTENTS
	
	m_type = String;
	m_u.pString = new QString(szString);
}

void KviKvsVariant::setReal(double dReal)
{
	DELETE_CONTENTS
	
	m_type = Real;
	m_u.pReal = new double;
	*(m_u.pReal) = dReal;
}

void KviKvsVariant::setReal(double * pReal)
{
	DELETE_CONTENTS

	m_type = Real;
	m_u.pReal = pReal;
}

void KviKvsVariant::setInteger(int iInteger)
{
	DELETE_CONTENTS
	
	m_type = Integer;
	m_u.iInteger = iInteger;
}

void KviKvsVariant::setArray(KviKvsArray * pArray)
{
	DELETE_CONTENTS
	
	m_type = Array;
	m_u.pArray = pArray;
}

void KviKvsVariant::setHash(KviKvsHash * pHash)
{
	DELETE_CONTENTS

	m_type = Hash;
	m_u.pHash = pHash;
}

void KviKvsVariant::setNothing()
{
	DELETE_CONTENTS
	
	m_type = Nothing;
}

bool KviKvsVariant::isEmpty() const
{
	switch(m_type)
	{
		case Nothing: return true; break;
		case String: return m_u.pString->isEmpty(); break;
		case Array: return m_u.pArray->isEmpty(); break;
		case Hash: return m_u.pHash->isEmpty(); break;
		default: /* make gcc happy */ break;
	}
	return false;
}

bool KviKvsVariant::asNumber(KviKvsNumber &n) const
{
	if(isInteger())
	{
		n.m_u.iInteger = m_u.iInteger;
		n.m_type = KviKvsNumber::Integer;
		return true;
	}
	if(isReal())
	{
		n.m_u.dReal = *(m_u.pReal);
		n.m_type = KviKvsNumber::Real;
		return true;
	}
	if(asInteger(n.m_u.iInteger))
	{
		n.m_type = KviKvsNumber::Integer;
		return true;
	}
	if(asReal(n.m_u.dReal))
	{
		n.m_type = KviKvsNumber::Real;
		return true;
	}
	return false;
}


bool KviKvsVariant::asNumberExt(KviKvsNumber &n) const
{
	if(isInteger())
	{
		n.m_u.iInteger = m_u.iInteger;
		n.m_type = KviKvsNumber::Integer;
		return true;
	}
	if(isReal())
	{
		n.m_u.dReal = *(m_u.pReal);
		n.m_type = KviKvsNumber::Real;
		return true;
	}
	if(asInteger(n.m_u.iInteger))
	{
		n.m_type = KviKvsNumber::Integer;
		return true;
	}
	if(asReal(n.m_u.dReal))
	{
		n.m_type = KviKvsNumber::Real;
		return true;
	}
	if(asIntegerExt(n.m_u.iInteger))
	{
		n.m_type = KviKvsNumber::Integer;
		return true;
	}
	return false;
}


bool KviKvsVariant::asInteger(int &iVal) const
{
	switch(m_type)
	{
		case Integer:
			iVal = m_u.iInteger;
			return true;
		break;
		case String:
		{
			bool bOk;
			iVal = m_u.pString->toInt(&bOk);
			return bOk;
		}
		break;
		case Real:
			// FIXME: this truncates the value!
			iVal = (int)*(m_u.pReal);
			return true;
		break;
		default: /* make gcc happy */ break;
	}
	return false;
}

bool KviKvsVariant::asIntegerExt(int &iVal) const
{
	switch(m_type)
	{
		case Integer:
			iVal = m_u.iInteger;
			return true;
		break;
		case String:
		{
			bool bOk;
			iVal = m_u.pString->toInt(&bOk);
			if(bOk)return true;
			iVal = m_u.pString->length();
			return true;
		}
		break;
		case Real:
			// FIXME: this truncates the value!
			iVal = (int)*(m_u.pReal);
			return true;
		break;
		case Array:
			iVal = m_u.pArray->size();
			return true;
		break;
		case Hash:
			iVal = m_u.pHash->size();
			return true;
		break;
		default: /* make gcc happy */ break;
	}
	return false;
}


bool KviKvsVariant::asReal(double & dVal) const
{
	switch(m_type)
	{
		case Integer:
			dVal = m_u.iInteger;
			return true;
		break;
		case String:
		{
			bool bOk;
			dVal = m_u.pString->toDouble(&bOk);
			return bOk;
		}
		break;
		case Real:
			dVal = *(m_u.pReal);
			return true;
		break;
		default: /* make gcc happy */ break;
	}
	return false;
}

void KviKvsVariant::asString(QString &szBuffer) const
{
	switch(m_type)
	{
		case String: szBuffer = *(m_u.pString); break;
		case Array: szBuffer = QString::null; m_u.pArray->appendAsString(szBuffer); break;
		case Hash: szBuffer = QString::null; m_u.pHash->appendAsString(szBuffer); break;
		case Integer:  szBuffer.setNum(m_u.iInteger); break;
		case Real: szBuffer.setNum(*(m_u.pReal)); break;
		case Nothing: szBuffer = QString::null; break;
		default: /* make gcc happy */ break;
	}
}

void KviKvsVariant::appendAsString(QString &szBuffer) const
{
	switch(m_type)
	{
		case String: szBuffer.append(*(m_u.pString)); break;
		case Array: m_u.pArray->appendAsString(szBuffer); break;
		case Hash: m_u.pHash->appendAsString(szBuffer); break;
		case Integer:  KviQString::appendNumber(szBuffer,m_u.iInteger); break;
		case Real: KviQString::appendNumber(szBuffer,*(m_u.pReal)); break;
		default: /* make gcc happy */ break;
	}
}

void KviKvsVariant::dump(const char * prefix) const
{
	switch(m_type)
	{
		case String: debug("%s String(%s)",prefix,m_u.pString->latin1()); break;
		case Array: debug("%s Array",prefix); break;
		case Hash: debug("%s Hash",prefix); break;
		case Integer:  debug("%s Integer(%d)",prefix,m_u.iInteger); break;
		case Real: debug("%s Real(%f)",prefix,*(m_u.pReal)); break;
		case Nothing: debug("%s Nothing",prefix); break;
		default: /* make gcc happy */ break;
	}
}

void KviKvsVariant::copyFrom(const KviKvsVariant * v)
{
	DELETE_CONTENTS
	
	m_type = v->m_type;
	switch(m_type)
	{
		case String: m_u.pString = new QString(*(v->m_u.pString)); break;
		case Array: m_u.pArray = new KviKvsArray(*(v->m_u.pArray)); break;
		case Hash: m_u.pHash = new KviKvsHash(*(v->m_u.pHash)); break;
		case Integer: m_u.iInteger = v->m_u.iInteger; break;
		case Real: m_u.pReal = new double; *(m_u.pReal) = *(v->m_u.pReal); break;
		//case Nothing: do nothing :); break;
		default: /* make gcc happy */ break;
	}
}

void KviKvsVariant::copyFrom(const KviKvsVariant &v)
{
	DELETE_CONTENTS
	
	m_type = v.m_type;
	switch(m_type)
	{
		case String: m_u.pString = new QString(*(v.m_u.pString)); break;
		case Array: m_u.pArray = new KviKvsArray(*(v.m_u.pArray)); break;
		case Hash: m_u.pHash = new KviKvsHash(*(v.m_u.pHash)); break;
		case Integer: m_u.iInteger = v.m_u.iInteger; break;
		case Real: m_u.pReal = new double; *(m_u.pReal) = *(v.m_u.pReal); break;
		//case Nothing: do nothing :); break;
		default: /* make gcc happy */ break;
	}
}

void KviKvsVariant::takeFrom(KviKvsVariant * v)
{
	DELETE_CONTENTS
	
	m_type = v->m_type;
	// WARNING: if some of the union types are bigger than a pointer
	// this operation could fail!
	m_u.pString = v->m_u.pString;
	v->m_type = Nothing;
}

void KviKvsVariant::takeFrom(KviKvsVariant &v)
{
	DELETE_CONTENTS
	
	m_type = v.m_type;
	// WARNING: if some of the union types are bigger than a pointer
	// this operation could fail!
	m_u.pString = v.m_u.pString;
	v.m_type = Nothing;
}

void KviKvsVariant::getTypeName(QString &szBuffer) const
{
	switch(m_type)
	{
		case Nothing: szBuffer = "nothing"; break;
		case String: szBuffer = "string"; break;
		case Hash: szBuffer = "hash"; break;
		case Array: szBuffer = "array"; break;
		case Real: szBuffer = "real"; break;
		case Integer: szBuffer = "integer"; break;
		default: szBuffer = "internal_error"; break;
	}
}

#endif //!SHARED_VARIANT
