//=============================================================================
//
//   File : kvi_processmanager.cpp
//   Created on Wed 07 Apr 2004 03:03:52 by Szymon Stefanek
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 2004 Szymon Stefanek <pragma at kvirc dot net>
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//=============================================================================

#define __KVIRC__

#include "kvi_processmanager.h"
#include "kvi_uparser.h"
#include "kvi_window.h"
#include "kvi_out.h"
#include "kvi_locale.h"
#include "kvi_command.h"
#include "kvi_parameterlist.h"
#include "kvi_app.h"
#include "kvi_console.h"

KviProcessManager * KviProcessManager::m_pInstance = 0;



KviProcessDescriptor::KviProcessDescriptor(KviProcessDescriptorData * d,KviProcessManager * pParent)
: QObject(pParent)
{
	m_pData = d;
	m_pProcess = 0;
	m_pDataContainer = new KviDataContainer();
	m_pPingTimer = 0;
	m_pRunTimeTimer = 0;
	m_bDeletePending = false;
}

KviProcessDescriptor::~KviProcessDescriptor()
{
	if(m_pPingTimer)delete m_pPingTimer;
	if(m_pRunTimeTimer)delete m_pRunTimeTimer;
	if(m_pProcess)
	{
		QObject::disconnect(m_pProcess,0,this,0);
		m_pProcess->kill();
		delete m_pProcess;
	}
	delete m_pDataContainer;
	delete m_pData;
}

bool KviProcessDescriptor::start()
{
	QStringList args;
	
	if(m_pData->iFlags & KVI_PROCESSDESCRIPTOR_NOSHELL)
	{
		args = QStringList::split(" ",m_pData->szCommandline);
	} else {
		QString szShell = m_pData->szShell;
		if(szShell.isEmpty())szShell = "sh -c";
		args = QStringList::split(" ",szShell);
		args.append(m_pData->szCommandline);
	}
	
	m_pProcess = new QProcess(args);
	int c = QProcess::Stdin;
	
	if(m_pData->iFlags & KVI_PROCESSDESCRIPTOR_TRIGGERSTDOUT)
	{
		connect(m_pProcess,SIGNAL(readyReadStdout()),this,SLOT(readStdout()));
		c |= QProcess::Stdout;
	}
	
	if(m_pData->iFlags & KVI_PROCESSDESCRIPTOR_TRIGGERSTDERR)
	{
		connect(m_pProcess,SIGNAL(readyReadStderr()),this,SLOT(readStderr()));
		c |= QProcess::Stderr;
	}

	m_pProcess->setCommunication(c);
	
	connect(m_pProcess,SIGNAL(processExited()),this,SLOT(processExited()));
	
	if(!m_pProcess->start())
	{
		return false;
	}
	
	if(m_pData->iFlags & KVI_PROCESSDESCRIPTOR_TRIGGERSTARTED)
	{
		QString szPid; 
		szPid.setNum((int)(m_pProcess->processIdentifier()));
		if(trigger(EventStarted,szPid))
		{
			triggerSelfDelete();
			return true;
		}
	}

	if(m_pData->iMaxRunTime > 0)
	{
		m_pRunTimeTimer = new QTimer(this);
		connect(m_pRunTimeTimer,SIGNAL(timeout()),this,SLOT(maxRunTimeExpired()));
		m_pRunTimeTimer->start(m_pData->iMaxRunTime);
	}
	
	if(m_pData->iPingTimeout > 0)
	{
		m_pPingTimer = new QTimer(this);
		connect(m_pPingTimer,SIGNAL(timeout()),this,SLOT(ping()));
		m_pPingTimer->start(m_pData->iPingTimeout);
	}
	
	return true;
}

void KviProcessDescriptor::ping()
{
	if(trigger(EventPing,QString::null))
	{
		triggerSelfDelete();
	}
}

void KviProcessDescriptor::triggerSelfDelete()
{
	if(m_bDeletePending)return;
	m_bDeletePending = true;
	QTimer::singleShot(m_pData->iMaxRunTime,this,SLOT(selfDelete()));
}

void KviProcessDescriptor::selfDelete()
{
	delete this;
}

void KviProcessDescriptor::maxRunTimeExpired()
{
	trigger(EventTerminated,"0");
	triggerSelfDelete();
}

bool KviProcessDescriptor::trigger(CallbackEvent e,const QString &szData)
{
	if(m_bDeletePending)return false;
	
	if(!g_pApp->windowExists(m_pData->pWnd))
	{
		if(m_pData->iFlags & KVI_PROCESSDESCRIPTOR_KILLIFNOWINDOW)
		{
			return true;
		}
		m_pData->pWnd = g_pApp->activeConsole();
	}

	
	KviStr * szEventName;
	
	switch(e)
	{
		case EventStdout:
			szEventName = new KviStr("stdout");
		break;
		case EventStderr:
			szEventName = new KviStr("stderr");
		break;
		case EventTerminated:
			szEventName = new KviStr("terminated");
		break;
		case EventStarted:
			szEventName = new KviStr("started");
		break;
		case EventPing:
			szEventName = new KviStr("ping");
		break;
		default:
			debug("Ops... unknown trigger() CallbackEvent parameter in KviProcessDescriptor::trigger()");
			return false;
		break;
	}
	

	KviCommand cmd(m_pData->szCallback.ptr(),m_pData->pWnd,0,m_pDataContainer);

	cmd.setParams(new KviParameterList(szEventName,new KviStr(szData),new KviStr(m_pData->szMagic)));
	bool bResult = g_pUserParser->parseCommand(&cmd);
	cmd.forgetExtendedScopeDataContainer();


	if(!bResult)
	{
		if(cmd.hasError())
		{
			g_pUserParser->printError(&cmd);
			m_pData->pWnd->output(KVI_OUT_PARSERERROR,
				__tr2qs("Error triggered from process callback handler: killing process"));
			return true;
		}
		// if we have encountered a halt then return true to kill the process
		if(cmd.haltEncountered())return true;
		
		// no halt and no error... it should be a return call...
		// check for returned value
		if(cmd.returnEncountered())
		{
			if(cmd.returnValue().hasData())
			{
				m_pProcess->writeToStdin(QString(cmd.returnValue().ptr()));
			}
		} // else what ?
	}

	// must return true if the process has to be killed
	return false;
}

void KviProcessDescriptor::readStdout()
{
	if(m_bDeletePending)return;
	if(m_pData->iFlags & KVI_PROCESSDESCRIPTOR_OUTPUTBYBLOCKS)
	{
		QByteArray a = m_pProcess->readStdout();
		if(a.size() > 0)
			m_szStdoutBuffer += QString(a);
	} else {
		QString l = m_pProcess->readLineStdout();
		bool bBreak = false;
		while((!l.isNull()) && (!bBreak))
		{
			if(m_pData->iFlags & KVI_PROCESSDESCRIPTOR_TRIGGERSTDOUT)
			{
				if(trigger(EventStdout,l))
				{
					bBreak = true;
					triggerSelfDelete();
				}
			}

			l = m_pProcess->readLineStdout();
		}
	}
}

void KviProcessDescriptor::readStderr()
{
	if(m_bDeletePending)return;
	if(m_pData->iFlags & KVI_PROCESSDESCRIPTOR_OUTPUTBYBLOCKS)
	{
		QByteArray a = m_pProcess->readStderr();
		if(a.size() > 0)
			m_szStderrBuffer += QString(a);
	} else {
		QString l = m_pProcess->readLineStderr();
		bool bBreak = false;
		while((!l.isNull()) && (!bBreak))
		{
			if(m_pData->iFlags & KVI_PROCESSDESCRIPTOR_TRIGGERSTDERR)
			{
				if(trigger(EventStderr,l))
				{
					bBreak = true;
					triggerSelfDelete();
				}
			}

			l = m_pProcess->readLineStderr();
		}
	}
}


void KviProcessDescriptor::processExited()
{
	if(m_bDeletePending)return;

	readStdout(); // just to make sure
	readStderr(); // just to make sure
	
	if(m_pData->iFlags & KVI_PROCESSDESCRIPTOR_OUTPUTBYBLOCKS)
	{
		// trigger Stdout and Stderr once
		if(m_pData->iFlags & KVI_PROCESSDESCRIPTOR_TRIGGERSTDOUT)
		{
			if(trigger(EventStdout,m_szStdoutBuffer))
			{
				triggerSelfDelete();
				return;
			}
		}
		
		if(m_pData->iFlags & KVI_PROCESSDESCRIPTOR_TRIGGERSTDERR)
		{
			if(trigger(EventStdout,m_szStderrBuffer))
			{
				triggerSelfDelete();
				return;
			}
		}
	}

	if(m_pData->iFlags & KVI_PROCESSDESCRIPTOR_TRIGGERTERMINATED)
	{
		QString szRetVal;
		szRetVal.setNum(m_pProcess->exitStatus());
		trigger(EventTerminated,szRetVal);
	}
	
	triggerSelfDelete();
}




KviProcessManager::KviProcessManager()
: QObject()
{
}

KviProcessManager::~KviProcessManager()
{
}

void KviProcessManager::init()
{
	if(m_pInstance)return;
	m_pInstance = new KviProcessManager();
}

void KviProcessManager::done()
{
	if(!m_pInstance)return;
	delete m_pInstance;
	m_pInstance = 0;
}

bool KviProcessManager::execute(KviProcessDescriptorData * d)
{
	KviProcessDescriptor * pd = new KviProcessDescriptor(d,this);
	if(!pd->start())
	{
		delete d;
		delete pd;
		return false;
	}
	return true;
}

