//
//   File : kvi_scriptobject.cpp
//   Creation date : Tue Sep 05 2000 15:31:12 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
#define __KVIRC__
#define _KVI_SCRIPTOBJECT_CPP_

#define _KVI_DEBUG_CHECK_RANGE_
#include "kvi_debug.h"

#include "kvi_scriptobject.h"
#include "kvi_modulemanager.h"
#include "kvi_datacontainer.h"
#include "kvi_command.h"
#include "kvi_window.h"
#include "kvi_app.h"
#include "kvi_uparser.h"
#include "kvi_console.h"
#include "kvi_locale.h"
#include "kvi_error.h"
#include "kvi_out.h"

#include <qtimer.h>
#include <time.h>

// #warning "FINISH THE OBJECTS DOC"

/*
	@doc: objects
	@title:
		Object scripting
	@short:
		Object scripting overview
	@keyterms:
		objects , object scripting , complex data structures
	@body:
		[big]Introduction[/big]

		The KVIrc scripting language is not object oriented in nature.
		Anyway , objects are a highlevel abstraction that allow
		to write complex code in a "clean" way.
		So I've added at least some pseudo-object support.[br][br]

		[big]Basic concepts[/big]

		Objects are arranged in tree structures.
		Each object that you create is either toplevel object or a children
		of another object. A toplevel object is a parentless one.
		Obviously all objects can have children objects.[br][br]

		When an object is destroyed , all its children are also destroyed.
		The toplevel objects are automatically destroyed when KVIrc quits.
		The objects are global to the entire application (this is different
		from previous KVIrc releases where the objects were local
		to the current frame window and arranged in a single tree
		with a builtin root object).[br][br]

		Each object is an instance of a class that defines its features.
		Each object has also a name , that is not necessary unique and is assigned
		by the programmer; the name is just a mnemonic expedient, and
		you may also not need it.[br][br]

		Each object is identified by an [b]OPAQUE UNIQUE ID[/b].
		The ID is assigned by KVIrc and can be held in any variable.
		You can think the object id as a "handle for the object" or the object's pointer.
		Any action performed on the object will require its ID.[br][br]

		[big]Creation and destruction[/big]

		To create an object you must use the [fnc]$new[/fnc]()
		function. [fnc]$new[/fnc]() requires three parameters:[br]
		- The object class (more about object classes later in this document)[br]
		- The ID of the parent object , (that can be 0 for toplevel objects).[br]
		- The object name (eventually empty)[br]
		[example]
			%myobject = [fnc]$new[/fnc]([class]object[/class],0,theName)
		[/example]
		[fnc]$new[/fnc]() returns the ID of the newly created object, or
		the STRING "0" if the object creation fails
		(it is a string because the object id's are generally strings, and 0 is an "invalid object id").
		In well written scripts it is not common that the object creation fails, anyway
		you can check if the creation has failed in the following way:[br]
		[example]
			[cmd]if[/cmd](%myobject)[cmd]echo[/cmd] "Object created!"
			else [cmd]echo[/cmd] "Object creation failed!"
		[/example]
		You can also test the object ID's for equality:[br]
		[example]
			[cmd]if[/cmd](%myobject == %anotherobject)[cmd]echo[/cmd] "This is the same object!";
		[/example]
		The parent object id is optional, if not specified it is assumed to be 0.
		The object name is optional , but it may help you later in finding the object.[br][br]

		To destroy an object use the [cmd]delete[/cmd] command. (In previous versions
		this command was named "destroy" and delete is currently aliased to that name too).[br]
		[example]
			[cmd]delete[/cmd] %myobject
		[/example]
		If the destroyed object has children objects , these are destroyed too.[br][br]

		[big]Fields : objects as pseudo-structures[/big]

		All the objects can contain variable fields.
		You can set an object's field by using the object scope operator "->":[br]
		[example]
			%myobject-&gt;%fieldVariable = dataString
		[/example]
		To unset a field set it with empty data string (just like with a normal variable).
		To retrieve the field data use the object scope operator in the same way:[br]
		[example]
			[cmd]echo[/cmd] %myobject->%fieldVariable
		[/example]
		The '-&gt;' operator has been stolen from the C language.
		In the KVIrc scripting language it switches from the global namespace
		to the object's one.[br]
		So in the above example %fieldVariable is owned by the object.[br]
		The first character of the variable name has no special meaning in the
		object namespace (in the global namespace the variables starting
		with an uppercase letter are global to the application , the other ones are local
		to the command sequence). The variable names are completely case insensitive.[br][br]

		Any [doc:operators]operator[/doc] can be used with the object field variables:[br]
		[example]
			%myobject-&gt;%fieldVariable = 0
			%myobject-&gt;%fieldVarialbe ++
			[cmd]if[/cmd]0(%myobject->%fieldVariable != 1)[cmd]echo[/cmd] KVIrc is drunk , maybe a reboot will help ?
		[/example]
		You can simulate C structures "on the fly" by using objects and fields:[br]
		[example]
			# Create an user description on the fly
			%myobj = [fnc]$new[/fnc]([class]object[/class],0,userDescription)
			# Set the fields
			%myobj-&gt;%nickname = Pragma
			%myobj-&gt;%username = daemon
			%myobj-&gt;%hostname = pippo.pragma.org
			%myobj-&gt;%info = Pragma goes always sleep too late
			%myobj-&gt;%info [doc:operators]&lt;&lt;[/doc] and wakes up too late too!
			# Call an (user defined) alias that stores the data to a file
			storetofile %myobj
			# Destroy the object
			[cmd]delete[/cmd] %myobj
		[/example]
		The field variables can be also dictionaries:[br]
		[example]
			%theobj-&gt;%field[key] = something
		[/example]
		Unlike in C , there is no need to declare object fields.
		Any object can have any field variable ; an "unset" field is equivalent to an "empty" field.[br]
		Note:[br]
		The KVIrc scripting language is not typed.
		Any object class (be patient...I'll explain classes in a while) identifier can be stored in any KVIrc variable:
		it is not possible to find out the object features by "examining" its identifier.
		This may make the usage of objects a bit "unclear";
		Howewer , with some experience you will be able to use the objects in a very powerful way.
		The type-safety can be also simulated by a careful usage of object names;
		in the above example , the %myobj object was created with the "userDescription" name.
		The storetofile alias could check the passed object's name and refuse to work
		if that does not match "userDescription".[br][br]

		A more complex use of fields will be described later in this document.[br][br]

		[big]Member functions[/big]

		Just like in C++ , the objects have member functions.
		For example , the "object" class (again...read on) objects export the [classfnc:object]$name[/classfnc]()
		and [classfnc:object]$className[/classfnc]() functions.[br]
		[example]
			%tmp = [fnc]$new[/fnc]([class]object[/class],0,myobject)
			[cmd]echo[/cmd] The object's name is %tmp->[classfnc:object]$name[/classfnc]() , the class name is %tmp->[classfnc:object]$className[/classfnc]()
			# Destroy the object
			[cmd]delete[/cmd] %tmp
		[/example]
		Another cool function exported by the [class:object]object[/class] class is the
		[classfnc:object]$children[/classfnc]() function.
		It returns a comma separated list of children identifiers.[br]
		[example]
			%tmp = [fnc]$new[/fnc]([class]object[/class],0,myobject)
			%tmpchild = [fnc]$new[/fnc]([class]object[/class],%tmp,child1)
			%tmpchild = [fnc]$new[/fnc]([class]object[/class],%tmp,child2)
			%tmpchild = [fnc]$new[/fnc]([class]object[/class],%tmp,child3)
			[cmd]echo[/cmd] The object's children list is : %tmp->[classfnc:object]$children[/classfnc]()
			# Destroy the object and the children
			[cmd]delete[/cmd] %tmp
		[/example]
		There are two special functions for each objects: the "constructor" and the "destructor".
		You will find more informations on constructors and destructors later in this document,
		for now it's enough that you know that these functions are called automatically by KVirc:
		the constructor is called when the object is created and the destructor is called when the
		object is being destroyed with [cmd]delete[/cmd].[br][br]

		The object functions can be reimplemented on-the-fly
		by using the [cmd]privateimpl[/cmd] command: you can simply modify the behaviour of the function
		by writing your own function body.
		(This is an uncommon feature: unlike many other languages , you can reimplement object
		functions at run-time, when the object has been already created.)[br][br]

		A more complex example[br]
		[example]
			%tmp = [fnc]$new[/fnc]([class]object[/class],0,myobject)
			[cmd]foreach[/cmd](%i,1,2,3)
			{
				%tmpchild = [fnc]$new[/fnc]([class]object[/class],%tmp,child%i)
				[cmd]privateimpl[/cmd](%tmpchild,destructor){ [cmd]echo[/cmd] Object [fnc]$this[/fnc] ([fnc]$this[/fnc]-&gt;[classfnc:object]$name[/classfnc]()) destroyed; }
			}
			[cmd]privateimpl[/cmd](%tmp,destructor)
			{
				%count = 0;
				[cmd]foreach[/cmd](%t,[fnc]$this[/fnc]-&gt;[classfnc:object]$children[/classfnc]())
				{
					[cmd]echo[/cmd] Children : %t-&gt;[classfnc:object]$name[/classfnc]() with class %t-&gt;[classfnc:object]$class[/classfnc]()
					%count++
				}
				[cmd]echo[/cmd] Just before destroying my %count children.
			}
			# Destroy the object and the children
			[cmd]delete[/cmd] %tmp
		[/example]

		In the example above four objects have been created.
		A "parent" object named "myobject" , and three children objects.
		The destructor has been reimplemented for each child object,
		to make it "say" its name (Please note the usage of [fnc]$this[/fnc]).
		In the parent destructor the children have been counted and listed.[br]
		Then the parent object is destroyed causing to:[br]
		- trigger the parent destructor.[br]
		- destroy all the children (and conseguently trigger all the "individual" destructors).[br][br]

		Not all the object functions must return a value:
		If a function does not return a meaningful value , or you just want to ignore it , you can call it in the following way:[br]
		[example]
			%anyobject-&gt;$functionname()
		[/example]
		[br]

		[big]Classes[/big]
		As said before , all objects are instances of a specific class.
		This concept is common to almost all object oriented languages.
		A class is a collection of methods that define an object's behaviour.
		Hehe...it is not easy to explain it , so I'll try with an example:[br]
		[example]
		class HostAddress
		{
			field hostname
			function ipnumber()
			function isLocalhost()
		}
		[/example]
		The above class is a rappresentation of a host address.
		You create an [b]instance of this class[/b] and set the hostname field, for example,
		to www.kernel.org.
		The object is now able to give you informations about the hostname in a transparent way:
		You can call the ipnumber() function, and the object will return you the
		digits and dots rappresentation of www.kernel.org.
		The isLocalhost() function will return true if the hostname refers to the local machine
		The object internal job is hidden from the user , but probably it will be a huge job.
		To obtain the IP number from the hostname , the object will probably have to perform a DNS call (usually a complex task).
		To check if the hostname references the local machine , the object will have to obtain the local hostname
		from the system (in some "unspecified" way) and then compare it with the given "hostname" field.[br][br]

		The internal job of the object is defined by the "implementation of the class".
		Obviously , the programmer that creates the class has to write that implementation.[br]

		[example]
		class HostAddress
		{
			field hostname
			function ipnumber()
			{
				find the nearest DNS server
				make the dns call
				wait for the response
				decode the response
			}
			function isLocalhost()
			{
				query the kernel for the local hostname
				compare the obtained hostname with the hostname field
			}
		}
		[/example]
		In the above example I have "implemented" the two functions by using a "fantastic" language.[br][br]

		Let's go back to the real world.[br][br]

		KVirc contains a [doc:classes]set of built-in ready-to-use classes[/doc].
		The basic class is [class]object[/class]: all the other classes are derived from this (more about
		object inheritance later in this doc).[br][br]

		Another available class is [class]socket[/class] that is an interface to the real system sockets.
		An instance of the [class]socket[/class] class can connect and communicate with other hosts on the net.[br][br]

		The [b]class definitions are GLOBAL to the entire application[/b]: all server windows share them.[br][br]

		So now we can say that in KVIrc
		[b]a CLASS is a collection of features that define the behaviour of an object.
		The user interface to the class are the member functions and the events.[/b][br][br]

		[big]Inheritance[/big]

		Someone asked for derived classes ?[br]
		Here we go:[br]
		The [cmd]class[/cmd] command allows you to define new object classes.
		In KVI++, A new class must be always derived from some other class: the lowest possible
		level of inheritance is 1: deriving from class [class]object[/class].[br]
		[example]
			[cmd]class[/cmd](helloworld,object)
			{
				sayhello()
				{
					[cmd]echo[/cmd] Hello world!
				}
			}
		[/example]

		The above class is named "helloworld". It inherits the [class]object[/class] class.
		This means that it acquires all the [class]object[/class] fuunctions: [classfnc:object]$name[/classfnc]() ,
		[classfnc:object]$class[/class]() , [classfnc:object]$children[/classfnc]()...
		Additionally , it has the $sayhello() function, that "echoes Hello world" to the console.
		Now you can create an instance of this class:
		[example]
			%instance = [fnc]$new[/fnc](helloworld)
			%instance->$sayhello()
		[/example]
		You should see "Hello world" printed in the console.
		Easy job...let's make the things a bit more complex now:
		derive another class from helloworld and make it say "hello" in two different languages:[br]
		[example]
		[cmd]class[/cmd](localizedhelloworld,helloworld)
		{
			[comment]# define the setlanguage function[/comment]
			[comment]# note that <$0 = language> is just a programmer reminder[/comment]
			setlanguage(<$0 = language>)
			{
				[cmd]if[/cmd](($0 == english) || ($0 == italian))
				{
					[fnc:$this]$$[/fnc]->%lang = $0
					[cmd]return[/cmd] 1
				} else {
					[cmd]echo[/cmd] I don't know that language ($0)
					[cmd]echo[/cmd] defaulting to english
					[fnc:$this]$$[/fnc]->%lang = english
					[cmd]return[/cmd] 0
				}
			}

			sayhello()
			{
				[cmd]if[/cmd]([fnc:$this]$$[/fnc]->%lang == italian)[cmd]echo[/cmd] Ciao mondo!
				else [fnc:$this]$$[/fnc]->$helloworld:sayhello()
			}
		}
		[/example]
		Now you can call:[br]
		[example]
		%m = [fnc]$new[/fnc](localizedhelloworld)
		%m->$setLanguage(italian)
		%m->$sayhello()
		%m->$setLanguage(english)
		%m->$sayhello()
		%m->$setLanguage(turkish)
		%m->$sayhello()
		[cmd]delete[/cmd] %myobj
		[/example]
		The class defined above is inherited from the previously defined helloworld class:
		so it inherits the "object" class functions and events and the sayhello function from "helloworld".
		In addition a setlanguage function is defined that stores in a variable the language name passed
		as a parameter (after checking its validity). ($0 evaluates to the first parameter passed)
		If the language is unknown the setlanguage function will return 0 (false).
		Now we want to be able to say "hello world" in italian and english.
		So we [b]override[/b] the inherited sayhello function.
		"To override" means "to reimplement" : if you call %object->$sayhello() and %object
		contains the ID of an instance of class "localizedhelloworld" , the new implementation of that function willl be called (executed).
		The inherited sayhello was able to say "hello world" only in english , so we can still use it in the new implementation
		without rewriting its contents. So if the language set is "not italian" we assume that it is english and
		call the [b]base class implementation[/b].[br]
		[example]
			[fnc]$this/[fnc]->$helloworld:sayhello()
			[comment]# equivalent to $$->$helloworld:sayhello(),[/comment]
			[comment]# to $this->$helloworld::sayhello(),[/comment]
			[comment]# and to $$->$helloworld::sayhello()[/comment]
		[/example]
		otherwise the language is italian and we say "hello" in italian :).
		So , to call a base class implementation of a function we "prepend" the base class name before the function name in the call.
		The base class name could be also [class]object[/class] in this case , but the [class]object[/class] class has no "sayhello" function defined
		so it would result in an error.[br][br]
		In the above example , all the values of [fnc]$this[/fnc]</a>-&gt;%language
		that are not equal to "italian" are assumed to be "english".
		This is not always true , for example , just after the object creation the %language variable field
		is effectively empty. The above class works correctly in this case , but we might want to have always
		a coherent state of the field variables , so we need another concept: the class [b]constructor[/b]
		that will be discussed in the next paragraph.[br][br]
	
		Note: multiple inheritance (inheritance from more than one base class) is not implemented , KVIrc is not a compiler :)[br][br]

		Objects are much more powerful....[br][br]

		Do a [cmd]clearobjects[/cmd] to cleanup the old class definitions , and read on.[br][br]

		[big]Constructors and destructors[/big]

		The class constructor is a [b]function[/b] that is called automatically just after the object
		has been created internally by KVIrc and just before the [fnc]$new[/fnc]
		function returns. It should be used to setup the internal object state.[br]
		Unlike in C++ , in KVIrc , the constructor CAN return a value:[br]
		If it returns 0 it signals the object creation failure : the object
		is immediately destroyed and [fnc]$new[/fnc]() returns 0 to the caller.
		Any other return value is treated as success , so the object is effectively
		created and [fnc]$new[/fnc]() returns its ID to the caller.[br]
		All the builtin classes have a constructor defined that will almost never fail (only if we run out of memory),
		so you can avoid to check the [fnc]$new[/fnc]() return value
		when creating the instances of the built-in classes.[br][br]

		In derived classes you can override the constructor to setup your object's state.[br]
		You should [b]always call the base class constructor[/b] in your overridden one , to setup
		the base class state , and propagate its return value (eventually modified if the base class
		constructor is succesfull but your derived class initialization fails).[br]
		In practice , the builtin class constructors do nothing other than setting the return
		value to 1 so you can even avoid to call them, but in any other case you must do it.[br][br]

		This is different from C (for example), where the constructors are called (more or less)
		automatically.[br][br]
		
		[big]Signals and slots[/big]

		The signals and slots are a powerful mean of inter-object communication.
		A signal is emitted by an object to notify a change in its state.
		For example , the [class:button]button class[/class] emits the
		[classsignal:button]clicked[/classsignal] signal when the user clicks the button.[br][br]
		A signal is emitted by an object and can be received and handled by any other existing object
		(including the object that emits the signal).[br]
		The handler function for a signal is called "slot".[br]
		It is just a convention : in fact , a slot is a normal object function (and any object function can be a slot).
		More than one slot can be connected to a single signal , and more signals can be connected to a single slot.[br]
		In this way , many objects can be notified of a change in a single object , as well as a single object
		can easily handle state-changes for many objects.[br]
		The signal / slot behaviour could be easily implemented by a careful usage of object functions.
		[b]So why signals and slots ?[/b][br]
		Because signals are much more powerful in many situations.
		The signals have no equivalent in C/C++...but they have been implemented in many highlevel
		C/C++ libraries and development kits (including the system-wide signal/handler mechanism implemented
		by all the modern kernels and used in inter-process communication).[br]
*/


KviScriptObjectController * g_pScriptObjectController = 0;

KviScriptObjectClass::KviScriptObjectClass(KviScriptObjectClass * par,const char * name,
		KviScriptObjectAllocateInstanceProc proc,bool bBuiltin)
{
	m_pParentClass = par;
	if(m_pParentClass)m_pParentClass->registerChildClass(this);
	m_szName       = name;
	m_bBuiltin     = bBuiltin;
	m_pFunctionHandlers = new QAsciiDict<KviScriptObjectFunctionHandler>(17,false,true);
	m_pFunctionHandlers->setAutoDelete(true);
	m_pChildClasses = new KviPtrList<KviScriptObjectClass>;
	m_pChildClasses->setAutoDelete(false);
	m_allocProc    = proc ? proc : par->m_allocProc;

	// inherit everything from the class above
	if(par)
	{
		QAsciiDictIterator<KviScriptObjectFunctionHandler> it(*(par->m_pFunctionHandlers));
		while(it.current())
		{
			KviScriptObjectFunctionHandler * aux = new KviScriptObjectFunctionHandler;
			aux->szScriptHandler = it.current()->szScriptHandler;
			aux->iFlags = KVI_SCRIPT_OBJECT_FUNCTION_INHERITED;
			aux->proc = it.current()->proc;
			m_pFunctionHandlers->insert(it.currentKey(),aux);
			++it;
		}
	}
	// "object" class is automatically registered in the controller constructor
	if(g_pScriptObjectController)g_pScriptObjectController->registerClass(this);
}

KviScriptObjectClass::~KviScriptObjectClass()
{
//	debug("KviScriptObjectClass::~KviScriptObjectClass(%s)",name());
	g_pScriptObjectController->killAllObjectsWithClass(this);
//	debug("All objects with class %s killed",name());
	if(m_pParentClass)m_pParentClass->unregisterChildClass(this);
	g_pScriptObjectController->unregisterClass(this);
	delete m_pFunctionHandlers;
	while(m_pChildClasses->first())delete m_pChildClasses->first();
	delete m_pChildClasses;
}

void KviScriptObjectClass::registerFunctionHandler(const char * func,
		KviScriptObjectFunctionHandlerProc proc,const char * buffer,bool bBuiltin)
{
	KviScriptObjectFunctionHandler * h = new KviScriptObjectFunctionHandler;
	h->proc = proc;
	h->szScriptHandler = buffer ? buffer : "";
	KviScriptObjectFunctionHandler * old = m_pFunctionHandlers->find(func);
	if(old)h->iFlags = bBuiltin ? KVI_SCRIPT_OBJECT_FUNCTION_BUILTINOVERRIDE : KVI_SCRIPT_OBJECT_FUNCTION_OVERRIDE;
	else h->iFlags = bBuiltin ? KVI_SCRIPT_OBJECT_FUNCTION_BUILTIN : 0;
	m_pFunctionHandlers->replace(func,h);
}

void KviScriptObjectClass::registerEmptyFunctionHandler(const char * func)
{
	registerFunctionHandler(func,(KviScriptObjectFunctionHandlerProc)0,0,true);
}

KviScriptObject * KviScriptObjectClass::allocateInstance(KviScriptObject * par,const char * nam,KviCommand *c,KviParameterList * params)
{
	if(!m_allocProc)return 0;
	KviScriptObject * o = m_allocProc(this,par,nam);
	if(!o)return 0;

	if(!o->init(c))
	{
		// internal init failure : abort
		delete o;
		return 0;
	}

	KviStr ret;

	KviParameterList * copy = new KviParameterList();
	copy->setAutoDelete(true);
	while(KviStr * tmp = params->next())
	{
		KviStr * szCopy = new KviStr(*tmp);
		copy->append(szCopy);
	}

	if(!o->callEventFunction("constructor",&ret,copy))
	{
		// ops...constructor failed (script error!)
		delete o;
		return 0;
	} else {
		if(kvi_strEqualCI(ret.ptr(),"0"))
		{
			// implementation failure...
			delete o;
			return 0;
		}
	}

	return o;
}

void KviScriptObjectClass::registerChildClass(KviScriptObjectClass *c)
{
	m_pChildClasses->append(c);
}

void KviScriptObjectClass::unregisterChildClass(KviScriptObjectClass *c)
{
	m_pChildClasses->removeRef(c);
}


////////////////////////////////////////////////////////////////////////////////////////

static KviScriptObject * objectClassCreateInstance(KviScriptObjectClass * cls,KviScriptObject *par,const char * nam)
{
	return new KviScriptObject(cls,par,nam);
}

////////////////////////////////////////////////////////////////////////////////////////

KviScriptObjectController::KviScriptObjectController()
: QObject(0,"script_object_controller")
{
	m_pTopLevelObjectList = new KviPtrList<KviScriptObject>;
	m_pTopLevelObjectList->setAutoDelete(false);
	m_pObjectDict = new QAsciiDict<KviScriptObject>(31,true,true);
	m_pObjectDict->setAutoDelete(false);
	m_pClassDict = new QAsciiDict<KviScriptObjectClass>(17,true,true);
	m_pClassDict->setAutoDelete(false);

	// allocate the "object" builtin class
	// this is the only one that is always in core memory
	m_pObjectClass = new KviScriptObjectClass(0,"object",objectClassCreateInstance,true);
	m_pObjectClass->registerFunctionHandler("name",KVI_PTR2MEMBER(KviScriptObject::functionName),0,true);
	m_pObjectClass->registerFunctionHandler("startTimer",KVI_PTR2MEMBER(KviScriptObject::functionStartTimer),0,true);
	m_pObjectClass->registerFunctionHandler("killTimer",KVI_PTR2MEMBER(KviScriptObject::functionKillTimer),0,true);
	m_pObjectClass->registerFunctionHandler("killTimers",KVI_PTR2MEMBER(KviScriptObject::functionKillTimers),0,true);
	m_pObjectClass->registerFunctionHandler("className",KVI_PTR2MEMBER(KviScriptObject::functionClassName),0,true);
	m_pObjectClass->registerFunctionHandler("findChild",KVI_PTR2MEMBER(KviScriptObject::functionFindChild),0,true);
	m_pObjectClass->registerFunctionHandler("childCount",KVI_PTR2MEMBER(KviScriptObject::functionChildCount),0,true);
	m_pObjectClass->registerFunctionHandler("emit",KVI_PTR2MEMBER(KviScriptObject::functionEmit),0,true);
	m_pObjectClass->registerFunctionHandler("children",KVI_PTR2MEMBER(KviScriptObject::functionChildren),0,true);
	m_pObjectClass->registerFunctionHandler("signalSender",KVI_PTR2MEMBER(KviScriptObject::functionSignalSender),0,true);
	m_pObjectClass->registerFunctionHandler("signalName",KVI_PTR2MEMBER(KviScriptObject::functionSignalName),0,true);
	m_pObjectClass->registerFunctionHandler("destructor",KVI_PTR2MEMBER(KviScriptObject::functionDestructor),0,true);
	m_pObjectClass->registerFunctionHandler("parent",KVI_PTR2MEMBER(KviScriptObject::functionParent),0,true);
	m_pObjectClass->registerEmptyFunctionHandler("constructor");
	m_pObjectClass->registerEmptyFunctionHandler("timerEvent");
	m_pClassDict->insert("object",m_pObjectClass);
}

KviScriptObjectController::~KviScriptObjectController()
{
	while(m_pTopLevelObjectList->first())delete m_pTopLevelObjectList->first();
	delete m_pTopLevelObjectList; // empty list
	delete m_pObjectDict; // empty dict
	m_pObjectDict = 0;
	delete m_pObjectClass; // delete the class tree
	delete m_pClassDict;  // empty dict
}

void KviScriptObjectController::killAllObjectsWithClass(KviScriptObjectClass * cl)
{
	if(!m_pObjectDict)return; // no more objects at all...
	KviPtrList<KviScriptObject> l;
	l.setAutoDelete(true);

	for(KviScriptObject * o = m_pTopLevelObjectList->first();o;o = m_pTopLevelObjectList->next())
	{
		if(o->getClass() == cl)l.append(o);
		else o->killAllChildrenWithClass(cl);
	}
}

void KviScriptObjectController::clearUserClasses()
{
	QAsciiDictIterator<KviScriptObjectClass> it(*m_pClassDict);
	KviPtrList<KviScriptObjectClass> l;
	l.setAutoDelete(true);
	while(it.current())
	{
		if(!(it.current()->isBuiltin()))l.append(it.current());
		++it;
	}
}

void KviScriptObjectController::clearInstances()
{
	while(m_pTopLevelObjectList->first())delete m_pTopLevelObjectList->first();
	delete m_pTopLevelObjectList; // empty list
	delete m_pObjectDict; // empty dict
	m_pTopLevelObjectList = new KviPtrList<KviScriptObject>;
	m_pTopLevelObjectList->setAutoDelete(false);
	m_pObjectDict = new QAsciiDict<KviScriptObject>(31,true,true);
	m_pObjectDict->setAutoDelete(false);
}

void KviScriptObjectController::registerClass(KviScriptObjectClass *c)
{
	m_pClassDict->insert(c->name(),c);
}

void KviScriptObjectController::unregisterClass(KviScriptObjectClass *c)
{
	m_pClassDict->remove(c->name());
}

void KviScriptObjectController::registerObject(KviScriptObject *o)
{
	m_pObjectDict->insert(o->m_szId.ptr(),o);
	if(o->parent() == 0)m_pTopLevelObjectList->append(o);
}

void KviScriptObjectController::unregisterObject(KviScriptObject *o)
{
	m_pObjectDict->remove(o->m_szId.ptr());
	if(o->parent() == 0)m_pTopLevelObjectList->removeRef(o);
}

KviScriptObjectClass * KviScriptObjectController::lookupClass(const char * cl)
{
	KviScriptObjectClass * pC = m_pClassDict->find(cl);
	if(!pC)
	{
		// maybe we need to load the object library ?
		KviModule * pModule = g_pModuleManager->getModule("objects");
		if(!pModule)
		{
			debug("ops...something wrong with the libkviobjects module!");
		} else pC = m_pClassDict->find(cl);
	}
	return pC;
};

bool KviScriptObjectController::objectExists(KviScriptObject * o)
{
	QAsciiDictIterator<KviScriptObject> it(*m_pObjectDict);
	while(it.current())
	{
		if(it.current() == o)return true;
		++it;
	}
	return false;
}

///////////////////////////////////////////////////////////////////////////////////////

/*
	@doc: object
	@keyterms:
		object class, object, class
	@title:
		object class
	@type:
		class
	@short:
		Base class for all the KVIrc objects
	@inherits:
		none
	@description:
		This is the base class for all the builtin KVirc object classes.
		It exports the functions to retrieve the object name, to iterate
		through children objects and to lookup a child object by name or class.
		Additionally , this class provides the builtin timer functionality.
		The [classfnc]$constructor[/classfnc] and [classfnc]$destructor[/classfnc]
		functions are empty implementations that all the other classes inherit.
	@functions:
		!fn: $constructor()
		Constructor for this object class.
		The default implementation does nothing.
		!fn: $destructor()
		Destructor for this object class.
		The default implementation emits the signal "[classsignal]destroyed[/classsignal]".
		!fn: $name()
		Returns the name of this object.
		!fn: $parent()
		Returns the parent object of this object or 0 if this object has no parent.
		!fn: $timerEvent(<timerId>)
		Handler for the timer events.
		The default implementation does nothing.
		See also [classfnc]$startTimer[/classfnc]()
		and [classfnc]$killTimer[/classfnc]().
		!fn: $startTimer(<timeout>)
		Starts a builtin timer for this object and returns its timer id
		as a string or '-1' if the <timeout> was invalid.
		The [classfnc]$timerEvent[/classfnc]() handler function
		will be called at each <timeout>. The <timeout> is in milliseconds.
		!fn: $killTimer(<timer id>)
		Stops the timer specified by <timer id>.
		!fn: $killTimers()
		Stops all the builtin timers running.
		!fn: $className()
		Returns the class name of this object instance
		!fn: $findChild(<class>,<name>)
		Returns the first child that matches <class> and <name>.
		If <class> is an empty string, any class matches,
		if <name> is an empty string, any name matches.
		This function traverses the entire tree of children
		but is NOT recursive.
		!fn: $childCount()
		Returns the number of children objects
		!fn: $emit(<signal_name>[,parameters])
		Emits the signal <signal_name> passing the optional [parameters].
		See the [doc:objects]objects documentation[/doc] for an overview of signals and slots.
		!fn: $children()
		Returns an array of children object identifiers.
		!fn: $signalSender()
		Returns the current signal sender when in a slot connected to a signal.
		In other contexts this function returns an empty string.
		You can safely use it to test if the current function has been
		triggered directly or from a signal emission.
		!fn: $signalName()
		Returns the name of the signal last signal that has triggered
		one of this object's slots.
		This means that in a slot handler it returns the name of the signal
		that has triggered it.
	@signals:
		!sg: destroyed()
		Emitted by the default implementation of [classfnc]$destructor[/classfnc].
		If you reimplement [classfnc]$destructor[/classfnc] in one of the derived
		classes (or as a private implementation), and still want this signal
		to be emitted you must emit it by yourself, or (better) call the base class
		destructor.
*/




// This will overflow at 2^32 (4.294.967.296) , and when it will happen
// the lowest numbered objects will be SURELY dead
// (otherwise all the objects will occupy some terabytes of memory)
static unsigned int g_uScriptObjectId = 1;

KviScriptObject::KviScriptObject(KviScriptObjectClass * cla,KviScriptObject * par,const char * nam)
: QObject(par,nam)
{
	m_pObject            = 0;
	m_bObjectOwner       = true; // true by default

	m_pClass             = cla;
	// to be sure that it is unique just add the time string
	m_szId.sprintf("%u.%u",g_uScriptObjectId,(unsigned int)time(0));
	g_uScriptObjectId++;

	m_pChildList         = new KviPtrList<KviScriptObject>;
	m_pChildList->setAutoDelete(false);

	m_pDataContainer     = new KviDataContainer(false);

	m_pFunctionHandlers  = 0; // no local function handlers yet!

	m_bInDelayedDeath    = false;

	m_pSignalDict        = 0; // no signals connected to remote slots
	m_pConnectionList    = 0; // no local slots connected to remote signals

	if(par)par->registerChild(this);

	g_pScriptObjectController->registerObject(this);

//	debug("Hello world!");
//	[root@localhost cvs3]# kvirc
//	Hello world!
//	[root@localhost cvs3]# date
//	Tue Sep  5 21:53:54 CEST 2000
//	[root@localhost cvs3]#
}


KviScriptObject::~KviScriptObject()
{
	m_bInDelayedDeath = true;

	callEventFunction("destructor");

	while(m_pChildList->first())delete m_pChildList->first();
	delete m_pChildList;

	// Disconnect all the signals

	if(m_pSignalDict)
	{
		//debug("DISCONNECTING SIGNALS");
		QAsciiDictIterator<KviScriptObjectConnectionList> it(*m_pSignalDict);
	
		//debug("SIGNAL DICT HAS %d ENTRIES",m_pSignalDict->count());


		while(it.current())
		{
			//debug("WILL DISCONNECT it.current() = 
			KviScriptObjectConnectionListIterator cit(*(it.current()));
			while(cit.current())
			{
				disconnectSignal(it.currentKey(),cit.current());
				// ++cit // NO!...we point to the next now!
			}
			// the iterator should automatically point to the next now
			//if(m_pSignalDict)++it;
		}
	
		__range_invalid(m_pSignalDict);
	}

	// Disconnect all the slots

	if(m_pConnectionList)
	{
		KviScriptObjectConnectionListIterator cit(*m_pConnectionList);
		while(cit.current())
		{
			cit.current()->pSourceObject->disconnectSignal(cit.current()->szSignal,cit.current());
			//++cit;// NO!... we point to the next now!
		}
		__range_invalid(m_pConnectionList);
	}

	g_pScriptObjectController->unregisterObject(this);

	if(parentObject())parentObject()->unregisterChild(this);

	if(m_pObject)
	{
		disconnect(m_pObject,SIGNAL(destroyed()),this,SLOT(objectDestroyed()));
		if(m_bObjectOwner)delete m_pObject;
	}

	
	delete m_pDataContainer;
	if(m_pFunctionHandlers)delete m_pFunctionHandlers;
}

bool KviScriptObject::init(KviCommand *)
{
	return true;
}


QWidget * KviScriptObject::parentScriptWidget()
{
	if(parentObject())
	{
		if(parentObject()->object())
		{
			if(parentObject()->object()->isWidgetType())
				return (QWidget *)(parentObject()->object());
		}
	}
	return 0;
}

void KviScriptObject::unregisterChild(KviScriptObject *c)
{
	m_pChildList->removeRef(c);
}

void KviScriptObject::registerChild(KviScriptObject *c)
{
	m_pChildList->append(c);
}

// SIGNAL/SLOT stuff

bool KviScriptObject::connectSignal(const char * sigName,KviScriptObject * target,const char * slotName)
{
	if(!target->lookupFunctionHandler(slotName))return false; // no such slot

	if(!m_pSignalDict)
	{
		m_pSignalDict = new QAsciiDict<KviScriptObjectConnectionList>(7,false,true);
		m_pSignalDict->setAutoDelete(true);
	}

	KviScriptObjectConnectionList * l = m_pSignalDict->find(sigName);
	if(!l)
	{
		l = new KviScriptObjectConnectionList;
		l->setAutoDelete(true);
		m_pSignalDict->insert(sigName,l);
	}

	KviScriptObjectConnection * con = new KviScriptObjectConnection;

	con->pSourceObject = this;
	con->pTargetObject = target;
	con->szSignal      = sigName;
	con->szSlot        = slotName;

	l->append(con);
	target->registerConnection(con);

	return true;
}

void KviScriptObject::registerConnection(KviScriptObjectConnection *con)
{
	if(!m_pConnectionList)
	{
		m_pConnectionList = new KviScriptObjectConnectionList;
		m_pConnectionList->setAutoDelete(false);
	}
	m_pConnectionList->append(con);
}

bool KviScriptObject::disconnectSignal(const char * sigName,KviScriptObject * target,const char * slotName)
{
	if(!m_pSignalDict)return false; //no such signal to disconnect

	KviScriptObjectConnectionList * l = m_pSignalDict->find(sigName);
	if(!l)return false;

	KviScriptObjectConnectionListIterator it(*l);

	while(KviScriptObjectConnection * sl = it.current())
	{
		if(sl->pTargetObject == target)
		{
			if(kvi_strEqualCI(sl->szSlot.ptr(),slotName))
			{
				target->unregisterConnection(sl);
				l->removeRef(sl);
				if(l->isEmpty())m_pSignalDict->remove(sigName);
				if(m_pSignalDict->isEmpty())
				{
					delete m_pSignalDict;
					m_pSignalDict = 0;
				}
				return true;
			}
		}
		++it;
	}
	return false;
}

bool KviScriptObject::disconnectSignal(const char * sigName,KviScriptObjectConnection * con)
{
	//debug("DISCONNECTING SIGNAL %s from connection %d",sigName,con);
	__range_valid(m_pSignalDict);
	if(!m_pSignalDict)return false;
	KviScriptObjectConnectionList * l = m_pSignalDict->find(sigName);
	__range_valid(l);
	if(!l)return false;
	//debug("CONNECTION LIST FOUND %d",l);
	con->pTargetObject->unregisterConnection(con);
	__range_valid(l->findRef(con) > -1);
	l->removeRef(con);
	if(l->isEmpty())m_pSignalDict->remove(sigName);
	//else debug("CONNECTION LIST STILL NOT EMPTY :%d ITEMS",l->count());
	if(m_pSignalDict->isEmpty())
	{
		//debug("SIGNAL DICT EMPTY : KILLING");
		delete m_pSignalDict;
		m_pSignalDict = 0;
	}// else debug("SIGNAL DICT NON EMPTY : still containing %d ITEMS",m_pSignalDict->count());
	return true;
}

bool KviScriptObject::unregisterConnection(KviScriptObjectConnection * con)
{
	if(!m_pConnectionList)return false;
	bool bOk = m_pConnectionList->removeRef(con); // no auto delete !
	if(!bOk)return false;
	if(m_pConnectionList->isEmpty())
	{
		delete m_pConnectionList;
		m_pConnectionList = 0;
	}
	return true;
}

void KviScriptObject::emitSignal(const char * sigName,KviStr * pBuffer,KviParameterList * params,KviCommand *c)
{
	if(!m_pSignalDict)return;

	KviScriptObjectConnectionList * l = m_pSignalDict->find(sigName);
	if(!l)return; // no slots registered

	if(!params)params = new KviParameterList;
	KviWindow * wnd = c ? c->window() : g_pApp->activeConsole();

	KviPtrList<KviScriptObjectConnection> dis;
	dis.setAutoDelete(false);

	KviScriptObjectConnectionListIterator it(*l);

	int emitted = 0;

	while(KviScriptObjectConnection * s = it.current())
	{
		KviScriptObjectFunctionHandler * h = s->pTargetObject->lookupFunctionHandler(s->szSlot.ptr(),0);
		if(h)
		{
			if(h->szScriptHandler.hasData() || (h->proc))
			{
				KviStr tmp(KviStr::Format,"builtinEmitSignal(%s,src:%s)",sigName,id());
				KviCommand cmd(tmp.ptr(),wnd,c);
				cmd.setScopeObject(s->pTargetObject);
				emitted ++;
				KviStr old = s->pTargetObject->signalSender();
				s->pTargetObject->setSignalSender(id());
				s->pTargetObject->setSignalName(sigName);
				if(!g_pUserParser->callObjectFunction(h,params,tmp,&cmd))
				{
					if(cmd.hasError())
					{
						g_pUserParser->printError(&cmd);
						if(g_pScriptObjectController->objectExists(cmd.scopeObject()) && it.current())
						{
							wnd->output(KVI_OUT_PARSERWARNING,
								__tr("Broken slot '%s' in target object '%s' while emitting signal '%s' from object '%s': disconnecting"),
								s->szSlot.ptr(),s->pTargetObject->id(),sigName,id());
							dis.append(s);
						} // else destroyed in the call! (already disconnected)
					}
				}
				if(g_pScriptObjectController->objectExists(cmd.scopeObject()))
				{
					cmd.scopeObject()->setSignalSender(old.ptr());
				}
			} // else empty handlers are OK
		} else {
			wnd->output(KVI_OUT_PARSERWARNING,
				__tr("No slot function '%s' exported by target object '%s' while emitting signal '%s' from object '%s': disconnecting"),
				s->szSlot.ptr(),s->pTargetObject->id(),sigName,id());
			dis.append(s);
		}
		++it;
	}

	for(KviScriptObjectConnection * con = dis.first();con;con = dis.next())disconnectSignal(sigName,con);

	delete params;

	if(pBuffer)pBuffer->append(KviStr::Format,"%d",emitted);
}

bool KviScriptObject::functionName(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	buffer.append(name());
	return true;
}

bool KviScriptObject::functionParent(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	KviScriptObject * o = parentObject();
	if(o)buffer.append(o->id());
	else buffer.append('0');
	return true;
}

bool KviScriptObject::functionClassName(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	buffer.append(getClass()->name());
	return true;
}

bool KviScriptObject::functionChildCount(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	buffer.append(KviStr::Format,"%u",m_pChildList->count());
	return true;
}

bool KviScriptObject::functionSignalSender(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	buffer.append(m_szSignalSender);
	return true;
}

bool KviScriptObject::functionSignalName(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	buffer.append(m_szSignalName);
	return true;
}

bool KviScriptObject::functionDestructor(KviCommand *c,KviParameterList *,KviStr &)
{
	ENTER_STACK_FRAME(c,"object::destructor");
	emitSignal("destroyed",0,0,c);
	return c->leaveStackFrame();
}

bool KviScriptObject::functionChildren(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	c->beginListArrayOrDictionaryReturnIdentifier();
	int id=0;
	for(KviScriptObject * o = m_pChildList->first();o;o = m_pChildList->next())
	{
		c->addListArrayOrDictionaryReturnValue(id++,o->id(),buffer);
	}
	return true;
}

bool KviScriptObject::functionFindChild(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	KviStr * pClass = params->safeFirst();
	KviStr * pName = params->safeNext();

	KviScriptObject * o = findChild(pClass->ptr(),pName->ptr());
	if(o)buffer.append(o->id());
	else buffer.append('0');

	return true;
}

bool KviScriptObject::functionEmit(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"object::emit");
	KviStr * pSignal = params->safeFirst();

	KviParameterList * l = new KviParameterList;
	while(KviStr * p = params->next())
	{
		l->append(new KviStr(*p));
	}

	emitSignal(pSignal->ptr(),&buffer,l,c);
	return c->leaveStackFrame();
}

bool KviScriptObject::functionStartTimer(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"object::startTimer");
	bool bOk;
	int timeout = params->getInt(&bOk);
	if(bOk)buffer.append(KviStr::Format,"%d",startTimer(timeout));
	else {
		c->warning(__tr("Invalid timeout"));
		buffer.append("-1");
	}
	return c->leaveStackFrame();
}

bool KviScriptObject::functionKillTimer(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"object::killTimer");
	bool bOk;
	int id = params->getInt(&bOk);
	if(bOk)killTimer(id);
	else c->warning(__tr("Invalid timer id"));
	return c->leaveStackFrame();
}

bool KviScriptObject::functionKillTimers(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	killTimers();
	return true;
}

void KviScriptObject::killAllChildrenWithClass(KviScriptObjectClass *cl)
{
//	debug("object %s : killAllChildrenWithClass %s",id(),cl->name());
	KviPtrList<KviScriptObject> l;
	l.setAutoDelete(true);
	for(KviScriptObject * o=m_pChildList->first();o;o=m_pChildList->next())
	{
		if(o->getClass() == cl)
		{
			l.append(o);
//			debug("Killing child %s",o->id());
		} else o->killAllChildrenWithClass(cl);
	}
//	debug("Ok...all the children appended...destructors..");
}

KviScriptObjectClass * KviScriptObject::getClass(const char * classOverride)
{
	if(!classOverride)return m_pClass;
	KviScriptObjectClass * cl = m_pClass; // class override can be also THIS class
	// if object->$function() is a local override, class::object->$function()
	// is the class member function (not the local override)
	while(cl)
	{
		if(kvi_strEqualCI(cl->name(),classOverride))break;
		else cl = cl->m_pParentClass;
	}
	return cl;
}

KviScriptObjectFunctionHandler * KviScriptObject::lookupFunctionHandler(
		const char * funcName,const char * classOverride)
{
	KviScriptObjectFunctionHandler * h = 0;

	if(!classOverride && m_pFunctionHandlers)
	{
		// lookup the local overrides
		h = m_pFunctionHandlers->find(funcName);
	}

	if(!h)
	{
		// not a local override function... lookup in the class
		KviScriptObjectClass * cl = getClass(classOverride);
		if(cl)return cl->lookupFunctionHandler(funcName);
	}

	return h;
}


bool KviScriptObject::die()
{
	if(m_bInDelayedDeath)return false;
	m_bInDelayedDeath = true;
	QTimer::singleShot(0,this,SLOT(delayedDie()));
	return true;
}

bool KviScriptObject::dieNow()
{
	if(m_bInDelayedDeath)return false;
	m_bInDelayedDeath = true;
	delete this;
	return true;
}

void KviScriptObject::delayedDie()
{
	delete this; // byez!
}

void KviScriptObject::setObject(QObject * o,bool bIsOwned)
{
	__range_invalid(m_pObject);
	m_bObjectOwner = bIsOwned;
	m_pObject = o;
	o->installEventFilter(this);
	connect(m_pObject,SIGNAL(destroyed()),this,SLOT(objectDestroyed()));
}

void KviScriptObject::objectDestroyed()
{
	m_pObject = 0;
	die();
}

bool KviScriptObject::eventFilter(QObject *o,QEvent *e)
{
	return false; // do not stop
}

void KviScriptObject::timerEvent(QTimerEvent *e)
{
	callEventFunction("timerEvent",0,new KviParameterList(
			new KviStr(KviStr::Format,"%d",e->timerId())));
}


bool KviScriptObject::callEventFunction(const char * fncName,KviStr * pBuffer,
	KviParameterList * params,KviWindow * wnd)
{
	if(!params)params = new KviParameterList;

	KviScriptObjectFunctionHandler * h = lookupFunctionHandler(fncName,0);

	if(!h)
	{
		debug("No object event function handler %s",fncName);
		delete params;
		return true;
	}

/*
	Not only gcc spits out compiler errors:
	25.09.2001 , at this point in file

	c:\programmi\microsoft visual studio\myprojects\kvirc3\src\kvirc\uparser\kvi_scriptobject.cpp(1234) : fatal error C1001: INTERNAL COMPILER ERROR
    (compiler file 'E:\8168\vc98\p2\src\P2\main.c', line 494)
    Please choose the Technical Support command on the Visual C++
    Help menu, or open the Technical Support help file for more information
*/

	if(h->szScriptHandler.isEmpty() && (h->proc == ((KviScriptObjectFunctionHandlerProc)0)))
	{
		delete params;
		return true;
	}


	if(!wnd)wnd = g_pApp->activeConsole();

	KviStr tmp(KviStr::Format,"builtinTriggerObjectEvent(%s)",fncName);
	KviCommand c(tmp.ptr(),wnd);
	c.setScopeObject(this);

	if(!g_pUserParser->callObjectFunction(h,params,pBuffer ? *pBuffer : tmp,&c))
	{
		if(c.hasError())
		{
			g_pUserParser->printError(&c);
			delete params;
			return false;
		}
	}
	delete params;
	return true;
}

void KviScriptObject::registerPrivateImplementation(const char * fncName,const char * buffer)
{
	if(buffer)
	{
		if(!m_pFunctionHandlers)
		{
			m_pFunctionHandlers = new QAsciiDict<KviScriptObjectFunctionHandler>(7,false,true);
			m_pFunctionHandlers->setAutoDelete(true);
		}
		KviScriptObjectFunctionHandler * h = new KviScriptObjectFunctionHandler;
		h->proc = 0;
		h->szScriptHandler = buffer ? buffer : "";
		m_pFunctionHandlers->replace(fncName,h);
	} else {
		if(m_pFunctionHandlers)
		{
			m_pFunctionHandlers->remove(fncName);
			if(m_pFunctionHandlers->isEmpty())
			{
				delete m_pFunctionHandlers;
				m_pFunctionHandlers = 0;
			}
		}
	}
}

KviScriptObject * KviScriptObject::findChild(const char * cl,const char * n)
{
	__range_valid(cl && n);
	for(KviScriptObject * o = m_pChildList->first();o;o= m_pChildList->next())
	{
		if(*cl)
		{
			if(kvi_strEqualCI(cl,o->getClass()->name()))
			{
				if(*n)
				{
					// name must match
					if(kvi_strEqualCI(n,o->name()))return o;
				} else return o; //any name matches
			}
		} else {
			// any class matches
			if(*n)
			{
				// name must match
				if(kvi_strEqualCI(n,o->name()))return o;
			} else return o; // any name matches
		}
		KviScriptObject * c = o->findChild(cl,n);
		if(c)return c;
	}
	return 0;
}



#include "kvi_scriptobject.moc"
