//=============================================================================
//
//   File : kvi_up_cmd.cpp
//   Creation date : Mon Jul 03 2000 12:14:22 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2004 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//=============================================================================

#define __KVIRC__

#include "kvi_debug.h"
#include "kvi_uparser.h"
#include "kvi_command.h"
#include "kvi_window.h"
#include "kvi_out.h"
#include "kvi_iconmanager.h"
#include "kvi_options.h"
#include "kvi_locale.h"
#include "kvi_app.h"
#include "kvi_error.h"
#include "kvi_dns.h"
#include "kvi_console.h"
#include "kvi_query.h"
#include "kvi_scriptobject.h"
#include "kvi_fileutils.h"
#include "kvi_ircserverdb.h"
#include "kvi_popupmanager.h"
#include "kvi_event.h"
#include "kvi_frame.h"
#include "kvi_sharedfiles.h"
#include "kvi_kvs_aliasmanager.h"
#include "kvi_filedialog.h"
//#include "kvi_process.h"
#include "kvi_mediatype.h"
#include "kvi_ircurl.h"
#include "kvi_cmdformatter.h"
#include "kvi_timermanager.h"
#include "kvi_time.h"
#include "kvi_irctoolbar.h"
#include "kvi_ircview.h"
#include "kvi_datacontainer.h"
#include "kvi_scriptbutton.h"
#include "kvi_channel.h"
#include "kvi_menubar.h"
#include "kvi_processmanager.h"
#include "kvi_ircconnection.h"
#include "kvi_ircconnectionuserinfo.h"
#include "kvi_ircconnectionasyncwhoisdata.h"
#include "kvi_asynchronousconnectiondata.h"
#include "kvi_kvs_eventmanager.h"
#include "kvi_kvs_eventhandler.h"

#ifdef COMPILE_IPV6_SUPPORT
	#include "kvi_netutils.h" // for kvi_isValidStringIp_V6()
#endif

#ifndef COMPILE_NO_X_BELL
	#include "kvi_xlib.h" // XBell
	#include <unistd.h>   // for usleep();
#endif

#include <stdlib.h> // for srand() now

#include <qcursor.h>
#include <qpixmap.h>
#include <qregexp.h>
#include <qhbox.h>
#include <qprocess.h>

// FIXME: #warning "$frameCaption"
// FIXME: #warning "backward goto"
// FIXME: #warning "IDLE (processEvents())"
// FIXME: #warning "LISTTIMER"

// kvi_app.cpp
extern KviSharedFilesManager   * g_pSharedFilesManager;
extern KviMediaManager         * g_pMediaManager;
extern KviTimerManager         * g_pTimerManager;



/*
	@doc: command_rebinding
	@type:
		language
	@keyterms:
		Rebinding commands to another window
	@title:
		Standard rebinding switch
	@short:
		Standard rebinding switch
	@syntax:
		<command> -r=<window_id> <parameters>
	@body:
		The -r switch is standardized along all the commands. It rebinds a command
		to the windows specified by <window_id>. It is useful to launch commands
		in windows that are not the current one. For example, you might want to
		say something in a specific channel while processing an event bound to
		a console, or say something in all the channels bound to the current irc context.
		The examples below will make everything clear.
	@examples:
		[example]
		[comment]# Run a command in the console of the current IRC context[/comment]
		[cmd]echo[/cmd] -r=$console This command is executed in the console ($window.caption)
		[comment]# Say something to all the channels of the current IRC context[/comment]
		[cmd]foreach[/cmd](%w,[fnc]$window.list[/fnc](channel))[cmd]say[/cmd] -r=%w Hi ppl on [fnc]$chan.name[/fnc]
		[/example]
*/

	/*
		@doc: window_naming_conventions
		@type:
			language
		@title:
			Window naming conventions
		@keyterms:
			IRC context,window ID,frame window,connection ID
		@short:
			KVIrc window structure and the window naming conventions
		@body:
			[big]Introduction[/big][br]
			Starting from the release 3.0.0 KVIrc window structure has
			grown in complexity. Older releases allowed one connetion
			per "frame window" and thus had a dedicated command parser
			for each connection. Finding a window in that scenario
			was quite easy: it was enough to designate it by "name"
			(that was exactly the text displayed in the window caption).
			It was sufficient to have an "unique" name for ever window;
			condition that was granted by the underlying IRC protocol
			and by the KVIrc core design.[br]
			In this version, the unique window names are impossible to be granted.[br]
			[big]Scenario[/big][br]
			The command parser is now "global" to the application.
			There can be two or more consoles in each frame and the user
			is able to join the same channel with two different nicknames
			using two separate connections.
			[ul]
			[li]
			Application (Unique command parser)
				[ul]
				[li]
				Frame X
					[ul]
					[li]
					Console M (IRC context)
						[ul]
						[li]Channel windows[/li]
						[li]Query windows[/li]
						[li]Other connection related windows[/li]
						[/ul]
					[/li]
					[li]
					Console N (IRC context)
						[ul]
						[li]Channel windows[/li]
						[li]Query windows[/li]
						[li]Other connection related windows[/li]
						[/ul]
					[/li]
					[li]
					Other windows
					[/li]
					[li]
					...
					[/li]
					[/ul]
				[/li]
				[li]
				Frame Y
					[ul]
					[li]
					Console O (IRC context)
						[ul]
						[li]Channel windows[/li]
						[li]Query windows[/li]
						[li]Other connection related windows[/li]
						[/ul]
					[/li]
					[li]
					Console P (IRC context)
						[ul]
						[li]Channel windows[/li]
						[li]Query windows[/li]
						[li]Other connection related windows[/li]
						[/ul]
					[/li]
					[li]
					Other windows
					[/li]
					[li]
					...
					[/li]
					[/ul]
				[/li]
				[li]
				...
				[/li]
				[/ul]
			[/li]
			[/ul]
			[br]
			A naming convention has becomed necessary to resolve ambiguities.[br]
			[big]Basic assumptions[/big]
			Every KVIrc window has four main properties:[br]
			-[b]an unique numeric identifier[/b][br]
			-[b]the logical name[/b][br]
			-[b]the type identifier[/b][br]
			-[b]the caption text[/b][br]
			The [b]numeric identifier[/b] is unique to the whole application,
			and is the one returned by the [fnc]$window[/fnc] function.[br]
			The identifier is assigned by KVIrc when the window is created
			and is not changed until the window is destroyed.
			This identifier will be referred as [b]window ID[/b].[br]
			The [b]logical name[/b] is a property of some kind of windows.
			It usually corresponds to the first part of the window caption.
			For example, for channel windows it is the channel name, for
			queries it is the list of the targets. For some other windows 
			the logical name corresponds to the caption text. This will be discussed later.[br]
			The [b]type identifier[/b] describes the properties of a certain window.
			For channel windows the type identifier is "channel" , for query windows is "query" ,
			for console windows it is "console", etc..[br]

			[big]Irc contexts[/big][br]
			The KVIrc frame windows are numbered starting from 0 and named
			"frame_<number>". Each frame can contain an unlimited number of consoles.[br]
			Each console is bound to an [b]IRC context[/b]. (The part "is bound to" could
			be substituted by "defines" or "is contained in").[br]
			[i]An [b]IRC context[/b] is a set of resources that can deal with a single
			IRC connection.[/i][br]
			The association between an [b]IRC context[/b]
			and a console is bijective: each [b]IRC context[/b] is associated
			to a single console window.[br]
			An [b]IRC context[/b] can be in connected or not-connected state.
			When in connected state, it contains a set of windows beside the console:
			mainly channels and query windows.
			The channels and query windows can exist ONLY if the associated
			[b]IRC context[/b] exists.[br]
			Channels and queries have unique names inside a connection so
			there is no way to confuse it. (Theoretically there can
			be more than one query window with the same name, but in fact
			all the windows refer to the same target so they are instances
			of the same resource).
			All this creates a sort of namespace: the channels and queries can be identified
			as "bound" to a specific [b]IRC context[/b].[br]
			An [b]IRC context[/b] can "contain" other windows, such as the "sockets"
			window or the "list" window. KVIrc takes care of making them
			unique inside the [b]IRC context[/b] namespace.[br]
			Each [b]IRC context[/b] has its own unique [b]IRC context ID[/b] (see [fnc]$context[/fnc]).[br]
			Since to a single [b]IRC context[/b] may correspond only a single irc connection,
			when in connected state, the [b]IRC context[/b] may be referred also as [b]connection[/b]
			or [b]connection context[/b], and the associated [b]IRC context Id[/b] can be
			referred as [b]connection ID[/b] or [b]connection context ID[/b].[br]
			There are classes of windows that are not bound to any [b]IRC context[/b]:
			this includes user created windows, DCC windows, browsers etc.[br]
			KVIrc will try to keep that windows with unique logical names.[br]
			[big]How to identify a window[/big][br]
			So what we have until now is:[br]
			[ul]
			[li]Each window has its own unique [b]window ID[/b]: we
			will refer windows always using this identifier.[/li]
			[li]Each window has a set of properties including:
			window type, logical name.[/li]
			[li]Subsets of windows are bound to a single [b]IRC context[/b][/li]
			[/ul]
			The simplest (but also the less significant) method of looking for
			a window is to finding it by caption.[br]
			The [fnc]$window[/fnc] function finds the first KVIrc window matching
			the "caption text" and returns its [b]window ID[/b].[br]
			This method will likely fail when there are more windows with the same
			caption text; for this reason several specific functions
			have been added to allow finding the correct window.[br]
			The [fnc]$console[/fnc] finds a console window bound to a specified
			[b]IRC context[/b].[br]
			The [fnc]$channel[/fnc] finds a channel window matching the specified
			name and bound to a specified [b]IRC context[/b].[br]
			The [fnc]$query[/fnc] finds a query window that has a specified target
			and is bound to a specified [b]IRC context[/b].[br]
	*/


	/*
		@doc: connection_dependant_commands
		@type:
			language
		@title:
			Connection dependant commands
		@keyterms:
			IRC context, connection dependant commands
		@body:
			Many KVIrc commands are connection dependant:
			you need an IRC connection to succesfully execute them;
			usually because some data needs to be sent to the server.
			This includes commands like [cmd]whois[/cmd],[cmd]raw[/cmd],[cmd]query[/cmd],
			[cmd]msg[/cmd],[cmd]notice[/cmd],[cmd]op[/cmd],[cmd]ctcp[/cmd]...[br]
			These commands must be executed in a window that is bound to a
			[b]connected [doc:window_naming_conventions]IRC context[/doc][/b].
			You will obviously get an error message if you try to use them in a window
			that has no associated IRC connection.[br]
			For instance: [cmd]whois[/cmd] will work only if you execute it
			in a console , channel or query window.[br]
			If you want to use these commands in a window that is not associated to
			any IRC context you may use the [doc:command_rebinding]standard -r switch[/doc].
			You can use the same switch to execute a command in an [b]IRC context[/b] that is
			not the current one.
	*/


/*
	@doc: aliases
	@type:
		language
	@keyterms:
		aliases
	@title:
		Aliases
	@short:
		Aliases : user definable command sequences
	@body:
		An alias is an user defined command.  It can be used to rename the builtin kvirc commands or functions,
		to automatize complex tasks or as structured programming mean.
		Aliases can be created or destroyed by using the scriptcenter (graphic interface)
		or from the commandline (or script) by using the [cmd]alias[/cmd] command.
		Once created, an alias remains stored permanently in the KVIrc configuration files
		until it is explicitly deleted.
		A couple of examples will make the things clear.
		join is a really commonly used command. It might be a good idea to rename it to
		simply "j" .. just to type it faster.
		Nothing easier in KVirc: just try this commandline:
		[example]
			[cmd]alias[/cmd](j){ [cmd]join[/cmd] $0-; };
		[/example]

		This will create the alias "j". From this moment you can use /j as it was a normal command.
		[example]
		j #kvirc
		[/example]
		You may have notices the strange $0- function in the alias body: it stands for
		"all parameters passed to the alias". This means that when you call
		[example]
			j #kvirc testpassword
		[/example]
		then both the parameters (#kvirc and testpassword) are passed to the join command.
		The $N functions are special functions that return the positional parameters passed
		to the current script context. In an alias the script context is the script body and
		it is the alias caller that generates the parameters.
		$N (where N is a digit) returns the (N-1)-th positional parameter passed by the caller.
		It returns the parameter numbered N-1 and not N since the parameters are indexed starting
		from zero ($0 is the first parameter!).
		$N-M returns the parameters from (N-1)-th to the (M-1)-th (a parameter range) and $N- returns
		all the parameters from (N-1)-th to the last one. In the example above $0- stands for
		all the parameters starting from the first one.
		[/p]
		[p]
		To remove an alias use again the alias command with an empty body:
		[example]
			[cmd]alias[/cmd](j){}
		[/example]
		This will remove the alias "j" defined above.
		[/p]
		[p]
		A common task in channel management is the kick & ban action.
		You first ban an user from the channel and then eventually kick him
		(obviously assuming that he is actually on the channel).
		This involves using two commands: ban and then kick.
		It could be a nice idea to have a single "kb" command to perform this action.
		Well...easy:
		[example]
			[cmd]alias[/cmd](kb){ [cmd]ban[/cmd] $0; [cmd]kick[/cmd] $0-; };
		[/example]
		This adds the "kb" alias: it can be called as a normal command:
		[example]
			kb spammer You're not welcome here!
		[/example]
		This will first execute "ban spammer" and then "kick spammer You're not welcome here".
		Our kb is a really simple example... it doesn't check for the validity of the parameters:
		the server will warn us if the parameters passed to kb were empty.
		[/p]
		[p]
		The alias can be modified at any time by re-using the alias command.
		Let's make our "kb" a bit more intelligent and add a check for the parameters.
		TIP: It is a good idea to write the following examples in a text file and then use /parse <filename> to execute it.
		[example]
			[cmd]alias[/cmd](kb)
			{
				[cmd]if[/cmd]("$0" == "")
				{
					[cmd]echo[/cmd] "Usage: /kb <nickname> <kick reason>"
					[cmd]return[/cmd]
				}
				[cmd]ban[/cmd] $0
				%reason = $1-
				[cmd]if[/cmd]("%reason" == "")%reason = "You're not welcome here!"
				[cmd]kick[/cmd] $0 %reason
			}
		[/example]
		The example above will first check the validity of the <nickname> passed to kb: 
		if no nickname was passed , it will warn the user and stop.
		The next step will be the "ban <nickname>" call. Another enchancement is the "default reason":
		we first assign the remaining parameters ($1- means "from $1 to the end") to a temporary variable,
		if the variable is empty , a default kick reason is assigned.
		Finally the "kick <nickname> <reason>" will be executed.
		Get used to looking at the single command documentation pages, they will give
		you the hints necessary to fully understand the above piece of code.
		[/p]
		[p]
		Aliases can be used as a mean for structured programming.
		In large scripts you will SURELY have "common tasks" to perform (like having specially
		colored output or calculating a value from a set of other values)...
		Aliases are the way of writing the common tasks: they are equivalent to the "procedures"
		or "functions" in many high-level programming languages.
		The alias as a procedure (subroutine or sub-task) has been shown in the "kb" example above:
		it might be commonly called from complexier scripts or other aliases in case that a
		kick & ban action is needed.
		[/p]
		[p]
		The aliases can be used also as functions.
		Assume that you need really often to calculate the sum of three numbers: a function-alias is the way.
		[example]
			[cmd]alias[/cmd](sum3){ [cmd]return[/cmd] $($0 + $1 + $2); };
		[/example]
		This will add the alias "sum3" and make it available both as a command and a function.
		The "return" command sets the return value of a sequence of commands
		(an alias is a sequence of commands...remember ?) and terminates the execution (by returning
		the control to the caller).
		So return $($0 + $1 + $2); will set the return value of the alias to the value
		computed by $($0 + $1 + $2) that actually is the sum of the first three parameters passed.
		You will then use it in the following way:
		[example]
			...
			%myfirstsum = $sum3(%somevalue,%someothervalue,4)
			%anothersum = $sum3(12,%somevalue,%anothervalue)
			...
		[/example]
		Ops.. I've used some variables without actually explaining them... hehe.. please forgive me and read on.
		This example is again really simple , but you might have complexier function-aliases.
		The function-aliases are also normal aliases.... you can use it as a command:
		[example]
			/sum3 1 2 3
		[/example]
		Is a perfectly valid call.... it's just that it will have no visible results
		(just because a command call implies ignoring the return value.
		In fact there is no difference al all between function-aliases and normal-aliases: 
		the caller makes the difference: by calling an alias as a command the return value 
		just disappears in hyperspace, by calling an alias as a function , the return value
		is propagated (and in fact "used").
		(There are some "nice" exceptions to this rule...but you don't need to care about it, for now).
		If return is not called inside an alias body , the return value will be just a null value.
		[/p]
		[p]
		Aliases can accept switches just like any other command. The [fnc]$sw[/fnc] is there
		exactly for that purpose. Check it out.
		[/p]
*/

/*
	@doc: datatypes
	@type:
		language
	@keyterms:
		variable,variables,array,arrays,dictionary,dictionaries,percent sign,global variables,local variables,
		global variable,local variable,variable creation and destruction,variable evaluation,dictionary keys,
		associative arrays,hash,hashes,scalar,scalars,command scope,data types,extended scope,array reference,
		dictionary reference
	@title:
		Data types
	@short:
		KVIrc built-in data types
	@body:
		KVirc has three basic built-in data types: simple variables, arrays of
		simple variables and associative arrays of variables.[br]
		There is a fourth builtin data type: object
		that allows creation of complex structures and object-oriented programming;
		the object data type is described in another document.[br]

		[big]Variables[/big]
		The variables can be either global to the application or local
		to the command scope. The content of a variable is basically a string:
		any sequence of ASCII characters. In some contests the string may be interpreted
		as a number. All the variable names start with a percent '%' sign.
		
		[big]Global variables[/big][br]
		A global variable name is formed by a "percent" sign (%),
		followed by an uppercase letter from A to Z, followed
		by a sequence of characters in range ('a' to 'z','0' to '9','.','_').[br]
		"%INDEX","%My_nickname","%Foo","%Bar1" and "%Foo.BAR" are examples of valid
		global variable names.[br]
		A global variable is [b]global to the entire application[/b], not
		to the current frame or irc context; be sure to remember this.[br]
		You can type[br]
		[example]
			/%Hello = "Hello world!"
		[/example]
		in the commandline of any KVIrc window and then execute[br]
		[example]
			echo %Hello
		[/example]
		in the commandline of any other KVIrc window. You will see "Hello world!"
		printed in the second window view.

		[big]Local variables[/big][br]
		A local variable name is formed by a "percent" sign (%),
		followed by an lowercase letter from a to z, followed
		by a sequence of characters in range ('a' to 'z','0' to '9','.','_').[br]
		"%index","%my_nickname","%foo","%bAR1" and "%foo.BAR" are examples of valid
		local variable names.[br]
		A local variable exists only in the current command scope.
		The exact command scope definition is rather tricky and depends
		on the internal KVIrc implementation. Just be aware that:[br]
		- An alias body is a command scope.[br]
		- An event body is a command scope.[br]
		- Any sequence of commands executed [b]at once[/b] in a window commandline
		is a command scope.[br]
		You will notice that finding out the current command scope is rather intuitive.[br]
		When you type[br]
		[example]
			%text = "very important text";
		[/example]
		in the commandline of any KVIrc window and then try to execute[br]
		[example]
			echo %text
		[/example]
		you will see nothing printed. The variable %text was local to the "command scope"
		and disappeared immediately after the execution of the assignment.[br]
		But if you execute[br]
		[example]
			%text = "hello"; echo %text wold!
		[/example]
		you will see  "hello world!" printed in the current window.[br]

		[big]Extended scope variables[/big]
		Variables that start with a ':' character are "extended scope" variables.
		"%:index" , "%:Hello" , "%:something.else" are all valid special scope variable names.[br]
		They're actually used in popups and in timers (but later I might find other usages as well :).[br]
		"Extended scope" means that these variables are somewhere in the middle between
		global and local variables. They normally act as local , but in some cases their [b]lifetime[/b] and [b]visibility[/b]
		may be extended.[br]
		For example , in the popups , all the special scope variables
		are visible during all the "lifetime" of a popup (so from the prologue code call to
		the moment when the user selects an item and the corresponding code is executed).[br]
		This allows you to pre-calculate some data or conditions in the popup prologue
		and use this data in the popup item conditions and item handlers.[br]

		[big]Variable creation and destruction[/big]
		You don't need to declare variables.
		A variable starts to exist at the time that you assing
		something to it.[br]
		[example]
			%MyVar = some funky text
			%X = 2445833
			%SevenSpaces = "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;"
		[/example]
		It terminates its existence at the time that you assingn it an empty string:
		[example]
			%MyVar = ""
			%X = %MyVar
			%SevenSpaces =
		[/example]
		The example is based on global variables, but it is valid for local ones as well.[br]
		A non existing variable is equivalent to an empty one: you will
		never get a warning about non existing variables. This convention
		allows KVIrc to automatically manage the variable creation and destruction.[br]

		[big]Variable evaluation[/big]
		A variable can appear in every place where a parameter
		is expected: so after the command name, after a switch or inside
		an identifier parameters.
		KVirc will try to extract the longest possible variable name after a literal percent '%'
		sign everywhere in the parameter string. So the command sequence[br]
		[example]
			%number = 1st; echo this is my %number variable test
		[/example]
		will first assign "1st" to the variable "%number" and then execute
		"echo this is my 1st variable test".
		The following example will NOT work as expected.[br]
		[example]
			%number = 1; echo this is my %numberst variable test
		[/example]
		KVirc will assign "1" to %number in this case but the next variable
		name extracted will be "%numberst" that is actually empty; so finally
		"echo this is my variable test" will be executed.
		To avoid this problem you can use the backslash escape character:[br]
		[example]
			%number = 1; echo this is my %number\st variable test
		[/example]

		[big]Arrays[/big]
		Arrays are collections of items indexed by numbers.[br]
		The general syntax for an array is:[br]
		[b]%<name>[<index>][/b][br]
		<name> is the name of the array and follows the rules
		valid for the simple variables: the names starting with an uppercase letter
		designate a global array, the others designate local ones.[br]
		Extended scope arrays can be created as well: normally they act as local,
		but may have extended lifetime in some scopes.[br]
		The arrays have its own namespace thus %Array[] has nothing to do with %Array
		(that is a simple variable).[br]
		<index> must be a subscript that evaluates to a positive integer and it selects an item in the array.
		The first index of the array is 0 and the last is equal to size-1.[br]
		You can obtain the size of the array by evaluating %<name>[]#.[br]
		You don't need to declare the size of the array: it is automatically
		handled. When you assign a non-empty string to an item, the array
		is automatically enlarged to contain the index that you are assigning to.
		If the array was not existing before the assignment, it is created.[br]
		If the first assignment index is greater than 0 the items below that index
		will be empty: will just behave as empty/unset variables.[br]
		[example]
			[comment]# Create an array with 21 elements[/comment]
			%Names[20]=Pragma
		[/example]
		To remove an item from the array you assign it an empty string.[br]
		When you remove the highest indexed item the array is automatically shrunk to
		the next highest non-empty item. If there are no other non-empty items the array is destroyed.[br]
		Please note that the memory occupation of the array depends on the <index>.[br]
		By assigning a value to the 10000'th item (index 9999) you allocate 10000 entries! (in fact
		at least ((10000 * 4) + value size) bytes).[br]
		An array is destroyed when it contains no more items or when it goes out of scope (a local array obviously).[br]
		[example]
			[comment]# Creating an array with 800 entries[/comment]
			%Array[799]=test
			echo %Array[]#
			%Array[299]=test
			echo %Array[]#
			%Array[799]=
			echo %Array[]#
			[comment]# Now it contains 300 elements[/comment]
		[/example]
		[br]
		[big]Array references[/big][br]
		When <index> is an empty string then the subscript %<name>[] is called "array reference" (maybe a bit unproperly :).
		You may think that notation as referencing the "whole array".[br]
		The following example shows how the array assignment.[br]
		[example]
			%A[100]=x
			%A[200]=y
			%B[]=%A[]
			echo %B[200]
		[/example]
		When you pass an array reference to a command or function, it is evaluated
		as a comma separated list of entries.[br]
		[example]
			%Array[0]=Pippo
			%Array[1]=Pluto
			%Array[2]=Paperino
			echo %Array[]
		[/example]
		By assigning a string to an array you assign it to all the array entries:[br]
		[example]
			%Array[0]=Pippo
			%Array[1]=Pluto
			%Array[2]=Paperino
			echo %Array[]
			%Array[]=undefined
			echo %Array[]
		[/example]
		This is useful when you want to unset an array: just assign an empty string to all its entries:[br]
		[example]
			%Array[200]=Test
			echo %Array[]#
			%Array[]=
			echo %Array[]#
		[/example]
		You can loop through all the array items by using the [cmd]foreach[/cmd] command:[br]
		[example]
			%Array[0]=Pippo
			%Array[1]=Pluto
			%Array[2]=Paperino
			[cmd]foreach[/cmd](%item,%Array[])[cmd]echo[/cmd] %item
		[/example]
		Obviously also the traditional [cmd]for[/cmd] and [cmd]while[/cmd] indexed-looping methods
		are available:[br]
		[example]
			%Array[0]=Pippo
			%Array[1]=Never show this
			%Array[2]=Pluto
			%Array[5]=Hidden again
			%Array[8]=Paperino
			for(%i=0;%i < %Array[]#;%i+=2)echo Entry %i: \"%Array[%i]\";
		[/example]
		[br]
		[big]Dictionaries[/big]
		Dictionaries are associative arrays of strings. They look close
		to the perl hashes. The general syntax for a dictionary name is:[br]
		[b]%<name>{<key>}[/b][br]
		<name> is the name of the dictionary and follows the same rule
		as for the variables: the names starting with an uppercase letter
		designate a global dictionary, the others designate local ones.[br]
		Again , the dictionaries have its own namespace: you can safely use
		%Names , %Names[] and %Names{} as different entities in your script.[br]
		Extended scope dictionaries can be created as well: normally they act as local,
		but may have extended lifetime in some scopes.[br]
		A dictionary associates a "data string" to each "key string" used.
		The key can be [b]any string[/b] not containing an "unescaped" '}' character
		and is [b]case insensitive[/b]: "key" is equivalent to "KEY" or "KeY".[br]
		[example]
			%Ages{Pragma} = 24
		[/example]
		This assignment associates the key "Pragma" to the number "24" in
		the global dictionary "%Ages". If the array was not existing yet,
		it is created first. If the key "Pragma" was already existing,
		the value associated is replaced with the new value.[br]
		To remove an association you simply assign the empty string to it:[br]
		[example]
			%Ages{pragma} =
		[/example]
		The dictionary is automatically destroyed when there are no more associations
		in it (eg. when you assign the empty string to the last key).[br]
		Dictionaries can be used easily to simulate arrays:[br]
		[example]
			%Links{21} = "http://www.kvirc.net"
		[/example]
		Even multidimensional ones:[br]
		[example]
			%Pixels{324.312} = 1
		[/example]
		Remember that in the example above the key "324.312" is a single string.[br]
		Obviously the key can contain variables and functions just as any other parameter.[br]
		An empty key performs operations on the whole dictionary (just like for the arrays):[br]
		You can assign dictionaries:[br]
		[example]
			%Test2{}=%Test1{}
		[/example]
		Assign a scalar to all the items
		[example]
			%Data{} = "unspecified"
		[/example]
		The example above associates the string "unspecified" to all the existing
		keys in the dictionary %Data. By using this method you can easily destroy
		a dictionary: simply assign the empty string to it.[br]
		[example]
			%Data{} =
		[/example]
		Other [doc]operators[/doc] have similar semantic when working on "empty keys".[br]

		[big]Dictionary evaluation[/big]
		A dictionary can appear in every place where a variable can appear.[br]
		[example]
			[cmd]echo[/cmd] My age is %Ages{[fnc]$mynick[/fnc]}
		[/example]
		If you pass an empty key, the dictionary evaluates to the comma separated
		list of values stored in it. The values have no defined order.
		The list is interpreted as single "string" in contexts where a "string" is required,
		and as a list of "strings" in context where lists of strings are expected. (<-- hehe :)[br]
		The special syntax %<name>{}# returns the number of keys in the
		dictionary.[br]
		The special syntax %<name>{}@ returns a comma
		separated list of keys in the dictionary; the keys have no
		defined order (well, you may be only sure that the order
		of the keys is exactly equal to the values order (%name{})).[br]
		[example]
			%Songs{Jimi Hendrix} = Voodo child
			%Songs{Shawn Lane} = Gray piano's flying
			%Songs{Mina} = Brava
			%Songs{Greg Howe} = "Full Throttle"
			# This is a "single string" evaluation context
			[cmd]echo[/cmd] %Songs{}
			# This is a "list of strings" evaluation context
			[cmd]foreach[/cmd](%var,%Songs{})[cmd]echo[/cmd] %var
		[/example]
*/


// kvi_app.cpp
extern KviIrcServerDataBase * g_pIrcServerDataBase;

#define cmdRegSpec(_keyIdx,_proc1,_proc2) \
	p = new KviCommandParseProc; \
	p->proc = KVI_PTR2MEMBER(KviUserParser::_proc1); \
	m_pCommandExecDict->insert(cmdNamesTable[_keyIdx],p); \
	p = new KviCommandParseProc; \
	p->proc = KVI_PTR2MEMBER(KviUserParser::_proc2); \
	m_pCommandSkipDict->insert(cmdNamesTable[_keyIdx],p)

#define cmdRegNorm(_keyIdx,_proc) cmdRegSpec(_keyIdx,_proc,skipNormalCommand)


// FIXME: #warning "KILL,REHASH,DIE,PING,PONG,RESTART,SUMMON,WALLOPS,USERHOST,ISON"
// FIXME: #warning "An echo that outputs to the lower channel view.."
// FIXME: #warning "PRIVMSG as alias to MSG"

// FIXME: #warning "SWITCH"

static char * cmdNamesTable[]=
{
	"ECHO"        , "HOST"        , "QUERY"       , "RAW"         , "QUOTE"       , //<--4
	"NICK"        , "PRIVMSG"     , "MSG"         , "NOTICE"      , "JOIN"        , //<--9
	"PART"        , "LEAVE"       , "TOPIC"       , "OP"          , "DEOP"        , //<--14
	"VOICE"       , "DEVOICE"     , "MODE"        , "DELETE"      , "DESTROY"     , //<--19
	"PARSE"       , "CLASS"       , "PRIVATEIMPL" , "SETRETURN"   , "IF"          , //<--24
	"SRAND"       , "SERVER"      , "CONNECT"     , "DISCONNECT"  , "POPUP"       , //<--29
	"DEFPOPUP"    , "EVENT"       , "EVENTCTL"    , "BAN"         , "UNBAN"       , //<--34
	"ME"          , "TIMER"       , "CTCP"        , "WHOIS"       , "WHOWAS"      , //<--39
	"AVATAR"      , "AWHOIS"      , "OPTION"      , "ALIAS"       , "HALT"        , //<--44
	"KICK"        , "WHO"         , "OPER"        , "SERVICE"     , "SQUIT"       , //<--49
	"LIST"        , "INVITE"      , "MOTD"        , "LUSERS"      , "VERSION"     , //<--54
	"STATS"       , "LINKS"       , "TIME"        , "TRACE"       , "ADMIN"       , //<--59
	"INFO"        , "SQUERY"      , "AWAY"        , "BACK"        , "RUN"         , //<--64
	"PLAY"        , "QUIT"        , "OPENURL"     , "HELP"        , "KILLTIMER"   , //<--69
	"WHILE"       , "BREAK"       , "RAISE"       , "DELPOPUPITEM", "TRLOAD"      , //<--74
	"SWITCH"      , "BUTTON"      , "CLEAR"       , "FOREACH"     , "BUTTONCTL"   , //<--79
	"SOUND"       , "PASS"        , "SETMENU"     , "BEEP"        , "RETURN"      , //<--84
	"CLEAROBJECTS", "FOR"         , "EVAL"        , "SAY"         , "EXEC"        , //<--89
	"ECHOPRIVMSG" , "AHOST"       , "REBIND"      , "CODEPAGE"    , "TRUNLOAD"      //<--94
};


void KviUserParser::initCommandDict()
{
	// case sensitive (faster) and do NOT copy keys
	m_pCommandExecDict = new QAsciiDict<KviCommandParseProc>(109,true,false);
	m_pCommandExecDict->setAutoDelete(true);
	m_pCommandSkipDict = new QAsciiDict<KviCommandParseProc>(109,true,false);
	m_pCommandSkipDict->setAutoDelete(true);

	KviCommandParseProc * p;

	cmdRegNorm(0,parseCmd_ECHO);
	cmdRegNorm(1,parseCmd_HOST);
	cmdRegNorm(2,parseCmd_QUERY);
	cmdRegNorm(3,parseCmd_RAW);
	cmdRegNorm(4,parseCmd_RAW);
	cmdRegNorm(5,parseCmd_NICK);
	cmdRegNorm(6,parseCmd_PRIVMSG);
	cmdRegNorm(7,parseCmd_PRIVMSG);
	cmdRegNorm(8,parseCmd_NOTICE);
	cmdRegNorm(9,parseCmd_JOIN);

	cmdRegNorm(10,parseCmd_PART);
	cmdRegNorm(11,parseCmd_PART);
	cmdRegNorm(12,parseCmd_TOPIC);
	cmdRegNorm(13,parseCmd_OP);
	cmdRegNorm(14,parseCmd_DEOP);
	cmdRegNorm(15,parseCmd_VOICE);
	cmdRegNorm(16,parseCmd_DEVOICE);
	cmdRegNorm(17,parseCmd_MODE);
	cmdRegNorm(18,parseCmd_DELETE);
	cmdRegNorm(19,parseCmd_DELETE);

	cmdRegNorm(20,parseCmd_PARSE);
	cmdRegSpec(21,parseCmd_CLASS,skipClass);
	cmdRegSpec(22,parseCmd_PRIVATEIMPL,skipCallbackCommand);
	cmdRegNorm(23,parseCmd_SETRETURN);
	cmdRegSpec(24,parseCmd_IF,skipIf);
	cmdRegNorm(25,parseCmd_SRAND);
	cmdRegNorm(26,parseCmd_SERVER);
	cmdRegNorm(27,parseCmd_CONNECT);
	cmdRegNorm(28,parseCmd_DISCONNECT);
	cmdRegNorm(29,parseCmd_POPUP);

	cmdRegSpec(30,parseCmd_DEFPOPUP,skipDefpopup);
	cmdRegSpec(31,parseCmd_EVENT,skipCallbackCommand);
	cmdRegNorm(32,parseCmd_EVENTCTL);
	cmdRegNorm(33,parseCmd_BAN);
	cmdRegNorm(34,parseCmd_UNBAN);
	cmdRegNorm(35,parseCmd_ME);
	cmdRegSpec(36,parseCmd_TIMER,skipCallbackCommand);
	cmdRegNorm(37,parseCmd_CTCP);
	cmdRegNorm(38,parseCmd_WHOIS);
	cmdRegNorm(39,parseCmd_WHOWAS);

	cmdRegNorm(40,parseCmd_AVATAR);
	cmdRegSpec(41,parseCmd_AWHOIS,skipCallbackCommand);
	cmdRegNorm(42,parseCmd_OPTION);
	cmdRegSpec(43,parseCmd_ALIAS,skipCallbackCommand);
	cmdRegNorm(44,parseCmd_HALT);

	cmdRegNorm(45,parseCmd_KICK);
	cmdRegNorm(46,parseCmd_RFC2812WRAPPER); // who
	cmdRegNorm(47,parseCmd_RFC2812WRAPPER); // oper
	cmdRegNorm(48,parseCmd_RFC2812WRAPPER); // service
	cmdRegNorm(49,parseCmd_RFC2812WRAPPER); // squit

	cmdRegNorm(50,parseCmd_RFC2812WRAPPER); // list
	cmdRegNorm(51,parseCmd_RFC2812WRAPPER); // invite
	cmdRegNorm(52,parseCmd_RFC2812WRAPPER); // motd
	cmdRegNorm(53,parseCmd_RFC2812WRAPPER); // lusers
	cmdRegNorm(54,parseCmd_RFC2812WRAPPER); // version

	cmdRegNorm(55,parseCmd_RFC2812WRAPPER); // stats
	cmdRegNorm(56,parseCmd_RFC2812WRAPPER); // sockets
	cmdRegNorm(57,parseCmd_RFC2812WRAPPER); // time
	cmdRegNorm(58,parseCmd_RFC2812WRAPPER); // trace
	cmdRegNorm(59,parseCmd_RFC2812WRAPPER); // admin

	cmdRegNorm(60,parseCmd_RFC2812WRAPPER); // info
	cmdRegNorm(61,parseCmd_RFC2812WRAPPER); // squery
	cmdRegNorm(62,parseCmd_AWAY);
	cmdRegNorm(63,parseCmd_BACK);
	cmdRegNorm(64,parseCmd_RUN);

	cmdRegNorm(65,parseCmd_PLAY);
	cmdRegNorm(66,parseCmd_QUIT);
	cmdRegNorm(67,parseCmd_OPENURL);
	cmdRegNorm(68,parseCmd_HELP);
	cmdRegNorm(69,parseCmd_KILLTIMER);

	cmdRegSpec(70,parseCmd_WHILE,skipIterationCommand);
	cmdRegNorm(71,parseCmd_BREAK);
	cmdRegNorm(72,parseCmd_RAISE);
	cmdRegNorm(73,parseCmd_DELPOPUPITEM);
	cmdRegNorm(74,parseCmd_TRLOAD);

	cmdRegSpec(75,parseCmd_SWITCH,skipSwitch);
	cmdRegSpec(76,parseCmd_BUTTON,skipCallbackCommand);
	cmdRegNorm(77,parseCmd_CLEAR);
	cmdRegSpec(78,parseCmd_FOREACH,skipIterationCommand);
	cmdRegNorm(79,parseCmd_BUTTONCTL);

	cmdRegNorm(80,parseCmd_SOUND);
	cmdRegNorm(81,parseCmd_RFC2812WRAPPER); // pass
	cmdRegNorm(82,parseCmd_SETMENU);
	cmdRegNorm(83,parseCmd_BEEP);
	cmdRegNorm(84,parseCmd_RETURN);

	cmdRegNorm(85,parseCmd_CLEAROBJECTS);
	cmdRegSpec(86,parseCmd_FOR,skipFor);
	cmdRegNorm(87,parseCmd_EVAL);
	cmdRegNorm(88,parseCmd_SAY);
	
	cmdRegSpec(89,parseCmd_EXEC,skipCallbackCommand);
	cmdRegNorm(90,parseCmd_ECHOPRIVMSG);
	cmdRegSpec(91,parseCmd_AHOST,skipCallbackCommand);
	cmdRegNorm(92,parseCmd_REBIND);
	cmdRegNorm(93,parseCmd_RFC2812WRAPPER); // codepage
	cmdRegNorm(94,parseCmd_TRUNLOAD);

	// FIXME: #warning "SPLITQUERY,JOINQUERY (?)"
}

// FIXME: #warning "DO{} while()"

// FIXME: #warning "SET and UNSET, that can be REBOUND TO ANOTHER WINDOW"
// FIXME: #warning "<module>.lock <module>.unlock <module>.version <module>.info"

bool KviUserParser::parseCmd_TRLOAD(KviCommand *c)
{
	/*
		@doc: trload
		@type:
			command
		@title:
			trload
		@syntax:
			trload [-v] <catalogue> [<catalogue_dir>]
		@short:
			Loads a translation catalogue
		@description:
			Loads the specified translation <catalogue> in the current user language.[br]
			The translation catalogues are *.mo files compiled with the standard
			msgfmt program from the gettext package.[br]
			The syntax of the catalogue names is:[br]
			[b]<catalogue_dir>/<catalogue>_<language>.mo[/b][br]
			KVIrc looks for the catalogues in <catalogue_dir> (if specified)
			and in the locale subdirectory of the global and local KVIrc directories.[br]
			<language> is choose automatically by KVIrc depending on the user's locale.[br]
			Catalogues MUST be in UTF8 encoding.[br]
			After a translation catalogue has been loaded you can
			access its strings by the means of [fnc]$tr[/fnc].[br]
			This function doesn't print any warning unless -v is specified.[br]
			[b]NOTE[/b][br]
			This feature is currently work in progress: the scripting engine
			is not yet fully unicode and thus some more exotic translations may still suffer...
		@seealso:
			[fnc]$tr[/fnc], [cmd]trunload[/cmd]
	*/

	ENTER_STACK_FRAME(c,"trload");

	KviStr cat,dir;
	if(!parseCmdSingleToken(c,cat))return false;
	if(!parseCmdFinalPart(c,dir))return false;

	if(!dir.isEmpty())
	{
		if(KviLocale::loadCatalogue(cat.ptr(),dir.ptr()))
			return c->leaveStackFrame();
	}

	g_pApp->getGlobalKvircDirectory(dir,KviApp::Locale);

	if(KviLocale::loadCatalogue(cat.ptr(),dir.ptr()))
		return c->leaveStackFrame();

	g_pApp->getLocalKvircDirectory(dir,KviApp::Locale);

	if(KviLocale::loadCatalogue(cat.ptr(),dir.ptr()))
		return c->leaveStackFrame();

	if(c->hasSwitch('v'))
		c->warning(__tr2qs("No catalogue %s for the current language found"),cat.ptr());

	return c->leaveStackFrame();
}


bool KviUserParser::parseCmd_TRUNLOAD(KviCommand *c)
{
	/*
		@doc: trunload
		@type:
			command
		@title:
			trunload
		@syntax:
			trunload [-v] <catalogue>
		@short:
			Unloads a translation catalogue
		@description:
			Unloads a specified translation catalogue.[br]
			When you no longer need the translation strings contained in the
			catalogue just unload it to free memory.[br]
			This command prints no warnings unload -v is used.[br]
		@seealso:
			[fnc]$tr[/fnc], [cmd]trload[/cmd]
	*/

	ENTER_STACK_FRAME(c,"trunload");

	KviStr cat;
	if(!parseCmdFinalPart(c,cat))return false;

	if(!KviLocale::unloadCatalogue(cat.ptr()))
	{
		if(c->hasSwitch('v'))
			c->warning(__tr2qs("The catalogue %s was not loaded"),cat.ptr());
	}

	return c->leaveStackFrame();
}


bool KviUserParser::parseCmd_HOST(KviCommand *c)
{
	/*
		@doc: host
		@type:
			command
		@title:
			host
		@syntax:
			host [-i] [-a] <hostname>
		@short:
			DNS lookup
		@description:
			Starts a DNS lookup for the <hostname> and prints
			the results in the current window.[br]
			The -i switch causes the command to execute
			in IpV6 mode (and lookup ONLY IpV6 hosts!).[br]
			The -a switch causes the command to run in "unspecified" mode
			and return any available address: IpV4 or Ipv6.[br]
			This command also performs reverse lookups (if you pass an IP address as <hostname>).[br]
		@examples:
			[example]
			host localhost
			host irc.flashnet.it
			host -i irc.flashnet.it
			host -i A0E0:FFFF:::1010:1111:E0E0:A0A0
			host -a 127.0.0.1
			host 255.41.31.23
			host -a irc.stealth.net
			[/example]
		@seealso:
			[cmd]ahost[/cmd]
	*/

	ENTER_STACK_FRAME(c,"host");

	KviStr buffer;
	if(!parseCmdFinalPart(c,buffer))return false;

	KviDns * d = new KviDns();
	d->setAuxData((void *)(c->window()));
	connect(d,SIGNAL(lookupDone(KviDns *)),this,SLOT(parseCmd_HOST_dnsSlot(KviDns *)));
	KviDns::QueryType queryType = KviDns::IpV4;
	if(c->hasSwitch('i'))queryType = KviDns::IpV6;
	if(c->hasSwitch('a'))queryType = KviDns::Any;
	if(!d->lookup(buffer.ptr(),queryType))parseCmd_HOST_dnsSlot(d);

	return c->leaveStackFrame();
}

void KviUserParser::parseCmd_HOST_dnsSlot(KviDns * dns)
{
	KviWindow * pWnd = (KviWindow *)dns->releaseAuxData();
	if(!g_pApp->windowExists(pWnd))pWnd = g_pActiveWindow;
	pWnd->output(KVI_OUT_HOSTLOOKUP,__tr2qs("DNS Lookup result for query \"%s\""),dns->query());

	if(dns->state() == KviDns::Failure)
	{
		QString strDescription(KviError::getDescription(dns->error()));
		pWnd->output(KVI_OUT_HOSTLOOKUP,__tr2qs("Error: %Q"),&strDescription);
//		pWnd->output(KVI_OUT_HOSTLOOKUP,__tr2qs("Error: %Q"),&(KviError::getDescription(dns->error())));
	} else {
		int idx = 1;
		for(KviStr * h = dns->hostnameList()->first();h;h = dns->hostnameList()->next())
		{
			pWnd->output(KVI_OUT_HOSTLOOKUP,__tr2qs("Hostname %d: %s"),idx,h->ptr());
			idx++;
		}
		idx = 1;
		for(KviStr * a = dns->ipAddressList()->first();a;a = dns->ipAddressList()->next())
		{
			pWnd->output(KVI_OUT_HOSTLOOKUP,__tr2qs("IP address %d: %s"),idx,a->ptr());
			idx++;
		}
	}
	delete dns;
}

typedef struct _KviAHOSTCommandData
{
	KviWindow * pWindow;
	KviStr      szMagic;
	KviStr      szQuery;
	KviStr      szCode;
} KviAHOSTCommandData;

bool KviUserParser::parseCmd_AHOST(KviCommand *c)
{
	/*
		@doc: ahost
		@type:
			command
		@title:
			host
		@syntax:
			ahost [-i] [-a] (<dnsquery>[,<magicdata>]){ <callback command> }
		@short:
			DNS lookup
		@description:
			Starts a DNS lookup for the <dnsquery> and reports the
			results by calling the callback routine.
			The -i switch causes the command to execute
			in IpV6 mode (and lookup ONLY IpV6 hosts!).[br]
			The -a switch causes the command to run in "unspecified" mode
			and return any available address: IpV4 or Ipv6.[br]
			This command also performs reverse lookups (if you pass an IP address as <hostname>).[br]
			The callback command gets passed five parameters:[br]
			$0 contains the query string (<dnsquery> in fact)[br]
			$1 contains the value 1 if the query was succesfull.[br]
			In that case the remaining parameters are set as follows:[br]
			$2 contains the first ip address associated to the <dnsquery>[br]
			$3 contains the hostname associated to the <dnsquery>[br]
			$4 contains the eventual <magicdata> passed.[br]
			If $1 contains the value 0 then the query has failed and[br]
			$2 contains an error message explaining the failure.[br]
			$3 is empty[br]
			$4 contains the eventual <magicdata> passed.[br]
			Please note that if the dns query fails to even start for some
			reason then your callback MAY be called even before ahost() returns.[br]
		@switches:
			!sw: -i
			Causes the command to execute in IpV6 mode (and lookup ONLY IpV6 hosts!).
			!sw: -a
			The -a switch causes the command to run in "unspecified" mode
			and return any available address: IpV4 or Ipv6.
		@examples:
			[example]
			ahost("localhost")
			{
				[cmd]echo[/cmd] "Lookup: "$0;
				if($1)
				{
					[cmd]echo[/cmd] "Ip address: "$2;
					[cmd]echo[/cmd] "Hostname: "$3;
				} else {
					[cmd]echo[/cmd] "Error: $2"; 
				}
			}
			ahost -i ("irc.flashnet.it","Hello :)")
			{
				[cmd]echo[/cmd] "Lookup: "$0;
				[cmd]echo[/cmd] "Magic: $3";
				if($1)
				{
					[cmd]echo[/cmd] "Ip address: "$2;
					[cmd]echo[/cmd] "Hostname: "$3;
				} else {
					[cmd]echo[/cmd] "Error: $2"; 
				}
			}
			ahost -a ("cafe:babe::dead:beef")
			{
				[cmd]echo[/cmd] "Lookup: "$0;
				[cmd]echo[/cmd] "Magic: $3";
				if($1)
				{
					[cmd]echo[/cmd] "Ip address: "$2;
					[cmd]echo[/cmd] "Hostname: "$3;
				} else {
					[cmd]echo[/cmd] "Error: $2"; 
				}
			}
			[/example]
		@seealso:
			[cmd]host[/cmd]
	*/

	ENTER_STACK_FRAME(c,"ahost");

	KviParameterList paramList;
	paramList.setAutoDelete(true);

	KviStr cmd;

	if(!parseCallbackCommand(c,&paramList,&cmd))return false;

	KviStr * query = paramList.safeFirst();
	KviStr * magic = paramList.safeNext();

	KviAHOSTCommandData * dat = new KviAHOSTCommandData;
	dat->szQuery = *query;
	dat->szMagic = *magic;
	dat->pWindow = c->window();
	dat->szCode = cmd;

	KviDns * d = new KviDns();
	d->setAuxData((void *)(dat));

	connect(d,SIGNAL(lookupDone(KviDns *)),this,SLOT(parseCmd_AHOST_dnsSlot(KviDns *)));
	KviDns::QueryType queryType = KviDns::IpV4;
	if(c->hasSwitch('i'))queryType = KviDns::IpV6;
	if(c->hasSwitch('a'))queryType = KviDns::Any;
	if(!d->lookup(query->ptr(),queryType))parseCmd_AHOST_dnsSlot(d);

	return c->leaveStackFrame();
}

void KviUserParser::parseCmd_AHOST_dnsSlot(KviDns * dns)
{
	KviAHOSTCommandData * dat = (KviAHOSTCommandData *)dns->releaseAuxData();
	if(!g_pApp->windowExists(dat->pWindow))dat->pWindow = g_pActiveWindow;
	
	KviCommand cmd(dat->szCode.ptr(),dat->pWindow,0,0);

	if(dns->state() == KviDns::Failure)
	{
		cmd.setParams(new KviParameterList(
			new KviStr(dat->szQuery),
			new KviStr("0"),
			new KviStr(KviError::getDescription(dns->error())),
			new KviStr(),
			new KviStr(dat->szMagic)));
	} else {
		KviStr * fh = dns->hostnameList()->first();
		KviStr * fi = dns->ipAddressList()->first();
		cmd.setParams(new KviParameterList(
			new KviStr(dat->szQuery),
			new KviStr("1"),
			new KviStr(fi ? fi->ptr() : "?.?.?.?"),
			new KviStr(fh ? fh->ptr() : "?.?"),
			new KviStr(dat->szMagic)));
	}

	delete dns;

	bool bResult = parseCommand(&cmd);

	if(!bResult)
	{
		if(cmd.hasError())
		{
			printError(&cmd);
		}
	}
}



bool KviUserParser::parseCmd_ECHO(KviCommand *c)
{
	/*
		@doc: echo
		@type:
			command
		@title:
			echo
		@syntax:
			echo [-w=<window_id>] [-i=<color_set>] [-n] <text>
		@short:
			Outputs text to a KVIrc window
		@description:
			Outputs <text> to the current window.[br]
			If the 'w' switch is present, output <text>
			to the specified window instead of the current one.
			The <window_id> parameter is the [doc:window_naming_conventions]global ID[/doc] of the window
			that has to be used.[br]
			Please note that this is not the same as the standard
			[doc:command_rebinding]-r rebinding switch[/doc]:
			-w causes the specified window to be used only for output,
			but the command parameters are evaluated in the current window.[br]
			If the 'i' switch is given , it uses the specified
			icon scheme (icon and colors) , otherwise it uses
			the default one (0).[br]
			For more informations about the icon/color schemes see [fnc]$msgtype[/fnc].
			The KVIrc view widgets support clickable sockets that can be realized by using special [doc:escape_sequences]escape sequences[/doc].[br]
			The -n switch disables the message timestamping (so you can output your own
			timestamp or no timestamp at all).[br]
		@examples:
			[example]
			echo Hey! this is my first echo test!
			echo -i 10 This text has a specified icon and colors
			echo -i [fnc]$msgtype[/fnc](parser error) this has the colors of the parser error messages
			[/example]
		@seealso:
			[fnc]$window[/fnc],
			[fnc]$caption[/fnc],
			[fnc]$msgtype[/fnc],
			[cmd]echoprivmsg[/cmd],
			[doc:window_naming_conventions]Window Naming Conventions[/doc]
	*/

	ENTER_STACK_FRAME(c,"echo");

	KviStr buffer;
	if(!parseCmdFinalPart(c,buffer))return false;

	int type = KVI_OUT_NONE;
	KviWindow * pWnd = c->window();

	QString szBuffer(buffer.ptr()); // from utf8

	if(c->hasSwitch('i'))
	{
		KviStr icon;
		if(c->getSwitchValue('i',icon))
		{
			bool bOk;
			int msgType = icon.toInt(&bOk);
			if(bOk)
			{
				if(msgType < 0)msgType = -msgType;
				type = msgType % KVI_NUM_MSGTYPE_OPTIONS;
			} else c->warning(__tr2qs("Invalid icon specification '%s', using default"),icon.ptr());
		} else c->warning(__tr2qs("Missing icon number after the 'i' switch"));
	}

	if(c->hasSwitch('w'))
	{
		KviStr window;
		if(c->getSwitchValue('w',window))
		{
			pWnd = g_pApp->findWindow(window.ptr());
			if(!pWnd)
			{
				c->warning(__tr2qs("Window '%s' not found, using current one"),window.ptr());
				pWnd = c->window();
			}
		} else c->warning(__tr2qs("Missing window ID after the 'w' switch"));
	}

	int iFlags = c->hasSwitch('n') ? KviIrcView::NoTimestamp : 0;

	pWnd->outputNoFmt(type,szBuffer,iFlags);

	return c->leaveStackFrame();
}


bool KviUserParser::parseCmd_ECHOPRIVMSG(KviCommand *c)
{
	/*
		@doc: echoprivmsg
		@type:
			command
		@title:
			echoprivmsg
		@syntax:
			echoprivmsg [switches] <nick> <user> <host> <text>
		@short:
			Outputs text to a KVIrc window
		@switches:
			!sw: -p[=<nick_prefix>]
			The message is printed with the specified custom nickname prefix.
			If <nick_prefix> is omitted then an empty string is assumed.
			!sw: -s[=<nick_suffix>]
			The message is printed with the specified custom nickname suffix.
			If <nick_suffix> is omitted then an empty string is assumed.
			!sw: -w=<window_id>
			The message is printed to the window specified by [doc:window_naming_conventions]window_id[/doc]
			!sw: -i=<color_set>
			Causes the message to use the specified icon scheme (icon and colors).
			!sw: -n
			Do not apply the highlighting rules
			!sw: -x
			Never cause the notifier window to pop up
			!sw: -f
			Never cause the window taskbar entry to flash (this works only on some platforms)
		@description:
			Outputs a <text> to the current window in the privmsg format.[br]
			The <nick> <user> and <host> parameters are formatted
			as specified by the user preferences (for example
pears in.)
			the nick may use smart colorisation).
			If you don't know the username and host then just use '*' for
			that parameters.
			The message will also get the highlighting rules applied.
			If the 'w' switch is present, outputs <text>
			to the specified window instead of the current one.
			The <window_id> parameter is the [doc:window_naming_conventions]global ID[/doc] of the window
			that has to be used.[br]
			Please note that this is not the same as the standard
			[doc:command_rebinding]-r rebinding switch[/doc]:
			-w causes the specified window to be used only for output,
			but the command parameters are evaluated in the current window.[br]
			If the 'i' switch is given , it uses the specified
			icon scheme (icon and colors) , otherwise it uses
			the default one (0).[br]
			If the -n switch is present then the highlighting rules
			are not applied.[br]
			If the -x switch is present then the message will never cause
			the notifier window to be popped up.[br]
			If the -f switch is present then the message will never cause
			the system taskbar to flash.[br]
			Actually -x and -f have a sense only if highlighting is used and thus -n is not present.[br]
			For more informations about the icon/color schemes see [fnc]$msgtype[/fnc].
			The KVIrc view widgets support clickable sockets that can be realized by using special [doc:escape_sequences]escape sequences[/doc].[br]
		@examples:
			[example]
			echoprivmsg Test * * This is a test message
			echoprivmsg -i=$msgtype(ChanPrivmsgCrypted) Pragma pragma staff.kvirc.net Hi people! :)
			[/example]
		@seealso:
			[fnc]$window[/fnc],
			[fnc]$caption[/fnc],
			[fnc]$msgtype[/fnc],
			[cmd]echo[/cmd],
			[doc:window_naming_conventions]Window Naming Conventions[/doc]
	*/

	ENTER_STACK_FRAME(c,"echoprivmsg");

	KviStr nick;
	KviStr user;
	KviStr host;

	KviStr buffer;
	if(!parseCmdSingleToken(c,nick))return false;
	if(!parseCmdSingleToken(c,user))return false;
	if(!parseCmdSingleToken(c,host))return false;
	if(!parseCmdFinalPart(c,buffer))return false;

	QString szNick(nick.ptr()); // from utf8
	QString szUser(user.ptr()); // from utf8
	QString szHost(host.ptr()); // from utf8
	QString szBuff(buffer.ptr()); // from utf8

	int type = KVI_OUT_NONE;
	KviWindow * pWnd = c->window();
	KviConsole * pConsole = pWnd->console();
	if(!pConsole)pConsole = g_pApp->activeConsole();

	if(c->hasSwitch('i'))
	{
		KviStr icon;
		if(c->getSwitchValue('i',icon))
		{
			bool bOk;
			int msgType = icon.toInt(&bOk);
			if(bOk)
			{
				if(msgType < 0)msgType = -msgType;
				type = msgType % KVI_NUM_MSGTYPE_OPTIONS;
			} else c->warning(__tr2qs("Invalid icon specification '%s', using default"),icon.ptr());
		} else c->warning(__tr2qs("Missing icon number after the 'i' switch"));
	}

	if(c->hasSwitch('w'))
	{
		KviStr window;
		if(c->getSwitchValue('w',window))
		{
			pWnd = g_pApp->findWindow(window.ptr());
			if(!pWnd)
			{
				c->warning(__tr2qs("Window '%s' not found, using current one"),window.ptr());
				pWnd = c->window();
			}
		} else c->warning(__tr2qs("Missing window ID after the 'w' switch"));
	}

	KviStr szPrefix,szSuffix;
	bool bPrefix = false;
	bool bSuffix = true;
	if(c->hasSwitch('p'))c->getSwitchValue('p',szPrefix), bPrefix = true;
	if(c->hasSwitch('s'))c->getSwitchValue('s',szSuffix), bSuffix = true;

	int iFlags = 0;
	if(c->hasSwitch('n'))iFlags |= KviConsole::NoHighlighting;
	if(c->hasSwitch('f'))iFlags |= KviConsole::NoWindowFlashing;
	if(c->hasSwitch('x'))iFlags |= KviConsole::NoNotifier;
	pConsole->outputPrivmsg(pWnd,type,
				szNick,szUser,szHost,szBuff,
				iFlags,
				bPrefix ? QString(szPrefix.ptr()) : QString::null,bSuffix ? QString(szSuffix.ptr()) : QString::null);

	return c->leaveStackFrame();
}


bool KviUserParser::parseCmd_QUERY(KviCommand *c)
{
	/*
		@doc: query
		@type:
			command
		@title:
			query
		@syntax:
			query <nickname list> [text]
		@short:
			Opens one or more query windows
		@description:
			Opens a query window for each user specified in <nickname list>
			which is a [b]comma separated[/b] list of nicknames.[br]
			If [text] is speficied, it is sent to the
			query window just as it would have been written in the query itself.
			If a query with one of the specified targets already exists,
			it is simply focused and the [text] is sent to the target.[br]
			This command is [doc:connection_dependant_commands]connection dependant[/doc].[br]
		@examples:
			[example]
			[comment]# Open a single query to Pragma[/comment]
			query Pragma
			[comment]# Open a query to Pragma and to Crocodile , say "Hello!" in both windows[/comment]
			query Pragma,Crocodile Hello !
			[/example]
	*/

	ENTER_STACK_FRAME(c,"query");

	KviStr targets;
	if(!parseCmdSingleToken(c,targets))return false;
	KviStr text;
	if(!parseCmdFinalPart(c,text))return false;

	if(targets.isEmpty())c->warning(__tr2qs("No targets specified"));


	if(!c->window()->console())return c->noIrcContext();
	if(!c->window()->console()->isConnected())return c->notConnectedToServer();

	KviQuery * query;
	KviStr nick;

	while(targets.hasData())
	{
		targets.getToken(nick,',');
		if(nick.isEmpty())c->warning(__tr2qs("Empty target specified"));
		else {
			QString szNick(nick.ptr()); // from utf8
		
			query = c->window()->console()->connection()->findQuery(szNick);
			if(query)query->autoRaise();
			else {
				query = c->window()->console()->connection()->createQuery(szNick);
				QString user;
				QString host;
				KviIrcUserDataBase * db = c->window()->connection()->userDataBase();
				if(db)
				{
					KviIrcUserEntry * e = db->find(szNick);
					if(e)
					{
						user = e->user();
						host = e->host();
					}
				}
				query->setTarget(szNick,user,host);
			}

			if(text.hasData())
			{
				QString szWText(text.ptr()); // from utf8
				query->ownMessage(szWText);
			}
		}
	}
	return c->leaveStackFrame();
}


bool KviUserParser::parseCmd_RAW(KviCommand *c)
{
	/*
		@doc: raw
		@type:
			command
		@title:
			raw
		@syntax:
			raw [-q] <raw data>
		@short:
			Sends raw data to the server
		@description:
			Sends a raw data string to the server of the current [b]IRC context[/b].[br]
			If the -q switch (quiet) is specified, no output is printed.
			This command is [doc:connection_dependant_commands]connection dependant[/doc].[br]
			This command is equivalent to [cmd]quote[/cmd].
		@examples:
			[example]
			[comment]# Send a private message "by hand"[/comment]
			raw PRIVMSG Pragma :hello!
			[comment]# Send a private message thru another connection[/comment]
			raw -r=[fnc]$console[/fnc]([fnc]$ic[/fnc](irc.otherserver.com,othernick)) PRIVMSG Pragma :heya on this side!
			[/example]
	*/

	/*
		@doc: quote
		@type:
			command
		@title:
			quote
		@syntax:
			quote [-q] <raw data>
		@short:
			Builtin alias for raw
		@description:
			Builtin alias for the command [cmd]raw[/cmd].
	*/

	ENTER_STACK_FRAME(c,"raw");

	KviStr text;
	if(!parseCmdFinalPart(c,text))return false;
	QString szWText = QString::fromUtf8(text.ptr()); // from utf8

	if(!c->window()->console())return c->noIrcContext();
	if(!c->window()->connection())return c->notConnectedToServer();
	
	QCString szData = c->window()->connection()->encodeText(szWText);
	if(szData.data())
	{
		if(!(c->window()->connection()->sendData(szData.data())))
			return c->notConnectedToServer();

		if(!c->hasSwitch('q'))c->window()->output(KVI_OUT_RAW,__tr2qs("[RAW]: %Q"),&szWText);
	}

	return c->leaveStackFrame();
}


bool KviUserParser::parseCmd_CTCP(KviCommand *c)
{
	/*
		@doc: ctcp
		@type:
			command
		@title:
			ctcp
		@syntax:
			ctcp <target> <ctcp_data>
		@short:
			Sends a CTCP message
		@description:
			Sends a CTCP message to the specified <target>.[br]
			The target may be a nickname , a channel, or a comma separated list of nicknames.[br]
			The <ctcp_data> is a string containing the ctcp type followed by the ctcp parameters.[br]
			For more info take a look at the [doc:ctcp_handling]ctcp protocol implementation notes[/doc].[br]
			The CTCP message will be a request (sent through a PRIVMSG) unless the -n switch
			specified: in that case it will be a reply (sent through a NOTICE).[br]
			If <ctcp_data> is the single string "ping" then a trailing time string argument
			is added in order to determine the round trip time when the ping reply comes back.
			To override this behaviour simply specify your own time string parameter.[br]
			This command is [doc:connection_dependant_commands]connection dependant[/doc].[br]
		@examples:
			[example]
			ctcp Pragma VERSION
			[/example]
	*/

	ENTER_STACK_FRAME(c,"ctcp");

	KviStr target,ctcpData;
	if(!parseCmdSingleToken(c,target))return false;
	if(!parseCmdFinalPart(c,ctcpData))return false;

	QString szTarget = QString::fromUtf8(target.ptr()); // from utf8

	if(!c->window()->console())return c->noIrcContext();
	if(!c->window()->connection())return c->notConnectedToServer();

	if(kvi_strEqualCI(ctcpData.ptr(),"PING"))
	{
		struct timeval tv;
		kvi_gettimeofday(&tv,0);
		ctcpData.append(KviStr::Format," %d.%d",tv.tv_sec,tv.tv_usec);
	}

	QString szData = QString::fromUtf8(ctcpData.ptr()); // from utf8
	
	QCString szT = c->window()->connection()->encodeText(szTarget);
	QCString szD = c->window()->connection()->encodeText(szData);

	if(!(c->window()->connection()->sendFmtData("%s %s :%c%s%c",
			c->hasSwitch('n') ? "NOTICE" : "PRIVMSG",szT.data(),0x01,szD.data(),0x01)))
		return c->notConnectedToServer();

	return c->leaveStackFrame();
}

bool KviUserParser::parseCmd_NICK(KviCommand *c)
{
	/*
		@doc: nick
		@type:
			command
		@title:
			nick
		@syntax:
			nick <new nickname>
		@short:
			Changes your nickname
		@description:
			Changes your nickname in the connection associated to the
			current [b]IRC context[/b].[br] This command is "server based";
			this means that the effects will be visible only after the
			server has acknowledged the change.[br]
			This command is [doc:connection_dependant_commands]connection dependant[/doc].[br]
		@examples:
			[example]
			nick Pragma
			[/example]
	*/

	ENTER_STACK_FRAME(c,"nick");

	KviStr newnick;
	if(!parseCmdFinalPart(c,newnick))return false;

	QString szNick = QString::fromUtf8(newnick.ptr()); // from utf8

	if(newnick.isEmpty())
	{
		c->warning(__tr2qs("Missing new nickname"));
		return c->leaveStackFrame();
	}

	if(!c->window()->console())return c->noIrcContext();
	if(!c->window()->connection())return c->notConnectedToServer();

	QCString szN = c->window()->connection()->encodeText(szNick);

	if(!(c->window()->connection()->sendFmtData("NICK %s",szN.data())))
		return c->notConnectedToServer();

	return c->leaveStackFrame();
}

bool KviUserParser::parseCmd_PRIVMSG(KviCommand *c)
{
	/*
		@doc: privmsg
		@type:
			command
		@title:
			privmsg
		@syntax:
			privmsg <target> <text>
		@short:
			Sends a private message
		@description:
			Sends a private message to the specified <target>.
			<target> may be any PRIVMSG target allowed by the underlying
			IRC protocol (see RFC1459).
			This command is really similar to [cmd]msg[/cmd] but it does not
			output the message locally.[br]
			This command is [doc:connection_dependant_commands]connection dependant[/doc].[br]
			This command is equivalent to [cmd]msg[/cmd]
		@examples:
			[example]
			privmsg Pragma Hello!
			privmsg Pragma,Crocodile Hello to you both!
			privmsg #kvirc Hello from outside!
			[/example]
	*/

	/*
		@doc: msg
		@type:
			command
		@title:
			msg
		@syntax:
			msg [-q] <target> <text>
		@short:
			Alias for privmsg
		@description:
			Sends a private message to the specified <target>.
			<target> may be any PRIVMSG target allowed by the underlying
			IRC protocol (see RFC1459).
			This is really similar to [cmd]privmsg[/cmd] but also outputs the
			message locally (unless the [-q] switch is used).[br]
			This command is [doc:connection_dependant_commands]connection dependant[/doc].[br]
	*/

	ENTER_STACK_FRAME(c,"privmsg");

	bool bIsMsgCommand = (tolower(*(c->currentEntity())) == 'm');

	KviStr target;
	if(!parseCmdSingleToken(c,target))return false;
	QString szWTarget = QString::fromUtf8(target.ptr()); // from utf8

	KviStr text;
	if(!parseCmdFinalPart(c,text))return false;
	QString szWText = QString::fromUtf8(text); // from utf8

	if(target.isEmpty())
	{
		c->warning(__tr2qs("Missing target"));
		return c->leaveStackFrame();
	}

	if(!c->window()->console())return c->noIrcContext();
	if(!c->window()->connection())return c->notConnectedToServer();

	KviWindow * w = c->window()->connection()->findChannel(szWTarget);
	if(!w)w = c->window()->connection()->findQuery(szWTarget);

	if(w)w->ownMessage(szWText);
	else {

		QCString szTarget = c->window()->connection()->encodeText(szWTarget);
		QCString szText = w ? w->encodeText(szWText) : c->window()->connection()->encodeText(szWText);

		if(!(c->window()->connection()->sendFmtData("PRIVMSG %s :%s",szTarget.data(),szText.data())))
			return c->notConnectedToServer();

		if(bIsMsgCommand && (!c->hasSwitch('q')))c->window()->output(KVI_OUT_OWNPRIVMSG,
			"[PRIVMSG >>> %Q]: %Q",&szWTarget,&szWText);
	}

	return c->leaveStackFrame();
}

bool KviUserParser::parseCmd_NOTICE(KviCommand *c)
{
	/*
		@doc: notice
		@type:
			command
		@title:
			notice
		@syntax:
			notice [-q] <target> <text>
		@short:
			Sends a private notice
		@description:
			Sends a private notice to the specified <target>.
			<target> may be any NOTICE target allowed by the underlying
			IRC protocol (see RFC1459).
			If the [-q] switch is specified, no output is printed.
			This command is [doc:connection_dependant_commands]connection dependant[/doc].[br]
		@examples:
			[example]
			notice Pragma Hello!
			notice Pragma,Crocodile Hello to you both!
			notice #kvirc Hello from outside!
			[/example]
	*/

	ENTER_STACK_FRAME(c,"notice");

	KviStr target;
	if(!parseCmdSingleToken(c,target))return false;
	QString szWTarget = QString::fromUtf8(target.ptr()); // from utf8

	KviStr text;
	if(!parseCmdFinalPart(c,text))return false;
	QString szWText = QString::fromUtf8(text.ptr()); // from utf8

	if(target.isEmpty())
	{
		c->warning(__tr2qs("Missing target"));
		return c->leaveStackFrame();
	}

	if(!c->window()->console())return c->noIrcContext();
	if(!c->window()->connection())return c->notConnectedToServer();

	KviWindow * w = c->window()->connection()->findQuery(szWTarget);
	if(!w)w = c->window()->connection()->findChannel(szWTarget);

	QCString szTarget = c->window()->connection()->encodeText(szWTarget);
	QCString szText = w ? w->encodeText(szWText) : c->window()->connection()->encodeText(szWText);

	if(!(c->window()->connection()->sendFmtData("NOTICE %s :%s",szTarget.data(),szText.data())))
		return c->notConnectedToServer();

// FIXME: #warning "KVI_OUT_OWNNOTICE ?"

	if(!c->hasSwitch('q'))c->window()->output(KVI_OUT_OWNPRIVMSG,
		"[NOTICE >>> %Q]: %Q",&szWTarget,&szWText);

	return c->leaveStackFrame();
}

bool KviUserParser::parseCmd_JOIN(KviCommand *c)
{
	/*
		@doc: join
		@type:
			command
		@title:
			join
		@syntax:
			join <channel list> [key list]
		@short:
			Joins the specified channels
		@description:
			Joins the channels specified in <channel list> by using the
			optional keys in the [key list].[br]
			This command is "server based"; you will see its results only
			after the server acknowledges it.[br]
			This command is [doc:connection_dependant_commands]connection dependant[/doc].[br]
		@examples:
			[example]
			join #kvirc
			join #linux-it,#xmms,#kde linus
			[/example]
	*/

	ENTER_STACK_FRAME(c,"join");

	KviStr chans;
	KviStr keys;
	if(!parseCmdSingleToken(c,chans))return false;
	if(!parseCmdFinalPart(c,keys))return false;

	if(chans.isEmpty())
	{
		c->warning(__tr2qs("Missing channel list"));
		return c->leaveStackFrame();
	}

	KviStr ** strarray = chans.splitToArray(',',100,0);

	for(int i=0;strarray[i];i++)
	{
		if(!(strarray[i]->firstCharIs('#') ||
				strarray[i]->firstCharIs('&') ||
				strarray[i]->firstCharIs('!') ||
				strarray[i]->firstCharIs('+')))
			strarray[i]->prepend('#');
	}

	chans.joinFromArray(strarray,",");
	chans.freeArray(strarray);

	if(!c->window()->console())return c->noIrcContext();
	if(!c->window()->connection())return c->notConnectedToServer();

	QCString szChans = c->window()->connection()->encodeText(QString::fromUtf8(chans.ptr()));

	if(keys.isEmpty())
	{
		if(!(c->window()->connection()->sendFmtData("JOIN %s",szChans.data())))
			return c->notConnectedToServer();
	} else {
		QCString szKeys  = c->window()->connection()->encodeText(QString::fromUtf8(keys.ptr()));
		if(!(c->window()->connection()->sendFmtData("JOIN %s %s",szChans.data(),szKeys.data())))
			return c->notConnectedToServer();
	}

	return c->leaveStackFrame();
}

bool KviUserParser::parseCmd_PART(KviCommand *c)
{
	/*
		@doc: part
		@type:
			command
		@title:
			part
		@syntax:
			part [-k] [-q] [-s] [<channel list> [part message]]
		@short:
			Leaves the specified channels
		@switches:
			!sw: -k
			Keeps the channel windows open after the part.
			!sw: -q
			Runs the command in quiet mode (no warnings printed)
			!sw: -s
			Does not show the part message in the channel window
		@description:
			Leaves the channels specified in the <channel list>.
			If no channels are specified, the current channel is used.
			This command is "server based"; you will see its results only
			after the server acknowledges it.[br]
			If the -k switch is specified then the channel windows are left open
			after the part (the channels become dead channels).[br]
			If the -s switch is specified then the part message is not shown
			in the channel window.[br]
			This command is [doc:connection_dependant_commands]connection dependant[/doc].[br]
			This command is equivalent to [cmd]leave[/cmd]
		@examples:
			[example]
			part #kvirc Byez!
			part #linux-it,#xmms,#kde Time to sleep
			[/example]
	*/

	/*
		@doc: leave
		@type:
			command
		@title:
			leave
		@syntax:
			leave [-k] [-q] [-s] <channel list> [part message]
		@short:
			Alias for part
		@description:
			Builtin alias for the command [cmd]part[/cmd].
	*/

	ENTER_STACK_FRAME(c,"part");

	KviStr chans;
	if(!parseCmdSingleToken(c,chans))return false;

	KviStr text;
	if(!parseCmdFinalPart(c,text))return false;

	if(chans.isEmpty())
	{
		if(c->window()->type() == KVI_WINDOW_TYPE_CHANNEL)
			chans = c->window()->target();
		else {
			if(!c->hasSwitch('q'))c->warning(__tr2qs("Missing channel list"));
			return c->leaveStackFrame();
		}
	}

	if(!c->window()->console())return c->noIrcContext();
	if(!c->window()->connection())return c->notConnectedToServer();

	QString szUniChans = QString::fromUtf8(chans.ptr());
	QCString szChans = c->window()->connection()->encodeText(szUniChans);

	QStringList sl = QStringList::split(",",szUniChans);

	if(text.hasData())
	{
		QCString szText;
		if(sl.count() == 1)
		{
			// single chan , use channel encoding if possible
			KviChannel * ch = c->window()->connection()->findChannel(szUniChans);
			if(ch)
				szText = ch->encodeText(QString::fromUtf8(text.ptr()));
			else
				szText = c->window()->connection()->encodeText(QString::fromUtf8(text.ptr()));
		} else {
			// multiple chans, use connection encoding
			szText = c->window()->connection()->encodeText(QString::fromUtf8(text.ptr()));
		}
		if(!(c->window()->connection()->sendFmtData("PART %s :%s",szChans.data(),szText.data())))
			return c->notConnectedToServer();
	} else {
		if(!(c->window()->connection()->sendFmtData("PART %s",szChans.data())))
			return c->notConnectedToServer();
	}

	for(QStringList::Iterator it=sl.begin();it != sl.end();it++)
	{
		KviChannel * ch = c->window()->connection()->findChannel(*it);
		if(ch)
		{
			ch->partMessageSent(!c->hasSwitch('k'),!c->hasSwitch('s'));
		} else {
			if(!c->hasSwitch('q'))c->warning(__tr2qs("You don't appear to be on channel %s"),(*it).latin1());
		}
	}

	return c->leaveStackFrame();
}

bool KviUserParser::parseCmd_TOPIC(KviCommand *c)
{
	/*
		@doc: topic
		@type:
			command
		@title:
			topic
		@syntax:
			topic [<channel> [topic text]]
		@short:
			Changes topic for the specified channel
		@description:
			Changes the topic for the specified <channel> to <topic text>.
			If no topic is specified, the current channel topic will be returned.[br]
			This command is "server based"; you will see its results only
			after the server acknowledges it.[br]
			This command is [doc:connection_dependant_commands]connection dependant[/doc].
		@examples:
			[example]
			topic #kvirc KVIrc 3.0.0 has been released!
			[/example]
	*/

	ENTER_STACK_FRAME(c,"topic");

	KviStr chan;
	if(!parseCmdSingleToken(c,chan))return false;

	KviStr text;
	if(!parseCmdFinalPart(c,text))return false;

	if(chan.isEmpty())
	{
		if(c->window()->type() == KVI_WINDOW_TYPE_CHANNEL)
			chan = c->window()->target();
		else {
			c->warning(__tr2qs("Missing target channel"));
			return c->leaveStackFrame();
		}
	}

	if(!c->window()->console())return c->noIrcContext();
	if(!c->window()->connection())return c->notConnectedToServer();

	QString szChan = QString::fromUtf8(chan.ptr()); // from utf8

	// decode from utf8
	QString szTopic(text.ptr());
	// re-encode in window/connection encoding
	QCString ch = c->window()->connection()->encodeText(szChan);
	QCString t = c->window()->encodeText(szTopic);


	if(!(c->window()->connection()->sendFmtData("TOPIC %s :%s",ch.data(),t.data())))
		return c->notConnectedToServer();

	return c->leaveStackFrame();
}


bool KviUserParser::parseCmd_OP(KviCommand *c)
{
	/*
		@doc: op
		@type:
			command
		@title:
			op
		@syntax:
			op <nickname_list>
		@short:
			Sets chanop status to the specified users
		@description:
			Sets channel operator status to the users specified in <nickname_list>,
			which is a comma separated list of nicknames. 
			This command works only if executed in a channel window.
			The command is translated to a set of MODE messages containing
			a variable number of +o flags. All servers should support
			up to three mode changes in a single message; some servers
			will support even more (this is user definable,
			see [fnc]$option[/fnc](uintModeChangesPerMessage)).[br]
			This command is [doc:connection_dependant_commands]connection dependant[/doc].
		@examples:
			[example]
			op Pragma,Crocodile
			[/example]
		@seealso:
			[cmd]deop[/cmd],[cmd]voice[/cmd],[cmd]devoice[/cmd]
	*/

	ENTER_STACK_FRAME(c,"op");
	return parseMultipleModeCommand(c,'+','o') ? c->leaveStackFrame() : false;
	return c->leaveStackFrame();
}

bool KviUserParser::parseCmd_DEOP(KviCommand *c)
{
	/*
		@doc: deop
		@type:
			command
		@title:
			deop
		@syntax:
			deop <nickname_list>
		@short:
			Removes chanop status from the specified users
		@description:
			Removes channel operator status to the users specified in <nickname_list>,
			which is a comma separated list of nicknames. 
			This command works only if executed in a channel window.
			The command is translated to a set of MODE messages containing
			a variable number of -o flags. All servers should support
			up to three mode changes in a single message; some servers
			will support even more (this is user definable,
			see [fnc]$option[/fnc](uintModeChangesPerMessage)).[br]
			This command is [doc:connection_dependant_commands]connection dependant[/doc].
		@examples:
			[example]
			deop Pragma,Crocodile
			[/example]
		@seealso:
			[cmd]op[/cmd],[cmd]voice[/cmd],[cmd]devoice[/cmd]
	*/

	ENTER_STACK_FRAME(c,"deop");
	return parseMultipleModeCommand(c,'-','o') ? c->leaveStackFrame() : false;
	return c->leaveStackFrame();
}

bool KviUserParser::parseCmd_VOICE(KviCommand *c)
{
	/*
		@doc: voice
		@type:
			command
		@title:
			voice
		@syntax:
			voice <nickname_list>
		@short:
			Sets voice flag for the specified users
		@description:
			Sets the voice flag for the users specified in <nickname_list>,
			which is a comma separated list of nicknames. 
			This command works only if executed in a channel window.
			The command is translated to a set of MODE messages containing
			a variable number of +v flags. All servers should support
			up to three mode changes in a single message; some servers
			will support even more (this is user definable,
			see [fnc]$option[/fnc](uintModeChangesPerMessage)).
			This command is [doc:connection_dependant_commands]connection dependant[/doc].
		@examples:
			[example]
			voice Pragma,Crocodile
			[/example]
		@seealso:
			[cmd]op[/cmd],[cmd]deop[/cmd],[cmd]devoice[/cmd]
	*/

	ENTER_STACK_FRAME(c,"voice");
	return parseMultipleModeCommand(c,'+','v') ? c->leaveStackFrame() : false;
	return c->leaveStackFrame();
}

bool KviUserParser::parseCmd_DEVOICE(KviCommand *c)
{
	/*
		@doc: devoice
		@type:
			command
		@title:
			devoice
		@syntax:
			devoice <nickname_list>
		@short:
			Removes voice flag for the specified users
		@description:
			Removes the voice flag for the users specified in <nickname_list>,
			which is a comma separated list of nicknames. 
			This command works only if executed in a channel window.
			The command is translated to a set of MODE messages containing
			a variable number of -v flags. All servers should support
			up to three mode changes in a single message; some servers
			will support even more (this is user definable,
			see [fnc]$option[/fnc](uintModeChangesPerMessage)).
			This command is [doc:connection_dependant_commands]connection dependant[/doc].
		@examples:
			[example]
			devoice Pragma,Crocodile
			[/example]
		@seealso:
			[cmd]op[/cmd],[cmd]deop[/cmd],[cmd]voice[/cmd]
	*/

	ENTER_STACK_FRAME(c,"devoice");
	return parseMultipleModeCommand(c,'-','v') ? c->leaveStackFrame() : false;
	return c->leaveStackFrame();
}

bool KviUserParser::parseCmd_BAN(KviCommand *c)
{
	/*
		@doc: ban
		@type:
			command
		@title:
			ban
		@syntax:
			ban <mask_list>
		@short:
			Sets ban masks for the channel
		@description:
			Sets the ban masks specified in the <mask_list>,
			which is a comma separated list of nicknames. 
			This command works only if executed in a channel window.
			The command is translated to a set of MODE messages containing
			a variable number of +b flags. All servers should support
			up to three mode changes in a single message; some servers
			will support even more (this is user definable,
			see [fnc]$option[/fnc](uintModeChangesPerMessage)).[br]
			This command is [doc:connection_dependant_commands]connection dependant[/doc].
		@examples:
			[example]
			ban Maxim,Gizmo!*@*,*!root@*
			[/example]
		@seealso:
			[cmd]op[/cmd],[cmd]deop[/cmd],[cmd]voice[/cmd],[cmd]devoice[/cmd],[cmd]unban[/cmd]
	*/

	ENTER_STACK_FRAME(c,"ban");
	return parseMultipleModeCommand(c,'+','b') ? c->leaveStackFrame() : false;
	return c->leaveStackFrame();
}

bool KviUserParser::parseCmd_UNBAN(KviCommand *c)
{
	/*
		@doc: unban
		@type:
			command
		@title:
			unban
		@syntax:
			unban <mask_list>
		@short:
			Unsets ban masks for the channel
		@description:
			Removes the ban masks specified in the <mask_list>,
			which is a comma separated list of nicknames. 
			This command works only if executed in a channel window.
			The command is translated to a set of MODE messages containing
			a variable number of -b flags. All servers should support
			up to three mode changes in a single message; some servers
			will support even more (this is user definable,
			see [fnc]$option[/fnc](uintModeChangesPerMessage)).[br]
			This command is [doc:connection_dependant_commands]connection dependant[/doc].
		@examples:
			[example]
			# Unban people on the current channel (say #kvirc)
			unban Maxim,Gizmo!*@*,*!root@*
			# Do the same but from another window belongin to this IRC context
			unban -r=[fnc]$channel[/fnc](#kvirc) Maxim,Gizmo!*@*,*!root@*
			# Do the same from any window
			unban -r=[fnc]$channel[/fnc](#kvirc,[fnc]$context[/fnc](irc.myirc.net,Pragma)) Maxim,Gizmo!*@*,*!root*@*
			[/example]
		@seealso:
			[cmd]op[/cmd],[cmd]deop[/cmd],[cmd]voice[/cmd],[cmd]devoice[/cmd],[cmd]ban[/cmd]
	*/

	ENTER_STACK_FRAME(c,"unban");
	return parseMultipleModeCommand(c,'-','b') ? c->leaveStackFrame() : false;
	return c->leaveStackFrame();
}

bool KviUserParser::parseMultipleModeCommand(KviCommand *c,char plusminus,char flag)
{
	ENTER_STACK_FRAME(c,"kvirc::multipleModeCommand");

	KviStr tokens;
	if(!parseCmdFinalPart(c,tokens))return false;

	if(c->window()->type() != KVI_WINDOW_TYPE_CHANNEL)
	{
		c->warning(__tr2qs("This is not a channel"));
		return c->leaveStackFrame();
	}


	KviStr message;
	KviStr flags;
	KviStr token;

	int i = 0;

	// sanity check
	if(KVI_OPTION_UINT(KviOption_uintModeChangesPerMessage) < 1)
		KVI_OPTION_UINT(KviOption_uintModeChangesPerMessage) = 1;


	tokens.getToken(token,',');
	if(!c->window()->connection())return c->notConnectedToServer(); // Hm ? should never happen
	QCString sz = c->window()->connection()->encodeText(c->window()->windowName());

	while(token.hasData())
	{
		if(message.hasData())message.append(' ');
		message.append(token);
		flags.append(flag);
		i++;
		if(((unsigned int)i) == KVI_OPTION_UINT(KviOption_uintModeChangesPerMessage))
		{
			QCString szm = c->window()->connection()->encodeText(QString::fromUtf8(message.ptr()));
			if(szm.data())
			{
				if(!c->window()->connection()->sendFmtData("MODE %s %c%s %s",
					sz.data(),plusminus,flags.ptr(),szm.data()))
						return c->notConnectedToServer();
			}
			i = 0;
			flags = "";
			message = "";
		}
		tokens.getToken(token,',');
	}

	if(message.hasData())
	{
		QCString szm = c->window()->connection()->encodeText(QString::fromUtf8(message.ptr()));
		c->window()->connection()->sendFmtData("MODE %s %c%s %s",
			sz.data(),plusminus,flags.ptr(),szm.data());
	}

	return c->leaveStackFrame();
}

bool KviUserParser::parseCmd_MODE(KviCommand *c)
{
	/*
		@doc: mode
		@type:
			command
		@title:
			mode
		@syntax:
			mode <target> <modeflags> [mode parameters]
		@short:
			Sends a MODE irc message
		@description:
			Sends a MODE irc message to the server of the current IRC context.[br]
			The parameters are not modified in any way by KVIrc: so
			you should use the RFC1459 syntax.[br]
			This command is [doc:connection_dependant_commands]connection dependant[/doc].
		@examples:
			[example]
			mode #kvirc +oo-b Pragma Buti *!*root@*
			[/example]
	*/

	ENTER_STACK_FRAME(c,"mode");

	KviStr text;
	if(!parseCmdFinalPart(c,text))return false;

	if(text.isEmpty())
		if(c->window()->type() == KVI_WINDOW_TYPE_CHANNEL)
			text = c->window()->target();

	if(!c->window()->console())return c->noIrcContext();
	if(!c->window()->connection())return c->notConnectedToServer();
	QCString szTxt = c->window()->connection()->encodeText(QString::fromUtf8(text.ptr()));
	if(szTxt.data())
	{
		if(!(c->window()->connection()->sendFmtData("MODE %s",szTxt.data())))
			return c->notConnectedToServer();
	}

	return c->leaveStackFrame();
}

bool KviUserParser::parseCmd_KICK(KviCommand *c)
{
	/*
		@doc: kick
		@type:
			command
		@title:
			kick
		@syntax:
			kick <user> [reason]
		@short:
			Kicks an user from a channel
		@description:
			Kicks the specified <user> from the current channel.[br]
			This command is a simple wrapper over the IRC KICK command.
			This command is [doc:connection_dependant_commands]connection dependant[/doc].
		@examples:
			[example]
			kick Pragma Go coding in Pascal!
			[/example]
	*/

	ENTER_STACK_FRAME(c,"kick");

	KviStr nick,reason;
	if(!parseCmdSingleToken(c,nick))return false;
	if(!parseCmdFinalPart(c,reason))return false;
	if(c->window()->type() != KVI_WINDOW_TYPE_CHANNEL)
	{
		c->warning(__tr2qs("This is not a channel"));
		return c->leaveStackFrame();
	}

// FIXME: #warning "Allow to kick multiple users ?"
// FIXME: #warning "Default reason ?"
	if(!c->window()->console())return c->noIrcContext();
	if(!c->window()->connection())return c->notConnectedToServer();

	// decode from utf8
	QString szReason = QString::fromUtf8(reason.ptr());
	QString szNick = QString::fromUtf8(nick.ptr());
	// re-encode in window/connection encoding
	QCString n = c->window()->connection()->encodeText(szNick);
	QCString r = c->window()->encodeText(szReason);

	if(!(c->window()->connection()->sendFmtData(reason.isEmpty() ? "KICK %s %s" : "KICK %s %s :%s",c->window()->windowName().latin1(),n.data(),r.data())))
		return c->notConnectedToServer();

	return c->leaveStackFrame();
}

bool KviUserParser::parseCmd_SETRETURN(KviCommand *c)
{
	/*
		@doc: setreturn
		@type:
			command
		@title:
			setreturn
		@syntax:
			setreturn <string>
		@short:
			Sets the return value of the current command sequence
		@description:
			Sets the return value of the current command sequence to <string>.
			You can call this command multiple times in a command sequence
			(so in fact use it as a write-only variable).
		@examples:
		@seealso:
			[cmd]return[/cmd]
	*/

	ENTER_STACK_FRAME(c,"setreturn");


	// FIXME: We could avoid return value propagation if we'd set the "topmost parent command" return buffer...
	//        Think about it...
	c->m_szRetBuffer = "";
	if(!parseCmdFinalPart(c,c->m_szRetBuffer))return false;

	return c->leaveStackFrame();
}

bool KviUserParser::parseCmd_RETURN(KviCommand *c)
{
	/*
		@doc: return
		@type:
			command
		@title:
			return
		@syntax:
			return <string>
		@short:
			Returns from the current command sequence and sets it return value
		@description:
			Sets the return value of the current command sequence to <string>
			and stops the execution.[br]
			This is more ro less equivalent to calling [cmd]setreturn[/cmd] <string>
			and then [cmd]halt[/cmd], but has no additional semantics in events.[br]
		@examples:
		@seealso:
			[cmd]setreturn[/cmd], [cmd]halt[/cmd], [cmd]break[/cmd]
	*/

	ENTER_STACK_FRAME(c,"return");
	// FIXME: We could avoid return value propagation if we'd set the "topmost parent command" return buffer...
	//        Think about it...
	c->m_szRetBuffer = "";
	if(!parseCmdFinalPart(c,c->m_szRetBuffer))return false;
	c->setReturnEncountered();
	return false;
}

bool KviUserParser::parseCmd_SRAND(KviCommand *c)
{
	/*
		@doc: srand
		@type:
			command
		@title:
			srand
		@syntax:
			srand <seed>
		@short:
			Initializes the random number generator
		@description:
			Initializes the random number generator "seeding" it with
			<seed> which must be an integer. If no <seed> is passed it defaults to 1.
			Sequences of random numbers can be repeated by using the
			same <seed> value.
		@examples:
	*/

	ENTER_STACK_FRAME(c,"srand");

	KviStr seed;
	if(!parseCmdFinalPart(c,seed))return false;
	bool bOk;
	int iSeed = seed.toInt(&bOk);
	if(!bOk)iSeed = 1;
	srand(iSeed);
	return c->leaveStackFrame();
}

bool KviUserParser::parseCmd_ME(KviCommand *c)
{
	/*
		@doc: me
		@type:
			command
		@title:
			me
		@syntax:
			me <text>
		@short:
			Sends a CTCP ACTION
		@description:
			Sends a CTCP ACTION to the current channel, query or dcc chat.[br]
			If you execute it in any other window type, you will get an error.[br]
			If you want to use this command in a window that is not a channel
			query or dcc chat, you may use the [doc:command_rebinding]standard -r switch[/doc].
		@examples:
			[example]
			me is Hungry!
			[/example]
	*/
	ENTER_STACK_FRAME(c,"me");

	KviStr text;
	if(!parseCmdFinalPart(c,text))return false;

	QString sz(text.ptr()); // this decodes from utf8!

	switch(c->window()->type())
	{
		case KVI_WINDOW_TYPE_CHANNEL:
		case KVI_WINDOW_TYPE_QUERY:
		case KVI_WINDOW_TYPE_DCCCHAT:
			c->window()->ownAction(sz);
		break;
		default:
			return c->noIrcContext();
		break;
	}

	return c->leaveStackFrame();
}



bool KviUserParser::parseCmd_SWITCH(KviCommand *c)
{
	/*
		@doc: switch
		@type:
			command
		@title:
			switch
		@syntax:
			switch [-s] (<expression>)
			{
				case(<value>)[:]<command>
				[break]
				match(<wildcard_expression>)[:]<command>
				[break]
				regexpr(<regular_expression>)[:]<command>
				[break]
				default[:]<command>
				[break]
			}
		@keyterms:
			conditional commands, flow control commands
		@short:
			Flow control command
		@description:
			This command is similar to the C switch: it
			just has some extended "case" labels.
			Fist evaluates expression, then executes each 
			internal label in the following way.
			The <value> in the "case" label is compared
			with the result of the expresison evaluation.
			If the result is equal to <value> then the corresponding
			<command1> is executed.
			In the "match" labels the result of the expression
			evaluation is matched in <wildcard_expression>. <wildcard_expression>
			can contain the usual * and ? wildcard characters.
			If the expression matches then the corresponding <command> is executed.
			In the "regexpr" labels the result of <expression> evaluation is matched against
			a real regular expression. If the regular expression matches then
			the corresponding command is executed.
			The default label is executed unconditionally if no other
			labels matched. If the break command is used in an executed
			<command> then the next labels are not evaluated. If break is not
			used then all the labels are evaluated sequentially and
			multiple matches are possible.
			If the -s switch is used then the comparisons
			become case sensitive.
			TODO: Write some examples :D
		@examples:
			[example]
				%x = 1
				switch(%x)
				{
					case(0): echo It was 0!
					case(1): echo It was 1!
				}
			[/example]
	*/

	ENTER_STACK_FRAME(c,"switch");

	c->skipWhiteSpace();
	if(*(c->m_ptr) != '(')return c->error(KviError_openParenthesisExpected);
	KviParameterList mainParmList;
	if(!extractFunctionParameters(c,&mainParmList))return false;
	KviStr * string = mainParmList.safeFirst();

	c->skipWhiteSpace();

	if(*(c->m_ptr) != '{')return c->error(KviError_openBraceExpected);
	++(c->m_ptr);

	c->skipWhiteSpace();

	int callStackSize = c->actualStackSize();
	bool bBreak = false;

	bool bCaseS = c->hasSwitch('s');

	bool (*cmpFunc)(const char *,const char *) = bCaseS ? kvi_strEqualCS : kvi_strEqualCI;

	bool bLastLabelExecuted = false;

	KviParameterList parmList;

	while(*(c->m_ptr) && (*(c->m_ptr) != '}'))
	{
		if(kvi_strEqualCIN(c->m_ptr,"case",4))
		{
			c->m_ptr += 4;

			if(bBreak)
			{
				if(!skipCaseLabel(c))return false;
			} else {
				char * pBegin = c->m_ptr;
	
				c->skipWhiteSpace();
				if(*(c->m_ptr) != '(')return c->error(KviError_openParenthesisExpected);
				parmList.clear();
				if(!extractFunctionParameters(c,&parmList))return false;
				KviStr * caseVal = parmList.safeFirst();
				c->skipWhiteSpace();
				if(*(c->m_ptr) == ':')
				{
					++(c->m_ptr);
					c->skipWhiteSpace();
				}

				if(bLastLabelExecuted = cmpFunc(caseVal->ptr(),string->ptr()))
				{
					bool bRetVal;
					if(*(c->m_ptr) == '{')bRetVal = parseCommandBlock(c);
					else bRetVal = parseSingleCommand(c);
					if(!bRetVal)
					{
						if(c->breakEncountered())
						{
							bBreak = true;
							c->setBreakHandled(callStackSize); // ensure that the call stack is returned to coherent state
							c->m_ptr = pBegin;
							if(!skipCaseLabel(c))return false;
						} else return false;
					}
				} else if(!skipCommand(c))return false;
			}
		} else if(kvi_strEqualCIN(c->m_ptr,"match",5))
		{
			c->m_ptr += 5;

			if(bBreak)
			{
				if(!skipCaseLabel(c))return false;
			} else {
				char * pBegin = c->m_ptr;
				c->skipWhiteSpace();
				if(*(c->m_ptr) != '(')return c->error(KviError_openParenthesisExpected);
				parmList.clear();
				if(!extractFunctionParameters(c,&parmList))return false;
				KviStr * caseVal = parmList.safeFirst();
				c->skipWhiteSpace();
				if(*(c->m_ptr) == ':')
				{
					++(c->m_ptr);
					c->skipWhiteSpace();
				}
				QRegExp exp(caseVal->ptr(),bCaseS,true);
			
				int iResult = exp.search(string->ptr(),0);
				if(bLastLabelExecuted = ((iResult == 0) && (exp.matchedLength() == string->len())))
				{
					bool bRetVal;
					if(*(c->m_ptr) == '{')bRetVal = parseCommandBlock(c);
					else bRetVal = parseSingleCommand(c);
					if(!bRetVal)
					{
						if(c->breakEncountered())
						{
							bBreak = true;
							c->setBreakHandled(callStackSize); // ensure that the call stack is returned to coherent state
							c->m_ptr = pBegin;
							if(!skipCaseLabel(c))return false;
						} else return false;
					}	
				} else if(!skipCommand(c))return false;
			}

		} else if(kvi_strEqualCIN(c->m_ptr,"regexp",6))
		{
			c->m_ptr += 6;
			if(bBreak)
			{
				if(!skipCaseLabel(c))return false;
			} else {
				char * pBegin = c->m_ptr;
				c->skipWhiteSpace();
				if(*(c->m_ptr) != '(')return c->error(KviError_openParenthesisExpected);
				parmList.clear();
				if(!extractFunctionParameters(c,&parmList))return false;
				KviStr * caseVal = parmList.safeFirst();
				c->skipWhiteSpace();
				if(*(c->m_ptr) == ':')
				{
					++(c->m_ptr);
					c->skipWhiteSpace();
				}
				QRegExp exp(caseVal->ptr(),bCaseS,false);

				int iResult = exp.search(string->ptr(),0);
				if(bLastLabelExecuted = ((iResult == 0) && (exp.matchedLength() == string->len())))
				{
					bool bRetVal;
					if(*(c->m_ptr) == '{')bRetVal = parseCommandBlock(c);
					else bRetVal = parseSingleCommand(c);
					if(!bRetVal)
					{
						if(c->breakEncountered())
						{
							bBreak = true;
							c->setBreakHandled(callStackSize); // ensure that the call stack is returned to coherent state
							c->m_ptr = pBegin;
							if(!skipCaseLabel(c))return false;
						} else return false;
					}
				} else if(!skipCommand(c))return false;
			}
		} else if(kvi_strEqualCIN(c->m_ptr,"default",7))
		{
			c->m_ptr += 7;
			if(bBreak)
			{
				if(!skipDefaultLabel(c))return false;
			} else {
				bLastLabelExecuted = true;
				char * pBegin = c->m_ptr;
				c->skipWhiteSpace();
				if(*(c->m_ptr) == ':')
				{
					++(c->m_ptr);
					c->skipWhiteSpace();
				}
				bool bRetVal;
				if(*(c->m_ptr) == '{')bRetVal = parseCommandBlock(c);
				else bRetVal = parseSingleCommand(c);
				if(!bRetVal)
				{
					if(c->breakEncountered())
					{
						bBreak = true;
						c->setBreakHandled(callStackSize); // ensure that the call stack is returned to coherent state
						c->m_ptr = pBegin;
						if(!skipDefaultLabel(c))return false;
					} else return false;
				}
			}
		} else if(kvi_strEqualCIN(c->m_ptr,"break",5))
		{
			c->m_ptr += 5;
			if(!skipNormalCommand(c))return false;
			c->skipWhiteSpace();
			bBreak = bBreak || bLastLabelExecuted;
		} else if(*(c->m_ptr) == '#')
		{
			// A comment
			skipComment(c);
		} else return c->error(KviError_caseMatchRegexpDefaultOrBreakExpected);

		c->skipWhiteSpace();
	}

	if(!(*(c->m_ptr)))return c->error(KviError_missingClosingBrace);
	++(c->m_ptr);

	return c->leaveStackFrame();
}

bool KviUserParser::parseCmd_IF(KviCommand *c)
{
	/*
		@doc: if
		@type:
			command
		@title:
			if
		@syntax:
			if (<condition>) <command1> [else <command2>]
		@keyterms:
			conditional commands, flow control commands
		@short:
			Flow control command
		@description:
			Executes <command1> if the <condition> evaluates
			to true (non zero result).
			If the "else part" is given <command2> is executed
			if the <condition> evaluates to false (result == '0')
		@examples:
	*/

	ENTER_STACK_FRAME(c,"if");

	c->skipWhiteSpace();
	if(*(c->m_ptr) != '(')return c->error(KviError_openParenthesisExpected);
	++(c->m_ptr);
	long res;
	if(!evaluateExpression(c,&res))return false;
	c->skipWhiteSpace();
	if(res)
	{
		if(*(c->m_ptr) == '{')
		{
			if(!parseCommandBlock(c))return false;
		} else {
			if(!parseSingleCommand(c))return false;
		}
		char * aux = c->m_ptr;
		c->skipWhiteSpace();
		if(kvi_strEqualCIN(c->m_ptr,"else",4))
		{
			c->m_ptr += 4;
			c->skipWhiteSpace();
			if(!skipCommand(c))return false;
		} else c->m_ptr = aux;
	} else {
		if(!skipCommand(c))return false;
		char * aux = c->m_ptr;
		c->skipWhiteSpace();
		if(kvi_strEqualCIN(c->m_ptr,"else",4))
		{
			c->m_ptr += 4;
			c->skipWhiteSpace();
			if(*(c->m_ptr) == '{')
			{
				if(!parseCommandBlock(c))return false;
			} else {
				if(!parseSingleCommand(c))return false;
			}
		} else c->m_ptr = aux;
	}
	return c->leaveStackFrame();
}

bool KviUserParser::parseCmd_BREAK(KviCommand *c)
{
	/*
		@doc: break
		@type:
			command
		@title:
			break
		@syntax:
			break
		@short:
			Interrupts an iteration loop
		@description:
			Interrupts an iteration loop like [cmd]while[/cmd].[br]
			This command always jumps out of a single code block.[br]
			If called outside an iteration loop , will act just like [cmd]halt[/cmd]
			has been called but has no additional semantics for events.[br]
	*/
	ENTER_STACK_FRAME(c,"break");
	KviStr dummy;
	if(!parseCmdFinalPart(c,dummy))return false;
	c->setBreakEncountered();
	return false;
}

bool KviUserParser::parseCmd_RAISE(KviCommand *c)
{
	/*
		@doc: raise
		@type:
			command
		@title:
			raise
		@syntax:
			raise
		@short:
			Raises a KVIrc frame window
		@description:
			Raises and activates the current KVIrc frame window....assuming that your window manager supports it.[br]
	*/
	ENTER_STACK_FRAME(c,"raise");
	KviStr dummy;
	if(!parseCmdFinalPart(c,dummy))return false;
	if(!c->window()->frame()->isVisible())c->window()->frame()->show();
	c->window()->frame()->raise();
	//c->window()->frame()->setActiveWindow();
	//c->window()->frame()->setFocus();
	((KviTalMainWindow *)c->window()->frame())->setActiveWindow();
	return c->leaveStackFrame();
}


bool KviUserParser::parseCmd_WHILE(KviCommand *c)
{
	/*
		@doc: while
		@type:
			command
		@title:
			while
		@syntax:
			while (<condition>) <command>
		@keyterms:
			iteration commands, flow control commands
		@short:
			Iteration command
		@description:
			Executes <command> while the <condition> evaluates
			to true (non zero result).[br]
			<command> may be either a single command or a block of commands.[br]
			It can contain the [cmd]break[/cmd] command: in that case the
			execution of the <command> will be immediately interrupted and the control
			transferred to the command following this while block.[br]
		@examples:
			[example]
			%i = 0;
			while(%i < 100)%i++
			while(%i > 0)
			{
				%i -= 10
				if(%i < 20)break;
			}
			echo %i
			[/example]
	*/

	ENTER_STACK_FRAME(c,"while");


	c->skipWhiteSpace();
	if(*(c->m_ptr) != '(')return c->error(KviError_openParenthesisExpected);
	char * pExpression = ++(c->m_ptr);

	long res;
	if(!evaluateExpression(c,&res))return false;
	c->skipWhiteSpace();

	int callStackSize = c->actualStackSize();

	if(!res)
	{
		// False condition...just skip and return
		if(!skipCommand(c))return false;
	} else {
		char * pCommand = c->m_ptr;
		char * pEnd = 0;
	
		while(res)
		{
			c->m_ptr = pCommand;

			bool bRetVal;

			if(*(c->m_ptr) == '{')bRetVal = parseCommandBlock(c);
			else bRetVal = parseSingleCommand(c);

			if(!bRetVal)
			{
				if(c->breakEncountered())
				{
					if(!pEnd)
					{
						// not reached the end yet...must reach it
						c->m_ptr = pCommand;
						if(!skipCommand(c))return false;
					} else c->m_ptr = pEnd;
					c->setBreakHandled(callStackSize); // ensure that the call stack is returned to coherent state
					return c->leaveStackFrame();

				} else return false;
			}

			pEnd = c->m_ptr;

			c->m_ptr = pExpression;
			if(!evaluateExpression(c,&res))return false;
		}

		c->m_ptr = pEnd; // non zero for sure
	}
	return c->leaveStackFrame();
}

bool KviUserParser::parseCmd_DELETE(KviCommand *c)
{
	/*
		@doc: delete
		@type:
			command
		@title:
			delete
		@syntax:
			delete [-i] <object_id>
		@short:
			Destroys an object
		@description:
			Schedules for destruction the object designed by <object_id>.
			This command is internally aliased to [cmd]destroy[/cmd].
			Please note that the object is NOT immediately destroyed:
			it will be destroyed when KVIrc returns to the main event loop,
			so after the current script code part has been executed.
			This behaviour makes the object destruction safe in any
			part of the script, but may lead to problems when
			using signals and slots.[br]
			For example, when you delete an object that emits some signals,
			the signals may be still emitted after the delete call.
			You have to disconnect the signals explicitly if you don't want it
			to happen.[br]
			Alternatively you can use the -i switch: it causes the object
			to be destructed immediately but is intrinsicly unsafe:
			in complex script scenarios it may lead to a SIGSEGV;
			usually when called from one of the deleted object function
			handlers, or from a slot connected to one of the deleted object
			signals. Well, it actually does not SIGSEGV, but I can't guarantee it;
			so, if use the -i switch, test your script 10 times before releasing it.
		@examples:
			[example]
			[/example]
	*/

	/*
		@doc: destroy
		@type:
			command
		@title:
			destroy
		@syntax:
			destroy [-i] <object_id>
		@short:
			Destroys an object
		@description:
			This is a builtin alias for the command [cmd]delete[/cmd]
	*/


	ENTER_STACK_FRAME(c,"delete");

	KviStr obj;
	if(!parseCmdFinalPart(c,obj))return false;

	KviScriptObject * ob = g_pScriptObjectController->lookupObject(obj.ptr());

	if(ob)
	{
		if(c->hasSwitch('i'))ob->dieNow();
		else ob->die();
		
	} else c->warning(__tr2qs("No such object (%s)"),obj.ptr());

	return c->leaveStackFrame();
}

bool KviUserParser::parseCmd_PARSE(KviCommand *c)
{
	/*
		@doc: parse
		@type:
			command
		@title:
			parse
		@syntax:
			parse <filename> [parameters]
		@short:
			Executes commands from a file
		@description:
			Executes commands from the external file <filename>.[br]
			[parameters] is a space separated string of parameters to be
			passed to the script. The parameter $0 will contain the
			name of the file being parsed, the other parameters will
			be available starting from $1.[br]
			If <filename> is a directory name a file dialog 
			will pop up, allowing you to choose a file to parse. 
			Parameters are passed normally in this situation.
	*/
	ENTER_STACK_FRAME(c,"parse");

	KviStr filename;
	if(!parseCmdSingleToken(c,filename)) return false;

	kvi_adjustFilePath(filename);

	if(kvi_directoryExists(filename.ptr())/* ||
	   kvi_strEqualCS(filename.ptr(), ".")*/)
		if(!KviFileDialog::askForOpenFileName(filename, \
			__tr("Choose a file to parse"), filename.ptr(),"*.kvs"))
		return c->leaveStackFrame();

	kvi_adjustFilePath(filename);

	KviStr buffer;
	if(!kvi_loadFile(filename.ptr(),buffer))
	{
		return c->error(KviError_noSuchFile,"%s",filename.ptr());
	}

	KviParameterList * l = new KviParameterList;
	l->append(new KviStr(filename));

	KviStr token;
	while((*(c->m_ptr) != '\n') && *(c->m_ptr) && (*(c->m_ptr) != ';'))
	{
		if(!parseCmdSingleToken(c,token))
		{
			delete l;
			return false;
		}
		l->append(new KviStr(token));
		token = "";
	}

	KviCommand cmd(buffer.ptr(),c->window(),c);
	cmd.setParams(l,true); // transfer parameter ownership too
	if(!parseCommand(&cmd))
	{
		if(cmd.hasError())
		{
			printError(&cmd);
			return false; // just a halt...we could do better ?
		}
	}

	c->m_szRetBuffer = cmd.m_szRetBuffer;

	return parseCmdFinalPart(c,token) ? c->leaveStackFrame() : false;
}

bool KviUserParser::parseCmd_CLASS(KviCommand *c)
{
	ENTER_STACK_FRAME(c,"class");

	/*
		@doc: class
		@title:
			class
		@short:
			Defines a new object class
		@keyterms:
			defining an object class
		@type:
			command
		@syntax:
			class(<class>,<base_name>)
			{
				<function_name>[([<parameter reminder>])]
				{
					<function body>
				}

				...
			}
		@description:
			Defines a new implementation of the class <class>.
			If an implementation of that class was already existing
			it is removed with all the derived classes (and all the instances of this class
			and the derived ones are destroyed).
			<base_name> is the name of the class that the 
			new class has to inherit from.[br]
			Note:[br]
			The keywords "function" and "event" that were used in KVIrc versions
			previous to 3.0.0 have been removed since "useless".[br]
			The <parameter reminder> part is an optional string
			that can be used to sign the parameters that the function expects;
			it acts as a programmer reminder or comment and it has no other
			meaning in KVIrc scripting. The <parameter reminder> respects the syntax
			of an expression, so it is terminated by a closed parenthesis.
			It's rather dangerous to use this command inside an object
			function handler: if the class definition <class> was already 
			existing and it is a parent of the object's class, you might
			end up executing "inexisting" code.[br]
			As a thumb rule, use this command only outside object function handlers.[br]
		@examples:
			[example]
				class(myclass,[class]object[/class])
				{
					constructor
					{
						[cmd]echo[/cmd] Hey this is my constructor
						[cmd]echo[/cmd] I have been just created
					}

					destructor
					{
						[cmd]echo[/cmd] Ops...being destroyed
					}

					sayHello(this function expects no parameters)
					{
						[cmd]echo[/cmd] Hello world!
					}
				}
			[/example]
		@seealso:
			[cmd]privateimpl[/cmd], [cmd]killclass[/cmd], [cmd]clearobjects[/cmd], [fnc]$classDefined[/fnc](),
			[doc:objects]Objects documentation[/doc]
	*/

	c->skipWhiteSpace();
	if(*(c->m_ptr) != '(')return c->error(KviError_openParenthesisExpected);


	KviParameterList paramList;
	paramList.setAutoDelete(true);

	if(!extractFunctionParameters(c,&paramList))return false;

	KviStr * clName = paramList.safeFirst();
	KviStr * base   = paramList.safeNext();

	if(clName->isEmpty())return c->error(KviError_missingObjectClassName);

	KviScriptObjectClass * pBase = g_pScriptObjectController->lookupClass(base->ptr());
	if(!pBase)return c->error(KviError_noSuchObjectClass,"%s",base->ptr());

	KviScriptObjectClass * pClass = g_pScriptObjectController->lookupClass(clName->ptr());
	if(pClass)
	{
		if(pClass->isBuiltin())return c->error(KviError_cantKillABuiltinClass,"%s",clName->ptr());
		else delete pClass; //Yahoo!!
	}

	KviPtrList<KviStr> fncList;
	fncList.setAutoDelete(true);

	KviPtrList<KviStr> cmdList;
	cmdList.setAutoDelete(true);

	c->skipWhiteSpace();

	if(*(c->m_ptr) != '{')return c->error(KviError_openBraceExpected);

	++(c->m_ptr);

	while(*(c->m_ptr) && (*(c->m_ptr) != '}'))
	{
		c->skipWhiteSpace();
		if(*(c->m_ptr) == '}')break;

		if(*(c->m_ptr) == '#')skipComment(c);
		else {
			if(isalpha(*(c->m_ptr)) || (*(c->m_ptr) == '_'))
			{
				const char * aux = c->m_ptr;
				while(*(c->m_ptr) && (isalnum(*(c->m_ptr)) || (*(c->m_ptr) == '_')))++(c->m_ptr);
				KviStr * fName = new KviStr(aux,c->m_ptr);
				fncList.append(fName);
				c->skipWhiteSpace();
				if(*(c->m_ptr) == '(')
				{
					// parameter reminder
					if(!skipExpressionBody(c))return false;
					c->skipWhiteSpace();
				}
				if(*(c->m_ptr) != '{')return c->error(KviError_openBraceExpected);
				aux = c->m_ptr;
				if(!skipCommand(c))return false;
				KviStr * cmd = new KviStr(aux,c->m_ptr);
//				cmd->cutLeft(1); // cut the leading '{'
//				cmd->cutRight(1); // cut the leading '}'
//				cmd->stripWhiteSpace(); // kill whitespace
				KviCommandFormatter::bufferFromBlock(*cmd);
				cmdList.append(cmd);
			} else return c->error(KviError_unexpectedCharacter,__tr("Function name expected"));
		}
	}

	if(!(*(c->m_ptr)))return c->error(KviError_missingClosingBrace);

	++(c->m_ptr);

	// ok...can register the class I guess..
	pClass = new KviScriptObjectClass(pBase,clName->ptr(),0,false);

	KviStr * b = cmdList.first();
	for(KviStr *f=fncList.first();f;f=fncList.next(),b=cmdList.next())
	{
		pClass->registerFunctionHandler(f->ptr(),0,b->ptr(),false);
	}

	return c->leaveStackFrame();
}

bool KviUserParser::parseCmd_KILLCLASS(KviCommand *c)
{
	ENTER_STACK_FRAME(c,"killclass");
	/*
		@doc: killclass
		@title:
			killclass
		@type:
			command
		@short:
			Removes a class definition
		@syntax:
			killclass [-q] <class_name>
		@description:
			Removes the definition of the class <class_name>.[br]
			All the instances of the class are killed (thus children objects are killed too).[br]
			All the derived class definitions are removed as well.[br]
			If the class <class_name> has not been defined yet
			this command prints a warning unless the -q switch is used.[br]
			Builtin class definitions cannot be removed and this commands
			kills only all the instances of that class (derived class definitions
			and instances are NOT killed).[br]
		@seealso:
			[cmd]class[/cmd], [cmd]clearobjects[/cmd], [fnc]$classDefined[/fnc](),
			[doc:objects]Objects documentation[/doc]
	*/
	KviStr clName;
	if(!parseCmdFinalPart(c,clName))return false;

	KviScriptObjectClass * pClass = g_pScriptObjectController->lookupClass(clName.ptr());
	if(pClass)
	{
		if(pClass->isBuiltin())
		{
			g_pScriptObjectController->killAllObjectsWithClass(pClass);
		} else delete pClass; //Yahoo!!
	} else {
		if(!c->hasSwitch('q'))c->warning(__tr2qs("Class %s is undefined"),clName.ptr());
	}
	return c->leaveStackFrame();
}


bool KviUserParser::parseCmd_CLEAROBJECTS(KviCommand *c)
{
	ENTER_STACK_FRAME(c,"clearobjects");
	/*
		@doc: clearobjects
		@title:
			clearobjects
		@type:
			command
		@short:
			Removes all the user class definitions
		@syntax:
			clearobjects [-i]
		@description:
			Removes the definition of all the user classes and kill all the object
			instances (also instances of the builtin classes).[br]
			If the -i switch is used, only the object instances are cleared
			but the class definitions are left unchanged.[br]
			This command should be used only for debugging & testing purposes
			and in particular should be NEVER used from inside an object
			function handler (leads to SIGSEGV :).[br]
		@seealso:
			[cmd]class[/cmd], [cmd]killclass[/cmd],
			[doc:objects]Objects documentation[/doc]
	*/
	KviStr dummy;
	if(!parseCmdFinalPart(c,dummy))return false;

	g_pScriptObjectController->clearInstances();
	if(!(c->hasSwitch('i')))g_pScriptObjectController->clearUserClasses();

	return c->leaveStackFrame();
}

bool KviUserParser::parseCmd_PRIVATEIMPL(KviCommand *c)
{
	ENTER_STACK_FRAME(c,"privateimpl");
	/*
		@doc: privateimpl
		@title:
			privateimpl
		@type:
			command
		@short:
			Adds a private implementation of a function
		@syntax:
			privateimpl(<object_id>,<function_name>)<implementation>
		@description:
			Adds a private implementation of function <function_name> to the
			existing object designed by <object_id>.
			<implementation> must be a valid command sequence.[br]
			Side note:[br]
			This command can not succesfully implement
			the "constructor" function since it must be called
			after this one has already been executed.[br]
			To implement a constructor you MUST write your own class definition.[br]
		@seealso:
			[cmd]class[/cmd],
			[doc:objects]Objects documentation[/doc]
	*/

	c->skipWhiteSpace();
	if(*(c->m_ptr) != '(')return c->error(KviError_openParenthesisExpected);

	KviParameterList paramList;
	paramList.setAutoDelete(true);

	if(!extractFunctionParameters(c,&paramList))return false;

	KviStr * objId = paramList.safeFirst();
	KviStr * fncName   = paramList.safeNext();

	KviScriptObject * o = g_pScriptObjectController->lookupObject(objId->ptr());
	if(!o)return c->error(KviError_noSuchObject,"%s",objId->ptr());

	c->skipWhiteSpace();

	if(*(c->m_ptr) != '{')return c->error(KviError_openBraceExpected);
	const char * aux = c->m_ptr;
	if(!skipCommand(c))return false;

	KviStr cmd(aux,c->m_ptr);
	KviCommandFormatter::bufferFromBlock(cmd);
//	cmd.cutLeft(1);
//	cmd.cutRight(1);
//	cmd.stripWhiteSpace();

	o->registerPrivateImplementation(fncName->ptr(),cmd.hasData() ? cmd.ptr() : 0);
	return c->leaveStackFrame();
}

bool KviUserParser::parseCmd_CONNECT(KviCommand *c)
{
	ENTER_STACK_FRAME(c,"connect");
	/*
		@doc: connect
		@title:
			connect
		@type:
			command
		@short:
			Connects a signal to a slot
		@syntax:
			connect <source_object> <signal_name> <target_object> <slot_name>
		@description:
			Connects the <source_object>'s signal <signal_name> to the
			<target_object>'s slot <slot_name>.
			When one of the two objects is destroyed, the signal/slot
			connection is automatically removed.[br]
			WARNING: This command name collides with the [doc:rfc2812]RFC2812[/doc]
			CONNECT IRC Op command: this IRC command is available to operators only
			and is rather rarely used: you can use it by the means of [doc:raw]raw[/doc].
		@seealso:
			[cmd]class[/cmd], [cmd]disconnect[/cmd], [doc:objects]objects documentation[/doc]
	*/
	KviStr src,sig,trg,slt,dummy;
	if(!parseCmdSingleToken(c,src))return false;
	if(!parseCmdSingleToken(c,sig))return false;
	if(!parseCmdSingleToken(c,trg))return false;
	if(!parseCmdSingleToken(c,slt))return false;
	if(!parseCmdFinalPart(c,dummy))return false;

	KviScriptObject * pSrc = g_pScriptObjectController->lookupObject(src.ptr());
	if(!pSrc)return c->error(KviError_noSuchObject,src.ptr());

	KviScriptObject * pTrg = g_pScriptObjectController->lookupObject(trg.ptr());
	if(!pTrg)return c->error(KviError_noSuchObject,trg.ptr());

	if(!pSrc->connectSignal(sig.ptr(),pTrg,slt.ptr()))
		return c->error(KviError_noSuchObjectFunction,slt.ptr());

	return c->leaveStackFrame();
}

bool KviUserParser::parseCmd_DISCONNECT(KviCommand *c)
{
	ENTER_STACK_FRAME(c,"disconnect");
	/*
		@doc: disconnect
		@title:
			disconnect
		@type:
			command
		@short:
			Disconnects a signal from a slot
		@syntax:
			disconnect <source_object> <signal_name> <target_object> <slot_name>
		@description:
			Disconnects the <source_object>'s signal <signal_name> from the
			<target_object>'s slot <slot_name>.
			When one of the two objects is destroyed, the signal/slot
			connection is automatically removed.
		@seealso:
			[cmd]class[/cmd], [cmd]connect[/cmd], [doc:objects]objects documentation[/doc]
	*/

	KviStr src,sig,trg,slt,dummy;
	if(!parseCmdSingleToken(c,src))return false;
	if(!parseCmdSingleToken(c,sig))return false;
	if(!parseCmdSingleToken(c,trg))return false;
	if(!parseCmdSingleToken(c,slt))return false;
	if(!parseCmdFinalPart(c,dummy))return false;

	KviScriptObject * pSrc = g_pScriptObjectController->lookupObject(src.ptr());
	if(!pSrc)
	{
		c->warning(__tr2qs("The source object '%s' is not existing"),src.ptr());
		return c->leaveStackFrame();
	}

	KviScriptObject * pTrg = g_pScriptObjectController->lookupObject(trg.ptr());
	if(!pTrg)
	{
		c->warning(__tr2qs("The target object '%s' is not existing"),trg.ptr());
		return c->leaveStackFrame();
	}

	if(pSrc->disconnectSignal(sig.ptr(),pTrg,slt.ptr()))
		c->warning(__tr2qs("No signal '%s' to disconnect"),sig.ptr());

	return c->leaveStackFrame();
}


bool KviUserParser::parseCmd_SERVER(KviCommand *c)
{
	/*
		@doc: server
		@type:
			command
		@title:
			server
		@syntax:
			server [switches] [<server> [<port>]]
		@keyterms:
			connecting to a server
		@short:
			Connects to the specified server
		@switches:
			!sw: -b=<local address>
			Bind the outgoing connection to <local address>. The <local address> must be
			the IP address of a local interface suitable for contacting the remote <server>:
			this means that if you're going to use the -i switch, the <local address>
			must be an IPV6 address.
			On most systems it is also possible to use a local interface name as <local address>;
			in this way the interface address can be resolved at runtime by the KVIrc network engine.

			!sw: -c=<command sequence>
			The <command sequence> will be executed just after the login operations
			have been completed.

			!sw: -f=<socket filter>
			Activates <socket filter> on the connection.
			A socket filter is a plugin that can adapt the IRC protocol to other existing protocols such as OpenNap.
			For plain IRC it is not needed.

			!sw: -i
			Makes the connection by using the IPV6 protocol
			(if supported by the executable)

			!sw: -n
			Forces the connection to be attempted in a new IRC context instead of the current one.

			!sw: -p=<password>
			Uses <password> to login to the server (the password will be stored in the server
			entry too).

			!sw: -q=<nick>
			Uses <nick> to login to the server (the nicknames will be stored in the server
			entry too).

			!sw: -s
			Activates the SSL support for this connection

			!sw: -u
			Forces the connection to be attempted in the first IRC context that has
			no connection in progress. If all the IRC contexts have connections in progress
			then a new one is created.

		@description:
			Attempts a connection to the specified <server>
			on the specified <port>. If no <port> is specified
			the default 6667 is used. If no <server> is specified,
			the currently selected server from the server list is used.
			If the <server> is a valid IPV6 address, the -i switch is implied.[br]
			The <server> is added to the server list (if not already
			there) and set as current.[br]
			If <server> doesn't seem to be a valid ip address or hostname
			it is assumed to be a network name and if such a network is found
			in the server list then the best server for that network is contacted.
			(KVIrc tries to guess the round-robin (random) servers first).[br]
			The connection is attempted in the current IRC context. If there is already
			a connection in progress then it is "brutally" terminated.[br]
		@examples:
			[example]
				server irc.tin.it
				server -i irc6.ircd.it
				server -i ircd.stealth.net 6667
				server irc.undernet.org 6665
				server -n irc.openprojects.net
				server -c="join #kvirc" irc.stealth.net 6665
				server -b=ppp0 irc.tin.it
				server undernet
			[/example]
	*/

	ENTER_STACK_FRAME(c,"server");

	KviStr server;
	KviStr port;
	if(!parseCmdSingleToken(c,server))return false;
	if(!parseCmdFinalPart(c,port))return false;

	KviConsole * console = 0;
	if(c->hasSwitch('n'))console = c->window()->frame()->createNewConsole();
	else {
		if(c->hasSwitch('u'))
		{
			// first try the current console
			console = c->window()->console();
			if(console)
			{
				// if there is a connection in progress, proceed in searching
				if(console->connectionInProgress())
				{
					console = 0;
				}
			}
			if(!console)
			{
				// yep , have to search
				console = c->window()->frame()->firstNotConnectedConsole();
				if(!console)
				{
					console = c->window()->frame()->createNewConsole();
				}
			}
		} else console = c->window()->console();
	}

	KviStr cmd;
	c->getSwitchValue('c',cmd);
	KviStr bindAddr;
	c->getSwitchValue('b',bindAddr);
	KviStr socketFilter;
	c->getSwitchValue('f',socketFilter);
	KviStr pass;
	c->getSwitchValue('p',pass);
	KviStr nick;
	c->getSwitchValue('q',nick);

	if(!console)return c->noIrcContext();

	if(server.isEmpty())
	{
		// just as hitting "connect"
		if(console->connectionInProgress())return c->error(KviError_anotherConnectionInProgress);
		else {
			KviAsynchronousConnectionData * d = new KviAsynchronousConnectionData();
			d->szCommandToExecAfterConnect = cmd;
			d->szBindAddress = bindAddr;
			console->context()->setAsynchronousConnectionData(d);
			console->context()->connectToCurrentServer();
		}
	} else {
		// server is not empty.
		bool bOk;
		unsigned short int uPort = port.toUShort(&bOk);
		KviAsynchronousConnectionData * d = new KviAsynchronousConnectionData();
		d->bUseSSL = c->hasSwitch('s');
		d->szServer = server;
		d->uPort = uPort;
		d->szLinkFilter = socketFilter;
		d->bPortIsOk = bOk;
#ifdef COMPILE_IPV6_SUPPORT
		d->bUseIpV6 = c->hasSwitch('i') || kvi_isValidStringIp_V6(server.ptr());
#else
		d->bUseIpV6 = false;
#endif
		d->szCommandToExecAfterConnect = cmd;
		d->szBindAddress = bindAddr;
		d->szPass = pass;
		d->szNick = nick;
		console->context()->setAsynchronousConnectionData(d);
		console->context()->beginAsynchronousConnect(0);
	}

	return c->leaveStackFrame();
}

bool KviUserParser::parseCmd_POPUP(KviCommand *c)
{
	/*
		@doc: popup
		@type:
			command
		@title:
			popup
		@syntax:
			popup [-p=<screen_coordinates>] <popup_name> [parameters]
		@short:
			Shows a popup menu
		@description:
			Shows the popup menu <popup_name> at the current cursor position,
			eventually passing the [parameters]. Please note that you can't
			use this command inside one of the [cmd]defpopup[/cmd] command
			handlers for <popup_name>. In other words, you can't "popup
			a popup" while it is being popped up. :) (This would
			be an endless recursive behaviour).[br]
			If the -p switch is used , then <screen_coordinates> must be
			in the form <x>,<y> and will be used as screen coordinates
			for the placement of the popup (instead of the current cursor position).[br]
		@seealso:
			[cmd]defpopup[/cmd]
	*/

	ENTER_STACK_FRAME(c,"popup");

	KviStr popupname;
	if(!parseCmdSingleToken(c,popupname))return false;

	KviParameterList * params = new KviParameterList;

	while(*(c->m_ptr) && (*(c->m_ptr) != '\n') && (*(c->m_ptr) != ';'))
	{
		KviStr tmp;
		if(!parseCmdSingleToken(c,tmp))
		{
			delete params;
			return false;
		}
		params->append(new KviStr(tmp));
	}

	KviPopupMenu * m = g_pPopupManager->lookupPopup(popupname.ptr());

	if(m)
	{
		if(m->isLocked())
		{
			delete params;
			return c->error(KviError_selfModificationNotAllowed);
		}

		QPoint pnt = QCursor::pos();
		if(c->hasSwitch('p'))
		{
			KviStr coords;
			c->getSwitchValue('p',coords);
			int idx = coords.findFirstIdx(',');
			if(idx != -1)
			{
				KviStr theX = coords.left(idx);
				coords.cutLeft(idx + 1);
				bool bOk1,bOk2;
				int iX = theX.toInt(&bOk1);
				int iY = coords.toInt(&bOk2);
				if(!(bOk1 && bOk2))c->warning(__tr("Invalid screen coordinates"));
				else pnt = QPoint(iX,iY);
			} else c->warning(__tr2qs("Invalid syntax for the screen coordinates"));
		}
		m->doPopup(pnt,c->window(),params);
	} else {
		delete params;
		c->warning(__tr2qs("The popup '%s' is not defined"),popupname.ptr());
	}

	return c->leaveStackFrame();
}


bool KviUserParser::parseCmd_SETMENU(KviCommand *c)
{
	/*
		@doc: setmenu
		@type:
			command
		@title:
			setmenu
		@syntax:
			setmenu [-q] [-i=<index>] <visible_text> [menu_name]
		@short:
			Adds a popup menu to the menu bar
		@description:
			Adds an [b]existing[/b] popup menu to the current frame's menu bar.[br]
			<visible_text> is used as menu identification and obviously as the visible
			menu bar text for the new popup.[br]
			If a popup menu with the same <visible_text> already exists in the current frame
			it is removed first.[br]
			<visible_text> can optionally contain the "&" character to identify
			the popup menu accelerator.[br]
			[menu_name] , if given , must be avalid name of a popup menu created earlier
			with [cmd]defpopup[/cmd] (or the script center).[br]
			If [menu_name] is not given , the popup menu identified by <visible_text> is removed.[br]
			If the -i switch is used , <index> is a [b]zero based index[/b] of the default
			menu-bar text items: the new popup menu will be placed on the left of that default item.[br]
			If -q is used , this command prints no warnings (so you can safely use it to remove
			popup menus without being sure that they actually are in there).
			The warning itself serves to debugging purposes for the scripter.[br]
		@seealso:
			[cmd]defpopup[/cmd]
		@examples:
			setmenu -q -i=2 Test apopup
			setmenu Test
			setmenu ThisDoesNotExist
			setmenu -q ThisDoesNotExist
	*/

	ENTER_STACK_FRAME(c,"setmenu");

	KviStr text,menuname;
	if(!parseCmdSingleToken(c,text))return false;
	if(!parseCmdFinalPart(c,menuname))return false;

	if(menuname.isEmpty())
	{
		if(!c->window()->frame()->mainMenuBar()->removeMenu(text.ptr()))
		{
			if(!c->hasSwitch('q'))c->warning(__tr2qs("No menu bar item with text '%s'"),text.ptr());
		}
	} else {
		KviPopupMenu * p = g_pPopupManager->lookupPopup(menuname.ptr());
		int idx = -1;
		if(c->hasSwitch('i'))
		{
			KviStr szIndex;
			c->getSwitchValue('i',szIndex);
			bool bOk;
			int index = szIndex.toInt(&bOk);
			if(bOk)idx = index;	
		}
		c->window()->frame()->mainMenuBar()->removeMenu(text.ptr());

		idx = c->window()->frame()->mainMenuBar()->getDefaultItemRealIndex(idx);
		
		//debug("Menu index now %d",idx);
		if(p)c->window()->frame()->mainMenuBar()->addMenu(text.ptr(),p,idx);
		else {
			if(!c->hasSwitch('q'))c->warning(__tr2qs("The popup '%s' is not defined"),menuname.ptr());
		}
	}



	return c->leaveStackFrame();
}

bool KviUserParser::parseCmd_DELPOPUPITEM(KviCommand *c)
{
	/*
		@doc: delpopupitem
		@type:
			command
		@title:
			delpopupitem
		@syntax:
			delpopupitem [-d] [-q] <popupname> <id>
		@short:
			Deletes an item from a popup
		@description:
			Deletes the item specified by <id> from the poup <popupname>.
			If the -d flag is specified then the item with the specified
			<id> is seached in the whole popup tree (containing submenus)
			otherwise it is searched only in the first level.[br]
			If the -q flag is specified the command does not complain
			about inexisting items or inexisting popup menus.[br]
			See [cmd]defpopup[/cmd] for more informations.[br]
		@seealso:
			[cmd]defpopup[/cmd],[cmd]popup[/cmd]
	*/

	ENTER_STACK_FRAME(c,"delpopupitem");

	KviStr pop,id;
	if(!parseCmdSingleToken(c,pop))return false;
	if(!parseCmdFinalPart(c,id))return false;

	if(pop.isEmpty())return c->error(KviError_missingPopupName);
	if(id.isEmpty())return c->error(KviError_notEnoughParameters);

	KviPopupMenu * p = g_pPopupManager->lookupPopup(pop.ptr());
	if(!p)
	{
		if(!c->hasSwitch('q'))
			c->warning(__tr2qs("Inexisting popup \"%s\""),pop.ptr());
		return c->leaveStackFrame();
	}

	if(p->isLocked())return c->error(KviError_selfModificationNotAllowed);

	if(!p->removeItemById(id.ptr(),c->hasSwitch('d')))
	{
		if(!c->hasSwitch('q'))
			c->warning(__tr2qs("The menu item with id \"%s\" does not exist in popup \"%s\""),id.ptr(),pop.ptr());
	}

	return c->leaveStackFrame();
}


// FIXME: #warning "Separator should have the expression too ?"


bool KviUserParser::parseCmd_DEFPOPUP(KviCommand *c)
{
	/*
		@doc: defpopup
		@type:
			command
		@title:
			defpopup
		@syntax:
			defpopup [-m] (<popup_name>)
			{
				prologue[(<id>)] <prologue_command>

				epilogue[(<id>)] <epilogue_command>

				label(<text>[,<id>])[(<expression>)][;]

				item(<text>[,<icon>[,<id>]])[(<expression>)]<command>

				popup(<text>[,<icon>[,<id>]])[(<expression>)]
				{
					<popup body>
				}

				extpopup(<text>,<name>[,<icon>[,<id>]])[(<expression>)][;]

				separator[(<id>)][;]
				...
			}
		@short:
			Defines a popup menu
		@description:
			Defines the popup menu <popup_name>. If the -m switch is NOT used
			the previous contents of the popups are cleared, otherwise are preserved.[br]
			The popup is generated 'on the fly' when the [cmd]popup[/cmd] command
			is called.[br]
			The 'item' keyword adds a menu item with visible <text> ,
			the optional <icon> and <command> as code to be executed when the item
			is clicked. <text> is a string that is evaluated at [cmd]popup[/cmd]
			call time and may contain identifiers and variables. If <expression>
			is given, it is evaluated at [cmd]popup[/cmd] call time and if the result
			is 0, the item is not shown in the physical popup.[br]
			The 'popup' keyword adds a submenu with visible <text> , the optional
			<icon> and a popup body that follows exactly the same syntax
			as the defpopup body. The <expression> has the same meaning as with the
			'item' keyword.[br]
			The 'extpopup' keyword adds a submenu with visible <text> , the optional
			icon and a popup body that is defined by the popup menu <name>. This
			basically allows to nest popup menus and define their parts separately.
			<icon> and <expression> have the same meaning as with the 'item' keyword.[br]
			The 'separator' keyword adds a straight line between items (separator).[br]
			The 'label' keywork adds a descriptive label that acts like a separator.[br]
			The 'prologue' keyword adds a <prologue_command> to be executed
			just before the popup is filled at [cmd]popup[/cmd] command call.[br]
			The 'epilogue' keyword adds an <epilogue_command> to be executed
			just after the popup has been filled at [cmd]popup[/cmd] command call.[br]
			There can be multiple prologue and epilogue commands: their execution order
			is undefined.[br]
			<icon> is always an [doc:image_id]image identifier[/doc].[br]
			<id> can be any string identifying the specific item: it will be
			possible to use [cmd]delpopupitem[/cmd] to remove it.[br]
			Please note that using this command inside the prologue , epilogue
			or item code of the modified popup menu is forbidden.
			In other words: self modification of popup menus is NOT allowed.[br]
			To remove a popup menu use this command with an empty body:[br]
			[example]
				defpopup(test){}
			[/example]
			This will remove the popup 'test' and free its memory.
			Popups have a special kind of local variables that have an extended lifetime:
			these are called "extended scope variables" and are described in the [doc:data_structures]Data structures documentation[/doc].[br]
			The syntax for these variables is:[br]
			[b]%:<variable name>[/b][br]
			These variables are visible during all the "visible lifetime" of the popup:
			from the [cmd]popup[/cmd] command call to the moment in that the user selects an item
			and the corresponding code is executed (substantially from a [cmd]popup[/cmd] call to the next one).[br]
			This allows you to pre-calculate data and conditions in the porologue of the popup
			and then use it in the item handlers or item conditions.[br]
		@seealso:
			[cmd]popup[/cmd], [cmd]delpopupitem[/cmd]
		@examples:
	*/

	ENTER_STACK_FRAME(c,"defpopup");

	if(*(c->m_ptr) != '(')return c->error(KviError_openParenthesisExpected);

	KviParameterList paramList;

	do {
		++(c->m_ptr);
		if(!parseFncSingleParam(c,&paramList))return false;
		__range_valid((*(c->m_ptr) == ')') || (*(c->m_ptr) == ','));
	} while(*(c->m_ptr) != ')');

	++(c->m_ptr);

	QString popupName = paramList.first() ? paramList.first()->ptr() : "";
	if(popupName.isEmpty())return c->error(KviError_missingPopupName);

	c->skipWhiteSpace();

	if(*(c->m_ptr) != '{')return c->error(KviError_openBraceExpected);

	KviPopupMenu * p = g_pPopupManager->getPopup(popupName);

	if(p->isLocked())return c->error(KviError_selfModificationNotAllowed);

	if(!c->hasSwitch('m'))p->doClear();

	if(!buildPopupMenu(c,p))
	{
		g_pPopupManager->removePopup(popupName);
		return false;
	}

	if(p->isEmpty())g_pPopupManager->removePopup(popupName);

	return c->leaveStackFrame();
}

bool KviUserParser::buildPopupMenu(KviCommand *c,KviPopupMenu *p)
{
	ENTER_STACK_FRAME(c,"kvirc::buildPopupMenu");
	__range_valid(*(c->m_ptr) == '{');

	++(c->m_ptr);
	c->skipWhiteSpace();


	while(*(c->m_ptr) && (*(c->m_ptr) != '}'))
	{
		if(*(c->m_ptr) == '#')skipComment(c);
		else {

#define EXTRACT_PARAM_LIST \
				++(c->m_ptr); \
				KviParameterList paramList; \
				if(!parseFncSingleParam(c,&paramList,true))return false; \
				if(*(c->m_ptr) == ',')++(c->m_ptr); \
				while(*(c->m_ptr) && (*(c->m_ptr) != ')')) \
				{ \
					if(!parseFncSingleParam(c,&paramList))return false; \
					__range_valid((*(c->m_ptr) == ')') || (*(c->m_ptr) == ',')); \
					if(*(c->m_ptr) == ',')++(c->m_ptr); \
				} \
				++(c->m_ptr); \
				c->skipWhiteSpace();

#define EXTRACT_EXPRESSION \
				KviStr expression; \
				if(*(c->m_ptr) == '(') \
				{ \
					char * expBegin = c->m_ptr; \
					if(!skipExpressionBody(c))return false; \
					expression.extractFromString(expBegin,c->m_ptr); \
					expression.cutLeft(1); \
					expression.stripWhiteSpace(); \
					expression.cutRight(1); \
					c->skipWhiteSpace(); \
				}

			if(kvi_strEqualCIN(c->m_ptr,"item",4))
			{
				c->m_ptr += 4;
				c->skipWhiteSpace();

				if(*(c->m_ptr) != '(')return c->error(KviError_openParenthesisExpected);
		
				EXTRACT_PARAM_LIST

				KviStr * pText = paramList.first();
				KviStr * pIcon = paramList.next();
				KviStr * pId = paramList.next();

				EXTRACT_EXPRESSION

				char * aux = c->m_ptr;
				if(!skipCommand(c))return false;

				KviStr body(aux,c->m_ptr);
				KviCommandFormatter::bufferFromBlock(body);

				KviPopupMenuItem * it = new KviPopupMenuItem(
					KviPopupMenuItem::Item,
					pText ? pText->ptr() : __tr("unnamed"),
					pIcon ? QString::fromUtf8(pIcon->ptr()) : QString::null,
					expression.ptr(),
					body.ptr(),
					0,
					pId ? QString::fromUtf8(pId->ptr()) : QString::null);

				p->addItem(it);
			} else if(kvi_strEqualCIN(c->m_ptr,"extpopup",8))
			{
				c->m_ptr += 8;
				c->skipWhiteSpace();

				if(*(c->m_ptr) != '(')return c->error(KviError_openParenthesisExpected);
				
				EXTRACT_PARAM_LIST

				KviStr * pText = paramList.first();
				KviStr * pName = paramList.next();
				KviStr * pIcon = paramList.next();
				KviStr * pId = paramList.next();

				if(!pName)return c->error(KviError_missingPopupName);
				if(pName->isEmpty())return c->error(KviError_missingPopupName);

				EXTRACT_EXPRESSION

				if(*(c->m_ptr) == ';')++(c->m_ptr);

				KviPopupMenuItem * it = new KviPopupMenuItem(
					KviPopupMenuItem::ExtMenu,
					pText ? pText->ptr() : __tr("unnamed"),
					pIcon ? QString::fromUtf8(pIcon->ptr()) : QString::null,
					expression.ptr(),
					pName->ptr(),
					0,
					pId ? QString::fromUtf8(pId->ptr()) : QString::null);

				p->addItem(it);

			} else if(kvi_strEqualCIN(c->m_ptr,"label",5))
			{
				c->m_ptr += 5;
				c->skipWhiteSpace();
				if(*(c->m_ptr) != '(')return c->error(KviError_openParenthesisExpected);
				EXTRACT_PARAM_LIST
				KviStr * pText = paramList.first();
				KviStr * pId = paramList.next();

				EXTRACT_EXPRESSION

				if(*(c->m_ptr) == ';')++(c->m_ptr);

				KviPopupMenuItem * it = new KviPopupMenuItem(KviPopupMenuItem::Label,
					pText ? pText->ptr() : __tr("unnamed"),
					QString::null, // icon
					expression.ptr(),
					QString::null, // code
					0, // popup
					pId ? QString(pId->ptr()) : QString::null);

				p->addItem(it);

			} else if(kvi_strEqualCIN(c->m_ptr,"popup",5))
			{
				c->m_ptr += 5;
				c->skipWhiteSpace();

				if(*(c->m_ptr) != '(')return c->error(KviError_openParenthesisExpected);
		
				EXTRACT_PARAM_LIST

				KviStr * pText = paramList.first();
				KviStr * pIcon = paramList.next();
				KviStr * pId = paramList.next();

				EXTRACT_EXPRESSION

				if(*(c->m_ptr) != '{')return c->error(KviError_openBraceExpected);

				KviPopupMenu * popup = new KviPopupMenu("submenu");
				if(!buildPopupMenu(c,popup))
				{
					delete popup;
					return false;
				}

				KviPopupMenuItem * it = new KviPopupMenuItem(
					KviPopupMenuItem::Menu,
					pText ? pText->ptr() : __tr("unnamed"),
					pIcon ? QString::fromUtf8(pIcon->ptr()) : QString::null,
					expression.ptr(),
					QString::null,
					popup,
					pId ? QString::fromUtf8(pId->ptr()) : QString::null);

				p->addItem(it);
			} else if(kvi_strEqualCIN(c->m_ptr,"prologue",8))
			{
				c->m_ptr += 8;
				c->skipWhiteSpace();
				KviStr szId;
				if(*(c->m_ptr) == '(')
				{
					EXTRACT_PARAM_LIST
					KviStr * pId = paramList.first();
					if(pId)szId = *pId;
				}

				char * aux = c->m_ptr;
				if(!skipCommand(c))return false;
				KviStr szTmp(aux,c->m_ptr);
				QString tmp(szTmp.ptr());
				KviCommandFormatter::bufferFromBlock(tmp);
				if(!tmp.isEmpty())
				{
					KviPopupMenuCodeSnippet * s = new KviPopupMenuCodeSnippet;
					s->m_szCode = tmp;
					if(!szId.isEmpty())s->m_szId = szId.ptr();
					p->addPrologue(s);
				}
			} else if(kvi_strEqualCIN(c->m_ptr,"epilogue",8))
			{
				c->m_ptr += 8;
				c->skipWhiteSpace();
				KviStr szId;
				if(*(c->m_ptr) == '(')
				{
					EXTRACT_PARAM_LIST
					KviStr * pId = paramList.first();
					if(pId)szId = *pId;
				}

				char * aux = c->m_ptr;
				if(!skipCommand(c))return false;

				KviStr szTmp(aux,c->m_ptr);
				QString tmp(szTmp.ptr());
				KviCommandFormatter::bufferFromBlock(tmp);
				if(!tmp.isEmpty())
				{
					KviPopupMenuCodeSnippet * s = new KviPopupMenuCodeSnippet;
					s->m_szCode = tmp;
					if(!szId.isEmpty())s->m_szId = szId.ptr();
					p->addEpilogue(s);
				}

			} else if(kvi_strEqualCIN(c->m_ptr,"separator",9))
			{
				c->m_ptr += 9;
				c->skipSpace();
				KviStr szId;
				if(*(c->m_ptr) == '(')
				{
					EXTRACT_PARAM_LIST
					KviStr * pId = paramList.first();
					if(pId)szId = *pId;
				}

				if(*(c->m_ptr) == ';')++(c->m_ptr);
				p->addItem(new KviPopupMenuItem(KviPopupMenuItem::Separator,QString::null,QString::null,QString::null,QString::null,0,szId.isEmpty() ? QString::null : QString(szId.ptr())));
			} else return c->error(KviError_itemPopupOrSeparatorExpected);
		}
		c->skipWhiteSpace();

	}

	if(!(*(c->m_ptr)))return c->error(KviError_missingClosingBrace);

	++(c->m_ptr);

	return c->leaveStackFrame();
}

bool KviUserParser::parseCmd_TIMER(KviCommand *c)
{
	ENTER_STACK_FRAME(c,"timer");

	/*
		@doc: timer
		@title:
			timer
		@type:
			command
		@short:
			Starts a timer
		@syntax:
			timer [-s] [-p] (<name>,<delay_in_msecs>[,<callback_param1>[,<callback_param2>[,...]]])
			{
				<callback_command>
			}
		@description:
			Starts a new timer named <name> with the specified delay (in milliseconds).[br]
			If a timer with the same name already exists, it is replaced by this one.[br]
			The timer periodically calls the specified <callback_command> code passing the
			eventual <callback_param> strings as positional parameters.[br]
			[b]The <callback_command> is evaluated at timer "shot" time and NOT while
			this command is being parsed. This means that the identifiers that you put
			inside <callback_command> will NOT have the current values.[/b]
			The values will be assigned at timer "shot" time.[br]
			This is a common scripters error and problem: if it is not clear, look at the examples below.[br]
			The timer is bound to the window in that this command is executed in.[br]
			If the window gets destroyed, the timer is stopped; unless the -p switch is used.[br]
			The -p switch causes the timer to be persistent across the application and exists until
			the last window has been closed: it is basically rebound to another (random) window when the
			original window is destroyed.[br]
			The -s switch cuases this timer to trigger only once: it will be automatically destroyed after that.[br]
			The time has an associated set of [doc:data_types]extended scope variables[/doc]:
			the variables that begin with "%:" have their life extended to the whole "life" of the timer.[br]
		@seealso:
			[cmd]killtimer[/cmd]
		@examples:
			[example]
				[comment]# Just a plain timer[/comment]
				timer(test,1000){ echo "Hello!"; }
				[comment]# Now watch the timer running[/comment]
				killtimer test
				[comment]# Single shot timer[/comment]
				timer -s (test,1000){ echo "This will fire only once!"; }
				[comment]# The call above is equivalent to[/comment]
				timer(test,1000){ echo "This will file only once!"; killtimer test; }
				[comment]# Callback parameters: consider the following code[/comment]
				%parameter = "some string value"
				echo "Before calling /timer \%parameter is \"%parameter\""
				timer -s (test,1000,%parameter){ echo "inside the callback \%parameter is \"%parameter\" but \$0 is \"$0\""; }
				[comment]# watch the timer running , and note the behaviour of the %parameter variable[/comment]
				killtimer test
				[comment]# Use the extended scope timer variables[/comment]
				timer(test,1000)
				{
					[comment]# Use the extended scope %:count variable to keep track[/comment]
					[comment]# of the times that this timer has been called[/comment]
					if("%:count" == "")%:count = 1
					else %:count++
					echo "This timer has fired %:count times"
					if(%:count == 10)killtimer test
				}
				[comment]# Use isTimer to check if the timer exists[/comment]
				echo $isTimer(test)
				[comment]# Repeat the command above after the 10th timeout...[/comment]
			[/example]
	*/

	c->skipWhiteSpace();
	if(*(c->m_ptr) != '(')return c->error(KviError_openParenthesisExpected);

	KviParameterList * paramList = new KviParameterList();
	paramList->setAutoDelete(true);

	if(!extractFunctionParameters(c,paramList))
	{
		delete paramList;
		return false;
	}

	KviStr * tName = paramList->safeFirst();
	KviStr * delay = paramList->safeNext();

	if(tName->isEmpty())
	{
		delete paramList;
		return c->error(KviError_notEnoughParameters,__tr("Missing timer name"));
	}

	if(delay->isEmpty())
	{
		delete paramList;
		return c->error(KviError_notEnoughParameters,__tr("Missing timeout delay"));
	}

	bool bOk;
	int iDelay = delay->toInt(&bOk);

	if(!bOk || (iDelay < 0))
	{
		delete paramList;
		return c->error(KviError_invalidParameter,__tr("Invalid timeout delay '%s'"),delay->ptr());
	}

	c->skipWhiteSpace();

	if(*(c->m_ptr) != '{')
	{
		delete paramList;
		return c->error(KviError_openBraceExpected);
	}

	const char * aux = c->m_ptr;
	if(!skipCommand(c))
	{
		delete paramList;
		return false;
	}

	KviStr cmd(aux,c->m_ptr);
	KviCommandFormatter::bufferFromBlock(cmd);

	KviStr szName = *tName; // copy it...we need it
	paramList->removeFirst();
	paramList->removeFirst();

	if(cmd.isEmpty())
	{
		c->warning(__tr2qs("Empty command block for timer '%s', timer not started"),szName.ptr());
		delete paramList;
	} else {
		unsigned int uFlags = c->hasSwitch('s') ? KVI_USER_TIMER_FLAG_SINGLE_SHOT : 0;
		if(c->hasSwitch('p'))uFlags |= KVI_USER_TIMER_FLAG_PERSISTENT;

		if(paramList->count() < 1)
		{
			delete paramList;
			paramList = 0;
		}

		// if paramList is non 0 , it will be destroyed by the timer manager

		if(!g_pTimerManager->addTimer(szName,c->window(),cmd,paramList,iDelay,uFlags))
		{
			c->warning(__tr2qs("Internal error: Unable to add timer %s, insufficient system resources"),szName.ptr());
		}
	}

	return c->leaveStackFrame();
}

bool KviUserParser::parseCmd_KILLTIMER(KviCommand *c)
{
	ENTER_STACK_FRAME(c,"killtimer");

	/*
		@doc: killtimer
		@title:
			killtimer
		@type:
			command
		@short:
			Stops a timer
		@syntax:
			killtimer <name>
		@description:
			Kills a timer named <name>
		@seealso:
			[cmd]timer[/cmd], [fnc]$isTimer[/fnc]
	*/

	KviStr szName;
	if(!parseCmdFinalPart(c,szName))return false;
	if(!g_pTimerManager->delTimer(szName.ptr()))c->warning(__tr2qs("No such timer (%s)"),szName.ptr());
	return c->leaveStackFrame();
}

bool KviUserParser::parseCmd_EVENT(KviCommand *c)
{
	ENTER_STACK_FRAME(c,"event");
	/*
		@doc: event
		@title:
			event
		@type:
			command
		@short:
			Adds a new event handler
		@syntax:
			event(<event_name>,<handler_name>)
			{
				<implementation>
			}
		@description:
			Adds the handler <handler_name> with <implementation> to
			the list of handlers for the event <event_name>.[br]
			If the <implementation> is empty
			the handler <handler_name> is removed from the handler
			list instead of being added.[br]
			The <event_name> may be one of the kvirc-builtin event names
			or a numeric code (from 0 to 999) of a raw server message.[br]
		@seealso:
			[cmd]eventctl[/cmd]
	*/

	c->skipWhiteSpace();
	if(*(c->m_ptr) != '(')return c->error(KviError_openParenthesisExpected);

	KviParameterList paramList;
	paramList.setAutoDelete(true);

	if(!extractFunctionParameters(c,&paramList))return false;

	KviStr * eveName = paramList.safeFirst();
	KviStr * hndName = paramList.safeNext();

	c->skipWhiteSpace();

	if(*(c->m_ptr) != '{')return c->error(KviError_openBraceExpected);
	const char * aux = c->m_ptr;
	if(!skipCommand(c))return false;

	KviStr cmd(aux,c->m_ptr);
	KviCommandFormatter::bufferFromBlock(cmd);

	bool bOk;
	int iNumber = eveName->toInt(&bOk);
	bool bIsRaw = (bOk && (iNumber >= 0) && (iNumber < 1000));

	if(bIsRaw)
	{
		if(!KviKvsEventManager::instance()->isValidRawEvent(iNumber))
		{
			c->warning(__tr2qs("No such event (%s)"),eveName->ptr());
			return c->leaveStackFrame();
		}
	} else {
		iNumber = KviKvsEventManager::instance()->findAppEventIndexByName(eveName->ptr());
		if(!KviKvsEventManager::instance()->isValidAppEvent(iNumber))
		{
			c->warning(__tr2qs("No such event (%s)"),eveName->ptr());
			return c->leaveStackFrame();
		}
	}

	if(cmd.isEmpty())
	{
		if(bIsRaw)
		{
			if(!KviKvsEventManager::instance()->removeScriptRawHandler(iNumber,hndName->ptr()))
				c->warning(__tr2qs("No handler '%s' for raw numeric event '%d'"),hndName->ptr(),iNumber);
		} else {
			if(!KviKvsEventManager::instance()->removeScriptAppHandler(iNumber,hndName->ptr()))
				c->warning(__tr2qs("No handler '%s' for event '%s'"),hndName->ptr(),eveName->ptr());
		}
	} else {
		if(bIsRaw)
		{
			// remove the old handler
			KviKvsEventManager::instance()->removeScriptRawHandler(iNumber,hndName->ptr());
			QString contextName;
			KviQString::sprintf(contextName,"RawEvent%d::%s",iNumber,hndName->ptr());
			KviKvsScriptEventHandler * pHandler = new KviKvsScriptEventHandler(hndName->ptr(),contextName,new QString(cmd.ptr()));
			KviKvsEventManager::instance()->addRawHandler(iNumber,pHandler);
		} else {
			// remove the old handler
			KviKvsEventManager::instance()->removeScriptAppHandler(iNumber,hndName->ptr());
			QString contextName;
			KviQString::sprintf(contextName,"%s::%s",eveName->ptr(),hndName->ptr());
			KviKvsScriptEventHandler * pHandler = new KviKvsScriptEventHandler(hndName->ptr(),contextName,new QString(cmd.ptr()));
			KviKvsEventManager::instance()->addAppHandler(iNumber,pHandler);
		}
	}
	return c->leaveStackFrame();
}

bool KviUserParser::parseCmd_EVENTCTL(KviCommand *c)
{
	ENTER_STACK_FRAME(c,"eventctl");
	/*
		@doc: eventctl
		@title:
			eventctl
		@type:
			command
		@short:
			Controls the execution of event handlers
		@syntax:
			eventctl [-u] [-e] [-d] <event_name> <handler_name> [parameters]
		@description:
			Performs control actions on the handler <handler_name> for
			the event <event_name>.
			Without any switch it triggers the handler for testing purposes,
			eventually passing [parameters].[br]
			With the -u switch the handler <handler_name> is unregistered.[br]
			With the -d swtich is is disabled (so it is never executed)
			and with -e is enabled again.[br]
			The <event_name> may be one of the kvirc-builtin event names
			or a numeric code (from 0 to 999) of a raw server message.[br]
		@seealso:
			[cmd]event[/cmd]
	*/

	KviStr eveName;
	if(!parseCmdSingleToken(c,eveName))return false;
	KviStr hndName;
	if(!parseCmdSingleToken(c,hndName))return false;

	KviParameterList params;

	while(*(c->m_ptr) && (*(c->m_ptr) != '\n') && (*(c->m_ptr) != ';'))
	{
		KviStr tmp;
		if(!parseCmdSingleToken(c,tmp))return false;
		params.append(new KviStr(tmp));
	}

	bool bOk;
	int iNumber = eveName.toInt(&bOk);
	bool bIsRaw = (bOk && (iNumber >= 0) && (iNumber < 1000));

	if(bIsRaw)
	{
		if(!KviKvsEventManager::instance()->isValidRawEvent(iNumber))
		{
			c->warning(__tr2qs("No such event (%s)"),eveName.ptr());
			return c->leaveStackFrame();
		}
	} else {
		iNumber = KviKvsEventManager::instance()->findAppEventIndexByName(eveName.ptr());
		if(!KviKvsEventManager::instance()->isValidAppEvent(iNumber))
		{
			c->warning(__tr2qs("No such event (%s)"),eveName.ptr());
			return c->leaveStackFrame();
		}
	}



	if(c->hasSwitch('u'))
	{
		// unregister it
		if(bIsRaw)
		{
			if(!KviKvsEventManager::instance()->removeScriptRawHandler(iNumber,hndName.ptr()))
				c->warning(__tr2qs("No handler '%s' for raw numeric event '%d'"),hndName.ptr(),iNumber);
		} else {
			if(!KviKvsEventManager::instance()->removeScriptAppHandler(iNumber,hndName.ptr()))
				c->warning(__tr2qs("No handler '%s' for event '%s'"),hndName.ptr(),eveName.ptr());
		}
	} else if(c->hasSwitch('e') || c->hasSwitch('d'))
	{
		// enable it
		if(bIsRaw)
		{
			if(!KviKvsEventManager::instance()->enableScriptRawHandler(iNumber,hndName.ptr(),c->hasSwitch('e')))
				c->warning(__tr2qs("No handler '%s' for raw numeric event '%d'"),hndName.ptr(),iNumber);
		} else {
			if(!KviKvsEventManager::instance()->enableScriptAppHandler(iNumber,hndName.ptr(),c->hasSwitch('e')))
				c->warning(__tr2qs("No handler '%s' for event '%s'"),hndName.ptr(),eveName.ptr());
		}
	} else {
		// trigger it
		KviKvsScriptEventHandler * h;
		QString code;
		
		if(bIsRaw)
		{
			h = KviKvsEventManager::instance()->findScriptRawHandler(iNumber,hndName.ptr());
		} else {
			h = KviKvsEventManager::instance()->findScriptAppHandler(iNumber,hndName.ptr());
		}

		if(h)
		{
			KviCommand cmd(h->code().utf8().data(),c->window(),c);
			cmd.setParams(&params,false); // do not transfer param ownership
			if(!parseCommand(&cmd))
			{
				if(cmd.hasError())printError(&cmd);
			}
		} else c->warning(__tr2qs("No handler '%s' for event '%s'"),hndName.ptr(),eveName.ptr());
	}

	return c->leaveStackFrame();
}



bool KviUserParser::parseCmd_WHOIS(KviCommand *c)
{
	/*
		@doc: whois
		@type:
			command
		@title:
			whois
		@syntax:
			whois [server] <nickname>
		@short:
			Requests user information
		@description:
			Requests information about an irc user specified by <nickname>.[br]
			If [server] is specified, the request is directed to that one. [server]
			may be a nickname so that the request is redirected to the server that
			the user with that nickname is connected to.[br]
			An useful trick is to use the following syntax:[br]
			whois <nickname> <nickname>[br]
			This will requests information about the user <nickname> to the server that he is currently
			connected to; returning the idle time among the other common information.[br]
			This command is [doc:connection_dependant_commands]connection dependant[/doc].
		@examples:
			[example]
			[comment]# Query info about Pragma[/comment]
			whois Pragma
			[comment]# Get the Pragma idle time[/comment]
			whois Pragma Pragma
			[/example]
		@seealso:
			[cmd]whowas[/cmd]
	*/

	ENTER_STACK_FRAME(c,"whois");

	KviStr text;
	if(!parseCmdFinalPart(c,text))return false;

	if(!c->window()->console())return c->noIrcContext();
	if(!c->window()->connection())return c->notConnectedToServer();
	QString tmp = QString::fromUtf8(text.ptr());
	QCString tmp2 = c->window()->connection()->encodeText(tmp);
	if(tmp2.data())
	{
		if(!(c->window()->connection()->sendFmtData("WHOIS %s",tmp2.data())))
			return c->notConnectedToServer();
	}

	return c->leaveStackFrame();
}

bool KviUserParser::parseCmd_WHOWAS(KviCommand *c)
{
	/*
		@doc: whowas
		@type:
			command
		@title:
			whowas
		@syntax:
			whowas <nickname>
		@short:
			Requests user information
		@description:
			Requests information about an irc user specified by <nickname>.[br]
			Whowas requests "history" records. See [doc:rfc2812]RFC2812[/doc] for more info.[br]
			This command is [doc:connection_dependant_commands]connection dependant[/doc].
		@examples:
			[example]
			[comment]# Query past info about Pragma[/comment]
			whowas Pragma
			[/example]
		@seealso:
			[cmd]whois[/cmd]
	*/

	ENTER_STACK_FRAME(c,"whowas");

	KviStr text;
	if(!parseCmdFinalPart(c,text))return false;

	if(!c->window()->console())return c->noIrcContext();
	if(!c->window()->connection())return c->notConnectedToServer();
	QString tmp = QString::fromUtf8(text.ptr());
	QCString tmp2 = c->window()->connection()->encodeText(tmp);
	if(tmp2.data())
	{
		if(!(c->window()->connection()->sendFmtData("WHOWAS %s",tmp2.data())))
			return c->notConnectedToServer();
	}

	return c->leaveStackFrame();
}


bool KviUserParser::parseCmd_AVATAR(KviCommand *c)
{
	// FIXME: #warning "A WAY TO OMIT THE TARGET ON CHANNELS AND QUERIES ?"
	/*
		@doc: avatar
		@type:
			command
		@title:
			avatar
		@syntax:
		@short:
			Obsolete
		@description:
			This command is obsolete and has been replaced by [cmd]avatar.set[/cmd]
			and [cmd]avatar.notify[/cmd]. It is provided only for backward compatibility
			and will be removed in future versions.
	*/

	ENTER_STACK_FRAME(c,"avatar");

	KviStr target,avatar;
	if(!parseCmdSingleToken(c,target))return false;
	if(!parseCmdFinalPart(c,avatar))return false;

	if(!c->window()->console())return c->noIrcContext();
	if(!c->window()->connection())return c->notConnectedToServer();

	if(target.isEmpty())
	{
		c->warning(__tr2qs("Missing target, no action taken"));
		return c->leaveStackFrame();
	}

	int iTimeout = (int)KVI_OPTION_UINT(KviOption_uintAvatarOfferTimeoutInSecs);
	if(c->hasSwitch('t'))
	{
		KviStr szTimeout;
		if(c->getSwitchValue('t',szTimeout))
		{
			if(szTimeout.isUnsignedNum())iTimeout = szTimeout.toInt();
			else c->warning(__tr2qs("Invalid timeout specified, using default"));
		}
	}

	KviStr absPath;

	if(c->hasSwitch('u'))
	{
		// Unset the current avatar
		KviIrcUserEntry * e = c->window()->connection()->userDataBase()->find(
								c->window()->connection()->currentNickName());
		if(e)
		{
			e->setAvatar(0);
			c->window()->console()->avatarChanged(0,
						c->window()->connection()->userInfo()->nickName(),
						c->window()->connection()->userInfo()->userName(),
						c->window()->connection()->userInfo()->hostName(),
						0);
		} else debug("oops...I'm not in the user database ?");
		avatar = "";
	} else {
		if(avatar.hasData())
		{
			bool bIsUrl = kvi_strEqualCIN(avatar.ptr(),"http://",7) && (avatar.len() > 7);
			// new avatar specified...try to load it
			KviIrcUserEntry * e = c->window()->connection()->userDataBase()->find(c->window()->connection()->currentNickName());
			if(e)
			{
				KviAvatar * av = g_pIconManager->getAvatar(QString::null,avatar.ptr());
				if(av)
				{
					// Ok...got it...
					e->setAvatar(av);
					c->window()->console()->avatarChanged(av,
								c->window()->connection()->userInfo()->nickName(),
								c->window()->connection()->userInfo()->userName(),
								c->window()->connection()->userInfo()->hostName(),
								QString::null);

					avatar = av->name();
					if(!av->isRemote())absPath = av->localPath();
				} else {
					if(bIsUrl)
					{
						// This is an url, and we don't have a cached copy for now
						KviStr szLocalFilePath;
						KviStr szLocalFile = avatar;
						g_pIconManager->urlToCachedFileName(szLocalFile);
						g_pApp->getLocalKvircDirectory(szLocalFilePath,KviApp::Avatars,szLocalFile.ptr());

						szLocalFilePath.replaceAll("\\","\\\\");

						KviStr szCommand = "http.get -w=nm ";
						szCommand += avatar;
						szCommand += " ";
						szCommand += szLocalFilePath;

						if(g_pUserParser->parseCommandBuffer(szCommand.ptr(),c->window()->console()))
						{
							g_pApp->setAvatarOnFileReceived(c->window()->console(),
								avatar.ptr(),
								c->window()->connection()->userInfo()->nickName(),
								c->window()->connection()->userInfo()->userName(),
								c->window()->connection()->userInfo()->hostName());
						} else {
							c->warning(__tr2qs("Can't set the current avatar to '%s' (failed to start the http transfer): no message sent"),avatar.ptr());
							return c->leaveStackFrame();
						}
					} else {
						c->warning(__tr2qs("Can't set the current avatar to '%s' (can't load the image): no message sent"),avatar.ptr());
						return c->leaveStackFrame();
					}
				}
			} else debug("oops...I'm not in the user database ?");
		} else {
			KviAvatar * avt = c->window()->console()->currentAvatar();
			if(avt)
			{
				avatar = avt->name();
				if(!avt->isRemote())absPath = avt->localPath();
			} // else have no current avatar
		}
	}

	KviSharedFile * o = 0;

	if(absPath.hasData() && avatar.hasData())
	{
		bool bTargetIsChan = (target.contains('#') || target.contains('&') || target.contains('!'));

		if(bTargetIsChan)o = g_pSharedFilesManager->lookupSharedFile(avatar.ptr(),0);
		else {
			KviIrcMask u(target.ptr());
			o = g_pSharedFilesManager->lookupSharedFile(avatar.ptr(),&u);
		}

		if(!o)
		{
// FIXME: #warning "OPTION FOR PERMANENT OR TIMEDOUT OFFER...TIMEDOUT WOULD ALSO NEED TO EXTEND EXISTING OFFERS LIFETIME"

			KviStr szUserMask(KviStr::Format,"%s!*@*",bTargetIsChan ? "*" : target.ptr());

			o = g_pSharedFilesManager->addSharedFile(avatar.ptr(),absPath.ptr(),szUserMask.ptr(),iTimeout);
			if(!o)
			{
				// Don't delete o...it has been already deleted by g_pFileTrader
				c->warning(__tr2qs("Can't add a file offer for file %s (huh ? file not readable ?)"),absPath.ptr());
				return c->leaveStackFrame();
			}

			if(_OUTPUT_VERBOSE)
			{
				c->window()->output(KVI_OUT_SYSTEMMESSAGE,__tr2qs("Added %d secs file offer for file %s (%s) and receiver %s"),
						iTimeout,o->absFilePath().latin1(),avatar.ptr(),o->userMask().latin1());
			}
		}
	}

	if(!c->hasSwitch('q'))
	{
		c->window()->output(KVI_OUT_AVATAR,__tr2qs("Notifying avatar '%s' to %s"),avatar.ptr(),target.ptr());
	}

	if(avatar.hasData())
	{
		if(o)
		{
			c->window()->connection()->sendFmtData("NOTICE %s :%cAVATAR %s %u%c",target.ptr(),0x01,
					avatar.ptr(),o->fileSize(),0x01);
		} else {
			c->window()->connection()->sendFmtData("NOTICE %s :%cAVATAR %s%c",target.ptr(),0x01,
					avatar.ptr(),0x01);
		}
	} else {
		c->window()->connection()->sendFmtData("NOTICE %s :%cAVATAR%c",target.ptr(),0x01,0x01);
	}

	return c->leaveStackFrame();
}


bool KviUserParser::parseCmd_AWHOIS(KviCommand *c)
{
	/*
		@doc: awhois
		@type:
			command
		@title:
			awhois
		@syntax:
			awhois [-i] (<nickname>[,<magic>])
			{
				<callback command>
			}
		@short:
			Asynchronous WHOIS
		@description:
			AWHOIS stands for Asynchronous WHOIS. It is used to obtain data for a specified
			irc user (designated by <nickname>). This command sends a WHOIS query to the
			server and silently awaits the sequence of replies. When the "End of WHOIS" message
			is received from server the <callback command> is executed passing the WHOIS
			information as positional parameters.[br]
			The <magic> string is an optional string to be evaluated at AWHOIS execution time.
			It is passed as the last positional parameter.[br]
			Callback command parameters:[br]
			$0 = nickname[br]
			$1 = username[br]
			$2 = hostname[br]
			$3 = realname (may be empty)[br]
			$4 = server[br]
			$5 = idle time (may be empty)[br]
			$6 = signon time (may be empty)[br]
			$7 = channels (may be empty)[br]
			$8 = server that provided the information[br]
			$9 = special information (may be empty)[br]
			$10 = magic string evaluated at awhois call (may be empty)[br]
			If the -i switch is specified , the whois message is sent to the server
			that the <nickname> user is connected to; in this way you will probably
			get the idle time of the user too.[br]
			If the server replies with a "No such nick/channel error message" the
			<callback command> will be still triggered , but will have all the parameters
			empty with the exception of $0.[br]
			If the connection gets interrupted before all the information have been received,
			the <callback command> will never be triggered.[br]
			This command is [doc:connection_dependant_commands]connection dependant[/doc].[br]
		@examples:
			[example]
			awhois(pragma){ echo $0-; }
			[/example]
	*/

	ENTER_STACK_FRAME(c,"awhois");

	if(!c->window()->console())return c->noIrcContext();
	if(!c->window()->connection())return c->notConnectedToServer();

	KviParameterList paramList;
	paramList.setAutoDelete(true);

	KviStr cmd;

	if(!parseCallbackCommand(c,&paramList,&cmd))return false;

	KviStr * nick = paramList.safeFirst();
	KviStr * magic = paramList.safeNext();

	KviAsyncWhoisInfo * info = new KviAsyncWhoisInfo;
	info->szCode = cmd.ptr();
	info->szMagic = magic->ptr();
	info->szNick = nick->hasData() ? QString(nick->ptr()) : c->window()->connection()->currentNickName();
	info->pWindow = c->window();

	c->window()->connection()->asyncWhoisData()->add(info);
	if(c->hasSwitch('i'))c->window()->connection()->sendFmtData("WHOIS %s %s",info->szNick.latin1(),info->szNick.latin1());
	else c->window()->connection()->sendFmtData("WHOIS %s",info->szNick.latin1());

	return c->leaveStackFrame();
}


bool KviUserParser::parseCmd_ALIAS(KviCommand *c)
{
	ENTER_STACK_FRAME(c,"alias");
	/*
		@doc: alias
		@title:
			alias
		@type:
			command
		@short:
			Adds a new alias or modifies an existing one
		@syntax:
			alias(<alias_name>)
			{
				<implementation>
			}
		@description:
			Adds the alias <alias_name> with <implementation>.
			If the alias was already existing, it is replaced with the
			new implementation.[br]
			If the <implementation> is empty
			the alias <alias_name> is removed instead of being added.
		@examples:
			[example]
				[comment]# Add the alias j[/comment]
				alias(j){ [cmd]join[/cmd] $0; }
				[comment]# Remove the alias j[/comment]
				alias(j){}
			[/example]
	*/

	c->skipWhiteSpace();
	if(*(c->m_ptr) != '(')return c->error(KviError_openParenthesisExpected);

	KviParameterList paramList;
	paramList.setAutoDelete(true);

	if(!extractFunctionParameters(c,&paramList))return false;
// FIXME: #warning "SHould ensure that the alias name does not contain invalid chars... '.' dot for example"
	KviStr * aliName = paramList.safeFirst();

	c->skipWhiteSpace();

	if(*(c->m_ptr) != '{')return c->error(KviError_openBraceExpected);
	const char * aux = c->m_ptr;
	if(!skipCommand(c))return false;

	KviStr cmd(aux,c->m_ptr);
	KviCommandFormatter::bufferFromBlock(cmd);

	if(cmd.isEmpty())KviKvsAliasManager::instance()->remove(aliName->ptr());
	else {
		KviKvsScript * a = new KviKvsScript(aliName->ptr(),new QString(cmd.ptr()));
		KviKvsAliasManager::instance()->add(aliName->ptr(),a);
	}
	return c->leaveStackFrame();
}

bool KviUserParser::parseCmd_OPTION(KviCommand *c)
{
	/*
		@doc: option
		@type:
			command
		@title:
			option
		@syntax:
			option [<optName> <optValue>]
		@short:
			Sets an internal option
		@description:
			Sets an internal option named <optName> to the value <optValue>. The value
			must be appropriate for the type of option that you're going to set.[br]
			With no parameters this command lists all the available option names
			sorted by option type.[br] The possible option types are:[br]
			[b]Boolean[/b]: <optValue> must be "1" or "0"[br]
			[b]String[/b]: <optValue> can be any string[br]
			[b]StringList[/b]: <optValue> must be a comma separated list of strings (eventually empty)[br]
			[b]Color[/b]: <optValue> must have the format #RRGGBB where R G and B are hex digits[br]
			[b]Font[/b]: <optValue> is a comma separated list of font properties:
				<family>,<pointsize>,<style>,<charset>,<weight>,<flags>. <family> is the font face name
				("arial" , "helvetica" etc..), <pointsize> is more or less the indicator of the size of the font
				(try sth like 12,14), <style> is an integer that indicates the font style (fixed for example),
				<charset> is an integer that indicates the charset of the font (maybe one day I'll document it better),
				<weight> indicates the (ehm) weight of the font (50,100,150...matches Bold,DemiBold etc..),
				<flags> is a combination of 'b','i','u','s,' and 'f' that match respectively bold,italic
				underline,strikeout and fixed.[br]
			[b]MessageType[/b]: <optValue> is a comma separated list of message type properties:
				<icon>,<foreground>,<background>,<logBoolean>,<level>: <icon> is the index of the internal
				small icon that has to be shown with the message type, <foreground> is an integer
				indicating the mirc color to be used for the message type text (0-15),
				<background> is similar to foreground and accepts also the value of 100 that means transparent.
				<logBoolean> is a boolean value (0/1) that indicates wheter this message type has to be logged or not.
				<level> is the message level (actually from 0 to 5).[br]
			[b]Rectangle[/b]: <optValue> is a comma separated list of integer values that indicate <x>,<y>,<width> and <height>.[br]
			[b]Pixmap[/b]: <optValue> must be an ABSOLUTE path of the image that you want to load.[br]
			[b]Int[/b]: <optValue> must be an integer.[br]
			[b]Uint[/b]: <optValue> must be an UNSIGNED positive integer.[br]
			Almost all the options available in the option dialog can be set by this command.[br]
			Some GUI options might require a KVIrc restart to work properly (altough I've tried to avoid that when possible).
		@examples:
			[example]
				[comment]# List available options[/comment]
				option
				[comment]# Set the mdi mananger background image[/comment]
				option pixmapMdiBackground /home/pragma/myback1.png
				[comment]# Set the frame caption text[/comment]
				option stringFrameCaption KVIrc rulez!
				[comment]# Enable verbose mode[/comment]
				option boolBeVerbose 1
				[comment]# Set the irc view font[/comment]
				option fontIrcView helvetica,24,5,1,50
			[/example]
		@seealso:
			[fnc]$option[/fnc]()
	*/

	ENTER_STACK_FRAME(c,"option");

	KviStr key,value;
	if(!parseCmdSingleToken(c,key))return false;
	if(!parseCmdFinalPart(c,value))return false;

	if(key.isEmpty())
	{
		// list available options
		g_pApp->listAvailableOptions(c->window());
	} else {
		if(!g_pApp->setOptionValue(key.ptr(),value.ptr()))c->warning(__tr2qs("Option setting error: Unknown option or invalid value for option type"));
	}

	return c->leaveStackFrame();
}

bool KviUserParser::parseCmd_HALT(KviCommand *c)
{
	/*
		@doc: halt
		@type:
			command
		@title:
			halt
		@syntax:
			halt
		@short:
			Stops the execution of the current script
		@description:
			Stops the execution of the current script.[br]
			Note that this is different from [cmd]return[/cmd]: halt doesn't
			set the return value (use [cmd]setreturn[/cmd] for that) and
			in the event handlers this command may have additional semantics:
			refer to the specific event documentation for more info.[br]
			In fact , this command should be used only in the event handlers,
			use [cmd]return[/cmd] in any other context.[br]
	*/
	c->setHaltEncountered();
	return false;
}

bool KviUserParser::parseCmd_RFC2812WRAPPER(KviCommand *c)
{
	/*
		@doc: rfc2821wrappers
		@title:
			RFC2821 Wrappers
		@type:
			generic
		@short:
			Wrappers for rfc2821 commands
		@body:
			Some less used KVIrc commands are implemented as simple wrappers
			around the standard [doc]rfc2821[/doc] commands.[br]
			For example, the /OPER command will be used only by server operators
			and probably only once per connection.[br]
			These commands should only be needed by experienced IRC users.
			Due to this fact, these commands have no special kind of parsing performed by KVIrc;
			the parameters are extracted, the identifiers are parsed normally
			and are sent to the server without any semantic check.[br]
			This means that while OPER requires two parameters, KVIrc will not
			complain if you send a single parameter only or use ten parameters.[br]
			The exact parameter syntax/semantic checking is left to you.[br]
			This also means that if some of these commands accept parameters with spaces,
			it is left to you to add the leading ':' before the last parameter.[br]
			You should refer to [doc]rfc2812[/doc] if any of these commands do not work properly for you.[br]
	*/

	/*
		@doc: who
		@type:
			command
		@title:
			who
		@syntax:
			who {[mask] | [channel]}
		@short:
			Requests WHO information
		@description:
			Requests WHO information about the specified user or channel.[br]
			This command is a [doc:rfc2821wrappers]RFC2821 command wrapper[/doc]; see that document for more information.[br]
	*/

	/*
		@doc: oper
		@type:
			command
		@title:
			oper
		@syntax:
			oper <name> <password>
		@short:
			Requests IRC operator status
		@description:
			Requests IRC operator status.[br]
			This command is a [doc:rfc2821wrappers]RFC2821 command wrapper[/doc]; see that document for more information.[br]
	*/

	/*
		@doc: service
		@type:
			command
		@title:
			service
		@syntax:
			service <nickname> <reserved> <distribution> <type> <reserved> [:]<info>
		@short:
			Registers a new service
		@description:
			Registers a new IRC service: this is for IRC operators only.[br]
			This command is a [doc:rfc2821wrappers]RFC2821 command wrapper[/doc]; see that document for more information.[br]
			WARNING: the <info> parameter may contain spaces: in that case you MUST add the leading ':' character.[br]
	*/

	/*
		@doc: squit
		@type:
			command
		@title:
			squit
		@syntax:
			squit <server> [:]<comment>
		@short:
			Disconnects a server socket (ircops only)
		@description:
			Disconnects a server socket: this is for IRC operators only.[br]
			This command is a [doc:rfc2821wrappers]RFC2821 command wrapper[/doc]; see that document for more information.[br]
			WARNING: the <comment> parameter may contain spaces: in that case you MUST add the leading ':' character
			before.[br]
	*/


	/*
		@doc: list
		@type:
			command
		@title:
			list
		@syntax:
			list [ <channel> *( "," <channel> ) [ <target> ] ]
		@short:
			Lists channels
		@description:
			List channels.[br]
			The syntax of the parameters of this command is different on many servers.[br]
			This command is a [doc:rfc2821wrappers]RFC2821 command wrapper[/doc]; see that document for more information.[br]
	*/

	/*
		@doc: invite
		@type:
			command
		@title:
			invite
		@syntax:
			invite <nickname> <channel>
		@short:
			Invites an user to a channel
		@description:
			Invites the specified user to the <channel>.[br]
			This command is a [doc:rfc2821wrappers]RFC2821 command wrapper[/doc]; see that document for more information.[br]
	*/

	/*
		@doc: motd
		@type:
			command
		@title:
			motd
		@syntax:
			motd [target server]
		@short:
			Requests the Message of the day
		@description:
			Requests the Message of the day from the specified server or the current server if no [target server] is specified.[br]
			This command is a [doc:rfc2821wrappers]RFC2821 command wrapper[/doc]; see that document for more information.[br]
	*/

	/*
		@doc: lusers
		@type:
			command
		@title:
			lusers
		@syntax:
			lusers [ <mask> [ <target> ] ]
		@short:
			Requests statistics about users on IRC
		@description:
			The LUSERS command is used to get statistics about the size of the
			IRC network.[br]If no parameter is given, the reply will be about the
			whole net.[br]If a <mask> is specified, then the reply will only
			concern the part of the network formed by the servers matching the
			mask.[br]Finally, if the <target> parameter is specified, the request
			is forwarded to that server which will generate the reply.[br]
			This command is a [doc:rfc2821wrappers]RFC2821 command wrapper[/doc]; see that document for more information.[br]
	*/

	/*
		@doc: version
		@type:
			command
		@title:
			version
		@syntax:
			version [target server]
		@short:
			Requests the version of a server
		@description:
			Requests the version information from the specified server or the current server if no [target server] is specified.[br]
			This command is a [doc:rfc2821wrappers]RFC2821 command wrapper[/doc]; see that document for more information.[br]
	*/

	/*
		@doc: stats
		@type:
			command
		@title:
			stats
		@syntax:
			stats [query [target server]]
		@short:
			Requests the stats info from  a server
		@description:
			Requests the stats information from the specified server or the current server if no [target server] is specified.[br]
			The query parameter is usually a letter indicating the type of the query requested: some query letters
			are defined by [doc]rfc2812[/doc] and some others are server implementation dependant.[br]
			This command is a [doc:rfc2821wrappers]RFC2821 command wrapper[/doc]; see that document for more information.[br]
	*/

	/*
		@doc: sockets
		@type:
			command
		@title:
			sockets
		@syntax:
			sockets [[remote server] server mask]
		@short:
			Requests the sockets info from a server
		@description:
			Requests the sockets information from the specified server or the current server if no [remote server] is specified.[br]
			The reply can be filtered by <server mask>.[br]
			This command is a [doc:rfc2821wrappers]RFC2821 command wrapper[/doc]; see that document for more information.[br]
	*/

	/*
		@doc: time
		@type:
			command
		@title:
			time
		@syntax:
			time [target server]
		@short:
			Requests the current local time info from a server
		@description:
			Requests time information from the specified server or the current server if no [target server] is specified.[br]
			This command is a [doc:rfc2821wrappers]RFC2821 command wrapper[/doc]; see that document for more information.[br]
	*/

	/*
		@doc: trace
		@type:
			command
		@title:
			trace
		@syntax:
			trace <target>
		@short:
			Requests trace information about a specified machine in the network
		@description:
			Requests trace information about a specified machine in the network.[br]
			This command is a [doc:rfc2821wrappers]RFC2821 command wrapper[/doc]; see that document for more information.[br]
	*/


	/*
		@doc: pass
		@type:
			command
		@title:
			pass
		@syntax:
			pass <password>
		@short:
			Sends a password to the server
		@description:
			Sends a password to the server: usually KVIrc takes care of it , but well.. this is useful with bouncers.[br]
			This command is a [doc:rfc2821wrappers]RFC2821 command wrapper[/doc]; see that document for more information.[br]
	*/

	/*
		@doc: admin
		@type:
			command
		@title:
			admin
		@syntax:
			admin [target server]
		@short:
			Requests the admin info from a server
		@description:
			Requests admin information from the specified server or the current server if no [target server] is specified.[br]
			This command is a [doc:rfc2821wrappers]RFC2821 command wrapper[/doc]; see that document for more information.[br]
	*/


	/*
		@doc: info
		@type:
			command
		@title:
			info
		@syntax:
			info [target server]
		@short:
			Requests the server daemon info from a server
		@description:
			Requests server daemon information from the specified server or the current server if no [target server] is specified.[br]
			This command is a [doc:rfc2821wrappers]RFC2821 command wrapper[/doc]; see that document for more information.[br]
	*/

	/*
		@doc: squery
		@type:
			command
		@title:
			squery
		@syntax:
			squery <target> :<text>
		@short:
			Sends a message to a service
		@description:
			Sends a message to a service in a form similar to [cmd]privmsg[/cmd].[br]
			This command is a [doc:rfc2821wrappers]RFC2821 command wrapper[/doc]; see that document for more information.[br]
	*/

	/*
		@doc: codepage
		@type:
			command
		@title:
			codepage
		@syntax:
			codepage <encoding name>
		@short:
			Tries to set the codepage on server
		@description:
			This is a not-widely implemented extension
			that allows the user to set the codepage mapping
			on server.
			This command is a [doc:rfc2821wrappers]RFC2821 command wrapper[/doc]; see that document for more information.[br]
	*/

// NAME CLASH!
//	/*
//		@doc: connect
//		@type:
//			command
//		@title:
//			connect
//		@syntax:
//			connect [target server]
//		@short:
//			Requests the current local time info from a server
//		@description:
//			Requests time information from the specified server or the current server if no [target server] is specified.[br]
//			This command is a [doc:rfc2821wrappers]RFC2821 command wrapper[/doc]; see that document for more information.[br]
//	*/



	KviStr contextName(KviStr::Format,"%s (RFC2812 wrapper)",c->currentEntity());
	KviStr command = c->currentEntity();

	ENTER_STACK_FRAME(c,contextName.ptr());

	KviStr text;
	if(!parseCmdFinalPart(c,text))return false;

	if(!c->window()->console())return c->noIrcContext();
	if(!c->window()->connection())return c->notConnectedToServer();

	QCString tmp2 = c->window()->connection()->encodeText(QString::fromUtf8(text.ptr()));
	if(!(c->window()->connection()->sendFmtData("%s %s",command.ptr(),tmp2.data() ? tmp2.data() : "")))
		return c->notConnectedToServer();

	return c->leaveStackFrame();
}

bool KviUserParser::parseCmd_AWAY(KviCommand * c)
{
	/*
		@doc: away
		@type:
			command
		@title:
			away
		@syntax:
			away <reason>
		@short:
			Puts you into 'away' state
		@description:
			Puts you into 'away' state in the connection associated to the
			current [b]IRC context[/b].[br] This command is "server based";
			this means that the effects will be visible only after the
			server has acknowledged the change.[br]
			When you use this command, other people will know that you are 
			away from the keyboard, and they will know why you're not here.[br]
			To return from being away you must use [cmd]back[/cmd].[br]
			This command is [doc:connection_dependant_commands]connection dependant[/doc].[br]
		@examples:
			[example]
			away I'm asleep. Don't wake me up.
			[/example]
	*/

	ENTER_STACK_FRAME(c,"away");

	KviStr reason;
	if(!parseCmdFinalPart(c,reason))return false;

	if(reason.isEmpty())reason = KVI_OPTION_STRING(KviOption_stringAwayMessage);

	if(!c->window()->console())return c->noIrcContext();
	if(!c->window()->connection())return c->notConnectedToServer();

	QCString tmp2 = c->window()->connection()->encodeText(QString::fromUtf8(reason.ptr()));
	if(tmp2.data())
	{
		if(!(c->window()->connection()->sendFmtData("AWAY :%s",tmp2.data())))
			return c->notConnectedToServer();
	}

	return c->leaveStackFrame();
}

bool KviUserParser::parseCmd_BACK(KviCommand * c)
{
	/*
		@doc: back
		@type:
			command
		@title:
			back
		@syntax:
			back
		@short:
			Allows you to return from being away
		@description:
			Using this command makes you return from being [cmd]away[/cmd] in the connection associated to the
			current [b]IRC context[/b].[br] This command is "server based";
			this means that the effects will be visible only after the
			server has acknowledged the change.[br]
			This command is [doc:connection_dependant_commands]connection dependant[/doc].[br]
		@examples:
			[example]
			back
			[/example]
	*/

	ENTER_STACK_FRAME(c,"back");

	KviStr dummy;
	if(!parseCmdFinalPart(c,dummy))
		return false;

	if(!c->window()->console())
		return c->noIrcContext();
	if(!c->window()->connection())return c->notConnectedToServer();

	if(!(c->window()->connection()->sendFmtData("AWAY")))
		return c->notConnectedToServer();

	return c->leaveStackFrame();
}

static bool file_in_path(QStringList &pathlist,QString &file)
{
	for(QStringList::Iterator it = pathlist.begin();it != pathlist.end();++it)
	{
		QString tmp = *it;
		tmp.append('/');
		tmp.append(file);
		KviFileUtils::adjustFilePath(tmp);
		if(KviFileUtils::fileExists(tmp))return true;
	}
	return false;
}

bool KviUserParser::parseCmd_HELP(KviCommand *c)
{
	/*
		@doc: help
		@type:
			command
		@title:
			help
		@syntax:
			help [-s] <topic>
		@short:
			Shows a help page
		@description:
			Shows a help page about <topic>.[br]
			<topic> can be a command name , or an identifier.[br]
			If <topic> starts with a slash , it is assumed to be a full path
			to a html file to be displayed in the help browser.[br]
			This command is a shortcut for the [cmd]help.open[/cmd] module command.[br]
		@examples:
			[example]
			help run
			help help
			help index
			help $mask
			help /home/pragma/myfile.html
			[/example]
	*/

	ENTER_STACK_FRAME(c,"help");

	const char * aux = c->ptr();

	if(!skipNormalCommand(c))return false;

	KviStr topic(aux,c->ptr() - aux);
	topic.stripWhiteSpace();
	if(topic.lastCharIs(';'))
	{
		topic.cutRight(1);
		topic.stripWhiteSpace();
	}

	KviStr szCommand;

	if(topic.isEmpty())szCommand = "help.open index.html";
	else {
		bool bWasFunc = false;
		if(*(topic.ptr()) == '$')
		{
			bWasFunc = true;
			topic.cutLeft(1);
		}

		KviStr szOriginal = topic;

		QString topik = topic.ptr();
		QStringList pl = QMimeSourceFactory::defaultFactory()->filePath();

		bool bLowCaseTried = false;
		bool bFound = KviFileUtils::fileExists(topik);

try_again:
		if((!bFound) && bWasFunc)
		{
			topik = topic.ptr();
			topik += ".html";
			topik.prepend("fnc_");
			bFound = file_in_path(pl,topik);
		}

		if(!bFound)
		{
			topik = topic.ptr();
			bFound = file_in_path(pl,topik);
		}

		if(!bFound)
		{
			topik = topic.ptr();
			topik += ".html";
			bFound = file_in_path(pl,topik);
		}

		if(!bFound)
		{
			topik = topic.ptr();
			topik += ".html";
			topik.prepend("cmd_");
			bFound = file_in_path(pl,topik);
		}

		if(!bFound)
		{
			topik = topic.ptr();
			topik += ".html";
			topik.prepend("doc_");
			bFound = file_in_path(pl,topik);
		}

		if(!bFound)
		{
			topik = topic.ptr();
			topik += ".html";
			topik.prepend("event_");
			bFound = file_in_path(pl,topik);
		}

		if(!bFound)
		{
			topik = topic.ptr();
			topik += ".html";
			topik.prepend("module_");
			bFound = file_in_path(pl,topik);
		}

		if(!bFound)
		{
			topik = topic.ptr();
			topik += ".html";
			topik.prepend("widget_");
			bFound = file_in_path(pl,topik);
		}

		if(!bFound)
		{
			topik = topic.ptr();
			topik += ".html";
			topik.prepend("class_");
			bFound = file_in_path(pl,topik);
		}
		
		if(!bFound && !bLowCaseTried)
		{
			// try again with lowercase
			topic.toLower();
			bLowCaseTried = true;
			goto try_again;
		}
		
		if(!bFound)
		{
			// let the help.open function decide...
			topik = szOriginal.ptr();
			topik += ".html";
		}

		szCommand.sprintf("help.open %s",topik.latin1());
	}


	// We don't care about the return value...
	parseCommandBuffer(szCommand.ptr(),c->window());

	return c->leaveStackFrame();
}

bool KviUserParser::parseCmd_RUN(KviCommand *c)
{
	/*
		@doc: run
		@type:
			command
		@title:
			run
		@syntax:
			run <commandline>
		@short:
			Runs an external process
		@description:
			Runs an external process. This command does not allow
			any other interaction with the process started. If you want
			to grab the process output you probably need to use [cmd]exec[/cmd].
			Run doesn't spawn a subshell in order to execute the <commandline>
			and thus if you need shell substitutions to be made you must invoke
			the shell by yourself (probably by passing the real command as the -c argument).
		@examples:
			[example]
			run xterm
			run xmessage -center *
			[comment]# note the difference in * processing[/comment]
			run /bin/bash -c "xmessage -center *"
			[/example]
	*/

	ENTER_STACK_FRAME(c,"run");

	KviStr command;

	if(!parseCmdSingleToken(c,command))return false;

	c->skipSpace();

	KviPtrList<KviStr> args;
	args.setAutoDelete(true);

	while(*(c->m_ptr) && (*(c->m_ptr) != '\n') && (*(c->m_ptr) != ';'))
	{
		KviStr * token = new KviStr();
		args.append(token);
		if(!parseCmdSingleToken(c,*token))return false;
		c->skipSpace();
	}

	KviStr dummy;

	if(!parseCmdFinalPart(c,dummy))return false;

	if(command.hasData())
	{
		//if(!kvi_runProcess(command,&args))c->warning(__tr2qs("Failed to execute command '%s'"),command.ptr());
		QProcess proc(QString(command.ptr()));
		for(KviStr * s = args.first();s;s = args.next())proc.addArgument(QString(s->ptr()));
		if(!proc.start())c->warning(__tr2qs("Failed to execute command '%s'"),command.ptr());
	} else c->warning(__tr2qs("No commandline specified"));

	return c->leaveStackFrame();
}

bool KviUserParser::parseCmd_SOUND(KviCommand *c)
{
	/*
		@doc: sound
		@type:
			command
		@title:
			sound
		@syntax:
			sound [-t=<timeout>] <target> <filename>
		@short:
			Plays a file
		@description:
			Plays a multimedia file by using the media type matching and
			sends a CTCP SOUND to the <target>.[br]
			(target may be a channel name or a nickname).[br]
			The <filename> has the same meaning as in [cmd]play[/cmd].[br]
			In fact , this CTCP should be something as "MULTIMEDIA" or "MM",
			but "SOUND" is used for other clients compatibility.[br]
			KVIrc accepts also "MULTIMEDIA" or "MM" as aliases to this CTCP.[br]
			The CTCP is sent thru a NOTICE and the <filename>
			is added to the public offer list for <timeout> seconds (or a default timeout if the -t switch is not used).[br]
	*/

	ENTER_STACK_FRAME(c,"sound");

	KviStr filename;
	KviStr target;
	if(!parseCmdSingleToken(c,target))return false;
	if(!parseCmdFinalPart(c,filename))return false;

	if(!c->window()->console())return c->noIrcContext();
	if(!c->window()->connection())return c->notConnectedToServer();

	if(kvi_strEqualCIN(filename.ptr(),"file:",5))filename.cutLeft(5);

	KviStr buffer;
	if(g_pApp->findUserFile(buffer,filename.ptr()))filename = buffer;

	if(!kvi_fileExists(filename.ptr()))
	{
		c->warning(__tr2qs("Can't find the multimedia file %s"),filename.ptr());
		return c->leaveStackFrame();
	}

	KviStr error;
	if(!g_pApp->playFile(filename.ptr(),error,c->window()))c->warning(error.ptr());

	// Now notify the MUTLIMEDIA file to <target> and add the offer

	if(target.isEmpty())
	{
		c->warning(__tr2qs("Missing target, no action taken"));
		return c->leaveStackFrame();
	}

	bool bTargetIsChan = (target.contains('#') || target.contains('&') || target.contains('!'));

	int iTimeout = (int)KVI_OPTION_UINT(KviOption_uintSoundOfferTimeoutInSecs);
	if(c->hasSwitch('t'))
	{
		KviStr szTimeout;
		if(c->getSwitchValue('t',szTimeout))
		{
			if(szTimeout.isUnsignedNum())iTimeout = szTimeout.toInt();
			else c->warning(__tr2qs("Invalid timeout specified , using default"));
		}
	}

	KviStr absPath = filename;
	if(filename.contains(KVI_PATH_SEPARATOR_CHAR))filename.cutToLast(KVI_PATH_SEPARATOR_CHAR);

	if(filename.hasData())
	{
		KviSharedFile * o = 0;

		if(bTargetIsChan)o = g_pSharedFilesManager->lookupSharedFile(filename.ptr(),0);
		else {
			KviIrcMask u(target.ptr());
			o = g_pSharedFilesManager->lookupSharedFile(filename.ptr(),&u);
		}

		if(!o)
		{
			KviStr szUserMask(KviStr::Format,"%s!*@*",bTargetIsChan ? "*" : target.ptr());

			o = g_pSharedFilesManager->addSharedFile(filename.ptr(),absPath.ptr(),szUserMask.ptr(),iTimeout);
			if(!o)
			{
				c->warning(__tr2qs("Can't add a file offer for file %s (huh ? file not readable ?)"),absPath.ptr());
				return c->leaveStackFrame();
			}

			if(_OUTPUT_VERBOSE)
			{
				c->window()->output(KVI_OUT_SYSTEMMESSAGE,__tr2qs("Added %d secs file offer for file %s (%s) and receiver %s"),
						iTimeout,o->absFilePath().latin1(),filename.ptr(),o->userMask().latin1());
			}
		}
	}

	c->window()->connection()->sendFmtData("NOTICE %s :%cSOUND %s%c",target.ptr(),0x01,
				filename.ptr(),0x01);

	if(c->hasSwitch('q'))return c->leaveStackFrame();

	// Lookup the window
	KviWindow * w = 0;
	if(bTargetIsChan)w = c->window()->console()->connection()->findChannel(target.ptr());
	else w = c->window()->console()->connection()->findQuery(target.ptr());	

	if(w)
	{
		w->output(KVI_OUT_MULTIMEDIA,__tr2qs("%s plays '%s'"),
			c->window()->connection()->currentNickName().latin1(),
			filename.ptr());
	} else {
		c->window()->output(KVI_OUT_MULTIMEDIA,__tr2qs("%s plays '%s' to %s"),
			c->window()->connection()->currentNickName().latin1(),
			filename.ptr(),target.ptr());
	}

	return c->leaveStackFrame();
}

bool KviUserParser::parseCmd_PLAY(KviCommand *c)
{
	/*
		@doc: play
		@type:
			command
		@title:
			play
		@syntax:
			play <filename>
		@short:
			Plays a file
		@description:
			Plays a file by using the media type matching.[br]
			If <filename> is a relative path, KVIrc will firs make a pre-guess
			of the media type by matching the filename (no magic check),
			then look for the file in the save directory for the media type guessed,
			then look in a set of standard directories (including the kvirc local
			pics, incoming and audio directories and the home directory).[br]
			If the <filename> is an absolute path, no lookup will be made.[br]
			If the file can be found, a second media type guess will be performed,
			this time eventually matching also the magic bytes.[br]
			If the file matches a media type, the commandline of that media type
			will be executed, passing the filename as the first parameter ($0).[br]
		@examples:
			[example]
				play test.wav
				play diagram.jpg
			[/example]
	*/

	ENTER_STACK_FRAME(c,"play");

	KviStr filename;
	if(!parseCmdFinalPart(c,filename))return false;

	if(kvi_strEqualCIN(filename.ptr(),"file:",5))filename.cutLeft(5);

	KviStr buffer;
	if(g_pApp->findUserFile(buffer,filename.ptr()))filename = buffer;
	kvi_adjustFilePath(filename);

	KviStr error;
	if(!g_pApp->playFile(filename.ptr(),error,c->window()))c->warning(error.ptr());

	return c->leaveStackFrame();
}


bool KviUserParser::parseCmd_OPENURL(KviCommand *c)
{
	/*
		@doc: openurl
		@type:
			command
		@title:
			openurl
		@syntax:
			openurl <url>
		@short:
			Opens an url
		@description:
			Opens the specified <url> with an appropriate handler.<br>
			The handlers for the supported url types are specified in the options dialog.<br>
			Each handler is a kvirc commandline that the url will be passed to as the first parameter ($0).<br>
			The supported url types are:<br>
			HTTP: http://&lt;url&gt; or sth that begins with "www." <br>
			HTTPS: https://&lt;url&gt;<br>
			FILE: file://&lt;url&gt;<br>
			IRC: irc[s][6]://<server>[:<port>][/<channel>[?<pass>]] (Handled internally)<br>
			FTP: ftp://<url> or sth that begins with "ftp."<br>
			MAIL: mailto:<mailaddress><br>
		@examples:
			[example]
				openurl http://www.kvirc.net
				openurl irc://irc.eu.dal.net:6667
				openurl irc6://irc.ircd.it/#kvirc
				openurl ircs://crypto.azzurra.org:9999
				openurl ircs6://ngnet.azzurra.org:9999
				openurl www.kvirc.net
				openurl ftp://ftp.localhost.net/pub/kvirc/
				openurl https://www.secure.net
				openurl file://home/pragma/pippo.txt
			[/example]
	*/

	ENTER_STACK_FRAME(c,"openurl");

	KviStr url;
	if(!parseCmdFinalPart(c,url))return false;

	KviStr command;
	if(kvi_strEqualCIN(url.ptr(),"www.",4))
	{
		command = KVI_OPTION_STRING(KviOption_stringUrlHttpCommand);
		url.prepend("http://");
	} else if(kvi_strEqualCIN(url.ptr(),"http:",5))
	{
		command = KVI_OPTION_STRING(KviOption_stringUrlHttpCommand);
	} else if(kvi_strEqualCIN(url.ptr(),"https:",6))
	{
		command = KVI_OPTION_STRING(KviOption_stringUrlHttpsCommand);
	} else if(kvi_strEqualCIN(url.ptr(),"ftp",3))
	{
		command = KVI_OPTION_STRING(KviOption_stringUrlFtpCommand);
		if(kvi_strEqualCIN(url.ptr(),"ftp.",4))url.prepend("ftp://");
	} else if(kvi_strEqualCIN(url.ptr(),"file",4))
	{
		command = KVI_OPTION_STRING(KviOption_stringUrlFileCommand);
	} else if(kvi_strEqualCIN(url.ptr(),"irc",3))
	{
		if(!parseIrcUrl(url.ptr(),command))
		{
			c->warning(__tr2qs("Invalid IRC url (%s)"),url.ptr());
			return c->leaveStackFrame();
		}
	} else if(kvi_strEqualCIN(url.ptr(),"mailto",6))
	{
		command = KVI_OPTION_STRING(KviOption_stringUrlMailtoCommand);
	}

	if(command.isEmpty())command = KVI_OPTION_STRING(KviOption_stringUrlUnknownCommand);

	if(command.hasData())
	{
		KviParameterList * l = new KviParameterList;
		l->append(new KviStr(url));

		if(!parseCommandBuffer(command.ptr(),c->window(),l))
			c->warning(__tr2qs("The commandline for this url type seems to be broken (%s)"),url.ptr());

	} else c->warning(__tr2qs("No commandline specified for this type of url (%s)"),url.ptr());

	return c->leaveStackFrame();
}

bool KviUserParser::parseCmd_QUIT(KviCommand *c)
{
	/*
		@doc: quit
		@type:
			command
		@title:
			quit
		@syntax:
			quit [-f] [quit-message]
      		quit -q
		@short:
			Terminates the current IRC connection or the entier application
		@description:
			Terminates the current IRC session.[br]
			By default this command sends a QUIT message 
			and waits for the server to close the connection.[br]
			If you want to force KVIrc to close the connection 
			immediately after sending the QUIT message you must use the -f switch.[br]
			Forcing the connection may cause your quit message to not be 
			displayed to the other IRC users: most likely it will be replaced 
			by a 'Connection reset by peer' or a 'EOF from client'.[br]
			If the -q switch is specified , this command terminates KVIrc immediately.[br]
		@examples:
			[example]
				quit Time to sleep
			[/example]
	*/
	ENTER_STACK_FRAME(c, "quit");

	KviStr reason;
	if(!parseCmdFinalPart(c,reason))return false;

	if(c->hasSwitch('q'))
	{
		QTimer::singleShot(0,g_pApp,SLOT(quit()));
	} else {
		if(!c->window()->context())return c->noIrcContext();
		if(!c->window()->context()->isConnected())return c->notConnectedToServer();
		c->window()->context()->terminateConnectionRequest(c->hasSwitch('f'),reason.ptr());
	}
	return c->leaveStackFrame();
}

bool KviUserParser::parseCmd_CLEAR(KviCommand *c)
{
	/*
	 	@doc: clear
		@type:
			command
		@title:
			clear
		@syntax:
			clear
		@short:
			Clears the IRC view
		@description:
			Clear the text buffer of the current active IRC view. Useful to know what
			people have said since you went AFK.
		*/
	ENTER_STACK_FRAME(c, "clear");
	KviStr reason;
	if(!parseCmdFinalPart(c,reason))return false;
    
	if (c->window()->view())c->window()->view()->emptyBuffer(true);
   
    if (c->window()->type() == KVI_WINDOW_TYPE_CHANNEL) 
    {
        KviChannel *chan = (KviChannel *)c->window();
        if(chan->messageView()) chan->messageView()->emptyBuffer(true);
    }

	return c->leaveStackFrame();
}

bool KviUserParser::parseCmd_BEEP(KviCommand *c)
{
	/*
	 	@doc: beep
		@type:
			command
		@title:
			beep
		@syntax:
			beep [-p=<pitch>] [-d=<duration>] [-s] [volume]
		@short:
			Beep beep!
		@description:
			Beeps (when possible :D)[br]
			...[br]
			No , really..[br]
			This command rings the bell on the keyboard (the PC speaker).
			The volume must be in range 0-100; the default is 100.[br]
			The pitch is specified in Hz and must be positive.[br]
			The duration is specified in milliseconds.[br]
			An invalid (or unspecified) pitch, volume or duration
			makes KVIrc to use the default values set by the system.[br]
			The duration of the bell is only indicative and
			can be shortened by a subsequent call to /beep (that
			will override the currently playing one).[br]
			On Windows, the bell is always synchronous and it is not
			event granted that the bell will be a bell at all... you might
			get the system default sound instead.. so be careful if you
			want to write portable scripts :)[br]
			If the -s switch is specified the bell becomes synchronous:
			KVIrc waits the bell to complete before continuing.[br]
			Obviously -s is senseless on Windows.[br]
			(WARNING : the main KVIrc thread is stopped in that case
			so if you play long notes (duration > 100)
			the entire application will appear to freeze for a while).[br]
			The precision of the bell pitch, duration and
			volume is strongly dependant on the system and the underlying hardware.[br]
		*/
	ENTER_STACK_FRAME(c, "beep");

	int pitch    = -1;
	int duration = -1;
	bool bSync   = c->hasSwitch('s');
	bool bOk = false;

	KviStr szVolume;
	if(!parseCmdFinalPart(c,szVolume))return false;


	if(c->hasSwitch('p'))
	{
		KviStr szPitch;
		c->getSwitchValue('p',szPitch);
		pitch = szPitch.toInt(&bOk);
		if(!bOk || pitch < 0)
		{
			c->warning(__tr2qs("Invalid pitch (%s): using default"),szPitch.ptr());
			pitch = -1;
		}
	}
	if(c->hasSwitch('d'))
	{
		KviStr szDuration;
		c->getSwitchValue('d',szDuration);
		duration = szDuration.toInt(&bOk);
		if(!bOk || duration < 0)
		{
			c->warning(__tr2qs("Invalid duration (%s): using default"),szDuration.ptr());
			duration = -1;
		}
	}

	int volume = szVolume.toInt(&bOk);
	if(!bOk || (volume > 100) || (volume < 0))volume = 100;

#ifdef COMPILE_ON_WINDOWS
	Beep(pitch,duration);
#else
	#ifndef COMPILE_NO_X_BELL

		XKeyboardState st;
		XKeyboardControl ctl;

		XGetKeyboardControl(qt_xdisplay(),&st);

		unsigned long mask = KBBellPercent;
		ctl.bell_percent = volume;
		if(pitch >= 0)
		{
			ctl.bell_pitch    = pitch;
			mask             |= KBBellPitch;
		}
		if(duration >= 0)
		{
			ctl.bell_duration = duration;
			mask             |= KBBellDuration;
		}
		XChangeKeyboardControl(qt_xdisplay(),mask,&ctl);

		XBell(qt_xdisplay(),100);

		if(bSync)
		{
			if(duration >= 0)usleep(duration * 1000);
			else usleep(st.bell_duration * 1000);
		}

		ctl.bell_pitch = st.bell_pitch;
		ctl.bell_duration = st.bell_duration;
		ctl.bell_percent = st.bell_percent;

		XChangeKeyboardControl(qt_xdisplay(),mask,&ctl);

	#endif //COMPILE_NO_X_BELL
#endif
	return c->leaveStackFrame();
}

bool KviUserParser::parseCmd_BUTTONCTL(KviCommand *c)
{
	ENTER_STACK_FRAME(c,"buttonctl");
	/*
		@doc: buttonctl
		@title:
			buttonctl
		@type:
			command
		@short:
			changes user definable buttons
		@syntax:
			buttonctl [-q] <type> <name> <operation> [parameter]
		@description:
			Changes an existing user defined button.[br]
			<type_unused> is ignored and present only for backward compatibility.[br]
			<name> is the name of the button.[br]
			<operation> may be one of the constant strings "enable", "disable", "image",
			"text".[br]
			Operations "enable" and "disable" do not require the fourth [parameter] and have
			the obvious meaning.[br] Operation "image" requires the [parameter] to be
			a valid [doc:image_id]image_id[/doc] and sets the button image.[br]
			Operation "text" requires the [parameter] (and in fact all the following ones)
			to be a string containing the button text label.[br]
			The <operation> constants may be abbreviated, even to the single letters 'e','d','i' and 't'.[br]
			The -q switch causes the command to be quiet about errors and warnings.[br]
		@seealso:
			[cmd]button[/cmd]
	*/

	KviStr tbTypeUnused, tbName, tbOp, tbPar;

	if(!parseCmdSingleToken(c,tbTypeUnused))return false;
	if(!parseCmdSingleToken(c,tbName))return false;
	if(!parseCmdSingleToken(c,tbOp))return false;
	if(!parseCmdFinalPart(c,tbPar))return false;

	KviScriptUserButton * pButton = 0;

	if(!c->window()->buttonContainer())
	{
		if(!c->hasSwitch('q'))c->warning(__tr2qs("The specified window has no button containers"));
		return c->leaveStackFrame();
	}
	
	pButton = (KviScriptUserButton *)(c->window()->buttonContainer())->child(tbName.ptr(),"KviWindowScriptButton");

	if(!pButton)
	{
		if(!c->hasSwitch('q'))c->warning(__tr2qs("No button with type %s named %s"),tbTypeUnused.ptr(),tbName.ptr());
		return c->leaveStackFrame();
	}

	switch(*(tbOp.ptr()))
	{
		case 't':
			QToolTip::remove(pButton);
			QToolTip::add(pButton,tbPar.ptr());
			pButton->setButtonText(tbPar.ptr());
		break;
		case 'i':
			if(tbPar.hasData())
			{
				QPixmap * pix = g_pIconManager->getImage(tbPar.ptr());
				if(pix)
				{
					pButton->setButtonPixmap(*pix);
				} else {
					if(!c->hasSwitch('q'))c->warning(__tr2qs("Can't find the icon '%s'"),tbPar.ptr());
				}
			}
		break;
		case 'e':
			pButton->setEnabled(true);
		break;
		case 'd':
			pButton->setEnabled(false);
		break;
	}
	return c->leaveStackFrame();
}

bool KviUserParser::parseCmd_BUTTON(KviCommand *c)
{
	ENTER_STACK_FRAME(c,"button");
	/*
		@doc: button
		@title:
			button
		@type:
			command
		@short:
			Adds/removes/changes user definable buttons
		@syntax:
			button [-d] [-q] (<type_unused>,<name>[,<image_id>[,<label_text>]])
			{
				<callback_code>
			}
		@description:
			Adds a new user defined button with the specified <name>.[br]
			[br]
			The <type_unused> parameter is ignored and is present only for
			backward compatibility.
			[br]
			The button image is specified by the [doc:image_id]<image_id>[/doc].[br]
			The optional button text is specified by <label_text>.[br]
			The <callback_code> will be executed as reaction to a button press.[br]
			[br]
			The "window" type button can be added only to the windows that have a button container: this
			actually includes at least console , channels and queries.[br]
			The button is added to the current window; if you want to add it to a different
			window , use the [doc:command_rebinding]standard -r command rebinding[/doc] switch.[br]
			The <callback_code> will be executed as reaction to a button press; the
			code execution will be bound to the window that the button is attacched to.[br]
			If a button with <name> already exists in the current window, its parameters are changed
			according to the passed values (<image_id>, <label_text> and <callback_code>).[br]
			[br]
			Passing an empty <callback_value> removes the button.[br]
			The callback parameters $0 and $1 will contain the screen coordinates of the bottom-left
			corner of the button: this is useful for showing a popup menu in response to the click.[br]
			If the -q switch is used , this command prints no warnings.[br]
			The -d switch causes the button to be disabled (grayed).[br]
		@examples:
			[example]
				button(w,test,-1,Test button){ echo Test!; }
				button(w,test){}
			[/example]
			
	*/

	c->skipWhiteSpace();
	if(*(c->m_ptr) != '(')return c->error(KviError_openParenthesisExpected);

	KviParameterList paramList;
	paramList.setAutoDelete(true);

	if(!extractFunctionParameters(c,&paramList))return false;

	c->skipWhiteSpace();

	if(*(c->m_ptr) != '{')return c->error(KviError_openBraceExpected);
	const char * aux = c->m_ptr;
	if(!skipCommand(c))return false;

	KviStr cmd(aux,c->m_ptr);
	KviCommandFormatter::bufferFromBlock(cmd);

	KviStr * tbTypeUnused;
	tbTypeUnused = paramList.safeFirst();
	KviStr * tbName = paramList.safeNext();
	KviStr * tbIco  = paramList.next();
	KviStr * tbLab  = paramList.next();

	KviScriptUserButton * pButton = 0;


	if(!c->window()->buttonContainer())
	{
		if(!c->hasSwitch('q'))c->warning(__tr2qs("The specified window has no button containers"));
		return c->leaveStackFrame();
	}

	pButton = (KviScriptUserButton *)(c->window()->buttonContainer())->child(tbName->ptr(),"KviWindowScriptButton");

	if(cmd.isEmpty())
	{
		if(pButton)delete pButton;
		else {
			if(!c->hasSwitch('q'))c->warning(__tr2qs("Window button '%s' not found"),tbName->ptr());
		}
		return c->leaveStackFrame();
	} else {
		if(!pButton)
		{
			pButton = new KviWindowScriptButton(c->window()->buttonContainer(),c->window(),tbName->ptr());
			pButton->show();
		}
		QToolTip::remove(pButton);
		if(tbLab)
		{
			pButton->setButtonText(tbLab->ptr());
			QToolTip::add(pButton,tbLab->ptr());
		}
	}

	pButton->setButtonCode(cmd.ptr());

	if(tbIco)
	{
		if(tbIco->hasData())
		{
			QPixmap * pix = g_pIconManager->getImage(tbIco->ptr());
			if(pix)
			{
				pButton->setButtonPixmap(*pix);
			} else {
				if(!c->hasSwitch('q'))c->warning(__tr2qs("Can't find the icon '%s'"),tbIco->ptr());
			}
		}
	}

	pButton->setEnabled(!(c->hasSwitch('d')));

	return c->leaveStackFrame();
}


bool KviUserParser::parseCmd_EVAL(KviCommand *c)
{
	/*
		@doc: eval
		@type:
			command
		@title:
			eval
		@syntax:
			eval [-q] [-r=<window>] <command>
		@short:
			Change the behaviour of a set of commands
		@description:
			This command is useful to execute variable command sequences.[br]
			<command> is first evaluated as a normal parameter (thus identifiers
			and variables are substituted) then the evaluated string is executed
			as a command sequence.[br]
			-q causes eval to silently ignore the errors inside <command> and continue execution.[br]
			This command may be used to rebind the <command> to a specified window.
			<command> shares the local variables with this command scope so you
			can easily exchange data with it.
			Remember that <command> is still a normal parameter and it must be
			enclosed in quotes if youwant it to be a complex command sequence.
			eval propagates the <command> return value.[br]
		@examples:
			[example]
				[comment]# evaluate a variable command[/comment]
				[cmd]if[/cmd](%somecondition)%tmp = "echo yeah"
				else %tmp = "echo -i=10 yeah"
				eval %tmp
				[comment]# Rebind the command to the #linux channel to get the user list[/comment]
				eval -r=[fnc]$channel[/fnc](#linux) "%Nicks[]=$chan.array;"
				[comment]# A nice alias that allows iterating commands through all the consoles[/comment]
				[comment]# This is by LatinSuD :)[/comment]
				[cmd]alias[/cmd](iterate)
				{
					%ctxt[]=[fnc]$window.list[/fnc](console,all)
					[cmd]for[/cmd](%i=0;%i<%ctxt[]#;%i++)
					{
						[cmd]eval[/cmd] -r=%ctxt[%i] $0-
					}
				}
				iterate [cmd]echo[/cmd] Hi ppl! :)
				[comment]# A little bit shorter (but less "colorful") version might be...[/comment]
				[cmd]alias[/cmd](iterate)
				{
					[cmd]foreach[/cmd](%x,[fnc]$window.list[/fnc](console,all))
							[cmd]eval[/cmd] -r=%x $0-;
				}
				iterate [cmd]echo[/cmd] Hi again!
				[comment]# Evaluate a command block[/comment]
				eval "{ echo First command!; echo Second command!; }"
			[/example]
	*/

	ENTER_STACK_FRAME(c,"eval");

	KviStr szBuffer;

	if(!parseCmdFinalPart(c,szBuffer))return false;

	KviCommand cmd(szBuffer,c->window(),c);
	cmd.setDataContainer(c->dataContainer());
	cmd.setScopeObject(c->scopeObject());
	bool bRet = parseCommand(&cmd);
	cmd.forgetDataContainer();
	if(!bRet)
	{
		if(cmd.hasError())
		{
			if(!c->hasSwitch('q'))
			{
				printError(&cmd);
				return false; // We could do better ? (just a halt)
			}
		}
	}

	c->m_szRetBuffer = cmd.m_szRetBuffer;

	return c->leaveStackFrame();
}


bool KviUserParser::parseCmd_SAY(KviCommand *c)
{
	/*
		@doc: say
		@type:
			command
		@title:
			say
		@syntax:
			say [-q] [-r=<window_id>] <text>
		@short:
			Type text in a window
		@description:
			This command is more or less equivalent to typing text in the input
			entry of the current window. If the -r switch is used
			then the command is rebound to the window specified by <window_id>.
			The main difference is that the variables and identifiers in <text>
			are always parsed (when typing this happen only if the text is a command).<br>
			You can use this command to execute "variable" commands too:
			if <text> begins with a slash then it will be treated as a command
			to be executed (after parsing the identifiers etc.).<br>
			-q causes the command to run quietly.<br>
			<b>Handle with care.</b>
	*/

	ENTER_STACK_FRAME(c,"say");

	KviStr szText;

	if(!parseCmdFinalPart(c,szText))return false;

	if(!parseUserCommand(szText,c->window()))
	{
		if(!c->hasSwitch('q'))
			c->warning(__tr("Say parse error: Broken command"));
	}

	return c->leaveStackFrame();
}


bool KviUserParser::parseCmd_REBIND(KviCommand *c)
{
	/*
		@doc: rebind
		@type:
			command
		@title:
			rebind
		@syntax:
			rebind [-q] <window_id>
		@short:
			Rebinds the command sequence to a specified window
		@description:
			Rebinds the current command sequence to the window specified by <window_id>.[br]
			The new window will be inherited by all the subcommands and aliases called.[br]
			-q causes the command to run quietly.[br]
			[b]Handle with care.[/b]
		@examples:
			[comment]# Try this example in a channel or query window[/comment]
			[comment]# Remember the current window id[/comment]
			%winid = $window
			[comment]# Rebind to the console of the current irc context[/comment]
			rebind $console
			echo "Hello from the console :)"
			echo "Hello again.. still in the console"
			[comment]# Rebind back[/comment]
			rebind %winid
			echo "Again in this window :)"
	*/

	ENTER_STACK_FRAME(c,"rebind");

	KviStr winId;

	if(!parseCmdFinalPart(c,winId))return false;

	if(winId.isEmpty())
	{
		if(!c->hasSwitch('q'))
			c->warning(__tr2qs("Can't rebind the command sequence: missing window identifier"));
	} else {
		KviWindow * pAux = g_pApp->findWindow(winId.ptr());
		if(pAux)c->rebindToWindow(pAux);
		else {
			if(!c->hasSwitch('q'))
				c->warning(__tr2qs("Can't rebind the command sequence: window with id %s not found"),winId.ptr());
		}
	}
	
	return c->leaveStackFrame();
}

bool KviUserParser::parseCmd_FOR(KviCommand *c)
{
	/*
		@doc: for
		@type:
			command
		@title:
			for
		@syntax:
			for([initialization];<condition>;[operation]) <command>
		@keyterms:
			iteration commands, flow control commands
		@description:
			Performs <initialization> and then enters the following loop:[br]
			1) check <condition> , if it evaluates to false , terminate the loop (skip AFTER point 4).[br]
			2) execute <command>[br]
			3) execute <operation> (that must be a single l-value command!)[br]
			4) return to point 1[br]
			You can always jump out of the <command> by using [cmd]break[/cmd].[br]
			[initialization] and [operation] can be empty.[br]
			An empty <condition> (unlike other languages) is evaluated to false (treated as empty string);
			thus if you want an endless loop, you must use for(;1;) or (even better) [cmd]while[/cmd](1)
			that is a lot faster since there is no check for the [operation].[br]
		@examples:
			[example]
			for(%i = 0;%i < 100;%i++)[cmd]echo[/cmd] %i
			for(%i = 100;%i;%i -= 10)[cmd]echo[/cmd] %i
			%i = 0;
			for(;1;)
			{
				echo %i
				%i++;
				if(%i > 10)[cmd]break[/cmd];
			}
			[/example]
	*/

	ENTER_STACK_FRAME(c,"for");

	c->skipWhiteSpace();
	if(*(c->m_ptr) != '(')return c->error(KviError_openParenthesisExpected);
	++(c->m_ptr);
	c->skipWhiteSpace();
	if(!parseSingleCommand(c))return false;
	c->skipWhiteSpace();

	char * conditionBegin = c->m_ptr;
	if(!skipForCondition(c))return false;
	char * conditionEnd = c->m_ptr;
	if(!(*(c->m_ptr)))return c->error(KviError_unexpectedEndInExpression);
	++(c->m_ptr);

	c->skipWhiteSpace();
	char * operationBegin = c->m_ptr;
	if(*(c->m_ptr) == ')')
	{
		++(c->m_ptr);
	} else {
		if(!skipLValueCommand(c,true))return false;
	}
	char * operationEnd = c->m_ptr;
	if(!(*(c->m_ptr)))return c->error(KviError_unexpectedEndInExpression);
	++(c->m_ptr);

	c->skipWhiteSpace();
	char * commandBegin = c->m_ptr;

	*operationEnd = '\0';
	*conditionEnd = '\0';

	char * commandEnd = 0;

	c->m_ptr = conditionBegin;
	long result;
	bool bRet = evaluateExpression(c,&result,false);
	if(!bRet)goto return_false;


	while(result)
	{
		c->m_ptr = commandBegin;

		int callStackSize = c->actualStackSize();

		if(*(c->m_ptr) == '{')bRet = parseCommandBlock(c);
		else bRet = parseSingleCommand(c);

		if(!bRet)
		{
			if(c->breakEncountered())
			{
				c->setBreakHandled(callStackSize); // ensure that the call stack is returned to coherent state
				break;
			} else goto return_false;
		}

		commandEnd = c->m_ptr;

		c->m_ptr = operationBegin;
		bRet = parseLValueCommand(c);
		if(!bRet)goto return_false;

		c->m_ptr = conditionBegin;
		bRet = evaluateExpression(c,&result,false);
		if(!bRet)goto return_false;
	}

	if(!commandEnd)
	{
		c->m_ptr = commandBegin;
		if(!skipCommand(c))goto return_false;
	} else {
		c->m_ptr = commandEnd;
	}

	*operationEnd = ';';
	*conditionEnd = ';';
	return c->leaveStackFrame();

return_false:
	*operationEnd = ';';
	*conditionEnd = ';';
	return false;
}

bool KviUserParser::parseCmd_FOREACH(KviCommand *c)
{
	/*
		@doc: foreach
		@type:
			command
		@title:
			foreach
		@syntax:
			foreach(<variable>,[<item>[,<item>[,<item>[...]]]) <command>
		@keyterms:
			iteration commands, flow control commands
		@short:
			Iteration command
		@description:
			Executed <command> while assigning to <variable> each <item>.[br]
			<item> may be a constant , a variable , an array , a dictionary or a function returning
			either a constant string an array reference or a dictionary reference.[br]
			If <item> is an array , a dictionary or a function that returns a dictionary or array reference
			the iteration is done through all the dictionary/array items.[br]
			Please note that the iteration order of dictionary items is undefined.[br]
			You can always break from the loop by using the [cmd]break[/cmd] command.[br]
		@examples:
			[example]
				foreach(%i,1,2,3,4,5,6,7,8,9)[cmd]echo[/cmd] %i
				foreach(%chan,[fnc]$window.list[/fnc](channel))[cmd]me[/cmd] -r=%chan This is a test!
				[comment]# This will work too, and will do the same job[/comment]
				%windows[] = [fnc]$window.list[/fnc](channel)
				foreach(%chan,%windows[])[cmd]me[/cmd] -r=%chan This is a test!
				[comment]# And this too[/comment]
				%windows[] = [fnc]$window.list[/fnc](channel)
				foreach(%key,%windows[]@)[cmd]me[/cmd] -r=%windows[%key] This is a test!
			[/example]
	*/

	ENTER_STACK_FRAME(c,"foreach");

	c->skipWhiteSpace();
	if(*(c->m_ptr) != '(')return c->error(KviError_openParenthesisExpected);

	++(c->m_ptr);

//	KviStr szSeparator;
//
//	if(c->hasSwitch('s'))c->getSwitchValue('s',szSeparator);
//	else szSeparator = ",";
//
//	if(szSeparator.isEmpty())return c->error(KviError_theSeparatorMayNotBeEmpty);

	// Ok...first of all , we expect a loop variable

	if((*(c->m_ptr) != '%')  && (*(c->m_ptr) != '$'))return c->error(KviError_variableOrIdentifierExpected);

	__range_invalid(c->scopeObject());

	bool bLastWasAVariable = false;

	KviDataType d;
	KviStr dummy;

	do {
		if(*(c->m_ptr) == '%')
		{
			bLastWasAVariable = true;
			if(!parseVariable(c,dummy,&d,false))return false;
		} else if(*(c->m_ptr) == '$')
		{
			bLastWasAVariable = false;
			if(!parseIdentifier(c,dummy))return false;
		} else return c->error(KviError_variableOrIdentifierExpected);
	} while(c->scopeObject());

	// check object scope

	if(!bLastWasAVariable)return c->error(KviError_variableExpected);

	switch(d.iDataType)
	{
		case KVI_DATA_TYPE_VARIABLE:
			d.pVariable = d.pDataContainer->lookupVariable(d.szName.ptr(),false); // RW access
		break;
		case KVI_DATA_TYPE_DICTVARIABLE:
			d.pDictionary = d.pDataContainer->lookupDictionary(d.szName.ptr(),false);
			d.pVariable = d.pDataContainer->lookupDictionaryVariable(d.szName.ptr(),d.szKey.ptr(),false);
		break;
		case KVI_DATA_TYPE_DICTREFERENCE:
		case KVI_DATA_TYPE_DICTKEYSREFERENCE:
		case KVI_DATA_TYPE_DICTITEMCOUNT:
		case KVI_DATA_TYPE_NONE:
			return c->error(KviError_variableExpected);
		break;
	}

	c->skipSpace();

	if(*(c->m_ptr) == ',')c->m_ptr++;

	// Oki..we have the loop variable in d
	// Now parse the data... need to parse it two times in fact!

	KviParameterList parms;

	if(!extractParameters(c,&parms))return false;

	// ok...now we can parse the command

	c->skipWhiteSpace();

	if(parms.isEmpty())
	{
		if(!skipCommand(c))return false;
		return c->leaveStackFrame();
	}

	int callStackSize = c->actualStackSize();

	char * pCommand = c->m_ptr;
	char * pEnd = 0;


	for(KviStr *param = parms.first();param;param = parms.next())
	{
		*(d.pVariable) = *param;

		c->m_ptr = pCommand;

		bool bRetVal;

		if(*(c->m_ptr) == '{')bRetVal = parseCommandBlock(c);
		else bRetVal = parseSingleCommand(c);

		if(!bRetVal)
		{
			if(c->breakEncountered())
			{
				if(!pEnd)
				{
					// not reached the end yet...must reach it
					c->m_ptr = pCommand;
					if(!skipCommand(c))return false;
				} else c->m_ptr = pEnd;
				c->setBreakHandled(callStackSize); // ensure that the call stack is returned to coherent state
				return c->leaveStackFrame();

			} else return false;
		}

		pEnd = c->m_ptr;
	}

	c->m_ptr = pEnd; // non zero for sure

	return c->leaveStackFrame();
}

bool KviUserParser::parseCmd_EXEC(KviCommand *c)
{
	/*
		@doc: exec
		@type:
			command
		@title:
			exec
		@syntax:
			exec [switches] (<commandline>[,<magic data>])
			{
				<callback command>
			}
		@short:
			Asynchronous execution of external programs
		@switches:
			!sw: -q
				Quiet: do not print any warnings
			!sw: -t
				Trigger the termination event
			!sw: -x
				Trigger the startup event
			!sw: -n
				Do NOT trigger any stdout events
			!sw: -e
				Trigger stderr events
			!sw: -b
				Trigger the <callback comand> with the stdout and stderr events exactly once,
				passing the complete block of process output. The events are triggered even
				if the process output is empty.
			!sw: -k=<maximum run time>
				Kill the process unconditionally after <maximum run time> milliseconds.
				If the -t switch is used then the termination event will be
				triggered just after the process has been killed.
			!sw: -p=<timeout>
				Trigger <callback command> with "ping" events every <timeout> milliseconds.
			!sw: -w
				Kill the process if the current window is closed. In this case the
				termination event is NOT triggered (since the parent window has been lost).
				If this switch is not used then the process is rebound to
				the active console window and continues running.
			!sw: -s=<interpreter command>
				Use <interpreter command> instead of the default interpreter "sh -c".
				The <interpreter command> should be able to launch the interpeter
				and should contain the necessary arguments in order to allow
				KVirc to pass the "commandline" by appending it as the last parameter.
			!sw: -d
				Use no command interpreter at all: run the command directly.
				Takes precedence over -s.
		@description:
			[b]Overview[/b][br]
			Executes the <commandline> by passing it to a command interpreter.
			The <commandline> is executed asynchronously: this means that
			when exec returns the control to the next command, <commandline>
			may be still running.[br]
			[br]
			[b]The callback[/b][br]
			The <callback command> is triggered on several events related to the
			child process and it gets passed the following parameters:[br]
			$0 = <event cause>[br]
			$1 = <event parameter>[br]
			$2 = <magic data>[br]
			The first parameter specifies the event cause and contains one of the
			following strings: "stdout","stderr","terminated","started" and "ping".
			[b]By default (if no switches are used) only "stdout" type events are triggered[/b].
			The second parameter depends on the event cause and contains data sensible
			to each event type. The third parameter is the eventual <magic data>
			passed to the exec command call.[br]
			[br]
			[b]Interacting with the process[/b][br]
			If you use [cmd]halt[/cmd] to terminate
			the callback then the slave process is killed immediately and
			no other callback events are triggered.[br] If you return some non empty string
			then this string will be written to the process stdin stream. This trick
			can be used to control interactive processes. Please note that you must
			include all the relevant carriage returns and newlines in the return value
			(see [fnc]$cr[/fnc] and [fnc]$lf[/fnc]).[br]
			[br]
			[b]Startup event[/b][br]
			If the -x switch is used then the startup event is triggered
			just after the process has been succesfully launched.
			The $0 parameter passed to the callback contains the string "started".
			Parameter $1 contains the pid of the slave process.[br]
			[br]
			[b]Stdout data event[/b][br]
			The stdout data event is triggered when the process prints some output
			on its stdout stream. This event is triggered by default and to disable
			it you must use the -n switch. $0 contains the string "stdout".
			If the -b switch is not used then $1 contains a single line of process
			output with the trailing carriage return and/or line feed stripped.
			If -b is used then $1 contains the whole process output 
			block (eventually empty) with all the cr/lf pairs.[br]
			[br]
			[b]Stderr data event[/b][br]
			The stderr data event is similar to the stdout one but there are three differences.
			The first one is that the stderr event is NOT triggered by default: you must
			use the -e switch to enable it. The second difference is that $0
			contains "stderr" instead of "stdout". The last difference is that $1 contains data
			coming from the slave process stderr stream.[br]
			[br]
			[b]Termination event[/b][br]
			The termination event is triggered after the slave process has terminated its
			execution. You must use the -t switch to enable it since it is
			disabled by default. $0 contains the string "terminated". $1 contains the process exit
			status value. (Note that if the process has crashed or has been terminated
			by an external singnal then this value will be 0).[br]
			[br]
			[b]Ping event[/b][br]
			The ping event is triggered only if the -p=<timeout> switch is passed.[br]
			This event may be useful to monitor the process status while it is not
			emitting any output, to write data to its stdin stream (by the means of [cmd]return[/cmd])
			or simply to give some feedback to the user while the slave process is
			doing a long computation.[br]
			[br]
			[b]The extended scope variables[/b][br]
			The <callback command> has a set of [doc:data_types]extended scope variables[/doc]
			that conserve their value during the whole life time of the slave process.[br]
			These variables can be accessed through the %:<varname> syntax and are
			useful to store process private data between multiple <callback command> calls.[br]
			[b]Some words about the switches[/b][br]
			If the -b switch is used then the <callback command> is called only once
			for the events stdout and stderr (if enabled) with the complete output block from the process.
			With the -b switch the events stdout and stderr are triggered once even if the process
			emits no output.
			The -s=<interpreter> switch may be used to specify the path of the command interpreter
			that is "sh -c" by default. The interpreter executable is searched on the system PATH.
			If the process can't be started then a warning message is printed in the current window
			unless the -q (quiet) flag is used.[br]
			[br]
		@examples:
			[example]
			[comment]# Really simple example: print only the stdout of a slave process[/comment]
			exec("cat /proc/cpuinfo"){ echo $1; };
			[comment]# Now print only stderr: enable stderr and disable stdout[/comment]
			exec -e -n ("sed -senseless"){ echo $1; };
			[comment]# Do it another way: enable stderr and filter out stdout[/comment]
			exec -e ("sed -senseless"){ if($0 == "stderr")echo $1; }
			[comment]# Now enable all (almost) events and print them[/comment]
			exec -e -t -s ("cat /proc/cpuinfo && sed -senseless"){ echo [event:$0] $1; }
			[comment]# Now see what happens if -b is used[/comment]
			exec -b -e -t -s ("cat /proc/cpuinfo && sed -senseless"){ echo [event:$0] $1; }
			[comment]# Run an iterative script and kill it after 20 seconds[/comment]
			exec -k=20000 ("while true; do sleep 1; echo \"Tic\"; done"){ echo [event:$0] $1; }
			[comment]# Run a blocking process, kill it after 20 seconds[/comment]
			[comment]# and give feedback to the user by the means of ping[/comment]
			exec -k=20000 -p=1000 -t ("cat")
			{
				if($0 == "ping")echo "[event:$0] Please wait while doing a huge computation ..."
				else if($0 == "terminated")echo "[event:$0] Ok, done :)"
			}
			[comment]# Do the same but this time use the extended scope vars[/comment]
			[comment]# Use also a nicer syntax[/comment]
			exec -k=20000 -p=1000 -t ("cat")
			{
				switch($0)
				{
					case("ping"):
					{
						if(%:x == 1)
						{
							%:x = 0;
							echo "Tic!"
						} else {
							%:x = 1;
							echo "Tac!"
						}
					}
					break;
					case("terminated"):
					{
						echo "Ok, done :)"
					}
					break;
				}
			}
			[comment]# Again do the same but kill the process explicitly[/comment]
			exec -x -p=1000 -t ("cat")
			{
				switch($0)
				{
					case("started"):
					{
						[comment]# Initialize the counter[/comment]
						%:x = 10;
					}
					break;
					case("ping"):
					{
						echo %:x
						%:x--
						[comment]# When the counter reaches zero, kill the process with halt[/comment]
						if(%:x == 0)halt;
					}
					break;
					case("terminated"):
					{
						echo "Boom!"
					}
					break;
				}
			}
			[comment]# Now play with an interactive process[/comment]
			[comment]# WARNING: Please note that spam is illegal and generates bad karma[/comment]
			[comment]# Try it only with your own e-mail address as recipient[/comment]
			exec -s -k=60000 -t ("telnet my.mail.server.com 25")
			{
				if($0 == "started")
				{
					%:state = 0
					[comment]# Returning an empty string does not write to stdin[/comment]
					return
				}
				
				if($1 == "stderr")
				{
					echo "[stderr] $1"
					return
				}
				
				if($1 == "terminated")
				{
					echo "[process terminated]"
					return
				}
			
				echo "[stdout] $1"
			
				switch(%:state)
				{
					case(0):
					{
						[comment]# Waiting for 220 (ready)[/comment]
						if($str.match("220*",$1))
						{
							%:state++
							echo "Sending HELO..."
							return "HELO myhostname$cr$lf";
						}
					}
					break
					case(1):
					{
						[comment]# Waiting for 250 (after the HELO)[/comment]
						if($str.match("250*",$1))
						{
							%:state++
							echo "Sending MAIL..."
							return "MAIL From: <myname@mydomain.com>$cr$lf"
						} else {
							echo "HELO command not accepted: $1"
							halt
						}
					}
					break;
					case(2):
					{
						[comment]# Waiting for another 250 (MAIL accepted)[/comment]
						if($str.match("250*",$1))
						{
							%:state++
							echo "Sending RCPT..."
							return "RCPT To: <me@myself.org>$cr$lf"
						} else {
							echo "MAIL command not accepted: $1"
							halt
						}
					}
					break;
					case(3):
					{
						[comment]# Waiting for another 250 (RCPT accepted)[/comment]
						if($str.match("250*",$1))
						{
							%:state++
							echo "Sending DATA..."
							return "DATA$cr$lf"
						} else {
							echo "RCPT not accepted: $1"
							halt
						}
					}
					break;
					case(4):
					{
						[comment]# Waiting for 354 (ok, go on)[/comment]
						if($str.match("354*",$1))
						{
							%:state++
							echo "Sending body..."
							return "This is a test message :)$cr$lf$cr$lf.$cr$lf"
						} else {
							echo "Mail body not accepted: $1"
							halt
						}
					}
					break;
					case(5):
					{
						[comment]# We don't wait anymore :)[/comment]
						%:state++
						echo "Sending QUIT..."
						return "QUIT$cr$lf"
					}
					break;
					default:
					{
						[comment]# Usually the mail server closes the connection[/comment]
						%:state++
						if(%:state > 10)
						{
							[comment]# But if it does not in few messages[/comment]
							[comment]# Then force the process to die[/comment]
							halt
						}
					}
				}
			}
			[/example]
	*/

	ENTER_STACK_FRAME(c,"exec");

	KviParameterList paramList;
	paramList.setAutoDelete(true);

	KviStr cmd;

	if(!parseCallbackCommand(c,&paramList,&cmd))return false;

	KviStr * cmdline = paramList.safeFirst();
	KviStr * magic = paramList.safeNext();

	int f = 0;
	if(c->hasSwitch('t'))f |= KVI_PROCESSDESCRIPTOR_TRIGGERTERMINATED;
	if(!c->hasSwitch('n'))f |= KVI_PROCESSDESCRIPTOR_TRIGGERSTDOUT;
	if(c->hasSwitch('e'))f |= KVI_PROCESSDESCRIPTOR_TRIGGERSTDERR;
	if(c->hasSwitch('x'))f |= KVI_PROCESSDESCRIPTOR_TRIGGERSTARTED;
	if(c->hasSwitch('b'))f |= KVI_PROCESSDESCRIPTOR_OUTPUTBYBLOCKS;
	if(c->hasSwitch('w'))f |= KVI_PROCESSDESCRIPTOR_KILLIFNOWINDOW;
	if(c->hasSwitch('d'))f |= KVI_PROCESSDESCRIPTOR_NOSHELL;
	
	KviStr shell;
	
	if(c->hasSwitch('s'))
	{
		c->getSwitchValue('s',shell);
	}
	
	int iPingTime = 0;
	int iMaxRunTime = 0;
	
	if(c->hasSwitch('p'))
	{
		KviStr tmp;
		c->getSwitchValue('p',tmp);
		bool bOk;
		iPingTime = tmp.toInt(&bOk);
		if(!bOk)
		{
			if(!c->hasSwitch('q'))c->warning(__tr2qs("The specified ping time is invalid: assuming zero (no ping)"));
			iPingTime = 0;
		}
	}

	if(c->hasSwitch('k'))
	{
		KviStr tmp;
		c->getSwitchValue('k',tmp);
		bool bOk;
		iMaxRunTime = tmp.toInt(&bOk);
		if(!bOk)
		{
			if(!c->hasSwitch('q'))c->warning(__tr2qs("The specified maximum run time is invalid: assuming zero (infinite)"));
			iMaxRunTime = 0;
		}
	}

	KviProcessDescriptorData * d = new KviProcessDescriptorData;
	
	d->szCommandline = cmdline->ptr();
	d->szShell = shell.ptr();
	d->pWnd = c->window();
	d->szMagic = magic->ptr();
	d->iFlags = f;
	d->szCallback = cmd;
	d->iMaxRunTime = iMaxRunTime;
	d->iPingTimeout = iPingTime;

	if(!KviProcessManager::instance()->execute(d))
	{
		if(!c->hasSwitch('q'))c->warning(__tr2qs("Failed to start the process"));
	}

	return c->leaveStackFrame();
}
