///////////////////////////////////////////////////////////////////////////////
//
//   File : libkviaction.cpp
//   Creation date : Tue 7 Dec 00:05:59 2002 GMT by Szymon Stefanek
//
//   This toolbar is part of the KVirc irc client distribution
//   Copyright (C) 2004 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
///////////////////////////////////////////////////////////////////////////////

#include "kvi_module.h"
#include "kvi_uparser.h"
#include "kvi_action.h"
#include "kvi_actionmanager.h"
#include "kvi_locale.h"
#include "kvi_qstring.h"
#include "kvi_parameterlist.h"
#include "kvi_command.h"
#include "kvi_cmdformatter.h"
#include "kvi_qstring.h"
#include "kvi_error.h"
#include "kvi_out.h"
#include "kvi_iconmanager.h"
#include "kvi_mirccntrl.h"
#include "kvi_kvs_useraction.h"

#define MODULE_COMMAND(__fncname) static bool __fncname(KviModule *m,KviCommand *c)
#define MODULE_COMMAND_SUCCESS return true;
#define MODULE_COMMAND_FAILURE return false;
#define MODULE_COMMAND_PROLOGUE(__module,__cmdname) ENTER_STACK_FRAME(c,__module "::" __cmdname);
#define MODULE_COMMAND_DECLARE_PARAM(__name) KviStr __name;
#define MODULE_COMMAND_PARAM_TO_QSTRING(__name) QString(__name.ptr())
#define MODULE_COMMAND_GET_FIRST_PARAM(__name) if(!g_pUserParser->parseCmdSingleToken(c,__name))return false;
#define MODULE_COMMAND_DISCARD_REMAINING_PARAMS { KviStr sZdUmMy; if(!g_pUserParser->parseCmdFinalPart(c,sZdUmMy))return false; }
#define MODULE_COMMAND_WARNING(__txt) c->warning(__txt);
#define MODULE_COMMAND_HAS_SWITCH(__char) c->hasSwitch(__char)
#define MODULE_COMMAND_DECLARE_SWITCH(__name) KviStr __name;
#define MODULE_COMMAND_GET_SWITCH_VALUE(__char,__value) c->getSwitchValue(__char,__value);
#define MODULE_COMMAND_SWITCH_VALUE_CONTAINS_CHAR(__sw,__char) (__sw.contains(__char))
#define MODULE_COMMAND_WINDOW c->window()

#define MODULE_CALLBACK_COMMAND(__fncname) static bool __fncname(KviModule *m,KviCommand *c)
#define MODULE_CALLBACK_COMMAND_GET_FIRST_PARAM(__name) __name = paramList.safeFirstParam();
#define MODULE_CALLBACK_COMMAND_GET_NEXT_PARAM(__name) __name = paramList.safeNextParam();
#define MODULE_CALLBACK_COMMAND_DECLARE_PARAM(__name) MODULE_COMMAND_DECLARE_PARAM(__name)
#define MODULE_CALLBACK_COMMAND_DECLARE_SWITCH(__name) MODULE_COMMAND_DECLARE_SWITCH(__name)
#define MODULE_CALLBACK_COMMAND_HAS_SWITCH(__char) MODULE_COMMAND_HAS_SWITCH(__char)
#define MODULE_CALLBACK_COMMAND_GET_SWITCH_VALUE(__char,__value) MODULE_COMMAND_GET_SWITCH_VALUE(__char,__value)
#define MODULE_CALLBACK_COMMAND_SUCCESS MODULE_COMMAND_SUCCESS
#define MODULE_CALLBACK_COMMAND_FAILURE MODULE_COMMAND_FAILURE
#define MODULE_CALLBACK_COMMAND_SWITCH_VALUE_CONTAINS_CHAR(__sw,__char) MODULE_COMMAND_SWITCH_VALUE_CONTAINS_CHAR(__sw,__char)
#define MODULE_CALLBACK_COMMAND_WARNING(__txt) MODULE_COMMAND_WARNING(__txt)
#define MODULE_CALLBACK_COMMAND_CALLBACK_TO_QSTRING(__qstr) __qstr = QString::fromUtf8(begin,c->ptr() - begin); KviCommandFormatter::bufferFromBlock(__qstr);



/*
	@doc: action.list
	@type:
		command
	@title:
		action.list
	@short:
		Lists the available actions
	@syntax:
		action.list
	@description:
		Lists the currently defined actions: nothing more, nothing less.
*/


MODULE_COMMAND(action_module_cmd_list)
{
	MODULE_COMMAND_PROLOGUE("action","list")
	MODULE_COMMAND_DISCARD_REMAINING_PARAMS

	KviWindow * pOut = MODULE_COMMAND_WINDOW;

	QDictIterator<KviAction> it(*(KviActionManager::instance()->actions()));
	while(KviAction * a = it.current())
	{
		if(a->isKviUserActionNeverOverrideThis())
			pOut->output(KVI_OUT_VERBOSE,__tr2qs("%cCore action: %Q"),KVI_TEXT_BOLD,&(a->name()));
		else
			pOut->output(KVI_OUT_VERBOSE,__tr2qs("%cUser action: %Q"),KVI_TEXT_BOLD,&(a->name()));
		pOut->output(KVI_OUT_VERBOSE,__tr2qs("Label: %Q"),&(a->visibleName()));
		pOut->output(KVI_OUT_VERBOSE,__tr2qs("Category: %Q"),&(a->category()->visibleName()));
		pOut->output(KVI_OUT_VERBOSE,__tr2qs("Description: %Q"),&(a->description()));
		pOut->output(KVI_OUT_VERBOSE,"  "); // spacer
		++it;
	}

	MODULE_COMMAND_SUCCESS
}


/*
	@doc: action.trigger
	@type:
		command
	@title:
		action.trigger
	@short:
		Triggers an action
	@syntax:
		action.trigger [-q] <name>
	@description:
		Triggers the action specified by <name>, if that action exists and is enabled.
		If the -q switch is specified the command runs in quiet mode and
		does not print warnings if the action is not defined or disabled.
	@examples:
		[example]
		action.trigger connect
		[/example]
*/


MODULE_COMMAND(action_module_cmd_trigger)
{
	MODULE_COMMAND_PROLOGUE("action","trigger")
	
	MODULE_COMMAND_DECLARE_PARAM(szCoreName)
	MODULE_COMMAND_GET_FIRST_PARAM(szCoreName)
	MODULE_COMMAND_DISCARD_REMAINING_PARAMS

	KviAction * a = KviActionManager::instance()->getAction(MODULE_COMMAND_PARAM_TO_QSTRING(szCoreName));
	if(a)
	{
		if(a->isEnabled())
		{
			a->activate();
		} else {
			if(!MODULE_COMMAND_HAS_SWITCH('q'))MODULE_COMMAND_WARNING(__tr2qs("The action \"%1\" is disabled").arg(MODULE_COMMAND_PARAM_TO_QSTRING(szCoreName)))
		}
	} else {
		MODULE_COMMAND_WARNING(__tr2qs("The action \"%1\" does not exist").arg(MODULE_COMMAND_PARAM_TO_QSTRING(szCoreName)))
		MODULE_COMMAND_SUCCESS
	}

	MODULE_COMMAND_SUCCESS
}

/*
	@doc: action.enable
	@type:
		command
	@title:
		action.enable
	@short:
		Enables an action
	@syntax:
		action.enable [-q] <name>
	@description:
		Enables the action specified by <name>, if that action exists.
		If the action was previously enabled then this command does nothing.
		otherwise all the toolbar buttons and menu items belonging to this
		action will get enabled.[br]
		Since [cmd]action.create[/cmd] command allows specifying contexts
		for actions and the action engine is able to enable/disable the actions
		automatically on certain events you will probably never need to use this command.
		If the -q switch is specified the command runs in quiet mode and
		does not print warnings if the action is not defined.
	@seealso:
		[cmd]action.disable[/cmd]
*/


MODULE_COMMAND(action_module_cmd_enable)
{
	MODULE_COMMAND_PROLOGUE("action","enable")
	
	MODULE_COMMAND_DECLARE_PARAM(szCoreName)
	MODULE_COMMAND_GET_FIRST_PARAM(szCoreName)
	MODULE_COMMAND_DISCARD_REMAINING_PARAMS

	KviAction * a = KviActionManager::instance()->getAction(MODULE_COMMAND_PARAM_TO_QSTRING(szCoreName));
	if(a)
	{
		if(!a->isEnabled())a->setEnabled(true);
	} else {
		MODULE_COMMAND_WARNING(__tr2qs("The action \"%1\" does not exist").arg(MODULE_COMMAND_PARAM_TO_QSTRING(szCoreName)))
		MODULE_COMMAND_SUCCESS
	}

	MODULE_COMMAND_SUCCESS
}


/*
	@doc: action.disable
	@type:
		command
	@title:
		action.disable
	@short:
		Disables an action
	@syntax:
		action.disable [-q] [-f] <name>
	@description:
		Disables the action specified by <name>, if that action exists.
		If the action was previously disabled then this command does nothing.
		otherwise all the toolbar buttons and menu items belonging to this
		action will get disabled.[br]
		Since [cmd]action.create[/cmd] command allows specifying contexts
		for actions and the action engine is able to enable/disable the actions
		automatically on certain events you will probably never need to use this command.
		For the same reason the action may get automatically re-enabled when
		one of the enabling events specified at [cmd]action.create[/cmd] time happens.
		If the -q switch is specified the command runs in quiet mode and
		does not print warnings if the action is not defined.
	@seealso:
		[cmd]action.enable[/cmd]
*/


MODULE_COMMAND(action_module_cmd_disable)
{
	MODULE_COMMAND_PROLOGUE("action","disable")
	
	MODULE_COMMAND_DECLARE_PARAM(szCoreName)
	MODULE_COMMAND_GET_FIRST_PARAM(szCoreName)
	MODULE_COMMAND_DISCARD_REMAINING_PARAMS

	KviAction * a = KviActionManager::instance()->getAction(MODULE_COMMAND_PARAM_TO_QSTRING(szCoreName));
	if(a)
	{
		if(a->isEnabled())a->setEnabled(false);
	} else {
		MODULE_COMMAND_WARNING(__tr2qs("The action \"%1\" does not exist").arg(MODULE_COMMAND_PARAM_TO_QSTRING(szCoreName)))
		MODULE_COMMAND_SUCCESS
	}

	MODULE_COMMAND_SUCCESS
}


/*
	@doc: action.destroy
	@type:
		command
	@title:
		action.destroy
	@short:
		Destroy an action
	@syntax:
		action.destroy [-q] <name>
	@description:
		Destroys the action specified by <name>, if that action exists.[br]
		Destroying an action will remove it from any toolbar or popup.[br]
		If the -q switch is specified the command runs in quiet mode and
		does not print warnings if the action is not defined.
		Please note that you CAN'T destroy core actions, you can destroy
		only script actions that have been defined by the user (or other scripts).
	@seealso:
		# this will trigger a warning and NOT work
		action.destroy connect 
*/


MODULE_COMMAND(action_module_cmd_destroy)
{
	MODULE_COMMAND_PROLOGUE("action","destroy")
	
	MODULE_COMMAND_DECLARE_PARAM(szCoreName)
	MODULE_COMMAND_GET_FIRST_PARAM(szCoreName)
	MODULE_COMMAND_DISCARD_REMAINING_PARAMS

	KviAction * a = KviActionManager::instance()->getAction(MODULE_COMMAND_PARAM_TO_QSTRING(szCoreName));
	if(a)
	{
		if(a->isKviUserActionNeverOverrideThis())
		{
			delete a; // kill it
		} else {
			if(!MODULE_COMMAND_HAS_SWITCH('q'))MODULE_COMMAND_WARNING(__tr2qs("The action \"%1\" is a core action and cannot be destroyed").arg(MODULE_COMMAND_PARAM_TO_QSTRING(szCoreName)))
		}
	} else {
		MODULE_COMMAND_WARNING(__tr2qs("The action \"%1\" does not exist").arg(MODULE_COMMAND_PARAM_TO_QSTRING(szCoreName)))
		MODULE_COMMAND_SUCCESS
	}

	MODULE_COMMAND_SUCCESS
}

// FIXME: In KVS this will end up as a SPECIAL command

/*
	@doc: action.create
	@type:
		command
	@title:
		action.create
	@short:
		Defines a new action
	@syntax:
		action.define [-i] [-c] [-l] [-w=<window_types>] [-s] [-t=<category>] (<name>,<visible label>,<description>,<big_icon_id>[,<small_icon_id>])
		{
			<action body>
		}
	@switches:
		!sw: -i
			Specifies that the action needs an IRC context to be executed.
			When the current window does not belong to an IRC context
			the action is automatically disabled
		!sw: -c
			Specifies that the action needs an IRC connection to be executed.
			When the current window has no associated IRC connection
			the action is automatically disabled.
			This switch implies -i.
		!sw: -l
			Specifies that the action needs to be enabled at login time, that is
			when a link to the server has been estabilished but the login
			operations haven't been carried out yet (and thus there is no real IRC connection).
			This switch requires -c to work.
		!sw: -w=<window_types>
			Causes the action to be enabled only when the active window
			is one of the specified types. <window_types> may be any combination
			of the letters 'q' (query), 'c' (channel), 'x' (console), 'd' (dccchat).
			If this switch is omitted then the action is enabled in all the windows.[br]
		!sw: -s
			Specifies that the action will be activated only if the active window
			has selected users in the userlist. This switch requires -w with a combination
			of flags 'q','c' and 'x' (it doesn't work for dcc chat).
		!sw: -t=<category>
			Causes the action to belong to the specified category.
			<category> can be one of "irc","scripting","settings","gui","channel" and "generic".
			If this switch is omitted the "generic" category is automatically assumed.
		!sw: -k=<key sequence>
			Specifies that the action will be activated also by the <key sequence>
			which is a string formed from up to four keyboard codes separated by
			commas optionally combined with the modifiers "Alt","Ctrl","Shift" and "Meta".[br]
			Examples of such sequences are: "Ctrl+X", "Ctrl+Alt+F" or "Ctrl+X,Alt+Space".
	@description:
		[p]
		Defines a new script action.
		[/p]
		[p]
		Each action has an unique <name> that must not collide with any core action
		(i.e. don't use the "kvirc." prefix).
		At any time you can check [cmd]action.list[/cmd] to verify that no core action
		is already using your <name>. If the <name> was already used for a script action
		then this action is simply replaced by the new one.
		[/p]
		[p]
		Each action has an associated <visible label> that is the name meant to be presented to the user,
		possibly even translated. This label will appear on the toolbar buttons, in the tooltips and
		in the popup menu items. The string will be evaluated just before the actions is displayed
		so the eventual runtime translation will fetch from the correct language catalogue.
		[/p]
		[p]
		<description> is the descriptive text that will be displayed in the action choice dialog
		(and maybe in other places).
		The string will be evaluated just before the actions is displayed
		so the eventual runtime translation will fetch from the correct language catalogue.
		[/p]
		[p]
		<big_icon_id> is the [doc:image_id]image identifier[/doc] of the icon that will
		appear on the toolbar buttons and in the action choice dialog.
		[/p]
		[p]
		<small_icon_id> is optional and is the [doc:image_id]image identifier[/doc] of the icon
		that will appear in the menu items. Since menu items can be also iconless then
		this parameter is optional.
		[/p]
		[p]
		<action body> is the callback code snippet that will be triggered when this action is activated
		either by the means of [cmd]action.trigger[/cmd], a toolbar button or a menu item selection.
		An empty <action body> causes this command to behave like [cmd]action.destoy[/cmd] <name>.
		[/p]
	@seealso:
		[cmd]action.destroy[/cmd], [cmd]action.trigger[/cmd]
	@examples:
		[example]
		
		[/example]
*/


MODULE_CALLBACK_COMMAND(action_module_cmd_create)
{
	ENTER_STACK_FRAME(c,"action::create");
	KviParameterList paramList;
	paramList.setAutoDelete(true);
	c->skipWhiteSpace();
	if(*(c->ptr()) != '(')return c->error(KviError_openParenthesisExpected);
	c->next();
	if(!g_pUserParser->parseFncSingleParam(c,&paramList))return false;
	if(*(c->ptr()) == ',')c->next();
	else return c->error(KviError_notEnoughParameters);
	if(!g_pUserParser->parseFncSingleParam(c,&paramList,true))return false;
	if(*(c->ptr()) == ',')c->next();
	else return c->error(KviError_notEnoughParameters);
	if(!g_pUserParser->parseFncSingleParam(c,&paramList,true))return false;
	if(*(c->ptr()) == ',')c->next();
	else return c->error(KviError_notEnoughParameters);
	while(*(c->ptr()) != ')')
	{
		if(!g_pUserParser->parseFncSingleParam(c,&paramList))return false;
		if(*(c->ptr()) == ',')c->next();
	}
	c->next();
	c->skipWhiteSpace();
	if(*(c->ptr()) != '{')return c->error(KviError_openBraceExpected);
	const char * begin = c->ptr();
	if(!g_pUserParser->skipCommand(c))return false;


	MODULE_CALLBACK_COMMAND_DECLARE_PARAM(szName)
	MODULE_CALLBACK_COMMAND_DECLARE_PARAM(szVisibleText)
	MODULE_CALLBACK_COMMAND_DECLARE_PARAM(szDescription)
	MODULE_CALLBACK_COMMAND_DECLARE_PARAM(szBigIconId)
	MODULE_CALLBACK_COMMAND_DECLARE_PARAM(szSmallIconId)
	MODULE_CALLBACK_COMMAND_DECLARE_SWITCH(szCategory)
	MODULE_CALLBACK_COMMAND_DECLARE_SWITCH(szWindows)
	MODULE_CALLBACK_COMMAND_DECLARE_SWITCH(szKeySequence)

	MODULE_CALLBACK_COMMAND_GET_FIRST_PARAM(szName)
	MODULE_CALLBACK_COMMAND_GET_NEXT_PARAM(szVisibleText)
	MODULE_CALLBACK_COMMAND_GET_NEXT_PARAM(szDescription)
	MODULE_CALLBACK_COMMAND_GET_NEXT_PARAM(szBigIconId)
	MODULE_CALLBACK_COMMAND_GET_NEXT_PARAM(szSmallIconId)

	int iFlags = 0;
	if(MODULE_CALLBACK_COMMAND_HAS_SWITCH('i'))iFlags |= KviAction::NeedsContext;
	if(MODULE_CALLBACK_COMMAND_HAS_SWITCH('c'))
	{
		iFlags |= KviAction::NeedsConnection | KviAction::NeedsContext;
	}
	if(MODULE_CALLBACK_COMMAND_HAS_SWITCH('l'))
	{
		if(iFlags & KviAction::NeedsConnection)
			iFlags |= KviAction::EnableAtLogin;
		else 
			MODULE_CALLBACK_COMMAND_WARNING(__tr2qs("The switch -l requires -c"))
	}
	if(MODULE_CALLBACK_COMMAND_HAS_SWITCH('t'))
	{
		MODULE_CALLBACK_COMMAND_GET_SWITCH_VALUE('t',szCategory)
	} else {
		szCategory = "generic";
	}
	if(MODULE_CALLBACK_COMMAND_HAS_SWITCH('w'))
	{
		MODULE_CALLBACK_COMMAND_GET_SWITCH_VALUE('w',szWindows)
		if(MODULE_CALLBACK_COMMAND_SWITCH_VALUE_CONTAINS_CHAR(szWindows,'c'))iFlags |= KviAction::WindowChannel;
		if(MODULE_CALLBACK_COMMAND_SWITCH_VALUE_CONTAINS_CHAR(szWindows,'x'))iFlags |= KviAction::WindowConsole;
		if(MODULE_CALLBACK_COMMAND_SWITCH_VALUE_CONTAINS_CHAR(szWindows,'d'))iFlags |= KviAction::WindowDccChat;
		if(MODULE_CALLBACK_COMMAND_SWITCH_VALUE_CONTAINS_CHAR(szWindows,'q'))iFlags |= KviAction::WindowQuery;
	}
	if(MODULE_CALLBACK_COMMAND_HAS_SWITCH('s'))
	{
		if(iFlags & (KviAction::WindowChannel | KviAction::WindowConsole | KviAction::WindowQuery))
			iFlags |= KviAction::WindowOnlyIfUsersSelected;
		else 
			MODULE_CALLBACK_COMMAND_WARNING(__tr2qs("The switch -s requires -w with a combination of flags 'c','x' and 'q'"))
	}

	if(MODULE_CALLBACK_COMMAND_HAS_SWITCH('k'))
	{
		MODULE_CALLBACK_COMMAND_GET_SWITCH_VALUE('k',szKeySequence)
	}
	
	QString szCmd;
	MODULE_CALLBACK_COMMAND_CALLBACK_TO_QSTRING(szCmd);


	KviAction * old = KviActionManager::instance()->getAction(MODULE_COMMAND_PARAM_TO_QSTRING(szName));
	if(old)
	{
		if(old->isKviUserActionNeverOverrideThis())
			delete old;
		else {
			MODULE_CALLBACK_COMMAND_WARNING(__tr2qs("The action \"%1\" is already defined as core action and cannot be overridden").arg(MODULE_COMMAND_PARAM_TO_QSTRING(szName)))
			MODULE_CALLBACK_COMMAND_FAILURE
		}
	}

	if(szCmd.isEmpty())
	{
		// just killed it :)
		MODULE_CALLBACK_COMMAND_SUCCESS;
	}

	int iOldFlags = iFlags;
	iFlags = KviAction::validateFlags(iFlags);
	if(iFlags != iOldFlags)
		debug("action.validate has provided invalid flags: %d fixed to %d",iOldFlags,iFlags);

	KviKvsUserAction * a = new KviKvsUserAction(KviActionManager::instance(),
											MODULE_COMMAND_PARAM_TO_QSTRING(szName),
											szCmd,
											MODULE_COMMAND_PARAM_TO_QSTRING(szVisibleText),
											MODULE_COMMAND_PARAM_TO_QSTRING(szDescription),
											MODULE_COMMAND_PARAM_TO_QSTRING(szCategory),
											MODULE_COMMAND_PARAM_TO_QSTRING(szBigIconId),
											MODULE_COMMAND_PARAM_TO_QSTRING(szSmallIconId),
											iFlags,
											MODULE_COMMAND_PARAM_TO_QSTRING(szKeySequence));

	KviActionManager::instance()->registerAction(a);

	MODULE_CALLBACK_COMMAND_SUCCESS
}


#define MODULE_INIT_FUNCTION(__name) static bool __name(KviModule *m)
#define MODULE_REGISTER_CALLBACK_COMMAND(__name,__fnc) m->registerCommand(__name,__fnc);
#define MODULE_REGISTER_COMMAND(__name,__fnc) m->registerCommand(__name,__fnc);

MODULE_INIT_FUNCTION(action_module_init)
{
	// setlabel , $label , $position , move , $itempos , $itemexists , $itemtype

	//m->registerCommand("create",toolbar_module_cmd_create);
	//m->registerCommand("show",toolbar_module_cmd_show);
	//m->registerCommand("hide",toolbar_module_cmd_hide);
	//m->registerCommand("destroy",toolbar_module_cmd_destroy);
	//m->registerCommand("addbutton",toolbar_module_cmd_addbutton);
	//m->registerCommand("addseparator",toolbar_module_cmd_addseparator);
	//m->registerCommand("removeitem",toolbar_module_cmd_removeitem);
	//m->registerCommand("setbutton",toolbar_module_cmd_setbutton);
	//m->registerCommand("define",toolbar_module_cmd_define);

	//m->registerFunction("exists",toolbar_module_fnc_exists);
	//m->registerFunction("list",toolbar_module_fnc_list);

	MODULE_REGISTER_CALLBACK_COMMAND("create",action_module_cmd_create)
	MODULE_REGISTER_COMMAND("destroy",action_module_cmd_destroy)
	MODULE_REGISTER_COMMAND("enable",action_module_cmd_enable)
	MODULE_REGISTER_COMMAND("disable",action_module_cmd_disable)
	MODULE_REGISTER_COMMAND("trigger",action_module_cmd_trigger)
	MODULE_REGISTER_COMMAND("list",action_module_cmd_list)
	return true;
}

static bool action_module_cleanup(KviModule *m)
{
	return true;
}

KVIRC_MODULE(
	"action",                                                      // module name
	"1.0.0",                                                        // module version
	"Copyright (C) 2004 Szymon Stefanek (pragma at kvirc dot net)", // author & (C)
	"Interface to the system actions",
	action_module_init,
	0,
	0,
	action_module_cleanup
)
