//
//   File : broker.cpp
//   Creation date : Tue Sep 19 09 2000 10:21:54 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "broker.h"
#include "dialogs.h"
#include "chat.h"
#include "send.h"
#include "canvas.h"
#include "voice.h"

#include "kvi_app.h"
#include "kvi_frame.h"
#include "kvi_locale.h"
#include "kvi_options.h"
#include "kvi_console.h"
#include "kvi_fileutils.h"
#include "kvi_out.h"
#include "kvi_mediatype.h"
#include "kvi_ircconnection.h"
#include "kvi_sharedfiles.h"

// kvi_app.cpp
extern KVIRC_API KviMediaManager * g_pMediaManager;
extern KVIRC_API KviSharedFilesManager * g_pSharedFilesManager;

#include <qfileinfo.h>


//#warning "The broker might lookup the remote host name"

KviDccBroker::KviDccBroker()
: QObject(0,"dcc_broker")
{
	KviDccFileTransfer::init();

	m_pBoxList = new KviPtrList<KviDccBox>;
	m_pBoxList->setAutoDelete(false);

	m_pDccWindowList = new KviPtrList<KviWindow>;
	m_pDccWindowList->setAutoDelete(false);
}

KviDccBroker::~KviDccBroker()
{
	while(m_pBoxList->first())delete m_pBoxList->first();
	delete m_pBoxList;
	m_pBoxList = 0;
	while(m_pDccWindowList->first())delete m_pDccWindowList->first();
	delete m_pDccWindowList;
	KviDccFileTransfer::done();
}

unsigned int KviDccBroker::dccBoxCount()
{
	return m_pBoxList->count();
}

void KviDccBroker::unregisterDccWindow(KviWindow *wnd)
{
	m_pDccWindowList->removeRef(wnd);
}

void KviDccBroker::unregisterDccBox(KviDccBox * box)
{
	//debug("Forgetting box %d",box);
	m_pBoxList->removeRef(box);
}


void KviDccBroker::cancelDcc(KviDccBox *box,KviDccDescriptor * dcc)
{
	if(box)box->forgetDescriptor();
	delete dcc;
    dcc = 0;
}

///////////////////////////////////////////////////////////////////////////////
// RSEND
///////////////////////////////////////////////////////////////////////////////

void KviDccBroker::rsendManage(KviDccDescriptor * dcc)
{
	// We need the filename...
	QFileInfo fi(dcc->szLocalFileName.ptr());
	if(fi.exists())rsendExecute(0,dcc);
	else rsendAskForFileName(dcc);
}

void KviDccBroker::rsendAskForFileName(KviDccDescriptor * dcc)
{
	KviDccLoadFileBox * box = new KviDccLoadFileBox(this,dcc);
	m_pBoxList->append(box);
	connect(box,SIGNAL(accepted(KviDccBox *,KviDccDescriptor *)),
			this,SLOT(rsendExecute(KviDccBox *,KviDccDescriptor *)));
	connect(box,SIGNAL(rejected(KviDccBox *,KviDccDescriptor *)),
			this,SLOT(cancelDcc(KviDccBox *,KviDccDescriptor *)));
	box->show();
}

void KviDccBroker::rsendExecute(KviDccBox * box,KviDccDescriptor * dcc)
{
	if(box)box->forgetDescriptor();

	if(!g_pApp->windowExists(dcc->console()))
	{
		// No way...we NEED the right IRC context...
		g_pApp->activeConsole()->output(KVI_OUT_DCCERROR,
			__tr2qs_ctx("Can't send DCC %s request to %s: IRC connection has been terminated","dcc"),
			dcc->szType.ptr(),dcc->szNick.ptr());
		delete dcc;
		return;
	}

	// Ok...we need the file to exist
	QFileInfo fi(dcc->szLocalFileName.ptr());
	if(!(fi.exists() && fi.isReadable() && (fi.isFile()) && (fi.size() > 0)))
	{
		dcc->console()->output(KVI_OUT_DCCERROR,__tr2qs_ctx("Can't open file %s for reading","dcc"),
			dcc->szLocalFileName.ptr());
		delete dcc;
		return;
	}

	dcc->szFileName = dcc->szLocalFileName;
	dcc->szFileName.cutToLast('/');
	dcc->szFileName.cutToLast('\\');

	KviStr fName = dcc->szFileName;
	fName.replaceAll(' ',"\\040"); // be cool :)

	dcc->console()->connection()->sendFmtData("PRIVMSG %s :%cDCC %s %s %u%c",
		dcc->szNick.ptr(),0x01,dcc->szType.ptr(),fName.ptr(),fi.size(),0x01);

	// now add a file offer , so he we will accept it automatically
	// 120 secs is a reasonable timeout
	KviStr mask(KviStr::Format,"%s!*@*",dcc->szNick.ptr());

	g_pSharedFilesManager->addSharedFile(dcc->szFileName.ptr(),dcc->szLocalFileName.ptr(),mask.ptr(),120);

	delete dcc;
}


///////////////////////////////////////////////////////////////////////////////
// DCC CHAT
///////////////////////////////////////////////////////////////////////////////

void KviDccBroker::handleChatRequest(KviDccDescriptor * dcc)
{

	if(!dcc->bAutoAccept)
	{
		// FIXME: better message ? Secure Direct Client Connection...eventually
		// need confirmation
		QString tmp = __tr2qs_ctx( \
				"<b>%1 [%2@%3]</b> requests a " \
				"<b>Direct Client Connection</b> in <b>%4</b> mode.<br>", \
				"dcc").arg(dcc->szNick.ptr()).arg(dcc->szUser.ptr()).arg(dcc->szHost.ptr()).arg(dcc->szType.ptr());

#ifdef COMPILE_SSL_SUPPORT
		if(dcc->bIsSSL)tmp += __tr2qs_ctx("The connection will be secured using SSL.<br>","dcc");
#endif

		tmp += __tr2qs_ctx( \
					"The connection target will be host <b>%1</b> on port <b>%2</b><br>" \
					,"dcc").arg(dcc->szIp.ptr()).arg(dcc->szPort.ptr());


		QString caption = __tr2qs_ctx("DCC %1 Request - KVIrc","dcc").arg(dcc->szType.ptr());

		KviDccAcceptBox * box = new KviDccAcceptBox(this,dcc,tmp,caption);

		m_pBoxList->append(box);
		connect(box,SIGNAL(accepted(KviDccBox *,KviDccDescriptor *)),
				this,SLOT(executeChat(KviDccBox *,KviDccDescriptor *)));
		connect(box,SIGNAL(rejected(KviDccBox *,KviDccDescriptor *)),
				this,SLOT(cancelDcc(KviDccBox *,KviDccDescriptor *)));
		box->show();
	} else {
		// auto accept
		executeChat(0,dcc);
	}
}

void KviDccBroker::executeChat(KviDccBox *box,KviDccDescriptor * dcc)
{
	if(box)box->forgetDescriptor();

	if(!g_pApp->windowExists(dcc->console()))
	{
		// rebind to the first available console....
		dcc->setConsole(g_pApp->activeConsole());
	}

	KviStr szSubProto = dcc->szType;
	szSubProto.toLower();

	KviStr tmp(KviStr::Format,"dcc: %s %s@%s:%s",szSubProto.ptr(),dcc->szNick.ptr(),dcc->szIp.ptr(),dcc->szPort.ptr());
	KviDccChat * chat = new KviDccChat(dcc->console()->frame(),dcc,tmp.ptr());

	bool bMinimized = dcc->bOverrideMinimize ? dcc->bShowMinimized : \
			(KVI_OPTION_BOOL(KviOption_boolCreateMinimizedDccChat) || \
				(dcc->bAutoAccept && KVI_OPTION_BOOL(KviOption_boolCreateMinimizedDccChatWhenAutoAccepted)));

	dcc->console()->frame()->addWindow(chat,!bMinimized);
	if(bMinimized)chat->minimize();
	m_pDccWindowList->append(chat);
}

///////////////////////////////////////////////////////////////////////////////
// ACTIVE VOICE
///////////////////////////////////////////////////////////////////////////////

void KviDccBroker::activeVoiceManage(KviDccDescriptor * dcc)
{
	if(!dcc->bAutoAccept)
	{
		// need confirmation
		QString tmp = __tr2qs_ctx(
					"<b>%1 [%2@%3]</b> requests a<br>" \
					"<b>Direct Client Connection</b> in <b>VOICE</b> mode.<br>" \
					"The connection target will be host <b>%4</b> on port <b>%5</b><br>" \
				,"dcc" \
			).arg(dcc->szNick.ptr()).arg(dcc->szUser.ptr()).arg(dcc->szHost.ptr()).arg(dcc->szIp.ptr()).arg(dcc->szPort.ptr());

		KviDccAcceptBox * box = new KviDccAcceptBox(this,dcc,tmp,__tr2qs_ctx("DCC VOICE request","dcc"));
		m_pBoxList->append(box);
		connect(box,SIGNAL(accepted(KviDccBox *,KviDccDescriptor *)),
				this,SLOT(activeVoiceExecute(KviDccBox *,KviDccDescriptor *)));
		connect(box,SIGNAL(rejected(KviDccBox *,KviDccDescriptor *)),
				this,SLOT(cancelDcc(KviDccBox *,KviDccDescriptor *)));
		box->show();
	} else {
		// auto accept
		activeVoiceExecute(0,dcc);
	}
}

void KviDccBroker::activeVoiceExecute(KviDccBox *box,KviDccDescriptor * dcc)
{
	if(box)box->forgetDescriptor();

	if(!g_pApp->windowExists(dcc->console()))
	{
		// rebind to the first available console....
		dcc->setConsole(g_pApp->activeConsole());
	}

	KviStr tmp(KviStr::Format,"dcc: voice %s@%s:%s",dcc->szNick.ptr(),dcc->szIp.ptr(),dcc->szPort.ptr());
	KviDccVoice * v = new KviDccVoice(dcc->console()->frame(),dcc,tmp.ptr());

	bool bMinimized = dcc->bOverrideMinimize ? dcc->bShowMinimized : \
			(KVI_OPTION_BOOL(KviOption_boolCreateMinimizedDccVoice) || \
				(dcc->bAutoAccept && KVI_OPTION_BOOL(KviOption_boolCreateMinimizedDccVoiceWhenAutoAccepted)));

	dcc->console()->frame()->addWindow(v,!bMinimized);
	if(bMinimized)v->minimize();

	m_pDccWindowList->append(v);
}


///////////////////////////////////////////////////////////////////////////////
// PASSIVE VOICE
///////////////////////////////////////////////////////////////////////////////

void KviDccBroker::passiveVoiceExecute(KviDccDescriptor * dcc)
{
	KviStr tmp(KviStr::Format,"dcc: voice %s@%s:%s",dcc->szNick.ptr(),dcc->szIp.ptr(),dcc->szPort.ptr());
	KviDccVoice * v = new KviDccVoice(dcc->console()->frame(),dcc,tmp.ptr());
//#warning "Create minimized dcc voice ?... or maybe it's too much ? :)"
	bool bMinimized = dcc->bOverrideMinimize ? dcc->bShowMinimized : KVI_OPTION_BOOL(KviOption_boolCreateMinimizedDccChat);
	dcc->console()->frame()->addWindow(v,!bMinimized);
	if(bMinimized)v->minimize();
	m_pDccWindowList->append(v);
}


///////////////////////////////////////////////////////////////////////////////
// ACTIVE CANVAS
///////////////////////////////////////////////////////////////////////////////

void KviDccBroker::activeCanvasManage(KviDccDescriptor * dcc)
{
	if(!dcc->bAutoAccept)
	{
		// need confirmation
		QString tmp = __tr2qs_ctx( \
					"<b>%1 [%2@%3]</b> requests a<br>" \
					"<b>Direct Client Connection</b> in <b>CANVAS</b> mode.<br>" \
					"The connection target will be host <b>%4</b> on port <b>%5</b><br>" \
				,"dcc" \
			).arg(dcc->szNick.ptr()).arg(dcc->szUser.ptr()).arg(dcc->szHost.ptr()).arg(dcc->szIp.ptr()).arg(dcc->szPort.ptr());

		KviDccAcceptBox * box = new KviDccAcceptBox(this,dcc,tmp,__tr2qs_ctx("DCC CANVAS request","dcc"));
		m_pBoxList->append(box);
		connect(box,SIGNAL(accepted(KviDccBox *,KviDccDescriptor *)),
				this,SLOT(activeCanvasExecute(KviDccBox *,KviDccDescriptor *)));
		connect(box,SIGNAL(rejected(KviDccBox *,KviDccDescriptor *)),
				this,SLOT(cancelDcc(KviDccBox *,KviDccDescriptor *)));
		box->show();
	} else {
		// auto accept
		activeCanvasExecute(0,dcc);
	}
}

void KviDccBroker::activeCanvasExecute(KviDccBox *box,KviDccDescriptor * dcc)
{
	if(box)box->forgetDescriptor();

	if(!g_pApp->windowExists(dcc->console()))
	{
		// rebind to the first available console....
		dcc->setConsole(g_pApp->activeConsole());
	}

	KviStr tmp(KviStr::Format,"dcc: canvas %s@%s:%s",dcc->szNick.ptr(),dcc->szIp.ptr(),dcc->szPort.ptr());
	KviDccCanvas * cnv = new KviDccCanvas(dcc->console()->frame(),dcc,tmp.ptr());

//#warning "This option should be dedicated to Dcc Canvas!....for now we are using the DccChat options"
	bool bMinimized = dcc->bOverrideMinimize ? dcc->bShowMinimized : \
			(KVI_OPTION_BOOL(KviOption_boolCreateMinimizedDccChat) || \
				(dcc->bAutoAccept && KVI_OPTION_BOOL(KviOption_boolCreateMinimizedDccChatWhenAutoAccepted)));

	dcc->console()->frame()->addWindow(cnv,!bMinimized);
	if(bMinimized)cnv->minimize();

	m_pDccWindowList->append(cnv);
}

///////////////////////////////////////////////////////////////////////////////
// PASSIVE CANVAS
///////////////////////////////////////////////////////////////////////////////

void KviDccBroker::passiveCanvasExecute(KviDccDescriptor * dcc)
{
	KviStr tmp(KviStr::Format,"dcc: canvas %s@%s:%s",dcc->szNick.ptr(),dcc->szIp.ptr(),dcc->szPort.ptr());
	KviDccCanvas * cnv = new KviDccCanvas(dcc->console()->frame(),dcc,tmp.ptr());
//#warning "This option should be dedicated to Dcc Canvas!....for now we are using the DccChat options"
	bool bMinimized = dcc->bOverrideMinimize ? dcc->bShowMinimized : KVI_OPTION_BOOL(KviOption_boolCreateMinimizedDccChat);
	dcc->console()->frame()->addWindow(cnv,!bMinimized);
	if(bMinimized)cnv->minimize();
	m_pDccWindowList->append(cnv);
}


///////////////////////////////////////////////////////////////////////////////
// SEND
///////////////////////////////////////////////////////////////////////////////

void KviDccBroker::recvFileManage(KviDccDescriptor * dcc)
{
	if(!dcc->bAutoAccept)
	{
		// need confirmation
		QString tmp;

		if(dcc->bActive)
		{
			// Normal active send: we will be connecting
			tmp = __tr2qs_ctx( \
						"<b>%1 [%2@%3]</b> " \
						"wants to send you the file " \
						"'<b>%4</b>', " \
						"<b>%5</b> bytes large.<br>" \
						"The connection target will be host <b>%6</b> on port <b>%7</b><br>" \
					,"dcc" \
				).arg(dcc->szNick.ptr()).arg(dcc->szUser.ptr()).arg(dcc->szHost.ptr()).arg(
					dcc->szFileName.ptr()).arg(dcc->szFileSize.ptr()).arg(
					dcc->szIp.ptr()).arg(dcc->szPort.ptr());

		} else {
			// passive: we will be listening!
			tmp = __tr2qs_ctx( \
						"<b>%1 [%2@%3]</b> "
						"wants to send you the file " \
						"'<b>%4</b>', " \
						"<b>%5</b> bytes large.<br>" \
						"You will be the passive side of the connection.<br>" \
					,"dcc" \
				).arg(dcc->szNick.ptr()).arg(dcc->szUser.ptr()).arg(dcc->szHost.ptr()).arg(
					dcc->szFileName.ptr()).arg(dcc->szFileSize.ptr());
		}

		if(dcc->bIsIncomingAvatar)
		{
			tmp += __tr2qs_ctx( \
					"<center><b>Note:</b></center>" \
					"The file appears to be an avatar that you have requested. " \
					"You should not change its filename. " \
					"Save it in a location where KVIrc can find it, such as " \
					"the 'avatars', 'incoming', or 'pics' directories, " \
					"your home directory, or the save directory for the incoming file type. " \
					"The default save path will probably work. " \
					"You can instruct KVIrc to accept incoming avatars automatically " \
					"by setting the option <tt>boolAutoAcceptIncomingAvatars</tt> to true.<br>" \
				,"dcc" \
				);
		}

//#warning "Maybe remove the pending avatar if rejected ?"

		QString title = __tr2qs_ctx("DCC %1 Request - KVIrc","dcc").arg(dcc->szType.ptr());

		KviDccAcceptBox * box = new KviDccAcceptBox(this,dcc,tmp,title);
		m_pBoxList->append(box);
		connect(box,SIGNAL(accepted(KviDccBox *,KviDccDescriptor *)),
				this,SLOT(chooseSaveFileName(KviDccBox *,KviDccDescriptor *)));
		connect(box,SIGNAL(rejected(KviDccBox *,KviDccDescriptor *)),
				this,SLOT(cancelDcc(KviDccBox *,KviDccDescriptor *)));
		box->show();
	} else {
		// auto accept

		if(_OUTPUT_VERBOSE)
		{
			dcc->console()->output(KVI_OUT_DCCMSG,__tr2qs_ctx("Auto-accepting DCC %s request from %s!%s@%s for file %s","dcc"),
				dcc->szType.ptr(),dcc->szNick.ptr(),dcc->szUser.ptr(),
				dcc->szHost.ptr(),dcc->szFileName.ptr());
		}
		chooseSaveFileName(0,dcc);
	}
}

void KviDccBroker::chooseSaveFileName(KviDccBox *box,KviDccDescriptor *dcc)
{
	if(box)box->forgetDescriptor();

	// Lookup the suggested save directory

	dcc->szLocalFileName = "";

	if(dcc->bIsIncomingAvatar)g_pApp->getLocalKvircDirectory(dcc->szLocalFileName,KviApp::Avatars);
	else {

		if(KVI_OPTION_BOOL(KviOption_boolUseIncomingDccMediaTypeSavePath))
		{
			g_pMediaManager->lock();
			if(KviMediaType * mt = g_pMediaManager->findMediaType(dcc->szFileName.ptr(),false))
			{
				if(mt->szSavePath.hasData())
				{
					if(kvi_directoryExists(mt->szSavePath.ptr()))dcc->szLocalFileName = mt->szSavePath;
					else {
						if(kvi_makeDir(mt->szSavePath.ptr()))dcc->szLocalFileName = mt->szSavePath;
					}
				}
			}
			g_pMediaManager->unlock();
		}
	
		if(dcc->szLocalFileName.isEmpty())g_pApp->getLocalKvircDirectory(dcc->szLocalFileName,KviApp::Incoming);
	}

	dcc->szLocalFileName.ensureLastCharIs('/');

	if(!(dcc->bAutoAccept))
	{
		KviDccSaveFileBox * box = new KviDccSaveFileBox(this,dcc);
		m_pBoxList->append(box);
		connect(box,SIGNAL(accepted(KviDccBox *,KviDccDescriptor *)),
				this,SLOT(renameOverwriteResume(KviDccBox *,KviDccDescriptor *)));
		connect(box,SIGNAL(rejected(KviDccBox *,KviDccDescriptor *)),
				this,SLOT(cancelDcc(KviDccBox *,KviDccDescriptor *)));
		box->show();
	} else {
		// auto accept
		// WE choose the filename
		dcc->szLocalFileName.append(dcc->szFileName);

		if(_OUTPUT_VERBOSE)
		{
			dcc->console()->output(KVI_OUT_DCCMSG,__tr2qs_ctx("Auto-saving DCC %s file %s as \r![!dbl]play $0\r%s\r","dcc"),
				dcc->szType.ptr(),dcc->szFileName.ptr(),dcc->szLocalFileName.ptr());
		}

		renameOverwriteResume(0,dcc);
	}
}

void KviDccBroker::renameOverwriteResume(KviDccBox *box,KviDccDescriptor * dcc)
{
	if(box)box->forgetDescriptor();

	// Check if file exists
	QFileInfo fi(dcc->szLocalFileName.ptr());
	if(fi.exists() && (fi.size() > 0)) // 0 byte files are senseless for us
	{
		dcc->szLocalFileSize.setNum(fi.size());
		
		bool bOk;
		int iRemoteSize = dcc->szFileSize.toInt(&bOk);
		if(!bOk)iRemoteSize = -1;

		// FIXME: Files downloaded succesfully shouldn't be resumed
		//        we should keep a db of downloaded files!

		if(!dcc->bAutoAccept)
		{
			QString tmp;
			bool bDisableResume = false;
			
			if((iRemoteSize > -1) || // remote size is unknown
				(iRemoteSize > ((int)(fi.size())))) // or it is larger than the actual size on disk
			{
				tmp = __tr2qs_ctx( \
							"The file '<b>%1</b>' already exists" \
							"and is <b>%2</b> bytes large.<br>" \
							"Do you wish to<br>" \
							"<b>overwrite</b> the existing file,<br> " \
							"<b>auto-rename</b> the new file, or<br>" \
							"<b>resume</b> an incomplete download?" \
						,"dcc" \
					).arg(dcc->szLocalFileName.ptr()).arg(fi.size());
			} else {
				bDisableResume = true;
				// the file on disk is larger or equal to the remote one
				tmp = __tr2qs_ctx( \
							"The file '<b>%1</b>' already exists" \
							"and is larger than the offered one.<br>" \
							"Do you wish to<br>" \
							"<b>overwrite</b> the existing file, or<br> " \
							"<b>auto-rename</b> the new file ?" \
						,"dcc" \
					).arg(dcc->szLocalFileName.ptr());
			}

			KviDccRenameBox * box = new KviDccRenameBox(this,dcc,tmp,bDisableResume);
			m_pBoxList->append(box);
			connect(box,SIGNAL(renameSelected(KviDccBox *,KviDccDescriptor *)),
					this,SLOT(renameDccSendFile(KviDccBox *,KviDccDescriptor *)));
			connect(box,SIGNAL(overwriteSelected(KviDccBox *,KviDccDescriptor *)),
					this,SLOT(recvFileExecute(KviDccBox *,KviDccDescriptor *)));
			connect(box,SIGNAL(cancelSelected(KviDccBox *,KviDccDescriptor *)),
					this,SLOT(cancelDcc(KviDccBox *,KviDccDescriptor *)));
			box->show();
			return;
		} else {
			// auto resume ?
			if(KVI_OPTION_BOOL(KviOption_boolAutoResumeDccSendWhenAutoAccepted) &&
				(iRemoteSize > -1) && // only if the remote size is really known
				(iRemoteSize > ((int)(fi.size()))) && // only if the remote size is larger than the local size
				(!KviDccFileTransfer::nonFailedTransferWithLocalFileName(dcc->szLocalFileName.ptr()))) // only if there is no transfer with this local file name yet
			{
				// yep, auto resume...
				dcc->bResume = true;
				recvFileExecute(0,dcc);
			} else {
				// otherwise auto rename
				renameDccSendFile(0,dcc);
			}
			return;
		}
	} else dcc->szLocalFileSize = "0";

	// everything OK
	recvFileExecute(0,dcc);
}

void KviDccBroker::renameDccSendFile(KviDccBox *box,KviDccDescriptor * dcc)
{
	if(box)box->forgetDescriptor();


	if(kvi_fileExists(dcc->szLocalFileName.ptr()))
	{
		KviStr szOrig = dcc->szLocalFileName;
		int i = 1;
		do {
			KviStr szNum;
			szNum.setNum(i);
			int idx = szOrig.findLastIdx('.');
			if(idx != -1)
			{
				dcc->szLocalFileName = szOrig.left(idx);
				dcc->szLocalFileName += ".";
				dcc->szLocalFileName += szNum;
				dcc->szLocalFileName += szOrig.right(szOrig.len() - idx);
			} else {
				dcc->szLocalFileName = szOrig;
				dcc->szLocalFileName += ".";
				dcc->szLocalFileName += szNum;
			}
			i++;
		} while(kvi_fileExists(dcc->szLocalFileName.ptr()));

		if(_OUTPUT_VERBOSE)
		{
			dcc->console()->output(KVI_OUT_DCCMSG,__tr2qs_ctx("File %s exists, auto-renaming to %s","dcc"),
				szOrig.ptr(),dcc->szLocalFileName.ptr());
		}
	}
	
	dcc->szLocalFileSize = "0"; // 0 for sure

	recvFileExecute(0,dcc);
}

void KviDccBroker::recvFileExecute(KviDccBox *box,KviDccDescriptor * dcc)
{
	if(box)box->forgetDescriptor();

	if(!g_pApp->windowExists(dcc->console()))
	{
		// rebind to the first available console....
		dcc->setConsole(g_pApp->activeConsole());
	}

//	KviStr szSubProto = dcc->szType;
//	szSubProto.toLower();

//	KviStr tmp(KviStr::Format,"dcc: %s %s@%s:%s %s",szSubProto.ptr(),dcc->szNick.ptr(),dcc->szIp.ptr(),dcc->szPort.ptr(),dcc->szLocalFileName.ptr());

	//KviDccSend * send = new KviDccSend(dcc->console()->frame(),dcc,tmp.ptr());
	KviDccFileTransfer * send = new KviDccFileTransfer(dcc);

	bool bMinimized = dcc->bOverrideMinimize ? dcc->bShowMinimized : \
			(KVI_OPTION_BOOL(KviOption_boolCreateMinimizedDccSend) || \
				(dcc->bAutoAccept && KVI_OPTION_BOOL(KviOption_boolCreateMinimizedDccSendWhenAutoAccepted)));

	send->invokeTransferWindow(dcc->console(),bMinimized,bMinimized);

	//dcc->console()->frame()->addWindow(send,!bMinimized);
	//if(bMinimized)send->minimize();
	//m_pDccWindowList->append(send);
}


void KviDccBroker::sendFileManage(KviDccDescriptor * dcc)
{
	KviDccLoadFileBox * box = new KviDccLoadFileBox(this,dcc);
	m_pBoxList->append(box);
	connect(box,SIGNAL(accepted(KviDccBox *,KviDccDescriptor *)),
			this,SLOT(sendFileExecute(KviDccBox *,KviDccDescriptor *)));
	connect(box,SIGNAL(rejected(KviDccBox *,KviDccDescriptor *)),
			this,SLOT(cancelDcc(KviDccBox *,KviDccDescriptor *)));
	box->show();
}

void KviDccBroker::sendFileExecute(KviDccBox * box,KviDccDescriptor *dcc)
{
	if(box)box->forgetDescriptor();

	if(!g_pApp->windowExists(dcc->console()))
	{
		// rebind to the first available console....
		dcc->setConsole(g_pApp->activeConsole());
	}

	QFileInfo fi(dcc->szLocalFileName.ptr());
	if(!(fi.exists() && fi.isReadable() && (fi.isFile()) && (fi.size() > 0)))
	{
		dcc->console()->output(KVI_OUT_DCCERROR,__tr2qs_ctx("Can't open file %s for reading","dcc"),
			dcc->szLocalFileName.ptr());
		delete dcc;
		return;
	}

	dcc->szFileName = dcc->szLocalFileName;
	dcc->szFileName.cutToLast('/');    // cut both!
	dcc->szFileName.cutToLast('\\');   // cut both!

	dcc->szLocalFileSize.setNum(fi.size());

//	KviStr szSubProto = dcc->szType;
//	szSubProto.toLower();

	//KviStr tmp(KviStr::Format,"dcc: %s %s@%s:%s %s",szSubProto.ptr(),dcc->szNick.ptr(),dcc->szIp.ptr(),dcc->szPort.ptr(),dcc->szLocalFileName.ptr());
//	KviDccSend * send = new KviDccSend(dcc->console()->frame(),dcc,tmp.ptr());
	KviDccFileTransfer * send = new KviDccFileTransfer(dcc);

	bool bMinimized = dcc->bOverrideMinimize ? dcc->bShowMinimized : KVI_OPTION_BOOL(KviOption_boolCreateMinimizedDccSend);

	send->invokeTransferWindow(dcc->console(),bMinimized,bMinimized);

//	dcc->console()->frame()->addWindow(send,!bMinimized);
//	if(bMinimized)send->minimize();
//	m_pDccWindowList->append(send);
}

bool KviDccBroker::canUnload()
{
	if(m_pBoxList)
	{
		if((m_pBoxList->count() != 0) ||
			(m_pDccWindowList->count() != 0) ||
			(KviDccFileTransfer::transferCount() != 0))return false;
	} // else in the destructor anyway (going to die)
	return true;
}

bool KviDccBroker::handleResumeAccepted(const char * filename,const char * port)
{
	return KviDccFileTransfer::handleResumeAccepted(filename,port);
}

bool KviDccBroker::handleResumeRequest(const char * filename,const char * port,unsigned int filePos)
{
	return KviDccFileTransfer::handleResumeRequest(filename,port,filePos);
}


#include "m_broker.moc"
