//==================================================================================
//
//   File : libkvihttp.cpp
//   Creation date : Tue Apr 22 2003 02:00:12 GMT by Szymon Stefanek
//
//   This config is part of the KVirc irc client distribution
//   Copyright (C) 2003 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//==================================================================================

#include "kvi_module.h"
#include "kvi_string.h"
#include "kvi_uparser.h"
#include "kvi_command.h"
#include "kvi_app.h"
#include "kvi_locale.h"
#include "kvi_filedialog.h"
#include "kvi_window.h"

#include "httpfiletransfer.h"


/*
	@doc: http.get
	@type:
		command
	@title:
		http.get
	@keyterms:
		HTTP extension
	@short:
		Retrieves a file via HTTP GET
	@syntax:
		http.get [switches] <http_url> [save_file_name]
	@description:
		Attempts to download the file at <http_url> by using the HTTP GET protocol.[br]
		If [save_file_name] is specified, then is is used as save file name, otherwise
		a save file dialog is displayed (unless -a is used).[br]
	@switches:
		!sw: -a=<auto_file_name>
		Don't show the savefile dialog but determine automatically a file name.
		The file is put in the KVIrc incoming directory and the file name
		is the processed url.[br]
		!sw: -e=<existing_file_action>
		Specifies the action to be taken when the local file already exists.[br]
		The action can be one of "i","e","o" or "r".[br]
		"i" causes the incoming file to be automatically renamed by appending a
		non colliding suffix. "e" causes the existing file to be renamed
		by appending a non colliding suffix (the incoming file will then have its name preserved).
		"o" causes the existing file to be overwritten and "r" will attempt to resume
		a interrupted transfer.[br]
		The default is to rename the incoming file.[br]
		!sw: -m=<max_content_length>
		Causes content longer than <max_content_length> to be discarded.[br]
		This is mainly to prevent you from automatically downloading 300 MB files
		and to prevent DOS attacks from malicious servers that do not report the Content-length header.[br]
		If the Content-length header is reported by the server then the transfer is aborted
		if the length exceedes <max_content_length>.[br]
		If the Content-length header is missing then the transfer is interrupted when
		the received data length exceeds <max_content_length>.[br]
		-m=0 means "accept any content length" (which is the default).[br]
		!sw: -o=<content_offset>
		Causes the download to start from position <content offset>.[br]
		This can be used to download only a part of the file starting at byte <content_offset>.[br]
		<content_offset> is used regardless if the file is resumed or not.
		Please note that you don't need to specify the content offset when using
		-e=r : the offset is automatically calculated. If you specify both -o=<content_offset>
		and -e=r then the file will be resumed, the transfer will start at the specified offset
		and the received stream will be appended to the existing file.(avoid it unless you know what you're doing:
		it's easy to download broken files).[br]
		!sw: -h
		Causes the connection to use the HTTP HEAD method that effectively
		does not transfer real data. The server sends only the response headers.
		This might be used in conjunction with the -v option to print the headers to the
		active window.[br]
		!sw: -w=<flags>
		This switch controls the creation and visualization of the transfer window.
		<flags> can be any combination of 'm','n' and 'h'.
		The flag 'h' causes the window to not be created. The transfer will simply run in background.
		Note that with 'h' the user has no possibility to interact with the transfer.
		The flag 'm' causes the transfer window to be created as "minimized". 'm' does nothing
		if the window alread exists. The flag 'n' causes the window to be NOT activated (brought to top).
	@seealso:
*/

static bool http_module_cmd_get(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"http::get");

	KviStr szUrl;
	KviStr szFileName;

	if(!g_pUserParser->parseCmdSingleToken(c,szUrl))return false;
	if(!g_pUserParser->parseCmdFinalPart(c,szFileName))return false;

	if(szUrl.isEmpty())
	{
		c->warning(__tr_ctx("No url specified","http"));
		return c->leaveStackFrame();
	}

	KviUrl url(szUrl.ptr());

	KviStr tmp;

	if(szFileName.isEmpty())
	{
		if(c->hasSwitch('a'))
		{
			tmp = szUrl;
			tmp.replaceAll('/',"_");
			tmp.replaceAll(':',"_");
			tmp.replaceAll('@',"_");
			// http____path_path2_path3_filename.ext
			g_pApp->getLocalKvircDirectory(szFileName,KviApp::Incoming,tmp.ptr());
		} else {
			if(!KviFileDialog::askForSaveFileName(szFileName,__tr2qs_ctx("Choose a save file name","http")))
				return c->leaveStackFrame();
			if(szFileName.isEmpty())return c->leaveStackFrame();
		}
	}

	KviHttpFileTransfer * hft = new KviHttpFileTransfer();

	hft->request()->setUrl(url);
	hft->request()->setProcessingType(c->hasSwitch('h') ? KviHttpRequest::HeadersOnly : KviHttpRequest::StoreToFile);
	hft->request()->setFileName(szFileName.ptr());

	if(c->getSwitchValue('e',tmp))
	{
		if(tmp.equalsCI("e"))
			hft->request()->setExistingFileAction(KviHttpRequest::RenameExisting);
		else if(tmp.equalsCI("i"))
			hft->request()->setExistingFileAction(KviHttpRequest::RenameIncoming);
		else if(tmp.equalsCI("o"))
			hft->request()->setExistingFileAction(KviHttpRequest::Overwrite);
		else if(tmp.equalsCI("r"))
			hft->request()->setExistingFileAction(KviHttpRequest::Resume);
	}

	if(c->getSwitchValue('m',tmp))
	{
		bool bOk;
		unsigned int uContentLength = tmp.toUInt(&bOk);
		if(bOk)hft->request()->setMaxContentLength(uContentLength);
	}

	if(c->getSwitchValue('o',tmp))
	{
		bool bOk;
		unsigned int uContentOffset = tmp.toUInt(&bOk);
		if(bOk)hft->request()->setContentOffset(uContentOffset);
	}

	if(c->getSwitchValue('w',tmp))
	{
		if(!tmp.contains('h'))hft->invokeTransferWindow(c->window(),tmp.contains('m'),tmp.contains('n'));
	} else {
		hft->invokeTransferWindow(c->window(),false,false);
	}

	if(!hft->startDownload())
	{
		KviStr tmp = hft->request()->lastError();
		c->warning(__tr_ctx("Failed to start the get request: %s","http"),tmp.ptr());
		delete hft;
	}

	return c->leaveStackFrame();
}


static bool http_module_init(KviModule * m)
{
	KviHttpFileTransfer::init();
	m->registerCommand("get",http_module_cmd_get);
	return true;
}

static bool http_module_cleanup(KviModule *m)
{
	KviHttpFileTransfer::done();
	return true;
}

static bool http_module_can_unload(KviModule *m)
{
	return (KviHttpFileTransfer::runningTransfers() == 0);
}

KVIRC_MODULE(
	"Http",                                               // module name
	"1.0.0",                                                // module version
	"Copyright (C) 2003 Szymon Stefanek (pragma at kvirc dot net)", // author & (C)
	"HTTP interface for KVIrc",
	http_module_init,
	http_module_can_unload,
	0,
	http_module_cleanup
)
