//
//   File : libkvimp3player.cpp
//   Creation date : Sat Nov  3 19:28:25 2001 GMT by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2001 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "kvi_module.h"
#include "kvi_library.h"
#include "kvi_locale.h"
#include "kvi_uparser.h"

#include "kvi_settings.h"

#include "libkvimp3player.h"

#include <qstring.h>


// need this for the mp3info header

#include "mp3tech.h"

/*
	@doc: mp3player
	@type:
		module
	@short:
		Interface to winamp/xmms
	@title:
		The mp3player interface module
	@body:
		The mp3player module is an interface to two widely user multimedia players:
		winamp on windows and xmms on unix.[br]
		On windows , the Winamp mp3 player is required , to make this module work.[br]
		You must install the "gen_kvirc.dll" plugin (included in the kvirc distribution)
		as a "general" winamp plugin. This is achieved by simpy copying the file gen_kvirc.dll
		to the winamp plugins directory.[br]
		On unix , the xmms mp3 player is required but no special installation is needed:
		kvirc will dynamically communicate with the xmms executable while it is running.[br]
		This module exports the following commands and functions:[br]
		[cmd]mp3player.play[/cmd] , 
      [cmd]mp3player.stop[/cmd] , 
      [cmd]mp3player.pause[/cmd] ,
		[cmd]mp3player.prev[/cmd] , 
      [cmd]mp3player.next[/cmd] , 
      [cmd]mp3player.quit[/cmd] ,
		[cmd]mp3player.add[/cmd] , 
      [cmd]mp3player.addurl[/cmd] ,
		[fnc]$mp3player.gettitle[/fnc] , 
      [fnc]$mp3player.getfilename[/fnc] , 
      [fnc]$mp3player.getlength[/fnc] ,
		[fnc]$mp3player.getfreq[/fnc] , 
      [fnc]$mp3player.getchannels[/fnc] , 
      [fnc]$mp3player.gettime[/fnc] ,
		[fnc]$mp3player.getposition[/fnc] , 
      [fnc]$mp3player.getrate[/fnc] , 
      [fnc]$mp3player.isplaying[/fnc] , 
		[fnc]$mp3player.ispaused[/fnc] , 
      [fnc]$mp3player.getmp3tag_title[/fnc] , 
      [fnc]$mp3player.getmp3tag_artist[/fnc] ,
      [fnc]$mp3player.getmp3tag_album[/fnc] , 
      [fnc]$mp3player.getmp3tag_date[/fnc] , 
      [fnc]$mp3player.getmp3tag_comment[/fnc] ,
      [fnc]$mp3player.getmp3tag_genre[/fnc] , 
      [fnc]$mp3player.getmp3tag_version[/fnc] , 
      [fnc]$mp3player.getmp3tag_layer[/fnc] ,
      [fnc]$mp3player.getmp3tag_crc[/fnc] , 
      [fnc]$mp3player.getmp3tag_copyright[/fnc] , 
      [fnc]$mp3player.getmp3tag_original[/fnc] ,
      [fnc]$mp3player.getmp3tag_emphasis[/fnc] , 
      [fnc]$mp3player.getmp3tag_tracknumber[/fnc]
*/


#ifdef COMPILE_ON_WINDOWS

	#define IPC_STARTPLAY 102
	#define IPC_ISPLAYING 104

	#define IPC_GETOUTPUTTIME 105
	//	** int res = SendMessage(hwnd_winamp,WM_WA_IPC,mode,IPC_GETOUTPUTTIME);
	//	**
	//	** IPC_GETOUTPUTTIME returns the position in milliseconds of the 
	//	** current song (mode = 0), or the song length, in seconds (mode = 1).
	//	** Returns -1 if not playing or error.

	#define IPC_SETPLAYLISTPOS 121
	//	** SendMessage(hwnd_winamp,WM_WA_IPC,position,IPC_SETPLAYLISTPOS)
	//	** IPC_SETPLAYLISTPOS sets the playlsit position to 'position'.

	#define IPC_GETLISTLENGTH 124
	//	** int length = SendMessage(hwnd_winamp,WM_WA_IPC,0,IPC_GETLISTLENGTH);
	//	** IPC_GETLISTLENGTH returns the length of the current playlist, in
	//	** tracks.

	#define IPC_GETPLAYLISTFILE 211
	//	** (requires Winamp 2.04+, only usable from plug-ins (not external apps))
	//	** char *name=SendMessage(hwnd_winamp,WM_WA_IPC,index,IPC_GETPLAYLISTFILE);
	//	** IPC_GETPLAYLISTFILE gets the filename of the playlist entry [index].
	//	** returns a pointer to it. returns NULL on error.

	#define IPC_GETPLAYLISTTITLE 212
	//	* (requires Winamp 2.04+, only usable from plug-ins (not external apps))
	//	** char *name=SendMessage(hwnd_winamp,WM_WA_IPC,index,IPC_GETPLAYLISTTITLE);
	//	** IPC_GETPLAYLISTTITLE gets the title of the playlist entry [index].
	//	** returns a pointer to it. returns NULL on error.

	#define IPC_GETLISTPOS 125
	//	** int pos=SendMessage(hwnd_winamp,WM_WA_IPC,0,IPC_GETLISTPOS);
	//	** IPC_GETLISTPOS returns the playlist position. A lot like IPC_WRITEPLAYLIST
	//	** only faster since it doesn't have to write out the list. Heh, silly me.

	#define IPC_GETINFO 126 
	//	** int inf=SendMessage(hwnd_winamp,WM_WA_IPC,mode,IPC_GETINFO);
	//	** IPC_GETINFO returns info about the current playing song. The value
	//	** it returns depends on the value of 'mode'.
	//	** Mode      Meaning
	//	** ------------------
	//	** 0         Samplerate (i.e. 44100)
	//	** 1         Bitrate  (i.e. 128)
	//	** 2         Channels (i.e. 2)

	#define IPC_PLAYFILE 100
	//	** COPYDATASTRUCT cds;
	//	** cds.dwData = IPC_PLAYFILE;
	//	** cds.lpData = (void *) "file.mp3";
	//	** cds.cbData = strlen((char *) cds.lpData)+1; // include space for null char
	//	** SendMessage(hwnd_winamp,WM_COPYDATA,(WPARAM)NULL,(LPARAM)&cds);
	//	**
	//	** This will play the file "file.mp3".

	#define IPC_ISPLAYING 104
	//	** int res = SendMessage(hwnd_winamp,WM_WA_IPC,0,IPC_ISPLAYING);
	//	** IPC_ISPLAYING returns the status of playback.
	//	** If it returns 1, it is playing. if it returns 3, it is paused, 
	//	** if it returns 0, it is not playing.



	/* The following stuff needs PostMessage(hwnd_winamp,WM_COMMAND,id,0); */
	#define WINAMP_BUTTON1                  40044
	#define WINAMP_BUTTON2                  40045
	#define WINAMP_BUTTON3                  40046
	#define WINAMP_BUTTON4                  40047
	#define WINAMP_BUTTON5                  40048

	#define WINAMP_CMD_PREV WINAMP_BUTTON1
	#define WINAMP_CMD_PLAY WINAMP_BUTTON2
	#define WINAMP_CMD_PAUSE WINAMP_BUTTON3
	#define WINAMP_CMD_STOP WINAMP_BUTTON4
	#define WINAMP_CMD_NEXT WINAMP_BUTTON5

	#define WINAMP_CMD_QUIT 40001


	#define KVIRC_WM_USER 63112

	#define KVIRC_WM_USER_CHECK 13123
	#define KVIRC_WM_USER_CHECK_REPLY 13124
	#define KVIRC_WM_USER_GETTITLE 5000
	#define KVIRC_WM_USER_GETFILE 10000
	#define KVIRC_WM_USER_TRANSFER 15000

	static HWND find_winamp(KviCommand *c)
	{
		HWND hWnd = FindWindow("Winamp v1.x",NULL);
		if(!hWnd)
		{
			// try to start the process ?
			if(c)c->warning(__tr("Can't find a running winamp window"));
		}
		return hWnd;
	}
#else
	static kvi_library_t g_hXmmsLib = 0;

	static kvi_library_t load_xmms_lib()
	{
		if(g_hXmmsLib)return g_hXmmsLib;
		g_hXmmsLib = kvi_library_load("libxmms.so");
		if(!g_hXmmsLib)g_hXmmsLib = kvi_library_load("libxmms.so.1"); // try this one too
		return g_hXmmsLib;
	}

   
	static void * lookup_xmms_symbol(KviCommand *c,const char * sym)
	{
		if(!load_xmms_lib())
		{
			c->warning(__tr("Can't load the xmms library (libxmms.so)"));
			return 0;
		}
		void * symptr = kvi_library_symbol(g_hXmmsLib,sym);
		if(!symptr)
		{
			c->warning(__tr("Can't find symbol %s in libxmms.so"),sym);
		}
		return symptr;
	}

#endif

/*
	@doc: mp3player.play
	@type:
		command
	@title:
		mp3player.play
	@short:
		Starts playback (mp3player interface module)
	@syntax:
		mp3player.play
	@description:
		Starts playback in the remote instance of the slave mp3 player.[br]
		Take a look at the [module:mp3player]mp3player module documentation[/module]
		for more details about how it works.[br]
	@seealso:
		[module:mp3player]mp3 player module documentation[/module]
*/


static bool mp3player_cmd_play(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"mp3player_cmd_play");
	KviStr dummy;
	if(!g_pUserParser->parseCmdFinalPart(c,dummy))return false;
#ifdef COMPILE_ON_WINDOWS
	HWND hWinamp = find_winamp(c);
	if(hWinamp)SendMessage(hWinamp,WM_USER,0,IPC_STARTPLAY);
#else
	void (*sym)(int) = (void (*)(int))lookup_xmms_symbol(c,"xmms_remote_play");
	if(sym)sym(0);
#endif
	return c->leaveStackFrame();
}

/*
	@doc: mp3player.stop
	@type:
		command
	@title:
		mp3player.stop
	@short:
		Stops playback (mp3player interface module)
	@syntax:
		mp3player.stop
	@description:
		Stops playback in the remote instance of the slave mp3 player.[br]
		Take a look at the [module:mp3player]mp3player module documentation[/module]
		for more details about how it works.[br]
	@seealso:
		[module:mp3player]mp3 player module documentation[/module]
*/

static bool mp3player_cmd_stop(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"mp3player_cmd_stop");
	KviStr dummy;
	if(!g_pUserParser->parseCmdFinalPart(c,dummy))return false;
#ifdef COMPILE_ON_WINDOWS
	HWND hWinamp = find_winamp(c);
	if(hWinamp)SendMessage(hWinamp,WM_COMMAND,WINAMP_CMD_STOP,0);
#else
	void (*sym)(int) = (void (*)(int))lookup_xmms_symbol(c,"xmms_remote_stop");
	if(sym)sym(0);
#endif
	return c->leaveStackFrame();
}

/*
	@doc: mp3player.next
	@type:
		command
	@title:
		mp3player.next
	@short:
		Starts playback of the next song (mp3player interface module)
	@syntax:
		mp3player.next
	@description:
		Starts playback of the next song in the remote instance of the slave mp3 player.[br]
		Take a look at the [module:mp3player]mp3player module documentation[/module]
		for more details about how it works.[br]
	@seealso:
		[module:mp3player]mp3 player module documentation[/module]
*/

static bool mp3player_cmd_next(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"mp3player_cmd_next");
	KviStr dummy;
	if(!g_pUserParser->parseCmdFinalPart(c,dummy))return false;
#ifdef COMPILE_ON_WINDOWS
	HWND hWinamp = find_winamp(c);
	if(hWinamp)SendMessage(hWinamp,WM_COMMAND,WINAMP_CMD_NEXT,0);
#else
	void (*sym)(int) = (void (*)(int))lookup_xmms_symbol(c,"xmms_remote_playlist_next");
	if(sym)sym(0);
#endif
	return c->leaveStackFrame();
}

/*
	@doc: mp3player.prev
	@type:
		command
	@title:
		mp3player.prev
	@short:
		Starts playback of the previous song (mp3player interface module)
	@syntax:
		mp3player.prev
	@description:
		Starts playback of the previous song in the remote instance of the slave mp3 player.[br]
		Take a look at the [module:mp3player]mp3player module documentation[/module]
		for more details about how it works.[br]
	@seealso:
		[module:mp3player]mp3 player module documentation[/module]
*/

static bool mp3player_cmd_prev(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"mp3player_cmd_prev");
	KviStr dummy;
	if(!g_pUserParser->parseCmdFinalPart(c,dummy))return false;
#ifdef COMPILE_ON_WINDOWS
	HWND hWinamp = find_winamp(c);
	if(hWinamp)SendMessage(hWinamp,WM_COMMAND,WINAMP_CMD_PREV,0);
#else
	void (*sym)(int) = (void (*)(int))lookup_xmms_symbol(c,"xmms_remote_playlist_prev");
	if(sym)sym(0);
#endif
	return c->leaveStackFrame();
}

/*
	@doc: mp3player.pause
	@type:
		command
	@title:
		mp3player.pause
	@short:
		Pauses playback (mp3player interface module)
	@syntax:
		mp3player.pause
	@description:
		Pauses playback in the remote instance of the slave mp3 player.[br]
		Take a look at the [module:mp3player]mp3player module documentation[/module]
		for more details about how it works.[br]
	@seealso:
		[module:mp3player]mp3 player module documentation[/module]
*/

static bool mp3player_cmd_pause(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"mp3player_cmd_pause");
	KviStr dummy;
	if(!g_pUserParser->parseCmdFinalPart(c,dummy))return false;
#ifdef COMPILE_ON_WINDOWS
	HWND hWinamp = find_winamp(c);
	if(hWinamp)SendMessage(hWinamp,WM_COMMAND,WINAMP_CMD_PAUSE,0);
#else
	void (*sym)(int) = (void (*)(int))lookup_xmms_symbol(c,"xmms_remote_pause");
	if(sym)sym(0);
#endif
	return c->leaveStackFrame();
}

/*
	@doc: mp3player.quit
	@type:
		command
	@title:
		mp3player.quit
	@short:
		Quits the remote mp3 player (mp3player interface module)
	@syntax:
		mp3player.quit
	@description:
		Quits the remote instance of the slave mp3 player.[br]
		Take a look at the [module:mp3player]mp3player module documentation[/module]
		for more details about how it works.[br]
	@seealso:
		[module:mp3player]mp3 player module documentation[/module]
*/

static bool mp3player_cmd_quit(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"mp3player_cmd_quit");
	KviStr dummy;
	if(!g_pUserParser->parseCmdFinalPart(c,dummy))return false;
#ifdef COMPILE_ON_WINDOWS
	HWND hWinamp = find_winamp(c);
	if(hWinamp)SendMessage(hWinamp,WM_COMMAND,WINAMP_CMD_QUIT,0);
#else
	void (*sym)(int) = (void (*)(int))lookup_xmms_symbol(c,"xmms_remote_quit");
	if(sym)sym(0);
#endif
	return c->leaveStackFrame();
}

#ifndef COMPILE_ON_WINDOWS
	typedef struct _KviGListDummy
	{
	 	void * data;
	 	struct _KviGListDummy *next;
	 	struct _KviGListDummy *prev;
	} KviGListDummy;
#endif

/*
	@doc: mp3player.add
	@type:
		command
	@title:
		mp3player.add
	@short:
		Adds a file to the playlist (mp3player interface module)
	@syntax:
		mp3player.add(<filename>)
	@description:
		Adds the <filename> to the playlist of the remote instance of the slave mp3 player.[br]
		Take a look at the [module:mp3player]mp3player module documentation[/module]
		for more details about how it works.[br]
	@seealso:
		[module:mp3player]mp3 player module documentation[/module]
*/

static bool mp3player_cmd_add(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"mp3player_cmd_add");
	KviStr fName;
	if(!g_pUserParser->parseCmdFinalPart(c,fName))return false;
#ifdef COMPILE_ON_WINDOWS
	HWND hWinamp = find_winamp(c);
	if(hWinamp && fName.hasData())
	{
		COPYDATASTRUCT cds;
		cds.dwData = IPC_PLAYFILE;
		cds.lpData = (void *)fName.ptr();
		cds.cbData = fName.len() + 1; // include space for null char
		SendMessage(hWinamp,WM_COPYDATA,(WPARAM)NULL,(LPARAM)&cds);
	}
#else
	void (*sym)(int,KviGListDummy *) = (void (*)(int,KviGListDummy *))lookup_xmms_symbol(c,"xmms_remote_playlist_add");

	if(fName.hasData())
	{
		KviGListDummy d;
		d.data = fName.ptr();
		d.prev = 0;
		d.next = 0;
		if(sym)sym(0,&d);
	}
#endif
	return c->leaveStackFrame();
}

/*
	@doc: mp3player.addurl
	@type:
		command
	@title:
		mp3player.addurl
	@short:
		Adds an url to the playlist (mp3player interface module)
	@syntax:
		mp3player.addurl(<url>)
	@description:
		Adds the <url> to the playlist of the remote instance of the slave mp3 player.[br]
		Take a look at the [module:mp3player]mp3player module documentation[/module]
		for more details about how it works.[br]
	@seealso:
		[module:mp3player]mp3 player module documentation[/module]
*/

static bool mp3player_cmd_addurl(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"mp3player_cmd_addurl");
	KviStr fName;
	if(!g_pUserParser->parseCmdFinalPart(c,fName))return false;
#ifdef COMPILE_ON_WINDOWS
	HWND hWinamp = find_winamp(c);
	if(hWinamp && fName.hasData())
	{
		COPYDATASTRUCT cds;
		cds.dwData = IPC_PLAYFILE;
		cds.lpData = (void *)fName.ptr();
		cds.cbData = fName.len() + 1; // include space for null char
		SendMessage(hWinamp,WM_COPYDATA,(WPARAM)NULL,(LPARAM)&cds);
	}
#else
	void (*sym)(int,char *) = (void (*)(int,char *))lookup_xmms_symbol(c,"xmms_remote_playlist_add_url_string");

	if(fName.hasData())
	{
		if(sym)sym(0,fName.ptr());
	}
#endif
	return c->leaveStackFrame();
}

/*
	@doc: mp3player.gettitle
	@type:
		function
	@title:
		$mp3player.gettitle
	@short:
		Returns the title of the current song (mp3player module)
	@syntax:
		$mp3player.gettitle
	@description:
		Returns the title of the multimedia file being played by the
		remote instance of the mp3 player.[br]
		Take a look at the [module:mp3player]mp3player module documentation[/module]
		for more details about how it works.[br]
	@seealso:
		[module:mp3player]mp3 player module documentation[/module]
*/


static bool mp3player_fnc_gettitle(KviModule *m,KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"mp3player_fnc_gettitle");
#ifdef COMPILE_ON_WINDOWS
	HWND hWinamp = find_winamp(c);
	if(hWinamp)
	{
		int ret = SendMessage(hWinamp,WM_USER,KVIRC_WM_USER,KVIRC_WM_USER_CHECK);
		if(ret != KVIRC_WM_USER_CHECK_REPLY)
		{
			c->warning(__tr("The winamp plugin has not been installed properly. Check /help $mp3player.gettitle"));
			return c->leaveStackFrame();
		}

		int len = SendMessage(hWinamp,WM_USER,KVIRC_WM_USER,KVIRC_WM_USER_GETTITLE);

		if(len < 4096)
		{
			char szBuffer[4096];

			for(int i=0;i<len;i++)
			{
				szBuffer[i] = SendMessage(hWinamp,WM_USER,KVIRC_WM_USER,KVIRC_WM_USER_TRANSFER + i);
			}
			szBuffer[len] = '\0';
			buffer.append(szBuffer);
		}
	}
#else
	int (*sym)(int) = (int (*)(int))lookup_xmms_symbol(c,"xmms_remote_get_playlist_pos");

	if(!sym)return c->leaveStackFrame();

	int pos = sym(0);

	char * (*sym2)(int,int) = (char * (*)(int,int))lookup_xmms_symbol(c,"xmms_remote_get_playlist_title");
	if(!sym2)return c->leaveStackFrame();

	buffer.append(sym2(0,pos));
#endif
	return c->leaveStackFrame();
}

/*
	@doc: mp3player.getfilename
	@type:
		function
	@title:
		$mp3player.getfilename
	@short:
		Returns the filename of the current song (mp3player module)
	@syntax:
		$mp3player.getfilename
	@description:
		Returns the file name (with path) of the multimedia file being played by the
		remote instance of the mp3 player.[br]
		Take a look at the [module:mp3player]mp3player module documentation[/module]
		for more details about how it works.[br]
	@seealso:
		[module:mp3player]mp3 player module documentation[/module]
*/

static bool mp3player_fnc_getfilename(KviModule *m,KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"mp3player_fnc_getfilename");
#ifdef COMPILE_ON_WINDOWS
	HWND hWinamp = find_winamp(c);
	if(hWinamp)
	{
		int ret = SendMessage(hWinamp,WM_USER,KVIRC_WM_USER,KVIRC_WM_USER_CHECK);

		if(ret != KVIRC_WM_USER_CHECK_REPLY)
		{
			c->warning(__tr("The winamp plugin has not been installed properly. Check /help $mp3player.gettitle"));
			return c->leaveStackFrame();
		}

		int len = SendMessage(hWinamp,WM_USER,KVIRC_WM_USER,KVIRC_WM_USER_GETFILE);

		if(len < 4096)
		{
			char szBuffer[4096];

			for(int i=0;i<len;i++)
			{
				szBuffer[i] = SendMessage(hWinamp,WM_USER,KVIRC_WM_USER,KVIRC_WM_USER_TRANSFER + i);
			}
			szBuffer[len] = '\0';
			buffer.append(szBuffer);
		}
	}
#else
	int (*sym)(int) = (int (*)(int))lookup_xmms_symbol(c,"xmms_remote_get_playlist_pos");

	if(!sym)return c->leaveStackFrame();

	int pos = sym(0);

	char * (*sym2)(int,int) = (char * (*)(int,int))lookup_xmms_symbol(c,"xmms_remote_get_playlist_file");
	if(!sym2)return c->leaveStackFrame();

	buffer.append(sym2(0,pos));
#endif
	return c->leaveStackFrame();
}

/*
	@doc: mp3player.getposition
	@type:
		function
	@title:
		$mp3player.getposition
	@short:
		Returns the position of the current song (mp3player module)
	@syntax:
		$mp3player.getposition
	@description:
		Returns the position in the playlist of the multimedia file being played by the
		remote instance of the mp3 player.[br]
		Take a look at the [module:mp3player]mp3player module documentation[/module]
		for more details about how it works.[br]
	@seealso:
		[module:mp3player]mp3 player module documentation[/module]
*/

static bool mp3player_fnc_getposition(KviModule *m,KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"mp3player_fnc_getposition");
#ifdef COMPILE_ON_WINDOWS
	HWND hWinamp = find_winamp(c);
	if(hWinamp)
	{
		int pos = SendMessage(hWinamp,WM_USER,0,IPC_GETLISTPOS);
		buffer.append(KviStr::Format,"%d",pos);
	}
#else
	int (*sym)(int) = (int (*)(int))lookup_xmms_symbol(c,"xmms_remote_get_playlist_pos");

	if(sym)
	{
		int pos = sym(0);
		buffer.append(KviStr::Format,"%d",pos);
	}
#endif
	return c->leaveStackFrame();
}

/*
	@doc: mp3player.getlength
	@type:
		function
	@title:
		$mp3player.getlength
	@short:
		Returns the length of the current song (mp3player module)
	@syntax:
		$mp3player.getlength
	@description:
		Returns the length in milliseconds of the multimedia file being played by the
		remote instance of the mp3 player.[br]
		Take a look at the [module:mp3player]mp3player module documentation[/module]
		for more details about how it works.[br]
	@seealso:
		[module:mp3player]mp3 player module documentation[/module]
*/

static bool mp3player_fnc_getlength(KviModule *m,KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"mp3player_fnc_getlength");
#ifdef COMPILE_ON_WINDOWS
	HWND hWinamp = find_winamp(c);
	if(hWinamp)
	{
		int leninsecs = SendMessage(hWinamp,WM_USER,1,IPC_GETOUTPUTTIME);
		buffer.append(KviStr::Format,"%d",leninsecs * 1000);
	}
#else
	int (*sym)(int) = (int (*)(int))lookup_xmms_symbol(c,"xmms_remote_get_playlist_pos");

	if(!sym)return c->leaveStackFrame();

	int pos = sym(0);

	int (*sym2)(int,int) = (int (*)(int,int))lookup_xmms_symbol(c,"xmms_remote_get_playlist_time");
	if(!sym2)return c->leaveStackFrame();

	buffer.append(KviStr::Format,"%d",sym2(0,pos));
#endif
	return c->leaveStackFrame();
}

/*
	@doc: mp3player.gettime
	@type:
		function
	@title:
		$mp3player.gettime
	@short:
		Returns the time of the current song (mp3player module)
	@syntax:
		$mp3player.gettime
	@description:
		Returns the currently elapsed time of the multimedia file being played by the
		remote instance of the mp3 player.[br]
		Take a look at the [module:mp3player]mp3player module documentation[/module]
		for more details about how it works.[br]
	@seealso:
		[module:mp3player]mp3 player module documentation[/module]
*/

static bool mp3player_fnc_gettime(KviModule *m,KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"mp3player_fnc_gettime");
#ifdef COMPILE_ON_WINDOWS
	HWND hWinamp = find_winamp(c);
	if(hWinamp)
	{
		int leninmsecs = SendMessage(hWinamp,WM_USER,0,IPC_GETOUTPUTTIME);
		buffer.append(KviStr::Format,"%d",leninmsecs);
	}
#else
	int (*sym)(int) = (int (*)(int))lookup_xmms_symbol(c,"xmms_remote_get_playlist_pos");

	if(!sym)return c->leaveStackFrame();

	int pos = sym(0);

	int (*sym2)(int,int) = (int (*)(int,int))lookup_xmms_symbol(c,"xmms_remote_get_output_time");
	if(!sym2)return c->leaveStackFrame();

	buffer.append(KviStr::Format,"%d",sym2(0,pos));
#endif
	return c->leaveStackFrame();
}

/*
	@doc: mp3player.getrate
	@type:
		function
	@title:
		$mp3player.getrate
	@short:
		Returns the rate of the current song (mp3player module)
	@syntax:
		$mp3player.getrate
	@description:
		Returns the rate of the multimedia file being played by the
		remote instance of the mp3 player.[br]
		Take a look at the [module:mp3player]mp3player module documentation[/module]
		for more details about how it works.[br]
	@seealso:
		[module:mp3player]mp3 player module documentation[/module]
*/


static bool mp3player_fnc_getrate(KviModule *m,KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"mp3player_fnc_getrate");
#ifdef COMPILE_ON_WINDOWS
	HWND hWinamp = find_winamp(c);
	if(hWinamp)
	{
		int leninmsecs = SendMessage(hWinamp,WM_USER,0,IPC_GETINFO);
		buffer.append(KviStr::Format,"%d",leninmsecs);
	}
#else
	void (*sym)(int,int *,int *,int *) = (void (*)(int,int *,int *,int *))lookup_xmms_symbol(c,"xmms_remote_get_info");

	if(sym)
	{
		int i1,i2,i3;
		sym(0,&i1,&i2,&i3);
		buffer.append(KviStr::Format,"%d",i1);
	}
#endif
	return c->leaveStackFrame();
}

/*
	@doc: mp3player.getchannels
	@type:
		function
	@title:
		$mp3player.getchannels
	@short:
		Returns the number of channels of the current song (mp3player module)
	@syntax:
		$mp3player.getchannels
	@description:
		Returns the number of channels of the multimedia file being played by the
		remote instance of the mp3 player.[br]
		Take a look at the [module:mp3player]mp3player module documentation[/module]
		for more details about how it works.[br]
	@seealso:
		[module:mp3player]mp3 player module documentation[/module]
*/

static bool mp3player_fnc_getchannels(KviModule *m,KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"mp3player_fnc_getchannels");
#ifdef COMPILE_ON_WINDOWS
	HWND hWinamp = find_winamp(c);
	if(hWinamp)
	{
		int leninmsecs = SendMessage(hWinamp,WM_USER,2,IPC_GETINFO);
		buffer.append(KviStr::Format,"%d",leninmsecs);
	}
#else
	void (*sym)(int,int *,int *,int *) = (void (*)(int,int *,int *,int *))lookup_xmms_symbol(c,"xmms_remote_get_info");

	if(sym)
	{
		int i1,i2,i3;
		sym(0,&i1,&i2,&i3);
		buffer.append(KviStr::Format,"%d",i3);
	}
#endif
	return c->leaveStackFrame();
}

/*
	@doc: mp3player.getfreq
	@type:
		function
	@title:
		$mp3player.getfreq
	@short:
		Returns the frequency of the current song (mp3player module)
	@syntax:
		$mp3player.getfreq
	@description:
		Returns the frequency of the multimedia file being played by the
		remote instance of the mp3 player.[br]
		Take a look at the [module:mp3player]mp3player module documentation[/module]
		for more details about how it works.[br]
	@seealso:
		[module:mp3player]mp3 player module documentation[/module]
*/

static bool mp3player_fnc_getfreq(KviModule *m,KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"mp3player_fnc_getfreq");
#ifdef COMPILE_ON_WINDOWS
	HWND hWinamp = find_winamp(c);
	if(hWinamp)
	{
		int leninmsecs = SendMessage(hWinamp,WM_USER,1,IPC_GETINFO);
		buffer.append(KviStr::Format,"%d",leninmsecs);
	}
#else
	void (*sym)(int,int *,int *,int *) = (void (*)(int,int *,int *,int *))lookup_xmms_symbol(c,"xmms_remote_get_info");

	if(sym)
	{
		int i1,i2,i3;
		sym(0,&i1,&i2,&i3);
		buffer.append(KviStr::Format,"%d",i2);
	}
#endif
	return c->leaveStackFrame();
}

/*
	@doc: mp3player.isplaying
	@type:
		function
	@title:
		$mp3player.isplaying
	@short:
		Checks the state of the mp3 player (mp3player module)
	@syntax:
		$mp3player.isplaying
	@description:
		Returns 1 if the remote instance of the mp3 player is in playback mode , 0 otherwise.[br]
		Take a look at the [module:mp3player]mp3player module documentation[/module]
		for more details about how it works.[br]
	@seealso:
		[module:mp3player]mp3 player module documentation[/module]
*/

static bool mp3player_fnc_isplaying(KviModule *m,KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"mp3player_fnc_isplaying");
#ifdef COMPILE_ON_WINDOWS
	HWND hWinamp = find_winamp(c);
	if(hWinamp)
	{
		int ret = SendMessage(hWinamp,WM_USER,0,IPC_ISPLAYING);
		buffer.append(KviStr::Format,"%d",(ret == 1) ? 1 : 0);
	}
#else
	bool (*sym)(int) = (bool (*)(int))lookup_xmms_symbol(c,"xmms_remote_is_playing");

	if(sym)
	{
		int pos = sym(0);
		buffer.append(KviStr::Format,"%d",pos);
	}
#endif
	return c->leaveStackFrame();
}

/*
	@doc: mp3player.ispaused
	@type:
		function
	@title:
		$mp3player.ispaused
	@short:
		Checks the state of the mp3 player (mp3player module)
	@syntax:
		$mp3player.ispaused
	@description:
		Returns 1 if the remote instance of the mp3 player is in pause mode , 0 otherwise.[br]
		Take a look at the [module:mp3player]mp3player module documentation[/module]
		for more details about how it works.[br]
	@seealso:
		[module:mp3player]mp3 player module documentation[/module]
*/

static bool mp3player_fnc_ispaused(KviModule *m,KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"mp3player_fnc_ispaused");
#ifdef COMPILE_ON_WINDOWS
	HWND hWinamp = find_winamp(c);
	if(hWinamp)
	{
		int ret = SendMessage(hWinamp,WM_USER,0,IPC_ISPLAYING);
		buffer.append(KviStr::Format,"%d",(ret == 3) ? 1 : 0);
	}
#else
	bool (*sym)(int) = (bool (*)(int))lookup_xmms_symbol(c,"xmms_remote_is_paused");

	if(sym)
	{
		int pos = sym(0);
		buffer.append(KviStr::Format,"%d",pos);
	}
#endif
	return c->leaveStackFrame();
}



static bool GetThatNameJohnny(KviCommand * c,KviStr &filename)
{
   #ifdef COMPILE_ON_WINDOWS
	HWND hWinamp = find_winamp(c);
	if(hWinamp)
	{
		int ret = SendMessage(hWinamp,WM_USER,KVIRC_WM_USER,KVIRC_WM_USER_CHECK);

		if(ret != KVIRC_WM_USER_CHECK_REPLY)
		{
			c->warning(__tr("The winamp plugin has not been installed properly. Check /help $mp3player.gettitle"));
			return false;
		}

		int len = SendMessage(hWinamp,WM_USER,KVIRC_WM_USER,KVIRC_WM_USER_GETFILE);

		if(len < 4096)
		{
			char szBuffer[4096];

			for(int i=0;i<len;i++)
			{
				szBuffer[i] = SendMessage(hWinamp,WM_USER,KVIRC_WM_USER,KVIRC_WM_USER_TRANSFER + i);
			}
			szBuffer[len] = '\0';
			filename.append(szBuffer);
		}
	}
#else
	int (*sym)(int) = (int (*)(int))lookup_xmms_symbol(c,"xmms_remote_get_playlist_pos");

	if(!sym)return false;

	int pos = sym(0);

	char * (*sym2)(int,int) = (char * (*)(int,int))lookup_xmms_symbol(c,"xmms_remote_get_playlist_file");
	if(!sym2)return false;

	filename.append(sym2(0,pos));
#endif

	return true;
}

static bool GetThatInfoJohnny(KviCommand * c,mp3info * inf)
{
	KviStr filename;
	if(!GetThatNameJohnny(c,filename))return false;

	memset(inf,0,sizeof(mp3info));
	inf->filename = filename.ptr(); //strdup(filename);
        
	inf->file = fopen(inf->filename, "rb");
	if(!inf->file)return false;

	get_mp3_info(inf,SCAN_QUICK,VBR_VARIABLE);
  
	fclose(inf->file);
	return true;
}


/*
	@doc: mp3player.getmp3tag_title
	@type:
		function
	@title:
		$mp3player.getmp3tag_title
	@short:
		reads title entry from mp3 tag (mp3player module)
	@syntax:
		$mp3player.getmp3tag_title
	@description:
		Returns 1 if successful , 0 otherwise.[br]
		Take a look at the [module:mp3player]mp3player module documentation[/module]
		for more details about how it works.[br]
	@seealso:
		[module:mp3player]mp3 player module documentation[/module]
*/

static bool mp3player_fnc_getmp3tag_title(KviModule *m,KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"mp3player_fnc_getmp3tag_title");
	mp3info mp3;
	if(GetThatInfoJohnny(c,&mp3))
	{
		if(mp3.id3_isvalid)buffer.append(mp3.id3.title);
	}

	return c->leaveStackFrame();
}


/*
	@doc: mp3player.getmp3tag_album
	@type:
		function
	@title:
		$mp3player.getmp3tag_album
	@short:
		reads album entry from mp3 tag (mp3player module)
	@syntax:
		$mp3player.getmp3tag_album
	@description:
		Returns 1 if successful , 0 otherwise.[br]
		Take a look at the [module:mp3player]mp3player module documentation[/module]
		for more details about how it works.[br]
	@seealso:
		[module:mp3player]mp3 player module documentation[/module]
*/

static bool mp3player_fnc_getmp3tag_album(KviModule *m,KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"mp3player_fnc_getmp3tag_album");
	mp3info mp3;
	if(GetThatInfoJohnny(c,&mp3))
	{
		if(mp3.id3_isvalid)buffer.append(mp3.id3.album);
	} 

	return c->leaveStackFrame();
}


/*
	@doc: mp3player.getmp3tag_date
	@type:
		function
	@title:
		$mp3player.getmp3tag_date
	@short:
		reads year entry from mp3 tag (mp3player module)
	@syntax:
		$mp3player.getmp3tag_date
	@description:
		Returns 1 if successful , 0 otherwise.[br]
		Take a look at the [module:mp3player]mp3player module documentation[/module]
		for more details about how it works.[br]
	@seealso:
		[module:mp3player]mp3 player module documentation[/module]
*/

static bool mp3player_fnc_getmp3tag_date(KviModule *m,KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"mp3player_fnc_getmp3tag_date");
	mp3info mp3;
	if(GetThatInfoJohnny(c,&mp3))
	{
		if(mp3.id3_isvalid)buffer.append(mp3.id3.year);
	}
    return c->leaveStackFrame();
}

/*
	@doc: mp3player.getmp3tag_comment
	@type:
		function
	@title:
		$mp3player.getmp3tag_comment
	@short:
		reads comment entry from mp3 tag (mp3player module)
	@syntax:
		$mp3player.getmp3tag_comment
	@description:
		Returns 1 if successful , 0 otherwise.[br]
		Take a look at the [module:mp3player]mp3player module documentation[/module]
		for more details about how it works.[br]
	@seealso:
		[module:mp3player]mp3 player module documentation[/module]
*/

static bool mp3player_fnc_getmp3tag_comment(KviModule *m,KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"mp3player_fnc_getmp3tag_comment");
	mp3info mp3;
	if(GetThatInfoJohnny(c,&mp3))
	{
		if(mp3.id3_isvalid)buffer.append(mp3.id3.comment);
	}
    return c->leaveStackFrame();
}


/*
	@doc: mp3player.getmp3tag_genre
	@type:
		function
	@title:
		$mp3player.getmp3tag_genre
	@short:
		reads genre entry from mp3 tag (mp3player module)
	@syntax:
		$mp3player.getmp3tag_genre
	@description:
		Returns 1 if successful , 0 otherwise.[br]
		Take a look at the [module:mp3player]mp3player module documentation[/module]
		for more details about how it works.[br]
	@seealso:
		[module:mp3player]mp3 player module documentation[/module]
*/

static bool mp3player_fnc_getmp3tag_genre(KviModule *m,KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"mp3player_fnc_getmp3tag_genre");
	mp3info mp3;
	if(GetThatInfoJohnny(c,&mp3))
	{
		if(mp3.id3_isvalid)buffer.append(get_typegenre(mp3.id3.genre[0]));
	}
    return c->leaveStackFrame();
}

/*
	@doc: mp3player.getmp3tag_artist
	@type:
		function
	@title:
		$mp3player.getmp3tag_artist
	@short:
		reads artist entry from mp3 tag (mp3player module)
	@syntax:
		$mp3player.getmp3tag_artist
	@description:
		Returns 1 if successful , 0 otherwise.[br]
		Take a look at the [module:mp3player]mp3player module documentation[/module]
		for more details about how it works.[br]
	@seealso:
		[module:mp3player]mp3 player module documentation[/module]
*/

static bool mp3player_fnc_getmp3tag_artist(KviModule *m,KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"mp3player_fnc_getmp3tag_artist");
	mp3info mp3;
	if(GetThatInfoJohnny(c,&mp3))
	{
		if(mp3.id3_isvalid)buffer.append(mp3.id3.artist);
	}    

	return c->leaveStackFrame();
}

/*
	@doc: mp3player.getmp3tag_version
	@type:
		function
	@title:
		$mp3player.getmp3tag_version
	@short:
		reads mpeg version from mp3 tag (mp3player module)
	@syntax:
		$mp3player.getmp3tag_version
	@description:
		Returns 1 if successful , 0 otherwise.[br]
		Take a look at the [module:mp3player]mp3player module documentation[/module]
		for more details about how it works.[br]
	@seealso:
		[module:mp3player]mp3 player module documentation[/module]
*/

static bool mp3player_fnc_getmp3tag_version(KviModule *m,KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"mp3player_fnc_getmp3tag_version");
	mp3info mp3;
	if(GetThatInfoJohnny(c,&mp3))
	{
		if(mp3.id3_isvalid)buffer.append(KviStr::Format,"%d",mp3.header.version);
	}
    return c->leaveStackFrame();
}

/*
	@doc: mp3player.getmp3tag_layer
	@type:
		function
	@title:
		$mp3player.getmp3tag_layer
	@short:
		reads mpeg layer entry from mp3 tag (mp3player module)
	@syntax:
		$mp3player.getmp3tag_layer
	@description:
		Returns 1 if successful , 0 otherwise.[br]
		Take a look at the [module:mp3player]mp3player module documentation[/module]
		for more details about how it works.[br]
	@seealso:
		[module:mp3player]mp3 player module documentation[/module]
*/

static bool mp3player_fnc_getmp3tag_layer(KviModule *m,KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"mp3player_fnc_getmp3tag_layer");
	mp3info mp3;
	if(GetThatInfoJohnny(c,&mp3))
	{
		int ret = header_layer(&mp3.header);
		if(mp3.id3_isvalid && ret)buffer.append(KviStr::Format,"%d",ret);
	}
    return c->leaveStackFrame();
}

/*
	@doc: mp3player.getmp3tag_crc
	@type:
		function
	@title:
		$mp3player.getmp3tag_crc
	@short:
		reads crc entry from mp3 tag (mp3player module)
	@syntax:
		$mp3player.getmp3tag_crc
	@description:
		Returns 1 if successful , 0 otherwise.[br]
		Take a look at the [module:mp3player]mp3player module documentation[/module]
		for more details about how it works.[br]
	@seealso:
		[module:mp3player]mp3 player module documentation[/module]
*/

static bool mp3player_fnc_getmp3tag_crc(KviModule *m,KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"mp3player_fnc_getmp3tag_crc");
	mp3info mp3;
	if(GetThatInfoJohnny(c,&mp3))
	{
		int ret = header_crc(&mp3.header);
		if(mp3.id3_isvalid)buffer.append(KviStr::Format,"%d",ret);
	}
    return c->leaveStackFrame();
}

/*
	@doc: mp3player.getmp3tag_copright
	@type:
		function
	@title:
		$mp3player.getmp3tag_copright
	@short:
		reads copright entry from mp3 tag (mp3player module)
	@syntax:
		$mp3player.getmp3tag_copright
	@description:
		Returns 1 if successful , 0 otherwise.[br]
		Take a look at the [module:mp3player]mp3player module documentation[/module]
		for more details about how it works.[br]
	@seealso:
		[module:mp3player]mp3 player module documentation[/module]
*/

static bool mp3player_fnc_getmp3tag_copright(KviModule *m,KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"mp3player_fnc_getmp3tag_copright");
	mp3info mp3;
	if(GetThatInfoJohnny(c,&mp3))
	{
		if(mp3.id3_isvalid && mp3.header.copyright != 0)buffer.append(KviStr::Format,"%d",mp3.header.copyright);
	}
    return c->leaveStackFrame();  
   
} 

/*
	@doc: mp3player.getmp3tag_original
	@type:
		function
	@title:
		$mp3player.getmp3tag_original
	@short:
		reads original entry from mp3 tag (mp3player module)
	@syntax:
		$mp3player.getmp3tag_original
	@description:
		Returns 1 if successful , 0 otherwise.[br]
		Take a look at the [module:mp3player]mp3player module documentation[/module]
		for more details about how it works.[br]
	@seealso:
		[module:mp3player]mp3 player module documentation[/module]
*/

static bool mp3player_fnc_getmp3tag_original(KviModule *m,KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"mp3player_fnc_getmp3tag_original");
	mp3info mp3;
	if(GetThatInfoJohnny(c,&mp3))
	{
		if(mp3.id3_isvalid)buffer.append(KviStr::Format,"%d",mp3.header.original);
	}
    return c->leaveStackFrame();
   
} 

/*
	@doc: mp3player.getmp3tag_emphasis
	@type:
		function
	@title:
		$mp3player.getmp3tag_emphasis
	@short:
		reads emphasis entry from mp3 tag (mp3player module)
	@syntax:
		$mp3player.getmp3tag_emphasis
	@description:
		Returns 1 if successful , 0 otherwise.[br]
		Take a look at the [module:mp3player]mp3player module documentation[/module]
		for more details about how it works.[br]
	@seealso:
		[module:mp3player]mp3 player module documentation[/module]
*/

static bool mp3player_fnc_getmp3tag_emphasis(KviModule *m,KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"mp3player_fnc_getmp3tag_emphasis");
	mp3info mp3;
	if(GetThatInfoJohnny(c,&mp3))
	{
		const char * ret = header_emphasis(&mp3.header);
		if(mp3.id3_isvalid && ret)buffer.append(ret);
	}
    return c->leaveStackFrame();
 
} 

/*
	@doc: mp3player.getmp3tag_tracknumber
	@type:
		function
	@title:
		$mp3player.getmp3tag_tracknumber
	@short:
		reads tracknumber entry from mp3 tag (mp3player module)
	@syntax:
		$mp3player.getmp3tag_tracknumber
	@description:
		Returns 1 if successful , 0 otherwise.[br]
		Take a look at the [module:mp3player]mp3player module documentation[/module]
		for more details about how it works.[br]
	@seealso:
		[module:mp3player]mp3 player module documentation[/module]
*/

static bool mp3player_fnc_getmp3tag_tracknumber(KviModule *m,KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"mp3player_fnc_getmp3tag_tracknumber");
	mp3info mp3;
	if(GetThatInfoJohnny(c,&mp3))
	{
		if(mp3.id3_isvalid)buffer.append(KviStr::Format,"%d",mp3.id3.track[0]);
	}
    return c->leaveStackFrame();
 
}


static bool mp3player_module_init(KviModule *m)
{
	m->registerCommand("play",mp3player_cmd_play);
	m->registerCommand("stop",mp3player_cmd_stop);
	m->registerCommand("next",mp3player_cmd_next);
	m->registerCommand("prev",mp3player_cmd_prev);
	m->registerCommand("pause",mp3player_cmd_pause);
	m->registerCommand("add",mp3player_cmd_add);
	m->registerCommand("addurl",mp3player_cmd_addurl);
	m->registerCommand("quit",mp3player_cmd_quit);
	m->registerFunction("gettitle",mp3player_fnc_gettitle);
	m->registerFunction("getposition",mp3player_fnc_getposition);
	m->registerFunction("getfilename",mp3player_fnc_getfilename);
	m->registerFunction("getrate",mp3player_fnc_getrate);
	m->registerFunction("getfreq",mp3player_fnc_getfreq);
	m->registerFunction("getchannels",mp3player_fnc_getchannels);
	m->registerFunction("getlength",mp3player_fnc_getlength);
	m->registerFunction("gettime",mp3player_fnc_gettime);
	m->registerFunction("isplaying",mp3player_fnc_isplaying);
	m->registerFunction("ispaused",mp3player_fnc_ispaused);
   
   // Crissi
   m->registerFunction("getmp3tag_title",mp3player_fnc_getmp3tag_title);
   m->registerFunction("getmp3tag_artist",mp3player_fnc_getmp3tag_artist);
   m->registerFunction("getmp3tag_album",mp3player_fnc_getmp3tag_album);
   m->registerFunction("getmp3tag_date",mp3player_fnc_getmp3tag_date);
   m->registerFunction("getmp3tag_comment",mp3player_fnc_getmp3tag_comment);
   m->registerFunction("getmp3tag_genre",mp3player_fnc_getmp3tag_genre);
   m->registerFunction("getmp3tag_version",mp3player_fnc_getmp3tag_version);
   m->registerFunction("getmp3tag_layer",mp3player_fnc_getmp3tag_layer);
   m->registerFunction("getmp3tag_crc",mp3player_fnc_getmp3tag_crc);
   m->registerFunction("getmp3tag_copyright",mp3player_fnc_getmp3tag_copright);
   m->registerFunction("getmp3tag_original",mp3player_fnc_getmp3tag_original);
   m->registerFunction("getmp3tag_emphasis",mp3player_fnc_getmp3tag_emphasis);
   m->registerFunction("getmp3tag_tracknumber",mp3player_fnc_getmp3tag_tracknumber);
   
	return true;
}

static bool mp3player_module_cleanup(KviModule *m)
{
#ifdef COMPILE_ON_WINDOWS

#else
	if(g_hXmmsLib)
	{
		kvi_library_close(g_hXmmsLib);
		g_hXmmsLib = 0;
	}
#endif
	return true;
}

static bool mp3player_module_can_unload(KviModule *m)
{
	return true;
}

KVIRC_MODULE(
	"mp3player",                                              // module name
	"1.0.0",                                                // module version
	"Copyright (C) 2001 Szymon Stefanek (pragma at kvirc dot net)", // author & (C)
	"Interface to mp3 players",
	mp3player_module_init,
	mp3player_module_can_unload,
	0,
	mp3player_module_cleanup
)
