//
//   File : libkvimy.cpp
//   Creation date : Mon Jul  1 02:46:49 2002 GMT by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2001 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "kvi_module.h"
#include "kvi_uparser.h"
#include "kvi_console.h"
#include "kvi_command.h"
#include "kvi_app.h"
#include "kvi_locale.h"
#include "kvi_ircconnection.h"
#include "kvi_ircconnectionuserinfo.h"
#include "kvi_ircconnectionserverinfo.h"

static KviConsole * my_module_get_console(KviCommand * c,KviParameterList * params)
{
	KviConsole *wnd = 0;

	KviStr * pCntx = params->first();

	if(pCntx)
	{
		bool bOk;
		unsigned int ircContextId = pCntx->toUInt(&bOk);
		if(bOk)wnd = g_pApp->findConsole(ircContextId);
		if(!wnd)c->warning(__tr("No such irc context (%d)"),pCntx->ptr());
	} else {
		if(c->window()->console())return c->window()->console();
		else c->warning(__tr("This window has no associated irc context"));
	}
	return wnd;
}


/*
	@doc: my.nick
	@type:
		function
	@title:
		$my.nick
	@short:
		Returns the current nickname
	@syntax:
		$my.nick
		$my.nick(<irc_context_id>)
	@description:
		Returns the nickname of the current irc context.[br]
		If the irc context is not connected then an empty string is returned.[br]
		If <irc_context_id> is specified this function returns acts as it was called
		in that irc_context.[br]
		Note that this function is different from [fnc]$me[/fnc] that will work also in a DCC CHAT.[br]
*/


static bool my_module_fnc_nick(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"$my.nick");
	KviConsole * cns = my_module_get_console(c,parms);
	if(cns)
	{
		if(cns->connection())
			buffer.append(cns->connection()->userInfo()->nickName());
	}
	return c->leaveStackFrame();
}

/*
	@doc: my.umode
	@type:
		function
	@title:
		$my.umode
	@short:
		Returns the current user mode flags
	@syntax:
		$my.umode
		$my.umode(<irc_context_id>)
	@description:
		Returns the user mode flags of the current irc context.[br]
		If the irc context is not connected then an empty string is returned.[br]
		If <irc_context_id> is specified this function returns acts as it was called
		in that irc_context.[br]
*/


static bool my_module_fnc_umode(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"$my.umode");
	KviConsole * cns = my_module_get_console(c,parms);
	if(cns)
	{
		if(cns->connection())
			buffer.append(cns->connection()->userInfo()->userMode());
	}
	return c->leaveStackFrame();
}

/*
	@doc: my.user
	@type:
		function
	@title:
		$my.user
	@short:
		Returns the current username
	@syntax:
		$my.user
		$my.user(<irc_context_id>)
	@description:
		Returns the username of the current irc context.[br]
		If the irc context is not connected then an empty string is returned.[br]
		If <irc_context_id> is specified this function returns acts as it was called
		in that irc_context.[br]
*/


static bool my_module_fnc_user(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"$my.user");
	KviConsole * cns = my_module_get_console(c,parms);
	if(cns)
	{
		if(cns->connection())
			buffer.append(cns->connection()->userInfo()->userName());
	}
	return c->leaveStackFrame();
}

/*
	@doc: my.host
	@type:
		function
	@title:
		$my.host
	@short:
		Returns the current hostname
	@syntax:
		$my.host
		$my.host(<irc_context_id>)
	@description:
		Returns the hostname of the current irc context as known by the IRC server.[br]
		If the irc context is not connected then an empty string is returned.[br]
		If <irc_context_id> is specified this function returns acts as it was called
		in that irc_context.[br]
*/


static bool my_module_fnc_host(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"$my.host");
	KviConsole * cns = my_module_get_console(c,parms);
	if(cns)
	{
		if(cns->connection())
			buffer.append(cns->connection()->userInfo()->hostName());
	}
	return c->leaveStackFrame();
}

/*
	@doc: my.ip
	@type:
		function
	@title:
		$my.ip
	@short:
		Returns the current ip address
	@syntax:
		$my.ip
		$my.ip(<irc_context_id>)
	@description:
		Returns the ip address of the current irc context as known by the IRC server.[br]
		The ip address is resolved as soon as the hostname is received from the server.
		If the hostname is masked or there is an error in the DNS lookup then
		you may get the real local host ip address as determined at connection startup.[br]
		If the irc context is not connected then an empty string is returned.[br]
		If <irc_context_id> is specified this function returns acts as it was called
		in that irc_context.[br]
*/


static bool my_module_fnc_ip(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"$my.ip");
	KviConsole * cns = my_module_get_console(c,parms);
	if(cns)
	{
		if(cns->connection())
			buffer.append(cns->connection()->userInfo()->hostIp());
	}
	return c->leaveStackFrame();
}

/*
	@doc: my.server
	@type:
		function
	@title:
		$my.server
	@short:
		Returns the current server name
	@syntax:
		$my.server
		$my.server(<irc_context_id>)
	@description:
		Returns the server name of the current irc context.[br]
		If the irc context is not connected then an empty string is returned.[br]
		If <irc_context_id> is specified this function returns acts as it was called
		in that irc_context.[br]
*/


static bool my_module_fnc_server(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"$my.server");
	KviConsole * cns = my_module_get_console(c,parms);
	if(cns)
	{
		if(cns->connection())
			buffer.append(cns->connection()->serverInfo()->name());
	}
	return c->leaveStackFrame();
}

/*
	@doc: my.network
	@type:
		function
	@title:
		$my.network
	@short:
		Returns the current network name
	@syntax:
		$my.network
		$my.network(<irc_context_id>)
	@description:
		Returns the network name of the current irc context.[br]
		If the irc context is not connected then an empty string is returned.[br]
		If <irc_context_id> is specified this function returns acts as it was called
		in that irc_context.[br]
*/


static bool my_module_fnc_network(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"$my.network");
	KviConsole * cns = my_module_get_console(c,parms);
	if(cns)buffer.append(cns->currentNetworkName());
	return c->leaveStackFrame();
}

static bool my_module_init(KviModule * m)
{
	m->registerFunction("nick",my_module_fnc_nick);
	m->registerFunction("user",my_module_fnc_user);
	m->registerFunction("host",my_module_fnc_host);
	m->registerFunction("ip",my_module_fnc_ip);
	m->registerFunction("server",my_module_fnc_server);
	m->registerFunction("network",my_module_fnc_network);
	m->registerFunction("umode",my_module_fnc_umode);
	return true;
}

static bool my_module_cleanup(KviModule *m)
{
	return true;
}

static bool my_module_can_unload(KviModule *)
{
	return true;
}

KVIRC_MODULE(
	"My",                                              // module name
	"1.0.0",                                                // module version
	"Copyright (C) 2002 Szymon Stefanek (pragma at kvirc dot net)", // author & (C)
	"Scripting irc-context related functions",
	my_module_init,
	my_module_can_unload,
	0,
	my_module_cleanup
)
