//==================================================================================
//
//   File : libkvinotifier.cpp
//   Creation date : Tue Jul 7 2004 20:21:12 CEST by Szymon Stefanek
//
//   Copyright (C) 2005 Iacopo Palazzi < iakko(at)siena(dot)linux(dot)it >
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//==================================================================================

#include "kvi_module.h"
#include "kvi_uparser.h"

#include "notifierwindow.h"
#include "notifiermessage.h"

#include "kvi_app.h"
#include "kvi_frame.h"
#include "kvi_window.h"
#include "kvi_locale.h"
#include "kvi_iconmanager.h"
#include "kvi_time.h"
#include "kvi_options.h"

#include <qsplitter.h>

KviNotifierWindow * g_pNotifierWindow = 0;
kvi_time_t g_tNotifierDisabledUntil = 0;

/*
	@doc: notifier.message
	@type:
		command
	@title:
		notifier.message
	@short:
		Adds a message to the notifier window
	@syntax:
		notifier.message [-q] [-n] [-w[=<window_id>]] [-i=<image_id>] [-t=<timeout>] <message>
	@switches:
		!sw: -q
		Does not show the notifier immediately
		!sw: -n
		Do not animate
		!sw: -w[=<window_id>]
		Attach the message to the specified window
		If the =<window_id> part is omitted then the current window is used.
		!sw: -i=<image_id>
		Display the specified image in the message
		!sw: -t=<timeout>
		Set the message lifetime to <timeout>
		If the timeout expires and is not overridden by any new message
		then the window will be automatically hidden.
		A zero timeout disables auto-hiding.
	@description:
		Adds a message to the notifier window.
		The notifier window is shown (if not already visible)
		unless the -q switch is present. If you use -q then
		you must explicitly call [cmd]notifier.show[/cmd] to
		show the notifier. If the -n switch is present then
		the show action will not be animated (the notifier
		will be shown immediately instead of fading in).
		Obviously -n has no meaning if -q is used.[br]
		If the -w=&lt;window_id&gt; switch is present then the
		message gets attacched to the specified window and
		the user is able to type commands in that window after
		showing up the notifier input. If the "=&lt;window_id&gt;" part
		is omitted then the current window is used.[br]
		If the -i=&lt;image_id&gt; switch is present then the
		message has the specified image displayed.
		See the [doc:image_id]documentation on the image identifier[/doc]
		for more informations about the image_id parameter.[br]
		The new message becomes the current message of the notifier
		unless the user is already typing in the input window
		and the typed message would be directed to a different window.
		In that case the message is appended at the end of the
		message queue and the user will be able to scroll to it
		by using the proper notifier buttons.[br]
		The &lt;message&gt; text can contain simple html tags: basically you are
		allowed to use &lt;b&gt; and &lt;i&gt;. The usage of other
		tags is possible but is discouraged since it tends to
		mess up the message display. In particular you should avoid
		any color and/or font specification since the notifier is
		skinnable and you don't know which color will result in a visible text.
		[b]Please note that the user can forcibly disable the notifier
		for a limited period of time (a sort of "don't bug me" option).[/b]
		If the -t switch is used then the notifier is automatically
		hidden after <timeout> seconds. Obviously this option has no
		meaning if the window is not going to be shown. The timeout
		may be overriddent by new messages but only in the future.
	@seealso:
		[cmd]notifier.show[/cmd] [cmd]notifier.hide[/cmd] [fnc]$notifier.isEnabled[/fnc]()
	@examples:
		[example]
			notifier.message Hello world!
			[cmd]notifier.hide[/cmd]
			notifier.message -q This is a hidden message!
			notifier.message -q -i=14 This is a second hidden message with an icon
			[cmd]notifier.show[/cmd]
			notifier.message -w This message has the current window associated
			notifier.message -w=[fnc]$window[/fnc] This is equivalent to the above
			notifier.message &lt;b&gt;Bold text&lt;/b&gt; and normal text
			[cmd]notifier.hide[/cmd]
			notifier.message -t=10 This message will be shown only for 10 seconds
		[/example]
*/

static bool notifier_module_cmd_message(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"notifier::message");

	KviStr szMsg;
	if(!g_pUserParser->parseCmdFinalPart(c,szMsg))return false;

	if(!g_pNotifierWindow)
		g_pNotifierWindow = new KviNotifierWindow();
		
	KviStr szIco;
	KviStr szWnd;
	
	KviWindow * pWnd = c->window();
	
	if(c->hasSwitch('w'))
	{
		if(c->getSwitchValue('w',szWnd))
		{
			pWnd = g_pApp->findWindow(szWnd.ptr());
			if(!pWnd)
				c->warning(__tr_ctx("The specified window does not exist","notifier"));
		}
	}
	c->getSwitchValue('i',szIco);
	
	unsigned int uTime = 0;
	if(c->hasSwitch('t'))
	{
		KviStr szTim;
		if(c->getSwitchValue('t',szTim))
		{
			bool bOk;
			uTime = szTim.toUInt(&bOk);
			if(!bOk)
			{
				uTime = 0;
				c->warning(__tr_ctx("The specified timeout is not valid, assuming 0","notifier"));
			}
		} else {
			c->warning(__tr_ctx("The -t switch expects a timeout in seconds","notifier"));
		}
	}

	g_pNotifierWindow->addMessage(pWnd,szIco.ptr(),szMsg.ptr(),uTime);

	if(!c->hasSwitch('q'))
		g_pNotifierWindow->doShow(!(c->hasSwitch('n')));

	return c->leaveStackFrame();
}


/*
	@doc: notifier.hide
	@type:
		command
	@title:
		notifier.hide
	@short:
		Hides the notifier window
	@syntax:
		notifier.hide [-n]
	@switches:
		!sw: -n
		Causes the supercazzola to be prematurated
	@description:
		Hide the notifier window
		If the -n switch is present then the hide operation is
		not animated.
		[b]Please note that the user can forcibly disable the notifier
		for a limited period of time (a sort of "don't bug me" option).[/b]
	@seealso:
		[cmd]notifier.show[/cmd] [cmd]notifier.message[/cmd] [fnc]$notifier.isEnabled[/fnc]
*/

static bool notifier_module_cmd_hide(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"notifier::hide");

	KviStr dummy;
	if(!g_pUserParser->parseCmdFinalPart(c,dummy))return false;

	if(g_pNotifierWindow)
	{
		g_pNotifierWindow->doHide(!(c->hasSwitch('n')));
	}

	return c->leaveStackFrame();
}

/*
	@doc: notifier.show
	@type:
		command
	@title:
		notifier.show
	@short:
		Shows the notifier window
	@syntax:
		notifier.show [-n]
	@switches:
		!sw: -n
		Disables the animation
	@description:
		Shows the notifier window if it is not already visible
		If the -n switch is present then the show operation is
		not animated.[br]
		The notifier is shown ONLY if it contains some messages.
		[b]Please note that the user can forcibly disable the notifier
		for a limited period of time (a sort of "don't bug me" option).[/b]
	@seealso:
		[cmd]notifier.hide[/cmd] [cmd]notifier.message[/cmd] [fnc]$notifier.isEnabled[/fnc]
*/

static bool notifier_module_cmd_show(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"notifier::show");

	KviStr dummy;
	if(!g_pUserParser->parseCmdFinalPart(c,dummy))return false;

	if(!g_pNotifierWindow)return c->leaveStackFrame();
	if(!g_pNotifierWindow->countTabs())return c->leaveStackFrame();

	g_pNotifierWindow->setDisableHideOnMainWindowGotAttention(true);
	g_pNotifierWindow->doShow(!(c->hasSwitch('n')));

	return c->leaveStackFrame();
}


/*
	@doc: notifier.isEnabled
	@type:
		function
	@title:
		$notifier.isEnabled
	@short:
		Returns 1 if the notifier window is enabled
	@syntax:
		$notifier.isEnabled
	@description:
		Returns 1 if the notifier window is enabled and 0 otherwise.
		The user can forcibly disable the notifier as a sort of "don't bug me"
		feature for a limited period of time. When the notifier
		is disabled the messages sent to it will not be shown.[br]
		The only method that you (scripter) can use to forcibly
		re-enable the notifier is to unload the module and
		reload it...but [b]DON'T do it[/b] :)[br]
		There is also a global option that allows forcibly disabling
		the notifier forever, this option could be overridden with [cmd]option[/cmd]
		instead.. but again [b]DON'T do it[/b] :)[br]
		
*/

static bool notifier_module_fnc_isEnabled(KviModule *m,KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"notifier::isEnabled");
	if(KVI_OPTION_BOOL(KviOption_boolForciblyDisableNotifier))
		buffer.append('0');
	else
		buffer.append((g_tNotifierDisabledUntil < kvi_unixTime()) ? '1' : '0');
	return c->leaveStackFrame();
}

static bool notifier_module_init(KviModule * m)
{
	m->registerCommand("show",notifier_module_cmd_show);
	m->registerCommand("hide",notifier_module_cmd_hide);
	m->registerCommand("message",notifier_module_cmd_message);
	m->registerFunction("isEnabled",notifier_module_fnc_isEnabled);
	return true;
}

static bool notifier_module_cleanup(KviModule *m)
{
	if(g_pNotifierWindow)
	{
		delete g_pNotifierWindow;
		g_pNotifierWindow = 0;
	}
	m->unregisterMetaObject("KviNotifierWindow");
	return true;
}

static bool notifier_module_can_unload(KviModule *m)
{
	return (!g_pNotifierWindow);
}

typedef struct _NotifierMessageSupaDupaParameterStruct
{
	KviWindow * pWindow;
	QString szIcon;
	QString szMessage;
	unsigned int uMessageLifetime; // 0 means no hide
} NotifierMessageSupaDupaParameterStruct;

static bool notifier_module_ctrl(KviModule *,const char *operation,void *param)
{
	if(kvi_strEqualCI("notifier::message",operation))
	{
		NotifierMessageSupaDupaParameterStruct * p = (NotifierMessageSupaDupaParameterStruct *)param;
		if(!p)return false;

		if(!g_pNotifierWindow)
			g_pNotifierWindow = new KviNotifierWindow();

		g_pNotifierWindow->addMessage(p->pWindow,p->szIcon,p->szMessage,p->uMessageLifetime);
		g_pNotifierWindow->doShow(true);

		return true;
	}
	return false;
}

KVIRC_MODULE(
	"Notifier",
	"2.0.0",
	"Copyright (C) 2005:\n" \
	"  Iacopo Palazzi (iakko at siena dot linux dot it)",
	"KVIrc Client - Taskbar Notifier",
	notifier_module_init,
	notifier_module_can_unload,
	notifier_module_ctrl,
	notifier_module_cleanup
)
