//
//   File : class_list.cpp
//   Creation date : Wed Sep 09 2000 21:07:55 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
#define _KVI_DEBUG_CHECK_RANGE_
#include "kvi_debug.h"
#include "kvi_command.h"
#include "kvi_malloc.h"
#include "kvi_locale.h"

#include "class_list.h"

#include <stdlib.h>

static KviScriptObjectClass * g_pListClass = 0;

static KviScriptObject * listClassCreateInstance(KviScriptObjectClass * cls,KviScriptObject *par,const char * nam)
{
	return new KviScriptListObject(cls,par,nam);
}

KviScriptListObject::KviScriptListObject(KviScriptObjectClass * cla,KviScriptObject * par,const char * nam)
: KviScriptObject(cla,par,nam)
{
	m_pDataList = new KviPtrList<KviStr>;
	m_pDataList->setAutoDelete(true);
}

KviScriptListObject::~KviScriptListObject()
{
	delete m_pDataList;
}

/*
	@doc: list
	@keyterms:
		list object class, sorted list
	@title:
		list class
	@type:
		class
	@short:
		Abstracts a double linked list of strings
	@inherits:
		[class]object[/class]
	@description:
		This object class abstracts a double linked list of strings.
		You can insert/remove items by zero based position, at the end (tail) or at the 
		beginning (head). The items can be sorted by using the [classfnc:list]$sort[/classfnc]()
		function (quicksort algorithm). The list incorporates an efficient iteration
		method by using the [classfnc:list]$first[/classfnc](),[classfnc:list]$last[/classfnc](),
		[classfnc:list]$next[/classfnc](),[classfnc:list]$prev[/classfnc](),
		[classfnc:list]$current[/classfnc]() and [classfnc:list]$canIterate[/classfnc]() functions.
		[example]
			%Nicknames = [fnc]$new[/fnc](list,0,mylist)
			%Nicknames->[classfnc:list]$append[/classfnc](Crocodile,Pragma,Number1,Kristoff,alch3m1st)
			%nick = %Nicknames->[classfnc:list]$first[/classfnc]();
			[cmd]while[/cmd](%Nicknames->[classfnc:list]$canIterate[/classfnc]())
			{
				[cmd]msg[/cmd] %nick Hello!
				%nick = %Nicknames->[classfnc:list]$next[/classfnc]()
			}
			[cmd]delete[/cmd] %Nicknames;
		[/example]
	@functions:
		!fn: $count()
		Returns the number of strings stored in the list
		!fn: $isEmpty()
		Returns '1' if there are no items stored in the list , '0' otherwise.
		!fn: $insert(<index>,<string>)
		Inserts the <string> at (zero based) position <index> in the list.
		!fn: $at(<index>)
		Returns the string at (zero based) position <index>, or an empty
		string if there is no item at position <index>.[br]
		Note: since empty strings are valid inside the list, there is
		no implicit method to know if the returned empty string is a
		valid one or it has been returned because <index> was out of range.
		You have to explicitly check <index> before.
		!fn: $remove(<index>)
		Removes the string at (zero based) position <index>. Does nothing if <index> is
		out of range.
		!fn: $append(<string_list>)
		Appends the items in the <string_list> to this list.
		<string_list> is expected to be a comma separated list of strings:[br]
		[example]
			%MyList->$append(Item1,Item2,Item3,%ListOfItems[])
		[/example]
		!fn: $prepend(<string_list>)
		Prepends the items in the <string_list> to this list.
		<string_list> is expected to be a comma separated list of strings:[br]
		[example]
			%MyList->$prepend(%LastNick,Pragma,%Nicknames[],[Pragma])
		[/example]
		!fn: $removeFirst()
		Removes the first item in the list. Does nothing if the list is empty.
		!fn: $removeLast()
		Removes the last item in the list. Does nothing if the list is empty.
		!fn: $first()
		Returns the first item in the list or an empty string if the list is empty.
		Moves the "current item" to the first item in the list.
		!fn: $last()
		Returns the last item in the list or an empty string if the list is empty.
		Moves the "current item" to the last item in the list.
		!fn: $prev()
		Moves the "current item" to the previous item and returns it.
		If the "current item" is no longer valid (if you've ran past the head of the list)
		the returned string is empty.
		!fn: $next()
		Moves the "current item" to the next item and returns it.
		If the "current item" is no longer valid (if you've ran past the tail of the list)
		the returned string is empty.
		!fn: $current()
		Returns the "current item". The returned string is empty if the current item is
		not valid (eg. poiting past the tail or head of the list).
		!fn: $canIterate()
		Returns '1' if the "current item" is valid (eg. points to an item in the list).
		!fn: $clear()
		Removes all the strings from the list.
		!fn: $sort()
		Sorts the list in case insensitive alphabetic order.
		!fn: $find(<string>)
		Returns the zero-based index of the first occurente of <string> in the list
		or -1 if the <string> is not in the list at all.
*/

void KviScriptListObject::registerSelf()
{
	KviScriptObjectClass * base = g_pScriptObjectController->lookupClass("object");
	__range_valid(base);
	g_pListClass = new KviScriptObjectClass(base,"list",listClassCreateInstance,true);
	g_pListClass->registerFunctionHandler("count",
		(KviScriptObjectFunctionHandlerProc)(KVI_PTR2MEMBER(KviScriptListObject::functionCount)),0,true);
	g_pListClass->registerFunctionHandler("isEmpty",
		(KviScriptObjectFunctionHandlerProc)(KVI_PTR2MEMBER(KviScriptListObject::functionIsEmpty)),0,true);
	g_pListClass->registerFunctionHandler("insert",
		(KviScriptObjectFunctionHandlerProc)(KVI_PTR2MEMBER(KviScriptListObject::functionInsert)),0,true);
	g_pListClass->registerFunctionHandler("at",
		(KviScriptObjectFunctionHandlerProc)(KVI_PTR2MEMBER(KviScriptListObject::functionAt)),0,true);
	g_pListClass->registerFunctionHandler("remove",
		(KviScriptObjectFunctionHandlerProc)(KVI_PTR2MEMBER(KviScriptListObject::functionRemove)),0,true);
	g_pListClass->registerFunctionHandler("append",
		(KviScriptObjectFunctionHandlerProc)(KVI_PTR2MEMBER(KviScriptListObject::functionAppend)),0,true);
	g_pListClass->registerFunctionHandler("prepend",
		(KviScriptObjectFunctionHandlerProc)(KVI_PTR2MEMBER(KviScriptListObject::functionPrepend)),0,true);
	g_pListClass->registerFunctionHandler("removeFirst",
		(KviScriptObjectFunctionHandlerProc)(KVI_PTR2MEMBER(KviScriptListObject::functionRemoveFirst)),0,true);
	g_pListClass->registerFunctionHandler("removeLast",
		(KviScriptObjectFunctionHandlerProc)(KVI_PTR2MEMBER(KviScriptListObject::functionRemoveLast)),0,true);
	g_pListClass->registerFunctionHandler("first",
		(KviScriptObjectFunctionHandlerProc)(KVI_PTR2MEMBER(KviScriptListObject::functionFirst)),0,true);
	g_pListClass->registerFunctionHandler("last",
		(KviScriptObjectFunctionHandlerProc)(KVI_PTR2MEMBER(KviScriptListObject::functionLast)),0,true);
	g_pListClass->registerFunctionHandler("next",
		(KviScriptObjectFunctionHandlerProc)(KVI_PTR2MEMBER(KviScriptListObject::functionNext)),0,true);
	g_pListClass->registerFunctionHandler("prev",
		(KviScriptObjectFunctionHandlerProc)(KVI_PTR2MEMBER(KviScriptListObject::functionPrev)),0,true);
	g_pListClass->registerFunctionHandler("current",
		(KviScriptObjectFunctionHandlerProc)(KVI_PTR2MEMBER(KviScriptListObject::functionCurrent)),0,true);
	g_pListClass->registerFunctionHandler("canIterate",
		(KviScriptObjectFunctionHandlerProc)(KVI_PTR2MEMBER(KviScriptListObject::functionCanIterate)),0,true);
	g_pListClass->registerFunctionHandler("clear",
		(KviScriptObjectFunctionHandlerProc)(KVI_PTR2MEMBER(KviScriptListObject::functionClear)),0,true);
	g_pListClass->registerFunctionHandler("sort",
		(KviScriptObjectFunctionHandlerProc)(KVI_PTR2MEMBER(KviScriptListObject::functionSort)),0,true);
	g_pListClass->registerFunctionHandler("find",
		(KviScriptObjectFunctionHandlerProc)(KVI_PTR2MEMBER(KviScriptListObject::functionFind)),0,true);
}

void KviScriptListObject::unregisterSelf()
{
	delete g_pListClass; // this will delete all the objects of this class
    g_pListClass = 0;
}

bool KviScriptListObject::functionCount(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	buffer.append(KviStr::Format,"%u",m_pDataList->count());
	return true;
}

bool KviScriptListObject::functionIsEmpty(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	buffer.append(m_pDataList->isEmpty() ? '1' : '0');
	return true;
}

bool KviScriptListObject::functionInsert(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"list::insert");
	KviStr * pIndex = params->first();
	KviStr * pValue = params->next();

	if(pIndex)
	{
		bool bOk;
		unsigned int idx = pIndex->toUInt(&bOk);
		if(bOk)
		{
			if(pValue)
			{
				m_pDataList->insert(idx,new KviStr(*pValue));
			} else c->warning(__tr("No value to insert"));
		} else c->warning(__tr("Invalid index(%s)"),pIndex->ptr());
	} else c->warning(__tr("No index specified"));

	return c->leaveStackFrame();
}

bool KviScriptListObject::functionAt(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"list::at");
	bool bOk;
	unsigned int idx = params->getUInt(&bOk);

	if(bOk)
	{
		KviStr * pData = m_pDataList->at(idx);
		if(pData)buffer.append(pData->ptr());
		else c->warning(__tr("index %u out of range"),idx);
	} else {
		c->warning(__tr("Invalid index (%s)"),params->safeFirst()->ptr());
	}

	return c->leaveStackFrame();
}

bool KviScriptListObject::functionRemove(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"list::remove");
	bool bOk;
	unsigned int idx = params->getUInt(&bOk);

	if(bOk)
	{
		if(!m_pDataList->remove(idx))c->warning(__tr("Index %u out of range"),idx);
	} else {
		c->warning(__tr("Invalid index (%s)"),params->safeFirst()->ptr());
	}

	return c->leaveStackFrame();
}

bool KviScriptListObject::functionAppend(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	KviStr * pValue = params->first();
	while(pValue)
	{
		m_pDataList->append(new KviStr(*pValue));
		pValue = params->next();
	}
	return true;
}

bool KviScriptListObject::functionPrepend(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	KviStr * pValue = params->last();
	while(pValue)
	{
		m_pDataList->insert(0,new KviStr(*pValue));
		pValue = params->prev();
	}
	return true;
}

bool KviScriptListObject::functionRemoveFirst(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	m_pDataList->removeFirst();
	return true;
}

bool KviScriptListObject::functionRemoveLast(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	m_pDataList->removeLast();
	return true;
}

bool KviScriptListObject::functionFirst(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	KviStr * pValue = m_pDataList->first();
	if(pValue)buffer.append(*pValue);
	return true;
}

bool KviScriptListObject::functionLast(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	KviStr * pValue = m_pDataList->last();
	if(pValue)buffer.append(*pValue);
	return true;
}

bool KviScriptListObject::functionNext(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	KviStr * pValue = m_pDataList->next();
	if(pValue)buffer.append(*pValue);
	return true;
}

bool KviScriptListObject::functionPrev(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	KviStr * pValue = m_pDataList->prev();
	if(pValue)buffer.append(*pValue);
	return true;
}

bool KviScriptListObject::functionCurrent(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	KviStr * pValue = m_pDataList->current();
	if(pValue)buffer.append(*pValue);
	return true;
}

bool KviScriptListObject::functionCanIterate(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	buffer.append(m_pDataList->current() ? '1' : '0');
	return true;
}

bool KviScriptListObject::functionClear(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	m_pDataList->clear();
	return true;
}

int default_cmp(const void *s1,const void *s2)
{
	return kvi_strcmpCI((*((KviStr**)s1))->ptr(),(*((KviStr**)s2))->ptr());
}

bool KviScriptListObject::functionSort(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
//#warning "SORT WITH $compareItems()"
	if(m_pDataList->count() > 1)
	{
		unsigned int cnt = m_pDataList->count();
		KviStr ** vector = (KviStr **)kvi_malloc(sizeof(KviStr *) * cnt);
		int i = 0;
		for(KviStr * s= m_pDataList->first();s;s= m_pDataList->next())
		{
			vector[i] = s;
			i++;
		}
		m_pDataList->setAutoDelete(false);
		delete m_pDataList;

		qsort((void *)vector,cnt,sizeof(KviStr *),default_cmp);

		m_pDataList = new KviPtrList<KviStr>;
		m_pDataList->setAutoDelete(true);
		for(unsigned int idx = 0;idx < cnt;idx++)
		{
			m_pDataList->append(vector[idx]);
		}
		kvi_free(vector);
	}
	return true;
}

bool KviScriptListObject::functionFind(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	KviStr * pStr = params->first();
	KviStr empty;
	if(!pStr)pStr = &empty;
	int idx = 0;
	for(KviStr * s = m_pDataList->first();s;s = m_pDataList->next())
	{
		if(kvi_strEqualCI(s->ptr(),pStr->ptr()))
		{
			buffer.append(KviStr::Format,"%d",idx);
			return true;
		}
		idx++;
	}
	buffer.append("-1");
	return true;
}


#include "m_class_list.moc"
