//vim: ts=8
//   File : class_multilineedit.cpp
//   Creation date : Mon Feb 19 00:45:34 CET by Krzysztof Godlewski
//
//	 Modification date: Thu 15 Feb 2005 22:00 :00
//	 by Tonino Imbesi(Grifisx) and Alessandro Carbone(Noldor)
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "kvi_command.h"
#include "kvi_error.h"
#include "kvi_debug.h"
#include "kvi_scriptobject.h"
#include "kvi_locale.h"
#include "kvi_malloc.h"
#include <qfile.h>


#include "class_multilineedit.h"

//->Tables for Text Format
static char * const mod_tbl[] =	{
					"PlainText",
					"RichText",
					"AutoText",
					"LogText"
				  };
			
static int mod_cod[] =	{
				Qt::PlainText,
				Qt::RichText,
				Qt::AutoText,
				Qt::LogText
			};

#define mod_num			(sizeof(mod_tbl) / sizeof(mod_tbl[0]))

static KviScriptObjectClass * g_pMultiLineEditClass = 0;

static KviScriptObject * multiLineEditClassCreateInstance( \
	KviScriptObjectClass * c, KviScriptObject * p, const char * n)
{
	return new KviScriptMLEditObject(c, p, n);
}

KviScriptMLEditObject::KviScriptMLEditObject(KviScriptObjectClass * c, KviScriptObject * p,\
	const char * n) : KviScriptWidgetObject(c, p, n)
{
}

KviScriptMLEditObject::~KviScriptMLEditObject()
{
}


/*
	@doc:	multilineedit
	@keyterms:
		multilineedit object class, line editor, input
	@title:
		multilineedit class
	@type:
		class
	@short:
		An editor for inputting larger portions of text
	@inherits:
		[class]object[/class]
		[class]widget[/class]
	@description:
		This is class an editor for inputting larger portions of text.
		Besides that, it has unfinished docs.
	@functions:
		!fn: $textLine(<line>)
		Returns the string at line <line>. Note that it can be an empty
		string.
		!fn: $insertLine(<text>, <line>)
		Inserts <text> at line number <line>. If line is less than
		zero, or larger than the number of rows, the new text is
		put at the end. If txt contains newline characters, several
		lines are inserted. The <line> parameter is optional. If it is
		ommited, the line is put at end.
		!fn: $removeLine(<line>)
		Removes line number <line>.
		!fn: $insertAt(<txt>, <line>, <col>)
		Inserts txt at line number line, after character number col in
		the line. If txt contains newline characters, new lines are
		inserted.
		!fn: $numLines()
		Returns number of lines in the widget.
		!fn: $setCursorPosition(<line>, <col>, <mark>)
		Sets the cursor position to character number col in line
		number line. The parameters are adjusted to lie within the
		legal range.[br]
		If mark is FALSE, the selection is cleared. otherwise it
		is extended
		!fn: $cursorPosition()
		Returns current cursor position in "<line>, <col>" format.
		!fn: $setReadOnly(<bReadOnly>)
		Sets the editor to be read-only if bReadOnly is 1 or
		removes the read-only status is ReadOnly is 0
		!fn: $atBeginning()
		Returns 1(TRUE) if the cursor is placed at the beginning of the text; otherwise returns 0(FALSE). 
		!fn: $atEnd()
		Returns 1(TRUE) if the cursor is placed at the end of the text; otherwise returns 0(FALSE). 
		!fn: $setWordWrap(<mode>)
		Sets the word wrap mode to mode. Valid Values are:[br]
		- NoWrap - Do not wrap the text.[br]
		- WidgetWidth - Wrap the text at the current width of the widget.[br]
		- FixedPixelWidth - Wrap the text at a fixed number of pixels from the widget's left side.[br]
		- FixedColumnWidth - Wrap the text at a fixed number of character columns from the widget's left side.[br]
		!fn: $wordWrap()
		Returns the word wrap mode. NoWrap, WidgetWidth, FixedPixelWidth, FixedColumnWidth.
		!fn: $text()
		Returns the multiline edit's text. 
		!fn: $length()
		Returns the number of characters in the text This function ignores newlines.
		!fn: $setMaxLines(<int a>)
		Sets the max number of the lines to <a>
		!fn: $maxLines()
		Returns the max number of the lines in the multiline edit.
		!fn: $insert(<text>)
		Inserts text at the current cursor position.
		!fn: $append()
		Appends a new paragraph with text to the end of the multiline edit.
		!fn: $copy()
		Copies any selected text from selection  to the clipboard.
		!fn: $cut()
		Copies the selected text from selection  to the clipboard and deletes it from the multiline edit.
		!fn: $paste()
		Pastes the text from the clipboard into the multiline edit at the current cursor position. 
		!fn: $setReadOnly()
		Sets the multilineedit to read only.
		!fn: $setFamily(<fontfamily>)
		Sets the font family of the current format to fontFamily.
		!fn: $setItalic(<bool 1 or 0>)
		If the bool value is 1 sets the current format to italic; otherwise, if it's 0 sets the current format to no-italic.
		!fn: $setBold(<bool 1 or 0>)
		If the bool value is 1 sets the current format to bold; otherwise, if it's 0 sets the current format to no-bold.
		!fn: $setUnderline(<bool 1 or 0>)
		If the bool value is 1 sets the current format to underline; otherwise, if it's 0 sets the current format to no-underline.
		!fn: $italic()
		Returns 1(TRUE) if the current format is italic; otherwise returns 0(FALSE).
		!fn: $bold()
		Returns 1(TRUE) if the current format is bold; otherwise returns 0(FALSE).
		!fn: $underline()
		Returns 1(TRUE) if the current format is underline; otherwise returns 0(FALSE).
		!fn: $zoomIn(<int range>)
		Zooms in on the text by making the base font size range points larger.
		!fn: $zoomOut(<int range>)
		Zooms out on the text by making the base font size range points smaller.
		!fn: $zoomTo(<int size>)
		Zooms the text by making the base font size <size> points.
		!fn: $undo()
		Undoes the last operation.
		!fn: $redo()
		Redoes the last operation.
		!fn: $clear()
		Deletes all the text in the multiline edit.
		!fn: $setUndoRedoEnabled(<bool 1 or 0>)
		Sets whether undo/redo is enabled to the bool value.
		!fn: $setUndoDepth(<int x>)
		Sets the depth of the undo history to x. 
		!fn: $isUndoRedoEnabled()
		Returns TRUE if undo/redo is enabled; otherwise returns FALSE. 
		!fn: $undoDepth()
		Returns the depth of the undo history.
		!fn: $indent()
		Re-indents the current paragraph.
		!fn: $setText(<txt>)
		Sets the text edit's text to txt. 
		!fn: $setPointSize(<int size))
		Sets the point size of the font.
		!fn: $setColor(<exadecimal value>)
		Sets the color of the text, to exadecimal colour values.[br]
		Es: %a->$setColor(<6C6A60>)
		!fn: $setModified(<bool 1 or 0>)
		Sets whether the document has been modified by the user. Valid Values are 1(TRUE) or 0(FALSE)
		!fn: $insertParagraph(<text>,<paragraph>)
		Inserts text as a new paragraph at position <paragraph>.
		!fn: $removeParagraph(<int par>)
		Removes the paragraph <par>.
		!fn: $setLinkUnderline(<bool 1 or 0 >)
		Sets to 1 if you want that hypertext links will be underlined; otherwise sets to 0.
		!fn: $setOverwriteMode(<bool 1 or 0>)
		Sets the Multilineedit edit's overwrite mode to b (1=Enabled or 0=Disabled) .
		!fn: $setTextFormat(<textformat>)
		Sets the text format. Correct values are RichText, PlainText, LogText or AutoText. 
		!fn: $textFormat()
		Returns the text format: rich text, plain text, log text or auto text.
		!fn: $setParagraphBackgroundColor(<int par>,<exadecimal color value>)
		Sets the background color of the paragraph <par> to color value specified.[br]
		Es:%mymultiline->$setParagraphBackgroundColor(2,0C686F)[br]
		!fn: $clearParagraphBackground(<int par>)
		Clear the background color of the paragraph <par>.
		!fn: $loadFile(<path>)
		Load the file specified in the <path> , also html files.
		!fn: $isUndoAvailable ()
		Returns 1(TRUE) if undo is available; otherwise returns 0(FALSE).
		!fn: $isRedoAvailable ()
		Returns 1(TRUE) if redo is available; otherwise returns 0(FALSE).
		!fn: $setAlignment(<alignment>)
		Sets the alignment of the current paragraph to <alignment>. Valid values are:[br]
		- AlignAuto - Aligns according to the language.[br]
		- AlignLeft - Aligns with the left edge.[br]
		- AlignRight - Aligns with the right edge.[br]
		- AlignCenter - Centers in both dimensions.[br]
		!fn: $setVerticalAlignment(<Valignment>)
		Sets the vertical alignment of the current format to <Valignemnt>. Valid Values are:[br]
		- AlignNormal - Normal alignment.[br]
		- AlignSuperScript - Superscript.[br]
		- AlignSubScript - Subscript.[br]
		!fn: $setAutoFormatting(<afvalue>)
		Sets the enabled set of auto formatting features to afvalue. Valid values are: [br]
		- AutoNone - Do not perform any automatic formatting[br]
		- AutoBulletList - Only automatically format bulletted lists[br]
		- AutoAll - Apply all available autoformatting[br]
		!fn: $setWrapPolicy(<wrpl>)
		Sets the word wrap policy, at <wrpl>. Valid values are:[br]
		- AtWhiteSpace - Don't use this deprecated value (it is a synonym for AtWordBoundary which you should use instead).[br]
		- Anywhere - Break anywhere, including within words.[br]
		- AtWordBoundary - Break lines at word boundaries, e.g. spaces or newlines[br]
		- AtWordOrDocumentBoundary - Break lines at whitespace, e.g. spaces or newlines if possible. Break it anywhere otherwise.[br]
		!fn: $paragraphs()
		Returns the number of paragraphs in the text.
		!fn: $lines()
		Returns the number of lines in the multilineedit.
		!fn: $linesOfParagraph(<int par>)
		Returns the number of lines in paragraph <par>, or -1 if there is no paragraph with index <par>.
		!fn: $lineOfChar(<int par>,<int index>)
		Returns the line number of the line in paragraph par in which the character at position index appears.[br]
		If there is no such paragraph or no such character at the index position  -1 is returned.
		!fn: $paragraphLenght(<int par>)
		Returns the length of the paragraph par, or -1 if there is no paragraph with index par.

*/

bool KviScriptMLEditObject::init(KviCommand *)
{
	setObject(new QMultiLineEdit(parentScriptWidget(), name()), true);
	return true;
}

#define mlEditFuncReg(__nam, __fnc) \
	g_pMultiLineEditClass->registerFunctionHandler(__nam, \
        	(KviScriptObjectFunctionHandlerProc)( \
		KVI_PTR2MEMBER(KviScriptMLEditObject::__fnc)), 0, true);

void KviScriptMLEditObject::registerSelf()
{
	KviScriptObjectClass * base = g_pScriptObjectController-> \
		lookupClass("widget");
	__range_valid(base);

	g_pMultiLineEditClass = new KviScriptObjectClass(base, "multilineedit",
		multiLineEditClassCreateInstance, true);

	mlEditFuncReg("textLine", functionTextLine);
	mlEditFuncReg("insertLine", functionInsertLine);
	mlEditFuncReg("removeLine", functionRemoveLine);
	mlEditFuncReg("insertAt", functionInsertAt);
	mlEditFuncReg("numLines", functionNumLines);
	mlEditFuncReg("setCursorPosition", functionSetCursorPosition);
	mlEditFuncReg("cursorPosition", functionCursorPosition);
	mlEditFuncReg("atBeginning", functionAtBeginning);
	mlEditFuncReg("atEnd", functionAtEnd);
	mlEditFuncReg("setWordWrap", functionSetWordWrap);
	mlEditFuncReg("wordWrap", functionWordWrap);
	mlEditFuncReg("text", functionText);
	mlEditFuncReg("length", functionLength);
	mlEditFuncReg("setMaxLines", functionSetMaxLines);
	mlEditFuncReg("maxLines", functionMaxLines);
	mlEditFuncReg("insert", functionInsert);
	mlEditFuncReg("append", functionAppend);
	mlEditFuncReg("copy", functionCopy);
	mlEditFuncReg("cut", functionCut);
	mlEditFuncReg("paste", functionPaste);

//->Set Style 
	mlEditFuncReg("setFamily" , functionsetFamily);
	mlEditFuncReg("setItalic", functionsetItalic);
	mlEditFuncReg("setBold", functionsetBold);
	mlEditFuncReg("setUnderline", functionsetUnderline);
	mlEditFuncReg("italic", functionitalic);
	mlEditFuncReg("bold", functionbold);
	mlEditFuncReg("underline", functionunderline);
//->Zoom In, Out, To
	mlEditFuncReg("zoomIn", functionzoomIn);
    mlEditFuncReg("zoomOut", functionzoomOut);
    mlEditFuncReg("zoomTo", functionzoomTo);
//->Undo & Redo
	mlEditFuncReg("undo", functionundo);
    mlEditFuncReg("redo", functionredo);
    mlEditFuncReg("del", functiondel);
    mlEditFuncReg("clear", functionclear);
    mlEditFuncReg("setUndoRedoEnabled", functionsetUndoRedoEnabled);
    mlEditFuncReg("setUndoDepth", functionsetUndoDepth);                
    mlEditFuncReg("isUndoRedoEnabled", functionsetUndoRedoEnabled);
    mlEditFuncReg("undoDepth", functionundoDepth);
	mlEditFuncReg("indent", functionundoDepth);
	mlEditFuncReg("isUndoAvailable", functionisUndoAvailable);
	mlEditFuncReg("isUndoRedoAvailable", functionisRedoAvailable);
//->Text color & others
	mlEditFuncReg("setText", functionsetText);
    mlEditFuncReg("setPointSize", functionsetPointSize);
    mlEditFuncReg("setColor", functionsetColor);
	mlEditFuncReg("setModified" , functionsetModified);
	mlEditFuncReg("setLinkUnderline" , functionsetLinkUnderline);
	mlEditFuncReg("setOverwriteMode" , functionsetOverwriteMode);
	mlEditFuncReg("setReadOnly",functionSetReadOnly);
	mlEditFuncReg("setTextFormat" , functionsetTextFormat);
	mlEditFuncReg("textFormat" , functiontextFormat);
	mlEditFuncReg("setParagraphBackgroundColor", functionsetParagraphBackgroundColor);
	mlEditFuncReg("clearParagraphBackgroundColor", functionsetParagraphBackgroundColor);
	mlEditFuncReg("loadFile" , functionloadFile); // let's fantasy
	mlEditFuncReg("setAlignment" , functionsetAlignment);
	mlEditFuncReg("setAutoFormatting" , functionsetAutoFormatting);
	mlEditFuncReg("setVerticalAlignment" , functionsetVerticalAlignment);
	mlEditFuncReg("setWrapPolicy" , functionsetWrapPolicy);
//->Paragraph & lines
	mlEditFuncReg("paragraphs" , functionparagraphs);
	mlEditFuncReg("paragraphLength" , functionparagraphLength); 
	mlEditFuncReg("insertParagraph" , functioninsertParagraph);
	mlEditFuncReg("removeParagraph" , functionremoveParagraph);
	mlEditFuncReg("lines" , functionlines);
	mlEditFuncReg("lineOfChar" , functionlineOfChar); 
	mlEditFuncReg("linesOfParagraph" , functionlinesOfParagraph);
	mlEditFuncReg("selectAll" , functionselectAll) 
	 
}

void KviScriptMLEditObject::unregisterSelf()
{
	delete g_pMultiLineEditClass;
    g_pMultiLineEditClass = 0;
}

bool KviScriptMLEditObject::functionSetReadOnly(KviCommand * c, KviParameterList * p, \
	KviStr & b)
{
	ENTER_STACK_FRAME(c, "multilineedit::setReadOnly");

	if(widget())
	{
		bool bVal = p->getBool();
		((QMultiLineEdit *)widget())->setReadOnly(bVal);
	}
	
	return c->leaveStackFrame();
}


bool KviScriptMLEditObject::functionTextLine(KviCommand * c, KviParameterList * p, \
	KviStr & b)
{
	ENTER_STACK_FRAME(c, "multilineedit::textLine");

	if(widget())
	{
		bool bOk;
		int lineNum = p->getInt(&bOk);
		if(!bOk)
			return c->error(KviError_integerParameterExpected);
			
		if(lineNum > ((QMultiLineEdit *)widget())->numLines() || \
			lineNum < 0)
			c->warning("No such line number");
			
		KviStr line = ((QMultiLineEdit *)widget())->textLine(lineNum);
		b.append(line);
	}
	
	return c->leaveStackFrame();
}

bool KviScriptMLEditObject::functionInsertLine(KviCommand * c, KviParameterList * p, \
	KviStr & b)
{
	ENTER_STACK_FRAME(c, "multilineedit::insertLine");

	if(widget())
	{
		bool bOk;
		KviStr * txt = p->safeFirst();
		if(txt->isEmpty())
			return c->error(KviError_notEnoughParameters);

		(void)(p->next());

		int line = p->getInt(&bOk);
		if(!bOk)
			return c->error(KviError_integerParameterExpected);

		((QMultiLineEdit *)widget())->insertLine(txt->ptr(), line);
	}
	
	return c->leaveStackFrame();
}

bool KviScriptMLEditObject::functionRemoveLine(KviCommand * c, KviParameterList * p,\
	KviStr & b)
{
	ENTER_STACK_FRAME(c, "multilineedit::removeLine");
	
	if(widget())
	{
		bool bOk;
		int line = p->getInt(&bOk);
		if(!bOk)
			return c->error(KviError_integerParameterExpected);

		if(line > ((QMultiLineEdit *)widget())->numLines() || line < 0)
			c->warning("No such line number");

		((QMultiLineEdit *)widget())->removeLine(line);
	}
	
	return c->leaveStackFrame();
}

bool KviScriptMLEditObject::functionInsertAt(KviCommand * c, KviParameterList * p, \
	KviStr & b)
{
	ENTER_STACK_FRAME(c, "multilineedit::insertAt");

	if(widget())
	{
		if(p->count() < 3)
			return c->error(KviError_notEnoughParameters);
			
		KviStr * txt = p->safeFirst();

		(void)(p->next());

		bool bOk1, bOk2;
		int line = p->getInt(&bOk1);
		int col = p->getInt(&bOk2);

		if(line < 0 || col < 0 || \
		  line > ((QMultiLineEdit *)widget())->numLines())
			c->warning("No such line number");	

		((QMultiLineEdit *)widget())->insertAt(txt->ptr(), line, col);
	}
	
	return c->leaveStackFrame();
}

bool KviScriptMLEditObject::functionNumLines(KviCommand *, KviParameterList *, \
	KviStr & b)
{
	if(widget())
		b.append(KviStr::Format, "%d", \
			((QMultiLineEdit *)widget())->numLines());

	return true;
}

bool KviScriptMLEditObject::functionSetCursorPosition(KviCommand * c, \
	KviParameterList * p, KviStr &)
{
	ENTER_STACK_FRAME(c, "multilineedit::setCursorPosition");

	if(widget())
	{
		if(p->count() < 3)
			return c->error(KviError_notEnoughParameters);

		bool bOk[2];
		int line = p->getInt(&bOk[1]);
		int col = p->getInt(&bOk[2]);
		bool m = p->getBool();

		if(!bOk[1] || !bOk[2])
			return c->error(KviError_integerParameterExpected);

		((QMultiLineEdit *)widget())->setCursorPosition(line, col, m);
	}
	
	return c->leaveStackFrame();
}

bool KviScriptMLEditObject::functionCursorPosition(KviCommand *, KviParameterList *, \
	KviStr & b)
{
	if(widget())
	{
		int line, col;
#if QT_VERSION >= 300
		((QMultiLineEdit *)widget())->getCursorPosition(&line, &col);
#else
		((QMultiLineEdit *)widget())->cursorPosition(&line, &col);
#endif
		b.append(KviStr::Format, "%d, %d", line, col);
	}

	return true;
}

bool KviScriptMLEditObject::functionAtBeginning(KviCommand *, KviParameterList *, \
	KviStr & b)
{
	if(widget())
		b.append((((QMultiLineEdit *)widget())->atBeginning())?'1':'0');

	return true;
}

bool KviScriptMLEditObject::functionAtEnd(KviCommand *, KviParameterList *, \
	KviStr & b)
{
	if(widget())
		b.append((((QMultiLineEdit *)widget())->atEnd()) ? '1' : '0');

	return true;
}

bool KviScriptMLEditObject::functionSetWordWrap(KviCommand *, KviParameterList * p, \
	KviStr &)
{
	KviStr * pModos = p->safeFirst();
	if(widget())
	{
		if(kvi_strEqualCI(pModos->ptr(),"NoWrap)"))
			((QMultiLineEdit *)widget())->setWordWrap(QTextEdit::NoWrap);
		if(kvi_strEqualCI(pModos->ptr(),"WidgetWidth"))
			((QMultiLineEdit *)widget())->setWordWrap(QTextEdit::WidgetWidth);
		if(kvi_strEqualCI(pModos->ptr(),"FixedPixelWidth"))
			((QMultiLineEdit *)widget())->setWordWrap(QTextEdit::FixedPixelWidth);
		if(kvi_strEqualCI(pModos->ptr(),"FixedColumnWidth"))
			((QMultiLineEdit *)widget())->setWordWrap(QTextEdit::FixedColumnWidth);
	}
	return true;
}

bool KviScriptMLEditObject::functionsetWrapPolicy(KviCommand *, KviParameterList * p, \
	KviStr &)
{
	KviStr * pModos = p->safeFirst();
	if(widget())
	{
		if(kvi_strEqualCI(pModos->ptr(),"AtWhiteSpace)"))
			((QMultiLineEdit *)widget())->setWrapPolicy(QTextEdit::AtWhiteSpace);
		if(kvi_strEqualCI(pModos->ptr(),"Anywhere"))
			((QMultiLineEdit *)widget())->setWrapPolicy(QTextEdit::Anywhere);
		if(kvi_strEqualCI(pModos->ptr(),"AtWordBoundary"))
			((QMultiLineEdit *)widget())->setWrapPolicy(QTextEdit::AtWordBoundary);
		if(kvi_strEqualCI(pModos->ptr(),"AtWordOrDocumentBoundary"))
			((QMultiLineEdit *)widget())->setWrapPolicy(QTextEdit::AtWordOrDocumentBoundary);
	}
	return true;
}
bool KviScriptMLEditObject::functionWordWrap(KviCommand *, KviParameterList *, \
	KviStr & b)
{
	if(widget())
	{
		bool on = ((QMultiLineEdit *)widget())->wordWrap() & \
			QMultiLineEdit::WidgetWidth;
		b.append(on ? '1' : '0');
	}

	return true;
}

bool KviScriptMLEditObject::functionText(KviCommand *, KviParameterList *, KviStr & b)
{
	if(widget())
		b.append(((QMultiLineEdit *)widget())->text());

	return true;
}

bool KviScriptMLEditObject::functionLength(KviCommand *, KviParameterList *, \
	KviStr & b)
{
	if(widget())
		b.append(KviStr::Format, "%d", \
			((QMultiLineEdit *)widget())->length());

	return true;
}

bool KviScriptMLEditObject::functionSetMaxLines(KviCommand * c, KviParameterList * p,\
	KviStr & b)
{
	ENTER_STACK_FRAME(c, "multilineedit::setMaxLines");

	if(widget())
	{
		bool bOk;
		int lines = p->getInt(&bOk);
		if(!bOk)
			return c->error(KviError_integerParameterExpected);

		((QMultiLineEdit *)widget())->setMaxLines(lines);
	}
	
	return c->leaveStackFrame();
}

bool KviScriptMLEditObject::functionMaxLines(KviCommand * c, KviParameterList * p, \
	KviStr & b)
{
	if(widget())
		b.append(KviStr::Format, "%d", \
			((QMultiLineEdit *)widget())->maxLines());

	return true;
}

bool KviScriptMLEditObject::functionInsert(KviCommand * c, KviParameterList * p, \
	KviStr & b)
{
	ENTER_STACK_FRAME(c, "multilineedit::insert");

	if(widget())
	{
		KviStr * pStr;
		if(!(pStr = p->safeFirst())->isEmpty())
			((QMultiLineEdit *)widget())->insert(pStr->ptr());
		else
			return c->error(KviError_notEnoughParameters);
	}
	
	return c->leaveStackFrame();
}

bool KviScriptMLEditObject::functionAppend(KviCommand * c, KviParameterList * p, \
	KviStr & b)
{
	ENTER_STACK_FRAME(c, "multilineedit::append");

	if(widget())
	{
		KviStr * pStr;
		if(!(pStr = p->safeFirst())->isEmpty())
			((QMultiLineEdit *)widget())->append(pStr->ptr());
		else
			return c->error(KviError_notEnoughParameters);
	}

	return c->leaveStackFrame();
}

bool KviScriptMLEditObject::functionCopy(KviCommand *, KviParameterList *, KviStr &)
{
	if(widget())
		((QMultiLineEdit *)widget())->copy();

	return true;
}

bool KviScriptMLEditObject::functionCut(KviCommand *, KviParameterList *, KviStr &)
{
	if(widget())
		((QMultiLineEdit *)widget())->cut();

	return true;
}

bool KviScriptMLEditObject::functionPaste(KviCommand *, KviParameterList *, KviStr &)
{
	if(widget())
		((QMultiLineEdit *)widget())->paste();

	return true;
}


//->Set Bold, Italic, Underline
bool KviScriptMLEditObject::functionsetBold(KviCommand *c,KviParameterList * p,KviStr &buffer)
{
	if(widget())
	{
		((QMultiLineEdit *)widget())->setBold(p->getBool());
	}
	return true;
}
bool KviScriptMLEditObject::functionsetUnderline(KviCommand *c,KviParameterList * p,KviStr &b)
{
	if(widget())
	{
		((QMultiLineEdit *)widget())->setUnderline(p->getBool());
	}
	return true;
}

bool KviScriptMLEditObject::functionsetItalic(KviCommand *c,KviParameterList * p,KviStr &)
{
	if(widget())
	{
		((QMultiLineEdit *)widget())->setItalic(p->getBool());
	}
	return true;
}


//-->Returns Italic , Bold, Underline. 
bool KviScriptMLEditObject::functionbold(KviCommand *c,KviParameterList * p,KviStr &b)
{
	if(widget())
		b.append(((QMultiLineEdit *)widget())->bold() ? '1' : '0');
	return true;
}
bool KviScriptMLEditObject::functionitalic(KviCommand *c,KviParameterList * params,KviStr &b)
{
	if(widget())
		b.append(((QMultiLineEdit *)widget())->italic() ? '1' : '0');
	return true;
}
bool KviScriptMLEditObject::functionunderline(KviCommand *c,KviParameterList * params,KviStr &b)
{
	if(widget())
		b.append(((QMultiLineEdit *)widget())->underline() ? '1' : '0');
	return true;
}
//->Zoom In, out at
bool KviScriptMLEditObject::functionzoomIn(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
        ENTER_STACK_FRAME(c,"multilineedit::zoomIn");

        if (!params->count()) 
        {
                ((QMultiLineEdit *)object())->zoomIn();
                        return c->leaveStackFrame();
        }

        KviStr * pVal = params->safeFirst();

        bool bOk;
        unsigned int uVal = pVal->toUInt(&bOk);
        
        if(!bOk)
        {
                c->warning(__tr("Invalid zoom value (%s)"),pVal->ptr());
                return c->leaveStackFrame();
        }

        ((QMultiLineEdit *)object())->zoomIn(uVal);

        return c->leaveStackFrame();
}

bool KviScriptMLEditObject::functionzoomOut(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
        ENTER_STACK_FRAME(c,"multilineedit::zoomOut");

        if (!params->count()) 
        {
                ((QMultiLineEdit *)object())->zoomOut();
                        return c->leaveStackFrame();
        }

        KviStr * pVal = params->safeFirst();

        bool bOk;
        unsigned int uVal = pVal->toUInt(&bOk);
        
        if(!bOk)
        {
                c->warning(__tr("Invalid zoom value (%s)"),pVal->ptr());
                return c->leaveStackFrame();
        }

        ((QMultiLineEdit *)object())->zoomOut(uVal);

        return c->leaveStackFrame();
}
bool KviScriptMLEditObject::functionzoomTo(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
        ENTER_STACK_FRAME(c,"multilineedit::zoomTo");

        KviStr * pVal = params->safeFirst();

        bool bOk;
        unsigned int uVal = pVal->toUInt(&bOk);
        
        if(!bOk)
        {
                c->warning(__tr("Invalid zoom value (%s)"),pVal->ptr());
                return c->leaveStackFrame();
        }

        ((QMultiLineEdit *)object())->zoomTo(uVal);

        return c->leaveStackFrame();
}
//-> Undo & Redo functions

bool KviScriptMLEditObject::functionundo(KviCommand *c,KviParameterList * p,KviStr &)
{
        if(widget())
        {
                ((QMultiLineEdit *)widget())->undo();
        }
        return true;
}
bool KviScriptMLEditObject::functionredo(KviCommand *c,KviParameterList * p,KviStr &)
{
        if(widget())
        {
                ((QMultiLineEdit *)widget())->redo();
        }
        return true;
}
bool KviScriptMLEditObject::functionclear(KviCommand *c,KviParameterList * p,KviStr &)
{
        if(widget())
        {
                ((QMultiLineEdit *)widget())->clear();
        }
        return true;
}
bool KviScriptMLEditObject::functiondel(KviCommand *c,KviParameterList * p,KviStr &)
{
        if(widget())
        {
                ((QMultiLineEdit *)widget())->del();
        }
        return true;
}
bool KviScriptMLEditObject::functionindent(KviCommand *c,KviParameterList * p,KviStr &)
{
        if(widget())
        {
                ((QMultiLineEdit *)widget())->indent();
        }
        return true;
}

bool KviScriptMLEditObject::functionsetUndoRedoEnabled(KviCommand *c,KviParameterList * p,KviStr &)
{
        if(widget())
        {
                ((QMultiLineEdit *)widget())->setUndoRedoEnabled(p->getBool());
        }
        return true;
}
 
bool KviScriptMLEditObject::functionisUndoRedoEnabled(KviCommand *c,KviParameterList * p,KviStr &b)
{
        if(widget())
                b.append(((QMultiLineEdit *)widget())->isUndoRedoEnabled() ? '1' : '0');
        return true;
}
bool KviScriptMLEditObject::functionisUndoAvailable(KviCommand *c,KviParameterList * p,KviStr &b)
{
        if(widget())
                b.append(((QMultiLineEdit *)widget())->isUndoAvailable() ? '1' : '0');
        return true;
}
bool KviScriptMLEditObject::functionisRedoAvailable(KviCommand *c,KviParameterList * p,KviStr &b)
{
        if(widget())
                b.append(((QMultiLineEdit *)widget())->isRedoAvailable() ? '1' : '0');
        return true;
}
bool KviScriptMLEditObject::functionsetUndoDepth(KviCommand *c,KviParameterList * params,KviStr &)
{
          ENTER_STACK_FRAME(c,"multilineedit::setUndoDepth");

        KviStr * pVal = params->safeFirst();

        bool bOk;
        unsigned int uVal = pVal->toUInt(&bOk);
        
        if(!bOk)
        {
                c->warning(__tr("Invalid depth value (%s)"),pVal->ptr());
                return c->leaveStackFrame();
        }

        ((QMultiLineEdit *)object())->setUndoDepth(uVal);

        return c->leaveStackFrame();
      
}
bool KviScriptMLEditObject::functionundoDepth(KviCommand *c,KviParameterList * p,KviStr &b)
{
        if(widget())
                b.append(((QMultiLineEdit *)widget())->undoDepth());
        return true;
}

//-->Text & Color & Family
bool KviScriptMLEditObject::functionsetText(KviCommand *c,KviParameterList * p,KviStr &buffer)
{
	if(widget())
	{
		if(p->first())
			((QMultiLineEdit *)widget())->setText(p->first()->ptr());
		else
			((QMultiLineEdit *)widget())->setText("");
	}
	return true;
}
bool KviScriptMLEditObject::functionsetColor(KviCommand *c,KviParameterList * p,KviStr &b)
{
    if(!widget())
            return true;
        if(KviStr * pS = p->first())
        {
                char * buff = 0;
                int len = pS->hexToBuffer(&buff, false);
                if(len == 3)
                {
                        ((QMultiLineEdit *)widget())->setColor( \
                                QColor((unsigned char)buff[0], \
                                (unsigned char)buff[1], \
                                (unsigned char)buff[2]));
                        b.append('1');
                        kvi_free(buff);
                        return c->leaveStackFrame();
                }
                if(len > 0)KviStr::freeBuffer(buff);
        }
        b.append('0');
        return true;
}
bool KviScriptMLEditObject::functionsetPointSize(KviCommand *c,KviParameterList * p,KviStr &b)
{
	ENTER_STACK_FRAME(c, "multilineedit::setPointSize");
	
	bool bOk;
	int cnt = p->getInt(&bOk);

	if(!bOk)
		return c->error(KviError_integerParameterExpected);

	((QMultiLineEdit *)widget())->setPointSize(cnt);
		
	return c->leaveStackFrame();
}

bool KviScriptMLEditObject::functionsetLinkUnderline(KviCommand *c,KviParameterList * p,KviStr &)
{
	if(widget())
	{
		((QMultiLineEdit *)widget())->setLinkUnderline(p->getBool());
	}
	return true;
}

bool KviScriptMLEditObject::functionsetFamily(KviCommand *c,KviParameterList * p,KviStr &)
{
	if(widget())
	{
		if(p->first())
			((QMultiLineEdit *)widget())->setFamily(p->first()->ptr());
	}
	return true;
}

bool KviScriptMLEditObject::functionsetModified(KviCommand *c,KviParameterList * p,KviStr &)
{
	if(widget())
	{
		((QMultiLineEdit *)widget())->setModified(p->getBool());
	}
	return true;
}

bool KviScriptMLEditObject::functioninsertParagraph(KviCommand *c,KviParameterList * p,KviStr &b)
{
	ENTER_STACK_FRAME(c,"multilineedit::insertparagraph");

	KviStr * szText = p->safeFirst();
	KviStr * siText = p->safeNext();

	bool bOk;
	int iText= siText->toUInt(&bOk);
	if(!bOk)
	{
		c->warning(__tr("Invalid  value (%s)"),siText->ptr());
		return c->leaveStackFrame();
	}
	((QMultiLineEdit *)widget())->insertParagraph(szText->ptr(),iText);


	return c->leaveStackFrame();
}

bool KviScriptMLEditObject::functionremoveParagraph(KviCommand *c,KviParameterList * p,KviStr &)
{
	ENTER_STACK_FRAME(c, "multilineedit::removeParagraph");
	
	bool bOk;
	int cnt = p->getInt(&bOk);

	if(!bOk)
		return c->error(KviError_integerParameterExpected);

	((QMultiLineEdit*)widget())->removeParagraph(cnt);
		
	return c->leaveStackFrame();
}

bool KviScriptMLEditObject::functionsetOverwriteMode(KviCommand *c,KviParameterList * p,KviStr &)
{
	if(widget())
	{
		((QMultiLineEdit *)widget())->setOverwriteMode(p->getBool());
	}
	return true;
}

bool KviScriptMLEditObject::functiontextFormat(KviCommand * c, KviParameterList * p,
	KviStr & b)
{

	KviStr ret;
	int fstyle = ((QMultiLineEdit *)widget())->textFormat();
	for(unsigned int i = 0; i < mod_num; i++)
	{
		if(fstyle & mod_cod[i])
		{
			ret.append(mod_tbl[i]);
			ret.append(",");
		}
	}
	ret.cutRight(1);	// get rid of last ","
	b.append(ret);
	return true;
}

bool KviScriptMLEditObject::functionsetTextFormat(KviCommand * c, KviParameterList *p,
	KviStr & b)
{
	KviStr * pModos = p->safeFirst();
	if(widget())
	{
		if(kvi_strEqualCI(pModos->ptr(),"PlainText"))
			((QMultiLineEdit *)widget())->setTextFormat(Qt::PlainText);
		if(kvi_strEqualCI(pModos->ptr(),"RichText"))
			((QMultiLineEdit *)widget())->setTextFormat(Qt::RichText);
		if(kvi_strEqualCI(pModos->ptr(),"LogText"))
			((QMultiLineEdit *)widget())->setTextFormat(Qt::LogText);
		if(kvi_strEqualCI(pModos->ptr(),"AutoText"))
			((QMultiLineEdit *)widget())->setTextFormat(Qt::AutoText);
	}
	return true;
}


bool KviScriptMLEditObject::functionsetParagraphBackgroundColor(KviCommand * c,
        KviParameterList * p, KviStr & b)
{
        if(!widget())
            return true;
                KviStr * pParag=p->safeFirst();
                bool bOk;
                int uParag = pParag->toUInt(&bOk);
        if(!bOk)
        {
                c->warning(__tr("Invalid paragraph value (%s)"),pParag->ptr());
                return c->leaveStackFrame();
        }
        if(KviStr * pS = p->next())
        {
                char * buff = 0;
                int len = pS->hexToBuffer(&buff, false);
                if(len == 3)
                {
                        ((QMultiLineEdit *)widget())->setParagraphBackgroundColor(uParag, \
                                QColor((unsigned char)buff[0], \
                                (unsigned char)buff[1], \
                                (unsigned char)buff[2]));
                        b.append('1');
                        kvi_free(buff);
                        return c->leaveStackFrame();
                }
                if(len > 0)KviStr::freeBuffer(buff);
        }
        b.append('0');

  return true;
}

bool KviScriptMLEditObject::functionclearParagraphBackground(KviCommand *c,KviParameterList * p,KviStr &)
{
	ENTER_STACK_FRAME(c, "multilineedit::clearParagraph");
	
	bool bOk;
	int cnt = p->getInt(&bOk);

	if(!bOk)
		return c->error(KviError_integerParameterExpected);

	((QMultiLineEdit*)widget())->clearParagraphBackground(cnt);
		
	return c->leaveStackFrame();
}

bool KviScriptMLEditObject::functionloadFile(KviCommand *c, KviParameterList *p, \
	KviStr & b)
{
	if(!widget())
	{
		c->warning(__tr("There is no widget"));
        return c->leaveStackFrame();
	}

	KviStr * pFile=p->safeFirst();
	QString nFile = pFile->ptr();

	if ( !QFile::exists(nFile))
	{
		c->warning(__tr(" I can't find the specified file."));
        return c->leaveStackFrame();
	}

	QFile file( nFile );
    if ( !file.open( IO_ReadOnly ) )
	{
		c->warning(__tr(" I cannot read the file."));
        return c->leaveStackFrame();
	}
    
	QTextStream ts( &file );
    QString txt = ts.read();
	if ( !QStyleSheet::mightBeRichText( txt ) )
	txt = QStyleSheet::convertFromPlainText( txt, QStyleSheetItem::WhiteSpacePre );
	((QMultiLineEdit *)widget())->setText( txt );
	file.close();
	return true;
}

bool KviScriptMLEditObject::functionsetAlignment(KviCommand * c, KviParameterList *p,
	KviStr & b)
{
	KviStr * pModos = p->safeFirst();
	if(widget())
	{
		if(kvi_strEqualCI(pModos->ptr(),"Left"))
			((QMultiLineEdit *)widget())->setAlignment(Qt::AlignLeft);
		if(kvi_strEqualCI(pModos->ptr(),"Right"))
			((QMultiLineEdit *)widget())->setAlignment(Qt::AlignRight);
		if(kvi_strEqualCI(pModos->ptr(),"Center"))
			((QMultiLineEdit *)widget())->setAlignment(Qt::AlignCenter);
		if(kvi_strEqualCI(pModos->ptr(),"Justify"))
			((QMultiLineEdit *)widget())->setAlignment(Qt::AlignJustify);
	}
	return true;
}

bool KviScriptMLEditObject::functionsetAutoFormatting(KviCommand * c, KviParameterList *p,
	KviStr & b)
{
	KviStr * pModos = p->safeFirst();
	if(widget())
	{
		if(kvi_strEqualCI(pModos->ptr(),"AutoNone"))
			((QMultiLineEdit *)widget())->setAlignment(QTextEdit::AutoNone);
		if(kvi_strEqualCI(pModos->ptr(),"BulletList"))
			((QMultiLineEdit *)widget())->setAlignment(QTextEdit::AutoBulletList);
		if(kvi_strEqualCI(pModos->ptr(),"AutoAll"))
			((QMultiLineEdit *)widget())->setAlignment(QTextEdit::AutoAll);
	}
	return true;
}

bool KviScriptMLEditObject::functionsetVerticalAlignment(KviCommand * c, KviParameterList *p,
	KviStr & b)
{
	KviStr * pModos = p->safeFirst();
	if(widget())
	{
		if(kvi_strEqualCI(pModos->ptr(),"Normal"))
			((QMultiLineEdit *)widget())->setVerticalAlignment(QTextEdit::AlignNormal);
		if(kvi_strEqualCI(pModos->ptr(),"SuperScript"))
			((QMultiLineEdit *)widget())->setVerticalAlignment(QTextEdit::AlignSuperScript);
		if(kvi_strEqualCI(pModos->ptr(),"SubScript"))
			((QMultiLineEdit *)widget())->setVerticalAlignment(QTextEdit::AlignSubScript);
	}
	return true;
}

bool KviScriptMLEditObject::functionparagraphs(KviCommand *c,KviParameterList * p,KviStr &b)
{
	if(widget())
		b.append(KviStr::Format, "%d", ((QMultiLineEdit *)widget())-> \
			paragraphs());
	else
		b.append("-1");
	return true;
}

bool KviScriptMLEditObject::functionlines(KviCommand *c,KviParameterList * p,KviStr &b)
{
	if(widget())
		b.append(KviStr::Format, "%d", ((QMultiLineEdit *)widget())-> \
			lines());
	else
		b.append("-1");
	return true;
}

bool KviScriptMLEditObject::functionlineOfChar(KviCommand *c,KviParameterList * p,KviStr &b)
{
	ENTER_STACK_FRAME(c, "multilineedit::linesOfParagraph");
	
	bool bOkp;
	bool bOki;
	int vall = p->getInt(&bOkp);
	int valp = p->getInt(&bOki);
	if(!bOkp &&  !bOki)

		return c->error(KviError_integerParameterExpected);

	b.append(KviStr::Format, "%d", ((QMultiLineEdit *)widget())->lineOfChar(valp, vall));

	return c->leaveStackFrame();


}

bool KviScriptMLEditObject::functionlinesOfParagraph(KviCommand *c,KviParameterList * p,KviStr &b)
{
	ENTER_STACK_FRAME(c, "multilineedit::linesOfParagraph");
	
	bool bOk;
	int val = p->getInt(&bOk);

	if(!bOk)

		return c->error(KviError_integerParameterExpected);

	b.append(KviStr::Format, "%d", ((QMultiLineEdit *)widget())->linesOfParagraph(val));
	return c->leaveStackFrame();


}


bool KviScriptMLEditObject::functionparagraphLength(KviCommand *c,KviParameterList * p,KviStr &b)
{
	ENTER_STACK_FRAME(c, "multilineedit::paragraphLength");
	
	bool bOk;
	int val = p->getInt(&bOk);

	if(!bOk)

		return c->error(KviError_integerParameterExpected);

	b.append(KviStr::Format, "%d", ((QMultiLineEdit *)widget())->paragraphLength(val));
	return c->leaveStackFrame();

}

bool KviScriptMLEditObject::functionselectAll(KviCommand *c,KviParameterList * p,KviStr &b)
{
	if(widget())
	{
		((QMultiLineEdit *)widget())->selectAll(p->getBool());
	}
	return true;
}

#include "m_class_multilineedit.moc"
