//mdm:
//   File : class_process.h
//   Creation date : Thu Feb 1 14:39:48 CEST 2005 
//	 by Tonino Imbesi(Grifisx) and Alessandro Carbone(Noldor)
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "class_process.h"
#include "kvi_error.h"
#include "kvi_debug.h"
#include "kvi_command.h"
#include "kvi_locale.h"
#include <qtimer.h>

static KviScriptObjectClass * g_pProcessClass = 0;

static KviScriptObject * ProcessClassCreateInstance(KviScriptObjectClass * c,
	KviScriptObject * p, const char * n)
{
	return new KviScriptProcessObject(c, p, n);
}

KviScriptProcessObject::KviScriptProcessObject(KviScriptObjectClass * c, KviScriptObject * p,
	const char *n) : KviScriptWidgetObject(c, p, n)
{
	m_pProcess = new QProcess();
	connect(m_pProcess,SIGNAL(readyReadStdout()),this,SLOT(slotReadStdout()));
	connect(m_pProcess,SIGNAL(readyReadStderr()),this,SLOT(slotReadStderr()));
}

KviScriptProcessObject::~KviScriptProcessObject()
{
	delete m_pProcess;
}

/*
	@doc:	process
	@keyterms:
		process object class, process
	@title:
		process class
	@type:
		class
	@short:
		A class to manage process. 
	@inherits:
		[class]object[/class]
	@description:
		The Process class  is used to start external programs and to communicate with them.[br]
		!WARNING AT LAST YOU HAVE TO CLOSE THE PROCESS!
	@functions:
		!fn: $addArg(<process-name>)
		With this command you give the process name (or more arguments) for comunication.
		Es: see the next example.
		!fn: $start()
		Tries to run the process.[br]
		Es: [br]
		%process=$new(process);[br]
		%process->$addArg("cmd.exe");[br]
		%process->$start();[br]
		!fn: $readStdout()
		Reads the data that the process has written to standard output.
		!fn: $readStderr()
		Reads the data that the process has written to standard error.
		!fn: $readyReadStdout()
		This signal is emitted when the process has written data to standard output.
		!fn: $readyReadStderr()
		This signal is emitted when the process has written data to standard error. 
		Es:[br]
		-------------------Start:[br]
		class (test,object)[br]
		{[br]
			slotReadStdout()[br]
			{[br]
	 			%stdo = %Process->$readStdout()[br]
				#%Aoutput->$append(%stdo);// coming soon in the new texteditor class[br]
		 		%Aoutput->$settext(%stdo);[br]
	 		}[br]
			slotReadStderr()[br]
			{[br]
	 			%stderr= %Process->$readStderr()[br]
		 		#%Aoutput->$append(%stderr);// coming soon in the new texteditor class[br]
				%Aoutput->$settext(%stderr);[br]
	 		}[br]
		}[br]
		
		%tt=$new(test)[br]
		%A=$new(widget)[br]
		%A->$setGeometry(100,100,400,300)[br]
		%layoutA=$new(layout,%A)[br]
		%Ainput=$new(lineedit,%A)[br]
		#%Aoutput=$new(textedit,%A)// coming soon in the new texteditor class[br]
		%Aoutput=$new(label,%A)[br]
		%bclosekill=$new(button,%A)[br]
		%bclosekill->$settext("&CloseKill ")[br]
		%bkill=$new(button,%A)[br]
		%bkill->$settext("&Kill ")[br]
		%bterminate=$new(button,%A)[br]
		%bterminate->$settext("&Ask to Terminate ")[br]
		%layoutA->$addwidget(%Ainput,0,0)[br]
		%layoutA->$addwidget(%Aoutput,1,0)[br]
		%layoutA->$addwidget(%bclosekill,3,0)[br]
		%layoutA->$addwidget(%bkill,4,0,)[br]
		%layoutA->$addwidget(%bterminate,5,0)[br]
				
		%Process=$new(process)[br]
		%Process->$addArg("cmd.exe")[br]
		%Process->$start();[br]
		
		connect %Process readyReadStdout %tt slotReadStdout[br]
		connect %Process readyReadStderr %tt slotReadStderr[br]
		privateimpl(%Ainput,returnPressedEvent)[br]
		{
		%command=%Ainput->$text() "\r\n"[br]
		%Process->$writeToStdin(%command);[br]
		%Ainput->$setText("");[br]
		}[br]
		
		privateimpl(%bclosekill,mousepressevent)[br]
		{[br]
			%Process->$closekill();[br]
			delete %A;[br]
		}[br]
		privateimpl(%bkill,mousepressevent)[br]
		{[br]
			%Process->$kill();[br]
			delete %A;[br]
		}[br]
		privateimpl(%bterminate,mousepressevent)[br]
		{[br]
			%Process->$tryTerminate();[br]
			delete %A;[br]
		}[br]
		%A->$show();[br]
		--------------------End.[br]
		!fn: $writeToStdin(<command>)
		Whit this command you send a command to the process:
		!fn: $closekill()
		This tries to terminate the process the nice way.[br]
		If the process is still running after 5 seconds, it terminates the process the hard way.[br]
		(I think that this is the better way.)[br]
		Es:
		%Process->close_kill();
		!fn: $kill()
		Kill the process in hard way.(Bad Idea)
		!fn: $tryTerminate()
		Tries to terminate the process.(It could be well but...)
		!fn: $closeStdin()
		Close the standard Input.
		!fn: $isRunning()
		Return 1 if the process is running, else return 0.
		!fn: $normalExit()
		Returns TRUE if the process has exited normally; otherwise returns FALSE.
		!fn: $readyReadStdoutEvent()
		This function is invoched by the process when there are new datas.[br]
		The default implementation emits the [classfnc]$readyReadStdout[/classfnc]() signal,[br]
		!fn: $readyReadStderrEvent()
		This function is invoched by the process when there are new error messages.[br]
		The default implementation emits the [classfnc]$readyReadStderr[/classfnc]() signal,[br]
		
	@signals:
		!sg: $readyReadStdout()
		This signal is emitted by the default implementation of [classfnc]$readyReadStdoutEvent[/classfnc]().[br]
		If you reimplement that function you will have to emit the signal manually (if you still need it).[br]
		!sg: $readyReadStderr()
		This signal is emitted by the default implementation of [classfnc]$readyReadStderrEvent[/classfnc]().[br]
*/

#define ProcessFuncReg(__nam, __func) \
	g_pProcessClass->registerFunctionHandler(__nam, \
	(KviScriptObjectFunctionHandlerProc)(KVI_PTR2MEMBER(KviScriptProcessObject::__func)), \
	0, true);

void KviScriptProcessObject::registerSelf()
{
	KviScriptObjectClass * base = g_pScriptObjectController-> \
		lookupClass("object");
	__range_valid(base);

	g_pProcessClass = new KviScriptObjectClass(base, "process", \
		ProcessClassCreateInstance, true);

	ProcessFuncReg("addArg", functionaddArgument);     
	ProcessFuncReg("start", functionstartProcess); 
	ProcessFuncReg("readStdout", functionreadStdout);
	ProcessFuncReg("readStderr", functionreadStderr);
	
	ProcessFuncReg("writeToStdin", functionwriteToStdin);
	ProcessFuncReg("closekill", functionclosekill);
	ProcessFuncReg("kill", functionkill);
	ProcessFuncReg("tryTerminate", functiontryTerminate);
	ProcessFuncReg("closeStdin", functioncloseStdin);
	ProcessFuncReg("isRunning",functionisRunning);
	ProcessFuncReg("normalExit",functionnormalExit);
// Events
	ProcessFuncReg("readyReadStdoutEvent",functionreadyReadStdoutEvent);
	ProcessFuncReg("readyReadStderrEvent",functionreadyReadStderrEvent);

}

void KviScriptProcessObject::unregisterSelf()
{
	delete g_pProcessClass;
    g_pProcessClass = 0;
}
//->Gives the process name or pass it an arguments.
bool KviScriptProcessObject::functionaddArgument(KviCommand * c, KviParameterList * p,	
	KviStr & b)
{
	ENTER_STACK_FRAME(c, "process::addArg");
	KviStr * pS = p->safeFirst();
	if(pS->isEmpty())
		return c->error(KviError_invalidParameter,__tr("Empty string"));
	m_pProcess->addArgument(pS->ptr());
	return c->leaveStackFrame();
}
//->Start the process.
bool KviScriptProcessObject::functionstartProcess(KviCommand * c, KviParameterList *, KviStr &)
{
	ENTER_STACK_FRAME(c, "process::start");

	if(!(m_pProcess->start()))
		return c->error(KviError_invalidParameter,__tr("Process could not be starded."));
	return c->leaveStackFrame();
}
//-->Read the standard output.
bool KviScriptProcessObject::functionreadStderr(KviCommand * c, KviParameterList *, KviStr & b)
{
	QString ng_Process =m_pProcess->readStderr();
	b.append(ng_Process);

	return true;
}
//-->Read the standard error.
bool KviScriptProcessObject::functionreadStdout(KviCommand * c, KviParameterList *, KviStr & b)
{
	QString ng_Process =m_pProcess->readStdout();
	b.append(ng_Process);

	return true;
}
//-->Signals and slot to manage reading output and error from the process.
bool KviScriptProcessObject::functionreadyReadStdoutEvent(KviCommand *c, KviParameterList *,	
	KviStr & b)
{

	ENTER_STACK_FRAME(c,"process::readyReadStout");
	emitSignal("readyReadStdout",0,0,c);
	return c->leaveStackFrame();

}

bool KviScriptProcessObject::functionreadyReadStderrEvent(KviCommand *c, KviParameterList *,	
	KviStr & b)
{

	ENTER_STACK_FRAME(c,"process::readyReadSterr");
	emitSignal("readyReadStderr",0,0,c);
	return c->leaveStackFrame();

}
void KviScriptProcessObject::slotReadStdout()
{

	callEventFunction("readyReadStdoutEvent");
}

void KviScriptProcessObject::slotReadStderr()
{

	callEventFunction("readyReadStderrEvent");
}
bool KviScriptProcessObject::functionwriteToStdin(KviCommand * c, KviParameterList * p,
	KviStr & buffer)
{

	KviStr *pS = p->safeFirst();
	m_pProcess->writeToStdin(pS->ptr());

	return true;
}
//-->The 3 Closing process finctions
bool KviScriptProcessObject::functionclosekill(KviCommand * c, KviParameterList *, KviStr &)
{
	//I try to  to terminate the process the nice way....
	m_pProcess->tryTerminate();
	//If the process is still running after 5 seconds, I'll terminate the process in the hard way.
	QTimer::singleShot( 5000, m_pProcess, SLOT( kill() ) );
	return true;
}
bool KviScriptProcessObject::functionkill(KviCommand * c, KviParameterList *, KviStr &)
{
	m_pProcess->kill();
	return true;
}
bool KviScriptProcessObject::functiontryTerminate(KviCommand * c, KviParameterList *, KviStr &)
{
	m_pProcess->tryTerminate();
	return true;
}
//-->Close the standard input.
bool KviScriptProcessObject::functioncloseStdin(KviCommand * c, KviParameterList *, KviStr &)
{
	
	m_pProcess->closeStdin();
	return true;
}
//->Returns if the process still runnig
bool KviScriptProcessObject::functionisRunning(KviCommand *, KviParameterList *, KviStr & b)
{
	b.append(m_pProcess->isRunning() ? '1' : '0');
	return true;
}
//->Returns if the process exited. 
bool KviScriptProcessObject::functionnormalExit(KviCommand *, KviParameterList *, KviStr & b)
{
	b.append(m_pProcess->normalExit() ? '1' : '0');
	return true;
}

#include "m_class_process.moc"
