//
//   File : class_widget.cpp 
//   Creation date : Mon Sep 11 16:35:32 CET 2000 by Krzysztof Godlewski
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "kvi_malloc.h"
#include "kvi_debug.h"
#include "kvi_command.h"
#include "kvi_error.h"
#include "kvi_locale.h"
#include "kvi_iconmanager.h"

#include "class_widget.h"

#include <qwidget.h>
#include <qtooltip.h>
#include <qfont.h>
static KviScriptObjectClass * g_pWidgetClass = 0;

static KviScriptObject * widgetClassCreateInstance(KviScriptObjectClass * cls,
	KviScriptObject * par, const char * name)
{
	return new KviScriptWidgetObject(cls, par, name);
}

KviScriptWidgetObject::KviScriptWidgetObject(KviScriptObjectClass *c, KviScriptObject *p,
	const char *n ) : KviScriptObject(c, p, n)
{
}

KviScriptWidgetObject::~KviScriptWidgetObject()
{
//	debug("KviScriptWidgetObject::~KviScriptWidgetObject:(%s)",id());
}

/*
	@doc: widget
	@keyterms:
		widget object class
	@title:
		widget class
	@type:
		class
	@short:
		Base class for all widgets
	@inherits:
		[class]object[/class]
	@description:
		This object class is the rappresentation of a widget.
		All the other widget-type classes inherit from this one.
	@functions:
		!fn: $show()
		Shows this widget and the children.
		See also [classfnc]$hide[/classfnc]() and [classfnc]$isVisible[/classfnc].
		!fn: $hide()
		Hides this widget (and conseguently all the children).
		See also [classfnc]$show[/classfnc]() and [classfnc]$isVisible[/classfnc].
		!fn: $x()
		Returns the x coordinate of the upper-left corner
		of this widget relative to the parent widget,
		or to the desktop if this widget is a toplevel one.
		!fn: $y()
		Returns the y coordinate of the uspper-left corner
		of this widget relative to the parent widget,
		or to the desktop if this widget is a toplevel one.
		!fn: $width()
		Returns the width of this widget in pixels.
		!fn: $height()
		Returns the height of this widget in pixels.
		!fn: $geometry()
		Returns the widget geometry in this form:[br]
		x, y, width, height.
		!fn: $setGeometry(<x>,<y>,<width>,<heigth>)
		Sets the geometry of this widget. <x> and <y> are relative
		to the parent widget or to the desktop (if this widget is
		a toplevel one). All the parameters are in pixels.
		!fn: $setMinimumWidth(<value>)
		Sets the minimum width of this widget to <value>.
		The user will not be able to resize the widget to a smaller
		value. This value is also used by the [class:layout]layout class[/class].
		!fn: $setMinimumHeight(<value>)
		Sets the minimum height of this widget to <value>.
		The user will not be able to resize the widget to a smaller
		value. This value is also used by the [class:layout]layout class[/class].
		!fn: $setMaximumWidth(<value>)
		Sets the maximum width of this widget to <value>.
		The user will not be able to resize the widget to a bigger
		value. This value is also used by the [class:layout]layout class[/class].
		!fn: $setMaximumHeight(<value>)
		Sets the maximum height of this widget to <value>.
		The user will not be able to resize the widget to a bigger
		value. This value is also used by the [class:layout]layout class[/class].
		!fn: $move(<x>,<y>)
		Moves this widget to the coordinate <x> and <y> relative to its
		parent widget (or the desktop if this widget is a toplevel one).
		This is equivalent to [classfnc]$setGeometry[/classfnc](<x>,<y>,
		[classfnc]$width[/classfnc](),[classfnc]$height[/classfnc]()).
		!fn: $resize(<width>,<height>)
		Changes the widget's width to <width> and height to <height>. 
		See also [classfnc]$setGeometry[/classfnc]().
		!fn: $isEnabled()
		Returns '1' if the widget is enabled , '0' otherwise.
		See also [classfnc:widget]$setEnabled[/classfnc]().
		!fn: $setEnabled(<bool>)
		Sets the widget state to enabled or disabled if <bool> is 1 or 0 respectively.
		A disabled widget does not receive keyboard nor mouse input.
		!fn: $setCaption(<text>)
		Sets the caption of this widget to <text>.
		This is meaningful for toplevel widgets only.
		!fn: $setToolTip(<tooltip_text>)
		Set the tooltip of this widget; the text can contain HTML formatting.
		!fn: $caption()
		Returns the caption text of this widget.
		!fn: $isTopLevel()
		Returns '1' if this widget is a toplevel (parentless) one,
		'0' otherwise.
		!fn: $isVisible()
		Returns '1' if this widget is currently visible (read: is managed
		by the window manager and displayed by the X server; the widget
		may be hidden behind other widgets). If the widget is not visible
		this function returns '0'.
		See also [classfnc]$show[/classfnc]() and [classfnc]$hide[/classfnc]().
		!fn: $raise()
		Moves this widget to the top of the stack of the widgets relative
		to its parent. See also [classfnc]$lower[/classfnc].
		!fn: $lower()
		Moves this widget to the bottom of the stack of the widgets relative
		to its parent. See also [classfnc]$raise[/classfnc]
		!fn: $hasFocus()
		Returns '1' if this widget has the keyboard focus.
		See also [classfnc]$setFocus[/classfnc].
		!fn: $setFocus()
		Sets this widget to be the one that receives keyboard events.
		See also [classfnc]$hasFocus[/classfnc]
		!fn: $parentWidget()
		Returns the object id of the parent widget, or '0' if this
		widget is a toplevel one.
		!fn: $backgroundColor()
		Returns the background color of this widget in hexadecimal
		html-like format. For example , for a black bacground you will get
		the string "000000" , for a red one , "FF0000", for a white one
		"FFFFFF". See also [classfnc]$setBackgroundColor[/classfnc]()
		!fn: $setBackgroundColor(<color>)
		Sets the background color of this widget to <color>.
		!fn: $setForegroundColor(<color>)
		Sets the foreground color of this widget to <color>.
		<color> must be a string with 6 hexadecimal digits (like the ones used to
		specify colors in html pages). The first two digits specify
		the RED component, the third and fourth digit specify the GREEN component
		and the last two specify the BLUE component.
		For example "FFFF00" means full red, full green and no blue that gives
		a yellow color, "808000" designates a brown color (dark yellow),
		"A000A0" is a kind of violet. See also [classfnc]$foregroundColor[/classfnc].
		!fn: $foregroundColor()
		Returns the foreground color of this widget in hexadecimal
		html-like format.
		See also [classfnc]$setForegroundColor[/classfnc].
		!fn: $setMouseTracking(<bool>)
		Enables or disables the mouse tracking if <bool> is '1' or '0' respectively.
		When mouse tracking is enabled you will receive mouse move events
		even if no button is pressed, otherwise you will receive it only
		when a mouse button is being pressed (so after a mousePressEvent).
		!fn: $mousePressEvent(<button>,<x>,<y>)
		This function is called when a mouse button is pressed while
		the cursor is in this widget. <button> is 0 if the pressed button
		is the left one, 1 if the button is the right one and 2 if it is the middle one.
		The <x> and <y> coordinates are relative to this widget upper-left corner
		and are expressed in pixels.
		If you call "[cmd]setreturn[/cmd] 1" you will stop the internal processing
		of this event. The default implementation does nothing.
		!fn: $mouseReleaseEvent(<button>,<x>,<y>)
		This function is called when a mouse button is released while
		the cursor is in this widget. <button> is 0 if the released button
		is the left one, 1 if the button is the right one and 2 if it is the middle one.
		The <x> and <y> coordinates are relative to this widget upper-left corner
		and are expressed in pixels.
		If you call "[cmd]setreturn[/cmd] 1" you will stop the internal processing
		of this event. The default implementation does nothing.
		!fn: $mouseDoubleClickEvent(<button>,<x>,<y>)
		This function is called when a mouse button is double clicked while
		the cursor is in this widget. <button> is 0 if the double clicked button
		is the left one, 1 if the button is the right one and 2 if it is the middle one.
		The <x> and <y> coordinates are relative to this widget upper-left corner
		and are expressed in pixels.
		If you call "[cmd]setreturn[/cmd] 1" you will stop the internal processing
		of this event. The default implementation does nothing.
		!fn: $mouseMoveEvent(<button>,<x>,<y>)
		This function is called when the mouse cursor moves inside this widget.
		<button> is 0 if the pressed button
		is the left one, 1 if the button is the right one and 2 if it is the middle one.
		The special value of -1 indicates that no button is being pressed.
		The <x> and <y> coordinates are relative to this widget upper-left corner
		and are expressed in pixels.
		Normally you will receive this event only if a mouse button is being pressed
		while moving. If you want to receive it also when the mouse buttons are not
		pressed, call [classfnc]$setMouseTracking[/classfnc]().
		If you call "[cmd]setreturn[/cmd] 1" you will stop the internal processing
		of this event. The default implementation does nothing.
		!fn: $focusInEvent()
		This function is called when this widget gains keyboard focus.
		If you call "[cmd]setreturn[/cmd] 1" you will stop the internal processing
		of this event. The default implementation does nothing.
		!fn: $focusOutEvent()
		This function is called when this widget looses keyboard focus.
		If you call "[cmd]setreturn[/cmd] 1" you will stop the internal processing
		of this event. The default implementation does nothing.
		!fn: $mouseLeaveEvent()
		This function is called when the mouse leaves this widget.
		If you call "[cmd]setreturn[/cmd] 1" you will stop the internal processing
		of this event. The default implementation does nothing.
		!fn: $mouseEnterEvent()
		This function is called when the mouse enters this widget.
		If you call "[cmd]setreturn[/cmd] 1" you will stop the internal processing
		of this event. The default implementation does nothing.
		!fn: $showEvent()
		This function is called when this widget is being shown.
		If you call "[cmd]setreturn[/cmd] 1" you will stop the internal processing
		of this event. The default implementation does nothing.
		!fn: $hideEvent()
		This function is called when this widget is being hidden.
		If you call "[cmd]setreturn[/cmd] 1" you will stop the internal processing
		of this event. The default implementation does nothing.
		!fn: $closeEvent()
		This function is called when this widget is going to be closed.
		If you call "[cmd]setreturn[/cmd] 1" you will ignore the close event.
		The default implementation does nothing.
		!fn: $resizeEvent()
		This function is called immediately after this widget has been resized.
		If you call "[cmd]setreturn[/cmd] 1" you will stop the internal processing
		of this event. The default implementation does nothing.
		!fn: $moveEvent()
		This function is called immediately after this widget has been moved.
		If you call "[cmd]setreturn[/cmd] 1" you will stop the internal processing
		of this event. The default implementation does nothing.
		!fn: $setIcon(<image_id>)
		Sets the icon for this widget. This is meaningful only for toplevel widgets.
		See the [doc:image_id]image identifier[/doc] documentation for the explaination
		of the <image_id> parameter.
		!fn: $setBackgroundImage(<image_id>)
		Sets the background image for this widget.
		See the [doc:image_id]image identifier[/doc] documentation for the explaination
		of the <image_id> parameter.
		For some kind of widgets, setting a background pixmap may have no effect or
		have strange results. Experiment with it.
		To unset the background image call [classfnc]$setBackgroundColor[/classfnc]
		!fn: $setFont(<size>,<family>,<style>)[br]
		Set the font's size, family and stile, valid flag for style are:[br]
		[pre]
		italic     [br]
		bold     [br]
		underline      [br]
		overline    [br]
		strikeout  [br]
		fixedpitch  [br]
		[/pre]
*/

bool KviScriptWidgetObject::init(KviCommand *)
{
	// All the WIDGET TYPE objects will create the widget in the "init" function
	// later we can force parameter passing here if we need it
	setObject(new QWidget(parentScriptWidget(),name()),true);
	return true;
}


#define widgetFuncReg(__nam, __func) \
	g_pWidgetClass->registerFunctionHandler( __nam, \
	(KviScriptObjectFunctionHandlerProc)(KVI_PTR2MEMBER(KviScriptWidgetObject::__func)), \
	0, true);

void KviScriptWidgetObject::registerSelf()
{
	KviScriptObjectClass * base = g_pScriptObjectController-> \
		lookupClass("object");
	__range_valid(base);

	g_pWidgetClass = new KviScriptObjectClass(base, "widget",
		widgetClassCreateInstance, true);
	__range_valid(base);
		
	widgetFuncReg("show", functionShow);
	widgetFuncReg("hide", functionHide);
	widgetFuncReg("x", functionX);
	widgetFuncReg("y", functionY);
	widgetFuncReg("width", functionWidth);
	widgetFuncReg("height", functionHeight);
	widgetFuncReg("setMinimumWidth", functionSetMinimumWidth);
	widgetFuncReg("setMinimumHeight", functionSetMinimumHeight);
	widgetFuncReg("setMaximumWidth", functionSetMaximumWidth);
	widgetFuncReg("setMaximumHeight", functionSetMaximumHeight);
	widgetFuncReg("geometry", functionGeometry);
	widgetFuncReg("setGeometry", functionSetGeometry);
	widgetFuncReg("move", functionMove);
	widgetFuncReg("resize", functionResize);
	widgetFuncReg("isEnabled", functionIsEnabled);
	widgetFuncReg("setEnabled", functionSetEnabled);
	widgetFuncReg("setCaption", functionSetCaption);
	widgetFuncReg("caption", functionCaption);
	widgetFuncReg("isTopLevel", functionIsTopLevel);
	widgetFuncReg("isVisible", functionIsVisible);
	widgetFuncReg("raise", functionRaise);
	widgetFuncReg("lower", functionLower);
	widgetFuncReg("hasFocus", functionHasFocus);
	widgetFuncReg("setFocus", functionSetFocus);
	widgetFuncReg("parentWidget", functionParentWidget);
	widgetFuncReg("setBackgroundColor", functionSetBackgroundColor);
	widgetFuncReg("setForegroundColor",functionsetPaletteForegroundColor); // Noldor/Grifisx
	widgetFuncReg("backgroundColor", functionBackgroundColor);
	widgetFuncReg("foregroundColor", functionforegroundColor);
	widgetFuncReg("setMouseTracking",functionSetMouseTracking);
	widgetFuncReg("setIcon",functionSetIcon);
	widgetFuncReg("setBackgroundImage",functionSetBackgroundImage); 
	widgetFuncReg("setToolTip", functionSetToolTip); // Noldor/Grifisx
	widgetFuncReg("setFont", functionSetFont); //Noldor & Grifisx
	widgetFuncReg("sizeHint", functionsizeHint);

	g_pWidgetClass->registerEmptyFunctionHandler("mousePressEvent");
	g_pWidgetClass->registerEmptyFunctionHandler("mouseReleaseEvent");
	g_pWidgetClass->registerEmptyFunctionHandler("mouseDoubleClickEvent");
	g_pWidgetClass->registerEmptyFunctionHandler("mouseMoveEvent");
	g_pWidgetClass->registerEmptyFunctionHandler("focusInEvent");
	g_pWidgetClass->registerEmptyFunctionHandler("focusOutEvent");
	g_pWidgetClass->registerEmptyFunctionHandler("mouseLeaveEvent");
	g_pWidgetClass->registerEmptyFunctionHandler("mouseEnterEvent");
	g_pWidgetClass->registerEmptyFunctionHandler("showEvent");
	g_pWidgetClass->registerEmptyFunctionHandler("hideEvent");
	g_pWidgetClass->registerEmptyFunctionHandler("closeEvent");
	g_pWidgetClass->registerEmptyFunctionHandler("resizeEvent");
	g_pWidgetClass->registerEmptyFunctionHandler("moveEvent");
}

void KviScriptWidgetObject::unregisterSelf()
{
	delete g_pWidgetClass;
    g_pWidgetClass = 0;
}

bool KviScriptWidgetObject::eventFilter(QObject *o,QEvent *e)
{
	if(o == object())
	{
		KviStr ret;
		int aparam;
		switch(e->type())
		{
			case QEvent::MouseButtonPress:
				if(((QMouseEvent *)e)->button() & LeftButton)aparam = 0;
				else {
					if(((QMouseEvent *)e)->button() & RightButton)aparam = 1;
					else aparam = 2;
				}
				if(!callEventFunction("mousePressEvent",&ret,
					new KviParameterList(
						new KviStr(KviStr::Format,"%d",aparam),
						new KviStr(KviStr::Format,"%d",((QMouseEvent *)e)->pos().x()),
						new KviStr(KviStr::Format,"%d",((QMouseEvent *)e)->pos().y())
					)
				))ret = ""; // ignore results of a broken event handler
			break;
			case QEvent::MouseButtonRelease:
				if(((QMouseEvent *)e)->button() & LeftButton)aparam = 0;
				else {
					if(((QMouseEvent *)e)->button() & RightButton)aparam = 1;
					else aparam = 2;
				}
				if(!callEventFunction("mouseReleaseEvent",&ret,
					new KviParameterList(
						new KviStr(KviStr::Format,"%d",aparam),
						new KviStr(KviStr::Format,"%d",((QMouseEvent *)e)->pos().x()),
						new KviStr(KviStr::Format,"%d",((QMouseEvent *)e)->pos().y())
					)
				))ret = ""; // ignore results of a broken event handler
			break;
			case QEvent::MouseButtonDblClick:
				if(((QMouseEvent *)e)->button() & LeftButton)aparam = 0;
				else {
					if(((QMouseEvent *)e)->button() & RightButton)aparam = 1;
					else aparam = 2;
				}
				if(!callEventFunction("mouseDoubleClickEvent",&ret,
					new KviParameterList(
						new KviStr(KviStr::Format,"%d",aparam),
						new KviStr(KviStr::Format,"%d",((QMouseEvent *)e)->pos().x()),
						new KviStr(KviStr::Format,"%d",((QMouseEvent *)e)->pos().y())
					)
				))ret = ""; // ignore results of a broken event handler
			break;
			case QEvent::MouseMove:
				if(((QMouseEvent *)e)->state() & LeftButton)aparam = 0;
				else {
					if(((QMouseEvent *)e)->state() & RightButton)aparam = 1;
					else {
						if(((QMouseEvent *)e)->state() & MidButton)aparam = 2;
						else aparam = -1;
					}
				}
				if(!callEventFunction("mouseMoveEvent",&ret,
					new KviParameterList(
						new KviStr(KviStr::Format,"%d",aparam),
						new KviStr(KviStr::Format,"%d",((QMouseEvent *)e)->pos().x()),
						new KviStr(KviStr::Format,"%d",((QMouseEvent *)e)->pos().y())
					)
				))ret = ""; // ignore results of a broken event handler
			break;
			case QEvent::FocusIn:
				if(!callEventFunction("focusInEvent",&ret,0))ret = "";
			break;
			case QEvent::FocusOut:
				if(!callEventFunction("focusOutEvent",&ret,0))ret = "";
			break;
			case QEvent::Resize:
				if(!callEventFunction("resizeEvent",&ret,0))ret = "";
			break;
			case QEvent::Move:
				if(!callEventFunction("moveEvent",&ret,0))ret = "";
			break;
			case QEvent::Close:
				if(!callEventFunction("closeEvent",&ret,0))ret = "";
			break;
			case QEvent::Enter:
				if(!callEventFunction("mouseEnterEvent",&ret,0))ret = "";
			break;
			case QEvent::Leave:
				if(!callEventFunction("mouseLeaveEvent",&ret,0))ret = "";
			break;
			case QEvent::Show:
				if(!callEventFunction("showEvent",&ret,0))ret = "";
			break;
			case QEvent::Hide:
				if(!callEventFunction("hideEvent",&ret,0))ret = "";
			break;
			default:
				return KviScriptObject::eventFilter(o,e);
			break;
		}
		if(ret.len() == 1)
		{
			if(kvi_strEqualCI("1",ret.ptr()))return true;
		}
	}
	return KviScriptObject::eventFilter(o,e);
}


//
// If you never call c->error() , c->warning() (your function never fails)
// and never call a function that can call c->error() or c->warning()
// you can avoid ENTER_STACK_FRAME and c->leaveStackFrame()
// Just return true.
// c->error() should be called only in really critical situations
// (we have to define better "critical situation")
// if possible , leave the things go after just a c->warning()
//

bool KviScriptWidgetObject::functionShow(KviCommand * c,KviParameterList * params, 
	KviStr & buffer)
{
	// widget() will be never 0 in THIS class
	// but in derived classes WILL be
	// ... now that I think about it , it
	// may happen that widget() will be zero here too:
	// If the Qt "physical" widget gets destroyed
	// by some external factor (for example when children
	// of a wrapper widget destroyed by KVIrc).
	//
	// as a convention:
	//   if widget() can be 0 in a class
	//   the user must have a function to check it
	//   (sth as object::$hasObject() could do the trick)
	//   obviously this will happen only in well defined cases
	//   as in a qtwrapper not yet wrapped or failed to wrap (so
	//   qtwrapper::$wrap() returned '0' for example)
	//   or after the widget has been destroyed...
	//   if widget() is 0 , the functions perform no actions
	//   return no errors and results that have to be assumed
	//   as invalid (this allows the minimum overhead: if widget()
	//   is 0 : do nothing)
	if(widget())
	    widget()->show();
	return true;
}

bool KviScriptWidgetObject::functionHide(KviCommand * c, KviParameterList * p,
	KviStr & b)
{
	if(widget())widget()->hide();
	return true;
}
// Start patch by Noldor/Grifisx
bool KviScriptWidgetObject::functionSetToolTip(KviCommand * c, KviParameterList * p, \
	KviStr & b)
{
	if(widget())
	{
		if(p->first())
			QToolTip::add( widget(), (p->first()->ptr() ) );
		else
			QToolTip::add( widget(), "" );
			}
	return true;
}
// End patch
bool KviScriptWidgetObject::functionX(KviCommand *c, KviParameterList *, KviStr & b)
{
	if(widget())b.append(KviStr::Format,"%d",widget()->x());
	return true;
}

bool KviScriptWidgetObject::functionY(KviCommand *c, KviParameterList *, KviStr & b)
{
	if(widget())b.append(KviStr::Format,"%d",widget()->y());
	return true;
}

bool KviScriptWidgetObject::functionWidth(KviCommand *c, KviParameterList *,	
	KviStr & b)
{
	if(widget())b.append(KviStr::Format,"%d",widget()->width());
	return true;
}

bool KviScriptWidgetObject::functionHeight(KviCommand *c, KviParameterList *,
	KviStr & b)
{
	if(widget())b.append(KviStr::Format,"%d",widget()->height());
	return true;
}

bool KviScriptWidgetObject::functionSetMinimumWidth(KviCommand *c,
	KviParameterList * p, KviStr & b)
{
	ENTER_STACK_FRAME(c, "widget::setMinimumWidth");
	if(!widget())
	    return c->leaveStackFrame(); // do nothing
	bool bOk;
	int val = p->getInt(&bOk);
	if(bOk)
	    widget()->setMinimumWidth(val);
	else
	    c->warning(__tr("Invalid width parameter"));
	return c->leaveStackFrame();
}

bool KviScriptWidgetObject::functionSetMinimumHeight(KviCommand *c,
	KviParameterList * p, KviStr & b)
{
	ENTER_STACK_FRAME(c, "widget::setMinimumHeight");
	if(!widget())
	    return c->leaveStackFrame(); // do nothing
	bool bOk;
	int val = p->getInt(&bOk);
	if(bOk)widget()->setMinimumHeight(val);
	else c->warning(__tr("Invalid height parameter"));
	return c->leaveStackFrame();
}

bool KviScriptWidgetObject::functionSetMaximumWidth(KviCommand *c,
	KviParameterList * p, KviStr & b)
{
	ENTER_STACK_FRAME(c, "widget::setMaximumWidth");
	if(!widget())
	    return c->leaveStackFrame(); // do nothing
	bool bOk;
	int val = p->getInt(&bOk);
	if(bOk)widget()->setMaximumWidth(val);
	else c->warning(__tr("Invalid width parameter"));
	return c->leaveStackFrame();
}

bool KviScriptWidgetObject::functionSetMaximumHeight(KviCommand *c,	
	KviParameterList * p, KviStr & b)
{
	ENTER_STACK_FRAME(c, "widget::setMaximumHeight");
	if(!widget())
	    return c->leaveStackFrame(); // do nothing
	bool bOk;
	int val = p->getInt(&bOk);
	if(bOk)widget()->setMaximumHeight(val);
	else c->warning(__tr("Invalid height parameter"));
	return c->leaveStackFrame();
}

bool KviScriptWidgetObject::functionGeometry(KviCommand *c, KviParameterList *p,
	KviStr & b)
{
	ENTER_STACK_FRAME(c, "widget::geometry");

	QRect r = widget()->geometry();

	b.append(KviStr::Format, "%d, %d, %d, %d", r.x(), r.y(), r.width(),
		r.height());

	return c->leaveStackFrame();
}

bool KviScriptWidgetObject::functionSetGeometry(KviCommand *c, KviParameterList *p,
	KviStr & b)
{
	ENTER_STACK_FRAME(c, "widget::setGeometry");
	if(!widget())
	    return c->leaveStackFrame(); // do nothing
	bool bOk;
	QRect r = p->getRect(&bOk);
	if(!bOk)c->warning(__tr("The parameters do not define a rectangle"));
	else widget()->setGeometry(r);
	return c->leaveStackFrame();
}

bool KviScriptWidgetObject::functionMove(KviCommand * c,KviParameterList * p,
	KviStr & b)
{
	ENTER_STACK_FRAME(c, "widget::move");
	if(!widget())
	    return c->leaveStackFrame(); // do nothing
	bool bOk;
	QPoint r = p->getPoint(&bOk);
	if(!bOk)c->warning(__tr("The parameters do not define a point"));
	else widget()->move(r);
	return c->leaveStackFrame();
}
bool KviScriptWidgetObject::functionsizeHint(KviCommand *c, KviParameterList *p,
	KviStr & b)
{
	ENTER_STACK_FRAME(c, "widget::sizeHint");
	QSize rect=widget()->sizeHint();
	if(widget())b.append(KviStr::Format,"%d,%d",rect.width(),rect.height());
	return c->leaveStackFrame();
}

bool KviScriptWidgetObject::functionResize(KviCommand * c, KviParameterList * p, 
	KviStr & b)
{
	ENTER_STACK_FRAME(c, "widget::resize");
	if(!widget())
		return c->leaveStackFrame();
	bool bOk;
	QSize s = p->getSize(&bOk);
	if(!bOk)
		c->warning(__tr("The parameters are do not define a valid " \
			"size"));
	else
		widget()->resize(s);
	return c->leaveStackFrame();
}

bool KviScriptWidgetObject::functionIsEnabled(KviCommand * c, KviParameterList * p,
	KviStr & b)
{
	if(widget())b.append(widget()->isEnabled() ? '1' : '0');
	return true;
}

bool KviScriptWidgetObject::functionSetEnabled(KviCommand * c, KviParameterList * p,
	KviStr & b)
{
	if(widget())
	{
		bool en = p->getBool();
		widget()->setEnabled(en);
	}
	return true;
}

bool KviScriptWidgetObject::functionSetMouseTracking(KviCommand * c, 
	KviParameterList * p, KviStr & b)
{
	if(widget())
	{
		bool en = p->getBool();
		widget()->setMouseTracking(en);
	}
	return true;
}

bool KviScriptWidgetObject::functionSetCaption(KviCommand * c, KviParameterList * p,
	KviStr & buffer)
{
	if(widget())
	{
		if(KviStr *pS = p->first())
		    widget()->setCaption(pS->ptr());
		else
		    // bee a good boy...default to "empty string"
		    widget()->setCaption("");
	}
	return true;
}

bool KviScriptWidgetObject::functionCaption(KviCommand * c, KviParameterList * p,
	KviStr & b)
{
	if(widget())
	    b.append(widget()->caption().latin1());
	return true;
}

bool KviScriptWidgetObject::functionIsTopLevel(KviCommand * c, KviParameterList * p,
	KviStr & b)
{
	if(widget())
	    b.append(widget()->isTopLevel() ? '1' : '0');
	return true;
}

bool KviScriptWidgetObject::functionIsVisible(KviCommand * c, KviParameterList * p,
	KviStr & b)
{
	if(widget())
	    b.append(widget()->isVisible() ? '1' : '0');
	return true;
}

bool KviScriptWidgetObject::functionRaise(KviCommand * c, KviParameterList * p,
	KviStr & b)
{
	if(widget())
	    widget()->raise();
	return true;
}

bool KviScriptWidgetObject::functionLower(KviCommand * c, KviParameterList * p,
	KviStr & b)
{
	if(widget())
	    widget()->lower();
	return true;
}

bool KviScriptWidgetObject::functionHasFocus(KviCommand * c, KviParameterList * p,
	KviStr & b)
{
	if(widget())
	    b.append(widget()->hasFocus() ? '1' : '0');
	return true;
}

bool KviScriptWidgetObject::functionSetFocus(KviCommand * c, KviParameterList * p,
	KviStr & b)
{
	if(widget())
	    widget()->setFocus();
	return true;
}

bool KviScriptWidgetObject::functionParentWidget(KviCommand * c, KviParameterList * p,
	KviStr & b)
{
    parentScriptWidget() ? b.append(parentObject()->id()) : b.append('0');
    return true;
//    if(parentScriptWidget()) {
//		b.append(parentObject()->id());
//		return c->leaveStackFrame();
//	}
//	b.append('0');
//	return true;
}

bool KviScriptWidgetObject::functionSetBackgroundColor(KviCommand * c,
	KviParameterList * p, KviStr & b)
{
	if(!widget())
	    return true;
	if(KviStr * pS = p->first())
	{
		char * buff = 0;
		int len = pS->hexToBuffer(&buff, false);
		if(len == 3)
		{
			widget()->setBackgroundColor( \
				QColor((unsigned char)buff[0], \
				(unsigned char)buff[1], \
				(unsigned char)buff[2]));
			b.append('1');
			kvi_free(buff);
			return c->leaveStackFrame();
		}
		if(len > 0)KviStr::freeBuffer(buff);
	}
	b.append('0');
	return true;
}
// Start patch by Noldor/Grifisx
bool KviScriptWidgetObject::functionsetPaletteForegroundColor(KviCommand * c,
	KviParameterList * p, KviStr & b)
{
	if(!widget())
	    return true;
	if(KviStr * pS = p->first())
	{
		char * buff = 0;
		int len = pS->hexToBuffer(&buff, false);
		if(len == 3)
		{
			widget()->setPaletteForegroundColor( \
				QColor((unsigned char)buff[0], \
				(unsigned char)buff[1], \
				(unsigned char)buff[2]));
			b.append('1');
			kvi_free(buff);
			return c->leaveStackFrame();
		}
		if(len > 0)KviStr::freeBuffer(buff);
	}
	b.append('0');
	return true;
}
// end patch
bool KviScriptWidgetObject::functionBackgroundColor(KviCommand * c,KviParameterList *,
	KviStr & b)
{
	if(!widget())return true;
	QColor col = widget()->backgroundColor();
	char buff[3];
	buff[0] = col.red();
	buff[1] = col.green();
	buff[2] = col.blue();
	KviStr tmp;
	tmp.bufferToHex(buff, 3);
	b.append(tmp);
	return true;
}

bool KviScriptWidgetObject::functionforegroundColor(KviCommand * c,KviParameterList *,
	KviStr & b)
{
	if(!widget())return true;
	QColor col = widget()->foregroundColor();
	char buff[3];
	buff[0] = col.red();
	buff[1] = col.green();
	buff[2] = col.blue();
	KviStr tmp;
	tmp.bufferToHex(buff, 3);
	b.append(tmp);
	return true;
}

bool KviScriptWidgetObject::functionSetBackgroundImage(KviCommand * c,KviParameterList *p,
	KviStr & b)
{
	if(!widget())return true;
	if(!p->first())return true; //< --- ?
	QPixmap * pix = g_pIconManager->getImage(p->first()->ptr());
	if(pix)widget()->setBackgroundPixmap(*pix);
	return true;
}

bool KviScriptWidgetObject::functionSetIcon(KviCommand * c,KviParameterList *p,
	KviStr & b)
{
	if(!widget())return true;
	if(!p->first())return true; //< --- ?
	QPixmap * pix = g_pIconManager->getImage(p->first()->ptr());
	if(pix)widget()->setIcon(*pix);
	return true;
}

//-Grifisx & Noldor Start:
bool KviScriptWidgetObject::functionSetFont(KviCommand * c, KviParameterList * params,
	KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"widget::setfont");
	KviStr * pSize = params->safeFirst();
	KviStr * pFamily = params->safeNext();
	KviStr * pStile = params->safeNext();

	bool bOk;
	int uSize = pSize->toInt(&bOk);
	if(!bOk)
	{
		c->warning(__tr("Invalid size parameter (%s)"),pSize->ptr());
		return c->leaveStackFrame();
	}

	QFont  grfont(widget()->font());
	grfont.setPointSize(uSize);
    widget()->setFont(grfont);
	
	QFont  nlfont(widget()->font());
	nlfont.setFamily(pFamily->ptr());  
    widget()->setFont(nlfont);

	QString stiloso = pStile->ptr();

	if( stiloso =="italic")
	{
		QFont  gffont(widget()->font());
		gffont.setItalic(TRUE);
		widget()->setFont(gffont);
	}
	if( stiloso =="bold")
	{
		QFont  gffont(widget()->font());
		gffont.setBold(TRUE);
		widget()->setFont(gffont);
	}
	if( stiloso =="underline")
	{
		QFont  gffont(widget()->font());
		gffont.setUnderline(TRUE);
		widget()->setFont(gffont);
	}
	if( stiloso =="overline")
	{
		QFont  gffont(widget()->font());
		gffont.setOverline(TRUE);
		widget()->setFont(gffont);
	}
	if( stiloso =="strikeout")
	{
		QFont  gffont(widget()->font());
		gffont.setStrikeOut(TRUE);
		widget()->setFont(gffont);
	}
	if( stiloso =="fixedpitch")
	{
		QFont  gffont(widget()->font());
		gffont.setFixedPitch(TRUE);
		widget()->setFont(gffont);
	}

	return c->leaveStackFrame();
}
//--G&N end


#include "m_class_widget.moc"
