//======================================================================================
//
//   File : libkviregchan.cpp
//   Creation date : Sat Jun 29 02:55:18 2002 GMT by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2002-2004 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//======================================================================================

#include "kvi_module.h"
#include "kvi_uparser.h"
#include "kvi_command.h"
#include "kvi_regchan.h"
#include "kvi_locale.h"
#include "kvi_out.h"
#include "kvi_mirccntrl.h"
#include "kvi_window.h"

// kvi_app.cpp
extern KVIRC_API KviRegisteredChannelDataBase * g_pRegisteredChannelDataBase;

/*
	@doc: regchan.add
	@type:
		command
	@title:
		regchan.add
	@keyterms:
		registering channels
	@short:
		Registers a channel
	@syntax:
		regchan.add <channel> [netmask]
	@description:
		Registers the <channel> with the specified [netmask].
		[netmask] can be a full network name or a wildcard string
		tht will be matched against network names.
		For example, if you want to register #kvirc on IRCNet
		you can use regchan.add #kvirc IRCNet.
		If you want to handle #kvirc on all the IRC networks then
		you can use regchan.add #kvirc *[br]
		Obviously if you register both #kvirc/IRCNet and #kvirc/*
		then #kvirc/IRCNet will be looked first.
		If netmask is not passed a netmask of * is assumed.[br]
	@examples:
		[example]
			regchan.add #kvirc openprojects
		[/example]
	@seealso:
*/

static bool regchan_module_cmd_add(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"regchan_module_cmd_add");

	KviStr name,mask;
	if(!g_pUserParser->parseCmdSingleToken(c,name))return false;
	if(!g_pUserParser->parseCmdFinalPart(c,mask))return false;

	if(name.isEmpty())
	{
		c->warning(__tr("No name specified"));
		return c->leaveStackFrame();
	}

	if(mask.isEmpty())mask="*";

	g_pRegisteredChannelDataBase->add(new KviRegisteredChannel(name,mask));
	return c->leaveStackFrame();
}

/*
	@doc: regchan.remove
	@type:
		command
	@title:
		regchan.remove
	@keyterms:
		unregistering channels
	@short:
		Unregisters a channel
	@syntax:
		regchan.remove [-q] [-e] <channel> <network>
	@description:
		Unregisters the channel named <channel> that best matches <network>
		with its netmask.[br]
		If no match for <channel>/<network> is found in the database then
		a warning is printed unless -q is used.[br]
		If -e is specified then <network> is not matched against the
		netmasks in the database but is compared directly. This means
		that <network> must be exactly the netmask used to register a channel.
	@examples:
		[example]
			regchan.remove #kvirc openprojects
		[/example]
	@seealso:
		regchan.add
*/

static bool regchan_module_cmd_remove(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"regchan_module_cmd_remove");

	KviStr name,mask;
	if(!g_pUserParser->parseCmdSingleToken(c,name))return false;
	if(!g_pUserParser->parseCmdFinalPart(c,mask))return false;

	if(name.isEmpty())
	{
		c->warning(__tr("No name specified"));
		return c->leaveStackFrame();
	}

	if(mask.isEmpty())
	{
		c->warning(__tr("No netmask specified"));
		return c->leaveStackFrame();
	}

	KviRegisteredChannel * ch;
	
	if(c->hasSwitch('e'))ch = g_pRegisteredChannelDataBase->findExact(name,mask);
	else ch = g_pRegisteredChannelDataBase->find(name,mask);
	if(ch)
	{
		g_pRegisteredChannelDataBase->remove(ch);
	} else {
		if(!c->hasSwitch('q'))c->warning(__tr("No such channel/netmask entry in the database"));
	}
	return c->leaveStackFrame();
}



/*
	@doc: regchan.setproperty
	@type:
		command
	@title:
		regchan.setproperty
	@short:
		Sets a property for a registered channel
	@syntax:
		regchan.setproperty [-e] [-q] <channel> <network> <property> <value>
	@description:
		Sets the <property> for the registered <channel> that best
		matches <network> to <value>.[br]
		If <value> is an empty string the <property> is removed.[br]
		If no match for <channel>/<network> is found in the database then
		a warning is printed unless -q is used.[br]
		If -e is specified then <network> is not matched against the
		netmasks in the database but is compared directly. This means
		that <network> must be exactly the netmask used to register a channel.
	@examples:
		[example]
			regchan.setproperty #kvirc openprojects autojoin true
		[/example]
	@seealso:
		regchan.add
*/

static bool regchan_module_cmd_setproperty(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"regchan_module_cmd_setproperty");

	KviStr name,mask,prop,value;
	if(!g_pUserParser->parseCmdSingleToken(c,name))return false;
	if(!g_pUserParser->parseCmdSingleToken(c,mask))return false;
	if(!g_pUserParser->parseCmdSingleToken(c,prop))return false;
	if(!g_pUserParser->parseCmdFinalPart(c,value))return false;

	if(name.isEmpty())
	{
		c->warning(__tr("No name specified"));
		return c->leaveStackFrame();
	}

	if(mask.isEmpty())
	{
		c->warning(__tr("No netmask specified"));
		return c->leaveStackFrame();
	}

	if(prop.isEmpty())
	{
		c->warning(__tr("No property name specified"));
		return c->leaveStackFrame();
	}


	
	KviRegisteredChannel * ch;
	if(c->hasSwitch('e'))ch = g_pRegisteredChannelDataBase->findExact(name,mask);
	else ch = g_pRegisteredChannelDataBase->find(name,mask);
	if(ch)
	{
		if(value.hasData())
		{
			ch->setProperty(prop.ptr(),new KviStr(value));
		} else {
			ch->removeProperty(prop.ptr());
		}
	} else {
		if(!c->hasSwitch('q'))c->warning(__tr("No such channel/netmask entry in the database"));
	}
	return c->leaveStackFrame();
}



/*
	@doc: regchan.list
	@type:
		command
	@title:
		regchan.list
	@short:
		Lists the registered channels
	@syntax:
		regchan.list
	@description:
		Lists the registered channels with their description
*/

static bool regchan_module_cmd_list(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"regchan_module_cmd_list");

	KviStr dummy;
	if(!g_pUserParser->parseCmdFinalPart(c,dummy))return false;

	c->window()->output(KVI_OUT_SYSTEMMESSAGE,__tr("Registered channel list:"));

	int tot = 0;

	QAsciiDictIterator<KviRegisteredChannelList> it(*(g_pRegisteredChannelDataBase->channelDict()));
	while(KviRegisteredChannelList * l = it.current())
	{
		for(KviRegisteredChannel * ch = l->first();ch;ch = l->next())
		{
			tot++;
			c->window()->output(KVI_OUT_SYSTEMMESSAGE,__tr("Channel: %c%s@%s"),
				KVI_TEXT_BOLD,ch->name().ptr(),ch->netMask().ptr());
			QAsciiDictIterator<KviStr> pit(*(ch->propertyDict()));
			while(KviStr * s = pit.current())
			{
				c->window()->output(KVI_OUT_SYSTEMMESSAGE,__tr("   Property: %s=%s"),pit.currentKey(),s->ptr());
				++pit;
			}
		}
		++it;
	}

	c->window()->output(KVI_OUT_SYSTEMMESSAGE,__tr("Total %d channels"),tot);
	return c->leaveStackFrame();
}


/*
	@doc: regchan.property
	@type:
		function
	@title:
		$regchan.property
	@short:
		Returns a property of an entry in the registered channels database
	@syntax:
		$regchan.property(<channel_name>,<network>,<property_name>)
	@description:
		Returns the value of the property identified by <property_name> and
		bound to the channel identified by <channel_name> and best
		matched by <network>.[br]
		If the property is not set or if there is no matching channel entry an empty string is returned.[br]
	@examples:
		[example]
			echo $regchan.property(#kvirc,[fnc]$my.network[/fnc],autojoin)
		[/example]
*/

static bool regchan_module_fnc_property(KviModule *m,KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"regchan_module_fnc_property");
	const char * p1 = params->safeFirstParam();
	const char * p2 = params->safeNextParam();
	const char * p3 = params->safeNextParam();
	KviRegisteredChannel * ch = g_pRegisteredChannelDataBase->find(p1,p2);
	if(ch)
	{
		if(p3)
		{
			KviStr * p = ch->property(p3);
			if(p)buffer.append(p->ptr());
		} else c->warning(__tr("Missing property name"));
	} //else c->warning(__tr("User %s not found"),parms->safeFirstParam());
	return c->leaveStackFrame();
}


/*
	@doc: regchan.match
	@type:
		function
	@title:
		$regchan.match
	@short:
		Checks if a channel is registered
	@syntax:
		$regchan.match(<channel_name>,<network>)
	@description:
		Returns 1 if <channel_name> and <network> match a registered channel entry
		and 0 otherwise.
	@examples:
		[example]
			echo $regchan.match(#kvirc,IRCNet)
			echo $regchan.match(#kvirc,$my.network)
		[/example]
	@seealso:
		$regchan.find
*/

static bool regchan_module_fnc_match(KviModule *m,KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"regchan::match");
	const char * p1 = params->safeFirstParam();
	const char * p2 = params->safeNextParam();
	KviRegisteredChannel * ch = g_pRegisteredChannelDataBase->find(p1,p2);
	buffer.append(ch ? '1' : '0');
	return c->leaveStackFrame();
}

/*
	@doc: regchan.find
	@type:
		function
	@title:
		$regchan.find
	@short:
		Checks if a channel is registered
	@syntax:
		$regchan.find(<channel_name>,<netmask>)
	@description:
		Returns 1 if the channel identified by <channel_name> and <netmask>
		is registered and 0 otherwise. This function is similar to $regchan.match
		but for $regchan.find the second parameter is the exact netmask
		of the registered channel entry.
	@examples:
		[example]
			echo $regchan.find(#kvirc,IRCNet)
			echo $regchan.find(#kvirc,*)
		[/example]
	@seealso:
		$regchan.match
*/

static bool regchan_module_fnc_find(KviModule *m,KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"regchan::match");
	const char * p1 = params->safeFirstParam();
	const char * p2 = params->safeNextParam();
	KviRegisteredChannel * ch = g_pRegisteredChannelDataBase->find(p1,p2);
	buffer.append(ch ? '1' : '0');
	return c->leaveStackFrame();
}


static bool regchan_module_init(KviModule * m)
{
	m->registerCommand("add",regchan_module_cmd_add);
	m->registerCommand("list",regchan_module_cmd_list);
	m->registerCommand("remove",regchan_module_cmd_remove);
	m->registerCommand("setproperty",regchan_module_cmd_setproperty);
	m->registerFunction("property",regchan_module_fnc_property);
	m->registerFunction("match",regchan_module_fnc_match);
	m->registerFunction("find",regchan_module_fnc_find);
	return true;
}

static bool regchan_module_cleanup(KviModule *m)
{
	return true;
}

static bool regchan_module_can_unload(KviModule *)
{
	return true;
}

KVIRC_MODULE(
	"RegChan",                                              // module name
	"1.0.0",                                                // module version
	"Copyright (C) 2002 Szymon Stefanek (pragma at kvirc dot net)", // author & (C)
	"Script interface to the registered channels database",
	regchan_module_init,
	regchan_module_can_unload,
	0,
	regchan_module_cleanup
)
