//
//   File : libkvisharedfile.cpp (orig : libkvioffer.cpp)
//   Creation date : Wed Sep 27 2000 20:59:02 CEST by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2003 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "kvi_module.h"
#include "kvi_uparser.h"
#include "kvi_sharedfiles.h"
#include "kvi_ircmask.h"
#include "kvi_fileutils.h"
#include "kvi_locale.h"
#include "kvi_command.h"
#include "kvi_out.h"
#include "kvi_mirccntrl.h"
#include "kvi_window.h"
#include "kvi_frame.h"


#include <time.h>
#include <qasciidict.h>

extern KVIRC_API KviSharedFilesManager * g_pSharedFilesManager;


/*
	@doc: sharedfile.add
	@type:
		command
	@title:
		sharedfile.add
	@keyterms:
		trading files by dcc
	@short:
		Adds a file sharedfile
	@syntax:
		sharedfile.add [-t=<timeout>] [-n=<visible name>] <filename> [user_mask]
	@description:
		Adds <filename> to the list of the active shared files.
		The users will be able to request the file via [cmd]dcc.get[/cmd].
		If [user_mask] is specified , the access to the file is limited
		to the users that match this mask, otherwise the mask will
		be automatically set to '*!*@*'.[br]
		If the 't' switch is used, the sharedfile will be removed after
		<timeout> seconds. If the 'n' switch is used, the sharedfile
		will be visible to the oter users as <visible name> instead
		of the real <filename> (stripped of the leading path).
		<filename> must be an absolute path.
	@seealso:
		[cmd]sharedfile.remove[/cmd], [cmd]sharedfile.list[/cmd],
		[cmd]dcc.get[/cmd]
*/

static bool sharedfile_module_cmd_add(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"sharedfile::add");

	KviStr filename;
	KviStr usermask;
	if(!g_pUserParser->parseCmdSingleToken(c,filename))return false;
	if(!g_pUserParser->parseCmdFinalPart(c,usermask))return false;

	if(filename.isEmpty())
	{
		c->warning(__tr_ctx("No filename specified","sharedfile"));
		return c->leaveStackFrame();
	}

	if(!kvi_fileIsReadable(filename.ptr()))
	{
		c->warning(__tr_ctx("The file '%s' is not readable","sharedfile"),filename.ptr());
		return c->leaveStackFrame();
	}

	if(usermask.isEmpty())usermask="*!*@*";
	KviIrcMask u(usermask.ptr());
	QString szm;
	u.mask(szm);

	int timeout = 0;

	if(c->hasSwitch('t'))
	{
		KviStr tmo;
		c->getSwitchValue('t',tmo);
		bool bOk;
		timeout = tmo.toInt(&bOk);
		if(!bOk)
		{
			c->warning(__tr_ctx("Invalid timeout, ignoring","sharedfile"));
			timeout = 0;
		}
	}

	KviStr vName = filename;
	vName.cutToLast('/');

	if(c->hasSwitch('n'))
	{
		KviStr tmp;
		c->getSwitchValue('n',tmp);
		if(tmp.isEmpty())
		{
			c->warning(__tr_ctx("Invalid visible name: using default","sharedfile"));
		} else vName = tmp;
	}

	if(!g_pSharedFilesManager->addSharedFile(vName.ptr(),filename.ptr(),szm.latin1(),timeout))
	{
		c->warning(__tr_ctx("Ops..failed to add the sharedfile...","sharedfile"));
	}

	return c->leaveStackFrame();
}

/*
	@doc: sharedfile.remove
	@type:
		command
	@title:
		sharedfile.remove
	@keyterms:
		trading files by dcc
	@short:
		Removes a shared file
	@syntax:
		sharedfile.remove <visible name> <user mask> [filesize]
	@description:
		Removes the shared file that matches <visible name> and <user mask>.
		If [filesize] is specified, then it must be matched by the entry
		to be removed.
	@seealso:
		[cmd]sharedfile.add[/cmd], [cmd]sharedfile.list[/cmd], [cmd]sharedfile.clear[/cmd]
*/

static bool sharedfile_module_cmd_remove(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"sharedfile::remove");
	KviStr vName,uMask,fSize;
	if(!g_pUserParser->parseCmdSingleToken(c,vName))return false;
	if(!g_pUserParser->parseCmdSingleToken(c,uMask))return false;
	if(!g_pUserParser->parseCmdFinalPart(c,fSize))return false;
	unsigned int uSize = 0;
	if(fSize.hasData())
	{
		bool bOk;
		uSize = fSize.toUInt(&bOk);
		if(!bOk)
		{
			c->warning(__tr_ctx("The specified file size is invalid","sharedfile"));
			return c->leaveStackFrame();
		}
	}
	if(!g_pSharedFilesManager->removeSharedFile(vName.ptr(),uMask.ptr(),uSize))
		c->warning(__tr_ctx("No sharedfile with visible name '%s' and user mask '%s'","sharedfile"),vName.ptr(),uMask.ptr());
	return c->leaveStackFrame();
}

/*
	@doc: sharedfile.clear
	@type:
		command
	@title:
		sharedfile.clear
	@keyterms:
		trading files by dcc, shared files
	@short:
		Clears the shared files list
	@syntax:
		sharedfile.clear
	@description:
		Clears the shared files list
	@seealso:
		[cmd]sharedfile.add[/cmd], [cmd]sharedfile.list[/cmd], [cmd]sharedfile.remove[/cmd]
*/

static bool sharedfile_module_cmd_clear(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"sharedfile::clear");
	KviStr dummy;
	if(!g_pUserParser->parseCmdFinalPart(c,dummy))return false;
	g_pSharedFilesManager->clear();
	return c->leaveStackFrame();
}

/*
	@doc: sharedfile.list
	@type:
		command
	@title:
		sharedfile.list
	@keyterms:
		trading files by dcc
	@short:
		Lists the active file sharedfile
	@syntax:
		sharedfile.list
	@description:
		Lists the active file sharedfile.
	@seealso:
		[cmd]sharedfile.add[/cmd], [cmd]sharedfile.remove[/cmd]
*/

static bool sharedfile_module_cmd_list(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"sharedfile::list");
	KviStr dummy;
	if(!g_pUserParser->parseCmdFinalPart(c,dummy))return false;

	QDictIterator<KviSharedFileList> it(*(g_pSharedFilesManager->sharedFileListDict()));

	int idx = 0;

	while(KviSharedFileList * l = it.current())
	{
		for(KviSharedFile * o = l->first();o;o = l->next())
		{
			c->window()->output(KVI_OUT_NONE,"%c%d. %s",
				KVI_TEXT_BOLD,idx + 1,it.currentKey().latin1());
			c->window()->output(KVI_OUT_NONE,__tr2qs_ctx("    File: %s (%u bytes)","sharedfile"),
				o->absFilePath().latin1(),o->fileSize());
			c->window()->output(KVI_OUT_NONE,__tr2qs_ctx("    Mask: %s","sharedfile"),
				o->userMask().latin1());
			if(o->expireTime() > 0)
			{
				int secs = ((int)(o->expireTime())) - ((int)(time(0)));
				int hour = secs / 3600;
				secs = secs % 3600;
				int mins = secs / 60;
				secs = secs % 60;
				c->window()->output(KVI_OUT_NONE,__tr2qs_ctx("    Expires in %d hours %d minutes %d seconds","sharedfile"),
					hour,mins,secs);
			}
			++idx;
		}
		++it;
	}

//#warning "FIND A BETTER KVI_OUT_*"

	if(idx == 0)c->window()->outputNoFmt(KVI_OUT_NONE,__tr2qs_ctx("No active file sharedfile","sharedfile"));
	else c->window()->output(KVI_OUT_NONE,__tr2qs_ctx("Total: %d sharedfile","sharedfile"),idx);

	return c->leaveStackFrame();
}

static bool sharedfile_module_init(KviModule * m)
{
	m->registerCommand("add",sharedfile_module_cmd_add);
	m->registerCommand("remove",sharedfile_module_cmd_remove);
	m->registerCommand("list",sharedfile_module_cmd_list);
	m->registerCommand("clear",sharedfile_module_cmd_clear);
	return true;
}

static bool sharedfile_module_can_unload(KviModule *m)
{
	return true;
}

static bool sharedfile_module_cleanup(KviModule *m)
{
	return true;
}

KVIRC_MODULE(
	"Offer",                                                // module name
	"1.0.0",                                                // module version
	"Copyright (C) 2000-2003 Szymon Stefanek (pragma at kvirc dot net)", // author & (C)
	"User interface to the file sharing system",
	sharedfile_module_init,
	sharedfile_module_can_unload,
	0,
	sharedfile_module_cleanup
)
