//   File : libkvitmphighlight.cpp
//   Creation date :  Oct 10 01:06:09 CEST 2002 by Juanjo lvarez
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2002 Juanjo lvarez (juanjux@yahoo.es)
//   Copyright (C) 2002 Szymon Stefanek (kvirc@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//


#include "kvi_module.h"
#include "kvi_uparser.h"
#include "kvi_locale.h"
#include "kvi_channel.h"

//-------------------------------------------------
/*
    @doc: tmphighlight.add
    @type:
        command
    @title:
        tmphighlight.add
    @short:
        Adds a user to the channel temporary highlight list
    @syntax:
        tmphighlight.add <nick>
    @description:
        This command adds a user to the channel temporary highlight list, so that user messages[br]
        will be highlighted until you close the channel. This is useful when you are in a very crowded [br]
        channel with lots of conversations running in parallel and you want to follow one of them.[br]
        
    @seealso:
        [cmd]tmphighlight.remove[/cmd]
        [cmd]tmphighlight.ishighlighted[/cmd]
*/
//-------------------------------------------
// tmphighlight.add
//-------------------------------------------
static bool tmphighlight_module_cmd_add(KviModule *m,KviCommand *c)
{
 	ENTER_STACK_FRAME(c,"tmphighlight_module_cmd_add");
    KviStr nick;
    if(!g_pUserParser->parseCmdSingleToken(c,nick))return false;
    if(nick.isEmpty())
    {
        c->warning(__tr("No nickname specified"));
        return false;
    }
    
    if( ( !c->window()->console()) || c->window()->console()->isNotConnected() )return c->noIrcContext();
    if(!c->window()->type() == KVI_WINDOW_TYPE_CHANNEL)
    {
        c->warning(__tr("Current window is not a channel"));
        return false;
    }

    ((KviChannel *)c->window())->addHighlightedUser(nick.ptr());
	return c->leaveStackFrame();
}
//-------------------------------------------------
/*
    @doc: tmphighlight.remove
    @type:
        command
    @title:
        tmphighlight.remove
    @short:
        Remove a user from the channel temporary highlight list
    @syntax:
        tmphighlight.remove <nick>
    @description:
        This command remove a user from the channel temporary highlight list, so that user messages[br]
        stop being highlighted.
        
    @seealso:
        [cmd]tmphighlight.add[/cmd]
        [cmd]tmphighlight.ishighlighted[/cmd]
*/
//-------------------------------------------
// tmphighlight.remove
//-------------------------------------------

static bool tmphighlight_module_cmd_remove(KviModule *m,KviCommand *c)
{
 	ENTER_STACK_FRAME(c,"tmphighlight_module_cmd_remove");
    KviStr nick;
    if(!g_pUserParser->parseCmdSingleToken(c,nick))return false;
    if(nick.isEmpty())
    {
        c->warning(__tr("No nickname specified"));
        return false;
    }
    
    if( (!c->window()->console()) || c->window()->console()->isNotConnected() )return c->noIrcContext();
    if(!c->window()->type() == KVI_WINDOW_TYPE_CHANNEL)
    {
        c->warning(__tr("Current window is not a channel"));
        return false;
    }
    
    ((KviChannel *)c->window())->removeHighlightedUser(nick.ptr());
	return c->leaveStackFrame();
}
//-------------------------------------------------
/*
    @doc: tmphighlight.ishighlighted
    @type:
        function
    @title:
        $tmphighlight.ishighlighted
    @short:
        Returns 1 if the user is highlighted on this channel, 0 otherwise
    @syntax:
        $tmphighlight.ishighlighted <nick>
    @description:
        This command returns 1 if the user is highlighted on this channel and on this session of 0 otherwise.
        
    @seealso:
        [cmd]tmphighlight.add[/cmd]
        [cmd]tmphighlight.remove[/cmd]
        
*/
//-------------------------------------------
// tmphighlight.ishighlighted
//-------------------------------------------
static bool tmphighlight_module_fnc_ishighlighted(KviModule *m,KviCommand *c,KviParameterList *parms,KviStr &buffer)
{
 	ENTER_STACK_FRAME(c,"tmphighlight_module_fnc_ishighlighted");
    KviStr nick(parms->safeFirstParam());
    if(nick.isEmpty())
    {
        c->warning(__tr("No nickname specified"));
        return false;
    }
    if( (!c->window()->console()) || c->window()->console()->isNotConnected())return c->noIrcContext();
    if(!c->window()->type() == KVI_WINDOW_TYPE_CHANNEL)
    {
        c->warning(__tr("Current window is not a channel"));
        return false;
    }
    
    if(((KviChannel *)c->window())->isHighlightedUser(nick.ptr()))
    {
        buffer.append(KviStr::Format,"%d", 1);
    } else
    {
        buffer.append(KviStr::Format,"%d", 0);
    }
    
        
	return c->leaveStackFrame();
}

//-------------------------------------------------    
static bool tmphighlight_module_init(KviModule * m)
{
	m->registerCommand("add", tmphighlight_module_cmd_add);
    m->registerCommand("remove", tmphighlight_module_cmd_remove);
    m->registerFunction("ishighlighted", tmphighlight_module_fnc_ishighlighted);
	return true;
}
//-------------------------------------------------
static bool tmphighlight_module_cleanup(KviModule *m)
{
	return true;
}
//-------------------------------------------------
static bool tmphighlight_module_can_unload(KviModule *m)
{
    return true;
}
//-------------------------------------------------
KVIRC_MODULE(
	"TmpHighlight",                                                 // module name
	"1.0.0",                                                // module version
	"          (C) 2002 Juanjo Alvarez (juanjux@yahoo.es)", // author & (C)
	"Temporal Highlightining of channel users",
	tmphighlight_module_init,
	tmphighlight_module_can_unload,
	0,
	tmphighlight_module_cleanup
)
