// =============================================================================
//
//      --- kvi_irc_user.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviIrcUser"

#include "kvi_debug.h"
#include "kvi_irc_user.h"
#include "kvi_malloc.h"
#include "kvi_memmove.h"
#include "kvi_string.h"

KviIrcUser::KviIrcUser(const KviIrcUser &ircUser)
{
	m_nick_ptr = 0;
	m_user_ptr = 0;
	m_host_ptr = 0;
	setNick(ircUser.nick());
	setUsername(ircUser.username());
	setHost(ircUser.host());
}

KviIrcUser::KviIrcUser()
{
	m_nick_ptr = 0;
	m_user_ptr = 0;
	m_host_ptr = 0;
	setNick(0);
	setUsername(0);
	setHost(0);
}

KviIrcUser::KviIrcUser(const char *szMask, char c)
{
	m_nick_ptr = 0;
	m_user_ptr = 0;
	m_host_ptr = 0;
	setMask(szMask, c);
}

KviIrcUser::~KviIrcUser()
{
	kvi_free(m_nick_ptr);
	kvi_free(m_user_ptr);
	kvi_free(m_host_ptr);
}

void KviIrcUser::setNick(const char *szNick)
{
	if( szNick ) {
		if( !(*szNick) ) szNick = "*";
	} else szNick = "*";
	int len = strlen(szNick) + 1;
	m_nick_ptr = (char *) kvi_realloc(m_nick_ptr, len);
	kvi_memmove((void *) m_nick_ptr, (void *) szNick, len);
}

void KviIrcUser::setUsername(const char *szUsername)
{
	if( szUsername ) {
		if( !(*szUsername) ) szUsername = "*";
	} else szUsername = "*";
	int len = strlen(szUsername) + 1;
	m_user_ptr = (char *) kvi_realloc(m_user_ptr, len);
	kvi_memmove((void *) m_user_ptr, (void *) szUsername, len);
}

void KviIrcUser::setHost(const char *szHost)
{
	if( szHost ) {
		if( !(*szHost) ) szHost = "*";
	} else szHost = "*";
	int len = strlen(szHost) + 1;
	m_host_ptr = (char *) kvi_realloc(m_host_ptr, len);
	kvi_memmove((void *) m_host_ptr, (void *) szHost, len);
}

const char *KviIrcUser::setMask(const char *szMask, char c)
{
	__range_valid(szMask);
	// nick!username@host.top
	// 0123456789
	register const char *p = szMask;
	// Run over nick
	while( *p && (*p != '!') ) p++;
	int len = p - szMask;
	if( len > 0 ) {
		m_nick_ptr = (char *) kvi_realloc(m_nick_ptr, len + 1);
		kvi_memmove((void *) m_nick_ptr, (void *) szMask, len);
	} else { // Empty nick... set it to "*"
		len = 1;
		m_nick_ptr = (char *) kvi_realloc(m_nick_ptr, len + 1);
		kvi_memmove((void *) m_nick_ptr, (void *) "*", len);
	}
	*(m_nick_ptr + len) = '\0'; // With zero length nick it will be just an empty-string
	if( !(*p) ) {
		setHost("*");
		setUsername("*");
		return p;
	}
	szMask = ++p;
	// The username
	while( *p && (*p != '@') ) p++;
	len = p - szMask;
	if( len > 0 ) {
		m_user_ptr = (char *) kvi_realloc(m_user_ptr, len + 1);
		kvi_memmove((void *) m_user_ptr, (void *) szMask, len);
	} else {
		len = 1;
		m_user_ptr = (char *) kvi_realloc(m_user_ptr, len + 1);
		kvi_memmove((void *) m_user_ptr, (void *) "*", len);
	}
	*(m_user_ptr + len) = '\0';
	if( !(*p) ) {
		setHost("*");
		return p;
	}
	szMask = ++p;
	// And finally the host
	while( *p && (*p != c) ) p++;
	len = p - szMask;
	if( len > 0 ) {
		m_host_ptr = (char *) kvi_realloc(m_host_ptr, len + 1);
		kvi_memmove((void *) m_host_ptr, (void *) szMask, len);
	} else {
		len = 1;
		m_host_ptr = (char *) kvi_realloc(m_host_ptr, len + 1);
		kvi_memmove((void *) m_host_ptr, (void *) "*", len);
	}
	*(m_host_ptr + len) = '\0';
	return p;
}

const char *KviIrcUser::setUserhostMask(const char *szMask)
{
	__range_valid(szMask);
	// nick[*]=<+!->username@host.top
	// 0123456789
	register const char *p = szMask;
	// Run over nick
	while( *p && (*p != '*') && (*p != '=') && !isspace(*p) ) p++;
	// Extract it
	int len = p - szMask;
	if( len > 0 ) {
		m_nick_ptr = (char *) kvi_realloc(m_nick_ptr, len + 1);
		kvi_memmove((void *) m_nick_ptr, (void *) szMask, len);
	} else { // Empty nick... set it to "*"
		len = 1;
		m_nick_ptr = (char *) kvi_realloc(m_nick_ptr, len + 1);
		kvi_memmove((void *) m_nick_ptr, (void *) "*", len);
	}
	*(m_nick_ptr + len) = '\0'; // With zero length nick it will be just an empty-string
	// Now skip all the flags
	while(*p && ((*p == '*') || (*p == '=') || (*p == '+') || (*p == '-')) && !isspace(*p) ) p++;
	// Check...
	if( !(*p) || isspace(*p) ) {
		// Oops, finished or isspace
		setHost("*");
		setUsername("*");
		while( *p && isspace(*p) ) p++;
		return p;
	}

	szMask = p;
	// The username
	while( *p && (*p != '@') && !isspace(*p) ) p++;
	len = p - szMask;
	if( len > 0 ) {
		m_user_ptr = (char *) kvi_realloc(m_user_ptr, len + 1);
		kvi_memmove((void *) m_user_ptr, (void *) szMask, len);
	} else {
		len = 1;
		m_user_ptr = (char *) kvi_realloc(m_user_ptr, len + 1);
		kvi_memmove((void *) m_user_ptr, (void *) "*", len);
	}
	*(m_user_ptr + len) = '\0';

	if( !(*p) || isspace(*p) ) {
		// Oops finished or isspace
		setHost("*");
		while( *p && isspace(*p) ) p++;
		return p;
	}
	szMask = ++p;
	// And finally the host
	while( *p && isspace(*p) ) p++;
	len = p - szMask;
	if( len > 0 ) {
		m_host_ptr = (char *) kvi_realloc(m_host_ptr, len + 1);
		kvi_memmove((void *) m_host_ptr, (void *) szMask, len);
	} else {
		len = 1;
		m_host_ptr = (char *) kvi_realloc(m_host_ptr, len + 1);
		kvi_memmove((void *) m_host_ptr, (void *) "*", len);
	}
	*(m_host_ptr + len) = '\0';
	while( *p && isspace(*p) ) p++;
	return p;
}

bool KviIrcUser::hasNumericHost() const
{
	register char *p = m_host_ptr;
	int nPoints = 0;
	while( *p ) {
		if( *p == '.' )
			nPoints++;
		else {
			if( (*p < '0') || (*p > '9') ) return false;
		}
		p++;
	}
	return (nPoints == 3);
}

/**
 * Returns in szMask the specified (if possible) mask of this user.<br>
 * If the host or username are not known, the mask may contain less information
 * than requested.<br>
 * Mask types:<br>
 * 0 : nick!user@machine.host.top  (nick!user@XXX.XXX.XXX.XXX) (default)<br>
 * 1 : nick!user@*.host.top        (nick!user@XXX.XXX.XXX.*)<br>
 * 2 : nick!user@*<br>
 * 3 : nick!*@machine.host.top     (nick!user@XXX.XXX.XXX.XXX)<br>
 * 4 : nick!*@*.host.top           (nick!user@XXX.XXX.XXX.*)<br>
 * 5 : nick!*@*<br>
 * 6 : *!user@machine.host.top     (*!user@XXX.XXX.XXX.XX)<br>
 * 7 : *!user@*.host.top           (*!user@XXX.XXX.XXX.*)<br>
 * 8 : *!user@*<br>
 * 9 : *!*@machine.host.top        (*!*@XXX.XXX.XXX.XXX)<br>
 * 10: *!*@*.host.top              (*!*@XXX.XXX.XXX.*)<br>
 * 11: nick!*user@machine.host.top (nick!*user@machine.host.top)<br>
 * 12: nick!*user@*.host.top       (nick!*user@*.host.top)<br>
 * 13: nick!*user@*<br>
 * 14: *!*user@machine.host.top    (*!*user@machine.host.top)<br>
 * 15: *!*user@*.host.top          (*!*user@*.host.top)<br>
 * 16: *!*user@*<br>
 * If some data is missing, these types may change:<br>
 * For example, if hostname is missing, then mask type 3 or 4 may be reduced to type 5
 */
static unsigned char maskTable[17][3] =
{
	{ 0, 0, 0 }, // 0 means normal block
	{ 0, 0, 2 }, // 2 in the third field means type *.host.top (or XXX.XXX.XXX.*) host mask
	{ 0, 0, 1 }, // 2 in the second field means *user (strip prefixes)
	{ 0, 1, 0 }, // 1 means *
	{ 0, 1, 2 },
	{ 0, 1, 1 },
	{ 1, 0, 0 },
	{ 1, 0, 2 },
	{ 1, 0, 1 },
	{ 1, 1, 0 },
	{ 1, 1, 2 },
	{ 0, 2, 0 },
	{ 0, 2, 2 },
	{ 0, 2, 1 },
	{ 1, 2, 0 },
	{ 1, 2, 2 },
	{ 1, 2, 1 }
};
void KviIrcUser::mask(KviStr &szMask, int mask_type)
{
	if( (mask_type > 16) || (mask_type < 0) ) mask_type = 0;
	szMask = maskTable[mask_type][0] ? "*" : m_nick_ptr;
	szMask.append("!");
	switch(maskTable[mask_type][1]){
		case 0:
			szMask.append(m_user_ptr);
			break;
		case 1:
			szMask.append('*');
			break;
		default:
			{
				szMask.append('*');
				/**
				 * ident is fun... ahem
				 * Prefixes used:
				 *	none   I line with ident
				 *	^      I line with OTHER type ident
				 *	~      I line, no ident
				 *	+      i line with ident
				 *	=      i line with OTHER type ident
				 *	-      i line, no ident
				 */
				if( ((*m_user_ptr) == '~') || ((*m_user_ptr) == '^') ||
					((*m_user_ptr) == '+') || ((*m_user_ptr) == '-') ||
					((*m_user_ptr) == '=') || ((*m_user_ptr) == '*'))
					szMask.append((m_user_ptr + 1));
				else
					szMask.append(m_user_ptr);
			}
			break;
	}
	szMask.append('@');
	switch( maskTable[mask_type][2] ) {
		case 0:
			szMask.append(m_host_ptr);
			break;
		case 1:
			szMask.append('*');
			break;
		default:
			if( hasHost() ) {
				if( hasNumericHost() ) {
					KviStr szHost(m_host_ptr, getIpDomainMaskLen());
					szMask.append(szHost.ptr());
					szMask.append("*");
				} else {
					szMask.append("*");
					szMask.append(getHostDomainMaskPtr());
				}
			} else szMask.append("*");
			break;
	}
}

bool KviIrcUser::matches(const char *szMask)
{
	KviStr str;
	mask(str, 0);
	return kvi_matchWildExpr(str.ptr(), szMask);
}

// ===========================================================================
// Internals for mask()
// ===========================================================================
int KviIrcUser::getIpDomainMaskLen()
{
	register char *p = m_host_ptr;
	// Run to the end
	while( *(++p) );
	p--;
	while( (m_host_ptr != p) && (*p != '.') ) p--;
	// 000.000.000.000
	//            p
	//
	return (p == m_host_ptr) ? 0 : ((p-m_host_ptr) + 1);
}

char *KviIrcUser::getHostDomainMaskPtr()
{
	register char *p = m_host_ptr;
	while( *p && *p != '.' ) p++;
	return p;
}
