// =============================================================================
//
//      --- kvi_expression.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviExpression"

#include "kvi_command.h"
#include "kvi_debug.h"
#include "kvi_error.h"
#include "kvi_exprtree.h"
#include "kvi_locale.h"
#include "kvi_userparser.h"

bool KviUserParser::evaluateExpr(KviCommand *cmd, long *result)
{
	// This one evaluates an expression
	KviExprTree *root = evaluateExpression(cmd);
	if( !root ) return false;

	if( *(cmd->m_ptr) != ')' ) {
		cmd->setError(KVI_ERROR_ParenthesisMismatch, _i18n_("EXPRESSION-END"), cmd->m_ptr);
		return false;
	}
	*result = root->calculate(cmd);
	delete root;
	++(cmd->m_ptr);
	return (!cmd->hasError());
}

KviExprTree *KviUserParser::evaluateExpression(KviCommand *cmd)
{
	// This one evaluates the expression up to a (not matching) ')' or a \0
	KviExprTree *op = evaluateOperand(cmd);
	if( !op ) return 0;

	cmd->skipSpace();
	if( (*(cmd->m_ptr) == ')') || (*(cmd->m_ptr) == '\0') )
		return op; // Single operand
	KviExprTree *curOper = evaluateOperator(cmd);
	if( !curOper ) {
		delete op;
		return 0;
	}
	curOper->setLeft(op);
	return evaluateExpression_RightOperand(curOper, cmd);
}

KviExprTree *KviUserParser::evaluateOperand(KviCommand *cmd)
{
	// This one tries to extract a numeric operand from the data buffer.
	// Moves m_ptr to the character after the last char of the operand.
	// Returns a valid KviExprTreeNode of successful,
	// otherwise returns 0 and sets cmd->m_err appropriately.
	cmd->skipSpace();

	KviExprTree *pTop  = 0;
	KviExprTree *pCur  = 0;
	KviExprTree *pLast = 0;

	// Check for unary ops...
	while( *(cmd->m_ptr) == '-' || *(cmd->m_ptr) == '~' || *(cmd->m_ptr) == '!' ) {
		// Unary op
		switch( *(cmd->m_ptr) ) {
			case '-':
				pCur = new KviExprTree(KviExprTree::UnaryOperator, OP_NEGATE);
				break;
			case '~':
				pCur = new KviExprTree(KviExprTree::UnaryOperator, OP_BITNOT);
				break;
			case '!':
				pCur = new KviExprTree(KviExprTree::UnaryOperator, OP_NOT);
				break;
		}
		// And we work with the child
		if( !pTop )
			pTop = pCur;
		else
			pLast->setLeft(pCur);
		pLast = pCur;
		++(cmd->m_ptr);
		cmd->skipSpace();
	}

	if( *(cmd->m_ptr) == '(' ) {
		// Subexpression
		++(cmd->m_ptr);
		pCur = evaluateExpression(cmd);
		if( !pCur ) { // Failed somewhere in the call
			if( pTop )
				delete pTop;
			return 0;
		}
		if( *(cmd->m_ptr) != ')' ) { // Failed here
			if( pTop )
				delete pTop;
			delete pCur;
			cmd->setError(KVI_ERROR_ParenthesisMismatch, _i18n_("SUBEXPRESSION"));
			return 0;
		}
		++(cmd->m_ptr);
	} else {
		// Here *m_ptr may be something that is not a digit too!
		pCur = evaluateSimpleOperand(cmd);
		if( !pCur ) { // Failed somewere
			if( pTop )
				delete pTop;
			return 0;
		}
	}
	if( pTop )
		pLast->setLeft(pCur);
	else
		return pCur;
	return pTop;
}

KviExprTree *KviUserParser::evaluateStringOperand(KviCommand *cmd)
{
	__range_valid(*(cmd->m_ptr) == '"');
	KviExprTree *node = new KviExprTree(KviExprTree::StringOperand, 0);
	const char *aux_ptr = ++(cmd->m_ptr);

	for( ;; ) {
		// Now skip all the uninteresting chars
		while( *aux_ptr && (*aux_ptr != '\\') &&
		      (*aux_ptr != '\n') && (*aux_ptr != '"') &&
			  (*aux_ptr != '$')  && (*aux_ptr != KVI_GLOBAL_VAR_CHAR)
		) {
			++aux_ptr;
		}
		// Interesting char
		switch( *aux_ptr ) {
			case '\n':
			case '\0': // End of command buffer... append the last block to the buffer and return
				cmd->setError(KVI_ERROR_UnexpectedEndInString, _i18n_("EXPRESSION-STRING-OPERAND"));
				delete node;
				return 0;
				break;
			case '"' : // Append the last block to the buffer and return
				node->m_string.append(cmd->m_ptr, aux_ptr - cmd->m_ptr);
				cmd->m_ptr = ++aux_ptr; // Do not forget to point to next char
				return node;
				break;
			case KVI_GLOBAL_VAR_CHAR: // Variable: append the last block to the buffer and process the var
				node->m_string.append(cmd->m_ptr, aux_ptr - cmd->m_ptr);
				cmd->m_ptr = aux_ptr;
				if( !processVariable(cmd, node->m_string) ) {
					delete node;
					return 0;
				}
				aux_ptr = cmd->m_ptr;
				break;
			case '$': // System identifier: append the last block to the buffer and process the ident
				node->m_string.append(cmd->m_ptr, aux_ptr - cmd->m_ptr);
				cmd->m_ptr = aux_ptr;
				if( !processIdentifier(cmd, node->m_string) ) {
					delete node;
					return 0;
				}
				aux_ptr = cmd->m_ptr;
				break;
			case '\\': // Escape character: append the last block to the processed buffer...
				node->m_string.append(cmd->m_ptr, aux_ptr - cmd->m_ptr);
				cmd->m_ptr = ++aux_ptr;
				switch( *aux_ptr ) {
					case '\0':
						cmd->setError(KVI_ERROR_UnexpectedEndInString, _i18n_("EXPRESSION-STRING-OPERAND"));
						delete node;
						return 0;
						break;
					case '\n':  // Escaped newline
						cmd->m_ptr = ++aux_ptr; // Skip it
						cmd->skipSpace();       // Skip leading spaces
						aux_ptr = cmd->m_ptr;   // Continue
						break;
					default: // Must be copied to the buffer without modification
						node->m_string.append(*aux_ptr);
						cmd->m_ptr = ++aux_ptr;
						break;
				}
			break;
		}
	}
	return 0;
}

KviExprTree *KviUserParser::evaluateSimpleOperand(KviCommand *cmd)
{
	// This one extracts a number from the data buffer
	KviStr tempNum;
	const char *aux =  cmd->m_ptr;
	switch( *aux ) {
		case '%': if( !processVariable(cmd, tempNum)   ) return 0; break;
		case '$': if( !processIdentifier(cmd, tempNum) ) return 0; break;
		case '"': return evaluateStringOperand(cmd); break;
		case '0':
			++(cmd->m_ptr);
			if( !(isalpha(*(cmd->m_ptr)) || isxdigit(*(cmd->m_ptr))) )
				tempNum = "0";
			else {
				++(cmd->m_ptr);
				if( isalpha(*(cmd->m_ptr)) || isxdigit(*(cmd->m_ptr)) )
					++(cmd->m_ptr);
				while( isxdigit(*(cmd->m_ptr)) )
					++(cmd->m_ptr);
				tempNum.setStr(aux, (cmd->m_ptr) - aux);
			}
			break;
		default:
			if( !isxdigit(*(cmd->m_ptr)) ) {
				cmd->setError(KVI_ERROR_UnexpectedCharacter, _i18n_("EXPRESSION-OPERAND"), cmd->m_ptr);
				return 0;
			}
			while( isxdigit(*(cmd->m_ptr)) )
				++(cmd->m_ptr);
			tempNum.setStr(aux, (cmd->m_ptr) - aux);
			break;
	}

	aux = tempNum.ptr();
	int base = 10;
	if( *aux == '0' ) {
		++aux;
		switch( tolower(*aux) ) {
			case 'x': tempNum.cutLeft(2); base = 16; break;
			case 'h': tempNum.cutLeft(2); base = 16; break;

			case 'i': tempNum.cutLeft(2); base =  2; break; // Cannot use b because it is a hex digit
			case 'j': tempNum.cutLeft(2); base =  3; break;
			case 'k': tempNum.cutLeft(2); base =  4; break;
			case 'l': tempNum.cutLeft(2); base =  5; break;
			case 'm': tempNum.cutLeft(2); base =  6; break;
			case 'n': tempNum.cutLeft(2); base =  7; break;
			case 'o': tempNum.cutLeft(2); base =  8; break;
			case 'p': tempNum.cutLeft(2); base =  9; break;
			case 'q': tempNum.cutLeft(2); base = 10; break;
			case 'r': tempNum.cutLeft(2); base = 11; break;
			case 's': tempNum.cutLeft(2); base = 12; break;
			case 't': tempNum.cutLeft(2); base = 13; break;
			case 'u': tempNum.cutLeft(2); base = 14; break;
			case 'v': tempNum.cutLeft(2); base = 15; break;
			default: break; // Unknown... maybe a single 0
		}
	}
	bool bOk;
	long res = tempNum.toLongExt(&bOk, base);
	if( !bOk ) {
		cmd->setError(KVI_ERROR_UnexpectedCharacter, _i18n_("EXPRESSION-OPERAND"), tempNum.ptr());
		return 0;
	}
	return new KviExprTree(KviExprTree::NumericOperand, res);
}

KviExprTree *KviUserParser::evaluateExpression_RightOperand(KviExprTree *curTopOperator, KviCommand *cmd)
{
	__range_valid(curTopOperator);
	// Now curTopOperator has the left subtree (one node) set
	// and it points to the TOP (=ROOT) node.
	// Evaluate the rest.
	KviExprTree *operand;
	KviExprTree *incompleteOperator = curTopOperator;
	KviExprTree *auxOperator;

	for( ;; ) {
		operand = evaluateOperand(cmd);
		if( !operand ) {
			delete curTopOperator;
			return 0;
		}
		cmd->skipSpace();
		if( *(cmd->m_ptr) == ')' || *(cmd->m_ptr) == '\0' ) {
			incompleteOperator->setRight(operand);
			return curTopOperator;
		}

		auxOperator = evaluateOperator(cmd);
		if( !auxOperator ) {
			delete curTopOperator;
			delete operand;
			return 0;
		}

		// Now compare operators...
		if( incompleteOperator->firstBinaryOperator() >= auxOperator->firstBinaryOperator() ) {
			incompleteOperator->setRight(auxOperator);
			auxOperator->setLeft(operand);
		} else {
			// Precedence of operators...
			incompleteOperator->setRight(operand); // Right tree complete
			// Go up until we find an operator with lower precedence than auxOperator (>=)
			KviExprTree *tempOperator = incompleteOperator->parentWithPrecedenceLowerThan(auxOperator->m_value);
			if( !tempOperator ) { // Reached the top
				auxOperator->setLeft(curTopOperator);
				curTopOperator = auxOperator;
			} else {
				__range_valid(tempOperator->right());
				auxOperator->setLeft(tempOperator->right());
				tempOperator->setRight(auxOperator);
			}
		}
		incompleteOperator = auxOperator;
		__range_valid(incompleteOperator->right() == 0);
	}
	return 0;
}

KviExprTree *KviUserParser::evaluateOperator(KviCommand *cmd)
{
	// This one extracts the first operator it finds
	cmd->skipSpace();
	long op;
	switch( *(cmd->m_ptr) ) {
		// Single operators
		case '^': op = OP_POWER; ++(cmd->m_ptr); break;
		case '*': op = OP_MUL;   ++(cmd->m_ptr); break;
		case '/': op = OP_DIV;   ++(cmd->m_ptr); break;
		case '%': op = OP_MOD;   ++(cmd->m_ptr); break;
		case '+': op = OP_ADD;   ++(cmd->m_ptr); break;
		case '-': op = OP_SUB;   ++(cmd->m_ptr); break;
		case '@':
			++(cmd->m_ptr);
			if( *(cmd->m_ptr) == '=' ) {
				op = OP_CSEQ;
				++(cmd->m_ptr);
			} else {
				cmd->setError(KVI_ERROR_UnknownOperator, _i18n_("EXPRESSION-OPERATOR"), cmd->m_ptr);
				return 0;
			}
			break;
		case '=':
			++(cmd->m_ptr);
			if( *(cmd->m_ptr) == '=' ) {
				op = OP_EQ;
				++(cmd->m_ptr);
			} else {
				cmd->setError(KVI_ERROR_UnknownOperator, _i18n_("EXPRESSION-OPERATOR"), cmd->m_ptr);
				return 0;
			}
			break;
		case '!':
			++(cmd->m_ptr);
			if( *(cmd->m_ptr) == '=' ) {
				op = OP_NE;
				++(cmd->m_ptr);
			} else {
				cmd->setError(KVI_ERROR_UnknownOperator, _i18n_("EXPRESSION-OPERATOR"), cmd->m_ptr);
				return 0;
			}
			break;
		case '>':
			++(cmd->m_ptr);
			if( *(cmd->m_ptr) == '=' ) {
				op = OP_GE;
				++(cmd->m_ptr);
			} else if( *(cmd->m_ptr) == '>' ) {
				op = OP_SHR;
				++(cmd->m_ptr);
			} else op = OP_GT;
			break;
		case '<':
			++(cmd->m_ptr);
			if( *(cmd->m_ptr) == '=' ) {
				op = OP_LE;
				++(cmd->m_ptr);
			} else if( *(cmd->m_ptr) == '<' ) {
				op = OP_SHL;
				++(cmd->m_ptr);
			} else op = OP_LT;
			break;
		case '&':
			++(cmd->m_ptr);
			if( *(cmd->m_ptr) == '&' ) {
				op = OP_AND;
				++(cmd->m_ptr);
			} else op = OP_BITAND;
			break;
		case '|':
			++(cmd->m_ptr);
			if( *(cmd->m_ptr) == '|' ) {
				op = OP_OR;
				++(cmd->m_ptr);
			} else op = OP_BITOR;
			break;
		default:
			cmd->setError(KVI_ERROR_UnknownOperator, _i18n_("EXPRESSION-OPERATOR"), cmd->m_ptr);
			return 0;
			break;
	}

	return new KviExprTree(KviExprTree::BinaryOperator, op);
}
