#ifndef _KVI_IRC_VIEW_H_INCLUDED_
#define _KVI_IRC_VIEW_H_INCLUDED_

// =============================================================================
//
//      --- kvi_irc_view.h ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#include <qptrlist.h>
#include <qwidget.h>

#include "kvi_string.h"

class QFile;
class QPainter;
class QPixmap;
class QScrollBar;

class KviWindow;
class KviFrame;

#define _KVI_PACKED_ __attribute__((__packed__))

typedef struct _KVI_PACKED_ KviIrcViewTextAttribute
{
	int   block_idx;                          // Index in the data_ptr of the beginning of the block.
	int   block_len;                          // Length in chars of the block (excluding the terminator).
	char  attribute;                          // Attribute type.
	char  attr_fore;                          // Optional foreground color for KVI_TEXT_COLOR attribute.
	char  attr_back;                          // Optional background color for KVI_TEXT_COLOR attribute.
	char *escape_cmd;                         // KVI_TEXT_ESCAPE attribute command buffer (non zeroed for other attributes!)
};

typedef struct _KVI_PACKED_ KviIrcViewTextBlock
{
	char                    *block_ptr;       // Pointer to the character that begins the block.
	int                      block_len;       // Length of the block in characters.
	int                      block_width;     // Width of the block in pixels.
	KviIrcViewTextAttribute *attr_ptr;        // Pointer to attributes or 0 for word wraps.
};

typedef struct _KVI_PACKED_ _KviIrcViewTextLine
{
	char                    *data_ptr;        // Pointer to the "string".
	int                      data_len;        // Length of the allocated buffer (including the terminator).
	int                      msg_type;
	int                      attr_len;        // Number of allocated attr structures.
	KviIrcViewTextAttribute *attr_ptr;        // Pointer to the allocated structures.

	/** Precalculated text blocks */
	int                      line_wraps;
	int                      max_line_width;  // Width that the blocks were calculated for.
	int                      num_text_blocks; // Number of allocated text blocks.
	KviIrcViewTextBlock     *text_blocks_ptr; // Pointer to the allocated text blocks.

	_KviIrcViewTextLine     *prev_line;
	_KviIrcViewTextLine     *next_line;
} KviIrcViewTextLine;

typedef struct KviIrcViewTextBlockSelectionInfo
{
	int selection_type;
	int part_1_length;
	int part_1_width;
	int part_2_length;
	int part_2_width;
	int part_3_length;
	int part_3_width;
};

/**
 * IrcView class. Do not take it as a good programming example.
 */
class KviIrcView : public QWidget
{
	Q_OBJECT
public:
	KviIrcView(QWidget *parent, KviFrame *pFrm, KviWindow *pWnd);
	~KviIrcView();
private:
	KviIrcViewTextLine  *m_pFirstLine;
	KviIrcViewTextLine  *m_pCurLine;    // Bottom line in the view
	KviIrcViewTextLine  *m_pLastLine;

	// Highlighting of links
	KviIrcViewTextBlock *m_pLastLinkUnderMouse;
	int                  m_iLastLinkRectTop;
	int                  m_iLastLinkRectHeight;

	int                  m_iNumLines;
	int                  m_iMaxLines;
	QPixmap             *m_pPrivateBackgroundPixmap;
	QScrollBar          *m_pScrollBar;
	int                  m_iLastScrollBarValue;
	int                  m_iFontLineSpacing;
	int                  m_iFontLineWidth;
	int                  m_iFontDescent;
	int                  m_iFontCharacterWidth[256];    // 256 bytes fixed
	int                  m_iWrapMargin;
	int                  m_iMouseButtonPressX;
	int                  m_iMouseButtonPressY;
	int                  m_iMouseButtonCurrentX;
	int                  m_iMouseButtonCurrentY;
	int                  m_iSelectionBegin;
	int                  m_iSelectionTop;
	int                  m_iSelectionEnd;
	int                  m_iSelectionBottom;
	int                  m_iSelectionLeft;
	int                  m_iSelectionRight;
	bool                 m_bSelecting;
	bool                 m_bDoubleClicked;
	bool                 m_bSkipScrollBarRepaint;
	bool                 m_bNeedScrollBar;
	bool                 m_bShowImages;
	int                  m_iSelectTimer;
	bool                 m_bTimestamp;
	KviStr               m_szLastSelection;
	KviStr               m_szLastStrippedSelection;
	KviStr               m_szLastSelectionLine;
	KviStr               m_szLastStrippedSelectionLine;
	KviStr               m_szLastDoubleClickedEscape;
	KviStr               m_szLastDoubleClickedEscapeCmd;
	int                  m_iLastSelectionLineLen;
	int                  m_iLastStrippedSelectionLineLen;
	KviWindow           *m_pKviWindow;

	QFile               *m_pLogFile;
	KviFrame            *m_pFrm;
	bool                 m_bAcceptDrops;
	int                  m_iUnprocessedPaintEventRequests;
	bool                 m_bPostedPaintEventPending;
	bool                 m_bBeeping;
	int                  m_iMinimumPaintWidth;
	int                  m_iRelativePixmapY;

	KviIrcViewTextBlockSelectionInfo  m_TextBlockSelectionInfo;
	QPtrList<KviIrcViewTextLine>     *m_pMessagesStoppedWhileSelecting;
public:
	/**
	 * The most useful API is this one...
	 */
	void enableDnD(bool bEnable);
	void appendText(int msg_type, const char *data_ptr, bool bRepaint = true);
	void removeHeadLine(bool bRepaint = false);
	void emptyBuffer(bool bRepaint = true);
	void getTextBuffer(KviStr &buffer);
	void setMaxBufferSize(int maxBufSize, bool bRepaint = true);
	void setShowImages(bool bShow, bool bRepaint = true);
	void setTimestamp(bool bTimestamp);

	bool timestamp();
	bool imagesVisible();

	bool saveBuffer(const char *filename);
	void findNext(const char *text);
	void findPrev(const char *text);
	KviWindow *parentKviWindow();

	/**
	 * A null pixmap passed here unsets the private background.
	 */
	void setPrivateBackgroundPixmap(const QPixmap &pixmap, bool bRepaint = true);

	bool hasPrivateBackgroundPixmap();

	/**
	 * Stops previous logging session too...
	 */
	bool startLogging(const char *filename);
	void add2Log(const char *buffer, int buf_len = -1);
	void stopLogging();
	bool isLogging();

	bool  initBeeping(const char *user);
	void startBeeping(const char *user);
	void  stopBeeping(const char *user);
	bool isBeeping();

	void prevLine();
	void nextLine();
	void nextPage();
	void prevPage();

	virtual void wheelEvent(QWheelEvent *);
public slots:
	void toggleBeeping();
	void toggleLogging();
	void toggleTimestamp();
	void toggleImages();
	void clearBuffer();
	void logToFile();
	void saveBufferToFile();
signals:
	/**
	 * The user right clicked on the view; what we do with that?
	 */
	void contextPopupRequested(KviIrcView *this_ptr);

	void dndEntered();
	void fileDropped(const char *);
private:
	void getDoubleClickCommand(KviStr &buffer, const char *escape_cmd);
	void  getRightClickCommand(KviStr &buffer, const char *escape_cmd);

	void resizeMemBuffer();
	void appendLine(KviIrcViewTextLine *ptr, bool bRepaint);
	void postUpdateEvent();
	void fastScroll(int lines = 1);

	const char *getTextLine(int msg_type, const char *data_ptr, KviIrcViewTextLine *line_ptr);

	void calculateLineWraps(KviIrcViewTextLine *ptr, int maxWidth);
	void recalcFontVariables(const QFont &font);

	bool              checkSelectionBlock(KviIrcViewTextLine *line, int left, int bottom, int bufIndex);
	void checkForDoubleClickedUrlOrEscape(KviIrcViewTextLine *line, int left, int bottom, int bufIndex);

	void calculateSelectionBounds();
	void drawBlockPart(
		QPainter *pa, int sb_width, char fg, char bg, bool bold, bool under,
		int &left, int bottom, const char *data, int len, int wdth
	);
	void drawSelectedText(
		QPainter *pa, int sb_width, char attr, char fg, char bg,
		int &left, int bottom, const char *data, int len, int wdth
	);

	KviIrcViewTextBlock *getLinkUnderMouse(int xPos, int yPos, int *rectTop, int *rectHeight);
protected:
	virtual void            paintEvent(QPaintEvent *);
	virtual void           resizeEvent(QResizeEvent *);
	virtual void       mousePressEvent(QMouseEvent *);
	virtual void     mouseReleaseEvent(QMouseEvent *);
	virtual void mouseDoubleClickEvent(QMouseEvent *);
	virtual void        mouseMoveEvent(QMouseEvent *);
	virtual void            timerEvent(QTimerEvent *);
	virtual void        dragEnterEvent(QDragEnterEvent *);
	virtual void             dropEvent(QDropEvent *);
	virtual bool                 event(QEvent *);
	virtual void fontChange(const QFont &oldFont);

	/**
	 * Sets whether the scrollbar is needed. Returns true if the status of
	 * the scrollbar has changed from before the call
	 */
	bool setNeedScrollBar(bool);
protected slots:
	virtual void scrollBarPositionChanged(int newValue);
};

#endif // _KVI_IRC_VIEW_H_INCLUDED_
