// =============================================================================
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviScriptFile"

#include <qfile.h>

#include "kvi_error.h"
#include "kvi_malloc.h"
#include "kvi_script_file.h"
#include "kvi_script_objectclassdefinition.h"

/*
	@class: file
	@short:
		A file interface object
	@inherits:
		[class]object[/class]<br>

	@functions:

		!fn: $setFileName(&lt;filename&gt;)
		Sets the filename of this file object.<br>
		If a file was previously open, it is closed first.<br>

		!fn: $fileName()
		Returns the filename for this file object.<br>

		!fn: $exists()
		Returns '1' if the file exists, '0' otherwise.<br>
		A filename must have been set previously.<br>

		!fn: $remove()
		Removes the file.<br>
		A filename must have been set previously.<br>
		Returns '1' if successful, '0' otherwise.<br>

		!fn: $size()
		Returns the size of the file.<br>
		A filename must have been set previously.<br>

		!fn: $open(&lt;mode&gt;)
		Opens the file in the specified mode.<br>
		&lt;mode&gt; can be any combination of the following flags:<br>
		<b>r</b> - Read access<br>
		<b>w</b> - Write access<br>
		<b>a</b> - Append (seek to end of the file)<br>
		<b>t</b> - Truncate the file to zero size<br>
		When the 'w' flag is specified and the file does not exist, it is created.<br>
		When ONLY the 'w' flag is specified, the file is truncated to zero size.<br>
		A filename must be set before this function can be called.<br>
		Returns '1' if successful, '0' otherwise.<br>

		!fn: $close()
		Closes this file object.<br>

		!fn: $flush()
		Flushes the cached file buffers to disk.<br>
		(The access is buffered to increase the performance).<br>
		[classfnc:file]$close[/classfnc]() calls it automatically, so
		you should not need this function.<br>

		!fn: $read(&lt;numBytes&gt;)
		Reads at most &lt;numBytes&gt; from the file and returns the string.<br>
		This function is not able to process NULL characters: you will be losing
		data if the file is not a text one.<br>

		!fn: $readLine(&lt;maxNumBytes&gt;)
		Reads a line of text from the file.<br>
		This function will read at most &lt;numBytes&gt;.<br>
		This function is not able to process NULL characters: you will be losing
		data if the file is not a text one.<br>

		!fn: $readHex(&lt;numBytes&gt;)
		Reads at most &lt;numBytes&gt; from the file and returns the hexadecimal
		encoding of the string (See [fnc]$strToHex[/fnc]() and [fnc]$hexToStr[/fnc]()).<br>

		!fn: $write(&lt;string_data&gt;)
		Writes the &lt;string_data&gt; to disk.<br>
		Returns the number of bytes written or '-1' in case of an error.<br>
		This function can NOT write 0 characters to the file, and is useful
		mainly for text files.<br>

		!fn: $writeHex(&lt;hex_data&gt;)
		Writes the &lt;hex_data&gt; to string.<br>
		The &lt;hex_data&gt; is a string of hexadecimal couples of digits.<br>
		Each couple of hex digits describes a byte.<br>
		Returns the number of bytes written to disk or '-1' in case of an error.<br>
		This function is useful mainly for binary files (the ones that usually
		contain unprintable characters and NULLs.<br>
		See also [fnc]$strToHex[/fnc]() and [fnc]$hexToStr[/fnc]().<br>

		!fn: $isEof()
		Returns '1' if the end of the file was reached, '0' otherwise.<br>

		!fn: $seek(&lt;position&gt;)
		Moves the file index to the specified &lt;position&gt;.<br>
		Returns '1' if successful, '0' otherwise.<br>

		!fn: $pos()
		Returns the file position.<br>

	@description:
		This object class is an interface to the system files.<br>
		You can open files, read data and write it.<br>

		The constructor accepts an additional parameter : the name of the file;
		in this way you can avoid to call [classfnc:file]$setFileName[/classfnc].<br>

		Note:<br>
		It is NOT a good idea to keep a file open for long time,
		you should open a file only when needed and close it immediately after
		you have terminated your operations.<br>
		This because the file may be accessed by other programs, and
		the disk storage I/O is really slower than the memory one (so you should
		perform most operations in memory, when possible, and not write & seek through the file).<br>

	@examples:
		A funky implementation of the cat command.<br>
		It reads a text file and echoes it to the current window.<br>
		You can easily convert it to an alias.<br>
		<example>
		&nbsp;%f = [fnc]$new[/fnc](file, [fnc]$root[/fnc], dummy, /afilenamewithpath)
		&nbsp;[cmd]if[/cmd](%f->[classfnc:file]$open[/classfnc](r))
		&nbsp;{
		&nbsp;	[cmd]while[/cmd](!%f->[classfnc:file]$isEof[/classfnc]())[cmd]echo[/cmd] %f->[classfnc:file]$readLine[/classfnc](1000)
		&nbsp;	# close()is not really needed, but let's be pedantic
		&nbsp;	%f->[classfnc:file]$close[/classfnc]()
		&nbsp;} [cmd:if]else[/cmd] [cmd]echo[/cmd] Cannot open the file for reading.
		&nbsp;[cmd]destroy[/cmd] %f
		</example>
	@seealso:
		class [class]object[/class], <br>
		<a href="syntax_objects.kvihelp">Objects documentation</a><br>
*/

/**
 * FILE class
 */
void KviScriptFile::initializeClassDefinition(KviScriptObjectClassDefinition *d)
{
	d->addBuiltinFunction("setFileName", (scriptObjectFunction) &KviScriptFile::builtinFunction_SETFILENAME);
	d->addBuiltinFunction("fileName",    (scriptObjectFunction) &KviScriptFile::builtinFunction_FILENAME);
	d->addBuiltinFunction("open",        (scriptObjectFunction) &KviScriptFile::builtinFunction_OPEN);
	d->addBuiltinFunction("close",       (scriptObjectFunction) &KviScriptFile::builtinFunction_CLOSE);
	d->addBuiltinFunction("flush",       (scriptObjectFunction) &KviScriptFile::builtinFunction_FLUSH);
	d->addBuiltinFunction("size",        (scriptObjectFunction) &KviScriptFile::builtinFunction_SIZE);
	d->addBuiltinFunction("exists",      (scriptObjectFunction) &KviScriptFile::builtinFunction_EXISTS);
	d->addBuiltinFunction("remove",      (scriptObjectFunction) &KviScriptFile::builtinFunction_REMOVE);
	d->addBuiltinFunction("isEof",       (scriptObjectFunction) &KviScriptFile::builtinFunction_ISEOF);
	d->addBuiltinFunction("pos",         (scriptObjectFunction) &KviScriptFile::builtinFunction_POS);
	d->addBuiltinFunction("seek",        (scriptObjectFunction) &KviScriptFile::builtinFunction_SEEK);
	d->addBuiltinFunction("write",       (scriptObjectFunction) &KviScriptFile::builtinFunction_WRITE);
	d->addBuiltinFunction("writeHex",    (scriptObjectFunction) &KviScriptFile::builtinFunction_WRITEHEX);
	d->addBuiltinFunction("read",        (scriptObjectFunction) &KviScriptFile::builtinFunction_READ);
	d->addBuiltinFunction("readLine",    (scriptObjectFunction) &KviScriptFile::builtinFunction_READLINE);
	d->addBuiltinFunction("readHex",     (scriptObjectFunction) &KviScriptFile::builtinFunction_READHEX);
}

KviScriptFile::KviScriptFile(
	KviScriptObjectController *cntrl, KviScriptObject *p, const char *name, KviScriptObjectClassDefinition *pDef)
	: KviScriptObject(cntrl, p, name, pDef)
{
	m_pFile = 0;
}

KviScriptFile::~KviScriptFile()
{
	delete m_pFile;
	m_pFile = 0;
}

bool KviScriptFile::init(QPtrList<KviStr> *params)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			m_pFile = new QFile(pS->ptr());
			return true;
		}
	}
	m_pFile = new QFile();
	return true;
}

int KviScriptFile::builtinFunction_SETFILENAME(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			delete m_pFile;
			m_pFile = new QFile(pS->ptr());
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptFile::builtinFunction_FILENAME(QPtrList<KviStr> *, KviStr &buffer)
{
	buffer.append(m_pFile->name());
	return KVI_ERROR_Success;
}

int KviScriptFile::builtinFunction_OPEN(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			int flags = 0;
			if( pS->contains('r', false) ) flags |= IO_ReadOnly;
			if( pS->contains('w', false) ) flags |= IO_WriteOnly;
			if( pS->contains('a', false) ) flags |= IO_Append;
			if( pS->contains('t', false) ) flags |= IO_Truncate;
			buffer.append(m_pFile->open(flags) ? '1' : '0');
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptFile::builtinFunction_SEEK(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			bool bOk = false;
			int pos  = pS->toInt(&bOk);
			if( bOk ) {
				buffer.append(m_pFile->at(pos) ? '1' : '0');
				return KVI_ERROR_Success;
			}
			return KVI_ERROR_InvalidParameter;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptFile::builtinFunction_WRITE(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			int written = m_pFile->writeBlock(pS->ptr(), pS->len());
			KviStr tmp(KviStr::Format, "%d", written);
			buffer.append(tmp);
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptFile::builtinFunction_READ(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			bool bOk = false;
			unsigned int len = pS->toUInt(&bOk);
			if( bOk && (len > 0) ) {
				char *buf = (char *) kvi_malloc(len);
				m_pFile->readBlock(buf, len);
				buffer.append(buf);
				kvi_free(buf);
				return KVI_ERROR_Success;
			}
			return KVI_ERROR_InvalidParameter;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptFile::builtinFunction_READHEX(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			bool bOk = false;
			unsigned int len = pS->toUInt(&bOk);
			if( bOk && (len > 0) ) {
				char *buf = (char *) kvi_malloc(len);
				int read  = m_pFile->readBlock(buf, len);
				if( read > 0 ) {
					KviStr tmp;
					tmp.bufferToHex(buf, read);
					buffer.append(tmp);
				}
				kvi_free(buf);
				return KVI_ERROR_Success;
			}
			return KVI_ERROR_InvalidParameter;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptFile::builtinFunction_READLINE(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			bool bOk = false;
			unsigned int len = pS->toUInt(&bOk);
			if( bOk && (len > 0) ) {
				char *buf = (char *) kvi_malloc(len);
				m_pFile->readLine(buf, len);
				buf[len - 1] = '\0'; // Remove the trailing newline
				buffer.append(buf);
				kvi_free(buf);
				return KVI_ERROR_Success;
			}
			return KVI_ERROR_InvalidParameter;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptFile::builtinFunction_WRITEHEX(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			char *buf = 0;
			int len = pS->hexToBuffer(&buf, false);
			if( buf ) {
				int written = m_pFile->writeBlock(buf, len);
				KviStr tmp(KviStr::Format, "%d", written + 1);
				buffer.append(tmp);
				kvi_free(buf);
			} else buffer.append("-1");
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptFile::builtinFunction_CLOSE(QPtrList<KviStr> *, KviStr &buffer)
{
	m_pFile->close();
	return KVI_ERROR_Success;
}

int KviScriptFile::builtinFunction_FLUSH(QPtrList<KviStr> *, KviStr &buffer)
{
	m_pFile->flush();
	return KVI_ERROR_Success;
}

int KviScriptFile::builtinFunction_SIZE(QPtrList<KviStr> *, KviStr &buffer)
{
	KviStr str(KviStr::Format, "%u", m_pFile->size());
	buffer.append(str);
	return KVI_ERROR_Success;
}

int KviScriptFile::builtinFunction_POS(QPtrList<KviStr> *, KviStr &buffer)
{
	KviStr str(KviStr::Format, "%d", m_pFile->at());
	buffer.append(str);
	return KVI_ERROR_Success;
}

int KviScriptFile::builtinFunction_EXISTS(QPtrList<KviStr> *, KviStr &buffer)
{
	buffer.append(m_pFile->exists() ? '1' : '0');
	return KVI_ERROR_Success;
}

int KviScriptFile::builtinFunction_REMOVE(QPtrList<KviStr> *, KviStr &buffer)
{
	buffer.append(m_pFile->remove() ? '1' : '0');
	return KVI_ERROR_Success;
}

int KviScriptFile::builtinFunction_ISEOF(QPtrList<KviStr> *, KviStr &buffer)
{
	buffer.append(m_pFile->atEnd() ? '1' : '0');
	return KVI_ERROR_Success;
}

#include "m_kvi_script_file.moc"
