/*****************************************************************************
 * Copyright (C) 2004-2009 Christoph Thielecke <crissi99@gmx.de>             *
 *                                                                           *
 * This program is free software; you can redistribute it and/or modify      *
 * it under the terms of the GNU General Public License as published by      *
 * the Free Software Foundation; either version 2 of the License, or         *
 * (at your option) any later version.                                       *
 *                                                                           *
 * This package is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
 * GNU General Public License for more details.                              *
 *                                                                           *
 * You should have received a copy of the GNU General Public License         *
 * along with this package; if not, write to the Free Software               *
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA *
 *****************************************************************************/

#ifndef VPNACCOUNTDATA_H
#define VPNACCOUNTDATA_H

#include <QtCore/QObject>
#include <QtCore/QString>
#include <QtCore/QStringList>

/**
 * holds the data for a account
 */
class VpnAccountData : public QObject
{
    Q_OBJECT

public:
    enum ConnectionType {
        cisco = 0,
        racoon = 1,
        freeswan = 2,
        pptp = 3,
        openvpn = 4,
        l2tpd_racoon = 5,
        l2tpd_freeswan = 6,
        ciscoorig = 7,
        vtun = 8,
        ssh = 9,
        other = 99
    };

    enum AuthenticationType {
        cert = 0,
        psk = 1,
        hybrid = 2,
        unknownauth = 99
    };

    enum L2tpDaemonType {
        l2tpd = 0,
        openl2tp = 1
    };

    enum AuthenticationMethod {
        chap = 0,
        pap = 1,
        mschap = 2,
        mschap_v2 = 3
    };

    /**
     * Constructor
     * @param ConnType Type (cisco, freeswan,racoon, etc)
     * @param Name The name of the account
     * @param Gateway The hostname/IP of the VPN gateway
     * @return nothing
     */
    VpnAccountData(ConnectionType ConnType, const QString &Name);

    /**
     * Destructor
     */
    ~VpnAccountData();

    /**
     * Copy constructor
     */
    VpnAccountData(const VpnAccountData&);

    ConnectionType getConnectionType() const; //< returns Connection type
    QString getName() const; //< returns Name
    QString getGateway() const; //< returns Gateway
    QString getID() const; //< returns ID
    QString getUserName() const; //< returns UserName
    QString getUserPassword() const; //< returns UserPassword
    QString getIkeGroup() const; //< returns IkeGroup
    QString getEncryptionAlgorithm() const; //< returns EncryptionAlgorithm
    QString getAuthenticationAlgorithm() const; //< returns AuthenticationAlgorithm
    QString getPerfectForwardSecrety() const; //< returns PerfectForwardSecrecy
    QString getNtDomainName() const; //< returns NtDomainName
    QString getApplicationVersion() const; //< returns ApplicationVersion
    QString getRemoteNetAddr() const; //<returns remoteNetAddr
    QString getRemoteNetMask() const; //<returns remoteNetMask
    QString getPreSharedKey() const; // returns PreSharedKey
    QString getX509Certificate() const; //< returns x509 certificate
    QString getCaCertificate() const; //< returns x509 certificate (CA)
    QString getNetworkDevice() const; //< returns NetworkDevice
    QString getCertPath() const; //<returns CertPath
    QString getSpecialRemoteID() const; //< returns SpecialRemoteID;
    QString getSpecialLocalID() const; //< returns SpecialLocalID;
    QString getLocalIDType() const; //< return Local ID type for racoon
    QString getRemoteIDType() const; //< return Remote ID type for racoon
    QString getPingHostIP() const; //< returns last part of IP (host) for ping test
    QString getPrivateKeyPass() const; //< returns PrivateKeyPass
    QString getPrivateKey() const; //< returns PrivateKey
    QString getDnsServer() const;
    QString getLocalVirtualIP() const; //< returns LocalVirtualIP
    QString getRemoteVirtualIP() const; //< returns RemoteVirtualIP (tun at openvpn,pptp) or Netmask for (tap at openvpn)
    QString getPreSharedKeyFile() const;
    QString getCommandBeforeConnect() const;
    QString getCommandAfterConnect() const;
    QString getCommandBeforeDisconnect() const;
    QString getCommandAfterDisconnect() const;
    QString getSpecialServerCertificate() const;
    QString getHashAlgo() const;
    QString getTunnelDeviceType() const;
    QString getDescription() const;
    QString getNsCertType() const;
    QString getVirtualIP() const;
    QString getUserdefiniedCipher() const;
    QString getTlsAuthFile() const;
    QString getHttpProxy() const;
    QString getExchangeMode() const;
    QString getTlsRemoteHost() const;
    QString getRightNextHop() const;
    QString getLeftNextHop() const;
    QString getHttpProxyAuthType() const;
    QString getHttpProxyUser() const;
    QString getHttpProxyPass() const;
    QString getPkcs11Providers() const;
    QString getPkcs11SlotType() const;
    QString getPkcs11Slot() const;
    QString getPkcs11IdType() const;
    QString getPkcs11Id() const;
    QString getPkcs11SignMode() const;
    QString getAuthMethod() const;
    QString getSearchDomainInResolvConf() const;
    QString getAuthenticationDirection() const;
    QString getCiscoNatMode() const;
    QString getIpsecEsp() const;
    QString getIpsecIke() const;
    QString getVtunProfile() const;
    QString getIpsecVpnMode() const;
    QString getLeftSourceIp() const;
    QString getRightSourceIp() const;
    QString getVirtualSubnetworks() const;
    QString getSshConfigRemoteScript() const;

    int getLocalPort() const; //< returns LocalPort
    int getRemotePort() const; //< returns RemotePort
    int getUdpPort() const; //< returns UdpPort
    int getMtu() const;
    int getMru() const;
    int getUserdefinedPort() const;
    int getConnectionStatusCheckSuccessCount() const;
    int getConnectionStatusInterval() const;
    int getHttpProxyPort() const;
    int getHttpProxyTimeout() const;
    int getReconnectDelay() const;
    int getMssfix() const;
    int getFragment() const;
    int getPeerTimeout() const;
    int getCommandAfterConnectDelayTime() const;
    int getDpdIdleTimeout() const;
    int getMaxConnectTries() const;
    int getRenegSec() const;
    int getTunnelPing() const;
    int getTunnelPingRestart() const;

    AuthenticationType getAuthType() const; //< returns AuthType
    QStringList getAdditionalNetworkRoutes() const;
    bool getSaveUserPassword() const; //< returns saveUserPassword
    bool getSavePsk() const; //< returns savePSK
    bool getUseSingleDes() const; //< returns useSingleDes
    bool getUseIkeGroup() const; //< returns useIkeGroup
    bool getUsePerfectForwardSecrety() const; //< returns usePerfectSecrecy
    bool getUseNtDomainName() const; //< returns useNtDomainName
    bool getUseApplicationVersion() const; //< returns useApplicationVersion
    bool getUseLocalPort() const; //< returns useLocalPort
    bool getUseRemotePort() const; //< returns useRemotePort
    bool getUseAdvancedSettings() const; //< returns useAdvancedSettings
    bool getUseGlobalIpsecSecret() const; //< returns useGlobalIpsecSecret
    bool getDoPingIP() const; //< returns doPingIP
    bool getUseSpecialRemoteID() const; //< returns useSpecialRemoteID
    bool getUseSpecialLocalID() const; //< returns useSpecialLocalID
    bool getUseUdp() const; //< returns useUdp
    bool getUseUdpPort() const; //< returns useUdpPort
    bool getUseDnsUpdate() const;
    bool getUseDnsServer() const;
    bool getRefuse40BitEncryption() const;
    bool getRefuse128BitEncryption() const;
    bool getAllowStatefulMode() const;
    bool getRequireMppe() const;
    bool getUseNoIpDefault() const;
    bool getDisableMPPEComp() const;
    bool getExecuteCmdBeforeConnect() const;
    bool getExecuteCmdAfterConnect() const;
    bool getExecuteCmdBeforeDisconnect() const;
    bool getExecuteCmdAfterDisconnect() const;
    bool getUseNoBsdComp() const;
    bool getUseNoDeflate() const;
    bool getReplaceDefaultRoute() const;
    bool getUseMtu() const;
    bool getUseMru() const;
    bool getUseVirtualIP() const;
    bool getPskIsInFile() const;
    bool getUseAdditionalNetworkRoutes() const;
    bool getUseSpecialServerCertificate() const;
    bool getUseUserdefinedPort() const;
    bool getUseNsCertType() const;
    bool getUseConnectionStatusCheck() const;
    bool getDoReconnectAfterConnectionLost() const;
    bool getDisableLzoCompression() const;
    bool getAuthWithUsernameAndPassword() const;
    bool getAllowEmptyGroupPassword() const;
    bool getUseUserdefiniedCipher() const;
    bool getUseRedirectGateway() const;
    bool getUseTlsAuth() const;
    bool getUseHttpProxy() const;
    bool getDontSaveUsername() const;
    bool getUseModeConfig() const;
    bool getUseReconnectDelay() const;//< if true kvpnc will wait ReconnectDelay before do reconnect instead of immediately reconnect
    bool getUseTlsRemoteHost() const;
    bool getDisableOpportunisticEncryption() const;
    bool getUseMssfix() const;
    bool getUseFragment() const;
    bool getDisableCcp() const;
    bool getUseSmartcard() const;
    bool getUseXauthInteractive() const;
    bool getSavePrivateKeyPassword() const;
    bool getUseHttpProxyAuth() const;
    bool getUseOnlyCaCertAndUserAuth() const;
    bool getUseMailAddressAsIdentifier() const;
    bool getUseRightNextHop() const;
    bool getUseLeftNextHop() const;
    bool getUsePkcs11Providers() const;
    bool getAllowEmptyPrivateKeyPassword() const;
    bool getAllowIpAddressChangeOfPeer() const;
    bool getUseSearchDomainInResolvConf() const;
    bool getDisableDataEncryption() const;
    bool getUseAuthenticationAlgorithm() const;
    bool getFixPathMtuDiscoveryProblem() const;
    bool getUseRemoteNetwork() const;
    bool getDisableHeaderCompression() const;
    bool getDisableMagicNumberNegotiation() const;
    bool getDisableIpx() const;
    bool getDisableProtocolFieldCompression() const;
    bool getDisableAddressControlCompression() const;
    bool getUseCustomIke() const;
    bool getUseCustomEsp() const;
    bool getVerifyCaCert() const;
    bool getUseDpdIdleTimeout() const;
    bool getHideGroupPasswordInAccountDataDialog() const;
    bool getUseLeftSourceIp() const;
    bool getUseRightSourceIp() const;
    bool getUseVirtualSubnetworks() const;
    bool getDisablePushFromServer() const;
    bool getDisableBind() const;
    bool getUseRenegSec() const;
    bool getUseTunnelPing() const;
    bool getUseTunnelPingRestart() const;
    bool getRequireEap() const;
    bool getUseSshConfigRemoteScript() const;
    bool getAskUserPasswordOnEachConnect() const;

    L2tpDaemonType getL2tpDaemon() const;

    /**
     * Sets the connection type
     * @param type The type
     */
    void setConnectionType(ConnectionType type);

    /**
     * Sets the name of the profile
     * @param Name The new name
     */
    void setName(const QString &Name);

    /**
     * Sets the IP/DNS name of the gateway
     * @param Gateway the IP/DNS name
     */
    void setGateway(const QString &Gateway);

    /**
     * Sets the remote ID (cisco)
     * @param ID The ID
     */
    void setID(const QString &ID);

    /**
     * Sets the group password (cisco)
     * @param GroupPassword The password
     */
    void setGroupPassword(const QString &GroupPassword);

    /**
     * Sets the username (cisco)
     * @param Username The username
     */
    void setUserName(const QString &Username);

    /**
     * Sets the user password
     * @param UserPassword The password
     */
    void setUserPassword(const QString &UserPassword);

    /**
     * Sets if user password should be stored or not
     * @param saveUserPassword True if password should be stored
     */
    void setSaveUserPassword(bool saveUserPassword);

    /**
     * Sets if group password should be stored or not
     * @param saveGroupPassword True if password should be stored
     */
    void setSavePsk(bool savePsk);

    /**
     * Sets the IKE group (cisco)
     * @param IkeGroup The group
     */
    void setIkeGroup(const QString &IkeGroup);
    void setEncryptionAlgorithm(const QString &EncryptionAlgorithm);
    void setAuthenticationAlgorithm(const QString &AuthenticationAlgorithm);

    void setPerfectForwardSecrety(const QString &PerfectForwardSecrecy);
    void setNtDomainName(const QString &Name);
    void setApplicationVersion(const QString &version);
    void setUseSingleDes(bool useSingleDes);
    void setLocalPort(int port);
    void setRemotePort(int port);
    void setUseIkeGroup(bool useIkeGroup);
    void setUsePerfectForwardSecrety(bool usePerfectForwardSecrety);
    void setUseNtDomainName(bool useNtDomainName);
    void setUseApplicationVersion(bool useApplicationVersion);
    void setUseLocalPort(bool useLocalPort);
    void setUseRemotePort(bool useRemotePort);
    void setUseAdvancedSettings(bool useAdvancedSettings);
    void setUseGlobalIpsecSecret(bool useGlobalIpsecSecret);
    void setRemoteNetAddr(const QString &RemoteNetAddr);
    void setRemoteNetMask(const QString &RemoteNetMask);
    void setPreSharedKey(const QString &PreSharedKey);
    void setX509Certificate(const QString &x509Certificate);
    void setCaCertificate(const QString &CaCertificate);
    void setNetworkDevice(const QString &NetworkDevice);
    void setCertPath(const QString &CertPath);
    void setAuthType(AuthenticationType authtype);
    void setDoPingIP(bool pingIP);
    void setPingHostIP(const QString &pingHostIP);
    void setPrivateKeyPass(const QString &PrivateKeyPass);
    void setPrivateKey(const QString &PrivateKey);
    void setUseSpecialRemoteID(bool useSpecialRemoteID);
    void setUseSpecialLocalID(bool useSpecialLocalID);
    void setSpecialRemoteID(const QString &SpecialRemoteID);
    void setSpecialLocalID(const QString &SpecialLocalID);
    void setLocalIDType(const QString &LocalIDType);
    void setRemoteIDType(const QString &RemoteIDType);
    void setUseUdp(bool useUdp);
    void setUseUdpPort(bool useUdpPort);
    void setUdpPort(int UdpPort);
    void setUseDnsUpdate(bool useDnsUpdate);
    void setUseDnsServer(bool useDnsServer);
    void setDnsServer(const QString &DnsServer);
    void setRefuse40BitEncryption(bool refuse40BitEncryption);
    void setRefuse128BitEncryption(bool refuse128BitEncryption);
    void setAllowStatefulMode(bool allowStatefulMode);
    void setRequireMppe(bool requireMppe);
    void setUseNoIpDefault(bool useNoIpDefault);
    void setDisableMPPEComp(bool disableMPPEComp);
    void setLocalVirtualIP(const QString &LocalVirtualIP);
    void setRemoteVirtualIP(const QString &RemoteVirtualIP);
    void setPreSharedKeyFile(const QString &PreSharedKeyFile);
    void setExecuteCmdBeforeConnect(bool executeCmdBeforeConnect);
    void setExecuteCmdAfterConnect(bool executeCmdAfterConnect);
    void setExecuteCmdBeforeDisconnect(bool executeCmdBeforeDisonnect);
    void setExecuteCmdAfterDisconnect(bool executeCmdAfterDisonnect);
    void setCommandBeforeConnect(const QString &CommandBeforeConnect);
    void setCommandAfterConnect(const QString &CommandAfterConnect);
    void setCommandBeforeDisconnect(const QString &CommandBeforeDisconnect);
    void setCommandAfterDisconnect(const QString &CommandAfterDisconnect);
    void setUseNoBsdComp(bool useNoBsdComp);
    void setUseNoDeflate(bool useNoDeflate);
    void setReplaceDefaultRoute(bool setReplaceDefaultRoute);
    void setUseMtu(bool useMtu);
    void setUseMru(bool useMru);
    void setMtu(int Mtu);
    void setMru(int Mru);
    void setUseVirtualIP(bool useVirtualIP);
    void setVirtualIP(const QString &VirtualIP);
    void setPskIsInFile(bool PskIsInFile);
    void setUseAdditionalNetworkRoutes(bool useAdditionalNetworkRoutes);
    void setAdditionalNetworkRoutes(QStringList AdditionalNetworkRoutes);
    void setUseSpecialServerCertificate(bool useSpecialServerCertificate);
    void setSpecialServerCertificate(const QString &SpecialServerCertificate);
    void setHashAlgo(const QString &HashAlgo);
    void setTunnelDeviceType(const QString &TunnelDeviceType);
    void setUseUserdefinedPort(bool useUserdefinedPort);
    void setUserdefinedPort(int UserdefinedPort);
    void setDescription(const QString &Description);
    void setNsCertType(const QString &NsCertType);
    void setUseNsCertType(bool useNsCertType);
    void setUseConnectionStatusCheck(bool useConnectionStatusCheck);
    void setConnectionStatusCheckSuccessCount(int connectionStatusCheckSuccessCount);
    void setConnectionStatusInterval(int connectionStatusInterval);
    void setDoReconnectAfterConnectionLost(bool doReconnectAfterConnectionLost);
    void setDisableLzoCompression(bool disableLzoCompression);
    void setAuthWithUsernameAndPassword(bool authWithUsernameAndPassword);
    void setAllowEmptyGroupPassword(bool allowEmptyGroupPassword);
    void setUseUserdefiniedCipher(bool useUserdefiniedCipher);
    void setUserdefiniedCipher(QString UserdefiniedCipher);
    void setUseRedirectGateway(bool useRedirectGateway);
    void setUseTlsAuth(bool useTlsAuth);
    void setTlsAuthFile(QString TlsAuthFile);
    void setUseHttpProxy(bool useHttpProxy);
    void setHttpProxy(QString HttpProxy);
    void setHttpProxyPort(int HttpProxyPort);
    void setHttpProxyTimeout(int HttpProxyTimeout);
    void setDontSaveUsername(bool dontSaveUsername);
    void setUseModeConfig(bool useModeConfig);
    void setExchangeMode(QString ExchangeMode);
    void setUseReconnectDelay(bool useReconnectDelay);
    void setReconnectDelay(int ReconnectDelay);
    void setUseTlsRemoteHost(bool useTlsRemoteHost);
    void setTlsRemoteHost(QString TlsRemoteHost);
    void setDisableOpportunisticEncryption(bool);
    void setRightNextHop(QString RightNextHop);
    void setLeftNextHop(QString LeftNextHop);
    void setUseMssfix(bool useMssfix);
    void setUseFragment(bool useFragment);
    void setMssfix(int Mssfix);
    void setFragment(int Fragment);
    void setPeerTimeout(int PeerTimeout);
    void setDisableCcp(bool disableCcp);
    void setUseSmartcard(bool useSmartcard);
    void setUseXauthInteractive(bool useXauthInteractive);
    void setSavePrivateKeyPassword(bool savePrivateKeyPassword);
    void setUseHttpProxyAuth(bool useHttpProxyAuth);
    void setHttpProxyAuthType(QString HttpProxyAuthType);
    void setHttpProxyUser(QString HttpProxyUser);
    void setHttpProxyPass(QString HttpProxyPass);
    void setUseOnlyCaCertAndUserAuth(bool useOnlyCaCertAndUserAuth);
    void setUseMailAddressAsIdentifier(bool useMailAddressAsIdentifier);
    void setUseRightNextHop(bool useRightNextHop);
    void setUseLeftNextHop(bool useLeftNextHop);
    void setPkcs11Providers(QString Pkcs11Providers);
    void setPkcs11SlotType(QString Pkcs11SlotType);
    void setPkcs11Slot(QString Pkcs11Slot);
    void setPkcs11IdType(QString Pkcs11IdType);
    void setPkcs11Id(QString Pkcs11Id);
    void setPkcs11SignMode(QString Pkcs11SignMode);
    void setUsePkcs11Providers(bool usePkcs11Providers);
    void setAllowEmptyPrivateKeyPassword(bool allowEmptyPrivateKeyPassword);
    void setAllowIpAddressChangeOfPeer(bool allowIpAddressChangeOfPeer);
    void setAuthMethod(QString AuthMethod);
    void setCommandAfterConnectDelayTime(int CommandAfterConnectDelayTime);
    void setUseSearchDomainInResolvConf(bool useSearchDomainInResolvConf);
    void setSearchDomainInResolvConf(QString SearchDomainInResolvConf);
    void setAuthenticationDirection(QString AuthenticationDirection);
    void setCiscoNatMode(QString CiscoNatMode);
    void setIpsecEsp(QString IpsecEsp);
    void setIpsecIke(QString IpsecIke);
    void setVtunProfile(QString VtunProfile);
    void setDisableDataEncryption(bool disableDataEncryption);
    void setUseAuthenticationAlgorithm(bool useAuthenticationAlgorithm);
    void setFixPathMtuDiscoveryProblem(bool fixPathMtuDiscoveryProblem);
    void setUseRemoteNetwork(bool useRemoteNetwork);
    void setIpsecVpnMode(QString IpsecVpnMode);
    void setDisableHeaderCompression(bool disableHeaderCompression);
    void setDisableMagicNumberNegotiation(bool disableMagicNumberNegotiation);
    void setDisableIpx(bool disableIpx);
    void setDisableProtocolFieldCompression(bool disableProtocolFieldCompression);
    void setDisableAddressControlCompression(bool disableAddressControlCompression);
    void setUseCustomIke(bool useCustomIke);
    void setUseCustomEsp(bool useCustomEsp);
    void setVerifyCaCert(bool verifyCaCert);
    void setUseDpdIdleTimeout(bool useDpdIdleTimeout);
    void setDpdIdleTimeout(int DpdIdleTimeout);
    void setHideGroupPasswordInAccountDataDialog(bool hideGroupPasswordInAccountDataDialog);
    void setMaxConnectTries(int MaxConnectTries);
    void setL2tpDaemon(L2tpDaemonType L2tpDaemon);
    void setUseLeftSourceIp(bool useLeftSourceIp);
    void setUseRightSourceIp(bool useLeftRightIp);
    void setLeftSourceIp(QString LeftSourceIp);
    void setRightSourceIp(QString RightSourceIp);
    void setUseVirtualSubnetworks(bool useVirtualSubnetworks);
    void setVirtualSubnetworks(QString VirtualSubnetworks);
    void setDisablePushFromServer(bool disablePushFromServer);
    void setDisableBind(bool disableBind);
    void setUseRenegSec(bool useRenegSec);
    void setRenegSec(int RenegSec);
    void setUseTunnelPing(bool useTunnelPing);
    void setTunnelPing(int TunnelPing);
    void setUseTunnelPingRestart(bool useTunnelPingRestart);
    void setTunnelPingRestart(int TunnelPingRestart);
    void setRequireEap(bool requireEap);
    void setUseSshConfigRemoteScript(bool useSshConfigRemoteScript);
    void setSshConfigRemoteScript(QString SshConfigRemoteScript);
    void setAskUserPasswordOnEachConnect(bool askUserPasswordOnEachConnect);

protected:
    QString Name; //< Name of the profile
    QString Gateway; //< IP/DNS name of the gateway
    QString ID; //< ID of remote gateway
    QString Username; //< Username (cisco)
    QString UserPassword; //< User password (cisco)
    QString IkeGroup; //< DH group for PFS (cisco,racoon)
    QString PerfectForwardSecrecy; //< Type of PFS (cisco, racoon)
    QString EncryptionAlgorithm; // phase 2 (ipsec)
    QString AuthenticationAlgorithm; // phase 2 (ipsec)
    QString NtDomainName; //< NT domain name for authentication (cisco/pptp)
    QString ApplicationVersion; //< userdefined application version (cisco)
    QString RemoteNetAddr; //< Remote network addr of VPN
    QString RemoteNetMask; //< Remote network mask of VPN
    QString PreSharedKey; //< Pre shared key (freeswan/racoon)
    QString x509Certificate; //< Path to X.509 certificate file
    QString CaCertificate; //< Path to CA certificate file
    QString NetworkDevice; //< Network device
    QString PrivateKeyPass; //< Password for private key
    QString PrivateKey; //< Path to private key file
    QString CertPath; //< Global certificate path
    QString AuthType; //< Authentication type (cert/psk)
    QString pingHostIP; //< IP of the host which should be pinged
    QString SpecialRemoteID; //< userdefined ID of remote gateway
    QString SpecialLocalID; //< userdefined ID of remote gateway
    QString LocalIDType; //< type of the local ID for racoon
    QString RemoteIDType; //< type of the remote ID for racoon
    QString LocalVirtualIP; //< Local virtual IP for tunnel (openvpn)
    QString RemoteVirtualIP; //< Remote virtual IP for tunnel (openvpn)
    QString PreSharedKeyFile; //< for openvpn
    QString HashAlgo; // Hash algorithm for racoon
    QString UseUserdefiniedCipher; //< Cipher for OpenVPN
    QString DnsServer; //< userdefined DNS server for pptp
    QString CommandBeforeConnect;
    QString CommandAfterConnect;
    QString CommandBeforeDisconnect;
    QString CommandAfterDisconnect;
    QString Description; //< Description of the profile
    QString SpecialServerCertificate;
    QString VirtualIP; //< Virtual IP for freeswan/racoon
    QString TunnelDeviceType; //< Type of tunnel device which should be used e.g. tun or tap  for openvpn
    QString NsCertType; //< openvpn
    QString UserdefiniedCipher; //< Ciphers for OpenVPN
    QString TlsAuthFile;
    QString HttpProxy;
    QString ExchangeMode; //< IKE exchange mode
    QString TlsRemoteHost;
    QString RightNextHop; // freeswan
    QString LeftNextHop; // freeswan
    QString HttpProxyAuthType;
    QString HttpProxyUser;
    QString HttpProxyPass;
    QString Pkcs11Providers;
    QString Pkcs11SlotType;
    QString Pkcs11Slot;
    QString Pkcs11IdType;
    QString Pkcs11Id;
    QString Pkcs11SignMode;
    QString AuthMethod; //< authorization method for pptp (chap/pap)
    QString SearchDomainInResolvConf;
    QString AuthenticationDirection; //< openvpn key and tls-auth direction (none,0,1)
    QString CiscoNatMode;
    QString IpsecEsp; // IPsec ESP mode
    QString IpsecIke; //< IPsec IKE mode
    QString VtunProfile;
    QString IpsecVpnMode; //< IPSec tunnel type (transport | tunnel)
    QString LeftSourceIp;
    QString RightSourceIp;
    QString VirtualSubnetworks;
    QString SshConfigRemoteScript;

    QStringList AdditionalNetworkRoutes;

    bool saveUserPassword; //< True if the user password should be stored (cisco)
    bool savePsk; //< True if the PSK should be stored
    bool useAdvancedSettings; //< True if advanced settings should be used (cisco)
    bool useIkeGroup; //< True if userdefined IKE group should be used
    bool usePerfectForwardSecrety; //< True if userdefined PFS should be used
    bool useNtDomainName; //< True if NT domain name should eb used for authentication
    bool useApplicationVersion; //< True if userdefined application version should be used
    bool useLocalPort; //< True if userdefined local port should be used
    bool useRemotePort; //< True if userdefined remote port should be used
    bool useSingleDes; //< True if DES instead of 3DES should be used
    bool useGlobalIpsecSecret; //< True if global group password from /etc/vpnc.conf should be used
    bool doPingIP; //< True if pingHostIP should be pinged after connection is established
    bool useSpecialRemoteID; //< True if userdefined ID of remote gateway instead of certificate issuer should be used
    bool useSpecialLocalID; //< True if userdefined ID of local instead of certificate client should be used
    bool useUdp; //< True if NAT-T should be used
    bool useUdpPort; //< True if userdefined UDP port should be used
    bool useDnsUpdate; //< False if we don't update resolv.conf
    bool useDnsServer; //< True if userdefined DNS server should be used
    bool refuse40BitEncryption; //< True if 40 bit encryption is refused (pptp)
    bool refuse128BitEncryption; //< True if 128 bit encryption is refused (pptp)
    bool allowStatefulMode;
    bool requireMppe; //< True if MPPE is required (pptp)
    bool useNoIpDefault;
    bool disableMPPEComp;
    bool executeCmdBeforeConnect;
    bool executeCmdAfterConnect;
    bool executeCmdBeforeDisconnect;
    bool executeCmdAfterDisconnect;
    bool useNoBsdComp;
    bool useNoDeflate;
    bool replaceDefaultRoute; //< for PPTP
    bool useMtu;
    bool useMru;
    bool useVirtualIP; //< freeswan/racoon
    bool PskIsInFile; //< Read PSK from file
    bool useAdditionalNetworkRoutes;
    bool useSpecialServerCertificate;
    bool useUserdefinedPort;
    bool useNsCertType; // <openvpn
    bool useConnectionStatusCheck; //< if true it checks if the gateway is alive within a given time
    bool doReconnectAfterConnectionLost;
    bool disableLzoCompression;
    bool authWithUsernameAndPassword; //< openvpn, xauth
    bool allowEmptyGroupPassword; //< allows empty group password for vpnc
    bool useUserdefiniedCipher;
    bool useRedirectGateway;//< redirect gateway for openvpn
    bool useTlsAuth; //< Add an additional layer of HMAC authentication on top of the TLS control channel to protect against DoS attacks (openvpn)
    bool useHttpProxy;
    bool dontSaveUsername;
    bool useModeConfig;
    bool useReconnectDelay;
    bool useTlsRemoteHost;
    bool disableOpportunisticEncryption;
    bool useMssfix;
    bool useFragment;
    bool disableCcp;
    bool useSmartcard;
    bool useXauthInteractive;
    bool savePrivateKeyPassword;
    bool useHttpProxyAuth;
    bool useOnlyCaCertAndUserAuth; //< use only a CA certificate and authentication with username and password (openvpn)
    bool useMailAddressAsIdentifier; //< x509 cert
    bool useRightNextHop;
    bool useLeftNextHop;
    bool usePkcs11Providers;
    bool allowEmptyPrivateKeyPassword;
    bool allowIpAddressChangeOfPeer;
    bool useSearchDomainInResolvConf;
    bool disableDataEncryption;
    bool useAuthenticationAlgorithm;
    bool fixPathMtuDiscoveryProblem; //< see http://pptpclient.sourceforge.net/howto-diagnosis.phtml#connections_freeze
    bool useRemoteNetwork; //< specify a remote network
    bool disableHeaderCompression; //< Disable Van Jacobson style TCP/IP header compression in both the transmit and the receive direction.
    bool disableMagicNumberNegotiation; //< Disable magic number negotiation. With this option, pppd cannot detect a looped-back line. This option should only be needed if the peer is buggy.
    bool disableIpx; //< Disable the IPXCP and IPX protocols.
    bool disableProtocolFieldCompression; //< Disable protocol field compression negotiation in both the receive and the transmit direction.
    bool disableAddressControlCompression; //< Disable Address/Control compression in both directions (send and receive).
    bool useCustomIke;
    bool useCustomEsp;
    bool verifyCaCert; //< verify ca cert of remote host on racoon
    bool useDpdIdleTimeout; //< dead peer detection idle timeout (vpnc)
    bool hideGroupPasswordInAccountDataDialog;
    bool useLeftSourceIp;
    bool useRightSourceIp;
    bool useVirtualSubnetworks;
    bool disablePushFromServer; //< openvpn, no pull mode
    bool disableBind; //< Do not bind to local address and port. The IP stack will allocate a dynamic port for returning packets. Since the value of the dynamic port could not be known in advance by a peer, this option is only suitable for peers which will be initiating connections by using the --remote option.
    bool useRenegSec;
    bool useTunnelPing;
    bool useTunnelPingRestart;
    bool requireEap;
    bool useSshConfigRemoteScript;
    bool askUserPasswordOnEachConnect; //< ask for user password on each connect
    int Mtu;
    int Mru;
    int UdpPort; //< Port number of UDP port
    int LocalPort; //< userdefined local port
    int RemotePort; //< userdefined remote port
    int UserdefinedPort;
    int connectionStatusCheckSuccessCount; //< in which count a check must be successful
    int connectionStatusInterval; //< interval which is used for check
    int HttpProxyPort;
    int HttpProxyTimeout;
    int ReconnectDelay;
    int MaxConnectTries;
    int Mssfix;
    int Fragment;
    int PeerTimeout; // timeout given from peer (cisco for example gives that)
    int CommandAfterConnectDelayTime; //< time which is wait after connect before the cmd after connect should be executed
    int DpdIdleTimeout;
    int RenegSec;
    int TunnelPing;
    int TunnelPingRestart;

public:
    ConnectionType connectiontype; //< Type of the connection (cisco, freeswan, racoon, etc.)
    AuthenticationType authtype; //< Type of authentication (cert, psk)
    L2tpDaemonType L2tpDaemon;
};

#endif
