###
### $Rev: 44 $
### $Release: 0.5.1 $
### copyright(c) 2005 kuwata-lab all rights reserved.
###

require 'yaml'
require 'kwalify'
require 'kwalify/util/yaml-helper'
#require 'kwalify/util/option-parser'


module Kwalify

   class CommandOptionError < KwalifyError
      def initialize(message, option, error_symbol)
         super(message)
         @option = option
         @error_symbol = error_symbol
      end
      attr_reader :option, :error_symbol
   end


   ##
   ## ex.
   ##   command = File.basename($0)
   ##   begin
   ##      main = Kwalify::Main.new(command)
   ##      s = main.execute
   ##      print s if s
   ##   rescue Kwalify::CommandOptionError => ex
   ##      $stderr.puts "ERROR: #{ex.message}"
   ##      exit 1
   ##   rescue Kwalify::KwalifyError => ex
   ##      $stderr.puts "ERROR: #{ex.message}"
   ##      exit 1
   ##   end
   ##
   class Main

      def initialize(command=nil)
         @command = command || File.basename($0)
         @flag_help     = false
         @flag_version  = false
         @flag_silent   = false
         @flag_meta     = false
         @flag_untabify = false
         @flag_emacs    = false
         @flag_linenum  = false
         @flag_debug    = false
         @schema_filename = nil
         @properties    = {}
      end


      def debug?
         @flag_debug
      end


      def _inspect()
         s =     ""
         s <<    "command       : #{@command}\n"
         s <<    "flag_help     : #{@flag_help}\n"
         s <<    "flag_version  : #{@flag_version}\n"
         s <<    "flag_silent   : #{@flag_silent}\n"
         s <<    "flag_meta     : #{@flag_meta}\n"
         s <<    "flag_untabify : #{@flag_untabify}\n"
         s <<    "flag_emacs    : #{@flag_emacs}\n"
         s <<    "flag_linenum  : #{@flag_linenum}\n"
         s <<    "flag_debug    : #{@flag_debug}\n"
         s <<    "schema_filename : #{@schema_filename != nil ? @schema_filename : 'null'}\n"
         s <<    "properties:\n"
         @properties.each do |key, val|
            s << "  #{key}: #{val}\n"
         end
         return s
      end


      def execute(argv=ARGV)
         # parse command-line options
         filenames = _parse_argv(argv)

         # help or version
         s = ''
         s << _version() << "\n" if @flag_version
         s << _usage()           if @flag_help
         return s unless s.empty?

         # validation
         if @flag_meta2
            s = _quick_meta_validate(filenames)
         elsif @flag_meta
            s = _meta_validate(filenames)
         elsif @schema_filename
            if @flag_debug
               s = _inspect_schema(@schema_filename)
            else
               s = _validate(filenames, @schema_filename)
            end
         else
            #* key=:command_option_noaction  msg="command-line option '-f' or '-m' required."
            raise option_error(:command_option_noaction, @command)
         end
         return s   # or return (s == nil || s.empty?) ? nil : s
      end


      def self.main(command, argv=ARGV)
         begin
            main = Kwalify::Main.new(command)
            s = main.execute(ARGV)
            print s if s
            #rescue Kwalify::CommandOptionError => ex
            #   $stderr.puts ex.message
            #   exit 1
            #rescue Kwalify::KwalifyError => ex
            #   $stderr.puts "ERROR: #{ex.message}"
            #   exit 1
         rescue => ex
            if main.debug?
               raise ex
            else
               $stderr.puts ex.message
               exit 1
            end
         end
      end


      private


      def option_error(error_symbol, arg)
         msg = Kwalify.msg(error_symbol) % arg
         return CommandOptionError.new(msg, arg, error_symbol)
      end


      def _inspect_schema(schema_filename)
         filename = schema_filename
         str = File.open(schema_filename) { |f| f.read() }
         str = YamlUtil.untabify(str) if @flag_untabify
         schema = YAML.load(str)
         return nil if schema == nil
         validator = Kwalify::Validator.new(schema)  # error raised when schema is wrong
         s = validator._inspect()
         s << "\n" unless s[-1] == ?\n
         return s
      end


      #class QuickMetaValidator
      #   def validate(schema)
      #      errors = []
      #      begin
      #         validator = Kwalify::Validator.new(schema) # error raised when schema is wrong
      #      rescue Kwalify::SchemaError => ex
      #         errors << ex
      #      end
      #      return errors
      #   end
      #end


      def _quick_meta_validate(filenames)
         meta_validator = Object.new
         def meta_validator.validate(schema)
            errors = []
            begin
               validator = Kwalify::Validator.new(schema) # error raised when schema is wrong
            rescue Kwalify::SchemaError => ex
               errors << ex
            end
            return errors
         end
         s = _validate_files(meta_validator, filenames)
         return s
      end


      def _meta_validate(filenames)
         meta_validator = Kwalify::MetaValidator.instance()
         s = _validate_files(meta_validator, filenames)
         return s
      end


      def _validate(filenames, schema_filename)
         filename = schema_filename
         str = File.open(filename) { |f| f.read() }   # or File.read(filename) in ruby1.8
         str = YamlHelper.untabify(str) if @flag_untabify
         schema = YAML.load(str)
         if schema
            validator = Kwalify::Validator.new(schema)
            s = _validate_files(validator, filenames)
         else
            #* key=:schema_empty  msg="%s: empty schema.\n"
            s = Kwalify.msg(:schema_emtpy) % filename
         end
         return s
      end


      def _validate_files(validator, filenames)
         s = ''
         filenames = [ nil ] if filenames.empty?
         filenames.each do |filename|
            if filename
               str = File.open(filename) { |f| f.read() }   # or File.read(filename) in ruby1.8
            else
               str = $stdin.read()
               filename = '(stdin)'
            end
            str = YamlHelper.untabify(str) if @flag_untabify
            if @flag_linenum
               parser = Kwalify::YamlParser.new(str)
               i = 0
               while parser.has_next?
                  doc = parser.parse()
                  s << _validate_document(validator, doc, filename, i, parser)
                  i += 1
               end
            else
               parser = nil
               i = 0
               YAML.load_documents(str) do |doc|
                  s << _validate_document(validator, doc, filename, i, nil)
                  i += 1
               end
            end
         end
         return s
      end


      def _validate_document(validator, doc, filename, i, parser=nil)
         s = ''
         if doc == nil
            #* key=:validation_empty  msg="%s#%d: empty.\n"
            s << kwalify.msg(:validation_empty) % [filename, i]
            return s
         end
         errors = validator.validate(doc)
         if errors == nil || errors.empty?
            #* key=:validation_valid  msg="%s#%d: valid.\n"
            s << Kwalify.msg(:validation_valid) % [filename, i] unless @flag_silent
         else
            #* key=:validation_invalid  msg="%s#%d: INVALID\n"
            s << Kwalify.msg(:validation_invalid) % [filename, i]
            if @flag_linenum
               #assert parser != nil
               raise unless parser != nil
               parser.set_errors_linenum(errors)
               errors.sort!
            end
            errors.each do |error|
               if @flag_emacs
                  #assert @flag_linenum
                  raise unless @flag_linenum
                  s << "#{filename}:#{error.linenum}: [#{error.path}] #{error.message}\n"
               elsif @flag_linenum
                  s << "  - (line #{error.linenum}) [#{error.path}] #{error.message}\n"
               else
                  s << "  - [#{error.path}] #{error.message}\n"
               end
            end
         end
         return s
      end


      def _usage()
         msg = Kwalify.msg(:command_help) % [@command, @command]
         return msg
      end


      def _version()
         return RELEASE
      end


      def _to_value(str)
         case str
         when nil, "null", "nil"         ;   return nil
         when "true", "yes"              ;   return true
         when "false", "no"              ;   return false
         when /\A\d+\z/                  ;   return str.to_i
         when /\A\d+\.\d+\z/             ;   return str.to_f
         when /\/(.*)\//                 ;   return Regexp.new($1)
         when /\A'.*'\z/, /\A".*"\z/     ;   return eval(str)
         else                            ;   return str
         end
      end


      def _parse_argv(argv)
         while argv[0] && argv[0][0] == ?-
            optstr = argv.shift
            optstr = optstr[1, optstr.length-1]
            # property
            if optstr[0] == ?-
               unless optstr =~ /\A\-([-\w]+)(?:=(.*))?\z/
                  #* key=:command_property_invalid  msg="%s: invalid property."
                  raise option_error(:command_property_invalid, optstr)
               end
               prop_name = $1;  prop_value = $2
               key   = prop_name.gsub(/-/, '_').intern
               value = prop_value == nil ? true : _to_value(prop_value)
               @properties[key] = value
            # option
            else
               while optstr && !optstr.empty?
                  optchar = optstr[0]
                  optstr[0,1] = ""
                  case optchar
                  when ?h  ;  @flag_help     = true
                  when ?v  ;  @flag_version  = true
                  when ?s  ;  @flag_silent   = true
                  when ?t  ;  @flag_untabify = true
                  when ?m  ;  @flag_meta     = true
                  when ?M  ;  @flag_meta2    = true
                  when ?E  ;  @flag_emacs    = true ; @flag_linenum = true
                  when ?l  ;  @flag_linenum  = true
                  when ?D  ;  @flag_debug    = true
                  when ?f  ;
                     arg = optstr.empty? ? argv.shift : optstr
                     #* key=:command_option_noarg  msg="-%s: argument required."
                     #* key=:command_option_noschema  msg="-%s: schema filename required."
                     raise option_error(:command_option_noschema, "f") unless arg
                     @schema_filename = arg
                     optstr = nil
                  else
                     #* key=:command_option_invalid  msg="-%s: invalid command option."
                     raise option_error(:command_option_invalid, optchar.chr)
                  end
               end
            end
         end  # end of while
         #
         @flag_help    = true if @properties[:help]
         @flag_version = true if @properties[:version]
         filenames = argv
         return filenames
      end


      def _domain_type?(doc)
         klass = defined?(YAML::DomainType) ? YAML::DomainType : YAML::Syck::DomainType
         return doc.is_a?(klass)
      end

   end

end
