###
### $Rev: 42 $
### $Release: 0.5.1 $
### copyright(c) 2005 kuwata-lab all rights reserved.
###

unless defined?(TESTDIR)
   TESTDIR = File.dirname(__FILE__)
   libdir  = File.dirname(TESTDIR) + "/lib"
   $LOAD_PATH << libdir << TESTDIR
end

require 'test/unit'
require 'test/unit/ui/console/testrunner'
require 'kwalify'
require 'kwalify/util/assert-diff'
require 'yaml'
require 'pp'
require 'stringio'


class YamlParserTest < Test::Unit::TestCase
   def _test()
      parser = Kwalify::YamlParser.new(@input)
      if @error_class
         assert_raise(@error_class) do
            doc = parser.parse()
         end
      else
         doc = parser.parse()
         sio = StringIO.new
         PP.pp(doc, sio)
         actual = sio.string
         if $log
            File.open("#{@name}.expected", 'w') { |f| f.write(@expected) }
            File.open("#{@name}.actual", 'w') { |f| f.write(actual) }
         end
         if $print
            print actual
         else
            assert_equal_with_diff(@expected, actual)
         end
      end
   end

   #str = DATA.read()        # doesn't work when required by other script
   str = File.read(__FILE__)
   str.gsub!(/.*^__END__$/m, '')
   doc_table = {}
   YAML.load_documents(str) do |doc|
      name = doc['name']
      raise "name '#{name}' is duplicated." if doc_table[name]
      doc_table[name] = doc
      next if $target && $target != doc['name']
      #if $target && $show
      #   puts doc.to_yaml
      #   exit
      #end
      s = <<-END
      def test_#{name}
         @name     = #{doc['name'].dump}
         @input    = #{doc['input'].dump}
         @expected = #{doc['expected'].dump}
         @error_class = #{doc['error'] ? doc['error'] : 'nil'}
         @option   = #{doc['option'] ? doc['option'].dump : 'nil'}
         _test()
      end
      END
      if $target && $show
         puts s
         exit
      end
      eval s
   end
end

#if $0 == __FILE__
#   Test::Unit::UI::Console::TestRunner.run(YamlParserTest)
#end

__END__
---
name:  parse01
desc:  basic
#flag:  print
#
input: |
  - AAA
  - BBB
  - CCC
#
expected: |
  ["AAA", "BBB", "CCC"]
#
---
name:  parse02
desc:  scalar
#
input: |
  - abc
  - 123
  - 3.14
  - true
  - false
  - yes
  - no
  - ~
  - null
  - "123"
  - '456'
  - 2005-01-01
  - :sym
#
expected: |
  ["abc",
   123,
   3.14,
   true,
   false,
   true,
   false,
   nil,
   nil,
   "123",
   "456",
   #<Date: 4906743/2,0,2299161>,
   :sym]
#  ["abc", 123, true, false, true, false, nil, nil, "123", "456", #<Date: 4906743/2,0,2299161>, :sym]
#
---
name:  parse03
desc:  nested sequence
#
input: |
  - AAA
  -
    - BBB1
    - BBB2
  - CCC
#
expected: |
  ["AAA", ["BBB1", "BBB2"], "CCC"]
#
---
name:  parse04
desc:  nested sequence #2
#
input: |
  - A
  -
    - B1
    -
      - B1-1
      - B1-2
  - C
#
expected: |
  ["A", ["B1", ["B1-1", "B1-2"]], "C"]
#
---
name:  parse05
desc:  null item of sequence
#
input: |
  - A
  -
  - C
  -
  -
  -
  - G
#
expected: |
  ["A", nil, "C", nil, nil, nil, "G"]
#
---
name:  parse06
desc:  null item of nested sequence
#
input: |
  -
    -
      -
      -
      -
  -
#
expected: |
  [[[nil, nil, nil]], nil]
#
---
name:  parse07
desc:  sequence with empty lines
#
input: |

  - A

  -


      - B


  -
#
expected: |
  ["A", ["B"], nil]
#
---
name:  parse08
desc:  syntax error - invalid indent of sequence.
error: Kwalify::YamlSyntaxError
#
input: |
  - AAA
    - BBB1
    - BBB2
  - CCC
#
expected: |
#
---
name:  parse09
desc:  syntax error - sequence item is exepcted.
error: Kwalify::YamlSyntaxError
#
input: |
  - 
    - a1
    - a2
    a3
  -
    - b1
    - b2
    b3
#
expected: |
#
---
name:  parse11
desc:  mapping
#
input: |
  A: foo
  B: bar
  C  : baz
#
expected: |
  {"A"=>"foo", "B"=>"bar", "C"=>"baz"}
---
name:  parse12
desc:  mapping of scalar
#
input: |
  abc: ABC
  123: 1.23
  3.14: 314
#
expected: |
 {"abc"=>"ABC", 3.14=>314, 123=>1.23}
---
name:  parse12b
desc:  mapping of scalar
#
input: |
  true: yes
  false: no
#
expected: |
 {false=>false, true=>true}
#
---
name:  parse12c
desc:  mapping of scalar
#
input: |
  :sym: :symbol
  --: -10
#
expected: |
  {"--"=>-10, :sym=>:symbol}
#
---
name:  parse13
desc:  nested mapping
#
input: |
  A: 10
  B:
    B1: 21
    B2: 22
  C: 30
#
expected: |
  {"A"=>10, "B"=>{"B1"=>21, "B2"=>22}, "C"=>30}
#
---
name:  parse14
desc:  nested mapping #2
#
input: |
  A: 10
  B:
    B1:
      B1-1: 21
      B1-2: 22
      B1-3: 23
  C: 30
#
expected: |
  {"A"=>10, "B"=>{"B1"=>{"B1-1"=>21, "B1-2"=>22, "B1-3"=>23}}, "C"=>30}
#
---
name:  parse15
desc:  null item in mapping
#
input: |
  A:
  B:
  C:
  D:
#
expected: |
  {"A"=>nil, "B"=>nil, "C"=>nil, "D"=>nil}
#
---
name:  parse16
desc:  null mapping
#
input: |
  A:
  B:
    B1:
      B1-2:
  C:
#
expected: |
  {"A"=>nil, "B"=>{"B1"=>{"B1-2"=>nil}}, "C"=>nil}
#
---
name:  parse17
desc:  mapping with empty lines
#
input: |

  A: 1

  B: 


    B1:



      B1a: 2
  C: 3
  
#
expected: |
  {"A"=>1, "B"=>{"B1"=>{"B1a"=>2}}, "C"=>3}
#
---
name:  parse18
desc:  parse error - invalid indent of mapping.
error:  Kwalify::YamlSyntaxError
#
input: |
  A: 10
  B: 20
    B1: 21
    B2: 22
  C: 30
#
expected: |
  
#
---
name:  parse19
desc:  parse error - mapping item is expected.
error: Kwalify::YamlSyntaxError
#
input: |
  A:
    a1: 1
    a2: 2
    a3  3
  B:
    b1: 1
    b2: 2
    b3  3
expected: |
#
---
name:  parse21
desc:  seq of mapping
#
input: |
  -
    x: 10
    y: 20
  -
    x: 15
    y: 25
#
expected: |
  [{"x"=>10, "y"=>20}, {"x"=>15, "y"=>25}]
---
name:  parse22
desc:  seq of mapping (inline)
#
input: |
  - x: 10
    y: 20
  - x: 15
    y: 25
#
expected: |
  [{"x"=>10, "y"=>20}, {"x"=>15, "y"=>25}]
---
name:  parse23
desc:  seq of seq of seq (inline)
#
input: |
  - - - a
      - b
  - - - c
      - d
#
expected: |
  [[["a", "b"]], [["c", "d"]]]
---
name:  parse24
desc:  map of sequence
#
input: |
  foo:
    - 1
    - 2
    - 3
  bar:
    - 4
    - 5
    - 6
#
expected: |
  {"foo"=>[1, 2, 3], "bar"=>[4, 5, 6]}
---
name:  parse25
desc:  map of sequence (inline)
#
input: |
  foo: - 1
       - 2
       - 3
  bar: - 4
       - 5
       - 6
#
expected: |
  {"foo"=>[1, 2, 3], "bar"=>[4, 5, 6]}
---
name:  parse26
desc:  map of map of map (inline)
#
input: |
  A: a: 1: 100
        2: 200
  B: b: 3: 300
        4: 400
#
expected: |
  {"A"=>{"a"=>{1=>100, 2=>200}}, "B"=>{"b"=>{3=>300, 4=>400}}}
---
name:  comment1
desc:  line comment
#
input: |
  # comment
  - A
  - B
    # comment
  -
      # comment
    - C
#
expected: |
  ["A", "B", ["C"]]
---
name:  comment2
desc:  escape line comment
#
input: |
  # comment
  - A
  - B:
     "# comment"
  -
    '# comment'
#
expected: |
  ["A", {"B"=>"# comment"}, "# comment"]
---
name:  comment3
desc:  line comment with seq and map
#
input: |
  - A             # comment
  - B:            # comment
      C: foo      # comment
      D: "bar#bar"    #comment
#
expected: |
  ["A", {"B"=>{"C"=>"foo", "D"=>"bar#bar"}}]
---
name:  comment4
desc:  line comment with anchor and alias
#
input: |
  - &a1           # comment
    foo
  - *a1           # comment
#
expected: |
  ["foo", "foo"]
---
name:  flowseq1
desc:  flow style seq
#
input: |
  - [ 10, 20 ]
  - [15,25,35]
#
expected: |
  [[10, 20], [15, 25, 35]]
#
---
name:  flowseq2
desc:  nested flow style seq 
#
input: |
  1: [ A, [B1, B2]]
  2: [[[X]]]
  3: [[x1,y1],[x2,"y2"],['x3',y3]]
#
expected: |
  {1=>["A", ["B1", "B2"]],
   2=>[[["X"]]],
   3=>[["x1", "y1"], ["x2", "y2"], ["x3", "y3"]]}
#
---
name:  flowseq3
desc:  flow style seq with some lines
#
input: |
  A: [ [10,20],
       [11,21],
       [12,22]]
  B: [
    [1.1,
     1.2,
     1.3
    ]
  ]
#
expected: |
  {"A"=>[[10, 20], [11, 21], [12, 22]], "B"=>[[1.1, 1.2, 1.3]]}
#
---
name:  flowseq4
desc:  invalid flow style seq (sequence item required (or last comma is extra).)
mesg:  sequence item required (or last comma is extra).
error: Kwalify::YamlSyntaxError
#
input: |
  A: [ [10,20], ]
#
expected: |
#
---
name:  flowseq5
desc:  invalid flow style seq (flow style sequence requires ']').
mesg:  flow style sequence requires ']'
error: Kwalify::YamlSyntaxError
#
input: |
  A: [ [10,20]
  B: [ [30, 40]]
#
expected: |
#
---
name:  flowseq6
desc:  invalid flow style seq (flow style sequence requires ']').
mesg:  flow style sequence is closed but got ']'.
error: Kwalify::YamlSyntaxError
#
input: |
  [ 10 ]]
#
expected: |
#
---
name:  flowmap1
desc:  flow style map
#
input: |
  - { A1: 10, A2: 20 }
  - {B1: 15, 'B2': 25, "B3": 35}
#
expected: |
  [{"A1"=>10, "A2"=>20}, {"B1"=>15, "B2"=>25, "B3"=>35}]
#
---
name:  flowmap2
desc:  flow style map nested
#
input: |
  A: { x: {y: {z: 10}}}
  B: { a: 1, b:{c: 2, d: 3, e:{f: 4}}, g: 5}
#
expected: |
  {"A"=>{"x"=>{"y"=>{"z"=>10}}},
   "B"=>{"a"=>1, "b"=>{"c"=>2, "d"=>3, "e"=>{"f"=>4}}, "g"=>5}}
#
---
name:  flowmap3
desc:  flow style map with some lines
#
input: |
  A: { x:
       {y:
         {z: 10}
       }
     }
  B: {
    a: 1,
    b: {
      c: 2,
      d: 3,
      e: {
        f: 4
      }
    },
    g: 5
  }
#
expected: |
  {"A"=>{"x"=>{"y"=>{"z"=>10}}},
   "B"=>{"a"=>1, "b"=>{"c"=>2, "d"=>3, "e"=>{"f"=>4}}, "g"=>5}}
#
---
name:  flowmap4
desc:  invalid flow style map (mapping item required (or last comma is extra).)
mesg:  mapping item required (or last comma is extra).
error: Kwalify::YamlSyntaxError
#
input: |
  - {A: 10, B: 20, }
#
expected: |
#
---
name:  flowmap5
desc:  invalid flow style map (flow style mapping requires '}').
mesg:  flow style mapping requires '}'
error: Kwalify::YamlSyntaxError
#
input: |
  - {A: { x: 10, y: 20 }
  - {A: { x: 11, y: 21 }}
#
expected: |
#
---
name:  flowmap6
desc:  invalid flow style map (flow style mapping requires ']').
mesg:  flow style mapping is closed but got '}'.
error: Kwalify::YamlSyntaxError
#
input: |
  { x: 10 }}
#
expected: |
#
---
name:  parse61
desc:  combination of flow style seq and map
#
input: |
  [
    {name: '  foo  ',
     e-mail: foo@mail.com},
    {name: ba   z,
     e-mail: ba__z@mail.com   }
  ]
#
expected: |
  [{"name"=>"  foo  ", "e-mail"=>"foo@mail.com"},
   {"name"=>"ba   z", "e-mail"=>"ba__z@mail.com"}]
#
---
name:  block_text1
desc:  parse_block_text
#
input: |
  - text1: |
     foo
     bar
     baz
  - text2: |
         aaa
        bbb
          ccc
  - |
   foo
   bar
   baz
  - |
         aaa
        bbb
          ccc
#
expected: |
  [{"text1"=>"foo\nbar\nbaz\n"},
   {"text2"=>" aaa\nbbb\n  ccc\n"},
   "foo\nbar\nbaz\n",
   " aaa\nbbb\n  ccc\n"]
#
---
name:  block_text2
desc:  block text with '|+' or '|-'
#
input: |
  - text1: |
      A
  
      B
      C
  
  
  - text2: |+
      A
  
      B
      C
  
  
  - text3: |-
      A
  
      B
      C
  
  
#
expected: |
  [{"text1"=>"A\n\nB\nC\n"},
   {"text2"=>"A\n\nB\nC\n\n\n"},
   {"text3"=>"A\n\nB\nC"}]
#
---
name:  block_text3
desc:  block text with '| foo'
#
input: |
  - text1: | foo  
     A
     
     B
     C
     
  - |  foo
    A
     B
      C
#
expected: |
  [{"text1"=>"foo  A\n\nB\nC\n"}, "fooA\n B\n  C\n"]
#
---
name:  block_text4
desc:  block text with '#' (comment)
#
input: |
  #
    - text1: |
       A
       #
       B
      #
      text2: |
       #
       #
      #
    - |
     A
     #
     B
    #
    - |
      #
      #
  #
    - x
  
#
expected: |
  [{"text1"=>"A\n#\nB\n", "text2"=>"#\n#\n"}, "A\n#\nB\n", "#\n#\n", "x"]
#
---
name:  block_text6
desc:  parse_block_text (>)
#
input: |
  - text1: >
     foo
     bar
     baz
  - text2: >
         aaa
        bbb
          ccc
  - >
   foo
   bar
   baz
  - >
         aaa
        bbb
          ccc
#
expected: |
  [{"text1"=>"foo bar baz\n"},
   {"text2"=>" aaa bbb   ccc\n"},
   "foo bar baz\n",
   " aaa bbb   ccc\n"]
#
---
name:  block_text7
desc:  block text with '>+' or '>-'
#
input: |
  - text1: >
      A
  
      B
      C
  
  
  - text2: >+
      A
  
      B
      C
  
  
  - text3: >-
      A
  
      B
      C
  
  
#
expected: |
  [{"text1"=>"A\nB C\n"}, {"text2"=>"A\nB C\n\n"}, {"text3"=>"A\nB C"}]
#
---
name:  block_text8
desc:  block text with '> foo'
#
input: |
  - text1: > foo  
     A
     
     B
     C
     
  - >  foo
    A
     B
      C
#
expected: |
  [{"text1"=>"foo  A\nB C\n"}, "fooA  B   C\n"]
#
---
name:  block_text9
desc:  block text with '#' (comment)
#
input: |
  #
    - text1: >
       AA
       ##
       BB
      #
      text2: >
       #
       #
      #
    - >
     AA
     ##
     BB
    #
    - >
      #
      #
  #
    - x
  
#
expected: |
  [{"text1"=>"AA ## BB\n", "text2"=>"# #\n"}, "AA ## BB\n", "# #\n", "x"]
#
---
name:  anchor1
desc:  parse_anchor, parse_alias
#
input: |
  - &a1 foo
  - &a2
   bar
  - *a1
  - *a2
#
expected: |
  ["foo", "bar", "foo", "bar"]
#
---
name:  anchor2
desc:  parse_anchor, parse_alias
#
input: |
  - A: &a1
     x: 10
     y: 20
  - B: bar
  - C: *a1
#
expected: |
  [{"A"=>{"x"=>10, "y"=>20}}, {"B"=>"bar"}, {"C"=>{"x"=>10, "y"=>20}}]
#
---
name:  anchor3
desc:  anchor on child node
#
input: |
  - A: &a1
     x: 10
     y: 20
     z: *a1
#
expected: |
  [{"A"=>{"x"=>10, "y"=>20, "z"=>{...}}}]
#
---
name:  anchor4
desc:  backward anchor
#
input: |
  - *a1
  - *a1
  - foo
  - &a1 bar
#
expected: |
  ["bar", "bar", "foo", "bar"]
#
---
name:  anchor5
desc:  anchor not found
error: Kwalify::YamlSyntaxError
mesg:  anchor 'a2' not found (Kwalify::YamlSyntaxError)
#
input: |
  - &a1 foo
  - bar
  - *a2
#
expected: |
#
---
name:  anchor6
desc:  anchor on child node
#
input: |
  type:        seq
  sequence:
    - &employee
      type:    map
      mapping:
        name:
          type:   str
        post:
          type:   str
          enum:
            - exective
            - manager
            - clerk
        supervisor: *employee
#
expected: |
  {"sequence"=>
    [{"type"=>"map",
      "mapping"=>
       {"name"=>{"type"=>"str"},
        "post"=>{"type"=>"str", "enum"=>["exective", "manager", "clerk"]},
        "supervisor"=>{...}}}],
   "type"=>"seq"}
#
---
name:  tag1
desc:  tag
#
input: |
  - !str 123
  - foo: !text 123
#
expected: |
  [123, {"foo"=>123}]
#
---
name:  docend1
desc:  ... (document end)
#
input: |
  - aaa
  - bbb
  ...
  - ccc
#
expected: |
  ["aaa", "bbb"]
#
---
name:  docend2
desc:  ... (document end) in block text
#
input: |
  - |
    foo
    ...
    bar
#
expected: |
  ["foo\n...\nbar\n"]
#
---
name:  docstart1
desc:  --- (document start)
#
input: |
  # comment
  ---
  - foo
  - bar
#
expected: |
  ["foo", "bar"]
#
---
name:  docstart2
desc:  --- (document start) with tag
#
input: |
  # comment
  --- %YAML !seq
  - foo
  - bar
#
expected: |
  ["foo", "bar"]
#
---
name:  docstart3
desc:  --- (document start) with tag
#
input: |
  - |
    foo
    ---
    bar
    ---
    baz
#
expected: |
  ["foo\n---\nbar\n---\nbaz\n"]
#
---
name:  default1
desc:  map default value
#
input: |
  A: 10
  B: 20
  =: 0
#
expected: |
  {"A"=>10, "B"=>20}
#
---
name:  merge1
desc:  merge key '<<'
#
input: |
  - &a1
    A: 10
    B: 20
  - C: 30
    <<: *a1
    A: ~
#
expected: |
  [{"A"=>10, "B"=>20}, {"A"=>nil, "B"=>20, "C"=>30}]
---
name:  example1
desc:  complex combination of seq and map
#
input: |
  columns:
    - name:       id
      type:       integer
      primary-key: yes
      sequence:   yes

    - name:       name
      type:       string
      not-null:   yes
      width:      63

    - name:       desc
      type:       string

    - name:       email
      type:       string
      width:      63

    - name:       birth
      type:       date

    - name:       age
      type:       integer

    - name:       last_update
      type:       timestamp

    - name:       gender
      type:       string
      values:
        - M
        - F

  tables:
    - name:               groups
      class:              Group
      desc:               Group master table
      columns:
        - name:           id
        - name:           name
        - name:           desc

    - name:               users
      class:              User
      desc:               User master table
      columns:
        - name:           id
        - name:           name
        - name:           desc
        - name:           email
        - name:           group_id
          type:           ref
          ref:            groups.id
        - name:           account
          type:           string
          maxlength:      31
          not-null:       yes
        - name:           password
          not-null:       yes
#
expected: |
  {"columns"=>
    [{"name"=>"id", "sequence"=>true, "type"=>"integer", "primary-key"=>true},
     {"name"=>"name", "not-null"=>true, "type"=>"string", "width"=>63},
     {"name"=>"desc", "type"=>"string"},
     {"name"=>"email", "type"=>"string", "width"=>63},
     {"name"=>"birth", "type"=>"date"},
     {"name"=>"age", "type"=>"integer"},
     {"name"=>"last_update", "type"=>"timestamp"},
     {"name"=>"gender", "type"=>"string", "values"=>["M", "F"]}],
   "tables"=>
    [{"columns"=>[{"name"=>"id"}, {"name"=>"name"}, {"name"=>"desc"}],
      "name"=>"groups",
      "class"=>"Group",
      "desc"=>"Group master table"},
     {"columns"=>
       [{"name"=>"id"},
        {"name"=>"name"},
        {"name"=>"desc"},
        {"name"=>"email"},
        {"name"=>"group_id", "type"=>"ref", "ref"=>"groups.id"},
        {"name"=>"account", "not-null"=>true, "type"=>"string", "maxlength"=>31},
        {"name"=>"password", "not-null"=>true}],
      "name"=>"users",
      "class"=>"User",
      "desc"=>"User master table"}]}
#
---
name:  example2
desc:  invoice
#
input: |
  invoice: 34843
  date   : 2001-01-23
  bill-to: &id001
      given  : Chris
      family : Dumars
      address:
          lines: |
              458 Walkman Dr.
              Suite #292
          city    : Royal Oak
          state   : MI
          postal  : 48046
  ship-to: *id001
  product:
      - sku         : BL394D
        quantity    : 4
        description : Basketball
        price       : 450.00
      - sku         : BL4438H
        quantity    : 1
        description : Super Hoop
        price       : 2392.00
  tax  : 251.42
  total: 4443.52
  comments: >
      Late afternoon is best.
      Backup contact is Nancy
      Billsmer @ 338-4338.
#
expected: |
  {"tax"=>251.42,
   "bill-to"=>
    {"given"=>"Chris",
     "family"=>"Dumars",
     "address"=>
      {"postal"=>48046,
       "city"=>"Royal Oak",
       "lines"=>"458 Walkman Dr.\nSuite #292\n",
       "state"=>"MI"}},
   "comments"=>
    "Late afternoon is best. Backup contact is Nancy Billsmer @ 338-4338.\n",
   "invoice"=>34843,
   "date"=>#<Date: 4903865/2,0,2299161>,
   "total"=>4443.52,
   "product"=>
    [{"price"=>450.0,
      "quantity"=>4,
      "description"=>"Basketball",
      "sku"=>"BL394D"},
     {"price"=>2392.0,
      "quantity"=>1,
      "description"=>"Super Hoop",
      "sku"=>"BL4438H"}],
   "ship-to"=>
    {"given"=>"Chris",
     "family"=>"Dumars",
     "address"=>
      {"postal"=>48046,
       "city"=>"Royal Oak",
       "lines"=>"458 Walkman Dr.\nSuite #292\n",
       "state"=>"MI"}}}
#
