#!/usr/bin/ruby

require 'test/unit'
require 'test/unit/ui/console/testrunner'

## --------------------

if test(?f, './kwartz')
    file = './kwartz'
elsif test(?f, '../kwartz')
    file = '../kwartz'
else
    $stderr.print "*** file 'kwartz' not found.\n"
    exit 1
end

if test(?f, './kwartz.rb')
    libpath = '.'
elsif test(?f, '../kwartz.rb')
    libpath = '..'
else
    $stderr.print "*** file 'kwartz.rb' not found.\n"
    exit 1
end

#$command = file
$command = "ruby -I#{libpath} #{file}"


### ====================
### argument test
### ====================
class ArgumentTest < Test::Unit::TestCase
    def setup
	@command = $command
    end
    
    def test_arg01	# -v + -h == -vh
	out_v  = `#{@command} -v 2>&1`
	out_h  = `#{@command} -h 2>&1`
	out_vh = `#{@command} -hv 2>&1`
	assert_equal(out_v + "\n" + out_h, out_vh)
    end

    def test_arg02	# invalid option
	output = `#{@command} -hoge 2>&1`
	expect = "kwartz: -o: invalid option.\n"
	assert_equal(expect, output)
    end

    def test_arg03	# argument required
	output = `#{@command} -a 2>&1`
	expect = "kwartz: -a: argument reguired.\n"
	assert_equal(expect, output)
    end
    
    def test_arg04	# invalid action
	output = `#{@command} -sal convert 2>&1`
	expect = "kwartz: -a l: invalid action.\n"
	assert_equal(expect, output)
    end

    def test_arg05	# invalid action
	output = `#{@command} -slperl 2>&1`
	expect = "kwartz: -l perl: unsupported language name.\n"
	assert_equal(expect, output)
    end
end


### ====================
### action test
### ====================
class ActionTest < Test::Unit::TestCase

    pdata = <<END
<tr id="mark:list">
 <td id="value:item">foo</td>
</tr>
END

    pdata2 = <<END
<tr id="list">
 <td id="value:item">foo</td>
</tr>
END

    plogic = <<END
:elem(list)
  :foreach(item = list)
    @stag
    @cont
    @etag
  :end
:end
END

    plogic_eruby = <<END
:elem(list)
  :foreach(item = list)
    @stag
    @cont
    @etag
  :end
:end

:macro(cont_list)
% if __lang__ == 'erb'
::: <td><<% %>%= CGI.escapeHTML(item) %<% %>></td>
% elsif __lang__ == 'php'
::: <td><?= htmlspecialchars($item) ?></td>
% else
  :print(' <td>', E(item), "</td>\n")
% end
:end
END

    convert = <<'END'
:macro(stag_list)
  :print("<tr>\n")
:end

:macro(cont_list)
  :print(" <td>", item, "</td>\n")
:end

:macro(etag_list)
  :print("</tr>\n")
:end

:macro(elem_list)
  :expand(stag_list)
  :expand(cont_list)
  :expand(etag_list)
:end

:expand(elem_list)
END

    merge = convert + "\n" + plogic

    translate_ruby = <<'END'
for item in list do
  print "<tr>\n"
  print " <td>", item, "</td>\n"
  print "</tr>\n"
end
END

    translate_ruby_s = <<'END'
for item in list do
  print "<tr>\n"
  print " <td>", CGI.escapeHTML((item).to_s), "</td>\n"
  print "</tr>\n"
end
END

    translate_php = <<'END'
<?php foreach ($list as $item) { ?>
<tr>
 <td><?php echo $item; ?></td>
</tr>
<?php } ?>
END

    translate_php_s = <<'END'
<?php foreach ($list as $item) { ?>
<tr>
 <td><?php echo htmlspecialchars($item); ?></td>
</tr>
<?php } ?>
END

    translate_jsp = <<'END'
<%@ taglib prefix="c" uri="http://java.sun.com/jstl/core" %>
<c:forEach var="item" items="${list}">
<tr>
 <td><c:out value="${item}" escapeXml="false"/></td>
</tr>
</c:forEach>
END

    translate_jsp_s = <<'END'
<%@ taglib prefix="c" uri="http://java.sun.com/jstl/core" %>
<c:forEach var="item" items="${list}">
<tr>
 <td><c:out value="${item}"/></td>
</tr>
</c:forEach>
END

    translate_ruby2 = <<'END'
for item in list do
  _s << "<tr>\n"
  _s << " <td>" << (item).to_s << "</td>\n"
  _s << "</tr>\n"
end
END

    translate_ruby2_s = <<'END'
for item in list do
  _s << "<tr>\n"
  _s << " <td>" << CGI.escapeHTML((item).to_s) << "</td>\n"
  _s << "</tr>\n"
end
END

    ## --------------------

    @@contents = {
	:pdata        => pdata,
	:pdata2       => pdata2,
	:plogic       => plogic,
	:plogic_eruby => plogic_eruby,
	:convert      => convert,
	:merge        => merge,
	:out_ruby     => translate_ruby,
	:out_ruby_s   => translate_ruby_s,
	:out_php      => translate_php,
	:out_php_s    => translate_php_s,
	:out_jsp      => translate_jsp,
	:out_jsp_s    => translate_jsp_s,
	:out_ruby2    => translate_ruby2,
	:out_ruby2_s  => translate_ruby2_s,
    }
    
    basedir = 'test.d'
    Dir.mkdir(basedir) if !test(?d, basedir)
    prefix  = basedir + "/kwartz01."
    @@filenames = {}
    @@contents.each do |key, content|
	filename = prefix + key.id2name.gsub(/_/, '.')
	File.open(filename, 'w') { |f| f.print(content) }
	@@filenames[key] = filename
    end

    ## --------------------

    def setup
	@command = $command
	@f = @@filenames
	@c = @@contents
    end
    
    def test_action01	# convert
	output = `#{@command} -a convert #{@f[:pdata]} 2>&1`
	expect = @c[:convert]
	assert_equal(expect, output)
    end

    def test_action02	# translate
	output = `#{@command} -a translate #{@f[:merge]} 2>&1`
	expect = @c[:out_ruby]
	assert_equal(expect, output)
    end

    def test_action03	# compile (-p filename)
	output = `#{@command} -p #{@f[:plogic]} #{@f[:pdata]} 2>&1`
	expect = @c[:out_ruby]
	assert_equal(expect, output)
    end

    def test_action04	# compile (-s)
	output = `#{@command} -p #{@f[:plogic]} -sl php #{@f[:pdata]} 2>&1`
	expect = @c[:out_php_s]
	assert_equal(expect, output)
    end

    def test_action05	# compile (--escape=true)
	output = `#{@command} -p #{@f[:plogic]} -ljsp --escape=true #{@f[:pdata]} 2>&1`
	expect = @c[:out_jsp_s]
	assert_equal(expect, output)
    end

    def test_action06	# compile (--enable_eruby=true)
	output = `#{@command} -p #{@f[:plogic_eruby]} -lphp --enable_eruby=true #{@f[:pdata]} 2>&1`
	expect = @c[:out_php]
	expect = expect.gsub(/<\?php echo \$item; \?>/, '<?= htmlspecialchars($item) ?>')
	assert_equal(expect, output)
    end
    
    def test_action07	# convert (--delete_idattr=true)
	output = `#{@command} -p #{@f[:plogic_eruby]} -a convert --delete_idattr #{@f[:pdata2]} 2>&1`
	expect = @c[:convert]
	assert_equal(expect, output)
    end


    def test_action11	# compile (-l jsp)
	output = `#{@command} -p #{@f[:plogic]} -ljsp #{@f[:pdata]} 2>&1`
	expect = @c[:out_jsp]
	assert_equal(expect, output)
    end

    def test_action12	# compile (-l jsp --charset=EUC-JP)
	output = `#{@command} -p #{@f[:plogic]} -ljsp --charset=EUC-JP #{@f[:pdata]} 2>&1`
	expect = @c[:out_jsp]
	expect = "<%@ page contentType=\"text/html; charset=EUC-JP\" %>\n" + expect
	assert_equal(expect, output)
    end

    def test_action13	# compile (-l jsp --header='')
	output = `#{@command} -p #{@f[:plogic]} -ljsp --header='' #{@f[:pdata]} 2>&1`
	expect = @c[:out_jsp]
	expect.sub!(/[^\n]*\n/, '')
	assert_equal(expect, output)
    end

    #def test_action14	# compile (-l ruby2)
    #	output = `#{@command} -p #{@f[:plogic]} -lruby2 #{@f[:pdata]} 2>&1`
    #	expect = @c[:out_ruby2]
    #	assert_equal(expect, output)
    #end
    #
    #def test_action15	# compile (-l ruby2 --header='')
    #	output = `#{@command} -p #{@f[:plogic]} -lruby2 --header='' #{@f[:pdata]} 2>&1`
    #	expect = @c[:out_ruby2]
    #	expect.sub!(/[^\n]*\n/, '')
    #	assert_equal(expect, output)
    #end


    def test_action21	# analyze
	output = `#{@command} -p #{@f[:plogic]} -a analyze #{@f[:pdata]} 2>&1`
	expect = <<END
global variables: list
local variables:  item
END
	assert_equal(expect, output)
    end
    
end


### ====================
### :load() and id="load:..."
### ====================
class LoadTest < Test::Unit::TestCase

    pdata = <<'END'
  <table>
   <tr id="mark:item_list">
    <td id="value:item">foo</td>
   </tr>
  </table>
  <span id="load:'loadtest1.plogic'"/>
END

    plogic = <<'END'
END

    plogic1 = <<'END'
:elem(item_list)
  :foreach(item=list)
    @stag
    @cont
    @etag
  :end
:end
:load("loadtest2.plogic")
END

    plogic2 = <<'END'
:macro(BEGIN)
  :print("<html>\n")
  :print(" <body>\n")
:end
:macro(END)
  :print(" </body>\n")
  :print("</html>\n")
:end
END

    translate_erb_s = <<'END'
<html>
 <body>
  <table>
% for item in list do
   <tr>
    <td><%= html_escape(item) %></td>
   </tr>
% end
  </table>
 </body>
</html>
END

    Dir.mkdir('test.d') if !test(?d, 'test.d')
    File.open('test.d/loadtest.html',    'w') { |f| f.write(pdata) }
    File.open('test.d/loadtest1.plogic', 'w') { |f| f.write(plogic1) }
    File.open('test.d/loadtest2.plogic', 'w') { |f| f.write(plogic2) }
    File.open('test.d/loadtest2.out.erb','w') { |f| f.write(translate_erb_s) }
    @@expect = translate_erb_s

    ## --------------------

    def setup
	@command = $command
	@pdata = 'test.d/loadtest.html'
    end
    
    def test_load	# id="load:..." and :load() statemtnt
	output = `#{@command} -slerb --load_path=.,test.d #{@pdata} 2>&1`
	expect = @@expect
	assert_equal(expect, output)
    end
end


### ====================
### batch mode
### ====================
class BatchmodeTest < Test::Unit::TestCase
    @@pdata = <<'END'
<ul id="list">
  <li id="value:item">foo</li>
</ul>
END

    @@plogic = <<'END'
:elem(list)
  @stag
  :foreach(item = list)
    @cont
  :end
  @etag
:end
END

    
    @@expected1 = <<'END'
<ul id="list">
% foreach item in list do
  <li><%= item %></li>
% end
</ul>
END
    @@expected0 = @@expected1.gsub(/^%.*\n/, '')
    @@expected2 = @@expected1.gsub(/<%= (\w+) %>/, '<%= CGI.escapeHTML((\1).to_s) %>')

    Dir.mkdir('test.d') unless test(?d, 'test.d')
    def create_files()
	File.open('test.d/batch1.html',    'w') { |f| f.print(@@pdata)  }
	File.open('test.d/batch1.plogic',  'w') { |f| f.print(@@plogic) }
	File.open('test.d/batch2.html',    'w') { |f| f.print(@@pdata)  }
	File.open('test.d/batch2.plogic',  'w') { |f| f.print(@@plogic) }
	File.open('test.d/batch3.html',    'w') { |f| f.print(@@pdata)  }
	File.open('test.d/batch3.plogic',  'w') { |f| f.print(@@plogic) }
    end
    
    File.open('test.d/batch.expected0', 'w') { |f| f.print(@@expected0) }
    File.open('test.d/batch.expected1', 'w') { |f| f.print(@@expected1) }
    File.open('test.d/batch.expected2', 'w') { |f| f.print(@@expected2) }

    ### --------------------
    
    def _touch(filename)
	File.open(filename, 'r+') do |file|
	    file.flock(File::LOCK_EX)
	    file.seek(0, IO::SEEK_SET)
	    ch = file.getc
	    file.seek(0, IO::SEEK_SET)
	    file.putc(ch)
	end
    end

    def setup
	@command = $command
	create_files()
    end
    
    def test_batch01
	sleep(1)
	_touch('test.d/batch1.html')
	_touch('test.d/batch2.html')
	_touch('test.d/batch3.html')
	output = `#{@command} -O .erb test.d/batch?.html 2>&1`
	expect = <<END
[OK] test.d/batch1.html => test.d/batch1.erb
[OK] test.d/batch2.html => test.d/batch2.erb
[OK] test.d/batch3.html => test.d/batch3.erb
END
	assert_equal(expect, output)
    end

    def test_batch02
	sleep(1)
	_touch('test.d/batch1.erb')
	_touch('test.d/batch2.erb')
	_touch('test.d/batch3.erb')
	output = `#{@command} -O .erb test.d/batch?.html 2>&1`
	expect = <<END
[OK] test.d/batch1.html => test.d/batch1.erb
[OK] test.d/batch2.html => test.d/batch2.erb
[OK] test.d/batch3.html => test.d/batch3.erb
END
	assert_equal(expect, output)
    end

    def test_batch03
	sleep(1)
	_touch('test.d/batch1.html')
	_touch('test.d/batch2.html')
	_touch('test.d/batch3.html')
	output = `#{@command} -O .erb -P.plogic test.d/batch?.html 2>&1`
	expect = <<END
[OK] test.d/batch1.html + test.d/batch1.plogic => test.d/batch1.erb
[OK] test.d/batch2.html + test.d/batch2.plogic => test.d/batch2.erb
[OK] test.d/batch3.html + test.d/batch3.plogic => test.d/batch3.erb
END
	assert_equal(expect, output)
    end

    def test_batch04
	sleep(1)
	_touch('test.d/batch1.erb')
	_touch('test.d/batch2.erb')
	_touch('test.d/batch3.erb')
	_touch('test.d/batch1.plogic')
	_touch('test.d/batch2.plogic')
	_touch('test.d/batch3.plogic')
	output = `#{@command} -O .erb -P.plogic test.d/batch?.html 2>&1`
	expect = <<END
[OK] test.d/batch1.html + test.d/batch1.plogic => test.d/batch1.erb
[OK] test.d/batch2.html + test.d/batch2.plogic => test.d/batch2.erb
[OK] test.d/batch3.html + test.d/batch3.plogic => test.d/batch3.erb
END
	assert_equal(expect, output)
    end

    def test_batch05	# -S
	output = `#{@command} -O .erb -P.plogic -S test.d/batch?.html 2>&1`
	expect = ''
	assert_equal(expect, output)
    end

    def test_batch06	# some files are missing
	File.unlink('test.d/batch1.erb')
	File.unlink('test.d/batch2.plogic')
	File.unlink('test.d/batch3.erb')
	File.unlink('test.d/batch3.plogic')
	output = `#{@command} -O .erb -P.plogic test.d/batch?.html 2>&1`
	expect = <<END
[OK] test.d/batch1.html + test.d/batch1.plogic => test.d/batch1.erb
[OK] test.d/batch2.html => test.d/batch2.erb
[OK] test.d/batch3.html => test.d/batch3.erb
END
	assert_equal(expect, output)
    end

    def test_batch11	# -T
	sleep(1)
	_touch('test.d/batch1.erb')
	_touch('test.d/batch2.html')	# html is newer
	_touch('test.d/batch3.erb')
	output = `#{@command} -O .erb -T test.d/batch?.html 2>&1`
	expect = <<END
[--] test.d/batch1.erb is newer than test.d/batch1.html
[OK] test.d/batch2.html => test.d/batch2.erb
[--] test.d/batch3.erb is newer than test.d/batch3.html
END
	assert_equal(expect, output)
    end

    def test_batch12	# -T -P.plogic
	_touch('test.d/batch1.plogic')
	_touch('test.d/batch3.html')
	sleep(1)
	_touch('test.d/batch1.erb')
	_touch('test.d/batch3.erb')
	sleep(1)
	_touch('test.d/batch1.html')		# html is newest
	_touch('test.d/batch2.erb')		# erb is newest
	_touch('test.d/batch3.plogic')		# plogic is newest
	output = `#{@command} -O .erb -P.plogic -T test.d/batch?.html 2>&1`
	expect = <<END
[OK] test.d/batch1.html + test.d/batch1.plogic => test.d/batch1.erb
[--] test.d/batch2.erb is newer than test.d/batch2.html and test.d/batch2.plogic
[OK] test.d/batch3.html + test.d/batch3.plogic => test.d/batch3.erb
END
	assert_equal(expect, output)
    end

    def test_batch13	# -T -P.plogic, and some files are missing
	File.unlink('test.d/batch1.plogic')
	File.unlink('test.d/batch2.plogic')
	sleep(1)
	_touch('test.d/batch1.html')
	_touch('test.d/batch2.erb')
	output = `#{@command} -O .erb -P.plogic -T test.d/batch?.html 2>&1`
	expect = <<END
[OK] test.d/batch1.html => test.d/batch1.erb
[--] test.d/batch2.erb is newer than test.d/batch2.html
[OK] test.d/batch3.html + test.d/batch3.plogic => test.d/batch3.erb
END
	assert_equal(expect, output)
    end

end


suite = Test::Unit::TestSuite.new
suite << ArgumentTest.suite
suite << ActionTest.suite
suite << LoadTest.suite
#suite << BatchTest.suite
suite << BatchmodeTest.suite
Test::Unit::UI::Console::TestRunner.run(suite)

__END__

### ====================
### batch mode
### ====================
#class BatchTest < Test::Unit::TestCase
#
#    def _touch(filename)
#	File.open(filename, 'r+') do |file|
#	    file.flock(File::LOCK_EX)
#	    file.seek(0, IO::SEEK_SET)
#	    ch = file.getc
#	    file.seek(0, IO::SEEK_SET)
#	    file.putc(ch)
#	end
#    end
#
#    def _test_batchmode(opts, flag_verbose=true)
#	input_fnames = opts[:input_fnames]
#	command = opts[:cmd_pattern] % input_fnames.join(' ')
#	if opts[:flag_touch] then
#	    input_fnames.each do |fname|
#		_touch(fname)
#	    end
#	end
#	#print "*** debug: command=`#{command}`\n"
#	log = `#{command}`
#	#print "--- result ------\n"
#	#print log
#	#print "-----------------\n"
#	expected_log = ''
#	input_fnames.each do |input_fname|
#	    base = input_fname.sub(/\.\w+$/, '')
#	    output_fname = base + opts[:output_suffix]
#	    expect_fname = base + opts[:expect_suffix]
#	    plogic_fname = opts[:plogic_fname]
#	    plogic_fname ||= base + opts[:plogic_suffix] if opts[:plogic_suffix]
#	    output = File.open(output_fname) { |f| f.read() }
#	    expect = File.open(expect_fname) { |f| f.read() }
#	    #print "*** debug: output_fname=#{output_fname}\n"
#	    #print "*** debug: expect_fname=#{expect_fname}\n"
#	    if flag_verbose then
#		assert_equal(expect, output)
#	    else
#		assert(expect == output)
#	    end
#	    expected_log << opts[:log_pattern] % [input_fname, output_fname, plogic_fname]
#	end
#	if flag_verbose then
#	    assert_equal(expected_log, log)
#	else
#	    assert(expected_log == log)
#	end
#	#assert(expected_log == log)
#	#print "--- expected ----\n"
#	#print expected_log
#	#print "-----------------\n"
#	#print "===\n"
#    end
#
#    FilePattern    = "test.d/compile??.in"
#    PlogicSuffix   = '.plogic'
#    InputFilenames = {}
#    InputFilenames[:all] = Dir.glob(FilePattern).sort
#    InputFilenames[:plogic_ok] = []
#    InputFilenames[:plogic_ng] = []
#    InputFilenames[:all].each do |input_fname|
#	plogic_fname = input_fname.sub(/.\w+$/, PlogicSuffix)
#	if test(?f, plogic_fname) then
#	    str = File.open(plogic_fname) { |f| f.read }
#	    if !str.empty? then
#		InputFilenames[:plogic_ok] << input_fname
#		next
#	    end
#	end
#	InputFilenames[:plogic_ng] << input_fname
#    end
#    LogPatternA1    = "[OK] %1$s => %2$s\n"
#    LogPatternA2    = "[OK] %1$s + %3$s => %2$s\n"
#    LogPatternB1    = "[--] '%2$s' is newer than '%1$s'.\n"
#    LogPatternB2    = "[--] '%2$s' is newer than '%1$s' and '%3$s'.\n"
#    if test(?f, './kwartz')
#	TestCommand = 'ruby ./kwartz'
#    elsif test(?f, '../kwartz')
#	TestCommand = 'ruby -I.. ../kwartz'
#    else
#	TestCommand = 'kwartz'
#    end
#
#    ### --------------------
#    ### batch mode (-O and -P)
#    ### --------------------
#    @@opts1 = {}
#    @@opts1[:cmd_pattern]     = "#{TestCommand} -O .out.ruby %s -l ruby --load_path=.,test.d %%s 2>&1"
#    @@opts1[:input_fnames]    = InputFilenames[:plogic_ng]
#    @@opts1[:log_pattern]     = LogPatternA1
#    @@opts1[:flag_touch]      = true
#    @@opts1[:output_suffix]   = '.out.ruby'
#    @@opts1[:expect_suffix]   = '.exp.ruby'
#    #@@opts1[:plogic_suffix]   = PlogicSuffix
#
#    def test_batch01
#	opts = @@opts1.dup
#	opts[:cmd_pattern] = @@opts1[:cmd_pattern] % ''
#	opts[:log_pattern] = LogPatternA1
#	opts[:flag_touch]  = true
#	sleep(1)
#	_test_batchmode(opts)
#    end
#    
#    def test_batch02
#	opts = @@opts1.dup
#	opts[:cmd_pattern] = @@opts1[:cmd_pattern] % ''
#	opts[:log_pattern] = LogPatternA1
#	opts[:flag_touch]  = false
#	sleep(1)
#	_test_batchmode(opts)
#    end
#
#    def test_batch03
#	opts = @@opts1.dup
#	opts[:cmd_pattern] = @@opts1[:cmd_pattern] % '-T'
#	opts[:log_pattern] = LogPatternA1
#	opts[:flag_touch]  = true
#	sleep(1)
#	_test_batchmode(opts)
#    end
#
#    def test_batch04
#	opts = @@opts1.dup
#	opts[:cmd_pattern] = @@opts1[:cmd_pattern] % '-T'
#	opts[:log_pattern] = LogPatternB1
#	opts[:flag_touch]  = false
#	sleep(1)
#	_test_batchmode(opts)
#    end
#
#    
#    ### --------------------
#    ### batch mode (-O, -P and -T)
#    ### --------------------
#    @@opts2 = {}
#    @@opts2[:cmd_pattern]     = "#{TestCommand} -O .out.erb -P #{PlogicSuffix} %s -l erb %%s 2>&1"
#    @@opts2[:input_fnames]    = InputFilenames[:plogic_ok]
#    @@opts2[:log_pattern]     = LogPatternA2
#    @@opts2[:flag_touch]      = true
#    @@opts2[:output_suffix]   = '.out.erb'
#    @@opts2[:expect_suffix]   = '.exp.erb'
#    @@opts2[:plogic_suffix]   = PlogicSuffix
#
#    def test_batch11
#	opts = @@opts2.dup
#	opts[:cmd_pattern] = @@opts2[:cmd_pattern] % ''
#	#opts[:log_pattern] = LogPatternA2
#	opts[:flag_touch] = true
#	sleep(1)
#	_test_batchmode(opts)
#    end
#
#    def test_batch12
#	opts = @@opts2.dup
#	opts[:cmd_pattern] = @@opts2[:cmd_pattern] % ''
#	#opts[:log_pattern] = LogPatternA2
#	opts[:flag_touch]  = false
#	sleep(1)
#	_test_batchmode(opts)
#    end
#
#    def test_batch13
#	opts = @@opts2.dup
#	opts[:cmd_pattern] = @@opts2[:cmd_pattern] % '-T'
#	#opts[:log_pattern] = LogPatternA2
#	opts[:flag_touch]  = true
#	sleep(1)
#	_test_batchmode(opts)
#    end
#
#    def test_batch14
#	opts = @@opts2.dup
#	opts[:cmd_pattern] = @@opts2[:cmd_pattern] % '-T'
#	opts[:log_pattern] = LogPatternB2
#	opts[:flag_touch]  = false
#	sleep(1)
#	_test_batchmode(opts)
#    end
#
#    
#
#    ### --------------------
#    ### sanitizing (-s)
#    ### --------------------
#    @@opts3 = {}
#    @@opts3[:cmd_pattern]   = "#{TestCommand} -O .out.php.s -P #{PlogicSuffix} -s -l php %s 2>&1"
#    @@opts3[:input_fnames]  = InputFilenames[:plogic_ok]
#    @@opts3[:log_pattern]   = LogPatternA2
#    @@opts3[:flag_touch]    = true
#    @@opts3[:output_suffix] = '.out.php.s'
#    @@opts3[:expect_suffix] = '.exp.php.s'
#    @@opts3[:plogic_suffix] = PlogicSuffix
#
#    def test_sanitize1
#	opts = @@opts3.dup
#	opts[:log_pattern] = LogPatternA2
#	opts[:flag_touch]  = true
#	sleep(1)
#	_test_batchmode(opts)
#    end
#
#    def test_sanitize2
#	opts = @@opts3.dup
#	opts[:log_pattern] = LogPatternA2
#	opts[:flag_touch]  = false
#	sleep(1)
#	_test_batchmode(opts)
#    end
#
#
#    ### --------------------
#    ### character set (-l jsp and --charset=xxx)
#    ### --------------------
#
#    JspHeader1 = "<%@ taglib prefix=\"c\" uri=\"http://java.sun.com/jstl/core\" %>\n"
#    JspHeader2 = "<%%@ page contentType=\"text/html; charset=%s\" %%>\n"
#
#    def _test_jspmode(charset=nil, flag_verbose=false)
#	basename = 'test.d/compile02'
#	input_filename  = "#{basename}.in"
#	plogic_filename = "#{basename}#{PlogicSuffix}"
#	output_filename = "#{basename}.out.jsp"
#	expect_filename = "#{basename}.exp.jsp"
#	opt =  charset ? "--charset=#{charset}" : ''
#	cmd = "#{TestCommand} -l jsp #{opt} -O .out.jsp"
#	cmd << " -p #{plogic_filename} #{input_filename} 2>&1"
#	log = `#{cmd}`
#	output = File.open(output_filename) { |f| f.read() }
#	expect = File.open(expect_filename) { |f| f.read() }
#	header = charset ? ((JspHeader2 % charset) + JspHeader1) : JspHeader1
#	expect = header + expect
#	#flag_verbose ? assert_equal(expect, output) : assert(expect == output)
#	if flag_verbose && expect != output then
#	    print "--- expect ----\n"
#	    print expect
#	    print "--- output ----\n"
#	    print output
#	    print "---------------\n"
#	end
#	assert(expect == output)
#    end
#
#    def test_jspmode1
#	_test_jspmode(nil, true)
#    end
#    
#    def test_jspmode2
#	_test_jspmode('EUC-JP')
#    end
#
#end
