/***************************************************************************
 *   Copyright (C) 2003 by Stephen Allewell                                *
 *   stephen@mirramar.fsnet.co.uk                                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#include <qfile.h>
#include <qfileinfo.h>
#include <qdir.h>
#include <qxml.h>
#include <kstandarddirs.h>
#include <kapplication.h>
#include <klocale.h>
#include <kmessagebox.h>
#include "schememanager.h"
#include "schemeparser.h"
#include "flossscheme.h"
#include "floss.h"

SchemeManager::SchemeManager()
{
  m_flossSchemes.setAutoDelete(true);
  refresh();
}

FlossScheme* SchemeManager::createScheme(QString schemeName)
{
  FlossScheme* flossScheme = 0;
  if (scheme(schemeName) == 0)
  {
    flossScheme = new FlossScheme;
    if (flossScheme)
    {
      flossScheme->setSchemeName(schemeName);
      m_flossSchemes.append(flossScheme);
    }
  }
  return flossScheme;
}

QStringList SchemeManager::schemes()
{
  QStringList schemeNames;
  for (FlossScheme *flossScheme = m_flossSchemes.first() ; flossScheme ; flossScheme = m_flossSchemes.next())
    schemeNames.append(flossScheme->schemeName());
  return schemeNames;
}

FlossScheme* SchemeManager::scheme(QString name)
{
  FlossScheme *flossScheme;
  for (flossScheme = m_flossSchemes.first() ; flossScheme ; flossScheme = m_flossSchemes.next())
  {
    if (flossScheme->schemeName() == name)
      break;
  }
  return flossScheme;
}

FlossScheme* SchemeManager::readScheme(QString name)
{
  SchemeParser handler;
  QFile xmlFile(name);
  QXmlInputSource source(xmlFile);
  QXmlSimpleReader reader;
  reader.setContentHandler(&handler);
  bool success = reader.parse(source);
  if (!success)
    KMessageBox::error(0,QString("Error reading scheme %1\n%2.").arg(name).arg(handler.errorString()),"Error reading floss scheme.");
  FlossScheme *flossScheme = handler.flossScheme();
  if (!success || scheme(flossScheme->schemeName()))
  {
    delete flossScheme;
    flossScheme = 0;
  }
  return flossScheme;
}

bool SchemeManager::writeScheme(QString name)
{
  QString writableDir = kapp->dirs()->saveLocation("appdata","schemes");
  if (writableDir.isNull())
  {
    KMessageBox::sorry(0,i18n("Unable to locate a writable directory\nto store the scheme."));
    // TODO Allow user to select a location to store the calibrated schemes
  }
  else
  {
    QFile schemeFile(QString("%1/%2.xml").arg(writableDir).arg(name.lower()));
    if (schemeFile.open(IO_WriteOnly))
    {
      QTextStream stream(&schemeFile);
      stream << "<!DOCTYPE flossScheme SYSTEM \"flossScheme.dtd\">\n<flossScheme>\n";
      stream << "<title>" << name << "</title>\n";
      FlossScheme *flossScheme = scheme(name);
      QPtrListIterator<Floss> it = flossScheme->flossIterator();
      for (Floss *floss = 0 ; (floss = it.current()) ; ++it)
      {
        stream << "<floss><name>" << floss->name << "</name><description>" << floss->description << "</description>";
        stream << "<color><red>" << floss->color.red() << "</red><green>" << floss->color.green() << "</green><blue>" << floss->color.blue() << "</blue></color></floss>\n";
      }
      stream << "</flossScheme>\n";
      schemeFile.close();
      return true;
    }
  }
  return false;
}

void SchemeManager::refresh()
{
  QStringList schemeList = kapp->dirs()->findAllResources("appdata", "schemes/*.xml");
  for (QStringList::Iterator it = schemeList.begin() ; it != schemeList.end() ; ++it)
  {
    FlossScheme *flossScheme = readScheme(*it);
    if (flossScheme)
      m_flossSchemes.append(flossScheme);
  }
}
