
// LabPlot : Plot.cc

#include <math.h>
#include <iostream>
#include <kdebug.h>
#include <klocale.h>
#include "Plot.h"
#include "Plot2DSurface.h"
#include "PlotQWT3D.h"

using namespace std;

//! general Plot class
Plot::Plot(Worksheet *w)
	: worksheet(w)
{
	kdDebug()<<"Plot::Plot()"<<endl;
	graphlist = new GraphList();

	bgcolor = QColor(Qt::white);
	gbgcolor = QColor(Qt::white);

	// title (richtext title)
	if(worksheet==0) kdDebug()<<"no Worksheet defined!"<<endl;
	QFont font = worksheet->getMainWin()->defaultFont();
	title = new Label(i18n("Title"),font,QColor(Qt::black));
	title->setPosition(0.4,0.04);

	font.setPointSize((int)(0.4*font.pointSize()));
	legend.setFont(font);
	
	position.setPoint(0.0,0.0);
	size.setPoint(1.0,1.0);
	p1.setPoint(.15,.15);
	p2.setPoint(.95,.85);

	baseline=0;
	baseline_enabled=0;
	xbaseline=0;
	xbaseline_enabled=0;

	region= new LRange(0.0,0.0);
	region_enabled=1;

	transparent=false;
	clipoffset=10;

	marks_enabled=false;
	markx = new LRange(0.0,1.0);
	marky = new LRange(0.0,1.0);

	kdDebug()<<"Plot OK"<<endl;
}

void Plot::drawStyle(QPainter *p, Style *style, Symbol *symbol, QPointArray pa,int xmin, int xmax,int ymin, int ymax) {
	kdDebug()<<"Plot::drawStyle()"<<endl;
	bool filled = style->isFilled();
	QColor c = style->FillColor();
 	QPen pen( style->Color(), style->Width(),(Qt::PenStyle)style->PenStyle() );
	p->setPen(pen);
	QBrush brush(c,(Qt::BrushStyle)style->Brush());

	// calculate baseline
	double min = actrange[1].rMin();
	double max = actrange[1].rMax();
	double minx = actrange[0].rMin();
	double maxx = actrange[0].rMax();

	int basey = ymax - (int) ((baseline-min)/(max-min)*(double)(ymax-ymin));
	int basex = xmin + (int) ((xbaseline-minx)/(maxx-minx)*(double)(xmax-xmin));
	int bw = style->BoxWidth();
	//kdDebug()<<"BASEX = "<<basex<<endl;
	//kdDebug()<<"BASEY = "<<basey<<endl;

//	kdDebug()<<"POINTARRAY "<<i<<'='<<pa[i].x()<<' '<<pa[i].y()<<endl;
		
	switch(style->Type()) {
	case 0:	// line
		if (filled) {
			QPointArray fillpa(pa.size()+2);
			p->setPen(Qt::NoPen);
			p->setBrush(brush);
			fillpa[0]=QPoint(pa[0].x(),basey);
			for(unsigned int i=0;i<pa.size();i++)
				fillpa[i+1]=pa[i];
			fillpa[pa.size()+1]=QPoint(pa[pa.size()-1].x(),basey);
			p->drawPolygon(fillpa);
			p->setPen(pen);
		}
		p->drawPolyline(pa);
		break;
	case 2:	{ // steps	: only for 2d
		QPointArray stepspa(2*pa.size());
		// start
		stepspa[0]=pa[0];		
		stepspa[1]=QPoint((pa[0].x()+pa[1].x())/2,pa[0].y());
		// end
		stepspa[2*pa.size()-2]=QPoint((int)((pa[pa.size()-2].x()+pa[pa.size()-1].x())/2.0),pa[pa.size()-1].y());
		stepspa[2*pa.size()-1]=pa[pa.size()-1];
		for(unsigned int i=1;i<pa.size()-1;i++) {
			stepspa[2*i]=QPoint((int)((pa[i-1].x()+pa[i].x())/2.0),pa[i].y());		// left
			stepspa[2*i+1]=QPoint((int)((pa[i+1].x()+pa[i].x())/2.0),pa[i].y());		// right
		}
		if (filled) {
			QPointArray fillpa(stepspa.size()+2);
			p->setPen(Qt::NoPen);
			p->setBrush(brush);
			fillpa[0]=QPoint(stepspa[0].x(),basey);
			for(unsigned int i=0;i<stepspa.size();i++)
				fillpa[i+1]=stepspa[i];
			fillpa[stepspa.size()+1]=QPoint(stepspa[stepspa.size()-1].x(),basey);
			p->drawPolygon(fillpa);
			p->setPen(pen);
		}
		p->drawPolyline(stepspa);
		}; break;
	case 3:	// boxes	: only for 2d
		for(unsigned int i=0;i<pa.size();i++) {
			QPointArray boxpa(4);
			
			if(style->AutoBoxWidth()) {
				if(i==0) {
					boxpa[0]=QPoint(pa[i].x()-(int)((pa[i+1].x()-pa[i].x())/2.0),pa[i].y());
					boxpa[1]=QPoint(pa[i].x()+(int)((pa[i+1].x()-pa[i].x())/2.0),pa[i].y());
					boxpa[2]=QPoint(pa[i].x()+(int)((pa[i+1].x()-pa[i].x())/2.0),basey);
					boxpa[3]=QPoint(pa[i].x()-(int)((pa[i+1].x()-pa[i].x())/2.0),basey);
				}
				else if(i==pa.size()-1) {
					boxpa[0]=QPoint(pa[i].x()-(int)((pa[i].x()-pa[i-1].x())/2.0),pa[i].y());
					boxpa[1]=QPoint(pa[i].x()+(int)((pa[i].x()-pa[i-1].x())/2.0),pa[i].y());
					boxpa[2]=QPoint(pa[i].x()+(int)((pa[i].x()-pa[i-1].x())/2.0),basey);
					boxpa[3]=QPoint(pa[i].x()-(int)((pa[i].x()-pa[i-1].x())/2.0),basey);
				}
				else {
					boxpa[0]=QPoint(pa[i].x()-(int)((pa[i].x()-pa[i-1].x())/2.0),pa[i].y());
					boxpa[1]=QPoint(pa[i].x()+(int)((pa[i+1].x()-pa[i].x())/2.0),pa[i].y());
					boxpa[2]=QPoint(pa[i].x()+(int)((pa[i+1].x()-pa[i].x())/2.0),basey);
					boxpa[3]=QPoint(pa[i].x()-(int)((pa[i].x()-pa[i-1].x())/2.0),basey);
				}
			}
			else {
				boxpa[0]=QPoint(pa[i].x()-bw/2,pa[i].y());
				boxpa[1]=QPoint(pa[i].x()+bw/2,pa[i].y());
				boxpa[2]=QPoint(pa[i].x()+bw/2,basey);
				boxpa[3]=QPoint(pa[i].x()-bw/2,basey);
			}

			p->setBrush(Qt::NoBrush);
			if (filled)
				p->setBrush(brush);
			p->drawPolygon(boxpa);
		}
		break;
	case 4:	// impulses	: only for 2d
		for (unsigned int i=0;i<pa.size();i++)
			p->drawLine(pa[i].x(),pa[i].y(),pa[i].x(),basey);
		break;
	case 5:	// yboxes	:only for 2d
		for(unsigned int i=0;i<pa.size();i++) {
			QPointArray boxpa(4);
			
			if(style->AutoBoxWidth()) {
				if(i==0) {
					boxpa[0]=QPoint(pa[i].x(),(int)(pa[i].y()-(pa[i+1].y()-pa[i].y())/2.0));
					boxpa[1]=QPoint(pa[i].x(),(int)(pa[i].y()+(pa[i+1].y()-pa[i].y())/2.0));
					boxpa[2]=QPoint(basex,pa[i].y()+(int)((pa[i+1].y()-pa[i].y())/2.0));
					boxpa[3]=QPoint(basex,pa[i].y()- (int)((pa[i+1].y()-pa[i].y())/2.0));
				}
				else if(i==pa.size()-1) {
					boxpa[0]=QPoint(pa[i].x(),pa[i].y()-(int)((pa[i].y()-pa[i-1].y())/2.0));
					boxpa[1]=QPoint(pa[i].x(),pa[i].y()+(int)((pa[i].y()-pa[i-1].y())/2.0));
					boxpa[2]=QPoint(basex,pa[i].y()+(int)((pa[i].y()-pa[i-1].y())/2.0));
					boxpa[3]=QPoint(basex, pa[i].y()-(int)((pa[i].y()-pa[i-1].y())/2.0));
				}
				else {
					boxpa[0]=QPoint(pa[i].x(),pa[i].y()-(int)((pa[i].y()-pa[i-1].y())/2.0));
					boxpa[1]=QPoint(pa[i].x(),pa[i].y()+(int)((pa[i+1].y()-pa[i].y())/2.0));
					boxpa[2]=QPoint(basex,pa[i].y()+(int)((pa[i+1].y()-pa[i].y())/2.0));
					boxpa[3]=QPoint(basex, pa[i].y()-(int)((pa[i].y()-pa[i-1].y())/2.0));
				}
			}
			else {
				boxpa[0]=QPoint(pa[i].x(),pa[i].y()-bw/2);
				boxpa[1]=QPoint(pa[i].x(),pa[i].y()+bw/2);
				boxpa[2]=QPoint(basex, pa[i].y()+bw/2);
				boxpa[3]=QPoint(basex, pa[i].y()-bw/2);
			}
			
			p->setBrush(Qt::NoBrush);
			if (filled)
				p->setBrush(brush);
			p->drawPolygon(boxpa);
		}
		break;
	}

	// draw symbol
	for (unsigned int i=0;i<pa.size();i++)
		symbol->draw(p,pa[i].x(),pa[i].y());
}


//! draw errorbars for x-y-dy, x-y-dx-dy and x-y-dy1-dy2
// at (x,y) from xleft to xright and ybottom to ytop
void Plot::drawErrorBar(QPainter *p, QPointArray pa, QPointArray hpa, QPointArray vpa) {
	for(unsigned int i=0;i<pa.size();i++) {
		int x=pa[i].x(), y=pa[i].y();
		int xleft=hpa[i].x(), xright=hpa[i].y();
		int ytop=vpa[i].x(),ybottom=vpa[i].y();
		
		if (xleft != x) {
			p->drawLine(xleft,y,x,y);
			p->drawLine(xleft,y-2,xleft,y+2);
		}
		if (xright != x) {
			p->drawLine(x,y,xright,y);
			p->drawLine(xright,y-2,xright,y+2);
		}
		if (ytop != y) {
			p->drawLine(x,y,x,ytop);
			p->drawLine(x-2,ytop,x+2,ytop);
		}
		if (ybottom != y) {
			p->drawLine(x,y,x,ybottom);
			p->drawLine(x-2,ybottom,x+2,ybottom);
		}
	}
}

void Plot::save(QTextStream *t) {
	kdDebug()<<"Plot::save()"<<endl;
	*t<<bgcolor.name()<<endl;
	*t<<gbgcolor.name()<<endl;
	*t<<transparent<<endl;
	*t<<clipoffset<<endl;

	for (int i=0;i<3;i++)
		*t<<actrange[i].rMin()<<' '<<actrange[i].rMax()<<endl;

	// save baselines
	*t<<baseline<<' '<<baseline_enabled<<' '<<xbaseline<<' '<<xbaseline_enabled<<endl;

	//save region
	*t<<region->rMin()<<' '<<region->rMax()<<' '<<region_enabled<<endl;

	*t<<position.X()<<' '<<position.Y()<<endl;
	*t<<size.X()<<' '<<size.Y()<<endl;
	*t<<p1.X()<<' '<<p1.Y()<<endl;
	*t<<p2.X()<<' '<<p2.Y()<<endl;
	
	//marks
	*t<<marks_enabled<<' '<<markx->rMin()<<' '<<markx->rMax()<<' '<<marky->rMin()<<' '<<marky->rMax()<<endl;

	title->save(t);
	legend.save(t);
	saveAxes(t);

	if (type == PSURFACE) {
		Plot2DSurface *plot = (Plot2DSurface *)this;
		*t<<plot->densityEnabled()<<' '<<plot->contourEnabled()<<endl;
		*t<<plot->Number()<<' '<<plot->Palette()<<endl;
		*t<<plot->ContourColor().name()<<endl;
		*t<<plot->Mesh()<<' '<<plot->ColoredContour()<<' '<<plot->Brush()<<' ';
		*t<<plot->Relative()<<' '<<plot->Threshold()<<endl;
	}
#ifdef HAVE_GL
	else if (type == PQWT3D) {
		PlotQWT3D *plot = (PlotQWT3D *)this;
		
		*t<<(int) plot->PlotStyle()<<endl;
		*t<<(int) plot->CoordinateStyle()<<endl;
		*t<<(int) plot->FloorStyle()<<endl;
		*t<<(int) plot->aspectRatio()<<endl;
		*t<<plot->isolines()<<endl;
		*t<<(int) plot->mouseDisabled()<<endl;
		for(int i=0;i<12;i++)
			*t<<plot->majorTicLength(i)<<' '<<plot->minorTicLength(i)<<endl;
		*t<<(int) plot->Resolution()<<endl;

		Qwt3D::ColorVector cv = plot->getColorVector();
		*t<<cv.size()<<endl;
		for(unsigned int i=0;i<cv.size();i++)
			*t<<cv[i].r<<' '<<cv[i].g<<' '<<cv[i].b<<' '<<cv[i].a<<endl;
	}
#endif

	// dump graphs
	QProgressDialog *progress = new QProgressDialog( i18n("Saving Project ..."), i18n("Cancel"), 100,
		worksheet->getMainWin(), "progress", true );
	progress->setMinimumDuration(1000);

	unsigned int i;
	for (i=0;i < graphlist->Number();i++) {
		switch (graphlist->getStruct(i)) {
		case GRAPH2D:
			*t<<"Graph2D "<<i<<endl;
			graphlist->getGraph2D(i)->save(t,progress);
			break;
		case GRAPH3D:
			*t<<"Graph3D "<<i<<endl;
			graphlist->getGraph3D(i)->save(t,progress);
			break;
		case GRAPHM:
			*t<<"GraphM "<<i<<endl;
			graphlist->getGraphM(i)->save(t,progress);
			break;
		case GRAPH4D:
			*t<<"Graph4D "<<i<<endl;
			graphlist->getGraph4D(i)->save(t,progress);
			break;
		case GRAPHIMAGE:
			*t<<"GraphIMAGE "<<i<<endl;
			graphlist->getGraphIMAGE(i)->save(t);
			break;
		default: break;
		}
	}
	*t<<"EOG "<<i<<endl;
}

void Plot::open(QTextStream *t, int version) {
	kdDebug()<<"Plot::open()"<<endl;

	QString family, color;
	double x,y;

	*t>>color;
	bgcolor = QColor(color);
	if(version > 3) {
		*t>>color;
		gbgcolor = QColor(color);

		// not used in newer versions
		if (version<11)
			*t>>x>>x>>x>>x;
		else {
			int tmp;
			*t>>tmp;
			transparent = (bool) tmp;
			if(version>13) {
				*t>>clipoffset;
			}
		}

		for (int i=0;i<3;i++) {
			*t>>x>>y;	// means min / max
			actrange[i].setMin(x);
			actrange[i].setMax(y);
			kdDebug()<<"	ActRange"<<i<<" min/max = "<<x<<' '<<y<<endl;
		}
		
		if(version>18) {
			int e1,e2;
			*t>>baseline>>e1>>xbaseline>>e2;
			baseline_enabled=(bool)e1;
			xbaseline_enabled=(bool)e2;
		}

		//region
		if (version>9) {
			int e;
			*t>>x>>y>>e;
			region->setMin(x);
			region->setMax(y);
			region_enabled=e;
			kdDebug()<<"Region : "<<x<<' '<<y<<' '<<e<<endl;
		}

		// position & size
		if(version>10) {
			*t>>x>>y;
			position.setPoint(x,y);
			*t>>x>>y;
			size.setPoint(x,y);
			*t>>x>>y;
			p1.setPoint(x,y);
			*t>>x>>y;
			p2.setPoint(x,y);
			
			if(version>19) {
				int e;
				double xmin, xmax, ymin,ymax;
				*t>>e>>xmin>>xmax>>ymin>>ymax;
				marks_enabled=e;
				markx->setMin(xmin);
				markx->setMax(xmax);
				marky->setMin(ymin);
				marky->setMax(ymax);
			}
		}
		else {
			position.setPoint(0,0);
			size.setPoint(1,1);
			p1.setPoint(.11,.15);
			p2.setPoint(.95,.85);
		}
	}

	// title
	kdDebug()<<"Opening title ..."<<endl;
	title->open(t,version);

	// legend
	kdDebug()<<"Opening legend ..."<<endl;
	legend.open(t,version);

	// axes
	kdDebug()<<"Opening axes ..."<<endl;
	openAxes(t,version);

	// plot type specific stuff
	if (type == PSURFACE && version > 5) {
		Plot2DSurface *plot = (Plot2DSurface *)this;
		int de, ce, n, p;
		QString tmp;
		*t>>de>>ce;
		plot->enableDensity(de);
		plot->enableContour(ce);
		*t>>n>>p;
		plot->setNumber(n);
		plot->setPalette(p);
		if (version>12) {
			*t>>tmp;
			plot->setContourColor(QColor(tmp));
			*t>>de>>ce>>n>>p>>tmp;
			plot->setMesh(de);
			plot->setColoredContour(ce);
			plot->setBrush(n);
			plot->setRelative(p);
			plot->setThreshold(tmp.toDouble());
		}
	}
	else if (type == PQWT3D && version > 17) {
		PlotQWT3D *plot = (PlotQWT3D *)this;
		int pstyle, cstyle, fstyle, ar, iso, mouse, resolution;
		double maj[12], min[12];
		*t>>pstyle;
		*t>>cstyle;
		*t>>fstyle;
		*t>>ar;
		*t>>iso;
		if(version > 20) {
			*t>>mouse;
			for(int i=0;i<12;i++)
				*t>>maj[i]>>min[i];
			*t>>resolution;
			kdDebug()<<"	RESOLUTION = "<<resolution<<endl;
		}
#ifdef HAVE_GL
		plot->setPlotStyle((Qwt3D::PLOTSTYLE) pstyle);
		plot->setCoordinateStyle((Qwt3D::COORDSTYLE) cstyle);
		plot->setFloorStyle((Qwt3D::FLOORSTYLE) fstyle);
		plot->setAspectRatio(ar);
		plot->setIsolines(iso);
		if(version > 20) {
			plot->disableMouse(mouse);
			for(int i=0;i<12;i++) {
				plot->setMajorTicLength(i,maj[i]);
				plot->setMinorTicLength(i,min[i]);
			}
			plot->setDataResolution(resolution);
		}

		//read colorvector
		Qwt3D::ColorVector cv;
		cv.clear();
		Qwt3D::RGBA rgb;
#endif
		
		int size;
		*t>>size;
		for(int i=0;i<size;i++) {
#ifdef HAVE_GL
			*t >> rgb.r >> rgb.g >> rgb.b>>rgb.a;
			cv.push_back(rgb);
#else
			int r,g,b;	// over read values
			*t>>r>>g>>b;
#endif
		}
#ifdef HAVE_GL
		plot->setColorVector(cv);
#endif
	}

	// get the data
	QString gstring;
	kdDebug()<<"Opening Graph ..."<<endl;
	QProgressDialog *progress = new QProgressDialog( i18n("Opening Project ..."), i18n("Cancel"), 100,
		worksheet->getMainWin(), "progress", true );
	progress->setMinimumDuration(1000);

	int n;
	*t>>gstring>>n;
	kdDebug()<<"string = "<<gstring<<" / n = "<<n<<endl;
	while (!strncmp("Graph",gstring,5) ) {
		kdDebug()<<" GRAPH "<<gstring<<' '<<n<<endl;

		if (gstring == "Graph2D") {
			Graph2D *g = new Graph2D();
			g->open(t,version,progress);
			worksheet->addGraph2D(g,type);
		}
		else if (gstring == "Graph3D") {
			Graph3D *g = new Graph3D();
			g->open(t,version,progress);
			worksheet->addGraph3D(g,type);
		}
		else if (gstring == "Graph4D") {
			Graph4D *g = new Graph4D();
			g->open(t,version,progress);
			worksheet->addGraph4D(g);
		}
		else if (gstring == "GraphM") {
			GraphM *g = new GraphM();
			g->open(t,version,progress);
			worksheet->addGraphM(g,type);
		}
		else if (gstring == "GraphIMAGE") {
			GraphIMAGE *g = new GraphIMAGE();
			g->open(t,version,progress);
			worksheet->addGraphIMAGE(g);
		}
		*t>>gstring>>n;
		kdDebug()<<" GRAPH LINE :  "<<gstring<<' '<<n<<endl;
	}
	kdDebug()<<"Plot::open() OK"<<endl;
}

// called from Plot<Type> with all axes
void Plot::saveAxis(QTextStream *t,Axis *axis,int gridenabled, int borderenabled, int minorgridenabled) {
	kdDebug()<<"Plot::saveAxis()"<<endl;
	*t<<axis->Scale()<<endl;

	//Grid & Border
	*t<<gridenabled<<' '<<borderenabled<<' '<<axis->enabled()<<endl;
	*t<<minorgridenabled<<endl;
	*t<<axis->majorGridColor().name()<<endl;
	*t<<axis->minorGridColor().name()<<endl;
	*t<<(int)axis->MajorGridType()<<' '<<(int)axis->MinorGridType()<<' '<<axis->borderWidth()<<endl;
	*t<<axis->majorGridWidth()<<' '<<axis->minorGridWidth()<<' '<<axis->majorTicsWidth()<<' '<<axis->minorTicsWidth()<<endl;

	axis->label()->save(t);

	*t<<axis->TicsPos()<<endl;
	*t<<axis->Scaling()<<' '<<axis->Shift()<<endl;
	*t<<axis->TicsLabelPrefix()<<endl;
	*t<<axis->TicsLabelSuffix()<<endl;
	*t<<axis->TicsLabelRotation()<<endl;
	*t<<axis->TicsLabelGap()<<endl;

	//Tics
	QFont tf = axis->TicsFont();
	*t<<tf.family()<<endl;
	*t<<tf.pointSize()<<' '<<tf.weight()<<' '<<tf.italic()<<endl;
	*t<<axis->MajorTics()<<' '<<axis->MinorTics()<<endl;
	*t<<axis->MajorTicsEnabled()<<' '<<axis->MinorTicsEnabled()<<endl;
	*t<<axis->TicsColor().name()<<endl;
	*t<<axis->TicsLabelColor().name()<<endl;
	*t<<axis->BorderColor().name()<<endl;
	*t<<axis->TicsLabelFormat()<<endl;
	*t<<axis->TicsLabelPrecision()<<endl;
	*t<<axis->DateTimeFormat()<<endl;
}

// called from Plot<Type> with all axes
void Plot::openAxis(QTextStream *t,int version,Axis *axis,bool *gridenabled,bool *borderenabled, bool *minorgridenabled) {
	kdDebug()<<"Plot::openAxis()"<<endl;
	QString family, color;
	int pointsize, weight, italic;
	double x,y;

	QString l;
	int major,minor;
	int majore,minore;
	int ge,be,e;
	int boxed=0;
	double rotation=0;
	int is_texlabel=0;

	int s=0;
	if(version > 7)
		*t>>s;
	axis->setScale((TScale)s);

	*t>>ge>>be>>e;
	*gridenabled=ge;
	*borderenabled=be;
	
	kdDebug()<<"GRID  enabled : "<<*gridenabled<<endl;
	kdDebug()<<"BORDER enabled : "<<*borderenabled<<endl;

	if(version > 14) {
		*t>>ge;
		*minorgridenabled=ge;
	}

	axis->enable(e);
	if (version > 3) {
		*t>>color;
		axis->setMajorGridColor(QColor(color));
		kdDebug()<<"MAJOR GRID COLOR = "<<color<<endl;
		if(version > 18) {
			int e1,e2,e3,e4;
			*t>>color;
			kdDebug()<<"MINOR GRID COLOR = "<<color<<endl;
			axis->setMinorGridColor(QColor(color));
			*t>>e1>>e2>>e3;
			axis->setMajorGridType((Qt::PenStyle)e1);
			axis->setMinorGridType((Qt::PenStyle)e2);
			axis->setBorderWidth(e3);
			*t>>e1>>e2>>e3>>e4;
			axis->setMajorGridWidth(e1);
			axis->setMinorGridWidth(e2);
			axis->setMajorTicsWidth(e3);
			axis->setMinorTicsWidth(e4);
			kdDebug()<<"GRID WIDTHs : "<<e1<<e2<<e3<<e4<<endl;
		}
	}
	
	l=t->readLine();	// needed. don't know why ...

	// open axis label
	l=t->readLine();

	kdDebug()<<"Label = "<<l<<endl;

	// Label::save() partly duplicated here !!!
	if (version > 6) {	// new order
		family=t->readLine();
		*t>>pointsize>>weight>>italic;
		*t>>color;
		*t>>x>>y;
		if(version > 8)
			*t>>boxed;
		if(version > 16)
			*t>>rotation;
		if(version > 20)
			*t>>is_texlabel;
	}
	else if (version > 3) {
		*t>>color;
		*t>>x>>y;
		axis->label()->setPosition(x,y);

		t->readLine();
		family=t->readLine();
		*t>>pointsize>>weight>>italic;
	}
	else {
		*t>>family>>pointsize>>weight>>italic;
	}

	kdDebug()<<"axis (Font) "<<family<<' '<<pointsize<<endl;

	Label *label = new Label(l,QFont(family,pointsize,weight,italic),QColor(color));
	label->setPosition(x,y);
	label->setBoxed(boxed);
	label->setRotation(rotation);
	label->setTeXLabel(is_texlabel);
	axis->setLabel(label);

	if (version > 10) {
		*t>>x;
		axis->setTicsPos((int)x);
		kdDebug()<<"	TIC POSITION = "<<x<<endl;
		*t>>x>>y;
		axis->setScaling(x);
		axis->setShift(y);
		kdDebug()<<"	SCALING/SHIFT = "<<x<<' '<<y<<endl;
		t->readLine();
		QString tmpprefix=t->readLine();
		axis->setTicsLabelPrefix(tmpprefix);
		kdDebug()<<"Prefix = "<<tmpprefix<<endl;
		QString tmpsuffix=t->readLine();
		axis->setTicsLabelSuffix(tmpsuffix);
		kdDebug()<<"Suffix = "<<tmpsuffix<<endl;
		if(version > 14) {
			*t>>x;
			axis->setTicsLabelRotation(x);
			if(version > 16) {
				*t>>x;
				axis->setTicsLabelGap((int)x);
			}
			t->readLine();
		}
	}

	// tics
	if (version > 3) {
		if (version<11)
			t->readLine();
		family=t->readLine();
		*t>>pointsize>>weight>>italic;
		axis->setTicsFont( QFont(family,pointsize,weight,italic));
	}

	kdDebug()<<"axis (Tics) "<<family<<pointsize<<endl;

	*t>>major>>minor;
	axis->setMajorTics(major);
	axis->setMinorTics(minor);
	*t>> majore>>minore;
	axis->enableMajorTics(majore);
	axis->enableMinorTics(minore);
	if (version > 3) {
		*t>>color;
		axis->setTicsColor(QColor(color));
		*t>>color;
		axis->setTicsLabelColor(QColor(color));
		*t>>color;
		axis->setBorderColor(QColor(color));
	}
	if (version > 4) {
		int tmp;
		*t>>tmp;
		axis->setTicsLabelFormat((TFormat)tmp);
		*t>>tmp;
		axis->setTicsLabelPrecision(tmp);
	}
	if(version >11) {
		QString tmp;
		t->readLine();
		tmp = t->readLine();
		axis->setDateTimeFormat(tmp);
	}

	kdDebug()<<"OK Axis : "<<l<<endl;
}

void Plot::autoScaleX() {
	TScale scale = ((Plot2D *)this)->getAxis(0)->Scale();
	double min=range[0].rMin(), max=range[0].rMax();

	worksheet->checkRanges(scale,&min,&max);
	
	actrange[0].setMin(min);
	actrange[0].setMax(max);
}

void Plot::autoScaleY() {
	TScale scale = ((Plot2D *)this)->getAxis(1)->Scale();
	double min=range[1].rMin(), max=range[1].rMax();

	worksheet->checkRanges(scale,&min,&max);
	
	actrange[1].setMin(min);
	actrange[1].setMax(max);
}

void Plot::autoScaleZ() {
	TScale scale = ((Plot2D *)this)->getAxis(2)->Scale();
	double min=range[2].rMin(), max=range[2].rMax();

	worksheet->checkRanges(scale,&min,&max);
	
	actrange[2].setMin(min);
	actrange[2].setMax(max);
}
	
//! build the tic label string according to atlf
QString Plot::TicLabel(int atlf, int prec, QString dtf, double value) {
	QString label;

	switch(atlf) {
	case AUTO:
		label = QString::number(value,'g',prec);
		break;
	case NORMAL:
		label = QString::number(value,'f',prec);
		break;
	case SCIENTIFIC:
		label = QString::number(value,'e',prec);
		break;
	case POWER10:
		label = "10<span style=\"vertical-align:super\">"+ QString::number(log10(value),'g',prec)+"</span>";
		break;
	case POWER2:
		label = "2<span style=\"vertical-align:super\">"+ QString::number(log2(value),'g',prec)+"</span>";
		break;
	case POWERE:
		label = "e<span style=\"vertical-align:super\">"+ QString::number(log(value),'g',prec)+"</span>";
		break;
	case FSQRT:
		label = "sqrt("+ QString::number(value*value,'g',prec) + ")";
		break;
	case TIME: {
		QTime time;
		time=time.addMSecs((int) (value*1000));
			
		QString format;
		if(fabs(value)<1)
			format="z";
		else if(fabs(value)<10) {
			format="s.zzz";
			if (prec==0)
				format="s";
			else if (prec==1) {
				// round to 100 ms
				int ms=time.msec();
				time=time.addMSecs(-ms);
				ms = 100*(int)rint(ms/100);
				time=time.addMSecs(ms);
			}
			else if (prec==2) {
				// round to 10 ms
				int ms=time.msec();
				time=time.addMSecs(-ms);
				ms = 10*(int)rint(ms/10);
				time=time.addMSecs(ms);
			}
		}
		else if (fabs(value)<3600) {
			format = "m:ss";
			if (prec==0) {
				int s=time.second();
				// round to full minute
				time=time.addSecs(-s);
				if(s>=30)
					time=time.addSecs(60);
				format="m";
			}
			else if (prec==1) {
				// round to 10 seconds
				int s=time.second();
				time=time.addSecs(-s);
				s = 10*(int)rint(s/(int)10);
				time=time.addSecs(s);
			}
		}
		else {
			// TODO : round minutes
			format="h:mm:ss";
		}
			
		// overwrite auto format
		if (dtf != i18n("auto"))
			format = dtf;
		label=time.toString(format);
		kdDebug()<<"VALUE in Time Format : "<<label<<endl;
		}
		break;
	case DATE: {
		QDate date(1970,1,1);
		date=date.addDays((int) value);
		QString format("dd.MM.yyyy");
		if (dtf != i18n("auto"))
			format = dtf;
			label=date.toString(format);
			kdDebug()<<"VALUE in Date Format ( "<<format<<") : "<<label<<endl;
		}
		break;
	case DATETIME: {
		QDate date(1970,1,1);
		QDateTime datetime(date);
//		kdDebug()<<"value = "<<(int) value<<endl;
		datetime=datetime.addSecs((int)value);
		QString format("dd.MM.yyyy h:mm:ss");
		if (dtf != i18n("auto"))
			format = dtf;
		label = datetime.toString(format);
//		kdDebug()<<"VALUE in DateTime Format ( "<<format<<") : "<<label<<endl;
		}
		break;
	case DEGREE:
		label = QString::number(180/M_PI*value,'f',prec)+'';
		break;
	}

	return label;
}

//! get the tic label value from the string according to atlf (range of axes)
double Plot::TicLabelValue(int atlf, QString string) {
	double value=0;
	
	switch(atlf) {
	case AUTO:
	case NORMAL:
	case SCIENTIFIC:
	case POWER10:
	case POWER2:
	case POWERE:
	case FSQRT:
		value = parse((char *)(string.latin1()));	// parse input
		break;
	case TIME: {
		QTime time;
		time = time.fromString(string);
		value = -1.0/1000.0*time.msecsTo(QTime()); // for reading msecs
		} break;
	case DATE: {
		QDate date;
		date = date.fromString(string,Qt::ISODate);              // must be yyyy-MM-dd
		//kdDebug()<<"DATE : "<<date.toString()<<endl;
		//kdDebug()<<"\tdaysTo1970 : "<<date.daysTo(QDate(1970,1,1))<<endl;
		value = -1.0*date.daysTo(QDate(1970,1,1));
		}
		break;
	case DATETIME: {
                QDateTime datetime;
                datetime = datetime.fromString(string,Qt::ISODate);
//              kdDebug()<<"DATETIME : "<<datetime.toString()<<endl;
//              kdDebug()<<"\tsecsTo1970 : "<<-datetime.secsTo(QDate(1970,1,1))<<endl;
                value = -1.0*datetime.secsTo(QDate(1970,1,1));
		}
		break;
	case DEGREE:
		string.remove('');
		value = M_PI/180.0*string.toDouble();
		break;
	}

	return value;
}

void Plot::shiftRight() {
	TScale scale = ((Plot2D *)this)->getAxis(0)->Scale();
	double r1=actrange[0].rMin(), r2=actrange[0].rMax();
	
	worksheet->checkRanges(scale,&r1,&r2);

	double r=worksheet->partRanges(scale,r1,r2);
	double nr1, nr2;
	switch(scale) {
	case LINEAR: case SQRT: case SX2: nr1 = r1-r; nr2 = r2-r; break;
	case LOG10: case LOG2: case LN: nr1 = r1/r; nr2 = r2/r; break;
	}

	// just checking
	worksheet->checkRanges(scale,&nr1,&nr2);

	actrange[0].setMin(nr1);
	actrange[0].setMax(nr2);
}

void Plot::shiftLeft() {
	TScale scale = ((Plot2D *)this)->getAxis(0)->Scale();
	double r1=actrange[0].rMin(), r2=actrange[0].rMax();
	
	worksheet->checkRanges(scale,&r1,&r2);
	
	double r=worksheet->partRanges(scale,r1,r2);
	double nr1, nr2;
	switch(scale) {
	case LINEAR: case SQRT: case SX2: nr1 = r1+r; nr2 = r2+r; break;
	case LOG10: case LOG2: case LN: nr1 = r1*r; nr2 = r2*r; break;
	}
	
	// just checking
	worksheet->checkRanges(scale,&nr1,&nr2);

	actrange[0].setMin(nr1);
	actrange[0].setMax(nr2);
}

void Plot::shiftUp() {
	TScale scale = ((Plot2D *)this)->getAxis(1)->Scale();
	double r1=actrange[1].rMin(), r2=actrange[1].rMax();
	
	worksheet->checkRanges(scale,&r1,&r2);
	
	double r=worksheet->partRanges(scale,r1,r2);
	double nr1, nr2;
	switch(scale) {
	case LINEAR: case SQRT: case SX2: nr1 = r1-r; nr2 = r2-r; break;
	case LOG10: case LOG2: case LN: nr1 = r1/r; nr2 = r2/r; break;
	}
	
	// just checking
	worksheet->checkRanges(scale,&nr1,&nr2);

	actrange[1].setMin(nr1);
	actrange[1].setMax(nr2);
}
	
void Plot::shiftDown() {
	TScale scale = ((Plot2D *)this)->getAxis(1)->Scale();
	double r1=actrange[1].rMin(), r2=actrange[1].rMax();
	
	worksheet->checkRanges(scale,&r1,&r2);
	
	double r=worksheet->partRanges(scale,r1,r2);
	double nr1, nr2;
	switch(scale) {
	case LINEAR: case SQRT: case SX2: nr1 = r1+r; nr2 = r2+r; break;
	case LOG10: case LOG2: case LN: nr1 = r1*r; nr2 = r2*r; break;
	}
	
	// just checking
	worksheet->checkRanges(scale,&nr1,&nr2);

	actrange[1].setMin(nr1);
	actrange[1].setMax(nr2);
}

void Plot::scaleXUp() {
	TScale scale = ((Plot2D *)this)->getAxis(0)->Scale();
	double r1=actrange[0].rMin(), r2=actrange[0].rMax();
	
	worksheet->checkRanges(scale,&r1,&r2);
	
	double r=worksheet->partRanges(scale,r1,r2);
	double nr1, nr2;
	switch(scale) {
	case LINEAR: case SQRT: case SX2: nr1 = r1+r; nr2 = r2-r; break;
	case LOG10: case LOG2: case LN: nr1 = r1*r; nr2 = r2/r; break;
	}
	
	// just checking
	worksheet->checkRanges(scale,&nr1,&nr2);

	actrange[0].setMin(nr1);
	actrange[0].setMax(nr2);
}

void Plot::scaleXDown() {
	TScale scale = ((Plot2D *)this)->getAxis(0)->Scale();
	double r1=actrange[0].rMin(), r2=actrange[0].rMax();
	
	worksheet->checkRanges(scale,&r1,&r2);
	
	double r=worksheet->partRanges(scale,r1,r2);
	double nr1, nr2;
	switch(scale) {
	case LINEAR: case SQRT: case SX2: nr1 = r1-r; nr2 = r2+r; break;
	case LOG10: case LOG2: case LN: nr1 = r1/r; nr2 = r2*r; break;
	}
	
	// just checking
	worksheet->checkRanges(scale,&nr1,&nr2);

	actrange[0].setMin(nr1);
	actrange[0].setMax(nr2);
}

void Plot::scaleYUp() {
	TScale scale = ((Plot2D *)this)->getAxis(1)->Scale();
	double r1=actrange[1].rMin(), r2=actrange[1].rMax();
	
	worksheet->checkRanges(scale,&r1,&r2);
	
	double r=worksheet->partRanges(scale,r1,r2);
	double nr1, nr2;
	switch(scale) {
	case LINEAR: case SQRT: case SX2: nr1 = r1+r; nr2 = r2-r; break;
	case LOG10: case LOG2: case LN: nr1 = r1*r; nr2 = r2/r; break;
	}
	
	// just checking
	worksheet->checkRanges(scale,&nr1,&nr2);

	actrange[1].setMin(nr1);
	actrange[1].setMax(nr2);
}
	
void Plot::scaleYDown() {
	TScale scale = ((Plot2D *)this)->getAxis(1)->Scale();
	double r1=actrange[1].rMin(), r2=actrange[1].rMax();
	
	worksheet->checkRanges(scale,&r1,&r2);
	
	double r=worksheet->partRanges(scale,r1,r2);
	double nr1, nr2;
	switch(scale) {
	case LINEAR: case SQRT: case SX2: nr1 = r1-r; nr2 = r2+r; break;
	case LOG10: case LOG2: case LN: nr1 = r1/r; nr2 = r2*r; break;
	}

	// just checking
	worksheet->checkRanges(scale,&nr1,&nr2);

	actrange[1].setMin(nr1);
	actrange[1].setMax(nr2);
}

void Plot::scaleZUp() {
	// does this work ?
	TScale scale = ((Plot2D *)this)->getAxis(2)->Scale();
	double r1=actrange[2].rMin(), r2=actrange[2].rMax();
	
	worksheet->checkRanges(scale,&r1,&r2);
	
	double r=worksheet->partRanges(scale,r1,r2);
	double nr1, nr2;
	switch(scale) {
	case LINEAR: case SQRT: case SX2: nr1 = r1+r; nr2 = r2-r; break;
	case LOG10: case LOG2: case LN: nr1 = r1*r; nr2 = r2/r; break;
	}
	
	// just checking
	worksheet->checkRanges(scale,&nr1,&nr2);

	actrange[2].setMin(nr1);
	actrange[2].setMax(nr2);
}

void Plot::scaleZDown() {
	// does this work ?
	TScale scale = ((Plot2D *)this)->getAxis(2)->Scale();
	double r1=actrange[2].rMin(), r2=actrange[2].rMax();
	
	worksheet->checkRanges(scale,&r1,&r2);
	
	double r=worksheet->partRanges(scale,r1,r2);
	double nr1, nr2;
	switch(scale) {
	case LINEAR: case SQRT: case SX2: nr1 = r1-r; nr2 = r2+r; break;
	case LOG10: case LOG2: case LN: nr1 = r1/r; nr2 = r2*r; break;
	}
	
	// just checking
	worksheet->checkRanges(scale,&nr1,&nr2);

	actrange[2].setMin(nr1);
	actrange[2].setMax(nr2);
}

void Plot::zoomIn() { 
	TScale xscale = ((Plot2D *)this)->getAxis(0)->Scale();
	TScale yscale = ((Plot2D *)this)->getAxis(1)->Scale();
	TScale zscale = ((Plot2D *)this)->getAxis(2)->Scale();
	double x1=actrange[0].rMin(), x2=actrange[0].rMax();
	double y1=actrange[1].rMin(), y2=actrange[1].rMax();
	double z1=actrange[2].rMin(), z2=actrange[2].rMax();
	
	worksheet->checkRanges(xscale,&x1,&x2);
	worksheet->checkRanges(yscale,&y1,&y2);
	worksheet->checkRanges(zscale,&z1,&z2);
	
	double x=worksheet->partRanges(xscale,x1,x2);
	double y=worksheet->partRanges(yscale,y1,y2);
	double z=worksheet->partRanges(zscale,z1,z2);
	double nx1, nx2, ny1, ny2, nz1, nz2;
	switch(xscale) {
	case LINEAR: case SQRT: case SX2: nx1 = x1+x; nx2 = x2-x;  break;
	case LOG10: case LOG2: case LN: nx1 = x1*x; nx2 = x2/x; break;
	}
	switch(yscale) {
	case LINEAR: case SQRT: case SX2: ny1 = y1+y; ny2 = y2-y; break;
	case LOG10: case LOG2: case LN: ny1 = y1*y; ny2 = y2/y; break;
	}
	switch(zscale) {
	case LINEAR: case SQRT: case SX2: nz1 = z1+z; nz2 = z2-z; break;
	case LOG10: case LOG2: case LN: nz1 = z1*z; nz2 = z2/z; break;
	}
	
	// just checking
	worksheet->checkRanges(xscale,&nx1,&nx2);
	worksheet->checkRanges(yscale,&ny1,&ny2);
	worksheet->checkRanges(zscale,&nz1,&nz2);

	actrange[0].setMin(nx1);actrange[0].setMax(nx2);
	actrange[1].setMin(ny1);actrange[1].setMax(ny2);
	actrange[2].setMin(nz1);actrange[2].setMax(nz2);
}

void Plot::zoomOut() { 
	TScale xscale = ((Plot2D *)this)->getAxis(0)->Scale();
	TScale yscale = ((Plot2D *)this)->getAxis(1)->Scale();
	TScale zscale = ((Plot2D *)this)->getAxis(2)->Scale();
	double x1=actrange[0].rMin(), x2=actrange[0].rMax();
	double y1=actrange[1].rMin(), y2=actrange[1].rMax();
	double z1=actrange[2].rMin(), z2=actrange[2].rMax();
	
	worksheet->checkRanges(xscale,&x1,&x2);
	worksheet->checkRanges(yscale,&y1,&y2);
	worksheet->checkRanges(zscale,&z1,&z2);
	
	double x=worksheet->partRanges(xscale,x1,x2);
	double y=worksheet->partRanges(yscale,y1,y2);
	double z=worksheet->partRanges(zscale,z1,z2);
	double nx1, nx2, ny1, ny2, nz1, nz2;
	switch(xscale) {
	case LINEAR: case SQRT: case SX2: nx1 = x1-x; nx2 = x2+x; break;
	case LOG10: case LOG2: case LN: nx1 = x1/x; nx2 = x2*x; break;
	}
	switch(yscale) {
	case LINEAR: case SQRT: case SX2: ny1 = y1-y; ny2 = y2+y; break;
	case LOG10: case LOG2: case LN: ny1 = y1/y; ny2 = y2*y; break;
	}
	switch(zscale) {
	case LINEAR: case SQRT: case SX2: nz1 = z1-z; nz2 = z2+z; break;
	case LOG10: case LOG2: case LN: nz1 = z1/z; nz2 = z2*z; break;
	}
	
	// just checking
	worksheet->checkRanges(xscale,&nx1,&nx2);
	worksheet->checkRanges(yscale,&ny1,&ny2);
	worksheet->checkRanges(zscale,&nz1,&nz2);

	actrange[0].setMin(nx1);actrange[0].setMax(nx2);
	actrange[1].setMin(ny1);actrange[1].setMax(ny2);
	actrange[2].setMin(nz1);actrange[2].setMax(nz2);
}

// find y value for x in data mode
Point Plot::dataValue(double x) {
	double y=0;

	double xmin=actrange[0].rMin(), xmax=actrange[0].rMax();
	double ymin=actrange[1].rMin(), ymax=actrange[1].rMax();
	
	//TODO : which graph
	int graph=0;
	
	if(x<p1.X())
		return Point(p1.X(),p2.Y());
	else if( x>p2.X())
		return Point(p2.X(),p2.Y());

//	kdDebug()<<"	X-VALUE for datamode = "<<x<<endl;

	// convert from world coordinates
	switch(type){
	case P2D:
		switch(((Plot2D *)this)->getAxis(0)->Scale()) {
		case LINEAR:	x = xmin+(x-p1.X())*(xmax-xmin)/(p2.X()-p1.X()); break;
		case LOG10: 	x = pow(10,log10(xmin)+(x-p1.X())*log10(xmax/xmin)/(p2.X()-p1.X())); break;
		case LOG2: 	x = pow(2,log2(xmin)+(x-p1.X())*log2(xmax/xmin)/(p2.X()-p1.X())); break;
		case LN: 		x = pow(M_E,log(xmin)+(x-p1.X())*log(xmax/xmin)/(p2.X()-p1.X())); break;
		case SQRT:
			//TODO
			break;
		case SX2:
			//TODO
			break;
		}
	default: break;
	}

//	kdDebug()<<"	X for datamode = "<<x<<endl;
	
	GRAPHType s = graphlist->getStruct(graph);
	
	switch(s) {
	case GRAPH2D: {
		Graph2D *g = graphlist->getGraph2D(graph);
		Point *ptr = g->Data();
		
		for(int i=0;i<g->Number()-1;i++) {
			if(ptr[i].X() < x && ptr[i+1].X() > x) {
				x = ptr[i].X();
				y = ptr[i].Y();
				break;
			}
		}
		}; break;
	case GRAPH3D: {
		Graph3D *g = graphlist->getGraph3D(graph);
		Point3D *ptr = g->Data();
		
		for(int i=0;i<g->Number()-1;i++) {
			if(ptr[i].X() < x && ptr[i+1].X() > x) {
				x = ptr[i].X();
				y = ptr[i].Y();
				break;
			}
		}
		}; break;
	case GRAPH4D: {
		Graph4D *g = graphlist->getGraph4D(graph);
		Point4D *ptr = g->Data();
		
		for(int i=0;i<g->Number()-1;i++) {
			if(ptr[i].X() < x && ptr[i+1].X() > x) {
				x = ptr[i].X();
				y = ptr[i].Y();
				break;
			}
		}
		}; break;
	default: break;
	}
	
	worksheet->getMainWin()->message("( " + QString::number(x) + " / " + QString::number(y) + " )");
	
	// convert to world coordinates
//	kdDebug()<<"	Y for datamode = "<<y<<endl;
	switch(type){
	case P2D:
		switch(((Plot2D *)this)->getAxis(1)->Scale()) {
		case LINEAR:	y = p2.Y()-(y-ymin)/(ymax-ymin)*(p2.Y()-p1.Y()); break;
		case LOG10:	y = p2.Y()-(log10(y)-log10(ymin))/log10(ymax/ymin)*(p2.Y()-p1.Y()); break;
		case LOG2:	y = p2.Y()-(log2(y)-log2(ymin))/log2(ymax/ymin)*(p2.Y()-p1.Y()); break;
		case LN:		y = p2.Y()-(log(y)-log(ymin))/log(ymax/ymin)*(p2.Y()-p1.Y()); break;
		case SQRT:
			//TODO
			break;
		case SX2:
			//TODO
			break;
		}
		switch(((Plot2D *)this)->getAxis(0)->Scale()) {
		case LINEAR:	x = p1.X()+(x-xmin)/(xmax-xmin)*(p2.X()-p1.X()); break;
		case LOG10:	x = p1.X()+(log10(x)-log10(xmin))/log10(xmax/xmin)*(p2.X()-p1.X()); break;
		case LOG2:	x = p1.X()+(log2(x)-log2(xmin))/log2(xmax/xmin)*(p2.X()-p1.X()); break;
		case LN:		x = p1.X()+(log(x)-log(xmin))/log(xmax/xmin)*(p2.X()-p1.X()); break;
		case SQRT:
			//TODO
			break;
		case SX2:
			//TODO
			break;
		}
	default: break;
	}
//	kdDebug()<<"	Y-VALUE for datamode = "<<y<<endl;
	
	return Point(x,y);
}
	
//! calcuate tic number for auto tics
int Plot::autoTics(double min, double max) {
//	kdDebug()<<"Plot::autoTics : min/max = "<<min<<' '<<max<<endl;
	if(max-min==0)
		return -1;
	int floorvalue = (int) floor(log10(max-min));
	int tics = (int)((max-min)/pow(10.0,(double)floorvalue));

	// just to make sure
	if(tics<=0)
		return -1;
	
	// always between 4 and 10 tics
	while(tics<4)
		tics *= 2;

	return tics;	
}
