// LabPlot : Spreadsheet.cc

#include <stdlib.h>
#include <time.h>
#ifdef HAVE_SGI_STL
#include <vector>
#endif
#include <qtable.h>
#include <qfile.h>
#include <qhbox.h>
#include <qcursor.h>
#include <qclipboard.h>
#include <qinputdialog.h>

#include <kmessagebox.h>
#include <kiconloader.h>
#include <kconfig.h>
#include <ktempfile.h>

#include "Spreadsheet.h"
#include "SpreadsheetValuesDialog.h"
#include "SpreadsheetPropertiesDialog.h"
#include "pixmaps/plotpixmap.h"
#include "DumpDialog.h"
#include "ListDialog.h"
#include "DestinationDialog.h"
#include "LTableItem.h"
#include "Plot2DSurface.h"

#include "spreadsheetProperties.h"
#include "source.h"
#include "defs.h"

#ifdef USE_NEW_TABLE
//#include <vector>
#endif

#include <exception>
using namespace std;

Spreadsheet::Spreadsheet(QWidget *p, MainWin *mw, const char *name)
	:QWidget(p,name), mw(mw)
{
	kdDebug()<<"Spreadsheet()"<<endl;

	ld=0;

	widgettype = WSPREADSHEET;

	// find free worksheet number
	QWidgetList list = mw->getWorkspace()->windowList();
	int nr=0, sheets = mw->NrWorksheets()+mw->NrSpreadsheets();
	bool found_free=false;
	while(nr <= sheets && !found_free ) {
		nr++;
		found_free=true;
		for(int i=0;i<sheets;i++) {
			if(list.at(i)->caption() == QString(i18n("Spreadsheet")+QString(" ")+QString::number(nr)))
				found_free=false;
		}
	}

	kdDebug()<<"	Using number "<<nr<<endl;
	title = i18n("Spreadsheet")+QString(" ")+QString::number(nr);
	setCaption(title);

#ifdef USE_NEW_TABLE
	table = new LTable(10, 2, this );
#else
	table = new QTable( 100, 2, this );
#endif
	table->setRowMovingEnabled (true);
	table->setColumnMovingEnabled (true);

	table->horizontalHeader()->setLabel( 0, QString( "A ")+i18n("{double}")+" [X]");
	table->horizontalHeader()->setLabel( 1, QString( "B ")+i18n("{double}")+" [Y]");
	table->horizontalHeader()->installEventFilter(this );
	
	table->setFocusPolicy(QWidget::StrongFocus);
	table->setFocus();

	resize(table->sizeHint());
	show();
	if(mw) mw->updateSheetList();
	
	destination=-1;

	// menus for context menu
	plotmenu = new QPopupMenu(this);
	destmenu = new QPopupMenu(this);
	selmenu = new QPopupMenu(this);
	fillmenu = new QPopupMenu(this);
	normmenu = new QPopupMenu(this);
	editmenu = new QPopupMenu(this);
	convertmenu = new QPopupMenu(this);
	sortmenu = new QPopupMenu(this);
	maskmenu = new QPopupMenu(this);
}

// used for properties dialog on header
bool Spreadsheet::eventFilter(QObject *object, QEvent *e) {
//	kdDebug()<<"Spreadsheet::eventFilter()"<<endl;

	switch(e->type()) {
	case QEvent::MouseButtonDblClick:	(new SpreadsheetPropertiesDialog(mw,table,caption()))->show(); break;
	default: break;
	}

	return QObject::eventFilter(object,e);
}

// used for left upper corner
void Spreadsheet::mousePressEvent ( QMouseEvent * e ) {
	kdDebug()<<"mousePressEvent() @ x/y "<<e->x()<<' '<<e->y()<<endl;
	if(e->x()<100 && e->y()<25)		// left upper corner
		for (int i=0;i<table->numCols();i++)
			table->selectColumn(i);
}

//! clear spreadsheet
void Spreadsheet::Clear() {
	kdDebug()<<"Spreadsheet::Clear()"<<endl;
	for (int i=0;i<table->numCols();i++) {
		for (int j=0;j<table->numRows();j++) {
			if(table->numSelections() == 0 || table->isSelected(j,i))
				table->setText(j,i,"");
		}
	}
	table->clearSelection();
}

QStringList Spreadsheet::Info() {
	kdDebug()<<"Spreadsheet::Info()"<<endl;
	QStringList s;
	s<<i18n("SIZE:")<<QString::number(table->numCols())<<i18n(" x ")<<QString::number(table->numRows());

	return s;
}

void Spreadsheet::setCurrentColumn(int i) {
	table->clearSelection();
	table->setCurrentCell(0,i);
}

//! calculate number of filled rows
int Spreadsheet::filledRows(int col) {
	int row=1;
	while (table->text(row,col).length()>0)
		row++;
	return row;
}

void Spreadsheet::resizeEvent(QResizeEvent *e) {
	table->resize(e->size());
}

void Spreadsheet::closeEvent(QCloseEvent *e) {
	kdDebug()<<"Spreadsheet::closeEvent()"<<endl;
	
	int ok = KMessageBox::warningContinueCancel(this, i18n("Do you really want to close the spreadsheet ?"));
	if(ok == KMessageBox::Cancel)
		return;
	
	if (ld) updateGraphList();
		
	e->accept();

	mw->deleteActiveSheet();
	mw->updateSheetList();
}

// used when editing data
void Spreadsheet::updateGraphList() {
	int rows = table->numRows();
	// -> see edit dialogx
	switch(g->Type()) {
	case GRAPH2D: {
		double x,y,xmin=0,xmax=0,ymin=0,ymax=0;
		Point *ptr = ((Graph2D *)g)->Data();
		for (int i=0;i<rows;i++) {
			x=(table->text(i,0)).toDouble();
			y=(table->text(i,1)).toDouble();
			
			ptr[i].setPoint(x,y);
		}
		
		mw->calculateRanges2D(ptr,rows,&xmin,&xmax,&ymin,&ymax);
		
		LRange range[2];
		range[0] = LRange(xmin,xmax);
		range[1] = LRange(ymin,ymax);
		((Graph2D *)g)->setRange(range);

		kdDebug()<<"EditDialog :"<<endl;
		kdDebug()<<"new 2D Range :"<<endl;
		kdDebug()<<"		"<<xmin<<' '<<xmax<<endl;
		kdDebug()<<"		"<<ymin<<' '<<ymax<<endl;

		((Graph2D *)g)->setNumber(rows);
		}; break;
	case GRAPH3D: {
		double x,y,z,xmin=0,xmax=0,ymin=0,ymax=0,zmin=0,zmax=0;
		Point3D *ptr = ((Graph3D *)g)->Data();

		for (int i=0;i<rows;i++) {
			x=(table->text(i,0)).toDouble();
			y=(table->text(i,1)).toDouble();
			z=(table->text(i,2)).toDouble();

			ptr[i].setPoint(x,y,z);
		}

		mw->calculateRanges3D(ptr,rows,&xmin,&xmax,&ymin,&ymax,&zmin,&zmax);
		
		LRange range[3];
		range[0] = LRange(xmin,xmax);
		range[1] = LRange(ymin,ymax);
		range[2] = LRange(zmin,zmax);

		kdDebug()<<"EditDialog :"<<endl;
		kdDebug()<<"3D Range :"<<endl;
		kdDebug()<<"		"<<xmin<<' '<<xmax<<endl;
		kdDebug()<<"		"<<ymin<<' '<<ymax<<endl;
		kdDebug()<<"		"<<zmin<<' '<<zmax<<endl;

		((Graph3D *)g)->setRange(range);
		((Graph3D *)g)->setNumber(rows);
		}; break;
	case GRAPHM : {
		double z, zmin=0,zmax=0;
		double *a = ((GraphM *)g)->Data();

		for (int i=0;i<rows;i++) {
			for (int j=0;j<table->numCols();j++) {
				z=(table->text(i,j)).toDouble();
				if (i == 0) {
					zmin=zmax=z;
				}
				else {
					z<zmin?zmin=z:0;
					z>zmax?zmax=z:0;
				}
				kdDebug()<<" EditDialog graphm : z = "<<z<<endl;

				a[j+i*(table->numCols())]=z;
			}
		}
		LRange range[1];
		range[0] = LRange(zmin,zmax);

		((GraphM *)g)->setZRange(range);
		((GraphM *)g)->setNumber(table->numCols(),rows);
		}; break;
	case GRAPH4D : {
		double x,y,z,t,xmin=0,xmax=0,ymin=0,ymax=0,zmin=0,zmax=0,tmin=0,tmax=0;
		Point4D *ptr = ((Graph4D *)g)->Data();

		for (int i=0;i<rows;i++) {
			x=(table->text(i,0)).toDouble();
			y=(table->text(i,1)).toDouble();
			z=(table->text(i,2)).toDouble();
			t=(table->text(i,3)).toDouble();

			ptr[i].setPoint(x,y,z,t);
		}
		
		mw->calculateRanges4D(ptr,rows,&xmin,&xmax,&ymin,&ymax,&zmin,&zmax,&tmin,&tmax);

		LRange range[4];
		range[0] = LRange(xmin,xmax);
		range[1] = LRange(ymin,ymax);
		range[2] = LRange(zmin,zmax);
		range[3] = LRange(tmin,tmax);

		kdDebug()<<"EditDialog :"<<endl;
		kdDebug()<<"4D Range :"<<endl;
		kdDebug()<<"		"<<xmin<<' '<<xmax<<endl;
		kdDebug()<<"		"<<ymin<<' '<<ymax<<endl;
		kdDebug()<<"		"<<zmin<<' '<<zmax<<endl;
		kdDebug()<<"		"<<tmin<<' '<<tmax<<endl;

		((Graph4D *)g)->setRange(range);
		((Graph4D *)g)->setNumber(rows);
		}; break;
	default: break;
	}

	ld->updateList();
}

// build menu for context or "Spreadsheet"
void Spreadsheet::Menu(QPopupMenu *menu) {
	kdDebug()<<"Spreadsheet::Menu()"<<endl;

	menu->clear();
	plotmenu->clear();
	destmenu->clear();
	selmenu->clear();
	fillmenu->clear();
	normmenu->clear();
	editmenu->clear();
	convertmenu->clear();
	sortmenu->clear();
	maskmenu->clear();

	QPixmap spreadsheetIcon = QPixmap(spreadsheet_xpm);
	
	QPixmap plot2dIcon = QPixmap(plot2d_xpm);
	QPixmap plot3dIcon = QPixmap(plot3d_xpm);
	QPixmap plotqwt3dIcon = QPixmap(plotqwt3d_xpm);
	QPixmap plotsurfaceIcon = QPixmap(plotsurface_xpm);
	QPixmap plotpieIcon = QPixmap(plotpie_xpm);
	QPixmap plotpolarIcon = QPixmap(plotpolar_xpm);
	QPixmap plotternaryIcon = QPixmap(plotternary_xpm);

	QPixmap selectIcon = KGlobal::iconLoader()->loadIcon("select", KIcon::Small);

	QPixmap cutIcon = KGlobal::iconLoader()->loadIcon("editcut", KIcon::Small);
	QPixmap copyIcon = KGlobal::iconLoader()->loadIcon("editcopy", KIcon::Small);
	QPixmap pasteIcon = KGlobal::iconLoader()->loadIcon("editpaste", KIcon::Small);
	QPixmap clearIcon = KGlobal::iconLoader()->loadIcon("editclear", KIcon::Small);

	QPixmap setIcon = KGlobal::iconLoader()->loadIcon("pagesetup", KIcon::Small);	
	QPixmap fillIcon = KGlobal::iconLoader()->loadIcon("fill", KIcon::Small);	
	QPixmap normIcon = KGlobal::iconLoader()->loadIcon("tool_normal", KIcon::Small);	
	QPixmap convertIcon = KGlobal::iconLoader()->loadIcon("transform", KIcon::Small);	
	QPixmap exportIcon = KGlobal::iconLoader()->loadIcon("fileexport", KIcon::Small);	

	QPixmap deleteIcon = KGlobal::iconLoader()->loadIcon("editdelete", KIcon::Small);
	QPixmap addIcon = KGlobal::iconLoader()->loadIcon("edit_add", KIcon::Small);
	QPixmap sortIcon = KGlobal::iconLoader()->loadIcon("unsorted_list", KIcon::Small);
	QPixmap sortascIcon = KGlobal::iconLoader()->loadIcon("sort_incr", KIcon::Small);
	QPixmap sortdesIcon = KGlobal::iconLoader()->loadIcon("sort_decrease", KIcon::Small);
	QPixmap propertiesIcon = KGlobal::iconLoader()->loadIcon("site_properties", KIcon::Small);
	
	QPixmap maskIcon = QPixmap(plotmask_xpm);

	menu->insertItem(plot2dIcon, i18n( "Plot" ),plotmenu );
	plotmenu->insertItem(plot2dIcon, i18n( "2D Plot (XY)" ),this,SLOT(plot2DSimple()) );
	plotmenu->insertItem(plot2dIcon, i18n( "2D Plot (XYDY)" ),this,SLOT(plot3DXYDY()) );
	plotmenu->insertItem(plot2dIcon, i18n( "2D Plot (XYDXDY)" ),this,SLOT(plot4DXYDXDY()) );
	plotmenu->insertItem(plot2dIcon, i18n( "2D Plot (XYDYDY)" ),this,SLOT(plot4DXYDYDY()) );
	plotmenu->insertItem(plot2dIcon, i18n( "2D Plot (XY%X)" ),this,SLOT(plot3DXYPX()) );
	plotmenu->insertItem(plot2dIcon, i18n( "2D Plot (XY%Y)" ),this,SLOT(plot3DXYPY()) );
	plotmenu->insertItem(plot2dIcon, i18n( "2D Plot (LINE)" ),this,SLOT(plot2DLine()) );
	plotmenu->insertSeparator();
	plotmenu->insertItem(plot3dIcon, i18n( "3D Plot (XYZ)" ),this,SLOT(plot3DSimple()) );
	plotmenu->insertItem(plot3dIcon, i18n( "3D Plot (MATRIX)" ),this,SLOT(plotM3D()) );
	plotmenu->insertItem(plotqwt3dIcon, i18n( "3D QWT Plot (XYZ)" ),this,SLOT(plotQWT3D()) );
	plotmenu->insertItem(plotqwt3dIcon, i18n( "3D QWT Plot (MATRIX)" ),this,SLOT(plotMQWT3D()) );
	plotmenu->insertSeparator();
	plotmenu->insertItem(plotsurfaceIcon, i18n( "Surface Plot(XYZ)" ),this,SLOT(plot3DSurface()) );
	plotmenu->insertItem(plotsurfaceIcon, i18n( "Surface Plot(MATRIX)" ),this,SLOT(plotMSurface()) );
	plotmenu->insertSeparator();
	plotmenu->insertItem(plotpieIcon, i18n( "Pie Plot (XY)" ),this,SLOT(plot2DPie()) );
	plotmenu->insertItem(plotpolarIcon, i18n( "Polar Plot (XY)" ),this,SLOT(plot2DPolar()) );
	plotmenu->insertItem(plotternaryIcon, i18n( "Ternary Plot (XYZ)" ),this,SLOT(plot3DTernary()) );
	menu->insertItem( selectIcon, i18n( "Set title" ),this,SLOT(setTitle()) );
	menu->insertItem( selectIcon, i18n( "Select destination" ),this,SLOT(selectDestination()) );
	menu->insertSeparator();
	menu->insertItem( cutIcon, i18n( "Cut" ), this, SLOT(cutSelection()), CTRL+Key_X );
	menu->insertItem( copyIcon, i18n( "Copy" ), this, SLOT(copySelection()), CTRL+Key_C );
	menu->insertItem( pasteIcon, i18n( "Paste" ), this, SLOT(pasteSelection()), CTRL+Key_V );
	menu->insertItem( clearIcon, i18n( "Clear" ), this, SLOT(clearSelection()) );
	menu->insertItem( selectIcon, i18n( "Selection" ), selmenu );
	selmenu->insertItem(selectIcon, i18n("Select all"),this,SLOT(selectAll()));
	selmenu->insertItem(selectIcon, i18n("Deselect all"),this,SLOT(selectNone()));
	selmenu->insertItem(selectIcon, i18n("Invert selection"),this,SLOT(selectInvert()));
	menu->insertSeparator();
	menu->insertItem( setIcon, i18n( "Set column values" ),this,SLOT(setValues()) );
	menu->insertItem( fillIcon, i18n( "Fill with" ),fillmenu );
	fillmenu->insertItem( i18n( "Row number" ),this,SLOT(fillRowNumber()) );
	fillmenu->insertItem( i18n( "Random values" ),this,SLOT(fillRandom()) );
	menu->insertItem( normIcon, i18n( "Normalize" ),normmenu );
	normmenu->insertItem( i18n( "Sum=1" ),this,SLOT(normSum()) );
	normmenu->insertItem( i18n( "Max=1" ),this,SLOT(normMax()) );
	menu->insertItem( copyIcon, i18n( "Edit with editor" ),editmenu );
	editmenu->insertItem( i18n( "vi" ),this,SLOT(editVI()) );
	editmenu->insertItem( i18n( "kvim" ),this,SLOT(editKVIM()) );
	editmenu->insertItem( i18n( "gvim" ),this,SLOT(editGVIM()) );
//	this doesn't work
//	editmenu->insertItem( i18n( "kate" ),this,SLOT(editKATE()) );
	editmenu->insertItem( i18n( "kwrite" ),this,SLOT(editKWRITE()) );
	editmenu->insertItem( i18n( "Emacs" ),this,SLOT(editEMACS()) );
	editmenu->insertItem( i18n( "Xemacs" ),this,SLOT(editXEMACS()) );
	editmenu->insertItem( i18n( "kword" ),this,SLOT(editKWORD()) );
	editmenu->insertItem( i18n( "Starwrite" ),this,SLOT(editSOFFICE()) );
	menu->insertSeparator();
	menu->insertItem( convertIcon, i18n( "Convert" ),convertmenu );
	convertmenu->insertItem( i18n( "Transpose Matrix" ),this,SLOT(transposeMatrix()) );
	convertmenu->insertItem( i18n( "Matrix -> XYZ" ),this,SLOT(convertMatrixtoXYZ()) );
	convertmenu->insertItem( i18n( "XYZ -> Matrix" ),this,SLOT(convertXYZtoMatrix()) );
	convertmenu->insertItem( i18n( "Column -> Matrix" ),this,SLOT(convertColumntoMatrix()) );
	menu->insertSeparator();
	menu->insertItem(exportIcon, i18n("Export data"),this,SLOT(exportData()));
	menu->insertSeparator();
	menu->insertItem( addIcon, i18n( "Add Column" ),this,SLOT(addColumn()) );
	menu->insertItem( deleteIcon, i18n( "Delete selected Columns" ),this,SLOT(deleteColumns()) );
	menu->insertItem( deleteIcon, i18n( "Delete selected Rows" ),this,SLOT(deleteRows()) );
	menu->insertItem( maskIcon, i18n( "Masking" ),maskmenu );
	maskmenu->insertItem( maskIcon, i18n( "Mask/Unmask selection" ),this,SLOT(maskSelection()) );
	maskmenu->insertItem( maskIcon, i18n( "Toggle Masking" ),this,SLOT(toggleMask()) );
	maskmenu->insertItem( maskIcon, i18n( "Unmask all" ),this,SLOT(unMask()) );
	maskmenu->insertItem( maskIcon, i18n( "Mask every n-th row" ),this,SLOT(maskNthRow()) );
	maskmenu->insertItem( maskIcon, i18n( "Mask all but n-th row" ),this,SLOT(maskFirstRow()) );
	menu->insertSeparator();
	menu->insertItem( sortIcon, i18n( "Sort" ),sortmenu );
	sortmenu->insertItem( sortascIcon, i18n( "Ascending" ),this,SLOT(sortAscending()) );
	sortmenu->insertItem( sortdesIcon, i18n( "Descending" ),this,SLOT(sortDescending()) );
	menu->insertSeparator();
	menu->insertItem(propertiesIcon, i18n( "Properties" ),this,SLOT(setProperties()) );
}

void Spreadsheet::contextMenuEvent(QContextMenuEvent *) {
	kdDebug()<<"Spreadsheet::contextMenuEvent()"<<endl;
	//TODO : if over header : select current column
/*	if ( object == (QObject *)table->horizontalHeader()) {
		const QMouseEvent *me = (const QMouseEvent *)e;
		int col=table->horizontalHeader()->sectionAt (me->pos().x());
		kdDebug()<<"	HEADER : selection col "<<col<<endl;
		if(!table->isColumnSelected(col))
			selectColumns(col);
	}*/
	
	QPopupMenu *menu = new QPopupMenu(this);
	Menu(menu);

	menu->exec(QCursor::pos());
}

void Spreadsheet::save(QTextStream *t) {
	kdDebug()<<"Spreadsheet::save()"<<endl;
	*t<<title<<endl;
	*t<<table->numCols()<<' '<<table->numRows()<<endl;
	
	// dump header
	for(int i=0;i<table->numCols();i++)
		*t<<table->horizontalHeader()->label(i)<<endl;
		
	// dump data
	for(int i=0;i<table->numRows();i++) {
		for(int j=0;j<table->numCols();j++) {
			*t<<"\""<<table->text(i,j)<<"\" ";
		}
		*t<<endl;
	}
}

void Spreadsheet::open(QTextStream *t, int version) {
	kdDebug()<<"Spreadsheet::open()"<<endl;
	if(version>20) {
		t->readLine();
		title = t->readLine();
		kdDebug()<<"TITLE = "<<title<<endl;
		setCaption(title);
	}
	int nx,ny;
	*t>>nx>>ny;
	table->setNumCols(nx);
	table->setNumRows(ny);
	kdDebug()<<"NX="<<nx<<" NY="<<ny<<endl;
		
	// read header
	QString tmp;
	t->readLine();
	for(int i=0;i<table->numCols();i++) {
		tmp = t->readLine();
		kdDebug()<<"Label "<<i<<'='<<tmp<<endl;
		table->horizontalHeader()->setLabel(i,tmp);
	}
	
	// read data
	for(int i=0;i<table->numRows();i++) {
		tmp = t->readLine();
		tmp = tmp.simplifyWhiteSpace();
		QStringList oneline;
		if(tmp.contains('"'))
			oneline = QStringList::split("\" ",tmp);	// new style
		else
			oneline = QStringList::split(" ",tmp);		// old style (onyl 1.4.0.pre/rc)
		int j=0;
		for ( QStringList::Iterator it = oneline.begin(); it != oneline.end(); ++it ) {
				QString entry=*it;
				entry.remove('"');
                                table->setText(i,j++,entry);
		}
	}
}

//! return format index for column col
int Spreadsheet::formatItem(int col) {
	QString header = table->horizontalHeader()->label(col);
	int pos1 = header.find(QRegExp("\\{"));
	int pos2 = header.find(QRegExp("\\}"));
	QString format = header.mid(pos1,pos2-pos1+1);
	for(int i=0;i<NR_FORMATS;i++)
		if(format==i18n(formatList[i]))
			return i;
	return -1;
}

// return name of column col
QString Spreadsheet::columnTitle(int col) {
	QString label = table->horizontalHeader()->label(col);
	label.remove(QRegExp(" \\{.+\\]"));
	
	return label;
}

void Spreadsheet::setColumnTitle(int col, QString name) {
	if(col==0)
		table->horizontalHeader()->setLabel(col, name+' '+i18n("{double}")+" [X]");
	else
		table->horizontalHeader()->setLabel(col, name+' '+i18n("{double}")+" [Y]");
}

void Spreadsheet::setColumnType(int col, QString type) {
	kdDebug()<<"setColumnType() : col="<<col<<" type = "<<type<<endl;
	QString label = table->horizontalHeader()->label(col);
	label.replace(QRegExp(" \\[.+\\]"),QString(" ["+type+"]"));
	table->horizontalHeader()->setLabel(col, label);
}

//! make 2d plot with x-y-dy
void Spreadsheet::plot3DXYDY(Input2D mode) {
	kdDebug()<<"Spreadsheet::plot3DXYDY("<<mode<<")"<<endl;
	
	// if less than 3 columns selected -> select all columns
	int selected_cols=0;
	for (int s=0;s<table->numSelections();s++) {
		QTableSelection ts = table->selection(s);
#if QT_VERSION > 0x030102
		selected_cols+=ts.numCols();
#else
		selected_cols+=ts.rightCol()-ts.leftCol();
#endif
	}
        if(selected_cols<2)
		for (int i=0;i<table->numCols();i++)
			table->selectColumn(i);

	// find x,y,dy column
	int indexx=-1,indexy=-1,indexz=-1;
	for(int i=table->numCols()-1;i>=0;i--) {
		if(table->isColumnSelected(i) && table->horizontalHeader()->label(i).findRev("[X]")>0)
			indexx=i;
		if(table->isColumnSelected(i) && table->horizontalHeader()->label(i).findRev("[Y]")>0)
			indexy=i;
		if(table->isColumnSelected(i) && table->horizontalHeader()->label(i).findRev("[DY]")>0)
			indexz=i;
	}

	Point3D *ptr = new Point3D[table->numRows()];
	// read x and y values
	double x, y, z, xmin=0,xmax=1,ymin=0,ymax=1, zmin=0, zmax=1;
	int numrows=0;

	// ask for percentage
	double percent=0.0;
	if(mode == I2DXYPX || mode == I2DXYPY) {
		bool ok;
		percent = QInputDialog::getDouble(i18n("2D error plot"),i18n("Percentage to use"),10,1,INF,1,&ok);
		if(!ok)
			return;
		kdDebug()<<"	percentage "<<percent<<endl;
	}
	
	// only selected rows
	QTableSelection ts = table->selection(table->currentSelection());
//	kdDebug()<<"NUM SEL = "<<table->numSelections()<<endl;
	int start=0, end=table->numRows();
	if(ts.bottomRow()>ts.topRow()) {
		start = ts.topRow();
		end =  ts.bottomRow();
	}
	kdDebug()<<"reading ROWS = "<<start<<' '<<end<<endl;
	for (int row = start; row < end + 1; row++) {
		if(indexx == -1)
			x = (double) row + 1;
		else
			x = mw->formatLabel(table->text(row,indexx),formatItem(indexx));
		y = mw->formatLabel(table->text(row,indexy),formatItem(indexy));
		if(mode == I2DXYPX)
			z = fabs(percent/100.0*x);
		else if(mode == I2DXYPY)
			z = fabs(percent/100.0*y);
		else
			z = mw->formatLabel(table->text(row,indexz),formatItem(indexz));
		//kdDebug()<<"x/y ="<<x<<' '<<y<<endl;

		// skip empty rows
		if (table->text(row,indexy).length()==0)
			continue;

		if(!finite(x)) x=0;
		if(!finite(y)) y=0;
		if(!finite(z)) z=0;
		
		if(row==start) {
			xmin=xmax=x;
			ymin=ymax=y;
			zmin=zmax=z;
		}
		
		x<xmin?xmin=x:0;
		x>xmax?xmax=x:0;
		y<ymin?ymin=y:0;
		y>ymax?ymax=y:0;
		z<zmin?zmin=z:0;
		z>zmax?zmax=z:0;

		ptr[numrows].setPoint(x,y,z);
		
		QTableItem *item = table->item(row,indexy);
		if(item && item->wordWrap())
			ptr[numrows].setMasked(true);
		numrows++;
	};
	
	// make new Graph3D
	LRange range[3];
	range[0] = LRange(xmin,xmax);
	range[1] = LRange(ymin,ymax);
	range[2] = LRange(zmin,zmax);
	Style *style = defaultStyle();
	Symbol *symbol = defaultSymbol();
	
	Graph3D *g = new Graph3D(caption(),columnTitle(indexy),range,SSPREADSHEET,P2D,style,symbol,ptr,numrows,1);

	mw->addGraph3D(g,destination,P2D);
}

//! make 2d plot with x-y-dx-dy
void Spreadsheet::plot4DXYDXDY() {
	kdDebug()<<"Spreadsheet::plot4DXYDXDY()"<<endl;
	
	// if less than 4 columns selected -> select all columns
	int selected_cols=0;
	for (int s=0;s<table->numSelections();s++) {
		QTableSelection ts = table->selection(s);
#if QT_VERSION > 0x030102
		selected_cols+=ts.numCols();
#else
		selected_cols+=ts.rightCol()-ts.leftCol();
#endif
	}
        if(selected_cols<3)
		for (int i=0;i<table->numCols();i++)
			table->selectColumn(i);

	// find x,y,dy column
	int indexx=-1,indexy=-1,indexz=-1, indext=-1;
	for(int i=table->numCols()-1;i>=0;i--) {
		if(table->isColumnSelected(i) && table->horizontalHeader()->label(i).findRev("[X]")>0)
			indexx=i;
		if(table->isColumnSelected(i) && table->horizontalHeader()->label(i).findRev("[Y]")>0)
			indexy=i;
		if(table->isColumnSelected(i) && table->horizontalHeader()->label(i).findRev("[DX]")>0)
			indexz=i;
		if(table->isColumnSelected(i) && table->horizontalHeader()->label(i).findRev("[DY]")>0)
			indext=i;
	}

	Point4D *ptr = new Point4D[table->numRows()];
	// read x and y values
	double x, y, z, t, xmin=0,xmax=1,ymin=0,ymax=1, zmin=0, zmax=1, tmin=0,tmax=1;
	int numrows=0;
	
	// only selected rows
	QTableSelection ts = table->selection(table->currentSelection());
//	kdDebug()<<"NUM SEL = "<<table->numSelections()<<endl;
	int start=0, end=table->numRows();
	if(ts.bottomRow()>ts.topRow()) {
		start = ts.topRow();
		end =  ts.bottomRow();
	}
	kdDebug()<<"reading ROWS = "<<start<<' '<<end<<endl;
	for (int row = start; row < end + 1; row++) {
		if(indexx == -1)
			x = (double) row + 1;
		else
			x = mw->formatLabel(table->text(row,indexx),formatItem(indexx));
		y = mw->formatLabel(table->text(row,indexy),formatItem(indexy));
		z = mw->formatLabel(table->text(row,indexz),formatItem(indexz));
		t = mw->formatLabel(table->text(row,indext),formatItem(indext));
		//kdDebug()<<"x/y ="<<x<<' '<<y<<endl;
		
		// skip empty rows
		if (table->text(row,indexy).length()==0)
			continue;

		if(!finite(x)) x=0;
		if(!finite(y)) y=0;
		if(!finite(z)) z=0;
		if(!finite(t)) t=0;
		
		if(row==start) {
			xmin=xmax=x;
			ymin=ymax=y;
			zmin=zmax=z;
			tmin=tmax=t;
		}
		
		x<xmin?xmin=x:0;
		x>xmax?xmax=x:0;
		y<ymin?ymin=y:0;
		y>ymax?ymax=y:0;
		z<zmin?zmin=z:0;
		z>zmax?zmax=z:0;
		t<tmin?tmin=t:0;
		t>tmax?tmax=t:0;

		ptr[numrows].setPoint(x,y,z,t);
		QTableItem *item = table->item(row,indexy);
		if(item && item->wordWrap())
			ptr[numrows].setMasked(true);
		numrows++;
	};
	
	// make new Graph4D
	LRange range[4];
	range[0] = LRange(xmin,xmax);
	range[1] = LRange(ymin,ymax);
	range[2] = LRange(zmin,zmax);
	range[3] = LRange(tmin,tmax);
	Style *style = defaultStyle();
	Symbol *symbol = defaultSymbol();
	
	Graph4D *g = new Graph4D(caption(),columnTitle(indexy),range,SSPREADSHEET,P2D,style,symbol,ptr,numrows,false);

	mw->addGraph4D(g,destination);
}

//! make 2d plot with x-y-dy1-dy2
void Spreadsheet::plot4DXYDYDY() {
	kdDebug()<<"Spreadsheet::plot4DXYDYDY()"<<endl;
	
	// if less than 4 columns selected -> select all columns
	int selected_cols=0;
	for (int s=0;s<table->numSelections();s++) {
		QTableSelection ts = table->selection(s);
#if QT_VERSION > 0x030102
		selected_cols+=ts.numCols();
#else
		selected_cols+=ts.rightCol()-ts.leftCol();
#endif
	}
        if(selected_cols<3)
		for (int i=0;i<table->numCols();i++)
			table->selectColumn(i);

	// find x,y,dy column
	int indexx=-1,indexy=-1,indexz=-1, indext=-1;
	for(int i=table->numCols()-1;i>=0;i--) {
		if(table->isColumnSelected(i) && table->horizontalHeader()->label(i).findRev("[X]")>0)
			indexx=i;
		if(table->isColumnSelected(i) && table->horizontalHeader()->label(i).findRev("[Y]")>0)
			indexy=i;
		if(table->isColumnSelected(i) && table->horizontalHeader()->label(i).findRev("[DY]")>0)
			indexz=i;
		if(table->isColumnSelected(i) && table->horizontalHeader()->label(i).findRev("[DY2]")>0)
			indext=i;
	}

	Point4D *ptr = new Point4D[table->numRows()];
	// read x and y values
	double x, y, z, t, xmin=0,xmax=1,ymin=0,ymax=1, zmin=0, zmax=1, tmin=0,tmax=1;
	int numrows=0;
	
	// only selected rows
	QTableSelection ts = table->selection(table->currentSelection());
//	kdDebug()<<"NUM SEL = "<<table->numSelections()<<endl;
	int start=0, end=table->numRows();
	if(ts.bottomRow()>ts.topRow()) {
		start = ts.topRow();
		end =  ts.bottomRow();
	}
	kdDebug()<<"reading ROWS = "<<start<<' '<<end<<endl;
	for (int row=start;row<end+1;row++) {
		if(indexx==-1)
			x = (double) row + 1;
		else
			x = mw->formatLabel(table->text(row,indexx),formatItem(indexx));
		y = mw->formatLabel(table->text(row,indexy),formatItem(indexy));
		z = mw->formatLabel(table->text(row,indexz),formatItem(indexz));
		t = mw->formatLabel(table->text(row,indext),formatItem(indext));
		//kdDebug()<<"x/y ="<<x<<' '<<y<<endl;

		// skip empty rows
		if (table->text(row,indexy).length()==0)
			continue;

		if(!finite(x)) x=0;
		if(!finite(y)) y=0;
		if(!finite(z)) z=0;
		if(!finite(t)) t=0;
		
		if(row==start) {
			xmin=xmax=x;
			ymin=ymax=y;
			zmin=zmax=z;
			tmin=tmax=t;
		}

		x<xmin?xmin=x:0;
		x>xmax?xmax=x:0;
		y<ymin?ymin=y:0;
		y>ymax?ymax=y:0;
		z<zmin?zmin=z:0;
		z>zmax?zmax=z:0;
		t<tmin?tmin=t:0;
		t>tmax?tmax=t:0;

		ptr[numrows].setPoint(x,y,z,t);
		QTableItem *item = table->item(row,indexy);
		if(item && item->wordWrap())
			ptr[numrows].setMasked(true);
		numrows++;
	};
	
	// make new Graph4D
	LRange range[4];
	range[0] = LRange(xmin,xmax);
	range[1] = LRange(ymin,ymax);
	range[2] = LRange(zmin,zmax);
	range[3] = LRange(tmin,tmax);
	Style *style = defaultStyle();
	Symbol *symbol = defaultSymbol();

	Graph4D *g = new Graph4D(caption(),columnTitle(indexy),range,SSPREADSHEET,P2D,style,symbol,ptr,numrows,true);

	mw->addGraph4D(g,destination);
}

void Spreadsheet::plot2D(PType type, bool line) {
	kdDebug()<<"Spreadsheet::plot2D() : "<<type<<endl;

	// if less than 2 columns selected -> select all columns
	int selected_cols=0;
	for (int s=0;s<table->numSelections();s++) {
		QTableSelection ts = table->selection(s);
#if QT_VERSION > 0x030102
		selected_cols+=ts.numCols();
#else
		selected_cols+=ts.rightCol()-ts.leftCol();
#endif
	}
        if(selected_cols<1 && !line)
		for (int i=0;i<table->numCols();i++)
			table->selectColumn(i);

	// find x column
	int indexx=-1;
	for(int i=table->numCols()-1;i>=0;i--) {
		// use only selected columns
		if(table->isColumnSelected(i) && table->horizontalHeader()->label(i).findRev("[X]")>0)
			indexx=i;
	}

	if(line) {
		Point *ptr = new Point[table->numCols()];
		// read x and y values
		double x, y, xmin=0,xmax=1,ymin=0,ymax=1;

		for (int s=0;s<table->numSelections();s++) {
			QTableSelection ts = table->selection(s);
			kdDebug()<<" rows "<<ts.topRow()<<" to row "<<ts.bottomRow()<<endl;
	
			for (int row=ts.topRow();row<=ts.bottomRow();row++) {
				for (int col=0;col<table->numCols();col++) {
					x =(double) col+1;
					// formatItem should be double for all columns. datetime ?
					y = mw->formatLabel(table->text(row,col),formatItem(col));
					kdDebug()<<"col/row : " <<col<<'/'<<row<<endl;
					
					if(!finite(x)) x=0;
					if(!finite(y)) y=0;
				
					if(col==0) {
						xmin=xmax=x;
						ymin=ymax=y;
					}
			
					x<xmin?xmin=x:0;
					x>xmax?xmax=x:0;
					y<ymin?ymin=y:0;
					y>ymax?ymax=y:0;
	
					ptr[col].setPoint(x,y);
					QTableItem *item = table->item(row,col+1);
					if(item && item->wordWrap())
						ptr[col].setMasked(true);
				}
		
				// make new Graph2D
				LRange range[2];
				range[0] = LRange(xmin,xmax);
				range[1] = LRange(ymin,ymax);
				Style *style = defaultStyle();
				Symbol *symbol = defaultSymbol();
	
				Graph2D *g = new Graph2D(caption(),i18n("row")+' '+QString::number(row+1),range,
					SSPREADSHEET,type,style,symbol,ptr,table->numCols());

				// use the same destination for every row
				destination = mw->addGraph2D(g,destination,type);
				kdDebug()<<"Setting destination to "<<destination<<endl;
			}
		}
	}
	else {	// no line data
		for(int i=0;i<table->numCols();i++) {
			// use only selected y cols
			if(table->isColumnSelected(i) && table->horizontalHeader()->label(i).findRev("[Y]")>0) {
				Point *ptr = new Point[table->numRows()];
				// read x and y values
				double x, y, xmin=0,xmax=1,ymin=0,ymax=1;
				int numrows=0;

				// only selected rows
				QTableSelection ts = table->selection(table->currentSelection());
//				kdDebug()<<"NUM SEL = "<<table->numSelections()<<endl;
				int start=0, end=table->numRows();
				if(ts.bottomRow()>ts.topRow()) {
					start = ts.topRow();
					end =  ts.bottomRow();
				}
				kdDebug()<<"reading ROWS = "<<start<<' '<<end<<endl;
				for (int row = start; row < end+1; row++) {
					if(indexx == -1)
						x =(double) row + 1;
					else
						x = mw->formatLabel(table->text(row,indexx),formatItem(indexx));

					// skip empty rows
					if (table->text(row,i).length()==0)
						continue;

					y = mw->formatLabel(table->text(row,i),formatItem(i));
					//kdDebug()<<"x/y ="<<x<<' '<<y<<endl;

					if(!finite(x)) x=0;
					if(!finite(y)) y=0;
			
					if(row==start) {
						xmin=xmax=x;
						ymin=ymax=y;
					}
			
					x<xmin?xmin=x:0;
					x>xmax?xmax=x:0;
					y<ymin?ymin=y:0;
					y>ymax?ymax=y:0;
	
					ptr[numrows].setPoint(x,y);
					QTableItem *item = table->item(row,i);
					if(item && item->wordWrap())
						ptr[numrows].setMasked(true);
					numrows++;
				}
				kdDebug()<<"xmin/xmax = "<<(int)xmin<<' '<<(int)xmax<<endl;
		
				// make new Graph2D
				LRange range[2];
				range[0] = LRange(xmin,xmax);
				range[1] = LRange(ymin,ymax);
				Style *style = defaultStyle();
				Symbol *symbol = defaultSymbol();
	
				Graph2D *g = new Graph2D(caption(),columnTitle(i),range,SSPREADSHEET,
					type,style,symbol,ptr,numrows);
	
				// use the same destination for every row
				destination = mw->addGraph2D(g,destination,type);
				kdDebug()<<"Setting destination to "<<destination<<endl;
			}
		}
	}
}

void Spreadsheet::plot3D(PType type) {
	kdDebug()<<"Spreadsheet::plot3D() : "<<type<<endl;
	
	// if less than 3 columns selected -> select all columns
	int selected_cols=0;
	for (int s=0;s<table->numSelections();s++) {
		QTableSelection ts = table->selection(s);
#if QT_VERSION > 0x030102
		selected_cols+=ts.numCols();
#else
		selected_cols+=ts.rightCol()-ts.leftCol();
#endif
	}
        if(selected_cols<3)
		for (int i=0;i<table->numCols();i++)
			table->selectColumn(i);

	// find x,y,z column
	int indexx=-1, indexy=-1, indexz=-1;
	for(int i=table->numCols()-1;i>=0;i--) {
		if(table->isColumnSelected(i) && table->horizontalHeader()->label(i).findRev("[X]")>0)
			indexx=i;
		if(table->isColumnSelected(i) && table->horizontalHeader()->label(i).findRev("[Y]")>0)
			indexy=i;
		if(table->isColumnSelected(i) && table->horizontalHeader()->label(i).findRev("[Z]")>0)
			indexz=i;
	}

	// use all z cols
	for(int i=0;i<table->numCols();i++) {
		// if no Z column : use the last one
		if((table->isColumnSelected(i) && table->horizontalHeader()->label(i).findRev("[Z]")>0) ||
			(indexz==-1 && i==table->numCols()-1)) {
			Point3D *ptr = new Point3D[table->numRows()];
			// read x and y values
			double xmin=0,xmax=1,ymin=0,ymax=1,zmin=0,zmax=1;
			int numrows=0;

			// only selected rows
			QTableSelection ts = table->selection(table->currentSelection());
//			kdDebug()<<"NUM SEL = "<<table->numSelections()<<endl;
			int start=0, end=table->numRows();
			if(ts.bottomRow()>ts.topRow()) {
				start = ts.topRow();
				end =  ts.bottomRow();
			}
			kdDebug()<<"reading ROWS = "<<start<<' '<<end<<endl;
			for (int row = start; row < end + 1; row++) {
				double x = mw->formatLabel(table->text(row,indexx),formatItem(indexx));
				double y = mw->formatLabel(table->text(row,indexy),formatItem(indexy));
				double z = mw->formatLabel(table->text(row,i),formatItem(i));
				// kdDebug()<<"x/y/z ="<<x<<' '<<y<<' '<<z<<endl;
		
				// skip empty rows
				if (table->text(row,i).length()==0)
					continue;
				
				if(!finite(x)) x=0;
				if(!finite(y)) y=0;
				if(!finite(z)) z=0;
		
				if (row == start) {
					kdDebug()<<"TYPE = "<<type<<endl; 
					if (type == PTERNARY) {
						xmin=0;
						xmax=x+y+z;
						kdDebug()<<"TERNARY MAX = "<<xmax<<endl; 
					}
					else
						xmin=xmax=x;	
					ymin=ymax=y;
					zmin=zmax=z;
				}
				else {
					if (type != PTERNARY) {
						x<xmin?xmin=x:0;
						x>xmax?xmax=x:0;
					}
					y<ymin?ymin=y:0;
					y>ymax?ymax=y:0;
					z<zmin?zmin=z:0;
					z>zmax?zmax=z:0;
				}
	
				ptr[numrows].setPoint(x,y,z);
				QTableItem *item = table->item(row,indexy);
				if(item && item->wordWrap())
					ptr[numrows].setMasked(true);
				numrows++;
			};
	
			// make new Graph3D
			LRange range[3];
			range[0] = LRange(xmin,xmax);
			range[1] = LRange(ymin,ymax);
			range[2] = LRange(zmin,zmax);
			Style *style = defaultStyle();
			Symbol *symbol = defaultSymbol();
			kdDebug()<<"X RANGE : "<<xmin<<' '<<xmax<<endl;
			kdDebug()<<"Y RANGE : "<<ymin<<' '<<ymax<<endl;
			kdDebug()<<"Z RANGE : "<<zmin<<' '<<zmax<<endl;
	
			int cols = table->numRows();
			if(type == PTERNARY)
				cols=1;
			else if (type == PSURFACE || type == P3D || type == PQWT3D ) {
				// calculate dimension
				for(int i=0;i<table->numRows();i++) {
					if(ptr[i].X()>ptr[i+1].X() || ptr[i].Y()>ptr[i+1].Y() ) {
						cols=i+1;
						i=table->numRows();
					}
				}
				numrows /= cols;

				// if unclear : ask for dimensions
				if(numrows <= 1 || cols <= 1) {
					bool ok;
					int res = QInputDialog::getInteger(i18n("Surface plot from X-Y-Z"),
						i18n("Please specify dimension in X direction : "),numrows,2,table->numRows()/2,1,&ok);
					if(ok)
						numrows=res;
					else
						return;
					cols = table->numRows()/numrows;
 				}
			}
			kdDebug()<<"	numrows/cols = "<<numrows<<'/'<<cols<<endl;
			Graph3D *g = new Graph3D(caption(),columnTitle(i),range,SSPREADSHEET,type,style,symbol,ptr,cols,numrows);

			// use the same destination for every row
			destination=mw->addGraph3D(g,destination,type);
			kdDebug()<<"Setting destination to "<<destination<<endl;
		}
	}
}

void Spreadsheet::plotMatrix(PType type) {
	kdDebug()<<"Spreadsheet::plotMatrix() : "<<type<<endl;

	int NX = table->numCols(), NY = table->numRows();
	double *a = new double[NY*NX];
	
	double z, zmin=0, zmax=1;
	// TODO : skip empty lines
	// TODO : only selected rows/cols
	for(int i=0;i<NY;i++) {
		for(int j=0;j<NX;j++) {
			z = mw->formatLabel(table->text(i,j),formatItem(j));
			if (!finite(z)) z=0;

			if(i==0 && j==0) {
				zmin=zmax=z;
			}

			if(!finite(z)) z=0;

			z<zmin?zmin=z:0;
			z>zmax?zmax=z:0;

			a[j+NX*i] = z;
		}
	}

	// make new GraphM
	LRange range[3];
	range[0] = LRange(0,NX);
	range[1] = LRange(0,NY);
	range[2] = LRange(zmin,zmax);

	Style *style= new Style(0);
	Symbol *symbol= new Symbol(SNONE);
	
	GraphM *g = new GraphM(caption(),i18n("Matrix"),range,SSPREADSHEET,type,style,symbol,a,NX,NY);

	mw->addGraphM(g,destination,type);
}

void Spreadsheet::setTitle(QString t) {
	kdDebug()<<"Spreadsheet::setTitle() : "<<t<<endl;
	bool ok=true;
	if(t.length()==0)
		t = QInputDialog::getText("LabPlot", i18n("Spreadsheet title : "),QLineEdit::Normal,title, &ok, this );
	if(!ok) return;

	if(t.length()>0) {
		title = t.remove(QChar('&'));
		setCaption(title);
		mw->updateSheetList();
	}
}

void Spreadsheet::selectDestination() {
	(new DestinationDialog(mw,0))->show();
}

void Spreadsheet::addGraph2D(Graph2D *g) {
	kdDebug()<<"Spreadsheet::addGraph2D()"<<endl;
	// add empty cols if necessary
	if(filledRows(table->numCols()-2)>1)
		table->setNumCols(table->numCols()+1);
	if(filledRows(table->numCols()-2)>1)
		table->setNumCols(table->numCols()+1);

	table->setNumRows(g->Number());
	
	Point *d = g->Data();
	table->horizontalHeader()->setLabel(table->numCols()-2 , QString( "A ")+i18n("{double}")+" [X]");
	table->horizontalHeader()->setLabel(table->numCols()-1 , QString( "B ")+i18n("{double}")+" [Y]");
	for(int i=0;i<g->Number();i++) {
		table->setText(i,table->numCols()-2,QString::number(d[i].X()));
		table->setText(i,table->numCols()-1,QString::number(d[i].Y()));
	}	
}

void Spreadsheet::addGraph3D(Graph3D *g) {
	kdDebug()<<"Spreadsheet::addGraph3D()"<<endl;
	// add empty cols if necessary
	if(filledRows(table->numCols()-3)>1)
		table->setNumCols(table->numCols()+1);
	if(filledRows(table->numCols()-3)>1)
		table->setNumCols(table->numCols()+1);
	if(filledRows(table->numCols()-3)>1)
		table->setNumCols(table->numCols()+1);
	
	table->setNumRows(g->Number());
	
	Point3D *d = g->Data();
	table->horizontalHeader()->setLabel(table->numCols()-3 , QString( "A ")+i18n("{double}")+" [X]");
	table->horizontalHeader()->setLabel(table->numCols()-2 , QString( "B ")+i18n("{double}")+" [Y]");
	table->horizontalHeader()->setLabel(table->numCols()-1 , QString( "C ")+i18n("{double}")+" [Z]");
	for(int i=0;i<g->Number();i++) {
		table->setText(i,table->numCols()-3,QString::number(d[i].X()));
		table->setText(i,table->numCols()-2,QString::number(d[i].Y()));
		table->setText(i,table->numCols()-1,QString::number(d[i].Z()));
	}	
}

void Spreadsheet::addGraph4D(Graph4D *g) {
	kdDebug()<<"Spreadsheet::addGraph4D()"<<endl;
	// add empty cols if necessary
	if(filledRows(table->numCols()-4)>1)
		table->setNumCols(table->numCols()+1);
	if(filledRows(table->numCols()-4)>1)
		table->setNumCols(table->numCols()+1);
	if(filledRows(table->numCols()-4)>1)
		table->setNumCols(table->numCols()+1);
	if(filledRows(table->numCols()-4)>1)
		table->setNumCols(table->numCols()+1);
	
	table->setNumRows(g->Number());
	
	Point4D *d = g->Data();
	table->horizontalHeader()->setLabel(table->numCols()-4 , QString( "A ")+i18n("{double}")+" [X]");
	table->horizontalHeader()->setLabel(table->numCols()-3 , QString( "B ")+i18n("{double}")+" [Y]");
	table->horizontalHeader()->setLabel(table->numCols()-2 , QString( "C ")+i18n("{double}")+" [DX]");
	table->horizontalHeader()->setLabel(table->numCols()-1 , QString( "D ")+i18n("{double}")+" [DY]");
	for(int i=0;i<g->Number();i++) {
		table->setText(i,table->numCols()-4,QString::number(d[i].X()));
		table->setText(i,table->numCols()-3,QString::number(d[i].Y()));
		table->setText(i,table->numCols()-2,QString::number(d[i].Z()));
		table->setText(i,table->numCols()-1,QString::number(d[i].T()));
	}	
}

void Spreadsheet::addGraphM(GraphM *g) {
	kdDebug()<<"Spreadsheet::addGraphM()"<<endl;
	int nx = g->NX();
	int ny = g->NY();
	double *data = g->Data();
	
	table->setNumCols(nx);
	table->setNumRows(ny);
	
	for (int i=0;i<ny;i++) {
		for (int j=0;j<nx;j++) {
			table->setText(i,j,QString::number(data[j+ny*i]));
		}
	}
}

//! create Graph2D from table data
Graph2D* Spreadsheet::getGraph2D() {
	kdDebug()<<"Spreadsheet::getGraph2D()"<<endl;
	int n=table->numRows();
	Point *ptr = new Point[n];

	double ymin=0,ymax=1, xmin=0, xmax=1;
	for(int i = 0;i <n ;i++) {
		double x = table->text(i,0).toDouble();
		double y = table->text(i,1).toDouble();

		ptr[i].setPoint(x,y);
	}
	
	mw->calculateRanges2D(ptr,n,&xmin,&xmax,&ymin,&ymax);

	QString fun("2d data");
	QString label("data");
	LRange range[2];
	range[0] = LRange(xmin,xmax);
	range[1] = LRange(ymin,ymax);

	Graph2D *g = new Graph2D(fun,label,range,SSPREADSHEET,P2D,0,0,ptr,n);

	return g;
}

Graph3D* Spreadsheet::getGraph3D() {
	kdDebug()<<"Spreadsheet::getGraph3D() TODO"<<endl;
	
	int n=table->numRows();
	Point3D *ptr = new Point3D[n];

	double ymin=0,ymax=1, xmin=0, xmax=1,zmin=0,zmax=1;
	for(int i = 0;i < n ;i++) {
		double x = table->text(i,0).toDouble();
		double y = table->text(i,1).toDouble();
		double z = table->text(i,2).toDouble();

		ptr[i].setPoint(x,y,z);

	}
	
	mw->calculateRanges3D(ptr,n,&xmin,&xmax,&ymin,&ymax,&zmin,&zmax);

	QString fun("3d data");
	QString label("data");
	LRange range[3];
	range[0] = LRange(xmin,xmax);
	range[1] = LRange(ymin,ymax);
	range[2] = LRange(zmin,zmax);

	Graph3D *g = new Graph3D(fun,label,range,SSPREADSHEET,P2D,0,0,ptr,n);
	
	return g;
}

Graph4D* Spreadsheet::getGraph4D() {
	kdDebug()<<"Spreadsheet::getGraph4D() TODO"<<endl;

	int n=table->numRows();
	Point4D *ptr = new Point4D[n];

	double ymin=0,ymax=1, xmin=0, xmax=1,zmin=0,zmax=1,tmin=0,tmax=1;
	for(int i = 0;i < n ;i++) {
		double x = table->text(i,0).toDouble();
		double y = table->text(i,1).toDouble();
		double z = table->text(i,2).toDouble();
		double t = table->text(i,3).toDouble();

		ptr[i].setPoint(x,y,z,t);
	}
	
	mw->calculateRanges4D(ptr,n,&xmin,&xmax,&ymin,&ymax,&zmin,&zmax,&tmin,&tmax);

	QString fun("4d data");
	QString label("data");
	LRange range[4];
	range[0] = LRange(xmin,xmax);
	range[1] = LRange(ymin,ymax);
	range[2] = LRange(zmin,zmax);
	range[4] = LRange(tmin,tmax);

	Graph4D *g = new Graph4D(fun,label,range,SSPREADSHEET,P2D,0,0,ptr,n);
	
	return g;
}

GraphM* Spreadsheet::getGraphM() {
	kdDebug()<<"Spreadsheet::getGraphM()"<<endl;
	int nx = table->numCols();
	int ny = table->numRows();

	double *data = new double[nx*ny];

	double zmin=0,zmax=1;
	for (int i=0;i<ny;i++) {
		for (int j=0;j<nx;j++) {
			double z = table->text(i,j).toDouble();
			if (i == 0 && j == 0) {
				zmin=z;
				zmax=z;
			}

			z<zmin?zmin=z:0;
			z>zmax?zmax=z:0;
			
			data[j+nx*i] = z;
		}
	}

	QString fun("2d data");
	QString label("data");
	LRange range[3];
	// old : used 0..NX,0..NY
	// new : use range
	range[0] = LRange(0,nx);
	range[1] = LRange(0,ny);
	range[2] = LRange(zmin,zmax);
		
	Style *style = new Style(0);
	Symbol *symbol = new Symbol(SNONE);
	GraphM *g = new GraphM(fun,label,range,SSPREADSHEET,PSURFACE,style,symbol,data,nx,ny);
	
	return g;
}

void Spreadsheet::cutSelection() {
	copySelection();
	clearSelection();
}

void Spreadsheet::copySelection() {
	QString text;
	for (int s=0; s<table->numSelections();s++) {
		QTableSelection ts = table->selection(s);
	
		for (int i=ts.topRow();i<=ts.bottomRow();i++) {
			for (int j=ts.leftCol();j<ts.rightCol();j++)
				text += table->text(i,j)+"\t";
			text += table->text(i,ts.rightCol())+"\n";
		}
	}
	// Copy text into the clipboard
	QApplication::clipboard()->setText(text);
	
	table->repaintContents();
}

void Spreadsheet::pasteSelection() {
	QString text = QApplication::clipboard()->text();

	QStringList rowTexts = QStringList::split ("\n",text,true);
	int rows = int(rowTexts.count())-1;
	QStringList cellTexts = QStringList::split ("\t",rowTexts[0],true);
	int cols = int(cellTexts.count());

	int top,bottom,right,left;
	
	QTableSelection ts = table->selection(table->currentSelection());
#if QT_VERSION > 0x030102
	if (!ts.isEmpty()) {
#else
	if (!((ts.bottomRow()-ts.topRow())*(ts.rightCol()-ts.leftCol()))) {
#endif
		top=ts.topRow();
		bottom=ts.bottomRow();
		left=ts.leftCol();
		right=ts.rightCol();
	}
	else {
		top=0;
		bottom=rows;
		left=ts.leftCol();
		right=ts.leftCol()+cols;
	
		int tableCols=table->numCols();
		if (right>tableCols)
			right=tableCols-1;
	}
	
	QStringList allCells;	
	for (int i=0;i<rows;i++) {
		cellTexts=QStringList::split ("\t",rowTexts[i],true);
		for (int j=0;j<cols;j++)
			allCells<<cellTexts[j];
	}
	
	// currentSelection < text
	int col=right-left+1;
	if (rows>(bottom-top+1) || cols>col) {
		switch( KMessageBox::questionYesNo(this,
			i18n("The text in the clipboard is larger then your current selection!\n")+
				i18n("Do you want to adjust it to the current selection?")))  {
		case KMessageBox::Yes:
			for (int i=top;i<=bottom;i++) {
				for (int j=left;j<=right;j++)
					table->setText(i,j,allCells[(i-top)*cols+j-left]);
			}
			return;
			break;
		}
	}
	
	// insert text
	for (int i=top;i<top+rows;i++) {
		for (int j=left;j<left+cols;j++)
			table->setText(i,j,allCells[(i-top)*cols+(j-left)]);
	}
	
	table->repaintContents();
}

void Spreadsheet::clearSelection() {
	for (int s=0; s<table->numSelections();s++) {
		QTableSelection ts = table->selection(s);
	
		for(int i=ts.topRow();i<=ts.bottomRow();i++) {
			for(int j=ts.leftCol();j<=ts.rightCol();j++) {
				table->clearCell(i,j);
				table->updateCell(i,j);
			}
		}
	}
	table->repaintContents();
}

void Spreadsheet::selectAll() {
	for(int i=0;i<=table->numCols();i++)
		table->selectColumn(i);
}

void Spreadsheet::selectNone() {
	table->clearSelection();
}

void Spreadsheet::selectInvert() {
	int rows = table->numRows();
	bool *selected = new bool[rows];
	for(int i=0;i<rows;i++) {
		if(table->isRowSelected(i)) {
//			kdDebug()<<"	ROW "<<i+1<<" is SELECTED"<<endl;
			selected[i]=true;
		}
		else
			selected[i]=false;
	}

	table->clearSelection();
	for(int i=0;i<rows;i++) {
		if(!selected[i])
			table->selectRow(i);
	}
}

void Spreadsheet::setValues(int startrow, int endrow, QString expression) {
	if(expression.length()<1)
		(new SpreadsheetValuesDialog(mw,table,caption()))->show();
	else{
		if (endrow<=0)
			endrow=table->numRows();
			
		for(int i=startrow;i<=endrow;i++) {
			QString tmp(expression.lower());
		
			// TODO
			// replace a,b,c,d,...
			for(int j=0;j<table->numCols();j++)
				tmp = mw->parseExpression(tmp,table->text(i-1,j).toDouble(),j);
		
			double result = parse((char *) tmp.latin1());
			
			if(parse_errors()>0) {
				KMessageBox::error(mw, i18n("Parse Error!\n Please check the given function."));
				return;
			}
			
			table->setText(i-1,table->currentColumn(),QString::number(result));
		}
	}	
}

void Spreadsheet::fillRowNumber() {
	for(int i=0;i<table->numRows();i++)
		table->setText(i,table->currentColumn(),QString::number(i+1));	
}

void Spreadsheet::fillRandom(double max) {
	srandom(time(0));
	for(int i=0;i<table->numRows();i++)
		table->setText(i,table->currentColumn(),QString::number(max*random()/(double)RAND_MAX));
}

void Spreadsheet::normSum() {
	// calculate sum
	double sum=0;
	int row=0, col = table->currentColumn();
	do {
		sum += table->text(row,col).toDouble();
		row++;
	}while(table->text(row,0).length()>0);
	
	// set value;
	row=0;
	do {
		double value = table->text(row,col).toDouble();
		table->setText(row,col,QString::number(value/sum));
		row++;
	}while(table->text(row,0).length()>0);
}

void Spreadsheet::normMax(double maximum) {
	// calculate max
	double max=0;
	int row=0, col = table->currentColumn();
	do {
		double value = table->text(row,col).toDouble();
		if(row==0)
			max=value;
		if(max<value)
			max = value;
		row++;
	}while(table->text(row,0).length()>0);
	
	// set value;
	row=0;
	do {
		double value = table->text(row,col).toDouble();
		table->setText(row,col,QString::number(maximum*value/max));
		row++;
	}while(table->text(row,0).length()>0);
}

void Spreadsheet::editEditor(int editor) {
	kdDebug()<<"	edit with editor "<<editor<<endl;
	KProcess* proc = new KProcess;

	switch(editor) {
	case 0:	*proc << "xterm"<<"-e"<<"vim"; break;
	case 1:	*proc << "kvim"<<"-f"; break;
	case 2:	*proc << "gvim"<<"-f"; break;
	case 3:	*proc << "kate" << "-n"; break;
	case 4:	*proc << "kwrite"; break;
	case 5:	*proc << "emacs"; break;
	case 6:	*proc << "xemacs"; break;
	case 7:	*proc << "kword"; break;
	case 8:	*proc << "soffice"; break;
	}

	// tmp file for vim;
	KTempFile *tmpfile = new KTempFile(QString::null,".dat");
	tmpfile->setAutoDelete(true);
	tmpfilename = tmpfile->name();

	// file as argument
	*proc<<tmpfilename;

	QTextStream *t = tmpfile->textStream();

	for (int i=0;i<table->numRows();i++) {
		for (int j=0;j<table->numCols();j++) {
				*t << table->text(i,j).toDouble()<<' ';
		}
		*t<<endl;
	}
	tmpfile->close();

	// TODO ; "kate" doesn't work : exit signal shortly after start (background job)
	connect(proc, SIGNAL(processExited(KProcess*)),this, SLOT(readfile(KProcess*)));
	if ( !proc->start(KProcess::NotifyOnExit) )
		KMessageBox::error( this,i18n("Could not start selected Editor."));
}

//read data from tmp file into table
void Spreadsheet::readfile(KProcess *) {
	kdDebug()<<"READING "<<tmpfilename<<endl;
	QFile file(tmpfilename);
	if ( ! file.open( IO_ReadOnly )) {
		KMessageBox::error(this,i18n("temporary data file not found!"));
		return;
	}
	QTextStream t(&file);

	int i=0;
	while(!(t.atEnd())) {
		QStringList line = QStringList::split(' ',t.readLine());
		int j=0;
		for ( QStringList::Iterator it = line.begin(); it != line.end(); ++it ) {
			table->setText(i,j++,(*it));
		}
		i++;
		// add additional rows
		if(i>=table->numRows())
			table->setNumRows ( table->numRows()+10 );
	}

	table->setNumRows ( i );

	file.close();
}

void Spreadsheet::transposeMatrix() {
	int nx = table->numCols();
	int ny = table->numRows();

	QString *data = new QString[nx*ny];
	for(int i=0;i<ny;i++)
		for(int j=0;j<nx;j++)
			data[j+i*nx]=table->text(i,j);

	table->setNumRows(nx);
	table->setNumCols(ny);
			
	for(int i=0;i<ny;i++)
		for(int j=0;j<nx;j++)
			table->setText(j,i,data[j+i*nx]);
}

void Spreadsheet::convertMatrixtoXYZ() {
	int nx = table->numCols();
	int ny = filledRows();
	
	double *data = new double[nx*ny];
	
	for(int i=0;i<ny;i++)
		for(int j=0;j<nx;j++)
			data[j+i*nx]=table->text(i,j).toDouble();

	table->setNumCols(3);
	table->horizontalHeader()->setLabel( 2, QString( "C ")+i18n("{double}")+" [Z]");
	table->setNumRows(nx*ny);
	for(int i=0;i<nx*ny;i++) {
		table->setText(i,0,QString::number(i%nx+1));
		table->setText(i,1,QString::number(i/nx+1));
		table->setText(i,2,QString::number(data[i]));
	}
}

void Spreadsheet::convertXYZtoMatrix() {
	int rows = filledRows();
	
	// find x,y,z column
	int indexx=-1, indexy=-1, indexz=-1;
	for(int i=table->numCols();i>=0;i--) {
		if(table->horizontalHeader()->label(i).findRev("[X]")>0)
			indexx=i;
		if(table->horizontalHeader()->label(i).findRev("[Y]")>0)
			indexy=i;
		if(table->horizontalHeader()->label(i).findRev("[Z]")>0)
			indexz=i;
	}
	
	// sort y
	table->selectColumn(indexy);
	sortAscending();
	// TODO : sort x in segment too
	
	double *data = new double[rows];
	for (int i=0;i<rows;i++)
		data[i]=table->text(i,indexz).toDouble();
	
	// build table
	int nx = (int) sqrt((double)rows);
	table->setNumCols(nx);
	table->setNumRows(nx);
	table->selectColumn(-1);
	
	// fill table
	for(int i=0;i<nx;i++)
		for(int j=0;j<nx;j++)
			table->setText(j,i,QString::number(data[i+j*nx]));
}

void Spreadsheet::convertColumntoMatrix() {
	kdDebug()<<"Spreadsheet::convertColumntoMatrix()"<<endl;
	// get selected column and fill tmp vector
	QTableSelection ts = table->selection(0);
	int col = ts.leftCol(), rows = table->numRows();
	double tmp[rows];
	kdDebug()<<"	selected "<<rows<<" rows from column "<<col<<endl;
	
	if(col==-1) col=0;

	for(int i=0;i<rows;i++)
		tmp[i]=table->text(i,col).toDouble();

	// dialog : "split after n rows. n=..."
	bool ok;
	int n = QInputDialog::getInteger(i18n("Convert Column -> Matrix"),i18n("split after row"),100,1,INF,1,&ok);
	if(!ok)
		return;
	kdDebug()<<"	split after row "<<n<<endl;

	// do conversion
	table->setNumRows(n);
	int entry=0, tmpcol=-1;
	while(entry<rows) {
		int row=entry%n;
		if(row==0) {
			tmpcol++;
			if(tmpcol>=table->numCols())
				addColumn();
		}
		table->setText(row,tmpcol,QString::number(tmp[entry]));

		entry++;
	}
}

void Spreadsheet::exportData() {
	(new DumpDialog(mw,"spreadsheet",-1))->show();
}

void Spreadsheet::selectColumns(int left, int right) {
	if(right==-1) right=left;
		
	kdDebug()<<"selecting columns "<<left<<" to "<<right<<endl;
	table->selectCells(0,left,table->numRows(),right);
}

void Spreadsheet::selectRows(int top, int bottom){
	if(bottom==-1) bottom=top;
	
	kdDebug()<<"selecting rows "<<top<<" to "<<bottom<<endl;
	table->selectCells(top,0,bottom,table->numCols());
}

// delete all currently selected rows
void Spreadsheet::deleteRows() {
	kdDebug()<<"Spreadsheet::deleteRows()"<<endl;
	table->setUpdatesEnabled(false);

	// much slower on SUSE 9.1 !
#if defined(HAVE_SGI_STL) && (QT_VERSION != 0x030301)
	vector<int> tmp;
	for (int s=0; s<table->numSelections();s++) {
		QTableSelection ts = table->selection(s);
		kdDebug()<<"removing row "<<ts.topRow()<<" to "<<ts.bottomRow()<<endl;

		for (int row=ts.topRow();row<=ts.bottomRow();tmp.push_back(row++))
		        ;
	}

	unsigned int rowcount = tmp.size();
	if (rowcount > 0) {
	        // sort temporary array to meet QTable::removeRows() requirements
	        stable_sort (tmp.begin(), tmp.end());

		QMemArray<int> selected_rows(rowcount);
		for (unsigned int i=0; i<rowcount; i++)
		        selected_rows[i] = tmp[i];

		table->removeRows(selected_rows);
	}
#else
	int rows=0;
	for (int s=0; s < table->numSelections(); s++) {
		QTableSelection ts = table->selection(s);
		kdDebug()<<"	removing rows "<<ts.topRow()+1<<" to "<<ts.bottomRow()+1<<endl;

		for (int row = ts.topRow(); row <= ts.bottomRow(); row++) {
			// move the row to the end
			for (int i = row-rows; i < table->numRows() - 1; i++)
				table->swapRows(i, i + 1);
			rows++;
		}
	}
	table->setNumRows(table->numRows() - rows);
	table->clearSelection();
#endif

	table->setCurrentCell(0,0);
	table->setUpdatesEnabled(true);
	table->repaintContents();
}

void Spreadsheet::addColumn() {
	int col = table->numCols();
	table->insertColumns(col);
	if(col<26)
		table->horizontalHeader()->setLabel( col, QChar(col+65)+' '+i18n("{double}")+" [Y]");
	else
		table->horizontalHeader()->setLabel( col, QChar(84)+' '+i18n("{double}")+" [Y]");
}

// delete all currently selected columns
void Spreadsheet::deleteColumns() {
	kdDebug()<<"Spreadsheet::deleteRows()"<<endl;
	table->setUpdatesEnabled(false);

#ifdef HAVE_SGI_STL
	vector<int> tmp;
	for (int s=0; s<table->numSelections();s++) {
		QTableSelection ts = table->selection(s);
		kdDebug()<<"removing column "<<ts.leftCol()<<" to "<<ts.rightCol()<<endl;

		for (int col=ts.leftCol();col<=ts.rightCol();tmp.push_back(col++))
		        ;
	}

	unsigned int colcount = tmp.size();
	if (colcount > 0) {
	        // sort temporary array to meet QTable::removeColumns() requirements
	        stable_sort (tmp.begin(), tmp.end());

		QMemArray<int> selected_cols(colcount);
		for (unsigned int i=0; i<colcount; i++)
		        selected_cols[i] = tmp[i];

		table->removeColumns(selected_cols);
	}
#else
	int cols=0;
	for (int s=0; s<table->numSelections();s++) {
		QTableSelection ts = table->selection(s);
		kdDebug()<<"removing column "<<ts.leftCol()<<" to column "<<ts.rightCol()<<endl;

		for (int col = ts.leftCol(); col <= ts.rightCol(); col++) {
			// move the col to the right
			for (int i = col-cols; i < table->numCols() - 1; i++)
				table->swapColumns(i, i + 1,true);
			cols++;
		}
	}
	table->setNumCols(table->numCols() - cols);
	table->clearSelection();
#endif

	table->setCurrentCell(0,0);
	table->setUpdatesEnabled(true);
	table->repaintContents();
}

void Spreadsheet::maskSelection() {
	for (int s=0; s<table->numSelections();s++) {
		QTableSelection ts = table->selection(s);

		for(int i=ts.topRow();i<=ts.bottomRow();i++) {
			for(int j=ts.leftCol();j<=ts.rightCol();j++) {
				QTableItem *item = table->item(i,j);
				bool masked=false;
				if(item)
					masked = item->wordWrap();	// HACK : use wordWrap as masked flag

				LTableItem *it;
				if(masked) {
					it = new LTableItem( table, QTableItem::OnTyping,table->text(i,j),Qt::black);
					it->setWordWrap(false);
				}
				else {
					it = new LTableItem( table, QTableItem::OnTyping,table->text(i,j),Qt::red);
					it->setWordWrap(true);
				}
				table->setItem( i, j, it);
			}
		}
	}
}

void Spreadsheet::toggleMask() {
	int nx = table->numCols();
	int ny = table->numRows();

	for(int i=0;i<ny;i++) {
		for(int j=0;j<nx;j++) {
			QTableItem *item = table->item(i,j);
			LTableItem *it;
			if(item->wordWrap()) {
				it = new LTableItem( table, QTableItem::OnTyping,table->text(i,j),Qt::black);
				it->setWordWrap(false);
			}
			else {
				it = new LTableItem( table, QTableItem::OnTyping,table->text(i,j),Qt::red);
				it->setWordWrap(true);
			}
			table->setItem( i, j, it);
		}
	}
}

void Spreadsheet::unMask() {
	int nx = table->numCols();
	int ny = table->numRows();

	for(int i=0;i<ny;i++) {
		for(int j=0;j<nx;j++) {
			LTableItem *it = new LTableItem( table, QTableItem::OnTyping,table->text(i,j),Qt::black);
			it->setWordWrap(false);
			table->setItem( i, j, it);
		}
	}
}

void Spreadsheet::maskNthRow() {
	bool ok;
	int n = QInputDialog::getInteger(i18n("Mask every n-th row"),i18n("n = "),10,1,INF,1,&ok);
	if(!ok) return;
		
	LTableItem *it;
	for(int i=0;i<table->numRows();i++) {
		if(i%n==0) {
			for(int j=0;j<table->numCols();j++) {
				it = new LTableItem( table, QTableItem::OnTyping,table->text(i,j),Qt::red);
				it->setWordWrap(true);
				table->setItem( i, j, it);
			}
		}
	}
}

void Spreadsheet::maskFirstRow() {
	bool ok;
	int n = QInputDialog::getInteger(i18n("Mask all but n-th row"),i18n("n = "),10,1,INF,1,&ok);
	if(!ok) return;
		
	LTableItem *it;
	for(int i=0;i<table->numRows();i++) {
		if(i%n) {
			for(int j=0;j<table->numCols();j++) {
				it = new LTableItem( table, QTableItem::OnTyping,table->text(i,j),Qt::red);
				it->setWordWrap(true);
				table->setItem( i, j, it);
			}
		}
	}
}

void Spreadsheet::sortAscending() {
	ascending=true;
	sort();
}

void Spreadsheet::sortDescending() {
	ascending=false;
	sort();
}

void Spreadsheet::sort() {
	for (int s=0; s<table->numSelections();s++) {
		QTableSelection ts = table->selection(s);
		if(ts.topRow()<ts.bottomRow())
			qsort(ts.topRow(),ts.bottomRow());
		else
			qsort(0,table->numRows()-1);
	}
	table->clearSelection();
	table->repaintContents();
}

void Spreadsheet::qsort(int s, int e) {
	if(e>s+1) {
		int col = table->currentColumn();
		double mid = table->text((s+e)/2,col).toDouble();
		
		int i=s, j=e;
		while (i<j) {
			if(ascending) {
				while (table->text(i,col).toDouble() < mid) i++;
				while (mid < table->text(j,col).toDouble())  j--;
			}
			else {
				while (table->text(i,col).toDouble() > mid) i++;
				while (mid > table->text(j,col).toDouble())  j--;
			}
			
			if(i<j) {
				table->swapRows(i,j);
				i++;j--;
			}
		}
		
		qsort(s,j);
		qsort(i,e);
	}
}

void Spreadsheet::setProperties(QString label, int type, int format, int rows) {
	if(label.length()<1)
		(new SpreadsheetPropertiesDialog(mw,table,caption()))->show();
	else {
		table->setNumRows(rows);
		table->horizontalHeader()->setLabel(table->currentColumn(),label +' '+ 
			formatList[format]+' '+columnTypeItems[type]);
	}
}

Style *Spreadsheet::defaultStyle() {
	kdDebug()<<"defaultStyle()"<<endl;
	KConfig *config = mw->Config();
	config->setGroup( "Plot Simple Style" );
	
	Style *style = new Style(
		config->readNumEntry("Graph Style",0),
		config->readColorEntry("Style Color",&Qt::blue),
		config->readBoolEntry("Filled",false),
		config->readColorEntry("Fill Color",&Qt::green),
		config->readNumEntry("Style Width",1),
		config->readNumEntry("Pen Style",1),
		config->readNumEntry("Brush",0)
		);
	
	style->setBoxWidth(config->readNumEntry("Box Width",10));
	style->setAutoBoxWidth(config->readBoolEntry("Auto Box Width",false));
		
/*	config->setGroup( "Plot Surface Style" );

	plot->enableDensity(config->readBoolEntry("Density Enabled",true));
	plot->enableContour(config->readBoolEntry("Contour Enabled",true));
	plot->setNumber(config->readNumEntry("Contour Level",10));
	plot->setPalette(config->readNumEntry("Palette",0));
	plot->setContourColor(config->readColorEntry("Contour Color",&Qt::black));
	plot->setColoredContour(config->readBoolEntry("Colored Contour",false));
	plot->setMesh(config->readBoolEntry("Show Mesh",false));
	plot->setRelative(config->readBoolEntry("Relative Colorscale",true));
	plot->setBrush(config->readNumEntry("Density Brush",1));
	plot->setThreshold(config->readDoubleNumEntry("Threshold",-INF));
*/
	return style;
}

Symbol *Spreadsheet::defaultSymbol() {
	kdDebug()<<"defaultSymbol()"<<endl;
	KConfig *config = mw->Config();
	config->setGroup( "Plot Simple Style" );
	
	return new Symbol(
		(SType) config->readNumEntry("Symbol Type",SNONE),
		config->readColorEntry("Symbol Color",&Qt::blue),
		config->readNumEntry("Symbol Size",5),
		(FType) config->readNumEntry("Symbol Fill",FNONE),
		config->readColorEntry("Symbol Fill Color",&Qt::red),
		config->readNumEntry("Symbol Brush",1));
}

