// Copyright 2008 Martin C. Frith

// This struct holds a pair of equal-length segments, in a pair of
// sequences.  In other words, it holds a gapless alignment.

#ifndef SEGMENTPAIR_HH
#define SEGMENTPAIR_HH

namespace cbrc{

struct SegmentPair{
  typedef unsigned indexT;
  typedef unsigned char uchar;

  enum { MAT = 64 };

  SegmentPair(){}

  SegmentPair( indexT s1, indexT s2, indexT sz, int sc = 0 )
    : start1(s1), start2(s2), size(sz), score(sc){}

  // Make a SegmentPair by doing gapless X-drop alignment in both
  // directions starting from a seed point.  The resulting SegmentPair
  // might not be "optimal" (see below).
  void makeXdrop( indexT seed1, indexT seed2,
		  const uchar* seq1, const uchar* seq2,
                  const int scoreMatrix[MAT][MAT], int maxDrop );

  // Check that the SegmentPair has no prefix with score <= 0, no
  // suffix with score <= 0, and no sub-segment with score < -maxDrop.
  bool isOptimal( const uchar* seq1, const uchar* seq2,
		  const int scoreMatrix[MAT][MAT], int maxDrop ) const;

  // Shrink the SegmentPair to the longest run of identical letters
  // within it.  Allow (upper/lower)case to differ, using "canonical".
  void maxIdenticalRun( const uchar* seq1, const uchar* seq2,
			const uchar* canonical,
			const int scoreMatrix[MAT][MAT] );

  indexT beg1() const{ return start1; }         // start in sequence 1
  indexT beg2() const{ return start2; }         // start in sequence 2
  indexT end1() const{ return start1 + size; }  // end in sequence 1
  indexT end2() const{ return start2 + size; }  // end in sequence 2
  indexT diagonal() const{ return start1 - start2; }  // may wrap around!

  indexT start1;
  indexT start2;
  indexT size;
  int score;
};

}  // end namespace cbrc
#endif  // SEGMENTPAIR_HH
