package latexDraw.parsers.svg.parsers;

import latexDraw.parsers.Parser;

/**
 * Defines a model for the SVG parsers.<br>
 *<br>
 * This file is part of LaTeXDraw.<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 10/20/07<br>
 * @author Arnaud BLOUIN<br>
 * @version 0.1<br>
 * @since 0.1
 */
public abstract class AbstractSVGParser extends Parser
{
	/** The token for the end of path. @since 0.1 */
	public static final int EOP = -1;
	
	/** The path to parse. @since 0.1 */
	protected String code;
	
	/** The current position in the path. @since 0.1 */
	protected int position;
	
	
	/**
	 * The constructor.
	 * @param code The path to parse.
	 */
	public AbstractSVGParser(String code)
	{
		if(code==null)
			throw new IllegalArgumentException();
		
		position = 0;
		this.code = code;
	}

	
	
	/**
	 * Skips the useless characters.
	 * @since 0.1
	 */
	public void skipWSP()
	{
    	int c = getChar();
    	boolean wsp = c!=EOP && (c==0x20 || c==0x9 || c==0xD || c==0xA);
    	
    	while(wsp)
    	{
    		c = nextChar();
    		
    		if(c==EOP || (c!=0x20 && c!=0x9 && c!=0xD && c!=0xA))
    			wsp = false;
    	}
	}
	
	
	
	/**
	 * Skips the useless characters and a possible comma.
	 * @since 0.1
	 */
    public void skipWSPComma()
	{
    	skipWSP();
    	
    	if(getChar()==',')
    	{
    		nextChar();
    		skipWSP();
    	}
	}
	
    
    /**
     * Reads the next char and returns it.
     * @return the next read character or EOS if the end of the path is reached.
     * @since 0.1
     */
    public int nextChar()
    {
    	position++;
    	
    	if(position>=code.length())
    		return EOP;
    	
    	return code.charAt(position);
    }
    
    
    
    /**
     * @return the current character or EOS if the end of the path is reached.
     * @since 0.1
     */
    public int getChar()
    {
    	if(position>=code.length())
    		return EOP;
    	
    	return code.charAt(position);
    }
    
    
    /**
     * @return True if the end of the path is reached.
     * @since 0.1
     */
    public boolean isEOP()
    {
    	return getChar()==EOP;
    }
    
    
	
	/**
	 * @return the position.
	 * @since 0.1
	 */
	public int getPosition()
	{
		return position;
	}


	
	/**
	 * @return the code.
	 * @since 0.1
	 */
	public String getCode()
	{
		return code;
	}
	
	
	/**
	 * Sets the current position.
	 * @param position THe position to set.
	 * @since 0.1
	 */
	public void setPosition(int position)
	{
		if(position>=0 && position<code.length())
			this.position = position;
	}
	
	
	
	/**
	 * Sets the new code to parse and set the position to 0.
	 * @param code The new code to parse (a null string has no effect).
	 * @since 0.1
	 */
	public void setCode(String code)
	{
		if(code!=null)
		{
			setPosition(0);
			this.code = code;
		}
	}
}
