package latexDraw.ui;

import java.awt.BorderLayout;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.ClipboardOwner;
import java.awt.datatransfer.StringSelection;
import java.awt.datatransfer.Transferable;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.ObjectInputStream;

import javax.swing.*;

import latexDraw.lang.LaTeXDrawLang;
import latexDraw.ui.codeEditorPane.AbstractCodeEditorPane;
import latexDraw.ui.codeEditorPane.PSTricksEditorPane;
import latexDraw.ui.components.CloseButton;
import latexDraw.util.LaTeXDrawNamespace;
import latexDraw.util.LaTeXDrawPoint2D;
import latexDraw.util.LaTeXDrawResources;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;


/** 
 * Defines the panel which contains the code generated from the drawing.<br>
 * <br>
 * This file is part of LaTeXDraw<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 06/03/07<br>
 * @author Arnaud BLOUIN<br>
 * @version 2.0.0<br>
 */
public class CodePanel extends JPanel implements ActionListener, ClipboardOwner
{
	private static final long serialVersionUID = 1L;

	/** The comments of the user */
	private String comments;
	
	/** The label of the drawing */
	private String label;
	
	/** The caption of the drawing */
	private String caption;
	
	/** The main frame of the program */
	private LaTeXDrawFrame mainFrame;
	
	/** The token of the position of the drawing */
	private String positionVertToken;
	
	/** The horizontal position of the drawing */
	private String positionHoriToken;
	
	public static final String TOKEN_HORI_CENTER_START = "\\begin{center}\n"; //$NON-NLS-1$
	
	public static final String TOKEN_HORI_CENTER_END = "\\end{center}\n"; //$NON-NLS-1$
	
	public static final String TOKEN_HORI_VERT_DEFAULT = ""; //$NON-NLS-1$
	
	public static final boolean DEFAULT_ISONFLOATSPAGE = false;
	
	public static final String TOKEN_VERT_TOP = "t"; //$NON-NLS-1$
	
	public static final String TOKEN_VERT_BOTTOM = "b"; //$NON-NLS-1$
	
	public static final String TOKEN_VERT_FLOATS_PAGE = "p"; //$NON-NLS-1$
	
	public static final String TOKEN_VERT_HERE = "h"; //$NON-NLS-1$
	
	public static final String TOKEN_VERT_HERE_HERE = "H"; //$NON-NLS-1$
	
	/** Defines the number of characters added at the beginning
	 * of each lines of the comments (these characters are "% ")*/
	public static final int LGTH_START_LINE_COMMENT = 2;
	
	/** The label of the button reload */
	public static final String LABEL_RELOAD = LaTeXDrawLang.getOthersString("CodePanel.reload"); //$NON-NLS-1$
	
	/** The label of the check-box autoUpdate */
	public static final String LABEL_AUTO_UPDATE = LaTeXDrawLang.getOthersString("CodePanel.autoUpdate"); //$NON-NLS-1$
	
	/** Allows to know if the code must be update in real time by default */
	public static final boolean DEFAULT_AUTO_UPDATE = true;
	
	/** The label of the button buttonCopySel */
	public static final String LABEL_COPY_SELECTION_CODE = LaTeXDrawLang.getOthersString("CodePanel.copySel"); //$NON-NLS-1$
	
	/** The label of the button which copy all the text in the clipboard */
	public static final String LABEL_COPY_ALL = LaTeXDrawLang.getOthersString("CodePanel.copyAll"); //$NON-NLS-1$
	
	public static final String LABEL_COPY_SELECTED_FIG = "Select code of selected figures"; //$NON-NLS-1$
	
	/** This button allows the user to copy the selected text in the clipboard */
	protected JButton buttonCopySel;
	
	/** Allows to know if the user wants to update in real time the code or not */
	protected JCheckBox autoUpdateCB;
	
	/** This button allows the user to update the code */
	protected JButton reloadButton;
	
	/** Allows to add comments to the code */
	protected JButton commentsButton;
	
	protected AbstractCodeEditorPane editor;
	
	
	
	
	
	/** The constructor by default */
	public CodePanel(LaTeXDrawFrame parent)
	{
		mainFrame = parent;
		
		positionHoriToken = TOKEN_HORI_VERT_DEFAULT;
		label   = "";//$NON-NLS-1$
		caption = "";//$NON-NLS-1$
		positionVertToken = "";//$NON-NLS-1$
		editor  = new PSTricksEditorPane(this);   
		
		JScrollPane scroller = new JScrollPane(); 
		JViewport vp = scroller.getViewport(); 
	 	vp.add(editor); 
		
		JToolBar toolbar = new JToolBar("Code toolbar"); //$NON-NLS-1$
		toolbar.setFloatable(false);
		
		autoUpdateCB = new JCheckBox(LABEL_AUTO_UPDATE);
		autoUpdateCB.setMargin(LaTeXDrawFrame.INSET_BUTTON);
		autoUpdateCB.setToolTipText(LaTeXDrawLang.getOthersString("CodePanel.tip")); //$NON-NLS-1$
		autoUpdateCB.setActionCommand(LABEL_AUTO_UPDATE);
		autoUpdateCB.setSelected(DEFAULT_AUTO_UPDATE);
		toolbar.add(autoUpdateCB);
		autoUpdateCB.addActionListener(this);
		
		reloadButton = new JButton(LaTeXDrawResources.reloadIcon);
		reloadButton.setMargin(LaTeXDrawFrame.INSET_BUTTON);
		reloadButton.setToolTipText(LaTeXDrawLang.getOthersString("CodePanel.updateCode")); //$NON-NLS-1$
		reloadButton.setActionCommand(LABEL_RELOAD);
		reloadButton.setEnabled(!DEFAULT_AUTO_UPDATE);
		toolbar.add(reloadButton);
		reloadButton.addActionListener(this);
		
		commentsButton = new JButton(LaTeXDrawResources.commentIcon);
		commentsButton.setMargin(LaTeXDrawFrame.INSET_BUTTON);
		commentsButton.setToolTipText(LaTeXDrawLang.getOthersString("CodePanel.addComment")); //$NON-NLS-1$
		commentsButton.setActionCommand(LaTeXDrawResources.LABEL_COMMENTS);
		commentsButton.addActionListener(mainFrame);
		toolbar.add(commentsButton);
		
		JButton but = new JButton(LaTeXDrawResources.copyIcon);
		but.setMargin(LaTeXDrawFrame.INSET_BUTTON);
		but.setToolTipText(LABEL_COPY_ALL);
		but.setActionCommand(LABEL_COPY_ALL);
		toolbar.add(but);
		but.addActionListener(this);
		
		buttonCopySel = new JButton(LaTeXDrawResources.copySelIcon);
		buttonCopySel.setMargin(LaTeXDrawFrame.INSET_BUTTON);
		buttonCopySel.setToolTipText(LABEL_COPY_SELECTION_CODE);
		buttonCopySel.setActionCommand(LABEL_COPY_SELECTION_CODE);
		buttonCopySel.setEnabled(false);
		toolbar.add(buttonCopySel);
		buttonCopySel.addActionListener(this);
		
		JToolBar toolbar2 = new JToolBar();
		toolbar2.setFloatable(false);
		CloseButton closeB = new CloseButton(this);
		closeB.addActionListener(mainFrame);
		closeB.setActionCommand(LaTeXDrawResources.LABEL_DISPLAY_CODE_PANEL);
		toolbar2.add(closeB);
		
		JPanel p = new JPanel();
		p.setLayout(new BorderLayout());
		p.add(toolbar2,BorderLayout.EAST);
		p.add(toolbar,BorderLayout.WEST);
		setLayout(new BorderLayout());
		add(p, BorderLayout.NORTH);
		add(scroller, BorderLayout.CENTER);
		
		comments = "";//$NON-NLS-1$
	}
	
	
	
	/**
	 * Allows to get the comments without any characters like "%"
	 * at the start of each lines. (these characters are used like comment symbol by LaTeX).
	 * @return The code without LaTeX comment symbol '%'
	 */
	public String getCommentsWithoutTag()
	{
		if(comments!=null)
		{
			int i=0, j=0, lgth = comments.length();
			char buffer[] = new char[lgth];
			boolean eol = true;
			
			while(i<lgth)
			{
				if(eol && comments.charAt(i)=='%') 
				{
					i+=LGTH_START_LINE_COMMENT;
					eol = false;
				}
				else
				{
					if(comments.charAt(i)=='\n')
						eol = true;
					
					buffer[j++] = comments.charAt(i);
					i++;
				}
			}
			
			String str = String.valueOf(buffer, 0, j);
			
			if(str.length()>1)
				return str.substring(0, str.length()-System.getProperty("line.separator").length());//$NON-NLS-1$
			return str;
		}
		return null;
	}
	
	
	
	
	/**
	 * Allows to get the comments of the code
	 * @return The comments of the code
	 */
	public String getComments()
	{
		return comments;
	}
	
	
	
	
	/**
	 * Empty the panel.
	 */
	public void setEmpty()
	{
		comments=""; //$NON-NLS-1$
		editor.setEmpty(); 
	}
	
	
	
	
	/**
	 * Allow to set the comments of the code
	 * @param newComments The new comments of the code
	 */
	public void setComments(String newComments)
	{
		if(newComments!=null && newComments.length()>0)
		{
			int i, j=0, lgth = newComments.length();
			char buffer[] = new char[lgth*3];
			boolean eol = true;
			
			for(i=0; i<newComments.length(); i++)
			{
				if(eol)
				{
					buffer[j++] = '%';
					buffer[j++] = ' ';
					eol = false;
				}
				
				if(newComments.charAt(i)=='\n')
					eol = true;
				
				buffer[j++] = newComments.charAt(i);
			}
			
			comments = String.valueOf(buffer, 0, j);
			comments+=System.getProperty("line.separator");//$NON-NLS-1$ 
		}
		else comments = ""; //$NON-NLS-1$
	}
	
	
	
	
	
	
	
	
	/**
	 * Allows to define if the text must be display or not
	 * @param display True : the text must be display
	 */
	public void setTextDisplayable(boolean display)
	{
		if(display)
			editor.updateText();
		else 
			editor.setText(""); //$NON-NLS-1$
	}



	private void readObject(@SuppressWarnings("unused") ObjectInputStream ois)
	{
		/* Nothing to do. */
	}




	public void actionPerformed(ActionEvent e) 
	{
		Object o = e.getSource();
	
		if(o instanceof JButton || o instanceof JCheckBox || o instanceof JMenuItem)
		{
			String l = ((AbstractButton)o).getActionCommand();
			
			if(l.equals(LABEL_RELOAD))
			{
				mainFrame.getDrawPanel().updateCode();
				mainFrame.setStatusBarText(LaTeXDrawLang.getOthersString("CodePanel.reloadText")); //$NON-NLS-1$
				mainFrame.requestFocus();
				return ;
			}
			
			if(l.equals(LABEL_AUTO_UPDATE))
			{
				if(o instanceof JCheckBoxMenuItem)
					autoUpdateCB.setSelected(mainFrame.menuBar.getMenuAutoUpdate().isSelected());
				else
					mainFrame.menuBar.getMenuAutoUpdate().setSelected(autoUpdateCB.isSelected());
				
				mainFrame.getDrawPanel().setIsModified(true);
				updateAutoUpdateComponents();
				mainFrame.requestFocus();
				return ;
			}
			
			if(l.equals(LABEL_COPY_ALL))
			{
				mainFrame.getDrawPanel().updateCode();
				String text = getText();
				
				if(text!=null && !text.equals("")) //$NON-NLS-1$
					copyTextToClipBoard(text);
				
				buttonCopySel.setEnabled(false);
				mainFrame.setStatusBarText(LaTeXDrawLang.getOthersString("CodePanel.copiedText")); //$NON-NLS-1$
				mainFrame.requestFocus();
				return ;
			}
			
			if(l.equals(LABEL_COPY_SELECTION_CODE))
			{
				mainFrame.setStatusBarText(LaTeXDrawLang.getOthersString("CodePanel.copiedSelText")); //$NON-NLS-1$
				String text = editor.getSelectedText();
				
				if(text!=null && !text.equals("")) //$NON-NLS-1$
					copyTextToClipBoard(text);
				
				buttonCopySel.setEnabled(false);
				mainFrame.requestFocus();
				return ;
			}
			
			if(l.equals(LABEL_COPY_SELECTED_FIG))
			{
				String code = mainFrame.getDrawPanel().getDraw().getCodeSelection();
				
				if(code!=null && code.length()>0)
				{
					mainFrame.setStatusBarText(LaTeXDrawLang.getOthersString("CodePanel.copiedSelText")); //$NON-NLS-1$
					copyTextToClipBoard(code);
				}

				mainFrame.requestFocus();
				return ;
			}
		} // if(o instanceof JButton)
	}

	
	
	
	
	/**
	 * Allows to copy a text in the clipboard
	 * @param text The text to copy
	 */
	public void copyTextToClipBoard(String text)
	{
		Clipboard cb = getToolkit().getSystemClipboard();
		cb.setContents(new StringSelection(text), this);
	}
	
	
	
	
	

	/**
	 * Allows to know if the code must be update in real time or not
	 * @return True if the code must be update in real time
	 */
	public boolean isInAutoUpdate()
	{
		return autoUpdateCB.isSelected();
	}
	
	
	
	
	/**
	 * Allows to set if the code panel must be displayed
	 * @param isIn If true, the code panel must be displayed
	 */
	public void setIsInAutoUpdate(boolean isIn)
	{
		if(isIn!=isInAutoUpdate())
			mainFrame.getDrawPanel().setIsModified(true);
			
		autoUpdateCB.setSelected(isIn);
		updateAutoUpdateComponents();
	}
	
	
	
	
	/**
	 * Allows to update components which depends of the checkbox "autoUpdateCB"
	 */
	protected void updateAutoUpdateComponents()
	{
		if(autoUpdateCB.isSelected())
		{
			mainFrame.menuBar.getMenuReloadCode().setEnabled(false);
			reloadButton.setEnabled(false);
			mainFrame.getDrawPanel().updateCode();
		}
		else
		{
			reloadButton.setEnabled(true);
			mainFrame.menuBar.getMenuReloadCode().setEnabled(true);
		}
		
	}
	
	
	
	/**
	 * Allows to get the text of the editor
	 * @return The text of the editor
	 */
	public String getText()
	{
		return editor.getText();
	}




	/**
	 * @return Returns the caption.
	 */
	public String getCaption()
	{
		return caption;
	}




	/**
	 * @param caption The caption to set.
	 */
	public void setCaption(String caption)
	{
		this.caption = caption;
	}




	/**
	 * @return Returns the label.
	 */
	public String getLabel()
	{
		return label;
	}




	/**
	 * @param label The label to set.
	 */
	public void setLabel(String label)
	{
		this.label = label;
	}




	/**
	 * @return Returns the positionVertToken.
	 */
	public String getPositionVertToken()
	{
		return positionVertToken;
	}




	/**
	 * @param positionToken The positionVertToken to set.
	 */
	public void setPositionVertToken(String positionToken)
	{
		if(positionToken.equals(TOKEN_VERT_BOTTOM) ||
			positionToken.equals(TOKEN_VERT_FLOATS_PAGE) || positionToken.equals(TOKEN_VERT_HERE) ||
			positionToken.equals(TOKEN_VERT_HERE_HERE) || positionToken.equals(TOKEN_VERT_TOP))
			positionVertToken = positionToken;
		else
			positionVertToken = TOKEN_HORI_VERT_DEFAULT;
	}






	/**
	 * @return Returns the positionHoriToken.
	 */
	public String getPositionHoriToken()
	{
		return positionHoriToken;
	}




	/**
	 * @param positionHori The positionHoriToken to set.
	 */
	public void setPositionHoriToken(String positionHori)
	{
		positionHoriToken = positionHori;
	}




	public void lostOwnership(Clipboard cb, Transferable t)
	{
		/*
		 * No code required.
		 */
	}




	
	/**
	 * Actions to do when a new project is created.
	 * @since 1.9
	 */
	public void newProject()
	{
		positionHoriToken = TOKEN_HORI_VERT_DEFAULT;
		positionVertToken = TOKEN_HORI_VERT_DEFAULT;
		label 	= "";//$NON-NLS-1$
		caption = "";//$NON-NLS-1$
	}



	/**
	 * @return the mainFrame.
	 * @since 1.9
	 */
	public LaTeXDrawFrame getMainFrame()
	{
		return mainFrame;
	}



	/**
	 * @return the buttonCopySel.
	 * @since 2.0.0
	 */
	public JButton getButtonCopySel()
	{
		return buttonCopySel;
	}



	/**
	 * Set the beginning of the code generated.
	 * @param SW The top right point.
	 * @param NE The bottom left point.
	 * @param pixelsPerCm The definition level of the drawing.
	 * @param origin The origin point of the system of coordinates.
	 */
	public void setStart(LaTeXDrawPoint2D SW, LaTeXDrawPoint2D NE, int pixelsPerCm, LaTeXDrawPoint2D origin)
	{
		editor.setStart(SW, NE, pixelsPerCm, origin);
	}



	/**
	 * Set the body (between the beginning and the end) of the code.
	 * @param code The new code.
	 */
	public void setBody(String code)
	{
		editor.setBody(code);
	}



	/**
	 * Update the text of the panel.
	 */
	public void updateText()
	{
		editor.updateText();
	}
	
	
	
	
	
	/**
	 * Sets the XML element "meta" with the parameters of the code panel.
	 * @param document The XML document.
	 * @param meta The Element to fill.
	 * @throws IllegalArgumentException If meta or document is null.
	 * @since 2.0.0
	 */
	public void getXMLMetadata(Document document, Element meta)
	{
		if(document==null || meta==null)
			throw new IllegalArgumentException();
		
        if(caption!=null && caption.length()!=0)
        {
        	Element cap = document.createElement(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE+':'+LaTeXDrawNamespace.XML_CAPTION);
        	cap.appendChild(document.createCDATASection(caption));
        	meta.appendChild(cap);
        }
        
        if(label!=null && label.length()!=0)
        {
        	Element lab = document.createElement(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE+':'+LaTeXDrawNamespace.XML_LABEL);
        	lab.appendChild(document.createCDATASection(label));
        	meta.appendChild(lab);
        }
        
        if(positionHoriToken!=null && positionHoriToken.length()!=0)
        {
        	Element tok = document.createElement(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE+':'+LaTeXDrawNamespace.XML_POSITION_HORIZ);
        	tok.appendChild(document.createTextNode(positionHoriToken));
        	meta.appendChild(tok);
        }
        
        if(positionVertToken!=null && positionVertToken.length()!=0)
        {
        	Element tok = document.createElement(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE+':'+LaTeXDrawNamespace.XML_POSITION_VERT);
        	tok.appendChild(document.createTextNode(positionVertToken));
        	meta.appendChild(tok);
        }
        
        if(comments!=null && comments.length()!=0)
        {
        	Element comm = document.createElement(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE+':'+LaTeXDrawNamespace.XML_COMMENTS);
        	comm.appendChild(document.createCDATASection(getCommentsWithoutTag()));
        	meta.appendChild(comm);
        }
        
        Element auto = document.createElement(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE+':'+LaTeXDrawNamespace.XML_AUTO_UPDATE);
        auto.appendChild(document.createTextNode(String.valueOf(isInAutoUpdate())));
    	meta.appendChild(auto);
	}



	/**
	 * Sets the latexdraw parameters from the <code>metadata</code> SVG tag.
	 * @param nl The list of the parameters.
	 * @since 2.0.0
	 */
	public void setXMLMetadata(NodeList nl)
	{
		if(nl==null)
			return ;
		
		Node n;
		String name;
		
		for(int i=0, size = nl.getLength(); i<size; i++)
		{
			n = nl.item(i);
			
			if(n!=null && LaTeXDrawNamespace.LATEXDRAW_NAMESPACE_URI.equals(n.getNamespaceURI()))
			{
				name = n.getNodeName();
				
				try
				{
					if(name.endsWith(LaTeXDrawNamespace.XML_CAPTION))
						setCaption(n.getTextContent());
					else if(name.endsWith(LaTeXDrawNamespace.XML_LABEL))
						setLabel(n.getTextContent());
					else if(name.endsWith(LaTeXDrawNamespace.XML_POSITION_HORIZ))
						setPositionHoriToken(n.getTextContent());
					else if(name.endsWith(LaTeXDrawNamespace.XML_POSITION_VERT))
						setPositionVertToken(n.getTextContent());
					else if(name.endsWith(LaTeXDrawNamespace.XML_COMMENTS))
						setComments(n.getTextContent());
					else if(name.endsWith(LaTeXDrawNamespace.XML_AUTO_UPDATE))
					{
						boolean is = Boolean.valueOf(n.getTextContent()).booleanValue();
						
						setIsInAutoUpdate(is);
						mainFrame.getLMenuBar().getMenuAutoUpdate().setSelected(is);
					}
				}
				catch(Exception e) { System.out.println(name + ": invalid value."); }//$NON-NLS-1$
			}
		}
	}
}
