package latexDraw.figures;

import java.awt.*;
import java.awt.geom.Area;
import java.awt.geom.Rectangle2D;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.util.Vector;

import latexDraw.psTricks.DviPsColors;
import latexDraw.psTricks.PSTricksConstants;
import latexDraw.ui.LaTeXDrawFrame;
import latexDraw.ui.components.Delimitor;
import latexDraw.ui.components.MagneticGrid;
import latexDraw.util.LaTeXDrawNumber;
import latexDraw.util.LaTeXDrawPoint2D;


/** 
 * This class defines a square.<br>
 *<br>
 * This file is part of LaTeXDraw.<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 01/16/06<br>
 * @author Arnaud BLOUIN<br>
 * @version 2.0.0<br>
 */
public class Square extends LaTeXDrawRectangle
{
	private static final long serialVersionUID = 1L;
	
	/** The width of the square */
	protected double width;
	
	
	
	/**
	 * @param f The figure to copy.
	 * @param sameNumber If the new figure must have the same id than the copy.
	 */
	public Square(Figure f, boolean sameNumber)
	{
		super(f, sameNumber);
		width = f.getTheSEPoint().x - f.getTheNWPoint().x;
		updateGravityCenter();
		setThickness(thickness);
	}




	/**
	 * The constructor by default
	 */
	public Square(boolean increaseMeter)
	{
		this(new LaTeXDrawPoint2D(), new LaTeXDrawPoint2D(), new LaTeXDrawPoint2D(), new LaTeXDrawPoint2D(), increaseMeter);
	}
	
	
	
	/**
	 * The constructor taking the four points of the square
	 * @param pt1 The first point of the square
	 * @param pt2 The second point of the square
	 * @param pt3 The third point of the square
	 * @param pt4 The fourth point of the square
	 */
	public Square(LaTeXDrawPoint2D pt1, LaTeXDrawPoint2D pt2, LaTeXDrawPoint2D pt3, LaTeXDrawPoint2D pt4, boolean increaseMeter)
	{
		super(pt1, pt2, pt3, pt4, increaseMeter);
		width = Math.abs(pt1.x-pt2.x);
		updateGravityCenter();
	}
	
	
	
	
	
	/**
	 * Sets the width of the square.
	 * @param w The new width of the square.
	 * @throws IllegalArgumentException if w<0.
	 */
	public synchronized void setWidth(double w)
	{
		if(w<=0)
			throw new IllegalArgumentException();
		
		width = w;
		updateGravityCenter();
		shape = getInsideOutsideOrMiddleBorders();
	}
	
	
	
	
	@Override
	public void draw(Graphics2D g, boolean drawBorders, Object antiAlias, Object rendering, Object alphaInter, Object colorRendering)
	{
		LaTeXDrawPoint2D NW = getTheNWPoint(), SE = getTheSEPoint();

		if(lineStyle.equals(PSTricksConstants.LINE_NONE_STYLE))
			g.setStroke(new BasicStroke(thickness, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER));
		else 
		if(lineStyle.equals(PSTricksConstants.LINE_DOTTED_STYLE))
			g.setStroke(new BasicStroke(thickness, BasicStroke.CAP_ROUND, BasicStroke.JOIN_MITER,
					1.f, new float[]{0,thickness+dotSep}, 0));
		else
		if(lineStyle.equals(PSTricksConstants.LINE_DASHED_STYLE))
			g.setStroke(new BasicStroke(thickness, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER,
					1.f, new float[]{blackDashLength, whiteDashLength}, 0));
		
		
		double cx = (NW.x+SE.x)/2., cy = (NW.y+SE.y)/2.;
		double c2x = Math.cos(rotationAngle)*cx - Math.sin(rotationAngle)*cy;
		double c2y = Math.sin(rotationAngle)*cx + Math.cos(rotationAngle)*cy;
		double c3x = Math.cos(-rotationAngle)*(cx-c2x) - Math.sin(-rotationAngle)*(cy-c2y);
		double c3y = Math.sin(-rotationAngle)*(cx-c2x) + Math.cos(-rotationAngle)*(cy-c2y);
		double dx=0, dy=0;
		boolean changeFillStyle = false;
		
		if(rotationAngle%(Math.PI*2)!=0)
		{
			g.rotate(rotationAngle);
			g.translate(c3x,c3y);
		}
		
		if(hasShadow)
		{
			LaTeXDrawPoint2D cg = getGravityCenter();
			LaTeXDrawPoint2D shadowCg = (LaTeXDrawPoint2D)cg.clone();
			shadowCg.setLocation(cg.x+shadowSize, cg.y);
			shadowCg = Figure.rotatePoint(shadowCg, cg, shadowAngle);
			dx = shadowCg.x-cg.x;
			dy = cg.y-shadowCg.y;
		}
		
		if(drawBorders)
		{
			Color formerCol = g.getColor();

			if(hasDoubleBoundary)
			{
				Shape s0 = shape;
				Shape s[] = getDbleBoundariesOutInOrMiddle(s0);
				Shape s1, s2, s3;

				if(bordersPosition.equals(PSTricksConstants.BORDERS_INSIDE))
				{
					s1 = s0;
					s2 = s[0];
					s3 = s[1];
				}
				else
					if(bordersPosition.equals(PSTricksConstants.BORDERS_MIDDLE))
					{
						s1 = s[0];
						s2 = s0;
						s3 = s[1];
					}
					else
					{
						s1 = s[0];
						s2 = s[1];
						s3 = s0;
					}

				if(lineStyle.equals(PSTricksConstants.LINE_NONE_STYLE))
				{
					g.setStroke(new BasicStroke(thickness, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER));
					if(hasShadow)
					{
						g.translate(dx, dy);
						g.setColor(shadowColor);
						g.fill(s1);
						g.draw(s1);
						g.translate(-dx, -dy);
						
						if(!isFilled)
						{
							changeFillStyle = true;
							isFilled = true;
						}
					}
					g.setColor(doubleColor);
					Area area = new Area(s1);
					area.subtract(new Area(s3));
					g.fill(area);
					fillFigure(g, antiAlias, rendering, alphaInter, colorRendering,s3);
					g.setColor(linesColor);
					g.setStroke(new BasicStroke(thickness, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER));
					g.draw(s1);
					g.draw(s3);
				}
				else
				{
					if(hasShadow)
					{
						g.setStroke(new BasicStroke(thickness, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER));
						g.translate(dx, dy);
						g.setColor(shadowColor);
						g.fill(s1);
						g.draw(s1);
						g.translate(-dx, -dy);
						g.setColor(interiorColor);
						g.setStroke(new BasicStroke((float)(thickness*2+doubleSep), BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER));
						g.draw(s2);
						
						
						if(!isFilled)
						{
							changeFillStyle = true;
							isFilled = true;
						}
					}

					if(lineStyle.equals(PSTricksConstants.LINE_DOTTED_STYLE))
						g.setStroke(new BasicStroke((float)(thickness*2+doubleSep), BasicStroke.CAP_ROUND,
								BasicStroke.JOIN_MITER, 1.f, new float[]
								{ 0, (float)(thickness*2+doubleSep + dotSep) }, 0));
					else
						g.setStroke(new BasicStroke((float)(thickness*2+doubleSep),
								BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER, 1.f,
								new float[] { blackDashLength, whiteDashLength }, 0));
						
					fillFigure(g, antiAlias, rendering, alphaInter, colorRendering, s2);
					g.setColor(linesColor);
					g.draw(s2);
					g.setStroke(new BasicStroke((float)doubleSep, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER));
					g.setColor(doubleColor);
					g.draw(s2);
				}				
			}
			else
			{
				Shape shape2 = shape;
				if(hasShadow)
				{
					g.setStroke(new BasicStroke(thickness, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER));
					g.translate(dx, dy);
					g.setColor(shadowColor);
					g.fill(shape);
					g.draw(shape);
					g.translate(-dx, -dy);
					if(!isFilled)
					{
						changeFillStyle = true;
						isFilled = true;
					}
					g.setColor(interiorColor);
					g.draw(shape);
				}

				if(lineStyle.equals(PSTricksConstants.LINE_NONE_STYLE))
					g.setStroke(new BasicStroke(thickness, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER));
				else
					if(lineStyle.equals(PSTricksConstants.LINE_DOTTED_STYLE))
					{
						g.setStroke(new BasicStroke(thickness, BasicStroke.CAP_ROUND,
								BasicStroke.JOIN_MITER, 1.f, new float[]{ 0, thickness + dotSep }, 0));
					}
					else
						g.setStroke(new BasicStroke(thickness, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER, 1.f,
								new float[] { blackDashLength, whiteDashLength }, 0));
				
				fillFigure(g, antiAlias, rendering, alphaInter, colorRendering,shape2);
				g.setColor(linesColor);
				g.draw(shape);
			}

			if(changeFillStyle) isFilled = false;
			g.setColor(formerCol);
		}
		
		if(isSelected)
		{
			dNW.draw(g);
			dSE.draw(g);
			dNE.draw(g);
			dSW.draw(g);
		}
		
		if(rotationAngle%(Math.PI*2)!=0)
		{
			g.translate(-c3x, -c3y);
			g.rotate(-rotationAngle);
		}
	}
	
	
	
	
	@Override
	public void draw(Graphics2D g, Object antiAlias, Object rendering, Object alphaInter, Object colorRendering) 
	{
		draw(g, true, antiAlias, rendering, alphaInter, colorRendering);
	}
	
	
	
	
	
	
	@Override
	public void onDragged(Point formerPt, Point newPt)
	{
		if(formerPt.equals(newPt)) return;
		
		if(dSelected!=null)
		{
			if(isOnRotation)
				rotate(formerPt, newPt);
			else
			{
				LaTeXDrawPoint2D newPt2 = rotateInvertPoint(newPt);
				LaTeXDrawPoint2D oldGc = (LaTeXDrawPoint2D)gravityCenter.clone();
				
				if(dSelected.getCenter()==pts.elementAt(0))
					setFirstPoint(newPt2);
				else
					if(dSelected.getCenter()==pts.elementAt(1))
						setSecondPoint(newPt2);
					else
						if(dSelected.getCenter()==pts.elementAt(2))
							setThirdPoint(newPt2);
						else
							setLastPoint(newPt2);
	
				width = Math.abs(pts.elementAt(0).x 
								- pts.elementAt(1).x);
				
				if(rotationAngle!=0)
					recenterDraggedOnRotation(oldGc);
				
				updateGravityCenter();		
			}
		}
		else //	 If the user has clicked on the line
		{
			shift(formerPt, newPt);		
			shape = getInsideOutsideOrMiddleBorders();
		}
	}
	
	
	
	

	@Override
	public void rescaleX(double formerX, double newX, double percent, LaTeXDrawRectangle bound)
	{
		if(percent==1.) return ;
		
		int i, size = getNbPoints();

		if(size>0)
		{
			LaTeXDrawPoint2D bNW = bound.getTheNWPoint(), bSE = bound.getTheSEPoint(),farest,p;
			
			if(Math.abs(newX-bSE.x)<Math.abs(newX-bNW.x))
				  farest = bNW;
			else  farest = bSE;
			
			for(i=0; i<size; i++)
			{// We rescale each point
				p = getPoint(i);
				p.x = farest.x+(p.x-farest.x)*percent;				
			}
			
			LaTeXDrawPoint2D pt1 = getPoint(0), pt4 = getPoint(3);
			width = Math.abs(pt1.x-pt4.x);
			
			if(pt1.x<pt4.x)
			{
				if(pt1.y<pt4.y)
				{
					setThirdPoint(new LaTeXDrawPoint2D(pt1.x,pt1.y+width));
					setLastPoint(pt1.x+width, pt1.y+width);
				}
				else
				{
					setThirdPoint(new LaTeXDrawPoint2D(pt1.x,pt1.y-width));
					setLastPoint(pt1.x+width, pt1.y-width);
				}
			}
			else
				if(pt1.y<pt4.y)
				{
					setThirdPoint(new LaTeXDrawPoint2D(pt1.x,pt1.y+width));
					setLastPoint(pt1.x-width, pt1.y+width);
				}
				else
				{
					setThirdPoint(new LaTeXDrawPoint2D(pt1.x,pt1.y-width));
					setLastPoint(pt1.x-width, pt1.y-width);
				}
			shape = getInsideOutsideOrMiddleBorders();
		}
	}
	
	
	
	

	@Override
	public void rescaleY(double formerY, double newY, double percent, LaTeXDrawRectangle bound)
	{
		if(percent==1.) return ;
		
		int i, size = getNbPoints();
		
		if(size>0)
		{
			LaTeXDrawPoint2D NW = bound.getTheNWPoint(), SE = bound.getTheSEPoint(),farest,p;
	
			if(Math.abs(newY-SE.y)<Math.abs(newY-NW.y))
				  farest = NW;
			else  farest = SE;
			
			for(i=0; i<size; i++)
			{// We rescale each point
				p = getPoint(i);
				p.y = farest.y+(p.y-farest.y)*percent;
			}
			LaTeXDrawPoint2D pt1 = getPoint(0), pt4 = getPoint(3);
			width = Math.abs(pt1.y-pt4.y);
			
			if(pt1.x<pt4.x)
			{
				if(pt1.y<pt4.y)
				{
					setSecondPoint(new LaTeXDrawPoint2D(pt1.x+width,pt1.y));
					setLastPoint(pt1.x+width, pt1.y+width);
				}
				else
				{
					setSecondPoint(new LaTeXDrawPoint2D(pt1.x-width,pt1.y));
					setLastPoint(pt1.x+width, pt1.y-width);
				}
			}
			else
				if(pt1.y<pt4.y)
				{
					setSecondPoint(new LaTeXDrawPoint2D(pt1.x+width,pt1.y));
					setLastPoint(pt1.x-width, pt1.y+width);
				}
				else
				{
					setSecondPoint(new LaTeXDrawPoint2D(pt1.x-width,pt1.y));
					setLastPoint(pt1.x-width, pt1.y-width);
				}
			shape = getInsideOutsideOrMiddleBorders();
		}
	}
	
	
	
	@Override
	public synchronized void setFirstPoint(double x, double y)
	{
		setFirstPoint(new LaTeXDrawPoint2D(x,y));
	}
	
	
	
	
	@Override
	public synchronized void setLastPoint(double x, double y)
	{
		setLastPoint(new LaTeXDrawPoint2D(x,y));
	}
	
	
	
	
	@Override
	public synchronized void setFirstPoint(LaTeXDrawPoint2D pt)
	{	
		// As the this second point pt doesn't necessary
		// creates a square but a rectangle, we need to change
		// this point in the goal that it allows to create a square
		LaTeXDrawPoint2D third = getPoint(2), second = getPoint(1);
		LaTeXDrawPoint2D first = getPoint(0), fourth = getPoint(3);

		width = Math.abs(second.x-pt.x);
		double x, y;
		
		if(fourth.y<pt.y)
			   y = fourth.y + width;
		else   y = fourth.y - width;
		
		if(fourth.x<pt.x)
			   x = fourth.x + width;
		else   x = fourth.x - width;
		
		// This new point allows to create a square and not a rectangle
		first.setLocation(x, y);
		third.x = x;
		second.y = y;
		updateGravityCenter();
		shape = getInsideOutsideOrMiddleBorders();
	}
	
	
	
	
	@Override
	public synchronized void setLastPoint(LaTeXDrawPoint2D pt)
	{	
		// As the this second point pt doesn't necessary
		// creates a square but a rectangle, we need to change
		// this point in the goal that it allows to create a square
		LaTeXDrawPoint2D third = getPoint(2), fourth = getPoint(3), second = getPoint(1);
		LaTeXDrawPoint2D first = getPoint(0);
		
		width = Math.abs(first.x-pt.x);
		double x, y;
		
		if(first.y<pt.y)
			   y = first.y + width;
		else   y = first.y - width;
		
		if(first.x<pt.x)
			   x = first.x + width;
		else   x = first.x - width;
		
		// This new point allows to create a square and not a rectangle
		fourth.setLocation(x, y);
		third.y = y;
		second.x = x;
		updateGravityCenter();
		shape = getInsideOutsideOrMiddleBorders();
	}
	
	
	
	
	/**
	 * This method allows to set the third point of the square
	 * @param pt The new third point
	 */
	public synchronized void setThirdPoint(LaTeXDrawPoint2D pt)
	{
		// As the this second point pt doesn't necessary
		// creates a square but a rectangle, we need to change
		// this point in the goal that it allows to create a square
		LaTeXDrawPoint2D third = getPoint(2), fourth = getPoint(3), second = getPoint(1);
		LaTeXDrawPoint2D first = getPoint(0);

		width = Math.abs(fourth.x-pt.x);
		double x, y;
		
		if(second.y<pt.y)
			   y = second.y + width;
		else   y = second.y - width;
		
		if(second.x<pt.x)
			   x = second.x + width;
		else   x = second.x - width;
		
		// This new point allows to create a square and not a rectangle
		third.setLocation(x, y);
		fourth.y = y;
		first.x = x;
		updateGravityCenter();
		shape = getInsideOutsideOrMiddleBorders();
	}
	
	
	
	/**
	 * This method allows to set the fourth point of the square
	 * @param pt The new fourth point
	 */
	public synchronized void setSecondPoint(LaTeXDrawPoint2D pt)
	{
		// As the this second point pt doesn't necessary
		// creates a square but a rectangle, we need to change
		// this point in the goal that it allows to create a square
		LaTeXDrawPoint2D third = getPoint(2), fourth = getPoint(3), second = getPoint(1);
		LaTeXDrawPoint2D first = getPoint(0);
		
		width = Math.abs(first.x-pt.x);
		double x, y;
		
		if(third.y<pt.y)
			   y = third.y + width;
		else   y = third.y - width;
		
		if(third.x<pt.x)
			   x = third.x + width;
		else   x = third.x - width;
		
		// This new point allows to create a square and not a rectangle
		second.setLocation(x, y);
		first.y  = y;
		fourth.x = x;
		updateGravityCenter();
		shape = getInsideOutsideOrMiddleBorders();
	}
	
	
	
	
	@Override
	public synchronized String getCodePSTricks(DrawBorders drawBorders, float ppc)
	{
		LaTeXDrawPoint2D pt1 = pts.elementAt(0);
		LaTeXDrawPoint2D pt2 = pts.elementAt(1);
		LaTeXDrawPoint2D d = drawBorders.getOriginPoint();
		boolean isFilledWasChanged = false;
		double x, y;
		String add="", addBegin="", addEnd="", fillType=""; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
		double threshold = 0.001;
		
		y = pt1.y>pt2.y ? pt1.y - width : pt1.y;
		x = pt1.x>pt2.x ? pt1.x - width : pt1.x;
		x = x - d.x;
		y = d.y - y;
		
		if(hasShadow)
		{
			fillType+=",shadow=true";//$NON-NLS-1$
			if(Math.toDegrees(shadowAngle)!=PSTricksConstants.DEFAULT_SHADOW_ANGLE)
				fillType+=",shadowangle="+(float)Math.toDegrees(shadowAngle);//$NON-NLS-1$
			
			if(((float)shadowSize)!=((float)DEFAULT_SHADOW_SIZE))
				fillType+=",shadowsize="+LaTeXDrawNumber.getCutNumber((float)(shadowSize/PPC),threshold);//$NON-NLS-1$
			
			if(!shadowColor.equals(PSTricksConstants.DEFAULT_SHADOW_COLOR))
			{
				String name = DviPsColors.getColourName(shadowColor);
				if(name==null)
				{
					name = "color"+number+'e';//$NON-NLS-1$
					DviPsColors.addUserColour(shadowColor, name); 
				}
				add += ",shadowcolor=" + name; //$NON-NLS-1$
			}
			if(!isFilled)
			{
				isFilled = true;
				isFilledWasChanged = true;
			}
		}

		String str = getPSTricksCodeFilling(ppc);
		if(str.length()>0) fillType=fillType+','+str;
		
		str = getPSTricksCodeLine(ppc);
		if(str.length()>0) add=add+','+str;
		
		if(isRound)
			add+=",framearc="+(float)frameArc; //$NON-NLS-1$
		
		if(rotationAngle%(Math.PI*2)!=0.)
		{
			double angle = -Math.toDegrees(rotationAngle);
			double cx = (gravityCenter.x-d.x)/ppc;
			double cy = (d.y-gravityCenter.y)/ppc;
			double x2 = -Math.cos(-rotationAngle)*cx+
						Math.sin(-rotationAngle)*cy+cx;
			double y2 =  -Math.sin(-rotationAngle)*cx-
						Math.cos(-rotationAngle)*cy+cy;
			
			if(Math.abs(x2) < 0.001) x2 = 0;
			if(Math.abs(y2) < 0.001) y2 = 0;
			
			addBegin +="\\rput{"+(float)angle+ "}("+(float)x2+','+(float)y2+"){"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			addEnd = "}"; //$NON-NLS-1$
		}
		
		add+=",dimen="+bordersPosition;  //$NON-NLS-1$
		
		if(hasDoubleBoundary)
		{
			add+=",doubleline=true,doublesep="+LaTeXDrawNumber.getCutNumber((float)(doubleSep/ppc),threshold); //$NON-NLS-1$
			
			if(doubleColor!=PSTricksConstants.DEFAULT_DOUBLE_COLOR)
			{
				String name = DviPsColors.getColourName(doubleColor);
				if(name==null)
				{
					name = "color"+number+'d';//$NON-NLS-1$
					DviPsColors.addUserColour(doubleColor, name); 
				}
				add+= ",doublecolor="+name; //$NON-NLS-1$
			}
		}
		
		if(isFilledWasChanged) isFilled = false;
		
		return addBegin+"\\psframe[linewidth=" + LaTeXDrawNumber.getCutNumber(thickness/ppc,threshold) +  //$NON-NLS-1$
				add + fillType+"](" + LaTeXDrawNumber.getCutNumber((float)((x+width)/ppc),threshold) + ',' +  //$NON-NLS-1$
				LaTeXDrawNumber.getCutNumber((float)(y/ppc),threshold) + ")("  //$NON-NLS-1$
				+ LaTeXDrawNumber.getCutNumber((float)(x/ppc),threshold) + ',' + LaTeXDrawNumber.getCutNumber((float)((y-width)/ppc),threshold) + ')'+addEnd;
	}

	
	
	
	@Override
	public Square clone() throws CloneNotSupportedException
	{
		Square s = (Square) super.clone();
		s.width = width;

		return s;
	}
	
	
	
	
	@SuppressWarnings("unchecked")
	private void readObject(ObjectInputStream ois) throws IOException, ClassNotFoundException
	{
		interiorColor = (Color) ois.readObject();
		lineStyle = (String) ois.readObject();
		rotationAngle = ois.readDouble();
		float thick = ois.readFloat();
		thickness = DEFAULT_THICKNESS;
		isFilled = ois.readBoolean();
		isSelected = ois.readBoolean();
		isOnRotation = ois.readBoolean();
		linesColor = (Color) ois.readObject();
		blackDashLength = ois.readFloat();
		dotSep = ois.readFloat();
		whiteDashLength = ois.readFloat();
		pts = (Vector) ois.readObject();
		isRound = ois.readBoolean();
		frameArc = ois.readDouble();
		width = ois.readDouble();
		
		if(pts.size()!=NB_POINTS_FRAME)
			throw new ArrayIndexOutOfBoundsException();
		
		dNW = new Delimitor(pts.firstElement());
		dNE = new Delimitor(pts.elementAt(1));
		dSW = new Delimitor(pts.elementAt(2));
		dSE = new Delimitor(pts.elementAt(3));

		if(LaTeXDrawFrame.getVersionOfFile().compareTo("1.5")>=0)//$NON-NLS-1$
		{
			hasDoubleBoundary = ois.readBoolean();
			doubleColor = (Color)ois.readObject();
			doubleSep = ois.readDouble();
			bordersPosition = (String)ois.readObject();
			if(!(LaTeXDrawFrame.getVersionOfFile().compareTo("1.6")>=0)) //$NON-NLS-1$
				ois.readBoolean();
			hatchingAngle = ois.readDouble();
			hatchingColor = (Color)ois.readObject();
			hatchingStyle = (String)ois.readObject();
			hatchingWidth = ois.readFloat();
			
			if(LaTeXDrawFrame.getVersionOfFile().compareTo("1.6") < 0)//$NON-NLS-1$
			{
				if(hatchingStyle.equals(DECREPETED_FILL_CROSS))
					hatchingStyle = PSTricksConstants.TOKEN_FILL_CROSSHATCH;
				else if(hatchingStyle.equals(DECREPETED_FILL_HORIZ))
					hatchingStyle = PSTricksConstants.TOKEN_FILL_HLINES;
				else if(hatchingStyle.equals(DECREPETED_FILL_VERT))
					hatchingStyle = PSTricksConstants.TOKEN_FILL_VLINES;
				else if(hatchingStyle.equals(DECREPETED_FILL_NO))
					hatchingStyle = PSTricksConstants.TOKEN_FILL_NONE;
			}
		}
		else
		{
			hasDoubleBoundary  = DEFAULT_HAS_DOUBLE_BOUNDARY;
			doubleColor = DEFAULT_DOUBLE_COLOR;
			doubleSep   = DEFAULT_DOUBLESEP;
			bordersPosition = DEFAULT_BORDERS_POSITION;
			hatchingAngle = DEFAULT_HATCH_ANGLE;
			hatchingColor = DEFAULT_HATCH_COL;
			hatchingStyle = DEFAULT_HATCH_STYLE;
			hatchingWidth = DEFAULT_HATCH_WIDTH;
		}
		
		if(LaTeXDrawFrame.getVersionOfFile().compareTo("1.7")>=0) //$NON-NLS-1$
		{
			hasShadow 	= ois.readBoolean();
			shadowAngle = ois.readDouble();
			shadowSize	= ois.readDouble();
			shadowColor	= (Color)ois.readObject();
			gradientEndColor = (Color)ois.readObject();
			gradientStartColor = (Color)ois.readObject();
			gradientAngle = ois.readDouble();
			gradientMidPoint = ois.readDouble();
		}
		else
		{
			hasShadow 	= DEFAULT_SHADOW_HAS;
			shadowAngle	= DEFAULT_SHADOW_ANGLE;
			shadowSize	= DEFAULT_SHADOW_SIZE;
			shadowColor	= DEFAULT_SHADOW_COLOR;
			gradientEndColor = PSTricksConstants.DEFAULT_GRADIENT_END_COLOR;
			gradientStartColor = PSTricksConstants.DEFAULT_GRADIENT_START_COLOR;
			gradientAngle = DEFAULT_GRADIENT_ANGLE;
			gradientMidPoint = DEFAULT_GRADIENT_MID_POINT;
		}
		
		if(LaTeXDrawFrame.getVersionOfFile().compareTo("1.8")>=0) //$NON-NLS-1$
			hatchingSep = ois.readDouble();
		else
			hatchingSep = DEFAULT_HATCH_SEP;
		
		canHaveShadow = true;
		setThickness(thick);
		updateStyleOfDelimitors();
		shape = getInsideOutsideOrMiddleBorders();
	}


	

	@Override
	public boolean isIn(LaTeXDrawPoint2D p)
	{
		LaTeXDrawPoint2D pt = rotateInvertPoint(p);
		
		if(isSelected() && (dNW.isIn(pt) || dSE.isIn(pt) || dSW.isIn(pt) || dNE.isIn(pt)))
			return true;

		LaTeXDrawPoint2D NW = getTheNWNonRotatedBoundPoint();
		LaTeXDrawPoint2D SE = getTheSENonRotatedBoundPoint();
		Rectangle2D.Double s = new Rectangle2D.Double(NW.x, NW.y, 
				Math.abs(NW.x - SE.x), Math.abs(NW.y - SE.y));

		if (!s.contains(pt))
			return false;

		if(isFilled || hasShadow ||  hasGradient())
			return true;

		Shape s2;

		if (hasDoubleBoundary)
			s2 = new Rectangle2D.Double(NW.x + thickness * 2 + doubleSep,
					NW.y + thickness * 2 + doubleSep, Math.abs(NW.x - SE.x)
					- 4 * thickness - 2 * doubleSep, Math.abs(NW.y - SE.y) - 4 * thickness - 2 * doubleSep);
		else
			s2 = new Rectangle2D.Double(NW.x + thickness, NW.y + thickness,
					Math.abs(NW.x - SE.x) - 2 * thickness, Math.abs(NW.y - SE.y) - 2 * thickness);

		return !s2.contains(pt);
	}


	
	
	@Override
	public void onClick(Point p)
	{
		updateNSEWDelimitors();
		
		LaTeXDrawPoint2D pt = rotateInvertPoint(p);
		
		if(dSE.isIn(pt))      dSelected = dSE;
		else if(dNW.isIn(pt)) dSelected = dNW;
		else if(dNE.isIn(pt)) dSelected = dNE;
		else if(dSW.isIn(pt)) dSelected = dSW;
		isSelected = true;
	}




	@Override
	public void updateToGrid(MagneticGrid grid)
	{
		LaTeXDrawPoint2D pt0 = getPoint(0);
		LaTeXDrawPoint2D pt1 = getPoint(1);
		LaTeXDrawPoint2D pt2 = getPoint(2);
		LaTeXDrawPoint2D pt3 = getPoint(3);
		
		pt0.setLocation(grid.getTransformedPointToGrid(pt0, false));
		pt1.setLocation(grid.getTransformedPointToGrid(pt1, false));
		width = Math.abs(pt0.x-pt1.x);
		pt2.x = pt0.x;
		pt2.y = pt0.y + width;
		pt3.x = pt1.x;
		pt3.y = pt1.y + width;
		
		dNW.setCoordinates(pt0);
		dNE.setCoordinates(pt1);
		dSW.setCoordinates(getPoint(2));
		dSE.setCoordinates(getPoint(3));
		
		updateShape();
	}
	
	
	
	@Override
	public int hashCode()
	{
		return super.hashCode()+(int)width;
	}
}
