/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */

/*!
    \page lcmaps_voms.mod voms plugin

    \section vomssyn SYNOPSIS
        \b lcmaps_voms.mod -vomsdir \<vomsdir\> -certdir \<certdir\>


    \section vomsdescr DESCRIPTION

    This plugin forms the link between the VOMS data found in the user grid credential (X509
    certificate) and the lcmaps system.
    It will retrieve the VOMS data by using the VOMS API.
    The plugin stores the VOMS data in the LCMAPS process space, where it is accessible by
    other 'VOMS-aware' plugins, and should, therefore, be evaluated before the other
    plugins, that actually gather the local credentials based on the VOMS information
    (e.g. \ref lcmaps_voms_poolaccount.mod "lcmaps_voms_poolaccount.mod",
          \ref lcmaps_voms_poolgroup.mod "lcmaps_voms_poolgroup.mod" and
          \ref lcmaps_voms_localgroup.mod "lcmaps_voms_localgroup.mod").


    \section vomsoptions OPTIONS
    \subsection vomsoptie1 -VOMSDIR \<vomsdir\>
        See \ref vomsoptie2 "-vomsdir"

    \subsection vomsoptie2 -vomsdir \<vomsdir\>
        This is the directory which contains the certificates of the VOMS servers

    \subsection vomsoptie3 -CERTDIR \<certdir\>
        See \ref vomsoptie4 "-certdir"
 
    \subsection vomsoptie4 -certdir \<certdir\>
        This is the directory which contains the CA certificates


\section vomsReturnvalue RETURN VALUES
        \li LCMAPS_MOD_SUCCESS : Success
        \li LCMAPS_MOD_FAIL    : Failure
 
 
\section vomsErrors ERRORS
        See bugzilla for known errors (http://marianne.in2p3.fr/datagrid/bugzilla/)
 
\section vomsSeeAlso SEE ALSO
        \ref lcmaps_voms_poolaccount.mod "lcmaps_voms_poolaccount.mod",
        \ref lcmaps_voms_poolgroup.mod "lcmaps_voms_poolgroup.mod",
        \ref lcmaps_voms_localgroup.mod "lcmaps_voms_localgroup.mod"
        \ref lcmaps_localaccount.mod "lcmaps_localaccount.mod",
        \ref lcmaps_poolaccount.mod "lcmaps_poolaccount.mod",
        \ref lcmaps_posix_enf.mod "lcmaps_posix_enf.mod",
        \ref lcmaps_ldap_enf.mod "lcmaps_ldap_enf.mod",
*/

/*!
    \file   lcmaps_voms.c
    \brief  Interface to the LCMAPS plugins
    \author Martijn Steenbakkers for the EU DataGrid.

    This file contains the code for the voms plugin (extracts the VOMS info from the
    certificate). The interface consists of the following functions:
    -# plugin_initialize()
    -# plugin_run()
    -# plugin_terminate()
    -# plugin_introspect()
*/

/*****************************************************************************
                            Include header files
******************************************************************************/
#include "lcmaps_voms_config.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pwd.h>
#include <openssl/x509.h>
#include <gssapi.h>

#include <lcmaps/lcmaps_modules.h>
#include <lcmaps/lcmaps_arguments.h>
#include <lcmaps/lcmaps_cred_data.h>
#include <lcmaps/lcmaps_vo_data.h>

#include "lcmaps_voms_gsi_utils.h"

#include <voms/voms_apic.h>
#include <globus_gss_assist.h>

#undef TESTBIO
#ifdef TESTBIO
#    include <openssl/pem.h>
#    include <openssl/bio.h>
#endif


/******************************************************************************
                                Definitions
******************************************************************************/
#define PLUGIN_RUN      0
#define PLUGIN_VERIFY   1

#define VOMS_BUFFER_SIZE 1024


/******************************************************************************
                          Module specific prototypes
******************************************************************************/
static int plugin_run_or_verify(int, lcmaps_argument_t *, int);
static void print_vomsdata(struct vomsdata *);

#ifdef TESTBIO
static STACK_OF(X509) *load_chain(char *certfile);
static char *retmsg[] = { "VERR_NONE", "VERR_NOSOCKET", "VERR_NOIDENT", "VERR_COMM", 
                          "VERR_PARAM", "VERR_NOEXT", "VERR_NOINIT",
                          "VERR_TIME", "VERR_IDCHECK", "VERR_EXTRAINFO",
                          "VERR_FORMAT", "VERR_NODATA", "VERR_PARSE",
                          "VERR_DIR", "VERR_SIGN", "VERR_SERVER", 
                          "VERR_MEM", "VERR_VERIFY", "VERR_IDENT",
                          "VERR_TYPE", "VERR_ORDER" };
#endif

/******************************************************************************
                       Define module specific variables
******************************************************************************/


static char * certdir = NULL;
static char * vomsdir = NULL;
static char   voms_buffer[VOMS_BUFFER_SIZE];

#ifdef TESTBIO
static STACK_OF(X509) *load_chain(char *certfile)
{
    STACK_OF(X509_INFO) *sk=NULL;
    STACK_OF(X509) *stack=NULL, *ret=NULL;
    BIO *in=NULL;
    X509_INFO *xi;
    int first = 1;

    if(!(stack = sk_X509_new_null()))
    {
        printf("%s: memory allocation failure\n", logstr);
        goto end;
    }

    if(!(in=BIO_new_file(certfile, "r")))
    {
        printf("%s: error opening the file, %s\n", logstr,certfile);
        goto end;
    }

    /* This loads from a file, a stack of x509/crl/pkey sets */
    if(!(sk=PEM_X509_INFO_read_bio(in,NULL,NULL,NULL)))
    {
        printf("%s: error reading the file, %s\n", logstr,certfile);
        goto end;
    }

    /* scan over it and pull out the certs */
    while (sk_X509_INFO_num(sk))
    {
        /* skip first cert */
        if (first)
        {
            first = 0;
            continue;
        }
        xi=sk_X509_INFO_shift(sk);
        if (xi->x509 != NULL)
        {
            sk_X509_push(stack,xi->x509);
            xi->x509=NULL;
        }
        X509_INFO_free(xi);
    }
    if(!sk_X509_num(stack))
    {
        printf("%s: no certificates in file, %s\n", logstr,certfile);
        sk_X509_free(stack);
        goto end;
    }
    ret=stack;
end:
    BIO_free(in);
    sk_X509_INFO_free(sk);
    return(ret);
}
#endif

/******************************************************************************
Function:   plugin_initialize
Description:
    Initialize plugin
Parameters:
    argc, argv
    argv[0]: the name of the plugin
Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
    LCMAPS_MOD_NOFILE  : db file not found (will halt LCMAPS initialization)
******************************************************************************/
int plugin_initialize(
        int argc,
        char ** argv
)
{
    char * logstr = "lcmaps_plugin_voms-plugin_initialize()";
    int i;

    lcmaps_log_debug(1,"%s: passed arguments:\n", logstr);
    for (i=0; i < argc; i++)
    {
       lcmaps_log_debug(2,"%s: arg %d is %s\n", logstr, i, argv[i]);
    }

    /*
     * CERTDIR = The directory which contains the CA certificates
     * VOMSDIR = The directory which contains the certificates of the VOMS servers 
     */

    /*
     * Parse arguments, argv[0] = name of plugin, so start with i = 1
     */
    for (i = 1; i < argc; i++)
    {
        if ( ((strcmp(argv[i], "-vomsdir") == 0) ||
              (strcmp(argv[i], "-VOMSDIR") == 0))
             && (i + 1 < argc))
        {
            if ((argv[i + 1] != NULL) && (strlen(argv[i + 1]) > 0))
            {
                 vomsdir = strdup(argv[i + 1]);
            }
            i++;
        }
        else if ( ((strcmp(argv[i], "-certdir") == 0) ||
                   (strcmp(argv[i], "-CERTDIR") == 0))
                   && (i + 1 < argc))
        {
            if ((argv[i + 1] != NULL) && (strlen(argv[i + 1]) > 0))
            {
                 certdir = strdup(argv[i + 1]);
            }
            i++;
        }
        else
        {
            lcmaps_log(LOG_ERR,"%s: Error in initialization parameter: %s (failure)\n", logstr,
                       argv[i]);
            return LCMAPS_MOD_FAIL;
        }
    }
    return LCMAPS_MOD_SUCCESS;
} 


/******************************************************************************
Function:   plugin_introspect
Description:
    return list of required arguments
Parameters:

Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
******************************************************************************/
int plugin_introspect(
        int * argc,
        lcmaps_argument_t ** argv
)
{
    char * logstr = "\tlcmaps_plugin_voms-plugin_introspect()";
    static lcmaps_argument_t argList[] = {
        { "user_dn"   , "char *"        ,  1, NULL},
        { "user_cred" , "gss_cred_id_t" ,  0, NULL},
        { NULL        , NULL            , -1, NULL}
    };

    lcmaps_log_debug(4,"%s: introspecting\n", logstr);

    *argv = argList;
    lcmaps_log_debug(5,"%s: before lcmaps_cntArgs()\n", logstr);
    *argc = lcmaps_cntArgs(argList);
    lcmaps_log_debug(5,"%s: address first argument: 0x%x\n", logstr,argList);

    return LCMAPS_MOD_SUCCESS;
}

/******************************************************************************
Function:   plugin_run
Description:
    Gather credentials for LCMAPS
Parameters:
    argc: number of arguments
    argv: list of arguments
Returns:
    LCMAPS_MOD_SUCCESS: authorization succeeded
    LCMAPS_MOD_FAIL   : authorization failed
******************************************************************************/
int plugin_run(
        int argc,
        lcmaps_argument_t * argv
)
{
    return plugin_run_or_verify(argc, argv, PLUGIN_RUN);
}

/******************************************************************************
Function:   plugin_verify
Description:
    Verify if user is entitled to use local credentials based on his grid
    credentials. This means that the site should already have been set up
    by, e.g., LCMAPS in a previous run. This method will not try to setup
    account leases, modify (distributed) passwd/group files, etc. etc.
    The outcome should be identical to that of plugin_run().
    In this particular case "plugin_verify()" is identical to "plugin_run()"

Parameters:
    argc: number of arguments
    argv: list of arguments
Returns:
    LCMAPS_MOD_SUCCESS: authorization succeeded
    LCMAPS_MOD_FAIL   : authorization failed
******************************************************************************/
int plugin_verify(
        int argc,
        lcmaps_argument_t * argv
)
{
    return plugin_run_or_verify(argc, argv, PLUGIN_VERIFY);
}

static int plugin_run_or_verify(
        int argc,
        lcmaps_argument_t * argv,
        int lcmaps_mode
)
{
    char * logstr = "lcmaps_plugin_voms-plugin_run()";
    char *              dn          = NULL; 
    struct vomsdata *   vd          = NULL;
    int                 errNo       = 0;
    gss_cred_id_t *     pcred       = NULL;
    gss_cred_id_t       cred        = GSS_C_NO_CREDENTIAL;
    X509 *              px509_cred  = NULL;
    STACK_OF(X509) *    px509_chain = NULL;
    void *              value       = NULL;

#ifdef TESTBIO
    int err;
    int res;
    BIO *in = NULL;
    X509 *x = NULL;
    STACK_OF(X509) *chain;
#endif


    /*
     * The beginning
     */
    if (lcmaps_mode == PLUGIN_RUN)
        logstr = "lcmaps_plugin_voms-plugin_run()";
    else if (lcmaps_mode == PLUGIN_VERIFY)
        logstr = "lcmaps_plugin_voms-plugin_verify()";
    else
    {
        lcmaps_log(LOG_ERR, "lcmaps_plugin_voms-plugin_run_or_verify(): attempt to run plugin in invalid mode: %d\n", lcmaps_mode);
        goto fail_voms;
    }
    lcmaps_log_debug(LOG_DEBUG,"%s:\n", logstr);

    /*
     * Try to get the ordered values:
     */
    if ( ( value = lcmaps_getArgValue("user_dn", "char *", argc, argv) ) )  {
	dn= *(char **)value;
        lcmaps_log_debug(1,"%s: found dn: %s\n", logstr,dn);
    } else {
	dn = NULL;
        lcmaps_log_debug(1,"%s: could not get value of dn !\n", logstr);
    }

    /* Fetch user gss credential */
    if ( ( value = lcmaps_getArgValue("user_cred", "gss_cred_id_t", argc, argv) ) )
    {
	pcred = (gss_cred_id_t *) value;
        lcmaps_log_debug(2,"%s: address user_cred: %p\n", logstr,pcred);
        cred = *pcred;
        if (cred == GSS_C_NO_CREDENTIAL)
        {
            lcmaps_log(LOG_NOTICE,"%s: user gss credential is empty ! (exit voms)\n", logstr);
            goto fail_voms;
        }
    }
    else
    {	
	pcred = NULL;
        lcmaps_log(LOG_DEBUG,"%s: could not get address of user_cred (exit voms)!\n", logstr);
        goto fail_voms;
    }

    /*
     * Retrieve a newly created X509 struct and X509 chain from gss credential (should be freed)
     */
    if ( ( px509_cred = lcmaps_cred_to_x509(cred) ) )
    {
        lcmaps_log_debug(1,"%s: found X509 struct inside gss credential\n", logstr);
        lcmaps_log_debug(5,"%s: just for kicks: X509->name %s\n", logstr,px509_cred->name);
    }
    else
    {
        lcmaps_log(LOG_ERR,"%s: could not get X509 cred (exit voms)!\n", logstr);
        goto fail_voms;
    }
    if ( ( px509_chain = lcmaps_cred_to_x509_chain(cred) ) )
    {
        lcmaps_log_debug(1,"%s: found X509 chain inside gss credential\n", logstr);
    }
    else
    {
        lcmaps_log(LOG_ERR,"%s: could not get X509 chain (exit voms)!\n", logstr);
        goto fail_voms;
    }

    lcmaps_log_debug(1,"%s: vomsdir = %s\n", logstr, vomsdir);
    lcmaps_log_debug(1,"%s: certdir = %s\n", logstr, certdir);
    if ((vd = VOMS_Init(vomsdir, certdir)) == NULL)
    {
        lcmaps_log(LOG_ERR,"%s: failed to initialize voms data structure. This may be because either the specified voms directory (%s) or the specified CA certificates directory (%s) does not exist\n", logstr, vomsdir, certdir);
        goto fail_voms;
    }
    lcmaps_log_debug(1,"%s: voms data structure initialized\n", logstr);

#ifdef TESTBIO
    in = BIO_new(BIO_s_file());
    chain = load_chain("/home/gridtest/cvs/fabric_mgt/gridification/lcmaps/modules/voms/x509up_u500");
    if (in)
    {
        if (BIO_read_filename(in, "/home/gridtest/cvs/fabric_mgt/gridification/lcmaps/modules/voms/x509up_u500") > 0)
        {
            x = PEM_read_bio_X509(in, NULL, 0, NULL);

            res = VOMS_Retrieve(x, chain, RECURSE_CHAIN, vd, &err);

            if (res)
                print_vomsdata(vd);
            else
                printf("%s: ERROR!\n", logstr);
        }
    }

    if (!res)
    {
        printf("%s: err: %s\n", logstr, retmsg[err]);
    }
#else
    if (VOMS_Retrieve(px509_cred, px509_chain, RECURSE_CHAIN, 
                         vd, &errNo))
#endif
    {
        lcmaps_vo_data_t * lcmaps_vo_data = NULL;
        struct voms **     volist = vd->data;
        struct voms *      vo;
        char *             bufptr = NULL;
        int k = 0;
        int j = 0;

        lcmaps_log_debug(1,"%s: We got something, errNo = %d\n", logstr, errNo);
        print_vomsdata(vd);

        while(volist[k]) {
            vo = volist[k++];
            lcmaps_log_debug(1,"%s: setting voms data for VO == %s\n", logstr,
                             vo->voname);

            switch (vo->type) {
                case TYPE_NODATA:
                    lcmaps_log_debug(1,"%s: NO DATA\n", logstr);
                    break;
                case TYPE_CUSTOM:
                    lcmaps_log_debug(1,"%s: %*s\n", logstr, vo->datalen - 10, vo->custom);
                    break;
                case TYPE_STD:
                    j = 0;
                    while (vo->std[j]) {
                        lcmaps_vo_data=lcmaps_createVoData(vo->voname,vo->std[j]->group,
                                                           NULL, vo->std[j]->role, vo->std[j]->cap
                        );
                        if (! lcmaps_vo_data)
                        {
                            lcmaps_log(LOG_ERR,"%s: could not create VoData structure (failure)\n", logstr);
                            goto fail_voms;
                        }
//                        lcmaps_printVoData(2,lcmaps_vo_data);
                        if ( lcmaps_stringVoData(lcmaps_vo_data, voms_buffer, VOMS_BUFFER_SIZE) )
                        {
                            lcmaps_log(LOG_ERR,"%s: error in casting VoData structure into string (failure)\n", logstr);
                            goto fail_voms;
                        }
//                        lcmaps_log_debug(1,"%s: buffer: %s\n", logstr, voms_buffer);
                        /* Add credential */
                        /* copy address of voms_buffer[0] in bufptr, because you cannot take the address of the array voms_buffer */
                        bufptr = voms_buffer;
                        addCredentialData(LCMAPS_VO_CRED_STRING, (void *) &bufptr);
                        addCredentialData(LCMAPS_VO_CRED, (void *) lcmaps_vo_data);
                        if ( lcmaps_deleteVoData(&lcmaps_vo_data) )
                        {
                            lcmaps_log(LOG_ERR,"%s: error while deleting VoData structure (failure)\n", logstr);
                            goto fail_voms;
                        }
                        j++;
                    }
                    break;
            }
        }
        lcmaps_log_debug(1,"%s: doing VOMS_Destroy\n", logstr);
        VOMS_Destroy(vd);
        lcmaps_log_debug(1,"%s: done\n", logstr);
    }
#ifdef TESTBIO
#else
    else if (errNo == VERR_NOEXT)
    {
        lcmaps_log(LOG_ERR,"%s: VOMS extensions missing from certificate (failure)!\n", logstr);
        goto fail_voms;
    }
    else if (errNo == VERR_IDCHECK)
    {
        lcmaps_log(LOG_ERR,"%s: VOMS User data in extension different from the real ones (failure)!\n", logstr);
        goto fail_voms;
    }
    else if (errNo == VERR_TIME)
    {
        lcmaps_log(LOG_ERR,"%s: VOMS extensions expired for at least one of the VOs (failure)!\n", logstr);
        goto fail_voms;
    }
    else if (errNo == VERR_ORDER)
    {
        lcmaps_log(LOG_ERR,"%s: The ordering of the VOMS groups, as required by the client, was not delivered by VOMS (failure)!\n", logstr);
        goto fail_voms;
    }
    else if (errNo == VERR_NOSOCKET)
    {
        lcmaps_log(LOG_ERR,"%s: VOMS Socket problem (failure)!\n", logstr);
        goto fail_voms;
    }
    else if (errNo == VERR_NOIDENT)
    {
        lcmaps_log(LOG_ERR,"%s: VOMS Cannot identify itself (certificate problem) (failure)!\n", logstr);
        goto fail_voms;
    }
    else if (errNo == VERR_COMM)
    {
        lcmaps_log(LOG_ERR,"%s: VOMS server problem (failure)!\n", logstr);
        goto fail_voms;
    }
    else if (errNo == VERR_PARAM)
    {
        lcmaps_log(LOG_ERR,"%s: Wrong parameters for VOMS (failure)!\n", logstr);
        goto fail_voms;
    }
    else if (errNo == VERR_NOINIT)
    {
        lcmaps_log(LOG_ERR,"%s: VOMS initialization error (failure)!\n", logstr);
        goto fail_voms;
    }
    else if (errNo == VERR_EXTRAINFO)
    {
        lcmaps_log(LOG_ERR,"%s: VO name and URI missing (in proxy ?) (failure)!\n", logstr);
        goto fail_voms;
    }
    else if (errNo == VERR_FORMAT)
    {
        lcmaps_log(LOG_ERR,"%s: Wrong VOMS data format (in proxy ?) (failure)!\n", logstr);
        goto fail_voms;
    }
    else if (errNo == VERR_NODATA)
    {
        lcmaps_log(LOG_ERR,"%s: Empty VOMS extension (failure)!\n", logstr);
        goto fail_voms;
    }
    else if (errNo == VERR_PARSE)
    {
        lcmaps_log(LOG_ERR,"%s: VOMS parse error (failure)!\n", logstr);
        goto fail_voms;
    }
    else if (errNo == VERR_DIR)
    {
        lcmaps_log(LOG_ERR,"%s: VOMS directory error (failure)!\n", logstr);
        goto fail_voms;
    }
    else if (errNo == VERR_SIGN)
    {
        lcmaps_log(LOG_ERR,"%s: VOMS Signature error (failure)!\n", logstr);
        goto fail_voms;
    }
    else if (errNo == VERR_SERVER)
    {
        lcmaps_log(LOG_ERR,"%s: Unidentifiable VOMS server (failure)!\n", logstr);
        goto fail_voms;
    }
    else if (errNo == VERR_MEM)
    {
        lcmaps_log(LOG_ERR,"%s: Memory problems in VOMS_Retrieve() (failure)!\n", logstr);
        goto fail_voms;
    }
    else if (errNo == VERR_VERIFY)
    {
        lcmaps_log(LOG_ERR,"%s: Generic verification error for VOMS (failure)!\n", logstr);
        goto fail_voms;
    }
    else if (errNo == VERR_TYPE)
    {
        lcmaps_log(LOG_ERR,"%s: Returned VOMS data of unknown type (failure)!\n", logstr);
        goto fail_voms;
    }
    else
    {
        lcmaps_log(LOG_ERR,"%s: VOMS_Retrieve() error --> %d (failure)!\n", logstr, errNo);
        goto fail_voms;
    }
#endif

    /* succes */
 success_voms:
    if (px509_cred) X509_free(px509_cred);
    if (px509_chain) sk_X509_free(px509_chain);
    lcmaps_log(LOG_INFO,"%s: voms plugin succeeded\n", logstr);
    return LCMAPS_MOD_SUCCESS;

 fail_voms:
    if (px509_cred) X509_free(px509_cred);
    if (px509_chain) sk_X509_free(px509_chain);
    lcmaps_log(LOG_INFO,"%s: voms plugin failed\n", logstr);
    return LCMAPS_MOD_FAIL;
}

/******************************************************************************
Function:   plugin_terminate
Description:
    Terminate plugin
Parameters:

Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
******************************************************************************/
int plugin_terminate()
{
    char * logstr = "lcmaps_plugin_voms-plugin_terminate()";
    lcmaps_log_debug(LOG_DEBUG,"%s: terminating\n", logstr);
    if (vomsdir) free(vomsdir);
    if (certdir) free(certdir);

    return LCMAPS_MOD_SUCCESS;
}

static void print_vomsdata(struct vomsdata *d)
{
    char * logstr = "lcmaps_plugin_voms-print_vomsdata()";
    struct voms **vo = d->data;
    struct voms *v;
    int k = 0;
    int j =0;

    while(vo[k])
    {
        v = vo[k++];
        lcmaps_log_debug(1,"%s: %d *******************************************\n", logstr,k);
        lcmaps_log_debug(1,"%s: SIGLEN: %d\n", logstr, v->siglen);
        lcmaps_log_a_string_debug(1, "lcmaps_plugin_voms-print_vomsdata(): USER:   %s\n", v->user);
        lcmaps_log_a_string_debug(1, "lcmaps_plugin_voms-print_vomsdata(): UCA:    %s\n", v->userca);
        lcmaps_log_a_string_debug(1, "lcmaps_plugin_voms-print_vomsdata(): SERVER: %s\n", v->server);
        lcmaps_log_a_string_debug(1, "lcmaps_plugin_voms-print_vomsdata(): SCA:    %s\n", v->serverca);
        lcmaps_log_a_string_debug(1, "lcmaps_plugin_voms-print_vomsdata(): VO:     %s\n", v->voname);
        lcmaps_log_a_string_debug(1, "lcmaps_plugin_voms-print_vomsdata(): URI:    %s\n", v->uri);
        lcmaps_log_a_string_debug(1, "lcmaps_plugin_voms-print_vomsdata(): DATE1:  %s\n", v->date1);
        lcmaps_log_a_string_debug(1, "lcmaps_plugin_voms-print_vomsdata(): DATE2:  %s\n", v->date2);

        switch (v->type)
        {
        case TYPE_NODATA:
            lcmaps_log_debug(1,"%s: NO DATA\n", logstr);
            break;
        case TYPE_CUSTOM:
            lcmaps_log_debug(1,"%s: VOMS custom type. Wont print.\n", logstr);
            break;
        case TYPE_STD:
            j = 0;
            if (v->fqan)
            {
                while ((v->fqan)[j] != NULL)
                {
                    lcmaps_log_a_string_debug(1,
                        "lcmaps_plugin_voms-print_vomsdata(): fqan:   %s\n",
                        (v->fqan)[j]);
                    j++;
                }
            }
            j = 0;
            if (v->std)
            {
                while (v->std[j])
                {
                    lcmaps_log_a_string_debug(1,
                        "lcmaps_plugin_voms-print_vomsdata(): GROUP:  %s\n", v->std[j]->group);
                    lcmaps_log_a_string_debug(1,
                        "lcmaps_plugin_voms-print_vomsdata(): ROLE:   %s\n", v->std[j]->role);
                    lcmaps_log_a_string_debug(1,
                        "lcmaps_plugin_voms-print_vomsdata(): CAP:    %s\n", v->std[j]->cap);
                    j++;
                }
            }
            break;
        }
        lcmaps_log_debug(1,"%s: %d *******************************************\n", logstr,k);
    }

    if (d->workvo)
    {
        lcmaps_log_a_string_debug(1,
            "lcmaps_plugin_voms-print_vomsdata(): WORKVO: %s\n", d->workvo);
    }

    if (d->extra_data)
    {
        lcmaps_log_a_string_debug(1,
            "lcmaps_plugin_voms-print_vomsdata(): EXTRA: %s\n", d->extra_data);
    }
}

/******************************************************************************
CVS Information:
    $Source: /srv/home/dennisvd/svn/mw-security/lcmaps-plugins-voms/src/voms/lcmaps_voms.c,v $
    $Date: 2010-02-19 06:01:37 $
    $Revision: 1.6 $
    $Author: okoeroo $
******************************************************************************/
