/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \file   lcmaps_pluginmanager.c
    \brief  the plugin manager for LCMAPS
    \author Martijn Steenbakkers for the EU DataGrid.

    The interface to the PluginManager module is composed of:
    -# lcmaps_startPluginManager(): start the PluginManager --> load plugins, start evaluation manager
    -# lcmaps_runPluginManager(): run the PluginManager --> run evaluation manager --> run plugins
    -# lcmaps_stopPluginManager(): stop the PluginManager
    -# reloadPluginManager(): reload the PluginManager ? (will we implement this ?)
    -# lcmaps_runPlugin(): run the specified plugin. (used by Evaluation Manager)
    -# lcmaps_resetCredentialData(): reset the Credential Data. (used by Evaluation Manager)
*/

#ifndef LCMAPS_PLUGINMANAGER_C
#define LCMAPS_PLUGINMANAGER_C

/*****************************************************************************
                            Include header files
******************************************************************************/
#include "lcmaps_config.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <signal.h>

#include <errno.h>
#include <sys/stat.h>

#if HAVE_MALLOC_H
#    include <malloc.h>
#endif

/* Programming interface to dynamic linking loader */
#if HAVE_DLFCN_H
#    include <dlfcn.h>
#endif

/* LCMAPS includes */
#include "lcmaps_types.h"
#include "lcmaps_log.h"
#include "lcmaps_arguments.h"
#include "lcmaps_defines.h"
#include "_lcmaps_utils.h"
#include "_lcmaps_db_read.h"
#include "_lcmaps_runvars.h"
#include "_lcmaps_cred_data.h"
#include "_lcmaps_pluginmanager.h"
/* interface to evaluationmanager */
#include "../evaluationmanager/evaluationmanager.h"

/******************************************************************************
                             Define constants
******************************************************************************/

#ifndef NUL
#define NUL '\0' /*!< NUL character \internal */
#endif

/* LCMAPS authorization module definitions */
#define MAXPROCS    5 /*!< maximum number of interface symbols in plugin modules \internal */
#define USE_EVALUATIONMANAGER 1


/******************************************************************************
                               Type definitions
******************************************************************************/
/*!
    \enum lcmaps_proctype_e
    \brief This enumeration type gives the different plugin symbol/function types.
    \internal
*/
enum lcmaps_proctype_e
{
    INITPROC,   /*!< this value corresponds to the plugin initialization function */
    RUNPROC,    /*!< this value corresponds to the plugin run function (get credentials) */
    TERMPROC,   /*!< this value corresponds to the plugin termination function */
    INTROPROC,  /*!< this value corresponds to the plugin introspect function */
    VERIFYPROC, /*!< this value corresponds to the plugin verification function */
    ENDOFPROCS  /*< this is the last enumeration value */
};

/*!
    \typedef lcmaps_proc_t
    \brief this type corresponds to the types of the plugin interface functions
    \internal
*/
typedef int (*lcmaps_proc_t)();

/*!
    \typedef lcmaps_plugindl_t
    \brief the type definition of the lcmaps plugin module structure
    \internal
*/
/*!
    \struct lcmaps_plugindl_s
    \brief the lcmaps plugin module structure
    \internal
*/
typedef struct lcmaps_plugindl_s
{
    void *                      handle;                            /*!< dlopen handle to plugin module */
    lcmaps_proc_t               procs[MAXPROCS];                   /*!< list of handles to interface functions of plugin */
    char                        pluginshortname[LCMAPS_MAXPATHLEN+1];   /*!< name of plugin without path */
    char                        pluginabsname[LCMAPS_MAXPATHLEN+1];   /*!< absolute name of plugin */
    char                        pluginargs[LCMAPS_MAXARGSTRING+1]; /*!< argument string */
    int                         init_argc;                         /*!< number of arguments for the initialization function */
    char *                      init_argv[LCMAPS_MAXARGS];         /*!< list of arguments for the initialization function */
    int                         run_argc;                          /*!< number of arguments for the plugin run function (get credentials) */
    lcmaps_argument_t *         run_argv;                          /*!< list of arguments for the plugin run function (get credentials) */
    struct lcmaps_plugindl_s *  next;                              /*!< pointer to the next plugin in the plugin list */
}
lcmaps_plugindl_t;

/******************************************************************************
                          Module specific prototypes
******************************************************************************/
static lcmaps_plugindl_t * PluginInit(lcmaps_db_entry_t *, lcmaps_plugindl_t **);
static lcmaps_proc_t       get_procsymbol(void *, char *);
static int                 print_lcmaps_plugin(int, lcmaps_plugindl_t *);
static int                 parse_args_plugin(const char *, const char *, char **, int *);
static int                 clean_plugin_list(lcmaps_plugindl_t **);
static int                 lcmaps_set_sigpipe_handler(void);
static int                 lcmaps_reset_sigpipe_handler(void);
static void                lcmaps_sigpipe_handler(int sig);
static int    fexist(char *);
static char * lcmaps_findplugin(char *);

/******************************************************************************
                       Define module specific variables
******************************************************************************/
static unsigned int         lcmaps_mode                    = LCMAPS_NORMAL_MODE; /*!< \internal */
static char *               lcmaps_db_file_default         = NULL; /*!< \internal */
static char *               lcmaps_dir                     = NULL; /*!< \internal */
static lcmaps_plugindl_t *  plugin_list                    = NULL; /*!< \internal */
static void *               lcmaps_stored_sigpipe_handler  = NULL; /*!< \internal */

/******************************************************************************
Function:       fexist()
Description:    check the existence of file corresponding to <path>
Parameters:     path
Returns:        1, if file exists
******************************************************************************/
/*!
    \fn fexist(
        char * path
        )
    \brief check the existence of file corresponding to \<path\>
    \param path absolute filename to be checked.
    \retval 1 file exists.
    \retval 0 failure.
*/
static int fexist(
        char * path
)
{
  struct stat sbuf;
  int res;

  if(!path || !*path) return 0;

  res=stat(path,&sbuf);
  if (res)
  {
      if (errno==ENOENT)
      {
          return 0;
      }
      else
      {
          return -1;
      }
  }
  return 1;
}


/******************************************************************************
Function:       lcmaps_findplugin()
Description:    Checks for file in standard directories
Parameters:     name
Returns:        returns filename found (should be freeed) or NULL
******************************************************************************/
/*!
    \fn lcmaps_findplugin(
        char * name
        )
    \brief Checks for file in standard directories

    The directories that are checked are:
    - value of LCMAPS_MODULES_DIR env var (if set)
    - LCMAPS_MOD_HOME

    \param name string containing the file name
    \return pointer to a string containing the absolute path to the file, which
    has to be freed or NULL.
*/
char * lcmaps_findplugin(
        char * name
)
{
    char *tmpname=NULL;
    char *lcmaps_modules_dir=NULL;
    char *pdl_path=NULL;

    /* New format for db: either path is set in db file and plugins have an
     * absolute path with respect to that (when path is relative, the eval
     * manager prefixes with $libdir) or there isn't a path set and the plugin
     * name is to be searched in either env var LCMAPS_MODULES_DIR if set, or in
     * build-time default LCMAPS_MOD_HOME */
    pdl_path=lcmaps_pdl_path();
    if (pdl_path)   {
        /* Absolute path: test it */
        tmpname=lcmaps_genfilename(pdl_path,name,NULL);
        if (!tmpname)    { /* calloc error */
            lcmaps_log(LOG_ERR,"lcmaps_findplugin(): Cannot calloc\n");
            return NULL;
        }
        lcmaps_log_debug(LOG_DEBUG, "Looking for %s at %s...\n", name, tmpname);
        if (fexist(tmpname)) 
            return tmpname;
        free(tmpname);
    } else { /* relative path:
        - do we have an LCMAPS_MODULES_DIR
        - if so then test there first */
        lcmaps_modules_dir=getenv("LCMAPS_MODULES_DIR");
        if (lcmaps_modules_dir)	{
            /* test in and in LCMAPS_MODULES_DIR and build-time default
             * LCMAPS_MOD_HOME */
            tmpname=lcmaps_genfilename(lcmaps_modules_dir,name,NULL);
            if (!tmpname)    { /* calloc error */
            lcmaps_log(LOG_ERR,"lcmaps_findplugin(): Cannot calloc\n");
            return NULL;
            }
            lcmaps_log_debug(LOG_DEBUG, "Looking for %s at %s...\n", name, tmpname);
            if (fexist(tmpname)) return tmpname;
            free(tmpname);
        }
        /* No name so far: test default build-in location */
        tmpname=lcmaps_genfilename(LCMAPS_MOD_HOME,name,NULL);
        if (!tmpname)    { /* calloc error */
            lcmaps_log(LOG_ERR,"lcmaps_findplugin(): Cannot calloc\n");
            return NULL;
        }
        lcmaps_log_debug(5, "Looking for %s at %s...\n", name, tmpname);
        if (fexist(tmpname)) return tmpname;
        free(tmpname);
    }
    return NULL;
}


/******************************************************************************
Function:   lcmaps_startPluginManager
Description:
    Start PluginManager module:
    read from LCMAPS config file, the plugins to be loaded
    Start evaluation manager (get policy)

Parameters:
Returns:
    0: succes
    1: failure
******************************************************************************/
/*!
    \fn lcmaps_startPluginManager(
        )
    \brief start the PluginManager

    start the PluginManager --> load plugins, start evaluation manager

    \retval 0 succes
    \retval 1 failure
*/
int lcmaps_startPluginManager(
)
{
#if USE_EVALUATIONMANAGER
    lcmaps_db_entry_t *   lcmaps_db_handle=NULL;
#else
    lcmaps_db_entry_t **  lcmaps_db_handle=NULL;
#endif
    lcmaps_db_entry_t *   ihandle=NULL;
    lcmaps_plugindl_t *   plugin_entry=NULL;
    lcmaps_plugindl_t *   plugin_entry_cmp=NULL;
    char *                lcmaps_db_file=NULL;
    int                   ientry;
    int                   npol = 0;
    char *                pol1 = NULL;

    /* get LCMAPS (database) home directory and set lcmaps_db_file */
    lcmaps_dir = getenv("LCMAPS_DIR");
    lcmaps_dir = (lcmaps_dir ? lcmaps_dir : getenv("LCMAPS_ETC_DIR") );
    lcmaps_dir = (lcmaps_dir ? lcmaps_dir : LCMAPS_ETC_HOME );

    lcmaps_db_file_default = getenv("LCMAPS_DB_FILE");
    lcmaps_db_file_default = (lcmaps_db_file_default ? lcmaps_db_file_default : "lcmaps.db" );

    lcmaps_db_file=lcmaps_genfilename(lcmaps_dir, lcmaps_db_file_default, NULL);


    /* Setting LCMAPS SIGPIPE handler - old SIGPIPE handler will be preserved */
    lcmaps_set_sigpipe_handler();


#if USE_EVALUATIONMANAGER
    /* To be done:
     * Start evaluation manager (evaluates the  LCMAPS policy database)
     */
    lcmaps_log_debug(4,"lcmaps.mod-lcmaps_startPluginManager(): doing lcmaps_startEvaluationManager(%s)\n", lcmaps_db_file);
    if (lcmaps_startEvaluationManager(lcmaps_db_file, npol, &pol1 ) != 0)
    {
        lcmaps_log(LOG_ERR,"lcmaps.mod-lcmaps_startPluginManager(): lcmaps_startEvaluationManager(%s) failed\n", lcmaps_db_file);
        goto fail_lcmaps_startPluginManager;
    }
#endif

    /* 
     * retrieve PLUGIN MODULE info from evaluation manager and store
     * the info in a lcmaps_db structure
     */
    lcmaps_log(LOG_DEBUG, "lcmaps.mod-lcmaps_startPluginManager(): Reading LCMAPS database %s\n", lcmaps_db_file);
#if USE_EVALUATIONMANAGER
    if (lcmaps_getPluginNameAndArgs(&lcmaps_db_handle) != 0)
    {
        lcmaps_log(LOG_ERR,"lcmaps.mod-lcmaps_startPluginManager(): Error fetching list of plugins from evaluation manager\n");
        goto fail_lcmaps_startPluginManager;
    }
#else
    lcmaps_db_handle=lcmaps_db_read(lcmaps_db_file);
#endif

    if (lcmaps_db_handle == NULL)
    {
#if USE_EVALUATIONMANAGER
        lcmaps_log_debug(1,"lcmaps.mod-lcmaps_startPluginManager(): Evaluation manager found no plugins\n");
#else
        lcmaps_log_debug(1,"lcmaps.mod-lcmaps_startPluginManager(): no plugins found by lcmaps_db_read()\n");
#endif
        goto fail_lcmaps_startPluginManager;
    }

    /*
     * init the PLUGIN AUTHORIZATION MODULES (PluginInit)
     * - open plugins and check the symbols plugin_init and confirm_authorization
     * - run plugin_init
     */
    ientry=0;
#if USE_EVALUATIONMANAGER
    ihandle=lcmaps_db_handle;
#else
    ihandle=*lcmaps_db_handle;
#endif
    while (ihandle)
    {
        if (strlen(ihandle->pluginname) > 0)
        {
            int plugin_count=0;

            lcmaps_log_debug(4,"lcmaps.mod-lcmaps_startPluginManager(): initializing plugin %s (db entry %d)\n", ihandle->pluginname, ientry);
            if ((plugin_entry=PluginInit(ihandle,&plugin_list)) == NULL)
            {
                lcmaps_log(LOG_ERR,"lcmaps.mod-lcmaps_startPluginManager(): error initializing plugin: %s\n",ihandle->pluginname);
                goto fail_lcmaps_startPluginManager;
            }
            /* 
             * Check if plugin module is already registered as standard authorization module,
             * by comparing the complete path names of the plugin and modules
             */
            plugin_entry_cmp=plugin_list;
            while (plugin_entry_cmp)
            {
                if ( (strncmp(plugin_entry_cmp->pluginabsname,
                              plugin_entry->pluginabsname,
                              LCMAPS_MAXPATHLEN) == 0)
                   )
                    plugin_count++;
                if (plugin_count > 1)
                {
                    lcmaps_log(LOG_ERR,"lcmaps.mod-lcmaps_startPluginManager() error: plugin %s already registered as\n", plugin_entry->pluginabsname);
                    goto fail_lcmaps_startPluginManager;
                }
                plugin_entry_cmp=plugin_entry_cmp->next;
            }
        }
        ientry++;
        ihandle=ihandle->next;
    }

    plugin_entry=plugin_list;
    while (plugin_entry)
    {
        print_lcmaps_plugin(4,plugin_entry);
        plugin_entry=plugin_entry->next;
    }

    /* clean PLUGIN authorization module database structure */
#if USE_EVALUATIONMANAGER
    /*
     * Do not clean lcmaps_db_entry_t structure since it is managed by the evaluation manager
     * and will be taken care of in lcmaps_stopEvaluationManager
     */
    /* if (lcmaps_db_clean_list(&lcmaps_db_handle)) */
    /* { */
        /* lcmaps_log(1,"lcmaps.mod-lcmaps_startPluginManager() error: could not clean up plugin db structure\n"); */
        /* goto fail_lcmaps_startPluginManager; */
    /* } */
    /* lcmaps_log_debug(5,"lcmaps.mod-lcmaps_startPluginManager(): cleaned lcmaps_db list: %p\n",lcmaps_db_handle); */
#else
    if (lcmaps_db_clean())
    {
        lcmaps_log(LOG_ERR,"lcmaps.mod-lcmaps_startPluginManager() error: could not clean up plugin db structure\n");
        goto fail_lcmaps_startPluginManager;
    }
    lcmaps_log_debug(5,"lcmaps.mod-lcmaps_startPluginManager(): cleaned lcmaps_db list: %p\n",lcmaps_db_handle);
#endif

    /* success */
    if (lcmaps_db_file) free(lcmaps_db_file);
    return 0;

 fail_lcmaps_startPluginManager:
    /* failure */
#if USE_EVALUATIONMANAGER
    /*
     * Do not clean lcmaps_db_entry_t structure since it is managed by the evaluation manager
     * and will be taken care of in lcmaps_stopEvaluationManager
     */
/*    lcmaps_db_clean_list(&lcmaps_db_handle); */
#else
    lcmaps_db_clean();
#endif
    if (clean_plugin_list(&plugin_list)!=0)
    {
        lcmaps_log(LOG_ERR,"lcmaps.mod-lcmaps_startPluginManager() error: could not clean up plugin list\n");
    }
    if (lcmaps_db_file) free(lcmaps_db_file);

    /* It is necessary to call lcmaps_free_resources() here in case LCMAPS
     * runs multiple times such as in gsisshd. Otherwise, on e.g. OpenSUSE,
     * memory corruption and/or improper close down of flex/yacc leads to a
     * segfaults. */
    lcmaps_free_resources();

    return 1;
}


/******************************************************************************
Function:   PluginInit
Description:
    Initialize the plugin
    - Create entry in (plugin)list
    - Open plugins and check the symbols plugin_init and confirm_authorization
    - run plugin_init

Parameters:
    db_handle: handle to LCMAPS db (containing pluginname and pluginargs)
    list:      pointer to list to which (plugin) module has to be added
Returns:
    pointer to plugin structure
    NULL: could not initialize plugin
******************************************************************************/
/*!
    \fn PluginInit(
        lcmaps_db_entry_t * db_handle,
        lcmaps_plugindl_t ** list
        )
    \brief Initialize the plugin.

    This function takes a plugin LCMAPS database entry and performs the following tasks:
    - Create entry in (plugin)list
    - Open the plugins and check the symbols plugin_init and confirm_authorization
    - run plugin_init

    \param db_handle handle to LCMAPS db (containing pluginname and pluginargs)
    \param list      pointer to plugin structure list to which (plugin) module has to be added

    \return pointer to newly created plugin structure or NULL in case of failure
    \internal
*/
static lcmaps_plugindl_t * PluginInit(
        lcmaps_db_entry_t * db_handle,
        lcmaps_plugindl_t ** list
)
{
    char *                 name;
    char *                 pname=NULL;
    char *                 args=NULL;
    void *                 plugin_handle;
    lcmaps_proc_t          plugin_procs[MAXPROCS];
    lcmaps_plugindl_t *    pplugin=NULL;
    int                    i;
    int                    retval;

    name = db_handle->pluginname;
    args = db_handle->pluginargs;

    /* Find plugin module */
    pname = lcmaps_findplugin(name);
    if (pname == NULL)
    {
        lcmaps_log(LOG_ERR, "lcmaps.mod-PluginInit(): plugin %s not found (arguments: %s)\n", name, args );
        goto fail_PluginInit;
    }

    /* Try a dlopen() */
    plugin_handle=dlopen(pname,RTLD_NOW);
    if (!plugin_handle)
    {
        lcmaps_log(LOG_ERR,"lcmaps.mod-PluginInit(): dlopen error for %s:\n    %s\n", pname,dlerror() );
        goto fail_PluginInit;
    }

    /* Check symbols */
    for (i=0; i < MAXPROCS; ++i)
    {
        switch (i)
        {
            case INITPROC:
                plugin_procs[i]=get_procsymbol(plugin_handle,"plugin_initialize");
                if (plugin_procs[i] == NULL)
                {
                    lcmaps_log(LOG_ERR,"lcmaps.mod-PluginInit(): plugin %s not compliant. Missing function \"plugin_initialize()\"\n", name);
                    goto fail_PluginInit;
                }
                else
                {
                    lcmaps_log_debug(3,"lcmaps.mod-PluginInit(): found \"plugin_initialize()\"\n");
                }
                break;
            case RUNPROC:
                plugin_procs[i]=get_procsymbol(plugin_handle,"plugin_run");
                if (plugin_procs[i] == NULL)
                {
                    lcmaps_log(LOG_ERR,"lcmaps.mod-PluginInit(): plugin %s not compliant. Missing function \"plugin_run()\"\n", name);
                    goto fail_PluginInit;
                }
                else
                {
                    lcmaps_log_debug(3,"lcmaps.mod-PluginInit(): found \"plugin_run()\"\n");
                }
                break;
            case TERMPROC:
                plugin_procs[i]=get_procsymbol(plugin_handle,"plugin_terminate");
                if (plugin_procs[i] == NULL)
                {
                    lcmaps_log(LOG_ERR,"lcmaps.mod-PluginInit(): plugin %s not compliant. Missing function \"plugin_terminate()\"\n", name);
                    goto fail_PluginInit;
                }
                else
                {
                    lcmaps_log_debug(3,"lcmaps.mod-PluginInit(): found \"plugin_terminate()\"\n");
                }
                break;
            case INTROPROC:
                plugin_procs[i]=get_procsymbol(plugin_handle,"plugin_introspect");
                if (plugin_procs[i] == NULL)
                {
                    lcmaps_log(LOG_ERR,"lcmaps.mod-PluginInit(): plugin %s not compliant. Missing function \"plugin_introspect()\"\n", name);
                    goto fail_PluginInit;
                }
                else
                {
                    lcmaps_log_debug(3,"lcmaps.mod-PluginInit(): found \"plugin_introspect()\"\n");
                }
                break;
            case VERIFYPROC:
                plugin_procs[i]=get_procsymbol(plugin_handle,"plugin_verify");
                if (plugin_procs[i] == NULL)
                {
                    lcmaps_log(LOG_INFO,"lcmaps.mod-PluginInit(): no \"plugin_verify()\" method found in %s: old style plugin\n", name);
                }
                else
                {
                    lcmaps_log_debug(3,"lcmaps.mod-PluginInit(): found \"plugin_verify()\"\n");
                }
                break;
            default:
                /* do nothing */
                plugin_procs[i]=NULL;
                break;
        }
    }
    /* Make a new entry at end of plugin list */
    if (!*list)
    {
        lcmaps_log_debug(4,"lcmaps.mod-PluginInit(): creating first pluginlist entry\n");
        *list=pplugin=(lcmaps_plugindl_t *)malloc(sizeof(lcmaps_plugindl_t));
    }
    else
    {
        lcmaps_log_debug(4,"lcmaps.mod-PluginInit(): creating new pluginlist entry\n");
        pplugin=*list;
        while (pplugin->next) pplugin=pplugin->next;
        pplugin=pplugin->next=(lcmaps_plugindl_t *)malloc(sizeof(lcmaps_plugindl_t));
    }
    if (!pplugin)
    {
        lcmaps_log(LOG_ERR,"lcmaps.mod-PluginInit(): error creating new pluginlist entry\n");
        goto fail_PluginInit;
    }
    /* Set plugin list elements */
    pplugin->next=NULL;
    pplugin->handle=plugin_handle;
    pplugin->run_argc=0;
    pplugin->run_argv=NULL;
    for (i=0; i < MAXPROCS; ++i)
        pplugin->procs[i]=plugin_procs[i];

    if (pname != NULL)
    {
        /* name without dir */
        strncpy(pplugin->pluginshortname,name,LCMAPS_MAXPATHLEN);
        (pplugin->pluginshortname)[LCMAPS_MAXPATHLEN]=NUL;
        /* name with dir */
        strncpy(pplugin->pluginabsname,pname,LCMAPS_MAXPATHLEN);
        (pplugin->pluginabsname)[LCMAPS_MAXPATHLEN]=NUL;
    }
    else    {
        strncpy(pplugin->pluginshortname,"",LCMAPS_MAXPATHLEN);
        strncpy(pplugin->pluginabsname,"",LCMAPS_MAXPATHLEN);
    }


    if (args != NULL)
    {
        strncpy(pplugin->pluginargs,args,LCMAPS_MAXARGSTRING);
        (pplugin->pluginargs)[LCMAPS_MAXARGSTRING]=NUL;
    }
    else
        strncpy(pplugin->pluginargs,"",LCMAPS_MAXARGSTRING);

    /* Parse options */
    if (parse_args_plugin(pname,args,pplugin->init_argv,&(pplugin->init_argc)) )
    {
        lcmaps_log(LOG_ERR,"lcmaps.mod-PluginInit(): Could not parse arguments for plugin module %s\n", pname);
        goto fail_PluginInit;
    }
    for (i=0; i < pplugin->init_argc; ++i)
        lcmaps_log_debug(4,"%s arg %d = %s\n",pname,i,(pplugin->init_argv)[i]);


    /* Run plugin_initialize() */
    retval=(*(pplugin->procs[INITPROC]))(pplugin->init_argc,pplugin->init_argv);
    if (retval != LCMAPS_MOD_SUCCESS )
    {
        lcmaps_log_debug(1,"lcmaps.mod-PluginInit(): \"plugin_initialize()\" failed\n");
        goto fail_PluginInit;
    }
    else
    {
        lcmaps_log_debug(4,"lcmaps.mod-PluginInit(): \"plugin_initialize()\" succeeded\n");
    }

    /* Run plugin_introspect() */
    retval=(*(pplugin->procs[INTROPROC]))(&(pplugin->run_argc),&(pplugin->run_argv));
    if (retval != LCMAPS_MOD_SUCCESS )
    {
        lcmaps_log(LOG_ERR,"lcmaps.mod-PluginInit(): \"plugin_introspect()\" failed\n");
        goto fail_PluginInit;
    }
    else
    {
        lcmaps_log_debug(4,"lcmaps.mod-PluginInit(): \"plugin_introspect()\" succeeded\n");
    }

    /* free stuff */
    if (pname) { free(pname); pname=NULL; }

    return pplugin;

 fail_PluginInit:
    if (pname) { free(pname); pname=NULL; }
    return NULL;
}

/******************************************************************************
Function:   parse_args_plugin()
Description:
    Parse the argument string of the plugin and create xargv and xargc

Parameters:
    name: name of the plugin (goes into argv[0])
    argstring: string containing the arguments
    xargv: array of argument strings (has to be freed later)
    xargc: number of arguments
Returns:
    0 on succes
    1 on failure
******************************************************************************/
/*!
    \fn parse_args_plugin(
        const char * name,
        const char * argstring,
        char ** xargv,
        int * xargc
        )
    \brief convert plugin argument string into xargc, xargv

    Parse the argument string of the plugin and create xargv and xargc
    \param name      name of the plugin (goes into xargv[0])
    \param argstring string containing the arguments
    \param xargv     array of argument strings (has to be freed later)
    \param xargc     number of arguments
    \retval 0 succes.
    \retval 1 failure.
    \internal
*/
static int parse_args_plugin(
        const char * name,
        const char * argstring,
        char ** xargv,
        int * xargc
)
{
    int len=0;
    int i;
    int rc;

    /* set modulename */
    len=strlen(name);
    if ( (len > 0) && (len < LCMAPS_MAXPATHLEN) )
    {
        xargv[0]=(char *)malloc(len+1);
        if (xargv[0] != NULL)
            strncpy(xargv[0],name,len+1);
        else
            return 1;
    }

    *xargc=LCMAPS_MAXARGS-1;
    if ( ( rc = lcmaps_tokenize(argstring, &xargv[1], xargc, " \t\n") ) )
    {
        lcmaps_log(LOG_ERR,"lcmaps.mod-parse_args_plugin(): something wrong parsing arguments of %s, rc=%d\n", name, rc);
        (*xargc)++;
        return 1;
    }
    (*xargc)++; /* increase by one for xargv[0] */
    lcmaps_log_debug(4,"lcmaps.mod-parse_args_plugin(): Found %d arguments:\n", *xargc);
    for (i=0; i < *xargc; i++)
        lcmaps_log_debug(4,"lcmaps.mod-parse_args_plugin(): %d --> %s\n", i, xargv[i]);

    return 0;
}

/******************************************************************************
Function:   get_procsymbol
Description:
    get procedure symbol from dlopen-ed library

Parameters:
    handle:  handle of dynamic library
    symname: name of procedure symbol
Returns:
    handle to procedure symbol
    NULL: Could not get symbol
******************************************************************************/
/*!
    \fn get_procsymbol(
        void * handle,
        char * symname
        )
    \brief get procedure symbol from dlopen-ed library
    \param handle  handle of dynamic library
    \param symname name of procedure symbol
    \return handle to procedure symbol or NUll
    \internal
*/
static lcmaps_proc_t get_procsymbol(
        void * handle,
        char * symname
)
{
    lcmaps_proc_t symhandle;
    char *error;

    symhandle=dlsym(handle,symname);
    if ((error = dlerror()) != NULL)
    {
        lcmaps_log(LOG_ERR,"lcmaps.mod-get_procsymbol(): dlsym error: %s\n",error);
        return NULL;
    }
    return symhandle;
}

/******************************************************************************
Function:   print_lcmaps_plugin
Description:
    print the lcmaps_plugindl_t structure

Parameters:
    debug_lvl: debugging level
    plugin:    plugin structure
Returns:
    1: succes
    0: failure
******************************************************************************/
/*!
    \fn print_lcmaps_plugin(
        int debug_lvl,
        lcmaps_plugindl_t * plugin
        )
    \brief print the lcmaps_plugindl_t structure
    \param debug_lvl debugging level
    \param plugin    plugin structure
    \retval 0 succes.
    \retval 1 failure.
    \internal
*/
static int print_lcmaps_plugin(
        int debug_lvl,
        lcmaps_plugindl_t * plugin
)
{
    char ** pstring = NULL;
    char *  argType = NULL;
    int  *  pint    = NULL;
    int     i       = 0;

#ifdef LCMAPS_DEBUG
    lcmaps_log_debug(debug_lvl,"plugin name                         : %s\n",plugin->pluginabsname);
    lcmaps_log_debug(debug_lvl,"plugin arguments                    : %s\n",plugin->pluginargs);
    lcmaps_log_debug(debug_lvl,"plugin address                      : 0x%x\n",plugin);
    lcmaps_log_debug(debug_lvl,"plugin size                         : %d\n",sizeof(lcmaps_plugindl_t));
    lcmaps_log_debug(debug_lvl,"plugin handle                       : 0x%x\n",plugin->handle);
    lcmaps_log_debug(debug_lvl,"plugin_initialize()                 : 0x%x\n",plugin->procs[INITPROC]);
    lcmaps_log_debug(debug_lvl,"plugin_run()                        : 0x%x\n",plugin->procs[RUNPROC]);
    lcmaps_log_debug(debug_lvl,"plugin_verify()                     : 0x%x\n",plugin->procs[VERIFYPROC]);
    lcmaps_log_debug(debug_lvl,"plugin_terminate()                  : 0x%x\n",plugin->procs[TERMPROC]);
    lcmaps_log_debug(debug_lvl,"plugin_introspect()                 : 0x%x\n",plugin->procs[INTROPROC]);
    lcmaps_log_debug(debug_lvl,"plugin init_argc                    : %d\n",plugin->init_argc);
    for (i=0; i < plugin->init_argc; i++)
        lcmaps_log_debug(debug_lvl,"plugin init_argv[%2d]                : %s\n",i,(plugin->init_argv)[i]);

    lcmaps_log_debug(debug_lvl,"plugin run_argc                     : %d\n",plugin->run_argc);
#endif
    for (i=0; i < plugin->run_argc; i++)
    {
        argType=((plugin->run_argv)[i]).argType;
#ifdef LCMAPS_DEBUG
        lcmaps_log_debug(debug_lvl,"plugin run_argv[%2d]                 : 0x%x\n",i,(plugin->run_argv)+i);
        lcmaps_log_debug(debug_lvl,"plugin run_argv[%2d], argName        : %s\n",i,((plugin->run_argv)[i]).argName);
        lcmaps_log_debug(debug_lvl,"plugin run_argv[%2d], argType        : %s\n",i,((plugin->run_argv)[i]).argType);
        lcmaps_log_debug(debug_lvl,"plugin run_argv[%2d], argInOut       : %d\n",i,((plugin->run_argv)[i]).argInOut);
        lcmaps_log_debug(debug_lvl,"plugin run_argv[%2d], value address  : 0x%x\n",i,((plugin->run_argv)[i]).value);
#endif
        if (strcmp("char *",argType) == 0)
        {
            if ( ((plugin->run_argv)[i]).value != NULL)
            {
                /* pstring = (char **) (((plugin->run_argv)[i]).value); */
                pstring = (char **) lcmaps_getArgValue(
                        ((plugin->run_argv)[i]).argName,
                        ((plugin->run_argv)[i]).argType,
                        plugin->run_argc,
                        plugin->run_argv
                );
#ifdef LCMAPS_DEBUG
                lcmaps_log_debug(debug_lvl,"plugin run_argv[%2d], char * value   : %s\n",i,*pstring);
#endif
            }
#ifdef LCMAPS_DEBUG
            else
            {
                lcmaps_log_debug(debug_lvl,"plugin run_argv[%2d], char * value   : none yet\n",i);
            }
#endif
        }
        if (strcmp("int",argType) == 0)
        {
            if ( ((plugin->run_argv)[i]).value != NULL)
            {
                /* pint = (int *) (((plugin->run_argv)[i]).value); */
                pint = (int *) lcmaps_getArgValue(
                        ((plugin->run_argv)[i]).argName,
                        ((plugin->run_argv)[i]).argType,
                        plugin->run_argc,
                        plugin->run_argv
                );
#ifdef LCMAPS_DEBUG
                lcmaps_log_debug(debug_lvl,"plugin run_argv[%2d], int value      : %d\n",i,*pint);
#endif
            }
#ifdef LCMAPS_DEBUG
            else
            {
                lcmaps_log_debug(debug_lvl,"plugin run_argv[%2d], int value      : none yet\n",i);
            }
#endif
        }
    }
#ifdef LCMAPS_DEBUG
    lcmaps_log_debug(debug_lvl,"plugin next                         : 0x%x\n",plugin->next);
    if (plugin->next != NULL)
        lcmaps_log_debug(debug_lvl,"plugin_next                         : %s\n",(plugin->next)->pluginabsname);
    else
        lcmaps_log_debug(debug_lvl,"plugin_next                         : last plugin in list\n");
#endif

    return 1;
}

/******************************************************************************
Function:   clean_plugin_list
Description:
    clean (free) the list of plugins and call the plugin termination functions

Parameters:
    list:      pointer to list of plugins which has to be freeed.

Returns:
    1: succes
    0: failure
******************************************************************************/
/*!
    \fn clean_plugin_list(
        lcmaps_plugindl_t ** list
        )
    \brief clean (free) the list of plugins and call the plugin termination functions
    \param list
    \param list      pointer to list of plugins which has to be freeed.
    \retval 0 succes.
    \retval 1 failure.
    \internal
*/
static int clean_plugin_list(
        lcmaps_plugindl_t ** list
)
{
    int                           rc;
    lcmaps_plugindl_t *             plugin_entry=NULL;

    plugin_entry=*list;
    while (plugin_entry)
    {
        lcmaps_plugindl_t * plugin_next;
        int               i;

        rc = plugin_entry->procs[TERMPROC]();
        if (rc != LCMAPS_MOD_SUCCESS)
        {
            lcmaps_log(LOG_WARNING,"lcmaps.mod-clean_plugin_list(): failed to terminate plugin module %s\n", plugin_entry->pluginabsname);
        }
        lcmaps_log_debug(4, "lcmaps.mod-clean_plugin_list(): plugin module %s terminated\n",
                plugin_entry->pluginabsname);
        dlclose(plugin_entry->handle);

        plugin_next=plugin_entry->next;
        for (i=0; i < plugin_entry->init_argc; i++)
        {
            if ((plugin_entry->init_argv)[i] != NULL)
            {
#ifdef LCMAPS_DEBUG
                lcmaps_log_debug(5,"Freeing %d - %s\n",i,(plugin_entry->init_argv)[i]);
#endif
                free((plugin_entry->init_argv)[i]);
            }
        }
        free(plugin_entry);
        plugin_entry=plugin_next;
    }
    *list=plugin_entry=NULL;
    return 0;
}

/******************************************************************************
Function:   lcmaps_runPluginManager
Description:
    Run PluginManager:
    Contact Evaluation Manager with list of possible policies --> runs plugins
    The plugin manager can be run in verification mode or normal mode, depending on
    "verification switch".

Parameters:
    request     : RSL request
    lcmaps_cred : user credential
    npols       : number of policies to be considered for evaluation
    policynames : the names of the policies to be considered for evaluation
    mode        : Mode of running lcmaps: LCMAPS_NORMAL_MODE or
                  LCMAPS_VERIFICATION_MODE

Returns:
    0: user mapping or mapping verification succeeded
    1: user mapping or mapping verification failed
******************************************************************************/
/*!
    \fn lcmaps_runPluginManager(
        lcmaps_request_t request,
        lcmaps_cred_id_t lcmaps_cred,
        int npols,
        char ** policynames,
        unsigned int mode
        )
    \brief Run the PluginManager

    This function runs the PluginManager for user mapping.
    Contact Evaluation Manager --> runs plugins

    \param request      RSL request (job request)
    \param lcmaps_cred  user credential
    \param npols        number of policies to be considered for evaluation
    \param policynames  the names of the policies to be considered for evaluation
    \param mode         Mode of running lcmaps: LCMAPS_NORMAL_MODE or
                        LCMAPS_VERIFICATION_MODE

    \retval 0 user mapping or mapping verification succeeded
    \retval 1 user mapping or mapping verification failed
*/
int lcmaps_runPluginManager(
        lcmaps_request_t request,
        lcmaps_cred_id_t lcmaps_cred,
        int npols,
        char ** policynames,
        unsigned int mode
)
{
    lcmaps_plugindl_t * plugin_entry=NULL;

    /* Set lcmaps mode */
    lcmaps_mode = mode;
    if (lcmaps_mode == LCMAPS_NORMAL_MODE)
    {
        lcmaps_log_debug(4,"lcmaps.mod-lcmaps_runPluginManager(): Running in normal mode\n");
    }
    else if (lcmaps_mode == LCMAPS_VERIFICATION_MODE)
    {
        lcmaps_log_debug(4,"lcmaps.mod-lcmaps_runPluginManager(): Running in verification mode\n");
    }
    else
    {
        lcmaps_log(LOG_ERR, "lcmaps.mod-lcmaps_runPluginManager(): Attempt to run in illegal mode: %d (failure)\n", lcmaps_mode);
        return 1;
    }

    lcmaps_log_debug(5,"lcmaps.mod-lcmaps_runPluginManager(): extract RunVars\n");
    if (lcmaps_extractRunVars(request, lcmaps_cred) != 0)
    {
        lcmaps_log(LOG_ERR,"lcmaps.mod-lcmaps_runPluginManager(): error in creating list of run variables\n");
        return 1;
    }

    plugin_entry=plugin_list;
    while (plugin_entry)
    {
        void * pvalue=NULL;
        char * argName=NULL;
        char * argType=NULL;
        int iarg=0;

        /*
         * Now that we know that we're running in verification mode or not,
         * check for the plugin_verify() method
         */
        if ((lcmaps_mode == LCMAPS_VERIFICATION_MODE) && (plugin_entry->procs[VERIFYPROC] == NULL))
        {
            lcmaps_log(LOG_ERR, "lcmaps.mod-lcmaps_runPluginManager(): Cannot find required \"plugin_verify()\" method in plugin \"%s\" (failure)\n", plugin_entry->pluginabsname);
            return 1;
        }
        for (iarg=0; iarg < plugin_entry->run_argc; iarg++)
        {
            argName=(plugin_entry->run_argv+iarg)->argName;
            argType=(plugin_entry->run_argv+iarg)->argType;
#ifdef LCMAPS_DEBUG
            lcmaps_log_debug(5,
                "lcmaps.mod-lcmaps_runPluginManager(): Getting value of \"%s\" of type \"%s\" for plugin \"%s\"\n",
                argName, argType, plugin_entry->pluginabsname);
#endif
            if ((pvalue=lcmaps_getRunVars(argName, argType)) == NULL)
            {
                lcmaps_log(LOG_ERR, "lcmaps.mod-lcmaps_runPluginManager(): could not GET requested variable \"%s\" of type \"%s\" for plugin \"%s\"\n", argName, argType, plugin_entry->pluginabsname);
                return 1;
            }
#ifdef LCMAPS_DEBUG
            lcmaps_log_debug(5, "lcmaps.mod-lcmaps_runPluginManager(): Setting value of \"%s\" of type \"%s\" for plugin \"%s\", address: 0x%x\n", argName, argType, plugin_entry->pluginabsname, pvalue);
#endif
            if (lcmaps_setArgValue(argName, argType, pvalue, plugin_entry->run_argc, &(plugin_entry->run_argv)) != 0)
            {
                lcmaps_log(LOG_ERR, "lcmaps.mod-lcmaps_runPluginManager(): could not SET requested variable \"%s\" of type \"%s\" for plugin \"%s\"\n", argName, argType, plugin_entry->pluginabsname);
                return 1;
            }
        }
        plugin_entry=plugin_entry->next;
    }

#if USE_EVALUATIONMANAGER
    if (npols > 0)
    {
        int ipol = 0;

        lcmaps_log_debug(4,"lcmaps.mod-lcmaps_runPluginManager(): Do lcmaps_runEvaluationManager with the following policies:\n");
        for (ipol = 0; ipol < npols; ipol ++)
            lcmaps_log_debug(1,"lcmaps.mod-lcmaps_runPluginManager():     %s\n", policynames[ipol]);
    }
    else
        lcmaps_log_debug(4,"lcmaps.mod-lcmaps_runPluginManager(): Do lcmaps_runEvaluationManager()\n");

    if (lcmaps_runEvaluationManager(npols, policynames) != 0)
    {
        lcmaps_log_debug(1,"lcmaps.mod-lcmaps_runPluginManager(): Error running evaluation manager\n");

        /* Print Post Run LCMAPS credentials */
        lcmaps_printCredData(1);

        /* Failure */
        return 1;
    }
    lcmaps_log_debug(3,"lcmaps.mod-lcmaps_runPluginManager(): lcmaps_runEvaluationManager() succeeded.\n");
#else
    /* The following will be taken over by the evaluation manager
     * It will call lcmaps_runPluginManager() directly and evaluate its return values
     */
    plugin_entry=plugin_list;
    while (plugin_entry)
    {
        print_lcmaps_plugin(4,plugin_entry);
        lcmaps_log_debug(3,"lcmaps.mod-lcmaps_runPluginManager(): running %s\n",plugin_entry->pluginabsname);
/*        if (lcmaps_runPlugin(plugin_entry->pluginabsname)) {*/
        if (lcmaps_runPlugin(plugin_entry->pluginshortname)) {
            lcmaps_log_debug(3,"lcmaps.mod-lcmaps_runPluginManager(): plugin %s failed\n",plugin_entry->pluginabsname);
        }
        plugin_entry=plugin_entry->next;
    }
#endif

    /* Print Post Run LCMAPS credentials */
    lcmaps_printCredData(1);

    /* Success */
    return 0;
}

/******************************************************************************
Function:   lcmaps_resetCredentialData
Description:
    Reset the Credential data.
    Wrapper around lcmaps_cleanCredentialData() to be used
    by the Evaluation Manager, before a new policy is
    evaluated

Parameters:
    none
Returns:
    the return value of lcmaps_cleanCredentialData()
******************************************************************************/
/*!
    \fn lcmaps_resetCredentialData()
    \brief Reset the Credential data

    Wrapper around lcmaps_cleanCredentialData() to be used
    by the Evaluation Manager, before a new policy is
    evaluated

    \return the return value of lcmaps_cleanCredentialData()
*/
int lcmaps_resetCredentialData()
{
    lcmaps_log_debug(3,"lcmaps.mod-lcmaps_resetCredentialData(): Called\n");
    return lcmaps_cleanCredentialData();
}

/******************************************************************************
Function:   runPlugin
Description:
    Run a plugin for the Evaluation Manager
    the result (succes or not will be passed to the Evaluation Manager)

Parameters:
    pluginname: the name of the plugin module
Returns:
    0: plugin run succeeded
    1: plugin run failed
******************************************************************************/
/*!
    \fn lcmaps_runPlugin(
        const char * pluginname
        )
    \brief Run a plugin

    Run a plugin for the Evaluation Manager
    the result (succes or not will be passed to the Evaluation Manager)

    \param pluginname the name of the plugin module

    \retval 0 plugin run succeeded
    \retval 1 plugin run failed
*/
int lcmaps_runPlugin(
        const char * pluginname
)
{
    lcmaps_plugindl_t * plugin_entry=NULL;
    int rc=1;
    int foundplugin=0;

    plugin_entry=plugin_list;
    while (plugin_entry)
    {
#ifdef LCMAPS_DEBUG
        lcmaps_log_debug(5,"lcmaps.mod-lcmaps_runPlugin(): looking for plugin %s\n",pluginname);
#endif
        /* Look for shortname since that's how they're in the policy! */
        if (strcmp(plugin_entry->pluginshortname,pluginname) == 0) {
            lcmaps_log_debug (5,"lcmaps.mod-lcmaps_runPlugin(): found plugin %s\n",plugin_entry->pluginabsname);
            foundplugin++;
            break;
        }
        plugin_entry=plugin_entry->next;
    }
    if (!foundplugin) {
        lcmaps_log(LOG_ERR,"lcmaps.mod-lcmaps_runPlugin(): could not find plugin %s\n",pluginname);
        return 1;
    }

    /* Actually run the plugin */
    lcmaps_log_debug(3, "lcmaps.mod-lcmaps_runPlugin(): running plugin %s\n",plugin_entry->pluginabsname);
    if (lcmaps_mode == LCMAPS_VERIFICATION_MODE)
    {
        rc = plugin_entry->procs[VERIFYPROC](plugin_entry->run_argc, plugin_entry->run_argv);
    }
    else /* LCMAPS_NORMAL_MODE */
    {
        rc = plugin_entry->procs[RUNPROC](plugin_entry->run_argc, plugin_entry->run_argv);
    }

    return rc;
}

/******************************************************************************
Function:   lcmaps_stopPluginManager
Description:
    Terminate PluginManager module.

Parameters:
Returns:
    0: termination succeeded
    1: termination failed
******************************************************************************/
/*!
    \fn lcmaps_stopPluginManager(
        )
    \brief Terminate the PluginManager module.

    stop the PluginManager --> terminate plugins, clean plugin list, (stop evaluation manager)

    \retval 0 succes
    \retval 1 failure
*/
int lcmaps_stopPluginManager(
)
{
    lcmaps_log_debug(3,"lcmaps.mod-lcmaps_stopPluginManager(): cleaning credential data\n");

    /* If I don't do lcmaps_term(), this will never be triggered */
    /* lcmaps_printCredData(1); */
    if (lcmaps_cleanCredentialData()!=0)
    {
        lcmaps_log(LOG_ERR,"lcmaps.mod-lcmaps_stopPluginManager() error: could not clean credential data list\n");

        /* Resetting the LCMAPS SIGPIPE handler - old SIGPIPE handler will be preserved */
        lcmaps_reset_sigpipe_handler();

        return 1;
    }
    if (clean_plugin_list(&plugin_list)!=0)
    {
        lcmaps_log(LOG_ERR,"lcmaps.mod-lcmaps_stopPluginManager() error: could not clean up plugin list\n");

        /* Resetting the LCMAPS SIGPIPE handler - old SIGPIPE handler will be preserved */
        lcmaps_reset_sigpipe_handler();

        return 1;
    }
#if USE_EVALUATIONMANAGER
    if (lcmaps_stopEvaluationManager() != 0)
    {
        lcmaps_log(LOG_ERR,"lcmaps.mod-lcmaps_stopPluginManager(): error in lcmaps_stopEvaluationManager()\n");

        /* Resetting the LCMAPS SIGPIPE handler - old SIGPIPE handler will be preserved */
        lcmaps_reset_sigpipe_handler();

        return 1;
    }
#endif

    /* Resetting the LCMAPS SIGPIPE handler - old SIGPIPE handler will be preserved */
    lcmaps_reset_sigpipe_handler();

    return 0;
}


/******************************************************************************
Function:   lcmaps_set_sigpipe_handler
Description:
    Sets a new SIGPIPE signal handler, the old signal handler will be kept
    and nicely reset to the previous setting.

Parameters:
Returns:
    0: termination succeeded
    1: termination failed
******************************************************************************/
/*!
    \fn lcmaps_set_sigpipe_handler(
        )
    \brief Sets a new SIGPIPE handler for LCMAPS.

    Sets a new SIGPIPE signal handler, the old signal handler will be kept
    and nicely reset to the previous setting.

    \retval 0 succes
    \retval 1 failure
*/
int lcmaps_set_sigpipe_handler()
{
    /* Setting the SIGPIPE handler to the LCMAPS SIGPIPE handler */
    lcmaps_stored_sigpipe_handler = signal(SIGPIPE, lcmaps_sigpipe_handler);

    return 0;
}


/******************************************************************************
Function:   lcmaps_reset_sigpipe_handler
Description:
    Resets the old SIGPIPE signal handler (if there was any).

Parameters:
Returns:
    0: termination succeeded
    1: termination failed
******************************************************************************/
/*!
    \fn lcmaps_reset_sigpipe_handler(
        )
    \brief Resets the previous SIGPIPE handler for LCMAPS.

    Resets the old SIGPIPE signal handler (if there was any).

    \retval 0 succes
    \retval 1 failure
*/
int lcmaps_reset_sigpipe_handler()
{
    /* Resetting the SIGPIPE handler to the previous handler */
    signal(SIGPIPE, lcmaps_stored_sigpipe_handler);

    return 0;
}


/******************************************************************************
Function:   lcmaps_sigpipe_handler
Description:
    The SIGPIPE handler.
    Will log to the lcmaps log facility about the SIGPIPE, when caught.

Parameters:
    int sig : The caught signal, see signal.h for details.
Returns:
    void
******************************************************************************/
/*!
    \fn lcmaps_reset_sigpipe_handler(
        )
    \brief The SIGPIPE handler.

*/
void lcmaps_sigpipe_handler(int sig)
{
    if (sig == SIGPIPE)
    {
        lcmaps_log(LOG_NOTICE, "Caught signal SIGPIPE\n");
    }
}

#endif /* LCMAPS_PLUGINMANAGER_C */
