/*
 *
 * This file is part of lcrash, an analysis tool for Linux memory dumps.
 *
 * Created by Silicon Graphics, Inc.
 * Contributions by IBM, and others
 *
 * Copyright (C) 1999 - 2002 Silicon Graphics, Inc. All rights reserved.
 * Copyright (C) 2001, 2002 IBM Deutschland Entwicklung GmbH, IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 *
 * Added support for X86_64 architecture Mar 2004
 *      Prashanth Tamraparni (prasht@in.ibm.com)
 *      Sachin Sant (sachinp@in.ibm.com)
 */

#include <lcrash.h>

extern int defcpu_x86_64;

#define _RD_USAGE "[-w outfile]"

void
rd_x86_64_usage(command_t *cmd)
{
	CMD_USAGE(cmd, _RD_USAGE);
}

void
rd_x86_64_help(command_t *cmd)
{
	CMD_HELP(cmd, _RD_USAGE,
			"Display the register contents of the default cpu."
			"This command can't be used on a live system ");
}

int
rd_x86_64_parse(command_t *cmd)
{
	if (set_cmd_flags(cmd, (C_WRITE), 0)) {
		return(1);
	}
	return 0;
}

int
rd_x86_64_cmd(command_t *cmd)
{
	kl_dump_header_x86_64_t dha;
	unsigned seg;

	struct kl_pt_regs_x86_64 *regs = &dha.smp_regs[defcpu_x86_64];

	if (cmd->nargs != 0) {
		rd_x86_64_usage(cmd);
		return(1);
	}	

	if (CORE_IS_KMEM) {
		fprintf(cmd->efp, "Can't use this command on live system\n");
		return(1);
	}
	
	if (kl_get_dump_header_x86_64(&dha))
		return(1);

	if (defcpu_x86_64 == -1)
		defcpu_x86_64 = KL_GET_UINT32(&dha.dumping_cpu);
	
		
	fprintf (cmd->ofp, "CPU: %d \nEIP <%04"FMTPTR"x>:<%016"FMTPTR"x>\n", 
		 defcpu_x86_64, regs->xcs & 0xffff, regs->rip);
	
	if (!regs->rip) {
		fprintf(cmd->efp, "RIP is NULL\n");
		fprintf(cmd->efp, "One probable cause:  pt_regs is NULL when Alt-SysRq-D is used to take the dump\n");
	}
	fprintf(cmd->ofp, "EFLAGS: %08"FMTPTR"x\n", regs->eflags);

	fprintf(cmd->ofp, "R15: %016"FMTPTR"x  R14: %016"FMTPTR"x\n", regs->r15, regs->r14);
	fprintf(cmd->ofp, "R13: %016"FMTPTR"x  R12: %016"FMTPTR"x\n", regs->r13, regs->r12);
	fprintf(cmd->ofp, "R11: %016"FMTPTR"x  R10: %016"FMTPTR"x\n", regs->r11, regs->r10);
	fprintf(cmd->ofp, "R9: %016"FMTPTR"x  R8: %016"FMTPTR"x\n", regs->r9, regs->r8);
	fprintf(cmd->ofp, "RAX: %016"FMTPTR"x  RBX: %016"FMTPTR"x\n", regs->rax, regs->rbx);
	fprintf(cmd->ofp, "RCX: %016"FMTPTR"x  RDX: %016"FMTPTR"x\n", regs->rcx, regs->rdx);
	fprintf(cmd->ofp, "RSI: %016"FMTPTR"x  RDI: %016"FMTPTR"x\n", regs->rsi, regs->rdi);
	fprintf(cmd->ofp, "RBP: %016"FMTPTR"x  RSP: %016"FMTPTR"x\n", regs->rbp, regs->rsp);
	return(0);
}
