/*
 * $Id: cmd_symtab.c,v 1.1 2004/12/21 23:26:19 tjm Exp $
 *
 * This file is part of lcrash, an analysis tool for Linux memory dumps.
 *
 * Created by Silicon Graphics, Inc.
 * Contributions by IBM, and others
 *
 * Copyright (C) 1999 - 2002 Silicon Graphics, Inc. All rights reserved.
 * Copyright (C) 2001, 2002 IBM Deutschland Entwicklung GmbH, IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */

#include <lcrash.h>

#define C_AFLAG (0x01 << C_LFLG_SHFT)
#define C_DFLAG (0x02 << C_LFLG_SHFT)
#define C_FFLAG (0x04 << C_LFLG_SHFT)
#define C_LFLAG (0x08 << C_LFLG_SHFT)
#define C_MFLAG (0x10 << C_LFLG_SHFT)
#define C_RFLAG (0x20 << C_LFLG_SHFT)
#define C_SFLAG (0x40 << C_LFLG_SHFT)
#define C_TFLAG (0x80 << C_LFLG_SHFT)

#define _KSYMTAB        "ksyms"
#define _ALLMODS        "all_modules"
#define _SYMBYNAME      "name"
#define _SYMBYADDR      "address"

static char *filename;
static char *modname;
static char *moddir;
static char *sortby;

/* Forward declaration
 */
void symtab_usage(command_t *);

int
symtab_cmd(command_t *cmd)
{
			
	int rc = 0;
	int validopt = 0;

	if((cmd->flags & C_AFLAG) &&
	   !(cmd->flags & (C_RFLAG|C_LFLAG|C_TFLAG|C_SFLAG))){
		if((cmd->nargs == 0) && (cmd->flags & C_MFLAG)){
			if(KL_LINUX_RELEASE >= LINUX_2_6_0){
				fprintf(cmd->ofp, "Adding module maps is not supported for this kernel version\n");
				rc = 1;
				goto out;
			}
				
			/* add symbol table for one module */
			validopt = 1;
			fprintf(cmd->ofp, "Loading symbol table ...\n");
			if((rc = kl_load_module_sym(modname, filename,
							moddir)) == 1){
				if(KL_ERROR == KLE_MAP_FILE_PRESENT){
					fprintf(cmd->ofp, "Warning: Symbol "
						"table already loaded\n");
				} else if(KL_ERROR == KLE_ARCHIVE_FILE){
					kl_error("Error: Tried to read symbol "
						 "table from archive file\n");
				} else {
					kl_error("Error: Could not load "
						 "symbol table\n");
			}
			} else {
				fprintf(cmd->ofp, "Done\n");
			}
		} else if((cmd->nargs == 0) && (cmd->flags & C_FFLAG) &&
			  !(cmd->flags & (C_MFLAG|C_DFLAG))){
			/* add "absolute" symbol table */
			validopt = 1;
			fprintf(cmd->ofp, "Loading symbol table ...\n");
			if(kl_load_sym(filename)){
				if(KL_ERROR == KLE_MAP_FILE_PRESENT){
					fprintf(cmd->ofp, "Warning: Symbol "
						"table already loaded\n");
				} else if(KL_ERROR == KLE_ARCHIVE_FILE){
					kl_error("Error: Tried to read symbol "
						 "table from archive file\n");
				} else {
					kl_error("Error: Could not load "
						 "symbol table\n");
			}
			} else {
				fprintf(cmd->ofp, "Done\n");
			}
		} else if((cmd->nargs == 1) &&
			  !(cmd->flags & (C_FFLAG|C_MFLAG))) {
			/* add ksyms */
			if((!strcmp(cmd->args[0], _KSYMTAB)) &&
			   !(cmd->flags & C_DFLAG)){
				validopt = 1;
				fprintf(cmd->ofp, "Loading ksyms"
					" from dump ...");
				if(KL_LINUX_RELEASE >= LINUX_2_6_0)
					rc = kl_load_ksyms_2_6(0);
				else
					rc = kl_load_ksyms(0);

				if(rc == 1){
					kl_error("\nError: Could not load "
						 "symbol table for ksyms\n");
				} else if(rc == 2) {
					fprintf(cmd->ofp, "\nWarning: Symbol "
						"table already loaded\n");
				} else {
					fprintf(cmd->ofp, "\nDone\n");
					}
			} else if(!strcmp(cmd->args[0], _ALLMODS)){
				if(KL_LINUX_RELEASE >= LINUX_2_6_0){
					fprintf(cmd->ofp, "Adding module maps is not supported for this kernel version\n");
					rc = 1;
					goto out;
				}
				/* add symbol tables for all modules */
				/* C_DFLAGS could be set */
				validopt = 1;
				fprintf(cmd->ofp, "Load symbol tables for "
					"all modules ...\n");
				if((kl_autoload_module_info(moddir)) &&
				   (KL_ERROR != KLE_MAP_FILE_PRESENT)){
					kl_error("Error: Could not load "
						 "symbol tables for all "
						 "modules\n");
				} else {
					fprintf(cmd->ofp, "Done\n");
				}
			}
		}
	} else if((cmd->flags & C_RFLAG) &&
		  !(cmd->flags & (C_LFLAG|C_TFLAG|C_SFLAG|C_DFLAG))) {
		if((cmd->nargs == 0) && (cmd->flags & C_MFLAG)){
			validopt = 1;
			if((rc = kl_unload_module_sym(modname))){
				fprintf(cmd->ofp, "Warning: No symbol table "
					"for module %s\n", modname);
			} else {
				fprintf(cmd->ofp, "Removed symbol table for "
					"module %s\n", modname);
		}
		} else if((cmd->nargs == 0) && (cmd->flags & C_FFLAG) &&
			  !(cmd->flags & C_MFLAG)){
			validopt = 1;
			if((rc = kl_free_syminfo(filename))){
				fprintf(cmd->ofp, "Warning: No symbol table "
					"for file %s\n", filename);
			} else {
				fprintf(cmd->ofp, "Removed symbol table(s) for"
					" file %s\n", filename);
			}
		} else 	if((cmd->nargs == 1) && 
			   !(cmd->flags & (C_MFLAG|C_FFLAG))){
			if(!strcmp(cmd->args[0], _KSYMTAB)) {
				validopt = 1;
				if((rc = kl_unload_ksyms())) {
					fprintf(cmd->ofp, "Warning: No symbol"
					" table for ksyms\n");
				} else {
					fprintf(cmd->ofp, "Removed symbol "
						"table for ksyms\n");
		}
			} else if(!strcmp(cmd->args[0], _ALLMODS)){
				/* remove symbol tables for all modules */
				validopt = 1;
				if((rc = kl_unload_module_sym(NULL))){
					fprintf(cmd->ofp, "Warning: No symbol"
					" table for any module\n");
		} else {
					fprintf(cmd->ofp, "Removed symbol "
						"tables for all modules\n");
				}
			}
		}
	} else if((cmd->flags & C_LFLAG) &&
		  !(cmd->flags & (C_DFLAG))){
		int symflags=0;
		if(cmd->flags & C_TFLAG){
			symflags |= KL_SYMFULL;
		}
		if(cmd->flags & C_SFLAG){
			if(!strcmp(_SYMBYNAME, sortby)){
				symflags |= KL_SYMBYNAME;
			} /* default sorted by address */
			}
		if(cmd->nargs == 0) {
			validopt = 1;
			if((rc=kl_print_symtables(modname, filename,
						  SYM_MAP_ANY, symflags))){
				kl_error("No such symbol table\n");
			}
		} else if((cmd->nargs == 1) &&
			  !(cmd->flags & (C_MFLAG|C_FFLAG))){
			if(!strcmp(cmd->args[0], _KSYMTAB)) {
				validopt = 1;
				if((rc=kl_print_symtables(modname, filename,
							  SYM_MAP_KSYM,
							  symflags))){
					kl_error("No such symbol table\n");
			}
			} else if(!strcmp(cmd->args[0], _ALLMODS)){
				/* list symbol tables for all modules */
				validopt = 1;
				if((rc=kl_print_symtables(modname, filename,
							  SYM_MAP_MODULE,
							  symflags))){
					kl_error("No such symbol table\n");
				}
			}
		}
	} 

	if(!validopt){
		kl_error("Invalid command line.\n");
		symtab_usage(cmd);
		return(1);
	}
out:
	return(rc);
}

#define _SYMTAB_USAGE \
"\n        -l | -a | -r [ksyms | all_modules] [-f filename] [-m modname]"\
"\n        [-d moddir] [-t [-s name | address]] [-w outfile]"

#define _SYMTAB_HELP \
"Add, list or remove symbol tables.\n"\
"\nOPTIONS:"\
"\n -l    Display loaded symbol tables."\
"\n -a    Add symbol table(s). One of or a combination of 'ksyms',"\
"\n       'all_modules', '-f filename', '-m modname' must be specified."\
"\n -r    Remove symbol table(s). One of or a combination of 'ksyms',"\
"\n       'all_modules', '-f filename', '-m modname' must be specified."\
"\n ksyms"\
"\n       Add, display or remove symbol table of ksyms."\
"\n all_modules"\
"\n       Automatically add symbol tables for all kernel modules"\
"\n       in the dump."\
"\n       List or remove symbol tables which belong to a kernel module."\
"\n -f filename"\
"\n       Filename of symbol table to be loaded, displayed or removed."\
"\n       This option can be combined with -m"\
"\n -m modname"\
"\n       Name of kernel module in the dump for which a symbol table"\
"\n       should be loaded, displayed or removed."\
"\n -d moddir"\
"\n       Name of directory which should replace the part \"/lib/modules\""\
"\n       when symbol tables are automatically loaded for modules."\
"\n       Can be combined with \"all_modules\" or option \"-m\"."\
"\n -t    Display full table of symbols when \"-l\" is used."\
"\n -s name | address"\
"\n       Specify sorting order when \"-t\" is used. Default is to sort by"\
"\n       symbol addreesses. Use \"-s name\" to sort by symbol name."\

/*
 * symtab_usage() -- Print the usage string for the 'symbol' command.
 */
void
symtab_usage(command_t *cmd)
{
	CMD_USAGE(cmd, _SYMTAB_USAGE);
}

/*
 * symtab_help() -- Print the help information for the 'symbol' command.
 */
void
symtab_help(command_t *cmd)
{
	CMD_HELP(cmd, _SYMTAB_USAGE, _SYMTAB_HELP);
}

/*
 * symtab_parse() -- Parse the command line arguments for 'symbol'.
 */
int
symtab_parse(command_t *cmd)
{
	option_t *op;
	filename = NULL;
	modname = NULL;
	moddir = NULL;
	sortby = NULL;

	if (set_cmd_flags(cmd, (C_WRITE), "ald:f:m:rs:t")) {
		return(1);
	}
	op = cmd->options;
	while (op) {
		switch(op->op_char) {
			case 'a':
				cmd->flags |= C_AFLAG;
				break;
			case 'd':
				cmd->flags |= C_DFLAG;
				moddir = op->op_arg;
				break;
			case 'f':
				cmd->flags |= C_FFLAG;
				filename = op->op_arg;
				break;
			case 'l':
				cmd->flags |= C_LFLAG;
				break;
			case 'm':
				cmd->flags |= C_MFLAG;
				modname = op->op_arg;
				break;
			case 'r':
				cmd->flags |= C_RFLAG;
				break;
			case 's':
				cmd->flags |= C_SFLAG;
				sortby = op->op_arg;
				break;
			case 't':
				cmd->flags |= C_TFLAG;
				break;
		}
		op = op->op_next;
	}
	return(0);
}

/*
 * symtab_complete() -- Complete arguments of 'symtab' command.
 */
char *
symtab_complete(command_t *cmd)
{
	char *ret;

	/* complete standard options (for example, -w option) arguments
	 */
	if ((ret = complete_standard_options(cmd)) != NOT_COMPLETED) {
		return(ret);
	}
	if ((cmd->nargs > 1) &&
	    !strcmp(cmd->args[cmd->nargs - 2],"-f")) {
		return(complete_file_name(cmd->args[cmd->nargs -1], 100));
	} else if((cmd->nargs > 1) &&
	    !strcmp(cmd->args[cmd->nargs - 2],"-d")) {
		return(complete_file_name(cmd->args[cmd->nargs -1], 100));
	} else {
		fprintf(cmd->ofp, "\n");
		symtab_usage(cmd);
		return(DRAW_NEW_ENTIRE_LINE);
	}
}
