//Written in the D programming language

/++
    Module containing Date/Time functionality.

    This module provides:
    $(UL
        $(LI Types to represent points in time: $(D SysTime), $(D Date),
             $(D TimeOfDay), and $(D DateTime).)
        $(LI Types to represent intervals of time.)
        $(LI Types to represent ranges over intervals of time.)
        $(LI Types to represent time zones (used by $(D SysTime)).)
        $(LI A platform-independent, high precision stopwatch type:
             $(D StopWatch))
        $(LI Benchmarking functions.)
        $(LI Various helper functions.)
    )

    Closely related to std.datetime is <a href="core_time.html">$(D core.time)</a>,
    and some of the time types used in std.datetime come from there - such as
    $(CXREF time, Duration), $(CXREF time, TickDuration), and
    $(CXREF time, FracSec).
    core.time is publically imported into std.datetime, it isn't necessary
    to import it separately.

    Three of the main concepts used in this module are time points, time
    durations, and time intervals.

    A time point is a specific point in time. e.g. January 5th, 2010
    or 5:00.

    A time duration is a length of time with units. e.g. 5 days or 231 seconds.

    A time interval indicates a period of time associated with a fixed point in
    time. It is either two time points associated with each other,
    indicating the time starting at the first point up to, but not including,
    the second point - e.g. [January 5th, 2010 - March 10th, 2010$(RPAREN) - or
    it is a time point and a time duration associated with one another. e.g.
    January 5th, 2010 and 5 days, indicating [January 5th, 2010 -
    January 10th, 2010$(RPAREN).

    Various arithmetic operations are supported between time points and
    durations (e.g. the difference between two time points is a time duration),
    and ranges can be gotten from time intervals, so range-based operations may
    be done on a series of time points.

    The types that the typical user is most likely to be interested in are
    $(D Date) (if they want dates but don't care about time), $(D DateTime)
    (if they want dates and times but don't care about time zones), $(D SysTime)
    (if they want the date and time from the OS and/or do care about time
    zones), and StopWatch (a platform-independent, high precision stop watch).
    $(D Date) and $(D DateTime) are optimized for calendar-based operations,
    while $(D SysTime) is designed for dealing with time from the OS. Check out
    their specific documentation for more details.

    To get the current time, use $(D Clock.currTime). It will return the current
    time as a $(D SysTime). To print it, $(D toString) is
    sufficient, but if using $(D toISOString), $(D toISOExtString), or
    $(D toSimpleString), use the corresponding $(D fromISOString),
    $(D fromISOExtString), or $(D fromISOExtString) to create a
    $(D SysTime) from the string.

--------------------
auto currentTime = Clock.currTime();
auto timeString = currentTime.toISOExtString();
auto restoredTime = SysTime.fromISOExtString(timeString);
--------------------

    Various functions take a string (or strings) to represent a unit of time
    (e.g. $(D convert!("days", "hours")(numDays))). The valid strings to use
    with such functions are $(D "years"), $(D "months"), $(D "weeks"),
    $(D "days"), $(D "hours"), $(D "minutes"), $(D "seconds"),
    $(D "msecs") (milliseconds), $(D "usecs") (microseconds),
    $(D "hnsecs") (hecto-nanoseconds - i.e. 100 ns), or some subset thereof.
    There are a few functions in core.time which take $(D "nsecs"), but because
    nothing in std.datetime has precision greater than hnsecs, and very little
    in core.time does, no functions in std.datetime accept $(D "nsecs").
    To remember which units are abbreviated and which aren't,
    all units seconds and greater use their full names, and all
    sub-second units are abbreviated (since they'd be rather long if they
    weren't).

    Note:
        $(D DateTimeException) is an alias for core.time's $(D TimeException),
        so you don't need to worry about core.time functions and std.datetime
        functions throwing different exception types (except in the rare case
        that they throw something other than $(D TimeException) or
        $(D DateTimeException)).

    See_Also:
        <a href="../intro-to-datetime.html">Introduction to std&#46;_datetime </a><br>
        $(WEB en.wikipedia.org/wiki/ISO_8601, ISO 8601)<br>
        $(WEB en.wikipedia.org/wiki/Tz_database,
              Wikipedia entry on TZ Database)<br>
        $(WEB en.wikipedia.org/wiki/List_of_tz_database_time_zones,
              List of Time Zones)<br>

    Copyright: Copyright 2010 - 2011
    License:   $(WEB www.boost.org/LICENSE_1_0.txt, Boost License 1.0).
    Authors:   Jonathan M Davis and Kato Shoichi
    Source:    $(PHOBOSSRC std/_datetime.d)
+/
module std.datetime;

public import core.time;

import core.exception;
import core.stdc.time;

import std.array;
import std.algorithm;
import std.ascii;
import std.conv;
import std.exception;
import std.file;
import std.functional;
import std.math;
import std.metastrings;
import std.path;
import std.range;
import std.stdio;
import std.string;
import std.system;
import std.traits;
import std.typecons;
import std.utf;

version(Windows)
{
    import core.sys.windows.windows;
    import std.c.windows.winsock;
    import std.windows.registry;
}
else version(Posix)
{
    import core.sys.posix.arpa.inet;
    import core.sys.posix.stdlib;
    import core.sys.posix.time;
    import core.sys.posix.sys.time;
}

//Comment this out to disable std.datetime's unit tests.
version = testStdDateTime;

version(unittest)
{
    import std.c.string;
    import std.stdio;
}

//I'd just alias it to indexOf, but
//http://d.puremagic.com/issues/show_bug.cgi?id=6013 would mean that that would
//pollute the global namespace. So, for now, I've created an alias which is
//highly unlikely to conflict with anything that anyone else is doing.
private alias std.string.indexOf stds_indexOf;

version(testStdDateTime) unittest
{
    initializeTests();
}

//Verify module example.
version(testStdDateTime) unittest
{
    auto currentTime = Clock.currTime();
    auto timeString = currentTime.toISOExtString();
    auto restoredTime = SysTime.fromISOExtString(timeString);
}

//Verify Examples for core.time.Duration which couldn't be in core.time.
version(testStdDateTime) unittest
{
    assert(std.datetime.Date(2010, 9, 7) + dur!"days"(5) ==
           std.datetime.Date(2010, 9, 12));

    assert(std.datetime.Date(2010, 9, 7) - std.datetime.Date(2010, 10, 3) ==
           dur!"days"(-26));
}

//Note: There various functions which void as their return type and ref of the
//      struct type which they're in as a commented out return type. Ideally,
//      they would return the ref, but there are several dmd bugs which prevent
//      that, relating to both ref and invariants. So, I've left the ref return
//      types commented out with the idea that those functions can be made to
//      return a ref to this once those bugs have been fixed.

//==============================================================================
// Section with public enums and constants.
//==============================================================================

/++
    Represents the 12 months of the Gregorian year (January is 1).
  +/
enum Month : ubyte { jan = 1, ///
                     feb,     ///
                     mar,     ///
                     apr,     ///
                     may,     ///
                     jun,     ///
                     jul,     ///
                     aug,     ///
                     sep,     ///
                     oct,     ///
                     nov,     ///
                     dec      ///
                   }

/++
    Represents the 7 days of the Gregorian week (Sunday is 0).
  +/
enum DayOfWeek : ubyte { sun = 0, ///
                         mon,     ///
                         tue,     ///
                         wed,     ///
                         thu,     ///
                         fri,     ///
                         sat      ///
                       }

/++
    In some date calculations, adding months or years can cause the date to fall
    on a day of the month which is not valid (e.g. February 29th 2001 or
    June 31st 2000). If overflow is allowed (as is the default), then the month
    will be incremented accordingly (so, February 29th 2001 would become
    March 1st 2001, and June 31st 2000 would become July 1st 2000). If overflow
    is not allowed, then the day will be adjusted to the last valid day in that
    month (so, February 29th 2001 would become February 28th 2001 and
    June 31st 2000 would become June 30th 2000).

    AllowDayOverflow only applies to calculations involving months or years.
  +/
enum AllowDayOverflow
{
    /// No, don't allow day overflow.
    no,

    /// Yes, allow day overflow.
    yes
}

/++
    Indicates a direction in time. One example of its use is $(D Interval)'s
    $(D expand) function which uses it to indicate whether the interval should
    be expanded backwards (into the past), forwards (into the future), or both.
  +/
enum Direction
{
    /// Backward.
    bwd,

    /// Forward.
    fwd,

    /// Both backward and forward.
    both
}

/++
    Used to indicate whether $(D popFront) should be called immediately upon
    creating a range. The idea is that for some functions used to generate a
    range for an interval, $(D front) is not necessarily a time point which
    would ever be generated by the range. To get the first time point
    in the range to match what the function generates, then use
    $(D PopFirst.yes) to indicate that the range should have $(D popFront)
    called on it before the range is returned so that $(D front) is a time point
    which the function would generate.

    For instance, if the function used to generate a range of time points
    generated successive Easters (i.e. you're iterating over all of the Easters
    within the interval), the initial date probably isn't an Easter. Using
    $(D PopFirst.yes) would tell the function which returned the
    range that $(D popFront) was to be called so that front would then be
    an Easter - the next one generated by the function (which when
    iterating forward would be the Easter following the original $(D front),
    while when iterating backward, it would be the Easter prior to the
    original $(D front)). If $(D PopFirst.no) were used, then $(D front) would
    remain the original time point and it would not necessarily be a time point
    which would be generated by the range-generating function (which in many
    cases is exactly what is desired -
    e.g. if iterating over every day starting at the beginning
    of the interval).
  +/
enum PopFirst
{
    /// No, don't call popFront() before returning the range.
    no,

    /// Yes, call popFront() before returning the range.
    yes
}

/++
   Used by StopWatch to indicate whether it should start immediately upon
   construction.
  +/
enum AutoStart
{
    /// No, don't start the StopWatch when it is constructed.
    no,

    /// Yes, do start the StopWatch when it is constructed.
    yes
}

/++
    Array of the strings representing time units, starting with the smallest
    unit and going to the largest. It does not include $(D "nsecs").

   Includes $(D "hnsecs") (hecto-nanoseconds (100 ns)),
   $(D "usecs") (microseconds), $(D "msecs") (milliseconds), $(D "seconds"),
   $(D "minutes"), $(D "hours"), $(D "days"), $(D "weeks"), $(D "months"), and
   $(D "years")
  +/
immutable string[] timeStrings = ["hnsecs", "usecs", "msecs", "seconds", "minutes",
                                  "hours", "days", "weeks", "months", "years"];


//==============================================================================
// Section with private constants.
//==============================================================================

/++
    Array of integers representing the last days of each month in a year.
  +/
private immutable int[13] lastDayNonLeap = [0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334, 365];

/++
    Array of integers representing the last days of each month in a leap year.
  +/
private immutable int[13] lastDayLeap = [0, 31, 60, 91, 121, 152, 182, 213, 244, 274, 305, 335, 366];

/++
    Array of the long names of each month.
  +/
private immutable string[12] longMonthNames = [
    "January",
    "February",
    "March",
    "April",
    "May",
    "June",
    "July",
    "August",
    "September",
    "October",
    "November",
    "December"
];

/++
    Array of the short (three letter) names of each month.
  +/
private immutable string[12] shortMonthNames = [
    "Jan",
    "Feb",
    "Mar",
    "Apr",
    "May",
    "Jun",
    "Jul",
    "Aug",
    "Sep",
    "Oct",
    "Nov",
    "Dec"
];

//==============================================================================
// Section with other types.
//==============================================================================

/++
    Exception type used by std.datetime. It's an alias to TimeException, which
    is what core.time uses. Either can be caught without concern about which
    module it came from.
  +/
alias TimeException DateTimeException;

/++
    Effectively a namespace to make it clear that the methods it contains are
    getting the time from the system clock. It cannot be instantiated.
 +/
final class Clock
{
public:

    /++
        Returns the current time in the given time zone.

        Throws:
            $(D ErrnoException) (on Posix) or $(D Exception) (on Windows)
            if it fails to get the time of day.
      +/
    static SysTime currTime(immutable TimeZone tz = LocalTime())
    {
        return SysTime(currStdTime, tz);
    }

    version(testStdDateTime) unittest
    {
        assert(currTime(UTC()).timezone is UTC());

        //I have no idea why, but for some reason, Windows/Wine likes to get
        //time_t wrong when getting it with core.stdc.time.time. On one box
        //I have (which has its local time set to UTC), it always gives time_t
        //in the real local time (America/Los_Angeles), and after the most recent
        //DST switch, every Windows box that I've tried it in is reporting
        //time_t as being 1 hour off of where it's supposed to be. So, I really
        //don't know what the deal is, but given what I'm seeing, I don't trust
        //core.stdc.time.time on Windows, so I'm just going to disable this test
        //on Windows.
        version(Posix)
        {
            immutable unixTimeD = currTime().toUnixTime();
            immutable unixTimeC = core.stdc.time.time(null);
            immutable diff = unixTimeC - unixTimeD;

            _assertPred!">="(diff, -2);
            _assertPred!"<="(diff, 2);
        }
    }

    /++
        Returns the number of hnsecs since midnight, January 1st, 1 A.D. for the
        current time.

        Throws:
            $(D DateTimeException) if it fails to get the time.
      +/
    @trusted
    static @property long currStdTime()
    {
        version(Windows)
        {
            FILETIME fileTime;
            GetSystemTimeAsFileTime(&fileTime);

            return FILETIMEToStdTime(&fileTime);
        }
        else version(Posix)
        {
            enum hnsecsToUnixEpoch = 621_355_968_000_000_000L;

            static if(is(typeof(clock_gettime)))
            {
                timespec ts;

                if(clock_gettime(CLOCK_REALTIME, &ts) != 0)
                    throw new TimeException("Failed in clock_gettime().");

                return convert!("seconds", "hnsecs")(ts.tv_sec) +
                       ts.tv_nsec / 100 +
                       hnsecsToUnixEpoch;
            }
            else
            {
                timeval tv;

                if(gettimeofday(&tv, null) != 0)
                    throw new TimeException("Failed in gettimeofday().");

                return convert!("seconds", "hnsecs")(tv.tv_sec) +
                       convert!("usecs", "hnsecs")(tv.tv_usec) +
                       hnsecsToUnixEpoch;
            }
        }
    }

    /++
        The current system tick. The number of ticks per second varies from
        system to system. currSystemTick uses a monotonic clock, so it's
        intended for precision timing by comparing relative time values, not
        for getting the current system time.

        Warning:
            On some systems, the monotonic clock may stop counting when
            the computer goes to sleep or hibernates. So, the monotonic
            clock could be off if that occurs. This is known to happen
            on Mac OS X. It has not been tested whether it occurs on
            either Windows or Linux.

        Throws:
            $(D DateTimeException) if it fails to get the time.
      +/
    @safe
    static @property TickDuration currSystemTick()
    {
        return TickDuration.currSystemTick();
    }

    version(testStdDateTime) unittest
    {
        assert(Clock.currSystemTick.length > 0);
    }

    /++
        The current number of system ticks since the application started.
        The number of ticks per second varies from system to system.
        This uses a monotonic clock.

        Warning:
            On some systems, the monotonic clock may stop counting when
            the computer goes to sleep or hibernates. So, the monotonic
            clock could be off if that occurs. This is known to happen
            on Mac OS X. It has not been tested whether it occurs on
            either Windows or on Linux.

        Throws:
            $(D DateTimeException) if it fails to get the time.
      +/
    @safe
    static @property TickDuration currAppTick()
    {
        return currSystemTick - TickDuration.appOrigin;
    }

    version(testStdDateTime) unittest
    {
        auto a = Clock.currSystemTick;
        auto b = Clock.currAppTick;
        assert(a.length);
        assert(b.length);
        assert(a > b);
    }

private:

    @disable this() {}
}

//==============================================================================
// Section with time points.
//==============================================================================

/++
    $(D SysTime) is the type used to get the current time from the
    system or doing anything that involves time zones. Unlike
    $(D DateTime), the time zone is an integral part of $(D SysTime) (though for
    local time applications, time zones can be ignored and
    it will work, since it defaults to using the local time zone). It holds its
    internal time in std time (hnsecs since midnight, January 1st, 1 A.D. UTC),
    so it interfaces well with the system time. However, that means that, unlike
    $(D DateTime), it is not optimized for calendar-based operations, and
    getting individual units from it such as years or days is going to involve
    conversions and be less efficient.

    For calendar-based operations that don't
    care about time zones, then $(D DateTime) would be the type to
    use. For system time, use $(D SysTime).

    $(D Clock.currTime) will return the current time as a $(D SysTime).
    To convert a $(D SysTime) to a $(D Date) or $(D DateTime), simply cast
    it. To convert a $(D Date) or $(D DateTime) to a
    $(D SysTime), use $(D SysTime)'s constructor, and pass in the
    intended time zone with it (or don't pass in a $(D TimeZone), and the local
    time zone will be used). Be aware, however, that converting from a
    $(D DateTime) to a $(D SysTime) will not necessarily be 100% accurate due to
    DST (one hour of the year doesn't exist and another occurs twice).
    To not risk any conversion errors, keep times as
    $(D SysTime)s. Aside from DST though, there shouldn't be any conversion
    problems.

    For using time zones other than local time or UTC, use
    $(D PosixTimeZone) on Posix systems (or on Windows, if providing the TZ
    Database files), and use $(D WindowsTimeZone) on Windows systems.
    The time in $(D SysTime) is kept internally in hnsecs from midnight,
    January 1st, 1 A.D. UTC. Conversion error cannot happen when changing
    the time zone of a $(D SysTime). $(D LocalTime) is the $(D TimeZone) class
    which represents the local time, and $(D UTC) is the $(D TimeZone) class
    which represents UTC. $(D SysTime) uses $(D LocalTime) if no $(D TimeZone)
    is provided. For more details on time zones, see the documentation for
    $(D TimeZone), $(D PosixTimeZone), and $(D WindowsTimeZone).

    $(D SysTime)'s range is from approximately 29,000 B.C. to approximately
    29,000 A.D.
  +/
struct SysTime
{
public:

    /++
        Params:
            dateTime = The $(D DateTime) to use to set this $(D SysTime)'s
                       internal std time. As $(D DateTime) has no concept of
                       time zone, tz is used as its time zone.
            tz       = The $(D TimeZone) to use for this $(D SysTime). If null,
                       $(D LocalTime) will be used. The given $(D DateTime) is
                       assumed to be in the given time zone.
      +/
    this(in DateTime dateTime, immutable TimeZone tz = null) nothrow
    {
        try
            this(dateTime, FracSec.from!"hnsecs"(0), tz);
        catch(Exception e)
            assert(0, "FracSec's constructor threw when it shouldn't have.");
    }

    version(testStdDateTime) unittest
    {
        static void test(DateTime dt, immutable TimeZone tz, long expected)
        {
            auto sysTime = SysTime(dt, tz);
            _assertPred!"=="(sysTime._stdTime, expected);
            assert(sysTime._timezone is (tz is null ? LocalTime() : tz),
                   format("Given DateTime: %s", dt));
        }

        test(DateTime.init, UTC(), 0);
        test(DateTime(1, 1, 1, 12, 30, 33), UTC(), 450_330_000_000L);
        test(DateTime(0, 12, 31, 12, 30, 33), UTC(), -413_670_000_000L);
        test(DateTime(1, 1, 1, 0, 0, 0), UTC(), 0);
        test(DateTime(1, 1, 1, 0, 0, 1), UTC(), 10_000_000L);
        test(DateTime(0, 12, 31, 23, 59, 59), UTC(), -10_000_000L);

        test(DateTime(1, 1, 1, 0, 0, 0), new SimpleTimeZone(-60),
             36_000_000_000L);
        test(DateTime(1, 1, 1, 0, 0, 0), new SimpleTimeZone(0), 0);
        test(DateTime(1, 1, 1, 0, 0, 0), new SimpleTimeZone(60),
             -36_000_000_000L);
    }

    /++
        Params:
            dateTime = The $(D DateTime) to use to set this $(D SysTime)'s
                       internal std time. As $(D DateTime) has no concept of
                       time zone, tz is used as its time zone.
            fracSec  = The fractional seconds portion of the time.
            tz       = The $(D TimeZone) to use for this $(D SysTime). If null,
                       $(D LocalTime) will be used. The given $(D DateTime) is
                       assumed to be in the given time zone.

        Throws:
            $(D DateTimeException) if $(D fracSec) is negative.
      +/
    this(in DateTime dateTime, in FracSec fracSec, immutable TimeZone tz = null)
    {
        immutable fracHNSecs = fracSec.hnsecs;
        enforce(fracHNSecs >= 0, new DateTimeException("A SysTime cannot have negative fractional seconds."));
        _timezone = tz is null ? LocalTime() : tz;

        try
        {
            immutable dateDiff = (dateTime.date - Date(1, 1, 1)).total!"hnsecs";
            immutable todDiff = (dateTime.timeOfDay - TimeOfDay(0, 0, 0)).total!"hnsecs";

            immutable adjustedTime = dateDiff + todDiff + fracHNSecs;
            immutable standardTime = _timezone.tzToUTC(adjustedTime);

            this(standardTime, _timezone);
        }
        catch(Exception e)
        {
            assert(0, "Date, TimeOfDay, or DateTime's constructor threw when " ~
                      "it shouldn't have.");
        }
    }

    version(testStdDateTime) unittest
    {
        static void test(DateTime dt,
                         FracSec fracSec,
                         immutable TimeZone tz,
                         long expected)
        {
            auto sysTime = SysTime(dt, fracSec, tz);
            _assertPred!"=="(sysTime._stdTime, expected);
            assert(sysTime._timezone is (tz is null ? LocalTime() : tz),
                   format("Given DateTime: %s, Given FracSec: %s", dt, fracSec));
        }

        test(DateTime.init, FracSec.init, UTC(), 0);
        test(DateTime(1, 1, 1, 12, 30, 33), FracSec.init, UTC(), 450_330_000_000L);
        test(DateTime(0, 12, 31, 12, 30, 33), FracSec.init, UTC(), -413_670_000_000L);
        test(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"msecs"(1), UTC(), 10_000L);
        test(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"msecs"(999), UTC(), -10_000L);

        test(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999), UTC(), -1);
        test(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(1), UTC(), -9_999_999);
        test(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(0), UTC(), -10_000_000);

        assertThrown!DateTimeException(SysTime(DateTime.init, FracSec.from!"hnsecs"(-1), UTC()));
    }

    /++
        Params:
            date = The $(D Date) to use to set this $(D SysTime)'s internal std
                   time. As $(D Date) has no concept of time zone, tz is used as
                   its time zone.
            tz   = The $(D TimeZone) to use for this $(D SysTime). If null,
                   $(D LocalTime) will be used. The given $(D Date) is assumed
                   to be in the given time zone.
      +/
    this(in Date date, immutable TimeZone tz = null) nothrow
    {
        _timezone = tz is null ? LocalTime() : tz;

        try
        {
            immutable adjustedTime = (date - Date(1, 1, 1)).total!"hnsecs";
            immutable standardTime = _timezone.tzToUTC(adjustedTime);

            this(standardTime, _timezone);
        }
        catch(Exception e)
            assert(0, "Date's constructor through when it shouldn't have.");
    }

    version(testStdDateTime) unittest
    {
        static void test(Date d, immutable TimeZone tz, long expected)
        {
            auto sysTime = SysTime(d, tz);
            _assertPred!"=="(sysTime._stdTime, expected);
            assert(sysTime._timezone is (tz is null ? LocalTime() : tz),
                   format("Given Date: %s", d));
        }

        test(Date.init, UTC(), 0);
        test(Date(1, 1, 1), UTC(), 0);
        test(Date(1, 1, 2), UTC(), 864000000000);
        test(Date(0, 12, 31), UTC(), -864000000000);
    }

    /++
        Note:
            Whereas the other constructors take in the given date/time, assume
            that it's in the given time zone, and convert it to hnsecs in UTC
            since midnight, January 1st, 1 A.D. UTC - i.e. std time - this
            constructor takes a std time, which is specifically already in UTC,
            so no conversion takes place. Of course, the various getter
            properties and functions will use the given time zone's conversion
            function to convert the results to that time zone, but no conversion
            of the arguments to this constructor takes place.

        Params:
            stdTime = The number of hnsecs since midnight, January 1st, 1 A.D. UTC.
            tz      = The $(D TimeZone) to use for this $(D SysTime). If null,
                      $(D LocalTime) will be used.
      +/
    this(long stdTime, immutable TimeZone tz = null) pure nothrow
    {
        _stdTime = stdTime;
        _timezone = tz is null ? LocalTime() : tz;
    }

    version(testStdDateTime) unittest
    {
        static void test(long stdTime, immutable TimeZone tz)
        {
            auto sysTime = SysTime(stdTime, tz);
            _assertPred!"=="(sysTime._stdTime, stdTime);
            assert(sysTime._timezone is (tz is null ? LocalTime() : tz),
                   format("Given stdTime: %s", stdTime));
        }

        foreach(stdTime; [-1234567890L, -250, 0, 250, 1235657390L])
        {
            foreach(tz; testTZs)
                test(stdTime, tz);
        }
    }

    /++
        Params:
            rhs = The $(D SysTime) to assign to this one.
      +/
    ref SysTime opAssign(const ref SysTime rhs) pure nothrow
    {
        _stdTime = rhs._stdTime;
        _timezone = rhs._timezone;

        return this;
    }

    /++
        Params:
            rhs = The $(D SysTime) to assign to this one.
      +/
    ref SysTime opAssign(SysTime rhs) pure nothrow
    {
        _stdTime = rhs._stdTime;
        _timezone = rhs._timezone;

        return this;
    }

    /++
        Checks for equality between this $(D SysTime) and the given
        $(D SysTime).

        Note that the time zone is ignored. Only the internal
        std times (which are in UTC) are compared.
     +/
    bool opEquals(const SysTime rhs) const pure nothrow
    {
        return opEquals(rhs);
    }

    /// ditto
    bool opEquals(const ref SysTime rhs) const pure nothrow
    {
        return _stdTime == rhs._stdTime;
    }

    version(testStdDateTime) unittest
    {
        _assertPred!"=="(SysTime(DateTime.init, UTC()), SysTime(0, UTC()));
        _assertPred!"=="(SysTime(DateTime.init, UTC()), SysTime(0));
        _assertPred!"=="(SysTime(Date.init, UTC()), SysTime(0));
        _assertPred!"=="(SysTime(0), SysTime(0));

        static void test(DateTime dt,
                         immutable TimeZone tz1,
                         immutable TimeZone tz2)
        {
            auto st1 = SysTime(dt);
            st1.timezone = tz1;

            auto st2 = SysTime(dt);
            st2.timezone = tz2;

            _assertPred!"=="(st1, st2);
        }

        foreach(tz1; testTZs)
        {
            foreach(tz2; testTZs)
            {
                foreach(dt; chain(testDateTimesBC, testDateTimesAD))
                        test(dt, tz1, tz2);
            }
        }

        auto st = SysTime(DateTime(1999, 7, 6, 12, 33, 30));
        const cst = SysTime(DateTime(1999, 7, 6, 12, 33, 30));
        //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 33, 30));
        static assert(__traits(compiles, st == st));
        static assert(__traits(compiles, st == cst));
        //static assert(__traits(compiles, st == ist));
        static assert(__traits(compiles, cst == st));
        static assert(__traits(compiles, cst == cst));
        //static assert(__traits(compiles, cst == ist));
        //static assert(__traits(compiles, ist == st));
        //static assert(__traits(compiles, ist == cst));
        //static assert(__traits(compiles, ist == ist));
    }

    /++
        Compares this $(D SysTime) with the given $(D SysTime).

        Time zone is irrelevant when comparing $(D SysTime)s.

        Returns:
            $(BOOKTABLE,
            $(TR $(TD this &lt; rhs) $(TD &lt; 0))
            $(TR $(TD this == rhs) $(TD 0))
            $(TR $(TD this &gt; rhs) $(TD &gt; 0))
            )
     +/
    int opCmp(in SysTime rhs) const pure nothrow
    {
        if(_stdTime < rhs._stdTime)
            return -1;
        if(_stdTime > rhs._stdTime)
            return 1;

        return 0;
    }

    version(testStdDateTime) unittest
    {
        _assertPred!("opCmp", "==")(SysTime(DateTime.init, UTC()),
                                    SysTime(0, UTC()));
        _assertPred!("opCmp", "==")(SysTime(DateTime.init, UTC()), SysTime(0));
        _assertPred!("opCmp", "==")(SysTime(Date.init, UTC()), SysTime(0));
        _assertPred!("opCmp", "==")(SysTime(0), SysTime(0));

        static void testEqual(SysTime st,
                              immutable TimeZone tz1,
                              immutable TimeZone tz2)
        {
            auto st1 = st;
            st1.timezone = tz1;

            auto st2 = st;
            st2.timezone = tz2;

            _assertPred!("opCmp", "==")(st1, st2);
        }

        auto sts = array(map!SysTime(chain(testDateTimesBC, testDateTimesAD)));

        foreach(st; sts)
            foreach(tz1; testTZs)
                foreach(tz2; testTZs)
                    testEqual(st, tz1, tz2);

        static void testCmp(SysTime st1,
                            immutable TimeZone tz1,
                            SysTime st2,
                            immutable TimeZone tz2)
        {
            st1.timezone = tz1;
            st2.timezone = tz2;
            _assertPred!("opCmp", "<")(st1, st2);
            _assertPred!("opCmp", ">")(st2, st1);
        }

        foreach(si, st1; sts)
            foreach(st2; sts[si+1 .. $])
                foreach(tz1; testTZs)
                    foreach(tz2; testTZs)
                        testCmp(st1, tz1, st2, tz2);

        auto st = SysTime(DateTime(1999, 7, 6, 12, 33, 30));
        const cst = SysTime(DateTime(1999, 7, 6, 12, 33, 30));
        //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 33, 30));
        static assert(__traits(compiles, st.opCmp(st)));
        static assert(__traits(compiles, st.opCmp(cst)));
        //static assert(__traits(compiles, st.opCmp(ist)));
        static assert(__traits(compiles, cst.opCmp(st)));
        static assert(__traits(compiles, cst.opCmp(cst)));
        //static assert(__traits(compiles, cst.opCmp(ist)));
        //static assert(__traits(compiles, ist.opCmp(st)));
        //static assert(__traits(compiles, ist.opCmp(cst)));
        //static assert(__traits(compiles, ist.opCmp(ist)));
    }

    /++
        Year of the Gregorian Calendar. Positive numbers are A.D. Non-positive
        are B.C.
     +/
    @property short year() const nothrow
    {
        return (cast(Date)this).year;
    }

    version(testStdDateTime) unittest
    {
        static void test(SysTime sysTime, long expected, size_t line = __LINE__)
        {
            _assertPred!"=="(sysTime.year, expected,
                             format("Value given: %s", sysTime), __FILE__, line);
        }

        test(SysTime(0, UTC()), 1);
        test(SysTime(1, UTC()), 1);
        test(SysTime(-1, UTC()), 0);

        foreach(year; chain(testYearsBC, testYearsAD))
        {
            foreach(md; testMonthDays)
            {
                foreach(tod; testTODs)
                {
                    auto dt = DateTime(Date(year, md.month, md.day), tod);

                    foreach(tz; testTZs)
                    {
                        foreach(fs; testFracSecs)
                            test(SysTime(dt, fs, tz), year);
                    }
                }
            }
        }

        const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
        //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
        static assert(__traits(compiles, cst.year));
        //static assert(__traits(compiles, ist.year));
    }

    /++
        Year of the Gregorian Calendar. Positive numbers are A.D. Non-positive
        are B.C.

        Params:
            year = The year to set this $(D SysTime)'s year to.

        Throws:
            $(D DateTimeException) if the new year is not a leap year and the
            resulting date would be on February 29th.

        Examples:
--------------------
assert(SysTime(DateTime(1999, 7, 6, 9, 7, 5)).year == 1999);
assert(SysTime(DateTime(2010, 10, 4, 0, 0, 30)).year == 2010);
assert(SysTime(DateTime(-7, 4, 5, 7, 45, 2)).year == -7);
--------------------
     +/
    @property void year(int year)
    {
        auto hnsecs = adjTime;
        auto days = splitUnitsFromHNSecs!"days"(hnsecs) + 1;

        if(hnsecs < 0)
        {
            hnsecs += convert!("hours", "hnsecs")(24);
            --days;
        }

        auto date = Date(cast(int)days);
        date.year = year;

        immutable newDaysHNSecs = convert!("days", "hnsecs")(date.dayOfGregorianCal - 1);
        adjTime = newDaysHNSecs + hnsecs;
    }

    //Verify Examples.
    version(testStdDateTime) unittest
    {
        assert(SysTime(DateTime(1999, 7, 6, 9, 7, 5)).year == 1999);
        assert(SysTime(DateTime(2010, 10, 4, 0, 0, 30)).year == 2010);
        assert(SysTime(DateTime(-7, 4, 5, 7, 45, 2)).year == -7);
    }

    version(testStdDateTime) unittest
    {
        static void test(SysTime st, int year, in SysTime expected, size_t line = __LINE__)
        {
            st.year = year;
            _assertPred!"=="(st, expected, "", __FILE__, line);
        }

        foreach(st; chain(testSysTimesBC, testSysTimesAD))
        {
            auto dt = cast(DateTime)st;

            foreach(year; chain(testYearsBC, testYearsAD))
            {
                auto e = SysTime(DateTime(year, dt.month, dt.day, dt.hour, dt.minute, dt.second),
                                 st.fracSec,
                                 st.timezone);
                test(st, year, e);
            }
        }

        foreach(fs; testFracSecs)
        {
            foreach(tz; testTZs)
            {
                foreach(tod; testTODs)
                {
                    test(SysTime(DateTime(Date(1999, 2, 28), tod), fs, tz), 2000,
                         SysTime(DateTime(Date(2000, 2, 28), tod), fs, tz));
                    test(SysTime(DateTime(Date(2000, 2, 28), tod), fs, tz), 1999,
                         SysTime(DateTime(Date(1999, 2, 28), tod), fs, tz));
                }

                foreach(tod; testTODsThrown)
                {
                    auto st = SysTime(DateTime(Date(2000, 2, 29), tod), fs, tz);
                    assertThrown!DateTimeException(st.year = 1999);
                }
            }
        }

        const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
        //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
        static assert(!__traits(compiles, cst.year = 7));
        //static assert(!__traits(compiles, ist.year = 7));
    }

    /++
        Year B.C. of the Gregorian Calendar counting year 0 as 1 B.C.

        Throws:
            $(D DateTimeException) if $(D isAD) is true.

        Examples:
--------------------
assert(SysTime(DateTime(0, 1, 1, 12, 30, 33)).yearBC == 1);
assert(SysTime(DateTime(-1, 1, 1, 10, 7, 2)).yearBC == 2);
assert(SysTime(DateTime(-100, 1, 1, 4, 59, 0)).yearBC == 101);
--------------------
     +/
    @property ushort yearBC() const
    {
        return (cast(Date)this).yearBC;
    }

    //Verify Examples.
    version(testStdDateTime) unittest
    {
        assert(SysTime(DateTime(0, 1, 1, 12, 30, 33)).yearBC == 1);
        assert(SysTime(DateTime(-1, 1, 1, 10, 7, 2)).yearBC == 2);
        assert(SysTime(DateTime(-100, 1, 1, 4, 59, 0)).yearBC == 101);
    }

    version(testStdDateTime) unittest
    {
        foreach(st; testSysTimesBC)
        {
            auto msg = format("SysTime: %s", st);
            assertNotThrown!DateTimeException(st.yearBC, msg);
            _assertPred!"=="(st.yearBC, (st.year * -1) + 1, msg);
        }

        foreach(st; [testSysTimesAD[0], testSysTimesAD[$/2], testSysTimesAD[$-1]])
            assertThrown!DateTimeException(st.yearBC, format("SysTime: %s", st));

        auto st = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
        const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
        //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
        static assert(__traits(compiles, st.year = 12));
        static assert(!__traits(compiles, cst.year = 12));
        //static assert(!__traits(compiles, ist.year = 12));
    }


    /++
        Year B.C. of the Gregorian Calendar counting year 0 as 1 B.C.

        Params:
            year = The year B.C. to set this $(D SysTime)'s year to.

        Throws:
            $(D DateTimeException) if a non-positive value is given.

        Examples:
--------------------
auto st = SysTime(DateTime(2010, 1, 1, 7, 30, 0));
st.yearBC = 1;
assert(st == SysTime(DateTime(0, 1, 1, 7, 30, 0)));

st.yearBC = 10;
assert(st == SysTime(DateTime(-9, 1, 1, 7, 30, 0)));
--------------------
     +/
    @property void yearBC(int year)
    {
        auto hnsecs = adjTime;
        auto days = splitUnitsFromHNSecs!"days"(hnsecs) + 1;

        if(hnsecs < 0)
        {
            hnsecs += convert!("hours", "hnsecs")(24);
            --days;
        }

        auto date = Date(cast(int)days);
        date.yearBC = year;

        immutable newDaysHNSecs = convert!("days", "hnsecs")(date.dayOfGregorianCal - 1);
        adjTime = newDaysHNSecs + hnsecs;
    }

    //Verify Examples
    version(testStdDateTime) unittest
    {
        auto st = SysTime(DateTime(2010, 1, 1, 7, 30, 0));
        st.yearBC = 1;
        assert(st == SysTime(DateTime(0, 1, 1, 7, 30, 0)));

        st.yearBC = 10;
        assert(st == SysTime(DateTime(-9, 1, 1, 7, 30, 0)));
    }

    version(testStdDateTime) unittest
    {
        static void test(SysTime st, int year, in SysTime expected, size_t line = __LINE__)
        {
            st.yearBC = year;
            _assertPred!"=="(st, expected, format("SysTime: %s", st), __FILE__, line);
        }

        foreach(st; chain(testSysTimesBC, testSysTimesAD))
        {
            auto dt = cast(DateTime)st;

            foreach(year; testYearsBC)
            {
                auto e = SysTime(DateTime(year, dt.month, dt.day, dt.hour, dt.minute, dt.second),
                                 st.fracSec,
                                 st.timezone);

                test(st, (year * -1) + 1, e);
            }
        }

        foreach(st; [testSysTimesBC[0], testSysTimesBC[$ - 1],
                     testSysTimesAD[0], testSysTimesAD[$ - 1]])
        {
            foreach(year; testYearsBC)
                assertThrown!DateTimeException(st.yearBC = year);
        }

        foreach(fs; testFracSecs)
        {
            foreach(tz; testTZs)
            {
                foreach(tod; testTODs)
                {
                    test(SysTime(DateTime(Date(-1999, 2, 28), tod), fs, tz), 2001,
                         SysTime(DateTime(Date(-2000, 2, 28), tod), fs, tz));
                    test(SysTime(DateTime(Date(-2000, 2, 28), tod), fs, tz), 2000,
                         SysTime(DateTime(Date(-1999, 2, 28), tod), fs, tz));
                }

                foreach(tod; testTODsThrown)
                {
                    auto st = SysTime(DateTime(Date(-2000, 2, 29), tod), fs, tz);
                    assertThrown!DateTimeException(st.year = -1999);
                }
            }
        }

        auto st = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
        const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
        //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
        static assert(__traits(compiles, st.yearBC = 12));
        static assert(!__traits(compiles, cst.yearBC = 12));
        //static assert(!__traits(compiles, ist.yearBC = 12));
    }

    /++
        Month of a Gregorian Year.

        Examples:
--------------------
assert(SysTime(DateTime(1999, 7, 6, 9, 7, 5)).month == 7);
assert(SysTime(DateTime(2010, 10, 4, 0, 0, 30)).month == 10);
assert(SysTime(DateTime(-7, 4, 5, 7, 45, 2)).month == 4);
--------------------
     +/
    @property Month month() const nothrow
    {
        return (cast(Date)this).month;
    }

    //Verify Examples.
    version(testStdDateTime) unittest
    {
        assert(SysTime(DateTime(1999, 7, 6, 9, 7, 5)).month == 7);
        assert(SysTime(DateTime(2010, 10, 4, 0, 0, 30)).month == 10);
        assert(SysTime(DateTime(-7, 4, 5, 7, 45, 2)).month == 4);
    }

    version(testStdDateTime) unittest
    {
        static void test(SysTime sysTime, Month expected, size_t line = __LINE__)
        {
            _assertPred!"=="(sysTime.month, expected,
                             format("Value given: %s", sysTime), __FILE__, line);
        }

        test(SysTime(0, UTC()), Month.jan);
        test(SysTime(1, UTC()), Month.jan);
        test(SysTime(-1, UTC()), Month.dec);

        foreach(year; chain(testYearsBC, testYearsAD))
        {
            foreach(md; testMonthDays)
            {
                foreach(tod; testTODs)
                {
                    auto dt = DateTime(Date(year, md.month, md.day), tod);

                    foreach(fs; testFracSecs)
                    {
                        foreach(tz; testTZs)
                            test(SysTime(dt, fs, tz), md.month);
                    }
                }
            }
        }

        const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
        //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
        static assert(__traits(compiles, cst.month));
        //static assert(__traits(compiles, ist.month));
    }


    /++
        Month of a Gregorian Year.

        Params:
            month = The month to set this $(D SysTime)'s month to.

        Throws:
            $(D DateTimeException) if the given month is not a valid month.
     +/
    @property void month(Month month)
    {
        auto hnsecs = adjTime;
        auto days = splitUnitsFromHNSecs!"days"(hnsecs) + 1;

        if(hnsecs < 0)
        {
            hnsecs += convert!("hours", "hnsecs")(24);
            --days;
        }

        auto date = Date(cast(int)days);
        date.month = month;

        immutable newDaysHNSecs = convert!("days", "hnsecs")(date.dayOfGregorianCal - 1);
        adjTime = newDaysHNSecs + hnsecs;
    }

    version(testStdDateTime) unittest
    {
        static void test(SysTime st, Month month, in SysTime expected, size_t line = __LINE__)
        {
            st.month = cast(Month)month;
            _assertPred!"=="(st, expected, "", __FILE__, line);
        }

        foreach(st; chain(testSysTimesBC, testSysTimesAD))
        {
            auto dt = cast(DateTime)st;

            foreach(md; testMonthDays)
            {
                if(st.day > maxDay(dt.year, md.month))
                    continue;

                auto e = SysTime(DateTime(dt.year, md.month, dt.day, dt.hour, dt.minute, dt.second),
                                 st.fracSec,
                                 st.timezone);

                test(st, md.month, e);
            }
        }

        foreach(fs; testFracSecs)
        {
            foreach(tz; testTZs)
            {
                foreach(tod; testTODs)
                {
                    foreach(year; filter!((a){return yearIsLeapYear(a);})
                                         (chain(testYearsBC, testYearsAD)))
                    {
                        test(SysTime(DateTime(Date(year, 1, 29), tod), fs, tz),
                             Month.feb,
                             SysTime(DateTime(Date(year, 2, 29), tod), fs, tz));
                    }

                    foreach(year; chain(testYearsBC, testYearsAD))
                    {
                        test(SysTime(DateTime(Date(year, 1, 28), tod), fs, tz),
                             Month.feb,
                             SysTime(DateTime(Date(year, 2, 28), tod), fs, tz));
                        test(SysTime(DateTime(Date(year, 7, 30), tod), fs, tz),
                             Month.jun,
                             SysTime(DateTime(Date(year, 6, 30), tod), fs, tz));
                    }
                }
            }
        }

        foreach(fs; [testFracSecs[0], testFracSecs[$-1]])
        {
            foreach(tz; testTZs)
            {
                foreach(tod; testTODsThrown)
                {
                    foreach(year; [testYearsBC[$-3], testYearsBC[$-2],
                                   testYearsBC[$-2], testYearsAD[0],
                                   testYearsAD[$-2], testYearsAD[$-1]])
                    {
                        auto day = yearIsLeapYear(year) ? 30 : 29;
                        auto st1 = SysTime(DateTime(Date(year, 1, day), tod), fs, tz);
                        assertThrown!DateTimeException(st1.month = Month.feb);

                        auto st2 = SysTime(DateTime(Date(year, 7, 31), tod), fs, tz);
                        assertThrown!DateTimeException(st2.month = Month.jun);
                    }
                }
            }
        }

        const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
        //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
        static assert(!__traits(compiles, cst.month = 12));
        //static assert(!__traits(compiles, ist.month = 12));
    }

    /++
        Day of a Gregorian Month.

        Examples:
--------------------
assert(SysTime(DateTime(1999, 7, 6, 9, 7, 5)).day == 6);
assert(SysTime(DateTime(2010, 10, 4, 0, 0, 30)).day == 4);
assert(SysTime(DateTime(-7, 4, 5, 7, 45, 2)).day == 5);
--------------------
     +/
    @property ubyte day() const nothrow
    {
        return (cast(Date)this).day;
    }

    //Verify Examples.
    version(testStdDateTime) unittest
    {
        assert(SysTime(DateTime(1999, 7, 6, 9, 7, 5)).day == 6);
        assert(SysTime(DateTime(2010, 10, 4, 0, 0, 30)).day == 4);
        assert(SysTime(DateTime(-7, 4, 5, 7, 45, 2)).day == 5);
    }

    version(testStdDateTime) unittest
    {
        static void test(SysTime sysTime, int expected, size_t line = __LINE__)
        {
            _assertPred!"=="(sysTime.day, expected,
                             format("Value given: %s", sysTime), __FILE__, line);
        }

        test(SysTime(0, UTC()), 1);
        test(SysTime(1, UTC()), 1);
        test(SysTime(-1, UTC()), 31);

        foreach(year; chain(testYearsBC, testYearsAD))
        {
            foreach(md; testMonthDays)
            {
                foreach(tod; testTODs)
                {
                    auto dt = DateTime(Date(year, md.month, md.day), tod);

                    foreach(tz; testTZs)
                    {
                        foreach(fs; testFracSecs)
                            test(SysTime(dt, fs, tz), md.day);
                    }
                }
            }
        }

        const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
        //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
        static assert(__traits(compiles, cst.day));
        //static assert(__traits(compiles, ist.day));
    }


    /++
        Day of a Gregorian Month.

        Params:
            day = The day of the month to set this $(D SysTime)'s day to.

        Throws:
            $(D DateTimeException) if the given day is not a valid day of the
            current month.
     +/
    @property void day(int day)
    {
        auto hnsecs = adjTime;
        auto days = splitUnitsFromHNSecs!"days"(hnsecs) + 1;

        if(hnsecs < 0)
        {
            hnsecs += convert!("hours", "hnsecs")(24);
            --days;
        }

        auto date = Date(cast(int)days);
        date.day = day;

        immutable newDaysHNSecs = convert!("days", "hnsecs")(date.dayOfGregorianCal - 1);
        adjTime = newDaysHNSecs + hnsecs;
    }

    version(testStdDateTime) unittest
    {
        foreach(day; chain(testDays))
        {
            foreach(st; chain(testSysTimesBC, testSysTimesAD))
            {
                auto dt = cast(DateTime)st;

                if(day > maxDay(dt.year, dt.month))
                    continue;

                auto expected = SysTime(DateTime(dt.year, dt.month, day, dt.hour, dt.minute, dt.second),
                                        st.fracSec,
                                        st.timezone);

                st.day = day;
                assert(st == expected, format("[%s] [%s]", st, expected));
            }
        }

        foreach(tz; testTZs)
        {
            foreach(tod; testTODs)
            {
                foreach(fs; testFracSecs)
                {
                    foreach(year; chain(testYearsBC, testYearsAD))
                    {
                        foreach(month; EnumMembers!Month)
                        {
                            auto st = SysTime(DateTime(Date(year, month, 1), tod), fs, tz);
                            immutable max = maxDay(year, month);
                            auto expected = SysTime(DateTime(Date(year, month, max), tod), fs, tz);

                            st.day = max;
                            assert(st == expected, format("[%s] [%s]", st, expected));
                        }
                    }
                }
            }
        }

        foreach(tz; testTZs)
        {
            foreach(tod; testTODsThrown)
            {
                foreach(fs; [testFracSecs[0], testFracSecs[$-1]])
                {
                    foreach(year; [testYearsBC[$-3], testYearsBC[$-2],
                                   testYearsBC[$-2], testYearsAD[0],
                                   testYearsAD[$-2], testYearsAD[$-1]])
                    {
                        foreach(month; EnumMembers!Month)
                        {
                            auto st = SysTime(DateTime(Date(year, month, 1), tod), fs, tz);
                            immutable max = maxDay(year, month);

                            assertThrown!DateTimeException(st.day = max + 1);
                        }
                    }
                }
            }
        }

        const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
        //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
        static assert(!__traits(compiles, cst.day = 27));
        //static assert(!__traits(compiles, ist.day = 27));
    }


    /++
        Hours past midnight.
     +/
    @property ubyte hour() const nothrow
    {
        auto hnsecs = adjTime;
        auto days = splitUnitsFromHNSecs!"days"(hnsecs) + 1;

        if(hnsecs < 0)
        {
            hnsecs += convert!("hours", "hnsecs")(24);
            --days;
        }

        return cast(ubyte)getUnitsFromHNSecs!"hours"(hnsecs);
    }

    version(testStdDateTime) unittest
    {
        static void test(SysTime sysTime, int expected, size_t line = __LINE__)
        {
            _assertPred!"=="(sysTime.hour, expected,
                             format("Value given: %s", sysTime), __FILE__, line);
        }

        test(SysTime(0, UTC()), 0);
        test(SysTime(1, UTC()), 0);
        test(SysTime(-1, UTC()), 23);

        foreach(tz; testTZs)
        {
            foreach(year; chain(testYearsBC, testYearsAD))
            {
                foreach(md; testMonthDays)
                {
                    foreach(hour; testHours)
                    {
                        foreach(minute; testMinSecs)
                        {
                            foreach(second; testMinSecs)
                            {
                                auto dt = DateTime(Date(year, md.month, md.day),
                                                   TimeOfDay(hour, minute, second));

                                foreach(fs; testFracSecs)
                                    test(SysTime(dt, fs, tz), hour);
                            }
                        }
                    }
                }
            }
        }

        const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
        //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
        static assert(__traits(compiles, cst.hour));
        //static assert(__traits(compiles, ist.hour));
    }


    /++
        Hours past midnight.

        Params:
            hour = The hours to set this $(D SysTime)'s hour to.

        Throws:
            $(D DateTimeException) if the given hour are not a valid hour of
            the day.
     +/
    @property void hour(int hour)
    {
        enforceValid!"hours"(hour);

        auto hnsecs = adjTime;
        auto days = splitUnitsFromHNSecs!"days"(hnsecs);
        immutable daysHNSecs = convert!("days", "hnsecs")(days);
        immutable negative = hnsecs < 0;

        if(negative)
            hnsecs += convert!("hours", "hnsecs")(24);

        hnsecs = removeUnitsFromHNSecs!"hours"(hnsecs);
        hnsecs += convert!("hours", "hnsecs")(hour);

        if(negative)
            hnsecs -= convert!("hours", "hnsecs")(24);

        adjTime = daysHNSecs + hnsecs;
    }

    version(testStdDateTime) unittest
    {
        foreach(hour; chain(testHours))
        {
            foreach(st; chain(testSysTimesBC, testSysTimesAD))
            {
                auto dt = cast(DateTime)st;
                auto expected = SysTime(DateTime(dt.year, dt.month, dt.day, hour, dt.minute, dt.second),
                                        st.fracSec,
                                        st.timezone);

                st.hour = hour;
                assert(st == expected, format("[%s] [%s]", st, expected));
            }
        }

        auto st = testSysTimesAD[0];
        assertThrown!DateTimeException(st.hour = -1);
        assertThrown!DateTimeException(st.hour = 60);

        const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
        //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
        static assert(!__traits(compiles, cst.hour = 27));
        //static assert(!__traits(compiles, ist.hour = 27));
    }


    /++
        Minutes past the current hour.
     +/
    @property ubyte minute() const nothrow
    {
        auto hnsecs = adjTime;
        auto days = splitUnitsFromHNSecs!"days"(hnsecs) + 1;

        if(hnsecs < 0)
        {
            hnsecs += convert!("hours", "hnsecs")(24);
            --days;
        }

        hnsecs = removeUnitsFromHNSecs!"hours"(hnsecs);

        return cast(ubyte)getUnitsFromHNSecs!"minutes"(hnsecs);
    }

    version(testStdDateTime) unittest
    {
        static void test(SysTime sysTime, int expected, size_t line = __LINE__)
        {
            _assertPred!"=="(sysTime.minute, expected,
                             format("Value given: %s", sysTime), __FILE__, line);
        }

        test(SysTime(0, UTC()), 0);
        test(SysTime(1, UTC()), 0);
        test(SysTime(-1, UTC()), 59);

        foreach(tz; testTZs)
        {
            foreach(year; chain(testYearsBC, testYearsAD))
            {
                foreach(md; testMonthDays)
                {
                    foreach(hour; testHours)
                    {
                        foreach(minute; testMinSecs)
                        {
                            foreach(second; testMinSecs)
                            {
                                auto dt = DateTime(Date(year, md.month, md.day),
                                                   TimeOfDay(hour, minute, second));

                                foreach(fs; testFracSecs)
                                    test(SysTime(dt, fs, tz), minute);
                            }
                        }
                    }
                }
            }
        }

        const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
        //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
        static assert(__traits(compiles, cst.minute));
        //static assert(__traits(compiles, ist.minute));
    }


    /++
        Minutes past the current hour.

        Params:
            minutes = The minute to set this $(D SysTime)'s minute to.

        Throws:
            $(D DateTimeException) if the given minute are not a valid minute
            of an hour.
     +/
    @property void minute(int minute)
    {
        enforceValid!"minutes"(minute);

        auto hnsecs = adjTime;
        auto days = splitUnitsFromHNSecs!"days"(hnsecs);
        immutable daysHNSecs = convert!("days", "hnsecs")(days);
        immutable negative = hnsecs < 0;

        if(negative)
            hnsecs += convert!("hours", "hnsecs")(24);

        immutable hour = splitUnitsFromHNSecs!"hours"(hnsecs);
        hnsecs = removeUnitsFromHNSecs!"minutes"(hnsecs);

        hnsecs += convert!("hours", "hnsecs")(hour);
        hnsecs += convert!("minutes", "hnsecs")(minute);

        if(negative)
            hnsecs -= convert!("hours", "hnsecs")(24);

        adjTime = daysHNSecs + hnsecs;
    }

    version(testStdDateTime) unittest
    {
        foreach(minute; testMinSecs)
        {
            foreach(st; chain(testSysTimesBC, testSysTimesAD))
            {
                auto dt = cast(DateTime)st;
                auto expected = SysTime(DateTime(dt.year, dt.month, dt.day, dt.hour, minute, dt.second),
                                        st.fracSec,
                                        st.timezone);

                st.minute = minute;
                assert(st == expected, format("[%s] [%s]", st, expected));
            }
        }

        auto st = testSysTimesAD[0];
        assertThrown!DateTimeException(st.minute = -1);
        assertThrown!DateTimeException(st.minute = 60);

        const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
        //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
        static assert(!__traits(compiles, cst.minute = 27));
        //static assert(!__traits(compiles, ist.minute = 27));
    }


    /++
        Seconds past the current minute.
     +/
    @property ubyte second() const nothrow
    {
        auto hnsecs = adjTime;
        auto days = splitUnitsFromHNSecs!"days"(hnsecs) + 1;

        if(hnsecs < 0)
        {
            hnsecs += convert!("hours", "hnsecs")(24);
            --days;
        }

        hnsecs = removeUnitsFromHNSecs!"hours"(hnsecs);
        hnsecs = removeUnitsFromHNSecs!"minutes"(hnsecs);

        return cast(ubyte)getUnitsFromHNSecs!"seconds"(hnsecs);
    }

    version(testStdDateTime) unittest
    {
        static void test(SysTime sysTime, int expected, size_t line = __LINE__)
        {
            _assertPred!"=="(sysTime.second, expected,
                             format("Value given: %s", sysTime), __FILE__, line);
        }

        test(SysTime(0, UTC()), 0);
        test(SysTime(1, UTC()), 0);
        test(SysTime(-1, UTC()), 59);

        foreach(tz; testTZs)
        {
            foreach(year; chain(testYearsBC, testYearsAD))
            {
                foreach(md; testMonthDays)
                {
                    foreach(hour; testHours)
                    {
                        foreach(minute; testMinSecs)
                        {
                            foreach(second; testMinSecs)
                            {
                                auto dt = DateTime(Date(year, md.month, md.day),
                                                   TimeOfDay(hour, minute, second));

                                foreach(fs; testFracSecs)
                                    test(SysTime(dt, fs, tz), second);
                            }
                        }
                    }
                }
            }
        }

        const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
        //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
        static assert(__traits(compiles, cst.second));
        //static assert(__traits(compiles, ist.second));
    }


    /++
        Seconds past the current minute.

        Params:
            second = The second to set this $(D SysTime)'s second to.

        Throws:
            $(D DateTimeException) if the given second are not a valid second
            of a minute.
     +/
    @property void second(int second)
    {
        enforceValid!"seconds"(second);

        auto hnsecs = adjTime;
        auto days = splitUnitsFromHNSecs!"days"(hnsecs);
        immutable daysHNSecs = convert!("days", "hnsecs")(days);
        immutable negative = hnsecs < 0;

        if(negative)
            hnsecs += convert!("hours", "hnsecs")(24);

        immutable hour = splitUnitsFromHNSecs!"hours"(hnsecs);
        immutable minute = splitUnitsFromHNSecs!"minutes"(hnsecs);
        hnsecs = removeUnitsFromHNSecs!"seconds"(hnsecs);

        hnsecs += convert!("hours", "hnsecs")(hour);
        hnsecs += convert!("minutes", "hnsecs")(minute);
        hnsecs += convert!("seconds", "hnsecs")(second);

        if(negative)
            hnsecs -= convert!("hours", "hnsecs")(24);

        adjTime = daysHNSecs + hnsecs;
    }

    version(testStdDateTime) unittest
    {
        foreach(second; testMinSecs)
        {
            foreach(st; chain(testSysTimesBC, testSysTimesAD))
            {
                auto dt = cast(DateTime)st;
                auto expected = SysTime(DateTime(dt.year, dt.month, dt.day, dt.hour, dt.minute, second),
                                        st.fracSec,
                                        st.timezone);

                st.second = second;
                assert(st == expected, format("[%s] [%s]", st, expected));
            }
        }

        auto st = testSysTimesAD[0];
        assertThrown!DateTimeException(st.second = -1);
        assertThrown!DateTimeException(st.second = 60);

        const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
        //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
        static assert(!__traits(compiles, cst.seconds = 27));
        //static assert(!__traits(compiles, ist.seconds = 27));
    }


    /++
        Fractional seconds passed the second.
     +/
    @property FracSec fracSec() const nothrow
    {
        try
        {
            auto hnsecs = removeUnitsFromHNSecs!"days"(adjTime);

            if(hnsecs < 0)
                hnsecs += convert!("hours", "hnsecs")(24);

            hnsecs = removeUnitsFromHNSecs!"seconds"(hnsecs);

            return FracSec.from!"hnsecs"(cast(int)hnsecs);
        }
        catch(Exception e)
            assert(0, "FracSec.from!\"hnsecs\"() threw.");
    }

    version(testStdDateTime) unittest
    {
        static void test(SysTime sysTime, FracSec expected, size_t line = __LINE__)
        {
            _assertPred!"=="(sysTime.fracSec, expected,
                             format("Value given: %s", sysTime), __FILE__, line);
        }

        test(SysTime(0, UTC()), FracSec.from!"hnsecs"(0));
        test(SysTime(1, UTC()), FracSec.from!"hnsecs"(1));
        test(SysTime(-1, UTC()), FracSec.from!"hnsecs"(9_999_999));

        foreach(tz; testTZs)
        {
            foreach(year; chain(testYearsBC, testYearsAD))
            {
                foreach(md; testMonthDays)
                {
                    foreach(hour; testHours)
                    {
                        foreach(minute; testMinSecs)
                        {
                            foreach(second; testMinSecs)
                            {
                                auto dt = DateTime(Date(year, md.month, md.day),
                                                   TimeOfDay(hour, minute, second));

                                foreach(fs; testFracSecs)
                                    test(SysTime(dt, fs, tz), fs);
                            }
                        }
                    }
                }
            }
        }

        const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
        //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
        static assert(__traits(compiles, cst.fracSec));
        //static assert(__traits(compiles, ist.fracSec));
    }


    /++
        Fractional seconds passed the second.

        Params:
            fracSec = The fractional seconds to set this $(D SysTimes)'s
                      fractional seconds to.

        Throws:
            $(D DateTimeException) if $(D fracSec) is negative.
     +/
    @property void fracSec(FracSec fracSec)
    {
        immutable fracHNSecs = fracSec.hnsecs;
        enforce(fracHNSecs >= 0, new DateTimeException("A SysTime cannot have negative fractional seconds."));

        auto hnsecs = adjTime;
        auto days = splitUnitsFromHNSecs!"days"(hnsecs);
        immutable daysHNSecs = convert!("days", "hnsecs")(days);
        immutable negative = hnsecs < 0;

        if(negative)
            hnsecs += convert!("hours", "hnsecs")(24);

        immutable hour = splitUnitsFromHNSecs!"hours"(hnsecs);
        immutable minute = splitUnitsFromHNSecs!"minutes"(hnsecs);
        immutable second = getUnitsFromHNSecs!"seconds"(hnsecs);

        hnsecs = fracHNSecs;
        hnsecs += convert!("hours", "hnsecs")(hour);
        hnsecs += convert!("minutes", "hnsecs")(minute);
        hnsecs += convert!("seconds", "hnsecs")(second);

        if(negative)
            hnsecs -= convert!("hours", "hnsecs")(24);

        adjTime = daysHNSecs + hnsecs;
    }

    version(testStdDateTime) unittest
    {
        foreach(fracSec; testFracSecs)
        {
            foreach(st; chain(testSysTimesBC, testSysTimesAD))
            {
                auto dt = cast(DateTime)st;
                auto expected = SysTime(DateTime(dt.year, dt.month, dt.day, dt.hour, dt.minute, dt.second),
                                        fracSec,
                                        st.timezone);

                st.fracSec = fracSec;
                assert(st == expected, format("[%s] [%s]", st, expected));
            }
        }

        auto st = testSysTimesAD[0];
        assertThrown!DateTimeException(st.fracSec = FracSec.from!"hnsecs"(-1));

        const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
        //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
        static assert(!__traits(compiles, cst.fracSec = FracSec.from!"msecs"(7)));
        //static assert(!__traits(compiles, ist.fracSec = FracSec.from!"msecs"(7)));
    }


    /++
        The total hnsecs from midnight, January 1st, 1 A.D. UTC. This is the
        internal representation of $(D SysTime).
     +/
    @property long stdTime() const pure nothrow
    {
        return _stdTime;
    }

    version(testStdDateTime) unittest
    {
        _assertPred!"=="(SysTime(0).stdTime, 0);
        _assertPred!"=="(SysTime(1).stdTime, 1);
        _assertPred!"=="(SysTime(-1).stdTime, -1);
        _assertPred!"=="(SysTime(DateTime(1, 1, 1, 0, 0, 33), FracSec.from!"hnsecs"(502), UTC()).stdTime,
                         330000502L);
        _assertPred!"=="(SysTime(DateTime(1970, 1, 1, 0, 0, 0), UTC()).stdTime,
                         621355968000000000L);

        const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
        //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
        static assert(__traits(compiles, cst.stdTime));
        //static assert(__traits(compiles, ist.stdTime));
    }


    /++
        The total hnsecs from midnight, January 1st, 1 A.D. UTC. This is the
        internal representation of $(D SysTime).

        Params:
            stdTime = The number of hnsecs since January 1st, 1 A.D. UTC.
     +/
    @property void stdTime(long stdTime) pure nothrow
    {
        _stdTime = stdTime;
    }

    version(testStdDateTime) unittest
    {
        static void test(long stdTime, in SysTime expected, size_t line = __LINE__)
        {
            auto st = SysTime(0, UTC());
            st.stdTime = stdTime;
            _assertPred!"=="(st, expected);
        }

        test(0, SysTime(Date(1, 1, 1), UTC()));
        test(1, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(1), UTC()));
        test(-1, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999), UTC()));
        test(330_000_502L, SysTime(DateTime(1, 1, 1, 0, 0, 33), FracSec.from!"hnsecs"(502), UTC()));
        test(621_355_968_000_000_000L, SysTime(DateTime(1970, 1, 1, 0, 0, 0), UTC()));

        const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
        //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
        static assert(!__traits(compiles, cst.stdTime = 27));
        //static assert(!__traits(compiles, ist.stdTime = 27));
    }


    /++
        The current time zone of this $(D SysTime). Its internal time is always
        kept in UTC, so there are no conversion issues between time zones due to
        DST. Functions which return all or part of the time - such as hours -
        adjust the time to this $(D SysTime)'s time zone before returning.
      +/
    @property immutable(TimeZone) timezone() const pure nothrow
    {
        return _timezone;
    }


    /++
        The current time zone of this $(D SysTime). It's internal time is always
        kept in UTC, so there are no conversion issues between time zones due to
        DST. Functions which return all or part of the time - such as hours -
        adjust the time to this $(D SysTime)'s time zone before returning.

        Params:
            tz = The $(D TimeZone) to set this $(D SysTime)'s time zone to.
      +/
    @property void timezone(immutable TimeZone timezone) pure nothrow
    {
        if(timezone is null)
            _timezone = LocalTime();
        else
            _timezone = timezone;
    }


    /++
        Returns whether DST is in effect for this $(D SysTime).
      +/
    @property bool dstInEffect() const nothrow
    {
        return _timezone.dstInEffect(_stdTime);
        //This function's unit testing is done in the time zone classes.
    }


    /++
        Returns what the offset from UTC is for this $(D SysTime).
        It includes the DST offset in effect at that time (if any).
      +/
    @property Duration utcOffset() const nothrow
    {
        return _timezone.utcOffsetAt(_stdTime);
    }


    /++
        Returns a $(D SysTime) with the same std time as this one, but with
        $(D LocalTime) as its time zone.
      +/
    SysTime toLocalTime() const nothrow
    {
        return SysTime(_stdTime, LocalTime());
    }

    unittest
    {
        version(testStdDateTime)
        {
            {
                auto sysTime = SysTime(DateTime(1982, 1, 4, 8, 59, 7), FracSec.from!"hnsecs"(27));
                _assertPred!"=="(sysTime, sysTime.toLocalTime());
                _assertPred!"=="(sysTime._stdTime, sysTime.toLocalTime()._stdTime);
                assert(sysTime.toLocalTime().timezone is LocalTime());
                assert(sysTime.toLocalTime().timezone is sysTime.timezone);
                assert(sysTime.toLocalTime().timezone !is UTC());
            }

            {
                immutable stz = new SimpleTimeZone(-3 * 60);
                auto sysTime = SysTime(DateTime(1982, 1, 4, 8, 59, 7), FracSec.from!"hnsecs"(27), stz);
                _assertPred!"=="(sysTime, sysTime.toLocalTime());
                _assertPred!"=="(sysTime._stdTime, sysTime.toLocalTime()._stdTime);
                assert(sysTime.toLocalTime().timezone is LocalTime());
                assert(sysTime.toLocalTime().timezone !is UTC());
                assert(sysTime.toLocalTime().timezone !is stz);
            }
        }
    }


    /++
        Returns a $(D SysTime) with the same std time as this one, but with
        $(D UTC) as its time zone.
      +/
    SysTime toUTC() const pure nothrow
    {
        return SysTime(_stdTime, UTC());
    }

    unittest
    {
        version(testStdDateTime)
        {
            auto sysTime = SysTime(DateTime(1982, 1, 4, 8, 59, 7), FracSec.from!"hnsecs"(27));
            _assertPred!"=="(sysTime, sysTime.toUTC());
            _assertPred!"=="(sysTime._stdTime, sysTime.toUTC()._stdTime);
            assert(sysTime.toUTC().timezone is UTC());
            assert(sysTime.toUTC().timezone !is LocalTime());
            assert(sysTime.toUTC().timezone !is sysTime.timezone);
        }
    }


    /++
        Returns a $(D SysTime) with the same std time as this one, but with
        given time zone as its time zone.
      +/
    SysTime toOtherTZ(immutable TimeZone tz) const pure nothrow
    {
        if(tz is null)
            return SysTime(_stdTime, LocalTime());
        else
            return SysTime(_stdTime, tz);
    }

    unittest
    {
        version(testStdDateTime)
        {
            immutable stz = new SimpleTimeZone(11 * 60);
            auto sysTime = SysTime(DateTime(1982, 1, 4, 8, 59, 7), FracSec.from!"hnsecs"(27));
            _assertPred!"=="(sysTime, sysTime.toOtherTZ(stz));
            _assertPred!"=="(sysTime._stdTime, sysTime.toOtherTZ(stz)._stdTime);
            assert(sysTime.toOtherTZ(stz).timezone is stz);
            assert(sysTime.toOtherTZ(stz).timezone !is LocalTime());
            assert(sysTime.toOtherTZ(stz).timezone !is UTC());
        }
    }


    /++
        Returns a $(D time_t) which represents the same time as this
        $(D SysTime).

        Note that like all conversions in std.datetime, this is a truncating
        conversion.

        If $(D time_t) is 32 bits, rather than 64, and the result can't fit in a
        32-bit value, then the closest value that can be held in 32 bits will be
        used (so $(D time_t.max) if it goes over and $(D time_t.min) if it goes
        under).
      +/
    time_t toUnixTime() const pure nothrow
    {
        return stdTimeToUnixTime(_stdTime);
    }

    unittest
    {
        version(testStdDateTime)
        {
            _assertPred!"=="(SysTime(DateTime(1970, 1, 1), UTC()).toUnixTime(), 0);
            _assertPred!"=="(SysTime(DateTime(1970, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(1), UTC()).toUnixTime(), 0);
            _assertPred!"=="(SysTime(DateTime(1970, 1, 1, 0, 0, 0), FracSec.from!"usecs"(1), UTC()).toUnixTime(), 0);
            _assertPred!"=="(SysTime(DateTime(1970, 1, 1, 0, 0, 0), FracSec.from!"msecs"(1), UTC()).toUnixTime(), 0);
            _assertPred!"=="(SysTime(DateTime(1970, 1, 1, 0, 0, 1), UTC()).toUnixTime(), 1);
            _assertPred!"=="(SysTime(DateTime(1969, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999), UTC()).toUnixTime(), 0);
            _assertPred!"=="(SysTime(DateTime(1969, 12, 31, 23, 59, 59), FracSec.from!"usecs"(999_999), UTC()).toUnixTime(), 0);
            _assertPred!"=="(SysTime(DateTime(1969, 12, 31, 23, 59, 59), FracSec.from!"msecs"(999), UTC()).toUnixTime(), 0);
            _assertPred!"=="(SysTime(DateTime(1969, 12, 31, 23, 59, 59), UTC()).toUnixTime(), -1);
        }
    }


    /++
        Returns a $(D timeval) which represents this $(D SysTime).

        Note that like all conversions in std.datetime, this is a truncating
        conversion.

        If $(D time_t) is 32 bits, rather than 64, and the result can't fit in a
        32-bit value, then the closest value that can be held in 32 bits will be
        used for $(D tv_sec). (so $(D time_t.max) if it goes over and
        $(D time_t.min) if it goes under).
      +/
    timeval toTimeVal() const pure nothrow
    {
        immutable tv_sec = toUnixTime();

        immutable fracHNSecs = removeUnitsFromHNSecs!"seconds"(_stdTime - 621355968000000000L);
        immutable tv_usec = cast(int)convert!("hnsecs", "usecs")(fracHNSecs);

        return timeval(tv_sec, tv_usec);
    }

    unittest
    {
        version(testStdDateTime)
        {
            assert(SysTime(DateTime(1970, 1, 1), UTC()).toTimeVal() == timeval(0, 0));
            assert(SysTime(DateTime(1970, 1, 1), FracSec.from!"hnsecs"(9), UTC()).toTimeVal() == timeval(0, 0));
            assert(SysTime(DateTime(1970, 1, 1), FracSec.from!"hnsecs"(10), UTC()).toTimeVal() == timeval(0, 1));
            assert(SysTime(DateTime(1970, 1, 1), FracSec.from!"usecs"(7), UTC()).toTimeVal() == timeval(0, 7));

            assert(SysTime(DateTime(1970, 1, 1, 0, 0, 1), UTC()).toTimeVal() == timeval(1, 0));
            assert(SysTime(DateTime(1970, 1, 1, 0, 0, 1), FracSec.from!"hnsecs"(9), UTC()).toTimeVal() == timeval(1, 0));
            assert(SysTime(DateTime(1970, 1, 1, 0, 0, 1), FracSec.from!"hnsecs"(10), UTC()).toTimeVal() == timeval(1, 1));
            assert(SysTime(DateTime(1970, 1, 1, 0, 0, 1), FracSec.from!"usecs"(7), UTC()).toTimeVal() == timeval(1, 7));

            assert(SysTime(DateTime(1969, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999), UTC()).toTimeVal() ==
                   timeval(0, 0));
            assert(SysTime(DateTime(1969, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_990), UTC()).toTimeVal() ==
                   timeval(0, -1));

            assert(SysTime(DateTime(1969, 12, 31, 23, 59, 59), FracSec.from!"usecs"(999_999), UTC()).toTimeVal() ==
                   timeval(0, -1));
            assert(SysTime(DateTime(1969, 12, 31, 23, 59, 59), FracSec.from!"usecs"(999), UTC()).toTimeVal() ==
                   timeval(0, -999_001));
            assert(SysTime(DateTime(1969, 12, 31, 23, 59, 59), FracSec.from!"msecs"(999), UTC()).toTimeVal() ==
                   timeval(0, -1000));
            assert(SysTime(DateTime(1969, 12, 31, 23, 59, 59), UTC()).toTimeVal() == timeval(-1, 0));
            assert(SysTime(DateTime(1969, 12, 31, 23, 59, 58), FracSec.from!"usecs"(17), UTC()).toTimeVal() ==
                   timeval(-1, -999_983));
        }
    }


    /++
        Returns a $(D tm) which represents this $(D SysTime).
      +/
    tm toTM() const nothrow
    {
        try
        {
            auto dateTime = cast(DateTime)this;
            tm timeInfo;

            timeInfo.tm_sec = dateTime.second;
            timeInfo.tm_min = dateTime.minute;
            timeInfo.tm_hour = dateTime.hour;
            timeInfo.tm_mday = dateTime.day;
            timeInfo.tm_mon = dateTime.month - 1;
            timeInfo.tm_year = dateTime.year - 1900;
            timeInfo.tm_wday = dateTime.dayOfWeek;
            timeInfo.tm_yday = dateTime.dayOfYear - 1;
            timeInfo.tm_isdst = _timezone.dstInEffect(_stdTime);

            version(Posix)
            {
                char[] zone = (timeInfo.tm_isdst ? _timezone.dstName : _timezone.stdName).dup;
                zone ~= "\0";

                timeInfo.tm_gmtoff = cast(int)convert!("hnsecs", "seconds")(adjTime - _stdTime);
                timeInfo.tm_zone = zone.ptr;
            }

            return timeInfo;
        }
        catch(Exception e)
            assert(0, "Either DateTime's constructor threw.");
    }

    unittest
    {
        version(testStdDateTime)
        {
            version(Posix)
            {
                scope(exit) clearTZEnvVar();
                setTZEnvVar("America/Los_Angeles");
            }

            {
                auto timeInfo = SysTime(DateTime(1970, 1, 1)).toTM();

                _assertPred!"=="(timeInfo.tm_sec, 0);
                _assertPred!"=="(timeInfo.tm_min, 0);
                _assertPred!"=="(timeInfo.tm_hour, 0);
                _assertPred!"=="(timeInfo.tm_mday, 1);
                _assertPred!"=="(timeInfo.tm_mon, 0);
                _assertPred!"=="(timeInfo.tm_year, 70);
                _assertPred!"=="(timeInfo.tm_wday, 4);
                _assertPred!"=="(timeInfo.tm_yday, 0);

                version(Posix)
                    _assertPred!"=="(timeInfo.tm_isdst, 0);
                else version(Windows)
                    assert(timeInfo.tm_isdst == 0 || timeInfo.tm_isdst == 1);

                version(Posix)
                {
                    _assertPred!"=="(timeInfo.tm_gmtoff, -8 * 60 * 60);
                    _assertPred!"=="(to!string(timeInfo.tm_zone), "PST");
                }
            }

            {
                auto timeInfo = SysTime(DateTime(2010, 7, 4, 12, 15, 7), FracSec.from!"hnsecs"(15)).toTM();

                _assertPred!"=="(timeInfo.tm_sec, 7);
                _assertPred!"=="(timeInfo.tm_min, 15);
                _assertPred!"=="(timeInfo.tm_hour, 12);
                _assertPred!"=="(timeInfo.tm_mday, 4);
                _assertPred!"=="(timeInfo.tm_mon, 6);
                _assertPred!"=="(timeInfo.tm_year, 110);
                _assertPred!"=="(timeInfo.tm_wday, 0);
                _assertPred!"=="(timeInfo.tm_yday, 184);

                version(Posix)
                    _assertPred!"=="(timeInfo.tm_isdst, 1);
                else version(Windows)
                    assert(timeInfo.tm_isdst == 0 || timeInfo.tm_isdst == 1);

                version(Posix)
                {
                    _assertPred!"=="(timeInfo.tm_gmtoff, -7 * 60 * 60);
                    _assertPred!"=="(to!string(timeInfo.tm_zone), "PDT");
                }
            }
        }
    }


    /++
        Adds the given number of years or months to this $(D SysTime). A
        negative number will subtract.

        Note that if day overflow is allowed, and the date with the adjusted
        year/month overflows the number of days in the new month, then the month
        will be incremented by one, and the day set to the number of days
        overflowed. (e.g. if the day were 31 and the new month were June, then
        the month would be incremented to July, and the new day would be 1). If
        day overflow is not allowed, then the day will be set to the last valid
        day in the month (e.g. June 31st would become June 30th).

        Params:
            units         = The type of units to add ("years" or "months").
            value         = The number of months or years to add to this
                            $(D SysTime).
            allowOverflow = Whether the days should be allowed to overflow,
                            causing the month to increment.

        Examples:
--------------------
auto st1 = SysTime(DateTime(2010, 1, 1, 12, 30, 33));
st1.add!"months"(11);
assert(st1 == SysTime(DateTime(2010, 12, 1, 12, 30, 33)));

auto st2 = SysTime(DateTime(2010, 1, 1, 12, 30, 33));
st2.add!"months"(-11);
assert(st2 == SysTime(DateTime(2009, 2, 1, 12, 30, 33)));

auto st3 = SysTime(DateTime(2000, 2, 29, 12, 30, 33));
st3.add!"years"(1);
assert(st3 == SysTime(DateTime(2001, 3, 1, 12, 30, 33)));

auto st4 = SysTime(DateTime(2000, 2, 29, 12, 30, 33));
st4.add!"years"(1, AllowDayOverflow.no);
assert(st4 == SysTime(DateTime(2001, 2, 28, 12, 30, 33)));
--------------------
      +/
    ref SysTime add(string units)(long value, AllowDayOverflow allowOverflow = AllowDayOverflow.yes) nothrow
        if(units == "years" ||
           units == "months")
    {
        auto hnsecs = adjTime;
        auto days = splitUnitsFromHNSecs!"days"(hnsecs) + 1;

        if(hnsecs < 0)
        {
            hnsecs += convert!("hours", "hnsecs")(24);
            --days;
        }

        auto date = Date(cast(int)days);
        date.add!units(value, allowOverflow);
        days = date.dayOfGregorianCal - 1;

        if(days < 0)
        {
            hnsecs -= convert!("hours", "hnsecs")(24);
            ++days;
        }

        immutable newDaysHNSecs = convert!("days", "hnsecs")(days);

        adjTime = newDaysHNSecs + hnsecs;

        return this;
    }

    //Verify Examples.
    unittest
    {
        version (testStdDateTime)
        {
            auto st1 = SysTime(DateTime(2010, 1, 1, 12, 30, 33));
            st1.add!"months"(11);
            assert(st1 == SysTime(DateTime(2010, 12, 1, 12, 30, 33)));

            auto st2 = SysTime(DateTime(2010, 1, 1, 12, 30, 33));
            st2.add!"months"(-11);
            assert(st2 == SysTime(DateTime(2009, 2, 1, 12, 30, 33)));

            auto st3 = SysTime(DateTime(2000, 2, 29, 12, 30, 33));
            st3.add!"years"(1);
            assert(st3 == SysTime(DateTime(2001, 3, 1, 12, 30, 33)));

            auto st4 = SysTime(DateTime(2000, 2, 29, 12, 30, 33));
            st4.add!"years"(1, AllowDayOverflow.no);
            assert(st4 == SysTime(DateTime(2001, 2, 28, 12, 30, 33)));
        }
    }

    //Test add!"years"() with AllowDayOverlow.yes
    unittest
    {
        version(testStdDateTime)
        {
            //Test A.D.
            {
                auto sysTime = SysTime(Date(1999, 7, 6));
                sysTime.add!"years"(7);
                _assertPred!"=="(sysTime, SysTime(Date(2006, 7, 6)));
                sysTime.add!"years"(-9);
                _assertPred!"=="(sysTime, SysTime(Date(1997, 7, 6)));
            }

            {
                auto sysTime = SysTime(Date(1999, 2, 28));
                sysTime.add!"years"(1);
                _assertPred!"=="(sysTime, SysTime(Date(2000, 2, 28)));
            }

            {
                auto sysTime = SysTime(Date(2000, 2, 29));
                sysTime.add!"years"(-1);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 3, 1)));
            }

            {
                auto sysTime = SysTime(DateTime(1999, 7, 6, 12, 7, 3), FracSec.from!"msecs"(234));
                sysTime.add!"years"(7);
                _assertPred!"=="(sysTime, SysTime(DateTime(2006, 7, 6, 12, 7, 3), FracSec.from!"msecs"(234)));
                sysTime.add!"years"(-9);
                _assertPred!"=="(sysTime, SysTime(DateTime(1997, 7, 6, 12, 7, 3), FracSec.from!"msecs"(234)));
            }

            {
                auto sysTime = SysTime(DateTime(1999, 2, 28, 0, 7, 2), FracSec.from!"usecs"(1207));
                sysTime.add!"years"(1);
                _assertPred!"=="(sysTime, SysTime(DateTime(2000, 2, 28, 0, 7, 2), FracSec.from!"usecs"(1207)));
            }

            {
                auto sysTime = SysTime(DateTime(2000, 2, 29, 0, 7, 2), FracSec.from!"usecs"(1207));
                sysTime.add!"years"(-1);
                _assertPred!"=="(sysTime, SysTime(DateTime(1999, 3, 1, 0, 7, 2), FracSec.from!"usecs"(1207)));
            }

            //Test B.C.
            {
                auto sysTime = SysTime(Date(-1999, 7, 6));
                sysTime.add!"years"(-7);
                _assertPred!"=="(sysTime, SysTime(Date(-2006, 7, 6)));
                sysTime.add!"years"(9);
                _assertPred!"=="(sysTime, SysTime(Date(-1997, 7, 6)));
            }

            {
                auto sysTime = SysTime(Date(-1999, 2, 28));
                sysTime.add!"years"(-1);
                _assertPred!"=="(sysTime, SysTime(Date(-2000, 2, 28)));
            }

            {
                auto sysTime = SysTime(Date(-2000, 2, 29));
                sysTime.add!"years"(1);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 3, 1)));
            }

            {
                auto sysTime = SysTime(DateTime(-1999, 7, 6, 12, 7, 3), FracSec.from!"msecs"(234));
                sysTime.add!"years"(-7);
                _assertPred!"=="(sysTime, SysTime(DateTime(-2006, 7, 6, 12, 7, 3), FracSec.from!"msecs"(234)));
                sysTime.add!"years"(9);
                _assertPred!"=="(sysTime, SysTime(DateTime(-1997, 7, 6, 12, 7, 3), FracSec.from!"msecs"(234)));
            }

            {
                auto sysTime = SysTime(DateTime(-1999, 2, 28, 3, 3, 3), FracSec.from!"hnsecs"(3));
                sysTime.add!"years"(-1);
                _assertPred!"=="(sysTime, SysTime(DateTime(-2000, 2, 28, 3, 3, 3), FracSec.from!"hnsecs"(3)));
            }

            {
                auto sysTime = SysTime(DateTime(-2000, 2, 29, 3, 3, 3), FracSec.from!"hnsecs"(3));
                sysTime.add!"years"(1);
                _assertPred!"=="(sysTime, SysTime(DateTime(-1999, 3, 1, 3, 3, 3), FracSec.from!"hnsecs"(3)));
            }

            //Test Both
            {
                auto sysTime = SysTime(Date(4, 7, 6));
                sysTime.add!"years"(-5);
                _assertPred!"=="(sysTime, SysTime(Date(-1, 7, 6)));
                sysTime.add!"years"(5);
                _assertPred!"=="(sysTime, SysTime(Date(4, 7, 6)));
            }

            {
                auto sysTime = SysTime(Date(-4, 7, 6));
                sysTime.add!"years"(5);
                _assertPred!"=="(sysTime, SysTime(Date(1, 7, 6)));
                sysTime.add!"years"(-5);
                _assertPred!"=="(sysTime, SysTime(Date(-4, 7, 6)));
            }

            {
                auto sysTime = SysTime(Date(4, 7, 6));
                sysTime.add!"years"(-8);
                _assertPred!"=="(sysTime, SysTime(Date(-4, 7, 6)));
                sysTime.add!"years"(8);
                _assertPred!"=="(sysTime, SysTime(Date(4, 7, 6)));
            }

            {
                auto sysTime = SysTime(Date(-4, 7, 6));
                sysTime.add!"years"(8);
                _assertPred!"=="(sysTime, SysTime(Date(4, 7, 6)));
                sysTime.add!"years"(-8);
                _assertPred!"=="(sysTime, SysTime(Date(-4, 7, 6)));
            }

            {
                auto sysTime = SysTime(Date(-4, 2, 29));
                sysTime.add!"years"(5);
                _assertPred!"=="(sysTime, SysTime(Date(1, 3, 1)));
            }

            {
                auto sysTime = SysTime(Date(4, 2, 29));
                sysTime.add!"years"(-5);
                _assertPred!"=="(sysTime, SysTime(Date(-1, 3, 1)));
            }

            {
                auto sysTime = SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0));
                sysTime.add!"years"(-1);
                _assertPred!"=="(sysTime, SysTime(DateTime(0, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)));
                sysTime.add!"years"(1);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)));
            }

            {
                auto sysTime = SysTime(DateTime(1, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999));
                sysTime.add!"years"(-1);
                _assertPred!"=="(sysTime, SysTime(DateTime(0, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
                sysTime.add!"years"(1);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            }

            {
                auto sysTime = SysTime(DateTime(0, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0));
                sysTime.add!"years"(1);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)));
                sysTime.add!"years"(-1);
                _assertPred!"=="(sysTime, SysTime(DateTime(0, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)));
            }

            {
                auto sysTime = SysTime(DateTime(0, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999));
                sysTime.add!"years"(1);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
                sysTime.add!"years"(-1);
                _assertPred!"=="(sysTime, SysTime(DateTime(0, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            }

            {
                auto sysTime = SysTime(DateTime(4, 7, 6, 14, 7, 1), FracSec.from!"usecs"(54329));
                sysTime.add!"years"(-5);
                _assertPred!"=="(sysTime, SysTime(DateTime(-1, 7, 6, 14, 7, 1), FracSec.from!"usecs"(54329)));
                sysTime.add!"years"(5);
                _assertPred!"=="(sysTime, SysTime(DateTime(4, 7, 6, 14, 7, 1), FracSec.from!"usecs"(54329)));
            }

            {
                auto sysTime = SysTime(DateTime(-4, 7, 6, 14, 7, 1), FracSec.from!"usecs"(54329));
                sysTime.add!"years"(5);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 7, 6, 14, 7, 1), FracSec.from!"usecs"(54329)));
                sysTime.add!"years"(-5);
                _assertPred!"=="(sysTime, SysTime(DateTime(-4, 7, 6, 14, 7, 1), FracSec.from!"usecs"(54329)));
            }

            {
                auto sysTime = SysTime(DateTime(-4, 2, 29, 5, 5, 5), FracSec.from!"msecs"(555));
                sysTime.add!"years"(5);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 3, 1, 5, 5, 5), FracSec.from!"msecs"(555)));
            }

            {
                auto sysTime = SysTime(DateTime(4, 2, 29, 5, 5, 5), FracSec.from!"msecs"(555));
                sysTime.add!"years"(-5);
                _assertPred!"=="(sysTime, SysTime(DateTime(-1, 3, 1, 5, 5, 5), FracSec.from!"msecs"(555)));
            }

            const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            static assert(!__traits(compiles, cst.add!"years"(4)));
            //static assert(!__traits(compiles, ist.add!"years"(4)));
        }
    }

    //Test add!"years"() with AllowDayOverlow.no
    unittest
    {
        version(testStdDateTime)
        {
            //Test A.D.
            {
                auto sysTime = SysTime(Date(1999, 7, 6));
                sysTime.add!"years"(7, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(2006, 7, 6)));
                sysTime.add!"years"(-9, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(1997, 7, 6)));
            }

            {
                auto sysTime = SysTime(Date(1999, 2, 28));
                sysTime.add!"years"(1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(2000, 2, 28)));
            }

            {
                auto sysTime = SysTime(Date(2000, 2, 29));
                sysTime.add!"years"(-1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 2, 28)));
            }

            {
                auto sysTime = SysTime(DateTime(1999, 7, 6, 12, 7, 3), FracSec.from!"msecs"(234));
                sysTime.add!"years"(7, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(2006, 7, 6, 12, 7, 3), FracSec.from!"msecs"(234)));
                sysTime.add!"years"(-9, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(1997, 7, 6, 12, 7, 3), FracSec.from!"msecs"(234)));
            }

            {
                auto sysTime = SysTime(DateTime(1999, 2, 28, 0, 7, 2), FracSec.from!"usecs"(1207));
                sysTime.add!"years"(1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(2000, 2, 28, 0, 7, 2), FracSec.from!"usecs"(1207)));
            }

            {
                auto sysTime = SysTime(DateTime(2000, 2, 29, 0, 7, 2), FracSec.from!"usecs"(1207));
                sysTime.add!"years"(-1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(1999, 2, 28, 0, 7, 2), FracSec.from!"usecs"(1207)));
            }

            //Test B.C.
            {
                auto sysTime = SysTime(Date(-1999, 7, 6));
                sysTime.add!"years"(-7, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-2006, 7, 6)));
                sysTime.add!"years"(9, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-1997, 7, 6)));
            }

            {
                auto sysTime = SysTime(Date(-1999, 2, 28));
                sysTime.add!"years"(-1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-2000, 2, 28)));
            }

            {
                auto sysTime = SysTime(Date(-2000, 2, 29));
                sysTime.add!"years"(1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 2, 28)));
            }

            {
                auto sysTime = SysTime(DateTime(-1999, 7, 6, 12, 7, 3), FracSec.from!"msecs"(234));
                sysTime.add!"years"(-7, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(-2006, 7, 6, 12, 7, 3), FracSec.from!"msecs"(234)));
                sysTime.add!"years"(9, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(-1997, 7, 6, 12, 7, 3), FracSec.from!"msecs"(234)));
            }

            {
                auto sysTime = SysTime(DateTime(-1999, 2, 28, 3, 3, 3), FracSec.from!"hnsecs"(3));
                sysTime.add!"years"(-1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(-2000, 2, 28, 3, 3, 3), FracSec.from!"hnsecs"(3)));
            }

            {
                auto sysTime = SysTime(DateTime(-2000, 2, 29, 3, 3, 3), FracSec.from!"hnsecs"(3));
                sysTime.add!"years"(1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(-1999, 2, 28, 3, 3, 3), FracSec.from!"hnsecs"(3)));
            }

            //Test Both
            {
                auto sysTime = SysTime(Date(4, 7, 6));
                sysTime.add!"years"(-5, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-1, 7, 6)));
                sysTime.add!"years"(5, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(4, 7, 6)));
            }

            {
                auto sysTime = SysTime(Date(-4, 7, 6));
                sysTime.add!"years"(5, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(1, 7, 6)));
                sysTime.add!"years"(-5, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-4, 7, 6)));
            }

            {
                auto sysTime = SysTime(Date(4, 7, 6));
                sysTime.add!"years"(-8, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-4, 7, 6)));
                sysTime.add!"years"(8, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(4, 7, 6)));
            }

            {
                auto sysTime = SysTime(Date(-4, 7, 6));
                sysTime.add!"years"(8, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(4, 7, 6)));
                sysTime.add!"years"(-8, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-4, 7, 6)));
            }

            {
                auto sysTime = SysTime(Date(-4, 2, 29));
                sysTime.add!"years"(5, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(1, 2, 28)));
            }

            {
                auto sysTime = SysTime(Date(4, 2, 29));
                sysTime.add!"years"(-5, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-1, 2, 28)));
            }

            {
                auto sysTime = SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0));
                sysTime.add!"years"(-1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(0, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)));
                sysTime.add!"years"(1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)));
            }

            {
                auto sysTime = SysTime(DateTime(1, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999));
                sysTime.add!"years"(-1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(0, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
                sysTime.add!"years"(1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            }

            {
                auto sysTime = SysTime(DateTime(0, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0));
                sysTime.add!"years"(1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)));
                sysTime.add!"years"(-1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(0, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)));
            }

            {
                auto sysTime = SysTime(DateTime(0, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999));
                sysTime.add!"years"(1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
                sysTime.add!"years"(-1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(0, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            }

            {
                auto sysTime = SysTime(DateTime(4, 7, 6, 14, 7, 1), FracSec.from!"usecs"(54329));
                sysTime.add!"years"(-5);
                _assertPred!"=="(sysTime, SysTime(DateTime(-1, 7, 6, 14, 7, 1), FracSec.from!"usecs"(54329)));
                sysTime.add!"years"(5);
                _assertPred!"=="(sysTime, SysTime(DateTime(4, 7, 6, 14, 7, 1), FracSec.from!"usecs"(54329)));
            }

            {
                auto sysTime = SysTime(DateTime(4, 7, 6, 14, 7, 1), FracSec.from!"usecs"(54329));
                sysTime.add!"years"(-5, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(-1, 7, 6, 14, 7, 1), FracSec.from!"usecs"(54329)));
                sysTime.add!"years"(5, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(4, 7, 6, 14, 7, 1), FracSec.from!"usecs"(54329)));
            }

            {
                auto sysTime = SysTime(DateTime(-4, 7, 6, 14, 7, 1), FracSec.from!"usecs"(54329));
                sysTime.add!"years"(5, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 7, 6, 14, 7, 1), FracSec.from!"usecs"(54329)));
                sysTime.add!"years"(-5, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(-4, 7, 6, 14, 7, 1), FracSec.from!"usecs"(54329)));
            }

            {
                auto sysTime = SysTime(DateTime(-4, 2, 29, 5, 5, 5), FracSec.from!"msecs"(555));
                sysTime.add!"years"(5, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 2, 28, 5, 5, 5), FracSec.from!"msecs"(555)));
            }

            {
                auto sysTime = SysTime(DateTime(4, 2, 29, 5, 5, 5), FracSec.from!"msecs"(555));
                sysTime.add!"years"(-5, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(-1, 2, 28, 5, 5, 5), FracSec.from!"msecs"(555)));
            }
        }
    }

    //Test add!"months"() with AllowDayOverlow.yes
    unittest
    {
        version(testStdDateTime)
        {
            //Test A.D.
            {
                auto sysTime = SysTime(Date(1999, 7, 6));
                sysTime.add!"months"(3);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 10, 6)));
                sysTime.add!"months"(-4);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 6, 6)));
            }

            {
                auto sysTime = SysTime(Date(1999, 7, 6));
                sysTime.add!"months"(6);
                _assertPred!"=="(sysTime, SysTime(Date(2000, 1, 6)));
                sysTime.add!"months"(-6);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 7, 6)));
            }

            {
                auto sysTime = SysTime(Date(1999, 7, 6));
                sysTime.add!"months"(27);
                _assertPred!"=="(sysTime, SysTime(Date(2001, 10, 6)));
                sysTime.add!"months"(-28);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 6, 6)));
            }

            {
                auto sysTime = SysTime(Date(1999, 5, 31));
                sysTime.add!"months"(1);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 7, 1)));
            }

            {
                auto sysTime = SysTime(Date(1999, 5, 31));
                sysTime.add!"months"(-1);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 5, 1)));
            }

            {
                auto sysTime = SysTime(Date(1999, 2, 28));
                sysTime.add!"months"(12);
                _assertPred!"=="(sysTime, SysTime(Date(2000, 2, 28)));
            }

            {
                auto sysTime = SysTime(Date(2000, 2, 29));
                sysTime.add!"months"(12);
                _assertPred!"=="(sysTime, SysTime(Date(2001, 3, 1)));
            }

            {
                auto sysTime = SysTime(Date(1999, 7, 31));
                sysTime.add!"months"(1);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 8, 31)));
                sysTime.add!"months"(1);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 10, 1)));
            }

            {
                auto sysTime = SysTime(Date(1998, 8, 31));
                sysTime.add!"months"(13);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 10, 1)));
                sysTime.add!"months"(-13);
                _assertPred!"=="(sysTime, SysTime(Date(1998, 9, 1)));
            }

            {
                auto sysTime = SysTime(Date(1997, 12, 31));
                sysTime.add!"months"(13);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 1, 31)));
                sysTime.add!"months"(-13);
                _assertPred!"=="(sysTime, SysTime(Date(1997, 12, 31)));
            }

            {
                auto sysTime = SysTime(Date(1997, 12, 31));
                sysTime.add!"months"(14);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 3, 3)));
                sysTime.add!"months"(-14);
                _assertPred!"=="(sysTime, SysTime(Date(1998, 1, 3)));
            }

            {
                auto sysTime = SysTime(Date(1998, 12, 31));
                sysTime.add!"months"(14);
                _assertPred!"=="(sysTime, SysTime(Date(2000, 3, 2)));
                sysTime.add!"months"(-14);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 1, 2)));
            }

            {
                auto sysTime = SysTime(Date(1999, 12, 31));
                sysTime.add!"months"(14);
                _assertPred!"=="(sysTime, SysTime(Date(2001, 3, 3)));
                sysTime.add!"months"(-14);
                _assertPred!"=="(sysTime, SysTime(Date(2000, 1, 3)));
            }

            {
                auto sysTime = SysTime(DateTime(1999, 7, 6, 12, 2, 7), FracSec.from!"usecs"(5007));
                sysTime.add!"months"(3);
                _assertPred!"=="(sysTime, SysTime(DateTime(1999, 10, 6, 12, 2, 7), FracSec.from!"usecs"(5007)));
                sysTime.add!"months"(-4);
                _assertPred!"=="(sysTime, SysTime(DateTime(1999, 6, 6, 12, 2, 7), FracSec.from!"usecs"(5007)));
            }

            {
                auto sysTime = SysTime(DateTime(1998, 12, 31, 7, 7, 7), FracSec.from!"hnsecs"(422202));
                sysTime.add!"months"(14);
                _assertPred!"=="(sysTime, SysTime(DateTime(2000, 3, 2, 7, 7, 7), FracSec.from!"hnsecs"(422202)));
                sysTime.add!"months"(-14);
                _assertPred!"=="(sysTime, SysTime(DateTime(1999, 1, 2, 7, 7, 7), FracSec.from!"hnsecs"(422202)));
            }

            {
                auto sysTime = SysTime(DateTime(1999, 12, 31, 7, 7, 7), FracSec.from!"hnsecs"(422202));
                sysTime.add!"months"(14);
                _assertPred!"=="(sysTime, SysTime(DateTime(2001, 3, 3, 7, 7, 7), FracSec.from!"hnsecs"(422202)));
                sysTime.add!"months"(-14);
                _assertPred!"=="(sysTime, SysTime(DateTime(2000, 1, 3, 7, 7, 7), FracSec.from!"hnsecs"(422202)));
            }

            //Test B.C.
            {
                auto sysTime = SysTime(Date(-1999, 7, 6));
                sysTime.add!"months"(3);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 10, 6)));
                sysTime.add!"months"(-4);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 6, 6)));
            }

            {
                auto sysTime = SysTime(Date(-1999, 7, 6));
                sysTime.add!"months"(6);
                _assertPred!"=="(sysTime, SysTime(Date(-1998, 1, 6)));
                sysTime.add!"months"(-6);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 7, 6)));
            }

            {
                auto sysTime = SysTime(Date(-1999, 7, 6));
                sysTime.add!"months"(-27);
                _assertPred!"=="(sysTime, SysTime(Date(-2001, 4, 6)));
                sysTime.add!"months"(28);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 8, 6)));
            }

            {
                auto sysTime = SysTime(Date(-1999, 5, 31));
                sysTime.add!"months"(1);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 7, 1)));
            }

            {
                auto sysTime = SysTime(Date(-1999, 5, 31));
                sysTime.add!"months"(-1);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 5, 1)));
            }

            {
                auto sysTime = SysTime(Date(-1999, 2, 28));
                sysTime.add!"months"(-12);
                _assertPred!"=="(sysTime, SysTime(Date(-2000, 2, 28)));
            }

            {
                auto sysTime = SysTime(Date(-2000, 2, 29));
                sysTime.add!"months"(-12);
                _assertPred!"=="(sysTime, SysTime(Date(-2001, 3, 1)));
            }

            {
                auto sysTime = SysTime(Date(-1999, 7, 31));
                sysTime.add!"months"(1);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 8, 31)));
                sysTime.add!"months"(1);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 10, 1)));
            }

            {
                auto sysTime = SysTime(Date(-1998, 8, 31));
                sysTime.add!"months"(13);
                _assertPred!"=="(sysTime, SysTime(Date(-1997, 10, 1)));
                sysTime.add!"months"(-13);
                _assertPred!"=="(sysTime, SysTime(Date(-1998, 9, 1)));
            }

            {
                auto sysTime = SysTime(Date(-1997, 12, 31));
                sysTime.add!"months"(13);
                _assertPred!"=="(sysTime, SysTime(Date(-1995, 1, 31)));
                sysTime.add!"months"(-13);
                _assertPred!"=="(sysTime, SysTime(Date(-1997, 12, 31)));
            }

            {
                auto sysTime = SysTime(Date(-1997, 12, 31));
                sysTime.add!"months"(14);
                _assertPred!"=="(sysTime, SysTime(Date(-1995, 3, 3)));
                sysTime.add!"months"(-14);
                _assertPred!"=="(sysTime, SysTime(Date(-1996, 1, 3)));
            }

            {
                auto sysTime = SysTime(Date(-2002, 12, 31));
                sysTime.add!"months"(14);
                _assertPred!"=="(sysTime, SysTime(Date(-2000, 3, 2)));
                sysTime.add!"months"(-14);
                _assertPred!"=="(sysTime, SysTime(Date(-2001, 1, 2)));
            }

            {
                auto sysTime = SysTime(Date(-2001, 12, 31));
                sysTime.add!"months"(14);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 3, 3)));
                sysTime.add!"months"(-14);
                _assertPred!"=="(sysTime, SysTime(Date(-2000, 1, 3)));
            }

            {
                auto sysTime = SysTime(DateTime(-1999, 7, 6, 12, 2, 7), FracSec.from!"usecs"(5007));
                sysTime.add!"months"(3);
                _assertPred!"=="(sysTime, SysTime(DateTime(-1999, 10, 6, 12, 2, 7), FracSec.from!"usecs"(5007)));
                sysTime.add!"months"(-4);
                _assertPred!"=="(sysTime, SysTime(DateTime(-1999, 6, 6, 12, 2, 7), FracSec.from!"usecs"(5007)));
            }

            {
                auto sysTime = SysTime(DateTime(-2002, 12, 31, 7, 7, 7), FracSec.from!"hnsecs"(422202));
                sysTime.add!"months"(14);
                _assertPred!"=="(sysTime, SysTime(DateTime(-2000, 3, 2, 7, 7, 7), FracSec.from!"hnsecs"(422202)));
                sysTime.add!"months"(-14);
                _assertPred!"=="(sysTime, SysTime(DateTime(-2001, 1, 2, 7, 7, 7), FracSec.from!"hnsecs"(422202)));
            }

            {
                auto sysTime = SysTime(DateTime(-2001, 12, 31, 7, 7, 7), FracSec.from!"hnsecs"(422202));
                sysTime.add!"months"(14);
                _assertPred!"=="(sysTime, SysTime(DateTime(-1999, 3, 3, 7, 7, 7), FracSec.from!"hnsecs"(422202)));
                sysTime.add!"months"(-14);
                _assertPred!"=="(sysTime, SysTime(DateTime(-2000, 1, 3, 7, 7, 7), FracSec.from!"hnsecs"(422202)));
            }

            //Test Both
            {
                auto sysTime = SysTime(Date(1, 1, 1));
                sysTime.add!"months"(-1);
                _assertPred!"=="(sysTime, SysTime(Date(0, 12, 1)));
                sysTime.add!"months"(1);
                _assertPred!"=="(sysTime, SysTime(Date(1, 1, 1)));
            }

            {
                auto sysTime = SysTime(Date(4, 1, 1));
                sysTime.add!"months"(-48);
                _assertPred!"=="(sysTime, SysTime(Date(0, 1, 1)));
                sysTime.add!"months"(48);
                _assertPred!"=="(sysTime, SysTime(Date(4, 1, 1)));
            }

            {
                auto sysTime = SysTime(Date(4, 3, 31));
                sysTime.add!"months"(-49);
                _assertPred!"=="(sysTime, SysTime(Date(0, 3, 2)));
                sysTime.add!"months"(49);
                _assertPred!"=="(sysTime, SysTime(Date(4, 4, 2)));
            }

            {
                auto sysTime = SysTime(Date(4, 3, 31));
                sysTime.add!"months"(-85);
                _assertPred!"=="(sysTime, SysTime(Date(-3, 3, 3)));
                sysTime.add!"months"(85);
                _assertPred!"=="(sysTime, SysTime(Date(4, 4, 3)));
            }

            {
                auto sysTime = SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0));
                sysTime.add!"months"(-1);
                _assertPred!"=="(sysTime, SysTime(DateTime(0, 12, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)));
                sysTime.add!"months"(1);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)));
            }

            {
                auto sysTime = SysTime(DateTime(1, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999));
                sysTime.add!"months"(-1);
                _assertPred!"=="(sysTime, SysTime(DateTime(0, 12, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
                sysTime.add!"months"(1);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            }

            {
                auto sysTime = SysTime(DateTime(0, 12, 1, 0, 0, 0), FracSec.from!"hnsecs"(0));
                sysTime.add!"months"(1);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)));
                sysTime.add!"months"(-1);
                _assertPred!"=="(sysTime, SysTime(DateTime(0, 12, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)));
            }

            {
                auto sysTime = SysTime(DateTime(0, 12, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999));
                sysTime.add!"months"(1);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
                sysTime.add!"months"(-1);
                _assertPred!"=="(sysTime, SysTime(DateTime(0, 12, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            }

            {
                auto sysTime = SysTime(DateTime(1, 1, 1, 0, 7, 9), FracSec.from!"hnsecs"(17));
                sysTime.add!"months"(-1);
                _assertPred!"=="(sysTime, SysTime(DateTime(0, 12, 1, 0, 7, 9), FracSec.from!"hnsecs"(17)));
                sysTime.add!"months"(1);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 1, 1, 0, 7, 9), FracSec.from!"hnsecs"(17)));
            }

            {
                auto sysTime = SysTime(DateTime(4, 3, 31, 12, 11, 10), FracSec.from!"msecs"(9));
                sysTime.add!"months"(-85);
                _assertPred!"=="(sysTime, SysTime(DateTime(-3, 3, 3, 12, 11, 10), FracSec.from!"msecs"(9)));
                sysTime.add!"months"(85);
                _assertPred!"=="(sysTime, SysTime(DateTime(4, 4, 3, 12, 11, 10), FracSec.from!"msecs"(9)));
            }

            {
                auto sysTime = SysTime(DateTime(-3, 3, 31, 12, 11, 10), FracSec.from!"msecs"(9));
                sysTime.add!"months"(85);
                _assertPred!"=="(sysTime, SysTime(DateTime(4, 5, 1, 12, 11, 10), FracSec.from!"msecs"(9)));
                sysTime.add!"months"(-85);
                _assertPred!"=="(sysTime, SysTime(DateTime(-3, 4, 1, 12, 11, 10), FracSec.from!"msecs"(9)));
            }

            const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            static assert(!__traits(compiles, cst.add!"months"(4)));
            //static assert(!__traits(compiles, ist.add!"months"(4)));
        }
    }

    //Test add!"months"() with AllowDayOverlow.no
    unittest
    {
        version(testStdDateTime)
        {
            //Test A.D.
            {
                auto sysTime = SysTime(Date(1999, 7, 6));
                sysTime.add!"months"(3, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 10, 6)));
                sysTime.add!"months"(-4, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 6, 6)));
            }

            {
                auto sysTime = SysTime(Date(1999, 7, 6));
                sysTime.add!"months"(6, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(2000, 1, 6)));
                sysTime.add!"months"(-6, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 7, 6)));
            }

            {
                auto sysTime = SysTime(Date(1999, 7, 6));
                sysTime.add!"months"(27, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(2001, 10, 6)));
                sysTime.add!"months"(-28, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 6, 6)));
            }

            {
                auto sysTime = SysTime(Date(1999, 5, 31));
                sysTime.add!"months"(1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 6, 30)));
            }

            {
                auto sysTime = SysTime(Date(1999, 5, 31));
                sysTime.add!"months"(-1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 4, 30)));
            }

            {
                auto sysTime = SysTime(Date(1999, 2, 28));
                sysTime.add!"months"(12, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(2000, 2, 28)));
            }

            {
                auto sysTime = SysTime(Date(2000, 2, 29));
                sysTime.add!"months"(12, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(2001, 2, 28)));
            }

            {
                auto sysTime = SysTime(Date(1999, 7, 31));
                sysTime.add!"months"(1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 8, 31)));
                sysTime.add!"months"(1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 9, 30)));
            }

            {
                auto sysTime = SysTime(Date(1998, 8, 31));
                sysTime.add!"months"(13, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 9, 30)));
                sysTime.add!"months"(-13, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(1998, 8, 30)));
            }

            {
                auto sysTime = SysTime(Date(1997, 12, 31));
                sysTime.add!"months"(13, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 1, 31)));
                sysTime.add!"months"(-13, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(1997, 12, 31)));
            }

            {
                auto sysTime = SysTime(Date(1997, 12, 31));
                sysTime.add!"months"(14, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 2, 28)));
                sysTime.add!"months"(-14, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(1997, 12, 28)));
            }

            {
                auto sysTime = SysTime(Date(1998, 12, 31));
                sysTime.add!"months"(14, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(2000, 2, 29)));
                sysTime.add!"months"(-14, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(1998, 12, 29)));
            }

            {
                auto sysTime = SysTime(Date(1999, 12, 31));
                sysTime.add!"months"(14, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(2001, 2, 28)));
                sysTime.add!"months"(-14, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 12, 28)));
            }

            {
                auto sysTime = SysTime(DateTime(1999, 7, 6, 12, 2, 7), FracSec.from!"usecs"(5007));
                sysTime.add!"months"(3, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(1999, 10, 6, 12, 2, 7), FracSec.from!"usecs"(5007)));
                sysTime.add!"months"(-4, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(1999, 6, 6, 12, 2, 7), FracSec.from!"usecs"(5007)));
            }

            {
                auto sysTime = SysTime(DateTime(1998, 12, 31, 7, 7, 7), FracSec.from!"hnsecs"(422202));
                sysTime.add!"months"(14, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(2000, 2, 29, 7, 7, 7), FracSec.from!"hnsecs"(422202)));
                sysTime.add!"months"(-14, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(1998, 12, 29, 7, 7, 7), FracSec.from!"hnsecs"(422202)));
            }

            {
                auto sysTime = SysTime(DateTime(1999, 12, 31, 7, 7, 7), FracSec.from!"hnsecs"(422202));
                sysTime.add!"months"(14, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(2001, 2, 28, 7, 7, 7), FracSec.from!"hnsecs"(422202)));
                sysTime.add!"months"(-14, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(1999, 12, 28, 7, 7, 7), FracSec.from!"hnsecs"(422202)));
            }

            //Test B.C.
            {
                auto sysTime = SysTime(Date(-1999, 7, 6));
                sysTime.add!"months"(3, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 10, 6)));
                sysTime.add!"months"(-4, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 6, 6)));
            }

            {
                auto sysTime = SysTime(Date(-1999, 7, 6));
                sysTime.add!"months"(6, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-1998, 1, 6)));
                sysTime.add!"months"(-6, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 7, 6)));
            }

            {
                auto sysTime = SysTime(Date(-1999, 7, 6));
                sysTime.add!"months"(-27, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-2001, 4, 6)));
                sysTime.add!"months"(28, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 8, 6)));
            }

            {
                auto sysTime = SysTime(Date(-1999, 5, 31));
                sysTime.add!"months"(1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 6, 30)));
            }

            {
                auto sysTime = SysTime(Date(-1999, 5, 31));
                sysTime.add!"months"(-1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 4, 30)));
            }

            {
                auto sysTime = SysTime(Date(-1999, 2, 28));
                sysTime.add!"months"(-12, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-2000, 2, 28)));
            }

            {
                auto sysTime = SysTime(Date(-2000, 2, 29));
                sysTime.add!"months"(-12, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-2001, 2, 28)));
            }

            {
                auto sysTime = SysTime(Date(-1999, 7, 31));
                sysTime.add!"months"(1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 8, 31)));
                sysTime.add!"months"(1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 9, 30)));
            }

            {
                auto sysTime = SysTime(Date(-1998, 8, 31));
                sysTime.add!"months"(13, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-1997, 9, 30)));
                sysTime.add!"months"(-13, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-1998, 8, 30)));
            }

            {
                auto sysTime = SysTime(Date(-1997, 12, 31));
                sysTime.add!"months"(13, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-1995, 1, 31)));
                sysTime.add!"months"(-13, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-1997, 12, 31)));
            }

            {
                auto sysTime = SysTime(Date(-1997, 12, 31));
                sysTime.add!"months"(14, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-1995, 2, 28)));
                sysTime.add!"months"(-14, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-1997, 12, 28)));
            }

            {
                auto sysTime = SysTime(Date(-2002, 12, 31));
                sysTime.add!"months"(14, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-2000, 2, 29)));
                sysTime.add!"months"(-14, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-2002, 12, 29)));
            }

            {
                auto sysTime = SysTime(Date(-2001, 12, 31));
                sysTime.add!"months"(14, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 2, 28)));
                sysTime.add!"months"(-14, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-2001, 12, 28)));
            }

            {
                auto sysTime = SysTime(DateTime(-1999, 7, 6, 12, 2, 7), FracSec.from!"usecs"(5007));
                sysTime.add!"months"(3, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(-1999, 10, 6, 12, 2, 7), FracSec.from!"usecs"(5007)));
                sysTime.add!"months"(-4, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(-1999, 6, 6, 12, 2, 7), FracSec.from!"usecs"(5007)));
            }

            {
                auto sysTime = SysTime(DateTime(-2002, 12, 31, 7, 7, 7), FracSec.from!"hnsecs"(422202));
                sysTime.add!"months"(14, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(-2000, 2, 29, 7, 7, 7), FracSec.from!"hnsecs"(422202)));
                sysTime.add!"months"(-14, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(-2002, 12, 29, 7, 7, 7), FracSec.from!"hnsecs"(422202)));
            }

            {
                auto sysTime = SysTime(DateTime(-2001, 12, 31, 7, 7, 7), FracSec.from!"hnsecs"(422202));
                sysTime.add!"months"(14, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(-1999, 2, 28, 7, 7, 7), FracSec.from!"hnsecs"(422202)));
                sysTime.add!"months"(-14, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(-2001, 12, 28, 7, 7, 7), FracSec.from!"hnsecs"(422202)));
            }

            //Test Both
            {
                auto sysTime = SysTime(Date(1, 1, 1));
                sysTime.add!"months"(-1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(0, 12, 1)));
                sysTime.add!"months"(1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(1, 1, 1)));
            }

            {
                auto sysTime = SysTime(Date(4, 1, 1));
                sysTime.add!"months"(-48, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(0, 1, 1)));
                sysTime.add!"months"(48, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(4, 1, 1)));
            }

            {
                auto sysTime = SysTime(Date(4, 3, 31));
                sysTime.add!"months"(-49, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(0, 2, 29)));
                sysTime.add!"months"(49, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(4, 3, 29)));
            }

            {
                auto sysTime = SysTime(Date(4, 3, 31));
                sysTime.add!"months"(-85, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-3, 2, 28)));
                sysTime.add!"months"(85, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(4, 3, 28)));
            }

            {
                auto sysTime = SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0));
                sysTime.add!"months"(-1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(0, 12, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)));
                sysTime.add!"months"(1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)));
            }

            {
                auto sysTime = SysTime(DateTime(1, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999));
                sysTime.add!"months"(-1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(0, 12, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
                sysTime.add!"months"(1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            }

            {
                auto sysTime = SysTime(DateTime(0, 12, 1, 0, 0, 0), FracSec.from!"hnsecs"(0));
                sysTime.add!"months"(1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)));
                sysTime.add!"months"(-1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(0, 12, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)));
            }

            {
                auto sysTime = SysTime(DateTime(0, 12, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999));
                sysTime.add!"months"(1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
                sysTime.add!"months"(-1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(0, 12, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            }

            {
                auto sysTime = SysTime(DateTime(1, 1, 1, 0, 7, 9), FracSec.from!"hnsecs"(17));
                sysTime.add!"months"(-1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(0, 12, 1, 0, 7, 9), FracSec.from!"hnsecs"(17)));
                sysTime.add!"months"(1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 1, 1, 0, 7, 9), FracSec.from!"hnsecs"(17)));
            }

            {
                auto sysTime = SysTime(DateTime(4, 3, 31, 12, 11, 10), FracSec.from!"msecs"(9));
                sysTime.add!"months"(-85, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(-3, 2, 28, 12, 11, 10), FracSec.from!"msecs"(9)));
                sysTime.add!"months"(85, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(4, 3, 28, 12, 11, 10), FracSec.from!"msecs"(9)));
            }

            {
                auto sysTime = SysTime(DateTime(-3, 3, 31, 12, 11, 10), FracSec.from!"msecs"(9));
                sysTime.add!"months"(85, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(4, 4, 30, 12, 11, 10), FracSec.from!"msecs"(9)));
                sysTime.add!"months"(-85, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(-3, 3, 30, 12, 11, 10), FracSec.from!"msecs"(9)));
            }
        }
    }


    /++
        Adds the given number of years or months to this $(D SysTime). A
        negative number will subtract.

        The difference between rolling and adding is that rolling does not
        affect larger units. Rolling a $(D SysTime) 12 months
        gets the exact same $(D SysTime). However, the days can still be affected
        due to the differing number of days in each month.

        Because there are no units larger than years, there is no difference
        between adding and rolling years.

        Params:
            units         = The type of units to add ("years" or "months").
            value         = The number of months or years to add to this
                            $(D SysTime).
            allowOverflow = Whether the days should be allowed to overflow,
                            causing the month to increment.

        Examples:
--------------------
auto st1 = SysTime(DateTime(2010, 1, 1, 12, 33, 33));
st1.roll!"months"(1);
assert(st1 == SysTime(DateTime(2010, 2, 1, 12, 33, 33)));

auto st2 = SysTime(DateTime(2010, 1, 1, 12, 33, 33));
st2.roll!"months"(-1);
assert(st2 == SysTime(DateTime(2010, 12, 1, 12, 33, 33)));

auto st3 = SysTime(DateTime(1999, 1, 29, 12, 33, 33));
st3.roll!"months"(1);
assert(st3 == SysTime(DateTime(1999, 3, 1, 12, 33, 33)));

auto st4 = SysTime(DateTime(1999, 1, 29, 12, 33, 33));
st4.roll!"months"(1, AllowDayOverflow.no);
assert(st4 == SysTime(DateTime(1999, 2, 28, 12, 33, 33)));

auto st5 = SysTime(DateTime(2000, 2, 29, 12, 30, 33));
st5.roll!"years"(1);
assert(st5 == SysTime(DateTime(2001, 3, 1, 12, 30, 33)));

auto st6 = SysTime(DateTime(2000, 2, 29, 12, 30, 33));
st6.roll!"years"(1, AllowDayOverflow.no);
assert(st6 == SysTime(DateTime(2001, 2, 28, 12, 30, 33)));
--------------------
      +/
    /+ref SysTime+/ void roll(string units)(long value, AllowDayOverflow allowOverflow = AllowDayOverflow.yes) nothrow
        if(units == "years")
    {
        add!"years"(value, allowOverflow);
    }

    unittest
    {
        version(testStdDateTime)
        {
            //Verify Examples.
            auto st1 = SysTime(DateTime(2010, 1, 1, 12, 33, 33));
            st1.roll!"months"(1);
            assert(st1 == SysTime(DateTime(2010, 2, 1, 12, 33, 33)));

            auto st2 = SysTime(DateTime(2010, 1, 1, 12, 33, 33));
            st2.roll!"months"(-1);
            assert(st2 == SysTime(DateTime(2010, 12, 1, 12, 33, 33)));

            auto st3 = SysTime(DateTime(1999, 1, 29, 12, 33, 33));
            st3.roll!"months"(1);
            assert(st3 == SysTime(DateTime(1999, 3, 1, 12, 33, 33)));

            auto st4 = SysTime(DateTime(1999, 1, 29, 12, 33, 33));
            st4.roll!"months"(1, AllowDayOverflow.no);
            assert(st4 == SysTime(DateTime(1999, 2, 28, 12, 33, 33)));

            auto st5 = SysTime(DateTime(2000, 2, 29, 12, 30, 33));
            st5.roll!"years"(1);
            assert(st5 == SysTime(DateTime(2001, 3, 1, 12, 30, 33)));

            auto st6 = SysTime(DateTime(2000, 2, 29, 12, 30, 33));
            st6.roll!"years"(1, AllowDayOverflow.no);
            assert(st6 == SysTime(DateTime(2001, 2, 28, 12, 30, 33)));
        }
    }

    unittest
    {
        version(testStdDateTime)
        {
            auto st = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            static assert(__traits(compiles, st.roll!"years"(4)));
            static assert(!__traits(compiles, cst.roll!"years"(4)));
            //static assert(!__traits(compiles, ist.roll!"years"(4)));
        }
    }


    //Shares documentation with "years" version.
    /+ref SysTime+/ void roll(string units)(long value, AllowDayOverflow allowOverflow = AllowDayOverflow.yes) nothrow
        if(units == "months")
    {
        auto hnsecs = adjTime;
        auto days = splitUnitsFromHNSecs!"days"(hnsecs) + 1;

        if(hnsecs < 0)
        {
            hnsecs += convert!("hours", "hnsecs")(24);
            --days;
        }

        auto date = Date(cast(int)days);
        date.roll!"months"(value, allowOverflow);
        days = date.dayOfGregorianCal - 1;

        if(days < 0)
        {
            hnsecs -= convert!("hours", "hnsecs")(24);
            ++days;
        }

        immutable newDaysHNSecs = convert!("days", "hnsecs")(days);

        adjTime = newDaysHNSecs + hnsecs;
    }

    //Test roll!"months"() with AllowDayOverlow.yes
    unittest
    {
        version(testStdDateTime)
        {
            //Test A.D.
            {
                auto sysTime = SysTime(Date(1999, 7, 6));
                sysTime.roll!"months"(3);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 10, 6)));
                sysTime.roll!"months"(-4);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 6, 6)));
            }

            {
                auto sysTime = SysTime(Date(1999, 7, 6));
                sysTime.roll!"months"(6);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 1, 6)));
                sysTime.roll!"months"(-6);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 7, 6)));
            }

            {
                auto sysTime = SysTime(Date(1999, 7, 6));
                sysTime.roll!"months"(27);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 10, 6)));
                sysTime.roll!"months"(-28);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 6, 6)));
            }

            {
                auto sysTime = SysTime(Date(1999, 5, 31));
                sysTime.roll!"months"(1);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 7, 1)));
            }

            {
                auto sysTime = SysTime(Date(1999, 5, 31));
                sysTime.roll!"months"(-1);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 5, 1)));
            }

            {
                auto sysTime = SysTime(Date(1999, 2, 28));
                sysTime.roll!"months"(12);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 2, 28)));
            }

            {
                auto sysTime = SysTime(Date(2000, 2, 29));
                sysTime.roll!"months"(12);
                _assertPred!"=="(sysTime, SysTime(Date(2000, 2, 29)));
            }

            {
                auto sysTime = SysTime(Date(1999, 7, 31));
                sysTime.roll!"months"(1);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 8, 31)));
                sysTime.roll!"months"(1);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 10, 1)));
            }

            {
                auto sysTime = SysTime(Date(1998, 8, 31));
                sysTime.roll!"months"(13);
                _assertPred!"=="(sysTime, SysTime(Date(1998, 10, 1)));
                sysTime.roll!"months"(-13);
                _assertPred!"=="(sysTime, SysTime(Date(1998, 9, 1)));
            }

            {
                auto sysTime = SysTime(Date(1997, 12, 31));
                sysTime.roll!"months"(13);
                _assertPred!"=="(sysTime, SysTime(Date(1997, 1, 31)));
                sysTime.roll!"months"(-13);
                _assertPred!"=="(sysTime, SysTime(Date(1997, 12, 31)));
            }

            {
                auto sysTime = SysTime(Date(1997, 12, 31));
                sysTime.roll!"months"(14);
                _assertPred!"=="(sysTime, SysTime(Date(1997, 3, 3)));
                sysTime.roll!"months"(-14);
                _assertPred!"=="(sysTime, SysTime(Date(1997, 1, 3)));
            }

            {
                auto sysTime = SysTime(Date(1998, 12, 31));
                sysTime.roll!"months"(14);
                _assertPred!"=="(sysTime, SysTime(Date(1998, 3, 3)));
                sysTime.roll!"months"(-14);
                _assertPred!"=="(sysTime, SysTime(Date(1998, 1, 3)));
            }

            {
                auto sysTime = SysTime(Date(1999, 12, 31));
                sysTime.roll!"months"(14);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 3, 3)));
                sysTime.roll!"months"(-14);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 1, 3)));
            }

            {
                auto sysTime = SysTime(DateTime(1999, 7, 6, 12, 2, 7), FracSec.from!"usecs"(5007));
                sysTime.roll!"months"(3);
                _assertPred!"=="(sysTime, SysTime(DateTime(1999, 10, 6, 12, 2, 7), FracSec.from!"usecs"(5007)));
                sysTime.roll!"months"(-4);
                _assertPred!"=="(sysTime, SysTime(DateTime(1999, 6, 6, 12, 2, 7), FracSec.from!"usecs"(5007)));
            }

            {
                auto sysTime = SysTime(DateTime(1998, 12, 31, 7, 7, 7), FracSec.from!"hnsecs"(422202));
                sysTime.roll!"months"(14);
                _assertPred!"=="(sysTime, SysTime(DateTime(1998, 3, 3, 7, 7, 7), FracSec.from!"hnsecs"(422202)));
                sysTime.roll!"months"(-14);
                _assertPred!"=="(sysTime, SysTime(DateTime(1998, 1, 3, 7, 7, 7), FracSec.from!"hnsecs"(422202)));
            }

            {
                auto sysTime = SysTime(DateTime(1999, 12, 31, 7, 7, 7), FracSec.from!"hnsecs"(422202));
                sysTime.roll!"months"(14);
                _assertPred!"=="(sysTime, SysTime(DateTime(1999, 3, 3, 7, 7, 7), FracSec.from!"hnsecs"(422202)));
                sysTime.roll!"months"(-14);
                _assertPred!"=="(sysTime, SysTime(DateTime(1999, 1, 3, 7, 7, 7), FracSec.from!"hnsecs"(422202)));
            }

            //Test B.C.
            {
                auto sysTime = SysTime(Date(-1999, 7, 6));
                sysTime.roll!"months"(3);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 10, 6)));
                sysTime.roll!"months"(-4);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 6, 6)));
            }

            {
                auto sysTime = SysTime(Date(-1999, 7, 6));
                sysTime.roll!"months"(6);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 1, 6)));
                sysTime.roll!"months"(-6);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 7, 6)));
            }

            {
                auto sysTime = SysTime(Date(-1999, 7, 6));
                sysTime.roll!"months"(-27);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 4, 6)));
                sysTime.roll!"months"(28);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 8, 6)));
            }

            {
                auto sysTime = SysTime(Date(-1999, 5, 31));
                sysTime.roll!"months"(1);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 7, 1)));
            }

            {
                auto sysTime = SysTime(Date(-1999, 5, 31));
                sysTime.roll!"months"(-1);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 5, 1)));
            }

            {
                auto sysTime = SysTime(Date(-1999, 2, 28));
                sysTime.roll!"months"(-12);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 2, 28)));
            }

            {
                auto sysTime = SysTime(Date(-2000, 2, 29));
                sysTime.roll!"months"(-12);
                _assertPred!"=="(sysTime, SysTime(Date(-2000, 2, 29)));
            }

            {
                auto sysTime = SysTime(Date(-1999, 7, 31));
                sysTime.roll!"months"(1);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 8, 31)));
                sysTime.roll!"months"(1);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 10, 1)));
            }

            {
                auto sysTime = SysTime(Date(-1998, 8, 31));
                sysTime.roll!"months"(13);
                _assertPred!"=="(sysTime, SysTime(Date(-1998, 10, 1)));
                sysTime.roll!"months"(-13);
                _assertPred!"=="(sysTime, SysTime(Date(-1998, 9, 1)));
            }

            {
                auto sysTime = SysTime(Date(-1997, 12, 31));
                sysTime.roll!"months"(13);
                _assertPred!"=="(sysTime, SysTime(Date(-1997, 1, 31)));
                sysTime.roll!"months"(-13);
                _assertPred!"=="(sysTime, SysTime(Date(-1997, 12, 31)));
            }

            {
                auto sysTime = SysTime(Date(-1997, 12, 31));
                sysTime.roll!"months"(14);
                _assertPred!"=="(sysTime, SysTime(Date(-1997, 3, 3)));
                sysTime.roll!"months"(-14);
                _assertPred!"=="(sysTime, SysTime(Date(-1997, 1, 3)));
            }

            {
                auto sysTime = SysTime(Date(-2002, 12, 31));
                sysTime.roll!"months"(14);
                _assertPred!"=="(sysTime, SysTime(Date(-2002, 3, 3)));
                sysTime.roll!"months"(-14);
                _assertPred!"=="(sysTime, SysTime(Date(-2002, 1, 3)));
            }

            {
                auto sysTime = SysTime(Date(-2001, 12, 31));
                sysTime.roll!"months"(14);
                _assertPred!"=="(sysTime, SysTime(Date(-2001, 3, 3)));
                sysTime.roll!"months"(-14);
                _assertPred!"=="(sysTime, SysTime(Date(-2001, 1, 3)));
            }

            {
                auto sysTime = SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0));
                sysTime.roll!"months"(-1);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 12, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)));
                sysTime.roll!"months"(1);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)));
            }

            {
                auto sysTime = SysTime(DateTime(1, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999));
                sysTime.roll!"months"(-1);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 12, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
                sysTime.roll!"months"(1);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            }

            {
                auto sysTime = SysTime(DateTime(0, 12, 1, 0, 0, 0), FracSec.from!"hnsecs"(0));
                sysTime.roll!"months"(1);
                _assertPred!"=="(sysTime, SysTime(DateTime(0, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)));
                sysTime.roll!"months"(-1);
                _assertPred!"=="(sysTime, SysTime(DateTime(0, 12, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)));
            }

            {
                auto sysTime = SysTime(DateTime(0, 12, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999));
                sysTime.roll!"months"(1);
                _assertPred!"=="(sysTime, SysTime(DateTime(0, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
                sysTime.roll!"months"(-1);
                _assertPred!"=="(sysTime, SysTime(DateTime(0, 12, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            }

            {
                auto sysTime = SysTime(DateTime(-1999, 7, 6, 12, 2, 7), FracSec.from!"hnsecs"(5007));
                sysTime.roll!"months"(3);
                _assertPred!"=="(sysTime, SysTime(DateTime(-1999, 10, 6, 12, 2, 7), FracSec.from!"hnsecs"(5007)));
                sysTime.roll!"months"(-4);
                _assertPred!"=="(sysTime, SysTime(DateTime(-1999, 6, 6, 12, 2, 7), FracSec.from!"hnsecs"(5007)));
            }

            {
                auto sysTime = SysTime(DateTime(-2002, 12, 31, 7, 7, 7), FracSec.from!"hnsecs"(422202));
                sysTime.roll!"months"(14);
                _assertPred!"=="(sysTime, SysTime(DateTime(-2002, 3, 3, 7, 7, 7), FracSec.from!"hnsecs"(422202)));
                sysTime.roll!"months"(-14);
                _assertPred!"=="(sysTime, SysTime(DateTime(-2002, 1, 3, 7, 7, 7), FracSec.from!"hnsecs"(422202)));
            }

            {
                auto sysTime = SysTime(DateTime(-2001, 12, 31, 7, 7, 7), FracSec.from!"hnsecs"(422202));
                sysTime.roll!"months"(14);
                _assertPred!"=="(sysTime, SysTime(DateTime(-2001, 3, 3, 7, 7, 7), FracSec.from!"hnsecs"(422202)));
                sysTime.roll!"months"(-14);
                _assertPred!"=="(sysTime, SysTime(DateTime(-2001, 1, 3, 7, 7, 7), FracSec.from!"hnsecs"(422202)));
            }

            //Test Both
            {
                auto sysTime = SysTime(Date(1, 1, 1));
                sysTime.roll!"months"(-1);
                _assertPred!"=="(sysTime, SysTime(Date(1, 12, 1)));
                sysTime.roll!"months"(1);
                _assertPred!"=="(sysTime, SysTime(Date(1, 1, 1)));
            }

            {
                auto sysTime = SysTime(Date(4, 1, 1));
                sysTime.roll!"months"(-48);
                _assertPred!"=="(sysTime, SysTime(Date(4, 1, 1)));
                sysTime.roll!"months"(48);
                _assertPred!"=="(sysTime, SysTime(Date(4, 1, 1)));
            }

            {
                auto sysTime = SysTime(Date(4, 3, 31));
                sysTime.roll!"months"(-49);
                _assertPred!"=="(sysTime, SysTime(Date(4, 3, 2)));
                sysTime.roll!"months"(49);
                _assertPred!"=="(sysTime, SysTime(Date(4, 4, 2)));
            }

            {
                auto sysTime = SysTime(Date(4, 3, 31));
                sysTime.roll!"months"(-85);
                _assertPred!"=="(sysTime, SysTime(Date(4, 3, 2)));
                sysTime.roll!"months"(85);
                _assertPred!"=="(sysTime, SysTime(Date(4, 4, 2)));
            }

            {
                auto sysTime = SysTime(Date(-1, 1, 1));
                sysTime.roll!"months"(-1);
                _assertPred!"=="(sysTime, SysTime(Date(-1, 12, 1)));
                sysTime.roll!"months"(1);
                _assertPred!"=="(sysTime, SysTime(Date(-1, 1, 1)));
            }

            {
                auto sysTime = SysTime(Date(-4, 1, 1));
                sysTime.roll!"months"(-48);
                _assertPred!"=="(sysTime, SysTime(Date(-4, 1, 1)));
                sysTime.roll!"months"(48);
                _assertPred!"=="(sysTime, SysTime(Date(-4, 1, 1)));
            }

            {
                auto sysTime = SysTime(Date(-4, 3, 31));
                sysTime.roll!"months"(-49);
                _assertPred!"=="(sysTime, SysTime(Date(-4, 3, 2)));
                sysTime.roll!"months"(49);
                _assertPred!"=="(sysTime, SysTime(Date(-4, 4, 2)));
            }

            {
                auto sysTime = SysTime(Date(-4, 3, 31));
                sysTime.roll!"months"(-85);
                _assertPred!"=="(sysTime, SysTime(Date(-4, 3, 2)));
                sysTime.roll!"months"(85);
                _assertPred!"=="(sysTime, SysTime(Date(-4, 4, 2)));
            }

            {
                auto sysTime = SysTime(DateTime(1, 1, 1, 0, 7, 9), FracSec.from!"hnsecs"(17));
                sysTime.roll!"months"(-1);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 12, 1, 0, 7, 9), FracSec.from!"hnsecs"(17)));
                sysTime.roll!"months"(1);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 1, 1, 0, 7, 9), FracSec.from!"hnsecs"(17)));
            }

            {
                auto sysTime = SysTime(DateTime(4, 3, 31, 12, 11, 10), FracSec.from!"msecs"(9));
                sysTime.roll!"months"(-85);
                _assertPred!"=="(sysTime, SysTime(DateTime(4, 3, 2, 12, 11, 10), FracSec.from!"msecs"(9)));
                sysTime.roll!"months"(85);
                _assertPred!"=="(sysTime, SysTime(DateTime(4, 4, 2, 12, 11, 10), FracSec.from!"msecs"(9)));
            }

            {
                auto sysTime = SysTime(DateTime(-3, 3, 31, 12, 11, 10), FracSec.from!"msecs"(9));
                sysTime.roll!"months"(85);
                _assertPred!"=="(sysTime, SysTime(DateTime(-3, 5, 1, 12, 11, 10), FracSec.from!"msecs"(9)));
                sysTime.roll!"months"(-85);
                _assertPred!"=="(sysTime, SysTime(DateTime(-3, 4, 1, 12, 11, 10), FracSec.from!"msecs"(9)));
            }

            const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            static assert(!__traits(compiles, cst.roll!"months"(4)));
            //static assert(!__traits(compiles, ist.roll!"months"(4)));

            //Verify Examples.
            auto st1 = SysTime(DateTime(2010, 1, 1, 12, 33, 33));
            st1.roll!"months"(1);
            assert(st1 == SysTime(DateTime(2010, 2, 1, 12, 33, 33)));

            auto st2 = SysTime(DateTime(2010, 1, 1, 12, 33, 33));
            st2.roll!"months"(-1);
            assert(st2 == SysTime(DateTime(2010, 12, 1, 12, 33, 33)));

            auto st3 = SysTime(DateTime(1999, 1, 29, 12, 33, 33));
            st3.roll!"months"(1);
            assert(st3 == SysTime(DateTime(1999, 3, 1, 12, 33, 33)));

            auto st4 = SysTime(DateTime(1999, 1, 29, 12, 33, 33));
            st4.roll!"months"(1, AllowDayOverflow.no);
            assert(st4 == SysTime(DateTime(1999, 2, 28, 12, 33, 33)));
        }
    }

    //Test roll!"months"() with AllowDayOverlow.no
    unittest
    {
        version(testStdDateTime)
        {
            //Test A.D.
            {
                auto sysTime = SysTime(Date(1999, 7, 6));
                sysTime.roll!"months"(3, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 10, 6)));
                sysTime.roll!"months"(-4, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 6, 6)));
            }

            {
                auto sysTime = SysTime(Date(1999, 7, 6));
                sysTime.roll!"months"(6, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 1, 6)));
                sysTime.roll!"months"(-6, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 7, 6)));
            }

            {
                auto sysTime = SysTime(Date(1999, 7, 6));
                sysTime.roll!"months"(27, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 10, 6)));
                sysTime.roll!"months"(-28, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 6, 6)));
            }

            {
                auto sysTime = SysTime(Date(1999, 5, 31));
                sysTime.roll!"months"(1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 6, 30)));
            }

            {
                auto sysTime = SysTime(Date(1999, 5, 31));
                sysTime.roll!"months"(-1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 4, 30)));
            }

            {
                auto sysTime = SysTime(Date(1999, 2, 28));
                sysTime.roll!"months"(12, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 2, 28)));
            }

            {
                auto sysTime = SysTime(Date(2000, 2, 29));
                sysTime.roll!"months"(12, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(2000, 2, 29)));
            }

            {
                auto sysTime = SysTime(Date(1999, 7, 31));
                sysTime.roll!"months"(1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 8, 31)));
                sysTime.roll!"months"(1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 9, 30)));
            }

            {
                auto sysTime = SysTime(Date(1998, 8, 31));
                sysTime.roll!"months"(13, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(1998, 9, 30)));
                sysTime.roll!"months"(-13, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(1998, 8, 30)));
            }

            {
                auto sysTime = SysTime(Date(1997, 12, 31));
                sysTime.roll!"months"(13, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(1997, 1, 31)));
                sysTime.roll!"months"(-13, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(1997, 12, 31)));
            }

            {
                auto sysTime = SysTime(Date(1997, 12, 31));
                sysTime.roll!"months"(14, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(1997, 2, 28)));
                sysTime.roll!"months"(-14, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(1997, 12, 28)));
            }

            {
                auto sysTime = SysTime(Date(1998, 12, 31));
                sysTime.roll!"months"(14, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(1998, 2, 28)));
                sysTime.roll!"months"(-14, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(1998, 12, 28)));
            }

            {
                auto sysTime = SysTime(Date(1999, 12, 31));
                sysTime.roll!"months"(14, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 2, 28)));
                sysTime.roll!"months"(-14, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 12, 28)));
            }

            {
                auto sysTime = SysTime(DateTime(1999, 7, 6, 12, 2, 7), FracSec.from!"usecs"(5007));
                sysTime.roll!"months"(3, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(1999, 10, 6, 12, 2, 7), FracSec.from!"usecs"(5007)));
                sysTime.roll!"months"(-4, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(1999, 6, 6, 12, 2, 7), FracSec.from!"usecs"(5007)));
            }

            {
                auto sysTime = SysTime(DateTime(1998, 12, 31, 7, 7, 7), FracSec.from!"hnsecs"(422202));
                sysTime.roll!"months"(14, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(1998, 2, 28, 7, 7, 7), FracSec.from!"hnsecs"(422202)));
                sysTime.roll!"months"(-14, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(1998, 12, 28, 7, 7, 7), FracSec.from!"hnsecs"(422202)));
            }

            {
                auto sysTime = SysTime(DateTime(1999, 12, 31, 7, 7, 7), FracSec.from!"hnsecs"(422202));
                sysTime.roll!"months"(14, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(1999, 2, 28, 7, 7, 7), FracSec.from!"hnsecs"(422202)));
                sysTime.roll!"months"(-14, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(1999, 12, 28, 7, 7, 7), FracSec.from!"hnsecs"(422202)));
            }

            //Test B.C.
            {
                auto sysTime = SysTime(Date(-1999, 7, 6));
                sysTime.roll!"months"(3, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 10, 6)));
                sysTime.roll!"months"(-4, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 6, 6)));
            }

            {
                auto sysTime = SysTime(Date(-1999, 7, 6));
                sysTime.roll!"months"(6, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 1, 6)));
                sysTime.roll!"months"(-6, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 7, 6)));
            }

            {
                auto sysTime = SysTime(Date(-1999, 7, 6));
                sysTime.roll!"months"(-27, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 4, 6)));
                sysTime.roll!"months"(28, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 8, 6)));
            }

            {
                auto sysTime = SysTime(Date(-1999, 5, 31));
                sysTime.roll!"months"(1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 6, 30)));
            }

            {
                auto sysTime = SysTime(Date(-1999, 5, 31));
                sysTime.roll!"months"(-1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 4, 30)));
            }

            {
                auto sysTime = SysTime(Date(-1999, 2, 28));
                sysTime.roll!"months"(-12, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 2, 28)));
            }

            {
                auto sysTime = SysTime(Date(-2000, 2, 29));
                sysTime.roll!"months"(-12, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-2000, 2, 29)));
            }

            {
                auto sysTime = SysTime(Date(-1999, 7, 31));
                sysTime.roll!"months"(1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 8, 31)));
                sysTime.roll!"months"(1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 9, 30)));
            }

            {
                auto sysTime = SysTime(Date(-1998, 8, 31));
                sysTime.roll!"months"(13, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-1998, 9, 30)));
                sysTime.roll!"months"(-13, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-1998, 8, 30)));
            }

            {
                auto sysTime = SysTime(Date(-1997, 12, 31));
                sysTime.roll!"months"(13, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-1997, 1, 31)));
                sysTime.roll!"months"(-13, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-1997, 12, 31)));
            }

            {
                auto sysTime = SysTime(Date(-1997, 12, 31));
                sysTime.roll!"months"(14, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-1997, 2, 28)));
                sysTime.roll!"months"(-14, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-1997, 12, 28)));
            }

            {
                auto sysTime = SysTime(Date(-2002, 12, 31));
                sysTime.roll!"months"(14, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-2002, 2, 28)));
                sysTime.roll!"months"(-14, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-2002, 12, 28)));
            }

            {
                auto sysTime = SysTime(Date(-2001, 12, 31));
                sysTime.roll!"months"(14, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-2001, 2, 28)));
                sysTime.roll!"months"(-14, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-2001, 12, 28)));
            }

            {
                auto sysTime = SysTime(DateTime(-1999, 7, 6, 12, 2, 7), FracSec.from!"usecs"(5007));
                sysTime.roll!"months"(3, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(-1999, 10, 6, 12, 2, 7), FracSec.from!"usecs"(5007)));
                sysTime.roll!"months"(-4, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(-1999, 6, 6, 12, 2, 7), FracSec.from!"usecs"(5007)));
            }

            {
                auto sysTime = SysTime(DateTime(-2002, 12, 31, 7, 7, 7), FracSec.from!"hnsecs"(422202));
                sysTime.roll!"months"(14, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(-2002, 2, 28, 7, 7, 7), FracSec.from!"hnsecs"(422202)));
                sysTime.roll!"months"(-14, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(-2002, 12, 28, 7, 7, 7), FracSec.from!"hnsecs"(422202)));
            }

            {
                auto sysTime = SysTime(DateTime(-2001, 12, 31, 7, 7, 7), FracSec.from!"hnsecs"(422202));
                sysTime.roll!"months"(14, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(-2001, 2, 28, 7, 7, 7), FracSec.from!"hnsecs"(422202)));
                sysTime.roll!"months"(-14, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(-2001, 12, 28, 7, 7, 7), FracSec.from!"hnsecs"(422202)));
            }

            //Test Both
            {
                auto sysTime = SysTime(Date(1, 1, 1));
                sysTime.roll!"months"(-1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(1, 12, 1)));
                sysTime.roll!"months"(1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(1, 1, 1)));
            }

            {
                auto sysTime = SysTime(Date(4, 1, 1));
                sysTime.roll!"months"(-48, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(4, 1, 1)));
                sysTime.roll!"months"(48, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(4, 1, 1)));
            }

            {
                auto sysTime = SysTime(Date(4, 3, 31));
                sysTime.roll!"months"(-49, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(4, 2, 29)));
                sysTime.roll!"months"(49, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(4, 3, 29)));
            }

            {
                auto sysTime = SysTime(Date(4, 3, 31));
                sysTime.roll!"months"(-85, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(4, 2, 29)));
                sysTime.roll!"months"(85, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(4, 3, 29)));
            }

            {
                auto sysTime = SysTime(Date(-1, 1, 1));
                sysTime.roll!"months"(-1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-1, 12, 1)));
                sysTime.roll!"months"(1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-1, 1, 1)));
            }

            {
                auto sysTime = SysTime(Date(-4, 1, 1));
                sysTime.roll!"months"(-48, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-4, 1, 1)));
                sysTime.roll!"months"(48, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-4, 1, 1)));
            }

            {
                auto sysTime = SysTime(Date(-4, 3, 31));
                sysTime.roll!"months"(-49, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-4, 2, 29)));
                sysTime.roll!"months"(49, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-4, 3, 29)));
            }

            {
                auto sysTime = SysTime(Date(-4, 3, 31));
                sysTime.roll!"months"(-85, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-4, 2, 29)));
                sysTime.roll!"months"(85, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(Date(-4, 3, 29)));
            }

            {
                auto sysTime = SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0));
                sysTime.roll!"months"(-1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 12, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)));
                sysTime.roll!"months"(1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)));
            }

            {
                auto sysTime = SysTime(DateTime(1, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999));
                sysTime.roll!"months"(-1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 12, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
                sysTime.roll!"months"(1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            }

            {
                auto sysTime = SysTime(DateTime(0, 12, 1, 0, 0, 0), FracSec.from!"hnsecs"(0));
                sysTime.roll!"months"(1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(0, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)));
                sysTime.roll!"months"(-1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(0, 12, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)));
            }

            {
                auto sysTime = SysTime(DateTime(0, 12, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999));
                sysTime.roll!"months"(1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(0, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
                sysTime.roll!"months"(-1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(0, 12, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            }

            {
                auto sysTime = SysTime(DateTime(1, 1, 1, 0, 7, 9), FracSec.from!"hnsecs"(17));
                sysTime.roll!"months"(-1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 12, 1, 0, 7, 9), FracSec.from!"hnsecs"(17)));
                sysTime.roll!"months"(1, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 1, 1, 0, 7, 9), FracSec.from!"hnsecs"(17)));
            }

            {
                auto sysTime = SysTime(DateTime(4, 3, 31, 12, 11, 10), FracSec.from!"msecs"(9));
                sysTime.roll!"months"(-85, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(4, 2, 29, 12, 11, 10), FracSec.from!"msecs"(9)));
                sysTime.roll!"months"(85, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(4, 3, 29, 12, 11, 10), FracSec.from!"msecs"(9)));
            }

            {
                auto sysTime = SysTime(DateTime(-3, 3, 31, 12, 11, 10), FracSec.from!"msecs"(9));
                sysTime.roll!"months"(85, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(-3, 4, 30, 12, 11, 10), FracSec.from!"msecs"(9)));
                sysTime.roll!"months"(-85, AllowDayOverflow.no);
                _assertPred!"=="(sysTime, SysTime(DateTime(-3, 3, 30, 12, 11, 10), FracSec.from!"msecs"(9)));
            }
        }
    }


    /++
        Adds the given number of units to this $(D SysTime). A negative number
        will subtract.

        The difference between rolling and adding is that rolling does not
        affect larger units. For instance, rolling a $(D SysTime) one
        year's worth of days gets the exact same $(D SysTime).

        Accepted units are $(D "days"), $(D "minutes"), $(D "hours"),
        $(D "minutes"), $(D "seconds"), $(D "msecs"), $(D "usecs"), and
        $(D "hnsecs").

        Note that when rolling msecs, usecs or hnsecs, they all add up to a
        second. So, for example, rolling 1000 msecs is exactly the same as
        rolling 100,000 usecs.

        Params:
            units = The units to add.
            value = The number of $(D_PARAM units) to add to this $(D SysTime).

        Examples:
--------------------
auto st1 = SysTime(DateTime(2010, 1, 1, 11, 23, 12));
st1.roll!"days"(1);
assert(st1 == SysTime(DateTime(2010, 1, 2, 11, 23, 12)));
st1.roll!"days"(365);
assert(st1 == SysTime(DateTime(2010, 1, 26, 11, 23, 12)));
st1.roll!"days"(-32);
assert(st1 == SysTime(DateTime(2010, 1, 25, 11, 23, 12)));

auto st2 = SysTime(DateTime(2010, 7, 4, 12, 0, 0));
st2.roll!"hours"(1);
assert(st2 == SysTime(DateTime(2010, 7, 4, 13, 0, 0)));

auto st3 = SysTime(DateTime(2010, 1, 1, 0, 0, 0));
st3.roll!"seconds"(-1);
assert(st3 == SysTime(DateTime(2010, 1, 1, 0, 0, 59)));

auto st4 = SysTime(DateTime(2010, 1, 1, 0, 0, 0),
                   FracSec.from!"usecs"(2_400));
st4.roll!"usecs"(-1_200_000);
assert(st4 == SysTime(DateTime(2010, 1, 1, 0, 0, 0),
                      FracSec.from!"usecs"(802_400)));
--------------------
      +/
    /+ref SysTime+/ void roll(string units)(long value) nothrow
        if(units == "days")
    {
        auto hnsecs = adjTime;
        auto gdays = splitUnitsFromHNSecs!"days"(hnsecs) + 1;

        if(hnsecs < 0)
        {
            hnsecs += convert!("hours", "hnsecs")(24);
            --gdays;
        }

        auto date = Date(cast(int)gdays);
        date.roll!"days"(value);
        gdays = date.dayOfGregorianCal - 1;

        if(gdays < 0)
        {
            hnsecs -= convert!("hours", "hnsecs")(24);
            ++gdays;
        }

        immutable newDaysHNSecs = convert!("days", "hnsecs")(gdays);

        adjTime = newDaysHNSecs + hnsecs;
    }

    //Verify Examples.
    unittest
    {
        version(testStdDateTime)
        {
            auto st1 = SysTime(DateTime(2010, 1, 1, 11, 23, 12));
            st1.roll!"days"(1);
            assert(st1 == SysTime(DateTime(2010, 1, 2, 11, 23, 12)));
            st1.roll!"days"(365);
            assert(st1 == SysTime(DateTime(2010, 1, 26, 11, 23, 12)));
            st1.roll!"days"(-32);
            assert(st1 == SysTime(DateTime(2010, 1, 25, 11, 23, 12)));

            auto st2 = SysTime(DateTime(2010, 7, 4, 12, 0, 0));
            st2.roll!"hours"(1);
            assert(st2 == SysTime(DateTime(2010, 7, 4, 13, 0, 0)));

            auto st3 = SysTime(DateTime(2010, 1, 1, 0, 0, 0));
            st3.roll!"seconds"(-1);
            assert(st3 == SysTime(DateTime(2010, 1, 1, 0, 0, 59)));

            auto st4 = SysTime(DateTime(2010, 1, 1, 0, 0, 0),
                               FracSec.from!"usecs"(2_400));
            st4.roll!"usecs"(-1_200_000);
            assert(st4 == SysTime(DateTime(2010, 1, 1, 0, 0, 0),
                                  FracSec.from!"usecs"(802_400)));
        }
    }

    unittest
    {
        version(testStdDateTime)
        {
            //Test A.D.
            {
                auto sysTime = SysTime(Date(1999, 2, 28));
                sysTime.roll!"days"(1);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 2, 1)));
                sysTime.roll!"days"(-1);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 2, 28)));
            }

            {
                auto sysTime = SysTime(Date(2000, 2, 28));
                sysTime.roll!"days"(1);
                _assertPred!"=="(sysTime, SysTime(Date(2000, 2, 29)));
                sysTime.roll!"days"(1);
                _assertPred!"=="(sysTime, SysTime(Date(2000, 2, 1)));
                sysTime.roll!"days"(-1);
                _assertPred!"=="(sysTime, SysTime(Date(2000, 2, 29)));
            }

            {
                auto sysTime = SysTime(Date(1999, 6, 30));
                sysTime.roll!"days"(1);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 6, 1)));
                sysTime.roll!"days"(-1);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 6, 30)));
            }

            {
                auto sysTime = SysTime(Date(1999, 7, 31));
                sysTime.roll!"days"(1);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 7, 1)));
                sysTime.roll!"days"(-1);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 7, 31)));
            }

            {
                auto sysTime = SysTime(Date(1999, 1, 1));
                sysTime.roll!"days"(-1);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 1, 31)));
                sysTime.roll!"days"(1);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 1, 1)));
            }

            {
                auto sysTime = SysTime(Date(1999, 7, 6));
                sysTime.roll!"days"(9);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 7, 15)));
                sysTime.roll!"days"(-11);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 7, 4)));
                sysTime.roll!"days"(30);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 7, 3)));
                sysTime.roll!"days"(-3);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 7, 31)));
            }

            {
                auto sysTime = SysTime(Date(1999, 7, 6));
                sysTime.roll!"days"(365);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 7, 30)));
                sysTime.roll!"days"(-365);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 7, 6)));
                sysTime.roll!"days"(366);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 7, 31)));
                sysTime.roll!"days"(730);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 7, 17)));
                sysTime.roll!"days"(-1096);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 7, 6)));
            }

            {
                auto sysTime = SysTime(Date(1999, 2, 6));
                sysTime.roll!"days"(365);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 2, 7)));
                sysTime.roll!"days"(-365);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 2, 6)));
                sysTime.roll!"days"(366);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 2, 8)));
                sysTime.roll!"days"(730);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 2, 10)));
                sysTime.roll!"days"(-1096);
                _assertPred!"=="(sysTime, SysTime(Date(1999, 2, 6)));
            }

            {
                auto sysTime = SysTime(DateTime(1999, 2, 28, 7, 9, 2), FracSec.from!"usecs"(234578));
                sysTime.roll!"days"(1);
                _assertPred!"=="(sysTime, SysTime(DateTime(1999, 2, 1, 7, 9, 2), FracSec.from!"usecs"(234578)));
                sysTime.roll!"days"(-1);
                _assertPred!"=="(sysTime, SysTime(DateTime(1999, 2, 28, 7, 9, 2), FracSec.from!"usecs"(234578)));
            }

            {
                auto sysTime = SysTime(DateTime(1999, 7, 6, 7, 9, 2), FracSec.from!"usecs"(234578));
                sysTime.roll!"days"(9);
                _assertPred!"=="(sysTime, SysTime(DateTime(1999, 7, 15, 7, 9, 2), FracSec.from!"usecs"(234578)));
                sysTime.roll!"days"(-11);
                _assertPred!"=="(sysTime, SysTime(DateTime(1999, 7, 4, 7, 9, 2), FracSec.from!"usecs"(234578)));
                sysTime.roll!"days"(30);
                _assertPred!"=="(sysTime, SysTime(DateTime(1999, 7, 3, 7, 9, 2), FracSec.from!"usecs"(234578)));
                sysTime.roll!"days"(-3);
                _assertPred!"=="(sysTime, SysTime(DateTime(1999, 7, 31, 7, 9, 2), FracSec.from!"usecs"(234578)));
            }

            //Test B.C.
            {
                auto sysTime = SysTime(Date(-1999, 2, 28));
                sysTime.roll!"days"(1);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 2, 1)));
                sysTime.roll!"days"(-1);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 2, 28)));
            }

            {
                auto sysTime = SysTime(Date(-2000, 2, 28));
                sysTime.roll!"days"(1);
                _assertPred!"=="(sysTime, SysTime(Date(-2000, 2, 29)));
                sysTime.roll!"days"(1);
                _assertPred!"=="(sysTime, SysTime(Date(-2000, 2, 1)));
                sysTime.roll!"days"(-1);
                _assertPred!"=="(sysTime, SysTime(Date(-2000, 2, 29)));
            }

            {
                auto sysTime = SysTime(Date(-1999, 6, 30));
                sysTime.roll!"days"(1);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 6, 1)));
                sysTime.roll!"days"(-1);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 6, 30)));
            }

            {
                auto sysTime = SysTime(Date(-1999, 7, 31));
                sysTime.roll!"days"(1);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 7, 1)));
                sysTime.roll!"days"(-1);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 7, 31)));
            }

            {
                auto sysTime = SysTime(Date(-1999, 1, 1));
                sysTime.roll!"days"(-1);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 1, 31)));
                sysTime.roll!"days"(1);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 1, 1)));
            }

            {
                auto sysTime = SysTime(Date(-1999, 7, 6));
                sysTime.roll!"days"(9);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 7, 15)));
                sysTime.roll!"days"(-11);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 7, 4)));
                sysTime.roll!"days"(30);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 7, 3)));
                sysTime.roll!"days"(-3);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 7, 31)));
            }

            {
                auto sysTime = SysTime(Date(-1999, 7, 6));
                sysTime.roll!"days"(365);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 7, 30)));
                sysTime.roll!"days"(-365);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 7, 6)));
                sysTime.roll!"days"(366);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 7, 31)));
                sysTime.roll!"days"(730);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 7, 17)));
                sysTime.roll!"days"(-1096);
                _assertPred!"=="(sysTime, SysTime(Date(-1999, 7, 6)));
            }

            {
                auto sysTime = SysTime(DateTime(-1999, 2, 28, 7, 9, 2), FracSec.from!"usecs"(234578));
                sysTime.roll!"days"(1);
                _assertPred!"=="(sysTime, SysTime(DateTime(-1999, 2, 1, 7, 9, 2), FracSec.from!"usecs"(234578)));
                sysTime.roll!"days"(-1);
                _assertPred!"=="(sysTime, SysTime(DateTime(-1999, 2, 28, 7, 9, 2), FracSec.from!"usecs"(234578)));
            }

            {
                auto sysTime = SysTime(DateTime(-1999, 7, 6, 7, 9, 2), FracSec.from!"usecs"(234578));
                sysTime.roll!"days"(9);
                _assertPred!"=="(sysTime, SysTime(DateTime(-1999, 7, 15, 7, 9, 2), FracSec.from!"usecs"(234578)));
                sysTime.roll!"days"(-11);
                _assertPred!"=="(sysTime, SysTime(DateTime(-1999, 7, 4, 7, 9, 2), FracSec.from!"usecs"(234578)));
                sysTime.roll!"days"(30);
                _assertPred!"=="(sysTime, SysTime(DateTime(-1999, 7, 3, 7, 9, 2), FracSec.from!"usecs"(234578)));
                sysTime.roll!"days"(-3);
            }

            //Test Both
            {
                auto sysTime = SysTime(Date(1, 7, 6));
                sysTime.roll!"days"(-365);
                _assertPred!"=="(sysTime, SysTime(Date(1, 7, 13)));
                sysTime.roll!"days"(365);
                _assertPred!"=="(sysTime, SysTime(Date(1, 7, 6)));
                sysTime.roll!"days"(-731);
                _assertPred!"=="(sysTime, SysTime(Date(1, 7, 19)));
                sysTime.roll!"days"(730);
                _assertPred!"=="(sysTime, SysTime(Date(1, 7, 5)));
            }

            {
                auto sysTime = SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0));
                sysTime.roll!"days"(-1);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 1, 31, 0, 0, 0), FracSec.from!"hnsecs"(0)));
                sysTime.roll!"days"(1);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)));
            }

            {
                auto sysTime = SysTime(DateTime(1, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999));
                sysTime.roll!"days"(-1);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 1, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
                sysTime.roll!"days"(1);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            }

            {
                auto sysTime = SysTime(DateTime(0, 12, 31, 0, 0, 0), FracSec.from!"hnsecs"(0));
                sysTime.roll!"days"(1);
                _assertPred!"=="(sysTime, SysTime(DateTime(0, 12, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)));
                sysTime.roll!"days"(-1);
                _assertPred!"=="(sysTime, SysTime(DateTime(0, 12, 31, 0, 0, 0), FracSec.from!"hnsecs"(0)));
            }

            {
                auto sysTime = SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999));
                sysTime.roll!"days"(1);
                _assertPred!"=="(sysTime, SysTime(DateTime(0, 12, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
                sysTime.roll!"days"(-1);
                _assertPred!"=="(sysTime, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            }

            {
                auto sysTime = SysTime(DateTime(1, 7, 6, 13, 13, 9), FracSec.from!"msecs"(22));
                sysTime.roll!"days"(-365);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 7, 13, 13, 13, 9), FracSec.from!"msecs"(22)));
                sysTime.roll!"days"(365);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 7, 6, 13, 13, 9), FracSec.from!"msecs"(22)));
                sysTime.roll!"days"(-731);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 7, 19, 13, 13, 9), FracSec.from!"msecs"(22)));
                sysTime.roll!"days"(730);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 7, 5, 13, 13, 9), FracSec.from!"msecs"(22)));
            }

            {
                auto sysTime = SysTime(DateTime(0, 7, 6, 13, 13, 9), FracSec.from!"msecs"(22));
                sysTime.roll!"days"(-365);
                _assertPred!"=="(sysTime, SysTime(DateTime(0, 7, 13, 13, 13, 9), FracSec.from!"msecs"(22)));
                sysTime.roll!"days"(365);
                _assertPred!"=="(sysTime, SysTime(DateTime(0, 7, 6, 13, 13, 9), FracSec.from!"msecs"(22)));
                sysTime.roll!"days"(-731);
                _assertPred!"=="(sysTime, SysTime(DateTime(0, 7, 19, 13, 13, 9), FracSec.from!"msecs"(22)));
                sysTime.roll!"days"(730);
                _assertPred!"=="(sysTime, SysTime(DateTime(0, 7, 5, 13, 13, 9), FracSec.from!"msecs"(22)));
            }

            const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            static assert(!__traits(compiles, cst.roll!"days"(4)));
            //static assert(!__traits(compiles, ist.roll!"days"(4)));

            //Verify Examples.
            auto st = SysTime(DateTime(2010, 1, 1, 11, 23, 12));
            st.roll!"days"(1);
            assert(st == SysTime(DateTime(2010, 1, 2, 11, 23, 12)));
            st.roll!"days"(365);
            assert(st == SysTime(DateTime(2010, 1, 26, 11, 23, 12)));
            st.roll!"days"(-32);
            assert(st == SysTime(DateTime(2010, 1, 25, 11, 23, 12)));
        }
    }


    //Shares documentation with "days" version.
    /+ref SysTime+/ void roll(string units)(long value) nothrow
        if(units == "hours" ||
           units == "minutes" ||
           units == "seconds")
    {
        try
        {
            auto hnsecs = adjTime;
            auto days = splitUnitsFromHNSecs!"days"(hnsecs) + 1;

            if(hnsecs < 0)
            {
                hnsecs += convert!("hours", "hnsecs")(24);
                --days;
            }

            immutable hour = splitUnitsFromHNSecs!"hours"(hnsecs);
            immutable minute = splitUnitsFromHNSecs!"minutes"(hnsecs);
            immutable second = splitUnitsFromHNSecs!"seconds"(hnsecs);

            auto dateTime = DateTime(Date(cast(int)days), TimeOfDay(cast(int)hour, cast(int)minute, cast(int)second));
            dateTime.roll!units(value);
            --days;

            hnsecs += convert!("hours", "hnsecs")(dateTime.hour);
            hnsecs += convert!("minutes", "hnsecs")(dateTime.minute);
            hnsecs += convert!("seconds", "hnsecs")(dateTime.second);

            if(days < 0)
            {
                hnsecs -= convert!("hours", "hnsecs")(24);
                ++days;
            }

            immutable newDaysHNSecs = convert!("days", "hnsecs")(days);

            adjTime = newDaysHNSecs + hnsecs;
        }
        catch(Exception e)
            assert(0, "Either DateTime's constructor or TimeOfDay's constructor threw.");
    }

    //Test roll!"hours"().
    unittest
    {
        version(testStdDateTime)
        {
            static void TestST(SysTime orig, int hours, in SysTime expected, size_t line = __LINE__)
            {
                orig.roll!"hours"(hours);
                _assertPred!"=="(orig, expected, "", __FILE__, line);
            }

            //Test A.D.
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 0, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 1, SysTime(DateTime(1999, 7, 6, 13, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 2, SysTime(DateTime(1999, 7, 6, 14, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 3, SysTime(DateTime(1999, 7, 6, 15, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 4, SysTime(DateTime(1999, 7, 6, 16, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 5, SysTime(DateTime(1999, 7, 6, 17, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 6, SysTime(DateTime(1999, 7, 6, 18, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 7, SysTime(DateTime(1999, 7, 6, 19, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 8, SysTime(DateTime(1999, 7, 6, 20, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 9, SysTime(DateTime(1999, 7, 6, 21, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 10, SysTime(DateTime(1999, 7, 6, 22, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 11, SysTime(DateTime(1999, 7, 6, 23, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 12, SysTime(DateTime(1999, 7, 6, 0, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 13, SysTime(DateTime(1999, 7, 6, 1, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 14, SysTime(DateTime(1999, 7, 6, 2, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 15, SysTime(DateTime(1999, 7, 6, 3, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 16, SysTime(DateTime(1999, 7, 6, 4, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 17, SysTime(DateTime(1999, 7, 6, 5, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 18, SysTime(DateTime(1999, 7, 6, 6, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 19, SysTime(DateTime(1999, 7, 6, 7, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 20, SysTime(DateTime(1999, 7, 6, 8, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 21, SysTime(DateTime(1999, 7, 6, 9, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 22, SysTime(DateTime(1999, 7, 6, 10, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 23, SysTime(DateTime(1999, 7, 6, 11, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 24, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 25, SysTime(DateTime(1999, 7, 6, 13, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 50, SysTime(DateTime(1999, 7, 6, 14, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 10_000, SysTime(DateTime(1999, 7, 6, 4, 30, 33), FracSec.from!"msecs"(45)));

            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -1, SysTime(DateTime(1999, 7, 6, 11, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -2, SysTime(DateTime(1999, 7, 6, 10, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -3, SysTime(DateTime(1999, 7, 6, 9, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -4, SysTime(DateTime(1999, 7, 6, 8, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -5, SysTime(DateTime(1999, 7, 6, 7, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -6, SysTime(DateTime(1999, 7, 6, 6, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -7, SysTime(DateTime(1999, 7, 6, 5, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -8, SysTime(DateTime(1999, 7, 6, 4, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -9, SysTime(DateTime(1999, 7, 6, 3, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -10, SysTime(DateTime(1999, 7, 6, 2, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -11, SysTime(DateTime(1999, 7, 6, 1, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -12, SysTime(DateTime(1999, 7, 6, 0, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -13, SysTime(DateTime(1999, 7, 6, 23, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -14, SysTime(DateTime(1999, 7, 6, 22, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -15, SysTime(DateTime(1999, 7, 6, 21, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -16, SysTime(DateTime(1999, 7, 6, 20, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -17, SysTime(DateTime(1999, 7, 6, 19, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -18, SysTime(DateTime(1999, 7, 6, 18, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -19, SysTime(DateTime(1999, 7, 6, 17, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -20, SysTime(DateTime(1999, 7, 6, 16, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -21, SysTime(DateTime(1999, 7, 6, 15, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -22, SysTime(DateTime(1999, 7, 6, 14, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -23, SysTime(DateTime(1999, 7, 6, 13, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -24, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -25, SysTime(DateTime(1999, 7, 6, 11, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -50, SysTime(DateTime(1999, 7, 6, 10, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -10_000, SysTime(DateTime(1999, 7, 6, 20, 30, 33), FracSec.from!"msecs"(45)));

            TestST(SysTime(DateTime(1999, 7, 6, 0, 30, 33), FracSec.from!"msecs"(45)), 1, SysTime(DateTime(1999, 7, 6, 1, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 0, 30, 33), FracSec.from!"msecs"(45)), 0, SysTime(DateTime(1999, 7, 6, 0, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 0, 30, 33), FracSec.from!"msecs"(45)), -1, SysTime(DateTime(1999, 7, 6, 23, 30, 33), FracSec.from!"msecs"(45)));

            TestST(SysTime(DateTime(1999, 7, 6, 23, 30, 33), FracSec.from!"msecs"(45)), 1, SysTime(DateTime(1999, 7, 6, 0, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 23, 30, 33), FracSec.from!"msecs"(45)), 0, SysTime(DateTime(1999, 7, 6, 23, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 7, 6, 23, 30, 33), FracSec.from!"msecs"(45)), -1, SysTime(DateTime(1999, 7, 6, 22, 30, 33), FracSec.from!"msecs"(45)));

            TestST(SysTime(DateTime(1999, 7, 31, 23, 30, 33), FracSec.from!"msecs"(45)), 1, SysTime(DateTime(1999, 7, 31, 0, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 8, 1, 0, 30, 33), FracSec.from!"msecs"(45)), -1, SysTime(DateTime(1999, 8, 1, 23, 30, 33), FracSec.from!"msecs"(45)));

            TestST(SysTime(DateTime(1999, 12, 31, 23, 30, 33), FracSec.from!"msecs"(45)), 1, SysTime(DateTime(1999, 12, 31, 0, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(2000, 1, 1, 0, 30, 33), FracSec.from!"msecs"(45)), -1, SysTime(DateTime(2000, 1, 1, 23, 30, 33), FracSec.from!"msecs"(45)));

            TestST(SysTime(DateTime(1999, 2, 28, 23, 30, 33), FracSec.from!"msecs"(45)), 25, SysTime(DateTime(1999, 2, 28, 0, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1999, 3, 2, 0, 30, 33), FracSec.from!"msecs"(45)), -25, SysTime(DateTime(1999, 3, 2, 23, 30, 33), FracSec.from!"msecs"(45)));

            TestST(SysTime(DateTime(2000, 2, 28, 23, 30, 33), FracSec.from!"msecs"(45)), 25, SysTime(DateTime(2000, 2, 28, 0, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(2000, 3, 1, 0, 30, 33), FracSec.from!"msecs"(45)), -25, SysTime(DateTime(2000, 3, 1, 23, 30, 33), FracSec.from!"msecs"(45)));

            //Test B.C.
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 0, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 1, SysTime(DateTime(-1999, 7, 6, 13, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 2, SysTime(DateTime(-1999, 7, 6, 14, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 3, SysTime(DateTime(-1999, 7, 6, 15, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 4, SysTime(DateTime(-1999, 7, 6, 16, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 5, SysTime(DateTime(-1999, 7, 6, 17, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 6, SysTime(DateTime(-1999, 7, 6, 18, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 7, SysTime(DateTime(-1999, 7, 6, 19, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 8, SysTime(DateTime(-1999, 7, 6, 20, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 9, SysTime(DateTime(-1999, 7, 6, 21, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 10, SysTime(DateTime(-1999, 7, 6, 22, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 11, SysTime(DateTime(-1999, 7, 6, 23, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 12, SysTime(DateTime(-1999, 7, 6, 0, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 13, SysTime(DateTime(-1999, 7, 6, 1, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 14, SysTime(DateTime(-1999, 7, 6, 2, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 15, SysTime(DateTime(-1999, 7, 6, 3, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 16, SysTime(DateTime(-1999, 7, 6, 4, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 17, SysTime(DateTime(-1999, 7, 6, 5, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 18, SysTime(DateTime(-1999, 7, 6, 6, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 19, SysTime(DateTime(-1999, 7, 6, 7, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 20, SysTime(DateTime(-1999, 7, 6, 8, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 21, SysTime(DateTime(-1999, 7, 6, 9, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 22, SysTime(DateTime(-1999, 7, 6, 10, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 23, SysTime(DateTime(-1999, 7, 6, 11, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 24, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 25, SysTime(DateTime(-1999, 7, 6, 13, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 50, SysTime(DateTime(-1999, 7, 6, 14, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), 10_000, SysTime(DateTime(-1999, 7, 6, 4, 30, 33), FracSec.from!"msecs"(45)));

            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -1, SysTime(DateTime(-1999, 7, 6, 11, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -2, SysTime(DateTime(-1999, 7, 6, 10, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -3, SysTime(DateTime(-1999, 7, 6, 9, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -4, SysTime(DateTime(-1999, 7, 6, 8, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -5, SysTime(DateTime(-1999, 7, 6, 7, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -6, SysTime(DateTime(-1999, 7, 6, 6, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -7, SysTime(DateTime(-1999, 7, 6, 5, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -8, SysTime(DateTime(-1999, 7, 6, 4, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -9, SysTime(DateTime(-1999, 7, 6, 3, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -10, SysTime(DateTime(-1999, 7, 6, 2, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -11, SysTime(DateTime(-1999, 7, 6, 1, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -12, SysTime(DateTime(-1999, 7, 6, 0, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -13, SysTime(DateTime(-1999, 7, 6, 23, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -14, SysTime(DateTime(-1999, 7, 6, 22, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -15, SysTime(DateTime(-1999, 7, 6, 21, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -16, SysTime(DateTime(-1999, 7, 6, 20, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -17, SysTime(DateTime(-1999, 7, 6, 19, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -18, SysTime(DateTime(-1999, 7, 6, 18, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -19, SysTime(DateTime(-1999, 7, 6, 17, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -20, SysTime(DateTime(-1999, 7, 6, 16, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -21, SysTime(DateTime(-1999, 7, 6, 15, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -22, SysTime(DateTime(-1999, 7, 6, 14, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -23, SysTime(DateTime(-1999, 7, 6, 13, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -24, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -25, SysTime(DateTime(-1999, 7, 6, 11, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -50, SysTime(DateTime(-1999, 7, 6, 10, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(45)), -10_000, SysTime(DateTime(-1999, 7, 6, 20, 30, 33), FracSec.from!"msecs"(45)));

            TestST(SysTime(DateTime(-1999, 7, 6, 0, 30, 33), FracSec.from!"msecs"(45)), 1, SysTime(DateTime(-1999, 7, 6, 1, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 0, 30, 33), FracSec.from!"msecs"(45)), 0, SysTime(DateTime(-1999, 7, 6, 0, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 0, 30, 33), FracSec.from!"msecs"(45)), -1, SysTime(DateTime(-1999, 7, 6, 23, 30, 33), FracSec.from!"msecs"(45)));

            TestST(SysTime(DateTime(-1999, 7, 6, 23, 30, 33), FracSec.from!"msecs"(45)), 1, SysTime(DateTime(-1999, 7, 6, 0, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 23, 30, 33), FracSec.from!"msecs"(45)), 0, SysTime(DateTime(-1999, 7, 6, 23, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 7, 6, 23, 30, 33), FracSec.from!"msecs"(45)), -1, SysTime(DateTime(-1999, 7, 6, 22, 30, 33), FracSec.from!"msecs"(45)));

            TestST(SysTime(DateTime(-1999, 7, 31, 23, 30, 33), FracSec.from!"msecs"(45)), 1, SysTime(DateTime(-1999, 7, 31, 0, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-1999, 8, 1, 0, 30, 33), FracSec.from!"msecs"(45)), -1, SysTime(DateTime(-1999, 8, 1, 23, 30, 33), FracSec.from!"msecs"(45)));

            TestST(SysTime(DateTime(-2001, 12, 31, 23, 30, 33), FracSec.from!"msecs"(45)), 1, SysTime(DateTime(-2001, 12, 31, 0, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-2000, 1, 1, 0, 30, 33), FracSec.from!"msecs"(45)), -1, SysTime(DateTime(-2000, 1, 1, 23, 30, 33), FracSec.from!"msecs"(45)));

            TestST(SysTime(DateTime(-2001, 2, 28, 23, 30, 33), FracSec.from!"msecs"(45)), 25, SysTime(DateTime(-2001, 2, 28, 0, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-2001, 3, 2, 0, 30, 33), FracSec.from!"msecs"(45)), -25, SysTime(DateTime(-2001, 3, 2, 23, 30, 33), FracSec.from!"msecs"(45)));

            TestST(SysTime(DateTime(-2000, 2, 28, 23, 30, 33), FracSec.from!"msecs"(45)), 25, SysTime(DateTime(-2000, 2, 28, 0, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(-2000, 3, 1, 0, 30, 33), FracSec.from!"msecs"(45)), -25, SysTime(DateTime(-2000, 3, 1, 23, 30, 33), FracSec.from!"msecs"(45)));

            //Test Both
            TestST(SysTime(DateTime(-1, 1, 1, 11, 30, 33), FracSec.from!"msecs"(45)), 17_546, SysTime(DateTime(-1, 1, 1, 13, 30, 33), FracSec.from!"msecs"(45)));
            TestST(SysTime(DateTime(1, 1, 1, 13, 30, 33), FracSec.from!"msecs"(45)), -17_546, SysTime(DateTime(1, 1, 1, 11, 30, 33), FracSec.from!"msecs"(45)));

            {
                auto sysTime = SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0));
                sysTime.roll!"hours"(-1);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 1, 1, 23, 0, 0), FracSec.from!"hnsecs"(0)));
                sysTime.roll!"hours"(1);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)));
            }

            {
                auto sysTime = SysTime(DateTime(1, 1, 1, 0, 59, 59), FracSec.from!"hnsecs"(9_999_999));
                sysTime.roll!"hours"(-1);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
                sysTime.roll!"hours"(1);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 1, 1, 0, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            }

            {
                auto sysTime = SysTime(DateTime(0, 12, 31, 23, 0, 0), FracSec.from!"hnsecs"(0));
                sysTime.roll!"hours"(1);
                _assertPred!"=="(sysTime, SysTime(DateTime(0, 12, 31, 0, 0, 0), FracSec.from!"hnsecs"(0)));
                sysTime.roll!"hours"(-1);
                _assertPred!"=="(sysTime, SysTime(DateTime(0, 12, 31, 23, 0, 0), FracSec.from!"hnsecs"(0)));
            }

            {
                auto sysTime = SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999));
                sysTime.roll!"hours"(1);
                _assertPred!"=="(sysTime, SysTime(DateTime(0, 12, 31, 0, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
                sysTime.roll!"hours"(-1);
                _assertPred!"=="(sysTime, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            }

            const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            static assert(!__traits(compiles, cst.roll!"hours"(4)));
            //static assert(!__traits(compiles, ist.roll!"hours"(4)));

            //Verify Examples.
            auto st1 = SysTime(DateTime(2010, 7, 4, 12, 0, 0));
            st1.roll!"hours"(1);
            assert(st1 == SysTime(DateTime(2010, 7, 4, 13, 0, 0)));

            auto st2 = SysTime(DateTime(2010, 2, 12, 12, 0, 0));
            st2.roll!"hours"(-1);
            assert(st2 == SysTime(DateTime(2010, 2, 12, 11, 0, 0)));

            auto st3 = SysTime(DateTime(2009, 12, 31, 0, 0, 0));
            st3.roll!"minutes"(1);
            assert(st3 == SysTime(DateTime(2009, 12, 31, 0, 1, 0)));

            auto st4 = SysTime(DateTime(2010, 1, 1, 0, 0, 0));
            st4.roll!"minutes"(-1);
            assert(st4 == SysTime(DateTime(2010, 1, 1, 0, 59, 0)));

            auto st5 = SysTime(DateTime(2009, 12, 31, 0, 0, 0));
            st5.roll!"seconds"(1);
            assert(st5 == SysTime(DateTime(2009, 12, 31, 0, 0, 1)));

            auto st6 = SysTime(DateTime(2010, 1, 1, 0, 0, 0));
            st6.roll!"seconds"(-1);
            assert(st6 == SysTime(DateTime(2010, 1, 1, 0, 0, 59)));
        }
    }

    //Test roll!"minutes"().
    unittest
    {
        version(testStdDateTime)
        {
            static void TestST(SysTime orig, int minutes, in SysTime expected, size_t line = __LINE__)
            {
                orig.roll!"minutes"(minutes);
                _assertPred!"=="(orig, expected, "", __FILE__, line);
            }

            //Test A.D.
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), 0, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), 1, SysTime(DateTime(1999, 7, 6, 12, 31, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), 2, SysTime(DateTime(1999, 7, 6, 12, 32, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), 3, SysTime(DateTime(1999, 7, 6, 12, 33, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), 4, SysTime(DateTime(1999, 7, 6, 12, 34, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), 5, SysTime(DateTime(1999, 7, 6, 12, 35, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), 10, SysTime(DateTime(1999, 7, 6, 12, 40, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), 15, SysTime(DateTime(1999, 7, 6, 12, 45, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), 29, SysTime(DateTime(1999, 7, 6, 12, 59, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), 30, SysTime(DateTime(1999, 7, 6, 12, 0, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), 45, SysTime(DateTime(1999, 7, 6, 12, 15, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), 60, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), 75, SysTime(DateTime(1999, 7, 6, 12, 45, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), 90, SysTime(DateTime(1999, 7, 6, 12, 0, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), 100, SysTime(DateTime(1999, 7, 6, 12, 10, 33), FracSec.from!"usecs"(7203)));

            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), 689, SysTime(DateTime(1999, 7, 6, 12, 59, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), 690, SysTime(DateTime(1999, 7, 6, 12, 0, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), 691, SysTime(DateTime(1999, 7, 6, 12, 1, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), 960, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), 1439, SysTime(DateTime(1999, 7, 6, 12, 29, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), 1440, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), 1441, SysTime(DateTime(1999, 7, 6, 12, 31, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), 2880, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)));

            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), -1, SysTime(DateTime(1999, 7, 6, 12, 29, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), -2, SysTime(DateTime(1999, 7, 6, 12, 28, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), -3, SysTime(DateTime(1999, 7, 6, 12, 27, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), -4, SysTime(DateTime(1999, 7, 6, 12, 26, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), -5, SysTime(DateTime(1999, 7, 6, 12, 25, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), -10, SysTime(DateTime(1999, 7, 6, 12, 20, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), -15, SysTime(DateTime(1999, 7, 6, 12, 15, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), -29, SysTime(DateTime(1999, 7, 6, 12, 1, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), -30, SysTime(DateTime(1999, 7, 6, 12, 0, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), -45, SysTime(DateTime(1999, 7, 6, 12, 45, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), -60, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), -75, SysTime(DateTime(1999, 7, 6, 12, 15, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), -90, SysTime(DateTime(1999, 7, 6, 12, 0, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), -100, SysTime(DateTime(1999, 7, 6, 12, 50, 33), FracSec.from!"usecs"(7203)));

            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), -749, SysTime(DateTime(1999, 7, 6, 12, 1, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), -750, SysTime(DateTime(1999, 7, 6, 12, 0, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), -751, SysTime(DateTime(1999, 7, 6, 12, 59, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), -960, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), -1439, SysTime(DateTime(1999, 7, 6, 12, 31, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), -1440, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), -1441, SysTime(DateTime(1999, 7, 6, 12, 29, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), -2880, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)));

            TestST(SysTime(DateTime(1999, 7, 6, 12, 0, 33), FracSec.from!"usecs"(7203)), 1, SysTime(DateTime(1999, 7, 6, 12, 1, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 0, 33), FracSec.from!"usecs"(7203)), 0, SysTime(DateTime(1999, 7, 6, 12, 0, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 0, 33), FracSec.from!"usecs"(7203)), -1, SysTime(DateTime(1999, 7, 6, 12, 59, 33), FracSec.from!"usecs"(7203)));

            TestST(SysTime(DateTime(1999, 7, 6, 11, 59, 33), FracSec.from!"usecs"(7203)), 1, SysTime(DateTime(1999, 7, 6, 11, 0, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 6, 11, 59, 33), FracSec.from!"usecs"(7203)), 0, SysTime(DateTime(1999, 7, 6, 11, 59, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 6, 11, 59, 33), FracSec.from!"usecs"(7203)), -1, SysTime(DateTime(1999, 7, 6, 11, 58, 33), FracSec.from!"usecs"(7203)));

            TestST(SysTime(DateTime(1999, 7, 6, 0, 0, 33), FracSec.from!"usecs"(7203)), 1, SysTime(DateTime(1999, 7, 6, 0, 1, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 6, 0, 0, 33), FracSec.from!"usecs"(7203)), 0, SysTime(DateTime(1999, 7, 6, 0, 0, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 6, 0, 0, 33), FracSec.from!"usecs"(7203)), -1, SysTime(DateTime(1999, 7, 6, 0, 59, 33), FracSec.from!"usecs"(7203)));

            TestST(SysTime(DateTime(1999, 7, 5, 23, 59, 33), FracSec.from!"usecs"(7203)), 1, SysTime(DateTime(1999, 7, 5, 23, 0, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 5, 23, 59, 33), FracSec.from!"usecs"(7203)), 0, SysTime(DateTime(1999, 7, 5, 23, 59, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1999, 7, 5, 23, 59, 33), FracSec.from!"usecs"(7203)), -1, SysTime(DateTime(1999, 7, 5, 23, 58, 33), FracSec.from!"usecs"(7203)));

            TestST(SysTime(DateTime(1998, 12, 31, 23, 59, 33), FracSec.from!"usecs"(7203)), 1, SysTime(DateTime(1998, 12, 31, 23, 0, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1998, 12, 31, 23, 59, 33), FracSec.from!"usecs"(7203)), 0, SysTime(DateTime(1998, 12, 31, 23, 59, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1998, 12, 31, 23, 59, 33), FracSec.from!"usecs"(7203)), -1, SysTime(DateTime(1998, 12, 31, 23, 58, 33), FracSec.from!"usecs"(7203)));

            //Test B.C.
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), 0, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), 1, SysTime(DateTime(-1999, 7, 6, 12, 31, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), 2, SysTime(DateTime(-1999, 7, 6, 12, 32, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), 3, SysTime(DateTime(-1999, 7, 6, 12, 33, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), 4, SysTime(DateTime(-1999, 7, 6, 12, 34, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), 5, SysTime(DateTime(-1999, 7, 6, 12, 35, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), 10, SysTime(DateTime(-1999, 7, 6, 12, 40, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), 15, SysTime(DateTime(-1999, 7, 6, 12, 45, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), 29, SysTime(DateTime(-1999, 7, 6, 12, 59, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), 30, SysTime(DateTime(-1999, 7, 6, 12, 0, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), 45, SysTime(DateTime(-1999, 7, 6, 12, 15, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), 60, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), 75, SysTime(DateTime(-1999, 7, 6, 12, 45, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), 90, SysTime(DateTime(-1999, 7, 6, 12, 0, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), 100, SysTime(DateTime(-1999, 7, 6, 12, 10, 33), FracSec.from!"usecs"(7203)));

            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), 689, SysTime(DateTime(-1999, 7, 6, 12, 59, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), 690, SysTime(DateTime(-1999, 7, 6, 12, 0, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), 691, SysTime(DateTime(-1999, 7, 6, 12, 1, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), 960, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), 1439, SysTime(DateTime(-1999, 7, 6, 12, 29, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), 1440, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), 1441, SysTime(DateTime(-1999, 7, 6, 12, 31, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), 2880, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)));

            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), -1, SysTime(DateTime(-1999, 7, 6, 12, 29, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), -2, SysTime(DateTime(-1999, 7, 6, 12, 28, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), -3, SysTime(DateTime(-1999, 7, 6, 12, 27, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), -4, SysTime(DateTime(-1999, 7, 6, 12, 26, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), -5, SysTime(DateTime(-1999, 7, 6, 12, 25, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), -10, SysTime(DateTime(-1999, 7, 6, 12, 20, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), -15, SysTime(DateTime(-1999, 7, 6, 12, 15, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), -29, SysTime(DateTime(-1999, 7, 6, 12, 1, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), -30, SysTime(DateTime(-1999, 7, 6, 12, 0, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), -45, SysTime(DateTime(-1999, 7, 6, 12, 45, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), -60, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), -75, SysTime(DateTime(-1999, 7, 6, 12, 15, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), -90, SysTime(DateTime(-1999, 7, 6, 12, 0, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), -100, SysTime(DateTime(-1999, 7, 6, 12, 50, 33), FracSec.from!"usecs"(7203)));

            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), -749, SysTime(DateTime(-1999, 7, 6, 12, 1, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), -750, SysTime(DateTime(-1999, 7, 6, 12, 0, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), -751, SysTime(DateTime(-1999, 7, 6, 12, 59, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), -960, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), -1439, SysTime(DateTime(-1999, 7, 6, 12, 31, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), -1440, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), -1441, SysTime(DateTime(-1999, 7, 6, 12, 29, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)), -2880, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7203)));

            TestST(SysTime(DateTime(-1999, 7, 6, 12, 0, 33), FracSec.from!"usecs"(7203)), 1, SysTime(DateTime(-1999, 7, 6, 12, 1, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 0, 33), FracSec.from!"usecs"(7203)), 0, SysTime(DateTime(-1999, 7, 6, 12, 0, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 0, 33), FracSec.from!"usecs"(7203)), -1, SysTime(DateTime(-1999, 7, 6, 12, 59, 33), FracSec.from!"usecs"(7203)));

            TestST(SysTime(DateTime(-1999, 7, 6, 11, 59, 33), FracSec.from!"usecs"(7203)), 1, SysTime(DateTime(-1999, 7, 6, 11, 0, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 6, 11, 59, 33), FracSec.from!"usecs"(7203)), 0, SysTime(DateTime(-1999, 7, 6, 11, 59, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 6, 11, 59, 33), FracSec.from!"usecs"(7203)), -1, SysTime(DateTime(-1999, 7, 6, 11, 58, 33), FracSec.from!"usecs"(7203)));

            TestST(SysTime(DateTime(-1999, 7, 6, 0, 0, 33), FracSec.from!"usecs"(7203)), 1, SysTime(DateTime(-1999, 7, 6, 0, 1, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 6, 0, 0, 33), FracSec.from!"usecs"(7203)), 0, SysTime(DateTime(-1999, 7, 6, 0, 0, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 6, 0, 0, 33), FracSec.from!"usecs"(7203)), -1, SysTime(DateTime(-1999, 7, 6, 0, 59, 33), FracSec.from!"usecs"(7203)));

            TestST(SysTime(DateTime(-1999, 7, 5, 23, 59, 33), FracSec.from!"usecs"(7203)), 1, SysTime(DateTime(-1999, 7, 5, 23, 0, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 5, 23, 59, 33), FracSec.from!"usecs"(7203)), 0, SysTime(DateTime(-1999, 7, 5, 23, 59, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-1999, 7, 5, 23, 59, 33), FracSec.from!"usecs"(7203)), -1, SysTime(DateTime(-1999, 7, 5, 23, 58, 33), FracSec.from!"usecs"(7203)));

            TestST(SysTime(DateTime(-2000, 12, 31, 23, 59, 33), FracSec.from!"usecs"(7203)), 1, SysTime(DateTime(-2000, 12, 31, 23, 0, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-2000, 12, 31, 23, 59, 33), FracSec.from!"usecs"(7203)), 0, SysTime(DateTime(-2000, 12, 31, 23, 59, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(-2000, 12, 31, 23, 59, 33), FracSec.from!"usecs"(7203)), -1, SysTime(DateTime(-2000, 12, 31, 23, 58, 33), FracSec.from!"usecs"(7203)));

            //Test Both
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0)), -1, SysTime(DateTime(1, 1, 1, 0, 59, 0)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 0)), 1, SysTime(DateTime(0, 12, 31, 23, 0, 0)));

            TestST(SysTime(DateTime(0, 1, 1, 0, 0, 0)), -1, SysTime(DateTime(0, 1, 1, 0, 59, 0)));
            TestST(SysTime(DateTime(-1, 12, 31, 23, 59, 0)), 1, SysTime(DateTime(-1, 12, 31, 23, 0, 0)));

            TestST(SysTime(DateTime(-1, 1, 1, 11, 30, 33), FracSec.from!"usecs"(7203)), 1_052_760, SysTime(DateTime(-1, 1, 1, 11, 30, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1, 1, 1, 13, 30, 33), FracSec.from!"usecs"(7203)), -1_052_760, SysTime(DateTime(1, 1, 1, 13, 30, 33), FracSec.from!"usecs"(7203)));

            TestST(SysTime(DateTime(-1, 1, 1, 11, 30, 33), FracSec.from!"usecs"(7203)), 1_052_782, SysTime(DateTime(-1, 1, 1, 11, 52, 33), FracSec.from!"usecs"(7203)));
            TestST(SysTime(DateTime(1, 1, 1, 13, 52, 33), FracSec.from!"usecs"(7203)), -1_052_782, SysTime(DateTime(1, 1, 1, 13, 30, 33), FracSec.from!"usecs"(7203)));

            {
                auto sysTime = SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0));
                sysTime.roll!"minutes"(-1);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 1, 1, 0, 59, 0), FracSec.from!"hnsecs"(0)));
                sysTime.roll!"minutes"(1);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)));
            }

            {
                auto sysTime = SysTime(DateTime(1, 1, 1, 0, 0, 59), FracSec.from!"hnsecs"(9_999_999));
                sysTime.roll!"minutes"(-1);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 1, 1, 0, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
                sysTime.roll!"minutes"(1);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 1, 1, 0, 0, 59), FracSec.from!"hnsecs"(9_999_999)));
            }

            {
                auto sysTime = SysTime(DateTime(0, 12, 31, 23, 59, 0), FracSec.from!"hnsecs"(0));
                sysTime.roll!"minutes"(1);
                _assertPred!"=="(sysTime, SysTime(DateTime(0, 12, 31, 23, 0, 0), FracSec.from!"hnsecs"(0)));
                sysTime.roll!"minutes"(-1);
                _assertPred!"=="(sysTime, SysTime(DateTime(0, 12, 31, 23, 59, 0), FracSec.from!"hnsecs"(0)));
            }

            {
                auto sysTime = SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999));
                sysTime.roll!"minutes"(1);
                _assertPred!"=="(sysTime, SysTime(DateTime(0, 12, 31, 23, 0, 59), FracSec.from!"hnsecs"(9_999_999)));
                sysTime.roll!"minutes"(-1);
                _assertPred!"=="(sysTime, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            }

            const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            static assert(!__traits(compiles, cst.roll!"minutes"(4)));
            //static assert(!__traits(compiles, ist.roll!"minutes"(4)));
        }
    }

    //Test roll!"seconds"().
    unittest
    {
        version(testStdDateTime)
        {
            static void TestST(SysTime orig, int seconds, in SysTime expected, size_t line = __LINE__)
            {
                orig.roll!"seconds"(seconds);
                _assertPred!"=="(orig, expected, "", __FILE__, line);
            }

            //Test A.D.
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 0, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 1, SysTime(DateTime(1999, 7, 6, 12, 30, 34), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 2, SysTime(DateTime(1999, 7, 6, 12, 30, 35), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 3, SysTime(DateTime(1999, 7, 6, 12, 30, 36), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 4, SysTime(DateTime(1999, 7, 6, 12, 30, 37), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 5, SysTime(DateTime(1999, 7, 6, 12, 30, 38), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 10, SysTime(DateTime(1999, 7, 6, 12, 30, 43), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 15, SysTime(DateTime(1999, 7, 6, 12, 30, 48), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 26, SysTime(DateTime(1999, 7, 6, 12, 30, 59), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 27, SysTime(DateTime(1999, 7, 6, 12, 30, 0), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 30, SysTime(DateTime(1999, 7, 6, 12, 30, 3), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 59, SysTime(DateTime(1999, 7, 6, 12, 30, 32), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 60, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 61, SysTime(DateTime(1999, 7, 6, 12, 30, 34), FracSec.from!"msecs"(274)));

            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 1766, SysTime(DateTime(1999, 7, 6, 12, 30, 59), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 1767, SysTime(DateTime(1999, 7, 6, 12, 30, 0), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 1768, SysTime(DateTime(1999, 7, 6, 12, 30, 1), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 2007, SysTime(DateTime(1999, 7, 6, 12, 30, 0), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 3599, SysTime(DateTime(1999, 7, 6, 12, 30, 32), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 3600, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 3601, SysTime(DateTime(1999, 7, 6, 12, 30, 34), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 7200, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)));

            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -1, SysTime(DateTime(1999, 7, 6, 12, 30, 32), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -2, SysTime(DateTime(1999, 7, 6, 12, 30, 31), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -3, SysTime(DateTime(1999, 7, 6, 12, 30, 30), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -4, SysTime(DateTime(1999, 7, 6, 12, 30, 29), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -5, SysTime(DateTime(1999, 7, 6, 12, 30, 28), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -10, SysTime(DateTime(1999, 7, 6, 12, 30, 23), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -15, SysTime(DateTime(1999, 7, 6, 12, 30, 18), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -33, SysTime(DateTime(1999, 7, 6, 12, 30, 0), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -34, SysTime(DateTime(1999, 7, 6, 12, 30, 59), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -35, SysTime(DateTime(1999, 7, 6, 12, 30, 58), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -59, SysTime(DateTime(1999, 7, 6, 12, 30, 34), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -60, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -61, SysTime(DateTime(1999, 7, 6, 12, 30, 32), FracSec.from!"msecs"(274)));

            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 0), FracSec.from!"msecs"(274)), 1, SysTime(DateTime(1999, 7, 6, 12, 30, 1), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 0), FracSec.from!"msecs"(274)), 0, SysTime(DateTime(1999, 7, 6, 12, 30, 0), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 0), FracSec.from!"msecs"(274)), -1, SysTime(DateTime(1999, 7, 6, 12, 30, 59), FracSec.from!"msecs"(274)));

            TestST(SysTime(DateTime(1999, 7, 6, 12, 0, 0), FracSec.from!"msecs"(274)), 1, SysTime(DateTime(1999, 7, 6, 12, 0, 1), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 0, 0), FracSec.from!"msecs"(274)), 0, SysTime(DateTime(1999, 7, 6, 12, 0, 0), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 0, 0), FracSec.from!"msecs"(274)), -1, SysTime(DateTime(1999, 7, 6, 12, 0, 59), FracSec.from!"msecs"(274)));

            TestST(SysTime(DateTime(1999, 7, 6, 0, 0, 0), FracSec.from!"msecs"(274)), 1, SysTime(DateTime(1999, 7, 6, 0, 0, 1), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 0, 0, 0), FracSec.from!"msecs"(274)), 0, SysTime(DateTime(1999, 7, 6, 0, 0, 0), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 0, 0, 0), FracSec.from!"msecs"(274)), -1, SysTime(DateTime(1999, 7, 6, 0, 0, 59), FracSec.from!"msecs"(274)));

            TestST(SysTime(DateTime(1999, 7, 5, 23, 59, 59), FracSec.from!"msecs"(274)), 1, SysTime(DateTime(1999, 7, 5, 23, 59, 0), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 5, 23, 59, 59), FracSec.from!"msecs"(274)), 0, SysTime(DateTime(1999, 7, 5, 23, 59, 59), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 5, 23, 59, 59), FracSec.from!"msecs"(274)), -1, SysTime(DateTime(1999, 7, 5, 23, 59, 58), FracSec.from!"msecs"(274)));

            TestST(SysTime(DateTime(1998, 12, 31, 23, 59, 59), FracSec.from!"msecs"(274)), 1, SysTime(DateTime(1998, 12, 31, 23, 59, 0), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1998, 12, 31, 23, 59, 59), FracSec.from!"msecs"(274)), 0, SysTime(DateTime(1998, 12, 31, 23, 59, 59), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1998, 12, 31, 23, 59, 59), FracSec.from!"msecs"(274)), -1, SysTime(DateTime(1998, 12, 31, 23, 59, 58), FracSec.from!"msecs"(274)));

            //Test B.C.
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 0, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 1, SysTime(DateTime(-1999, 7, 6, 12, 30, 34), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 2, SysTime(DateTime(-1999, 7, 6, 12, 30, 35), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 3, SysTime(DateTime(-1999, 7, 6, 12, 30, 36), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 4, SysTime(DateTime(-1999, 7, 6, 12, 30, 37), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 5, SysTime(DateTime(-1999, 7, 6, 12, 30, 38), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 10, SysTime(DateTime(-1999, 7, 6, 12, 30, 43), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 15, SysTime(DateTime(-1999, 7, 6, 12, 30, 48), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 26, SysTime(DateTime(-1999, 7, 6, 12, 30, 59), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 27, SysTime(DateTime(-1999, 7, 6, 12, 30, 0), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 30, SysTime(DateTime(-1999, 7, 6, 12, 30, 3), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 59, SysTime(DateTime(-1999, 7, 6, 12, 30, 32), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 60, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 61, SysTime(DateTime(-1999, 7, 6, 12, 30, 34), FracSec.from!"msecs"(274)));

            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 1766, SysTime(DateTime(-1999, 7, 6, 12, 30, 59), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 1767, SysTime(DateTime(-1999, 7, 6, 12, 30, 0), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 1768, SysTime(DateTime(-1999, 7, 6, 12, 30, 1), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 2007, SysTime(DateTime(-1999, 7, 6, 12, 30, 0), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 3599, SysTime(DateTime(-1999, 7, 6, 12, 30, 32), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 3600, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 3601, SysTime(DateTime(-1999, 7, 6, 12, 30, 34), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 7200, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)));

            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -1, SysTime(DateTime(-1999, 7, 6, 12, 30, 32), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -2, SysTime(DateTime(-1999, 7, 6, 12, 30, 31), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -3, SysTime(DateTime(-1999, 7, 6, 12, 30, 30), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -4, SysTime(DateTime(-1999, 7, 6, 12, 30, 29), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -5, SysTime(DateTime(-1999, 7, 6, 12, 30, 28), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -10, SysTime(DateTime(-1999, 7, 6, 12, 30, 23), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -15, SysTime(DateTime(-1999, 7, 6, 12, 30, 18), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -33, SysTime(DateTime(-1999, 7, 6, 12, 30, 0), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -34, SysTime(DateTime(-1999, 7, 6, 12, 30, 59), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -35, SysTime(DateTime(-1999, 7, 6, 12, 30, 58), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -59, SysTime(DateTime(-1999, 7, 6, 12, 30, 34), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -60, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -61, SysTime(DateTime(-1999, 7, 6, 12, 30, 32), FracSec.from!"msecs"(274)));

            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 0), FracSec.from!"msecs"(274)), 1, SysTime(DateTime(-1999, 7, 6, 12, 30, 1), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 0), FracSec.from!"msecs"(274)), 0, SysTime(DateTime(-1999, 7, 6, 12, 30, 0), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 0), FracSec.from!"msecs"(274)), -1, SysTime(DateTime(-1999, 7, 6, 12, 30, 59), FracSec.from!"msecs"(274)));

            TestST(SysTime(DateTime(-1999, 7, 6, 12, 0, 0), FracSec.from!"msecs"(274)), 1, SysTime(DateTime(-1999, 7, 6, 12, 0, 1), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 0, 0), FracSec.from!"msecs"(274)), 0, SysTime(DateTime(-1999, 7, 6, 12, 0, 0), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 0, 0), FracSec.from!"msecs"(274)), -1, SysTime(DateTime(-1999, 7, 6, 12, 0, 59), FracSec.from!"msecs"(274)));

            TestST(SysTime(DateTime(-1999, 7, 6, 0, 0, 0), FracSec.from!"msecs"(274)), 1, SysTime(DateTime(-1999, 7, 6, 0, 0, 1), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 0, 0, 0), FracSec.from!"msecs"(274)), 0, SysTime(DateTime(-1999, 7, 6, 0, 0, 0), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 0, 0, 0), FracSec.from!"msecs"(274)), -1, SysTime(DateTime(-1999, 7, 6, 0, 0, 59), FracSec.from!"msecs"(274)));

            TestST(SysTime(DateTime(-1999, 7, 5, 23, 59, 59), FracSec.from!"msecs"(274)), 1, SysTime(DateTime(-1999, 7, 5, 23, 59, 0), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 5, 23, 59, 59), FracSec.from!"msecs"(274)), 0, SysTime(DateTime(-1999, 7, 5, 23, 59, 59), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 5, 23, 59, 59), FracSec.from!"msecs"(274)), -1, SysTime(DateTime(-1999, 7, 5, 23, 59, 58), FracSec.from!"msecs"(274)));

            TestST(SysTime(DateTime(-2000, 12, 31, 23, 59, 59), FracSec.from!"msecs"(274)), 1, SysTime(DateTime(-2000, 12, 31, 23, 59, 0), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-2000, 12, 31, 23, 59, 59), FracSec.from!"msecs"(274)), 0, SysTime(DateTime(-2000, 12, 31, 23, 59, 59), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-2000, 12, 31, 23, 59, 59), FracSec.from!"msecs"(274)), -1, SysTime(DateTime(-2000, 12, 31, 23, 59, 58), FracSec.from!"msecs"(274)));

            //Test Both
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"msecs"(274)), -1, SysTime(DateTime(1, 1, 1, 0, 0, 59), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"msecs"(274)), 1, SysTime(DateTime(0, 12, 31, 23, 59, 0), FracSec.from!"msecs"(274)));

            TestST(SysTime(DateTime(0, 1, 1, 0, 0, 0), FracSec.from!"msecs"(274)), -1, SysTime(DateTime(0, 1, 1, 0, 0, 59), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-1, 12, 31, 23, 59, 59), FracSec.from!"msecs"(274)), 1, SysTime(DateTime(-1, 12, 31, 23, 59, 0), FracSec.from!"msecs"(274)));

            TestST(SysTime(DateTime(-1, 1, 1, 11, 30, 33), FracSec.from!"msecs"(274)), 63_165_600L, SysTime(DateTime(-1, 1, 1, 11, 30, 33), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1, 1, 1, 13, 30, 33), FracSec.from!"msecs"(274)), -63_165_600L, SysTime(DateTime(1, 1, 1, 13, 30, 33), FracSec.from!"msecs"(274)));

            TestST(SysTime(DateTime(-1, 1, 1, 11, 30, 33), FracSec.from!"msecs"(274)), 63_165_617L, SysTime(DateTime(-1, 1, 1, 11, 30, 50), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1, 1, 1, 13, 30, 50), FracSec.from!"msecs"(274)), -63_165_617L, SysTime(DateTime(1, 1, 1, 13, 30, 33), FracSec.from!"msecs"(274)));

            {
                auto sysTime = SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0));
                sysTime.roll!"seconds"(-1);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 1, 1, 0, 0, 59), FracSec.from!"hnsecs"(0)));
                sysTime.roll!"seconds"(1);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)));
            }

            {
                auto sysTime = SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(9_999_999));
                sysTime.roll!"seconds"(-1);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 1, 1, 0, 0, 59), FracSec.from!"hnsecs"(9_999_999)));
                sysTime.roll!"seconds"(1);
                _assertPred!"=="(sysTime, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(9_999_999)));
            }

            {
                auto sysTime = SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(0));
                sysTime.roll!"seconds"(1);
                _assertPred!"=="(sysTime, SysTime(DateTime(0, 12, 31, 23, 59, 0), FracSec.from!"hnsecs"(0)));
                sysTime.roll!"seconds"(-1);
                _assertPred!"=="(sysTime, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(0)));
            }

            {
                auto sysTime = SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999));
                sysTime.roll!"seconds"(1);
                _assertPred!"=="(sysTime, SysTime(DateTime(0, 12, 31, 23, 59, 0), FracSec.from!"hnsecs"(9_999_999)));
                sysTime.roll!"seconds"(-1);
                _assertPred!"=="(sysTime, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            }

            const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            static assert(!__traits(compiles, cst.roll!"seconds"(4)));
            //static assert(!__traits(compiles, ist.roll!"seconds"(4)));
        }
    }


    //Shares documentation with "days" version.
    /+ref SysTime+/ void roll(string units)(long value) nothrow
        if(units == "msecs" ||
           units == "usecs" ||
           units == "hnsecs")
    {
        auto hnsecs = adjTime;
        immutable days = splitUnitsFromHNSecs!"days"(hnsecs);
        immutable negative = hnsecs < 0;

        if(negative)
            hnsecs += convert!("hours", "hnsecs")(24);

        immutable seconds = splitUnitsFromHNSecs!"seconds"(hnsecs);

        hnsecs += convert!(units, "hnsecs")(value);
        hnsecs %= convert!("seconds", "hnsecs")(1);

        if(hnsecs < 0)
            hnsecs += convert!("seconds", "hnsecs")(1);

        hnsecs += convert!("seconds", "hnsecs")(seconds);

        if(negative)
            hnsecs -= convert!("hours", "hnsecs")(24);

        immutable newDaysHNSecs = convert!("days", "hnsecs")(days);

        adjTime = newDaysHNSecs + hnsecs;
    }


    //Test roll!"msecs"().
    unittest
    {
        version(testStdDateTime)
        {
            static void TestST(SysTime orig, int milliseconds, in SysTime expected, size_t line = __LINE__)
            {
                orig.roll!"msecs"(milliseconds);
                _assertPred!"=="(orig, expected, "", __FILE__, line);
            }

            //Test A.D.
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 0, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 1, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(275)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 2, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(276)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 10, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(284)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 100, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(374)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 725, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(999)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 726, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(0)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 1000, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 1001, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(275)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 2000, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 26_725, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(999)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 26_726, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(0)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 26_727, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(1)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 1_766_725, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(999)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 1_766_726, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(0)));

            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -1, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(273)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -2, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(272)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -10, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(264)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -100, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(174)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -274, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(0)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -275, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(999)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -1000, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -1001, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(273)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -2000, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -33_274, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(0)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -33_275, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(999)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -1_833_274, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(0)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -1_833_275, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(999)));

            //Test B.C.
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 0, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 1, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(275)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 2, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(276)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 10, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(284)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 100, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(374)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 725, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(999)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 726, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(0)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 1000, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 1001, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(275)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 2000, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 26_725, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(999)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 26_726, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(0)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 26_727, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(1)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 1_766_725, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(999)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), 1_766_726, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(0)));

            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -1, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(273)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -2, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(272)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -10, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(264)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -100, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(174)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -274, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(0)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -275, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(999)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -1000, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -1001, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(273)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -2000, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -33_274, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(0)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -33_275, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(999)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -1_833_274, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(0)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(274)), -1_833_275, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(999)));

            //Test Both
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"msecs"(0)), 1, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"msecs"(1)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"msecs"(0)), 0, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"msecs"(0)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"msecs"(0)), -1, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"msecs"(999)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"msecs"(0)), -2, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"msecs"(998)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"msecs"(0)), -1000, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"msecs"(0)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"msecs"(0)), -2000, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"msecs"(0)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"msecs"(0)), -2555, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"msecs"(445)));

            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), -1, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_989_999)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 0, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 1, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9999)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 2, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(19_999)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 1000, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 2000, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 2555, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(5_549_999)));

            const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            static assert(!__traits(compiles, cst.addMSecs(4)));
            //static assert(!__traits(compiles, ist.addMSecs(4)));
        }
    }

    //Test roll!"usecs"().
    unittest
    {
        version(testStdDateTime)
        {
            static void TestST(SysTime orig, long microseconds, in SysTime expected, size_t line = __LINE__)
            {
                orig.roll!"usecs"(microseconds);
                _assertPred!"=="(orig, expected, "", __FILE__, line);
            }

            //Test A.D.
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), 0, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), 1, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(275)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), 2, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(276)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), 10, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(284)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), 100, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(374)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), 725, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(999)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), 726, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(1000)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), 1000, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(1274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), 1001, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(1275)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), 2000, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(2274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), 26_725, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(26_999)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), 26_726, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(27_000)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), 26_727, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(27_001)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), 1_766_725, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(766_999)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), 1_766_726, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(767_000)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), 1_000_000, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), 60_000_000L, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), 3_600_000_000L, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)));

            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), -1, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(273)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), -2, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(272)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), -10, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(264)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), -100, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(174)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), -274, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(0)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), -275, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(999_999)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), -1000, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(999_274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), -1001, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(999_273)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), -2000, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(998_274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), -33_274, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(967_000)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), -33_275, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(966_999)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), -1_833_274, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(167_000)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), -1_833_275, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(166_999)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), -1_000_000, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), -60_000_000L, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), -3_600_000_000L, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)));

            //Test B.C.
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), 0, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), 1, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(275)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), 2, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(276)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), 10, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(284)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), 100, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(374)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), 725, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(999)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), 726, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(1000)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), 1000, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(1274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), 1001, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(1275)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), 2000, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(2274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), 26_725, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(26_999)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), 26_726, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(27_000)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), 26_727, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(27_001)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), 1_766_725, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(766_999)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), 1_766_726, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(767_000)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), 1_000_000, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), 60_000_000L, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), 3_600_000_000L, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)));

            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), -1, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(273)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), -2, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(272)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), -10, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(264)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), -100, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(174)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), -274, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(0)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), -275, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(999_999)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), -1000, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(999_274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), -1001, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(999_273)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), -2000, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(998_274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), -33_274, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(967_000)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), -33_275, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(966_999)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), -1_833_274, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(167_000)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), -1_833_275, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(166_999)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), -1_000_000, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), -60_000_000L, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)), -3_600_000_000L, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(274)));

            //Test Both
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"usecs"(0)), 1, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"usecs"(1)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"usecs"(0)), 0, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"usecs"(0)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"usecs"(0)), -1, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"usecs"(999_999)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"usecs"(0)), -2, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"usecs"(999_998)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"usecs"(0)), -1000, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"usecs"(999_000)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"usecs"(0)), -2000, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"usecs"(998_000)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"usecs"(0)), -2555, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"usecs"(997_445)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"usecs"(0)), -1_000_000, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"usecs"(0)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"usecs"(0)), -2_000_000, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"usecs"(0)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"usecs"(0)), -2_333_333, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"usecs"(666_667)));

            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), -1, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_989)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 0, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 1, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 2, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(19)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 1000, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9999)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 2000, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(19_999)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 2555, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(25_549)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 1_000_000, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 2_000_000, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 2_333_333, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(3_333_329)));

            const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            static assert(!__traits(compiles, cst.roll!"usecs"(4)));
            //static assert(!__traits(compiles, ist.roll!"usecs"(4)));
        }
    }

    //Test roll!"hnsecs"().
    unittest
    {
        version(testStdDateTime)
        {
            static void TestST(SysTime orig, long hnsecs, in SysTime expected, size_t line = __LINE__)
            {
                orig.roll!"hnsecs"(hnsecs);
                _assertPred!"=="(orig, expected, "", __FILE__, line);
            }

            //Test A.D.
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 0, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 1, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(275)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 2, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(276)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 10, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(284)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 100, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(374)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 725, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(999)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 726, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(1000)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 1000, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(1274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 1001, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(1275)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 2000, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(2274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 26_725, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(26_999)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 26_726, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(27_000)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 26_727, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(27_001)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 1_766_725, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(1_766_999)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 1_766_726, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(1_767_000)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 1_000_000, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(1_000_274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 60_000_000L, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 3_600_000_000L, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 600_000_000L, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 36_000_000_000L, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)));

            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -1, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(273)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -2, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(272)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -10, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(264)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -100, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(174)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -274, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(0)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -275, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(9_999_999)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -1000, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(9_999_274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -1001, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(9_999_273)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -2000, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(9_998_274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -33_274, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(9_967_000)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -33_275, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(9_966_999)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -1_833_274, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(8_167_000)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -1_833_275, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(8_166_999)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -1_000_000, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(9_000_274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -60_000_000L, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -3_600_000_000L, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -600_000_000L, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -36_000_000_000L, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)));

            //Test B.C.
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 0, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 1, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(275)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 2, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(276)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 10, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(284)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 100, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(374)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 725, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(999)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 726, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(1000)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 1000, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(1274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 1001, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(1275)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 2000, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(2274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 26_725, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(26_999)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 26_726, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(27_000)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 26_727, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(27_001)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 1_766_725, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(1_766_999)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 1_766_726, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(1_767_000)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 1_000_000, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(1_000_274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 60_000_000L, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 3_600_000_000L, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 600_000_000L, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 36_000_000_000L, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)));

            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -1, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(273)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -2, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(272)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -10, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(264)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -100, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(174)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -274, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(0)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -275, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(9_999_999)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -1000, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(9_999_274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -1001, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(9_999_273)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -2000, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(9_998_274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -33_274, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(9_967_000)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -33_275, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(9_966_999)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -1_833_274, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(8_167_000)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -1_833_275, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(8_166_999)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -1_000_000, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(9_000_274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -60_000_000L, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -3_600_000_000L, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -600_000_000L, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -36_000_000_000L, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)));

            //Test Both
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)), 1, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(1)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)), 0, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)), -1, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(9_999_999)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)), -2, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(9_999_998)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)), -1000, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(9_999_000)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)), -2000, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(9_998_000)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)), -2555, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(9_997_445)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)), -1_000_000, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(9_000_000)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)), -2_000_000, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(8_000_000)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)), -2_333_333, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(7_666_667)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)), -10_000_000, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)), -20_000_000, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)), -20_888_888, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(9_111_112)));

            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), -1, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_998)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 0, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 1, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(0)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 2, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(1)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 1000, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(999)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 2000, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(1999)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 2555, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(2554)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 1_000_000, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(999_999)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 2_000_000, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(1_999_999)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 2_333_333, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(2_333_332)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 10_000_000, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 20_000_000, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 20_888_888, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(888_887)));

            const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            static assert(!__traits(compiles, cst.roll!"hnsecs"(4)));
            //static assert(!__traits(compiles, ist.roll!"hnsecs"(4)));
        }
    }


    /++
        Gives the result of adding or subtracting a duration from this
        $(D SysTime).

        The legal types of arithmetic for $(D SysTime) using this operator are

        $(BOOKTABLE,
        $(TR $(TD SysTime) $(TD +) $(TD duration) $(TD -->) $(TD SysTime))
        $(TR $(TD SysTime) $(TD -) $(TD duration) $(TD -->) $(TD SysTime))
        )

        Params:
            duration = The duration to add to or subtract from this
                       $(D SysTime).
      +/
    SysTime opBinary(string op, D)(in D duration) const pure nothrow
        if((op == "+" || op == "-") &&
           (is(Unqual!D == Duration) ||
            is(Unqual!D == TickDuration)))
    {
        SysTime retval = SysTime(this._stdTime, this._timezone);

        static if(is(Unqual!D == Duration))
            immutable hnsecs = duration.total!"hnsecs";
        else static if(is(Unqual!D == TickDuration))
            immutable hnsecs = duration.hnsecs;

        //Ideally, this would just be
        //retval._stdTime += unaryFun!(op ~ "a")(hnsecs);
        //But there isn't currently a pure version of unaryFun!().

        static if(op == "+")
            immutable signedHNSecs = hnsecs;
        else static if(op == "-")
            immutable signedHNSecs = -hnsecs;
        else
            static assert(0);

        retval._stdTime += signedHNSecs;

        return retval;
    }

    unittest
    {
        version(testStdDateTime)
        {
            auto st = SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(2_345_678));

            _assertPred!"=="(st + dur!"weeks"(7), SysTime(DateTime(1999, 8, 24, 12, 30, 33), FracSec.from!"hnsecs"(2_345_678)));
            _assertPred!"=="(st + dur!"weeks"(-7), SysTime(DateTime(1999, 5, 18, 12, 30, 33), FracSec.from!"hnsecs"(2_345_678)));
            _assertPred!"=="(st + dur!"days"(7), SysTime(DateTime(1999, 7, 13, 12, 30, 33), FracSec.from!"hnsecs"(2_345_678)));
            _assertPred!"=="(st + dur!"days"(-7), SysTime(DateTime(1999, 6, 29, 12, 30, 33), FracSec.from!"hnsecs"(2_345_678)));
            _assertPred!"=="(st + dur!"hours"(7), SysTime(DateTime(1999, 7, 6, 19, 30, 33), FracSec.from!"hnsecs"(2_345_678)));
            _assertPred!"=="(st + dur!"hours"(-7), SysTime(DateTime(1999, 7, 6, 5, 30, 33), FracSec.from!"hnsecs"(2_345_678)));
            _assertPred!"=="(st + dur!"minutes"(7), SysTime(DateTime(1999, 7, 6, 12, 37, 33), FracSec.from!"hnsecs"(2_345_678)));
            _assertPred!"=="(st + dur!"minutes"(-7), SysTime(DateTime(1999, 7, 6, 12, 23, 33), FracSec.from!"hnsecs"(2_345_678)));
            _assertPred!"=="(st + dur!"seconds"(7), SysTime(DateTime(1999, 7, 6, 12, 30, 40), FracSec.from!"hnsecs"(2_345_678)));
            _assertPred!"=="(st + dur!"seconds"(-7), SysTime(DateTime(1999, 7, 6, 12, 30, 26), FracSec.from!"hnsecs"(2_345_678)));
            _assertPred!"=="(st + dur!"msecs"(7), SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(2_415_678)));
            _assertPred!"=="(st + dur!"msecs"(-7), SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(2_275_678)));
            _assertPred!"=="(st + dur!"usecs"(7), SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(2_345_748)));
            _assertPred!"=="(st + dur!"usecs"(-7), SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(2_345_608)));
            _assertPred!"=="(st + dur!"hnsecs"(7), SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(2_345_685)));
            _assertPred!"=="(st + dur!"hnsecs"(-7), SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(2_345_671)));

            //This probably only runs in cases where gettimeofday() is used, but it's
            //hard to do this test correctly with variable ticksPerSec.
            if(TickDuration.ticksPerSec == 1_000_000)
            {
                _assertPred!"=="(st + TickDuration.from!"usecs"(7), SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(2_345_748)));
                _assertPred!"=="(st + TickDuration.from!"usecs"(-7), SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(2_345_608)));
            }

            _assertPred!"=="(st - dur!"weeks"(-7), SysTime(DateTime(1999, 8, 24, 12, 30, 33), FracSec.from!"hnsecs"(2_345_678)));
            _assertPred!"=="(st - dur!"weeks"(7), SysTime(DateTime(1999, 5, 18, 12, 30, 33), FracSec.from!"hnsecs"(2_345_678)));
            _assertPred!"=="(st - dur!"days"(-7), SysTime(DateTime(1999, 7, 13, 12, 30, 33), FracSec.from!"hnsecs"(2_345_678)));
            _assertPred!"=="(st - dur!"days"(7), SysTime(DateTime(1999, 6, 29, 12, 30, 33), FracSec.from!"hnsecs"(2_345_678)));
            _assertPred!"=="(st - dur!"hours"(-7), SysTime(DateTime(1999, 7, 6, 19, 30, 33), FracSec.from!"hnsecs"(2_345_678)));
            _assertPred!"=="(st - dur!"hours"(7), SysTime(DateTime(1999, 7, 6, 5, 30, 33), FracSec.from!"hnsecs"(2_345_678)));
            _assertPred!"=="(st - dur!"minutes"(-7), SysTime(DateTime(1999, 7, 6, 12, 37, 33), FracSec.from!"hnsecs"(2_345_678)));
            _assertPred!"=="(st - dur!"minutes"(7), SysTime(DateTime(1999, 7, 6, 12, 23, 33), FracSec.from!"hnsecs"(2_345_678)));
            _assertPred!"=="(st - dur!"seconds"(-7), SysTime(DateTime(1999, 7, 6, 12, 30, 40), FracSec.from!"hnsecs"(2_345_678)));
            _assertPred!"=="(st - dur!"seconds"(7), SysTime(DateTime(1999, 7, 6, 12, 30, 26), FracSec.from!"hnsecs"(2_345_678)));
            _assertPred!"=="(st - dur!"msecs"(-7), SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(2_415_678)));
            _assertPred!"=="(st - dur!"msecs"(7), SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(2_275_678)));
            _assertPred!"=="(st - dur!"usecs"(-7), SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(2_345_748)));
            _assertPred!"=="(st - dur!"usecs"(7), SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(2_345_608)));
            _assertPred!"=="(st - dur!"hnsecs"(-7), SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(2_345_685)));
            _assertPred!"=="(st - dur!"hnsecs"(7), SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(2_345_671)));

            //This probably only runs in cases where gettimeofday() is used, but it's
            //hard to do this test correctly with variable ticksPerSec.
            if(TickDuration.ticksPerSec == 1_000_000)
            {
                _assertPred!"=="(st - TickDuration.from!"usecs"(-7), SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(2_345_748)));
                _assertPred!"=="(st - TickDuration.from!"usecs"(7), SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(2_345_608)));
            }

            static void TestST(in SysTime orig, long hnsecs, in SysTime expected, size_t line = __LINE__)
            {
                _assertPred!"=="(orig + dur!"hnsecs"(hnsecs), expected, "", __FILE__, line);
            }

            //Test A.D.
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 0, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 1, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(275)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 2, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(276)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 10, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(284)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 100, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(374)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 725, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(999)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 726, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(1000)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 1000, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(1274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 1001, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(1275)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 2000, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(2274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 26_725, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(26_999)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 26_726, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(27_000)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 26_727, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(27_001)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 1_766_725, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(1_766_999)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 1_766_726, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(1_767_000)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 1_000_000, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(1_000_274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 60_000_000L, SysTime(DateTime(1999, 7, 6, 12, 30, 39), FracSec.from!"hnsecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 3_600_000_000L, SysTime(DateTime(1999, 7, 6, 12, 36, 33), FracSec.from!"hnsecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 600_000_000L, SysTime(DateTime(1999, 7, 6, 12, 31, 33), FracSec.from!"hnsecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 36_000_000_000L, SysTime(DateTime(1999, 7, 6, 13, 30, 33), FracSec.from!"hnsecs"(274)));

            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -1, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(273)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -2, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(272)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -10, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(264)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -100, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(174)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -274, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(0)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -275, SysTime(DateTime(1999, 7, 6, 12, 30, 32), FracSec.from!"hnsecs"(9_999_999)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -1000, SysTime(DateTime(1999, 7, 6, 12, 30, 32), FracSec.from!"hnsecs"(9_999_274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -1001, SysTime(DateTime(1999, 7, 6, 12, 30, 32), FracSec.from!"hnsecs"(9_999_273)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -2000, SysTime(DateTime(1999, 7, 6, 12, 30, 32), FracSec.from!"hnsecs"(9_998_274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -33_274, SysTime(DateTime(1999, 7, 6, 12, 30, 32), FracSec.from!"hnsecs"(9_967_000)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -33_275, SysTime(DateTime(1999, 7, 6, 12, 30, 32), FracSec.from!"hnsecs"(9_966_999)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -1_833_274, SysTime(DateTime(1999, 7, 6, 12, 30, 32), FracSec.from!"hnsecs"(8_167_000)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -1_833_275, SysTime(DateTime(1999, 7, 6, 12, 30, 32), FracSec.from!"hnsecs"(8_166_999)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -1_000_000, SysTime(DateTime(1999, 7, 6, 12, 30, 32), FracSec.from!"hnsecs"(9_000_274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -60_000_000L, SysTime(DateTime(1999, 7, 6, 12, 30, 27), FracSec.from!"hnsecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -3_600_000_000L, SysTime(DateTime(1999, 7, 6, 12, 24, 33), FracSec.from!"hnsecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -600_000_000L, SysTime(DateTime(1999, 7, 6, 12, 29, 33), FracSec.from!"hnsecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -36_000_000_000L, SysTime(DateTime(1999, 7, 6, 11, 30, 33), FracSec.from!"hnsecs"(274)));

            //Test B.C.
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 0, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 1, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(275)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 2, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(276)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 10, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(284)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 100, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(374)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 725, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(999)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 726, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(1000)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 1000, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(1274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 1001, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(1275)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 2000, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(2274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 26_725, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(26_999)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 26_726, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(27_000)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 26_727, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(27_001)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 1_766_725, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(1_766_999)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 1_766_726, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(1_767_000)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 1_000_000, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(1_000_274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 60_000_000L, SysTime(DateTime(-1999, 7, 6, 12, 30, 39), FracSec.from!"hnsecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 3_600_000_000L, SysTime(DateTime(-1999, 7, 6, 12, 36, 33), FracSec.from!"hnsecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 600_000_000L, SysTime(DateTime(-1999, 7, 6, 12, 31, 33), FracSec.from!"hnsecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 36_000_000_000L, SysTime(DateTime(-1999, 7, 6, 13, 30, 33), FracSec.from!"hnsecs"(274)));

            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -1, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(273)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -2, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(272)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -10, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(264)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -100, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(174)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -274, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(0)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -275, SysTime(DateTime(-1999, 7, 6, 12, 30, 32), FracSec.from!"hnsecs"(9_999_999)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -1000, SysTime(DateTime(-1999, 7, 6, 12, 30, 32), FracSec.from!"hnsecs"(9_999_274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -1001, SysTime(DateTime(-1999, 7, 6, 12, 30, 32), FracSec.from!"hnsecs"(9_999_273)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -2000, SysTime(DateTime(-1999, 7, 6, 12, 30, 32), FracSec.from!"hnsecs"(9_998_274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -33_274, SysTime(DateTime(-1999, 7, 6, 12, 30, 32), FracSec.from!"hnsecs"(9_967_000)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -33_275, SysTime(DateTime(-1999, 7, 6, 12, 30, 32), FracSec.from!"hnsecs"(9_966_999)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -1_833_274, SysTime(DateTime(-1999, 7, 6, 12, 30, 32), FracSec.from!"hnsecs"(8_167_000)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -1_833_275, SysTime(DateTime(-1999, 7, 6, 12, 30, 32), FracSec.from!"hnsecs"(8_166_999)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -1_000_000, SysTime(DateTime(-1999, 7, 6, 12, 30, 32), FracSec.from!"hnsecs"(9_000_274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -60_000_000L, SysTime(DateTime(-1999, 7, 6, 12, 30, 27), FracSec.from!"hnsecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -3_600_000_000L, SysTime(DateTime(-1999, 7, 6, 12, 24, 33), FracSec.from!"hnsecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -600_000_000L, SysTime(DateTime(-1999, 7, 6, 12, 29, 33), FracSec.from!"hnsecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -36_000_000_000L, SysTime(DateTime(-1999, 7, 6, 11, 30, 33), FracSec.from!"hnsecs"(274)));

            //Test Both
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)), 1, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(1)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)), 0, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)), -1, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)), -2, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_998)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)), -1000, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_000)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)), -2000, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_998_000)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)), -2555, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_997_445)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)), -1_000_000, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_000_000)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)), -2_000_000, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(8_000_000)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)), -2_333_333, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(7_666_667)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)), -10_000_000, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(0)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)), -20_000_000, SysTime(DateTime(0, 12, 31, 23, 59, 58), FracSec.from!"hnsecs"(0)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)), -20_888_888, SysTime(DateTime(0, 12, 31, 23, 59, 57), FracSec.from!"hnsecs"(9_111_112)));

            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), -1, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_998)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 0, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 1, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 2, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(1)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 1000, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(999)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 2000, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(1999)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 2555, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(2554)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 1_000_000, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(999_999)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 2_000_000, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(1_999_999)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 2_333_333, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(2_333_332)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 10_000_000, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(9_999_999)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 20_000_000, SysTime(DateTime(1, 1, 1, 0, 0, 1), FracSec.from!"hnsecs"(9_999_999)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 20_888_888, SysTime(DateTime(1, 1, 1, 0, 0, 2), FracSec.from!"hnsecs"(888_887)));

            auto duration = dur!"seconds"(12);
            const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            static assert(__traits(compiles, cst + duration));
            //static assert(__traits(compiles, ist + duration));
            static assert(__traits(compiles, cst - duration));
            //static assert(__traits(compiles, ist - duration));
        }
    }


    /++
        Gives the result of adding or subtracting a duration from this
        $(D SysTime), as well as assigning the result to this $(D SysTime).

        The legal types of arithmetic for $(D SysTime) using this operator are

        $(BOOKTABLE,
        $(TR $(TD SysTime) $(TD +) $(TD duration) $(TD -->) $(TD SysTime))
        $(TR $(TD SysTime) $(TD -) $(TD duration) $(TD -->) $(TD SysTime))
        )

        Params:
            duration = The duration to add to or subtract from this
                       $(D SysTime).
      +/
    /+ref+/ SysTime opOpAssign(string op, D)(in D duration) pure nothrow
        if((op == "+" || op == "-") &&
           (is(Unqual!D == Duration) ||
            is(Unqual!D == TickDuration)))
    {
        static if(is(Unqual!D == Duration))
            auto hnsecs = duration.total!"hnsecs";
        else static if(is(Unqual!D == TickDuration))
            auto hnsecs = duration.hnsecs;

        //Ideally, this would just be
        //_stdTime += unaryFun!(op ~ "a")(hnsecs);
        //But there isn't currently a pure version of unaryFun!().

        static if(op == "+")
            immutable signedHNSecs = hnsecs;
        else static if(op == "-")
            immutable signedHNSecs = -hnsecs;
        else
            static assert(0);

        _stdTime += signedHNSecs;

        return this;
    }

    unittest
    {
        version(testStdDateTime)
        {
            _assertPred!"+="(SysTime(DateTime(1999, 7, 6, 12, 30, 33)), dur!"weeks"(7), SysTime(DateTime(1999, 8, 24, 12, 30, 33)));
            _assertPred!"+="(SysTime(DateTime(1999, 7, 6, 12, 30, 33)), dur!"weeks"(-7), SysTime(DateTime(1999, 5, 18, 12, 30, 33)));
            _assertPred!"+="(SysTime(DateTime(1999, 7, 6, 12, 30, 33)), dur!"days"(7), SysTime(DateTime(1999, 7, 13, 12, 30, 33)));
            _assertPred!"+="(SysTime(DateTime(1999, 7, 6, 12, 30, 33)), dur!"days"(-7), SysTime(DateTime(1999, 6, 29, 12, 30, 33)));

            _assertPred!"+="(SysTime(DateTime(1999, 7, 6, 12, 30, 33)), dur!"hours"(7), SysTime(DateTime(1999, 7, 6, 19, 30, 33)));
            _assertPred!"+="(SysTime(DateTime(1999, 7, 6, 12, 30, 33)), dur!"hours"(-7), SysTime(DateTime(1999, 7, 6, 5, 30, 33)));
            _assertPred!"+="(SysTime(DateTime(1999, 7, 6, 12, 30, 33)), dur!"minutes"(7), SysTime(DateTime(1999, 7, 6, 12, 37, 33)));
            _assertPred!"+="(SysTime(DateTime(1999, 7, 6, 12, 30, 33)), dur!"minutes"(-7), SysTime(DateTime(1999, 7, 6, 12, 23, 33)));
            _assertPred!"+="(SysTime(DateTime(1999, 7, 6, 12, 30, 33)), dur!"seconds"(7), SysTime(DateTime(1999, 7, 6, 12, 30, 40)));
            _assertPred!"+="(SysTime(DateTime(1999, 7, 6, 12, 30, 33)), dur!"seconds"(-7), SysTime(DateTime(1999, 7, 6, 12, 30, 26)));
            _assertPred!"+="(SysTime(DateTime(1999, 7, 6, 12, 30, 33)), dur!"msecs"(7), SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(7)));
            _assertPred!"+="(SysTime(DateTime(1999, 7, 6, 12, 30, 33)), dur!"msecs"(-7), SysTime(DateTime(1999, 7, 6, 12, 30, 32), FracSec.from!"msecs"(993)));
            _assertPred!"+="(SysTime(DateTime(1999, 7, 6, 12, 30, 33)), dur!"usecs"(7), SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7)));
            _assertPred!"+="(SysTime(DateTime(1999, 7, 6, 12, 30, 33)), dur!"usecs"(-7), SysTime(DateTime(1999, 7, 6, 12, 30, 32), FracSec.from!"usecs"(999_993)));
            _assertPred!"+="(SysTime(DateTime(1999, 7, 6, 12, 30, 33)), dur!"hnsecs"(7), SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(7)));
            _assertPred!"+="(SysTime(DateTime(1999, 7, 6, 12, 30, 33)), dur!"hnsecs"(-7), SysTime(DateTime(1999, 7, 6, 12, 30, 32), FracSec.from!"hnsecs"(9_999_993)));

            _assertPred!"-="(SysTime(DateTime(1999, 7, 6, 12, 30, 33)), dur!"weeks"(-7), SysTime(DateTime(1999, 8, 24, 12, 30, 33)));
            _assertPred!"-="(SysTime(DateTime(1999, 7, 6, 12, 30, 33)), dur!"weeks"(7), SysTime(DateTime(1999, 5, 18, 12, 30, 33)));
            _assertPred!"-="(SysTime(DateTime(1999, 7, 6, 12, 30, 33)), dur!"days"(-7), SysTime(DateTime(1999, 7, 13, 12, 30, 33)));
            _assertPred!"-="(SysTime(DateTime(1999, 7, 6, 12, 30, 33)), dur!"days"(7), SysTime(DateTime(1999, 6, 29, 12, 30, 33)));

            _assertPred!"-="(SysTime(DateTime(1999, 7, 6, 12, 30, 33)), dur!"hours"(-7), SysTime(DateTime(1999, 7, 6, 19, 30, 33)));
            _assertPred!"-="(SysTime(DateTime(1999, 7, 6, 12, 30, 33)), dur!"hours"(7), SysTime(DateTime(1999, 7, 6, 5, 30, 33)));
            _assertPred!"-="(SysTime(DateTime(1999, 7, 6, 12, 30, 33)), dur!"minutes"(-7), SysTime(DateTime(1999, 7, 6, 12, 37, 33)));
            _assertPred!"-="(SysTime(DateTime(1999, 7, 6, 12, 30, 33)), dur!"minutes"(7), SysTime(DateTime(1999, 7, 6, 12, 23, 33)));
            _assertPred!"-="(SysTime(DateTime(1999, 7, 6, 12, 30, 33)), dur!"seconds"(-7), SysTime(DateTime(1999, 7, 6, 12, 30, 40)));
            _assertPred!"-="(SysTime(DateTime(1999, 7, 6, 12, 30, 33)), dur!"seconds"(7), SysTime(DateTime(1999, 7, 6, 12, 30, 26)));
            _assertPred!"-="(SysTime(DateTime(1999, 7, 6, 12, 30, 33)), dur!"msecs"(-7), SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(7)));
            _assertPred!"-="(SysTime(DateTime(1999, 7, 6, 12, 30, 33)), dur!"msecs"(7), SysTime(DateTime(1999, 7, 6, 12, 30, 32), FracSec.from!"msecs"(993)));
            _assertPred!"-="(SysTime(DateTime(1999, 7, 6, 12, 30, 33)), dur!"usecs"(-7), SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(7)));
            _assertPred!"-="(SysTime(DateTime(1999, 7, 6, 12, 30, 33)), dur!"usecs"(7), SysTime(DateTime(1999, 7, 6, 12, 30, 32), FracSec.from!"usecs"(999_993)));
            _assertPred!"-="(SysTime(DateTime(1999, 7, 6, 12, 30, 33)), dur!"hnsecs"(-7), SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(7)));
            _assertPred!"-="(SysTime(DateTime(1999, 7, 6, 12, 30, 33)), dur!"hnsecs"(7), SysTime(DateTime(1999, 7, 6, 12, 30, 32), FracSec.from!"hnsecs"(9_999_993)));

            static void TestST(SysTime orig, long hnsecs, in SysTime expected, size_t line = __LINE__)
            {
                orig += dur!"hnsecs"(hnsecs);
                _assertPred!"=="(orig, expected, "", __FILE__, line);
            }

            //Test A.D.
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 0, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 1, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(275)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 2, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(276)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 10, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(284)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 100, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(374)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 725, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(999)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 726, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(1000)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 1000, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(1274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 1001, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(1275)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 2000, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(2274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 26_725, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(26_999)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 26_726, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(27_000)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 26_727, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(27_001)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 1_766_725, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(1_766_999)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 1_766_726, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(1_767_000)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 1_000_000, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(1_000_274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 60_000_000L, SysTime(DateTime(1999, 7, 6, 12, 30, 39), FracSec.from!"hnsecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 3_600_000_000L, SysTime(DateTime(1999, 7, 6, 12, 36, 33), FracSec.from!"hnsecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 600_000_000L, SysTime(DateTime(1999, 7, 6, 12, 31, 33), FracSec.from!"hnsecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 36_000_000_000L, SysTime(DateTime(1999, 7, 6, 13, 30, 33), FracSec.from!"hnsecs"(274)));

            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -1, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(273)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -2, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(272)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -10, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(264)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -100, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(174)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -274, SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(0)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -275, SysTime(DateTime(1999, 7, 6, 12, 30, 32), FracSec.from!"hnsecs"(9_999_999)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -1000, SysTime(DateTime(1999, 7, 6, 12, 30, 32), FracSec.from!"hnsecs"(9_999_274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -1001, SysTime(DateTime(1999, 7, 6, 12, 30, 32), FracSec.from!"hnsecs"(9_999_273)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -2000, SysTime(DateTime(1999, 7, 6, 12, 30, 32), FracSec.from!"hnsecs"(9_998_274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -33_274, SysTime(DateTime(1999, 7, 6, 12, 30, 32), FracSec.from!"hnsecs"(9_967_000)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -33_275, SysTime(DateTime(1999, 7, 6, 12, 30, 32), FracSec.from!"hnsecs"(9_966_999)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -1_833_274, SysTime(DateTime(1999, 7, 6, 12, 30, 32), FracSec.from!"hnsecs"(8_167_000)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -1_833_275, SysTime(DateTime(1999, 7, 6, 12, 30, 32), FracSec.from!"hnsecs"(8_166_999)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -1_000_000, SysTime(DateTime(1999, 7, 6, 12, 30, 32), FracSec.from!"hnsecs"(9_000_274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -60_000_000L, SysTime(DateTime(1999, 7, 6, 12, 30, 27), FracSec.from!"hnsecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -3_600_000_000L, SysTime(DateTime(1999, 7, 6, 12, 24, 33), FracSec.from!"hnsecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -600_000_000L, SysTime(DateTime(1999, 7, 6, 12, 29, 33), FracSec.from!"hnsecs"(274)));
            TestST(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -36_000_000_000L, SysTime(DateTime(1999, 7, 6, 11, 30, 33), FracSec.from!"hnsecs"(274)));

            //Test B.C.
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 0, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 1, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(275)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 2, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(276)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 10, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(284)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 100, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(374)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 725, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(999)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 726, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(1000)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 1000, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(1274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 1001, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(1275)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 2000, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(2274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 26_725, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(26_999)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 26_726, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(27_000)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 26_727, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(27_001)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 1_766_725, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(1_766_999)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 1_766_726, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(1_767_000)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 1_000_000, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(1_000_274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 60_000_000L, SysTime(DateTime(-1999, 7, 6, 12, 30, 39), FracSec.from!"hnsecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 3_600_000_000L, SysTime(DateTime(-1999, 7, 6, 12, 36, 33), FracSec.from!"hnsecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 600_000_000L, SysTime(DateTime(-1999, 7, 6, 12, 31, 33), FracSec.from!"hnsecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), 36_000_000_000L, SysTime(DateTime(-1999, 7, 6, 13, 30, 33), FracSec.from!"hnsecs"(274)));

            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -1, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(273)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -2, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(272)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -10, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(264)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -100, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(174)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -274, SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(0)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -275, SysTime(DateTime(-1999, 7, 6, 12, 30, 32), FracSec.from!"hnsecs"(9_999_999)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -1000, SysTime(DateTime(-1999, 7, 6, 12, 30, 32), FracSec.from!"hnsecs"(9_999_274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -1001, SysTime(DateTime(-1999, 7, 6, 12, 30, 32), FracSec.from!"hnsecs"(9_999_273)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -2000, SysTime(DateTime(-1999, 7, 6, 12, 30, 32), FracSec.from!"hnsecs"(9_998_274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -33_274, SysTime(DateTime(-1999, 7, 6, 12, 30, 32), FracSec.from!"hnsecs"(9_967_000)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -33_275, SysTime(DateTime(-1999, 7, 6, 12, 30, 32), FracSec.from!"hnsecs"(9_966_999)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -1_833_274, SysTime(DateTime(-1999, 7, 6, 12, 30, 32), FracSec.from!"hnsecs"(8_167_000)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -1_833_275, SysTime(DateTime(-1999, 7, 6, 12, 30, 32), FracSec.from!"hnsecs"(8_166_999)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -1_000_000, SysTime(DateTime(-1999, 7, 6, 12, 30, 32), FracSec.from!"hnsecs"(9_000_274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -60_000_000L, SysTime(DateTime(-1999, 7, 6, 12, 30, 27), FracSec.from!"hnsecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -3_600_000_000L, SysTime(DateTime(-1999, 7, 6, 12, 24, 33), FracSec.from!"hnsecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -600_000_000L, SysTime(DateTime(-1999, 7, 6, 12, 29, 33), FracSec.from!"hnsecs"(274)));
            TestST(SysTime(DateTime(-1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(274)), -36_000_000_000L, SysTime(DateTime(-1999, 7, 6, 11, 30, 33), FracSec.from!"hnsecs"(274)));

            //Test Both
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)), 1, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(1)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)), 0, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)), -1, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)), -2, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_998)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)), -1000, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_000)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)), -2000, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_998_000)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)), -2555, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_997_445)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)), -1_000_000, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_000_000)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)), -2_000_000, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(8_000_000)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)), -2_333_333, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(7_666_667)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)), -10_000_000, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(0)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)), -20_000_000, SysTime(DateTime(0, 12, 31, 23, 59, 58), FracSec.from!"hnsecs"(0)));
            TestST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)), -20_888_888, SysTime(DateTime(0, 12, 31, 23, 59, 57), FracSec.from!"hnsecs"(9_111_112)));

            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), -1, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_998)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 0, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 1, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 2, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(1)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 1000, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(999)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 2000, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(1999)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 2555, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(2554)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 1_000_000, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(999_999)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 2_000_000, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(1_999_999)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 2_333_333, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(2_333_332)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 10_000_000, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(9_999_999)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 20_000_000, SysTime(DateTime(1, 1, 1, 0, 0, 1), FracSec.from!"hnsecs"(9_999_999)));
            TestST(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 20_888_888, SysTime(DateTime(1, 1, 1, 0, 0, 2), FracSec.from!"hnsecs"(888_887)));

            auto duration = dur!"seconds"(12);
            const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            static assert(!__traits(compiles, cst += duration));
            //static assert(!__traits(compiles, ist += duration));
            static assert(!__traits(compiles, cst -= duration));
            //static assert(!__traits(compiles, ist -= duration));
        }
    }


    /++
        Gives the difference between two $(D SysTime)s.

        The legal types of arithmetic for $(D SysTime) using this operator are

        $(BOOKTABLE,
        $(TR $(TD SysTime) $(TD -) $(TD SysTime) $(TD -->) $(TD duration))
        )
      +/
    Duration opBinary(string op)(in SysTime rhs) const pure nothrow
        if(op == "-")
    {
        return dur!"hnsecs"(_stdTime - rhs._stdTime);
    }

    unittest
    {
        version(testStdDateTime)
        {
            _assertPred!"=="(SysTime(DateTime(1999, 7, 6, 12, 30, 33)) - SysTime(DateTime(1998, 7, 6, 12, 30, 33)),
                        dur!"seconds"(31_536_000));
            _assertPred!"=="(SysTime(DateTime(1998, 7, 6, 12, 30, 33)) - SysTime(DateTime(1999, 7, 6, 12, 30, 33)),
                        dur!"seconds"(-31_536_000));

            _assertPred!"=="(SysTime(DateTime(1999, 8, 6, 12, 30, 33)) - SysTime(DateTime(1999, 7, 6, 12, 30, 33)),
                        dur!"seconds"(26_78_400));
            _assertPred!"=="(SysTime(DateTime(1999, 7, 6, 12, 30, 33)) - SysTime(DateTime(1999, 8, 6, 12, 30, 33)),
                        dur!"seconds"(-26_78_400));

            _assertPred!"=="(SysTime(DateTime(1999, 7, 6, 12, 30, 33)) - SysTime(DateTime(1999, 7, 5, 12, 30, 33)),
                        dur!"seconds"(86_400));
            _assertPred!"=="(SysTime(DateTime(1999, 7, 5, 12, 30, 33)) - SysTime(DateTime(1999, 7, 6, 12, 30, 33)),
                        dur!"seconds"(-86_400));

            _assertPred!"=="(SysTime(DateTime(1999, 7, 6, 12, 30, 33)) - SysTime(DateTime(1999, 7, 6, 11, 30, 33)),
                        dur!"seconds"(3600));
            _assertPred!"=="(SysTime(DateTime(1999, 7, 6, 11, 30, 33)) - SysTime(DateTime(1999, 7, 6, 12, 30, 33)),
                        dur!"seconds"(-3600));

            _assertPred!"=="(SysTime(DateTime(1999, 7, 6, 12, 31, 33)) - SysTime(DateTime(1999, 7, 6, 12, 30, 33)),
                        dur!"seconds"(60));
            _assertPred!"=="(SysTime(DateTime(1999, 7, 6, 12, 30, 33)) - SysTime(DateTime(1999, 7, 6, 12, 31, 33)),
                        dur!"seconds"(-60));

            _assertPred!"=="(SysTime(DateTime(1999, 7, 6, 12, 30, 34)) - SysTime(DateTime(1999, 7, 6, 12, 30, 33)),
                        dur!"seconds"(1));
            _assertPred!"=="(SysTime(DateTime(1999, 7, 6, 12, 30, 33)) - SysTime(DateTime(1999, 7, 6, 12, 30, 34)),
                        dur!"seconds"(-1));

            _assertPred!"=="(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(532)) - SysTime(DateTime(1999, 7, 6, 12, 30, 33)),
                        dur!"msecs"(532));
            _assertPred!"=="(SysTime(DateTime(1999, 7, 6, 12, 30, 33)) - SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"msecs"(532)),
                        dur!"msecs"(-532));

            _assertPred!"=="(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(333_347)) - SysTime(DateTime(1999, 7, 6, 12, 30, 33)),
                        dur!"usecs"(333_347));
            _assertPred!"=="(SysTime(DateTime(1999, 7, 6, 12, 30, 33)) - SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"usecs"(333_347)),
                        dur!"usecs"(-333_347));

            _assertPred!"=="(SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(1_234_567)) - SysTime(DateTime(1999, 7, 6, 12, 30, 33)),
                        dur!"hnsecs"(1_234_567));
            _assertPred!"=="(SysTime(DateTime(1999, 7, 6, 12, 30, 33)) - SysTime(DateTime(1999, 7, 6, 12, 30, 33), FracSec.from!"hnsecs"(1_234_567)),
                        dur!"hnsecs"(-1_234_567));

            _assertPred!"=="(SysTime(DateTime(1, 1, 1, 12, 30, 33)) - SysTime(DateTime(1, 1, 1, 0, 0, 0)), dur!"seconds"(45033));
            _assertPred!"=="(SysTime(DateTime(1, 1, 1, 0, 0, 0)) - SysTime(DateTime(1, 1, 1, 12, 30, 33)), dur!"seconds"(-45033));
            _assertPred!"=="(SysTime(DateTime(0, 12, 31, 12, 30, 33)) - SysTime(DateTime(1, 1, 1, 0, 0, 0)), dur!"seconds"(-41367));
            _assertPred!"=="(SysTime(DateTime(1, 1, 1, 0, 0, 0)) - SysTime(DateTime(0, 12, 31, 12, 30, 33)), dur!"seconds"(41367));

            _assertPred!"=="(SysTime(DateTime(1, 1, 1, 0, 0, 0)) - SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)),
                            dur!"hnsecs"(1));
            _assertPred!"=="(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)) - SysTime(DateTime(1, 1, 1, 0, 0, 0)),
                            dur!"hnsecs"(-1));

            auto tz = TimeZone.getTimeZone("America/Los_Angeles");

            _assertPred!"=="(SysTime(DateTime(2011, 1, 13, 8, 17, 2), FracSec.from!"msecs"(296), tz) -
                            SysTime(DateTime(2011, 1, 13, 8, 17, 2), FracSec.from!"msecs"(296), tz),
                            dur!"hnsecs"(0));

            _assertPred!"=="(SysTime(DateTime(2011, 1, 13, 8, 17, 2), FracSec.from!"msecs"(296), tz) -
                            SysTime(DateTime(2011, 1, 13, 8, 17, 2), FracSec.from!"msecs"(296), UTC()),
                            dur!"hours"(8));

            _assertPred!"=="(SysTime(DateTime(2011, 1, 13, 8, 17, 2), FracSec.from!"msecs"(296), UTC()) -
                            SysTime(DateTime(2011, 1, 13, 8, 17, 2), FracSec.from!"msecs"(296), tz),
                            dur!"hours"(-8));

            auto st = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            static assert(__traits(compiles, st - st));
            static assert(__traits(compiles, cst - st));
            //static assert(__traits(compiles, ist - st));

            static assert(__traits(compiles, st - cst));
            static assert(__traits(compiles, cst - cst));
            //static assert(__traits(compiles, ist - cst));

            //static assert(__traits(compiles, st - ist));
            //static assert(__traits(compiles, cst - ist));
            //static assert(__traits(compiles, ist - ist));
        }
    }


    /++
        Returns the difference between the two $(D SysTime)s in months.

        To get the difference in years, subtract the year property
        of two $(D SysTime)s. To get the difference in days or weeks,
        subtract the $(D SysTime)s themselves and use the $(D Duration)
        that results. Because converting between months and smaller
        units requires a specific date (which $(D Duration)s don't have),
        getting the difference in months requires some math using both
        the year and month properties, so this is a convenience function for
        getting the difference in months.

        Note that the number of days in the months or how far into the month
        either date is is irrelevant. It is the difference in the month property
        combined with the difference in years * 12. So, for instance,
        December 31st and January 1st are one month apart just as December 1st
        and January 31st are one month apart.

        Params:
            rhs = The $(D SysTime) to subtract from this one.

        Examples:
--------------------
assert(SysTime(Date(1999, 2, 1)).diffMonths(SysTime(Date(1999, 1, 31))) == 1);
assert(SysTime(Date(1999, 1, 31)).diffMonths(SysTime(Date(1999, 2, 1))) == -1);
assert(SysTime(Date(1999, 3, 1)).diffMonths(SysTime(Date(1999, 1, 1))) == 2);
assert(SysTime(Date(1999, 1, 1)).diffMonths(SysTime(Date(1999, 3, 31))) == -2);
--------------------
      +/
    int diffMonths(in SysTime rhs) const nothrow
    {
        return (cast(Date)this).diffMonths(cast(Date)rhs);
    }

    unittest
    {
        version(testStdDateTime)
        {
            auto st = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            static assert(__traits(compiles, st.diffMonths(st)));
            static assert(__traits(compiles, cst.diffMonths(st)));
            //static assert(__traits(compiles, ist.diffMonths(st)));

            static assert(__traits(compiles, st.diffMonths(cst)));
            static assert(__traits(compiles, cst.diffMonths(cst)));
            //static assert(__traits(compiles, ist.diffMonths(cst)));

            //static assert(__traits(compiles, st.diffMonths(ist)));
            //static assert(__traits(compiles, cst.diffMonths(ist)));
            //static assert(__traits(compiles, ist.diffMonths(ist)));

            //Verify Examples.
            assert(SysTime(Date(1999, 2, 1)).diffMonths(SysTime(Date(1999, 1, 31))) == 1);
            assert(SysTime(Date(1999, 1, 31)).diffMonths(SysTime(Date(1999, 2, 1))) == -1);
            assert(SysTime(Date(1999, 3, 1)).diffMonths(SysTime(Date(1999, 1, 1))) == 2);
            assert(SysTime(Date(1999, 1, 1)).diffMonths(SysTime(Date(1999, 3, 31))) == -2);
        }
    }


    /++
        Whether this $(D SysTime) is in a leap year.
     +/
    @property bool isLeapYear() const nothrow
    {
        return (cast(Date)this).isLeapYear;
    }

    unittest
    {
        version(testStdDateTime)
        {
            auto st = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            static assert(__traits(compiles, st.isLeapYear));
            static assert(__traits(compiles, cst.isLeapYear));
            //static assert(__traits(compiles, ist.isLeapYear));
        }
    }


    /++
        Day of the week this $(D SysTime) is on.
      +/
    @property DayOfWeek dayOfWeek() const nothrow
    {
        return getDayOfWeek(dayOfGregorianCal);
    }

    unittest
    {
        version(testStdDateTime)
        {
            auto st = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            static assert(__traits(compiles, st.dayOfWeek));
            static assert(__traits(compiles, cst.dayOfWeek));
            //static assert(__traits(compiles, ist.dayOfWeek));
        }
    }


    /++
        Day of the year this $(D SysTime) is on.

        Examples:
--------------------
assert(SysTime(DateTime(1999, 1, 1, 12, 22, 7)).dayOfYear == 1);
assert(SysTime(DateTime(1999, 12, 31, 7, 2, 59)).dayOfYear == 365);
assert(SysTime(DateTime(2000, 12, 31, 21, 20, 0)).dayOfYear == 366);
--------------------
      +/
    @property ushort dayOfYear() const nothrow
    {
        return (cast(Date)this).dayOfYear;
    }

    unittest
    {
        version(testStdDateTime)
        {
            auto st = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            static assert(__traits(compiles, st.dayOfYear));
            static assert(__traits(compiles, cst.dayOfYear));
            //static assert(__traits(compiles, ist.dayOfYear));

            //Verify Examples.
            assert(SysTime(DateTime(1999, 1, 1, 12, 22, 7)).dayOfYear == 1);
            assert(SysTime(DateTime(1999, 12, 31, 7, 2, 59)).dayOfYear == 365);
            assert(SysTime(DateTime(2000, 12, 31, 21, 20, 0)).dayOfYear == 366);
        }
    }


    /++
        Day of the year.

        Params:
            day = The day of the year to set which day of the year this
                  $(D SysTime) is on.
      +/
    @property void dayOfYear(int day)
    {
        immutable hnsecs = adjTime;
        immutable days = convert!("hnsecs", "days")(hnsecs);
        immutable theRest = hnsecs - convert!("days", "hnsecs")(days);

        auto date = Date(cast(int)days);
        date.dayOfYear = day;

        immutable newDaysHNSecs = convert!("days", "hnsecs")(date.dayOfGregorianCal - 1);

        adjTime = newDaysHNSecs + theRest;
    }

    unittest
    {
        version(testStdDateTime)
        {
            auto st = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            static assert(__traits(compiles, st.dayOfYear = 12));
            static assert(!__traits(compiles, cst.dayOfYear = 12));
            //static assert(!__traits(compiles, ist.dayOfYear = 12));
        }
    }


    /++
        The Xth day of the Gregorian Calendar that this $(D SysTime) is on.

        Examples:
--------------------
assert(SysTime(DateTime(1, 1, 1, 0, 0, 0)).dayOfGregorianCal == 1);
assert(SysTime(DateTime(1, 12, 31, 23, 59, 59)).dayOfGregorianCal == 365);
assert(SysTime(DateTime(2, 1, 1, 2, 2, 2)).dayOfGregorianCal == 366);

assert(SysTime(DateTime(0, 12, 31, 7, 7, 7)).dayOfGregorianCal == 0);
assert(SysTime(DateTime(0, 1, 1, 19, 30, 0)).dayOfGregorianCal == -365);
assert(SysTime(DateTime(-1, 12, 31, 4, 7, 0)).dayOfGregorianCal == -366);

assert(SysTime(DateTime(2000, 1, 1, 9, 30, 20)).dayOfGregorianCal == 730_120);
assert(SysTime(DateTime(2010, 12, 31, 15, 45, 50)).dayOfGregorianCal == 734_137);
--------------------
     +/
    @property int dayOfGregorianCal() const nothrow
    {
        immutable adjustedTime = adjTime;

        //We have to add one because 0 would be midnight, January 1st, 1 A.D.,
        //which would be the 1st day of the Gregorian Calendar, not the 0th. So,
        //simply casting to days is one day off.
        if(adjustedTime > 0)
            return cast(int)getUnitsFromHNSecs!"days"(adjustedTime) + 1;

        auto hnsecs = adjustedTime;
        immutable days = cast(int)splitUnitsFromHNSecs!"days"(hnsecs);

        return hnsecs == 0 ? days + 1 : days;
    }

    unittest
    {
        version(testStdDateTime)
        {
            //Test A.D.
            _assertPred!"=="(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)).dayOfGregorianCal, 1);
            _assertPred!"=="(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(1)).dayOfGregorianCal, 1);
            _assertPred!"=="(SysTime(DateTime(1, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)).dayOfGregorianCal, 1);

            _assertPred!"=="(SysTime(DateTime(1, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, 1);
            _assertPred!"=="(SysTime(DateTime(1, 1, 2, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, 2);
            _assertPred!"=="(SysTime(DateTime(1, 2, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, 32);
            _assertPred!"=="(SysTime(DateTime(2, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, 366);
            _assertPred!"=="(SysTime(DateTime(3, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, 731);
            _assertPred!"=="(SysTime(DateTime(4, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, 1096);
            _assertPred!"=="(SysTime(DateTime(5, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, 1462);
            _assertPred!"=="(SysTime(DateTime(50, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, 17_898);
            _assertPred!"=="(SysTime(DateTime(97, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, 35_065);
            _assertPred!"=="(SysTime(DateTime(100, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, 36_160);
            _assertPred!"=="(SysTime(DateTime(101, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, 36_525);
            _assertPred!"=="(SysTime(DateTime(105, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, 37_986);
            _assertPred!"=="(SysTime(DateTime(200, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, 72_684);
            _assertPred!"=="(SysTime(DateTime(201, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, 73_049);
            _assertPred!"=="(SysTime(DateTime(300, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, 109_208);
            _assertPred!"=="(SysTime(DateTime(301, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, 109_573);
            _assertPred!"=="(SysTime(DateTime(400, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, 145_732);
            _assertPred!"=="(SysTime(DateTime(401, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, 146_098);
            _assertPred!"=="(SysTime(DateTime(500, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, 182_257);
            _assertPred!"=="(SysTime(DateTime(501, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, 182_622);
            _assertPred!"=="(SysTime(DateTime(1000, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, 364_878);
            _assertPred!"=="(SysTime(DateTime(1001, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, 365_243);
            _assertPred!"=="(SysTime(DateTime(1600, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, 584_023);
            _assertPred!"=="(SysTime(DateTime(1601, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, 584_389);
            _assertPred!"=="(SysTime(DateTime(1900, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, 693_596);
            _assertPred!"=="(SysTime(DateTime(1901, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, 693_961);
            _assertPred!"=="(SysTime(DateTime(1945, 11, 12, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, 710_347);
            _assertPred!"=="(SysTime(DateTime(1999, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, 729_755);
            _assertPred!"=="(SysTime(DateTime(2000, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, 730_120);
            _assertPred!"=="(SysTime(DateTime(2001, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, 730_486);

            _assertPred!"=="(SysTime(DateTime(2010, 1, 1, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, 733_773);
            _assertPred!"=="(SysTime(DateTime(2010, 1, 31, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, 733_803);
            _assertPred!"=="(SysTime(DateTime(2010, 2, 1, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, 733_804);
            _assertPred!"=="(SysTime(DateTime(2010, 2, 28, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, 733_831);
            _assertPred!"=="(SysTime(DateTime(2010, 3, 1, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, 733_832);
            _assertPred!"=="(SysTime(DateTime(2010, 3, 31, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, 733_862);
            _assertPred!"=="(SysTime(DateTime(2010, 4, 1, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, 733_863);
            _assertPred!"=="(SysTime(DateTime(2010, 4, 30, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, 733_892);
            _assertPred!"=="(SysTime(DateTime(2010, 5, 1, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, 733_893);
            _assertPred!"=="(SysTime(DateTime(2010, 5, 31, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, 733_923);
            _assertPred!"=="(SysTime(DateTime(2010, 6, 1, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, 733_924);
            _assertPred!"=="(SysTime(DateTime(2010, 6, 30, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, 733_953);
            _assertPred!"=="(SysTime(DateTime(2010, 7, 1, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, 733_954);
            _assertPred!"=="(SysTime(DateTime(2010, 7, 31, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, 733_984);
            _assertPred!"=="(SysTime(DateTime(2010, 8, 1, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, 733_985);
            _assertPred!"=="(SysTime(DateTime(2010, 8, 31, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, 734_015);
            _assertPred!"=="(SysTime(DateTime(2010, 9, 1, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, 734_016);
            _assertPred!"=="(SysTime(DateTime(2010, 9, 30, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, 734_045);
            _assertPred!"=="(SysTime(DateTime(2010, 10, 1, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, 734_046);
            _assertPred!"=="(SysTime(DateTime(2010, 10, 31, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, 734_076);
            _assertPred!"=="(SysTime(DateTime(2010, 11, 1, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, 734_077);
            _assertPred!"=="(SysTime(DateTime(2010, 11, 30, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, 734_106);
            _assertPred!"=="(SysTime(DateTime(2010, 12, 1, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, 734_107);
            _assertPred!"=="(SysTime(DateTime(2010, 12, 31, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, 734_137);

            _assertPred!"=="(SysTime(DateTime(2012, 2, 1, 0, 0, 0), FracSec.from!"msecs"(0)).dayOfGregorianCal, 734_534);
            _assertPred!"=="(SysTime(DateTime(2012, 2, 28, 0, 0, 0), FracSec.from!"msecs"(0)).dayOfGregorianCal, 734_561);
            _assertPred!"=="(SysTime(DateTime(2012, 2, 29, 0, 0, 0), FracSec.from!"msecs"(0)).dayOfGregorianCal, 734_562);
            _assertPred!"=="(SysTime(DateTime(2012, 3, 1, 0, 0, 0), FracSec.from!"msecs"(0)).dayOfGregorianCal, 734_563);

            //Test B.C.
            _assertPred!"=="(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)).dayOfGregorianCal, 0);
            _assertPred!"=="(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_998)).dayOfGregorianCal, 0);
            _assertPred!"=="(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(0)).dayOfGregorianCal, 0);
            _assertPred!"=="(SysTime(DateTime(0, 12, 31, 0, 0, 0), FracSec.from!"hnsecs"(1)).dayOfGregorianCal, 0);
            _assertPred!"=="(SysTime(DateTime(0, 12, 31, 0, 0, 0), FracSec.from!"hnsecs"(0)).dayOfGregorianCal, 0);

            _assertPred!"=="(SysTime(DateTime(-1, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)).dayOfGregorianCal, -366);
            _assertPred!"=="(SysTime(DateTime(-1, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_998)).dayOfGregorianCal, -366);
            _assertPred!"=="(SysTime(DateTime(-1, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(0)).dayOfGregorianCal, -366);
            _assertPred!"=="(SysTime(DateTime(-1, 12, 31, 0, 0, 0), FracSec.from!"hnsecs"(0)).dayOfGregorianCal, -366);

            _assertPred!"=="(SysTime(DateTime(0, 12, 31, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, 0);
            _assertPred!"=="(SysTime(DateTime(0, 12, 30, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, -1);
            _assertPred!"=="(SysTime(DateTime(0, 12, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, -30);
            _assertPred!"=="(SysTime(DateTime(0, 11, 30, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, -31);

            _assertPred!"=="(SysTime(DateTime(-1, 12, 31, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, -366);
            _assertPred!"=="(SysTime(DateTime(-1, 12, 30, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, -367);
            _assertPred!"=="(SysTime(DateTime(-1, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, -730);
            _assertPred!"=="(SysTime(DateTime(-2, 12, 31, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, -731);
            _assertPred!"=="(SysTime(DateTime(-2, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, -1095);
            _assertPred!"=="(SysTime(DateTime(-3, 12, 31, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, -1096);
            _assertPred!"=="(SysTime(DateTime(-3, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, -1460);
            _assertPred!"=="(SysTime(DateTime(-4, 12, 31, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, -1461);
            _assertPred!"=="(SysTime(DateTime(-4, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, -1826);
            _assertPred!"=="(SysTime(DateTime(-5, 12, 31, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, -1827);
            _assertPred!"=="(SysTime(DateTime(-5, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, -2191);
            _assertPred!"=="(SysTime(DateTime(-9, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, -3652);

            _assertPred!"=="(SysTime(DateTime(-49, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, -18_262);
            _assertPred!"=="(SysTime(DateTime(-50, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, -18_627);
            _assertPred!"=="(SysTime(DateTime(-97, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, -35_794);
            _assertPred!"=="(SysTime(DateTime(-99, 12, 31, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, -36_160);
            _assertPred!"=="(SysTime(DateTime(-99, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, -36_524);
            _assertPred!"=="(SysTime(DateTime(-100, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, -36_889);
            _assertPred!"=="(SysTime(DateTime(-101, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, -37_254);
            _assertPred!"=="(SysTime(DateTime(-105, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, -38_715);
            _assertPred!"=="(SysTime(DateTime(-200, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, -73_413);
            _assertPred!"=="(SysTime(DateTime(-201, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, -73_778);
            _assertPred!"=="(SysTime(DateTime(-300, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, -109_937);
            _assertPred!"=="(SysTime(DateTime(-301, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, -110_302);
            _assertPred!"=="(SysTime(DateTime(-400, 12, 31, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, -146_097);
            _assertPred!"=="(SysTime(DateTime(-400, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, -146_462);
            _assertPred!"=="(SysTime(DateTime(-401, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, -146_827);
            _assertPred!"=="(SysTime(DateTime(-499, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, -182_621);
            _assertPred!"=="(SysTime(DateTime(-500, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, -182_986);
            _assertPred!"=="(SysTime(DateTime(-501, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, -183_351);
            _assertPred!"=="(SysTime(DateTime(-1000, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, -365_607);
            _assertPred!"=="(SysTime(DateTime(-1001, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, -365_972);
            _assertPred!"=="(SysTime(DateTime(-1599, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, -584_387);
            _assertPred!"=="(SysTime(DateTime(-1600, 12, 31, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, -584_388);
            _assertPred!"=="(SysTime(DateTime(-1600, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, -584_753);
            _assertPred!"=="(SysTime(DateTime(-1601, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, -585_118);
            _assertPred!"=="(SysTime(DateTime(-1900, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, -694_325);
            _assertPred!"=="(SysTime(DateTime(-1901, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, -694_690);
            _assertPred!"=="(SysTime(DateTime(-1999, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, -730_484);
            _assertPred!"=="(SysTime(DateTime(-2000, 12, 31, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, -730_485);
            _assertPred!"=="(SysTime(DateTime(-2000, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, -730_850);
            _assertPred!"=="(SysTime(DateTime(-2001, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)).dayOfGregorianCal, -731_215);

            _assertPred!"=="(SysTime(DateTime(-2010, 1, 1, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, -734_502);
            _assertPred!"=="(SysTime(DateTime(-2010, 1, 31, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, -734_472);
            _assertPred!"=="(SysTime(DateTime(-2010, 2, 1, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, -734_471);
            _assertPred!"=="(SysTime(DateTime(-2010, 2, 28, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, -734_444);
            _assertPred!"=="(SysTime(DateTime(-2010, 3, 1, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, -734_443);
            _assertPred!"=="(SysTime(DateTime(-2010, 3, 31, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, -734_413);
            _assertPred!"=="(SysTime(DateTime(-2010, 4, 1, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, -734_412);
            _assertPred!"=="(SysTime(DateTime(-2010, 4, 30, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, -734_383);
            _assertPred!"=="(SysTime(DateTime(-2010, 5, 1, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, -734_382);
            _assertPred!"=="(SysTime(DateTime(-2010, 5, 31, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, -734_352);
            _assertPred!"=="(SysTime(DateTime(-2010, 6, 1, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, -734_351);
            _assertPred!"=="(SysTime(DateTime(-2010, 6, 30, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, -734_322);
            _assertPred!"=="(SysTime(DateTime(-2010, 7, 1, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, -734_321);
            _assertPred!"=="(SysTime(DateTime(-2010, 7, 31, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, -734_291);
            _assertPred!"=="(SysTime(DateTime(-2010, 8, 1, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, -734_290);
            _assertPred!"=="(SysTime(DateTime(-2010, 8, 31, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, -734_260);
            _assertPred!"=="(SysTime(DateTime(-2010, 9, 1, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, -734_259);
            _assertPred!"=="(SysTime(DateTime(-2010, 9, 30, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, -734_230);
            _assertPred!"=="(SysTime(DateTime(-2010, 10, 1, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, -734_229);
            _assertPred!"=="(SysTime(DateTime(-2010, 10, 31, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, -734_199);
            _assertPred!"=="(SysTime(DateTime(-2010, 11, 1, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, -734_198);
            _assertPred!"=="(SysTime(DateTime(-2010, 11, 30, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, -734_169);
            _assertPred!"=="(SysTime(DateTime(-2010, 12, 1, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, -734_168);
            _assertPred!"=="(SysTime(DateTime(-2010, 12, 31, 23, 59, 59), FracSec.from!"msecs"(999)).dayOfGregorianCal, -734_138);

            _assertPred!"=="(SysTime(DateTime(-2012, 2, 1, 0, 0, 0), FracSec.from!"msecs"(0)).dayOfGregorianCal, -735_202);
            _assertPred!"=="(SysTime(DateTime(-2012, 2, 28, 0, 0, 0), FracSec.from!"msecs"(0)).dayOfGregorianCal, -735_175);
            _assertPred!"=="(SysTime(DateTime(-2012, 2, 29, 0, 0, 0), FracSec.from!"msecs"(0)).dayOfGregorianCal, -735_174);
            _assertPred!"=="(SysTime(DateTime(-2012, 3, 1, 0, 0, 0), FracSec.from!"msecs"(0)).dayOfGregorianCal, -735_173);

            _assertPred!"=="(SysTime(DateTime(-3760, 9, 7, 0, 0, 0), FracSec.from!"msecs"(0)).dayOfGregorianCal, -1_373_427); //Start of Hebrew Calendar

            const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            static assert(__traits(compiles, cst.dayOfGregorianCal));
            //static assert(__traits(compiles, ist.dayOfGregorianCal));

            //Verify Examples.
            assert(SysTime(DateTime(1, 1, 1, 0, 0, 0)).dayOfGregorianCal == 1);
            assert(SysTime(DateTime(1, 12, 31, 23, 59, 59)).dayOfGregorianCal == 365);
            assert(SysTime(DateTime(2, 1, 1, 2, 2, 2)).dayOfGregorianCal == 366);

            assert(SysTime(DateTime(0, 12, 31, 7, 7, 7)).dayOfGregorianCal == 0);
            assert(SysTime(DateTime(0, 1, 1, 19, 30, 0)).dayOfGregorianCal == -365);
            assert(SysTime(DateTime(-1, 12, 31, 4, 7, 0)).dayOfGregorianCal == -366);

            assert(SysTime(DateTime(2000, 1, 1, 9, 30, 20)).dayOfGregorianCal == 730_120);
            assert(SysTime(DateTime(2010, 12, 31, 15, 45, 50)).dayOfGregorianCal == 734_137);
        }
    }


    //Test that the logic for the day of the Gregorian Calendar is consistent
    //between Date and SysTime.
    unittest
    {
        version(testStdDateTime)
        {
            //Test A.D.
            _assertPred!"=="(Date(1, 1, 1).dayOfGregorianCal, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)).dayOfGregorianCal);
            _assertPred!"=="(Date(1, 1, 2).dayOfGregorianCal, SysTime(DateTime(1, 1, 2, 0, 0, 0), FracSec.from!"hnsecs"(500)).dayOfGregorianCal);
            _assertPred!"=="(Date(1, 2, 1).dayOfGregorianCal, SysTime(DateTime(1, 2, 1, 0, 0, 0), FracSec.from!"hnsecs"(50_000)).dayOfGregorianCal);
            _assertPred!"=="(Date(2, 1, 1).dayOfGregorianCal, SysTime(DateTime(2, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(9_999_999)).dayOfGregorianCal);
            _assertPred!"=="(Date(3, 1, 1).dayOfGregorianCal, SysTime(DateTime(3, 1, 1, 12, 13, 14), FracSec.from!"hnsecs"(0)).dayOfGregorianCal);
            _assertPred!"=="(Date(4, 1, 1).dayOfGregorianCal, SysTime(DateTime(4, 1, 1, 12, 13, 14), FracSec.from!"hnsecs"(500)).dayOfGregorianCal);
            _assertPred!"=="(Date(5, 1, 1).dayOfGregorianCal, SysTime(DateTime(5, 1, 1, 12, 13, 14), FracSec.from!"hnsecs"(50_000)).dayOfGregorianCal);
            _assertPred!"=="(Date(50, 1, 1).dayOfGregorianCal, SysTime(DateTime(50, 1, 1, 12, 13, 14), FracSec.from!"hnsecs"(9_999_999)).dayOfGregorianCal);
            _assertPred!"=="(Date(97, 1, 1).dayOfGregorianCal, SysTime(DateTime(97, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(0)).dayOfGregorianCal);
            _assertPred!"=="(Date(100, 1, 1).dayOfGregorianCal, SysTime(DateTime(100, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(500)).dayOfGregorianCal);
            _assertPred!"=="(Date(101, 1, 1).dayOfGregorianCal, SysTime(DateTime(101, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(50_000)).dayOfGregorianCal);
            _assertPred!"=="(Date(105, 1, 1).dayOfGregorianCal, SysTime(DateTime(105, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)).dayOfGregorianCal);
            _assertPred!"=="(Date(200, 1, 1).dayOfGregorianCal, SysTime(DateTime(200, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)).dayOfGregorianCal);
            _assertPred!"=="(Date(201, 1, 1).dayOfGregorianCal, SysTime(DateTime(201, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(500)).dayOfGregorianCal);
            _assertPred!"=="(Date(300, 1, 1).dayOfGregorianCal, SysTime(DateTime(300, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(50_000)).dayOfGregorianCal);
            _assertPred!"=="(Date(301, 1, 1).dayOfGregorianCal, SysTime(DateTime(301, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(9_999_999)).dayOfGregorianCal);
            _assertPred!"=="(Date(400, 1, 1).dayOfGregorianCal, SysTime(DateTime(400, 1, 1, 12, 13, 14), FracSec.from!"hnsecs"(0)).dayOfGregorianCal);
            _assertPred!"=="(Date(401, 1, 1).dayOfGregorianCal, SysTime(DateTime(401, 1, 1, 12, 13, 14), FracSec.from!"hnsecs"(500)).dayOfGregorianCal);
            _assertPred!"=="(Date(500, 1, 1).dayOfGregorianCal, SysTime(DateTime(500, 1, 1, 12, 13, 14), FracSec.from!"hnsecs"(50_000)).dayOfGregorianCal);
            _assertPred!"=="(Date(501, 1, 1).dayOfGregorianCal, SysTime(DateTime(501, 1, 1, 12, 13, 14), FracSec.from!"hnsecs"(9_999_999)).dayOfGregorianCal);
            _assertPred!"=="(Date(1000, 1, 1).dayOfGregorianCal, SysTime(DateTime(1000, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(0)).dayOfGregorianCal);
            _assertPred!"=="(Date(1001, 1, 1).dayOfGregorianCal, SysTime(DateTime(1001, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(500)).dayOfGregorianCal);
            _assertPred!"=="(Date(1600, 1, 1).dayOfGregorianCal, SysTime(DateTime(1600, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(50_000)).dayOfGregorianCal);
            _assertPred!"=="(Date(1601, 1, 1).dayOfGregorianCal, SysTime(DateTime(1601, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)).dayOfGregorianCal);
            _assertPred!"=="(Date(1900, 1, 1).dayOfGregorianCal, SysTime(DateTime(1900, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)).dayOfGregorianCal);
            _assertPred!"=="(Date(1901, 1, 1).dayOfGregorianCal, SysTime(DateTime(1901, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(500)).dayOfGregorianCal);
            _assertPred!"=="(Date(1945, 11, 12).dayOfGregorianCal, SysTime(DateTime(1945, 11, 12, 0, 0, 0), FracSec.from!"hnsecs"(50_000)).dayOfGregorianCal);
            _assertPred!"=="(Date(1999, 1, 1).dayOfGregorianCal, SysTime(DateTime(1999, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(9_999_999)).dayOfGregorianCal);
            _assertPred!"=="(Date(1999, 7, 6).dayOfGregorianCal, SysTime(DateTime(1999, 7, 6, 12, 13, 14), FracSec.from!"hnsecs"(0)).dayOfGregorianCal);
            _assertPred!"=="(Date(2000, 1, 1).dayOfGregorianCal, SysTime(DateTime(2000, 1, 1, 12, 13, 14), FracSec.from!"hnsecs"(500)).dayOfGregorianCal);
            _assertPred!"=="(Date(2001, 1, 1).dayOfGregorianCal, SysTime(DateTime(2001, 1, 1, 12, 13, 14), FracSec.from!"hnsecs"(50_000)).dayOfGregorianCal);

            _assertPred!"=="(Date(2010, 1, 1).dayOfGregorianCal, SysTime(DateTime(2010, 1, 1, 12, 13, 14), FracSec.from!"hnsecs"(9_999_999)).dayOfGregorianCal);
            _assertPred!"=="(Date(2010, 1, 31).dayOfGregorianCal, SysTime(DateTime(2010, 1, 31, 23, 0, 0), FracSec.from!"hnsecs"(0)).dayOfGregorianCal);
            _assertPred!"=="(Date(2010, 2, 1).dayOfGregorianCal, SysTime(DateTime(2010, 2, 1, 23, 59, 59), FracSec.from!"hnsecs"(500)).dayOfGregorianCal);
            _assertPred!"=="(Date(2010, 2, 28).dayOfGregorianCal, SysTime(DateTime(2010, 2, 28, 23, 59, 59), FracSec.from!"hnsecs"(50_000)).dayOfGregorianCal);
            _assertPred!"=="(Date(2010, 3, 1).dayOfGregorianCal, SysTime(DateTime(2010, 3, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)).dayOfGregorianCal);
            _assertPred!"=="(Date(2010, 3, 31).dayOfGregorianCal, SysTime(DateTime(2010, 3, 31, 0, 0, 0), FracSec.from!"hnsecs"(0)).dayOfGregorianCal);
            _assertPred!"=="(Date(2010, 4, 1).dayOfGregorianCal, SysTime(DateTime(2010, 4, 1, 0, 0, 0), FracSec.from!"hnsecs"(500)).dayOfGregorianCal);
            _assertPred!"=="(Date(2010, 4, 30).dayOfGregorianCal, SysTime(DateTime(2010, 4, 30, 0, 0, 0), FracSec.from!"hnsecs"(50_000)).dayOfGregorianCal);
            _assertPred!"=="(Date(2010, 5, 1).dayOfGregorianCal, SysTime(DateTime(2010, 5, 1, 0, 0, 0), FracSec.from!"hnsecs"(9_999_999)).dayOfGregorianCal);
            _assertPred!"=="(Date(2010, 5, 31).dayOfGregorianCal, SysTime(DateTime(2010, 5, 31, 12, 13, 14), FracSec.from!"hnsecs"(0)).dayOfGregorianCal);
            _assertPred!"=="(Date(2010, 6, 1).dayOfGregorianCal, SysTime(DateTime(2010, 6, 1, 12, 13, 14), FracSec.from!"hnsecs"(500)).dayOfGregorianCal);
            _assertPred!"=="(Date(2010, 6, 30).dayOfGregorianCal, SysTime(DateTime(2010, 6, 30, 12, 13, 14), FracSec.from!"hnsecs"(50_000)).dayOfGregorianCal);
            _assertPred!"=="(Date(2010, 7, 1).dayOfGregorianCal, SysTime(DateTime(2010, 7, 1, 12, 13, 14), FracSec.from!"hnsecs"(9_999_999)).dayOfGregorianCal);
            _assertPred!"=="(Date(2010, 7, 31).dayOfGregorianCal, SysTime(DateTime(2010, 7, 31, 23, 59, 59), FracSec.from!"hnsecs"(0)).dayOfGregorianCal);
            _assertPred!"=="(Date(2010, 8, 1).dayOfGregorianCal, SysTime(DateTime(2010, 8, 1, 23, 59, 59), FracSec.from!"hnsecs"(500)).dayOfGregorianCal);
            _assertPred!"=="(Date(2010, 8, 31).dayOfGregorianCal, SysTime(DateTime(2010, 8, 31, 23, 59, 59), FracSec.from!"hnsecs"(50_000)).dayOfGregorianCal);
            _assertPred!"=="(Date(2010, 9, 1).dayOfGregorianCal, SysTime(DateTime(2010, 9, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)).dayOfGregorianCal);
            _assertPred!"=="(Date(2010, 9, 30).dayOfGregorianCal, SysTime(DateTime(2010, 9, 30, 12, 0, 0), FracSec.from!"hnsecs"(0)).dayOfGregorianCal);
            _assertPred!"=="(Date(2010, 10, 1).dayOfGregorianCal, SysTime(DateTime(2010, 10, 1, 0, 12, 0), FracSec.from!"hnsecs"(500)).dayOfGregorianCal);
            _assertPred!"=="(Date(2010, 10, 31).dayOfGregorianCal, SysTime(DateTime(2010, 10, 31, 0, 0, 12), FracSec.from!"hnsecs"(50_000)).dayOfGregorianCal);
            _assertPred!"=="(Date(2010, 11, 1).dayOfGregorianCal, SysTime(DateTime(2010, 11, 1, 23, 0, 0), FracSec.from!"hnsecs"(9_999_999)).dayOfGregorianCal);
            _assertPred!"=="(Date(2010, 11, 30).dayOfGregorianCal, SysTime(DateTime(2010, 11, 30, 0, 59, 0), FracSec.from!"hnsecs"(0)).dayOfGregorianCal);
            _assertPred!"=="(Date(2010, 12, 1).dayOfGregorianCal, SysTime(DateTime(2010, 12, 1, 0, 0, 59), FracSec.from!"hnsecs"(500)).dayOfGregorianCal);
            _assertPred!"=="(Date(2010, 12, 31).dayOfGregorianCal, SysTime(DateTime(2010, 12, 31, 0, 59, 59), FracSec.from!"hnsecs"(50_000)).dayOfGregorianCal);

            _assertPred!"=="(Date(2012, 2, 1).dayOfGregorianCal, SysTime(DateTime(2012, 2, 1, 23, 0, 59), FracSec.from!"hnsecs"(9_999_999)).dayOfGregorianCal);
            _assertPred!"=="(Date(2012, 2, 28).dayOfGregorianCal, SysTime(DateTime(2012, 2, 28, 23, 59, 0), FracSec.from!"hnsecs"(0)).dayOfGregorianCal);
            _assertPred!"=="(Date(2012, 2, 29).dayOfGregorianCal, SysTime(DateTime(2012, 2, 29, 7, 7, 7), FracSec.from!"hnsecs"(7)).dayOfGregorianCal);
            _assertPred!"=="(Date(2012, 3, 1).dayOfGregorianCal, SysTime(DateTime(2012, 3, 1, 7, 7, 7), FracSec.from!"hnsecs"(7)).dayOfGregorianCal);

            //Test B.C.
            _assertPred!"=="(Date(0, 12, 31).dayOfGregorianCal, SysTime(DateTime(0, 12, 31, 0, 0, 0), FracSec.from!"hnsecs"(0)).dayOfGregorianCal);
            _assertPred!"=="(Date(0, 12, 30).dayOfGregorianCal, SysTime(DateTime(0, 12, 30, 0, 0, 0), FracSec.from!"hnsecs"(500)).dayOfGregorianCal);
            _assertPred!"=="(Date(0, 12, 1).dayOfGregorianCal, SysTime(DateTime(0, 12, 1, 0, 0, 0), FracSec.from!"hnsecs"(50_000)).dayOfGregorianCal);
            _assertPred!"=="(Date(0, 11, 30).dayOfGregorianCal, SysTime(DateTime(0, 11, 30, 0, 0, 0), FracSec.from!"hnsecs"(9_999_999)).dayOfGregorianCal);

            _assertPred!"=="(Date(-1, 12, 31).dayOfGregorianCal, SysTime(DateTime(-1, 12, 31, 12, 13, 14), FracSec.from!"hnsecs"(0)).dayOfGregorianCal);
            _assertPred!"=="(Date(-1, 12, 30).dayOfGregorianCal, SysTime(DateTime(-1, 12, 30, 12, 13, 14), FracSec.from!"hnsecs"(500)).dayOfGregorianCal);
            _assertPred!"=="(Date(-1, 1, 1).dayOfGregorianCal, SysTime(DateTime(-1, 1, 1, 12, 13, 14), FracSec.from!"hnsecs"(50_000)).dayOfGregorianCal);
            _assertPred!"=="(Date(-2, 12, 31).dayOfGregorianCal, SysTime(DateTime(-2, 12, 31, 12, 13, 14), FracSec.from!"hnsecs"(9_999_999)).dayOfGregorianCal);
            _assertPred!"=="(Date(-2, 1, 1).dayOfGregorianCal, SysTime(DateTime(-2, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(0)).dayOfGregorianCal);
            _assertPred!"=="(Date(-3, 12, 31).dayOfGregorianCal, SysTime(DateTime(-3, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(500)).dayOfGregorianCal);
            _assertPred!"=="(Date(-3, 1, 1).dayOfGregorianCal, SysTime(DateTime(-3, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(50_000)).dayOfGregorianCal);
            _assertPred!"=="(Date(-4, 12, 31).dayOfGregorianCal, SysTime(DateTime(-4, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)).dayOfGregorianCal);
            _assertPred!"=="(Date(-4, 1, 1).dayOfGregorianCal, SysTime(DateTime(-4, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)).dayOfGregorianCal);
            _assertPred!"=="(Date(-5, 12, 31).dayOfGregorianCal, SysTime(DateTime(-5, 12, 31, 0, 0, 0), FracSec.from!"hnsecs"(500)).dayOfGregorianCal);
            _assertPred!"=="(Date(-5, 1, 1).dayOfGregorianCal, SysTime(DateTime(-5, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(50_000)).dayOfGregorianCal);
            _assertPred!"=="(Date(-9, 1, 1).dayOfGregorianCal, SysTime(DateTime(-9, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(9_999_999)).dayOfGregorianCal);

            _assertPred!"=="(Date(-49, 1, 1).dayOfGregorianCal, SysTime(DateTime(-49, 1, 1, 12, 13, 14), FracSec.from!"hnsecs"(0)).dayOfGregorianCal);
            _assertPred!"=="(Date(-50, 1, 1).dayOfGregorianCal, SysTime(DateTime(-50, 1, 1, 12, 13, 14), FracSec.from!"hnsecs"(500)).dayOfGregorianCal);
            _assertPred!"=="(Date(-97, 1, 1).dayOfGregorianCal, SysTime(DateTime(-97, 1, 1, 12, 13, 14), FracSec.from!"hnsecs"(50_000)).dayOfGregorianCal);
            _assertPred!"=="(Date(-99, 12, 31).dayOfGregorianCal, SysTime(DateTime(-99, 12, 31, 12, 13, 14), FracSec.from!"hnsecs"(9_999_999)).dayOfGregorianCal);
            _assertPred!"=="(Date(-99, 1, 1).dayOfGregorianCal, SysTime(DateTime(-99, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(0)).dayOfGregorianCal);
            _assertPred!"=="(Date(-100, 1, 1).dayOfGregorianCal, SysTime(DateTime(-100, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(500)).dayOfGregorianCal);
            _assertPred!"=="(Date(-101, 1, 1).dayOfGregorianCal, SysTime(DateTime(-101, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(50_000)).dayOfGregorianCal);
            _assertPred!"=="(Date(-105, 1, 1).dayOfGregorianCal, SysTime(DateTime(-105, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)).dayOfGregorianCal);
            _assertPred!"=="(Date(-200, 1, 1).dayOfGregorianCal, SysTime(DateTime(-200, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)).dayOfGregorianCal);
            _assertPred!"=="(Date(-201, 1, 1).dayOfGregorianCal, SysTime(DateTime(-201, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(500)).dayOfGregorianCal);
            _assertPred!"=="(Date(-300, 1, 1).dayOfGregorianCal, SysTime(DateTime(-300, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(50_000)).dayOfGregorianCal);
            _assertPred!"=="(Date(-301, 1, 1).dayOfGregorianCal, SysTime(DateTime(-301, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(9_999_999)).dayOfGregorianCal);
            _assertPred!"=="(Date(-400, 12, 31).dayOfGregorianCal, SysTime(DateTime(-400, 12, 31, 12, 13, 14), FracSec.from!"hnsecs"(0)).dayOfGregorianCal);
            _assertPred!"=="(Date(-400, 1, 1).dayOfGregorianCal, SysTime(DateTime(-400, 1, 1, 12, 13, 14), FracSec.from!"hnsecs"(500)).dayOfGregorianCal);
            _assertPred!"=="(Date(-401, 1, 1).dayOfGregorianCal, SysTime(DateTime(-401, 1, 1, 12, 13, 14), FracSec.from!"hnsecs"(50_000)).dayOfGregorianCal);
            _assertPred!"=="(Date(-499, 1, 1).dayOfGregorianCal, SysTime(DateTime(-499, 1, 1, 12, 13, 14), FracSec.from!"hnsecs"(9_999_999)).dayOfGregorianCal);
            _assertPred!"=="(Date(-500, 1, 1).dayOfGregorianCal, SysTime(DateTime(-500, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(0)).dayOfGregorianCal);
            _assertPred!"=="(Date(-501, 1, 1).dayOfGregorianCal, SysTime(DateTime(-501, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(500)).dayOfGregorianCal);
            _assertPred!"=="(Date(-1000, 1, 1).dayOfGregorianCal, SysTime(DateTime(-1000, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(50_000)).dayOfGregorianCal);
            _assertPred!"=="(Date(-1001, 1, 1).dayOfGregorianCal, SysTime(DateTime(-1001, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)).dayOfGregorianCal);
            _assertPred!"=="(Date(-1599, 1, 1).dayOfGregorianCal, SysTime(DateTime(-1599, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)).dayOfGregorianCal);
            _assertPred!"=="(Date(-1600, 12, 31).dayOfGregorianCal, SysTime(DateTime(-1600, 12, 31, 0, 0, 0), FracSec.from!"hnsecs"(500)).dayOfGregorianCal);
            _assertPred!"=="(Date(-1600, 1, 1).dayOfGregorianCal, SysTime(DateTime(-1600, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(50_000)).dayOfGregorianCal);
            _assertPred!"=="(Date(-1601, 1, 1).dayOfGregorianCal, SysTime(DateTime(-1601, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(9_999_999)).dayOfGregorianCal);
            _assertPred!"=="(Date(-1900, 1, 1).dayOfGregorianCal, SysTime(DateTime(-1900, 1, 1, 12, 13, 14), FracSec.from!"hnsecs"(0)).dayOfGregorianCal);
            _assertPred!"=="(Date(-1901, 1, 1).dayOfGregorianCal, SysTime(DateTime(-1901, 1, 1, 12, 13, 14), FracSec.from!"hnsecs"(500)).dayOfGregorianCal);
            _assertPred!"=="(Date(-1999, 1, 1).dayOfGregorianCal, SysTime(DateTime(-1999, 1, 1, 12, 13, 14), FracSec.from!"hnsecs"(50_000)).dayOfGregorianCal);
            _assertPred!"=="(Date(-1999, 7, 6).dayOfGregorianCal, SysTime(DateTime(-1999, 7, 6, 12, 13, 14), FracSec.from!"hnsecs"(9_999_999)).dayOfGregorianCal);
            _assertPred!"=="(Date(-2000, 12, 31).dayOfGregorianCal, SysTime(DateTime(-2000, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(0)).dayOfGregorianCal);
            _assertPred!"=="(Date(-2000, 1, 1).dayOfGregorianCal, SysTime(DateTime(-2000, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(500)).dayOfGregorianCal);
            _assertPred!"=="(Date(-2001, 1, 1).dayOfGregorianCal, SysTime(DateTime(-2001, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(50_000)).dayOfGregorianCal);

            _assertPred!"=="(Date(-2010, 1, 1).dayOfGregorianCal, SysTime(DateTime(-2010, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)).dayOfGregorianCal);
            _assertPred!"=="(Date(-2010, 1, 31).dayOfGregorianCal, SysTime(DateTime(-2010, 1, 31, 0, 0, 0), FracSec.from!"hnsecs"(0)).dayOfGregorianCal);
            _assertPred!"=="(Date(-2010, 2, 1).dayOfGregorianCal, SysTime(DateTime(-2010, 2, 1, 0, 0, 0), FracSec.from!"hnsecs"(500)).dayOfGregorianCal);
            _assertPred!"=="(Date(-2010, 2, 28).dayOfGregorianCal, SysTime(DateTime(-2010, 2, 28, 0, 0, 0), FracSec.from!"hnsecs"(50_000)).dayOfGregorianCal);
            _assertPred!"=="(Date(-2010, 3, 1).dayOfGregorianCal, SysTime(DateTime(-2010, 3, 1, 0, 0, 0), FracSec.from!"hnsecs"(9_999_999)).dayOfGregorianCal);
            _assertPred!"=="(Date(-2010, 3, 31).dayOfGregorianCal, SysTime(DateTime(-2010, 3, 31, 12, 13, 14), FracSec.from!"hnsecs"(0)).dayOfGregorianCal);
            _assertPred!"=="(Date(-2010, 4, 1).dayOfGregorianCal, SysTime(DateTime(-2010, 4, 1, 12, 13, 14), FracSec.from!"hnsecs"(500)).dayOfGregorianCal);
            _assertPred!"=="(Date(-2010, 4, 30).dayOfGregorianCal, SysTime(DateTime(-2010, 4, 30, 12, 13, 14), FracSec.from!"hnsecs"(50_000)).dayOfGregorianCal);
            _assertPred!"=="(Date(-2010, 5, 1).dayOfGregorianCal, SysTime(DateTime(-2010, 5, 1, 12, 13, 14), FracSec.from!"hnsecs"(9_999_999)).dayOfGregorianCal);
            _assertPred!"=="(Date(-2010, 5, 31).dayOfGregorianCal, SysTime(DateTime(-2010, 5, 31, 23, 59, 59), FracSec.from!"hnsecs"(0)).dayOfGregorianCal);
            _assertPred!"=="(Date(-2010, 6, 1).dayOfGregorianCal, SysTime(DateTime(-2010, 6, 1, 23, 59, 59), FracSec.from!"hnsecs"(500)).dayOfGregorianCal);
            _assertPred!"=="(Date(-2010, 6, 30).dayOfGregorianCal, SysTime(DateTime(-2010, 6, 30, 23, 59, 59), FracSec.from!"hnsecs"(50_000)).dayOfGregorianCal);
            _assertPred!"=="(Date(-2010, 7, 1).dayOfGregorianCal, SysTime(DateTime(-2010, 7, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)).dayOfGregorianCal);
            _assertPred!"=="(Date(-2010, 7, 31).dayOfGregorianCal, SysTime(DateTime(-2010, 7, 31, 0, 0, 0), FracSec.from!"hnsecs"(0)).dayOfGregorianCal);
            _assertPred!"=="(Date(-2010, 8, 1).dayOfGregorianCal, SysTime(DateTime(-2010, 8, 1, 0, 0, 0), FracSec.from!"hnsecs"(500)).dayOfGregorianCal);
            _assertPred!"=="(Date(-2010, 8, 31).dayOfGregorianCal, SysTime(DateTime(-2010, 8, 31, 0, 0, 0), FracSec.from!"hnsecs"(50_000)).dayOfGregorianCal);
            _assertPred!"=="(Date(-2010, 9, 1).dayOfGregorianCal, SysTime(DateTime(-2010, 9, 1, 0, 0, 0), FracSec.from!"hnsecs"(9_999_999)).dayOfGregorianCal);
            _assertPred!"=="(Date(-2010, 9, 30).dayOfGregorianCal, SysTime(DateTime(-2010, 9, 30, 12, 0, 0), FracSec.from!"hnsecs"(0)).dayOfGregorianCal);
            _assertPred!"=="(Date(-2010, 10, 1).dayOfGregorianCal, SysTime(DateTime(-2010, 10, 1, 0, 12, 0), FracSec.from!"hnsecs"(500)).dayOfGregorianCal);
            _assertPred!"=="(Date(-2010, 10, 31).dayOfGregorianCal, SysTime(DateTime(-2010, 10, 31, 0, 0, 12), FracSec.from!"hnsecs"(50_000)).dayOfGregorianCal);
            _assertPred!"=="(Date(-2010, 11, 1).dayOfGregorianCal, SysTime(DateTime(-2010, 11, 1, 23, 0, 0), FracSec.from!"hnsecs"(9_999_999)).dayOfGregorianCal);
            _assertPred!"=="(Date(-2010, 11, 30).dayOfGregorianCal, SysTime(DateTime(-2010, 11, 30, 0, 59, 0), FracSec.from!"hnsecs"(0)).dayOfGregorianCal);
            _assertPred!"=="(Date(-2010, 12, 1).dayOfGregorianCal, SysTime(DateTime(-2010, 12, 1, 0, 0, 59), FracSec.from!"hnsecs"(500)).dayOfGregorianCal);
            _assertPred!"=="(Date(-2010, 12, 31).dayOfGregorianCal, SysTime(DateTime(-2010, 12, 31, 0, 59, 59), FracSec.from!"hnsecs"(50_000)).dayOfGregorianCal);

            _assertPred!"=="(Date(-2012, 2, 1).dayOfGregorianCal, SysTime(DateTime(-2012, 2, 1, 23, 0, 59), FracSec.from!"hnsecs"(9_999_999)).dayOfGregorianCal);
            _assertPred!"=="(Date(-2012, 2, 28).dayOfGregorianCal, SysTime(DateTime(-2012, 2, 28, 23, 59, 0), FracSec.from!"hnsecs"(0)).dayOfGregorianCal);
            _assertPred!"=="(Date(-2012, 2, 29).dayOfGregorianCal, SysTime(DateTime(-2012, 2, 29, 7, 7, 7), FracSec.from!"hnsecs"(7)).dayOfGregorianCal);
            _assertPred!"=="(Date(-2012, 3, 1).dayOfGregorianCal, SysTime(DateTime(-2012, 3, 1, 7, 7, 7), FracSec.from!"hnsecs"(7)).dayOfGregorianCal);

            _assertPred!"=="(Date(-3760, 9, 7).dayOfGregorianCal, SysTime(DateTime(-3760, 9, 7, 0, 0, 0), FracSec.from!"hnsecs"(0)).dayOfGregorianCal);
        }
    }


    /++
        The Xth day of the Gregorian Calendar that this $(D SysTime) is on.
        Setting this property does not affect the time portion of $(D SysTime).

        Params:
            days = The day of the Gregorian Calendar to set this $(D SysTime)
                   to.

        Examples:
--------------------
auto st = SysTime(DateTime(0, 0, 0, 12, 0, 0));
st.dayOfGregorianCal = 1;
assert(st == SysTime(DateTime(1, 1, 1, 12, 0, 0)));

st.dayOfGregorianCal = 365;
assert(st == SysTime(DateTime(1, 12, 31, 12, 0, 0)));

st.dayOfGregorianCal = 366;
assert(st == SysTime(DateTime(2, 1, 1, 12, 0, 0)));

st.dayOfGregorianCal = 0;
assert(st == SysTime(DateTime(0, 12, 31, 12, 0, 0)));

st.dayOfGregorianCal = -365;
assert(st == SysTime(DateTime(-0, 1, 1, 12, 0, 0)));

st.dayOfGregorianCal = -366;
assert(st == SysTime(DateTime(-1, 12, 31, 12, 0, 0)));

st.dayOfGregorianCal = 730_120;
assert(st == SysTime(DateTime(2000, 1, 1, 12, 0, 0)));

st.dayOfGregorianCal = 734_137;
assert(st == SysTime(DateTime(2010, 12, 31, 12, 0, 0)));
--------------------
     +/
    @property void dayOfGregorianCal(int days) nothrow
    {
        auto hnsecs = adjTime;
        hnsecs = removeUnitsFromHNSecs!"days"(hnsecs);

        if(hnsecs < 0)
            hnsecs += convert!("hours", "hnsecs")(24);

        if(--days < 0)
        {
            hnsecs -= convert!("hours", "hnsecs")(24);
            ++days;
        }

        immutable newDaysHNSecs = convert!("days", "hnsecs")(days);

        adjTime = newDaysHNSecs + hnsecs;
    }

    unittest
    {
        version(testStdDateTime)
        {
            void testST(SysTime orig, int day, in SysTime expected, size_t line = __LINE__)
            {
                orig.dayOfGregorianCal = day;
                _assertPred!"=="(orig, expected, "", __FILE__, line);
            }

            //Test A.D.
            testST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)), 1, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)));
            testST(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(1)), 1, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(1)));
            testST(SysTime(DateTime(1, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 1, SysTime(DateTime(1, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));

            //Test B.C.
            testST(SysTime(DateTime(0, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)), 0, SysTime(DateTime(0, 12, 31, 0, 0, 0), FracSec.from!"hnsecs"(0)));
            testST(SysTime(DateTime(0, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 0, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            testST(SysTime(DateTime(0, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(1)), 0, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(1)));
            testST(SysTime(DateTime(0, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(0)), 0, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(0)));

            //Test Both.
            testST(SysTime(DateTime(-512, 7, 20, 0, 0, 0), FracSec.from!"hnsecs"(0)), 1, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(0)));
            testST(SysTime(DateTime(-513, 6, 6, 0, 0, 0), FracSec.from!"hnsecs"(1)), 1, SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(1)));
            testST(SysTime(DateTime(-511, 5, 7, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 1, SysTime(DateTime(1, 1, 1, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));

            testST(SysTime(DateTime(1607, 4, 8, 0, 0, 0), FracSec.from!"hnsecs"(0)), 0, SysTime(DateTime(0, 12, 31, 0, 0, 0), FracSec.from!"hnsecs"(0)));
            testST(SysTime(DateTime(1500, 3, 9, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)), 0, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            testST(SysTime(DateTime(999, 2, 10, 23, 59, 59), FracSec.from!"hnsecs"(1)), 0, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(1)));
            testST(SysTime(DateTime(2007, 12, 11, 23, 59, 59), FracSec.from!"hnsecs"(0)), 0, SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(0)));


            auto sysTime = SysTime(DateTime(1, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212));

            void testST2(int day, in SysTime expected, size_t line = __LINE__)
            {
                sysTime.dayOfGregorianCal = day;
                _assertPred!"=="(sysTime, expected, "", __FILE__, line);
            }

            //Test A.D.
            testST2(1, SysTime(DateTime(1, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(2, SysTime(DateTime(1, 1, 2, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(32, SysTime(DateTime(1, 2, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(366, SysTime(DateTime(2, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(731, SysTime(DateTime(3, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(1096, SysTime(DateTime(4, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(1462, SysTime(DateTime(5, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(17_898, SysTime(DateTime(50, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(35_065, SysTime(DateTime(97, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(36_160, SysTime(DateTime(100, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(36_525, SysTime(DateTime(101, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(37_986, SysTime(DateTime(105, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(72_684, SysTime(DateTime(200, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(73_049, SysTime(DateTime(201, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(109_208, SysTime(DateTime(300, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(109_573, SysTime(DateTime(301, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(145_732, SysTime(DateTime(400, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(146_098, SysTime(DateTime(401, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(182_257, SysTime(DateTime(500, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(182_622, SysTime(DateTime(501, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(364_878, SysTime(DateTime(1000, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(365_243, SysTime(DateTime(1001, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(584_023, SysTime(DateTime(1600, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(584_389, SysTime(DateTime(1601, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(693_596, SysTime(DateTime(1900, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(693_961, SysTime(DateTime(1901, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(729_755, SysTime(DateTime(1999, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(730_120, SysTime(DateTime(2000, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(730_486, SysTime(DateTime(2001, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));

            testST2(733_773, SysTime(DateTime(2010, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(733_803, SysTime(DateTime(2010, 1, 31, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(733_804, SysTime(DateTime(2010, 2, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(733_831, SysTime(DateTime(2010, 2, 28, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(733_832, SysTime(DateTime(2010, 3, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(733_862, SysTime(DateTime(2010, 3, 31, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(733_863, SysTime(DateTime(2010, 4, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(733_892, SysTime(DateTime(2010, 4, 30, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(733_893, SysTime(DateTime(2010, 5, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(733_923, SysTime(DateTime(2010, 5, 31, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(733_924, SysTime(DateTime(2010, 6, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(733_953, SysTime(DateTime(2010, 6, 30, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(733_954, SysTime(DateTime(2010, 7, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(733_984, SysTime(DateTime(2010, 7, 31, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(733_985, SysTime(DateTime(2010, 8, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(734_015, SysTime(DateTime(2010, 8, 31, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(734_016, SysTime(DateTime(2010, 9, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(734_045, SysTime(DateTime(2010, 9, 30, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(734_046, SysTime(DateTime(2010, 10, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(734_076, SysTime(DateTime(2010, 10, 31, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(734_077, SysTime(DateTime(2010, 11, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(734_106, SysTime(DateTime(2010, 11, 30, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(734_107, SysTime(DateTime(2010, 12, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(734_137, SysTime(DateTime(2010, 12, 31, 12, 2, 9), FracSec.from!"msecs"(212)));

            testST2(734_534, SysTime(DateTime(2012, 2, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(734_561, SysTime(DateTime(2012, 2, 28, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(734_562, SysTime(DateTime(2012, 2, 29, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(734_563, SysTime(DateTime(2012, 3, 1, 12, 2, 9), FracSec.from!"msecs"(212)));

            testST2(734_534,  SysTime(DateTime(2012, 2, 1, 12, 2, 9), FracSec.from!"msecs"(212)));

            testST2(734_561, SysTime(DateTime(2012, 2, 28, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(734_562, SysTime(DateTime(2012, 2, 29, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(734_563, SysTime(DateTime(2012, 3, 1, 12, 2, 9), FracSec.from!"msecs"(212)));

            //Test B.C.
            testST2(0, SysTime(DateTime(0, 12, 31, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-1, SysTime(DateTime(0, 12, 30, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-30, SysTime(DateTime(0, 12, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-31, SysTime(DateTime(0, 11, 30, 12, 2, 9), FracSec.from!"msecs"(212)));

            testST2(-366, SysTime(DateTime(-1, 12, 31, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-367, SysTime(DateTime(-1, 12, 30, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-730, SysTime(DateTime(-1, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-731, SysTime(DateTime(-2, 12, 31, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-1095, SysTime(DateTime(-2, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-1096, SysTime(DateTime(-3, 12, 31, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-1460, SysTime(DateTime(-3, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-1461, SysTime(DateTime(-4, 12, 31, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-1826, SysTime(DateTime(-4, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-1827, SysTime(DateTime(-5, 12, 31, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-2191, SysTime(DateTime(-5, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-3652, SysTime(DateTime(-9, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));

            testST2(-18_262, SysTime(DateTime(-49, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-18_627, SysTime(DateTime(-50, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-35_794, SysTime(DateTime(-97, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-36_160, SysTime(DateTime(-99, 12, 31, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-36_524, SysTime(DateTime(-99, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-36_889, SysTime(DateTime(-100, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-37_254, SysTime(DateTime(-101, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-38_715, SysTime(DateTime(-105, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-73_413, SysTime(DateTime(-200, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-73_778, SysTime(DateTime(-201, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-109_937, SysTime(DateTime(-300, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-110_302, SysTime(DateTime(-301, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-146_097, SysTime(DateTime(-400, 12, 31, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-146_462, SysTime(DateTime(-400, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-146_827, SysTime(DateTime(-401, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-182_621, SysTime(DateTime(-499, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-182_986, SysTime(DateTime(-500, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-183_351, SysTime(DateTime(-501, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-365_607, SysTime(DateTime(-1000, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-365_972, SysTime(DateTime(-1001, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-584_387, SysTime(DateTime(-1599, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-584_388, SysTime(DateTime(-1600, 12, 31, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-584_753, SysTime(DateTime(-1600, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-585_118, SysTime(DateTime(-1601, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-694_325, SysTime(DateTime(-1900, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-694_690, SysTime(DateTime(-1901, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-730_484, SysTime(DateTime(-1999, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-730_485, SysTime(DateTime(-2000, 12, 31, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-730_850, SysTime(DateTime(-2000, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-731_215, SysTime(DateTime(-2001, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));

            testST2(-734_502, SysTime(DateTime(-2010, 1, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-734_472, SysTime(DateTime(-2010, 1, 31, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-734_471, SysTime(DateTime(-2010, 2, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-734_444, SysTime(DateTime(-2010, 2, 28, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-734_443, SysTime(DateTime(-2010, 3, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-734_413, SysTime(DateTime(-2010, 3, 31, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-734_412, SysTime(DateTime(-2010, 4, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-734_383, SysTime(DateTime(-2010, 4, 30, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-734_382, SysTime(DateTime(-2010, 5, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-734_352, SysTime(DateTime(-2010, 5, 31, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-734_351, SysTime(DateTime(-2010, 6, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-734_322, SysTime(DateTime(-2010, 6, 30, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-734_321, SysTime(DateTime(-2010, 7, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-734_291, SysTime(DateTime(-2010, 7, 31, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-734_290, SysTime(DateTime(-2010, 8, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-734_260, SysTime(DateTime(-2010, 8, 31, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-734_259, SysTime(DateTime(-2010, 9, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-734_230, SysTime(DateTime(-2010, 9, 30, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-734_229, SysTime(DateTime(-2010, 10, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-734_199, SysTime(DateTime(-2010, 10, 31, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-734_198, SysTime(DateTime(-2010, 11, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-734_169, SysTime(DateTime(-2010, 11, 30, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-734_168, SysTime(DateTime(-2010, 12, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-734_138, SysTime(DateTime(-2010, 12, 31, 12, 2, 9), FracSec.from!"msecs"(212)));

            testST2(-735_202, SysTime(DateTime(-2012, 2, 1, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-735_175, SysTime(DateTime(-2012, 2, 28, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-735_174, SysTime(DateTime(-2012, 2, 29, 12, 2, 9), FracSec.from!"msecs"(212)));
            testST2(-735_173, SysTime(DateTime(-2012, 3, 1, 12, 2, 9), FracSec.from!"msecs"(212)));

            const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            static assert(!__traits(compiles, cst.dayOfGregorianCal = 7));
            //static assert(!__traits(compiles, ist.dayOfGregorianCal = 7));

            //Verify Examples.
            auto st = SysTime(DateTime(0, 1, 1, 12, 0, 0));
            st.dayOfGregorianCal = 1;
            assert(st == SysTime(DateTime(1, 1, 1, 12, 0, 0)));

            st.dayOfGregorianCal = 365;
            assert(st == SysTime(DateTime(1, 12, 31, 12, 0, 0)));

            st.dayOfGregorianCal = 366;
            assert(st == SysTime(DateTime(2, 1, 1, 12, 0, 0)));

            st.dayOfGregorianCal = 0;
            assert(st == SysTime(DateTime(0, 12, 31, 12, 0, 0)));

            st.dayOfGregorianCal = -365;
            assert(st == SysTime(DateTime(-0, 1, 1, 12, 0, 0)));

            st.dayOfGregorianCal = -366;
            assert(st == SysTime(DateTime(-1, 12, 31, 12, 0, 0)));

            st.dayOfGregorianCal = 730_120;
            assert(st == SysTime(DateTime(2000, 1, 1, 12, 0, 0)));

            st.dayOfGregorianCal = 734_137;
            assert(st == SysTime(DateTime(2010, 12, 31, 12, 0, 0)));
        }
    }


    /++
        The ISO 8601 week of the year that this $(D SysTime) is in.

        See_Also:
            $(WEB en.wikipedia.org/wiki/ISO_week_date, ISO Week Date).
      +/
    @property ubyte isoWeek() const nothrow
    {
        return (cast(Date)this).isoWeek;
    }

    unittest
    {
        version(testStdDateTime)
        {
            auto st = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            static assert(__traits(compiles, st.isoWeek));
            static assert(__traits(compiles, cst.isoWeek));
            //static assert(__traits(compiles, ist.isoWeek));
        }
    }


    /++
        $(D SysTime) for the last day in the month that this Date is in.
        The time portion of endOfMonth is always 23:59:59.9999999.

        Examples:
--------------------
assert(SysTime(DateTime(1999, 1, 6, 0, 0, 0)).endOfMonth ==
       SysTime(DateTime(1999, 1, 31, 23, 59, 59),
               FracSec.from!"hnsecs"(9_999_999)));

assert(SysTime(DateTime(1999, 2, 7, 19, 30, 0),
               FracSec.from!"msecs"(24)).endOfMonth ==
       SysTime(DateTime(1999, 2, 28, 23, 59, 59),
               FracSec.from!"hnsecs"(9_999_999)));

assert(SysTime(DateTime(2000, 2, 7, 5, 12, 27),
               FracSec.from!"usecs"(5203)).endOfMonth ==
       SysTime(DateTime(2000, 2, 29, 23, 59, 59),
               FracSec.from!"hnsecs"(9_999_999)));

assert(SysTime(DateTime(2000, 6, 4, 12, 22, 9),
               FracSec.from!"hnsecs"(12345)).endOfMonth ==
       SysTime(DateTime(2000, 6, 30, 23, 59, 59),
               FracSec.from!"hnsecs"(9_999_999)));
--------------------
      +/
    @property SysTime endOfMonth() const nothrow
    {
        immutable hnsecs = adjTime;
        immutable days = getUnitsFromHNSecs!"days"(hnsecs);

        auto date = Date(cast(int)days + 1).endOfMonth;
        auto newDays = date.dayOfGregorianCal - 1;
        long theTimeHNSecs;

        if(newDays < 0)
        {
            theTimeHNSecs = -1;
            ++newDays;
        }
        else
            theTimeHNSecs = convert!("days", "hnsecs")(1) - 1;

        immutable newDaysHNSecs = convert!("days", "hnsecs")(newDays);

        auto retval = SysTime(this._stdTime, this._timezone);
        retval.adjTime = newDaysHNSecs + theTimeHNSecs;

        return retval;
    }

    unittest
    {
        version(testStdDateTime)
        {
            //Test A.D.
            _assertPred!"=="(SysTime(Date(1999, 1, 1)).endOfMonth, SysTime(DateTime(1999, 1, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            _assertPred!"=="(SysTime(Date(1999, 2, 1)).endOfMonth, SysTime(DateTime(1999, 2, 28, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            _assertPred!"=="(SysTime(Date(2000, 2, 1)).endOfMonth, SysTime(DateTime(2000, 2, 29, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            _assertPred!"=="(SysTime(Date(1999, 3, 1)).endOfMonth, SysTime(DateTime(1999, 3, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            _assertPred!"=="(SysTime(Date(1999, 4, 1)).endOfMonth, SysTime(DateTime(1999, 4, 30, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            _assertPred!"=="(SysTime(Date(1999, 5, 1)).endOfMonth, SysTime(DateTime(1999, 5, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            _assertPred!"=="(SysTime(Date(1999, 6, 1)).endOfMonth, SysTime(DateTime(1999, 6, 30, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            _assertPred!"=="(SysTime(Date(1999, 7, 1)).endOfMonth, SysTime(DateTime(1999, 7, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            _assertPred!"=="(SysTime(Date(1999, 8, 1)).endOfMonth, SysTime(DateTime(1999, 8, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            _assertPred!"=="(SysTime(Date(1999, 9, 1)).endOfMonth, SysTime(DateTime(1999, 9, 30, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            _assertPred!"=="(SysTime(Date(1999, 10, 1)).endOfMonth, SysTime(DateTime(1999, 10, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            _assertPred!"=="(SysTime(Date(1999, 11, 1)).endOfMonth, SysTime(DateTime(1999, 11, 30, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            _assertPred!"=="(SysTime(Date(1999, 12, 1)).endOfMonth, SysTime(DateTime(1999, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));

            //Test B.C.
            _assertPred!"=="(SysTime(Date(-1999, 1, 1)).endOfMonth, SysTime(DateTime(-1999, 1, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            _assertPred!"=="(SysTime(Date(-1999, 2, 1)).endOfMonth, SysTime(DateTime(-1999, 2, 28, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            _assertPred!"=="(SysTime(Date(-2000, 2, 1)).endOfMonth, SysTime(DateTime(-2000, 2, 29, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            _assertPred!"=="(SysTime(Date(-1999, 3, 1)).endOfMonth, SysTime(DateTime(-1999, 3, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            _assertPred!"=="(SysTime(Date(-1999, 4, 1)).endOfMonth, SysTime(DateTime(-1999, 4, 30, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            _assertPred!"=="(SysTime(Date(-1999, 5, 1)).endOfMonth, SysTime(DateTime(-1999, 5, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            _assertPred!"=="(SysTime(Date(-1999, 6, 1)).endOfMonth, SysTime(DateTime(-1999, 6, 30, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            _assertPred!"=="(SysTime(Date(-1999, 7, 1)).endOfMonth, SysTime(DateTime(-1999, 7, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            _assertPred!"=="(SysTime(Date(-1999, 8, 1)).endOfMonth, SysTime(DateTime(-1999, 8, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            _assertPred!"=="(SysTime(Date(-1999, 9, 1)).endOfMonth, SysTime(DateTime(-1999, 9, 30, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            _assertPred!"=="(SysTime(Date(-1999, 10, 1)).endOfMonth, SysTime(DateTime(-1999, 10, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            _assertPred!"=="(SysTime(Date(-1999, 11, 1)).endOfMonth, SysTime(DateTime(-1999, 11, 30, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            _assertPred!"=="(SysTime(Date(-1999, 12, 1)).endOfMonth, SysTime(DateTime(-1999, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));

            const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            static assert(__traits(compiles, cst.endOfMonth));
            //static assert(__traits(compiles, ist.endOfMonth));

            //Verify Examples.
            assert(SysTime(DateTime(1999, 1, 6, 0, 0, 0)).endOfMonth == SysTime(DateTime(1999, 1, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            assert(SysTime(DateTime(1999, 2, 7, 19, 30, 0), FracSec.from!"msecs"(24)).endOfMonth == SysTime(DateTime(1999, 2, 28, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            assert(SysTime(DateTime(2000, 2, 7, 5, 12, 27), FracSec.from!"usecs"(5203)).endOfMonth == SysTime(DateTime(2000, 2, 29, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
            assert(SysTime(DateTime(2000, 6, 4, 12, 22, 9), FracSec.from!"hnsecs"(12345)).endOfMonth == SysTime(DateTime(2000, 6, 30, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999)));
        }
    }


    /++
        The last day in the month that this $(D SysTime) is in.

        Examples:
--------------------
assert(SysTime(DateTime(1999, 1, 6, 0, 0, 0)).daysInMonth == 31);
assert(SysTime(DateTime(1999, 2, 7, 19, 30, 0)).daysInMonth == 28);
assert(SysTime(DateTime(2000, 2, 7, 5, 12, 27)).daysInMonth == 29);
assert(SysTime(DateTime(2000, 6, 4, 12, 22, 9)).daysInMonth == 30);
--------------------
      +/
    @property ubyte daysInMonth() const nothrow
    {
        return Date(dayOfGregorianCal).daysInMonth;
    }

    /++
        $(RED Deprecated. It will be removed in September 2012.
              Please use daysInMonth instead.)
      +/
    deprecated @property ubyte endOfMonthDay() const nothrow
    {
        return Date(dayOfGregorianCal).daysInMonth;
    }

    unittest
    {
        version(testStdDateTime)
        {
            //Test A.D.
            _assertPred!"=="(SysTime(DateTime(1999, 1, 1, 12, 1, 13)).daysInMonth, 31);
            _assertPred!"=="(SysTime(DateTime(1999, 2, 1, 17, 13, 12)).daysInMonth, 28);
            _assertPred!"=="(SysTime(DateTime(2000, 2, 1, 13, 2, 12)).daysInMonth, 29);
            _assertPred!"=="(SysTime(DateTime(1999, 3, 1, 12, 13, 12)).daysInMonth, 31);
            _assertPred!"=="(SysTime(DateTime(1999, 4, 1, 12, 6, 13)).daysInMonth, 30);
            _assertPred!"=="(SysTime(DateTime(1999, 5, 1, 15, 13, 12)).daysInMonth, 31);
            _assertPred!"=="(SysTime(DateTime(1999, 6, 1, 13, 7, 12)).daysInMonth, 30);
            _assertPred!"=="(SysTime(DateTime(1999, 7, 1, 12, 13, 17)).daysInMonth, 31);
            _assertPred!"=="(SysTime(DateTime(1999, 8, 1, 12, 3, 13)).daysInMonth, 31);
            _assertPred!"=="(SysTime(DateTime(1999, 9, 1, 12, 13, 12)).daysInMonth, 30);
            _assertPred!"=="(SysTime(DateTime(1999, 10, 1, 13, 19, 12)).daysInMonth, 31);
            _assertPred!"=="(SysTime(DateTime(1999, 11, 1, 12, 13, 17)).daysInMonth, 30);
            _assertPred!"=="(SysTime(DateTime(1999, 12, 1, 12, 52, 13)).daysInMonth, 31);

            //Test B.C.
            _assertPred!"=="(SysTime(DateTime(-1999, 1, 1, 12, 1, 13)).daysInMonth, 31);
            _assertPred!"=="(SysTime(DateTime(-1999, 2, 1, 7, 13, 12)).daysInMonth, 28);
            _assertPred!"=="(SysTime(DateTime(-2000, 2, 1, 13, 2, 12)).daysInMonth, 29);
            _assertPred!"=="(SysTime(DateTime(-1999, 3, 1, 12, 13, 12)).daysInMonth, 31);
            _assertPred!"=="(SysTime(DateTime(-1999, 4, 1, 12, 6, 13)).daysInMonth, 30);
            _assertPred!"=="(SysTime(DateTime(-1999, 5, 1, 5, 13, 12)).daysInMonth, 31);
            _assertPred!"=="(SysTime(DateTime(-1999, 6, 1, 13, 7, 12)).daysInMonth, 30);
            _assertPred!"=="(SysTime(DateTime(-1999, 7, 1, 12, 13, 17)).daysInMonth, 31);
            _assertPred!"=="(SysTime(DateTime(-1999, 8, 1, 12, 3, 13)).daysInMonth, 31);
            _assertPred!"=="(SysTime(DateTime(-1999, 9, 1, 12, 13, 12)).daysInMonth, 30);
            _assertPred!"=="(SysTime(DateTime(-1999, 10, 1, 13, 19, 12)).daysInMonth, 31);
            _assertPred!"=="(SysTime(DateTime(-1999, 11, 1, 12, 13, 17)).daysInMonth, 30);
            _assertPred!"=="(SysTime(DateTime(-1999, 12, 1, 12, 52, 13)).daysInMonth, 31);

            const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            static assert(__traits(compiles, cst.daysInMonth));
            //static assert(__traits(compiles, ist.daysInMonth));

            //Verify Examples.
            assert(SysTime(DateTime(1999, 1, 6, 0, 0, 0)).daysInMonth == 31);
            assert(SysTime(DateTime(1999, 2, 7, 19, 30, 0)).daysInMonth == 28);
            assert(SysTime(DateTime(2000, 2, 7, 5, 12, 27)).daysInMonth == 29);
            assert(SysTime(DateTime(2000, 6, 4, 12, 22, 9)).daysInMonth == 30);
        }
    }


    /++
        Whether the current year is a date in A.D.

        Examples:
--------------------
assert(SysTime(DateTime(1, 1, 1, 12, 7, 0)).isAD);
assert(SysTime(DateTime(2010, 12, 31, 0, 0, 0)).isAD);
assert(!SysTime(DateTime(0, 12, 31, 23, 59, 59)).isAD);
assert(!SysTime(DateTime(-2010, 1, 1, 2, 2, 2)).isAD);
--------------------
      +/
    @property bool isAD() const nothrow
    {
        return adjTime >= 0;
    }

    unittest
    {
        version(testStdDateTime)
        {
            assert(SysTime(DateTime(2010, 7, 4, 12, 0, 9)).isAD);
            assert(SysTime(DateTime(1, 1, 1, 0, 0, 0)).isAD);
            assert(!SysTime(DateTime(0, 12, 31, 23, 59, 59)).isAD);
            assert(!SysTime(DateTime(0, 1, 1, 23, 59, 59)).isAD);
            assert(!SysTime(DateTime(-1, 1, 1, 23 ,59 ,59)).isAD);
            assert(!SysTime(DateTime(-2010, 7, 4, 12, 2, 2)).isAD);

            const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            static assert(__traits(compiles, cst.isAD));
            //static assert(__traits(compiles, ist.isAD));

            //Verify Examples.
            assert(SysTime(DateTime(1, 1, 1, 12, 7, 0)).isAD);
            assert(SysTime(DateTime(2010, 12, 31, 0, 0, 0)).isAD);
            assert(!SysTime(DateTime(0, 12, 31, 23, 59, 59)).isAD);
            assert(!SysTime(DateTime(-2010, 1, 1, 2, 2, 2)).isAD);
        }
    }


    /++
        The julian day for this $(D SysTime) at the given time. For example,
        prior to noon, 1996-03-31 would be the julian day number 2_450_173, so
        this function returns 2_450_173, while from noon onward, the julian
        day number would be 2_450_174, so this function returns 2_450_174.
      +/
    @property long julianDay() const nothrow
    {
        immutable jd = dayOfGregorianCal + 1_721_425;

        return hour < 12 ? jd - 1 : jd;
    }

    unittest
    {
        version(testStdDateTime)
        {
            _assertPred!"=="(SysTime(DateTime(-4713, 11, 24, 0, 0, 0)).julianDay, -1);
            _assertPred!"=="(SysTime(DateTime(-4713, 11, 24, 12, 0, 0)).julianDay, 0);

            _assertPred!"=="(SysTime(DateTime(0, 12, 31, 0, 0, 0)).julianDay, 1_721_424);
            _assertPred!"=="(SysTime(DateTime(0, 12, 31, 12, 0, 0)).julianDay, 1_721_425);

            _assertPred!"=="(SysTime(DateTime(1, 1, 1, 0, 0, 0)).julianDay, 1_721_425);
            _assertPred!"=="(SysTime(DateTime(1, 1, 1, 12, 0, 0)).julianDay, 1_721_426);

            _assertPred!"=="(SysTime(DateTime(1582, 10, 15, 0, 0, 0)).julianDay, 2_299_160);
            _assertPred!"=="(SysTime(DateTime(1582, 10, 15, 12, 0, 0)).julianDay, 2_299_161);

            _assertPred!"=="(SysTime(DateTime(1858, 11, 17, 0, 0, 0)).julianDay, 2_400_000);
            _assertPred!"=="(SysTime(DateTime(1858, 11, 17, 12, 0, 0)).julianDay, 2_400_001);

            _assertPred!"=="(SysTime(DateTime(1982, 1, 4, 0, 0, 0)).julianDay, 2_444_973);
            _assertPred!"=="(SysTime(DateTime(1982, 1, 4, 12, 0, 0)).julianDay, 2_444_974);

            _assertPred!"=="(SysTime(DateTime(1996, 3, 31, 0, 0, 0)).julianDay, 2_450_173);
            _assertPred!"=="(SysTime(DateTime(1996, 3, 31, 12, 0, 0)).julianDay, 2_450_174);

            _assertPred!"=="(SysTime(DateTime(2010, 8, 24, 0, 0, 0)).julianDay, 2_455_432);
            _assertPred!"=="(SysTime(DateTime(2010, 8, 24, 12, 0, 0)).julianDay, 2_455_433);

            const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            static assert(__traits(compiles, cst.julianDay));
            //static assert(__traits(compiles, ist.julianDay));
        }
    }


    /++
        The modified julian day for any time on this date (since, the modified
        julian day changes at midnight).
      +/
    @property long modJulianDay() const nothrow
    {
        return (dayOfGregorianCal + 1_721_425) - 2_400_001;
    }

    unittest
    {
        version(testStdDateTime)
        {
            _assertPred!"=="(SysTime(DateTime(1858, 11, 17, 0, 0, 0)).modJulianDay, 0);
            _assertPred!"=="(SysTime(DateTime(1858, 11, 17, 12, 0, 0)).modJulianDay, 0);

            _assertPred!"=="(SysTime(DateTime(2010, 8, 24, 0, 0, 0)).modJulianDay, 55_432);
            _assertPred!"=="(SysTime(DateTime(2010, 8, 24, 12, 0, 0)).modJulianDay, 55_432);

            const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            static assert(__traits(compiles, cst.modJulianDay));
            //static assert(__traits(compiles, ist.modJulianDay));
        }
    }


    /++
        Returns a $(D Date) equivalent to this $(D SysTime).
      +/
    Date opCast(T)() const nothrow
        if(is(Unqual!T == Date))
    {
        return Date(dayOfGregorianCal);
    }

    unittest
    {
        version(testStdDateTime)
        {
            _assertPred!"=="(cast(Date)SysTime(Date(1999, 7, 6)), Date(1999, 7, 6));
            _assertPred!"=="(cast(Date)SysTime(Date(2000, 12, 31)), Date(2000, 12, 31));
            _assertPred!"=="(cast(Date)SysTime(Date(2001, 1, 1)), Date(2001, 1, 1));

            _assertPred!"=="(cast(Date)SysTime(DateTime(1999, 7, 6, 12, 10, 9)), Date(1999, 7, 6));
            _assertPred!"=="(cast(Date)SysTime(DateTime(2000, 12, 31, 13, 11, 10)), Date(2000, 12, 31));
            _assertPred!"=="(cast(Date)SysTime(DateTime(2001, 1, 1, 14, 12, 11)), Date(2001, 1, 1));

            _assertPred!"=="(cast(Date)SysTime(Date(-1999, 7, 6)), Date(-1999, 7, 6));
            _assertPred!"=="(cast(Date)SysTime(Date(-2000, 12, 31)), Date(-2000, 12, 31));
            _assertPred!"=="(cast(Date)SysTime(Date(-2001, 1, 1)), Date(-2001, 1, 1));

            _assertPred!"=="(cast(Date)SysTime(DateTime(-1999, 7, 6, 12, 10, 9)), Date(-1999, 7, 6));
            _assertPred!"=="(cast(Date)SysTime(DateTime(-2000, 12, 31, 13, 11, 10)), Date(-2000, 12, 31));
            _assertPred!"=="(cast(Date)SysTime(DateTime(-2001, 1, 1, 14, 12, 11)), Date(-2001, 1, 1));

            const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            static assert(__traits(compiles, cast(Date)cst));
            //static assert(__traits(compiles, cast(Date)ist));
        }
    }


    /++
        Returns a $(D DateTime) equivalent to this $(D SysTime).
      +/
    DateTime opCast(T)() const nothrow
        if(is(Unqual!T == DateTime))
    {
        try
        {
            auto hnsecs = adjTime;
            auto days = splitUnitsFromHNSecs!"days"(hnsecs) + 1;

            if(hnsecs < 0)
            {
                hnsecs += convert!("hours", "hnsecs")(24);
                --days;
            }

            immutable hour = splitUnitsFromHNSecs!"hours"(hnsecs);
            immutable minute = splitUnitsFromHNSecs!"minutes"(hnsecs);
            immutable second = getUnitsFromHNSecs!"seconds"(hnsecs);

            return DateTime(Date(cast(int)days), TimeOfDay(cast(int)hour, cast(int)minute, cast(int)second));
        }
        catch(Exception e)
            assert(0, "Either DateTime's constructor or TimeOfDay's constructor threw.");
    }

    unittest
    {
        version(testStdDateTime)
        {
            _assertPred!"=="(cast(DateTime)SysTime(DateTime(1, 1, 6, 7, 12, 22)), DateTime(1, 1, 6, 7, 12, 22));
            _assertPred!"=="(cast(DateTime)SysTime(DateTime(1, 1, 6, 7, 12, 22), FracSec.from!"msecs"(22)), DateTime(1, 1, 6, 7, 12, 22));
            _assertPred!"=="(cast(DateTime)SysTime(Date(1999, 7, 6)), DateTime(1999, 7, 6, 0, 0, 0));
            _assertPred!"=="(cast(DateTime)SysTime(Date(2000, 12, 31)), DateTime(2000, 12, 31, 0, 0, 0));
            _assertPred!"=="(cast(DateTime)SysTime(Date(2001, 1, 1)), DateTime(2001, 1, 1, 0, 0, 0));

            _assertPred!"=="(cast(DateTime)SysTime(DateTime(1999, 7, 6, 12, 10, 9)), DateTime(1999, 7, 6, 12, 10, 9));
            _assertPred!"=="(cast(DateTime)SysTime(DateTime(2000, 12, 31, 13, 11, 10)), DateTime(2000, 12, 31, 13, 11, 10));
            _assertPred!"=="(cast(DateTime)SysTime(DateTime(2001, 1, 1, 14, 12, 11)), DateTime(2001, 1, 1, 14, 12, 11));

            _assertPred!"=="(cast(DateTime)SysTime(DateTime(-1, 1, 6, 7, 12, 22)), DateTime(-1, 1, 6, 7, 12, 22));
            _assertPred!"=="(cast(DateTime)SysTime(DateTime(-1, 1, 6, 7, 12, 22), FracSec.from!"msecs"(22)), DateTime(-1, 1, 6, 7, 12, 22));
            _assertPred!"=="(cast(DateTime)SysTime(Date(-1999, 7, 6)), DateTime(-1999, 7, 6, 0, 0, 0));
            _assertPred!"=="(cast(DateTime)SysTime(Date(-2000, 12, 31)), DateTime(-2000, 12, 31, 0, 0, 0));
            _assertPred!"=="(cast(DateTime)SysTime(Date(-2001, 1, 1)), DateTime(-2001, 1, 1, 0, 0, 0));

            _assertPred!"=="(cast(DateTime)SysTime(DateTime(-1999, 7, 6, 12, 10, 9)), DateTime(-1999, 7, 6, 12, 10, 9));
            _assertPred!"=="(cast(DateTime)SysTime(DateTime(-2000, 12, 31, 13, 11, 10)), DateTime(-2000, 12, 31, 13, 11, 10));
            _assertPred!"=="(cast(DateTime)SysTime(DateTime(-2001, 1, 1, 14, 12, 11)), DateTime(-2001, 1, 1, 14, 12, 11));

            _assertPred!"=="(cast(DateTime)SysTime(DateTime(2011, 1, 13, 8, 17, 2), FracSec.from!"msecs"(296), LocalTime()),
                            DateTime(2011, 1, 13, 8, 17, 2));

            const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            static assert(__traits(compiles, cast(DateTime)cst));
            //static assert(__traits(compiles, cast(DateTime)ist));
        }
    }


    /++
        Returns a $(D TimeOfDay) equivalent to this $(D SysTime).
      +/
    TimeOfDay opCast(T)() const nothrow
        if(is(Unqual!T == TimeOfDay))
    {
        try
        {
            auto hnsecs = adjTime;
            hnsecs = removeUnitsFromHNSecs!"days"(hnsecs);

            if(hnsecs < 0)
                hnsecs += convert!("hours", "hnsecs")(24);

            immutable hour = splitUnitsFromHNSecs!"hours"(hnsecs);
            immutable minute = splitUnitsFromHNSecs!"minutes"(hnsecs);
            immutable second = getUnitsFromHNSecs!"seconds"(hnsecs);

            return TimeOfDay(cast(int)hour, cast(int)minute, cast(int)second);
        }
        catch(Exception e)
            assert(0, "TimeOfDay's constructor threw.");
    }

    unittest
    {
        version(testStdDateTime)
        {
            _assertPred!"=="(cast(TimeOfDay)SysTime(Date(1999, 7, 6)), TimeOfDay(0, 0, 0));
            _assertPred!"=="(cast(TimeOfDay)SysTime(Date(2000, 12, 31)), TimeOfDay(0, 0, 0));
            _assertPred!"=="(cast(TimeOfDay)SysTime(Date(2001, 1, 1)), TimeOfDay(0, 0, 0));

            _assertPred!"=="(cast(TimeOfDay)SysTime(DateTime(1999, 7, 6, 12, 10, 9)), TimeOfDay(12, 10, 9));
            _assertPred!"=="(cast(TimeOfDay)SysTime(DateTime(2000, 12, 31, 13, 11, 10)), TimeOfDay(13, 11, 10));
            _assertPred!"=="(cast(TimeOfDay)SysTime(DateTime(2001, 1, 1, 14, 12, 11)), TimeOfDay(14, 12, 11));

            _assertPred!"=="(cast(TimeOfDay)SysTime(Date(-1999, 7, 6)), TimeOfDay(0, 0, 0));
            _assertPred!"=="(cast(TimeOfDay)SysTime(Date(-2000, 12, 31)), TimeOfDay(0, 0, 0));
            _assertPred!"=="(cast(TimeOfDay)SysTime(Date(-2001, 1, 1)), TimeOfDay(0, 0, 0));

            _assertPred!"=="(cast(TimeOfDay)SysTime(DateTime(-1999, 7, 6, 12, 10, 9)), TimeOfDay(12, 10, 9));
            _assertPred!"=="(cast(TimeOfDay)SysTime(DateTime(-2000, 12, 31, 13, 11, 10)), TimeOfDay(13, 11, 10));
            _assertPred!"=="(cast(TimeOfDay)SysTime(DateTime(-2001, 1, 1, 14, 12, 11)), TimeOfDay(14, 12, 11));

            const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            static assert(__traits(compiles, cast(TimeOfDay)cst));
            //static assert(__traits(compiles, cast(TimeOfDay)ist));
        }
    }


    //Temporary hack until bug http://d.puremagic.com/issues/show_bug.cgi?id=4867 is fixed.
    //This allows assignment from const(SysTime) to SysTime.
    //It may be a good idea to keep it though, since casting from a type to itself
    //should be allowed, and it doesn't work without this opCast() since opCast()
    //has already been defined for other types.
    SysTime opCast(T)() const pure nothrow
        if(is(Unqual!T == SysTime))
    {
        return SysTime(_stdTime, _timezone);
    }


    /++
        Converts this $(D SysTime) to a string with the format
        YYYYMMDDTHHMMSS.FFFFFFFTZ (where F is fractional seconds and TZ is time
        zone).

        Note that the number of digits in the fractional seconds varies with the
        number of fractional seconds. It's a maximum of 7 (which would be
        hnsecs), but only has as many as are necessary to hold the correct value
        (so no trailing zeroes), and if there are no fractional seconds, then
        there is no decimal point.

        If this $(D SysTime)'s time zone is $(D LocalTime), then TZ is empty.
        If its time zone is $(D UTC), then it is "Z". Otherwise, it is the
        offset from UTC (e.g. +1:00 or -7:00). Note that the offset from UTC
        is $(I not) enough to uniquely identify the time zone.

        Time zone offsets will be in the form +HH:MM or -HH:MM.

        Examples:
--------------------
assert(SysTime(DateTime(2010, 7, 4, 7, 6, 12)).toISOString() ==
       "20100704T070612");

assert(SysTime(DateTime(1998, 12, 25, 2, 15, 0),
               FracSec.from!"msecs"(24)).toISOString() ==
       "19981225T021500.024");

assert(SysTime(DateTime(0, 1, 5, 23, 9, 59)).toISOString() ==
       "00000105T230959");

assert(SysTime(DateTime(-4, 1, 5, 0, 0, 2),
               FracSec.from!"hnsecs"(520_920)).toISOString() ==
       "-00040105T000002.052092");
--------------------
      +/
    string toISOString() const nothrow
    {
        try
        {
            immutable adjustedTime = adjTime;
            long hnsecs = adjustedTime;

            auto days = splitUnitsFromHNSecs!"days"(hnsecs) + 1;

            if(hnsecs < 0)
            {
                hnsecs += convert!("hours", "hnsecs")(24);
                --days;
            }

            auto hour = splitUnitsFromHNSecs!"hours"(hnsecs);
            auto minute = splitUnitsFromHNSecs!"minutes"(hnsecs);
            auto second = splitUnitsFromHNSecs!"seconds"(hnsecs);

            auto dateTime = DateTime(Date(cast(int)days), TimeOfDay(cast(int)hour, cast(int)minute, cast(int)second));
            auto fracSecStr = fracSecToISOString(cast(int)hnsecs);

            if(_timezone is LocalTime())
                return dateTime.toISOString() ~ fracSecToISOString(cast(int)hnsecs);

            if(_timezone is UTC())
                return dateTime.toISOString() ~ fracSecToISOString(cast(int)hnsecs) ~ "Z";

            immutable utcOffset = cast(int)convert!("hnsecs", "minutes")(adjustedTime - stdTime);

            return dateTime.toISOString() ~ fracSecToISOString(cast(int)hnsecs) ~ SimpleTimeZone.toISOString(utcOffset);
        }
        catch(Exception e)
            assert(0, "format() threw.");
    }

    unittest
    {
        version(testStdDateTime)
        {
            //Test A.D.
            _assertPred!"=="(SysTime(DateTime.init, UTC()).toISOString(), "00010101T000000Z");
            _assertPred!"=="(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(1), UTC()).toISOString(), "00010101T000000.0000001Z");

            _assertPred!"=="(SysTime(DateTime(9, 12, 4, 0, 0, 0)).toISOString(), "00091204T000000");
            _assertPred!"=="(SysTime(DateTime(99, 12, 4, 5, 6, 12)).toISOString(), "00991204T050612");
            _assertPred!"=="(SysTime(DateTime(999, 12, 4, 13, 44, 59)).toISOString(), "09991204T134459");
            _assertPred!"=="(SysTime(DateTime(9999, 7, 4, 23, 59, 59)).toISOString(), "99990704T235959");
            _assertPred!"=="(SysTime(DateTime(10000, 10, 20, 1, 1, 1)).toISOString(), "+100001020T010101");

            _assertPred!"=="(SysTime(DateTime(9, 12, 4, 0, 0, 0), FracSec.from!"msecs"(42)).toISOString(), "00091204T000000.042");
            _assertPred!"=="(SysTime(DateTime(99, 12, 4, 5, 6, 12), FracSec.from!"msecs"(100)).toISOString(), "00991204T050612.1");
            _assertPred!"=="(SysTime(DateTime(999, 12, 4, 13, 44, 59), FracSec.from!"usecs"(45020)).toISOString(), "09991204T134459.04502");
            _assertPred!"=="(SysTime(DateTime(9999, 7, 4, 23, 59, 59), FracSec.from!"hnsecs"(12)).toISOString(), "99990704T235959.0000012");
            _assertPred!"=="(SysTime(DateTime(10000, 10, 20, 1, 1, 1), FracSec.from!"hnsecs"(507890)).toISOString(), "+100001020T010101.050789");

            _assertPred!"=="(SysTime(DateTime(2012, 12, 21, 12, 12, 12),
                                    new SimpleTimeZone(-360)).toISOString(),
                            "20121221T121212-06:00");

            _assertPred!"=="(SysTime(DateTime(2012, 12, 21, 12, 12, 12),
                                    new SimpleTimeZone(420)).toISOString(),
                            "20121221T121212+07:00");

            //Test B.C.
            _assertPred!"=="(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999), UTC()).toISOString(), "00001231T235959.9999999Z");
            _assertPred!"=="(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(1), UTC()).toISOString(), "00001231T235959.0000001Z");
            _assertPred!"=="(SysTime(DateTime(0, 12, 31, 23, 59, 59), UTC()).toISOString(), "00001231T235959Z");

            _assertPred!"=="(SysTime(DateTime(0, 12, 4, 0, 12, 4)).toISOString(), "00001204T001204");
            _assertPred!"=="(SysTime(DateTime(-9, 12, 4, 0, 0, 0)).toISOString(), "-00091204T000000");
            _assertPred!"=="(SysTime(DateTime(-99, 12, 4, 5, 6, 12)).toISOString(), "-00991204T050612");
            _assertPred!"=="(SysTime(DateTime(-999, 12, 4, 13, 44, 59)).toISOString(), "-09991204T134459");
            _assertPred!"=="(SysTime(DateTime(-9999, 7, 4, 23, 59, 59)).toISOString(), "-99990704T235959");
            _assertPred!"=="(SysTime(DateTime(-10000, 10, 20, 1, 1, 1)).toISOString(), "-100001020T010101");

            _assertPred!"=="(SysTime(DateTime(0, 12, 4, 0, 0, 0), FracSec.from!"msecs"(7)).toISOString(), "00001204T000000.007");
            _assertPred!"=="(SysTime(DateTime(-9, 12, 4, 0, 0, 0), FracSec.from!"msecs"(42)).toISOString(), "-00091204T000000.042");
            _assertPred!"=="(SysTime(DateTime(-99, 12, 4, 5, 6, 12), FracSec.from!"msecs"(100)).toISOString(), "-00991204T050612.1");
            _assertPred!"=="(SysTime(DateTime(-999, 12, 4, 13, 44, 59), FracSec.from!"usecs"(45020)).toISOString(), "-09991204T134459.04502");
            _assertPred!"=="(SysTime(DateTime(-9999, 7, 4, 23, 59, 59), FracSec.from!"hnsecs"(12)).toISOString(), "-99990704T235959.0000012");
            _assertPred!"=="(SysTime(DateTime(-10000, 10, 20, 1, 1, 1), FracSec.from!"hnsecs"(507890)).toISOString(), "-100001020T010101.050789");

            const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            static assert(__traits(compiles, cast(TimeOfDay)cst));
            //static assert(__traits(compiles, cast(TimeOfDay)ist));

            //Verify Examples.
            assert(SysTime(DateTime(2010, 7, 4, 7, 6, 12)).toISOString() ==
                   "20100704T070612");

            assert(SysTime(DateTime(1998, 12, 25, 2, 15, 0),
                           FracSec.from!"msecs"(24)).toISOString() ==
                   "19981225T021500.024");

            assert(SysTime(DateTime(0, 1, 5, 23, 9, 59)).toISOString() ==
                   "00000105T230959");

            assert(SysTime(DateTime(-4, 1, 5, 0, 0, 2),
                           FracSec.from!"hnsecs"(520_920)).toISOString() ==
                   "-00040105T000002.052092");
        }
    }



    /++
        Converts this $(D SysTime) to a string with the format
        YYYY-MM-DDTHH:MM:SS.FFFFFFFTZ (where F is fractional seconds and TZ
        is the time zone).

        Note that the number of digits in the fractional seconds varies with the
        number of fractional seconds. It's a maximum of 7 (which would be
        hnsecs), but only has as many as are necessary to hold the correct value
        (so no trailing zeroes), and if there are no fractional seconds, then
        there is no decimal point.

        If this $(D SysTime)'s time zone is $(D LocalTime), then TZ is empty. If
        its time zone is $(D UTC), then it is "Z". Otherwise, it is the offset
        from UTC (e.g. +1:00 or -7:00). Note that the offset from UTC is
        $(I not) enough to uniquely identify the time zone.

        Time zone offsets will be in the form +HH:MM or -HH:MM.

        Examples:
--------------------
assert(SysTime(DateTime(2010, 7, 4, 7, 6, 12)).toISOExtString() ==
       "2010-07-04T07:06:12");

assert(SysTime(DateTime(1998, 12, 25, 2, 15, 0),
               FracSec.from!"msecs"(24)).toISOExtString() ==
       "1998-12-25T02:15:00.024");

assert(SysTime(DateTime(0, 1, 5, 23, 9, 59)).toISOExtString() ==
       "0000-01-05T23:09:59");

assert(SysTime(DateTime(-4, 1, 5, 0, 0, 2),
               FracSec.from!"hnsecs"(520_920)).toISOExtString() ==
       "-0004-01-05T00:00:02.052092");
--------------------
      +/
    string toISOExtString() const nothrow
    {
        try
        {
            immutable adjustedTime = adjTime;
            long hnsecs = adjustedTime;

            auto days = splitUnitsFromHNSecs!"days"(hnsecs) + 1;

            if(hnsecs < 0)
            {
                hnsecs += convert!("hours", "hnsecs")(24);
                --days;
            }

            auto hour = splitUnitsFromHNSecs!"hours"(hnsecs);
            auto minute = splitUnitsFromHNSecs!"minutes"(hnsecs);
            auto second = splitUnitsFromHNSecs!"seconds"(hnsecs);

            auto dateTime = DateTime(Date(cast(int)days), TimeOfDay(cast(int)hour, cast(int)minute, cast(int)second));
            auto fracSecStr = fracSecToISOString(cast(int)hnsecs);

            if(_timezone is LocalTime())
                return dateTime.toISOExtString() ~ fracSecToISOString(cast(int)hnsecs);

            if(_timezone is UTC())
                return dateTime.toISOExtString() ~ fracSecToISOString(cast(int)hnsecs) ~ "Z";

            immutable utcOffset = cast(int)convert!("hnsecs", "minutes")(adjustedTime - stdTime);

            return dateTime.toISOExtString() ~ fracSecToISOString(cast(int)hnsecs) ~ SimpleTimeZone.toISOString(utcOffset);
        }
        catch(Exception e)
            assert(0, "format() threw.");
    }

    unittest
    {
        version(testStdDateTime)
        {
            //Test A.D.
            _assertPred!"=="(SysTime(DateTime.init, UTC()).toISOExtString(), "0001-01-01T00:00:00Z");
            _assertPred!"=="(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(1), UTC()).toISOExtString(), "0001-01-01T00:00:00.0000001Z");

            _assertPred!"=="(SysTime(DateTime(9, 12, 4, 0, 0, 0)).toISOExtString(), "0009-12-04T00:00:00");
            _assertPred!"=="(SysTime(DateTime(99, 12, 4, 5, 6, 12)).toISOExtString(), "0099-12-04T05:06:12");
            _assertPred!"=="(SysTime(DateTime(999, 12, 4, 13, 44, 59)).toISOExtString(), "0999-12-04T13:44:59");
            _assertPred!"=="(SysTime(DateTime(9999, 7, 4, 23, 59, 59)).toISOExtString(), "9999-07-04T23:59:59");
            _assertPred!"=="(SysTime(DateTime(10000, 10, 20, 1, 1, 1)).toISOExtString(), "+10000-10-20T01:01:01");

            _assertPred!"=="(SysTime(DateTime(9, 12, 4, 0, 0, 0), FracSec.from!"msecs"(42)).toISOExtString(), "0009-12-04T00:00:00.042");
            _assertPred!"=="(SysTime(DateTime(99, 12, 4, 5, 6, 12), FracSec.from!"msecs"(100)).toISOExtString(), "0099-12-04T05:06:12.1");
            _assertPred!"=="(SysTime(DateTime(999, 12, 4, 13, 44, 59), FracSec.from!"usecs"(45020)).toISOExtString(), "0999-12-04T13:44:59.04502");
            _assertPred!"=="(SysTime(DateTime(9999, 7, 4, 23, 59, 59), FracSec.from!"hnsecs"(12)).toISOExtString(), "9999-07-04T23:59:59.0000012");
            _assertPred!"=="(SysTime(DateTime(10000, 10, 20, 1, 1, 1), FracSec.from!"hnsecs"(507890)).toISOExtString(), "+10000-10-20T01:01:01.050789");

            _assertPred!"=="(SysTime(DateTime(2012, 12, 21, 12, 12, 12),
                                    new SimpleTimeZone(-360)).toISOExtString(),
                            "2012-12-21T12:12:12-06:00");

            _assertPred!"=="(SysTime(DateTime(2012, 12, 21, 12, 12, 12),
                                    new SimpleTimeZone(420)).toISOExtString(),
                            "2012-12-21T12:12:12+07:00");

            //Test B.C.
            _assertPred!"=="(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999), UTC()).toISOExtString(), "0000-12-31T23:59:59.9999999Z");
            _assertPred!"=="(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(1), UTC()).toISOExtString(), "0000-12-31T23:59:59.0000001Z");
            _assertPred!"=="(SysTime(DateTime(0, 12, 31, 23, 59, 59), UTC()).toISOExtString(), "0000-12-31T23:59:59Z");

            _assertPred!"=="(SysTime(DateTime(0, 12, 4, 0, 12, 4)).toISOExtString(), "0000-12-04T00:12:04");
            _assertPred!"=="(SysTime(DateTime(-9, 12, 4, 0, 0, 0)).toISOExtString(), "-0009-12-04T00:00:00");
            _assertPred!"=="(SysTime(DateTime(-99, 12, 4, 5, 6, 12)).toISOExtString(), "-0099-12-04T05:06:12");
            _assertPred!"=="(SysTime(DateTime(-999, 12, 4, 13, 44, 59)).toISOExtString(), "-0999-12-04T13:44:59");
            _assertPred!"=="(SysTime(DateTime(-9999, 7, 4, 23, 59, 59)).toISOExtString(), "-9999-07-04T23:59:59");
            _assertPred!"=="(SysTime(DateTime(-10000, 10, 20, 1, 1, 1)).toISOExtString(), "-10000-10-20T01:01:01");

            _assertPred!"=="(SysTime(DateTime(0, 12, 4, 0, 0, 0), FracSec.from!"msecs"(7)).toISOExtString(), "0000-12-04T00:00:00.007");
            _assertPred!"=="(SysTime(DateTime(-9, 12, 4, 0, 0, 0), FracSec.from!"msecs"(42)).toISOExtString(), "-0009-12-04T00:00:00.042");
            _assertPred!"=="(SysTime(DateTime(-99, 12, 4, 5, 6, 12), FracSec.from!"msecs"(100)).toISOExtString(), "-0099-12-04T05:06:12.1");
            _assertPred!"=="(SysTime(DateTime(-999, 12, 4, 13, 44, 59), FracSec.from!"usecs"(45020)).toISOExtString(), "-0999-12-04T13:44:59.04502");
            _assertPred!"=="(SysTime(DateTime(-9999, 7, 4, 23, 59, 59), FracSec.from!"hnsecs"(12)).toISOExtString(), "-9999-07-04T23:59:59.0000012");
            _assertPred!"=="(SysTime(DateTime(-10000, 10, 20, 1, 1, 1), FracSec.from!"hnsecs"(507890)).toISOExtString(), "-10000-10-20T01:01:01.050789");

            const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            static assert(__traits(compiles, cast(TimeOfDay)cst));
            //static assert(__traits(compiles, cast(TimeOfDay)ist));

            //Verify Examples.
            assert(SysTime(DateTime(2010, 7, 4, 7, 6, 12)).toISOExtString() ==
                   "2010-07-04T07:06:12");

            assert(SysTime(DateTime(1998, 12, 25, 2, 15, 0),
                           FracSec.from!"msecs"(24)).toISOExtString() ==
                   "1998-12-25T02:15:00.024");

            assert(SysTime(DateTime(0, 1, 5, 23, 9, 59)).toISOExtString() ==
                   "0000-01-05T23:09:59");

            assert(SysTime(DateTime(-4, 1, 5, 0, 0, 2),
                           FracSec.from!"hnsecs"(520_920)).toISOExtString() ==
                   "-0004-01-05T00:00:02.052092");
        }
    }

    /++
        Converts this $(D SysTime) to a string with the format
        YYYY-Mon-DD HH:MM:SS.FFFFFFFTZ (where F is fractional seconds and TZ
        is the time zone).

        Note that the number of digits in the fractional seconds varies with the
        number of fractional seconds. It's a maximum of 7 (which would be
        hnsecs), but only has as many as are necessary to hold the correct value
        (so no trailing zeroes), and if there are no fractional seconds, then
        there is no decimal point.

        If this $(D SysTime)'s time zone is $(D LocalTime), then TZ is empty. If
        its time zone is $(D UTC), then it is "Z". Otherwise, it is the offset
        from UTC (e.g. +1:00 or -7:00). Note that the offset from UTC is
        $(I not) enough to uniquely identify the time zone.

        Time zone offsets will be in the form +HH:MM or -HH:MM.

        Examples:
--------------------
assert(SysTime(DateTime(2010, 7, 4, 7, 6, 12)).toSimpleString() ==
       "2010-Jul-04 07:06:12");

assert(SysTime(DateTime(1998, 12, 25, 2, 15, 0),
               FracSec.from!"msecs"(24)).toSimpleString() ==
       "1998-Dec-25 02:15:00.024");

assert(SysTime(DateTime(0, 1, 5, 23, 9, 59)).toSimpleString() ==
       "0000-Jan-05 23:09:59");

assert(SysTime(DateTime(-4, 1, 5, 0, 0, 2),
               FracSec.from!"hnsecs"(520_920)).toSimpleString() ==
        "-0004-Jan-05 00:00:02.052092");
--------------------
      +/
    string toSimpleString() const nothrow
    {
        try
        {
            immutable adjustedTime = adjTime;
            long hnsecs = adjustedTime;

            auto days = splitUnitsFromHNSecs!"days"(hnsecs) + 1;

            if(hnsecs < 0)
            {
                hnsecs += convert!("hours", "hnsecs")(24);
                --days;
            }

            auto hour = splitUnitsFromHNSecs!"hours"(hnsecs);
            auto minute = splitUnitsFromHNSecs!"minutes"(hnsecs);
            auto second = splitUnitsFromHNSecs!"seconds"(hnsecs);

            auto dateTime = DateTime(Date(cast(int)days), TimeOfDay(cast(int)hour, cast(int)minute, cast(int)second));
            auto fracSecStr = fracSecToISOString(cast(int)hnsecs);

            if(_timezone is LocalTime())
                return dateTime.toSimpleString() ~ fracSecToISOString(cast(int)hnsecs);

            if(_timezone is UTC())
                return dateTime.toSimpleString() ~ fracSecToISOString(cast(int)hnsecs) ~ "Z";

            immutable utcOffset = cast(int)convert!("hnsecs", "minutes")(adjustedTime - stdTime);

            return dateTime.toSimpleString() ~ fracSecToISOString(cast(int)hnsecs) ~ SimpleTimeZone.toISOString(utcOffset);
        }
        catch(Exception e)
            assert(0, "format() threw.");
    }

    unittest
    {
        version(testStdDateTime)
        {
            //Test A.D.
            _assertPred!"=="(SysTime(DateTime.init, UTC()).toString(), "0001-Jan-01 00:00:00Z");
            _assertPred!"=="(SysTime(DateTime(1, 1, 1, 0, 0, 0), FracSec.from!"hnsecs"(1), UTC()).toString(), "0001-Jan-01 00:00:00.0000001Z");

            _assertPred!"=="(SysTime(DateTime(9, 12, 4, 0, 0, 0)).toSimpleString(), "0009-Dec-04 00:00:00");
            _assertPred!"=="(SysTime(DateTime(99, 12, 4, 5, 6, 12)).toSimpleString(), "0099-Dec-04 05:06:12");
            _assertPred!"=="(SysTime(DateTime(999, 12, 4, 13, 44, 59)).toSimpleString(), "0999-Dec-04 13:44:59");
            _assertPred!"=="(SysTime(DateTime(9999, 7, 4, 23, 59, 59)).toSimpleString(), "9999-Jul-04 23:59:59");
            _assertPred!"=="(SysTime(DateTime(10000, 10, 20, 1, 1, 1)).toSimpleString(), "+10000-Oct-20 01:01:01");

            _assertPred!"=="(SysTime(DateTime(9, 12, 4, 0, 0, 0), FracSec.from!"msecs"(42)).toSimpleString(), "0009-Dec-04 00:00:00.042");
            _assertPred!"=="(SysTime(DateTime(99, 12, 4, 5, 6, 12), FracSec.from!"msecs"(100)).toSimpleString(), "0099-Dec-04 05:06:12.1");
            _assertPred!"=="(SysTime(DateTime(999, 12, 4, 13, 44, 59), FracSec.from!"usecs"(45020)).toSimpleString(), "0999-Dec-04 13:44:59.04502");
            _assertPred!"=="(SysTime(DateTime(9999, 7, 4, 23, 59, 59), FracSec.from!"hnsecs"(12)).toSimpleString(), "9999-Jul-04 23:59:59.0000012");
            _assertPred!"=="(SysTime(DateTime(10000, 10, 20, 1, 1, 1), FracSec.from!"hnsecs"(507890)).toSimpleString(), "+10000-Oct-20 01:01:01.050789");

            _assertPred!"=="(SysTime(DateTime(2012, 12, 21, 12, 12, 12),
                                    new SimpleTimeZone(-360)).toSimpleString(),
                            "2012-Dec-21 12:12:12-06:00");

            _assertPred!"=="(SysTime(DateTime(2012, 12, 21, 12, 12, 12),
                                    new SimpleTimeZone(420)).toSimpleString(),
                            "2012-Dec-21 12:12:12+07:00");

            //Test B.C.
            _assertPred!"=="(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(9_999_999), UTC()).toSimpleString(), "0000-Dec-31 23:59:59.9999999Z");
            _assertPred!"=="(SysTime(DateTime(0, 12, 31, 23, 59, 59), FracSec.from!"hnsecs"(1), UTC()).toSimpleString(), "0000-Dec-31 23:59:59.0000001Z");
            _assertPred!"=="(SysTime(DateTime(0, 12, 31, 23, 59, 59), UTC()).toSimpleString(), "0000-Dec-31 23:59:59Z");

            _assertPred!"=="(SysTime(DateTime(0, 12, 4, 0, 12, 4)).toSimpleString(), "0000-Dec-04 00:12:04");
            _assertPred!"=="(SysTime(DateTime(-9, 12, 4, 0, 0, 0)).toSimpleString(), "-0009-Dec-04 00:00:00");
            _assertPred!"=="(SysTime(DateTime(-99, 12, 4, 5, 6, 12)).toSimpleString(), "-0099-Dec-04 05:06:12");
            _assertPred!"=="(SysTime(DateTime(-999, 12, 4, 13, 44, 59)).toSimpleString(), "-0999-Dec-04 13:44:59");
            _assertPred!"=="(SysTime(DateTime(-9999, 7, 4, 23, 59, 59)).toSimpleString(), "-9999-Jul-04 23:59:59");
            _assertPred!"=="(SysTime(DateTime(-10000, 10, 20, 1, 1, 1)).toSimpleString(), "-10000-Oct-20 01:01:01");

            _assertPred!"=="(SysTime(DateTime(0, 12, 4, 0, 0, 0), FracSec.from!"msecs"(7)).toSimpleString(), "0000-Dec-04 00:00:00.007");
            _assertPred!"=="(SysTime(DateTime(-9, 12, 4, 0, 0, 0), FracSec.from!"msecs"(42)).toSimpleString(), "-0009-Dec-04 00:00:00.042");
            _assertPred!"=="(SysTime(DateTime(-99, 12, 4, 5, 6, 12), FracSec.from!"msecs"(100)).toSimpleString(), "-0099-Dec-04 05:06:12.1");
            _assertPred!"=="(SysTime(DateTime(-999, 12, 4, 13, 44, 59), FracSec.from!"usecs"(45020)).toSimpleString(), "-0999-Dec-04 13:44:59.04502");
            _assertPred!"=="(SysTime(DateTime(-9999, 7, 4, 23, 59, 59), FracSec.from!"hnsecs"(12)).toSimpleString(), "-9999-Jul-04 23:59:59.0000012");
            _assertPred!"=="(SysTime(DateTime(-10000, 10, 20, 1, 1, 1), FracSec.from!"hnsecs"(507890)).toSimpleString(), "-10000-Oct-20 01:01:01.050789");

            const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            static assert(__traits(compiles, cast(TimeOfDay)cst));
            //static assert(__traits(compiles, cast(TimeOfDay)ist));

            //Verify Examples.
            assert(SysTime(DateTime(2010, 7, 4, 7, 6, 12)).toSimpleString() ==
                   "2010-Jul-04 07:06:12");

            assert(SysTime(DateTime(1998, 12, 25, 2, 15, 0),
                           FracSec.from!"msecs"(24)).toSimpleString() ==
                   "1998-Dec-25 02:15:00.024");

            assert(SysTime(DateTime(0, 1, 5, 23, 9, 59)).toSimpleString() ==
                   "0000-Jan-05 23:09:59");

            assert(SysTime(DateTime(-4, 1, 5, 0, 0, 2),
                           FracSec.from!"hnsecs"(520_920)).toSimpleString() ==
                    "-0004-Jan-05 00:00:02.052092");
        }
    }


    /+
        Converts this $(D SysTime) to a string.
      +/
    //Due to bug http://d.puremagic.com/issues/show_bug.cgi?id=3715 , we can't
    //have versions of toString() with extra modifiers, so we define one version
    //with modifiers and one without.
    string toString()
    {
        return toSimpleString();
    }

    /++
        Converts this $(D SysTime) to a string.
      +/
    //Due to bug http://d.puremagic.com/issues/show_bug.cgi?id=3715 , we can't
    //have versions of toString() with extra modifiers, so we define one version
    //with modifiers and one without.
    string toString() const nothrow
    {
        return toSimpleString();
    }

    unittest
    {
        version(testStdDateTime)
        {
            auto st = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            const cst = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            //immutable ist = SysTime(DateTime(1999, 7, 6, 12, 30, 33));
            static assert(__traits(compiles, st.toString()));
            static assert(__traits(compiles, cst.toString()));
            //static assert(__traits(compiles, ist.toString()));
        }
    }


    /++
        Creates a $(D SysTime) from a string with the format
        YYYYMMDDTHHMMSS.FFFFFFFTZ (where F is fractional seconds is the time
        zone). Whitespace is stripped from the given string.

        The exact format is exactly as described in $(D toISOString) except that
        trailing zeroes are permitted - including having fractional seconds with
        all zeroes. However, a decimal point with nothing following it is
        invalid.

        If there is no time zone in the string, then $(D LocalTime) is used. If
        the time zone is "Z", then $(D UTC) is used. Otherwise, a
        $(D SimpleTimeZone) which corresponds to the given offset from UTC is
        used. To get the returned $(D SysTime) to be a particular time
        zone, pass in that time zone and the $(D SysTime) to be returned
        will be converted to that time zone (though it will still be read in as
        whatever time zone is in its string).

        The accepted formats for time zone offsets
        are +H, -H, +HH, -HH, +H:MM, -H:MM, +HH:MM, and -HH:MM.

        Params:
            isoString = A string formatted in the ISO format for dates and times.
            tz        = The time zone to convert the given time to (no
                        conversion occurs if null).

        Throws:
            $(D DateTimeException) if the given string is not in the ISO format
            or if the resulting $(D SysTime) would not be valid.

        Examples:
--------------------
assert(SysTime.fromISOString("20100704T070612") ==
       SysTime(DateTime(2010, 7, 4, 7, 6, 12)));
assert(SysTime.fromISOString("19981225T021500.007") ==
       SysTime(DateTime(1998, 12, 25, 2, 15, 0), FracSec.from!"msecs"(7)));
assert(SysTime.fromISOString("00000105T230959.00002") ==
       SysTime(DateTime(0, 1, 5, 23, 9, 59), FracSec.from!"usecs"(20)));
assert(SysTime.fromISOString("-00040105T000002") ==
       SysTime(DateTime(-4, 1, 5, 0, 0, 2)));
assert(SysTime.fromISOString(" 20100704T070612 ") ==
       SysTime(DateTime(2010, 7, 4, 7, 6, 12)));

assert(SysTime.fromISOString("20100704T070612Z") ==
       SysTime(DateTime(2010, 7, 4, 7, 6, 12), UTC()));
assert(SysTime.fromISOString("20100704T070612-8:00") ==
       SysTime(DateTime(2010, 7, 4, 7, 6, 12), new SimpleTimeZone(-480)));
assert(SysTime.fromISOString("20100704T070612+8:00") ==
       SysTime(DateTime(2010, 7, 3, 7, 6, 12), new SimpleTimeZone(480)));
--------------------
      +/
    static SysTime fromISOString(S)(in S isoString, immutable TimeZone tz = null)
        if(isSomeString!S)
    {
        auto dstr = to!dstring(strip(isoString));
        immutable skipFirst = dstr.startsWith("+", "-") != 0;

        auto found = (skipFirst ? dstr[1..$] : dstr).find(".", "Z", "+", "-");
        auto dateTimeStr = dstr[0 .. $ - found[0].length];

        dstring fracSecStr;
        dstring zoneStr;

        if(found[1] != 0)
        {
            if(found[1] == 1)
            {
                auto foundTZ = found[0].find("Z", "+", "-");

                if(foundTZ[1] != 0)
                {
                    fracSecStr = found[0][0 .. $ - foundTZ[0].length];
                    zoneStr = foundTZ[0];
                }
                else
                    fracSecStr = found[0];
            }
            else
                zoneStr = found[0];
        }

        try
        {
            auto dateTime = DateTime.fromISOString(dateTimeStr);
            auto fracSec = fracSecFromISOString(fracSecStr);
            Rebindable!(immutable TimeZone) parsedZone;

            if(zoneStr.empty)
                parsedZone = LocalTime();
            else if(zoneStr == "Z")
                parsedZone = UTC();
            else
                parsedZone = SimpleTimeZone.fromISOString(zoneStr);

            auto retval = SysTime(dateTime, fracSec, parsedZone);

            if(tz !is null)
                retval.timezone = tz;

            return retval;
        }
        catch(DateTimeException dte)
            throw new DateTimeException(format("Invalid ISO String: %s", isoString));
    }

    unittest
    {
        version(testStdDateTime)
        {
            assertThrown!DateTimeException(SysTime.fromISOString(""));
            assertThrown!DateTimeException(SysTime.fromISOString("20100704000000"));
            assertThrown!DateTimeException(SysTime.fromISOString("20100704 000000"));
            assertThrown!DateTimeException(SysTime.fromISOString("20100704t000000"));
            assertThrown!DateTimeException(SysTime.fromISOString("20100704T000000."));
            assertThrown!DateTimeException(SysTime.fromISOString("20100704T000000.A"));
            assertThrown!DateTimeException(SysTime.fromISOString("20100704T000000.Z"));
            assertThrown!DateTimeException(SysTime.fromISOString("20100704T000000.00000000"));
            assertThrown!DateTimeException(SysTime.fromISOString("20100704T000000.00000000"));
            assertThrown!DateTimeException(SysTime.fromISOString("20100704T000000+"));
            assertThrown!DateTimeException(SysTime.fromISOString("20100704T000000-"));
            assertThrown!DateTimeException(SysTime.fromISOString("20100704T000000:"));
            assertThrown!DateTimeException(SysTime.fromISOString("20100704T000000-:"));
            assertThrown!DateTimeException(SysTime.fromISOString("20100704T000000+:"));
            assertThrown!DateTimeException(SysTime.fromISOString("20100704T000000-1:"));
            assertThrown!DateTimeException(SysTime.fromISOString("20100704T000000+1:"));
            assertThrown!DateTimeException(SysTime.fromISOString("20100704T000000+1:0"));
            assertThrown!DateTimeException(SysTime.fromISOString("20100704T000000-24.00"));
            assertThrown!DateTimeException(SysTime.fromISOString("20100704T000000+24.00"));

            assertThrown!DateTimeException(SysTime.fromISOString("2010-07-0400:00:00"));
            assertThrown!DateTimeException(SysTime.fromISOString("2010-07-04 00:00:00"));
            assertThrown!DateTimeException(SysTime.fromISOString("2010-07-04t00:00:00"));
            assertThrown!DateTimeException(SysTime.fromISOString("2010-07-04T00:00:00."));

            assertThrown!DateTimeException(SysTime.fromISOString("2010-Jul-0400:00:00"));
            assertThrown!DateTimeException(SysTime.fromISOString("2010-Jul-04 00:00:00"));
            assertThrown!DateTimeException(SysTime.fromISOString("2010-Jul-04t00:00:00"));
            assertThrown!DateTimeException(SysTime.fromISOString("2010-Jul-04T00:00:00"));
            assertThrown!DateTimeException(SysTime.fromISOString("2010-Jul-04 00:00:00."));

            assertThrown!DateTimeException(SysTime.fromISOString("2010-12-22T172201"));
            assertThrown!DateTimeException(SysTime.fromISOString("2010-Dec-22 17:22:01"));

            _assertPred!"=="(SysTime.fromISOString("20101222T172201"), SysTime(DateTime(2010, 12, 22, 17, 22, 01)));
            _assertPred!"=="(SysTime.fromISOString("19990706T123033"), SysTime(DateTime(1999, 7, 6, 12, 30, 33)));
            _assertPred!"=="(SysTime.fromISOString("-19990706T123033"), SysTime(DateTime(-1999, 7, 6, 12, 30, 33)));
            _assertPred!"=="(SysTime.fromISOString("+019990706T123033"), SysTime(DateTime(1999, 7, 6, 12, 30, 33)));
            _assertPred!"=="(SysTime.fromISOString("19990706T123033 "), SysTime(DateTime(1999, 7, 6, 12, 30, 33)));
            _assertPred!"=="(SysTime.fromISOString(" 19990706T123033"), SysTime(DateTime(1999, 7, 6, 12, 30, 33)));
            _assertPred!"=="(SysTime.fromISOString(" 19990706T123033 "), SysTime(DateTime(1999, 7, 6, 12, 30, 33)));

            _assertPred!"=="(SysTime.fromISOString("19070707T121212.0"), SysTime(DateTime(1907, 07, 07, 12, 12, 12)));
            _assertPred!"=="(SysTime.fromISOString("19070707T121212.0000000"), SysTime(DateTime(1907, 07, 07, 12, 12, 12)));
            _assertPred!"=="(SysTime.fromISOString("19070707T121212.0000001"), SysTime(DateTime(1907, 07, 07, 12, 12, 12), FracSec.from!"hnsecs"(1)));
            _assertPred!"=="(SysTime.fromISOString("19070707T121212.000001"), SysTime(DateTime(1907, 07, 07, 12, 12, 12), FracSec.from!"usecs"(1)));
            _assertPred!"=="(SysTime.fromISOString("19070707T121212.0000010"), SysTime(DateTime(1907, 07, 07, 12, 12, 12), FracSec.from!"usecs"(1)));
            _assertPred!"=="(SysTime.fromISOString("19070707T121212.001"), SysTime(DateTime(1907, 07, 07, 12, 12, 12), FracSec.from!"msecs"(1)));
            _assertPred!"=="(SysTime.fromISOString("19070707T121212.0010000"), SysTime(DateTime(1907, 07, 07, 12, 12, 12), FracSec.from!"msecs"(1)));

            _assertPred!"=="(SysTime.fromISOString("20101222T172201Z"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), UTC()));
            _assertPred!"=="(SysTime.fromISOString("20101222T172201-1:00"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), new SimpleTimeZone(-60)));
            _assertPred!"=="(SysTime.fromISOString("20101222T172201-1"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), new SimpleTimeZone(-60)));
            _assertPred!"=="(SysTime.fromISOString("20101222T172201-1:30"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), new SimpleTimeZone(-90)));
            _assertPred!"=="(SysTime.fromISOString("20101222T172201-8:00"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), new SimpleTimeZone(-480)));
            _assertPred!"=="(SysTime.fromISOString("20101222T172201+1:00"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), new SimpleTimeZone(60)));
            _assertPred!"=="(SysTime.fromISOString("20101222T172201+1"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), new SimpleTimeZone(60)));
            _assertPred!"=="(SysTime.fromISOString("20101222T172201+1:30"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), new SimpleTimeZone(90)));
            _assertPred!"=="(SysTime.fromISOString("20101222T172201+8:00"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), new SimpleTimeZone(480)));

            _assertPred!"=="(SysTime.fromISOString("20101103T065106.57159Z"), SysTime(DateTime(2010, 11, 3, 6, 51, 6), FracSec.from!"hnsecs"(5715900), UTC()));

            _assertPred!"=="(SysTime.fromISOString("20101222T172201.23412Z"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), FracSec.from!"hnsecs"(2_341_200), UTC()));
            _assertPred!"=="(SysTime.fromISOString("20101222T172201.23112-1:00"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), FracSec.from!"hnsecs"(2_311_200), new SimpleTimeZone(-60)));
            _assertPred!"=="(SysTime.fromISOString("20101222T172201.45-1"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), FracSec.from!"hnsecs"(4_500_000), new SimpleTimeZone(-60)));
            _assertPred!"=="(SysTime.fromISOString("20101222T172201.1-1:30"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), FracSec.from!"hnsecs"(1_000_000), new SimpleTimeZone(-90)));
            _assertPred!"=="(SysTime.fromISOString("20101222T172201.55-8:00"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), FracSec.from!"hnsecs"(5_500_000), new SimpleTimeZone(-480)));
            _assertPred!"=="(SysTime.fromISOString("20101222T172201.1234567+1:00"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), FracSec.from!"hnsecs"(1_234_567), new SimpleTimeZone(60)));
            _assertPred!"=="(SysTime.fromISOString("20101222T172201.0+1"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), FracSec.from!"hnsecs"(0), new SimpleTimeZone(60)));
            _assertPred!"=="(SysTime.fromISOString("20101222T172201.0000000+1:30"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), FracSec.from!"hnsecs"(0), new SimpleTimeZone(90)));
            _assertPred!"=="(SysTime.fromISOString("20101222T172201.45+8:00"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), FracSec.from!"hnsecs"(4_500_000), new SimpleTimeZone(480)));

            //Verify Examples.
            assert(SysTime.fromISOString("20100704T070612") == SysTime(DateTime(2010, 7, 4, 7, 6, 12)));
            assert(SysTime.fromISOString("19981225T021500.007") == SysTime(DateTime(1998, 12, 25, 2, 15, 0), FracSec.from!"msecs"(7)));
            assert(SysTime.fromISOString("00000105T230959.00002") == SysTime(DateTime(0, 1, 5, 23, 9, 59), FracSec.from!"usecs"(20)));
            assert(SysTime.fromISOString("-00040105T000002") == SysTime(DateTime(-4, 1, 5, 0, 0, 2)));
            assert(SysTime.fromISOString(" 20100704T070612 ") == SysTime(DateTime(2010, 7, 4, 7, 6, 12)));

            assert(SysTime.fromISOString("20100704T070612Z") == SysTime(DateTime(2010, 7, 4, 7, 6, 12), UTC()));
            assert(SysTime.fromISOString("20100704T070612-8:00") == SysTime(DateTime(2010, 7, 4, 7, 6, 12), new SimpleTimeZone(-480)));
            assert(SysTime.fromISOString("20100704T070612+8:00") == SysTime(DateTime(2010, 7, 4, 7, 6, 12), new SimpleTimeZone(480)));
        }
    }


    /++
        Creates a $(D SysTime) from a string with the format
        YYYY-MM-DDTHH:MM:SS.FFFFFFFTZ (where F is fractional seconds is the
        time zone). Whitespace is stripped from the given string.

        The exact format is exactly as described in $(D toISOExtString)
        except that trailing zeroes are permitted - including having fractional
        seconds with all zeroes. However, a decimal point with nothing following
        it is invalid.

        If there is no time zone in the string, then $(D LocalTime) is used. If
        the time zone is "Z", then $(D UTC) is used. Otherwise, a
        $(D SimpleTimeZone) which corresponds to the given offset from UTC is
        used. To get the returned $(D SysTime) to be a particular time
        zone, pass in that time zone and the $(D SysTime) to be returned
        will be converted to that time zone (though it will still be read in as
        whatever time zone is in its string).

        The accepted formats for time zone offsets
        are +H, -H, +HH, -HH, +H:MM, -H:MM, +HH:MM, and -HH:MM.

        Params:
            isoString = A string formatted in the ISO Extended format for dates
                        and times.
            tz        = The time zone to convert the given time to (no
                        conversion occurs if null).

        Throws:
            $(D DateTimeException) if the given string is not in the ISO format
            or if the resulting $(D SysTime) would not be valid.

        Examples:
--------------------
assert(SysTime.fromISOExtString("2010-07-04T07:06:12") ==
       SysTime(DateTime(2010, 7, 4, 7, 6, 12)));
assert(SysTime.fromISOExtString("1998-12-25T02:15:00.007") ==
       SysTime(DateTime(1998, 12, 25, 2, 15, 0), FracSec.from!"msecs"(7)));
assert(SysTime.fromISOExtString("0000-01-05T23:09:59.00002") ==
       SysTime(DateTime(0, 1, 5, 23, 9, 59), FracSec.from!"usecs"(20)));
assert(SysTime.fromISOExtString("-0004-01-05T00:00:02") ==
       SysTime(DateTime(-4, 1, 5, 0, 0, 2)));
assert(SysTime.fromISOExtString(" 2010-07-04T07:06:12 ") ==
       SysTime(DateTime(2010, 7, 4, 7, 6, 12)));

assert(SysTime.fromISOExtString("2010-07-04T07:06:12Z") ==
       SysTime(DateTime(2010, 7, 4, 7, 6, 12), UTC()));
assert(SysTime.fromISOExtString("2010-07-04T07:06:12-8:00") ==
       SysTime(DateTime(2010, 7, 4, 7, 6, 12), new SimpleTimeZone(-480)));
assert(SysTime.fromISOExtString("2010-07-04T07:06:12+8:00") ==
       SysTime(DateTime(2010, 7, 4, 7, 6, 12), new SimpleTimeZone(480)));
--------------------
      +/
    static SysTime fromISOExtString(S)(in S isoExtString, immutable TimeZone tz = null)
        if(isSomeString!(S))
    {
        auto dstr = to!dstring(strip(isoExtString));

        auto tIndex = dstr.stds_indexOf("T");
        enforce(tIndex != -1, new DateTimeException(format("Invalid ISO Extended String: %s", isoExtString)));

        auto found = dstr[tIndex + 1 .. $].find(".", "Z", "+", "-");
        auto dateTimeStr = dstr[0 .. $ - found[0].length];

        dstring fracSecStr;
        dstring zoneStr;

        if(found[1] != 0)
        {
            if(found[1] == 1)
            {
                auto foundTZ = found[0].find("Z", "+", "-");

                if(foundTZ[1] != 0)
                {
                    fracSecStr = found[0][0 .. $ - foundTZ[0].length];
                    zoneStr = foundTZ[0];
                }
                else
                    fracSecStr = found[0];
            }
            else
                zoneStr = found[0];
        }

        try
        {
            auto dateTime = DateTime.fromISOExtString(dateTimeStr);
            auto fracSec = fracSecFromISOString(fracSecStr);
            Rebindable!(immutable TimeZone) parsedZone;

            if(zoneStr.empty)
                parsedZone = LocalTime();
            else if(zoneStr == "Z")
                parsedZone = UTC();
            else
                parsedZone = SimpleTimeZone.fromISOString(zoneStr);

            auto retval = SysTime(dateTime, fracSec, parsedZone);

            if(tz !is null)
                retval.timezone = tz;

            return retval;
        }
        catch(DateTimeException dte)
            throw new DateTimeException(format("Invalid ISO Extended String: %s", isoExtString));
    }

    unittest
    {
        version(testStdDateTime)
        {
            assertThrown!DateTimeException(SysTime.fromISOExtString(""));
            assertThrown!DateTimeException(SysTime.fromISOExtString("20100704000000"));
            assertThrown!DateTimeException(SysTime.fromISOExtString("20100704 000000"));
            assertThrown!DateTimeException(SysTime.fromISOExtString("20100704t000000"));
            assertThrown!DateTimeException(SysTime.fromISOExtString("20100704T000000."));
            assertThrown!DateTimeException(SysTime.fromISOExtString("20100704T000000.0"));

            assertThrown!DateTimeException(SysTime.fromISOExtString("2010-07:0400:00:00"));
            assertThrown!DateTimeException(SysTime.fromISOExtString("2010-07-04 00:00:00"));
            assertThrown!DateTimeException(SysTime.fromISOExtString("2010-07-04 00:00:00"));
            assertThrown!DateTimeException(SysTime.fromISOExtString("2010-07-04t00:00:00"));
            assertThrown!DateTimeException(SysTime.fromISOExtString("2010-07-04T00:00:00."));
            assertThrown!DateTimeException(SysTime.fromISOExtString("2010-07-04T00:00:00.A"));
            assertThrown!DateTimeException(SysTime.fromISOExtString("2010-07-04T00:00:00.Z"));
            assertThrown!DateTimeException(SysTime.fromISOExtString("2010-07-04T00:00:00.00000000"));
            assertThrown!DateTimeException(SysTime.fromISOExtString("2010-07-04T00:00:00.00000000"));
            assertThrown!DateTimeException(SysTime.fromISOExtString("2010-07-04T00:00:00+"));
            assertThrown!DateTimeException(SysTime.fromISOExtString("2010-07-04T00:00:00-"));
            assertThrown!DateTimeException(SysTime.fromISOExtString("2010-07-04T00:00:00:"));
            assertThrown!DateTimeException(SysTime.fromISOExtString("2010-07-04T00:00:00-:"));
            assertThrown!DateTimeException(SysTime.fromISOExtString("2010-07-04T00:00:00+:"));
            assertThrown!DateTimeException(SysTime.fromISOExtString("2010-07-04T00:00:00-1:"));
            assertThrown!DateTimeException(SysTime.fromISOExtString("2010-07-04T00:00:00+1:"));
            assertThrown!DateTimeException(SysTime.fromISOExtString("2010-07-04T00:00:00+1:0"));
            assertThrown!DateTimeException(SysTime.fromISOExtString("2010-07-04T00:00:00-24.00"));
            assertThrown!DateTimeException(SysTime.fromISOExtString("2010-07-04T00:00:00+24.00"));

            assertThrown!DateTimeException(SysTime.fromISOExtString("2010-Jul-0400:00:00"));
            assertThrown!DateTimeException(SysTime.fromISOExtString("2010-Jul-04t00:00:00"));
            assertThrown!DateTimeException(SysTime.fromISOExtString("2010-Jul-04 00:00:00."));
            assertThrown!DateTimeException(SysTime.fromISOExtString("2010-Jul-04 00:00:00.0"));

            assertThrown!DateTimeException(SysTime.fromISOExtString("20101222T172201"));
            assertThrown!DateTimeException(SysTime.fromISOExtString("2010-Dec-22 17:22:01"));

            _assertPred!"=="(SysTime.fromISOExtString("2010-12-22T17:22:01"), SysTime(DateTime(2010, 12, 22, 17, 22, 01)));
            _assertPred!"=="(SysTime.fromISOExtString("1999-07-06T12:30:33"), SysTime(DateTime(1999, 7, 6, 12, 30, 33)));
            _assertPred!"=="(SysTime.fromISOExtString("-1999-07-06T12:30:33"), SysTime(DateTime(-1999, 7, 6, 12, 30, 33)));
            _assertPred!"=="(SysTime.fromISOExtString("+01999-07-06T12:30:33"), SysTime(DateTime(1999, 7, 6, 12, 30, 33)));
            _assertPred!"=="(SysTime.fromISOExtString("1999-07-06T12:30:33 "), SysTime(DateTime(1999, 7, 6, 12, 30, 33)));
            _assertPred!"=="(SysTime.fromISOExtString(" 1999-07-06T12:30:33"), SysTime(DateTime(1999, 7, 6, 12, 30, 33)));
            _assertPred!"=="(SysTime.fromISOExtString(" 1999-07-06T12:30:33 "), SysTime(DateTime(1999, 7, 6, 12, 30, 33)));

            _assertPred!"=="(SysTime.fromISOExtString("1907-07-07T12:12:12.0"), SysTime(DateTime(1907, 07, 07, 12, 12, 12)));
            _assertPred!"=="(SysTime.fromISOExtString("1907-07-07T12:12:12.0000000"), SysTime(DateTime(1907, 07, 07, 12, 12, 12)));
            _assertPred!"=="(SysTime.fromISOExtString("1907-07-07T12:12:12.0000001"), SysTime(DateTime(1907, 07, 07, 12, 12, 12), FracSec.from!"hnsecs"(1)));
            _assertPred!"=="(SysTime.fromISOExtString("1907-07-07T12:12:12.000001"), SysTime(DateTime(1907, 07, 07, 12, 12, 12), FracSec.from!"usecs"(1)));
            _assertPred!"=="(SysTime.fromISOExtString("1907-07-07T12:12:12.0000010"), SysTime(DateTime(1907, 07, 07, 12, 12, 12), FracSec.from!"usecs"(1)));
            _assertPred!"=="(SysTime.fromISOExtString("1907-07-07T12:12:12.001"), SysTime(DateTime(1907, 07, 07, 12, 12, 12), FracSec.from!"msecs"(1)));
            _assertPred!"=="(SysTime.fromISOExtString("1907-07-07T12:12:12.0010000"), SysTime(DateTime(1907, 07, 07, 12, 12, 12), FracSec.from!"msecs"(1)));

            _assertPred!"=="(SysTime.fromISOExtString("2010-12-22T17:22:01Z"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), UTC()));
            _assertPred!"=="(SysTime.fromISOExtString("2010-12-22T17:22:01-1:00"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), new SimpleTimeZone(-60)));
            _assertPred!"=="(SysTime.fromISOExtString("2010-12-22T17:22:01-1"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), new SimpleTimeZone(-60)));
            _assertPred!"=="(SysTime.fromISOExtString("2010-12-22T17:22:01-1:30"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), new SimpleTimeZone(-90)));
            _assertPred!"=="(SysTime.fromISOExtString("2010-12-22T17:22:01-8:00"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), new SimpleTimeZone(-480)));
            _assertPred!"=="(SysTime.fromISOExtString("2010-12-22T17:22:01+1:00"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), new SimpleTimeZone(60)));
            _assertPred!"=="(SysTime.fromISOExtString("2010-12-22T17:22:01+1"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), new SimpleTimeZone(60)));
            _assertPred!"=="(SysTime.fromISOExtString("2010-12-22T17:22:01+1:30"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), new SimpleTimeZone(90)));
            _assertPred!"=="(SysTime.fromISOExtString("2010-12-22T17:22:01+8:00"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), new SimpleTimeZone(480)));

            _assertPred!"=="(SysTime.fromISOExtString("2010-11-03T06:51:06.57159Z"), SysTime(DateTime(2010, 11, 3, 6, 51, 6), FracSec.from!"hnsecs"(5715900), UTC()));

            _assertPred!"=="(SysTime.fromISOExtString("2010-12-22T17:22:01.23412Z"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), FracSec.from!"hnsecs"(2_341_200), UTC()));
            _assertPred!"=="(SysTime.fromISOExtString("2010-12-22T17:22:01.23112-1:00"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), FracSec.from!"hnsecs"(2_311_200), new SimpleTimeZone(-60)));
            _assertPred!"=="(SysTime.fromISOExtString("2010-12-22T17:22:01.45-1"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), FracSec.from!"hnsecs"(4_500_000), new SimpleTimeZone(-60)));
            _assertPred!"=="(SysTime.fromISOExtString("2010-12-22T17:22:01.1-1:30"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), FracSec.from!"hnsecs"(1_000_000), new SimpleTimeZone(-90)));
            _assertPred!"=="(SysTime.fromISOExtString("2010-12-22T17:22:01.55-8:00"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), FracSec.from!"hnsecs"(5_500_000), new SimpleTimeZone(-480)));
            _assertPred!"=="(SysTime.fromISOExtString("2010-12-22T17:22:01.1234567+1:00"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), FracSec.from!"hnsecs"(1_234_567), new SimpleTimeZone(60)));
            _assertPred!"=="(SysTime.fromISOExtString("2010-12-22T17:22:01.0+1"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), FracSec.from!"hnsecs"(0), new SimpleTimeZone(60)));
            _assertPred!"=="(SysTime.fromISOExtString("2010-12-22T17:22:01.0000000+1:30"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), FracSec.from!"hnsecs"(0), new SimpleTimeZone(90)));
            _assertPred!"=="(SysTime.fromISOExtString("2010-12-22T17:22:01.45+8:00"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), FracSec.from!"hnsecs"(4_500_000), new SimpleTimeZone(480)));

            //Verify Examples.
            assert(SysTime.fromISOExtString("2010-07-04T07:06:12") == SysTime(DateTime(2010, 7, 4, 7, 6, 12)));
            assert(SysTime.fromISOExtString("1998-12-25T02:15:00.007") == SysTime(DateTime(1998, 12, 25, 2, 15, 0), FracSec.from!"msecs"(7)));
            assert(SysTime.fromISOExtString("0000-01-05T23:09:59.00002") == SysTime(DateTime(0, 1, 5, 23, 9, 59), FracSec.from!"usecs"(20)));
            assert(SysTime.fromISOExtString("-0004-01-05T00:00:02") == SysTime(DateTime(-4, 1, 5, 0, 0, 2)));
            assert(SysTime.fromISOExtString(" 2010-07-04T07:06:12 ") == SysTime(DateTime(2010, 7, 4, 7, 6, 12)));

            assert(SysTime.fromISOExtString("2010-07-04T07:06:12Z") == SysTime(DateTime(2010, 7, 4, 7, 6, 12), UTC()));
            assert(SysTime.fromISOExtString("2010-07-04T07:06:12-8:00") == SysTime(DateTime(2010, 7, 4, 7, 6, 12), new SimpleTimeZone(-480)));
            assert(SysTime.fromISOExtString("2010-07-04T07:06:12+8:00") == SysTime(DateTime(2010, 7, 4, 7, 6, 12), new SimpleTimeZone(480)));
        }
    }


    /++
        Creates a $(D SysTime) from a string with the format
        YYYY-MM-DD HH:MM:SS.FFFFFFFTZ (where F is fractional seconds is the
        time zone). Whitespace is stripped from the given string.

        The exact format is exactly as described in $(D toSimpleString) except
        that trailing zeroes are permitted - including having fractional seconds
        with all zeroes. However, a decimal point with nothing following it is
        invalid.

        If there is no time zone in the string, then $(D LocalTime) is used. If
        the time zone is "Z", then $(D UTC) is used. Otherwise, a
        $(D SimpleTimeZone) which corresponds to the given offset from UTC is
        used. To get the returned $(D SysTime) to be a particular time
        zone, pass in that time zone and the $(D SysTime) to be returned
        will be converted to that time zone (though it will still be read in as
        whatever time zone is in its string).

        The accepted formats for time zone offsets
        are +H, -H, +HH, -HH, +H:MM, -H:MM, +HH:MM, and -HH:MM.


        Params:
            simpleString = A string formatted in the way that
                           $(D toSimpleString) formats dates and times.
            tz           = The time zone to convert the given time to (no
                           conversion occurs if null).

        Throws:
            $(D DateTimeException) if the given string is not in the ISO format
            or if the resulting $(D SysTime) would not be valid.

        Examples:
--------------------
assert(SysTime.fromSimpleString("2010-Jul-04 07:06:12") ==
       SysTime(DateTime(2010, 7, 4, 7, 6, 12)));
assert(SysTime.fromSimpleString("1998-Dec-25 02:15:00.007") ==
       SysTime(DateTime(1998, 12, 25, 2, 15, 0), FracSec.from!"msecs"(7)));
assert(SysTime.fromSimpleString("0000-Jan-05 23:09:59.00002") ==
       SysTime(DateTime(0, 1, 5, 23, 9, 59), FracSec.from!"usecs"(20)));
assert(SysTime.fromSimpleString("-0004-Jan-05 00:00:02") ==
       SysTime(DateTime(-4, 1, 5, 0, 0, 2)));
assert(SysTime.fromSimpleString(" 2010-Jul-04 07:06:12 ") ==
       SysTime(DateTime(2010, 7, 4, 7, 6, 12)));

assert(SysTime.fromSimpleString("2010-Jul-04 07:06:12Z") ==
       SysTime(DateTime(2010, 7, 4, 7, 6, 12), UTC()));
assert(SysTime.fromSimpleString("2010-Jul-04 07:06:12-8:00") ==
       SysTime(DateTime(2010, 7, 4, 7, 6, 12), new SimpleTimeZone(-480)));
assert(SysTime.fromSimpleString("2010-Jul-04 07:06:12+8:00") ==
       SysTime(DateTime(2010, 7, 4, 7, 6, 12), new SimpleTimeZone(480)));
--------------------
      +/
    static SysTime fromSimpleString(S)(in S simpleString, immutable TimeZone tz = null)
        if(isSomeString!(S))
    {
        auto dstr = to!dstring(strip(simpleString));

        auto spaceIndex = dstr.stds_indexOf(" ");
        enforce(spaceIndex != -1, new DateTimeException(format("Invalid Simple String: %s", simpleString)));

        auto found = dstr[spaceIndex + 1 .. $].find(".", "Z", "+", "-");
        auto dateTimeStr = dstr[0 .. $ - found[0].length];

        dstring fracSecStr;
        dstring zoneStr;

        if(found[1] != 0)
        {
            if(found[1] == 1)
            {
                auto foundTZ = found[0].find("Z", "+", "-");

                if(foundTZ[1] != 0)
                {
                    fracSecStr = found[0][0 .. $ - foundTZ[0].length];
                    zoneStr = foundTZ[0];
                }
                else
                    fracSecStr = found[0];
            }
            else
                zoneStr = found[0];
        }

        try
        {
            auto dateTime = DateTime.fromSimpleString(dateTimeStr);
            auto fracSec = fracSecFromISOString(fracSecStr);
            Rebindable!(immutable TimeZone) parsedZone;

            if(zoneStr.empty)
                parsedZone = LocalTime();
            else if(zoneStr == "Z")
                parsedZone = UTC();
            else
                parsedZone = SimpleTimeZone.fromISOString(zoneStr);

            auto retval = SysTime(dateTime, fracSec, parsedZone);

            if(tz !is null)
                retval.timezone = tz;

            return retval;
        }
        catch(DateTimeException dte)
            throw new DateTimeException(format("Invalid Simple String: %s", simpleString));
    }

    unittest
    {
        version(testStdDateTime)
        {
            assertThrown!DateTimeException(SysTime.fromSimpleString(""));
            assertThrown!DateTimeException(SysTime.fromSimpleString("20100704000000"));
            assertThrown!DateTimeException(SysTime.fromSimpleString("20100704 000000"));
            assertThrown!DateTimeException(SysTime.fromSimpleString("20100704t000000"));
            assertThrown!DateTimeException(SysTime.fromSimpleString("20100704T000000."));
            assertThrown!DateTimeException(SysTime.fromSimpleString("20100704T000000.0"));

            assertThrown!DateTimeException(SysTime.fromSimpleString("2010-07-0400:00:00"));
            assertThrown!DateTimeException(SysTime.fromSimpleString("2010-07-04 00:00:00"));
            assertThrown!DateTimeException(SysTime.fromSimpleString("2010-07-04t00:00:00"));
            assertThrown!DateTimeException(SysTime.fromSimpleString("2010-07-04T00:00:00."));
            assertThrown!DateTimeException(SysTime.fromSimpleString("2010-07-04T00:00:00.0"));

            assertThrown!DateTimeException(SysTime.fromSimpleString("2010-Jul-0400:00:00"));
            assertThrown!DateTimeException(SysTime.fromSimpleString("2010-Jul-04t00:00:00"));
            assertThrown!DateTimeException(SysTime.fromSimpleString("2010-Jul-04T00:00:00"));
            assertThrown!DateTimeException(SysTime.fromSimpleString("2010-Jul-04 00:00:00."));
            assertThrown!DateTimeException(SysTime.fromSimpleString("2010-Jul-04 00:00:00.A"));
            assertThrown!DateTimeException(SysTime.fromSimpleString("2010-Jul-04 00:00:00.Z"));
            assertThrown!DateTimeException(SysTime.fromSimpleString("2010-Jul-04 00:00:00.00000000"));
            assertThrown!DateTimeException(SysTime.fromSimpleString("2010-Jul-04 00:00:00.00000000"));
            assertThrown!DateTimeException(SysTime.fromSimpleString("2010-Jul-04 00:00:00+"));
            assertThrown!DateTimeException(SysTime.fromSimpleString("2010-Jul-04 00:00:00-"));
            assertThrown!DateTimeException(SysTime.fromSimpleString("2010-Jul-04 00:00:00:"));
            assertThrown!DateTimeException(SysTime.fromSimpleString("2010-Jul-04 00:00:00-:"));
            assertThrown!DateTimeException(SysTime.fromSimpleString("2010-Jul-04 00:00:00+:"));
            assertThrown!DateTimeException(SysTime.fromSimpleString("2010-Jul-04 00:00:00-1:"));
            assertThrown!DateTimeException(SysTime.fromSimpleString("2010-Jul-04 00:00:00+1:"));
            assertThrown!DateTimeException(SysTime.fromSimpleString("2010-Jul-04 00:00:00+1:0"));
            assertThrown!DateTimeException(SysTime.fromSimpleString("2010-Jul-04 00:00:00-24.00"));
            assertThrown!DateTimeException(SysTime.fromSimpleString("2010-Jul-04 00:00:00+24.00"));

            assertThrown!DateTimeException(SysTime.fromSimpleString("20101222T172201"));
            assertThrown!DateTimeException(SysTime.fromSimpleString("2010-12-22T172201"));

            _assertPred!"=="(SysTime.fromSimpleString("2010-Dec-22 17:22:01"), SysTime(DateTime(2010, 12, 22, 17, 22, 01)));
            _assertPred!"=="(SysTime.fromSimpleString("1999-Jul-06 12:30:33"), SysTime(DateTime(1999, 7, 6, 12, 30, 33)));
            _assertPred!"=="(SysTime.fromSimpleString("-1999-Jul-06 12:30:33"), SysTime(DateTime(-1999, 7, 6, 12, 30, 33)));
            _assertPred!"=="(SysTime.fromSimpleString("+01999-Jul-06 12:30:33"), SysTime(DateTime(1999, 7, 6, 12, 30, 33)));
            _assertPred!"=="(SysTime.fromSimpleString("1999-Jul-06 12:30:33 "), SysTime(DateTime(1999, 7, 6, 12, 30, 33)));
            _assertPred!"=="(SysTime.fromSimpleString(" 1999-Jul-06 12:30:33"), SysTime(DateTime(1999, 7, 6, 12, 30, 33)));
            _assertPred!"=="(SysTime.fromSimpleString(" 1999-Jul-06 12:30:33 "), SysTime(DateTime(1999, 7, 6, 12, 30, 33)));

            _assertPred!"=="(SysTime.fromSimpleString("1907-Jul-07 12:12:12.0"), SysTime(DateTime(1907, 07, 07, 12, 12, 12)));
            _assertPred!"=="(SysTime.fromSimpleString("1907-Jul-07 12:12:12.0000000"), SysTime(DateTime(1907, 07, 07, 12, 12, 12)));
            _assertPred!"=="(SysTime.fromSimpleString("1907-Jul-07 12:12:12.0000001"), SysTime(DateTime(1907, 07, 07, 12, 12, 12), FracSec.from!"hnsecs"(1)));
            _assertPred!"=="(SysTime.fromSimpleString("1907-Jul-07 12:12:12.000001"), SysTime(DateTime(1907, 07, 07, 12, 12, 12), FracSec.from!"usecs"(1)));
            _assertPred!"=="(SysTime.fromSimpleString("1907-Jul-07 12:12:12.0000010"), SysTime(DateTime(1907, 07, 07, 12, 12, 12), FracSec.from!"usecs"(1)));
            _assertPred!"=="(SysTime.fromSimpleString("1907-Jul-07 12:12:12.001"), SysTime(DateTime(1907, 07, 07, 12, 12, 12), FracSec.from!"msecs"(1)));
            _assertPred!"=="(SysTime.fromSimpleString("1907-Jul-07 12:12:12.0010000"), SysTime(DateTime(1907, 07, 07, 12, 12, 12), FracSec.from!"msecs"(1)));

            _assertPred!"=="(SysTime.fromSimpleString("2010-Dec-22 17:22:01Z"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), UTC()));
            _assertPred!"=="(SysTime.fromSimpleString("2010-Dec-22 17:22:01-1:00"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), new SimpleTimeZone(-60)));
            _assertPred!"=="(SysTime.fromSimpleString("2010-Dec-22 17:22:01-1"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), new SimpleTimeZone(-60)));
            _assertPred!"=="(SysTime.fromSimpleString("2010-Dec-22 17:22:01-1:30"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), new SimpleTimeZone(-90)));
            _assertPred!"=="(SysTime.fromSimpleString("2010-Dec-22 17:22:01-8:00"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), new SimpleTimeZone(-480)));
            _assertPred!"=="(SysTime.fromSimpleString("2010-Dec-22 17:22:01+1:00"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), new SimpleTimeZone(60)));
            _assertPred!"=="(SysTime.fromSimpleString("2010-Dec-22 17:22:01+1"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), new SimpleTimeZone(60)));
            _assertPred!"=="(SysTime.fromSimpleString("2010-Dec-22 17:22:01+1:30"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), new SimpleTimeZone(90)));
            _assertPred!"=="(SysTime.fromSimpleString("2010-Dec-22 17:22:01+8:00"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), new SimpleTimeZone(480)));

            _assertPred!"=="(SysTime.fromSimpleString("2010-Nov-03 06:51:06.57159Z"), SysTime(DateTime(2010, 11, 3, 6, 51, 6), FracSec.from!"hnsecs"(5715900), UTC()));

            _assertPred!"=="(SysTime.fromSimpleString("2010-Dec-22 17:22:01.23412Z"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), FracSec.from!"hnsecs"(2_341_200), UTC()));
            _assertPred!"=="(SysTime.fromSimpleString("2010-Dec-22 17:22:01.23112-1:00"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), FracSec.from!"hnsecs"(2_311_200), new SimpleTimeZone(-60)));
            _assertPred!"=="(SysTime.fromSimpleString("2010-Dec-22 17:22:01.45-1"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), FracSec.from!"hnsecs"(4_500_000), new SimpleTimeZone(-60)));
            _assertPred!"=="(SysTime.fromSimpleString("2010-Dec-22 17:22:01.1-1:30"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), FracSec.from!"hnsecs"(1_000_000), new SimpleTimeZone(-90)));
            _assertPred!"=="(SysTime.fromSimpleString("2010-Dec-22 17:22:01.55-8:00"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), FracSec.from!"hnsecs"(5_500_000), new SimpleTimeZone(-480)));
            _assertPred!"=="(SysTime.fromSimpleString("2010-Dec-22 17:22:01.1234567+1:00"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), FracSec.from!"hnsecs"(1_234_567), new SimpleTimeZone(60)));
            _assertPred!"=="(SysTime.fromSimpleString("2010-Dec-22 17:22:01.0+1"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), FracSec.from!"hnsecs"(0), new SimpleTimeZone(60)));
            _assertPred!"=="(SysTime.fromSimpleString("2010-Dec-22 17:22:01.0000000+1:30"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), FracSec.from!"hnsecs"(0), new SimpleTimeZone(90)));
            _assertPred!"=="(SysTime.fromSimpleString("2010-Dec-22 17:22:01.45+8:00"), SysTime(DateTime(2010, 12, 22, 17, 22, 01), FracSec.from!"hnsecs"(4_500_000), new SimpleTimeZone(480)));

            //Verify Examples.
            assert(SysTime.fromSimpleString("2010-Jul-04 07:06:12") == SysTime(DateTime(2010, 7, 4, 7, 6, 12)));
            assert(SysTime.fromSimpleString("1998-Dec-25 02:15:00.007") == SysTime(DateTime(1998, 12, 25, 2, 15, 0), FracSec.from!"msecs"(7)));
            assert(SysTime.fromSimpleString("0000-Jan-05 23:09:59.00002") == SysTime(DateTime(0, 1, 5, 23, 9, 59), FracSec.from!"usecs"(20)));
            assert(SysTime.fromSimpleString("-0004-Jan-05 00:00:02") == SysTime(DateTime(-4, 1, 5, 0, 0, 2)));
            assert(SysTime.fromSimpleString(" 2010-Jul-04 07:06:12 ") == SysTime(DateTime(2010, 7, 4, 7, 6, 12)));

            assert(SysTime.fromSimpleString("2010-Jul-04 07:06:12Z") == SysTime(DateTime(2010, 7, 4, 7, 6, 12), UTC()));
            assert(SysTime.fromSimpleString("2010-Jul-04 07:06:12-8:00") == SysTime(DateTime(2010, 7, 4, 7, 6, 12), new SimpleTimeZone(-480)));
            assert(SysTime.fromSimpleString("2010-Jul-04 07:06:12+8:00") == SysTime(DateTime(2010, 7, 4, 7, 6, 12), new SimpleTimeZone(480)));
        }
    }


    //TODO Add function which takes a user-specified time format and produces a SysTime

    //TODO Add function which takes pretty much any time-string and produces a SysTime.
    //     Obviously, it will be less efficient, and it probably won't manage _every_
    //     possible date format, but a smart conversion function would be nice.


    /++
        Returns the $(D SysTime) farthest in the past which is representable
        by $(D SysTime).

        The $(D SysTime) which is returned is in UTC.
      +/
    @property static SysTime min() pure nothrow
    {
        return SysTime(long.min, UTC());
    }

    unittest
    {
        version(testStdDateTime)
        {
            assert(SysTime.min.year < 0);
            assert(SysTime.min < SysTime.max);
        }
    }


    /++
        Returns the $(D SysTime) farthest in the future which is representable
        by $(D SysTime).

        The $(D SysTime) which is returned is in UTC.
      +/
    @property static SysTime max() pure nothrow
    {
        return SysTime(long.max, UTC());
    }

    unittest
    {
        version(testStdDateTime)
        {
            assert(SysTime.max.year > 0);
            assert(SysTime.max > SysTime.min);
        }
    }


private:

    /+
        Returns $(D stdTime) converted to $(D SysTime)'s time zone.
      +/
    @property long adjTime() const nothrow
    {
        return _timezone.utcToTZ(_stdTime);
    }


    /+
        Converts the given hnsecs from $(D SysTime)'s time zone to std time.
      +/
    @property void adjTime(long adjTime) nothrow
    {
        _stdTime = _timezone.tzToUTC(adjTime);
    }


    //Commented out due to bug http://d.puremagic.com/issues/show_bug.cgi?id=5058
    /+
    invariant()
    {
        assert(_timezone !is null, "Invariant Failure: timezone is null. Were you foolish enough to use SysTime.init? (since timezone for SysTime.init can't be set at compile time).");
    }
    +/


    long  _stdTime;
    Rebindable!(immutable TimeZone) _timezone;
}


/++
    Represents a date in the Proleptic Gregorian Calendar ranging from
    32,768 B.C. to 32,767 A.D. Positive years are A.D. Non-positive years are
    B.C.

    Year, month, and day are kept separately internally so that $(D Date) is
    optimized for calendar-based operations.

    $(D Date) uses the Proleptic Gregorian Calendar, so it assumes the Gregorian
    leap year calculations for its entire length. And, as per
    $(WEB en.wikipedia.org/wiki/ISO_8601, ISO 8601), it also treats 1 B.C. as
    year 0. So, 1 B.C. is 0, 2 B.C. is -1, etc. Use $(D yearBC) if want B.C. as
    a positive integer with 1 B.C. being the year prior to 1 A.D.

    Year 0 is a leap year.
 +/
struct Date
{
public:

    /++
        Throws:
            $(D DateTimeException) if the resulting $(D Date) would not be valid.

        Params:
            year  = Year of the Gregorian Calendar. Positive values are A.D.
                    Non-positive values are B.C. with year 0 being the year
                    prior to 1 A.D.
            month = Month of the year.
            day   = Day of the month.
     +/
    this(int year, int month, int day) pure
    {
        enforceValid!"months"(cast(Month)month);
        enforceValid!"days"(year, cast(Month)month, day);

        _year  = cast(short)year;
        _month = cast(Month)month;
        _day   = cast(ubyte)day;
    }

    unittest
    {
        version(testStdDateTime)
        {
            _assertPred!"=="(Date(1, 1, 1), Date.init);

            static void testDate(in Date date, int year, int month, int day, size_t line = __LINE__)
            {
                _assertPred!"=="(date._year, year, "", __FILE__, line);
                _assertPred!"=="(date._month, month, "", __FILE__, line);
                _assertPred!"=="(date._day, day, "", __FILE__, line);
            }

            testDate(Date(1999, 1 , 1), 1999, Month.jan, 1);
            testDate(Date(1999, 7 , 1), 1999, Month.jul, 1);
            testDate(Date(1999, 7 , 6), 1999, Month.jul, 6);

            //Test A.D.
            assertThrown!DateTimeException(Date(1, 0, 1));
            assertThrown!DateTimeException(Date(1, 1, 0));
            assertThrown!DateTimeException(Date(1999, 13, 1));
            assertThrown!DateTimeException(Date(1999, 1, 32));
            assertThrown!DateTimeException(Date(1999, 2, 29));
            assertThrown!DateTimeException(Date(2000, 2, 30));
            assertThrown!DateTimeException(Date(1999, 3, 32));
            assertThrown!DateTimeException(Date(1999, 4, 31));
            assertThrown!DateTimeException(Date(1999, 5, 32));
            assertThrown!DateTimeException(Date(1999, 6, 31));
            assertThrown!DateTimeException(Date(1999, 7, 32));
            assertThrown!DateTimeException(Date(1999, 8, 32));
            assertThrown!DateTimeException(Date(1999, 9, 31));
            assertThrown!DateTimeException(Date(1999, 10, 32));
            assertThrown!DateTimeException(Date(1999, 11, 31));
            assertThrown!DateTimeException(Date(1999, 12, 32));

            assertNotThrown!DateTimeException(Date(1999, 1, 31));
            assertNotThrown!DateTimeException(Date(1999, 2, 28));
            assertNotThrown!DateTimeException(Date(2000, 2, 29));
            assertNotThrown!DateTimeException(Date(1999, 3, 31));
            assertNotThrown!DateTimeException(Date(1999, 4, 30));
            assertNotThrown!DateTimeException(Date(1999, 5, 31));
            assertNotThrown!DateTimeException(Date(1999, 6, 30));
            assertNotThrown!DateTimeException(Date(1999, 7, 31));
            assertNotThrown!DateTimeException(Date(1999, 8, 31));
            assertNotThrown!DateTimeException(Date(1999, 9, 30));
            assertNotThrown!DateTimeException(Date(1999, 10, 31));
            assertNotThrown!DateTimeException(Date(1999, 11, 30));
            assertNotThrown!DateTimeException(Date(1999, 12, 31));

            //Test B.C.
            assertNotThrown!DateTimeException(Date(0, 1, 1));
            assertNotThrown!DateTimeException(Date(-1, 1, 1));
            assertNotThrown!DateTimeException(Date(-1, 12, 31));
            assertNotThrown!DateTimeException(Date(-1, 2, 28));
            assertNotThrown!DateTimeException(Date(-4, 2, 29));

            assertThrown!DateTimeException(Date(-1, 2, 29));
            assertThrown!DateTimeException(Date(-2, 2, 29));
            assertThrown!DateTimeException(Date(-3, 2, 29));
            }
    }


    /++
        Params:
            day = The Xth day of the Gregorian Calendar that the constructed
                  $(D Date) will be for.
     +/
    this(int day) pure nothrow
    {
        if(day > 0)
        {
            int years = (day / daysIn400Years) * 400 + 1;
            day %= daysIn400Years;

            {
                immutable tempYears = day / daysIn100Years;

                if(tempYears == 4)
                {
                    years += 300;
                    day -= daysIn100Years * 3;
                }
                else
                {
                    years += tempYears * 100;
                    day %= daysIn100Years;
                }
            }

            years += (day / daysIn4Years) * 4;
            day %= daysIn4Years;

            {
                immutable tempYears = day / daysInYear;

                if(tempYears == 4)
                {
                    years += 3;
                    day -= daysInYear * 3;
                }
                else
                {
                    years += tempYears;
                    day %= daysInYear;
                }
            }

            if(day == 0)
            {
                _year = cast(short)(years - 1);
                _month = Month.dec;
                _day = 31;
            }
            else
            {
                _year = cast(short)years;

                try
                    dayOfYear = day;
                catch(Exception e)
                    assert(0, "dayOfYear assignment threw.");
            }
        }
        else if(day <= 0 && -day < daysInLeapYear)
        {
            _year = 0;

            try
                dayOfYear = (daysInLeapYear + day);
            catch(Exception e)
                assert(0, "dayOfYear assignment threw.");
        }
        else
        {
            day += daysInLeapYear - 1;
            int years = (day / daysIn400Years) * 400 - 1;
            day %= daysIn400Years;

            {
                immutable tempYears = day / daysIn100Years;

                if(tempYears == -4)
                {
                    years -= 300;
                    day += daysIn100Years * 3;
                }
                else
                {
                    years += tempYears * 100;
                    day %= daysIn100Years;
                }
            }

            years += (day / daysIn4Years) * 4;
            day %= daysIn4Years;

            {
                immutable tempYears = day / daysInYear;

                if(tempYears == -4)
                {
                    years -= 3;
                    day += daysInYear * 3;
                }
                else
                {
                    years += tempYears;
                    day %= daysInYear;
                }
            }

            if(day == 0)
            {
                _year = cast(short)(years + 1);
                _month = Month.jan;
                _day = 1;
            }
            else
            {
                _year = cast(short)years;
                immutable newDoY = (yearIsLeapYear(_year) ? daysInLeapYear : daysInYear) + day + 1;

                try
                    dayOfYear = newDoY;
                catch(Exception e)
                    assert(0, "dayOfYear assignment threw.");
            }
        }
    }

    version(testStdDateTime) unittest
    {
        //Test A.D.
        foreach(gd; chain(testGregDaysBC, testGregDaysAD))
            _assertPred!"=="(Date(gd.day), gd.date);
    }


    /++
        Compares this $(D Date) with the given $(D Date).

        Returns:
            $(BOOKTABLE,
            $(TR $(TD this &lt; rhs) $(TD &lt; 0))
            $(TR $(TD this == rhs) $(TD 0))
            $(TR $(TD this &gt; rhs) $(TD &gt; 0))
            )
     +/
    int opCmp(in Date rhs) const pure nothrow
    {
        if(_year < rhs._year)
            return -1;
        if(_year > rhs._year)
            return 1;

        if(_month < rhs._month)
            return -1;
        if(_month > rhs._month)
            return 1;

        if(_day < rhs._day)
            return -1;
        if(_day > rhs._day)
            return 1;

        return 0;
    }

    unittest
    {
        version(testStdDateTime)
        {
            //Test A.D.
            _assertPred!("opCmp", "==")(Date(1, 1, 1), Date.init);

            _assertPred!("opCmp", "==")(Date(1999, 1, 1), Date(1999, 1, 1));
            _assertPred!("opCmp", "==")(Date(1, 7, 1), Date(1, 7, 1));
            _assertPred!("opCmp", "==")(Date(1, 1, 6), Date(1, 1, 6));

            _assertPred!("opCmp", "==")(Date(1999, 7, 1), Date(1999, 7, 1));
            _assertPred!("opCmp", "==")(Date(1999, 7, 6), Date(1999, 7, 6));

            _assertPred!("opCmp", "==")(Date(1, 7, 6), Date(1, 7, 6));

            _assertPred!("opCmp", "<")(Date(1999, 7, 6), Date(2000, 7, 6));
            _assertPred!("opCmp", ">")(Date(2000, 7, 6), Date(1999, 7, 6));
            _assertPred!("opCmp", "<")(Date(1999, 7, 6), Date(1999, 8, 6));
            _assertPred!("opCmp", ">")(Date(1999, 8, 6), Date(1999, 7, 6));
            _assertPred!("opCmp", "<")(Date(1999, 7, 6), Date(1999, 7, 7));
            _assertPred!("opCmp", ">")(Date(1999, 7, 7), Date(1999, 7, 6));

            _assertPred!("opCmp", "<")(Date(1999, 8, 7), Date(2000, 7, 6));
            _assertPred!("opCmp", ">")(Date(2000, 8, 6), Date(1999, 7, 7));
            _assertPred!("opCmp", "<")(Date(1999, 7, 7), Date(2000, 7, 6));
            _assertPred!("opCmp", ">")(Date(2000, 7, 6), Date(1999, 7, 7));
            _assertPred!("opCmp", "<")(Date(1999, 7, 7), Date(1999, 8, 6));
            _assertPred!("opCmp", ">")(Date(1999, 8, 6), Date(1999, 7, 7));

            //Test B.C.
            _assertPred!("opCmp", "==")(Date(0, 1, 1), Date(0, 1, 1));
            _assertPred!("opCmp", "==")(Date(-1, 1, 1), Date(-1, 1, 1));
            _assertPred!("opCmp", "==")(Date(-1, 7, 1), Date(-1, 7, 1));
            _assertPred!("opCmp", "==")(Date(-1, 1, 6), Date(-1, 1, 6));

            _assertPred!("opCmp", "==")(Date(-1999, 7, 1), Date(-1999, 7, 1));
            _assertPred!("opCmp", "==")(Date(-1999, 7, 6), Date(-1999, 7, 6));

            _assertPred!("opCmp", "==")(Date(-1, 7, 6), Date(-1, 7, 6));

            _assertPred!("opCmp", "<")(Date(-2000, 7, 6), Date(-1999, 7, 6));
            _assertPred!("opCmp", ">")(Date(-1999, 7, 6), Date(-2000, 7, 6));
            _assertPred!("opCmp", "<")(Date(-1999, 7, 6), Date(-1999, 8, 6));
            _assertPred!("opCmp", ">")(Date(-1999, 8, 6), Date(-1999, 7, 6));
            _assertPred!("opCmp", "<")(Date(-1999, 7, 6), Date(-1999, 7, 7));
            _assertPred!("opCmp", ">")(Date(-1999, 7, 7), Date(-1999, 7, 6));

            _assertPred!("opCmp", "<")(Date(-2000, 8, 6), Date(-1999, 7, 7));
            _assertPred!("opCmp", ">")(Date(-1999, 8, 7), Date(-2000, 7, 6));
            _assertPred!("opCmp", "<")(Date(-2000, 7, 6), Date(-1999, 7, 7));
            _assertPred!("opCmp", ">")(Date(-1999, 7, 7), Date(-2000, 7, 6));
            _assertPred!("opCmp", "<")(Date(-1999, 7, 7), Date(-1999, 8, 6));
            _assertPred!("opCmp", ">")(Date(-1999, 8, 6), Date(-1999, 7, 7));

            //Test Both
            _assertPred!("opCmp", "<")(Date(-1999, 7, 6), Date(1999, 7, 6));
            _assertPred!("opCmp", ">")(Date(1999, 7, 6), Date(-1999, 7, 6));

            _assertPred!("opCmp", "<")(Date(-1999, 8, 6), Date(1999, 7, 6));
            _assertPred!("opCmp", ">")(Date(1999, 7, 6), Date(-1999, 8, 6));

            _assertPred!("opCmp", "<")(Date(-1999, 7, 7), Date(1999, 7, 6));
            _assertPred!("opCmp", ">")(Date(1999, 7, 6), Date(-1999, 7, 7));

            _assertPred!("opCmp", "<")(Date(-1999, 8, 7), Date(1999, 7, 6));
            _assertPred!("opCmp", ">")(Date(1999, 7, 6), Date(-1999, 8, 7));

            _assertPred!("opCmp", "<")(Date(-1999, 8, 6), Date(1999, 6, 6));
            _assertPred!("opCmp", ">")(Date(1999, 6, 8), Date(-1999, 7, 6));

            auto date = Date(1999, 7, 6);
            const cdate = Date(1999, 7, 6);
            immutable idate = Date(1999, 7, 6);
            static assert(__traits(compiles, date.opCmp(date)));
            static assert(__traits(compiles, date.opCmp(cdate)));
            static assert(__traits(compiles, date.opCmp(idate)));
            static assert(__traits(compiles, cdate.opCmp(date)));
            static assert(__traits(compiles, cdate.opCmp(cdate)));
            static assert(__traits(compiles, cdate.opCmp(idate)));
            static assert(__traits(compiles, idate.opCmp(date)));
            static assert(__traits(compiles, idate.opCmp(cdate)));
            static assert(__traits(compiles, idate.opCmp(idate)));
        }
    }


    /++
        Year of the Gregorian Calendar. Positive numbers are A.D. Non-positive
        are B.C.

        Examples:
--------------------
assert(Date(1999, 7, 6).year == 1999);
assert(Date(2010, 10, 4).year == 2010);
assert(Date(-7, 4, 5).year == -7);
--------------------
     +/
    @property short year() const pure nothrow
    {
        return _year;
    }

    unittest
    {
        version(testStdDateTime)
        {
            _assertPred!"=="(Date.init.year, 1);
            _assertPred!"=="(Date(1999, 7, 6).year, 1999);
            _assertPred!"=="(Date(-1999, 7, 6).year, -1999);

            const cdate = Date(1999, 7, 6);
            immutable idate = Date(1999, 7, 6);
            static assert(__traits(compiles, cdate.year == 1999));
            static assert(__traits(compiles, idate.year == 1999));

            //Verify Examples.
            assert(Date(1999, 7, 6).year == 1999);
            assert(Date(2010, 10, 4).year == 2010);
            assert(Date(-7, 4, 5).year == -7);
        }
    }

    /++
        Year of the Gregorian Calendar. Positive numbers are A.D. Non-positive
        are B.C.

        Params:
            year = The year to set this Date's year to.

        Throws:
            $(D DateTimeException) if the new year is not a leap year and the
            resulting date would be on February 29th.
     +/
    @property void year(int year) pure
    {
        enforceValid!"days"(year, _month, _day);
        _year = cast(short)year;
    }

    unittest
    {
        version(testStdDateTime)
        {
            static void testDateInvalid(Date date, int year)
            {
                date.year = year;
            }

            static void testDate(Date date, int year, in Date expected, size_t line = __LINE__)
            {
                date.year = year;
                _assertPred!"=="(date, expected, "", __FILE__, line);
            }

            assertThrown!DateTimeException(testDateInvalid(Date(4, 2, 29), 1));

            testDate(Date(1, 1, 1), 1999, Date(1999, 1, 1));
            testDate(Date(1, 1, 1), 0, Date(0, 1, 1));
            testDate(Date(1, 1, 1), -1999, Date(-1999, 1, 1));

            const cdate = Date(1999, 7, 6);
            immutable idate = Date(1999, 7, 6);
            static assert(!__traits(compiles, cdate.year = 1999));
            static assert(!__traits(compiles, idate.year = 1999));

            //Verify Examples.
            assert(Date(1999, 7, 6).year == 1999);
            assert(Date(2010, 10, 4).year == 2010);
            assert(Date(-7, 4, 5).year == -7);
        }
    }


    /++
        Year B.C. of the Gregorian Calendar counting year 0 as 1 B.C.

        Throws:
            $(D DateTimeException) if $(D isAD) is true.

        Examples:
--------------------
assert(Date(0, 1, 1).yearBC == 1);
assert(Date(-1, 1, 1).yearBC == 2);
assert(Date(-100, 1, 1).yearBC == 101);
--------------------
     +/
    @property ushort yearBC() const pure
    {
        if(isAD)
            throw new DateTimeException("Year " ~ numToString(_year) ~ " is A.D.");
            //Once format is pure, this would be a better error message.
            //throw new DateTimeException(format("%s is A.D.", this));

        return cast(ushort)((_year * -1) + 1);
    }

    unittest
    {
        version(testStdDateTime)
        {
            assertThrown!DateTimeException((in Date date){date.yearBC;}(Date(1, 1, 1)));

            auto date = Date(0, 7, 6);
            const cdate = Date(0, 7, 6);
            immutable idate = Date(0, 7, 6);
            static assert(__traits(compiles, date.yearBC));
            static assert(__traits(compiles, cdate.yearBC));
            static assert(__traits(compiles, idate.yearBC));

            //Verify Examples.
            assert(Date(0, 1, 1).yearBC == 1);
            assert(Date(-1, 1, 1).yearBC == 2);
            assert(Date(-100, 1, 1).yearBC == 101);
        }
    }


    /++
        Year B.C. of the Gregorian Calendar counting year 0 as 1 B.C.

        Params:
            year = The year B.C. to set this $(D Date)'s year to.

        Throws:
            $(D DateTimeException) if a non-positive value is given.

        Examples:
--------------------
auto date = Date(2010, 1, 1);
date.yearBC = 1;
assert(date == Date(0, 1, 1));

date.yearBC = 10;
assert(date == Date(-9, 1, 1));
--------------------
     +/
    @property void yearBC(int year) pure
    {
        if(year <= 0)
            throw new DateTimeException("The given year is not a year B.C.");

        _year = cast(short)((year - 1) * -1);
    }

    unittest
    {
        version(testStdDateTime)
        {
            assertThrown!DateTimeException((Date date){date.yearBC = -1;}(Date(1, 1, 1)));

            {
                auto date = Date(0, 7, 6);
                const cdate = Date(0, 7, 6);
                immutable idate = Date(0, 7, 6);
                static assert(__traits(compiles, date.yearBC = 7));
                static assert(!__traits(compiles, cdate.yearBC = 7));
                static assert(!__traits(compiles, idate.yearBC = 7));
            }

            //Verify Examples.
            {
                auto date = Date(2010, 1, 1);
                date.yearBC = 1;
                assert(date == Date(0, 1, 1));

                date.yearBC = 10;
                assert(date == Date(-9, 1, 1));
            }
        }
    }


    /++
        Month of a Gregorian Year.

        Examples:
--------------------
assert(Date(1999, 7, 6).month == 7);
assert(Date(2010, 10, 4).month == 10);
assert(Date(-7, 4, 5).month == 4);
--------------------
     +/
    @property Month month() const pure nothrow
    {
        return _month;
    }

    unittest
    {
        version(testStdDateTime)
        {
            _assertPred!"=="(Date.init.month, 1);
            _assertPred!"=="(Date(1999, 7, 6).month, 7);
            _assertPred!"=="(Date(-1999, 7, 6).month, 7);

            const cdate = Date(1999, 7, 6);
            immutable idate = Date(1999, 7, 6);
            static assert(__traits(compiles, cdate.month == 7));
            static assert(__traits(compiles, idate.month == 7));

            //Verify Examples.
            assert(Date(1999, 7, 6).month == 7);
            assert(Date(2010, 10, 4).month == 10);
            assert(Date(-7, 4, 5).month == 4);
        }
    }

    /++
        Month of a Gregorian Year.

        Params:
            month = The month to set this $(D Date)'s month to.

        Throws:
            $(D DateTimeException) if the given month is not a valid month or if
            the current day would not be valid in the given month.
     +/
    @property void month(Month month) pure
    {
        enforceValid!"months"(month);
        enforceValid!"days"(_year, month, _day);
        _month = cast(Month)month;
    }

    unittest
    {
        version(testStdDateTime)
        {
            static void testDate(Date date, Month month, in Date expected = Date.init, size_t line = __LINE__)
            {
                date.month = month;
                assert(expected != Date.init);
                _assertPred!"=="(date, expected, "", __FILE__, line);
            }

            assertThrown!DateTimeException(testDate(Date(1, 1, 1), cast(Month)0));
            assertThrown!DateTimeException(testDate(Date(1, 1, 1), cast(Month)13));
            assertThrown!DateTimeException(testDate(Date(1, 1, 29), cast(Month)2));
            assertThrown!DateTimeException(testDate(Date(0, 1, 30), cast(Month)2));

            testDate(Date(1, 1, 1), cast(Month)7, Date(1, 7, 1));
            testDate(Date(-1, 1, 1), cast(Month)7, Date(-1, 7, 1));

            const cdate = Date(1999, 7, 6);
            immutable idate = Date(1999, 7, 6);
            static assert(!__traits(compiles, cdate.month = 7));
            static assert(!__traits(compiles, idate.month = 7));
        }
    }


    /++
        Day of a Gregorian Month.

        Examples:
--------------------
assert(Date(1999, 7, 6).day == 6);
assert(Date(2010, 10, 4).day == 4);
assert(Date(-7, 4, 5).day == 5);
--------------------
     +/
    @property ubyte day() const pure nothrow
    {
        return _day;
    }

    //Verify Examples.
    version(testStdDateTime) unittest
    {
        assert(Date(1999, 7, 6).day == 6);
        assert(Date(2010, 10, 4).day == 4);
        assert(Date(-7, 4, 5).day == 5);
    }

    version(testStdDateTime) unittest
    {
        static void test(Date date, int expected, size_t line = __LINE__)
        {
            _assertPred!"=="(date.day, expected,
                             format("Value given: %s", date), __FILE__, line);
        }

        foreach(year; chain(testYearsBC, testYearsAD))
        {
            foreach(md; testMonthDays)
                test(Date(year, md.month, md.day), md.day);
        }

        const cdate = Date(1999, 7, 6);
        immutable idate = Date(1999, 7, 6);
        static assert(__traits(compiles, cdate.day == 6));
        static assert(__traits(compiles, idate.day == 6));
    }

    /++
        Day of a Gregorian Month.

        Params:
            day = The day of the month to set this $(D Date)'s day to.

        Throws:
            $(D DateTimeException) if the given day is not a valid day of the
            current month.
     +/
    @property void day(int day) pure
    {
        enforceValid!"days"(_year, _month, day);
        _day = cast(ubyte)day;
    }

    unittest
    {
        version(testStdDateTime)
        {
            static void testDate(Date date, int day)
            {
                date.day = day;
            }

            //Test A.D.
            assertThrown!DateTimeException(testDate(Date(1, 1, 1), 0));
            assertThrown!DateTimeException(testDate(Date(1, 1, 1), 32));
            assertThrown!DateTimeException(testDate(Date(1, 2, 1), 29));
            assertThrown!DateTimeException(testDate(Date(4, 2, 1), 30));
            assertThrown!DateTimeException(testDate(Date(1, 3, 1), 32));
            assertThrown!DateTimeException(testDate(Date(1, 4, 1), 31));
            assertThrown!DateTimeException(testDate(Date(1, 5, 1), 32));
            assertThrown!DateTimeException(testDate(Date(1, 6, 1), 31));
            assertThrown!DateTimeException(testDate(Date(1, 7, 1), 32));
            assertThrown!DateTimeException(testDate(Date(1, 8, 1), 32));
            assertThrown!DateTimeException(testDate(Date(1, 9, 1), 31));
            assertThrown!DateTimeException(testDate(Date(1, 10, 1), 32));
            assertThrown!DateTimeException(testDate(Date(1, 11, 1), 31));
            assertThrown!DateTimeException(testDate(Date(1, 12, 1), 32));

            assertNotThrown!DateTimeException(testDate(Date(1, 1, 1), 31));
            assertNotThrown!DateTimeException(testDate(Date(1, 2, 1), 28));
            assertNotThrown!DateTimeException(testDate(Date(4, 2, 1), 29));
            assertNotThrown!DateTimeException(testDate(Date(1, 3, 1), 31));
            assertNotThrown!DateTimeException(testDate(Date(1, 4, 1), 30));
            assertNotThrown!DateTimeException(testDate(Date(1, 5, 1), 31));
            assertNotThrown!DateTimeException(testDate(Date(1, 6, 1), 30));
            assertNotThrown!DateTimeException(testDate(Date(1, 7, 1), 31));
            assertNotThrown!DateTimeException(testDate(Date(1, 8, 1), 31));
            assertNotThrown!DateTimeException(testDate(Date(1, 9, 1), 30));
            assertNotThrown!DateTimeException(testDate(Date(1, 10, 1), 31));
            assertNotThrown!DateTimeException(testDate(Date(1, 11, 1), 30));
            assertNotThrown!DateTimeException(testDate(Date(1, 12, 1), 31));

            {
                auto date = Date(1, 1, 1);
                date.day = 6;
                _assertPred!"=="(date, Date(1, 1, 6));
            }

            //Test B.C.
            assertThrown!DateTimeException(testDate(Date(-1, 1, 1), 0));
            assertThrown!DateTimeException(testDate(Date(-1, 1, 1), 32));
            assertThrown!DateTimeException(testDate(Date(-1, 2, 1), 29));
            assertThrown!DateTimeException(testDate(Date(0, 2, 1), 30));
            assertThrown!DateTimeException(testDate(Date(-1, 3, 1), 32));
            assertThrown!DateTimeException(testDate(Date(-1, 4, 1), 31));
            assertThrown!DateTimeException(testDate(Date(-1, 5, 1), 32));
            assertThrown!DateTimeException(testDate(Date(-1, 6, 1), 31));
            assertThrown!DateTimeException(testDate(Date(-1, 7, 1), 32));
            assertThrown!DateTimeException(testDate(Date(-1, 8, 1), 32));
            assertThrown!DateTimeException(testDate(Date(-1, 9, 1), 31));
            assertThrown!DateTimeException(testDate(Date(-1, 10, 1), 32));
            assertThrown!DateTimeException(testDate(Date(-1, 11, 1), 31));
            assertThrown!DateTimeException(testDate(Date(-1, 12, 1), 32));

            assertNotThrown!DateTimeException(testDate(Date(-1, 1, 1), 31));
            assertNotThrown!DateTimeException(testDate(Date(-1, 2, 1), 28));
            assertNotThrown!DateTimeException(testDate(Date(0, 2, 1), 29));
            assertNotThrown!DateTimeException(testDate(Date(-1, 3, 1), 31));
            assertNotThrown!DateTimeException(testDate(Date(-1, 4, 1), 30));
            assertNotThrown!DateTimeException(testDate(Date(-1, 5, 1), 31));
            assertNotThrown!DateTimeException(testDate(Date(-1, 6, 1), 30));
            assertNotThrown!DateTimeException(testDate(Date(-1, 7, 1), 31));
            assertNotThrown!DateTimeException(testDate(Date(-1, 8, 1), 31));
            assertNotThrown!DateTimeException(testDate(Date(-1, 9, 1), 30));
            assertNotThrown!DateTimeException(testDate(Date(-1, 10, 1), 31));
            assertNotThrown!DateTimeException(testDate(Date(-1, 11, 1), 30));
            assertNotThrown!DateTimeException(testDate(Date(-1, 12, 1), 31));

            {
                auto date = Date(-1, 1, 1);
                date.day = 6;
                _assertPred!"=="(date, Date(-1, 1, 6));
            }

            const cdate = Date(1999, 7, 6);
            immutable idate = Date(1999, 7, 6);
            static assert(!__traits(compiles, cdate.day = 6));
            static assert(!__traits(compiles, idate.day = 6));
        }
    }


    /++
        Adds the given number of years or months to this $(D Date). A negative
        number will subtract.

        Note that if day overflow is allowed, and the date with the adjusted
        year/month overflows the number of days in the new month, then the month
        will be incremented by one, and the day set to the number of days
        overflowed. (e.g. if the day were 31 and the new month were June, then
        the month would be incremented to July, and the new day would be 1). If
        day overflow is not allowed, then the day will be set to the last valid
        day in the month (e.g. June 31st would become June 30th).

        Params:
            units         = The type of units to add ("years" or "months").
            value         = The number of months or years to add to this
                            $(D Date).
            allowOverflow = Whether the day should be allowed to overflow,
                            causing the month to increment.

        Examples:
--------------------
auto d1 = Date(2010, 1, 1);
d1.add!"months"(11);
assert(d1 == Date(2010, 12, 1));

auto d2 = Date(2010, 1, 1);
d2.add!"months"(-11);
assert(d2 == Date(2009, 2, 1));

auto d3 = Date(2000, 2, 29);
d3.add!"years"(1);
assert(d3 == Date(2001, 3, 1));

auto d4 = Date(2000, 2, 29);
d4.add!"years"(1, AllowDayOverflow.no);
assert(d4 == Date(2001, 2, 28));
--------------------
      +/
    /+ref Date+/ void add(string units)(long value, AllowDayOverflow allowOverflow = AllowDayOverflow.yes) pure nothrow
        if(units == "years")
    {
        immutable newYear = _year + value;

        _year += value;

        if(_month == Month.feb && _day == 29 && !yearIsLeapYear(_year))
        {
            if(allowOverflow == AllowDayOverflow.yes)
            {
                _month = Month.mar;
                _day = 1;
            }
            else
                _day = 28;
        }
    }

    //Verify Examples.
    unittest
    {
        version(stdStdDateTime)
        {
            auto d1 = Date(2010, 1, 1);
            d1.add!"months"(11);
            assert(d1 == Date(2010, 12, 1));

            auto d2 = Date(2010, 1, 1);
            d2.add!"months"(-11);
            assert(d2 == Date(2009, 2, 1));

            auto d3 = Date(2000, 2, 29);
            d3.add!"years"(1);
            assert(d3 == Date(2001, 3, 1));

            auto d4 = Date(2000, 2, 29);
            d4.add!"years"(1, AllowDayOverflow.no);
            assert(d4 == Date(2001, 2, 28));
        }
    }

    //Test add!"years"() with AllowDayOverlow.yes
    unittest
    {
        version(testStdDateTime)
        {
            //Test A.D.
            {
                auto date = Date(1999, 7, 6);
                date.add!"years"(7);
                _assertPred!"=="(date, Date(2006, 7, 6));
                date.add!"years"(-9);
                _assertPred!"=="(date, Date(1997, 7, 6));
            }

            {
                auto date = Date(1999, 2, 28);
                date.add!"years"(1);
                _assertPred!"=="(date, Date(2000, 2, 28));
            }

            {
                auto date = Date(2000, 2, 29);
                date.add!"years"(-1);
                _assertPred!"=="(date, Date(1999, 3, 1));
            }

            //Test B.C.
            {
                auto date = Date(-1999, 7, 6);
                date.add!"years"(-7);
                _assertPred!"=="(date, Date(-2006, 7, 6));
                date.add!"years"(9);
                _assertPred!"=="(date, Date(-1997, 7, 6));
            }

            {
                auto date = Date(-1999, 2, 28);
                date.add!"years"(-1);
                _assertPred!"=="(date, Date(-2000, 2, 28));
            }

            {
                auto date = Date(-2000, 2, 29);
                date.add!"years"(1);
                _assertPred!"=="(date, Date(-1999, 3, 1));
            }

            //Test Both
            {
                auto date = Date(4, 7, 6);
                date.add!"years"(-5);
                _assertPred!"=="(date, Date(-1, 7, 6));
                date.add!"years"(5);
                _assertPred!"=="(date, Date(4, 7, 6));
            }

            {
                auto date = Date(-4, 7, 6);
                date.add!"years"(5);
                _assertPred!"=="(date, Date(1, 7, 6));
                date.add!"years"(-5);
                _assertPred!"=="(date, Date(-4, 7, 6));
            }

            {
                auto date = Date(4, 7, 6);
                date.add!"years"(-8);
                _assertPred!"=="(date, Date(-4, 7, 6));
                date.add!"years"(8);
                _assertPred!"=="(date, Date(4, 7, 6));
            }

            {
                auto date = Date(-4, 7, 6);
                date.add!"years"(8);
                _assertPred!"=="(date, Date(4, 7, 6));
                date.add!"years"(-8);
                _assertPred!"=="(date, Date(-4, 7, 6));
            }

            {
                auto date = Date(-4, 2, 29);
                date.add!"years"(5);
                _assertPred!"=="(date, Date(1, 3, 1));
            }

            {
                auto date = Date(4, 2, 29);
                date.add!"years"(-5);
                _assertPred!"=="(date, Date(-1, 3, 1));
            }

            const cdate = Date(1999, 7, 6);
            immutable idate = Date(1999, 7, 6);
            static assert(!__traits(compiles, cdate.add!"years"(7)));
            static assert(!__traits(compiles, idate.add!"years"(7)));
        }
    }

    //Test add!"years"() with AllowDayOverlow.no
    unittest
    {
        version(testStdDateTime)
        {
            //Test A.D.
            {
                auto date = Date(1999, 7, 6);
                date.add!"years"(7, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(2006, 7, 6));
                date.add!"years"(-9, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(1997, 7, 6));
            }

            {
                auto date = Date(1999, 2, 28);
                date.add!"years"(1, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(2000, 2, 28));
            }

            {
                auto date = Date(2000, 2, 29);
                date.add!"years"(-1, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(1999, 2, 28));
            }

            //Test B.C.
            {
                auto date = Date(-1999, 7, 6);
                date.add!"years"(-7, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-2006, 7, 6));
                date.add!"years"(9, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-1997, 7, 6));
            }

            {
                auto date = Date(-1999, 2, 28);
                date.add!"years"(-1, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-2000, 2, 28));
            }

            {
                auto date = Date(-2000, 2, 29);
                date.add!"years"(1, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-1999, 2, 28));
            }

            //Test Both
            {
                auto date = Date(4, 7, 6);
                date.add!"years"(-5, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-1, 7, 6));
                date.add!"years"(5, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(4, 7, 6));
            }

            {
                auto date = Date(-4, 7, 6);
                date.add!"years"(5, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(1, 7, 6));
                date.add!"years"(-5, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-4, 7, 6));
            }

            {
                auto date = Date(4, 7, 6);
                date.add!"years"(-8, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-4, 7, 6));
                date.add!"years"(8, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(4, 7, 6));
            }

            {
                auto date = Date(-4, 7, 6);
                date.add!"years"(8, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(4, 7, 6));
                date.add!"years"(-8, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-4, 7, 6));
            }

            {
                auto date = Date(-4, 2, 29);
                date.add!"years"(5, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(1, 2, 28));
            }

            {
                auto date = Date(4, 2, 29);
                date.add!"years"(-5, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-1, 2, 28));
            }
        }
    }


    //Shares documentation with "years" version.
    /+ref Date+/ void add(string units)(long months, AllowDayOverflow allowOverflow = AllowDayOverflow.yes) pure nothrow
        if(units == "months")
    {
        auto years = months / 12;
        months %= 12;
        auto newMonth = _month + months;

        if(months < 0)
        {
            if(newMonth < 1)
            {
                newMonth += 12;
                --years;
            }
        }
        else if(newMonth > 12)
        {
            newMonth -= 12;
            ++years;
        }

        _year += years;
        _month = cast(Month)newMonth;

        immutable currMaxDay = maxDay(_year, _month);
        immutable overflow = _day - currMaxDay;

        if(overflow > 0)
        {
            if(allowOverflow == AllowDayOverflow.yes)
            {
                ++_month;
                _day = cast(ubyte)overflow;
            }
            else
                _day = cast(ubyte)currMaxDay;
        }
    }

    //Test add!"months"() with AllowDayOverlow.yes
    unittest
    {
        version(testStdDateTime)
        {
            //Test A.D.
            {
                auto date = Date(1999, 7, 6);
                date.add!"months"(3);
                _assertPred!"=="(date, Date(1999, 10, 6));
                date.add!"months"(-4);
                _assertPred!"=="(date, Date(1999, 6, 6));
            }

            {
                auto date = Date(1999, 7, 6);
                date.add!"months"(6);
                _assertPred!"=="(date, Date(2000, 1, 6));
                date.add!"months"(-6);
                _assertPred!"=="(date, Date(1999, 7, 6));
            }

            {
                auto date = Date(1999, 7, 6);
                date.add!"months"(27);
                _assertPred!"=="(date, Date(2001, 10, 6));
                date.add!"months"(-28);
                _assertPred!"=="(date, Date(1999, 6, 6));
            }

            {
                auto date = Date(1999, 5, 31);
                date.add!"months"(1);
                _assertPred!"=="(date, Date(1999, 7, 1));
            }

            {
                auto date = Date(1999, 5, 31);
                date.add!"months"(-1);
                _assertPred!"=="(date, Date(1999, 5, 1));
            }

            {
                auto date = Date(1999, 2, 28);
                date.add!"months"(12);
                _assertPred!"=="(date, Date(2000, 2, 28));
            }

            {
                auto date = Date(2000, 2, 29);
                date.add!"months"(12);
                _assertPred!"=="(date, Date(2001, 3, 1));
            }

            {
                auto date = Date(1999, 7, 31);
                date.add!"months"(1);
                _assertPred!"=="(date, Date(1999, 8, 31));
                date.add!"months"(1);
                _assertPred!"=="(date, Date(1999, 10, 1));
            }

            {
                auto date = Date(1998, 8, 31);
                date.add!"months"(13);
                _assertPred!"=="(date, Date(1999, 10, 1));
                date.add!"months"(-13);
                _assertPred!"=="(date, Date(1998, 9, 1));
            }

            {
                auto date = Date(1997, 12, 31);
                date.add!"months"(13);
                _assertPred!"=="(date, Date(1999, 1, 31));
                date.add!"months"(-13);
                _assertPred!"=="(date, Date(1997, 12, 31));
            }

            {
                auto date = Date(1997, 12, 31);
                date.add!"months"(14);
                _assertPred!"=="(date, Date(1999, 3, 3));
                date.add!"months"(-14);
                _assertPred!"=="(date, Date(1998, 1, 3));
            }

            {
                auto date = Date(1998, 12, 31);
                date.add!"months"(14);
                _assertPred!"=="(date, Date(2000, 3, 2));
                date.add!"months"(-14);
                _assertPred!"=="(date, Date(1999, 1, 2));
            }

            {
                auto date = Date(1999, 12, 31);
                date.add!"months"(14);
                _assertPred!"=="(date, Date(2001, 3, 3));
                date.add!"months"(-14);
                _assertPred!"=="(date, Date(2000, 1, 3));
            }

            //Test B.C.
            {
                auto date = Date(-1999, 7, 6);
                date.add!"months"(3);
                _assertPred!"=="(date, Date(-1999, 10, 6));
                date.add!"months"(-4);
                _assertPred!"=="(date, Date(-1999, 6, 6));
            }

            {
                auto date = Date(-1999, 7, 6);
                date.add!"months"(6);
                _assertPred!"=="(date, Date(-1998, 1, 6));
                date.add!"months"(-6);
                _assertPred!"=="(date, Date(-1999, 7, 6));
            }

            {
                auto date = Date(-1999, 7, 6);
                date.add!"months"(-27);
                _assertPred!"=="(date, Date(-2001, 4, 6));
                date.add!"months"(28);
                _assertPred!"=="(date, Date(-1999, 8, 6));
            }

            {
                auto date = Date(-1999, 5, 31);
                date.add!"months"(1);
                _assertPred!"=="(date, Date(-1999, 7, 1));
            }

            {
                auto date = Date(-1999, 5, 31);
                date.add!"months"(-1);
                _assertPred!"=="(date, Date(-1999, 5, 1));
            }

            {
                auto date = Date(-1999, 2, 28);
                date.add!"months"(-12);
                _assertPred!"=="(date, Date(-2000, 2, 28));
            }

            {
                auto date = Date(-2000, 2, 29);
                date.add!"months"(-12);
                _assertPred!"=="(date, Date(-2001, 3, 1));
            }

            {
                auto date = Date(-1999, 7, 31);
                date.add!"months"(1);
                _assertPred!"=="(date, Date(-1999, 8, 31));
                date.add!"months"(1);
                _assertPred!"=="(date, Date(-1999, 10, 1));
            }

            {
                auto date = Date(-1998, 8, 31);
                date.add!"months"(13);
                _assertPred!"=="(date, Date(-1997, 10, 1));
                date.add!"months"(-13);
                _assertPred!"=="(date, Date(-1998, 9, 1));
            }

            {
                auto date = Date(-1997, 12, 31);
                date.add!"months"(13);
                _assertPred!"=="(date, Date(-1995, 1, 31));
                date.add!"months"(-13);
                _assertPred!"=="(date, Date(-1997, 12, 31));
            }

            {
                auto date = Date(-1997, 12, 31);
                date.add!"months"(14);
                _assertPred!"=="(date, Date(-1995, 3, 3));
                date.add!"months"(-14);
                _assertPred!"=="(date, Date(-1996, 1, 3));
            }

            {
                auto date = Date(-2002, 12, 31);
                date.add!"months"(14);
                _assertPred!"=="(date, Date(-2000, 3, 2));
                date.add!"months"(-14);
                _assertPred!"=="(date, Date(-2001, 1, 2));
            }

            {
                auto date = Date(-2001, 12, 31);
                date.add!"months"(14);
                _assertPred!"=="(date, Date(-1999, 3, 3));
                date.add!"months"(-14);
                _assertPred!"=="(date, Date(-2000, 1, 3));
            }

            //Test Both
            {
                auto date = Date(1, 1, 1);
                date.add!"months"(-1);
                _assertPred!"=="(date, Date(0, 12, 1));
                date.add!"months"(1);
                _assertPred!"=="(date, Date(1, 1, 1));
            }

            {
                auto date = Date(4, 1, 1);
                date.add!"months"(-48);
                _assertPred!"=="(date, Date(0, 1, 1));
                date.add!"months"(48);
                _assertPred!"=="(date, Date(4, 1, 1));
            }

            {
                auto date = Date(4, 3, 31);
                date.add!"months"(-49);
                _assertPred!"=="(date, Date(0, 3, 2));
                date.add!"months"(49);
                _assertPred!"=="(date, Date(4, 4, 2));
            }

            {
                auto date = Date(4, 3, 31);
                date.add!"months"(-85);
                _assertPred!"=="(date, Date(-3, 3, 3));
                date.add!"months"(85);
                _assertPred!"=="(date, Date(4, 4, 3));
            }

            const cdate = Date(1999, 7, 6);
            immutable idate = Date(1999, 7, 6);
            static assert(!__traits(compiles, cdate.add!"months"(3)));
            static assert(!__traits(compiles, idate.add!"months"(3)));
        }
    }

    //Test add!"months"() with AllowDayOverlow.no
    unittest
    {
        version(testStdDateTime)
        {
            //Test A.D.
            {
                auto date = Date(1999, 7, 6);
                date.add!"months"(3, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(1999, 10, 6));
                date.add!"months"(-4, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(1999, 6, 6));
            }

            {
                auto date = Date(1999, 7, 6);
                date.add!"months"(6, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(2000, 1, 6));
                date.add!"months"(-6, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(1999, 7, 6));
            }

            {
                auto date = Date(1999, 7, 6);
                date.add!"months"(27, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(2001, 10, 6));
                date.add!"months"(-28, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(1999, 6, 6));
            }

            {
                auto date = Date(1999, 5, 31);
                date.add!"months"(1, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(1999, 6, 30));
            }

            {
                auto date = Date(1999, 5, 31);
                date.add!"months"(-1, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(1999, 4, 30));
            }

            {
                auto date = Date(1999, 2, 28);
                date.add!"months"(12, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(2000, 2, 28));
            }

            {
                auto date = Date(2000, 2, 29);
                date.add!"months"(12, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(2001, 2, 28));
            }

            {
                auto date = Date(1999, 7, 31);
                date.add!"months"(1, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(1999, 8, 31));
                date.add!"months"(1, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(1999, 9, 30));
            }

            {
                auto date = Date(1998, 8, 31);
                date.add!"months"(13, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(1999, 9, 30));
                date.add!"months"(-13, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(1998, 8, 30));
            }

            {
                auto date = Date(1997, 12, 31);
                date.add!"months"(13, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(1999, 1, 31));
                date.add!"months"(-13, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(1997, 12, 31));
            }

            {
                auto date = Date(1997, 12, 31);
                date.add!"months"(14, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(1999, 2, 28));
                date.add!"months"(-14, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(1997, 12, 28));
            }

            {
                auto date = Date(1998, 12, 31);
                date.add!"months"(14, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(2000, 2, 29));
                date.add!"months"(-14, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(1998, 12, 29));
            }

            {
                auto date = Date(1999, 12, 31);
                date.add!"months"(14, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(2001, 2, 28));
                date.add!"months"(-14, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(1999, 12, 28));
            }

            //Test B.C.
            {
                auto date = Date(-1999, 7, 6);
                date.add!"months"(3, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-1999, 10, 6));
                date.add!"months"(-4, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-1999, 6, 6));
            }

            {
                auto date = Date(-1999, 7, 6);
                date.add!"months"(6, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-1998, 1, 6));
                date.add!"months"(-6, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-1999, 7, 6));
            }

            {
                auto date = Date(-1999, 7, 6);
                date.add!"months"(-27, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-2001, 4, 6));
                date.add!"months"(28, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-1999, 8, 6));
            }

            {
                auto date = Date(-1999, 5, 31);
                date.add!"months"(1, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-1999, 6, 30));
            }

            {
                auto date = Date(-1999, 5, 31);
                date.add!"months"(-1, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-1999, 4, 30));
            }

            {
                auto date = Date(-1999, 2, 28);
                date.add!"months"(-12, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-2000, 2, 28));
            }

            {
                auto date = Date(-2000, 2, 29);
                date.add!"months"(-12, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-2001, 2, 28));
            }

            {
                auto date = Date(-1999, 7, 31);
                date.add!"months"(1, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-1999, 8, 31));
                date.add!"months"(1, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-1999, 9, 30));
            }

            {
                auto date = Date(-1998, 8, 31);
                date.add!"months"(13, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-1997, 9, 30));
                date.add!"months"(-13, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-1998, 8, 30));
            }

            {
                auto date = Date(-1997, 12, 31);
                date.add!"months"(13, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-1995, 1, 31));
                date.add!"months"(-13, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-1997, 12, 31));
            }

            {
                auto date = Date(-1997, 12, 31);
                date.add!"months"(14, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-1995, 2, 28));
                date.add!"months"(-14, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-1997, 12, 28));
            }

            {
                auto date = Date(-2002, 12, 31);
                date.add!"months"(14, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-2000, 2, 29));
                date.add!"months"(-14, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-2002, 12, 29));
            }

            {
                auto date = Date(-2001, 12, 31);
                date.add!"months"(14, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-1999, 2, 28));
                date.add!"months"(-14, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-2001, 12, 28));
            }

            //Test Both
            {
                auto date = Date(1, 1, 1);
                date.add!"months"(-1, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(0, 12, 1));
                date.add!"months"(1, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(1, 1, 1));
            }

            {
                auto date = Date(4, 1, 1);
                date.add!"months"(-48, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(0, 1, 1));
                date.add!"months"(48, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(4, 1, 1));
            }

            {
                auto date = Date(4, 3, 31);
                date.add!"months"(-49, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(0, 2, 29));
                date.add!"months"(49, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(4, 3, 29));
            }

            {
                auto date = Date(4, 3, 31);
                date.add!"months"(-85, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-3, 2, 28));
                date.add!"months"(85, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(4, 3, 28));
            }
        }
    }


    /++
        Adds the given number of years or months to this $(D Date). A negative
        number will subtract.

        The difference between rolling and adding is that rolling does not
        affect larger units. Rolling a $(D Date) 12 months gets
        the exact same $(D Date). However, the days can still be affected due to
        the differing number of days in each month.

        Because there are no units larger than years, there is no difference
        between adding and rolling years.

        Params:
            units         = The type of units to add ("years" or "months").
            value         = The number of months or years to add to this
                            $(D Date).

        Examples:
--------------------
auto d1 = Date(2010, 1, 1);
d1.roll!"months"(1);
assert(d1 == Date(2010, 2, 1));

auto d2 = Date(2010, 1, 1);
d2.roll!"months"(-1);
assert(d2 == Date(2010, 12, 1));

auto d3 = Date(1999, 1, 29);
d3.roll!"months"(1);
assert(d3 == Date(1999, 3, 1));

auto d4 = Date(1999, 1, 29);
d4.roll!"months"(1, AllowDayOverflow.no);
assert(d4 == Date(1999, 2, 28));

auto d5 = Date(2000, 2, 29);
d5.roll!"years"(1);
assert(d5 == Date(2001, 3, 1));

auto d6 = Date(2000, 2, 29);
d6.roll!"years"(1, AllowDayOverflow.no);
assert(d6 == Date(2001, 2, 28));
--------------------
      +/
    /+ref Date+/ void roll(string units)(long value, AllowDayOverflow allowOverflow = AllowDayOverflow.yes) pure nothrow
        if(units == "years")
    {
        add!"years"(value, allowOverflow);
    }

    //Verify Examples.
    unittest
    {
        version(testStdDateTime)
        {
            auto d1 = Date(2010, 1, 1);
            d1.roll!"months"(1);
            assert(d1 == Date(2010, 2, 1));

            auto d2 = Date(2010, 1, 1);
            d2.roll!"months"(-1);
            assert(d2 == Date(2010, 12, 1));

            auto d3 = Date(1999, 1, 29);
            d3.roll!"months"(1);
            assert(d3 == Date(1999, 3, 1));

            auto d4 = Date(1999, 1, 29);
            d4.roll!"months"(1, AllowDayOverflow.no);
            assert(d4 == Date(1999, 2, 28));

            auto d5 = Date(2000, 2, 29);
            d5.roll!"years"(1);
            assert(d5 == Date(2001, 3, 1));

            auto d6 = Date(2000, 2, 29);
            d6.roll!"years"(1, AllowDayOverflow.no);
            assert(d6 == Date(2001, 2, 28));
        }
    }

    unittest
    {
        version(testStdDateTime)
        {
            const cdate = Date(1999, 7, 6);
            immutable idate = Date(1999, 7, 6);
            static assert(!__traits(compiles, cdate.roll!"years"(3)));
            static assert(!__traits(compiles, idate.rolYears(3)));
        }
    }


    //Shares documentation with "years" version.
    /+ref Date+/ void roll(string units)(long months, AllowDayOverflow allowOverflow = AllowDayOverflow.yes) pure nothrow
        if(units == "months")
    {
        months %= 12;
        auto newMonth = _month + months;

        if(months < 0)
        {
            if(newMonth < 1)
                newMonth += 12;
        }
        else
        {
            if(newMonth > 12)
                newMonth -= 12;
        }

        _month = cast(Month)newMonth;

        immutable currMaxDay = maxDay(_year, _month);
        immutable overflow = _day - currMaxDay;

        if(overflow > 0)
        {
            if(allowOverflow == AllowDayOverflow.yes)
            {
                ++_month;
                _day = cast(ubyte)overflow;
            }
            else
                _day = cast(ubyte)currMaxDay;
        }
    }

    //Test roll!"months"() with AllowDayOverlow.yes
    unittest
    {
        version(testStdDateTime)
        {
            //Test A.D.
            {
                auto date = Date(1999, 7, 6);
                date.roll!"months"(3);
                _assertPred!"=="(date, Date(1999, 10, 6));
                date.roll!"months"(-4);
                _assertPred!"=="(date, Date(1999, 6, 6));
            }

            {
                auto date = Date(1999, 7, 6);
                date.roll!"months"(6);
                _assertPred!"=="(date, Date(1999, 1, 6));
                date.roll!"months"(-6);
                _assertPred!"=="(date, Date(1999, 7, 6));
            }

            {
                auto date = Date(1999, 7, 6);
                date.roll!"months"(27);
                _assertPred!"=="(date, Date(1999, 10, 6));
                date.roll!"months"(-28);
                _assertPred!"=="(date, Date(1999, 6, 6));
            }

            {
                auto date = Date(1999, 5, 31);
                date.roll!"months"(1);
                _assertPred!"=="(date, Date(1999, 7, 1));
            }

            {
                auto date = Date(1999, 5, 31);
                date.roll!"months"(-1);
                _assertPred!"=="(date, Date(1999, 5, 1));
            }

            {
                auto date = Date(1999, 2, 28);
                date.roll!"months"(12);
                _assertPred!"=="(date, Date(1999, 2, 28));
            }

            {
                auto date = Date(2000, 2, 29);
                date.roll!"months"(12);
                _assertPred!"=="(date, Date(2000, 2, 29));
            }

            {
                auto date = Date(1999, 7, 31);
                date.roll!"months"(1);
                _assertPred!"=="(date, Date(1999, 8, 31));
                date.roll!"months"(1);
                _assertPred!"=="(date, Date(1999, 10, 1));
            }

            {
                auto date = Date(1998, 8, 31);
                date.roll!"months"(13);
                _assertPred!"=="(date, Date(1998, 10, 1));
                date.roll!"months"(-13);
                _assertPred!"=="(date, Date(1998, 9, 1));
            }

            {
                auto date = Date(1997, 12, 31);
                date.roll!"months"(13);
                _assertPred!"=="(date, Date(1997, 1, 31));
                date.roll!"months"(-13);
                _assertPred!"=="(date, Date(1997, 12, 31));
            }

            {
                auto date = Date(1997, 12, 31);
                date.roll!"months"(14);
                _assertPred!"=="(date, Date(1997, 3, 3));
                date.roll!"months"(-14);
                _assertPred!"=="(date, Date(1997, 1, 3));
            }

            {
                auto date = Date(1998, 12, 31);
                date.roll!"months"(14);
                _assertPred!"=="(date, Date(1998, 3, 3));
                date.roll!"months"(-14);
                _assertPred!"=="(date, Date(1998, 1, 3));
            }

            {
                auto date = Date(1999, 12, 31);
                date.roll!"months"(14);
                _assertPred!"=="(date, Date(1999, 3, 3));
                date.roll!"months"(-14);
                _assertPred!"=="(date, Date(1999, 1, 3));
            }

            //Test B.C.
            {
                auto date = Date(-1999, 7, 6);
                date.roll!"months"(3);
                _assertPred!"=="(date, Date(-1999, 10, 6));
                date.roll!"months"(-4);
                _assertPred!"=="(date, Date(-1999, 6, 6));
            }

            {
                auto date = Date(-1999, 7, 6);
                date.roll!"months"(6);
                _assertPred!"=="(date, Date(-1999, 1, 6));
                date.roll!"months"(-6);
                _assertPred!"=="(date, Date(-1999, 7, 6));
            }

            {
                auto date = Date(-1999, 7, 6);
                date.roll!"months"(-27);
                _assertPred!"=="(date, Date(-1999, 4, 6));
                date.roll!"months"(28);
                _assertPred!"=="(date, Date(-1999, 8, 6));
            }

            {
                auto date = Date(-1999, 5, 31);
                date.roll!"months"(1);
                _assertPred!"=="(date, Date(-1999, 7, 1));
            }

            {
                auto date = Date(-1999, 5, 31);
                date.roll!"months"(-1);
                _assertPred!"=="(date, Date(-1999, 5, 1));
            }

            {
                auto date = Date(-1999, 2, 28);
                date.roll!"months"(-12);
                _assertPred!"=="(date, Date(-1999, 2, 28));
            }

            {
                auto date = Date(-2000, 2, 29);
                date.roll!"months"(-12);
                _assertPred!"=="(date, Date(-2000, 2, 29));
            }

            {
                auto date = Date(-1999, 7, 31);
                date.roll!"months"(1);
                _assertPred!"=="(date, Date(-1999, 8, 31));
                date.roll!"months"(1);
                _assertPred!"=="(date, Date(-1999, 10, 1));
            }

            {
                auto date = Date(-1998, 8, 31);
                date.roll!"months"(13);
                _assertPred!"=="(date, Date(-1998, 10, 1));
                date.roll!"months"(-13);
                _assertPred!"=="(date, Date(-1998, 9, 1));
            }

            {
                auto date = Date(-1997, 12, 31);
                date.roll!"months"(13);
                _assertPred!"=="(date, Date(-1997, 1, 31));
                date.roll!"months"(-13);
                _assertPred!"=="(date, Date(-1997, 12, 31));
            }

            {
                auto date = Date(-1997, 12, 31);
                date.roll!"months"(14);
                _assertPred!"=="(date, Date(-1997, 3, 3));
                date.roll!"months"(-14);
                _assertPred!"=="(date, Date(-1997, 1, 3));
            }

            {
                auto date = Date(-2002, 12, 31);
                date.roll!"months"(14);
                _assertPred!"=="(date, Date(-2002, 3, 3));
                date.roll!"months"(-14);
                _assertPred!"=="(date, Date(-2002, 1, 3));
            }

            {
                auto date = Date(-2001, 12, 31);
                date.roll!"months"(14);
                _assertPred!"=="(date, Date(-2001, 3, 3));
                date.roll!"months"(-14);
                _assertPred!"=="(date, Date(-2001, 1, 3));
            }

            //Test Both
            {
                auto date = Date(1, 1, 1);
                date.roll!"months"(-1);
                _assertPred!"=="(date, Date(1, 12, 1));
                date.roll!"months"(1);
                _assertPred!"=="(date, Date(1, 1, 1));
            }

            {
                auto date = Date(4, 1, 1);
                date.roll!"months"(-48);
                _assertPred!"=="(date, Date(4, 1, 1));
                date.roll!"months"(48);
                _assertPred!"=="(date, Date(4, 1, 1));
            }

            {
                auto date = Date(4, 3, 31);
                date.roll!"months"(-49);
                _assertPred!"=="(date, Date(4, 3, 2));
                date.roll!"months"(49);
                _assertPred!"=="(date, Date(4, 4, 2));
            }

            {
                auto date = Date(4, 3, 31);
                date.roll!"months"(-85);
                _assertPred!"=="(date, Date(4, 3, 2));
                date.roll!"months"(85);
                _assertPred!"=="(date, Date(4, 4, 2));
            }

            {
                auto date = Date(-1, 1, 1);
                date.roll!"months"(-1);
                _assertPred!"=="(date, Date(-1, 12, 1));
                date.roll!"months"(1);
                _assertPred!"=="(date, Date(-1, 1, 1));
            }

            {
                auto date = Date(-4, 1, 1);
                date.roll!"months"(-48);
                _assertPred!"=="(date, Date(-4, 1, 1));
                date.roll!"months"(48);
                _assertPred!"=="(date, Date(-4, 1, 1));
            }

            {
                auto date = Date(-4, 3, 31);
                date.roll!"months"(-49);
                _assertPred!"=="(date, Date(-4, 3, 2));
                date.roll!"months"(49);
                _assertPred!"=="(date, Date(-4, 4, 2));
            }

            {
                auto date = Date(-4, 3, 31);
                date.roll!"months"(-85);
                _assertPred!"=="(date, Date(-4, 3, 2));
                date.roll!"months"(85);
                _assertPred!"=="(date, Date(-4, 4, 2));
            }

            const cdate = Date(1999, 7, 6);
            immutable idate = Date(1999, 7, 6);
            static assert(!__traits(compiles, cdate.roll!"months"(3)));
            static assert(!__traits(compiles, idate.roll!"months"(3)));

            //Verify Examples.
            auto date1 = Date(2010, 1, 1);
            date1.roll!"months"(1);
            assert(date1 == Date(2010, 2, 1));

            auto date2 = Date(2010, 1, 1);
            date2.roll!"months"(-1);
            assert(date2 == Date(2010, 12, 1));

            auto date3 = Date(1999, 1, 29);
            date3.roll!"months"(1);
            assert(date3 == Date(1999, 3, 1));

            auto date4 = Date(1999, 1, 29);
            date4.roll!"months"(1, AllowDayOverflow.no);
            assert(date4 == Date(1999, 2, 28));
        }
    }

    //Test roll!"months"() with AllowDayOverlow.no
    unittest
    {
        version(testStdDateTime)
        {
            //Test A.D.
            {
                auto date = Date(1999, 7, 6);
                date.roll!"months"(3, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(1999, 10, 6));
                date.roll!"months"(-4, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(1999, 6, 6));
            }

            {
                auto date = Date(1999, 7, 6);
                date.roll!"months"(6, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(1999, 1, 6));
                date.roll!"months"(-6, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(1999, 7, 6));
            }

            {
                auto date = Date(1999, 7, 6);
                date.roll!"months"(27, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(1999, 10, 6));
                date.roll!"months"(-28, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(1999, 6, 6));
            }

            {
                auto date = Date(1999, 5, 31);
                date.roll!"months"(1, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(1999, 6, 30));
            }

            {
                auto date = Date(1999, 5, 31);
                date.roll!"months"(-1, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(1999, 4, 30));
            }

            {
                auto date = Date(1999, 2, 28);
                date.roll!"months"(12, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(1999, 2, 28));
            }

            {
                auto date = Date(2000, 2, 29);
                date.roll!"months"(12, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(2000, 2, 29));
            }

            {
                auto date = Date(1999, 7, 31);
                date.roll!"months"(1, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(1999, 8, 31));
                date.roll!"months"(1, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(1999, 9, 30));
            }

            {
                auto date = Date(1998, 8, 31);
                date.roll!"months"(13, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(1998, 9, 30));
                date.roll!"months"(-13, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(1998, 8, 30));
            }

            {
                auto date = Date(1997, 12, 31);
                date.roll!"months"(13, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(1997, 1, 31));
                date.roll!"months"(-13, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(1997, 12, 31));
            }

            {
                auto date = Date(1997, 12, 31);
                date.roll!"months"(14, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(1997, 2, 28));
                date.roll!"months"(-14, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(1997, 12, 28));
            }

            {
                auto date = Date(1998, 12, 31);
                date.roll!"months"(14, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(1998, 2, 28));
                date.roll!"months"(-14, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(1998, 12, 28));
            }

            {
                auto date = Date(1999, 12, 31);
                date.roll!"months"(14, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(1999, 2, 28));
                date.roll!"months"(-14, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(1999, 12, 28));
            }

            //Test B.C.
            {
                auto date = Date(-1999, 7, 6);
                date.roll!"months"(3, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-1999, 10, 6));
                date.roll!"months"(-4, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-1999, 6, 6));
            }

            {
                auto date = Date(-1999, 7, 6);
                date.roll!"months"(6, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-1999, 1, 6));
                date.roll!"months"(-6, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-1999, 7, 6));
            }

            {
                auto date = Date(-1999, 7, 6);
                date.roll!"months"(-27, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-1999, 4, 6));
                date.roll!"months"(28, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-1999, 8, 6));
            }

            {
                auto date = Date(-1999, 5, 31);
                date.roll!"months"(1, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-1999, 6, 30));
            }

            {
                auto date = Date(-1999, 5, 31);
                date.roll!"months"(-1, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-1999, 4, 30));
            }

            {
                auto date = Date(-1999, 2, 28);
                date.roll!"months"(-12, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-1999, 2, 28));
            }

            {
                auto date = Date(-2000, 2, 29);
                date.roll!"months"(-12, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-2000, 2, 29));
            }

            {
                auto date = Date(-1999, 7, 31);
                date.roll!"months"(1, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-1999, 8, 31));
                date.roll!"months"(1, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-1999, 9, 30));
            }

            {
                auto date = Date(-1998, 8, 31);
                date.roll!"months"(13, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-1998, 9, 30));
                date.roll!"months"(-13, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-1998, 8, 30));
            }

            {
                auto date = Date(-1997, 12, 31);
                date.roll!"months"(13, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-1997, 1, 31));
                date.roll!"months"(-13, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-1997, 12, 31));
            }

            {
                auto date = Date(-1997, 12, 31);
                date.roll!"months"(14, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-1997, 2, 28));
                date.roll!"months"(-14, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-1997, 12, 28));
            }

            {
                auto date = Date(-2002, 12, 31);
                date.roll!"months"(14, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-2002, 2, 28));
                date.roll!"months"(-14, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-2002, 12, 28));
            }

            {
                auto date = Date(-2001, 12, 31);
                date.roll!"months"(14, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-2001, 2, 28));
                date.roll!"months"(-14, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-2001, 12, 28));
            }

            //Test Both
            {
                auto date = Date(1, 1, 1);
                date.roll!"months"(-1, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(1, 12, 1));
                date.roll!"months"(1, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(1, 1, 1));
            }

            {
                auto date = Date(4, 1, 1);
                date.roll!"months"(-48, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(4, 1, 1));
                date.roll!"months"(48, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(4, 1, 1));
            }

            {
                auto date = Date(4, 3, 31);
                date.roll!"months"(-49, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(4, 2, 29));
                date.roll!"months"(49, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(4, 3, 29));
            }

            {
                auto date = Date(4, 3, 31);
                date.roll!"months"(-85, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(4, 2, 29));
                date.roll!"months"(85, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(4, 3, 29));
            }

            {
                auto date = Date(-1, 1, 1);
                date.roll!"months"(-1, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-1, 12, 1));
                date.roll!"months"(1, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-1, 1, 1));
            }

            {
                auto date = Date(-4, 1, 1);
                date.roll!"months"(-48, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-4, 1, 1));
                date.roll!"months"(48, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-4, 1, 1));
            }

            {
                auto date = Date(-4, 3, 31);
                date.roll!"months"(-49, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-4, 2, 29));
                date.roll!"months"(49, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-4, 3, 29));
            }

            {
                auto date = Date(-4, 3, 31);
                date.roll!"months"(-85, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-4, 2, 29));
                date.roll!"months"(85, AllowDayOverflow.no);
                _assertPred!"=="(date, Date(-4, 3, 29));
            }
        }
    }


    /++
        Adds the given number of units to this $(D Date). A negative number will
        subtract.

        The difference between rolling and adding is that rolling does not
        affect larger units. For instance, rolling a $(D Date) one
        year's worth of days gets the exact same $(D Date).

        The only accepted units are $(D "days").

        Params:
            units = The units to add. Must be $(D "days").
            value = The number of days to add to this $(D Date).

        Examples:
--------------------
auto d = Date(2010, 1, 1);
d.roll!"days"(1);
assert(d == Date(2010, 1, 2));
d.roll!"days"(365);
assert(d == Date(2010, 1, 26));
d.roll!"days"(-32);
assert(d == Date(2010, 1, 25));
--------------------
      +/
    /+ref Date+/ void roll(string units)(long days) pure nothrow
        if(units == "days")
    {
        immutable limit = maxDay(_year, _month);
        days %= limit;
        auto newDay = _day + days;

        if(days < 0)
        {
            if(newDay < 1)
                newDay += limit;
        }
        else if(newDay > limit)
            newDay -= limit;

        _day = cast(ubyte)newDay;
    }

    //Verify Examples.
    unittest
    {
        version(testStdDateTime)
        {
            auto d = Date(2010, 1, 1);
            d.roll!"days"(1);
            assert(d == Date(2010, 1, 2));
            d.roll!"days"(365);
            assert(d == Date(2010, 1, 26));
            d.roll!"days"(-32);
            assert(d == Date(2010, 1, 25));
        }
    }

    unittest
    {
        version(testStdDateTime)
        {
            //Test A.D.
            {
                auto date = Date(1999, 2, 28);
                date.roll!"days"(1);
                _assertPred!"=="(date, Date(1999, 2, 1));
                date.roll!"days"(-1);
                _assertPred!"=="(date, Date(1999, 2, 28));
            }

            {
                auto date = Date(2000, 2, 28);
                date.roll!"days"(1);
                _assertPred!"=="(date, Date(2000, 2, 29));
                date.roll!"days"(1);
                _assertPred!"=="(date, Date(2000, 2, 1));
                date.roll!"days"(-1);
                _assertPred!"=="(date, Date(2000, 2, 29));
            }

            {
                auto date = Date(1999, 6, 30);
                date.roll!"days"(1);
                _assertPred!"=="(date, Date(1999, 6, 1));
                date.roll!"days"(-1);
                _assertPred!"=="(date, Date(1999, 6, 30));
            }

            {
                auto date = Date(1999, 7, 31);
                date.roll!"days"(1);
                _assertPred!"=="(date, Date(1999, 7, 1));
                date.roll!"days"(-1);
                _assertPred!"=="(date, Date(1999, 7, 31));
            }

            {
                auto date = Date(1999, 1, 1);
                date.roll!"days"(-1);
                _assertPred!"=="(date, Date(1999, 1, 31));
                date.roll!"days"(1);
                _assertPred!"=="(date, Date(1999, 1, 1));
            }

            {
                auto date = Date(1999, 7, 6);
                date.roll!"days"(9);
                _assertPred!"=="(date, Date(1999, 7, 15));
                date.roll!"days"(-11);
                _assertPred!"=="(date, Date(1999, 7, 4));
                date.roll!"days"(30);
                _assertPred!"=="(date, Date(1999, 7, 3));
                date.roll!"days"(-3);
                _assertPred!"=="(date, Date(1999, 7, 31));
            }

            {
                auto date = Date(1999, 7, 6);
                date.roll!"days"(365);
                _assertPred!"=="(date, Date(1999, 7, 30));
                date.roll!"days"(-365);
                _assertPred!"=="(date, Date(1999, 7, 6));
                date.roll!"days"(366);
                _assertPred!"=="(date, Date(1999, 7, 31));
                date.roll!"days"(730);
                _assertPred!"=="(date, Date(1999, 7, 17));
                date.roll!"days"(-1096);
                _assertPred!"=="(date, Date(1999, 7, 6));
            }

            {
                auto date = Date(1999, 2, 6);
                date.roll!"days"(365);
                _assertPred!"=="(date, Date(1999, 2, 7));
                date.roll!"days"(-365);
                _assertPred!"=="(date, Date(1999, 2, 6));
                date.roll!"days"(366);
                _assertPred!"=="(date, Date(1999, 2, 8));
                date.roll!"days"(730);
                _assertPred!"=="(date, Date(1999, 2, 10));
                date.roll!"days"(-1096);
                _assertPred!"=="(date, Date(1999, 2, 6));
            }

            //Test B.C.
            {
                auto date = Date(-1999, 2, 28);
                date.roll!"days"(1);
                _assertPred!"=="(date, Date(-1999, 2, 1));
                date.roll!"days"(-1);
                _assertPred!"=="(date, Date(-1999, 2, 28));
            }

            {
                auto date = Date(-2000, 2, 28);
                date.roll!"days"(1);
                _assertPred!"=="(date, Date(-2000, 2, 29));
                date.roll!"days"(1);
                _assertPred!"=="(date, Date(-2000, 2, 1));
                date.roll!"days"(-1);
                _assertPred!"=="(date, Date(-2000, 2, 29));
            }

            {
                auto date = Date(-1999, 6, 30);
                date.roll!"days"(1);
                _assertPred!"=="(date, Date(-1999, 6, 1));
                date.roll!"days"(-1);
                _assertPred!"=="(date, Date(-1999, 6, 30));
            }

            {
                auto date = Date(-1999, 7, 31);
                date.roll!"days"(1);
                _assertPred!"=="(date, Date(-1999, 7, 1));
                date.roll!"days"(-1);
                _assertPred!"=="(date, Date(-1999, 7, 31));
            }

            {
                auto date = Date(-1999, 1, 1);
                date.roll!"days"(-1);
                _assertPred!"=="(date, Date(-1999, 1, 31));
                date.roll!"days"(1);
                _assertPred!"=="(date, Date(-1999, 1, 1));
            }

            {
                auto date = Date(-1999, 7, 6);
                date.roll!"days"(9);
                _assertPred!"=="(date, Date(-1999, 7, 15));
                date.roll!"days"(-11);
                _assertPred!"=="(date, Date(-1999, 7, 4));
                date.roll!"days"(30);
                _assertPred!"=="(date, Date(-1999, 7, 3));
                date.roll!"days"(-3);
                _assertPred!"=="(date, Date(-1999, 7, 31));
            }

            {
                auto date = Date(-1999, 7, 6);
                date.roll!"days"(365);
                _assertPred!"=="(date, Date(-1999, 7, 30));
                date.roll!"days"(-365);
                _assertPred!"=="(date, Date(-1999, 7, 6));
                date.roll!"days"(366);
                _assertPred!"=="(date, Date(-1999, 7, 31));
                date.roll!"days"(730);
                _assertPred!"=="(date, Date(-1999, 7, 17));
                date.roll!"days"(-1096);
                _assertPred!"=="(date, Date(-1999, 7, 6));
            }

            //Test Both
            {
                auto date = Date(1, 7, 6);
                date.roll!"days"(-365);
                _assertPred!"=="(date, Date(1, 7, 13));
                date.roll!"days"(365);
                _assertPred!"=="(date, Date(1, 7, 6));
                date.roll!"days"(-731);
                _assertPred!"=="(date, Date(1, 7, 19));
                date.roll!"days"(730);
                _assertPred!"=="(date, Date(1, 7, 5));
            }

            {
                auto date = Date(0, 7, 6);
                date.roll!"days"(-365);
                _assertPred!"=="(date, Date(0, 7, 13));
                date.roll!"days"(365);
                _assertPred!"=="(date, Date(0, 7, 6));
                date.roll!"days"(-731);
                _assertPred!"=="(date, Date(0, 7, 19));
                date.roll!"days"(730);
                _assertPred!"=="(date, Date(0, 7, 5));
            }

            const cdate = Date(1999, 7, 6);
            immutable idate = Date(1999, 7, 6);
            static assert(!__traits(compiles, cdate.roll!"days"(12)));
            static assert(!__traits(compiles, idate.roll!"days"(12)));

            //Verify Examples.
            auto date = Date(2010, 1, 1);
            date.roll!"days"(1);
            assert(date == Date(2010, 1, 2));
            date.roll!"days"(365);
            assert(date == Date(2010, 1, 26));
            date.roll!"days"(-32);
            assert(date == Date(2010, 1, 25));
        }
    }


    /++
        Gives the result of adding or subtracting a duration from this
        $(D Date).

        The legal types of arithmetic for Date using this operator are

        $(BOOKTABLE,
        $(TR $(TD Date) $(TD +) $(TD duration) $(TD -->) $(TD Date))
        $(TR $(TD Date) $(TD -) $(TD duration) $(TD -->) $(TD Date))
        )

        Params:
            duration = The duration to add to or subtract from this $(D Date).
      +/
    Date opBinary(string op, D)(in D duration) const pure nothrow
        if((op == "+" || op == "-") &&
           (is(Unqual!D == Duration) ||
            is(Unqual!D == TickDuration)))
    {
        Date retval = this;

        static if(is(Unqual!D == Duration))
            immutable days = duration.total!"days";
        else static if(is(Unqual!D == TickDuration))
            immutable days = convert!("hnsecs", "days")(duration.hnsecs);

        //Ideally, this would just be
        //return retval.addDays(unaryFun!(op ~ "a")(days));
        //But there isn't currently a pure version of unaryFun!().

        static if(op == "+")
            immutable signedDays = days;
        else static if(op == "-")
            immutable signedDays = -days;
        else
            static assert(0);

        return retval.addDays(signedDays);
    }

    unittest
    {
        version(testStdDateTime)
        {
            auto date = Date(1999, 7, 6);

            _assertPred!"=="(date + dur!"weeks"(7), Date(1999, 8, 24));
            _assertPred!"=="(date + dur!"weeks"(-7), Date(1999, 5, 18));
            _assertPred!"=="(date + dur!"days"(7), Date(1999, 7, 13));
            _assertPred!"=="(date + dur!"days"(-7), Date(1999, 6, 29));

            _assertPred!"=="(date + dur!"hours"(24), Date(1999, 7, 7));
            _assertPred!"=="(date + dur!"hours"(-24), Date(1999, 7, 5));
            _assertPred!"=="(date + dur!"minutes"(1440), Date(1999, 7, 7));
            _assertPred!"=="(date + dur!"minutes"(-1440), Date(1999, 7, 5));
            _assertPred!"=="(date + dur!"seconds"(86_400), Date(1999, 7, 7));
            _assertPred!"=="(date + dur!"seconds"(-86_400), Date(1999, 7, 5));
            _assertPred!"=="(date + dur!"msecs"(86_400_000), Date(1999, 7, 7));
            _assertPred!"=="(date + dur!"msecs"(-86_400_000), Date(1999, 7, 5));
            _assertPred!"=="(date + dur!"usecs"(86_400_000_000), Date(1999, 7, 7));
            _assertPred!"=="(date + dur!"usecs"(-86_400_000_000), Date(1999, 7, 5));
            _assertPred!"=="(date + dur!"hnsecs"(864_000_000_000), Date(1999, 7, 7));
            _assertPred!"=="(date + dur!"hnsecs"(-864_000_000_000), Date(1999, 7, 5));

            //This probably only runs in cases where gettimeofday() is used, but it's
            //hard to do this test correctly with variable ticksPerSec.
            if(TickDuration.ticksPerSec == 1_000_000)
            {
                _assertPred!"=="(date + TickDuration.from!"usecs"(86_400_000_000), Date(1999, 7, 7));
                _assertPred!"=="(date + TickDuration.from!"usecs"(-86_400_000_000), Date(1999, 7, 5));
            }

            _assertPred!"=="(date - dur!"weeks"(-7), Date(1999, 8, 24));
            _assertPred!"=="(date - dur!"weeks"(7), Date(1999, 5, 18));
            _assertPred!"=="(date - dur!"days"(-7), Date(1999, 7, 13));
            _assertPred!"=="(date - dur!"days"(7), Date(1999, 6, 29));

            _assertPred!"=="(date - dur!"hours"(-24), Date(1999, 7, 7));
            _assertPred!"=="(date - dur!"hours"(24), Date(1999, 7, 5));
            _assertPred!"=="(date - dur!"minutes"(-1440), Date(1999, 7, 7));
            _assertPred!"=="(date - dur!"minutes"(1440), Date(1999, 7, 5));
            _assertPred!"=="(date - dur!"seconds"(-86_400), Date(1999, 7, 7));
            _assertPred!"=="(date - dur!"seconds"(86_400), Date(1999, 7, 5));
            _assertPred!"=="(date - dur!"msecs"(-86_400_000), Date(1999, 7, 7));
            _assertPred!"=="(date - dur!"msecs"(86_400_000), Date(1999, 7, 5));
            _assertPred!"=="(date - dur!"usecs"(-86_400_000_000), Date(1999, 7, 7));
            _assertPred!"=="(date - dur!"usecs"(86_400_000_000), Date(1999, 7, 5));
            _assertPred!"=="(date - dur!"hnsecs"(-864_000_000_000), Date(1999, 7, 7));
            _assertPred!"=="(date - dur!"hnsecs"(864_000_000_000), Date(1999, 7, 5));

            //This probably only runs in cases where gettimeofday() is used, but it's
            //hard to do this test correctly with variable ticksPerSec.
            if(TickDuration.ticksPerSec == 1_000_000)
            {
                _assertPred!"=="(date - TickDuration.from!"usecs"(-86_400_000_000), Date(1999, 7, 7));
                _assertPred!"=="(date - TickDuration.from!"usecs"(86_400_000_000), Date(1999, 7, 5));
            }

            auto duration = dur!"days"(12);
            const cdate = Date(1999, 7, 6);
            immutable idate = Date(1999, 7, 6);
            static assert(__traits(compiles, date + duration));
            static assert(__traits(compiles, cdate + duration));
            static assert(__traits(compiles, idate + duration));

            static assert(__traits(compiles, date - duration));
            static assert(__traits(compiles, cdate - duration));
            static assert(__traits(compiles, idate - duration));
        }
    }


    /++
        Gives the result of adding or subtracting a duration from this
        $(D Date), as well as assigning the result to this $(D Date).

        The legal types of arithmetic for $(D Date) using this operator are

        $(BOOKTABLE,
        $(TR $(TD Date) $(TD +) $(TD duration) $(TD -->) $(TD Date))
        $(TR $(TD Date) $(TD -) $(TD duration) $(TD -->) $(TD Date))
        )

        Params:
            duration = The duration to add to or subtract from this $(D Date).
      +/
    /+ref+/ Date opOpAssign(string op, D)(in D duration) pure nothrow
        if((op == "+" || op == "-") &&
           (is(Unqual!D == Duration) ||
            is(Unqual!D == TickDuration)))
    {
        static if(is(Unqual!D == Duration))
            immutable days = duration.total!"days";
        else static if(is(Unqual!D == TickDuration))
            immutable days = convert!("hnsecs", "days")(duration.hnsecs);

        //Ideally, this would just be
        //return addDays(unaryFun!(op ~ "a")(days));
        //But there isn't currently a pure version of unaryFun!().

        static if(op == "+")
            immutable signedDays = days;
        else static if(op == "-")
            immutable signedDays = -days;
        else
            static assert(0);

        return addDays(signedDays);
    }

    unittest
    {
        version(testStdDateTime)
        {
            _assertPred!"+="(Date(1999, 7, 6), dur!"weeks"(7), Date(1999, 8, 24));
            _assertPred!"+="(Date(1999, 7, 6), dur!"weeks"(-7), Date(1999, 5, 18));
            _assertPred!"+="(Date(1999, 7, 6), dur!"days"(7), Date(1999, 7, 13));
            _assertPred!"+="(Date(1999, 7, 6), dur!"days"(-7), Date(1999, 6, 29));

            _assertPred!"+="(Date(1999, 7, 6), dur!"hours"(24), Date(1999, 7, 7));
            _assertPred!"+="(Date(1999, 7, 6), dur!"hours"(-24), Date(1999, 7, 5));
            _assertPred!"+="(Date(1999, 7, 6), dur!"minutes"(1440), Date(1999, 7, 7));
            _assertPred!"+="(Date(1999, 7, 6), dur!"minutes"(-1440), Date(1999, 7, 5));
            _assertPred!"+="(Date(1999, 7, 6), dur!"seconds"(86_400), Date(1999, 7, 7));
            _assertPred!"+="(Date(1999, 7, 6), dur!"seconds"(-86_400), Date(1999, 7, 5));
            _assertPred!"+="(Date(1999, 7, 6), dur!"msecs"(86_400_000), Date(1999, 7, 7));
            _assertPred!"+="(Date(1999, 7, 6), dur!"msecs"(-86_400_000), Date(1999, 7, 5));
            _assertPred!"+="(Date(1999, 7, 6), dur!"usecs"(86_400_000_000), Date(1999, 7, 7));
            _assertPred!"+="(Date(1999, 7, 6), dur!"usecs"(-86_400_000_000), Date(1999, 7, 5));
            _assertPred!"+="(Date(1999, 7, 6), dur!"hnsecs"(864_000_000_000), Date(1999, 7, 7));
            _assertPred!"+="(Date(1999, 7, 6), dur!"hnsecs"(-864_000_000_000), Date(1999, 7, 5));

            _assertPred!"-="(Date(1999, 7, 6), dur!"weeks"(-7), Date(1999, 8, 24));
            _assertPred!"-="(Date(1999, 7, 6), dur!"weeks"(7), Date(1999, 5, 18));
            _assertPred!"-="(Date(1999, 7, 6), dur!"days"(-7), Date(1999, 7, 13));
            _assertPred!"-="(Date(1999, 7, 6), dur!"days"(7), Date(1999, 6, 29));

            _assertPred!"-="(Date(1999, 7, 6), dur!"hours"(-24), Date(1999, 7, 7));
            _assertPred!"-="(Date(1999, 7, 6), dur!"hours"(24), Date(1999, 7, 5));
            _assertPred!"-="(Date(1999, 7, 6), dur!"minutes"(-1440), Date(1999, 7, 7));
            _assertPred!"-="(Date(1999, 7, 6), dur!"minutes"(1440), Date(1999, 7, 5));
            _assertPred!"-="(Date(1999, 7, 6), dur!"seconds"(-86_400), Date(1999, 7, 7));
            _assertPred!"-="(Date(1999, 7, 6), dur!"seconds"(86_400), Date(1999, 7, 5));
            _assertPred!"-="(Date(1999, 7, 6), dur!"msecs"(-86_400_000), Date(1999, 7, 7));
            _assertPred!"-="(Date(1999, 7, 6), dur!"msecs"(86_400_000), Date(1999, 7, 5));
            _assertPred!"-="(Date(1999, 7, 6), dur!"usecs"(-86_400_000_000), Date(1999, 7, 7));
            _assertPred!"-="(Date(1999, 7, 6), dur!"usecs"(86_400_000_000), Date(1999, 7, 5));
            _assertPred!"-="(Date(1999, 7, 6), dur!"hnsecs"(-864_000_000_000), Date(1999, 7, 7));
            _assertPred!"-="(Date(1999, 7, 6), dur!"hnsecs"(864_000_000_000), Date(1999, 7, 5));

            auto duration = dur!"days"(12);
            auto date = Date(1999, 7, 6);
            const cdate = Date(1999, 7, 6);
            immutable idate = Date(1999, 7, 6);
            static assert(__traits(compiles, date += duration));
            static assert(!__traits(compiles, cdate += duration));
            static assert(!__traits(compiles, idate += duration));

            static assert(__traits(compiles, date -= duration));
            static assert(!__traits(compiles, cdate -= duration));
            static assert(!__traits(compiles, idate -= duration));
        }
    }


    /++
        Gives the difference between two $(D Date)s.

        The legal types of arithmetic for Date using this operator are

        $(BOOKTABLE,
        $(TR $(TD Date) $(TD -) $(TD Date) $(TD -->) $(TD duration))
        )
      +/
    Duration opBinary(string op)(in Date rhs) const pure nothrow
        if(op == "-")
    {
        return dur!"days"(this.dayOfGregorianCal - rhs.dayOfGregorianCal);
    }

    unittest
    {
        version(testStdDateTime)
        {
            auto date = Date(1999, 7, 6);

            _assertPred!"=="(Date(1999, 7, 6) - Date(1998, 7, 6), dur!"days"(365));
            _assertPred!"=="(Date(1998, 7, 6) - Date(1999, 7, 6), dur!"days"(-365));
            _assertPred!"=="(Date(1999, 6, 6) - Date(1999, 5, 6), dur!"days"(31));
            _assertPred!"=="(Date(1999, 5, 6) - Date(1999, 6, 6), dur!"days"(-31));
            _assertPred!"=="(Date(1999, 1, 1) - Date(1998, 12, 31), dur!"days"(1));
            _assertPred!"=="(Date(1998, 12, 31) - Date(1999, 1, 1), dur!"days"(-1));

            const cdate = Date(1999, 7, 6);
            immutable idate = Date(1999, 7, 6);
            static assert(__traits(compiles, date - date));
            static assert(__traits(compiles, cdate - date));
            static assert(__traits(compiles, idate - date));

            static assert(__traits(compiles, date - cdate));
            static assert(__traits(compiles, cdate - cdate));
            static assert(__traits(compiles, idate - cdate));

            static assert(__traits(compiles, date - idate));
            static assert(__traits(compiles, cdate - idate));
            static assert(__traits(compiles, idate - idate));
        }
    }


    /++
        Returns the difference between the two $(D Date)s in months.

        To get the difference in years, subtract the year property
        of two $(D SysTime)s. To get the difference in days or weeks,
        subtract the $(D SysTime)s themselves and use the $(D Duration)
        that results. Because converting between months and smaller
        units requires a specific date (which $(D Duration)s don't have),
        getting the difference in months requires some math using both
        the year and month properties, so this is a convenience function for
        getting the difference in months.

        Note that the number of days in the months or how far into the month
        either $(D Date) is is irrelevant. It is the difference in the month
        property combined with the difference in years * 12. So, for instance,
        December 31st and January 1st are one month apart just as December 1st
        and January 31st are one month apart.

        Params:
            rhs = The $(D Date) to subtract from this one.

        Examples:
--------------------
assert(Date(1999, 2, 1).diffMonths(Date(1999, 1, 31)) == 1);
assert(Date(1999, 1, 31).diffMonths(Date(1999, 2, 1)) == -1);
assert(Date(1999, 3, 1).diffMonths(Date(1999, 1, 1)) == 2);
assert(Date(1999, 1, 1).diffMonths(Date(1999, 3, 31)) == -2);
--------------------
      +/
    int diffMonths(in Date rhs) const pure nothrow
    {
        immutable yearDiff = _year - rhs._year;
        immutable monthDiff = _month - rhs._month;

        return yearDiff * 12 + monthDiff;
    }

    unittest
    {
        version(testStdDateTime)
        {
            auto date = Date(1999, 7, 6);

            //Test A.D.
            _assertPred!"=="(date.diffMonths(Date(1998, 6, 5)), 13);
            _assertPred!"=="(date.diffMonths(Date(1998, 7, 5)), 12);
            _assertPred!"=="(date.diffMonths(Date(1998, 8, 5)), 11);
            _assertPred!"=="(date.diffMonths(Date(1998, 9, 5)), 10);
            _assertPred!"=="(date.diffMonths(Date(1998, 10, 5)), 9);
            _assertPred!"=="(date.diffMonths(Date(1998, 11, 5)), 8);
            _assertPred!"=="(date.diffMonths(Date(1998, 12, 5)), 7);
            _assertPred!"=="(date.diffMonths(Date(1999, 1, 5)), 6);
            _assertPred!"=="(date.diffMonths(Date(1999, 2, 6)), 5);
            _assertPred!"=="(date.diffMonths(Date(1999, 3, 6)), 4);
            _assertPred!"=="(date.diffMonths(Date(1999, 4, 6)), 3);
            _assertPred!"=="(date.diffMonths(Date(1999, 5, 6)), 2);
            _assertPred!"=="(date.diffMonths(Date(1999, 6, 6)), 1);
            _assertPred!"=="(date.diffMonths(date), 0);
            _assertPred!"=="(date.diffMonths(Date(1999, 8, 6)), -1);
            _assertPred!"=="(date.diffMonths(Date(1999, 9, 6)), -2);
            _assertPred!"=="(date.diffMonths(Date(1999, 10, 6)), -3);
            _assertPred!"=="(date.diffMonths(Date(1999, 11, 6)), -4);
            _assertPred!"=="(date.diffMonths(Date(1999, 12, 6)), -5);
            _assertPred!"=="(date.diffMonths(Date(2000, 1, 6)), -6);
            _assertPred!"=="(date.diffMonths(Date(2000, 2, 6)), -7);
            _assertPred!"=="(date.diffMonths(Date(2000, 3, 6)), -8);
            _assertPred!"=="(date.diffMonths(Date(2000, 4, 6)), -9);
            _assertPred!"=="(date.diffMonths(Date(2000, 5, 6)), -10);
            _assertPred!"=="(date.diffMonths(Date(2000, 6, 6)), -11);
            _assertPred!"=="(date.diffMonths(Date(2000, 7, 6)), -12);
            _assertPred!"=="(date.diffMonths(Date(2000, 8, 6)), -13);

            _assertPred!"=="(Date(1998, 6, 5).diffMonths(date), -13);
            _assertPred!"=="(Date(1998, 7, 5).diffMonths(date), -12);
            _assertPred!"=="(Date(1998, 8, 5).diffMonths(date), -11);
            _assertPred!"=="(Date(1998, 9, 5).diffMonths(date), -10);
            _assertPred!"=="(Date(1998, 10, 5).diffMonths(date), -9);
            _assertPred!"=="(Date(1998, 11, 5).diffMonths(date), -8);
            _assertPred!"=="(Date(1998, 12, 5).diffMonths(date), -7);
            _assertPred!"=="(Date(1999, 1, 5).diffMonths(date), -6);
            _assertPred!"=="(Date(1999, 2, 6).diffMonths(date), -5);
            _assertPred!"=="(Date(1999, 3, 6).diffMonths(date), -4);
            _assertPred!"=="(Date(1999, 4, 6).diffMonths(date), -3);
            _assertPred!"=="(Date(1999, 5, 6).diffMonths(date), -2);
            _assertPred!"=="(Date(1999, 6, 6).diffMonths(date), -1);
            _assertPred!"=="(Date(1999, 8, 6).diffMonths(date), 1);
            _assertPred!"=="(Date(1999, 9, 6).diffMonths(date), 2);
            _assertPred!"=="(Date(1999, 10, 6).diffMonths(date), 3);
            _assertPred!"=="(Date(1999, 11, 6).diffMonths(date), 4);
            _assertPred!"=="(Date(1999, 12, 6).diffMonths(date), 5);
            _assertPred!"=="(Date(2000, 1, 6).diffMonths(date), 6);
            _assertPred!"=="(Date(2000, 2, 6).diffMonths(date), 7);
            _assertPred!"=="(Date(2000, 3, 6).diffMonths(date), 8);
            _assertPred!"=="(Date(2000, 4, 6).diffMonths(date), 9);
            _assertPred!"=="(Date(2000, 5, 6).diffMonths(date), 10);
            _assertPred!"=="(Date(2000, 6, 6).diffMonths(date), 11);
            _assertPred!"=="(Date(2000, 7, 6).diffMonths(date), 12);
            _assertPred!"=="(Date(2000, 8, 6).diffMonths(date), 13);

            _assertPred!"=="(date.diffMonths(Date(1999, 6, 30)), 1);
            _assertPred!"=="(date.diffMonths(Date(1999, 7, 1)), 0);
            _assertPred!"=="(date.diffMonths(Date(1999, 7, 6)), 0);
            _assertPred!"=="(date.diffMonths(Date(1999, 7, 11)), 0);
            _assertPred!"=="(date.diffMonths(Date(1999, 7, 16)), 0);
            _assertPred!"=="(date.diffMonths(Date(1999, 7, 21)), 0);
            _assertPred!"=="(date.diffMonths(Date(1999, 7, 26)), 0);
            _assertPred!"=="(date.diffMonths(Date(1999, 7, 31)), 0);
            _assertPred!"=="(date.diffMonths(Date(1999, 8, 1)), -1);

            _assertPred!"=="(date.diffMonths(Date(1990, 6, 30)), 109);
            _assertPred!"=="(date.diffMonths(Date(1990, 7, 1)), 108);
            _assertPred!"=="(date.diffMonths(Date(1990, 7, 6)), 108);
            _assertPred!"=="(date.diffMonths(Date(1990, 7, 11)), 108);
            _assertPred!"=="(date.diffMonths(Date(1990, 7, 16)), 108);
            _assertPred!"=="(date.diffMonths(Date(1990, 7, 21)), 108);
            _assertPred!"=="(date.diffMonths(Date(1990, 7, 26)), 108);
            _assertPred!"=="(date.diffMonths(Date(1990, 7, 31)), 108);
            _assertPred!"=="(date.diffMonths(Date(1990, 8, 1)), 107);

            _assertPred!"=="(Date(1999, 6, 30).diffMonths(date), -1);
            _assertPred!"=="(Date(1999, 7, 1).diffMonths(date), 0);
            _assertPred!"=="(Date(1999, 7, 6).diffMonths(date), 0);
            _assertPred!"=="(Date(1999, 7, 11).diffMonths(date), 0);
            _assertPred!"=="(Date(1999, 7, 16).diffMonths(date), 0);
            _assertPred!"=="(Date(1999, 7, 21).diffMonths(date), 0);
            _assertPred!"=="(Date(1999, 7, 26).diffMonths(date), 0);
            _assertPred!"=="(Date(1999, 7, 31).diffMonths(date), 0);
            _assertPred!"=="(Date(1999, 8, 1).diffMonths(date), 1);

            _assertPred!"=="(Date(1990, 6, 30).diffMonths(date), -109);
            _assertPred!"=="(Date(1990, 7, 1).diffMonths(date), -108);
            _assertPred!"=="(Date(1990, 7, 6).diffMonths(date), -108);
            _assertPred!"=="(Date(1990, 7, 11).diffMonths(date), -108);
            _assertPred!"=="(Date(1990, 7, 16).diffMonths(date), -108);
            _assertPred!"=="(Date(1990, 7, 21).diffMonths(date), -108);
            _assertPred!"=="(Date(1990, 7, 26).diffMonths(date), -108);
            _assertPred!"=="(Date(1990, 7, 31).diffMonths(date), -108);
            _assertPred!"=="(Date(1990, 8, 1).diffMonths(date), -107);

            //Test B.C.
            auto dateBC = Date(-1999, 7, 6);

            _assertPred!"=="(dateBC.diffMonths(Date(-2000, 6, 5)), 13);
            _assertPred!"=="(dateBC.diffMonths(Date(-2000, 7, 5)), 12);
            _assertPred!"=="(dateBC.diffMonths(Date(-2000, 8, 5)), 11);
            _assertPred!"=="(dateBC.diffMonths(Date(-2000, 9, 5)), 10);
            _assertPred!"=="(dateBC.diffMonths(Date(-2000, 10, 5)), 9);
            _assertPred!"=="(dateBC.diffMonths(Date(-2000, 11, 5)), 8);
            _assertPred!"=="(dateBC.diffMonths(Date(-2000, 12, 5)), 7);
            _assertPred!"=="(dateBC.diffMonths(Date(-1999, 1, 5)), 6);
            _assertPred!"=="(dateBC.diffMonths(Date(-1999, 2, 6)), 5);
            _assertPred!"=="(dateBC.diffMonths(Date(-1999, 3, 6)), 4);
            _assertPred!"=="(dateBC.diffMonths(Date(-1999, 4, 6)), 3);
            _assertPred!"=="(dateBC.diffMonths(Date(-1999, 5, 6)), 2);
            _assertPred!"=="(dateBC.diffMonths(Date(-1999, 6, 6)), 1);
            _assertPred!"=="(dateBC.diffMonths(dateBC), 0);
            _assertPred!"=="(dateBC.diffMonths(Date(-1999, 8, 6)), -1);
            _assertPred!"=="(dateBC.diffMonths(Date(-1999, 9, 6)), -2);
            _assertPred!"=="(dateBC.diffMonths(Date(-1999, 10, 6)), -3);
            _assertPred!"=="(dateBC.diffMonths(Date(-1999, 11, 6)), -4);
            _assertPred!"=="(dateBC.diffMonths(Date(-1999, 12, 6)), -5);
            _assertPred!"=="(dateBC.diffMonths(Date(-1998, 1, 6)), -6);
            _assertPred!"=="(dateBC.diffMonths(Date(-1998, 2, 6)), -7);
            _assertPred!"=="(dateBC.diffMonths(Date(-1998, 3, 6)), -8);
            _assertPred!"=="(dateBC.diffMonths(Date(-1998, 4, 6)), -9);
            _assertPred!"=="(dateBC.diffMonths(Date(-1998, 5, 6)), -10);
            _assertPred!"=="(dateBC.diffMonths(Date(-1998, 6, 6)), -11);
            _assertPred!"=="(dateBC.diffMonths(Date(-1998, 7, 6)), -12);
            _assertPred!"=="(dateBC.diffMonths(Date(-1998, 8, 6)), -13);

            _assertPred!"=="(Date(-2000, 6, 5).diffMonths(dateBC), -13);
            _assertPred!"=="(Date(-2000, 7, 5).diffMonths(dateBC), -12);
            _assertPred!"=="(Date(-2000, 8, 5).diffMonths(dateBC), -11);
            _assertPred!"=="(Date(-2000, 9, 5).diffMonths(dateBC), -10);
            _assertPred!"=="(Date(-2000, 10, 5).diffMonths(dateBC), -9);
            _assertPred!"=="(Date(-2000, 11, 5).diffMonths(dateBC), -8);
            _assertPred!"=="(Date(-2000, 12, 5).diffMonths(dateBC), -7);
            _assertPred!"=="(Date(-1999, 1, 5).diffMonths(dateBC), -6);
            _assertPred!"=="(Date(-1999, 2, 6).diffMonths(dateBC), -5);
            _assertPred!"=="(Date(-1999, 3, 6).diffMonths(dateBC), -4);
            _assertPred!"=="(Date(-1999, 4, 6).diffMonths(dateBC), -3);
            _assertPred!"=="(Date(-1999, 5, 6).diffMonths(dateBC), -2);
            _assertPred!"=="(Date(-1999, 6, 6).diffMonths(dateBC), -1);
            _assertPred!"=="(Date(-1999, 8, 6).diffMonths(dateBC), 1);
            _assertPred!"=="(Date(-1999, 9, 6).diffMonths(dateBC), 2);
            _assertPred!"=="(Date(-1999, 10, 6).diffMonths(dateBC), 3);
            _assertPred!"=="(Date(-1999, 11, 6).diffMonths(dateBC), 4);
            _assertPred!"=="(Date(-1999, 12, 6).diffMonths(dateBC), 5);
            _assertPred!"=="(Date(-1998, 1, 6).diffMonths(dateBC), 6);
            _assertPred!"=="(Date(-1998, 2, 6).diffMonths(dateBC), 7);
            _assertPred!"=="(Date(-1998, 3, 6).diffMonths(dateBC), 8);
            _assertPred!"=="(Date(-1998, 4, 6).diffMonths(dateBC), 9);
            _assertPred!"=="(Date(-1998, 5, 6).diffMonths(dateBC), 10);
            _assertPred!"=="(Date(-1998, 6, 6).diffMonths(dateBC), 11);
            _assertPred!"=="(Date(-1998, 7, 6).diffMonths(dateBC), 12);
            _assertPred!"=="(Date(-1998, 8, 6).diffMonths(dateBC), 13);

            _assertPred!"=="(dateBC.diffMonths(Date(-1999, 6, 30)), 1);
            _assertPred!"=="(dateBC.diffMonths(Date(-1999, 7, 1)), 0);
            _assertPred!"=="(dateBC.diffMonths(Date(-1999, 7, 6)), 0);
            _assertPred!"=="(dateBC.diffMonths(Date(-1999, 7, 11)), 0);
            _assertPred!"=="(dateBC.diffMonths(Date(-1999, 7, 16)), 0);
            _assertPred!"=="(dateBC.diffMonths(Date(-1999, 7, 21)), 0);
            _assertPred!"=="(dateBC.diffMonths(Date(-1999, 7, 26)), 0);
            _assertPred!"=="(dateBC.diffMonths(Date(-1999, 7, 31)), 0);
            _assertPred!"=="(dateBC.diffMonths(Date(-1999, 8, 1)), -1);

            _assertPred!"=="(dateBC.diffMonths(Date(-2008, 6, 30)), 109);
            _assertPred!"=="(dateBC.diffMonths(Date(-2008, 7, 1)), 108);
            _assertPred!"=="(dateBC.diffMonths(Date(-2008, 7, 6)), 108);
            _assertPred!"=="(dateBC.diffMonths(Date(-2008, 7, 11)), 108);
            _assertPred!"=="(dateBC.diffMonths(Date(-2008, 7, 16)), 108);
            _assertPred!"=="(dateBC.diffMonths(Date(-2008, 7, 21)), 108);
            _assertPred!"=="(dateBC.diffMonths(Date(-2008, 7, 26)), 108);
            _assertPred!"=="(dateBC.diffMonths(Date(-2008, 7, 31)), 108);
            _assertPred!"=="(dateBC.diffMonths(Date(-2008, 8, 1)), 107);

            _assertPred!"=="(Date(-1999, 6, 30).diffMonths(dateBC), -1);
            _assertPred!"=="(Date(-1999, 7, 1).diffMonths(dateBC), 0);
            _assertPred!"=="(Date(-1999, 7, 6).diffMonths(dateBC), 0);
            _assertPred!"=="(Date(-1999, 7, 11).diffMonths(dateBC), 0);
            _assertPred!"=="(Date(-1999, 7, 16).diffMonths(dateBC), 0);
            _assertPred!"=="(Date(-1999, 7, 21).diffMonths(dateBC), 0);
            _assertPred!"=="(Date(-1999, 7, 26).diffMonths(dateBC), 0);
            _assertPred!"=="(Date(-1999, 7, 31).diffMonths(dateBC), 0);
            _assertPred!"=="(Date(-1999, 8, 1).diffMonths(dateBC), 1);

            _assertPred!"=="(Date(-2008, 6, 30).diffMonths(dateBC), -109);
            _assertPred!"=="(Date(-2008, 7, 1).diffMonths(dateBC), -108);
            _assertPred!"=="(Date(-2008, 7, 6).diffMonths(dateBC), -108);
            _assertPred!"=="(Date(-2008, 7, 11).diffMonths(dateBC), -108);
            _assertPred!"=="(Date(-2008, 7, 16).diffMonths(dateBC), -108);
            _assertPred!"=="(Date(-2008, 7, 21).diffMonths(dateBC), -108);
            _assertPred!"=="(Date(-2008, 7, 26).diffMonths(dateBC), -108);
            _assertPred!"=="(Date(-2008, 7, 31).diffMonths(dateBC), -108);
            _assertPred!"=="(Date(-2008, 8, 1).diffMonths(dateBC), -107);

            //Test Both
            _assertPred!"=="(Date(3, 3, 3).diffMonths(Date(-5, 5, 5)), 94);
            _assertPred!"=="(Date(-5, 5, 5).diffMonths(Date(3, 3, 3)), -94);

            const cdate = Date(1999, 7, 6);
            immutable idate = Date(1999, 7, 6);
            static assert(__traits(compiles, date.diffMonths(date)));
            static assert(__traits(compiles, cdate.diffMonths(date)));
            static assert(__traits(compiles, idate.diffMonths(date)));

            static assert(__traits(compiles, date.diffMonths(cdate)));
            static assert(__traits(compiles, cdate.diffMonths(cdate)));
            static assert(__traits(compiles, idate.diffMonths(cdate)));

            static assert(__traits(compiles, date.diffMonths(idate)));
            static assert(__traits(compiles, cdate.diffMonths(idate)));
            static assert(__traits(compiles, idate.diffMonths(idate)));

            //Verify Examples.
            assert(Date(1999, 2, 1).diffMonths(Date(1999, 1, 31)) == 1);
            assert(Date(1999, 1, 31).diffMonths(Date(1999, 2, 1)) == -1);
            assert(Date(1999, 3, 1).diffMonths(Date(1999, 1, 1)) == 2);
            assert(Date(1999, 1, 1).diffMonths(Date(1999, 3, 31)) == -2);
        }
    }


    /++
        Whether this $(D Date) is in a leap year.
     +/
    @property bool isLeapYear() const pure nothrow
    {
        return yearIsLeapYear(_year);
    }

    unittest
    {
        version(testStdDateTime)
        {
            auto date = Date(1999, 7, 6);
            const cdate = Date(1999, 7, 6);
            immutable idate = Date(1999, 7, 6);
            static assert(!__traits(compiles, date.isLeapYear = true));
            static assert(!__traits(compiles, cdate.isLeapYear = true));
            static assert(!__traits(compiles, idate.isLeapYear = true));
        }
    }


    /++
        Day of the week this $(D Date) is on.
      +/
    @property DayOfWeek dayOfWeek() const pure nothrow
    {
        return getDayOfWeek(dayOfGregorianCal);
    }

    unittest
    {
        version(testStdDateTime)
        {
            const cdate = Date(1999, 7, 6);
            immutable idate = Date(1999, 7, 6);
            static assert(__traits(compiles, cdate.dayOfWeek == DayOfWeek.sun));
            static assert(!__traits(compiles, cdate.dayOfWeek = DayOfWeek.sun));
            static assert(__traits(compiles, idate.dayOfWeek == DayOfWeek.sun));
            static assert(!__traits(compiles, idate.dayOfWeek = DayOfWeek.sun));
        }
    }


    /++
        Day of the year this $(D Date) is on.

        Examples:
--------------------
assert(Date(1999, 1, 1).dayOfYear == 1);
assert(Date(1999, 12, 31).dayOfYear == 365);
assert(Date(2000, 12, 31).dayOfYear == 366);
--------------------
      +/
    @property ushort dayOfYear() const pure nothrow
    {
        if (_month >= Month.jan && _month <= Month.dec)
        {
            immutable int[] lastDay = isLeapYear ? lastDayLeap : lastDayNonLeap;
            auto monthIndex = _month - Month.jan;

            return cast(ushort)(lastDay[monthIndex] + _day);
        }
        assert(0, "Invalid month.");
    }

    //Verify Examples.
    version(testStdDateTime) unittest
    {
        assert(Date(1999, 1, 1).dayOfYear == 1);
        assert(Date(1999, 12, 31).dayOfYear == 365);
        assert(Date(2000, 12, 31).dayOfYear == 366);
    }

    version(testStdDateTime) unittest
    {
        foreach(year; filter!((a){return !yearIsLeapYear(a);})
                             (chain(testYearsBC, testYearsAD)))
        {
            foreach(doy; testDaysOfYear)
            {
                _assertPred!"=="(Date(year, doy.md.month, doy.md.day).dayOfYear,
                                 doy.day);
            }
        }

        foreach(year; filter!((a){return yearIsLeapYear(a);})
                             (chain(testYearsBC, testYearsAD)))
        {
            foreach(doy; testDaysOfLeapYear)
            {
                _assertPred!"=="(Date(year, doy.md.month, doy.md.day).dayOfYear,
                                 doy.day);
            }
        }

        const cdate = Date(1999, 7, 6);
        immutable idate = Date(1999, 7, 6);
        static assert(__traits(compiles, cdate.dayOfYear == 187));
        static assert(__traits(compiles, idate.dayOfYear == 187));
    }

    /++
        Day of the year.

        Params:
            day = The day of the year to set which day of the year this
                  $(D Date) is on.

        Throws:
            $(D DateTimeException) if the given day is an invalid day of the
            year.
      +/
    @property void dayOfYear(int day) pure
    {
        immutable int[] lastDay = isLeapYear ? lastDayLeap : lastDayNonLeap;

        if(day <= 0 || day > (isLeapYear ? daysInLeapYear : daysInYear) )
            throw new DateTimeException("Invalid day of the year.");

        foreach (i; 1..lastDay.length)
        {
            if (day <= lastDay[i])
            {
                _month = cast(Month)(cast(int)Month.jan + i - 1);
                _day = cast(ubyte)(day - lastDay[i - 1]);
                return;
            }
        }
        assert(0, "Invalid day of the year.");
    }

    version(testStdDateTime) unittest
    {
        static void test(Date date, int day, MonthDay expected, size_t line = __LINE__)
        {
            date.dayOfYear = day;
            _assertPred!"=="(date.month, expected.month, "", __FILE__, line);
            _assertPred!"=="(date.day, expected.day, "", __FILE__, line);
        }

        foreach(doy; testDaysOfYear)
        {
            test(Date(1999, 1, 1), doy.day, doy.md);
            test(Date(-1, 1, 1), doy.day, doy.md);
        }

        foreach(doy; testDaysOfLeapYear)
        {
            test(Date(2000, 1, 1), doy.day, doy.md);
            test(Date(-4, 1, 1), doy.day, doy.md);
        }

        const cdate = Date(1999, 7, 6);
        immutable idate = Date(1999, 7, 6);
        static assert(!__traits(compiles, cdate.dayOfYear = 187));
        static assert(!__traits(compiles, idate.dayOfYear = 187));
    }


    /++
        The Xth day of the Gregorian Calendar that this $(D Date) is on.

        Examples:
--------------------
assert(Date(1, 1, 1).dayOfGregorianCal == 1);
assert(Date(1, 12, 31).dayOfGregorianCal == 365);
assert(Date(2, 1, 1).dayOfGregorianCal == 366);

assert(Date(0, 12, 31).dayOfGregorianCal == 0);
assert(Date(0, 1, 1).dayOfGregorianCal == -365);
assert(Date(-1, 12, 31).dayOfGregorianCal == -366);

assert(Date(2000, 1, 1).dayOfGregorianCal == 730_120);
assert(Date(2010, 12, 31).dayOfGregorianCal == 734_137);
--------------------
     +/
    @property int dayOfGregorianCal() const pure nothrow
    {
        if(isAD)
        {
            if(_year == 1)
                return dayOfYear;

            int years = _year - 1;
            auto days = (years / 400) * daysIn400Years;
            years %= 400;

            days += (years / 100) * daysIn100Years;
            years %= 100;

            days += (years / 4) * daysIn4Years;
            years %= 4;

            days += years * daysInYear;

            days += dayOfYear;

            return days;
        }
        else if(_year == 0)
            return dayOfYear - daysInLeapYear;
        else
        {
            int years = _year;
            auto days = (years / 400) * daysIn400Years;
            years %= 400;

            days += (years / 100) * daysIn100Years;
            years %= 100;

            days += (years / 4) * daysIn4Years;
            years %= 4;

            if(years < 0)
            {
                days -= daysInLeapYear;
                ++years;

                days += years * daysInYear;

                days -= daysInYear - dayOfYear;
            }
            else
                days -= daysInLeapYear - dayOfYear;

            return days;
        }
    }

    //Verify Examples.
    version(testStdDateTime) unittest
    {
        assert(Date(1, 1, 1).dayOfGregorianCal == 1);
        assert(Date(1, 12, 31).dayOfGregorianCal == 365);
        assert(Date(2, 1, 1).dayOfGregorianCal == 366);

        assert(Date(0, 12, 31).dayOfGregorianCal == 0);
        assert(Date(0, 1, 1).dayOfGregorianCal == -365);
        assert(Date(-1, 12, 31).dayOfGregorianCal == -366);

        assert(Date(2000, 1, 1).dayOfGregorianCal == 730_120);
        assert(Date(2010, 12, 31).dayOfGregorianCal == 734_137);
    }

    version(testStdDateTime) unittest
    {
        foreach(gd; chain(testGregDaysBC, testGregDaysAD))
            _assertPred!"=="(gd.date.dayOfGregorianCal, gd.day);

        auto date = Date(1999, 7, 6);
        const cdate = Date(1999, 7, 6);
        immutable idate = Date(1999, 7, 6);
        static assert(__traits(compiles, date.dayOfGregorianCal));
        static assert(__traits(compiles, cdate.dayOfGregorianCal));
        static assert(__traits(compiles, idate.dayOfGregorianCal));
    }

    /++
        The Xth day of the Gregorian Calendar that this $(D Date) is on.

        Params:
            day = The day of the Gregorian Calendar to set this $(D Date) to.

        Examples:
--------------------
auto date = Date.init;
date.dayOfGregorianCal = 1;
assert(date == Date(1, 1, 1));

date.dayOfGregorianCal = 365;
assert(date == Date(1, 12, 31));

date.dayOfGregorianCal = 366;
assert(date == Date(2, 1, 1));

date.dayOfGregorianCal = 0;
assert(date == Date(0, 12, 31));

date.dayOfGregorianCal = -365;
assert(date == Date(-0, 1, 1));

date.dayOfGregorianCal = -366;
assert(date == Date(-1, 12, 31));

date.dayOfGregorianCal = 730_120;
assert(date == Date(2000, 1, 1));

date.dayOfGregorianCal = 734_137;
assert(date == Date(2010, 12, 31));
--------------------
     +/
    @property void dayOfGregorianCal(int day) pure nothrow
    {
        this = Date(day);
    }

    unittest
    {
        version(testStdDateTime)
        {
            {
                auto date = Date(1999, 7, 6);
                const cdate = Date(1999, 7, 6);
                immutable idate = Date(1999, 7, 6);
                static assert(__traits(compiles, date.dayOfGregorianCal = 187));
                static assert(!__traits(compiles, cdate.dayOfGregorianCal = 187));
                static assert(!__traits(compiles, idate.dayOfGregorianCal = 187));
            }

            //Verify Examples.
            {
                auto date = Date.init;
                date.dayOfGregorianCal = 1;
                assert(date == Date(1, 1, 1));

                date.dayOfGregorianCal = 365;
                assert(date == Date(1, 12, 31));

                date.dayOfGregorianCal = 366;
                assert(date == Date(2, 1, 1));

                date.dayOfGregorianCal = 0;
                assert(date == Date(0, 12, 31));

                date.dayOfGregorianCal = -365;
                assert(date == Date(-0, 1, 1));

                date.dayOfGregorianCal = -366;
                assert(date == Date(-1, 12, 31));

                date.dayOfGregorianCal = 730_120;
                assert(date == Date(2000, 1, 1));

                date.dayOfGregorianCal = 734_137;
                assert(date == Date(2010, 12, 31));
            }
        }
    }


    /++
        The ISO 8601 week of the year that this $(D Date) is in.

        See_Also:
            $(WEB en.wikipedia.org/wiki/ISO_week_date, ISO Week Date)
      +/
    @property ubyte isoWeek() const pure nothrow
    {
        immutable weekday = dayOfWeek;
        immutable adjustedWeekday = weekday == DayOfWeek.sun ? 7 : weekday;
        immutable week = (dayOfYear - adjustedWeekday + 10) / 7;

        try
        {
            if(week == 53)
            {
                switch(Date(_year + 1, 1, 1).dayOfWeek)
                {
                    case DayOfWeek.mon:
                    case DayOfWeek.tue:
                    case DayOfWeek.wed:
                    case DayOfWeek.thu:
                        return 1;
                    case DayOfWeek.fri:
                    case DayOfWeek.sat:
                    case DayOfWeek.sun:
                        return 53;
                    default:
                        assert(0, "Invalid ISO Week");
                }
            }
            else if(week > 0)
                return cast(ubyte)week;
            else
                return Date(_year - 1, 12, 31).isoWeek;
        }
        catch(Exception e)
            assert(0, "Date's constructor threw.");
    }

    unittest
    {
        version(testStdDateTime)
        {
            //Test A.D.
            _assertPred!"=="(Date(2009, 12, 28).isoWeek, 53);
            _assertPred!"=="(Date(2009, 12, 29).isoWeek, 53);
            _assertPred!"=="(Date(2009, 12, 30).isoWeek, 53);
            _assertPred!"=="(Date(2009, 12, 31).isoWeek, 53);
            _assertPred!"=="(Date(2010, 1, 1).isoWeek, 53);
            _assertPred!"=="(Date(2010, 1, 2).isoWeek, 53);
            _assertPred!"=="(Date(2010, 1, 3).isoWeek, 53);
            _assertPred!"=="(Date(2010, 1, 4).isoWeek, 1);
            _assertPred!"=="(Date(2010, 1, 5).isoWeek, 1);
            _assertPred!"=="(Date(2010, 1, 6).isoWeek, 1);
            _assertPred!"=="(Date(2010, 1, 7).isoWeek, 1);
            _assertPred!"=="(Date(2010, 1, 8).isoWeek, 1);
            _assertPred!"=="(Date(2010, 1, 9).isoWeek, 1);
            _assertPred!"=="(Date(2010, 1, 10).isoWeek, 1);
            _assertPred!"=="(Date(2010, 1, 11).isoWeek, 2);
            _assertPred!"=="(Date(2010, 12, 31).isoWeek, 52);

            _assertPred!"=="(Date(2004, 12, 26).isoWeek, 52);
            _assertPred!"=="(Date(2004, 12, 27).isoWeek, 53);
            _assertPred!"=="(Date(2004, 12, 28).isoWeek, 53);
            _assertPred!"=="(Date(2004, 12, 29).isoWeek, 53);
            _assertPred!"=="(Date(2004, 12, 30).isoWeek, 53);
            _assertPred!"=="(Date(2004, 12, 31).isoWeek, 53);
            _assertPred!"=="(Date(2005, 1, 1).isoWeek, 53);
            _assertPred!"=="(Date(2005, 1, 2).isoWeek, 53);

            _assertPred!"=="(Date(2005, 12, 31).isoWeek, 52);
            _assertPred!"=="(Date(2007, 1, 1).isoWeek, 1);

            _assertPred!"=="(Date(2007, 12, 30).isoWeek, 52);
            _assertPred!"=="(Date(2007, 12, 31).isoWeek, 1);
            _assertPred!"=="(Date(2008, 1, 1).isoWeek, 1);

            _assertPred!"=="(Date(2008, 12, 28).isoWeek, 52);
            _assertPred!"=="(Date(2008, 12, 29).isoWeek, 1);
            _assertPred!"=="(Date(2008, 12, 30).isoWeek, 1);
            _assertPred!"=="(Date(2008, 12, 31).isoWeek, 1);
            _assertPred!"=="(Date(2009, 1, 1).isoWeek, 1);
            _assertPred!"=="(Date(2009, 1, 2).isoWeek, 1);
            _assertPred!"=="(Date(2009, 1, 3).isoWeek, 1);
            _assertPred!"=="(Date(2009, 1, 4).isoWeek, 1);

            //Test B.C.
            //The algorithm should work identically for both A.D. and B.C. since
            //it doesn't really take the year into account, so B.C. testing
            //probably isn't really needed.
            _assertPred!"=="(Date(0, 12, 31).isoWeek, 52);
            _assertPred!"=="(Date(0, 1, 4).isoWeek, 1);
            _assertPred!"=="(Date(0, 1, 1).isoWeek, 52);

            const cdate = Date(1999, 7, 6);
            immutable idate = Date(1999, 7, 6);
            static assert(__traits(compiles, cdate.isoWeek == 3));
            static assert(!__traits(compiles, cdate.isoWeek = 3));
            static assert(__traits(compiles, idate.isoWeek == 3));
            static assert(!__traits(compiles, idate.isoWeek = 3));
        }
    }


    /++
        $(D Date) for the last day in the month that this $(D Date) is in.

        Examples:
--------------------
assert(Date(1999, 1, 6).endOfMonth == Date(1999, 1, 31));
assert(Date(1999, 2, 7).endOfMonth == Date(1999, 2, 28));
assert(Date(2000, 2, 7).endOfMonth == Date(1999, 2, 29));
assert(Date(2000, 6, 4).endOfMonth == Date(1999, 6, 30));
--------------------
      +/
    @property Date endOfMonth() const pure nothrow
    {
        try
            return Date(_year, _month, maxDay(_year, _month));
        catch(Exception e)
            assert(0, "Date's constructor threw.");
    }

    unittest
    {
        version(testStdDateTime)
        {
            //Test A.D.
            _assertPred!"=="(Date(1999, 1, 1).endOfMonth, Date(1999, 1, 31));
            _assertPred!"=="(Date(1999, 2, 1).endOfMonth, Date(1999, 2, 28));
            _assertPred!"=="(Date(2000, 2, 1).endOfMonth, Date(2000, 2, 29));
            _assertPred!"=="(Date(1999, 3, 1).endOfMonth, Date(1999, 3, 31));
            _assertPred!"=="(Date(1999, 4, 1).endOfMonth, Date(1999, 4, 30));
            _assertPred!"=="(Date(1999, 5, 1).endOfMonth, Date(1999, 5, 31));
            _assertPred!"=="(Date(1999, 6, 1).endOfMonth, Date(1999, 6, 30));
            _assertPred!"=="(Date(1999, 7, 1).endOfMonth, Date(1999, 7, 31));
            _assertPred!"=="(Date(1999, 8, 1).endOfMonth, Date(1999, 8, 31));
            _assertPred!"=="(Date(1999, 9, 1).endOfMonth, Date(1999, 9, 30));
            _assertPred!"=="(Date(1999, 10, 1).endOfMonth, Date(1999, 10, 31));
            _assertPred!"=="(Date(1999, 11, 1).endOfMonth, Date(1999, 11, 30));
            _assertPred!"=="(Date(1999, 12, 1).endOfMonth, Date(1999, 12, 31));

            //Test B.C.
            _assertPred!"=="(Date(-1999, 1, 1).endOfMonth, Date(-1999, 1, 31));
            _assertPred!"=="(Date(-1999, 2, 1).endOfMonth, Date(-1999, 2, 28));
            _assertPred!"=="(Date(-2000, 2, 1).endOfMonth, Date(-2000, 2, 29));
            _assertPred!"=="(Date(-1999, 3, 1).endOfMonth, Date(-1999, 3, 31));
            _assertPred!"=="(Date(-1999, 4, 1).endOfMonth, Date(-1999, 4, 30));
            _assertPred!"=="(Date(-1999, 5, 1).endOfMonth, Date(-1999, 5, 31));
            _assertPred!"=="(Date(-1999, 6, 1).endOfMonth, Date(-1999, 6, 30));
            _assertPred!"=="(Date(-1999, 7, 1).endOfMonth, Date(-1999, 7, 31));
            _assertPred!"=="(Date(-1999, 8, 1).endOfMonth, Date(-1999, 8, 31));
            _assertPred!"=="(Date(-1999, 9, 1).endOfMonth, Date(-1999, 9, 30));
            _assertPred!"=="(Date(-1999, 10, 1).endOfMonth, Date(-1999, 10, 31));
            _assertPred!"=="(Date(-1999, 11, 1).endOfMonth, Date(-1999, 11, 30));
            _assertPred!"=="(Date(-1999, 12, 1).endOfMonth, Date(-1999, 12, 31));

            const cdate = Date(1999, 7, 6);
            immutable idate = Date(1999, 7, 6);
            static assert(!__traits(compiles, cdate.endOfMonth = Date(1999, 7, 30)));
            static assert(!__traits(compiles, idate.endOfMonth = Date(1999, 7, 30)));

            //Verify Examples.
            assert(Date(1999, 1, 6).endOfMonth == Date(1999, 1, 31));
            assert(Date(1999, 2, 7).endOfMonth == Date(1999, 2, 28));
            assert(Date(2000, 2, 7).endOfMonth == Date(2000, 2, 29));
            assert(Date(2000, 6, 4).endOfMonth == Date(2000, 6, 30));
        }
    }


    /++
        The last day in the month that this $(D Date) is in.

        Examples:
--------------------
assert(Date(1999, 1, 6).daysInMonth == 31);
assert(Date(1999, 2, 7).daysInMonth == 28);
assert(Date(2000, 2, 7).daysInMonth == 29);
assert(Date(2000, 6, 4).daysInMonth == 30);
--------------------
      +/
    @property ubyte daysInMonth() const pure nothrow
    {
        return maxDay(_year, _month);
    }

    /++
        $(RED Deprecated. It will be removed in September 2012.
              Please use daysInMonth instead.)
      +/
    deprecated @property ubyte endOfMonthDay() const pure nothrow
    {
        return maxDay(_year, _month);
    }

    unittest
    {
        version(testStdDateTime)
        {
            //Test A.D.
            _assertPred!"=="(Date(1999, 1, 1).daysInMonth, 31);
            _assertPred!"=="(Date(1999, 2, 1).daysInMonth, 28);
            _assertPred!"=="(Date(2000, 2, 1).daysInMonth, 29);
            _assertPred!"=="(Date(1999, 3, 1).daysInMonth, 31);
            _assertPred!"=="(Date(1999, 4, 1).daysInMonth, 30);
            _assertPred!"=="(Date(1999, 5, 1).daysInMonth, 31);
            _assertPred!"=="(Date(1999, 6, 1).daysInMonth, 30);
            _assertPred!"=="(Date(1999, 7, 1).daysInMonth, 31);
            _assertPred!"=="(Date(1999, 8, 1).daysInMonth, 31);
            _assertPred!"=="(Date(1999, 9, 1).daysInMonth, 30);
            _assertPred!"=="(Date(1999, 10, 1).daysInMonth, 31);
            _assertPred!"=="(Date(1999, 11, 1).daysInMonth, 30);
            _assertPred!"=="(Date(1999, 12, 1).daysInMonth, 31);

            //Test B.C.
            _assertPred!"=="(Date(-1999, 1, 1).daysInMonth, 31);
            _assertPred!"=="(Date(-1999, 2, 1).daysInMonth, 28);
            _assertPred!"=="(Date(-2000, 2, 1).daysInMonth, 29);
            _assertPred!"=="(Date(-1999, 3, 1).daysInMonth, 31);
            _assertPred!"=="(Date(-1999, 4, 1).daysInMonth, 30);
            _assertPred!"=="(Date(-1999, 5, 1).daysInMonth, 31);
            _assertPred!"=="(Date(-1999, 6, 1).daysInMonth, 30);
            _assertPred!"=="(Date(-1999, 7, 1).daysInMonth, 31);
            _assertPred!"=="(Date(-1999, 8, 1).daysInMonth, 31);
            _assertPred!"=="(Date(-1999, 9, 1).daysInMonth, 30);
            _assertPred!"=="(Date(-1999, 10, 1).daysInMonth, 31);
            _assertPred!"=="(Date(-1999, 11, 1).daysInMonth, 30);
            _assertPred!"=="(Date(-1999, 12, 1).daysInMonth, 31);

            const cdate = Date(1999, 7, 6);
            immutable idate = Date(1999, 7, 6);
            static assert(!__traits(compiles, cdate.daysInMonth = 30));
            static assert(!__traits(compiles, idate.daysInMonth = 30));

            //Verify Examples.
            assert(Date(1999, 1, 6).daysInMonth == 31);
            assert(Date(1999, 2, 7).daysInMonth == 28);
            assert(Date(2000, 2, 7).daysInMonth == 29);
            assert(Date(2000, 6, 4).daysInMonth == 30);
        }
    }


    /++
        Whether the current year is a date in A.D.

        Examples:
--------------------
assert(Date(1, 1, 1).isAD);
assert(Date(2010, 12, 31).isAD);
assert(!Date(0, 12, 31).isAD);
assert(!Date(-2010, 1, 1).isAD);
--------------------
      +/
    @property bool isAD() const pure nothrow
    {
        return _year > 0;
    }

    unittest
    {
        version(testStdDateTime)
        {
            assert(Date(2010, 7, 4).isAD);
            assert(Date(1, 1, 1).isAD);
            assert(!Date(0, 1, 1).isAD);
            assert(!Date(-1, 1, 1).isAD);
            assert(!Date(-2010, 7, 4).isAD);

            const cdate = Date(1999, 7, 6);
            immutable idate = Date(1999, 7, 6);
            static assert(__traits(compiles, cdate.isAD));
            static assert(__traits(compiles, idate.isAD));

            //Verify Examples.
            assert(Date(1, 1, 1).isAD);
            assert(Date(2010, 12, 31).isAD);
            assert(!Date(0, 12, 31).isAD);
            assert(!Date(-2010, 1, 1).isAD);
        }
    }


    /++
        The julian day for this $(D Date) at noon (since the julian day changes
        at noon).
      +/
    @property long julianDay() const pure nothrow
    {
        return dayOfGregorianCal + 1_721_425;
    }

    unittest
    {
        version(testStdDateTime)
        {
            _assertPred!"=="(Date(-4713, 11, 24).julianDay, 0);
            _assertPred!"=="(Date(0, 12, 31).julianDay, 1_721_425);
            _assertPred!"=="(Date(1, 1, 1).julianDay, 1_721_426);
            _assertPred!"=="(Date(1582, 10, 15).julianDay, 2_299_161);
            _assertPred!"=="(Date(1858, 11, 17).julianDay, 2_400_001);
            _assertPred!"=="(Date(1982, 1, 4).julianDay, 2_444_974);
            _assertPred!"=="(Date(1996, 3, 31).julianDay, 2_450_174);
            _assertPred!"=="(Date(2010, 8, 24).julianDay, 2_455_433);

            const cdate = Date(1999, 7, 6);
            immutable idate = Date(1999, 7, 6);
            static assert(__traits(compiles, cdate.julianDay));
            static assert(__traits(compiles, idate.julianDay));
        }
    }


    /++
        The modified julian day for any time on this date (since, the modified
        julian day changes at midnight).
      +/
    @property long modJulianDay() const pure nothrow
    {
        return julianDay - 2_400_001;
    }

    unittest
    {
        version(testStdDateTime)
        {
            _assertPred!"=="(Date(1858, 11, 17).modJulianDay, 0);
            _assertPred!"=="(Date(2010, 8, 24).modJulianDay, 55_432);

            const cdate = Date(1999, 7, 6);
            immutable idate = Date(1999, 7, 6);
            static assert(__traits(compiles, cdate.modJulianDay));
            static assert(__traits(compiles, idate.modJulianDay));
        }
    }


    /++
        Converts this $(D Date) to a string with the format YYYYMMDD.

        Examples:
--------------------
assert(Date(2010, 7, 4).toISOString() == "20100704");
assert(Date(1998, 12, 25).toISOString() == "19981225");
assert(Date(0, 1, 5).toISOString() == "00000105");
assert(Date(-4, 1, 5).toISOString() == "-00040105");
--------------------
      +/
    string toISOString() const nothrow
    {
        try
        {
            if(_year >= 0)
            {
                if(_year < 10_000)
                    return format("%04d%02d%02d", _year, _month, _day);
                else
                    return format("+%05d%02d%02d", _year, _month, _day);
            }
            else if(_year > -10_000)
                return format("%05d%02d%02d", _year, _month, _day);
            else
                return format("%06d%02d%02d", _year, _month, _day);
        }
        catch(Exception e)
            assert(0, "format() threw.");
    }

    unittest
    {
        version(testStdDateTime)
        {
            //Test A.D.
            _assertPred!"=="(Date(9, 12, 4).toISOString(), "00091204");
            _assertPred!"=="(Date(99, 12, 4).toISOString(), "00991204");
            _assertPred!"=="(Date(999, 12, 4).toISOString(), "09991204");
            _assertPred!"=="(Date(9999, 7, 4).toISOString(), "99990704");
            _assertPred!"=="(Date(10000, 10, 20).toISOString(), "+100001020");

            //Test B.C.
            _assertPred!"=="(Date(0, 12, 4).toISOString(), "00001204");
            _assertPred!"=="(Date(-9, 12, 4).toISOString(), "-00091204");
            _assertPred!"=="(Date(-99, 12, 4).toISOString(), "-00991204");
            _assertPred!"=="(Date(-999, 12, 4).toISOString(), "-09991204");
            _assertPred!"=="(Date(-9999, 7, 4).toISOString(), "-99990704");
            _assertPred!"=="(Date(-10000, 10, 20).toISOString(), "-100001020");

            const cdate = Date(1999, 7, 6);
            immutable idate = Date(1999, 7, 6);
            static assert(__traits(compiles, cdate.toISOString()));
            static assert(__traits(compiles, idate.toISOString()));

            //Verify Examples.
            assert(Date(2010, 7, 4).toISOString() == "20100704");
            assert(Date(1998, 12, 25).toISOString() == "19981225");
            assert(Date(0, 1, 5).toISOString() == "00000105");
            assert(Date(-4, 1, 5).toISOString() == "-00040105");
        }
    }

    /++
        Converts this $(D Date) to a string with the format YYYY-MM-DD.

        Examples:
--------------------
assert(Date(2010, 7, 4).toISOExtString() == "2010-07-04");
assert(Date(1998, 12, 25).toISOExtString() == "1998-12-25");
assert(Date(0, 1, 5).toISOExtString() == "0000-01-05");
assert(Date(-4, 1, 5).toISOExtString() == "-0004-01-05");
--------------------
      +/
    string toISOExtString() const nothrow
    {
        try
        {
            if(_year >= 0)
            {
                if(_year < 10_000)
                    return format("%04d-%02d-%02d", _year, _month, _day);
                else
                    return format("+%05d-%02d-%02d", _year, _month, _day);
            }
            else if(_year > -10_000)
                return format("%05d-%02d-%02d", _year, _month, _day);
            else
                return format("%06d-%02d-%02d", _year, _month, _day);
        }
        catch(Exception e)
            assert(0, "format() threw.");
    }

    unittest
    {
        version(testStdDateTime)
        {
            //Test A.D.
            _assertPred!"=="(Date(9, 12, 4).toISOExtString(), "0009-12-04");
            _assertPred!"=="(Date(99, 12, 4).toISOExtString(), "0099-12-04");
            _assertPred!"=="(Date(999, 12, 4).toISOExtString(), "0999-12-04");
            _assertPred!"=="(Date(9999, 7, 4).toISOExtString(), "9999-07-04");
            _assertPred!"=="(Date(10000, 10, 20).toISOExtString(), "+10000-10-20");

            //Test B.C.
            _assertPred!"=="(Date(0, 12, 4).toISOExtString(), "0000-12-04");
            _assertPred!"=="(Date(-9, 12, 4).toISOExtString(), "-0009-12-04");
            _assertPred!"=="(Date(-99, 12, 4).toISOExtString(), "-0099-12-04");
            _assertPred!"=="(Date(-999, 12, 4).toISOExtString(), "-0999-12-04");
            _assertPred!"=="(Date(-9999, 7, 4).toISOExtString(), "-9999-07-04");
            _assertPred!"=="(Date(-10000, 10, 20).toISOExtString(), "-10000-10-20");

            const cdate = Date(1999, 7, 6);
            immutable idate = Date(1999, 7, 6);
            static assert(__traits(compiles, cdate.toISOExtString()));
            static assert(__traits(compiles, idate.toISOExtString()));

            //Verify Examples.
            assert(Date(2010, 7, 4).toISOExtString() == "2010-07-04");
            assert(Date(1998, 12, 25).toISOExtString() == "1998-12-25");
            assert(Date(0, 1, 5).toISOExtString() == "0000-01-05");
            assert(Date(-4, 1, 5).toISOExtString() == "-0004-01-05");
        }
    }

    /++
        Converts this $(D Date) to a string with the format YYYY-Mon-DD.

        Examples:
--------------------
assert(Date(2010, 7, 4).toSimpleString() == "2010-Jul-04");
assert(Date(1998, 12, 25).toSimpleString() == "1998-Dec-25");
assert(Date(0, 1, 5).toSimpleString() == "0000-Jan-05");
assert(Date(-4, 1, 5).toSimpleString() == "-0004-Jan-05");
--------------------
      +/
    string toSimpleString() const nothrow
    {
        try
        {
            if(_year >= 0)
            {
                if(_year < 10_000)
                    return format("%04d-%s-%02d", _year, monthToString(_month, false), _day);
                else
                    return format("+%05d-%s-%02d", _year, monthToString(_month, false), _day);
            }
            else if(_year > -10_000)
                return format("%05d-%s-%02d", _year, monthToString(_month, false), _day);
            else
                return format("%06d-%s-%02d", _year, monthToString(_month, false), _day);
        }
        catch(Exception e)
            assert(0, "format() threw.");
    }

    unittest
    {
        version(testStdDateTime)
        {
            //Test A.D.
            _assertPred!"=="(Date(9, 12, 4).toSimpleString(), "0009-Dec-04");
            _assertPred!"=="(Date(99, 12, 4).toSimpleString(), "0099-Dec-04");
            _assertPred!"=="(Date(999, 12, 4).toSimpleString(), "0999-Dec-04");
            _assertPred!"=="(Date(9999, 7, 4).toSimpleString(), "9999-Jul-04");
            _assertPred!"=="(Date(10000, 10, 20).toSimpleString(), "+10000-Oct-20");

            //Test B.C.
            _assertPred!"=="(Date(0, 12, 4).toSimpleString(), "0000-Dec-04");
            _assertPred!"=="(Date(-9, 12, 4).toSimpleString(), "-0009-Dec-04");
            _assertPred!"=="(Date(-99, 12, 4).toSimpleString(), "-0099-Dec-04");
            _assertPred!"=="(Date(-999, 12, 4).toSimpleString(), "-0999-Dec-04");
            _assertPred!"=="(Date(-9999, 7, 4).toSimpleString(), "-9999-Jul-04");
            _assertPred!"=="(Date(-10000, 10, 20).toSimpleString(), "-10000-Oct-20");

            const cdate = Date(1999, 7, 6);
            immutable idate = Date(1999, 7, 6);
            static assert(__traits(compiles, cdate.toSimpleString()));
            static assert(__traits(compiles, idate.toSimpleString()));

            //Verify Examples.
            assert(Date(2010, 7, 4).toSimpleString() == "2010-Jul-04");
            assert(Date(1998, 12, 25).toSimpleString() == "1998-Dec-25");
            assert(Date(0, 1, 5).toSimpleString() == "0000-Jan-05");
            assert(Date(-4, 1, 5).toSimpleString() == "-0004-Jan-05");
        }
    }

    //TODO Add a function which returns a string in a user-specified format.


    /+
        Converts this $(D Date) to a string.
      +/
    //Due to bug http://d.puremagic.com/issues/show_bug.cgi?id=3715 , we can't
    //have versions of toString() with extra modifiers, so we define one version
    //with modifiers and one without.
    string toString()
    {
        return toSimpleString();
    }

    /++
        Converts this $(D Date) to a string.
      +/
    //Due to bug http://d.puremagic.com/issues/show_bug.cgi?id=3715 , we can't
    //have versions of toString() with extra modifiers, so we define one version
    //with modifiers and one without.
    string toString() const nothrow
    {
        return toSimpleString();
    }

    unittest
    {
        version(testStdDateTime)
        {
            auto date = Date(1999, 7, 6);
            const cdate = Date(1999, 7, 6);
            immutable idate = Date(1999, 7, 6);
            static assert(__traits(compiles, date.toString()));
            static assert(__traits(compiles, cdate.toString()));
            static assert(__traits(compiles, idate.toString()));
        }
    }


    /++
        Creates a $(D Date) from a string with the format YYYYMMDD. Whitespace
        is stripped from the given string.

        Params:
            isoString = A string formatted in the ISO format for dates.

        Throws:
            $(D DateTimeException) if the given string is not in the ISO format
            or if the resulting $(D Date) would not be valid.

        Examples:
--------------------
assert(Date.fromISOString("20100704") == Date(2010, 7, 4));
assert(Date.fromISOString("19981225") == Date(1998, 12, 25));
assert(Date.fromISOString("00000105") == Date(0, 1, 5));
assert(Date.fromISOString("-00040105") == Date(-4, 1, 5));
assert(Date.fromISOString(" 20100704 ") == Date(2010, 7, 4));
--------------------
      +/
    static Date fromISOString(S)(in S isoString)
        if(isSomeString!S)
    {
        auto dstr = to!dstring(strip(isoString));

        enforce(dstr.length >= 8, new DateTimeException(format("Invalid ISO String: %s", isoString)));

        auto day = dstr[$-2 .. $];
        auto month = dstr[$-4 .. $-2];
        auto year = dstr[0 .. $-4];

        enforce(!canFind!(not!isDigit)(day), new DateTimeException(format("Invalid ISO String: %s", isoString)));
        enforce(!canFind!(not!isDigit)(month), new DateTimeException(format("Invalid ISO String: %s", isoString)));

        if(year.length > 4)
        {
            enforce(year.startsWith("-") || year.startsWith("+"),
                    new DateTimeException(format("Invalid ISO String: %s", isoString)));
            enforce(!canFind!(not!isDigit)(year[1..$]),
                    new DateTimeException(format("Invalid ISO String: %s", isoString)));
        }
        else
            enforce(!canFind!(not!isDigit)(year), new DateTimeException(format("Invalid ISO String: %s", isoString)));

        return Date(to!short(year), to!ubyte(month), to!ubyte(day));
    }

    unittest
    {
        version(testStdDateTime)
        {
            assertThrown!DateTimeException(Date.fromISOString(""));
            assertThrown!DateTimeException(Date.fromISOString("990704"));
            assertThrown!DateTimeException(Date.fromISOString("0100704"));
            assertThrown!DateTimeException(Date.fromISOString("2010070"));
            assertThrown!DateTimeException(Date.fromISOString("2010070 "));
            assertThrown!DateTimeException(Date.fromISOString("120100704"));
            assertThrown!DateTimeException(Date.fromISOString("-0100704"));
            assertThrown!DateTimeException(Date.fromISOString("+0100704"));
            assertThrown!DateTimeException(Date.fromISOString("2010070a"));
            assertThrown!DateTimeException(Date.fromISOString("20100a04"));
            assertThrown!DateTimeException(Date.fromISOString("2010a704"));

            assertThrown!DateTimeException(Date.fromISOString("99-07-04"));
            assertThrown!DateTimeException(Date.fromISOString("010-07-04"));
            assertThrown!DateTimeException(Date.fromISOString("2010-07-0"));
            assertThrown!DateTimeException(Date.fromISOString("2010-07-0 "));
            assertThrown!DateTimeException(Date.fromISOString("12010-07-04"));
            assertThrown!DateTimeException(Date.fromISOString("-010-07-04"));
            assertThrown!DateTimeException(Date.fromISOString("+010-07-04"));
            assertThrown!DateTimeException(Date.fromISOString("2010-07-0a"));
            assertThrown!DateTimeException(Date.fromISOString("2010-0a-04"));
            assertThrown!DateTimeException(Date.fromISOString("2010-a7-04"));
            assertThrown!DateTimeException(Date.fromISOString("2010/07/04"));
            assertThrown!DateTimeException(Date.fromISOString("2010/7/04"));
            assertThrown!DateTimeException(Date.fromISOString("2010/7/4"));
            assertThrown!DateTimeException(Date.fromISOString("2010/07/4"));
            assertThrown!DateTimeException(Date.fromISOString("2010-7-04"));
            assertThrown!DateTimeException(Date.fromISOString("2010-7-4"));
            assertThrown!DateTimeException(Date.fromISOString("2010-07-4"));

            assertThrown!DateTimeException(Date.fromISOString("99Jul04"));
            assertThrown!DateTimeException(Date.fromISOString("010Jul04"));
            assertThrown!DateTimeException(Date.fromISOString("2010Jul0"));
            assertThrown!DateTimeException(Date.fromISOString("2010Jul0 "));
            assertThrown!DateTimeException(Date.fromISOString("12010Jul04"));
            assertThrown!DateTimeException(Date.fromISOString("-010Jul04"));
            assertThrown!DateTimeException(Date.fromISOString("+010Jul04"));
            assertThrown!DateTimeException(Date.fromISOString("2010Jul0a"));
            assertThrown!DateTimeException(Date.fromISOString("2010Jua04"));
            assertThrown!DateTimeException(Date.fromISOString("2010aul04"));

            assertThrown!DateTimeException(Date.fromISOString("99-Jul-04"));
            assertThrown!DateTimeException(Date.fromISOString("010-Jul-04"));
            assertThrown!DateTimeException(Date.fromISOString("2010-Jul-0"));
            assertThrown!DateTimeException(Date.fromISOString("2010-Jul-0 "));
            assertThrown!DateTimeException(Date.fromISOString("12010-Jul-04"));
            assertThrown!DateTimeException(Date.fromISOString("-010-Jul-04"));
            assertThrown!DateTimeException(Date.fromISOString("+010-Jul-04"));
            assertThrown!DateTimeException(Date.fromISOString("2010-Jul-0a"));
            assertThrown!DateTimeException(Date.fromISOString("2010-Jua-04"));
            assertThrown!DateTimeException(Date.fromISOString("2010-Jal-04"));
            assertThrown!DateTimeException(Date.fromISOString("2010-aul-04"));

            assertThrown!DateTimeException(Date.fromISOString("2010-07-04"));
            assertThrown!DateTimeException(Date.fromISOString("2010-Jul-04"));

            _assertPred!"=="(Date.fromISOString("19990706"), Date(1999, 7, 6));
            _assertPred!"=="(Date.fromISOString("-19990706"), Date(-1999, 7, 6));
            _assertPred!"=="(Date.fromISOString("+019990706"), Date(1999, 7, 6));
            _assertPred!"=="(Date.fromISOString("19990706 "), Date(1999, 7, 6));
            _assertPred!"=="(Date.fromISOString(" 19990706"), Date(1999, 7, 6));
            _assertPred!"=="(Date.fromISOString(" 19990706 "), Date(1999, 7, 6));

            //Verify Examples.
            assert(Date.fromISOString("20100704") == Date(2010, 7, 4));
            assert(Date.fromISOString("19981225") == Date(1998, 12, 25));
            assert(Date.fromISOString("00000105") == Date(0, 1, 5));
            assert(Date.fromISOString("-00040105") == Date(-4, 1, 5));
            assert(Date.fromISOString(" 20100704 ") == Date(2010, 7, 4));
        }
    }


    /++
        Creates a $(D Date) from a string with the format YYYY-MM-DD. Whitespace
        is stripped from the given string.

        Params:
            isoExtString = A string formatted in the ISO Extended format for
                           dates.

        Throws:
            $(D DateTimeException) if the given string is not in the ISO
            Extended format or if the resulting $(D Date) would not be valid.

        Examples:
--------------------
assert(Date.fromISOExtString("2010-07-04") == Date(2010, 7, 4));
assert(Date.fromISOExtString("1998-12-25") == Date(1998, 12, 25));
assert(Date.fromISOExtString("0000-01-05") == Date(0, 1, 5));
assert(Date.fromISOExtString("-0004-01-05") == Date(-4, 1, 5));
assert(Date.fromISOExtString(" 2010-07-04 ") == Date(2010, 7, 4));
--------------------
      +/
    static Date fromISOExtString(S)(in S isoExtString)
        if(isSomeString!(S))
    {
        auto dstr = to!dstring(strip(isoExtString));

        enforce(dstr.length >= 10, new DateTimeException(format("Invalid ISO Extended String: %s", isoExtString)));

        auto day = dstr[$-2 .. $];
        auto month = dstr[$-5 .. $-3];
        auto year = dstr[0 .. $-6];

        enforce(dstr[$-3] == '-', new DateTimeException(format("Invalid ISO Extended String: %s", isoExtString)));
        enforce(dstr[$-6] == '-', new DateTimeException(format("Invalid ISO Extended String: %s", isoExtString)));
        enforce(!canFind!(not!isDigit)(day),
                new DateTimeException(format("Invalid ISO Extended String: %s", isoExtString)));
        enforce(!canFind!(not!isDigit)(month),
                new DateTimeException(format("Invalid ISO Extended String: %s", isoExtString)));

        if(year.length > 4)
        {
            enforce(year.startsWith("-") || year.startsWith("+"),
                    new DateTimeException(format("Invalid ISO Extended String: %s", isoExtString)));
            enforce(!canFind!(not!isDigit)(year[1..$]),
                    new DateTimeException(format("Invalid ISO Extended String: %s", isoExtString)));
        }
        else
            enforce(!canFind!(not!isDigit)(year),
                    new DateTimeException(format("Invalid ISO Extended String: %s", isoExtString)));

        return Date(to!short(year), to!ubyte(month), to!ubyte(day));
    }

    unittest
    {
        version(testStdDateTime)
        {
            assertThrown!DateTimeException(Date.fromISOExtString(""));
            assertThrown!DateTimeException(Date.fromISOExtString("990704"));
            assertThrown!DateTimeException(Date.fromISOExtString("0100704"));
            assertThrown!DateTimeException(Date.fromISOExtString("2010070"));
            assertThrown!DateTimeException(Date.fromISOExtString("2010070 "));
            assertThrown!DateTimeException(Date.fromISOExtString("120100704"));
            assertThrown!DateTimeException(Date.fromISOExtString("-0100704"));
            assertThrown!DateTimeException(Date.fromISOExtString("+0100704"));
            assertThrown!DateTimeException(Date.fromISOExtString("2010070a"));
            assertThrown!DateTimeException(Date.fromISOExtString("20100a04"));
            assertThrown!DateTimeException(Date.fromISOExtString("2010a704"));

            assertThrown!DateTimeException(Date.fromISOExtString("99-07-04"));
            assertThrown!DateTimeException(Date.fromISOExtString("010-07-04"));
            assertThrown!DateTimeException(Date.fromISOExtString("2010-07-0"));
            assertThrown!DateTimeException(Date.fromISOExtString("2010-07-0 "));
            assertThrown!DateTimeException(Date.fromISOExtString("12010-07-04"));
            assertThrown!DateTimeException(Date.fromISOExtString("-010-07-04"));
            assertThrown!DateTimeException(Date.fromISOExtString("+010-07-04"));
            assertThrown!DateTimeException(Date.fromISOExtString("2010-07-0a"));
            assertThrown!DateTimeException(Date.fromISOExtString("2010-0a-04"));
            assertThrown!DateTimeException(Date.fromISOExtString("2010-a7-04"));
            assertThrown!DateTimeException(Date.fromISOExtString("2010/07/04"));
            assertThrown!DateTimeException(Date.fromISOExtString("2010/7/04"));
            assertThrown!DateTimeException(Date.fromISOExtString("2010/7/4"));
            assertThrown!DateTimeException(Date.fromISOExtString("2010/07/4"));
            assertThrown!DateTimeException(Date.fromISOExtString("2010-7-04"));
            assertThrown!DateTimeException(Date.fromISOExtString("2010-7-4"));
            assertThrown!DateTimeException(Date.fromISOExtString("2010-07-4"));

            assertThrown!DateTimeException(Date.fromISOExtString("99Jul04"));
            assertThrown!DateTimeException(Date.fromISOExtString("010Jul04"));
            assertThrown!DateTimeException(Date.fromISOExtString("2010Jul0"));
            assertThrown!DateTimeException(Date.fromISOExtString("2010-Jul-0 "));
            assertThrown!DateTimeException(Date.fromISOExtString("12010Jul04"));
            assertThrown!DateTimeException(Date.fromISOExtString("-010Jul04"));
            assertThrown!DateTimeException(Date.fromISOExtString("+010Jul04"));
            assertThrown!DateTimeException(Date.fromISOExtString("2010Jul0a"));
            assertThrown!DateTimeException(Date.fromISOExtString("2010Jua04"));
            assertThrown!DateTimeException(Date.fromISOExtString("2010aul04"));

            assertThrown!DateTimeException(Date.fromISOExtString("99-Jul-04"));
            assertThrown!DateTimeException(Date.fromISOExtString("010-Jul-04"));
            assertThrown!DateTimeException(Date.fromISOExtString("2010-Jul-0"));
            assertThrown!DateTimeException(Date.fromISOExtString("2010Jul0 "));
            assertThrown!DateTimeException(Date.fromISOExtString("12010-Jul-04"));
            assertThrown!DateTimeException(Date.fromISOExtString("-010-Jul-04"));
            assertThrown!DateTimeException(Date.fromISOExtString("+010-Jul-04"));
            assertThrown!DateTimeException(Date.fromISOExtString("2010-Jul-0a"));
            assertThrown!DateTimeException(Date.fromISOExtString("2010-Jua-04"));
            assertThrown!DateTimeException(Date.fromISOExtString("2010-Jal-04"));
            assertThrown!DateTimeException(Date.fromISOExtString("2010-aul-04"));

            assertThrown!DateTimeException(Date.fromISOExtString("20100704"));
            assertThrown!DateTimeException(Date.fromISOExtString("2010-Jul-04"));

            _assertPred!"=="(Date.fromISOExtString("1999-07-06"), Date(1999, 7, 6));
            _assertPred!"=="(Date.fromISOExtString("-1999-07-06"), Date(-1999, 7, 6));
            _assertPred!"=="(Date.fromISOExtString("+01999-07-06"), Date(1999, 7, 6));
            _assertPred!"=="(Date.fromISOExtString("1999-07-06 "), Date(1999, 7, 6));
            _assertPred!"=="(Date.fromISOExtString(" 1999-07-06"), Date(1999, 7, 6));
            _assertPred!"=="(Date.fromISOExtString(" 1999-07-06 "), Date(1999, 7, 6));

            //Verify Examples.
            assert(Date.fromISOExtString("2010-07-04") == Date(2010, 7, 4));
            assert(Date.fromISOExtString("1998-12-25") == Date(1998, 12, 25));
            assert(Date.fromISOExtString("0000-01-05") == Date(0, 1, 5));
            assert(Date.fromISOExtString("-0004-01-05") == Date(-4, 1, 5));
            assert(Date.fromISOExtString(" 2010-07-04 ") == Date(2010, 7, 4));
        }
    }


    /++
        Creates a $(D Date) from a string with the format YYYY-Mon-DD.
        Whitespace is stripped from the given string.

        Params:
            simpleString = A string formatted in the way that toSimpleString
                           formats dates.

        Throws:
            $(D DateTimeException) if the given string is not in the correct
            format or if the resulting $(D Date) would not be valid.

        Examples:
--------------------
assert(Date.fromSimpleString("2010-Jul-04") == Date(2010, 7, 4));
assert(Date.fromSimpleString("1998-Dec-25") == Date(1998, 12, 25));
assert(Date.fromSimpleString("0000-Jan-05") == Date(0, 1, 5));
assert(Date.fromSimpleString("-0004-Jan-05") == Date(-4, 1, 5));
assert(Date.fromSimpleString(" 2010-Jul-04 ") == Date(2010, 7, 4));
--------------------
      +/
    static Date fromSimpleString(S)(in S simpleString)
        if(isSomeString!(S))
    {
        auto dstr = to!dstring(strip(simpleString));

        enforce(dstr.length >= 11, new DateTimeException(format("Invalid string format: %s", simpleString)));

        auto day = dstr[$-2 .. $];
        auto month = monthFromString(to!string(dstr[$-6 .. $-3]));
        auto year = dstr[0 .. $-7];

        enforce(dstr[$-3] == '-', new DateTimeException(format("Invalid string format: %s", simpleString)));
        enforce(dstr[$-7] == '-', new DateTimeException(format("Invalid string format: %s", simpleString)));
        enforce(!canFind!(not!isDigit)(day), new DateTimeException(format("Invalid string format: %s", simpleString)));

        if(year.length > 4)
        {
            enforce(year.startsWith("-") || year.startsWith("+"),
                    new DateTimeException(format("Invalid string format: %s", simpleString)));
            enforce(!canFind!(not!isDigit)(year[1..$]),
                    new DateTimeException(format("Invalid string format: %s", simpleString)));
        }
        else
            enforce(!canFind!(not!isDigit)(year),
                    new DateTimeException(format("Invalid string format: %s", simpleString)));

        return Date(to!short(year), month, to!ubyte(day));
    }

    unittest
    {
        version(testStdDateTime)
        {
            assertThrown!DateTimeException(Date.fromSimpleString(""));
            assertThrown!DateTimeException(Date.fromSimpleString("990704"));
            assertThrown!DateTimeException(Date.fromSimpleString("0100704"));
            assertThrown!DateTimeException(Date.fromSimpleString("2010070"));
            assertThrown!DateTimeException(Date.fromSimpleString("2010070 "));
            assertThrown!DateTimeException(Date.fromSimpleString("120100704"));
            assertThrown!DateTimeException(Date.fromSimpleString("-0100704"));
            assertThrown!DateTimeException(Date.fromSimpleString("+0100704"));
            assertThrown!DateTimeException(Date.fromSimpleString("2010070a"));
            assertThrown!DateTimeException(Date.fromSimpleString("20100a04"));
            assertThrown!DateTimeException(Date.fromSimpleString("2010a704"));

            assertThrown!DateTimeException(Date.fromSimpleString("99-07-04"));
            assertThrown!DateTimeException(Date.fromSimpleString("010-07-04"));
            assertThrown!DateTimeException(Date.fromSimpleString("2010-07-0"));
            assertThrown!DateTimeException(Date.fromSimpleString("2010-07-0 "));
            assertThrown!DateTimeException(Date.fromSimpleString("12010-07-04"));
            assertThrown!DateTimeException(Date.fromSimpleString("-010-07-04"));
            assertThrown!DateTimeException(Date.fromSimpleString("+010-07-04"));
            assertThrown!DateTimeException(Date.fromSimpleString("2010-07-0a"));
            assertThrown!DateTimeException(Date.fromSimpleString("2010-0a-04"));
            assertThrown!DateTimeException(Date.fromSimpleString("2010-a7-04"));
            assertThrown!DateTimeException(Date.fromSimpleString("2010/07/04"));
            assertThrown!DateTimeException(Date.fromSimpleString("2010/7/04"));
            assertThrown!DateTimeException(Date.fromSimpleString("2010/7/4"));
            assertThrown!DateTimeException(Date.fromSimpleString("2010/07/4"));
            assertThrown!DateTimeException(Date.fromSimpleString("2010-7-04"));
            assertThrown!DateTimeException(Date.fromSimpleString("2010-7-4"));
            assertThrown!DateTimeException(Date.fromSimpleString("2010-07-4"));

            assertThrown!DateTimeException(Date.fromSimpleString("99Jul04"));
            assertThrown!DateTimeException(Date.fromSimpleString("010Jul04"));
            assertThrown!DateTimeException(Date.fromSimpleString("2010Jul0"));
            assertThrown!DateTimeException(Date.fromSimpleString("2010Jul0 "));
            assertThrown!DateTimeException(Date.fromSimpleString("12010Jul04"));
            assertThrown!DateTimeException(Date.fromSimpleString("-010Jul04"));
            assertThrown!DateTimeException(Date.fromSimpleString("+010Jul04"));
            assertThrown!DateTimeException(Date.fromSimpleString("2010Jul0a"));
            assertThrown!DateTimeException(Date.fromSimpleString("2010Jua04"));
            assertThrown!DateTimeException(Date.fromSimpleString("2010aul04"));

            assertThrown!DateTimeException(Date.fromSimpleString("99-Jul-04"));
            assertThrown!DateTimeException(Date.fromSimpleString("010-Jul-04"));
            assertThrown!DateTimeException(Date.fromSimpleString("2010-Jul-0"));
            assertThrown!DateTimeException(Date.fromSimpleString("2010-Jul-0 "));
            assertThrown!DateTimeException(Date.fromSimpleString("12010-Jul-04"));
            assertThrown!DateTimeException(Date.fromSimpleString("-010-Jul-04"));
            assertThrown!DateTimeException(Date.fromSimpleString("+010-Jul-04"));
            assertThrown!DateTimeException(Date.fromSimpleString("2010-Jul-0a"));
            assertThrown!DateTimeException(Date.fromSimpleString("2010-Jua-04"));
            assertThrown!DateTimeException(Date.fromSimpleString("2010-Jal-04"));
            assertThrown!DateTimeException(Date.fromSimpleString("2010-aul-04"));

            assertThrown!DateTimeException(Date.fromSimpleString("20100704"));
            assertThrown!DateTimeException(Date.fromSimpleString("2010-07-04"));

            _assertPred!"=="(Date.fromSimpleString("1999-Jul-06"), Date(1999, 7, 6));
            _assertPred!"=="(Date.fromSimpleString("-1999-Jul-06"), Date(-1999, 7, 6));
            _assertPred!"=="(Date.fromSimpleString("+01999-Jul-06"), Date(1999, 7, 6));
            _assertPred!"=="(Date.fromSimpleString("1999-Jul-06 "), Date(1999, 7, 6));
            _assertPred!"=="(Date.fromSimpleString(" 1999-Jul-06"), Date(1999, 7, 6));
            _assertPred!"=="(Date.fromSimpleString(" 1999-Jul-06 "), Date(1999, 7, 6));

            //Verify Examples.
            assert(Date.fromSimpleString("2010-Jul-04") == Date(2010, 7, 4));
            assert(Date.fromSimpleString("1998-Dec-25") == Date(1998, 12, 25));
            assert(Date.fromSimpleString("0000-Jan-05") == Date(0, 1, 5));
            assert(Date.fromSimpleString("-0004-Jan-05") == Date(-4, 1, 5));
            assert(Date.fromSimpleString(" 2010-Jul-04 ") == Date(2010, 7, 4));
        }
    }


    //TODO Add function which takes a user-specified time format and produces a Date

    //TODO Add function which takes pretty much any time-string and produces a Date
    //     Obviously, it will be less efficient, and it probably won't manage _every_
    //     possible date format, but a smart conversion function would be nice.


    /++
        Returns the $(D Date) farthest in the past which is representable by
        $(D Date).
      +/
    @property static Date min() pure nothrow
    {
        auto date = Date.init;
        date._year = short.min;
        date._month = Month.jan;
        date._day = 1;

        return date;
    }

    unittest
    {
        version(testStdDateTime)
        {
            assert(Date.min.year < 0);
            assert(Date.min < Date.max);
        }
    }


    /++
        Returns the $(D Date) farthest in the future which is representable by
        $(D Date).
      +/
    @property static Date max() pure nothrow
    {
        auto date = Date.init;
        date._year = short.max;
        date._month = Month.dec;
        date._day = 31;

        return date;
    }

    unittest
    {
        version(testStdDateTime)
        {
            assert(Date.max.year > 0);
            assert(Date.max > Date.min);
        }
    }


private:

    /+
        Whether the given values form a valid date.

        Params:
            year  = The year to test.
            month = The month of the Gregorian Calendar to test.
            day   = The day of the month to test.
     +/
    static bool _valid(int year, int month, int day) pure nothrow
    {
        if(!valid!"months"(month))
            return false;

        return valid!"days"(year, month, day);
    }

    /+
        Adds the given number of days to this $(D Date). A negative number will
        subtract.

        The month will be adjusted along with the day if the number of days
        added (or subtracted) would overflow (or underflow) the current month.
        The year will be adjusted along with the month if the increase (or
        decrease) to the month would cause it to overflow (or underflow) the
        current year.

        $(D addDays(numDays)) is effectively equivalent to
        $(D date.dayOfGregorianCal = date.dayOfGregorianCal + days).

        Params:
            days = The number of days to add to this Date.
      +/
    ref Date addDays(long days) pure nothrow
    {
        dayOfGregorianCal = cast(int)(dayOfGregorianCal + days);

        return this;
    }

    unittest
    {
        version(testStdDateTime)
        {
            //Test A.D.
            {
                auto date = Date(1999, 2, 28);
                date.addDays(1);
                _assertPred!"=="(date, Date(1999, 3, 1));
                date.addDays(-1);
                _assertPred!"=="(date, Date(1999, 2, 28));
            }

            {
                auto date = Date(2000, 2, 28);
                date.addDays(1);
                _assertPred!"=="(date, Date(2000, 2, 29));
                date.addDays(1);
                _assertPred!"=="(date, Date(2000, 3, 1));
                date.addDays(-1);
                _assertPred!"=="(date, Date(2000, 2, 29));
            }

            {
                auto date = Date(1999, 6, 30);
                date.addDays(1);
                _assertPred!"=="(date, Date(1999, 7, 1));
                date.addDays(-1);
                _assertPred!"=="(date, Date(1999, 6, 30));
            }

            {
                auto date = Date(1999, 7, 31);
                date.addDays(1);
                _assertPred!"=="(date, Date(1999, 8, 1));
                date.addDays(-1);
                _assertPred!"=="(date, Date(1999, 7, 31));
            }

            {
                auto date = Date(1999, 1, 1);
                date.addDays(-1);
                _assertPred!"=="(date, Date(1998, 12, 31));
                date.addDays(1);
                _assertPred!"=="(date, Date(1999, 1, 1));
            }

            {
                auto date = Date(1999, 7, 6);
                date.addDays(9);
                _assertPred!"=="(date, Date(1999, 7, 15));
                date.addDays(-11);
                _assertPred!"=="(date, Date(1999, 7, 4));
                date.addDays(30);
                _assertPred!"=="(date, Date(1999, 8, 3));
                date.addDays(-3);
                _assertPred!"=="(date, Date(1999, 7, 31));
            }

            {
                auto date = Date(1999, 7, 6);
                date.addDays(365);
                _assertPred!"=="(date, Date(2000, 7, 5));
                date.addDays(-365);
                _assertPred!"=="(date, Date(1999, 7, 6));
                date.addDays(366);
                _assertPred!"=="(date, Date(2000, 7, 6));
                date.addDays(730);
                _assertPred!"=="(date, Date(2002, 7, 6));
                date.addDays(-1096);
                _assertPred!"=="(date, Date(1999, 7, 6));
            }

            //Test B.C.
            {
                auto date = Date(-1999, 2, 28);
                date.addDays(1);
                _assertPred!"=="(date, Date(-1999, 3, 1));
                date.addDays(-1);
                _assertPred!"=="(date, Date(-1999, 2, 28));
            }

            {
                auto date = Date(-2000, 2, 28);
                date.addDays(1);
                _assertPred!"=="(date, Date(-2000, 2, 29));
                date.addDays(1);
                _assertPred!"=="(date, Date(-2000, 3, 1));
                date.addDays(-1);
                _assertPred!"=="(date, Date(-2000, 2, 29));
            }

            {
                auto date = Date(-1999, 6, 30);
                date.addDays(1);
                _assertPred!"=="(date, Date(-1999, 7, 1));
                date.addDays(-1);
                _assertPred!"=="(date, Date(-1999, 6, 30));
            }

            {
                auto date = Date(-1999, 7, 31);
                date.addDays(1);
                _assertPred!"=="(date, Date(-1999, 8, 1));
                date.addDays(-1);
                _assertPred!"=="(date, Date(-1999, 7, 31));
            }

            {
                auto date = Date(-1999, 1, 1);
                date.addDays(-1);
                _assertPred!"=="(date, Date(-2000, 12, 31));
                date.addDays(1);
                _assertPred!"=="(date, Date(-1999, 1, 1));
            }

            {
                auto date = Date(-1999, 7, 6);
                date.addDays(9);
                _assertPred!"=="(date, Date(-1999, 7, 15));
                date.addDays(-11);
                _assertPred!"=="(date, Date(-1999, 7, 4));
                date.addDays(30);
                _assertPred!"=="(date, Date(-1999, 8, 3));
                date.addDays(-3);
            }

            {
                auto date = Date(-1999, 7, 6);
                date.addDays(365);
                _assertPred!"=="(date, Date(-1998, 7, 6));
                date.addDays(-365);
                _assertPred!"=="(date, Date(-1999, 7, 6));
                date.addDays(366);
                _assertPred!"=="(date, Date(-1998, 7, 7));
                date.addDays(730);
                _assertPred!"=="(date, Date(-1996, 7, 6));
                date.addDays(-1096);
                _assertPred!"=="(date, Date(-1999, 7, 6));
            }

            //Test Both
            {
                auto date = Date(1, 7, 6);
                date.addDays(-365);
                _assertPred!"=="(date, Date(0, 7, 6));
                date.addDays(365);
                _assertPred!"=="(date, Date(1, 7, 6));
                date.addDays(-731);
                _assertPred!"=="(date, Date(-1, 7, 6));
                date.addDays(730);
                _assertPred!"=="(date, Date(1, 7, 5));
            }

            const cdate = Date(1999, 7, 6);
            immutable idate = Date(1999, 7, 6);
            static assert(!__traits(compiles, cdate.addDays(12)));
            static assert(!__traits(compiles, idate.addDays(12)));
        }
    }


    pure invariant()
    {
        assert(valid!"months"(_month), "Invariant Failure: year [" ~
                                       numToString(_year) ~
                                       "] month [" ~
                                       numToString(_month) ~
                                       "] day [" ~
                                       numToString(_day) ~
                                       "]");
        assert(valid!"days"(_year, _month, _day), "Invariant Failure: year [" ~
                                                  numToString(_year) ~
                                                  "] month [" ~
                                                  numToString(_month) ~
                                                  "] day [" ~
                                                  numToString(_day) ~
                                                  "]");
    }


    short _year  = 1;
    Month _month = Month.jan;
    ubyte _day   = 1;
}



/++
    Represents a time of day with hours, minutes, and seconds. It uses 24 hour
    time.
+/
struct TimeOfDay
{
public:

    /++
        Params:
            hour   = Hour of the day [0 - 24$(RPAREN).
            minute = Minute of the hour [0 - 60$(RPAREN).
            second = Second of the minute [0 - 60$(RPAREN).

        Throws:
            $(D DateTimeException) if the resulting $(D TimeOfDay) would be not
            be valid.
     +/
    this(int hour, int minute, int second = 0) pure
    {
        enforceValid!"hours"(hour);
        enforceValid!"minutes"(minute);
        enforceValid!"seconds"(second);

        _hour   = cast(ubyte)hour;
        _minute = cast(ubyte)minute;
        _second = cast(ubyte)second;
    }

    unittest
    {
        version(testStdDateTime)
        {
            assert(TimeOfDay(0, 0) == TimeOfDay.init);

            {
                auto tod = TimeOfDay(0, 0);
                _assertPred!"=="(tod._hour, 0);
                _assertPred!"=="(tod._minute, 0);
                _assertPred!"=="(tod._second, 0);
            }

            {
                auto tod = TimeOfDay(12, 30, 33);
                _assertPred!"=="(tod._hour, 12);
                _assertPred!"=="(tod._minute, 30);
                _assertPred!"=="(tod._second, 33);
            }

            {
                auto tod = TimeOfDay(23, 59, 59);
                _assertPred!"=="(tod._hour, 23);
                _assertPred!"=="(tod._minute, 59);
                _assertPred!"=="(tod._second, 59);
            }

            assertThrown!DateTimeException(TimeOfDay(24, 0, 0));
            assertThrown!DateTimeException(TimeOfDay(0, 60, 0));
            assertThrown!DateTimeException(TimeOfDay(0, 0, 60));
        }
    }


    /++
        Compares this $(D TimeOfDay) with the given $(D TimeOfDay).

        Returns:
            $(BOOKTABLE,
            $(TR $(TD this &lt; rhs) $(TD &lt; 0))
            $(TR $(TD this == rhs) $(TD 0))
            $(TR $(TD this &gt; rhs) $(TD &gt; 0))
            )
     +/
    int opCmp(in TimeOfDay rhs) const pure nothrow
    {
        if(_hour < rhs._hour)
            return -1;
        if(_hour > rhs._hour)
            return 1;

        if(_minute < rhs._minute)
            return -1;
        if(_minute > rhs._minute)
            return 1;

        if(_second < rhs._second)
            return -1;
        if(_second > rhs._second)
            return 1;

        return 0;
    }

    unittest
    {
        version(testStdDateTime)
        {
            _assertPred!("opCmp", "==")(TimeOfDay(0, 0, 0), TimeOfDay.init);

            _assertPred!("opCmp", "==")(TimeOfDay(0, 0, 0), TimeOfDay(0, 0, 0));
            _assertPred!("opCmp", "==")(TimeOfDay(12, 0, 0), TimeOfDay(12, 0, 0));
            _assertPred!("opCmp", "==")(TimeOfDay(0, 30, 0), TimeOfDay(0, 30, 0));
            _assertPred!("opCmp", "==")(TimeOfDay(0, 0, 33), TimeOfDay(0, 0, 33));

            _assertPred!("opCmp", "==")(TimeOfDay(12, 30, 0), TimeOfDay(12, 30, 0));
            _assertPred!("opCmp", "==")(TimeOfDay(12, 30, 33), TimeOfDay(12, 30, 33));

            _assertPred!("opCmp", "==")(TimeOfDay(0, 30, 33), TimeOfDay(0, 30, 33));
            _assertPred!("opCmp", "==")(TimeOfDay(0, 0, 33), TimeOfDay(0, 0, 33));

            _assertPred!("opCmp", "<")(TimeOfDay(12, 30, 33), TimeOfDay(13, 30, 33));
            _assertPred!("opCmp", ">")(TimeOfDay(13, 30, 33), TimeOfDay(12, 30, 33));
            _assertPred!("opCmp", "<")(TimeOfDay(12, 30, 33), TimeOfDay(12, 31, 33));
            _assertPred!("opCmp", ">")(TimeOfDay(12, 31, 33), TimeOfDay(12, 30, 33));
            _assertPred!("opCmp", "<")(TimeOfDay(12, 30, 33), TimeOfDay(12, 30, 34));
            _assertPred!("opCmp", ">")(TimeOfDay(12, 30, 34), TimeOfDay(12, 30, 33));

            _assertPred!("opCmp", ">")(TimeOfDay(13, 30, 33), TimeOfDay(12, 30, 34));
            _assertPred!("opCmp", "<")(TimeOfDay(12, 30, 34), TimeOfDay(13, 30, 33));
            _assertPred!("opCmp", ">")(TimeOfDay(13, 30, 33), TimeOfDay(12, 31, 33));
            _assertPred!("opCmp", "<")(TimeOfDay(12, 31, 33), TimeOfDay(13, 30, 33));

            _assertPred!("opCmp", ">")(TimeOfDay(12, 31, 33), TimeOfDay(12, 30, 34));
            _assertPred!("opCmp", "<")(TimeOfDay(12, 30, 34), TimeOfDay(12, 31, 33));

            const ctod = TimeOfDay(12, 30, 33);
            immutable itod = TimeOfDay(12, 30, 33);
            static assert(__traits(compiles, ctod.opCmp(itod)));
            static assert(__traits(compiles, itod.opCmp(ctod)));
        }
    }


    /++
        Hours passed midnight.
     +/
    @property ubyte hour() const pure nothrow
    {
        return _hour;
    }

    unittest
    {
        version(testStdDateTime)
        {
            _assertPred!"=="(TimeOfDay.init.hour, 0);
            _assertPred!"=="(TimeOfDay(12, 0, 0).hour, 12);

            const ctod = TimeOfDay(12, 0, 0);
            immutable itod = TimeOfDay(12, 0, 0);
            static assert(__traits(compiles, ctod.hour == 12));
            static assert(__traits(compiles, itod.hour == 12));
        }
    }


    /++
        Hours passed midnight.

        Params:
            hour = The hour of the day to set this $(D TimeOfDay)'s hour to.

        Throws:
            $(D DateTimeException) if the given hour would result in an invalid
            $(D TimeOfDay).
     +/
    @property void hour(int hour) pure
    {
        enforceValid!"hours"(hour);
        _hour = cast(ubyte)hour;
    }

    unittest
    {
        version(testStdDateTime)
        {
            assertThrown!DateTimeException((){TimeOfDay(0, 0, 0).hour = 24;}());

            auto tod = TimeOfDay(0, 0, 0);
            tod.hour = 12;
            _assertPred!"=="(tod, TimeOfDay(12, 0, 0));

            const ctod = TimeOfDay(0, 0, 0);
            immutable itod = TimeOfDay(0, 0, 0);
            static assert(!__traits(compiles, ctod.hour = 12));
            static assert(!__traits(compiles, itod.hour = 12));
        }
    }


    /++
        Minutes passed the hour.
     +/
    @property ubyte minute() const pure nothrow
    {
        return _minute;
    }

    unittest
    {
        version(testStdDateTime)
        {
            _assertPred!"=="(TimeOfDay.init.minute, 0);
            _assertPred!"=="(TimeOfDay(0, 30, 0).minute, 30);

            const ctod = TimeOfDay(0, 30, 0);
            immutable itod = TimeOfDay(0, 30, 0);
            static assert(__traits(compiles, ctod.minute == 30));
            static assert(__traits(compiles, itod.minute == 30));
        }
    }


    /++
        Minutes passed the hour.

        Params:
            minute = The minute to set this $(D TimeOfDay)'s minute to.

        Throws:
            $(D DateTimeException) if the given minute would result in an
            invalid $(D TimeOfDay).
     +/
    @property void minute(int minute) pure
    {
        enforceValid!"minutes"(minute);
        _minute = cast(ubyte)minute;
    }

    unittest
    {
        version(testStdDateTime)
        {
            assertThrown!DateTimeException((){TimeOfDay(0, 0, 0).minute = 60;}());

            auto tod = TimeOfDay(0, 0, 0);
            tod.minute = 30;
            _assertPred!"=="(tod, TimeOfDay(0, 30, 0));

            const ctod = TimeOfDay(0, 0, 0);
            immutable itod = TimeOfDay(0, 0, 0);
            static assert(!__traits(compiles, ctod.minute = 30));
            static assert(!__traits(compiles, itod.minute = 30));
        }
    }


    /++
        Seconds passed the minute.
     +/
    @property ubyte second() const pure nothrow
    {
        return _second;
    }

    unittest
    {
        version(testStdDateTime)
        {
            _assertPred!"=="(TimeOfDay.init.second, 0);
            _assertPred!"=="(TimeOfDay(0, 0, 33).second, 33);

            const ctod = TimeOfDay(0, 0, 33);
            immutable itod = TimeOfDay(0, 0, 33);
            static assert(__traits(compiles, ctod.second == 33));
            static assert(__traits(compiles, itod.second == 33));
        }
    }


    /++
        Seconds passed the minute.

        Params:
            second = The second to set this $(D TimeOfDay)'s second to.

        Throws:
            $(D DateTimeException) if the given second would result in an
            invalid $(D TimeOfDay).
     +/
    @property void second(int second) pure
    {
        enforceValid!"seconds"(second);
        _second = cast(ubyte)second;
    }

    unittest
    {
        version(testStdDateTime)
        {
            assertThrown!DateTimeException((){TimeOfDay(0, 0, 0).second = 60;}());

            auto tod = TimeOfDay(0, 0, 0);
            tod.second = 33;
            _assertPred!"=="(tod, TimeOfDay(0, 0, 33));

            const ctod = TimeOfDay(0, 0, 0);
            immutable itod = TimeOfDay(0, 0, 0);
            static assert(!__traits(compiles, ctod.second = 33));
            static assert(!__traits(compiles, itod.second = 33));
        }
    }


    /++
        Adds the given number of units to this $(D TimeOfDay). A negative number
        will subtract.

        The difference between rolling and adding is that rolling does not
        affect larger units. For instance, rolling a $(D TimeOfDay)
        one hours's worth of minutes gets the exact same
        $(D TimeOfDay).

        Accepted units are $(D "hours"), $(D "minutes"), and $(D "seconds").

        Params:
            units = The units to add.
            value = The number of $(D_PARAM units) to add to this
                    $(D TimeOfDay).

        Examples:
--------------------
auto tod1 = TimeOfDay(7, 12, 0);
tod1.roll!"hours"(1);
assert(tod1 == TimeOfDay(8, 12, 0));

auto tod2 = TimeOfDay(7, 12, 0);
tod2.roll!"hours"(-1);
assert(tod2 == TimeOfDay(6, 12, 0));

auto tod3 = TimeOfDay(23, 59, 0);
tod3.roll!"minutes"(1);
assert(tod3 == TimeOfDay(23, 0, 0));

auto tod4 = TimeOfDay(0, 0, 0);
tod4.roll!"minutes"(-1);
assert(tod4 == TimeOfDay(0, 59, 0));

auto tod5 = TimeOfDay(23, 59, 59);
tod5.roll!"seconds"(1);
assert(tod5 == TimeOfDay(23, 59, 0));

auto tod6 = TimeOfDay(0, 0, 0);
tod6.roll!"seconds"(-1);
assert(tod6 == TimeOfDay(0, 0, 59));
--------------------
      +/
    /+ref TimeOfDay+/ void roll(string units)(long value) pure nothrow
        if(units == "hours")
    {
        this += dur!"hours"(value);
    }

    //Verify Examples.
    unittest
    {
        version(testStdDateTime)
        {
            auto tod1 = TimeOfDay(7, 12, 0);
            tod1.roll!"hours"(1);
            assert(tod1 == TimeOfDay(8, 12, 0));

            auto tod2 = TimeOfDay(7, 12, 0);
            tod2.roll!"hours"(-1);
            assert(tod2 == TimeOfDay(6, 12, 0));

            auto tod3 = TimeOfDay(23, 59, 0);
            tod3.roll!"minutes"(1);
            assert(tod3 == TimeOfDay(23, 0, 0));

            auto tod4 = TimeOfDay(0, 0, 0);
            tod4.roll!"minutes"(-1);
            assert(tod4 == TimeOfDay(0, 59, 0));

            auto tod5 = TimeOfDay(23, 59, 59);
            tod5.roll!"seconds"(1);
            assert(tod5 == TimeOfDay(23, 59, 0));

            auto tod6 = TimeOfDay(0, 0, 0);
            tod6.roll!"seconds"(-1);
            assert(tod6 == TimeOfDay(0, 0, 59));
        }
    }

    unittest
    {
        version(testStdDateTime)
        {
            const ctod = TimeOfDay(0, 0, 0);
            immutable itod = TimeOfDay(0, 0, 0);
            static assert(!__traits(compiles, ctod.roll!"hours"(53)));
            static assert(!__traits(compiles, itod.roll!"hours"(53)));
        }
    }


    //Shares documentation with "hours" version.
    /+ref TimeOfDay+/ void roll(string units)(long value) pure nothrow
        if(units == "minutes" ||
           units == "seconds")
    {
        static if(units == "minutes")
            enum memberVarStr = "minute";
        else static if(units == "seconds")
            enum memberVarStr = "second";
        else
            static assert(0);

        value %= 60;
        mixin("auto newVal = cast(ubyte)(_" ~ memberVarStr ~ ") + value;");

        if(value < 0)
        {
            if(newVal < 0)
                newVal += 60;
        }
        else if(newVal >= 60)
            newVal -= 60;

        mixin("_" ~ memberVarStr ~ " = cast(ubyte)newVal;");
    }

    //Test roll!"minutes"().
    unittest
    {
        version(testStdDateTime)
        {
            static void testTOD(TimeOfDay orig, int minutes, in TimeOfDay expected, size_t line = __LINE__)
            {
                orig.roll!"minutes"(minutes);
                _assertPred!"=="(orig, expected, "", __FILE__, line);
            }

            testTOD(TimeOfDay(12, 30, 33), 0, TimeOfDay(12, 30, 33));
            testTOD(TimeOfDay(12, 30, 33), 1, TimeOfDay(12, 31, 33));
            testTOD(TimeOfDay(12, 30, 33), 2, TimeOfDay(12, 32, 33));
            testTOD(TimeOfDay(12, 30, 33), 3, TimeOfDay(12, 33, 33));
            testTOD(TimeOfDay(12, 30, 33), 4, TimeOfDay(12, 34, 33));
            testTOD(TimeOfDay(12, 30, 33), 5, TimeOfDay(12, 35, 33));
            testTOD(TimeOfDay(12, 30, 33), 10, TimeOfDay(12, 40, 33));
            testTOD(TimeOfDay(12, 30, 33), 15, TimeOfDay(12, 45, 33));
            testTOD(TimeOfDay(12, 30, 33), 29, TimeOfDay(12, 59, 33));
            testTOD(TimeOfDay(12, 30, 33), 30, TimeOfDay(12, 0, 33));
            testTOD(TimeOfDay(12, 30, 33), 45, TimeOfDay(12, 15, 33));
            testTOD(TimeOfDay(12, 30, 33), 60, TimeOfDay(12, 30, 33));
            testTOD(TimeOfDay(12, 30, 33), 75, TimeOfDay(12, 45, 33));
            testTOD(TimeOfDay(12, 30, 33), 90, TimeOfDay(12, 0, 33));
            testTOD(TimeOfDay(12, 30, 33), 100, TimeOfDay(12, 10, 33));

            testTOD(TimeOfDay(12, 30, 33), 689, TimeOfDay(12, 59, 33));
            testTOD(TimeOfDay(12, 30, 33), 690, TimeOfDay(12, 0, 33));
            testTOD(TimeOfDay(12, 30, 33), 691, TimeOfDay(12, 1, 33));
            testTOD(TimeOfDay(12, 30, 33), 960, TimeOfDay(12, 30, 33));
            testTOD(TimeOfDay(12, 30, 33), 1439, TimeOfDay(12, 29, 33));
            testTOD(TimeOfDay(12, 30, 33), 1440, TimeOfDay(12, 30, 33));
            testTOD(TimeOfDay(12, 30, 33), 1441, TimeOfDay(12, 31, 33));
            testTOD(TimeOfDay(12, 30, 33), 2880, TimeOfDay(12, 30, 33));

            testTOD(TimeOfDay(12, 30, 33), -1, TimeOfDay(12, 29, 33));
            testTOD(TimeOfDay(12, 30, 33), -2, TimeOfDay(12, 28, 33));
            testTOD(TimeOfDay(12, 30, 33), -3, TimeOfDay(12, 27, 33));
            testTOD(TimeOfDay(12, 30, 33), -4, TimeOfDay(12, 26, 33));
            testTOD(TimeOfDay(12, 30, 33), -5, TimeOfDay(12, 25, 33));
            testTOD(TimeOfDay(12, 30, 33), -10, TimeOfDay(12, 20, 33));
            testTOD(TimeOfDay(12, 30, 33), -15, TimeOfDay(12, 15, 33));
            testTOD(TimeOfDay(12, 30, 33), -29, TimeOfDay(12, 1, 33));
            testTOD(TimeOfDay(12, 30, 33), -30, TimeOfDay(12, 0, 33));
            testTOD(TimeOfDay(12, 30, 33), -45, TimeOfDay(12, 45, 33));
            testTOD(TimeOfDay(12, 30, 33), -60, TimeOfDay(12, 30, 33));
            testTOD(TimeOfDay(12, 30, 33), -75, TimeOfDay(12, 15, 33));
            testTOD(TimeOfDay(12, 30, 33), -90, TimeOfDay(12, 0, 33));
            testTOD(TimeOfDay(12, 30, 33), -100, TimeOfDay(12, 50, 33));

            testTOD(TimeOfDay(12, 30, 33), -749, TimeOfDay(12, 1, 33));
            testTOD(TimeOfDay(12, 30, 33), -750, TimeOfDay(12, 0, 33));
            testTOD(TimeOfDay(12, 30, 33), -751, TimeOfDay(12, 59, 33));
            testTOD(TimeOfDay(12, 30, 33), -960, TimeOfDay(12, 30, 33));
            testTOD(TimeOfDay(12, 30, 33), -1439, TimeOfDay(12, 31, 33));
            testTOD(TimeOfDay(12, 30, 33), -1440, TimeOfDay(12, 30, 33));
            testTOD(TimeOfDay(12, 30, 33), -1441, TimeOfDay(12, 29, 33));
            testTOD(TimeOfDay(12, 30, 33), -2880, TimeOfDay(12, 30, 33));

            testTOD(TimeOfDay(12, 0, 33), 1, TimeOfDay(12, 1, 33));
            testTOD(TimeOfDay(12, 0, 33), 0, TimeOfDay(12, 0, 33));
            testTOD(TimeOfDay(12, 0, 33), -1, TimeOfDay(12, 59, 33));

            testTOD(TimeOfDay(11, 59, 33), 1, TimeOfDay(11, 0, 33));
            testTOD(TimeOfDay(11, 59, 33), 0, TimeOfDay(11, 59, 33));
            testTOD(TimeOfDay(11, 59, 33), -1, TimeOfDay(11, 58, 33));

            testTOD(TimeOfDay(0, 0, 33), 1, TimeOfDay(0, 1, 33));
            testTOD(TimeOfDay(0, 0, 33), 0, TimeOfDay(0, 0, 33));
            testTOD(TimeOfDay(0, 0, 33), -1, TimeOfDay(0, 59, 33));

            testTOD(TimeOfDay(23, 59, 33), 1, TimeOfDay(23, 0, 33));
            testTOD(TimeOfDay(23, 59, 33), 0, TimeOfDay(23, 59, 33));
            testTOD(TimeOfDay(23, 59, 33), -1, TimeOfDay(23, 58, 33));

            const ctod = TimeOfDay(0, 0, 0);
            immutable itod = TimeOfDay(0, 0, 0);
            static assert(!__traits(compiles, ctod.roll!"minutes"(7)));
            static assert(!__traits(compiles, itod.roll!"minutes"(7)));

            //Verify Examples.
            auto tod1 = TimeOfDay(7, 12, 0);
            tod1.roll!"minutes"(1);
            assert(tod1 == TimeOfDay(7, 13, 0));

            auto tod2 = TimeOfDay(7, 12, 0);
            tod2.roll!"minutes"(-1);
            assert(tod2 == TimeOfDay(7, 11, 0));

            auto tod3 = TimeOfDay(23, 59, 0);
            tod3.roll!"minutes"(1);
            assert(tod3 == TimeOfDay(23, 0, 0));

            auto tod4 = TimeOfDay(0, 0, 0);
            tod4.roll!"minutes"(-1);
            assert(tod4 == TimeOfDay(0, 59, 0));

            auto tod5 = TimeOfDay(7, 32, 12);
            tod5.roll!"seconds"(1);
            assert(tod5 == TimeOfDay(7, 32, 13));

            auto tod6 = TimeOfDay(7, 32, 12);
            tod6.roll!"seconds"(-1);
            assert(tod6 == TimeOfDay(7, 32, 11));

            auto tod7 = TimeOfDay(23, 59, 59);
            tod7.roll!"seconds"(1);
            assert(tod7 == TimeOfDay(23, 59, 0));

            auto tod8 = TimeOfDay(0, 0, 0);
            tod8.roll!"seconds"(-1);
            assert(tod8 == TimeOfDay(0, 0, 59));
        }
    }

    //Test roll!"seconds"().
    unittest
    {
        version(testStdDateTime)
        {
            static void testTOD(TimeOfDay orig, int seconds, in TimeOfDay expected, size_t line = __LINE__)
            {
                orig.roll!"seconds"(seconds);
                _assertPred!"=="(orig, expected, "", __FILE__, line);
            }

            testTOD(TimeOfDay(12, 30, 33), 0, TimeOfDay(12, 30, 33));
            testTOD(TimeOfDay(12, 30, 33), 1, TimeOfDay(12, 30, 34));
            testTOD(TimeOfDay(12, 30, 33), 2, TimeOfDay(12, 30, 35));
            testTOD(TimeOfDay(12, 30, 33), 3, TimeOfDay(12, 30, 36));
            testTOD(TimeOfDay(12, 30, 33), 4, TimeOfDay(12, 30, 37));
            testTOD(TimeOfDay(12, 30, 33), 5, TimeOfDay(12, 30, 38));
            testTOD(TimeOfDay(12, 30, 33), 10, TimeOfDay(12, 30, 43));
            testTOD(TimeOfDay(12, 30, 33), 15, TimeOfDay(12, 30, 48));
            testTOD(TimeOfDay(12, 30, 33), 26, TimeOfDay(12, 30, 59));
            testTOD(TimeOfDay(12, 30, 33), 27, TimeOfDay(12, 30, 0));
            testTOD(TimeOfDay(12, 30, 33), 30, TimeOfDay(12, 30, 3));
            testTOD(TimeOfDay(12, 30, 33), 59, TimeOfDay(12, 30, 32));
            testTOD(TimeOfDay(12, 30, 33), 60, TimeOfDay(12, 30, 33));
            testTOD(TimeOfDay(12, 30, 33), 61, TimeOfDay(12, 30, 34));

            testTOD(TimeOfDay(12, 30, 33), 1766, TimeOfDay(12, 30, 59));
            testTOD(TimeOfDay(12, 30, 33), 1767, TimeOfDay(12, 30, 0));
            testTOD(TimeOfDay(12, 30, 33), 1768, TimeOfDay(12, 30, 1));
            testTOD(TimeOfDay(12, 30, 33), 2007, TimeOfDay(12, 30, 0));
            testTOD(TimeOfDay(12, 30, 33), 3599, TimeOfDay(12, 30, 32));
            testTOD(TimeOfDay(12, 30, 33), 3600, TimeOfDay(12, 30, 33));
            testTOD(TimeOfDay(12, 30, 33), 3601, TimeOfDay(12, 30, 34));
            testTOD(TimeOfDay(12, 30, 33), 7200, TimeOfDay(12, 30, 33));

            testTOD(TimeOfDay(12, 30, 33), -1, TimeOfDay(12, 30, 32));
            testTOD(TimeOfDay(12, 30, 33), -2, TimeOfDay(12, 30, 31));
            testTOD(TimeOfDay(12, 30, 33), -3, TimeOfDay(12, 30, 30));
            testTOD(TimeOfDay(12, 30, 33), -4, TimeOfDay(12, 30, 29));
            testTOD(TimeOfDay(12, 30, 33), -5, TimeOfDay(12, 30, 28));
            testTOD(TimeOfDay(12, 30, 33), -10, TimeOfDay(12, 30, 23));
            testTOD(TimeOfDay(12, 30, 33), -15, TimeOfDay(12, 30, 18));
            testTOD(TimeOfDay(12, 30, 33), -33, TimeOfDay(12, 30, 0));
            testTOD(TimeOfDay(12, 30, 33), -34, TimeOfDay(12, 30, 59));
            testTOD(TimeOfDay(12, 30, 33), -35, TimeOfDay(12, 30, 58));
            testTOD(TimeOfDay(12, 30, 33), -59, TimeOfDay(12, 30, 34));
            testTOD(TimeOfDay(12, 30, 33), -60, TimeOfDay(12, 30, 33));
            testTOD(TimeOfDay(12, 30, 33), -61, TimeOfDay(12, 30, 32));

            testTOD(TimeOfDay(12, 30, 0), 1, TimeOfDay(12, 30, 1));
            testTOD(TimeOfDay(12, 30, 0), 0, TimeOfDay(12, 30, 0));
            testTOD(TimeOfDay(12, 30, 0), -1, TimeOfDay(12, 30, 59));

            testTOD(TimeOfDay(12, 0, 0), 1, TimeOfDay(12, 0, 1));
            testTOD(TimeOfDay(12, 0, 0), 0, TimeOfDay(12, 0, 0));
            testTOD(TimeOfDay(12, 0, 0), -1, TimeOfDay(12, 0, 59));

            testTOD(TimeOfDay(0, 0, 0), 1, TimeOfDay(0, 0, 1));
            testTOD(TimeOfDay(0, 0, 0), 0, TimeOfDay(0, 0, 0));
            testTOD(TimeOfDay(0, 0, 0), -1, TimeOfDay(0, 0, 59));

            testTOD(TimeOfDay(23, 59, 59), 1, TimeOfDay(23, 59, 0));
            testTOD(TimeOfDay(23, 59, 59), 0, TimeOfDay(23, 59, 59));
            testTOD(TimeOfDay(23, 59, 59), -1, TimeOfDay(23, 59, 58));

            const ctod = TimeOfDay(0, 0, 0);
            immutable itod = TimeOfDay(0, 0, 0);
            static assert(!__traits(compiles, ctod.roll!"seconds"(7)));
            static assert(!__traits(compiles, itod.roll!"seconds"(7)));
        }
    }


    /++
        Gives the result of adding or subtracting a duration from this
        $(D TimeOfDay).

        The legal types of arithmetic for $(D TimeOfDay) using this operator are

        $(BOOKTABLE,
        $(TR $(TD TimeOfDay) $(TD +) $(TD duration) $(TD -->) $(TD TimeOfDay))
        $(TR $(TD TimeOfDay) $(TD -) $(TD duration) $(TD -->) $(TD TimeOfDay))
        )

        Params:
            duration = The duration to add to or subtract from this
                       $(D TimeOfDay).
      +/
    TimeOfDay opBinary(string op, D)(in D duration) const pure nothrow
        if((op == "+" || op == "-") &&
           (is(Unqual!D == Duration) ||
            is(Unqual!D == TickDuration)))
    {
        TimeOfDay retval = this;

        static if(is(Unqual!D == Duration))
            immutable hnsecs = duration.total!"hnsecs";
        else static if(is(Unqual!D == TickDuration))
            immutable hnsecs = duration.hnsecs;

        //Ideally, this would just be
        //return retval.addSeconds(convert!("hnsecs", "seconds")(unaryFun!(op ~ "a")(hnsecs)));
        //But there isn't currently a pure version of unaryFun!().

        static if(op == "+")
            immutable signedHNSecs = hnsecs;
        else static if(op == "-")
            immutable signedHNSecs = -hnsecs;
        else
            static assert(0);

        return retval.addSeconds(convert!("hnsecs", "seconds")(signedHNSecs));
    }

    unittest
    {
        version(testStdDateTime)
        {
            auto tod = TimeOfDay(12, 30, 33);

            _assertPred!"=="(tod + dur!"hours"(7), TimeOfDay(19, 30, 33));
            _assertPred!"=="(tod + dur!"hours"(-7), TimeOfDay(5, 30, 33));
            _assertPred!"=="(tod + dur!"minutes"(7), TimeOfDay(12, 37, 33));
            _assertPred!"=="(tod + dur!"minutes"(-7), TimeOfDay(12, 23, 33));
            _assertPred!"=="(tod + dur!"seconds"(7), TimeOfDay(12, 30, 40));
            _assertPred!"=="(tod + dur!"seconds"(-7), TimeOfDay(12, 30, 26));

            _assertPred!"=="(tod + dur!"msecs"(7000), TimeOfDay(12, 30, 40));
            _assertPred!"=="(tod + dur!"msecs"(-7000), TimeOfDay(12, 30, 26));
            _assertPred!"=="(tod + dur!"usecs"(7_000_000), TimeOfDay(12, 30, 40));
            _assertPred!"=="(tod + dur!"usecs"(-7_000_000), TimeOfDay(12, 30, 26));
            _assertPred!"=="(tod + dur!"hnsecs"(70_000_000), TimeOfDay(12, 30, 40));
            _assertPred!"=="(tod + dur!"hnsecs"(-70_000_000), TimeOfDay(12, 30, 26));

            //This probably only runs in cases where gettimeofday() is used, but it's
            //hard to do this test correctly with variable ticksPerSec.
            if(TickDuration.ticksPerSec == 1_000_000)
            {
                _assertPred!"=="(tod + TickDuration.from!"usecs"(7_000_000), TimeOfDay(12, 30, 40));
                _assertPred!"=="(tod + TickDuration.from!"usecs"(-7_000_000), TimeOfDay(12, 30, 26));
            }

            _assertPred!"=="(tod - dur!"hours"(-7), TimeOfDay(19, 30, 33));
            _assertPred!"=="(tod - dur!"hours"(7), TimeOfDay(5, 30, 33));
            _assertPred!"=="(tod - dur!"minutes"(-7), TimeOfDay(12, 37, 33));
            _assertPred!"=="(tod - dur!"minutes"(7), TimeOfDay(12, 23, 33));
            _assertPred!"=="(tod - dur!"seconds"(-7), TimeOfDay(12, 30, 40));
            _assertPred!"=="(tod - dur!"seconds"(7), TimeOfDay(12, 30, 26));

            _assertPred!"=="(tod - dur!"msecs"(-7000), TimeOfDay(12, 30, 40));
            _assertPred!"=="(tod - dur!"msecs"(7000), TimeOfDay(12, 30, 26));
            _assertPred!"=="(tod - dur!"usecs"(-7_000_000), TimeOfDay(12, 30, 40));
            _assertPred!"=="(tod - dur!"usecs"(7_000_000), TimeOfDay(12, 30, 26));
            _assertPred!"=="(tod - dur!"hnsecs"(-70_000_000), TimeOfDay(12, 30, 40));
            _assertPred!"=="(tod - dur!"hnsecs"(70_000_000), TimeOfDay(12, 30, 26));

            //This probably only runs in cases where gettimeofday() is used, but it's
            //hard to do this test correctly with variable ticksPerSec.
            if(TickDuration.ticksPerSec == 1_000_000)
            {
                _assertPred!"=="(tod - TickDuration.from!"usecs"(-7_000_000), TimeOfDay(12, 30, 40));
                _assertPred!"=="(tod - TickDuration.from!"usecs"(7_000_000), TimeOfDay(12, 30, 26));
            }

            auto duration = dur!"hours"(11);
            const ctod = TimeOfDay(12, 33, 30);
            immutable itod = TimeOfDay(12, 33, 30);
            static assert(__traits(compiles, tod + duration));
            static assert(__traits(compiles, ctod + duration));
            static assert(__traits(compiles, itod + duration));

            static assert(__traits(compiles, tod - duration));
            static assert(__traits(compiles, ctod - duration));
            static assert(__traits(compiles, itod - duration));
        }
    }


    /++
        Gives the result of adding or subtracting a duration from this
        $(D TimeOfDay), as well as assigning the result to this
        $(D TimeOfDay).

        The legal types of arithmetic for $(D TimeOfDay) using this operator are

        $(BOOKTABLE,
        $(TR $(TD TimeOfDay) $(TD +) $(TD duration) $(TD -->) $(TD TimeOfDay))
        $(TR $(TD TimeOfDay) $(TD -) $(TD duration) $(TD -->) $(TD TimeOfDay))
        )

        Params:
            duration = The duration to add to or subtract from this
                       $(D TimeOfDay).
      +/
    /+ref+/ TimeOfDay opOpAssign(string op, D)(in D duration) pure nothrow
        if((op == "+" || op == "-") &&
           (is(Unqual!D == Duration) ||
            is(Unqual!D == TickDuration)))
    {
        static if(is(Unqual!D == Duration))
            immutable hnsecs = duration.total!"hnsecs";
        else static if(is(Unqual!D == TickDuration))
            immutable hnsecs = duration.hnsecs;

        //Ideally, this would just be
        //return addSeconds(convert!("hnsecs", "seconds")(unaryFun!(op ~ "a")(hnsecs)));
        //But there isn't currently a pure version of unaryFun!().

        static if(op == "+")
            immutable signedHNSecs = hnsecs;
        else static if(op == "-")
            immutable signedHNSecs = -hnsecs;
        else
            static assert(0);

        return addSeconds(convert!("hnsecs", "seconds")(signedHNSecs));
    }

    unittest
    {
        version(testStdDateTime)
        {
            auto duration = dur!"hours"(12);

            _assertPred!"+="(TimeOfDay(12, 30, 33), dur!"hours"(7), TimeOfDay(19, 30, 33));
            _assertPred!"+="(TimeOfDay(12, 30, 33), dur!"hours"(-7), TimeOfDay(5, 30, 33));
            _assertPred!"+="(TimeOfDay(12, 30, 33), dur!"minutes"(7), TimeOfDay(12, 37, 33));
            _assertPred!"+="(TimeOfDay(12, 30, 33), dur!"minutes"(-7), TimeOfDay(12, 23, 33));
            _assertPred!"+="(TimeOfDay(12, 30, 33), dur!"seconds"(7), TimeOfDay(12, 30, 40));
            _assertPred!"+="(TimeOfDay(12, 30, 33), dur!"seconds"(-7), TimeOfDay(12, 30, 26));

            _assertPred!"+="(TimeOfDay(12, 30, 33), dur!"msecs"(7000), TimeOfDay(12, 30, 40));
            _assertPred!"+="(TimeOfDay(12, 30, 33), dur!"msecs"(-7000), TimeOfDay(12, 30, 26));
            _assertPred!"+="(TimeOfDay(12, 30, 33), dur!"usecs"(7_000_000), TimeOfDay(12, 30, 40));
            _assertPred!"+="(TimeOfDay(12, 30, 33), dur!"usecs"(-7_000_000), TimeOfDay(12, 30, 26));
            _assertPred!"+="(TimeOfDay(12, 30, 33), dur!"hnsecs"(70_000_000), TimeOfDay(12, 30, 40));
            _assertPred!"+="(TimeOfDay(12, 30, 33), dur!"hnsecs"(-70_000_000), TimeOfDay(12, 30, 26));

            _assertPred!"-="(TimeOfDay(12, 30, 33), dur!"hours"(-7), TimeOfDay(19, 30, 33));
            _assertPred!"-="(TimeOfDay(12, 30, 33), dur!"hours"(7), TimeOfDay(5, 30, 33));
            _assertPred!"-="(TimeOfDay(12, 30, 33), dur!"minutes"(-7), TimeOfDay(12, 37, 33));
            _assertPred!"-="(TimeOfDay(12, 30, 33), dur!"minutes"(7), TimeOfDay(12, 23, 33));
            _assertPred!"-="(TimeOfDay(12, 30, 33), dur!"seconds"(-7), TimeOfDay(12, 30, 40));
            _assertPred!"-="(TimeOfDay(12, 30, 33), dur!"seconds"(7), TimeOfDay(12, 30, 26));

            _assertPred!"-="(TimeOfDay(12, 30, 33), dur!"msecs"(-7000), TimeOfDay(12, 30, 40));
            _assertPred!"-="(TimeOfDay(12, 30, 33), dur!"msecs"(7000), TimeOfDay(12, 30, 26));
            _assertPred!"-="(TimeOfDay(12, 30, 33), dur!"usecs"(-7_000_000), TimeOfDay(12, 30, 40));
            _assertPred!"-="(TimeOfDay(12, 30, 33), dur!"usecs"(7_000_000), TimeOfDay(12, 30, 26));
            _assertPred!"-="(TimeOfDay(12, 30, 33), dur!"hnsecs"(-70_000_000), TimeOfDay(12, 30, 40));
            _assertPred!"-="(TimeOfDay(12, 30, 33), dur!"hnsecs"(70_000_000), TimeOfDay(12, 30, 26));

            const ctod = TimeOfDay(12, 33, 30);
            immutable itod = TimeOfDay(12, 33, 30);
            static assert(!__traits(compiles, ctod += duration));
            static assert(!__traits(compiles, itod += duration));
            static assert(!__traits(compiles, ctod -= duration));
            static assert(!__traits(compiles, itod -= duration));
        }
    }


    /++
        Gives the difference between two $(D TimeOfDay)s.

        The legal types of arithmetic for $(D TimeOfDay) using this operator are

        $(BOOKTABLE,
        $(TR $(TD TimeOfDay) $(TD -) $(TD TimeOfDay) $(TD -->) $(TD duration))
        )

        Params:
            rhs = The $(D TimeOfDay) to subtract from this one.
      +/
    Duration opBinary(string op)(in TimeOfDay rhs) const pure nothrow
        if(op == "-")
    {
        immutable lhsSec = _hour * 3600 + _minute * 60 + _second;
        immutable rhsSec = rhs._hour * 3600 + rhs._minute * 60 + rhs._second;

        return dur!"seconds"(lhsSec - rhsSec);
    }

    unittest
    {
        version(testStdDateTime)
        {
            auto tod = TimeOfDay(12, 30, 33);

            _assertPred!"=="(TimeOfDay(7, 12, 52) - TimeOfDay(12, 30, 33), dur!"seconds"(-19_061));
            _assertPred!"=="(TimeOfDay(12, 30, 33) - TimeOfDay(7, 12, 52), dur!"seconds"(19_061));
            _assertPred!"=="(TimeOfDay(12, 30, 33) - TimeOfDay(14, 30, 33), dur!"seconds"(-7200));
            _assertPred!"=="(TimeOfDay(14, 30, 33) - TimeOfDay(12, 30, 33), dur!"seconds"(7200));
            _assertPred!"=="(TimeOfDay(12, 30, 33) - TimeOfDay(12, 34, 33), dur!"seconds"(-240));
            _assertPred!"=="(TimeOfDay(12, 34, 33) - TimeOfDay(12, 30, 33), dur!"seconds"(240));
            _assertPred!"=="(TimeOfDay(12, 30, 33) - TimeOfDay(12, 30, 34), dur!"seconds"(-1));
            _assertPred!"=="(TimeOfDay(12, 30, 34) - TimeOfDay(12, 30, 33), dur!"seconds"(1));

            const ctod = TimeOfDay(12, 30, 33);
            immutable itod = TimeOfDay(12, 30, 33);
            static assert(__traits(compiles, tod - tod));
            static assert(__traits(compiles, ctod - tod));
            static assert(__traits(compiles, itod - tod));

            static assert(__traits(compiles, tod - ctod));
            static assert(__traits(compiles, ctod - ctod));
            static assert(__traits(compiles, itod - ctod));

            static assert(__traits(compiles, tod - itod));
            static assert(__traits(compiles, ctod - itod));
            static assert(__traits(compiles, itod - itod));
        }
    }


    /++
        Converts this $(D TimeOfDay) to a string with the format HHMMSS.

        Examples:
--------------------
assert(TimeOfDay(0, 0, 0).toISOString() == "000000");
assert(TimeOfDay(12, 30, 33).toISOString() == "123033");
--------------------
      +/
    string toISOString() const nothrow
    {
        try
            return format("%02d%02d%02d", _hour, _minute, _second);
        catch(Exception e)
            assert(0, "format() threw.");
    }

    unittest
    {
        version(testStdDateTime)
        {
            auto tod = TimeOfDay(12, 30, 33);
            const ctod = TimeOfDay(12, 30, 33);
            immutable itod = TimeOfDay(12, 30, 33);
            static assert(__traits(compiles, tod.toISOString()));
            static assert(__traits(compiles, ctod.toISOString()));
            static assert(__traits(compiles, itod.toISOString()));

            //Verify Examples.
            assert(TimeOfDay(0, 0, 0).toISOString() == "000000");
            assert(TimeOfDay(12, 30, 33).toISOString() == "123033");
        }
    }


    /++
        Converts this $(D TimeOfDay) to a string with the format HH:MM:SS.

        Examples:
--------------------
assert(TimeOfDay(0, 0, 0).toISOExtString() == "000000");
assert(TimeOfDay(12, 30, 33).toISOExtString() == "123033");
--------------------
      +/
    string toISOExtString() const nothrow
    {
        try
            return format("%02d:%02d:%02d", _hour, _minute, _second);
        catch(Exception e)
            assert(0, "format() threw.");
    }

    unittest
    {
        version(testStdDateTime)
        {
            auto tod = TimeOfDay(12, 30, 33);
            const ctod = TimeOfDay(12, 30, 33);
            immutable itod = TimeOfDay(12, 30, 33);
            static assert(__traits(compiles, tod.toISOExtString()));
            static assert(__traits(compiles, ctod.toISOExtString()));
            static assert(__traits(compiles, itod.toISOExtString()));

            //Verify Examples.
            assert(TimeOfDay(0, 0, 0).toISOExtString() == "00:00:00");
            assert(TimeOfDay(12, 30, 33).toISOExtString() == "12:30:33");
        }
    }


    /+
        Converts this $(D TimeOfDay) to a string.
      +/
    //Due to bug http://d.puremagic.com/issues/show_bug.cgi?id=3715 , we can't
    //have versions of toString() with extra modifiers, so we define one version
    //with modifiers and one without.
    string toString()
    {
        return toISOExtString();
    }


    /++
        Converts this TimeOfDay to a string.
      +/
    //Due to bug http://d.puremagic.com/issues/show_bug.cgi?id=3715 , we can't
    //have versions of toString() with extra modifiers, so we define one version
    //with modifiers and one without.
    string toString() const nothrow
    {
        return toISOExtString();
    }

    unittest
    {
        version(testStdDateTime)
        {
            auto tod = TimeOfDay(12, 30, 33);
            const ctod = TimeOfDay(12, 30, 33);
            immutable itod = TimeOfDay(12, 30, 33);
            static assert(__traits(compiles, tod.toString()));
            static assert(__traits(compiles, ctod.toString()));
            static assert(__traits(compiles, itod.toString()));
        }
    }

    //TODO Add a function which returns a string in a user-specified format.



    /++
        Creates a $(D TimeOfDay) from a string with the format HHMMSS.
        Whitespace is stripped from the given string.

        Params:
            isoString = A string formatted in the ISO format for times.

        Throws:
            $(D DateTimeException) if the given string is not in the ISO format
            or if the resulting $(D TimeOfDay) would not be valid.

        Examples:
--------------------
assert(TimeOfDay.fromISOString("000000") == TimeOfDay(0, 0, 0));
assert(TimeOfDay.fromISOString("123033") == TimeOfDay(12, 30, 33));
assert(TimeOfDay.fromISOString(" 123033 ") == TimeOfDay(12, 30, 33));
--------------------
      +/
    static TimeOfDay fromISOString(S)(in S isoString)
        if(isSomeString!S)
    {
        auto dstr = to!dstring(strip(isoString));

        enforce(dstr.length == 6, new DateTimeException(format("Invalid ISO String: %s", isoString)));

        auto hours = dstr[0 .. 2];
        auto minutes = dstr[2 .. 4];
        auto seconds = dstr[4 .. $];

        enforce(!canFind!(not!isDigit)(hours), new DateTimeException(format("Invalid ISO String: %s", isoString)));
        enforce(!canFind!(not!isDigit)(minutes), new DateTimeException(format("Invalid ISO String: %s", isoString)));
        enforce(!canFind!(not!isDigit)(seconds), new DateTimeException(format("Invalid ISO String: %s", isoString)));

        return TimeOfDay(to!int(hours), to!int(minutes), to!int(seconds));
    }

    unittest
    {
        version(testStdDateTime)
        {
            assertThrown!DateTimeException(TimeOfDay.fromISOString(""));
            assertThrown!DateTimeException(TimeOfDay.fromISOString("0"));
            assertThrown!DateTimeException(TimeOfDay.fromISOString("00"));
            assertThrown!DateTimeException(TimeOfDay.fromISOString("000"));
            assertThrown!DateTimeException(TimeOfDay.fromISOString("0000"));
            assertThrown!DateTimeException(TimeOfDay.fromISOString("00000"));
            assertThrown!DateTimeException(TimeOfDay.fromISOString("13033"));
            assertThrown!DateTimeException(TimeOfDay.fromISOString("1277"));
            assertThrown!DateTimeException(TimeOfDay.fromISOString("12707"));
            assertThrown!DateTimeException(TimeOfDay.fromISOString("12070"));
            assertThrown!DateTimeException(TimeOfDay.fromISOString("12303a"));
            assertThrown!DateTimeException(TimeOfDay.fromISOString("1230a3"));
            assertThrown!DateTimeException(TimeOfDay.fromISOString("123a33"));
            assertThrown!DateTimeException(TimeOfDay.fromISOString("12a033"));
            assertThrown!DateTimeException(TimeOfDay.fromISOString("1a0033"));
            assertThrown!DateTimeException(TimeOfDay.fromISOString("a20033"));
            assertThrown!DateTimeException(TimeOfDay.fromISOString("1200330"));
            assertThrown!DateTimeException(TimeOfDay.fromISOString("0120033"));
            assertThrown!DateTimeException(TimeOfDay.fromISOString("-120033"));
            assertThrown!DateTimeException(TimeOfDay.fromISOString("+120033"));
            assertThrown!DateTimeException(TimeOfDay.fromISOString("120033am"));
            assertThrown!DateTimeException(TimeOfDay.fromISOString("120033pm"));

            assertThrown!DateTimeException(TimeOfDay.fromISOString("0::"));
            assertThrown!DateTimeException(TimeOfDay.fromISOString(":0:"));
            assertThrown!DateTimeException(TimeOfDay.fromISOString("::0"));
            assertThrown!DateTimeException(TimeOfDay.fromISOString("0:0:0"));
            assertThrown!DateTimeException(TimeOfDay.fromISOString("0:0:00"));
            assertThrown!DateTimeException(TimeOfDay.fromISOString("0:00:0"));
            assertThrown!DateTimeException(TimeOfDay.fromISOString("00:0:0"));
            assertThrown!DateTimeException(TimeOfDay.fromISOString("00:00:0"));
            assertThrown!DateTimeException(TimeOfDay.fromISOString("00:0:00"));
            assertThrown!DateTimeException(TimeOfDay.fromISOString("13:0:33"));
            assertThrown!DateTimeException(TimeOfDay.fromISOString("12:7:7"));
            assertThrown!DateTimeException(TimeOfDay.fromISOString("12:7:07"));
            assertThrown!DateTimeException(TimeOfDay.fromISOString("12:07:0"));
            assertThrown!DateTimeException(TimeOfDay.fromISOString("12:30:3a"));
            assertThrown!DateTimeException(TimeOfDay.fromISOString("12:30:a3"));
            assertThrown!DateTimeException(TimeOfDay.fromISOString("12:3a:33"));
            assertThrown!DateTimeException(TimeOfDay.fromISOString("12:a0:33"));
            assertThrown!DateTimeException(TimeOfDay.fromISOString("1a:00:33"));
            assertThrown!DateTimeException(TimeOfDay.fromISOString("a2:00:33"));
            assertThrown!DateTimeException(TimeOfDay.fromISOString("12:003:30"));
            assertThrown!DateTimeException(TimeOfDay.fromISOString("120:03:30"));
            assertThrown!DateTimeException(TimeOfDay.fromISOString("012:00:33"));
            assertThrown!DateTimeException(TimeOfDay.fromISOString("01:200:33"));
            assertThrown!DateTimeException(TimeOfDay.fromISOString("-12:00:33"));
            assertThrown!DateTimeException(TimeOfDay.fromISOString("+12:00:33"));
            assertThrown!DateTimeException(TimeOfDay.fromISOString("12:00:33am"));
            assertThrown!DateTimeException(TimeOfDay.fromISOString("12:00:33pm"));

            assertThrown!DateTimeException(TimeOfDay.fromISOString("12:00:33"));

            _assertPred!"=="(TimeOfDay.fromISOString("011217"), TimeOfDay(1, 12, 17));
            _assertPred!"=="(TimeOfDay.fromISOString("001412"), TimeOfDay(0, 14, 12));
            _assertPred!"=="(TimeOfDay.fromISOString("000007"), TimeOfDay(0, 0, 7));
            _assertPred!"=="(TimeOfDay.fromISOString("011217 "), TimeOfDay(1, 12, 17));
            _assertPred!"=="(TimeOfDay.fromISOString(" 011217"), TimeOfDay(1, 12, 17));
            _assertPred!"=="(TimeOfDay.fromISOString(" 011217 "), TimeOfDay(1, 12, 17));

            //Verify Examples.
            assert(TimeOfDay.fromISOString("000000") == TimeOfDay(0, 0, 0));
            assert(TimeOfDay.fromISOString("123033") == TimeOfDay(12, 30, 33));
            assert(TimeOfDay.fromISOString(" 123033 ") == TimeOfDay(12, 30, 33));
        }
    }


    /++
        Creates a $(D TimeOfDay) from a string with the format HH:MM:SS.
        Whitespace is stripped from the given string.

        Params:
            isoString = A string formatted in the ISO Extended format for times.

        Throws:
            $(D DateTimeException) if the given string is not in the ISO
            Extended format or if the resulting $(D TimeOfDay) would not be
            valid.

        Examples:
--------------------
assert(TimeOfDay.fromISOExtString("00:00:00") == TimeOfDay(0, 0, 0));
assert(TimeOfDay.fromISOExtString("12:30:33") == TimeOfDay(12, 30, 33));
assert(TimeOfDay.fromISOExtString(" 12:30:33 ") == TimeOfDay(12, 30, 33));
--------------------
      +/
    static TimeOfDay fromISOExtString(S)(in S isoExtString)
        if(isSomeString!S)
    {
        auto dstr = to!dstring(strip(isoExtString));

        enforce(dstr.length == 8, new DateTimeException(format("Invalid ISO Extended String: %s", isoExtString)));

        auto hours = dstr[0 .. 2];
        auto minutes = dstr[3 .. 5];
        auto seconds = dstr[6 .. $];

        enforce(dstr[2] == ':', new DateTimeException(format("Invalid ISO Extended String: %s", isoExtString)));
        enforce(dstr[5] == ':', new DateTimeException(format("Invalid ISO Extended String: %s", isoExtString)));
        enforce(!canFind!(not!isDigit)(hours),
                new DateTimeException(format("Invalid ISO Extended String: %s", isoExtString)));
        enforce(!canFind!(not!isDigit)(minutes),
                new DateTimeException(format("Invalid ISO Extended String: %s", isoExtString)));
        enforce(!canFind!(not!isDigit)(seconds),
                new DateTimeException(format("Invalid ISO Extended String: %s", isoExtString)));

        return TimeOfDay(to!int(hours), to!int(minutes), to!int(seconds));
    }

    unittest
    {
        version(testStdDateTime)
        {
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString(""));
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("0"));
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("00"));
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("000"));
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("0000"));
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("00000"));
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("13033"));
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("1277"));
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("12707"));
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("12070"));
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("12303a"));
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("1230a3"));
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("123a33"));
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("12a033"));
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("1a0033"));
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("a20033"));
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("1200330"));
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("0120033"));
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("-120033"));
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("+120033"));
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("120033am"));
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("120033pm"));

            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("0::"));
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString(":0:"));
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("::0"));
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("0:0:0"));
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("0:0:00"));
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("0:00:0"));
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("00:0:0"));
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("00:00:0"));
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("00:0:00"));
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("13:0:33"));
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("12:7:7"));
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("12:7:07"));
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("12:07:0"));
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("12:30:3a"));
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("12:30:a3"));
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("12:3a:33"));
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("12:a0:33"));
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("1a:00:33"));
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("a2:00:33"));
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("12:003:30"));
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("120:03:30"));
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("012:00:33"));
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("01:200:33"));
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("-12:00:33"));
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("+12:00:33"));
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("12:00:33am"));
            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("12:00:33pm"));

            assertThrown!DateTimeException(TimeOfDay.fromISOExtString("120033"));

            _assertPred!"=="(TimeOfDay.fromISOExtString("01:12:17"), TimeOfDay(1, 12, 17));
            _assertPred!"=="(TimeOfDay.fromISOExtString("00:14:12"), TimeOfDay(0, 14, 12));
            _assertPred!"=="(TimeOfDay.fromISOExtString("00:00:07"), TimeOfDay(0, 0, 7));
            _assertPred!"=="(TimeOfDay.fromISOExtString("01:12:17 "), TimeOfDay(1, 12, 17));
            _assertPred!"=="(TimeOfDay.fromISOExtString(" 01:12:17"), TimeOfDay(1, 12, 17));
            _assertPred!"=="(TimeOfDay.fromISOExtString(" 01:12:17 "), TimeOfDay(1, 12, 17));

            //Verify Examples.
            assert(TimeOfDay.fromISOExtString("00:00:00") == TimeOfDay(0, 0, 0));
            assert(TimeOfDay.fromISOExtString("12:30:33") == TimeOfDay(12, 30, 33));
            assert(TimeOfDay.fromISOExtString(" 12:30:33 ") == TimeOfDay(12, 30, 33));
        }
    }


    //TODO Add function which takes a user-specified time format and produces a TimeOfDay

    //TODO Add function which takes pretty much any time-string and produces a TimeOfDay
    //     Obviously, it will be less efficient, and it probably won't manage _every_
    //     possible date format, but a smart conversion function would be nice.


    /++
        Returns midnight.
      +/
    @property static TimeOfDay min() pure nothrow
    {
        return TimeOfDay.init;
    }

    unittest
    {
        version(testStdDateTime)
        {
            assert(TimeOfDay.min.hour == 0);
            assert(TimeOfDay.min.minute == 0);
            assert(TimeOfDay.min.second == 0);
            assert(TimeOfDay.min < TimeOfDay.max);
        }
    }


    /++
        Returns one second short of midnight.
      +/
    @property static TimeOfDay max() pure nothrow
    {
        auto tod = TimeOfDay.init;
        tod._hour = maxHour;
        tod._minute = maxMinute;
        tod._second = maxSecond;

        return tod;
    }

    unittest
    {
        version(testStdDateTime)
        {
            assert(TimeOfDay.max.hour == 23);
            assert(TimeOfDay.max.minute == 59);
            assert(TimeOfDay.max.second == 59);
            assert(TimeOfDay.max > TimeOfDay.min);
        }
    }


private:

    /+
        Add seconds to the time of day. Negative values will subtract. If the
        number of seconds overflows (or underflows), then the seconds will wrap,
        increasing (or decreasing) the number of minutes accordingly. If the
        number of minutes overflows (or underflows), then the minutes will wrap.
        If the number of minutes overflows(or underflows), then the hour will
        wrap. (e.g. adding 90 seconds to 23:59:00 would result in 00:00:30).

        Params:
            seconds = The number of seconds to add to this TimeOfDay.
      +/
    ref TimeOfDay addSeconds(long seconds) pure nothrow
    {
        long hnsecs = convert!("seconds", "hnsecs")(seconds);
        hnsecs += convert!("hours", "hnsecs")(_hour);
        hnsecs += convert!("minutes", "hnsecs")(_minute);
        hnsecs += convert!("seconds", "hnsecs")(_second);

        hnsecs %= convert!("days", "hnsecs")(1);

        if(hnsecs < 0)
            hnsecs += convert!("days", "hnsecs")(1);

        immutable newHours = splitUnitsFromHNSecs!"hours"(hnsecs);
        immutable newMinutes = splitUnitsFromHNSecs!"minutes"(hnsecs);
        immutable newSeconds = splitUnitsFromHNSecs!"seconds"(hnsecs);

        _hour = cast(ubyte)newHours;
        _minute = cast(ubyte)newMinutes;
        _second = cast(ubyte)newSeconds;

        return this;
    }

    unittest
    {
        version(testStdDateTime)
        {
            static void testTOD(TimeOfDay orig, int seconds, in TimeOfDay expected, size_t line = __LINE__)
            {
                orig.addSeconds(seconds);
                _assertPred!"=="(orig, expected, "", __FILE__, line);
            }

            testTOD(TimeOfDay(12, 30, 33), 0, TimeOfDay(12, 30, 33));
            testTOD(TimeOfDay(12, 30, 33), 1, TimeOfDay(12, 30, 34));
            testTOD(TimeOfDay(12, 30, 33), 2, TimeOfDay(12, 30, 35));
            testTOD(TimeOfDay(12, 30, 33), 3, TimeOfDay(12, 30, 36));
            testTOD(TimeOfDay(12, 30, 33), 4, TimeOfDay(12, 30, 37));
            testTOD(TimeOfDay(12, 30, 33), 5, TimeOfDay(12, 30, 38));
            testTOD(TimeOfDay(12, 30, 33), 10, TimeOfDay(12, 30, 43));
            testTOD(TimeOfDay(12, 30, 33), 15, TimeOfDay(12, 30, 48));
            testTOD(TimeOfDay(12, 30, 33), 26, TimeOfDay(12, 30, 59));
            testTOD(TimeOfDay(12, 30, 33), 27, TimeOfDay(12, 31, 0));
            testTOD(TimeOfDay(12, 30, 33), 30, TimeOfDay(12, 31, 3));
            testTOD(TimeOfDay(12, 30, 33), 59, TimeOfDay(12, 31, 32));
            testTOD(TimeOfDay(12, 30, 33), 60, TimeOfDay(12, 31, 33));
            testTOD(TimeOfDay(12, 30, 33), 61, TimeOfDay(12, 31, 34));

            testTOD(TimeOfDay(12, 30, 33), 1766, TimeOfDay(12, 59, 59));
            testTOD(TimeOfDay(12, 30, 33), 1767, TimeOfDay(13, 0, 0));
            testTOD(TimeOfDay(12, 30, 33), 1768, TimeOfDay(13, 0, 1));
            testTOD(TimeOfDay(12, 30, 33), 2007, TimeOfDay(13, 4, 0));
            testTOD(TimeOfDay(12, 30, 33), 3599, TimeOfDay(13, 30, 32));
            testTOD(TimeOfDay(12, 30, 33), 3600, TimeOfDay(13, 30, 33));
            testTOD(TimeOfDay(12, 30, 33), 3601, TimeOfDay(13, 30, 34));
            testTOD(TimeOfDay(12, 30, 33), 7200, TimeOfDay(14, 30, 33));

            testTOD(TimeOfDay(12, 30, 33), -1, TimeOfDay(12, 30, 32));
            testTOD(TimeOfDay(12, 30, 33), -2, TimeOfDay(12, 30, 31));
            testTOD(TimeOfDay(12, 30, 33), -3, TimeOfDay(12, 30, 30));
            testTOD(TimeOfDay(12, 30, 33), -4, TimeOfDay(12, 30, 29));
            testTOD(TimeOfDay(12, 30, 33), -5, TimeOfDay(12, 30, 28));
            testTOD(TimeOfDay(12, 30, 33), -10, TimeOfDay(12, 30, 23));
            testTOD(TimeOfDay(12, 30, 33), -15, TimeOfDay(12, 30, 18));
            testTOD(TimeOfDay(12, 30, 33), -33, TimeOfDay(12, 30, 0));
            testTOD(TimeOfDay(12, 30, 33), -34, TimeOfDay(12, 29, 59));
            testTOD(TimeOfDay(12, 30, 33), -35, TimeOfDay(12, 29, 58));
            testTOD(TimeOfDay(12, 30, 33), -59, TimeOfDay(12, 29, 34));
            testTOD(TimeOfDay(12, 30, 33), -60, TimeOfDay(12, 29, 33));
            testTOD(TimeOfDay(12, 30, 33), -61, TimeOfDay(12, 29, 32));

            testTOD(TimeOfDay(12, 30, 33), -1833, TimeOfDay(12, 0, 0));
            testTOD(TimeOfDay(12, 30, 33), -1834, TimeOfDay(11, 59, 59));
            testTOD(TimeOfDay(12, 30, 33), -3600, TimeOfDay(11, 30, 33));
            testTOD(TimeOfDay(12, 30, 33), -3601, TimeOfDay(11, 30, 32));
            testTOD(TimeOfDay(12, 30, 33), -5134, TimeOfDay(11, 4, 59));
            testTOD(TimeOfDay(12, 30, 33), -7200, TimeOfDay(10, 30, 33));

            testTOD(TimeOfDay(12, 30, 0), 1, TimeOfDay(12, 30, 1));
            testTOD(TimeOfDay(12, 30, 0), 0, TimeOfDay(12, 30, 0));
            testTOD(TimeOfDay(12, 30, 0), -1, TimeOfDay(12, 29, 59));

            testTOD(TimeOfDay(12, 0, 0), 1, TimeOfDay(12, 0, 1));
            testTOD(TimeOfDay(12, 0, 0), 0, TimeOfDay(12, 0, 0));
            testTOD(TimeOfDay(12, 0, 0), -1, TimeOfDay(11, 59, 59));

            testTOD(TimeOfDay(0, 0, 0), 1, TimeOfDay(0, 0, 1));
            testTOD(TimeOfDay(0, 0, 0), 0, TimeOfDay(0, 0, 0));
            testTOD(TimeOfDay(0, 0, 0), -1, TimeOfDay(23, 59, 59));

            testTOD(TimeOfDay(23, 59, 59), 1, TimeOfDay(0, 0, 0));
            testTOD(TimeOfDay(23, 59, 59), 0, TimeOfDay(23, 59, 59));
            testTOD(TimeOfDay(23, 59, 59), -1, TimeOfDay(23, 59, 58));

            const ctod = TimeOfDay(0, 0, 0);
            immutable itod = TimeOfDay(0, 0, 0);
            static assert(!__traits(compiles, ctod.addSeconds(7)));
            static assert(!__traits(compiles, itod.addSeconds(7)));
        }
    }


    /+
        Whether the given values form a valid $(D TimeOfDay).
     +/
    static bool _valid(int hour, int minute, int second) pure nothrow
    {
        return valid!"hours"(hour) && valid!"minutes"(minute) && valid!"seconds"(second);
    }


    pure invariant()
    {
        assert(_valid(_hour, _minute, _second),
               "Invariant Failure: hour [" ~
               numToString(_hour) ~
               "] minute [" ~
               numToString(_minute) ~
               "] second [" ~
               numToString(_second) ~
               "]");
    }

    ubyte _hour;
    ubyte _minute;
    ubyte _second;

    enum ubyte maxHour   = 24 - 1;
    enum ubyte maxMinute = 60 - 1;
    enum ubyte maxSecond = 60 - 1;
}


/++
   Combines the $(D Date) and $(D TimeOfDay) structs to give an object
   which holds both the date and the time. It is optimized for calendar-based
   operations and has no concept of time zone. For an object which is
   optimized for time operations based on the system time, use
   $(D SysTime). $(D SysTime) has a concept of time zone and has much higher
   precision (hnsecs). $(D DateTime) is intended primarily for calendar-based
   uses rather than precise time operations.
  +/
struct DateTime
{
public:

    /++
        Params:
            date = The date portion of $(D DateTime).
            tod  = The time portion of $(D DateTime).
      +/
    this(in Date date, in TimeOfDay tod = TimeOfDay.init) pure nothrow
    {
        _date = date;
        _tod = tod;
    }

    unittest
    {
        version(testStdDateTime)
        {
            {
                auto dt = DateTime.init;
                _assertPred!"=="(dt._date, Date.init);
                _assertPred!"=="(dt._tod, TimeOfDay.init);
            }

            {
                auto dt = DateTime(Date(1999, 7 ,6));
                _assertPred!"=="(dt._date, Date(1999, 7, 6));
                _assertPred!"=="(dt._tod, TimeOfDay.init);
            }

            {
                auto dt = DateTime(Date(1999, 7 ,6), TimeOfDay(12, 30, 33));
                _assertPred!"=="(dt._date, Date(1999, 7, 6));
                _assertPred!"=="(dt._tod, TimeOfDay(12, 30, 33));
            }
        }
    }


    /++
        Params:
            year   = The year portion of the date.
            month  = The month portion of the date.
            day    = The day portion of the date.
            hour   = The hour portion of the time;
            minute = The minute portion of the time;
            second = The second portion of the time;
      +/
    this(int year, int month, int day,
         int hour = 0, int minute = 0, int second = 0) pure
    {
        _date = Date(year, month, day);
        _tod = TimeOfDay(hour, minute, second);
    }

    unittest
    {
        version(testStdDateTime)
        {
            {
                auto dt = DateTime(1999, 7 ,6);
                _assertPred!"=="(dt._date, Date(1999, 7, 6));
                _assertPred!"=="(dt._tod, TimeOfDay.init);
            }

            {
                auto dt = DateTime(1999, 7 ,6, 12, 30, 33);
                _assertPred!"=="(dt._date, Date(1999, 7, 6));
                _assertPred!"=="(dt._tod, TimeOfDay(12, 30, 33));
            }
        }
    }


    /++
        Compares this $(D DateTime) with the given $(D DateTime.).

        Returns:
            $(BOOKTABLE,
            $(TR $(TD this &lt; rhs) $(TD &lt; 0))
            $(TR $(TD this == rhs) $(TD 0))
            $(TR $(TD this &gt; rhs) $(TD &gt; 0))
            )
     +/
    int opCmp(in DateTime rhs) const pure nothrow
    {
        immutable dateResult = _date.opCmp(rhs._date);

        if(dateResult != 0)
            return dateResult;

        return _tod.opCmp(rhs._tod);
    }

    unittest
    {
        version(testStdDateTime)
        {
            //Test A.D.
            _assertPred!("opCmp", "==")(DateTime(Date.init, TimeOfDay.init), DateTime.init);

            _assertPred!("opCmp", "==")(DateTime(Date(1999, 1, 1)), DateTime(Date(1999, 1, 1)));
            _assertPred!("opCmp", "==")(DateTime(Date(1, 7, 1)), DateTime(Date(1, 7, 1)));
            _assertPred!("opCmp", "==")(DateTime(Date(1, 1, 6)), DateTime(Date(1, 1, 6)));

            _assertPred!("opCmp", "==")(DateTime(Date(1999, 7, 1)), DateTime(Date(1999, 7, 1)));
            _assertPred!("opCmp", "==")(DateTime(Date(1999, 7, 6)), DateTime(Date(1999, 7, 6)));

            _assertPred!("opCmp", "==")(DateTime(Date(1, 7, 6)), DateTime(Date(1, 7, 6)));

            _assertPred!("opCmp", "<")(DateTime(Date(1999, 7, 6)), DateTime(Date(2000, 7, 6)));
            _assertPred!("opCmp", ">")(DateTime(Date(2000, 7, 6)), DateTime(Date(1999, 7, 6)));
            _assertPred!("opCmp", "<")(DateTime(Date(1999, 7, 6)), DateTime(Date(1999, 8, 6)));
            _assertPred!("opCmp", ">")(DateTime(Date(1999, 8, 6)), DateTime(Date(1999, 7, 6)));
            _assertPred!("opCmp", "<")(DateTime(Date(1999, 7, 6)), DateTime(Date(1999, 7, 7)));
            _assertPred!("opCmp", ">")(DateTime(Date(1999, 7, 7)), DateTime(Date(1999, 7, 6)));

            _assertPred!("opCmp", "<")(DateTime(Date(1999, 8, 7)), DateTime(Date(2000, 7, 6)));
            _assertPred!("opCmp", ">")(DateTime(Date(2000, 8, 6)), DateTime(Date(1999, 7, 7)));
            _assertPred!("opCmp", "<")(DateTime(Date(1999, 7, 7)), DateTime(Date(2000, 7, 6)));
            _assertPred!("opCmp", ">")(DateTime(Date(2000, 7, 6)), DateTime(Date(1999, 7, 7)));
            _assertPred!("opCmp", "<")(DateTime(Date(1999, 7, 7)), DateTime(Date(1999, 8, 6)));
            _assertPred!("opCmp", ">")(DateTime(Date(1999, 8, 6)), DateTime(Date(1999, 7, 7)));


            _assertPred!("opCmp", "==")(DateTime(Date(1999, 7, 6), TimeOfDay(0, 0, 0)),
                                       DateTime(Date(1999, 7, 6), TimeOfDay(0, 0, 0)));
            _assertPred!("opCmp", "==")(DateTime(Date(1999, 7, 6), TimeOfDay(12, 0, 0)),
                                       DateTime(Date(1999, 7, 6), TimeOfDay(12, 0, 0)));
            _assertPred!("opCmp", "==")(DateTime(Date(1999, 7, 6), TimeOfDay(0, 30, 0)),
                                       DateTime(Date(1999, 7, 6), TimeOfDay(0, 30, 0)));
            _assertPred!("opCmp", "==")(DateTime(Date(1999, 7, 6), TimeOfDay(0, 0, 33)),
                                       DateTime(Date(1999, 7, 6), TimeOfDay(0, 0, 33)));

            _assertPred!("opCmp", "==")(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 0)),
                                       DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 0)));
            _assertPred!("opCmp", "==")(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)),
                                       DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));

            _assertPred!("opCmp", "==")(DateTime(Date(1999, 7, 6), TimeOfDay(0, 30, 33)),
                                       DateTime(Date(1999, 7, 6), TimeOfDay(0, 30, 33)));
            _assertPred!("opCmp", "==")(DateTime(Date(1999, 7, 6), TimeOfDay(0, 0, 33)),
                                       DateTime(Date(1999, 7, 6), TimeOfDay(0, 0, 33)));

            _assertPred!("opCmp", "<")(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)),
                                      DateTime(Date(1999, 7, 6), TimeOfDay(13, 30, 33)));
            _assertPred!("opCmp", ">")(DateTime(Date(1999, 7, 6), TimeOfDay(13, 30, 33)),
                                      DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
            _assertPred!("opCmp", "<")(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)),
                                      DateTime(Date(1999, 7, 6), TimeOfDay(12, 31, 33)));
            _assertPred!("opCmp", ">")(DateTime(Date(1999, 7, 6), TimeOfDay(12, 31, 33)),
                                      DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
            _assertPred!("opCmp", "<")(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)),
                                      DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 34)));
            _assertPred!("opCmp", ">")(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 34)),
                                      DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));

            _assertPred!("opCmp", ">")(DateTime(Date(1999, 7, 6), TimeOfDay(13, 30, 33)),
                                      DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 34)));
            _assertPred!("opCmp", "<")(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 34)),
                                      DateTime(Date(1999, 7, 6), TimeOfDay(13, 30, 33)));
            _assertPred!("opCmp", ">")(DateTime(Date(1999, 7, 6), TimeOfDay(13, 30, 33)),
                                      DateTime(Date(1999, 7, 6), TimeOfDay(12, 31, 33)));
            _assertPred!("opCmp", "<")(DateTime(Date(1999, 7, 6), TimeOfDay(12, 31, 33)),
                                      DateTime(Date(1999, 7, 6), TimeOfDay(13, 30, 33)));

            _assertPred!("opCmp", ">")(DateTime(Date(1999, 7, 6), TimeOfDay(12, 31, 33)),
                                      DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 34)));
            _assertPred!("opCmp", "<")(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 34)),
                                      DateTime(Date(1999, 7, 6), TimeOfDay(12, 31, 33)));

            _assertPred!("opCmp", "<")(DateTime(Date(1999, 7, 6), TimeOfDay(13, 30, 33)),
                                      DateTime(Date(2000, 7, 6), TimeOfDay(12, 30, 33)));
            _assertPred!("opCmp", ">")(DateTime(Date(2000, 7, 6), TimeOfDay(12, 30, 33)),
                                      DateTime(Date(1999, 7, 6), TimeOfDay(13, 30, 33)));
            _assertPred!("opCmp", "<")(DateTime(Date(1999, 7, 6), TimeOfDay(12, 31, 33)),
                                      DateTime(Date(2000, 7, 6), TimeOfDay(12, 30, 33)));
            _assertPred!("opCmp", ">")(DateTime(Date(2000, 7, 6), TimeOfDay(12, 30, 33)),
                                      DateTime(Date(1999, 7, 6), TimeOfDay(12, 31, 33)));
            _assertPred!("opCmp", "<")(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 34)),
                                      DateTime(Date(2000, 7, 6), TimeOfDay(12, 30, 33)));
            _assertPred!("opCmp", ">")(DateTime(Date(2000, 7, 6), TimeOfDay(12, 30, 33)),
                                      DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 34)));

            _assertPred!("opCmp", "<")(DateTime(Date(1999, 7, 6), TimeOfDay(13, 30, 33)),
                                      DateTime(Date(1999, 8, 6), TimeOfDay(12, 30, 33)));
            _assertPred!("opCmp", ">")(DateTime(Date(1999, 8, 6), TimeOfDay(12, 30, 33)),
                                      DateTime(Date(1999, 7, 6), TimeOfDay(13, 30, 33)));
            _assertPred!("opCmp", "<")(DateTime(Date(1999, 7, 6), TimeOfDay(12, 31, 33)),
                                      DateTime(Date(1999, 8, 6), TimeOfDay(12, 30, 33)));
            _assertPred!("opCmp", ">")(DateTime(Date(1999, 8, 6), TimeOfDay(12, 30, 33)),
                                      DateTime(Date(1999, 7, 6), TimeOfDay(12, 31, 33)));
            _assertPred!("opCmp", "<")(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 34)),
                                      DateTime(Date(1999, 8, 6), TimeOfDay(12, 30, 33)));
            _assertPred!("opCmp", ">")(DateTime(Date(1999, 8, 6), TimeOfDay(12, 30, 33)),
                                      DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 34)));

            _assertPred!("opCmp", "<")(DateTime(Date(1999, 7, 6), TimeOfDay(13, 30, 33)),
                                      DateTime(Date(1999, 7, 7), TimeOfDay(12, 30, 33)));
            _assertPred!("opCmp", ">")(DateTime(Date(1999, 7, 7), TimeOfDay(12, 30, 33)),
                                      DateTime(Date(1999, 7, 6), TimeOfDay(13, 30, 33)));
            _assertPred!("opCmp", "<")(DateTime(Date(1999, 7, 6), TimeOfDay(12, 31, 33)),
                                      DateTime(Date(1999, 7, 7), TimeOfDay(12, 31, 33)));
            _assertPred!("opCmp", ">")(DateTime(Date(1999, 7, 7), TimeOfDay(12, 30, 33)),
                                      DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
            _assertPred!("opCmp", "<")(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 34)),
                                      DateTime(Date(1999, 7, 7), TimeOfDay(12, 30, 33)));
            _assertPred!("opCmp", ">")(DateTime(Date(1999, 7, 7), TimeOfDay(12, 30, 33)),
                                      DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 34)));

            //Test B.C.
            _assertPred!("opCmp", "==")(DateTime(Date(-1, 1, 1), TimeOfDay(12, 30, 33)),
                                       DateTime(Date(-1, 1, 1), TimeOfDay(12, 30, 33)));
            _assertPred!("opCmp", "==")(DateTime(Date(-1, 7, 1), TimeOfDay(12, 30, 33)),
                                       DateTime(Date(-1, 7, 1), TimeOfDay(12, 30, 33)));
            _assertPred!("opCmp", "==")(DateTime(Date(-1, 1, 6), TimeOfDay(12, 30, 33)),
                                       DateTime(Date(-1, 1, 6), TimeOfDay(12, 30, 33)));

            _assertPred!("opCmp", "==")(DateTime(Date(-1999, 7, 1), TimeOfDay(12, 30, 33)),
                                       DateTime(Date(-1999, 7, 1), TimeOfDay(12, 30, 33)));
            _assertPred!("opCmp", "==")(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)),
                                       DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)));

            _assertPred!("opCmp", "==")(DateTime(Date(-1, 7, 6), TimeOfDay(12, 30, 33)),
                                       DateTime(Date(-1, 7, 6), TimeOfDay(12, 30, 33)));

            _assertPred!("opCmp", "<")(DateTime(Date(-2000, 7, 6), TimeOfDay(12, 30, 33)),
                                      DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)));
            _assertPred!("opCmp", ">")(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)),
                                      DateTime(Date(-2000, 7, 6), TimeOfDay(12, 30, 33)));
            _assertPred!("opCmp", "<")(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)),
                                      DateTime(Date(-1999, 8, 6), TimeOfDay(12, 30, 33)));
            _assertPred!("opCmp", ">")(DateTime(Date(-1999, 8, 6), TimeOfDay(12, 30, 33)),
                                      DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)));
            _assertPred!("opCmp", "<")(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)),
                                      DateTime(Date(-1999, 7, 7), TimeOfDay(12, 30, 33)));
            _assertPred!("opCmp", ">")(DateTime(Date(-1999, 7, 7), TimeOfDay(12, 30, 33)),
                                      DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)));

            _assertPred!("opCmp", "<")(DateTime(Date(-2000, 8, 6), TimeOfDay(12, 30, 33)),
                                      DateTime(Date(-1999, 7, 7), TimeOfDay(12, 30, 33)));
            _assertPred!("opCmp", ">")(DateTime(Date(-1999, 8, 7), TimeOfDay(12, 30, 33)),
                                      DateTime(Date(-2000, 7, 6), TimeOfDay(12, 30, 33)));
            _assertPred!("opCmp", "<")(DateTime(Date(-2000, 7, 6), TimeOfDay(12, 30, 33)),
                                      DateTime(Date(-1999, 7, 7), TimeOfDay(12, 30, 33)));
            _assertPred!("opCmp", ">")(DateTime(Date(-1999, 7, 7), TimeOfDay(12, 30, 33)),
                                      DateTime(Date(-2000, 7, 6), TimeOfDay(12, 30, 33)));
            _assertPred!("opCmp", "<")(DateTime(Date(-1999, 7, 7), TimeOfDay(12, 30, 33)),
                                      DateTime(Date(-1999, 8, 6), TimeOfDay(12, 30, 33)));
            _assertPred!("opCmp", ">")(DateTime(Date(-1999, 8, 6), TimeOfDay(12, 30, 33)),
                                      DateTime(Date(-1999, 7, 7), TimeOfDay(12, 30, 33)));

            //Test Both
            _assertPred!("opCmp", "<")(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)),
                                      DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
            _assertPred!("opCmp", ">")(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)),
                                      DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)));

            _assertPred!("opCmp", "<")(DateTime(Date(-1999, 8, 6), TimeOfDay(12, 30, 33)),
                                      DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
            _assertPred!("opCmp", ">")(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)),
                                      DateTime(Date(-1999, 8, 6), TimeOfDay(12, 30, 33)));

            _assertPred!("opCmp", "<")(DateTime(Date(-1999, 7, 7), TimeOfDay(12, 30, 33)),
                                      DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
            _assertPred!("opCmp", ">")(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)),
                                      DateTime(Date(-1999, 7, 7), TimeOfDay(12, 30, 33)));

            _assertPred!("opCmp", "<")(DateTime(Date(-1999, 8, 7), TimeOfDay(12, 30, 33)),
                                      DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
            _assertPred!("opCmp", ">")(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)),
                                      DateTime(Date(-1999, 8, 7), TimeOfDay(12, 30, 33)));

            _assertPred!("opCmp", "<")(DateTime(Date(-1999, 8, 6), TimeOfDay(12, 30, 33)),
                                      DateTime(Date(1999, 6, 6), TimeOfDay(12, 30, 33)));
            _assertPred!("opCmp", ">")(DateTime(Date(1999, 6, 8), TimeOfDay(12, 30, 33)),
                                      DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)));

            auto dt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 33, 30));
            const cdt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 33, 30));
            immutable idt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 33, 30));
            static assert(__traits(compiles, dt.opCmp(dt)));
            static assert(__traits(compiles, dt.opCmp(cdt)));
            static assert(__traits(compiles, dt.opCmp(idt)));
            static assert(__traits(compiles, cdt.opCmp(dt)));
            static assert(__traits(compiles, cdt.opCmp(cdt)));
            static assert(__traits(compiles, cdt.opCmp(idt)));
            static assert(__traits(compiles, idt.opCmp(dt)));
            static assert(__traits(compiles, idt.opCmp(cdt)));
            static assert(__traits(compiles, idt.opCmp(idt)));
        }
    }


    /++
        The date portion of $(D DateTime).
      +/
    @property Date date() const pure nothrow
    {
        return _date;
    }

    unittest
    {
        version(testStdDateTime)
        {
            {
                auto dt = DateTime.init;
                _assertPred!"=="(dt.date, Date.init);
            }

            {
                auto dt = DateTime(Date(1999, 7, 6));
                _assertPred!"=="(dt.date, Date(1999, 7, 6));
            }

            const cdt = DateTime(1999, 7, 6);
            immutable idt = DateTime(1999, 7, 6);
            static assert(__traits(compiles, cdt.date == Date(2010, 1, 1)));
            static assert(__traits(compiles, idt.date == Date(2010, 1, 1)));
        }
    }


    /++
        The date portion of $(D DateTime).

        Params:
            date = The Date to set this $(D DateTime)'s date portion to.
      +/
    @property void date(in Date date) pure nothrow
    {
        _date = date;
    }

    unittest
    {
        version(testStdDateTime)
        {
            auto dt = DateTime.init;
            dt.date = Date(1999, 7, 6);
            _assertPred!"=="(dt._date, Date(1999, 7, 6));
            _assertPred!"=="(dt._tod, TimeOfDay.init);

            const cdt = DateTime(1999, 7, 6);
            immutable idt = DateTime(1999, 7, 6);
            static assert(!__traits(compiles, cdt.date = Date(2010, 1, 1)));
            static assert(!__traits(compiles, idt.date = Date(2010, 1, 1)));
        }
    }


    /++
        The time portion of $(D DateTime).
      +/
    @property TimeOfDay timeOfDay() const pure nothrow
    {
        return _tod;
    }

    unittest
    {
        version(testStdDateTime)
        {
            {
                auto dt = DateTime.init;
                _assertPred!"=="(dt.timeOfDay, TimeOfDay.init);
            }

            {
                auto dt = DateTime(Date.init, TimeOfDay(12, 30, 33));
                _assertPred!"=="(dt.timeOfDay, TimeOfDay(12, 30, 33));
            }

            const cdt = DateTime(1999, 7, 6, 12, 30, 33);
            immutable idt = DateTime(1999, 7, 6, 12, 30, 33);
            static assert(__traits(compiles, cdt.timeOfDay == TimeOfDay(12, 30, 33)));
            static assert(__traits(compiles, idt.timeOfDay == TimeOfDay(12, 30, 33)));
        }
    }


    /++
        The time portion of $(D DateTime).

        Params:
            tod = The $(D TimeOfDay) to set this $(D DateTime)'s time portion
                  to.
      +/
    @property void timeOfDay(in TimeOfDay tod) pure nothrow
    {
        _tod = tod;
    }

    unittest
    {
        version(testStdDateTime)
        {
            auto dt = DateTime.init;
            dt.timeOfDay = TimeOfDay(12, 30, 33);
            _assertPred!"=="(dt._date, date.init);
            _assertPred!"=="(dt._tod, TimeOfDay(12, 30, 33));

            const cdt = DateTime(1999, 7, 6, 12, 30, 33);
            immutable idt = DateTime(1999, 7, 6, 12, 30, 33);
            static assert(!__traits(compiles, cdt.timeOfDay = TimeOfDay(12, 30, 33)));
            static assert(!__traits(compiles, idt.timeOfDay = TimeOfDay(12, 30, 33)));
        }
    }


    /++
        Year of the Gregorian Calendar. Positive numbers are A.D. Non-positive
        are B.C.
     +/
    @property short year() const pure nothrow
    {
        return _date.year;
    }

    unittest
    {
        version(testStdDateTime)
        {
            _assertPred!"=="(Date.init.year, 1);
            _assertPred!"=="(Date(1999, 7, 6).year, 1999);
            _assertPred!"=="(Date(-1999, 7, 6).year, -1999);

            const cdt = DateTime(1999, 7, 6, 12, 30, 33);
            immutable idt = DateTime(1999, 7, 6, 12, 30, 33);
            static assert(__traits(compiles, idt.year));
            static assert(__traits(compiles, idt.year));
        }
    }


    /++
        Year of the Gregorian Calendar. Positive numbers are A.D. Non-positive
        are B.C.

        Params:
            year = The year to set this $(D DateTime)'s year to.

        Throws:
            $(D DateTimeException) if the new year is not a leap year and if the
            resulting date would be on February 29th.

        Examples:
--------------------
assert(DateTime(Date(1999, 7, 6), TimeOfDay(9, 7, 5)).year == 1999);
assert(DateTime(Date(2010, 10, 4), TimeOfDay(0, 0, 30)).year == 2010);
assert(DateTime(Date(-7, 4, 5), TimeOfDay(7, 45, 2)).year == -7);
--------------------
     +/
    @property void year(int year) pure
    {
        _date.year = year;
    }

    unittest
    {
        version(testStdDateTime)
        {
            static void testDT(DateTime dt, int year, in DateTime expected, size_t line = __LINE__)
            {
                dt.year = year;
                _assertPred!"=="(dt, expected, "", __FILE__, line);
            }

            testDT(DateTime(Date(1, 1, 1), TimeOfDay(12, 30, 33)), 1999, DateTime(Date(1999, 1, 1), TimeOfDay(12, 30, 33)));
            testDT(DateTime(Date(1, 1, 1), TimeOfDay(12, 30, 33)), 0, DateTime(Date(0, 1, 1), TimeOfDay(12, 30, 33)));
            testDT(DateTime(Date(1, 1, 1), TimeOfDay(12, 30, 33)), -1999, DateTime(Date(-1999, 1, 1), TimeOfDay(12, 30, 33)));

            const cdt = DateTime(1999, 7, 6, 12, 30, 33);
            immutable idt = DateTime(1999, 7, 6, 12, 30, 33);
            static assert(!__traits(compiles, cdt.year = 7));
            static assert(!__traits(compiles, idt.year = 7));

            //Verify Examples.
            assert(DateTime(Date(1999, 7, 6), TimeOfDay(9, 7, 5)).year == 1999);
            assert(DateTime(Date(2010, 10, 4), TimeOfDay(0, 0, 30)).year == 2010);
            assert(DateTime(Date(-7, 4, 5), TimeOfDay(7, 45, 2)).year == -7);
        }
    }


    /++
        Year B.C. of the Gregorian Calendar counting year 0 as 1 B.C.

        Throws:
            $(D DateTimeException) if $(D isAD) is true.

        Examples:
--------------------
assert(DateTime(Date(0, 1, 1), TimeOfDay(12, 30, 33)).yearBC == 1);
assert(DateTime(Date(-1, 1, 1), TimeOfDay(10, 7, 2)).yearBC == 2);
assert(DateTime(Date(-100, 1, 1), TimeOfDay(4, 59, 0)).yearBC == 101);
--------------------
     +/
    @property short yearBC() const pure
    {
        return _date.yearBC;
    }

    unittest
    {
        version(testStdDateTime)
        {
            assertThrown!DateTimeException((in DateTime dt){dt.yearBC;}(DateTime(Date(1, 1, 1))));

            auto dt = DateTime(1999, 7, 6, 12, 30, 33);
            const cdt = DateTime(1999, 7, 6, 12, 30, 33);
            immutable idt = DateTime(1999, 7, 6, 12, 30, 33);
            static assert(__traits(compiles, dt.yearBC = 12));
            static assert(!__traits(compiles, cdt.yearBC = 12));
            static assert(!__traits(compiles, idt.yearBC = 12));

            //Verify Examples.
            assert(DateTime(Date(0, 1, 1), TimeOfDay(12, 30, 33)).yearBC == 1);
            assert(DateTime(Date(-1, 1, 1), TimeOfDay(10, 7, 2)).yearBC == 2);
            assert(DateTime(Date(-100, 1, 1), TimeOfDay(4, 59, 0)).yearBC == 101);
        }
    }


    /++
        Year B.C. of the Gregorian Calendar counting year 0 as 1 B.C.

        Params:
            year = The year B.C. to set this $(D DateTime)'s year to.

        Throws:
            $(D DateTimeException) if a non-positive value is given.

        Examples:
--------------------
auto dt = DateTime(Date(2010, 1, 1), TimeOfDay(7, 30, 0));
dt.yearBC = 1;
assert(dt == DateTime(Date(0, 1, 1), TimeOfDay(7, 30, 0)));

dt.yearBC = 10;
assert(dt == DateTime(Date(-9, 1, 1), TimeOfDay(7, 30, 0)));
--------------------
     +/
    @property void yearBC(int year) pure
    {
        _date.yearBC = year;
    }

    unittest
    {
        version(testStdDateTime)
        {
            assertThrown!DateTimeException((DateTime dt){dt.yearBC = -1;}(DateTime(Date(1, 1, 1))));

            {
                auto dt = DateTime(1999, 7, 6, 12, 30, 33);
                const cdt = DateTime(1999, 7, 6, 12, 30, 33);
                immutable idt = DateTime(1999, 7, 6, 12, 30, 33);
                static assert(__traits(compiles, dt.yearBC = 12));
                static assert(!__traits(compiles, cdt.yearBC = 12));
                static assert(!__traits(compiles, idt.yearBC = 12));
            }

            //Verify Examples.
            {
                auto dt = DateTime(Date(2010, 1, 1), TimeOfDay(7, 30, 0));
                dt.yearBC = 1;
                assert(dt == DateTime(Date(0, 1, 1), TimeOfDay(7, 30, 0)));

                dt.yearBC = 10;
                assert(dt == DateTime(Date(-9, 1, 1), TimeOfDay(7, 30, 0)));
            }
        }
    }


    /++
        Month of a Gregorian Year.

        Examples:
--------------------
assert(DateTime(Date(1999, 7, 6), TimeOfDay(9, 7, 5)).month == 7);
assert(DateTime(Date(2010, 10, 4), TimeOfDay(0, 0, 30)).month == 10);
assert(DateTime(Date(-7, 4, 5), TimeOfDay(7, 45, 2)).month == 4);
--------------------
     +/
    @property Month month() const pure nothrow
    {
        return _date.month;
    }

    unittest
    {
        version(testStdDateTime)
        {
            _assertPred!"=="(DateTime.init.month, 1);
            _assertPred!"=="(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)).month, 7);
            _assertPred!"=="(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)).month, 7);

            const cdt = DateTime(1999, 7, 6, 12, 30, 33);
            immutable idt = DateTime(1999, 7, 6, 12, 30, 33);
            static assert(__traits(compiles, cdt.month));
            static assert(__traits(compiles, idt.month));

            //Verify Examples.
            assert(DateTime(Date(1999, 7, 6), TimeOfDay(9, 7, 5)).month == 7);
            assert(DateTime(Date(2010, 10, 4), TimeOfDay(0, 0, 30)).month == 10);
            assert(DateTime(Date(-7, 4, 5), TimeOfDay(7, 45, 2)).month == 4);
        }
    }


    /++
        Month of a Gregorian Year.

        Params:
            month = The month to set this $(D DateTime)'s month to.

        Throws:
            $(D DateTimeException) if the given month is not a valid month.
     +/
    @property void month(Month month) pure
    {
        _date.month = month;
    }

    unittest
    {
        version(testStdDateTime)
        {
            static void testDT(DateTime dt, Month month, in DateTime expected = DateTime.init, size_t line = __LINE__)
            {
                dt.month = month;
                assert(expected != DateTime.init);
                _assertPred!"=="(dt, expected, "", __FILE__, line);
            }

            assertThrown!DateTimeException(testDT(DateTime(Date(1, 1, 1), TimeOfDay(12, 30, 33)), cast(Month)0));
            assertThrown!DateTimeException(testDT(DateTime(Date(1, 1, 1), TimeOfDay(12, 30, 33)), cast(Month)13));

            testDT(DateTime(Date(1, 1, 1), TimeOfDay(12, 30, 33)), cast(Month)7, DateTime(Date(1, 7, 1), TimeOfDay(12, 30, 33)));
            testDT(DateTime(Date(-1, 1, 1), TimeOfDay(12, 30, 33)), cast(Month)7, DateTime(Date(-1, 7, 1), TimeOfDay(12, 30, 33)));

            const cdt = DateTime(1999, 7, 6, 12, 30, 33);
            immutable idt = DateTime(1999, 7, 6, 12, 30, 33);
            static assert(!__traits(compiles, cdt.month = 12));
            static assert(!__traits(compiles, idt.month = 12));
        }
    }


    /++
        Day of a Gregorian Month.

        Examples:
--------------------
assert(DateTime(Date(1999, 7, 6), TimeOfDay(9, 7, 5)).day == 6);
assert(DateTime(Date(2010, 10, 4), TimeOfDay(0, 0, 30)).day == 4);
assert(DateTime(Date(-7, 4, 5), TimeOfDay(7, 45, 2)).day == 5);
--------------------
     +/
    @property ubyte day() const pure nothrow
    {
        return _date.day;
    }

    //Verify Examples.
    version(testStdDateTime) unittest
    {
        assert(DateTime(Date(1999, 7, 6), TimeOfDay(9, 7, 5)).day == 6);
        assert(DateTime(Date(2010, 10, 4), TimeOfDay(0, 0, 30)).day == 4);
        assert(DateTime(Date(-7, 4, 5), TimeOfDay(7, 45, 2)).day == 5);
    }

    version(testStdDateTime) unittest
    {
        static void test(DateTime dateTime, int expected, size_t line = __LINE__)
        {
            _assertPred!"=="(dateTime.day, expected,
                             format("Value given: %s", dateTime), __FILE__, line);
        }

        foreach(year; chain(testYearsBC, testYearsAD))
        {
            foreach(md; testMonthDays)
            {
                foreach(tod; testTODs)
                    test(DateTime(Date(year, md.month, md.day), tod), md.day);
            }
        }

        const cdt = DateTime(1999, 7, 6, 12, 30, 33);
        immutable idt = DateTime(1999, 7, 6, 12, 30, 33);
        static assert(__traits(compiles, cdt.day));
        static assert(__traits(compiles, idt.day));
    }


    /++
        Day of a Gregorian Month.

        Params:
            day = The day of the month to set this $(D DateTime)'s day to.

        Throws:
            $(D DateTimeException) if the given day is not a valid day of the
            current month.
     +/
    @property void day(int day) pure
    {
        _date.day = day;
    }

    unittest
    {
        version(testStdDateTime)
        {
            static void testDT(DateTime dt, int day)
            {
                dt.day = day;
            }

            //Test A.D.
            assertThrown!DateTimeException(testDT(DateTime(Date(1, 1, 1)), 0));
            assertThrown!DateTimeException(testDT(DateTime(Date(1, 1, 1)), 32));
            assertThrown!DateTimeException(testDT(DateTime(Date(1, 2, 1)), 29));
            assertThrown!DateTimeException(testDT(DateTime(Date(4, 2, 1)), 30));
            assertThrown!DateTimeException(testDT(DateTime(Date(1, 3, 1)), 32));
            assertThrown!DateTimeException(testDT(DateTime(Date(1, 4, 1)), 31));
            assertThrown!DateTimeException(testDT(DateTime(Date(1, 5, 1)), 32));
            assertThrown!DateTimeException(testDT(DateTime(Date(1, 6, 1)), 31));
            assertThrown!DateTimeException(testDT(DateTime(Date(1, 7, 1)), 32));
            assertThrown!DateTimeException(testDT(DateTime(Date(1, 8, 1)), 32));
            assertThrown!DateTimeException(testDT(DateTime(Date(1, 9, 1)), 31));
            assertThrown!DateTimeException(testDT(DateTime(Date(1, 10, 1)), 32));
            assertThrown!DateTimeException(testDT(DateTime(Date(1, 11, 1)), 31));
            assertThrown!DateTimeException(testDT(DateTime(Date(1, 12, 1)), 32));

            assertNotThrown!DateTimeException(testDT(DateTime(Date(1, 1, 1)), 31));
            assertNotThrown!DateTimeException(testDT(DateTime(Date(1, 2, 1)), 28));
            assertNotThrown!DateTimeException(testDT(DateTime(Date(4, 2, 1)), 29));
            assertNotThrown!DateTimeException(testDT(DateTime(Date(1, 3, 1)), 31));
            assertNotThrown!DateTimeException(testDT(DateTime(Date(1, 4, 1)), 30));
            assertNotThrown!DateTimeException(testDT(DateTime(Date(1, 5, 1)), 31));
            assertNotThrown!DateTimeException(testDT(DateTime(Date(1, 6, 1)), 30));
            assertNotThrown!DateTimeException(testDT(DateTime(Date(1, 7, 1)), 31));
            assertNotThrown!DateTimeException(testDT(DateTime(Date(1, 8, 1)), 31));
            assertNotThrown!DateTimeException(testDT(DateTime(Date(1, 9, 1)), 30));
            assertNotThrown!DateTimeException(testDT(DateTime(Date(1, 10, 1)), 31));
            assertNotThrown!DateTimeException(testDT(DateTime(Date(1, 11, 1)), 30));
            assertNotThrown!DateTimeException(testDT(DateTime(Date(1, 12, 1)), 31));

            {
                auto dt = DateTime(Date(1, 1, 1), TimeOfDay(7, 12, 22));
                dt.day = 6;
                _assertPred!"=="(dt, DateTime(Date(1, 1, 6), TimeOfDay(7, 12, 22)));
            }

            //Test B.C.
            assertThrown!DateTimeException(testDT(DateTime(Date(-1, 1, 1)), 0));
            assertThrown!DateTimeException(testDT(DateTime(Date(-1, 1, 1)), 32));
            assertThrown!DateTimeException(testDT(DateTime(Date(-1, 2, 1)), 29));
            assertThrown!DateTimeException(testDT(DateTime(Date(0, 2, 1)), 30));
            assertThrown!DateTimeException(testDT(DateTime(Date(-1, 3, 1)), 32));
            assertThrown!DateTimeException(testDT(DateTime(Date(-1, 4, 1)), 31));
            assertThrown!DateTimeException(testDT(DateTime(Date(-1, 5, 1)), 32));
            assertThrown!DateTimeException(testDT(DateTime(Date(-1, 6, 1)), 31));
            assertThrown!DateTimeException(testDT(DateTime(Date(-1, 7, 1)), 32));
            assertThrown!DateTimeException(testDT(DateTime(Date(-1, 8, 1)), 32));
            assertThrown!DateTimeException(testDT(DateTime(Date(-1, 9, 1)), 31));
            assertThrown!DateTimeException(testDT(DateTime(Date(-1, 10, 1)), 32));
            assertThrown!DateTimeException(testDT(DateTime(Date(-1, 11, 1)), 31));
            assertThrown!DateTimeException(testDT(DateTime(Date(-1, 12, 1)), 32));

            assertNotThrown!DateTimeException(testDT(DateTime(Date(-1, 1, 1)), 31));
            assertNotThrown!DateTimeException(testDT(DateTime(Date(-1, 2, 1)), 28));
            assertNotThrown!DateTimeException(testDT(DateTime(Date(0, 2, 1)), 29));
            assertNotThrown!DateTimeException(testDT(DateTime(Date(-1, 3, 1)), 31));
            assertNotThrown!DateTimeException(testDT(DateTime(Date(-1, 4, 1)), 30));
            assertNotThrown!DateTimeException(testDT(DateTime(Date(-1, 5, 1)), 31));
            assertNotThrown!DateTimeException(testDT(DateTime(Date(-1, 6, 1)), 30));
            assertNotThrown!DateTimeException(testDT(DateTime(Date(-1, 7, 1)), 31));
            assertNotThrown!DateTimeException(testDT(DateTime(Date(-1, 8, 1)), 31));
            assertNotThrown!DateTimeException(testDT(DateTime(Date(-1, 9, 1)), 30));
            assertNotThrown!DateTimeException(testDT(DateTime(Date(-1, 10, 1)), 31));
            assertNotThrown!DateTimeException(testDT(DateTime(Date(-1, 11, 1)), 30));
            assertNotThrown!DateTimeException(testDT(DateTime(Date(-1, 12, 1)), 31));

            auto dt = DateTime(Date(-1, 1, 1), TimeOfDay(7, 12, 22));
            dt.day = 6;
            _assertPred!"=="(dt, DateTime(Date(-1, 1, 6), TimeOfDay(7, 12, 22)));

            const cdt = DateTime(1999, 7, 6, 12, 30, 33);
            immutable idt = DateTime(1999, 7, 6, 12, 30, 33);
            static assert(!__traits(compiles, cdt.day = 27));
            static assert(!__traits(compiles, idt.day = 27));
        }
    }


    /++
        Hours passed midnight.
     +/
    @property ubyte hour() const pure nothrow
    {
        return _tod.hour;
    }

    unittest
    {
        version(testStdDateTime)
        {
            _assertPred!"=="(DateTime.init.hour, 0);
            _assertPred!"=="(DateTime(Date.init, TimeOfDay(12, 0, 0)).hour, 12);

            const cdt = DateTime(1999, 7, 6, 12, 30, 33);
            immutable idt = DateTime(1999, 7, 6, 12, 30, 33);
            static assert(__traits(compiles, cdt.hour));
            static assert(__traits(compiles, idt.hour));
        }
    }


    /++
        Hours passed midnight.

        Params:
            hour = The hour of the day to set this $(D DateTime)'s hour to.

        Throws:
            $(D DateTimeException) if the given hour would result in an invalid
            $(D DateTime).
     +/
    @property void hour(int hour) pure
    {
        _tod.hour = hour;
    }

    unittest
    {
        version(testStdDateTime)
        {
            assertThrown!DateTimeException((){DateTime(Date(1999, 7, 6), TimeOfDay(0, 0, 0)).hour = 24;}());

            auto dt = DateTime.init;
            dt.hour = 12;
            _assertPred!"=="(dt, DateTime(1, 1, 1, 12, 0, 0));

            const cdt = DateTime(1999, 7, 6, 12, 30, 33);
            immutable idt = DateTime(1999, 7, 6, 12, 30, 33);
            static assert(!__traits(compiles, cdt.hour = 27));
            static assert(!__traits(compiles, idt.hour = 27));
        }
    }


    /++
        Minutes passed the hour.
     +/
    @property ubyte minute() const pure nothrow
    {
        return _tod.minute;
    }

    unittest
    {
        version(testStdDateTime)
        {
            _assertPred!"=="(DateTime.init.minute, 0);
            _assertPred!"=="(DateTime(1, 1, 1, 0, 30, 0).minute, 30);

            const cdt = DateTime(1999, 7, 6, 12, 30, 33);
            immutable idt = DateTime(1999, 7, 6, 12, 30, 33);
            static assert(__traits(compiles, cdt.minute));
            static assert(__traits(compiles, idt.minute));
        }
    }


    /++
        Minutes passed the hour.

        Params:
            minute = The minute to set this $(D DateTime)'s minute to.

        Throws:
            $(D DateTimeException) if the given minute would result in an
            invalid $(D DateTime).
     +/
    @property void minute(int minute) pure
    {
        _tod.minute = minute;
    }

    unittest
    {
        version(testStdDateTime)
        {
            assertThrown!DateTimeException((){DateTime.init.minute = 60;}());

            auto dt = DateTime.init;
            dt.minute = 30;
            _assertPred!"=="(dt, DateTime(1, 1, 1, 0, 30, 0));

            const cdt = DateTime(1999, 7, 6, 12, 30, 33);
            immutable idt = DateTime(1999, 7, 6, 12, 30, 33);
            static assert(!__traits(compiles, cdt.minute = 27));
            static assert(!__traits(compiles, idt.minute = 27));
        }
    }


    /++
        Seconds passed the minute.
     +/
    @property ubyte second() const pure nothrow
    {
        return _tod.second;
    }

    unittest
    {
        version(testStdDateTime)
        {
            _assertPred!"=="(DateTime.init.second, 0);
            _assertPred!"=="(DateTime(1, 1, 1, 0, 0, 33).second, 33);

            const cdt = DateTime(1999, 7, 6, 12, 30, 33);
            immutable idt = DateTime(1999, 7, 6, 12, 30, 33);
            static assert(__traits(compiles, cdt.second));
            static assert(__traits(compiles, idt.second));
        }
    }


    /++
        Seconds passed the minute.

        Params:
            second = The second to set this $(D DateTime)'s second to.

        Throws:
            $(D DateTimeException) if the given seconds would result in an
            invalid $(D DateTime).
     +/
    @property void second(int second) pure
    {
        _tod.second = second;
    }

    unittest
    {
        version(testStdDateTime)
        {
            assertThrown!DateTimeException((){DateTime.init.second = 60;}());

            auto dt = DateTime.init;
            dt.second = 33;
            _assertPred!"=="(dt, DateTime(1, 1, 1, 0, 0, 33));

            const cdt = DateTime(1999, 7, 6, 12, 30, 33);
            immutable idt = DateTime(1999, 7, 6, 12, 30, 33);
            static assert(!__traits(compiles, cdt.second = 27));
            static assert(!__traits(compiles, idt.second = 27));
        }
    }


    /++
        Adds the given number of years or months to this $(D DateTime). A
        negative number will subtract.

        Note that if day overflow is allowed, and the date with the adjusted
        year/month overflows the number of days in the new month, then the month
        will be incremented by one, and the day set to the number of days
        overflowed. (e.g. if the day were 31 and the new month were June, then
        the month would be incremented to July, and the new day would be 1). If
        day overflow is not allowed, then the day will be set to the last valid
        day in the month (e.g. June 31st would become June 30th).

        Params:
            units         = The type of units to add ("years" or "months").
            value         = The number of months or years to add to this
                            $(D DateTime).
            allowOverflow = Whether the days should be allowed to overflow,
                            causing the month to increment.

        Examples:
--------------------
auto dt1 = DateTime(2010, 1, 1, 12, 30, 33);
dt1.add!"months"(11);
assert(dt1 == DateTime(2010, 12, 1, 12, 30, 33));

auto dt2 = DateTime(2010, 1, 1, 12, 30, 33);
dt2.add!"months"(-11);
assert(dt2 == DateTime(2009, 2, 1, 12, 30, 33));

auto dt3 = DateTime(2000, 2, 29, 12, 30, 33);
dt3.add!"years"(1);
assert(dt3 == DateTime(2001, 3, 1, 12, 30, 33));

auto dt4 = DateTime(2000, 2, 29, 12, 30, 33);
dt4.add!"years"(1, AllowDayOverflow.no);
assert(dt4 == DateTime(2001, 2, 28, 12, 30, 33));
--------------------
      +/
    /+ref DateTime+/ void add(string units)(long value, AllowDayOverflow allowOverflow = AllowDayOverflow.yes) pure nothrow
        if(units == "years" ||
           units == "months")
    {
        _date.add!units(value, allowOverflow);
    }

    //Verify Examples.
    unittest
    {
        version(testStdDateTime)
        {
            auto dt1 = DateTime(2010, 1, 1, 12, 30, 33);
            dt1.add!"months"(11);
            assert(dt1 == DateTime(2010, 12, 1, 12, 30, 33));

            auto dt2 = DateTime(2010, 1, 1, 12, 30, 33);
            dt2.add!"months"(-11);
            assert(dt2 == DateTime(2009, 2, 1, 12, 30, 33));

            auto dt3 = DateTime(2000, 2, 29, 12, 30, 33);
            dt3.add!"years"(1);
            assert(dt3 == DateTime(2001, 3, 1, 12, 30, 33));

            auto dt4 = DateTime(2000, 2, 29, 12, 30, 33);
            dt4.add!"years"(1, AllowDayOverflow.no);
            assert(dt4 == DateTime(2001, 2, 28, 12, 30, 33));
        }
    }

    unittest
    {
        version(testStdDateTime)
        {
            const cdt = DateTime(1999, 7, 6, 12, 30, 33);
            immutable idt = DateTime(1999, 7, 6, 12, 30, 33);
            static assert(!__traits(compiles, cdt.add!"years"(4)));
            static assert(!__traits(compiles, idt.add!"years"(4)));
            static assert(!__traits(compiles, cdt.add!"months"(4)));
            static assert(!__traits(compiles, idt.add!"months"(4)));
        }
    }


    /++
        Adds the given number of years or months to this $(D DateTime). A
        negative number will subtract.

        The difference between rolling and adding is that rolling does not
        affect larger units. Rolling a $(D DateTime) 12 months
        gets the exact same $(D DateTime). However, the days can still be
        affected due to the differing number of days in each month.

        Because there are no units larger than years, there is no difference
        between adding and rolling years.

        Params:
            units         = The type of units to add ("years" or "months").
            value         = The number of months or years to add to this
                            $(D DateTime).
            allowOverflow = Whether the days should be allowed to overflow,
                            causing the month to increment.

        Examples:
--------------------
auto dt1 = DateTime(2010, 1, 1, 12, 33, 33);
dt1.roll!"months"(1);
assert(dt1 == DateTime(2010, 2, 1, 12, 33, 33));

auto dt2 = DateTime(2010, 1, 1, 12, 33, 33);
dt2.roll!"months"(-1);
assert(dt2 == DateTime(2010, 12, 1, 12, 33, 33));

auto dt3 = DateTime(1999, 1, 29, 12, 33, 33);
dt3.roll!"months"(1);
assert(dt3 == DateTime(1999, 3, 1, 12, 33, 33));

auto dt4 = DateTime(1999, 1, 29, 12, 33, 33);
dt4.roll!"months"(1, AllowDayOverflow.no);
assert(dt4 == DateTime(1999, 2, 28, 12, 33, 33));

auto dt5 = DateTime(2000, 2, 29, 12, 30, 33);
dt5.roll!"years"(1);
assert(dt5 == DateTime(2001, 3, 1, 12, 30, 33));

auto dt6 = DateTime(2000, 2, 29, 12, 30, 33);
dt6.roll!"years"(1, AllowDayOverflow.no);
assert(dt6 == DateTime(2001, 2, 28, 12, 30, 33));
--------------------
      +/
    /+ref DateTime+/ void roll(string units)(long value, AllowDayOverflow allowOverflow = AllowDayOverflow.yes) pure nothrow
        if(units == "years" ||
           units == "months")
    {
        _date.roll!units(value, allowOverflow);
    }

    //Verify Examples.
    unittest
    {
        version(testdStdDateTime)
        {
            auto dt1 = DateTime(2010, 1, 1, 12, 33, 33);
            dt1.roll!"months"(1);
            assert(dt1 == DateTime(2010, 2, 1, 12, 33, 33));

            auto dt2 = DateTime(2010, 1, 1, 12, 33, 33);
            dt2.roll!"months"(-1);
            assert(dt2 == DateTime(2010, 12, 1, 12, 33, 33));

            auto dt3 = DateTime(1999, 1, 29, 12, 33, 33);
            dt3.roll!"months"(1);
            assert(dt3 == DateTime(1999, 3, 1, 12, 33, 33));

            auto dt4 = DateTime(1999, 1, 29, 12, 33, 33);
            dt4.roll!"months"(1, AllowDayOverflow.no);
            assert(dt4 == DateTime(1999, 2, 28, 12, 33, 33));

            auto dt5 = DateTime(2000, 2, 29, 12, 30, 33);
            dt5.roll!"years"(1);
            assert(dt5 == DateTime(2001, 3, 1, 12, 30, 33));

            auto dt6 = DateTime(2000, 2, 29, 12, 30, 33);
            dt6.roll!"years"(1, AllowDayOverflow.no);
            assert(dt6 == DateTime(2001, 2, 28, 12, 30, 33));
        }
    }

    unittest
    {
        version(testStdDateTime)
        {
            const cdt = DateTime(1999, 7, 6, 12, 30, 33);
            immutable idt = DateTime(1999, 7, 6, 12, 30, 33);
            static assert(!__traits(compiles, cdt.roll!"years"(4)));
            static assert(!__traits(compiles, idt.roll!"years"(4)));
            static assert(!__traits(compiles, cdt.roll!"months"(4)));
            static assert(!__traits(compiles, idt.roll!"months"(4)));
            static assert(!__traits(compiles, cdt.roll!"days"(4)));
            static assert(!__traits(compiles, idt.roll!"days"(4)));
        }
    }


    /++
        Adds the given number of units to this $(D DateTime). A negative number
        will subtract.

        The difference between rolling and adding is that rolling does not
        affect larger units. For instance, rolling a $(D DateTime) one
        year's worth of days gets the exact same $(D DateTime).

        Accepted units are $(D "days"), $(D "minutes"), $(D "hours"),
        $(D "minutes"), and $(D "seconds").

        Params:
            units = The units to add.
            value = The number of $(D_PARAM units) to add to this $(D DateTime).

        Examples:
--------------------
auto dt1 = DateTime(2010, 1, 1, 11, 23, 12);
dt1.roll!"days"(1);
assert(dt1 == DateTime(2010, 1, 2, 11, 23, 12));
dt1.roll!"days"(365);
assert(dt1 == DateTime(2010, 1, 26, 11, 23, 12));
dt1.roll!"days"(-32);
assert(dt1 == DateTime(2010, 1, 25, 11, 23, 12));

auto dt2 = DateTime(2010, 7, 4, 12, 0, 0);
dt2.roll!"hours"(1);
assert(dt2 == DateTime(2010, 7, 4, 13, 0, 0));

auto dt3 = DateTime(2010, 1, 1, 0, 0, 0);
dt3.roll!"seconds"(-1);
assert(dt3 == DateTime(2010, 1, 1, 0, 0, 59));
--------------------
      +/
    /+ref DateTime+/ void roll(string units)(long days) pure nothrow
        if(units == "days")
    {
        _date.roll!"days"(days);
    }

    //Verify Examples.
    unittest
    {
        version(testStdDateTime)
        {
            auto dt1 = DateTime(2010, 1, 1, 11, 23, 12);
            dt1.roll!"days"(1);
            assert(dt1 == DateTime(2010, 1, 2, 11, 23, 12));
            dt1.roll!"days"(365);
            assert(dt1 == DateTime(2010, 1, 26, 11, 23, 12));
            dt1.roll!"days"(-32);
            assert(dt1 == DateTime(2010, 1, 25, 11, 23, 12));

            auto dt2 = DateTime(2010, 7, 4, 12, 0, 0);
            dt2.roll!"hours"(1);
            assert(dt2 == DateTime(2010, 7, 4, 13, 0, 0));

            auto dt3 = DateTime(2010, 1, 1, 0, 0, 0);
            dt3.roll!"seconds"(-1);
            assert(dt3 == DateTime(2010, 1, 1, 0, 0, 59));
        }
    }

    unittest
    {
        version(testStdDateTime)
        {
            const cdt = DateTime(1999, 7, 6, 12, 30, 33);
            immutable idt = DateTime(1999, 7, 6, 12, 30, 33);
            static assert(!__traits(compiles, cdt.roll!"days"(4)));
            static assert(!__traits(compiles, idt.roll!"days"(4)));
        }
    }


    //Shares documentation with "days" version.
    /+ref DateTime+/ void roll(string units)(long value) pure nothrow
        if(units == "hours" ||
           units == "minutes" ||
           units == "seconds")
    {
        _tod.roll!units(value);
    }

    //Test roll!"hours"().
    unittest
    {
        version(testStdDateTime)
        {
            static void testDT(DateTime orig, int hours, in DateTime expected, size_t line = __LINE__)
            {
                orig.roll!"hours"(hours);
                _assertPred!"=="(orig, expected, "", __FILE__, line);
            }

            //Test A.D.
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 0, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 1, DateTime(Date(1999, 7, 6), TimeOfDay(13, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 2, DateTime(Date(1999, 7, 6), TimeOfDay(14, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 3, DateTime(Date(1999, 7, 6), TimeOfDay(15, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 4, DateTime(Date(1999, 7, 6), TimeOfDay(16, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 5, DateTime(Date(1999, 7, 6), TimeOfDay(17, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 6, DateTime(Date(1999, 7, 6), TimeOfDay(18, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 7, DateTime(Date(1999, 7, 6), TimeOfDay(19, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 8, DateTime(Date(1999, 7, 6), TimeOfDay(20, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 9, DateTime(Date(1999, 7, 6), TimeOfDay(21, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 10, DateTime(Date(1999, 7, 6), TimeOfDay(22, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 11, DateTime(Date(1999, 7, 6), TimeOfDay(23, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 12, DateTime(Date(1999, 7, 6), TimeOfDay(0, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 13, DateTime(Date(1999, 7, 6), TimeOfDay(1, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 14, DateTime(Date(1999, 7, 6), TimeOfDay(2, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 15, DateTime(Date(1999, 7, 6), TimeOfDay(3, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 16, DateTime(Date(1999, 7, 6), TimeOfDay(4, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 17, DateTime(Date(1999, 7, 6), TimeOfDay(5, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 18, DateTime(Date(1999, 7, 6), TimeOfDay(6, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 19, DateTime(Date(1999, 7, 6), TimeOfDay(7, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 20, DateTime(Date(1999, 7, 6), TimeOfDay(8, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 21, DateTime(Date(1999, 7, 6), TimeOfDay(9, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 22, DateTime(Date(1999, 7, 6), TimeOfDay(10, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 23, DateTime(Date(1999, 7, 6), TimeOfDay(11, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 24, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 25, DateTime(Date(1999, 7, 6), TimeOfDay(13, 30, 33)));

            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -1, DateTime(Date(1999, 7, 6), TimeOfDay(11, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -2, DateTime(Date(1999, 7, 6), TimeOfDay(10, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -3, DateTime(Date(1999, 7, 6), TimeOfDay(9, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -4, DateTime(Date(1999, 7, 6), TimeOfDay(8, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -5, DateTime(Date(1999, 7, 6), TimeOfDay(7, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -6, DateTime(Date(1999, 7, 6), TimeOfDay(6, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -7, DateTime(Date(1999, 7, 6), TimeOfDay(5, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -8, DateTime(Date(1999, 7, 6), TimeOfDay(4, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -9, DateTime(Date(1999, 7, 6), TimeOfDay(3, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -10, DateTime(Date(1999, 7, 6), TimeOfDay(2, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -11, DateTime(Date(1999, 7, 6), TimeOfDay(1, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -12, DateTime(Date(1999, 7, 6), TimeOfDay(0, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -13, DateTime(Date(1999, 7, 6), TimeOfDay(23, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -14, DateTime(Date(1999, 7, 6), TimeOfDay(22, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -15, DateTime(Date(1999, 7, 6), TimeOfDay(21, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -16, DateTime(Date(1999, 7, 6), TimeOfDay(20, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -17, DateTime(Date(1999, 7, 6), TimeOfDay(19, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -18, DateTime(Date(1999, 7, 6), TimeOfDay(18, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -19, DateTime(Date(1999, 7, 6), TimeOfDay(17, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -20, DateTime(Date(1999, 7, 6), TimeOfDay(16, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -21, DateTime(Date(1999, 7, 6), TimeOfDay(15, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -22, DateTime(Date(1999, 7, 6), TimeOfDay(14, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -23, DateTime(Date(1999, 7, 6), TimeOfDay(13, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -24, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -25, DateTime(Date(1999, 7, 6), TimeOfDay(11, 30, 33)));

            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(0, 30, 33)), 1, DateTime(Date(1999, 7, 6), TimeOfDay(1, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(0, 30, 33)), 0, DateTime(Date(1999, 7, 6), TimeOfDay(0, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(0, 30, 33)), -1, DateTime(Date(1999, 7, 6), TimeOfDay(23, 30, 33)));

            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(23, 30, 33)), 1, DateTime(Date(1999, 7, 6), TimeOfDay(0, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(23, 30, 33)), 0, DateTime(Date(1999, 7, 6), TimeOfDay(23, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(23, 30, 33)), -1, DateTime(Date(1999, 7, 6), TimeOfDay(22, 30, 33)));

            testDT(DateTime(Date(1999, 7, 31), TimeOfDay(23, 30, 33)), 1, DateTime(Date(1999, 7, 31), TimeOfDay(0, 30, 33)));
            testDT(DateTime(Date(1999, 8, 1), TimeOfDay(0, 30, 33)), -1, DateTime(Date(1999, 8, 1), TimeOfDay(23, 30, 33)));

            testDT(DateTime(Date(1999, 12, 31), TimeOfDay(23, 30, 33)), 1, DateTime(Date(1999, 12, 31), TimeOfDay(0, 30, 33)));
            testDT(DateTime(Date(2000, 1, 1), TimeOfDay(0, 30, 33)), -1, DateTime(Date(2000, 1, 1), TimeOfDay(23, 30, 33)));

            testDT(DateTime(Date(1999, 2, 28), TimeOfDay(23, 30, 33)), 25, DateTime(Date(1999, 2, 28), TimeOfDay(0, 30, 33)));
            testDT(DateTime(Date(1999, 3, 2), TimeOfDay(0, 30, 33)), -25, DateTime(Date(1999, 3, 2), TimeOfDay(23, 30, 33)));

            testDT(DateTime(Date(2000, 2, 28), TimeOfDay(23, 30, 33)), 25, DateTime(Date(2000, 2, 28), TimeOfDay(0, 30, 33)));
            testDT(DateTime(Date(2000, 3, 1), TimeOfDay(0, 30, 33)), -25, DateTime(Date(2000, 3, 1), TimeOfDay(23, 30, 33)));

            //Test B.C.
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 0, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 1, DateTime(Date(-1999, 7, 6), TimeOfDay(13, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 2, DateTime(Date(-1999, 7, 6), TimeOfDay(14, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 3, DateTime(Date(-1999, 7, 6), TimeOfDay(15, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 4, DateTime(Date(-1999, 7, 6), TimeOfDay(16, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 5, DateTime(Date(-1999, 7, 6), TimeOfDay(17, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 6, DateTime(Date(-1999, 7, 6), TimeOfDay(18, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 7, DateTime(Date(-1999, 7, 6), TimeOfDay(19, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 8, DateTime(Date(-1999, 7, 6), TimeOfDay(20, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 9, DateTime(Date(-1999, 7, 6), TimeOfDay(21, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 10, DateTime(Date(-1999, 7, 6), TimeOfDay(22, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 11, DateTime(Date(-1999, 7, 6), TimeOfDay(23, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 12, DateTime(Date(-1999, 7, 6), TimeOfDay(0, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 13, DateTime(Date(-1999, 7, 6), TimeOfDay(1, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 14, DateTime(Date(-1999, 7, 6), TimeOfDay(2, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 15, DateTime(Date(-1999, 7, 6), TimeOfDay(3, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 16, DateTime(Date(-1999, 7, 6), TimeOfDay(4, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 17, DateTime(Date(-1999, 7, 6), TimeOfDay(5, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 18, DateTime(Date(-1999, 7, 6), TimeOfDay(6, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 19, DateTime(Date(-1999, 7, 6), TimeOfDay(7, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 20, DateTime(Date(-1999, 7, 6), TimeOfDay(8, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 21, DateTime(Date(-1999, 7, 6), TimeOfDay(9, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 22, DateTime(Date(-1999, 7, 6), TimeOfDay(10, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 23, DateTime(Date(-1999, 7, 6), TimeOfDay(11, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 24, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 25, DateTime(Date(-1999, 7, 6), TimeOfDay(13, 30, 33)));

            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -1, DateTime(Date(-1999, 7, 6), TimeOfDay(11, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -2, DateTime(Date(-1999, 7, 6), TimeOfDay(10, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -3, DateTime(Date(-1999, 7, 6), TimeOfDay(9, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -4, DateTime(Date(-1999, 7, 6), TimeOfDay(8, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -5, DateTime(Date(-1999, 7, 6), TimeOfDay(7, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -6, DateTime(Date(-1999, 7, 6), TimeOfDay(6, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -7, DateTime(Date(-1999, 7, 6), TimeOfDay(5, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -8, DateTime(Date(-1999, 7, 6), TimeOfDay(4, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -9, DateTime(Date(-1999, 7, 6), TimeOfDay(3, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -10, DateTime(Date(-1999, 7, 6), TimeOfDay(2, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -11, DateTime(Date(-1999, 7, 6), TimeOfDay(1, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -12, DateTime(Date(-1999, 7, 6), TimeOfDay(0, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -13, DateTime(Date(-1999, 7, 6), TimeOfDay(23, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -14, DateTime(Date(-1999, 7, 6), TimeOfDay(22, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -15, DateTime(Date(-1999, 7, 6), TimeOfDay(21, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -16, DateTime(Date(-1999, 7, 6), TimeOfDay(20, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -17, DateTime(Date(-1999, 7, 6), TimeOfDay(19, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -18, DateTime(Date(-1999, 7, 6), TimeOfDay(18, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -19, DateTime(Date(-1999, 7, 6), TimeOfDay(17, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -20, DateTime(Date(-1999, 7, 6), TimeOfDay(16, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -21, DateTime(Date(-1999, 7, 6), TimeOfDay(15, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -22, DateTime(Date(-1999, 7, 6), TimeOfDay(14, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -23, DateTime(Date(-1999, 7, 6), TimeOfDay(13, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -24, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -25, DateTime(Date(-1999, 7, 6), TimeOfDay(11, 30, 33)));

            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(0, 30, 33)), 1, DateTime(Date(-1999, 7, 6), TimeOfDay(1, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(0, 30, 33)), 0, DateTime(Date(-1999, 7, 6), TimeOfDay(0, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(0, 30, 33)), -1, DateTime(Date(-1999, 7, 6), TimeOfDay(23, 30, 33)));

            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(23, 30, 33)), 1, DateTime(Date(-1999, 7, 6), TimeOfDay(0, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(23, 30, 33)), 0, DateTime(Date(-1999, 7, 6), TimeOfDay(23, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(23, 30, 33)), -1, DateTime(Date(-1999, 7, 6), TimeOfDay(22, 30, 33)));

            testDT(DateTime(Date(-1999, 7, 31), TimeOfDay(23, 30, 33)), 1, DateTime(Date(-1999, 7, 31), TimeOfDay(0, 30, 33)));
            testDT(DateTime(Date(-1999, 8, 1), TimeOfDay(0, 30, 33)), -1, DateTime(Date(-1999, 8, 1), TimeOfDay(23, 30, 33)));

            testDT(DateTime(Date(-2001, 12, 31), TimeOfDay(23, 30, 33)), 1, DateTime(Date(-2001, 12, 31), TimeOfDay(0, 30, 33)));
            testDT(DateTime(Date(-2000, 1, 1), TimeOfDay(0, 30, 33)), -1, DateTime(Date(-2000, 1, 1), TimeOfDay(23, 30, 33)));

            testDT(DateTime(Date(-2001, 2, 28), TimeOfDay(23, 30, 33)), 25, DateTime(Date(-2001, 2, 28), TimeOfDay(0, 30, 33)));
            testDT(DateTime(Date(-2001, 3, 2), TimeOfDay(0, 30, 33)), -25, DateTime(Date(-2001, 3, 2), TimeOfDay(23, 30, 33)));

            testDT(DateTime(Date(-2000, 2, 28), TimeOfDay(23, 30, 33)), 25, DateTime(Date(-2000, 2, 28), TimeOfDay(0, 30, 33)));
            testDT(DateTime(Date(-2000, 3, 1), TimeOfDay(0, 30, 33)), -25, DateTime(Date(-2000, 3, 1), TimeOfDay(23, 30, 33)));

            //Test Both
            testDT(DateTime(Date(-1, 1, 1), TimeOfDay(11, 30, 33)), 17_546, DateTime(Date(-1, 1, 1), TimeOfDay(13, 30, 33)));
            testDT(DateTime(Date(1, 1, 1), TimeOfDay(13, 30, 33)), -17_546, DateTime(Date(1, 1, 1), TimeOfDay(11, 30, 33)));

            const cdt = DateTime(1999, 7, 6, 12, 30, 33);
            immutable idt = DateTime(1999, 7, 6, 12, 30, 33);
            static assert(!__traits(compiles, cdt.roll!"hours"(4)));
            static assert(!__traits(compiles, idt.roll!"hours"(4)));

            //Verify Examples.
            auto dt1 = DateTime(Date(2010, 7, 4), TimeOfDay(12, 0, 0));
            dt1.roll!"hours"(1);
            assert(dt1 == DateTime(Date(2010, 7, 4), TimeOfDay(13, 0, 0)));

            auto dt2 = DateTime(Date(2010, 2, 12), TimeOfDay(12, 0, 0));
            dt2.roll!"hours"(-1);
            assert(dt2 == DateTime(Date(2010, 2, 12), TimeOfDay(11, 0, 0)));

            auto dt3 = DateTime(Date(2009, 12, 31), TimeOfDay(23, 0, 0));
            dt3.roll!"hours"(1);
            assert(dt3 == DateTime(Date(2009, 12, 31), TimeOfDay(0, 0, 0)));

            auto dt4 = DateTime(Date(2010, 1, 1), TimeOfDay(0, 0, 0));
            dt4.roll!"hours"(-1);
            assert(dt4 == DateTime(Date(2010, 1, 1), TimeOfDay(23, 0, 0)));
        }
    }

    //Test roll!"minutes"().
    unittest
    {
        version(testStdDateTime)
        {
            static void testDT(DateTime orig, int minutes, in DateTime expected, size_t line = __LINE__)
            {
                orig.roll!"minutes"(minutes);
                _assertPred!"=="(orig, expected, "", __FILE__, line);
            }

            //Test A.D.
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 0, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 1, DateTime(Date(1999, 7, 6), TimeOfDay(12, 31, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 2, DateTime(Date(1999, 7, 6), TimeOfDay(12, 32, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 3, DateTime(Date(1999, 7, 6), TimeOfDay(12, 33, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 4, DateTime(Date(1999, 7, 6), TimeOfDay(12, 34, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 5, DateTime(Date(1999, 7, 6), TimeOfDay(12, 35, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 10, DateTime(Date(1999, 7, 6), TimeOfDay(12, 40, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 15, DateTime(Date(1999, 7, 6), TimeOfDay(12, 45, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 29, DateTime(Date(1999, 7, 6), TimeOfDay(12, 59, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 30, DateTime(Date(1999, 7, 6), TimeOfDay(12, 0, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 45, DateTime(Date(1999, 7, 6), TimeOfDay(12, 15, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 60, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 75, DateTime(Date(1999, 7, 6), TimeOfDay(12, 45, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 90, DateTime(Date(1999, 7, 6), TimeOfDay(12, 0, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 100, DateTime(Date(1999, 7, 6), TimeOfDay(12, 10, 33)));

            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 689, DateTime(Date(1999, 7, 6), TimeOfDay(12, 59, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 690, DateTime(Date(1999, 7, 6), TimeOfDay(12, 0, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 691, DateTime(Date(1999, 7, 6), TimeOfDay(12, 1, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 960, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 1439, DateTime(Date(1999, 7, 6), TimeOfDay(12, 29, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 1440, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 1441, DateTime(Date(1999, 7, 6), TimeOfDay(12, 31, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 2880, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));

            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -1, DateTime(Date(1999, 7, 6), TimeOfDay(12, 29, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -2, DateTime(Date(1999, 7, 6), TimeOfDay(12, 28, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -3, DateTime(Date(1999, 7, 6), TimeOfDay(12, 27, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -4, DateTime(Date(1999, 7, 6), TimeOfDay(12, 26, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -5, DateTime(Date(1999, 7, 6), TimeOfDay(12, 25, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -10, DateTime(Date(1999, 7, 6), TimeOfDay(12, 20, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -15, DateTime(Date(1999, 7, 6), TimeOfDay(12, 15, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -29, DateTime(Date(1999, 7, 6), TimeOfDay(12, 1, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -30, DateTime(Date(1999, 7, 6), TimeOfDay(12, 0, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -45, DateTime(Date(1999, 7, 6), TimeOfDay(12, 45, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -60, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -75, DateTime(Date(1999, 7, 6), TimeOfDay(12, 15, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -90, DateTime(Date(1999, 7, 6), TimeOfDay(12, 0, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -100, DateTime(Date(1999, 7, 6), TimeOfDay(12, 50, 33)));

            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -749, DateTime(Date(1999, 7, 6), TimeOfDay(12, 1, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -750, DateTime(Date(1999, 7, 6), TimeOfDay(12, 0, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -751, DateTime(Date(1999, 7, 6), TimeOfDay(12, 59, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -960, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -1439, DateTime(Date(1999, 7, 6), TimeOfDay(12, 31, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -1440, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -1441, DateTime(Date(1999, 7, 6), TimeOfDay(12, 29, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -2880, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));

            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 0, 33)), 1, DateTime(Date(1999, 7, 6), TimeOfDay(12, 1, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 0, 33)), 0, DateTime(Date(1999, 7, 6), TimeOfDay(12, 0, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 0, 33)), -1, DateTime(Date(1999, 7, 6), TimeOfDay(12, 59, 33)));

            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(11, 59, 33)), 1, DateTime(Date(1999, 7, 6), TimeOfDay(11, 0, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(11, 59, 33)), 0, DateTime(Date(1999, 7, 6), TimeOfDay(11, 59, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(11, 59, 33)), -1, DateTime(Date(1999, 7, 6), TimeOfDay(11, 58, 33)));

            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(0, 0, 33)), 1, DateTime(Date(1999, 7, 6), TimeOfDay(0, 1, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(0, 0, 33)), 0, DateTime(Date(1999, 7, 6), TimeOfDay(0, 0, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(0, 0, 33)), -1, DateTime(Date(1999, 7, 6), TimeOfDay(0, 59, 33)));

            testDT(DateTime(Date(1999, 7, 5), TimeOfDay(23, 59, 33)), 1, DateTime(Date(1999, 7, 5), TimeOfDay(23, 0, 33)));
            testDT(DateTime(Date(1999, 7, 5), TimeOfDay(23, 59, 33)), 0, DateTime(Date(1999, 7, 5), TimeOfDay(23, 59, 33)));
            testDT(DateTime(Date(1999, 7, 5), TimeOfDay(23, 59, 33)), -1, DateTime(Date(1999, 7, 5), TimeOfDay(23, 58, 33)));

            testDT(DateTime(Date(1998, 12, 31), TimeOfDay(23, 59, 33)), 1, DateTime(Date(1998, 12, 31), TimeOfDay(23, 0, 33)));
            testDT(DateTime(Date(1998, 12, 31), TimeOfDay(23, 59, 33)), 0, DateTime(Date(1998, 12, 31), TimeOfDay(23, 59, 33)));
            testDT(DateTime(Date(1998, 12, 31), TimeOfDay(23, 59, 33)), -1, DateTime(Date(1998, 12, 31), TimeOfDay(23, 58, 33)));

            //Test B.C.
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 0, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 1, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 31, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 2, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 32, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 3, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 33, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 4, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 34, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 5, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 35, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 10, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 40, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 15, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 45, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 29, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 59, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 30, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 0, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 45, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 15, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 60, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 75, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 45, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 90, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 0, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 100, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 10, 33)));

            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 689, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 59, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 690, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 0, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 691, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 1, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 960, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 1439, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 29, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 1440, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 1441, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 31, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 2880, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)));

            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -1, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 29, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -2, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 28, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -3, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 27, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -4, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 26, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -5, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 25, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -10, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 20, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -15, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 15, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -29, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 1, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -30, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 0, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -45, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 45, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -60, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -75, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 15, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -90, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 0, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -100, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 50, 33)));

            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -749, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 1, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -750, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 0, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -751, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 59, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -960, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -1439, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 31, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -1440, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -1441, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 29, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -2880, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)));

            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 0, 33)), 1, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 1, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 0, 33)), 0, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 0, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 0, 33)), -1, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 59, 33)));

            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(11, 59, 33)), 1, DateTime(Date(-1999, 7, 6), TimeOfDay(11, 0, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(11, 59, 33)), 0, DateTime(Date(-1999, 7, 6), TimeOfDay(11, 59, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(11, 59, 33)), -1, DateTime(Date(-1999, 7, 6), TimeOfDay(11, 58, 33)));

            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(0, 0, 33)), 1, DateTime(Date(-1999, 7, 6), TimeOfDay(0, 1, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(0, 0, 33)), 0, DateTime(Date(-1999, 7, 6), TimeOfDay(0, 0, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(0, 0, 33)), -1, DateTime(Date(-1999, 7, 6), TimeOfDay(0, 59, 33)));

            testDT(DateTime(Date(-1999, 7, 5), TimeOfDay(23, 59, 33)), 1, DateTime(Date(-1999, 7, 5), TimeOfDay(23, 0, 33)));
            testDT(DateTime(Date(-1999, 7, 5), TimeOfDay(23, 59, 33)), 0, DateTime(Date(-1999, 7, 5), TimeOfDay(23, 59, 33)));
            testDT(DateTime(Date(-1999, 7, 5), TimeOfDay(23, 59, 33)), -1, DateTime(Date(-1999, 7, 5), TimeOfDay(23, 58, 33)));

            testDT(DateTime(Date(-2000, 12, 31), TimeOfDay(23, 59, 33)), 1, DateTime(Date(-2000, 12, 31), TimeOfDay(23, 0, 33)));
            testDT(DateTime(Date(-2000, 12, 31), TimeOfDay(23, 59, 33)), 0, DateTime(Date(-2000, 12, 31), TimeOfDay(23, 59, 33)));
            testDT(DateTime(Date(-2000, 12, 31), TimeOfDay(23, 59, 33)), -1, DateTime(Date(-2000, 12, 31), TimeOfDay(23, 58, 33)));

            //Test Both
            testDT(DateTime(Date(1, 1, 1), TimeOfDay(0, 0, 0)), -1, DateTime(Date(1, 1, 1), TimeOfDay(0, 59, 0)));
            testDT(DateTime(Date(0, 12, 31), TimeOfDay(23, 59, 0)), 1, DateTime(Date(0, 12, 31), TimeOfDay(23, 0, 0)));

            testDT(DateTime(Date(0, 1, 1), TimeOfDay(0, 0, 0)), -1, DateTime(Date(0, 1, 1), TimeOfDay(0, 59, 0)));
            testDT(DateTime(Date(-1, 12, 31), TimeOfDay(23, 59, 0)), 1, DateTime(Date(-1, 12, 31), TimeOfDay(23, 0, 0)));

            testDT(DateTime(Date(-1, 1, 1), TimeOfDay(11, 30, 33)), 1_052_760, DateTime(Date(-1, 1, 1), TimeOfDay(11, 30, 33)));
            testDT(DateTime(Date(1, 1, 1), TimeOfDay(13, 30, 33)), -1_052_760, DateTime(Date(1, 1, 1), TimeOfDay(13, 30, 33)));

            testDT(DateTime(Date(-1, 1, 1), TimeOfDay(11, 30, 33)), 1_052_782, DateTime(Date(-1, 1, 1), TimeOfDay(11, 52, 33)));
            testDT(DateTime(Date(1, 1, 1), TimeOfDay(13, 52, 33)), -1_052_782, DateTime(Date(1, 1, 1), TimeOfDay(13, 30, 33)));

            const cdt = DateTime(1999, 7, 6, 12, 30, 33);
            immutable idt = DateTime(1999, 7, 6, 12, 30, 33);
            static assert(!__traits(compiles, cdt.roll!"minutes"(4)));
            static assert(!__traits(compiles, idt.roll!"minutes"(4)));
        }
    }

    //Test roll!"seconds"().
    unittest
    {
        version(testStdDateTime)
        {
            static void testDT(DateTime orig, int seconds, in DateTime expected, size_t line = __LINE__)
            {
                orig.roll!"seconds"(seconds);
                _assertPred!"=="(orig, expected, "", __FILE__, line);
            }

            //Test A.D.
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 0, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 1, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 34)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 2, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 35)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 3, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 36)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 4, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 37)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 5, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 38)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 10, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 43)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 15, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 48)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 26, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 59)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 27, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 0)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 30, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 3)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 59, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 32)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 60, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 61, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 34)));

            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 1766, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 59)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 1767, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 0)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 1768, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 1)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 2007, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 0)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 3599, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 32)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 3600, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 3601, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 34)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), 7200, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));

            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -1, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 32)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -2, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 31)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -3, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 30)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -4, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 29)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -5, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 28)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -10, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 23)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -15, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 18)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -33, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 0)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -34, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 59)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -35, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 58)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -59, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 34)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -60, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), -61, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 32)));

            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 0)), 1, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 1)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 0)), 0, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 0)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 0)), -1, DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 59)));

            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 0, 0)), 1, DateTime(Date(1999, 7, 6), TimeOfDay(12, 0, 1)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 0, 0)), 0, DateTime(Date(1999, 7, 6), TimeOfDay(12, 0, 0)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(12, 0, 0)), -1, DateTime(Date(1999, 7, 6), TimeOfDay(12, 0, 59)));

            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(0, 0, 0)), 1, DateTime(Date(1999, 7, 6), TimeOfDay(0, 0, 1)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(0, 0, 0)), 0, DateTime(Date(1999, 7, 6), TimeOfDay(0, 0, 0)));
            testDT(DateTime(Date(1999, 7, 6), TimeOfDay(0, 0, 0)), -1, DateTime(Date(1999, 7, 6), TimeOfDay(0, 0, 59)));

            testDT(DateTime(Date(1999, 7, 5), TimeOfDay(23, 59, 59)), 1, DateTime(Date(1999, 7, 5), TimeOfDay(23, 59, 0)));
            testDT(DateTime(Date(1999, 7, 5), TimeOfDay(23, 59, 59)), 0, DateTime(Date(1999, 7, 5), TimeOfDay(23, 59, 59)));
            testDT(DateTime(Date(1999, 7, 5), TimeOfDay(23, 59, 59)), -1, DateTime(Date(1999, 7, 5), TimeOfDay(23, 59, 58)));

            testDT(DateTime(Date(1998, 12, 31), TimeOfDay(23, 59, 59)), 1, DateTime(Date(1998, 12, 31), TimeOfDay(23, 59, 0)));
            testDT(DateTime(Date(1998, 12, 31), TimeOfDay(23, 59, 59)), 0, DateTime(Date(1998, 12, 31), TimeOfDay(23, 59, 59)));
            testDT(DateTime(Date(1998, 12, 31), TimeOfDay(23, 59, 59)), -1, DateTime(Date(1998, 12, 31), TimeOfDay(23, 59, 58)));

            //Test B.C.
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 0, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 1, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 34)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 2, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 35)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 3, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 36)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 4, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 37)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 5, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 38)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 10, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 43)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 15, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 48)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 26, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 59)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 27, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 0)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 30, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 3)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 59, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 32)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 60, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 61, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 34)));

            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 1766, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 59)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 1767, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 0)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 1768, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 1)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 2007, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 0)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 3599, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 32)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 3600, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 3601, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 34)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), 7200, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)));

            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -1, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 32)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -2, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 31)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -3, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 30)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -4, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 29)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -5, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 28)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -10, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 23)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -15, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 18)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -33, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 0)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -34, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 59)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -35, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 58)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -59, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 34)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -60, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)), -61, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 32)));

            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 0)), 1, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 1)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 0)), 0, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 0)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 0)), -1, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 59)));

            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 0, 0)), 1, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 0, 1)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 0, 0)), 0, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 0, 0)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(12, 0, 0)), -1, DateTime(Date(-1999, 7, 6), TimeOfDay(12, 0, 59)));

            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(0, 0, 0)), 1, DateTime(Date(-1999, 7, 6), TimeOfDay(0, 0, 1)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(0, 0, 0)), 0, DateTime(Date(-1999, 7, 6), TimeOfDay(0, 0, 0)));
            testDT(DateTime(Date(-1999, 7, 6), TimeOfDay(0, 0, 0)), -1, DateTime(Date(-1999, 7, 6), TimeOfDay(0, 0, 59)));

            testDT(DateTime(Date(-1999, 7, 5), TimeOfDay(23, 59, 59)), 1, DateTime(Date(-1999, 7, 5), TimeOfDay(23, 59, 0)));
            testDT(DateTime(Date(-1999, 7, 5), TimeOfDay(23, 59, 59)), 0, DateTime(Date(-1999, 7, 5), TimeOfDay(23, 59, 59)));
            testDT(DateTime(Date(-1999, 7, 5), TimeOfDay(23, 59, 59)), -1, DateTime(Date(-1999, 7, 5), TimeOfDay(23, 59, 58)));

            testDT(DateTime(Date(-2000, 12, 31), TimeOfDay(23, 59, 59)), 1, DateTime(Date(-2000, 12, 31), TimeOfDay(23, 59, 0)));
            testDT(DateTime(Date(-2000, 12, 31), TimeOfDay(23, 59, 59)), 0, DateTime(Date(-2000, 12, 31), TimeOfDay(23, 59, 59)));
            testDT(DateTime(Date(-2000, 12, 31), TimeOfDay(23, 59, 59)), -1, DateTime(Date(-2000, 12, 31), TimeOfDay(23, 59, 58)));

            //Test Both
            testDT(DateTime(Date(1, 1, 1), TimeOfDay(0, 0, 0)), -1, DateTime(Date(1, 1, 1), TimeOfDay(0, 0, 59)));
            testDT(DateTime(Date(0, 12, 31), TimeOfDay(23, 59, 59)), 1, DateTime(Date(0, 12, 31), TimeOfDay(23, 59, 0)));

            testDT(DateTime(Date(0, 1, 1), TimeOfDay(0, 0, 0)), -1, DateTime(Date(0, 1, 1), TimeOfDay(0, 0, 59)));
            testDT(DateTime(Date(-1, 12, 31), TimeOfDay(23, 59, 59)), 1, DateTime(Date(-1, 12, 31), TimeOfDay(23, 59, 0)));

            testDT(DateTime(Date(-1, 1, 1), TimeOfDay(11, 30, 33)), 63_165_600L, DateTime(Date(-1, 1, 1), TimeOfDay(11, 30, 33)));
            testDT(DateTime(Date(1, 1, 1), TimeOfDay(13, 30, 33)), -63_165_600L, DateTime(Date(1, 1, 1), TimeOfDay(13, 30, 33)));

            testDT(DateTime(Date(-1, 1, 1), TimeOfDay(11, 30, 33)), 63_165_617L, DateTime(Date(-1, 1, 1), TimeOfDay(11, 30, 50)));
            testDT(DateTime(Date(1, 1, 1), TimeOfDay(13, 30, 50)), -63_165_617L, DateTime(Date(1, 1, 1), TimeOfDay(13, 30, 33)));

            const cdt = DateTime(1999, 7, 6, 12, 30, 33);
            immutable idt = DateTime(1999, 7, 6, 12, 30, 33);
            static assert(!__traits(compiles, cdt.roll!"seconds"(4)));
            static assert(!__traits(compiles, idt.roll!"seconds"(4)));
        }
    }


    /++
        Gives the result of adding or subtracting a duration from this
        $(D DateTime).

        The legal types of arithmetic for $(D DateTime) using this operator are

        $(BOOKTABLE,
        $(TR $(TD DateTime) $(TD +) $(TD duration) $(TD -->) $(TD DateTime))
        $(TR $(TD DateTime) $(TD -) $(TD duration) $(TD -->) $(TD DateTime))
        )

        Params:
            duration = The duration to add to or subtract from this
                       $(D DateTime).
      +/
    DateTime opBinary(string op, D)(in D duration) const pure nothrow
        if((op == "+" || op == "-") &&
           (is(Unqual!D == Duration) ||
            is(Unqual!D == TickDuration)))
    {
        DateTime retval = this;

        static if(is(Unqual!D == Duration))
            immutable hnsecs = duration.total!"hnsecs";
        else static if(is(Unqual!D == TickDuration))
            immutable hnsecs = duration.hnsecs;

        //Ideally, this would just be
        //return retval.addSeconds(convert!("hnsecs", "seconds")(unaryFun!(op ~ "a")(hnsecs)));
        //But there isn't currently a pure version of unaryFun!().

        static if(op == "+")
            immutable signedHNSecs = hnsecs;
        else static if(op == "-")
            immutable signedHNSecs = -hnsecs;
        else
            static assert(0);

        return retval.addSeconds(convert!("hnsecs", "seconds")(signedHNSecs));
    }

    unittest
    {
        version(testStdDateTime)
        {
            auto dt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));

            _assertPred!"=="(dt + dur!"weeks"(7), DateTime(Date(1999, 8, 24), TimeOfDay(12, 30, 33)));
            _assertPred!"=="(dt + dur!"weeks"(-7), DateTime(Date(1999, 5, 18), TimeOfDay(12, 30, 33)));
            _assertPred!"=="(dt + dur!"days"(7), DateTime(Date(1999, 7, 13), TimeOfDay(12, 30, 33)));
            _assertPred!"=="(dt + dur!"days"(-7), DateTime(Date(1999, 6, 29), TimeOfDay(12, 30, 33)));

            _assertPred!"=="(dt + dur!"hours"(7), DateTime(Date(1999, 7, 6), TimeOfDay(19, 30, 33)));
            _assertPred!"=="(dt + dur!"hours"(-7), DateTime(Date(1999, 7, 6), TimeOfDay(5, 30, 33)));
            _assertPred!"=="(dt + dur!"minutes"(7), DateTime(Date(1999, 7, 6), TimeOfDay(12, 37, 33)));
            _assertPred!"=="(dt + dur!"minutes"(-7), DateTime(Date(1999, 7, 6), TimeOfDay(12, 23, 33)));
            _assertPred!"=="(dt + dur!"seconds"(7), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 40)));
            _assertPred!"=="(dt + dur!"seconds"(-7), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 26)));
            _assertPred!"=="(dt + dur!"msecs"(7_000), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 40)));
            _assertPred!"=="(dt + dur!"msecs"(-7_000), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 26)));
            _assertPred!"=="(dt + dur!"usecs"(7_000_000), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 40)));
            _assertPred!"=="(dt + dur!"usecs"(-7_000_000), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 26)));
            _assertPred!"=="(dt + dur!"hnsecs"(70_000_000), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 40)));
            _assertPred!"=="(dt + dur!"hnsecs"(-70_000_000), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 26)));

            //This probably only runs in cases where gettimeofday() is used, but it's
            //hard to do this test correctly with variable ticksPerSec.
            if(TickDuration.ticksPerSec == 1_000_000)
            {
                _assertPred!"=="(dt + TickDuration.from!"usecs"(7_000_000), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 40)));
                _assertPred!"=="(dt + TickDuration.from!"usecs"(-7_000_000), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 26)));
            }

            _assertPred!"=="(dt - dur!"weeks"(-7), DateTime(Date(1999, 8, 24), TimeOfDay(12, 30, 33)));
            _assertPred!"=="(dt - dur!"weeks"(7), DateTime(Date(1999, 5, 18), TimeOfDay(12, 30, 33)));
            _assertPred!"=="(dt - dur!"days"(-7), DateTime(Date(1999, 7, 13), TimeOfDay(12, 30, 33)));
            _assertPred!"=="(dt - dur!"days"(7), DateTime(Date(1999, 6, 29), TimeOfDay(12, 30, 33)));

            _assertPred!"=="(dt - dur!"hours"(-7), DateTime(Date(1999, 7, 6), TimeOfDay(19, 30, 33)));
            _assertPred!"=="(dt - dur!"hours"(7), DateTime(Date(1999, 7, 6), TimeOfDay(5, 30, 33)));
            _assertPred!"=="(dt - dur!"minutes"(-7), DateTime(Date(1999, 7, 6), TimeOfDay(12, 37, 33)));
            _assertPred!"=="(dt - dur!"minutes"(7), DateTime(Date(1999, 7, 6), TimeOfDay(12, 23, 33)));
            _assertPred!"=="(dt - dur!"seconds"(-7), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 40)));
            _assertPred!"=="(dt - dur!"seconds"(7), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 26)));
            _assertPred!"=="(dt - dur!"msecs"(-7_000), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 40)));
            _assertPred!"=="(dt - dur!"msecs"(7_000), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 26)));
            _assertPred!"=="(dt - dur!"usecs"(-7_000_000), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 40)));
            _assertPred!"=="(dt - dur!"usecs"(7_000_000), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 26)));
            _assertPred!"=="(dt - dur!"hnsecs"(-70_000_000), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 40)));
            _assertPred!"=="(dt - dur!"hnsecs"(70_000_000), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 26)));

            //This probably only runs in cases where gettimeofday() is used, but it's
            //hard to do this test correctly with variable ticksPerSec.
            if(TickDuration.ticksPerSec == 1_000_000)
            {
                _assertPred!"=="(dt - TickDuration.from!"usecs"(-7_000_000), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 40)));
                _assertPred!"=="(dt - TickDuration.from!"usecs"(7_000_000), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 26)));
            }

            auto duration = dur!"seconds"(12);
            const cdt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            immutable idt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            static assert(__traits(compiles, cdt + duration));
            static assert(__traits(compiles, idt + duration));
            static assert(__traits(compiles, cdt - duration));
            static assert(__traits(compiles, idt - duration));
        }
    }


    /++
        Gives the result of adding or subtracting a duration from this
        $(D DateTime), as well as assigning the result to this $(D DateTime).

        The legal types of arithmetic for $(D DateTime) using this operator are

        $(BOOKTABLE,
        $(TR $(TD DateTime) $(TD +) $(TD duration) $(TD -->) $(TD DateTime))
        $(TR $(TD DateTime) $(TD -) $(TD duration) $(TD -->) $(TD DateTime))
        )

        Params:
            duration = The duration to add to or subtract from this
                       $(D DateTime).
      +/
    /+ref+/ DateTime opOpAssign(string op, D)(in D duration) pure nothrow
        if((op == "+" || op == "-") &&
           (is(Unqual!D == Duration) ||
            is(Unqual!D == TickDuration)))
    {
        DateTime retval = this;

        static if(is(Unqual!D == Duration))
            immutable hnsecs = duration.total!"hnsecs";
        else static if(is(Unqual!D == TickDuration))
            immutable hnsecs = duration.hnsecs;

        //Ideally, this would just be
       //return addSeconds(convert!("hnsecs", "seconds")(unaryFun!(op ~ "a")(hnsecs)));
        //But there isn't currently a pure version of unaryFun!().

        static if(op == "+")
            immutable signedHNSecs = hnsecs;
        else static if(op == "-")
            immutable signedHNSecs = -hnsecs;
        else
            static assert(0);

        return addSeconds(convert!("hnsecs", "seconds")(signedHNSecs));
    }

    unittest
    {
        version(testStdDateTime)
        {
            _assertPred!"+="(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), dur!"weeks"(7), DateTime(Date(1999, 8, 24), TimeOfDay(12, 30, 33)));
            _assertPred!"+="(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), dur!"weeks"(-7), DateTime(Date(1999, 5, 18), TimeOfDay(12, 30, 33)));
            _assertPred!"+="(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), dur!"days"(7), DateTime(Date(1999, 7, 13), TimeOfDay(12, 30, 33)));
            _assertPred!"+="(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), dur!"days"(-7), DateTime(Date(1999, 6, 29), TimeOfDay(12, 30, 33)));

            _assertPred!"+="(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), dur!"hours"(7), DateTime(Date(1999, 7, 6), TimeOfDay(19, 30, 33)));
            _assertPred!"+="(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), dur!"hours"(-7), DateTime(Date(1999, 7, 6), TimeOfDay(5, 30, 33)));
            _assertPred!"+="(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), dur!"minutes"(7), DateTime(Date(1999, 7, 6), TimeOfDay(12, 37, 33)));
            _assertPred!"+="(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), dur!"minutes"(-7), DateTime(Date(1999, 7, 6), TimeOfDay(12, 23, 33)));
            _assertPred!"+="(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), dur!"seconds"(7), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 40)));
            _assertPred!"+="(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), dur!"seconds"(-7), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 26)));
            _assertPred!"+="(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), dur!"msecs"(7_000), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 40)));
            _assertPred!"+="(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), dur!"msecs"(-7_000), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 26)));
            _assertPred!"+="(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), dur!"usecs"(7_000_000), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 40)));
            _assertPred!"+="(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), dur!"usecs"(-7_000_000), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 26)));
            _assertPred!"+="(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), dur!"hnsecs"(70_000_000), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 40)));
            _assertPred!"+="(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), dur!"hnsecs"(-70_000_000), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 26)));

            _assertPred!"-="(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), dur!"weeks"(-7), DateTime(Date(1999, 8, 24), TimeOfDay(12, 30, 33)));
            _assertPred!"-="(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), dur!"weeks"(7), DateTime(Date(1999, 5, 18), TimeOfDay(12, 30, 33)));
            _assertPred!"-="(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), dur!"days"(-7), DateTime(Date(1999, 7, 13), TimeOfDay(12, 30, 33)));
            _assertPred!"-="(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), dur!"days"(7), DateTime(Date(1999, 6, 29), TimeOfDay(12, 30, 33)));

            _assertPred!"-="(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), dur!"hours"(-7), DateTime(Date(1999, 7, 6), TimeOfDay(19, 30, 33)));
            _assertPred!"-="(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), dur!"hours"(7), DateTime(Date(1999, 7, 6), TimeOfDay(5, 30, 33)));
            _assertPred!"-="(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), dur!"minutes"(-7), DateTime(Date(1999, 7, 6), TimeOfDay(12, 37, 33)));
            _assertPred!"-="(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), dur!"minutes"(7), DateTime(Date(1999, 7, 6), TimeOfDay(12, 23, 33)));
            _assertPred!"-="(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), dur!"seconds"(-7), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 40)));
            _assertPred!"-="(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), dur!"seconds"(7), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 26)));
            _assertPred!"-="(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), dur!"msecs"(-7_000), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 40)));
            _assertPred!"-="(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), dur!"msecs"(7_000), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 26)));
            _assertPred!"-="(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), dur!"usecs"(-7_000_000), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 40)));
            _assertPred!"-="(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), dur!"usecs"(7_000_000), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 26)));
            _assertPred!"-="(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), dur!"hnsecs"(-70_000_000), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 40)));
            _assertPred!"-="(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)), dur!"hnsecs"(70_000_000), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 26)));

            auto duration = dur!"seconds"(12);
            const cdt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            immutable idt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            static assert(!__traits(compiles, cdt += duration));
            static assert(!__traits(compiles, idt += duration));
            static assert(!__traits(compiles, cdt -= duration));
            static assert(!__traits(compiles, idt -= duration));
        }
    }


    /++
        Gives the difference between two $(D DateTime)s.

        The legal types of arithmetic for $(D DateTime) using this operator are

        $(BOOKTABLE,
        $(TR $(TD DateTime) $(TD -) $(TD DateTime) $(TD -->) $(TD duration))
        )
      +/
    Duration opBinary(string op)(in DateTime rhs) const pure nothrow
        if(op == "-")
    {
        immutable dateResult = _date - rhs.date;
        immutable todResult = _tod - rhs._tod;

        return dur!"hnsecs"(dateResult.total!"hnsecs" + todResult.total!"hnsecs");
    }

    unittest
    {
        version(testStdDateTime)
        {
            auto dt = DateTime(1999, 7, 6, 12, 30, 33);

            _assertPred!"=="(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)) -
                         DateTime(Date(1998, 7, 6), TimeOfDay(12, 30, 33)),
                        dur!"seconds"(31_536_000));
            _assertPred!"=="(DateTime(Date(1998, 7, 6), TimeOfDay(12, 30, 33)) -
                         DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)),
                        dur!"seconds"(-31_536_000));

            _assertPred!"=="(DateTime(Date(1999, 8, 6), TimeOfDay(12, 30, 33)) -
                         DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)),
                        dur!"seconds"(26_78_400));
            _assertPred!"=="(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)) -
                         DateTime(Date(1999, 8, 6), TimeOfDay(12, 30, 33)),
                        dur!"seconds"(-26_78_400));

            _assertPred!"=="(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)) -
                         DateTime(Date(1999, 7, 5), TimeOfDay(12, 30, 33)),
                        dur!"seconds"(86_400));
            _assertPred!"=="(DateTime(Date(1999, 7, 5), TimeOfDay(12, 30, 33)) -
                         DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)),
                        dur!"seconds"(-86_400));

            _assertPred!"=="(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)) -
                         DateTime(Date(1999, 7, 6), TimeOfDay(11, 30, 33)),
                        dur!"seconds"(3600));
            _assertPred!"=="(DateTime(Date(1999, 7, 6), TimeOfDay(11, 30, 33)) -
                         DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)),
                        dur!"seconds"(-3600));

            _assertPred!"=="(DateTime(Date(1999, 7, 6), TimeOfDay(12, 31, 33)) -
                         DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)),
                        dur!"seconds"(60));
            _assertPred!"=="(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)) -
                         DateTime(Date(1999, 7, 6), TimeOfDay(12, 31, 33)),
                        dur!"seconds"(-60));

            _assertPred!"=="(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 34)) -
                         DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)),
                        dur!"seconds"(1));
            _assertPred!"=="(DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)) -
                         DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 34)),
                        dur!"seconds"(-1));

            _assertPred!"=="(DateTime(1, 1, 1, 12, 30, 33) - DateTime(1, 1, 1, 0, 0, 0), dur!"seconds"(45033));
            _assertPred!"=="(DateTime(1, 1, 1, 0, 0, 0) - DateTime(1, 1, 1, 12, 30, 33), dur!"seconds"(-45033));
            _assertPred!"=="(DateTime(0, 12, 31, 12, 30, 33) - DateTime(1, 1, 1, 0, 0, 0), dur!"seconds"(-41367));
            _assertPred!"=="(DateTime(1, 1, 1, 0, 0, 0) - DateTime(0, 12, 31, 12, 30, 33), dur!"seconds"(41367));

            const cdt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            immutable idt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            static assert(__traits(compiles, dt - dt));
            static assert(__traits(compiles, cdt - dt));
            static assert(__traits(compiles, idt - dt));

            static assert(__traits(compiles, dt - cdt));
            static assert(__traits(compiles, cdt - cdt));
            static assert(__traits(compiles, idt - cdt));

            static assert(__traits(compiles, dt - idt));
            static assert(__traits(compiles, cdt - idt));
            static assert(__traits(compiles, idt - idt));
        }
    }


    /++
        Returns the difference between the two $(D DateTime)s in months.

        To get the difference in years, subtract the year property
        of two $(D SysTime)s. To get the difference in days or weeks,
        subtract the $(D SysTime)s themselves and use the $(D Duration)
        that results. Because converting between months and smaller
        units requires a specific date (which $(D Duration)s don't have),
        getting the difference in months requires some math using both
        the year and month properties, so this is a convenience function for
        getting the difference in months.

        Note that the number of days in the months or how far into the month
        either date is is irrelevant. It is the difference in the month property
        combined with the difference in years * 12. So, for instance,
        December 31st and January 1st are one month apart just as December 1st
        and January 31st are one month apart.

        Params:
            rhs = The $(D DateTime) to subtract from this one.

        Examples:
--------------------
assert(DateTime(1999, 2, 1, 12, 2, 3).diffMonths(
            DateTime(1999, 1, 31, 23, 59, 59)) == 1);

assert(DateTime(1999, 1, 31, 0, 0, 0).diffMonths(
            DateTime(1999, 2, 1, 12, 3, 42)) == -1);

assert(DateTime(1999, 3, 1, 5, 30, 0).diffMonths(
            DateTime(1999, 1, 1, 2, 4, 7)) == 2);

assert(DateTime(1999, 1, 1, 7, 2, 4).diffMonths(
            DateTime(1999, 3, 31, 0, 30, 58)) == -2);
--------------------
      +/
    int diffMonths(in DateTime rhs) const pure nothrow
    {
        return _date.diffMonths(rhs._date);
    }

    unittest
    {
        version(testStdDateTime)
        {
            auto dt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            const cdt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            immutable idt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            static assert(__traits(compiles, dt.diffMonths(dt)));
            static assert(__traits(compiles, cdt.diffMonths(dt)));
            static assert(__traits(compiles, idt.diffMonths(dt)));

            static assert(__traits(compiles, dt.diffMonths(cdt)));
            static assert(__traits(compiles, cdt.diffMonths(cdt)));
            static assert(__traits(compiles, idt.diffMonths(cdt)));

            static assert(__traits(compiles, dt.diffMonths(idt)));
            static assert(__traits(compiles, cdt.diffMonths(idt)));
            static assert(__traits(compiles, idt.diffMonths(idt)));

            //Verify Examples.
            assert(DateTime(1999, 2, 1, 12, 2, 3).diffMonths(DateTime(1999, 1, 31, 23, 59, 59)) == 1);
            assert(DateTime(1999, 1, 31, 0, 0, 0).diffMonths(DateTime(1999, 2, 1, 12, 3, 42)) == -1);
            assert(DateTime(1999, 3, 1, 5, 30, 0).diffMonths(DateTime(1999, 1, 1, 2, 4, 7)) == 2);
            assert(DateTime(1999, 1, 1, 7, 2, 4).diffMonths(DateTime(1999, 3, 31, 0, 30, 58)) == -2);
        }
    }


    /++
        Whether this $(D DateTime) is in a leap year.
     +/
    @property bool isLeapYear() const pure nothrow
    {
        return _date.isLeapYear;
    }

    unittest
    {
        version(testStdDateTime)
        {
            auto dt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            const cdt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            immutable idt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            static assert(__traits(compiles, dt.isLeapYear));
            static assert(__traits(compiles, cdt.isLeapYear));
            static assert(__traits(compiles, idt.isLeapYear));
        }
    }


    /++
        Day of the week this $(D DateTime) is on.
      +/
    @property DayOfWeek dayOfWeek() const pure nothrow
    {
        return _date.dayOfWeek;
    }

    unittest
    {
        version(testStdDateTime)
        {
            auto dt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            const cdt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            immutable idt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            static assert(__traits(compiles, dt.dayOfWeek));
            static assert(__traits(compiles, cdt.dayOfWeek));
            static assert(__traits(compiles, idt.dayOfWeek));
        }
    }


    /++
        Day of the year this $(D DateTime) is on.

        Examples:
--------------------
assert(DateTime(Date(1999, 1, 1), TimeOfDay(12, 22, 7)).dayOfYear == 1);
assert(DateTime(Date(1999, 12, 31), TimeOfDay(7, 2, 59)).dayOfYear == 365);
assert(DateTime(Date(2000, 12, 31), TimeOfDay(21, 20, 0)).dayOfYear == 366);
--------------------
      +/
    @property ushort dayOfYear() const pure nothrow
    {
        return _date.dayOfYear;
    }

    unittest
    {
        version(testStdDateTime)
        {
            auto dt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            const cdt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            immutable idt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            static assert(__traits(compiles, dt.dayOfYear));
            static assert(__traits(compiles, cdt.dayOfYear));
            static assert(__traits(compiles, idt.dayOfYear));

            //Verify Examples.
            assert(DateTime(Date(1999, 1, 1), TimeOfDay(12, 22, 7)).dayOfYear == 1);
            assert(DateTime(Date(1999, 12, 31), TimeOfDay(7, 2, 59)).dayOfYear == 365);
            assert(DateTime(Date(2000, 12, 31), TimeOfDay(21, 20, 0)).dayOfYear == 366);
        }
    }


    /++
        Day of the year.

        Params:
            day = The day of the year to set which day of the year this
                  $(D DateTime) is on.
      +/
    @property void dayOfYear(int day) pure
    {
        _date.dayOfYear = day;
    }

    unittest
    {
        version(testStdDateTime)
        {
            auto dt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            const cdt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            immutable idt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            static assert(__traits(compiles, dt.dayOfYear = 12));
            static assert(!__traits(compiles, cdt.dayOfYear = 12));
            static assert(!__traits(compiles, idt.dayOfYear = 12));
        }
    }


    /++
        The Xth day of the Gregorian Calendar that this $(D DateTime) is on.

        Examples:
--------------------
assert(DateTime(Date(1, 1, 1), TimeOfDay(0, 0, 0)).dayOfGregorianCal ==
       1);
assert(DateTime(Date(1, 12, 31), TimeOfDay(23, 59, 59)).dayOfGregorianCal ==
       365);
assert(DateTime(Date(2, 1, 1), TimeOfDay(2, 2, 2)).dayOfGregorianCal ==
       366);

assert(DateTime(Date(0, 12, 31), TimeOfDay(7, 7, 7)).dayOfGregorianCal ==
       0);
assert(DateTime(Date(0, 1, 1), TimeOfDay(19, 30, 0)).dayOfGregorianCal ==
       -365);
assert(DateTime(Date(-1, 12, 31), TimeOfDay(4, 7, 0)).dayOfGregorianCal ==
       -366);

assert(DateTime(Date(2000, 1, 1), TimeOfDay(9, 30, 20)).dayOfGregorianCal ==
       730_120);
assert(DateTime(Date(2010, 12, 31), TimeOfDay(15, 45, 50)).dayOfGregorianCal ==
       734_137);
--------------------
     +/
    @property int dayOfGregorianCal() const pure nothrow
    {
        return _date.dayOfGregorianCal;
    }

    unittest
    {
        version(testStdDateTime)
        {
            const cdt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            immutable idt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            static assert(__traits(compiles, cdt.dayOfGregorianCal));
            static assert(__traits(compiles, idt.dayOfGregorianCal));

            //Verify Examples.
            assert(DateTime(Date(1, 1, 1), TimeOfDay(0, 0, 0)).dayOfGregorianCal == 1);
            assert(DateTime(Date(1, 12, 31), TimeOfDay(23, 59, 59)).dayOfGregorianCal == 365);
            assert(DateTime(Date(2, 1, 1), TimeOfDay(2, 2, 2)).dayOfGregorianCal == 366);

            assert(DateTime(Date(0, 12, 31), TimeOfDay(7, 7, 7)).dayOfGregorianCal == 0);
            assert(DateTime(Date(0, 1, 1), TimeOfDay(19, 30, 0)).dayOfGregorianCal == -365);
            assert(DateTime(Date(-1, 12, 31), TimeOfDay(4, 7, 0)).dayOfGregorianCal == -366);

            assert(DateTime(Date(2000, 1, 1), TimeOfDay(9, 30, 20)).dayOfGregorianCal == 730_120);
            assert(DateTime(Date(2010, 12, 31), TimeOfDay(15, 45, 50)).dayOfGregorianCal == 734_137);
        }
    }


    /++
        The Xth day of the Gregorian Calendar that this $(D DateTime) is on.
        Setting this property does not affect the time portion of
        $(D DateTime).

        Params:
            days = The day of the Gregorian Calendar to set this $(D DateTime)
                   to.

        Examples:
--------------------
auto dt = DateTime(Date.init, TimeOfDay(12, 0, 0));
dt.dayOfGregorianCal = 1;
assert(dt == DateTime(Date(1, 1, 1), TimeOfDay(12, 0, 0)));

dt.dayOfGregorianCal = 365;
assert(dt == DateTime(Date(1, 12, 31), TimeOfDay(12, 0, 0)));

dt.dayOfGregorianCal = 366;
assert(dt == DateTime(Date(2, 1, 1), TimeOfDay(12, 0, 0)));

dt.dayOfGregorianCal = 0;
assert(dt == DateTime(Date(0, 12, 31), TimeOfDay(12, 0, 0)));

dt.dayOfGregorianCal = -365;
assert(dt == DateTime(Date(-0, 1, 1), TimeOfDay(12, 0, 0)));

dt.dayOfGregorianCal = -366;
assert(dt == DateTime(Date(-1, 12, 31), TimeOfDay(12, 0, 0)));

dt.dayOfGregorianCal = 730_120;
assert(dt == DateTime(Date(2000, 1, 1), TimeOfDay(12, 0, 0)));

dt.dayOfGregorianCal = 734_137;
assert(dt == DateTime(Date(2010, 12, 31), TimeOfDay(12, 0, 0)));
--------------------
     +/
    @property void dayOfGregorianCal(int days) pure nothrow
    {
        _date.dayOfGregorianCal = days;
    }

    unittest
    {
        version(testStdDateTime)
        {
            const cdt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            immutable idt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            static assert(!__traits(compiles, cdt.dayOfGregorianCal = 7));
            static assert(!__traits(compiles, idt.dayOfGregorianCal = 7));

            //Verify Examples.
            auto dt = DateTime(Date.init, TimeOfDay(12, 0, 0));
            dt.dayOfGregorianCal = 1;
            assert(dt == DateTime(Date(1, 1, 1), TimeOfDay(12, 0, 0)));

            dt.dayOfGregorianCal = 365;
            assert(dt == DateTime(Date(1, 12, 31), TimeOfDay(12, 0, 0)));

            dt.dayOfGregorianCal = 366;
            assert(dt == DateTime(Date(2, 1, 1), TimeOfDay(12, 0, 0)));

            dt.dayOfGregorianCal = 0;
            assert(dt == DateTime(Date(0, 12, 31), TimeOfDay(12, 0, 0)));

            dt.dayOfGregorianCal = -365;
            assert(dt == DateTime(Date(-0, 1, 1), TimeOfDay(12, 0, 0)));

            dt.dayOfGregorianCal = -366;
            assert(dt == DateTime(Date(-1, 12, 31), TimeOfDay(12, 0, 0)));

            dt.dayOfGregorianCal = 730_120;
            assert(dt == DateTime(Date(2000, 1, 1), TimeOfDay(12, 0, 0)));

            dt.dayOfGregorianCal = 734_137;
            assert(dt == DateTime(Date(2010, 12, 31), TimeOfDay(12, 0, 0)));
        }
    }


    /++
        The ISO 8601 week of the year that this $(D DateTime) is in.

        See_Also:
            $(WEB en.wikipedia.org/wiki/ISO_week_date, ISO Week Date)
      +/
    @property ubyte isoWeek() const pure nothrow
    {
        return _date.isoWeek;
    }

    unittest
    {
        version(testStdDateTime)
        {
            auto dt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            const cdt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            immutable idt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            static assert(__traits(compiles, dt.isoWeek));
            static assert(__traits(compiles, cdt.isoWeek));
            static assert(__traits(compiles, idt.isoWeek));
        }
    }


    /++
        $(D DateTime) for the last day in the month that this $(D DateTime) is
        in. The time portion of endOfMonth is always 23:59:59.

        Examples:
--------------------
assert(DateTime(Date(1999, 1, 6), TimeOfDay(0, 0, 0)).endOfMonth ==
       DateTime(Date(1999, 1, 31), TimeOfDay(23, 59, 59)));

assert(DateTime(Date(1999, 2, 7), TimeOfDay(19, 30, 0)).endOfMonth ==
       DateTime(Date(1999, 2, 28), TimeOfDay(23, 59, 59)));

assert(DateTime(Date(2000, 2, 7), TimeOfDay(5, 12, 27)).endOfMonth ==
       DateTime(Date(2000, 2, 29), TimeOfDay(23, 59, 59)));

assert(DateTime(Date(2000, 6, 4), TimeOfDay(12, 22, 9)).endOfMonth ==
       DateTime(Date(2000, 6, 30), TimeOfDay(23, 59, 59)));
--------------------
      +/
    @property DateTime endOfMonth() const pure nothrow
    {
        try
            return DateTime(_date.endOfMonth, TimeOfDay(23, 59, 59));
        catch(Exception e)
            assert(0, "DateTime constructor threw.");
    }

    unittest
    {
        version(testStdDateTime)
        {
            //Test A.D.
            _assertPred!"=="(DateTime(1999, 1, 1, 0, 13, 26).endOfMonth, DateTime(1999, 1, 31, 23, 59, 59));
            _assertPred!"=="(DateTime(1999, 2, 1, 1, 14, 27).endOfMonth, DateTime(1999, 2, 28, 23, 59, 59));
            _assertPred!"=="(DateTime(2000, 2, 1, 2, 15, 28).endOfMonth, DateTime(2000, 2, 29, 23, 59, 59));
            _assertPred!"=="(DateTime(1999, 3, 1, 3, 16, 29).endOfMonth, DateTime(1999, 3, 31, 23, 59, 59));
            _assertPred!"=="(DateTime(1999, 4, 1, 4, 17, 30).endOfMonth, DateTime(1999, 4, 30, 23, 59, 59));
            _assertPred!"=="(DateTime(1999, 5, 1, 5, 18, 31).endOfMonth, DateTime(1999, 5, 31, 23, 59, 59));
            _assertPred!"=="(DateTime(1999, 6, 1, 6, 19, 32).endOfMonth, DateTime(1999, 6, 30, 23, 59, 59));
            _assertPred!"=="(DateTime(1999, 7, 1, 7, 20, 33).endOfMonth, DateTime(1999, 7, 31, 23, 59, 59));
            _assertPred!"=="(DateTime(1999, 8, 1, 8, 21, 34).endOfMonth, DateTime(1999, 8, 31, 23, 59, 59));
            _assertPred!"=="(DateTime(1999, 9, 1, 9, 22, 35).endOfMonth, DateTime(1999, 9, 30, 23, 59, 59));
            _assertPred!"=="(DateTime(1999, 10, 1, 10, 23, 36).endOfMonth, DateTime(1999, 10, 31, 23, 59, 59));
            _assertPred!"=="(DateTime(1999, 11, 1, 11, 24, 37).endOfMonth, DateTime(1999, 11, 30, 23, 59, 59));
            _assertPred!"=="(DateTime(1999, 12, 1, 12, 25, 38).endOfMonth, DateTime(1999, 12, 31, 23, 59, 59));

            //Test B.C.
            _assertPred!"=="(DateTime(-1999, 1, 1, 0, 13, 26).endOfMonth, DateTime(-1999, 1, 31, 23, 59, 59));
            _assertPred!"=="(DateTime(-1999, 2, 1, 1, 14, 27).endOfMonth, DateTime(-1999, 2, 28, 23, 59, 59));
            _assertPred!"=="(DateTime(-2000, 2, 1, 2, 15, 28).endOfMonth, DateTime(-2000, 2, 29, 23, 59, 59));
            _assertPred!"=="(DateTime(-1999, 3, 1, 3, 16, 29).endOfMonth, DateTime(-1999, 3, 31, 23, 59, 59));
            _assertPred!"=="(DateTime(-1999, 4, 1, 4, 17, 30).endOfMonth, DateTime(-1999, 4, 30, 23, 59, 59));
            _assertPred!"=="(DateTime(-1999, 5, 1, 5, 18, 31).endOfMonth, DateTime(-1999, 5, 31, 23, 59, 59));
            _assertPred!"=="(DateTime(-1999, 6, 1, 6, 19, 32).endOfMonth, DateTime(-1999, 6, 30, 23, 59, 59));
            _assertPred!"=="(DateTime(-1999, 7, 1, 7, 20, 33).endOfMonth, DateTime(-1999, 7, 31, 23, 59, 59));
            _assertPred!"=="(DateTime(-1999, 8, 1, 8, 21, 34).endOfMonth, DateTime(-1999, 8, 31, 23, 59, 59));
            _assertPred!"=="(DateTime(-1999, 9, 1, 9, 22, 35).endOfMonth, DateTime(-1999, 9, 30, 23, 59, 59));
            _assertPred!"=="(DateTime(-1999, 10, 1, 10, 23, 36).endOfMonth, DateTime(-1999, 10, 31, 23, 59, 59));
            _assertPred!"=="(DateTime(-1999, 11, 1, 11, 24, 37).endOfMonth, DateTime(-1999, 11, 30, 23, 59, 59));
            _assertPred!"=="(DateTime(-1999, 12, 1, 12, 25, 38).endOfMonth, DateTime(-1999, 12, 31, 23, 59, 59));

            const cdt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            immutable idt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            static assert(__traits(compiles, cdt.endOfMonth));
            static assert(__traits(compiles, idt.endOfMonth));

            //Verify Examples.
            assert(DateTime(Date(1999, 1, 6), TimeOfDay(0, 0, 0)).endOfMonth == DateTime(Date(1999, 1, 31), TimeOfDay(23, 59, 59)));
            assert(DateTime(Date(1999, 2, 7), TimeOfDay(19, 30, 0)).endOfMonth == DateTime(Date(1999, 2, 28), TimeOfDay(23, 59, 59)));
            assert(DateTime(Date(2000, 2, 7), TimeOfDay(5, 12, 27)).endOfMonth == DateTime(Date(2000, 2, 29), TimeOfDay(23, 59, 59)));
            assert(DateTime(Date(2000, 6, 4), TimeOfDay(12, 22, 9)).endOfMonth == DateTime(Date(2000, 6, 30), TimeOfDay(23, 59, 59)));
        }
    }


    /++
        The last day in the month that this $(D DateTime) is in.

        Examples:
--------------------
assert(DateTime(Date(1999, 1, 6), TimeOfDay(0, 0, 0)).daysInMonth == 31);
assert(DateTime(Date(1999, 2, 7), TimeOfDay(19, 30, 0)).daysInMonth == 28);
assert(DateTime(Date(2000, 2, 7), TimeOfDay(5, 12, 27)).daysInMonth == 29);
assert(DateTime(Date(2000, 6, 4), TimeOfDay(12, 22, 9)).daysInMonth == 30);
--------------------
      +/
    @property ubyte daysInMonth() const pure nothrow
    {
        return _date.daysInMonth;
    }

    /++
        $(RED Deprecated. It will be removed in September 2012.
              Please use daysInMonth instead.)
      +/
    deprecated @property ubyte endOfMonthDay() const pure nothrow
    {
        return _date.daysInMonth;
    }

    unittest
    {
        version(testStdDateTime)
        {
            const cdt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            immutable idt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            static assert(__traits(compiles, cdt.daysInMonth));
            static assert(__traits(compiles, idt.daysInMonth));

            //Verify Examples.
            assert(DateTime(Date(1999, 1, 6), TimeOfDay(0, 0, 0)).daysInMonth == 31);
            assert(DateTime(Date(1999, 2, 7), TimeOfDay(19, 30, 0)).daysInMonth == 28);
            assert(DateTime(Date(2000, 2, 7), TimeOfDay(5, 12, 27)).daysInMonth == 29);
            assert(DateTime(Date(2000, 6, 4), TimeOfDay(12, 22, 9)).daysInMonth == 30);
        }
    }


    /++
        Whether the current year is a date in A.D.

        Examples:
--------------------
assert(DateTime(Date(1, 1, 1), TimeOfDay(12, 7, 0)).isAD);
assert(DateTime(Date(2010, 12, 31), TimeOfDay(0, 0, 0)).isAD);
assert(!DateTime(Date(0, 12, 31), TimeOfDay(23, 59, 59)).isAD);
assert(!DateTime(Date(-2010, 1, 1), TimeOfDay(2, 2, 2)).isAD);
--------------------
      +/
    @property bool isAD() const pure nothrow
    {
        return _date.isAD;
    }

    unittest
    {
        version(testStdDateTime)
        {
            const cdt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            immutable idt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            static assert(__traits(compiles, cdt.isAD));
            static assert(__traits(compiles, idt.isAD));

            //Verify Examples.
            assert(DateTime(Date(1, 1, 1), TimeOfDay(12, 7, 0)).isAD);
            assert(DateTime(Date(2010, 12, 31), TimeOfDay(0, 0, 0)).isAD);
            assert(!DateTime(Date(0, 12, 31), TimeOfDay(23, 59, 59)).isAD);
            assert(!DateTime(Date(-2010, 1, 1), TimeOfDay(2, 2, 2)).isAD);
        }
    }


    /++
        The julian day for this $(D DateTime) at the given time. For example,
        prior to noon, 1996-03-31 would be the julian day number 2_450_173, so
        this function returns 2_450_173, while from noon onward, the julian
        day number would be 2_450_174, so this function returns 2_450_174.
      +/
    @property long julianDay() const pure nothrow
    {
        if(_tod._hour < 12)
            return _date.julianDay - 1;
        else
            return _date.julianDay;
    }

    unittest
    {
        version(testStdDateTime)
        {
            _assertPred!"=="(DateTime(Date(-4713, 11, 24), TimeOfDay(0, 0, 0)).julianDay, -1);
            _assertPred!"=="(DateTime(Date(-4713, 11, 24), TimeOfDay(12, 0, 0)).julianDay, 0);

            _assertPred!"=="(DateTime(Date(0, 12, 31), TimeOfDay(0, 0, 0)).julianDay, 1_721_424);
            _assertPred!"=="(DateTime(Date(0, 12, 31), TimeOfDay(12, 0, 0)).julianDay, 1_721_425);

            _assertPred!"=="(DateTime(Date(1, 1, 1), TimeOfDay(0, 0, 0)).julianDay, 1_721_425);
            _assertPred!"=="(DateTime(Date(1, 1, 1), TimeOfDay(12, 0, 0)).julianDay, 1_721_426);

            _assertPred!"=="(DateTime(Date(1582, 10, 15), TimeOfDay(0, 0, 0)).julianDay, 2_299_160);
            _assertPred!"=="(DateTime(Date(1582, 10, 15), TimeOfDay(12, 0, 0)).julianDay, 2_299_161);

            _assertPred!"=="(DateTime(Date(1858, 11, 17), TimeOfDay(0, 0, 0)).julianDay, 2_400_000);
            _assertPred!"=="(DateTime(Date(1858, 11, 17), TimeOfDay(12, 0, 0)).julianDay, 2_400_001);

            _assertPred!"=="(DateTime(Date(1982, 1, 4), TimeOfDay(0, 0, 0)).julianDay, 2_444_973);
            _assertPred!"=="(DateTime(Date(1982, 1, 4), TimeOfDay(12, 0, 0)).julianDay, 2_444_974);

            _assertPred!"=="(DateTime(Date(1996, 3, 31), TimeOfDay(0, 0, 0)).julianDay, 2_450_173);
            _assertPred!"=="(DateTime(Date(1996, 3, 31), TimeOfDay(12, 0, 0)).julianDay, 2_450_174);

            _assertPred!"=="(DateTime(Date(2010, 8, 24), TimeOfDay(0, 0, 0)).julianDay, 2_455_432);
            _assertPred!"=="(DateTime(Date(2010, 8, 24), TimeOfDay(12, 0, 0)).julianDay, 2_455_433);

            const cdt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            immutable idt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            static assert(__traits(compiles, cdt.julianDay));
            static assert(__traits(compiles, idt.julianDay));
        }
    }


    /++
        The modified julian day for any time on this date (since, the modified
        julian day changes at midnight).
      +/
    @property long modJulianDay() const pure nothrow
    {
        return _date.modJulianDay;
    }

    unittest
    {
        version(testStdDateTime)
        {
            _assertPred!"=="(DateTime(Date(1858, 11, 17), TimeOfDay(0, 0, 0)).modJulianDay, 0);
            _assertPred!"=="(DateTime(Date(1858, 11, 17), TimeOfDay(12, 0, 0)).modJulianDay, 0);

            _assertPred!"=="(DateTime(Date(2010, 8, 24), TimeOfDay(0, 0, 0)).modJulianDay, 55_432);
            _assertPred!"=="(DateTime(Date(2010, 8, 24), TimeOfDay(12, 0, 0)).modJulianDay, 55_432);

            const cdt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            immutable idt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            static assert(__traits(compiles, cdt.modJulianDay));
            static assert(__traits(compiles, idt.modJulianDay));
        }
    }


    /++
        Converts this $(D DateTime) to a string with the format YYYYMMDDTHHMMSS.

        Examples:
--------------------
assert(DateTime(Date(2010, 7, 4), TimeOfDay(7, 6, 12)).toISOString() ==
       "20100704T070612");

assert(DateTime(Date(1998, 12, 25), TimeOfDay(2, 15, 0)).toISOString() ==
       "19981225T021500");

assert(DateTime(Date(0, 1, 5), TimeOfDay(23, 9, 59)).toISOString() ==
       "00000105T230959");

assert(DateTime(Date(-4, 1, 5), TimeOfDay(0, 0, 2)).toISOString() ==
       "-00040105T000002");
--------------------
      +/
    string toISOString() const nothrow
    {
        try
            return format("%sT%s", _date.toISOString(), _tod.toISOString());
        catch(Exception e)
            assert(0, "format() threw.");
    }

    unittest
    {
        version(testStdDateTime)
        {
            //Test A.D.
            _assertPred!"=="(DateTime(Date(9, 12, 4), TimeOfDay(0, 0, 0)).toISOString(), "00091204T000000");
            _assertPred!"=="(DateTime(Date(99, 12, 4), TimeOfDay(5, 6, 12)).toISOString(), "00991204T050612");
            _assertPred!"=="(DateTime(Date(999, 12, 4), TimeOfDay(13, 44, 59)).toISOString(), "09991204T134459");
            _assertPred!"=="(DateTime(Date(9999, 7, 4), TimeOfDay(23, 59, 59)).toISOString(), "99990704T235959");
            _assertPred!"=="(DateTime(Date(10000, 10, 20), TimeOfDay(1, 1, 1)).toISOString(), "+100001020T010101");

            //Test B.C.
            _assertPred!"=="(DateTime(Date(0, 12, 4), TimeOfDay(0, 12, 4)).toISOString(), "00001204T001204");
            _assertPred!"=="(DateTime(Date(-9, 12, 4), TimeOfDay(0, 0, 0)).toISOString(), "-00091204T000000");
            _assertPred!"=="(DateTime(Date(-99, 12, 4), TimeOfDay(5, 6, 12)).toISOString(), "-00991204T050612");
            _assertPred!"=="(DateTime(Date(-999, 12, 4), TimeOfDay(13, 44, 59)).toISOString(), "-09991204T134459");
            _assertPred!"=="(DateTime(Date(-9999, 7, 4), TimeOfDay(23, 59, 59)).toISOString(), "-99990704T235959");
            _assertPred!"=="(DateTime(Date(-10000, 10, 20), TimeOfDay(1, 1, 1)).toISOString(), "-100001020T010101");

            const cdt = DateTime(1999, 7, 6, 12, 30, 33);
            immutable idt = DateTime(1999, 7, 6, 12, 30, 33);
            static assert(__traits(compiles, cdt.toISOString()));
            static assert(__traits(compiles, idt.toISOString()));

            //Verify Examples.
            assert(DateTime(Date(2010, 7, 4), TimeOfDay(7, 6, 12)).toISOString() == "20100704T070612");
            assert(DateTime(Date(1998, 12, 25), TimeOfDay(2, 15, 0)).toISOString() == "19981225T021500");
            assert(DateTime(Date(0, 1, 5), TimeOfDay(23, 9, 59)).toISOString() == "00000105T230959");
            assert(DateTime(Date(-4, 1, 5), TimeOfDay(0, 0, 2)).toISOString() == "-00040105T000002");
        }
    }


    /++
        Converts this $(D DateTime) to a string with the format
        YYYY-MM-DDTHH:MM:SS.

        Examples:
--------------------
assert(DateTime(Date(2010, 7, 4), TimeOfDay(7, 6, 12)).toISOExtString() ==
       "2010-07-04T07:06:12");

assert(DateTime(Date(1998, 12, 25), TimeOfDay(2, 15, 0)).toISOExtString() ==
       "1998-12-25T02:15:00");

assert(DateTime(Date(0, 1, 5), TimeOfDay(23, 9, 59)).toISOExtString() ==
       "0000-01-05T23:09:59");

assert(DateTime(Date(-4, 1, 5), TimeOfDay(0, 0, 2)).toISOExtString() ==
       "-0004-01-05T00:00:02");
--------------------
      +/
    string toISOExtString() const nothrow
    {
        try
            return format("%sT%s", _date.toISOExtString(), _tod.toISOExtString());
        catch(Exception e)
            assert(0, "format() threw.");
    }

    unittest
    {
        version(testStdDateTime)
        {
            //Test A.D.
            _assertPred!"=="(DateTime(Date(9, 12, 4), TimeOfDay(0, 0, 0)).toISOExtString(), "0009-12-04T00:00:00");
            _assertPred!"=="(DateTime(Date(99, 12, 4), TimeOfDay(5, 6, 12)).toISOExtString(), "0099-12-04T05:06:12");
            _assertPred!"=="(DateTime(Date(999, 12, 4), TimeOfDay(13, 44, 59)).toISOExtString(), "0999-12-04T13:44:59");
            _assertPred!"=="(DateTime(Date(9999, 7, 4), TimeOfDay(23, 59, 59)).toISOExtString(), "9999-07-04T23:59:59");
            _assertPred!"=="(DateTime(Date(10000, 10, 20), TimeOfDay(1, 1, 1)).toISOExtString(), "+10000-10-20T01:01:01");

            //Test B.C.
            _assertPred!"=="(DateTime(Date(0, 12, 4), TimeOfDay(0, 12, 4)).toISOExtString(), "0000-12-04T00:12:04");
            _assertPred!"=="(DateTime(Date(-9, 12, 4), TimeOfDay(0, 0, 0)).toISOExtString(), "-0009-12-04T00:00:00");
            _assertPred!"=="(DateTime(Date(-99, 12, 4), TimeOfDay(5, 6, 12)).toISOExtString(), "-0099-12-04T05:06:12");
            _assertPred!"=="(DateTime(Date(-999, 12, 4), TimeOfDay(13, 44, 59)).toISOExtString(), "-0999-12-04T13:44:59");
            _assertPred!"=="(DateTime(Date(-9999, 7, 4), TimeOfDay(23, 59, 59)).toISOExtString(), "-9999-07-04T23:59:59");
            _assertPred!"=="(DateTime(Date(-10000, 10, 20), TimeOfDay(1, 1, 1)).toISOExtString(), "-10000-10-20T01:01:01");

            const cdt = DateTime(1999, 7, 6, 12, 30, 33);
            immutable idt = DateTime(1999, 7, 6, 12, 30, 33);
            static assert(__traits(compiles, cdt.toISOExtString()));
            static assert(__traits(compiles, idt.toISOExtString()));

            //Verify Examples.
            assert(DateTime(Date(2010, 7, 4), TimeOfDay(7, 6, 12)).toISOExtString() == "2010-07-04T07:06:12");
            assert(DateTime(Date(1998, 12, 25), TimeOfDay(2, 15, 0)).toISOExtString() == "1998-12-25T02:15:00");
            assert(DateTime(Date(0, 1, 5), TimeOfDay(23, 9, 59)).toISOExtString() == "0000-01-05T23:09:59");
            assert(DateTime(Date(-4, 1, 5), TimeOfDay(0, 0, 2)).toISOExtString() == "-0004-01-05T00:00:02");
        }
    }

    /++
        Converts this $(D DateTime) to a string with the format
        YYYY-Mon-DD HH:MM:SS.

        Examples:
--------------------
assert(DateTime(Date(2010, 7, 4), TimeOfDay(7, 6, 12)).toSimpleString() ==
       "2010-Jul-04 07:06:12");

assert(DateTime(Date(1998, 12, 25), TimeOfDay(2, 15, 0)).toSimpleString() ==
       "1998-Dec-25 02:15:00");

assert(DateTime(Date(0, 1, 5), TimeOfDay(23, 9, 59)).toSimpleString() ==
       "0000-Jan-05 23:09:59");

assert(DateTime(Dte(-4, 1, 5), TimeOfDay(0, 0, 2)).toSimpleString() ==
       "-0004-Jan-05 00:00:02");
--------------------
      +/
    string toSimpleString() const nothrow
    {
        try
            return format("%s %s", _date.toSimpleString(), _tod.toString());
        catch(Exception e)
            assert(0, "format() threw.");
    }

    unittest
    {
        version(testStdDateTime)
        {
            //Test A.D.
            _assertPred!"=="(DateTime(Date(9, 12, 4), TimeOfDay(0, 0, 0)).toSimpleString(), "0009-Dec-04 00:00:00");
            _assertPred!"=="(DateTime(Date(99, 12, 4), TimeOfDay(5, 6, 12)).toSimpleString(), "0099-Dec-04 05:06:12");
            _assertPred!"=="(DateTime(Date(999, 12, 4), TimeOfDay(13, 44, 59)).toSimpleString(), "0999-Dec-04 13:44:59");
            _assertPred!"=="(DateTime(Date(9999, 7, 4), TimeOfDay(23, 59, 59)).toSimpleString(), "9999-Jul-04 23:59:59");
            _assertPred!"=="(DateTime(Date(10000, 10, 20), TimeOfDay(1, 1, 1)).toSimpleString(), "+10000-Oct-20 01:01:01");

            //Test B.C.
            _assertPred!"=="(DateTime(Date(0, 12, 4), TimeOfDay(0, 12, 4)).toSimpleString(), "0000-Dec-04 00:12:04");
            _assertPred!"=="(DateTime(Date(-9, 12, 4), TimeOfDay(0, 0, 0)).toSimpleString(), "-0009-Dec-04 00:00:00");
            _assertPred!"=="(DateTime(Date(-99, 12, 4), TimeOfDay(5, 6, 12)).toSimpleString(), "-0099-Dec-04 05:06:12");
            _assertPred!"=="(DateTime(Date(-999, 12, 4), TimeOfDay(13, 44, 59)).toSimpleString(), "-0999-Dec-04 13:44:59");
            _assertPred!"=="(DateTime(Date(-9999, 7, 4), TimeOfDay(23, 59, 59)).toSimpleString(), "-9999-Jul-04 23:59:59");
            _assertPred!"=="(DateTime(Date(-10000, 10, 20), TimeOfDay(1, 1, 1)).toSimpleString(), "-10000-Oct-20 01:01:01");

            const cdt = DateTime(1999, 7, 6, 12, 30, 33);
            immutable idt = DateTime(1999, 7, 6, 12, 30, 33);
            static assert(__traits(compiles, cdt.toSimpleString()));
            static assert(__traits(compiles, idt.toSimpleString()));

            //Verify Examples.
            assert(DateTime(Date(2010, 7, 4), TimeOfDay(7, 6, 12)).toSimpleString() == "2010-Jul-04 07:06:12");
            assert(DateTime(Date(1998, 12, 25), TimeOfDay(2, 15, 0)).toSimpleString() == "1998-Dec-25 02:15:00");
            assert(DateTime(Date(0, 1, 5), TimeOfDay(23, 9, 59)).toSimpleString() == "0000-Jan-05 23:09:59");
            assert(DateTime(Date(-4, 1, 5), TimeOfDay(0, 0, 2)).toSimpleString() == "-0004-Jan-05 00:00:02");
        }
    }


    /+
        Converts this $(D DateTime) to a string.
      +/
    //Due to bug http://d.puremagic.com/issues/show_bug.cgi?id=3715 , we can't
    //have versions of toString() with extra modifiers, so we define one version
    //with modifiers and one without.
    string toString()
    {
        return toSimpleString();
    }

    /++
        Converts this $(D DateTime) to a string.
      +/
    //Due to bug http://d.puremagic.com/issues/show_bug.cgi?id=3715 , we can't
    //have versions of toString() with extra modifiers, so we define one version
    //with modifiers and one without.
    string toString() const nothrow
    {
        return toSimpleString();
    }

    unittest
    {
        version(testStdDateTime)
        {
            auto dt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            const cdt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            immutable idt = DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33));
            static assert(__traits(compiles, dt.toString()));
            static assert(__traits(compiles, cdt.toString()));
            static assert(__traits(compiles, idt.toString()));
        }
    }



    /++
        Creates a $(D DateTime) from a string with the format YYYYMMDDTHHMMSS.
        Whitespace is stripped from the given string.

        Params:
            isoString = A string formatted in the ISO format for dates and times.

        Throws:
            $(D DateTimeException) if the given string is not in the ISO format
            or if the resulting $(D DateTime) would not be valid.

        Examples:
--------------------
assert(DateTime.fromISOString("20100704T070612") ==
       DateTime(Date(2010, 7, 4), TimeOfDay(7, 6, 12)));

assert(DateTime.fromISOString("19981225T021500") ==
       DateTime(Date(1998, 12, 25), TimeOfDay(2, 15, 0)));

assert(DateTime.fromISOString("00000105T230959") ==
       DateTime(Date(0, 1, 5), TimeOfDay(23, 9, 59)));

assert(DateTime.fromISOString("-00040105T000002") ==
       DateTime(Date(-4, 1, 5), TimeOfDay(0, 0, 2)));

assert(DateTime.fromISOString(" 20100704T070612 ") ==
       DateTime(Date(2010, 7, 4), TimeOfDay(7, 6, 12)));
--------------------
      +/
    static DateTime fromISOString(S)(in S isoString)
        if(isSomeString!S)
    {
        immutable dstr = to!dstring(strip(isoString));

        enforce(dstr.length >= 15, new DateTimeException(format("Invalid ISO String: %s", isoString)));
        auto t = dstr.stds_indexOf('T');

        enforce(t != -1, new DateTimeException(format("Invalid ISO String: %s", isoString)));

        immutable date = Date.fromISOString(dstr[0..t]);
        immutable tod = TimeOfDay.fromISOString(dstr[t+1 .. $]);

        return DateTime(date, tod);
    }

    unittest
    {
        version(testStdDateTime)
        {
            assertThrown!DateTimeException(DateTime.fromISOString(""));
            assertThrown!DateTimeException(DateTime.fromISOString("20100704000000"));
            assertThrown!DateTimeException(DateTime.fromISOString("20100704 000000"));
            assertThrown!DateTimeException(DateTime.fromISOString("20100704t000000"));
            assertThrown!DateTimeException(DateTime.fromISOString("20100704T000000."));
            assertThrown!DateTimeException(DateTime.fromISOString("20100704T000000.0"));

            assertThrown!DateTimeException(DateTime.fromISOString("2010-07-0400:00:00"));
            assertThrown!DateTimeException(DateTime.fromISOString("2010-07-04 00:00:00"));
            assertThrown!DateTimeException(DateTime.fromISOString("2010-07-04t00:00:00"));
            assertThrown!DateTimeException(DateTime.fromISOString("2010-07-04T00:00:00."));
            assertThrown!DateTimeException(DateTime.fromISOString("2010-07-04T00:00:00.0"));

            assertThrown!DateTimeException(DateTime.fromISOString("2010-Jul-0400:00:00"));
            assertThrown!DateTimeException(DateTime.fromISOString("2010-Jul-04 00:00:00"));
            assertThrown!DateTimeException(DateTime.fromISOString("2010-Jul-04t00:00:00"));
            assertThrown!DateTimeException(DateTime.fromISOString("2010-Jul-04T00:00:00"));
            assertThrown!DateTimeException(DateTime.fromISOString("2010-Jul-04 00:00:00."));
            assertThrown!DateTimeException(DateTime.fromISOString("2010-Jul-04 00:00:00.0"));

            assertThrown!DateTimeException(DateTime.fromISOString("2010-12-22T172201"));
            assertThrown!DateTimeException(DateTime.fromISOString("2010-Dec-22 17:22:01"));

            _assertPred!"=="(DateTime.fromISOString("20101222T172201"), DateTime(Date(2010, 12, 22), TimeOfDay(17, 22, 01)));
            _assertPred!"=="(DateTime.fromISOString("19990706T123033"), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
            _assertPred!"=="(DateTime.fromISOString("-19990706T123033"), DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)));
            _assertPred!"=="(DateTime.fromISOString("+019990706T123033"), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
            _assertPred!"=="(DateTime.fromISOString("19990706T123033 "), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
            _assertPred!"=="(DateTime.fromISOString(" 19990706T123033"), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
            _assertPred!"=="(DateTime.fromISOString(" 19990706T123033 "), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));

            //Verify Examples.
            assert(DateTime.fromISOString("20100704T070612") == DateTime(Date(2010, 7, 4), TimeOfDay(7, 6, 12)));
            assert(DateTime.fromISOString("19981225T021500") == DateTime(Date(1998, 12, 25), TimeOfDay(2, 15, 0)));
            assert(DateTime.fromISOString("00000105T230959") == DateTime(Date(0, 1, 5), TimeOfDay(23, 9, 59)));
            assert(DateTime.fromISOString("-00040105T000002") == DateTime(Date(-4, 1, 5), TimeOfDay(0, 0, 2)));
            assert(DateTime.fromISOString(" 20100704T070612 ") == DateTime(Date(2010, 7, 4), TimeOfDay(7, 6, 12)));
        }
    }


    /++
        Creates a $(D DateTime) from a string with the format
        YYYY-MM-DDTHH:MM:SS. Whitespace is stripped from the given string.

        Params:
            isoString = A string formatted in the ISO Extended format for dates
                        and times.

        Throws:
            $(D DateTimeException) if the given string is not in the ISO
            Extended format or if the resulting $(D DateTime) would not be
            valid.

        Examples:
--------------------
assert(DateTime.fromISOExtString("2010-07-04T07:06:12") ==
       DateTime(Date(2010, 7, 4), TimeOfDay(7, 6, 12)));

assert(DateTime.fromISOExtString("1998-12-25T02:15:00") ==
       DateTime(Date(1998, 12, 25), TimeOfDay(2, 15, 0)));

assert(DateTime.fromISOExtString("0000-01-05T23:09:59") ==
       DateTime(Date(0, 1, 5), TimeOfDay(23, 9, 59)));

assert(DateTime.fromISOExtString("-0004-01-05T00:00:02") ==
       DateTime(Date(-4, 1, 5), TimeOfDay(0, 0, 2)));

assert(DateTime.fromISOExtString(" 2010-07-04T07:06:12 ") ==
       DateTime(Date(2010, 7, 4), TimeOfDay(7, 6, 12)));
--------------------
      +/
    static DateTime fromISOExtString(S)(in S isoExtString)
        if(isSomeString!(S))
    {
        immutable dstr = to!dstring(strip(isoExtString));

        enforce(dstr.length >= 15, new DateTimeException(format("Invalid ISO Extended String: %s", isoExtString)));
        auto t = dstr.stds_indexOf('T');

        enforce(t != -1, new DateTimeException(format("Invalid ISO Extended String: %s", isoExtString)));

        immutable date = Date.fromISOExtString(dstr[0..t]);
        immutable tod = TimeOfDay.fromISOExtString(dstr[t+1 .. $]);

        return DateTime(date, tod);
    }

    unittest
    {
        version(testStdDateTime)
        {
            assertThrown!DateTimeException(DateTime.fromISOExtString(""));
            assertThrown!DateTimeException(DateTime.fromISOExtString("20100704000000"));
            assertThrown!DateTimeException(DateTime.fromISOExtString("20100704 000000"));
            assertThrown!DateTimeException(DateTime.fromISOExtString("20100704t000000"));
            assertThrown!DateTimeException(DateTime.fromISOExtString("20100704T000000."));
            assertThrown!DateTimeException(DateTime.fromISOExtString("20100704T000000.0"));

            assertThrown!DateTimeException(DateTime.fromISOExtString("2010-07:0400:00:00"));
            assertThrown!DateTimeException(DateTime.fromISOExtString("2010-07-04 00:00:00"));
            assertThrown!DateTimeException(DateTime.fromISOExtString("2010-07-04 00:00:00"));
            assertThrown!DateTimeException(DateTime.fromISOExtString("2010-07-04t00:00:00"));
            assertThrown!DateTimeException(DateTime.fromISOExtString("2010-07-04T00:00:00."));
            assertThrown!DateTimeException(DateTime.fromISOExtString("2010-07-04T00:00:00.0"));

            assertThrown!DateTimeException(DateTime.fromISOExtString("2010-Jul-0400:00:00"));
            assertThrown!DateTimeException(DateTime.fromISOExtString("2010-Jul-04t00:00:00"));
            assertThrown!DateTimeException(DateTime.fromISOExtString("2010-Jul-04 00:00:00."));
            assertThrown!DateTimeException(DateTime.fromISOExtString("2010-Jul-04 00:00:00.0"));

            assertThrown!DateTimeException(DateTime.fromISOExtString("20101222T172201"));
            assertThrown!DateTimeException(DateTime.fromISOExtString("2010-Dec-22 17:22:01"));

            _assertPred!"=="(DateTime.fromISOExtString("2010-12-22T17:22:01"), DateTime(Date(2010, 12, 22), TimeOfDay(17, 22, 01)));
            _assertPred!"=="(DateTime.fromISOExtString("1999-07-06T12:30:33"), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
            _assertPred!"=="(DateTime.fromISOExtString("-1999-07-06T12:30:33"), DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)));
            _assertPred!"=="(DateTime.fromISOExtString("+01999-07-06T12:30:33"), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
            _assertPred!"=="(DateTime.fromISOExtString("1999-07-06T12:30:33 "), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
            _assertPred!"=="(DateTime.fromISOExtString(" 1999-07-06T12:30:33"), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
            _assertPred!"=="(DateTime.fromISOExtString(" 1999-07-06T12:30:33 "), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));

            //Verify Examples.
            assert(DateTime.fromISOExtString("2010-07-04T07:06:12") == DateTime(Date(2010, 7, 4), TimeOfDay(7, 6, 12)));
            assert(DateTime.fromISOExtString("1998-12-25T02:15:00") == DateTime(Date(1998, 12, 25), TimeOfDay(2, 15, 0)));
            assert(DateTime.fromISOExtString("0000-01-05T23:09:59") == DateTime(Date(0, 1, 5), TimeOfDay(23, 9, 59)));
            assert(DateTime.fromISOExtString("-0004-01-05T00:00:02") == DateTime(Date(-4, 1, 5), TimeOfDay(0, 0, 2)));
            assert(DateTime.fromISOExtString(" 2010-07-04T07:06:12 ") == DateTime(Date(2010, 7, 4), TimeOfDay(7, 6, 12)));
        }
    }


    /++
        Creates a $(D DateTime) from a string with the format
        YYYY-Mon-DD HH:MM:SS. Whitespace is stripped from the given string.

        Params:
            simpleString = A string formatted in the way that toSimpleString
                           formats dates and times.

        Throws:
            $(D DateTimeException) if the given string is not in the correct
            format or if the resulting $(D DateTime) would not be valid.

        Examples:
--------------------
assert(DateTime.fromSimpleString("2010-Jul-04 07:06:12") ==
       DateTime(Date(2010, 7, 4), TimeOfDay(7, 6, 12)));
assert(DateTime.fromSimpleString("1998-Dec-25 02:15:00") ==
       DateTime(Date(1998, 12, 25), TimeOfDay(2, 15, 0)));
assert(DateTime.fromSimpleString("0000-Jan-05 23:09:59") ==
       DateTime(Date(0, 1, 5), TimeOfDay(23, 9, 59)));
assert(DateTime.fromSimpleString("-0004-Jan-05 00:00:02") ==
       DateTime(Date(-4, 1, 5), TimeOfDay(0, 0, 2)));
assert(DateTime.fromSimpleString(" 2010-Jul-04 07:06:12 ") ==
       DateTime(Date(2010, 7, 4), TimeOfDay(7, 6, 12)));
--------------------
      +/
    static DateTime fromSimpleString(S)(in S simpleString)
        if(isSomeString!(S))
    {
        immutable dstr = to!dstring(strip(simpleString));

        enforce(dstr.length >= 15, new DateTimeException(format("Invalid string format: %s", simpleString)));
        auto t = dstr.stds_indexOf(' ');

        enforce(t != -1, new DateTimeException(format("Invalid string format: %s", simpleString)));

        immutable date = Date.fromSimpleString(dstr[0..t]);
        immutable tod = TimeOfDay.fromISOExtString(dstr[t+1 .. $]);

        return DateTime(date, tod);
    }

    unittest
    {
        version(testStdDateTime)
        {
            assertThrown!DateTimeException(DateTime.fromISOString(""));
            assertThrown!DateTimeException(DateTime.fromISOString("20100704000000"));
            assertThrown!DateTimeException(DateTime.fromISOString("20100704 000000"));
            assertThrown!DateTimeException(DateTime.fromISOString("20100704t000000"));
            assertThrown!DateTimeException(DateTime.fromISOString("20100704T000000."));
            assertThrown!DateTimeException(DateTime.fromISOString("20100704T000000.0"));

            assertThrown!DateTimeException(DateTime.fromISOString("2010-07-0400:00:00"));
            assertThrown!DateTimeException(DateTime.fromISOString("2010-07-04 00:00:00"));
            assertThrown!DateTimeException(DateTime.fromISOString("2010-07-04t00:00:00"));
            assertThrown!DateTimeException(DateTime.fromISOString("2010-07-04T00:00:00."));
            assertThrown!DateTimeException(DateTime.fromISOString("2010-07-04T00:00:00.0"));

            assertThrown!DateTimeException(DateTime.fromISOString("2010-Jul-0400:00:00"));
            assertThrown!DateTimeException(DateTime.fromISOString("2010-Jul-04 00:00:00"));
            assertThrown!DateTimeException(DateTime.fromISOString("2010-Jul-04t00:00:00"));
            assertThrown!DateTimeException(DateTime.fromISOString("2010-Jul-04T00:00:00"));
            assertThrown!DateTimeException(DateTime.fromISOString("2010-Jul-04 00:00:00."));
            assertThrown!DateTimeException(DateTime.fromISOString("2010-Jul-04 00:00:00.0"));

            assertThrown!DateTimeException(DateTime.fromSimpleString("20101222T172201"));
            assertThrown!DateTimeException(DateTime.fromSimpleString("2010-12-22T172201"));

            _assertPred!"=="(DateTime.fromSimpleString("2010-Dec-22 17:22:01"), DateTime(Date(2010, 12, 22), TimeOfDay(17, 22, 01)));
            _assertPred!"=="(DateTime.fromSimpleString("1999-Jul-06 12:30:33"), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
            _assertPred!"=="(DateTime.fromSimpleString("-1999-Jul-06 12:30:33"), DateTime(Date(-1999, 7, 6), TimeOfDay(12, 30, 33)));
            _assertPred!"=="(DateTime.fromSimpleString("+01999-Jul-06 12:30:33"), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
            _assertPred!"=="(DateTime.fromSimpleString("1999-Jul-06 12:30:33 "), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
            _assertPred!"=="(DateTime.fromSimpleString(" 1999-Jul-06 12:30:33"), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));
            _assertPred!"=="(DateTime.fromSimpleString(" 1999-Jul-06 12:30:33 "), DateTime(Date(1999, 7, 6), TimeOfDay(12, 30, 33)));

            //Verify Examples.
            assert(DateTime.fromSimpleString("2010-Jul-04 07:06:12") == DateTime(Date(2010, 7, 4), TimeOfDay(7, 6, 12)));
            assert(DateTime.fromSimpleString("1998-Dec-25 02:15:00") == DateTime(Date(1998, 12, 25), TimeOfDay(2, 15, 0)));
            assert(DateTime.fromSimpleString("0000-Jan-05 23:09:59") == DateTime(Date(0, 1, 5), TimeOfDay(23, 9, 59)));
            assert(DateTime.fromSimpleString("-0004-Jan-05 00:00:02") == DateTime(Date(-4, 1, 5), TimeOfDay(0, 0, 2)));
            assert(DateTime.fromSimpleString(" 2010-Jul-04 07:06:12 ") == DateTime(Date(2010, 7, 4), TimeOfDay(7, 6, 12)));
        }
    }


    //TODO Add function which takes a user-specified time format and produces a DateTime

    //TODO Add function which takes pretty much any time-string and produces a DateTime
    //     Obviously, it will be less efficient, and it probably won't manage _every_
    //     possible date format, but a smart conversion function would be nice.


    /++
        Returns the $(D DateTime) farthest in the past which is representable by
        $(D DateTime).
      +/
    @property static DateTime min() pure nothrow
    out(result)
    {
        assert(result._date == Date.min);
        assert(result._tod == TimeOfDay.min);
    }
    body
    {
        auto dt = DateTime.init;
        dt._date._year = short.min;
        dt._date._month = Month.jan;
        dt._date._day = 1;

        return dt;
    }

    unittest
    {
        version(testStdDateTime)
        {
            assert(DateTime.min.year < 0);
            assert(DateTime.min < DateTime.max);
        }
    }


    /++
        Returns the $(D DateTime) farthest in the future which is representable
        by $(D DateTime).
      +/
    @property static DateTime max() pure nothrow
    out(result)
    {
        assert(result._date == Date.max);
        assert(result._tod == TimeOfDay.max);
    }
    body
    {
        auto dt = DateTime.init;
        dt._date._year = short.max;
        dt._date._month = Month.dec;
        dt._date._day = 31;
        dt._tod._hour = TimeOfDay.maxHour;
        dt._tod._minute = TimeOfDay.maxMinute;
        dt._tod._second = TimeOfDay.maxSecond;

        return dt;
    }

    unittest
    {
        version(testStdDateTime)
        {
            assert(DateTime.max.year > 0);
            assert(DateTime.max > DateTime.min);
        }
    }


private:

    /+
        Add seconds to the time of day. Negative values will subtract. If the
        number of seconds overflows (or underflows), then the seconds will wrap,
        increasing (or decreasing) the number of minutes accordingly. The
        same goes for any larger units.

        Params:
            seconds = The number of seconds to add to this $(D DateTime).
      +/
    ref DateTime addSeconds(long seconds) pure nothrow
    {
        long hnsecs = convert!("seconds", "hnsecs")(seconds);
        hnsecs += convert!("hours", "hnsecs")(_tod._hour);
        hnsecs += convert!("minutes", "hnsecs")(_tod._minute);
        hnsecs += convert!("seconds", "hnsecs")(_tod._second);

        auto days = splitUnitsFromHNSecs!"days"(hnsecs);

        if(hnsecs < 0)
        {
            hnsecs += convert!("days", "hnsecs")(1);
            --days;
        }

        _date.addDays(days);

        immutable newHours = splitUnitsFromHNSecs!"hours"(hnsecs);
        immutable newMinutes = splitUnitsFromHNSecs!"minutes"(hnsecs);
        immutable newSeconds = splitUnitsFromHNSecs!"seconds"(hnsecs);

        _tod._hour = cast(ubyte)newHours;
        _tod._minute = cast(ubyte)newMinutes;
        _tod._second = cast(ubyte)newSeconds;

        return this;
    }

    unittest
    {
        version(testStdDateTime)
        {
            static void testDT(DateTime orig, int seconds, in DateTime expected, size_t line = __LINE__)
            {
                orig.addSeconds(seconds);
                _assertPred!"=="(orig, expected, "", __FILE__, line);
            }

            //Test A.D.
            testDT(DateTime(1999, 7, 6, 12, 30, 33), 0, DateTime(1999, 7, 6, 12, 30, 33));
            testDT(DateTime(1999, 7, 6, 12, 30, 33), 1, DateTime(1999, 7, 6, 12, 30, 34));
            testDT(DateTime(1999, 7, 6, 12, 30, 33), 2, DateTime(1999, 7, 6, 12, 30, 35));
            testDT(DateTime(1999, 7, 6, 12, 30, 33), 3, DateTime(1999, 7, 6, 12, 30, 36));
            testDT(DateTime(1999, 7, 6, 12, 30, 33), 4, DateTime(1999, 7, 6, 12, 30, 37));
            testDT(DateTime(1999, 7, 6, 12, 30, 33), 5, DateTime(1999, 7, 6, 12, 30, 38));
            testDT(DateTime(1999, 7, 6, 12, 30, 33), 10, DateTime(1999, 7, 6, 12, 30, 43));
            testDT(DateTime(1999, 7, 6, 12, 30, 33), 15, DateTime(1999, 7, 6, 12, 30, 48));
            testDT(DateTime(1999, 7, 6, 12, 30, 33), 26, DateTime(1999, 7, 6, 12, 30, 59));
            testDT(DateTime(1999, 7, 6, 12, 30, 33), 27, DateTime(1999, 7, 6, 12, 31, 0));
            testDT(DateTime(1999, 7, 6, 12, 30, 33), 30, DateTime(1999, 7, 6, 12, 31, 3));
            testDT(DateTime(1999, 7, 6, 12, 30, 33), 59, DateTime(1999, 7, 6, 12, 31, 32));
            testDT(DateTime(1999, 7, 6, 12, 30, 33), 60, DateTime(1999, 7, 6, 12, 31, 33));
            testDT(DateTime(1999, 7, 6, 12, 30, 33), 61, DateTime(1999, 7, 6, 12, 31, 34));

            testDT(DateTime(1999, 7, 6, 12, 30, 33), 1766, DateTime(1999, 7, 6, 12, 59, 59));
            testDT(DateTime(1999, 7, 6, 12, 30, 33), 1767, DateTime(1999, 7, 6, 13, 0, 0));
            testDT(DateTime(1999, 7, 6, 12, 30, 33), 1768, DateTime(1999, 7, 6, 13, 0, 1));
            testDT(DateTime(1999, 7, 6, 12, 30, 33), 2007, DateTime(1999, 7, 6, 13, 4, 0));
            testDT(DateTime(1999, 7, 6, 12, 30, 33), 3599, DateTime(1999, 7, 6, 13, 30, 32));
            testDT(DateTime(1999, 7, 6, 12, 30, 33), 3600, DateTime(1999, 7, 6, 13, 30, 33));
            testDT(DateTime(1999, 7, 6, 12, 30, 33), 3601, DateTime(1999, 7, 6, 13, 30, 34));
            testDT(DateTime(1999, 7, 6, 12, 30, 33), 7200, DateTime(1999, 7, 6, 14, 30, 33));
            testDT(DateTime(1999, 7, 6, 23, 0, 0), 432_123, DateTime(1999, 7, 11, 23, 2, 3));

            testDT(DateTime(1999, 7, 6, 12, 30, 33), -1, DateTime(1999, 7, 6, 12, 30, 32));
            testDT(DateTime(1999, 7, 6, 12, 30, 33), -2, DateTime(1999, 7, 6, 12, 30, 31));
            testDT(DateTime(1999, 7, 6, 12, 30, 33), -3, DateTime(1999, 7, 6, 12, 30, 30));
            testDT(DateTime(1999, 7, 6, 12, 30, 33), -4, DateTime(1999, 7, 6, 12, 30, 29));
            testDT(DateTime(1999, 7, 6, 12, 30, 33), -5, DateTime(1999, 7, 6, 12, 30, 28));
            testDT(DateTime(1999, 7, 6, 12, 30, 33), -10, DateTime(1999, 7, 6, 12, 30, 23));
            testDT(DateTime(1999, 7, 6, 12, 30, 33), -15, DateTime(1999, 7, 6, 12, 30, 18));
            testDT(DateTime(1999, 7, 6, 12, 30, 33), -33, DateTime(1999, 7, 6, 12, 30, 0));
            testDT(DateTime(1999, 7, 6, 12, 30, 33), -34, DateTime(1999, 7, 6, 12, 29, 59));
            testDT(DateTime(1999, 7, 6, 12, 30, 33), -35, DateTime(1999, 7, 6, 12, 29, 58));
            testDT(DateTime(1999, 7, 6, 12, 30, 33), -59, DateTime(1999, 7, 6, 12, 29, 34));
            testDT(DateTime(1999, 7, 6, 12, 30, 33), -60, DateTime(1999, 7, 6, 12, 29, 33));
            testDT(DateTime(1999, 7, 6, 12, 30, 33), -61, DateTime(1999, 7, 6, 12, 29, 32));

            testDT(DateTime(1999, 7, 6, 12, 30, 33), -1833, DateTime(1999, 7, 6, 12, 0, 0));
            testDT(DateTime(1999, 7, 6, 12, 30, 33), -1834, DateTime(1999, 7, 6, 11, 59, 59));
            testDT(DateTime(1999, 7, 6, 12, 30, 33), -3600, DateTime(1999, 7, 6, 11, 30, 33));
            testDT(DateTime(1999, 7, 6, 12, 30, 33), -3601, DateTime(1999, 7, 6, 11, 30, 32));
            testDT(DateTime(1999, 7, 6, 12, 30, 33), -5134, DateTime(1999, 7, 6, 11, 4, 59));
            testDT(DateTime(1999, 7, 6, 23, 0, 0), -432_123, DateTime(1999, 7, 1, 22, 57, 57));

            testDT(DateTime(1999, 7, 6, 12, 30, 0), 1, DateTime(1999, 7, 6, 12, 30, 1));
            testDT(DateTime(1999, 7, 6, 12, 30, 0), 0, DateTime(1999, 7, 6, 12, 30, 0));
            testDT(DateTime(1999, 7, 6, 12, 30, 0), -1, DateTime(1999, 7, 6, 12, 29, 59));

            testDT(DateTime(1999, 7, 6, 12, 0, 0), 1, DateTime(1999, 7, 6, 12, 0, 1));
            testDT(DateTime(1999, 7, 6, 12, 0, 0), 0, DateTime(1999, 7, 6, 12, 0, 0));
            testDT(DateTime(1999, 7, 6, 12, 0, 0), -1, DateTime(1999, 7, 6, 11, 59, 59));

            testDT(DateTime(1999, 7, 6, 0, 0, 0), 1, DateTime(1999, 7, 6, 0, 0, 1));
            testDT(DateTime(1999, 7, 6, 0, 0, 0), 0, DateTime(1999, 7, 6, 0, 0, 0));
            testDT(DateTime(1999, 7, 6, 0, 0, 0), -1, DateTime(1999, 7, 5, 23, 59, 59));

            testDT(DateTime(1999, 7, 5, 23, 59, 59), 1, DateTime(1999, 7, 6, 0, 0, 0));
            testDT(DateTime(1999, 7, 5, 23, 59, 59), 0, DateTime(1999, 7, 5, 23, 59, 59));
            testDT(DateTime(1999, 7, 5, 23, 59, 59), -1, DateTime(1999, 7, 5, 23, 59, 58));

            testDT(DateTime(1998, 12, 31, 23, 59, 59), 1, DateTime(1999, 1, 1, 0, 0, 0));
            testDT(DateTime(1998, 12, 31, 23, 59, 59), 0, DateTime(1998, 12, 31, 23, 59, 59));
            testDT(DateTime(1998, 12, 31, 23, 59, 59), -1, DateTime(1998, 12, 31, 23, 59, 58));

            testDT(DateTime(1998, 1, 1, 0, 0, 0), 1, DateTime(1998, 1, 1, 0, 0, 1));
            testDT(DateTime(1998, 1, 1, 0, 0, 0), 0, DateTime(1998, 1, 1, 0, 0, 0));
            testDT(DateTime(1998, 1, 1, 0, 0, 0), -1, DateTime(1997, 12, 31, 23, 59, 59));

            //Test B.C.
            testDT(DateTime(-1999, 7, 6, 12, 30, 33), 0, DateTime(-1999, 7, 6, 12, 30, 33));
            testDT(DateTime(-1999, 7, 6, 12, 30, 33), 1, DateTime(-1999, 7, 6, 12, 30, 34));
            testDT(DateTime(-1999, 7, 6, 12, 30, 33), 2, DateTime(-1999, 7, 6, 12, 30, 35));
            testDT(DateTime(-1999, 7, 6, 12, 30, 33), 3, DateTime(-1999, 7, 6, 12, 30, 36));
            testDT(DateTime(-1999, 7, 6, 12, 30, 33), 4, DateTime(-1999, 7, 6, 12, 30, 37));
            testDT(DateTime(-1999, 7, 6, 12, 30, 33), 5, DateTime(-1999, 7, 6, 12, 30, 38));
            testDT(DateTime(-1999, 7, 6, 12, 30, 33), 10, DateTime(-1999, 7, 6, 12, 30, 43));
            testDT(DateTime(-1999, 7, 6, 12, 30, 33), 15, DateTime(-1999, 7, 6, 12, 30, 48));
            testDT(DateTime(-1999, 7, 6, 12, 30, 33), 26, DateTime(-1999, 7, 6, 12, 30, 59));
            testDT(DateTime(-1999, 7, 6, 12, 30, 33), 27, DateTime(-1999, 7, 6, 12, 31, 0));
            testDT(DateTime(-1999, 7, 6, 12, 30, 33), 30, DateTime(-1999, 7, 6, 12, 31, 3));
            testDT(DateTime(-1999, 7, 6, 12, 30, 33), 59, DateTime(-1999, 7, 6, 12, 31, 32));
            testDT(DateTime(-1999, 7, 6, 12, 30, 33), 60, DateTime(-1999, 7, 6, 12, 31, 33));
            testDT(DateTime(-1999, 7, 6, 12, 30, 33), 61, DateTime(-1999, 7, 6, 12, 31, 34));

            testDT(DateTime(-1999, 7, 6, 12, 30, 33), 1766, DateTime(-1999, 7, 6, 12, 59, 59));
            testDT(DateTime(-1999, 7, 6, 12, 30, 33), 1767, DateTime(-1999, 7, 6, 13, 0, 0));
            testDT(DateTime(-1999, 7, 6, 12, 30, 33), 1768, DateTime(-1999, 7, 6, 13, 0, 1));
            testDT(DateTime(-1999, 7, 6, 12, 30, 33), 2007, DateTime(-1999, 7, 6, 13, 4, 0));
            testDT(DateTime(-1999, 7, 6, 12, 30, 33), 3599, DateTime(-1999, 7, 6, 13, 30, 32));
            testDT(DateTime(-1999, 7, 6, 12, 30, 33), 3600, DateTime(-1999, 7, 6, 13, 30, 33));
            testDT(DateTime(-1999, 7, 6, 12, 30, 33), 3601, DateTime(-1999, 7, 6, 13, 30, 34));
            testDT(DateTime(-1999, 7, 6, 12, 30, 33), 7200, DateTime(-1999, 7, 6, 14, 30, 33));
            testDT(DateTime(-1999, 7, 6, 23, 0, 0), 432_123, DateTime(-1999, 7, 11, 23, 2, 3));

            testDT(DateTime(-1999, 7, 6, 12, 30, 33), -1, DateTime(-1999, 7, 6, 12, 30, 32));
            testDT(DateTime(-1999, 7, 6, 12, 30, 33), -2, DateTime(-1999, 7, 6, 12, 30, 31));
            testDT(DateTime(-1999, 7, 6, 12, 30, 33), -3, DateTime(-1999, 7, 6, 12, 30, 30));
            testDT(DateTime(-1999, 7, 6, 12, 30, 33), -4, DateTime(-1999, 7, 6, 12, 30, 29));
            testDT(DateTime(-1999, 7, 6, 12, 30, 33), -5, DateTime(-1999, 7, 6, 12, 30, 28));
            testDT(DateTime(-1999, 7, 6, 12, 30, 33), -10, DateTime(-1999, 7, 6, 12, 30, 23));
            testDT(DateTime(-1999, 7, 6, 12, 30, 33), -15, DateTime(-1999, 7, 6, 12, 30, 18));
            testDT(DateTime(-1999, 7, 6, 12, 30, 33), -33, DateTime(-1999, 7, 6, 12, 30, 0));
            testDT(DateTime(-1999, 7, 6, 12, 30, 33), -34, DateTime(-1999, 7, 6, 12, 29, 59));
            testDT(DateTime(-1999, 7, 6, 12, 30, 33), -35, DateTime(-1999, 7, 6, 12, 29, 58));
            testDT(DateTime(-1999, 7, 6, 12, 30, 33), -59, DateTime(-1999, 7, 6, 12, 29, 34));
            testDT(DateTime(-1999, 7, 6, 12, 30, 33), -60, DateTime(-1999, 7, 6, 12, 29, 33));
            testDT(DateTime(-1999, 7, 6, 12, 30, 33), -61, DateTime(-1999, 7, 6, 12, 29, 32));

            testDT(DateTime(-1999, 7, 6, 12, 30, 33), -1833, DateTime(-1999, 7, 6, 12, 0, 0));
            testDT(DateTime(-1999, 7, 6, 12, 30, 33), -1834, DateTime(-1999, 7, 6, 11, 59, 59));
            testDT(DateTime(-1999, 7, 6, 12, 30, 33), -3600, DateTime(-1999, 7, 6, 11, 30, 33));
            testDT(DateTime(-1999, 7, 6, 12, 30, 33), -3601, DateTime(-1999, 7, 6, 11, 30, 32));
            testDT(DateTime(-1999, 7, 6, 12, 30, 33), -5134, DateTime(-1999, 7, 6, 11, 4, 59));
            testDT(DateTime(-1999, 7, 6, 12, 30, 33), -7200, DateTime(-1999, 7, 6, 10, 30, 33));
            testDT(DateTime(-1999, 7, 6, 23, 0, 0), -432_123, DateTime(-1999, 7, 1, 22, 57, 57));

            testDT(DateTime(-1999, 7, 6, 12, 30, 0), 1, DateTime(-1999, 7, 6, 12, 30, 1));
            testDT(DateTime(-1999, 7, 6, 12, 30, 0), 0, DateTime(-1999, 7, 6, 12, 30, 0));
            testDT(DateTime(-1999, 7, 6, 12, 30, 0), -1, DateTime(-1999, 7, 6, 12, 29, 59));

            testDT(DateTime(-1999, 7, 6, 12, 0, 0), 1, DateTime(-1999, 7, 6, 12, 0, 1));
            testDT(DateTime(-1999, 7, 6, 12, 0, 0), 0, DateTime(-1999, 7, 6, 12, 0, 0));
            testDT(DateTime(-1999, 7, 6, 12, 0, 0), -1, DateTime(-1999, 7, 6, 11, 59, 59));

            testDT(DateTime(-1999, 7, 6, 0, 0, 0), 1, DateTime(-1999, 7, 6, 0, 0, 1));
            testDT(DateTime(-1999, 7, 6, 0, 0, 0), 0, DateTime(-1999, 7, 6, 0, 0, 0));
            testDT(DateTime(-1999, 7, 6, 0, 0, 0), -1, DateTime(-1999, 7, 5, 23, 59, 59));

            testDT(DateTime(-1999, 7, 5, 23, 59, 59), 1, DateTime(-1999, 7, 6, 0, 0, 0));
            testDT(DateTime(-1999, 7, 5, 23, 59, 59), 0, DateTime(-1999, 7, 5, 23, 59, 59));
            testDT(DateTime(-1999, 7, 5, 23, 59, 59), -1, DateTime(-1999, 7, 5, 23, 59, 58));

            testDT(DateTime(-2000, 12, 31, 23, 59, 59), 1, DateTime(-1999, 1, 1, 0, 0, 0));
            testDT(DateTime(-2000, 12, 31, 23, 59, 59), 0, DateTime(-2000, 12, 31, 23, 59, 59));
            testDT(DateTime(-2000, 12, 31, 23, 59, 59), -1, DateTime(-2000, 12, 31, 23, 59, 58));

            testDT(DateTime(-2000, 1, 1, 0, 0, 0), 1, DateTime(-2000, 1, 1, 0, 0, 1));
            testDT(DateTime(-2000, 1, 1, 0, 0, 0), 0, DateTime(-2000, 1, 1, 0, 0, 0));
            testDT(DateTime(-2000, 1, 1, 0, 0, 0), -1, DateTime(-2001, 12, 31, 23, 59, 59));

            //Test Both
            testDT(DateTime(1, 1, 1, 0, 0, 0), -1, DateTime(0, 12, 31, 23, 59, 59));
            testDT(DateTime(0, 12, 31, 23, 59, 59), 1, DateTime(1, 1, 1, 0, 0, 0));

            testDT(DateTime(0, 1, 1, 0, 0, 0), -1, DateTime(-1, 12, 31, 23, 59, 59));
            testDT(DateTime(-1, 12, 31, 23, 59, 59), 1, DateTime(0, 1, 1, 0, 0, 0));

            testDT(DateTime(-1, 1, 1, 11, 30, 33), 63_165_600L, DateTime(1, 1, 1, 13, 30, 33));
            testDT(DateTime(1, 1, 1, 13, 30, 33), -63_165_600L, DateTime(-1, 1, 1, 11, 30, 33));

            testDT(DateTime(-1, 1, 1, 11, 30, 33), 63_165_617L, DateTime(1, 1, 1, 13, 30, 50));
            testDT(DateTime(1, 1, 1, 13, 30, 50), -63_165_617L, DateTime(-1, 1, 1, 11, 30, 33));

            const cdt = DateTime(1999, 7, 6, 12, 30, 33);
            immutable idt = DateTime(1999, 7, 6, 12, 30, 33);
            static assert(!__traits(compiles, cdt.addSeconds(4)));
            static assert(!__traits(compiles, idt.addSeconds(4)));
        }
    }


    Date      _date;
    TimeOfDay _tod;
}


//==============================================================================
// Section with intervals.
//==============================================================================

/++
    Represents an interval of time.

    An $(D Interval) has a starting point and an end point. The interval of time
    is therefore the time starting at the starting point up to, but not
    including, the end point. e.g.

    $(BOOKTABLE,
    $(TR $(TD [January 5th, 2010 - March 10th, 2010$(RPAREN)))
    $(TR $(TD [05:00:30 - 12:00:00$(RPAREN)))
    $(TR $(TD [1982-01-04T08:59:00 - 2010-07-04T12:00:00$(RPAREN)))
    )

    A range can be obtained from an $(D Interval), allowing iteration over
    that interval, with the exact time points which are iterated over depending
    on the function which generates the range.
  +/
struct Interval(TP)
{
public:

    /++
        Params:
            begin = The time point which begins the interval.
            end   = The time point which ends (but is not included in) the
                    interval.

        Throws:
            $(D DateTimeException) if $(D_PARAM end) is before $(D_PARAM begin).

        Examples:
--------------------
Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1));
--------------------
      +/
    this(U)(in TP begin, in U end) pure
        if(is(Unqual!TP == Unqual!U))
    {
        if(!_valid(begin, end))
            throw new DateTimeException("Arguments would result in an invalid Interval.");

        _begin = cast(TP)begin;
        _end = cast(TP)end;
    }


    /++
        Params:
            begin    = The time point which begins the interval.
            duration = The duration from the starting point to the end point.

        Throws:
            $(D DateTimeException) if the resulting $(D end) is before
            $(D begin).

        Examples:
--------------------
assert(Interval!Date(Date(1996, 1, 2), dur!"years"(3)) ==
       Interval!Date(Date(1996, 1, 2), Date(1999, 1, 2)));
--------------------
      +/
    this(D)(in TP begin, in D duration) pure
        if(__traits(compiles, begin + duration))
    {
        _begin = cast(TP)begin;
        _end = begin + duration;

        if(!_valid(_begin, _end))
            throw new DateTimeException("Arguments would result in an invalid Interval.");
    }


    /++
        Params:
            rhs = The $(D Interval) to assign to this one.
      +/
    /+ref+/ Interval opAssign(const ref Interval rhs) pure nothrow
    {
        _begin = cast(TP)rhs._begin;
        _end = cast(TP)rhs._end;

        return this;
    }


    /++
        Params:
            rhs = The $(D Interval) to assign to this one.
      +/
    /+ref+/ Interval opAssign(Interval rhs) pure nothrow
    {
        _begin = cast(TP)rhs._begin;
        _end = cast(TP)rhs._end;

        return this;
    }


    /++
        The starting point of the interval. It is included in the interval.

        Examples:
--------------------
assert(Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).begin ==
       Date(1996, 1, 2));
--------------------
      +/
    @property TP begin() const pure nothrow
    {
        return cast(TP)_begin;
    }


    /++
        The starting point of the interval. It is included in the interval.

        Params:
            timePoint = The time point to set $(D begin) to.

        Throws:
            $(D DateTimeException) if the resulting interval would be invalid.
      +/
    @property void begin(TP timePoint) pure
    {
        if(!_valid(timePoint, _end))
            throw new DateTimeException("Arguments would result in an invalid Interval.");

        _begin = timePoint;
    }


    /++
        The end point of the interval. It is excluded from the interval.

        Examples:
--------------------
assert(Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).end ==
       Date(2012, 3, 1));
--------------------
      +/
    @property TP end() const pure nothrow
    {
        return cast(TP)_end;
    }


    /++
        The end point of the interval. It is excluded from the interval.

        Params:
            timePoint = The time point to set end to.

        Throws:
            $(D DateTimeException) if the resulting interval would be invalid.
      +/
    @property void end(TP timePoint) pure
    {
        if(!_valid(_begin, timePoint))
            throw new DateTimeException("Arguments would result in an invalid Interval.");

        _end = timePoint;
    }


    /++
        Returns the duration between $(D begin) and $(D end).

        Examples:
--------------------
assert(Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).length ==
       dur!"days"(5903));
--------------------
      +/
    @property typeof(end - begin) length() const pure nothrow
    {
        return _end - _begin;
    }


    /++
        Whether the interval's length is 0, that is, whether $(D begin == end).

        Examples:
--------------------
assert(Interval!Date(Date(1996, 1, 2), Date(1996, 1, 2)).empty);
assert(!Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).empty);
--------------------
      +/
    @property bool empty() const pure nothrow
    {
        return _begin == _end;
    }


    /++
        Whether the given time point is within this interval.

        Params:
            timePoint = The time point to check for inclusion in this interval.

        Throws:
            $(D DateTimeException) if this interval is empty.

        Examples:
--------------------
assert(!Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).contains(
            Date(1994, 12, 24)));

assert(Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).contains(
            Date(2000, 1, 5)));
assert(!Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).contains(
            Date(2012, 3, 1)));
--------------------
      +/
    bool contains(in TP timePoint) const pure
    {
        _enforceNotEmpty();

        return timePoint >= _begin && timePoint < _end;
    }


    /++
        Whether the given interval is completely within this interval.

        Params:
            interval = The interval to check for inclusion in this interval.

        Throws:
            $(D DateTimeException) if either interval is empty.

        Examples:
--------------------
assert(!Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).contains(
            Interval!Date(Date(1990, 7, 6), Date(2000, 8, 2))));

assert(Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).contains(
            Interval!Date(Date(1999, 1, 12), Date(2011, 9, 17))));

assert(!Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).contains(
            Interval!Date(Date(1998, 2, 28), Date(2013, 5, 1))));
--------------------
      +/
    bool contains(in Interval interval) const pure
    {
        _enforceNotEmpty();
        interval._enforceNotEmpty();

        return interval._begin >= _begin &&
               interval._begin < _end &&
               interval._end <= _end;
    }


    /++
        Whether the given interval is completely within this interval.

        Always returns false (unless this interval is empty), because an
        interval going to positive infinity can never be contained in a finite
        interval.

        Params:
            interval = The interval to check for inclusion in this interval.

        Throws:
            $(D DateTimeException) if this interval is empty.

        Examples:
--------------------
assert(!Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).contains(
            PosInfInterval!Date(Date(1999, 5, 4))));
--------------------
      +/
    bool contains(in PosInfInterval!TP interval) const pure
    {
        _enforceNotEmpty();

        return false;
    }


    /++
        Whether the given interval is completely within this interval.

        Always returns false (unless this interval is empty), because an
        interval beginning at negative infinity can never be contained in a
        finite interval.

        Params:
            interval = The interval to check for inclusion in this interval.

        Throws:
            $(D DateTimeException) if this interval is empty.

        Examples:
--------------------
assert(!Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).contains(
            NegInfInterval!Date(Date(1996, 5, 4))));
--------------------
      +/
    bool contains(in NegInfInterval!TP interval) const pure
    {
        _enforceNotEmpty();

        return false;
    }


    /++
        Whether this interval is before the given time point.

        Params:
            timePoint = The time point to check whether this interval is before
                        it.

        Throws:
            $(D DateTimeException) if this interval is empty.

        Examples:
--------------------
assert(!Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).isBefore(
            Date(1994, 12, 24)));

assert(!Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).isBefore(
            Date(2000, 1, 5)));

assert(Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).isBefore(
            Date(2012, 3, 1)));
--------------------
      +/
    bool isBefore(in TP timePoint) const pure
    {
        _enforceNotEmpty();

        return _end <= timePoint;
    }


    /++
        Whether this interval is before the given interval and does not
        intersect with it.

        Params:
            interval = The interval to check for against this interval.

        Throws:
            $(D DateTimeException) if either interval is empty.

        Examples:
--------------------
assert(!Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).isBefore(
            Interval!Date(Date(1990, 7, 6), Date(2000, 8, 2))));

assert(!Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).isBefore(
            Interval!Date(Date(1999, 1, 12), Date(2011, 9, 17))));

assert(Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).isBefore(
            Interval!Date(Date(2012, 3, 1), Date(2013, 5, 1))));
--------------------
      +/
    bool isBefore(in Interval interval) const pure
    {
        _enforceNotEmpty();
        interval._enforceNotEmpty();

        return _end <= interval._begin;
    }


    /++
        Whether this interval is before the given interval and does not
        intersect with it.

        Params:
            interval = The interval to check for against this interval.

        Throws:
            $(D DateTimeException) if this interval is empty.

        Examples:
--------------------
assert(!Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).isBefore(
            PosInfInterval!Date(Date(1999, 5, 4))));

assert(Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).isBefore(
            PosInfInterval!Date(Date(2013, 3, 7))));
--------------------
      +/
    bool isBefore(in PosInfInterval!TP interval) const pure
    {
        _enforceNotEmpty();

        return _end <= interval._begin;
    }


    /++
        Whether this interval is before the given interval and does not
        intersect with it.

        Always returns false (unless this interval is empty) because a finite
        interval can never be before an interval beginning at negative infinity.

        Params:
            interval = The interval to check for against this interval.

        Throws:
            $(D DateTimeException) if this interval is empty.

        Examples:
--------------------
assert(!Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).isBefore(
            NegInfInterval!Date(Date(1996, 5, 4))));
--------------------
      +/
    bool isBefore(in NegInfInterval!TP interval) const pure
    {
        _enforceNotEmpty();

        return false;
    }


    /++
        Whether this interval is after the given time point.

        Params:
            timePoint = The time point to check whether this interval is after
                        it.

        Throws:
            $(D DateTimeException) if this interval is empty.

        Examples:
--------------------
assert(Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).isAfter(
            Date(1994, 12, 24)));

assert(!Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).isAfter(
            Date(2000, 1, 5)));

assert(!Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).isAfter(
            Date(2012, 3, 1)));
--------------------
      +/
    bool isAfter(in TP timePoint) const pure
    {
        _enforceNotEmpty();

        return timePoint < _begin;
    }


    /++
        Whether this interval is after the given interval and does not intersect
        it.

        Params:
            interval = The interval to check against this interval.

        Throws:
            $(D DateTimeException) if either interval is empty.

        Examples:
--------------------
assert(!Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).isAfter(
            Interval!Date(Date(1990, 7, 6), Date(2000, 8, 2))));

assert(!Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).isAfter(
            Interval!Date(Date(1999, 1, 12), Date(2011, 9, 17))));

assert(Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).isAfter(
            Interval!Date(Date(1989, 3, 1), Date(1996, 1, 2))));
--------------------
      +/
    bool isAfter(in Interval interval) const pure
    {
        _enforceNotEmpty();
        interval._enforceNotEmpty();

        return _begin >= interval._end;
    }


    /++
        Whether this interval is after the given interval and does not intersect
        it.

        Always returns false (unless this interval is empty) because a finite
        interval can never be after an interval going to positive infinity.

        Params:
            interval = The interval to check against this interval.

        Throws:
            $(D DateTimeException) if this interval is empty.

        Examples:
--------------------
assert(!Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).isAfter(
            PosInfInterval!Date(Date(1999, 5, 4))));
--------------------
      +/
    bool isAfter(in PosInfInterval!TP interval) const pure
    {
        _enforceNotEmpty();

        return false;
    }


    /++
        Whether this interval is after the given interval and does not intersect
        it.

        Params:
            interval = The interval to check against this interval.

        Throws:
            $(D DateTimeException) if this interval is empty.

        Examples:
--------------------
assert(Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).isAfter(
            NegInfInterval!Date(Date(1996, 1, 2))));
--------------------
      +/
    bool isAfter(in NegInfInterval!TP interval) const pure
    {
        _enforceNotEmpty();

        return _begin >= interval._end;
    }


    /++
        Whether the given interval overlaps this interval.

        Params:
            interval = The interval to check for intersection with this interval.

        Throws:
            $(D DateTimeException) if either interval is empty.

        Examples:
--------------------
assert(Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).intersects(
            Interval!Date(Date(1990, 7, 6), Date(2000, 8, 2))));

assert(Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).intersects(
            Interval!Date(Date(1999, 1, 12), Date(2011, 9, 17))));

assert(!Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).intersects(
            Interval!Date(Date(1989, 3, 1), Date(1996, 1, 2))));
--------------------
      +/
    bool intersects(in Interval interval) const pure
    {
        _enforceNotEmpty();
        interval._enforceNotEmpty();

        return interval._begin < _end && interval._end > _begin;
    }


    /++
        Whether the given interval overlaps this interval.

        Params:
            interval = The interval to check for intersection with this interval.

        Throws:
            $(D DateTimeException) if this interval is empty.

        Examples:
--------------------
assert(Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).intersects(
            PosInfInterval!Date(Date(1999, 5, 4))));

assert(!Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).intersects(
            PosInfInterval!Date(Date(2012, 3, 1))));
--------------------
      +/
    bool intersects(in PosInfInterval!TP interval) const pure
    {
        _enforceNotEmpty();

        return _end > interval._begin;
    }


    /++
        Whether the given interval overlaps this interval.

        Params:
            interval = The interval to check for intersection with this interval.

        Throws:
            $(D DateTimeException) if this interval is empty.

        Examples:
--------------------
assert(!Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).intersects(
            NegInfInterval!Date(Date(1996, 1, 2))));

assert(Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).intersects(
            NegInfInterval!Date(Date(2000, 1, 2))));
--------------------
      +/
    bool intersects(in NegInfInterval!TP interval) const pure
    {
        _enforceNotEmpty();

        return _begin < interval._end;
    }


    /++
        Returns the intersection of two intervals

        Params:
            interval = The interval to intersect with this interval.

        Throws:
            $(D DateTimeException) if the two intervals do not intersect or if
            either interval is empty.

        Examples:
--------------------
assert(Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).intersection(
            Interval!Date(Date(1990, 7, 6), Date(2000, 8, 2))) ==
       Interval!Date(Date(1996, 1 , 2), Date(2000, 8, 2)));

assert(Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).intersection(
            Interval!Date(Date(1999, 1, 12), Date(2011, 9, 17))) ==
       Interval!Date(Date(1999, 1 , 12), Date(2011, 9, 17)));
--------------------
      +/
    Interval intersection(in Interval interval) const
    {
        enforce(this.intersects(interval), new DateTimeException(format("%s and %s do not intersect.", this, interval)));

        auto begin = _begin > interval._begin ? _begin : interval._begin;
        auto end = _end < interval._end ? _end : interval._end;

        return Interval(begin, end);
    }


    /++
        Returns the intersection of two intervals

        Params:
            interval = The interval to intersect with this interval.

        Throws:
            $(D DateTimeException) if the two intervals do not intersect or if
            this interval is empty.

        Examples:
--------------------
assert(Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).intersection(
            PosInfInterval!Date(Date(1990, 7, 6))) ==
       Interval!Date(Date(1996, 1 , 2), Date(2012, 3, 1)));

assert(Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).intersection(
            PosInfInterval!Date(Date(1999, 1, 12))) ==
       Interval!Date(Date(1999, 1 , 12), Date(2012, 3, 1)));
--------------------
      +/
    Interval intersection(in PosInfInterval!TP interval) const
    {
        enforce(this.intersects(interval), new DateTimeException(format("%s and %s do not intersect.", this, interval)));

        return Interval(_begin > interval._begin ? _begin : interval._begin, _end);
    }


    /++
        Returns the intersection of two intervals

        Params:
            interval = The interval to intersect with this interval.

        Throws:
            $(D DateTimeException) if the two intervals do not intersect or if
            this interval is empty.

        Examples:
--------------------
assert(Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).intersection(
            NegInfInterval!Date(Date(1999, 7, 6))) ==
       Interval!Date(Date(1996, 1 , 2), Date(1999, 7, 6)));

assert(Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).intersection(
            NegInfInterval!Date(Date(2013, 1, 12))) ==
       Interval!Date(Date(1996, 1 , 2), Date(2012, 3, 1)));
--------------------
      +/
    Interval intersection(in NegInfInterval!TP interval) const
    {
        enforce(this.intersects(interval), new DateTimeException(format("%s and %s do not intersect.", this, interval)));

        return Interval(_begin, _end < interval._end ? _end : interval._end);
    }


    /++
        Whether the given interval is adjacent to this interval.

        Params:
            interval = The interval to check whether its adjecent to this
                       interval.

        Throws:
            $(D DateTimeException) if either interval is empty.

        Examples:
--------------------
assert(Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).isAdjacent(
            Interval!Date(Date(1990, 7, 6), Date(1996, 1, 2))));

assert(Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).isAdjacent(
            Interval!Date(Date(2012, 3, 1), Date(2013, 9, 17))));

assert(!Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).isAdjacent(
            Interval!Date(Date(1989, 3, 1), Date(2012, 3, 1))));
--------------------
      +/
    bool isAdjacent(in Interval interval) const pure
    {
        _enforceNotEmpty();
        interval._enforceNotEmpty();

        return _begin == interval._end || _end == interval._begin;
    }


    /++
        Whether the given interval is adjacent to this interval.

        Params:
            interval = The interval to check whether its adjecent to this
                       interval.

        Throws:
            $(D DateTimeException) if this interval is empty.

        Examples:
--------------------
assert(!Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).isAdjacent(
            PosInfInterval!Date(Date(1999, 5, 4))));

assert(Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).isAdjacent(
            PosInfInterval!Date(Date(2012, 3, 1))));
--------------------
      +/
    bool isAdjacent(in PosInfInterval!TP interval) const pure
    {
        _enforceNotEmpty();

        return _end == interval._begin;
    }


    /++
        Whether the given interval is adjacent to this interval.

        Params:
            interval = The interval to check whether its adjecent to this
                       interval.

        Throws:
            $(D DateTimeException) if this interval is empty.

        Examples:
--------------------
assert(Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).isAdjacent(
            NegInfInterval!Date(Date(1996, 1, 2))));

assert(!Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).isAdjacent(
            NegInfInterval!Date(Date(2000, 1, 2))));
--------------------
      +/
    bool isAdjacent(in NegInfInterval!TP interval) const pure
    {
        _enforceNotEmpty();

        return _begin == interval._end;
    }


    /++
        Returns the union of two intervals

        Params:
            interval = The interval to merge with this interval.

        Throws:
            $(D DateTimeException) if the two intervals do not intersect and are
            not adjacent or if either interval is empty.

        Examples:
--------------------
assert(Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).merge(
            Interval!Date(Date(1990, 7, 6), Date(2000, 8, 2))) ==
       Interval!Date(Date(1990, 7 , 6), Date(2012, 3, 1)));

assert(Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).merge(
            Interval!Date(Date(2012, 3, 1), Date(2013, 5, 7))) ==
       Interval!Date(Date(1996, 1 , 2), Date(2013, 5, 7)));
--------------------
      +/
    Interval merge(in Interval interval) const
    {
        enforce(this.isAdjacent(interval) || this.intersects(interval),
                new DateTimeException(format("%s and %s are not adjacent and do not intersect.", this, interval)));

        auto begin = _begin < interval._begin ? _begin : interval._begin;
        auto end = _end > interval._end ? _end : interval._end;

        return Interval(begin, end);
    }


    /++
        Returns the union of two intervals

        Params:
            interval = The interval to merge with this interval.

        Throws:
            $(D DateTimeException) if the two intervals do not intersect and are
            not adjacent or if this interval is empty.

        Examples:
--------------------
assert(Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).merge(
            PosInfInterval!Date(Date(1990, 7, 6))) ==
       PosInfInterval!Date(Date(1990, 7 , 6)));

assert(Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).merge(
            PosInfInterval!Date(Date(2012, 3, 1))) ==
       PosInfInterval!Date(Date(1996, 1 , 2)));
--------------------
      +/
    PosInfInterval!TP merge(in PosInfInterval!TP interval) const
    {
        enforce(this.isAdjacent(interval) || this.intersects(interval),
                new DateTimeException(format("%s and %s are not adjacent and do not intersect.", this, interval)));

        return PosInfInterval!TP(_begin < interval._begin ? _begin : interval._begin);
    }


    /++
        Returns the union of two intervals

        Params:
            interval = The interval to merge with this interval.

        Throws:
            $(D DateTimeException) if the two intervals do not intersect and are not
            adjacent or if this interval is empty.

        Examples:
--------------------
assert(Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).merge(
            NegInfInterval!Date(Date(1996, 1, 2))) ==
       NegInfInterval!Date(Date(2012, 3 , 1)));

assert(Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).merge(
            NegInfInterval!Date(Date(2013, 1, 12))) ==
       NegInfInterval!Date(Date(2013, 1 , 12)));
--------------------
      +/
    NegInfInterval!TP merge(in NegInfInterval!TP interval) const
    {
        enforce(this.isAdjacent(interval) || this.intersects(interval),
                new DateTimeException(format("%s and %s are not adjacent and do not intersect.", this, interval)));

        return NegInfInterval!TP(_end > interval._end ? _end : interval._end);
    }


    /++
        Returns an interval that covers from the earliest time point of two
        intervals up to (but not including) the latest time point of two
        intervals.

        Params:
            interval = The interval to create a span together with this interval.

        Throws:
            $(D DateTimeException) if either interval is empty.

        Examples:
--------------------
assert(Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).span(
            Interval!Date(Date(1990, 7, 6), Date(1991, 1, 8))) ==
       Interval!Date(Date(1990, 7 , 6), Date(2012, 3, 1)));

assert(Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).span(
            Interval!Date(Date(2012, 3, 1), Date(2013, 5, 7))) ==
       Interval!Date(Date(1996, 1 , 2), Date(2013, 5, 7)));
--------------------
      +/
    Interval span(in Interval interval) const pure
    {
        _enforceNotEmpty();
        interval._enforceNotEmpty();

        auto begin = _begin < interval._begin ? _begin : interval._begin;
        auto end = _end > interval._end ? _end : interval._end;

        return Interval(begin, end);
    }


    /++
        Returns an interval that covers from the earliest time point of two
        intervals up to (but not including) the latest time point of two
        intervals.

        Params:
            interval = The interval to create a span together with this interval.

        Throws:
            $(D DateTimeException) if this interval is empty.

        Examples:
--------------------
assert(Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).span(
            PosInfInterval!Date(Date(1990, 7, 6))) ==
       PosInfInterval!Date(Date(1990, 7 , 6)));

assert(Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).span(
            PosInfInterval!Date(Date(2050, 1, 1))) ==
       PosInfInterval!Date(Date(1996, 1 , 2)));
--------------------
      +/
    PosInfInterval!TP span(in PosInfInterval!TP interval) const pure
    {
        _enforceNotEmpty();

        return PosInfInterval!TP(_begin < interval._begin ? _begin : interval._begin);
    }


    /++
        Returns an interval that covers from the earliest time point of two
        intervals up to (but not including) the latest time point of two
        intervals.

        Params:
            interval = The interval to create a span together with this interval.

        Throws:
            $(D DateTimeException) if this interval is empty.

        Examples:
--------------------
assert(Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).span(
            NegInfInterval!Date(Date(1602, 5, 21))) ==
       NegInfInterval!Date(Date(2012, 3 , 1)));

assert(Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).span(
            NegInfInterval!Date(Date(2013, 1, 12))) ==
       NegInfInterval!Date(Date(2013, 1 , 12)));
--------------------
      +/
    NegInfInterval!TP span(in NegInfInterval!TP interval) const pure
    {
        _enforceNotEmpty();

        return NegInfInterval!TP(_end > interval._end ? _end : interval._end);
    }


    /++
        Shifts the interval forward or backwards in time by the given duration
        (a positive duration shifts the interval forward; a negative duration
        shifts it backward). Effectively, it does $(D begin += duration) and
        $(D end += duration).

        Params:
            duration = The duration to shift the interval by.

        Throws:
            $(D DateTimeException) this interval is empty or if the resulting
            interval would be invalid.

        Examples:
--------------------
auto interval1 = Interval!Date(Date(1996, 1, 2), Date(2012, 4, 5));
auto interval2 = Interval!Date(Date(1996, 1, 2), Date(2012, 4, 5));

interval1.shift(dur!"days"(50));
assert(interval1 == Interval!Date(Date(1996, 2, 21), Date(2012, 5, 25)));

interval2.shift(dur!"days"(-50));
assert(interval2 == Interval!Date(Date(1995, 11, 13), Date(2012, 2, 15)));
--------------------
      +/
    void shift(D)(D duration) pure
        if(__traits(compiles, begin + duration))
    {
        _enforceNotEmpty();

        auto begin = _begin + duration;
        auto end = _end + duration;

        if(!_valid(begin, end))
            throw new DateTimeException("Argument would result in an invalid Interval.");

        _begin = begin;
        _end = end;
    }


    static if(__traits(compiles, begin.add!"months"(1)) &&
              __traits(compiles, begin.add!"years"(1)))
    {
        /++
            Shifts the interval forward or backwards in time by the given number
            of years and/or months (a positive number of years and months shifts
            the interval forward; a negative number shifts it backward).
            It adds the years the given years and months to both begin and end.
            It effectively calls $(D add!"years"()) and then $(D add!"months"())
            on begin and end with the given number of years and months.

            Params:
                years         = The number of years to shift the interval by.
                months        = The number of months to shift the interval by.
                allowOverflow = Whether the days should be allowed to overflow
                                on $(D begin) and $(D end), causing their month
                                to increment.

            Throws:
                $(D DateTimeException) if this interval is empty or if the
                resulting interval would be invalid.

            Examples:
--------------------
auto interval1 = Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1));
auto interval2 = Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1));

interval1.shift(2);
assert(interval1 == Interval!Date(Date(1998, 1, 2), Date(2014, 3, 1)));

interval2.shift(-2);
assert(interval2 == Interval!Date(Date(1994, 1, 2), Date(2010, 3, 1)));
--------------------
          +/
        void shift(T)(T years, T months = 0, AllowDayOverflow allowOverflow = AllowDayOverflow.yes)
            if(isIntegral!T)
        {
            _enforceNotEmpty();

            auto begin = _begin;
            auto end = _end;

            begin.add!"years"(years, allowOverflow);
            begin.add!"months"(months, allowOverflow);
            end.add!"years"(years, allowOverflow);
            end.add!"months"(months, allowOverflow);

            enforce(_valid(begin, end), new DateTimeException("Argument would result in an invalid Interval."));

            _begin = begin;
            _end = end;
        }
    }


    /++
        Expands the interval forwards and/or backwards in time. Effectively,
        it does $(D begin -= duration) and/or $(D end += duration). Whether
        it expands forwards and/or backwards in time is determined by
        $(D_PARAM dir).

        Params:
            duration = The duration to expand the interval by.
            dir      = The direction in time to expand the interval.

        Throws:
            $(D DateTimeException) this interval is empty or if the resulting
            interval would be invalid.

        Examples:
--------------------
auto interval1 = Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1));
auto interval2 = Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1));

interval1.expand(2);
assert(interval1 == Interval!Date(Date(1994, 1, 2), Date(2014, 3, 1)));

interval2.expand(-2);
assert(interval2 == Interval!Date(Date(1998, 1, 2), Date(2010, 3, 1)));
--------------------
      +/
    void expand(D)(D duration, Direction dir = Direction.both) pure
        if(__traits(compiles, begin + duration))
    {
        _enforceNotEmpty();

        switch(dir)
        {
            case Direction.both:
            {
                auto begin = _begin - duration;
                auto end = _end + duration;

                if(!_valid(begin, end))
                    throw new DateTimeException("Argument would result in an invalid Interval.");

                _begin = begin;
                _end = end;

                return;
            }
            case Direction.fwd:
            {
                auto end = _end + duration;

                if(!_valid(_begin, end))
                    throw new DateTimeException("Argument would result in an invalid Interval.");
                _end = end;

                return;
            }
            case Direction.bwd:
            {
                auto begin = _begin - duration;

                if(!_valid(begin, _end))
                    throw new DateTimeException("Argument would result in an invalid Interval.");
                _begin = begin;

                return;
            }
            default:
                assert(0, "Invalid Direction.");
        }
    }

    static if(__traits(compiles, begin.add!"months"(1)) &&
              __traits(compiles, begin.add!"years"(1)))
    {
        /++
            Expands the interval forwards and/or backwards in time. Effectively,
            it subtracts the given number of months/years from $(D begin) and
            adds them to $(D end). Whether it expands forwards and/or backwards
            in time is determined by $(D_PARAM dir).

            Params:
                years         = The number of years to expand the interval by.
                months        = The number of months to expand the interval by.
                allowOverflow = Whether the days should be allowed to overflow
                                on $(D begin) and $(D end), causing their month
                                to increment.

            Throws:
                $(D DateTimeException) if this interval is empty or if the
                resulting interval would be invalid.

            Examples:
--------------------
auto interval1 = Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1));
auto interval2 = Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1));

interval1.expand(2);
assert(interval1 == Interval!Date(Date(1994, 1, 2), Date(2014, 3, 1)));

interval2.expand(-2);
assert(interval2 == Interval!Date(Date(1998, 1, 2), Date(2010, 3, 1)));
--------------------
          +/
        void expand(T)(T years, T months = 0, AllowDayOverflow allowOverflow = AllowDayOverflow.yes, Direction dir = Direction.both)
            if(isIntegral!T)
        {
            _enforceNotEmpty();

            switch(dir)
            {
                case Direction.both:
                {
                    auto begin = _begin;
                    auto end = _end;

                    begin.add!"years"(-years, allowOverflow);
                    begin.add!"months"(-months, allowOverflow);
                    end.add!"years"(years, allowOverflow);
                    end.add!"months"(months, allowOverflow);

                    enforce(_valid(begin, end), new DateTimeException("Argument would result in an invalid Interval."));
                    _begin = begin;
                    _end = end;

                    return;
                }
                case Direction.fwd:
                {
                    auto end = _end;

                    end.add!"years"(years, allowOverflow);
                    end.add!"months"(months, allowOverflow);

                    enforce(_valid(_begin, end), new DateTimeException("Argument would result in an invalid Interval."));
                    _end = end;

                    return;
                }
                case Direction.bwd:
                {
                    auto begin = _begin;

                    begin.add!"years"(-years, allowOverflow);
                    begin.add!"months"(-months, allowOverflow);

                    enforce(_valid(begin, _end), new DateTimeException("Argument would result in an invalid Interval."));
                    _begin = begin;

                    return;
                }
                default:
                    assert(0, "Invalid Direction.");
            }
        }
    }


    /++
        Returns a range which iterates forward over the interval, starting
        at $(D begin), using $(D_PARAM func) to generate each successive time
        point.

        The range's $(D front) is the interval's $(D begin). $(D_PARAM func) is
        used to generate the next $(D front) when $(D popFront) is called. If
        $(D_PARAM popFirst) is $(D PopFirst.yes), then $(D popFront) is called
        before the range is returned (so that $(D front) is a time point which
        $(D_PARAM func) would generate).

        If $(D_PARAM func) ever generates a time point less than or equal to the
        current $(D front) of the range, then a $(D DateTimeException) will be
        thrown. The range will be empty and iteration complete when
        $(D_PARAM func) generates a time point equal to or beyond the $(D end)
        of the interval.

        There are helper functions in this module which generate common
        delegates to pass to $(D fwdRange). Their documentation starts with
        "Range-generating function," making them easily searchable.

        Params:
            func     = The function used to generate the time points of the
                       range over the interval.
            popFirst = Whether $(D popFront) should be called on the range
                       before returning it.

        Throws:
            $(D DateTimeException) if this interval is empty.

        Warning:
            $(D_PARAM func) must be logically pure. Ideally, $(D_PARAM func)
            would be a function pointer to a pure function, but forcing
            $(D_PARAM func) to be pure is far too restrictive to be useful, and
            in order to have the ease of use of having functions which generate
            functions to pass to $(D fwdRange), $(D_PARAM func) must be a
            delegate.

            If $(D_PARAM func) retains state which changes as it is called, then
            some algorithms will not work correctly, because the range's
            $(D save) will have failed to have really saved the range's state.
            To avoid such bugs, don't pass a delegate which is
            not logically pure to $(D fwdRange). If $(D_PARAM func) is given the
            same time point with two different calls, it must return the same
            result both times.

            Of course, none of the functions in this module have this problem,
            so it's only relevant if when creating a custom delegate.

        Examples:
--------------------
auto interval = Interval!Date(Date(2010, 9, 1), Date(2010, 9, 9));
auto func = (in Date date) //For iterating over even-numbered days.
            {
                if((date.day & 1) == 0)
                    return date + dur!"days"(2);

                return date + dur!"days"(1);
            };
auto range = interval.fwdRange(func);

 //An odd day. Using PopFirst.yes would have made this Date(2010, 9, 2).
assert(range.front == Date(2010, 9, 1));

range.popFront();
assert(range.front == Date(2010, 9, 2));

range.popFront();
assert(range.front == Date(2010, 9, 4));

range.popFront();
assert(range.front == Date(2010, 9, 6));

range.popFront();
assert(range.front == Date(2010, 9, 8));

range.popFront();
assert(range.empty);
--------------------
      +/
    IntervalRange!(TP, Direction.fwd) fwdRange(TP delegate(in TP) func, PopFirst popFirst = PopFirst.no) const
    {
        _enforceNotEmpty();

        auto range = IntervalRange!(TP, Direction.fwd)(this, func);

        if(popFirst == PopFirst.yes)
            range.popFront();

        return range;
    }


    /++
        Returns a range which iterates backwards over the interval, starting
        at $(D end), using $(D_PARAM func) to generate each successive time
        point.

        The range's $(D front) is the interval's $(D end). $(D_PARAM func) is
        used to generate the next $(D front) when $(D popFront) is called. If
        $(D_PARAM popFirst) is $(D PopFirst.yes), then $(D popFront) is called
        before the range is returned (so that $(D front) is a time point which
        $(D_PARAM func) would generate).

        If $(D_PARAM func) ever generates a time point greater than or equal to
        the current $(D front) of the range, then a $(D DateTimeException) will
        be thrown. The range will be empty and iteration complete when
        $(D_PARAM func) generates a time point equal to or less than the
        $(D begin) of the interval.

        There are helper functions in this module which generate common
        delegates to pass to $(D bwdRange). Their documentation starts with
        "Range-generating function," making them easily searchable.

        Params:
            func     = The function used to generate the time points of the
                       range over the interval.
            popFirst = Whether $(D popFront) should be called on the range
                       before returning it.

        Throws:
            $(D DateTimeException) if this interval is empty.

        Warning:
            $(D_PARAM func) must be logically pure. Ideally, $(D_PARAM func)
            would be a function pointer to a pure function, but forcing
            $(D_PARAM func) to be pure is far too restrictive to be useful, and
            in order to have the ease of use of having functions which generate
            functions to pass to $(D fwdRange), $(D_PARAM func) must be a
            delegate.

            If $(D_PARAM func) retains state which changes as it is called, then
            some algorithms will not work correctly, because the range's
            $(D save) will have failed to have really saved the range's state.
            To avoid such bugs, don't pass a delegate which is
            not logically pure to $(D fwdRange). If $(D_PARAM func) is given the
            same time point with two different calls, it must return the same
            result both times.

            Of course, none of the functions in this module have this problem,
            so it's only relevant for custom delegates.

        Examples:
--------------------
auto interval = Interval!Date(Date(2010, 9, 1), Date(2010, 9, 9));
auto func = (in Date date) //For iterating over even-numbered days.
            {
                if((date.day & 1) == 0)
                    return date - dur!"days"(2);

                return date - dur!"days"(1);
            };
auto range = interval.bwdRange(func);

//An odd day. Using PopFirst.yes would have made this Date(2010, 9, 8).
assert(range.front == Date(2010, 9, 9));

range.popFront();
assert(range.front == Date(2010, 9, 8));

range.popFront();
assert(range.front == Date(2010, 9, 6));

range.popFront();
assert(range.front == Date(2010, 9, 4));

range.popFront();
assert(range.front == Date(2010, 9, 2));

range.popFront();
assert(range.empty);
--------------------
      +/
    IntervalRange!(TP, Direction.bwd) bwdRange(TP delegate(in TP) func, PopFirst popFirst = PopFirst.no) const
    {
        _enforceNotEmpty();

        auto range = IntervalRange!(TP, Direction.bwd)(this, func);

        if(popFirst == PopFirst.yes)
            range.popFront();

        return range;
    }


    /+
        Converts this interval to a string.
      +/
    //Due to bug http://d.puremagic.com/issues/show_bug.cgi?id=3715 , we can't
    //have versions of toString() with extra modifiers, so we define one version
    //with modifiers and one without.
    string toString()
    {
        return _toStringImpl();
    }


    /++
        Converts this interval to a string.
      +/
    //Due to bug http://d.puremagic.com/issues/show_bug.cgi?id=3715 , we can't
    //have versions of toString() with extra modifiers, so we define one version
    //with modifiers and one without.
    string toString() const nothrow
    {
        return _toStringImpl();
    }


private:

    /+
        Since we have two versions of toString, we have _toStringImpl
        so that they can share implementations.
      +/
    string _toStringImpl() const nothrow
    {
        try
            return format("[%s - %s)", _begin, _end);
        catch(Exception e)
            assert(0, "format() threw.");
    }


    /+
        Throws:
            $(D DateTimeException) if this interval is empty.
      +/
    void _enforceNotEmpty(size_t line = __LINE__) const pure
    {
        if(empty)
            throw new DateTimeException("Invalid operation for an empty Interval.", __FILE__, line);
    }


    /+
        Whether the given values form a valid time interval.

        Params:
            begin = The starting point of the interval.
            end   = The end point of the interval.
     +/
    static bool _valid(in TP begin, in TP end) pure nothrow
    {
        return begin <= end;
    }


    pure invariant()
    {
        assert(_valid(_begin, _end), "Invariant Failure: begin is not before or equal to end.");
    }


    TP _begin;
    TP _end;
}

//Test Interval's constructors.
unittest
{
    version(testStdDateTime)
    {
        assertThrown!DateTimeException(Interval!Date(Date(2010, 1, 1), Date(1, 1, 1)));

        Interval!Date(Date.init, Date.init);
        Interval!TimeOfDay(TimeOfDay.init, TimeOfDay.init);
        Interval!DateTime(DateTime.init, DateTime.init);
        Interval!SysTime(SysTime(0), SysTime(0));

        Interval!DateTime(DateTime.init, dur!"days"(7));

        //Verify Examples.
        Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1));
        assert(Interval!Date(Date(1996, 1, 2), dur!"weeks"(3)) == Interval!Date(Date(1996, 1, 2), Date(1996, 1, 23)));
        assert(Interval!Date(Date(1996, 1, 2), dur!"days"(3)) == Interval!Date(Date(1996, 1, 2), Date(1996, 1, 5)));
        assert(Interval!DateTime(DateTime(1996, 1, 2, 12, 0, 0), dur!"hours"(3)) == Interval!DateTime(DateTime(1996, 1, 2, 12, 0, 0), DateTime(1996, 1, 2, 15, 0, 0)));
        assert(Interval!DateTime(DateTime(1996, 1, 2, 12, 0, 0), dur!"minutes"(3)) == Interval!DateTime(DateTime(1996, 1, 2, 12, 0, 0), DateTime(1996, 1, 2, 12, 3, 0)));
        assert(Interval!DateTime(DateTime(1996, 1, 2, 12, 0, 0), dur!"seconds"(3)) == Interval!DateTime(DateTime(1996, 1, 2, 12, 0, 0), DateTime(1996, 1, 2, 12, 0, 3)));
        assert(Interval!DateTime(DateTime(1996, 1, 2, 12, 0, 0), dur!"msecs"(3000)) == Interval!DateTime(DateTime(1996, 1, 2, 12, 0, 0), DateTime(1996, 1, 2, 12, 0, 3)));
    }
}

//Test Interval's begin.
unittest
{
    version(testStdDateTime)
    {
        _assertPred!"=="(Interval!Date(Date(1, 1, 1), Date(2010, 1, 1)).begin, Date(1, 1, 1));
        _assertPred!"=="(Interval!Date(Date(2010, 1, 1), Date(2010, 1, 1)).begin, Date(2010, 1, 1));
        _assertPred!"=="(Interval!Date(Date(1997, 12, 31), Date(1998, 1, 1)).begin, Date(1997, 12, 31));

        const cInterval = Interval!Date(Date(2010, 7, 4), Date(2012, 1, 7));
        immutable iInterval = Interval!Date(Date(2010, 7, 4), Date(2012, 1, 7));
        static assert(__traits(compiles, cInterval.begin));
        static assert(__traits(compiles, iInterval.begin));

        //Verify Examples.
        assert(Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).begin == Date(1996, 1, 2));
    }
}

//Test Interval's end.
unittest
{
    version(testStdDateTime)
    {
        _assertPred!"=="(Interval!Date(Date(1, 1, 1), Date(2010, 1, 1)).end, Date(2010, 1, 1));
        _assertPred!"=="(Interval!Date(Date(2010, 1, 1), Date(2010, 1, 1)).end, Date(2010, 1, 1));
        _assertPred!"=="(Interval!Date(Date(1997, 12, 31), Date(1998, 1, 1)).end, Date(1998, 1, 1));

        const cInterval = Interval!Date(Date(2010, 7, 4), Date(2012, 1, 7));
        immutable iInterval = Interval!Date(Date(2010, 7, 4), Date(2012, 1, 7));
        static assert(__traits(compiles, cInterval.end));
        static assert(__traits(compiles, iInterval.end));

        //Verify Examples.
        assert(Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).end == Date(2012, 3, 1));
    }
}

//Test Interval's length.
unittest
{
    version(testStdDateTime)
    {
        _assertPred!"=="(Interval!Date(Date(2010, 1, 1), Date(2010, 1, 1)).length, dur!"days"(0));
        _assertPred!"=="(Interval!Date(Date(2010, 1, 1), Date(2010, 4, 1)).length, dur!"days"(90));
        _assertPred!"=="(Interval!TimeOfDay(TimeOfDay(0, 30, 0), TimeOfDay(12, 22, 7)).length, dur!"seconds"(42_727));
        _assertPred!"=="(Interval!DateTime(DateTime(2010, 1, 1, 0, 30, 0), DateTime(2010, 1, 2, 12, 22, 7)).length, dur!"seconds"(129_127));
        _assertPred!"=="(Interval!SysTime(SysTime(DateTime(2010, 1, 1, 0, 30, 0)), SysTime(DateTime(2010, 1, 2, 12, 22, 7))).length, dur!"seconds"(129_127));

        const cInterval = Interval!Date(Date(2010, 7, 4), Date(2012, 1, 7));
        immutable iInterval = Interval!Date(Date(2010, 7, 4), Date(2012, 1, 7));
        static assert(__traits(compiles, cInterval.length));
        static assert(__traits(compiles, iInterval.length));

        //Verify Examples.
        assert(Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).length == dur!"days"(5903));
    }
}

//Test Interval's empty.
unittest
{
    version(testStdDateTime)
    {
        assert(Interval!Date(Date(2010, 1, 1), Date(2010, 1, 1)).empty);
        assert(!Interval!Date(Date(2010, 1, 1), Date(2010, 4, 1)).empty);
        assert(!Interval!TimeOfDay(TimeOfDay(0, 30, 0), TimeOfDay(12, 22, 7)).empty);
        assert(!Interval!DateTime(DateTime(2010, 1, 1, 0, 30, 0), DateTime(2010, 1, 2, 12, 22, 7)).empty);
        assert(!Interval!SysTime(SysTime(DateTime(2010, 1, 1, 0, 30, 0)), SysTime(DateTime(2010, 1, 2, 12, 22, 7))).empty);

        const cInterval = Interval!Date(Date(2010, 7, 4), Date(2012, 1, 7));
        immutable iInterval = Interval!Date(Date(2010, 7, 4), Date(2012, 1, 7));
        static assert(__traits(compiles, cInterval.empty));
        static assert(__traits(compiles, iInterval.empty));

        //Verify Examples.
        assert(Interval!Date(Date(1996, 1, 2), Date(1996, 1, 2)).empty);
        assert(!Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).empty);
    }
}

//Test Interval's contains(time point).
unittest
{
    version(testStdDateTime)
    {
        auto interval = Interval!Date(Date(2010, 7, 4), Date(2012, 1, 7));

        assertThrown!DateTimeException(Interval!Date(Date(2010, 7, 4), dur!"days"(0)).contains(Date(2010, 7, 4)));

        assert(!interval.contains(Date(2009, 7, 4)));
        assert(!interval.contains(Date(2010, 7, 3)));
        assert(interval.contains(Date(2010, 7, 4)));
        assert(interval.contains(Date(2010, 7, 5)));
        assert(interval.contains(Date(2011, 7, 1)));
        assert(interval.contains(Date(2012, 1, 6)));
        assert(!interval.contains(Date(2012, 1, 7)));
        assert(!interval.contains(Date(2012, 1, 8)));
        assert(!interval.contains(Date(2013, 1, 7)));

        const cdate = Date(2010, 7, 6);
        const cInterval = Interval!Date(Date(2010, 7, 4), Date(2012, 1, 7));
        immutable iInterval = Interval!Date(Date(2010, 7, 4), Date(2012, 1, 7));
        static assert(__traits(compiles, interval.contains(cdate)));
        static assert(__traits(compiles, cInterval.contains(cdate)));
        static assert(__traits(compiles, iInterval.contains(cdate)));

        //Verify Examples.
        assert(!Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).contains(Date(1994, 12, 24)));
        assert(Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).contains(Date(2000, 1, 5)));
        assert(!Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).contains(Date(2012, 3, 1)));
    }
}

//Test Interval's contains(Interval).
unittest
{
    version(testStdDateTime)
    {
        auto interval = Interval!Date(Date(2010, 7, 4), Date(2012, 1, 7));

        assertThrown!DateTimeException(interval.contains(Interval!Date(Date(2010, 7, 4), dur!"days"(0))));
        assertThrown!DateTimeException(Interval!Date(Date(2010, 7, 4), dur!"days"(0)).contains(interval));
        assertThrown!DateTimeException(Interval!Date(Date(2010, 7, 4), dur!"days"(0)).contains(Interval!Date(Date(2010, 7, 4), dur!"days"(0))));

        assert(interval.contains(interval));
        assert(!interval.contains(Interval!Date(Date(2010, 7, 1), Date(2010, 7, 3))));
        assert(!interval.contains(Interval!Date(Date(2010, 7, 1), Date(2013, 7, 3))));
        assert(!interval.contains(Interval!Date(Date(2010, 7, 3), Date(2010, 7, 4))));
        assert(!interval.contains(Interval!Date(Date(2010, 7, 3), Date(2010, 7, 5))));
        assert(!interval.contains(Interval!Date(Date(2010, 7, 3), Date(2012, 1, 7))));
        assert(!interval.contains(Interval!Date(Date(2010, 7, 3), Date(2012, 1, 8))));
        assert(interval.contains(Interval!Date(Date(2010, 7, 5), Date(2012, 1, 6))));
        assert(interval.contains(Interval!Date(Date(2010, 7, 5), Date(2012, 1, 7))));
        assert(interval.contains(Interval!Date(Date(2012, 1, 6), Date(2012, 1, 7))));
        assert(!interval.contains(Interval!Date(Date(2012, 1, 6), Date(2012, 1, 8))));
        assert(!interval.contains(Interval!Date(Date(2012, 1, 7), Date(2012, 1, 8))));
        assert(!interval.contains(Interval!Date(Date(2012, 1, 8), Date(2012, 1, 9))));

        assert(!Interval!Date(Date(2010, 7, 1), Date(2010, 7, 3)).contains(interval));
        assert(Interval!Date(Date(2010, 7, 1), Date(2013, 7, 3)).contains(interval));
        assert(!Interval!Date(Date(2010, 7, 3), Date(2010, 7, 4)).contains(interval));
        assert(!Interval!Date(Date(2010, 7, 3), Date(2010, 7, 5)).contains(interval));
        assert(Interval!Date(Date(2010, 7, 3), Date(2012, 1, 7)).contains(interval));
        assert(Interval!Date(Date(2010, 7, 3), Date(2012, 1, 8)).contains(interval));
        assert(!Interval!Date(Date(2010, 7, 5), Date(2012, 1, 6)).contains(interval));
        assert(!Interval!Date(Date(2010, 7, 5), Date(2012, 1, 7)).contains(interval));
        assert(!Interval!Date(Date(2012, 1, 6), Date(2012, 1, 7)).contains(interval));
        assert(!Interval!Date(Date(2012, 1, 6), Date(2012, 1, 8)).contains(interval));
        assert(!Interval!Date(Date(2012, 1, 7), Date(2012, 1, 8)).contains(interval));
        assert(!Interval!Date(Date(2012, 1, 8), Date(2012, 1, 9)).contains(interval));

        assert(!interval.contains(PosInfInterval!Date(Date(2010, 7, 3))));
        assert(!interval.contains(PosInfInterval!Date(Date(2010, 7, 4))));
        assert(!interval.contains(PosInfInterval!Date(Date(2010, 7, 5))));
        assert(!interval.contains(PosInfInterval!Date(Date(2012, 1, 6))));
        assert(!interval.contains(PosInfInterval!Date(Date(2012, 1, 7))));
        assert(!interval.contains(PosInfInterval!Date(Date(2012, 1, 8))));

        assert(!interval.contains(NegInfInterval!Date(Date(2010, 7, 3))));
        assert(!interval.contains(NegInfInterval!Date(Date(2010, 7, 4))));
        assert(!interval.contains(NegInfInterval!Date(Date(2010, 7, 5))));
        assert(!interval.contains(NegInfInterval!Date(Date(2012, 1, 6))));
        assert(!interval.contains(NegInfInterval!Date(Date(2012, 1, 7))));
        assert(!interval.contains(NegInfInterval!Date(Date(2012, 1, 8))));

        const cInterval = Interval!Date(Date(2010, 7, 4), Date(2012, 1, 7));
        immutable iInterval = Interval!Date(Date(2010, 7, 4), Date(2012, 1, 7));
        auto posInfInterval = PosInfInterval!Date(Date(2010, 7, 4));
        const cPosInfInterval = PosInfInterval!Date(Date(2010, 7, 4));
        immutable iPosInfInterval = PosInfInterval!Date(Date(2010, 7, 4));
        auto negInfInterval = NegInfInterval!Date(Date(2012, 1, 7));
        const cNegInfInterval = NegInfInterval!Date(Date(2012, 1, 7));
        immutable iNegInfInterval = NegInfInterval!Date(Date(2012, 1, 7));
        static assert(__traits(compiles, interval.contains(interval)));
        static assert(__traits(compiles, interval.contains(cInterval)));
        static assert(__traits(compiles, interval.contains(iInterval)));
        static assert(__traits(compiles, interval.contains(posInfInterval)));
        static assert(__traits(compiles, interval.contains(cPosInfInterval)));
        static assert(__traits(compiles, interval.contains(iPosInfInterval)));
        static assert(__traits(compiles, interval.contains(negInfInterval)));
        static assert(__traits(compiles, interval.contains(cNegInfInterval)));
        static assert(__traits(compiles, interval.contains(iNegInfInterval)));
        static assert(__traits(compiles, cInterval.contains(interval)));
        static assert(__traits(compiles, cInterval.contains(cInterval)));
        static assert(__traits(compiles, cInterval.contains(iInterval)));
        static assert(__traits(compiles, cInterval.contains(posInfInterval)));
        static assert(__traits(compiles, cInterval.contains(cPosInfInterval)));
        static assert(__traits(compiles, cInterval.contains(iPosInfInterval)));
        static assert(__traits(compiles, cInterval.contains(negInfInterval)));
        static assert(__traits(compiles, cInterval.contains(cNegInfInterval)));
        static assert(__traits(compiles, cInterval.contains(iNegInfInterval)));
        static assert(__traits(compiles, iInterval.contains(interval)));
        static assert(__traits(compiles, iInterval.contains(cInterval)));
        static assert(__traits(compiles, iInterval.contains(iInterval)));
        static assert(__traits(compiles, iInterval.contains(posInfInterval)));
        static assert(__traits(compiles, iInterval.contains(cPosInfInterval)));
        static assert(__traits(compiles, iInterval.contains(iPosInfInterval)));
        static assert(__traits(compiles, iInterval.contains(negInfInterval)));
        static assert(__traits(compiles, iInterval.contains(cNegInfInterval)));
        static assert(__traits(compiles, iInterval.contains(iNegInfInterval)));

        //Verify Examples.
        assert(!Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).contains(Interval!Date(Date(1990, 7, 6), Date(2000, 8, 2))));
        assert(Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).contains(Interval!Date(Date(1999, 1, 12), Date(2011, 9, 17))));
        assert(!Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).contains(Interval!Date(Date(1998, 2, 28), Date(2013, 5, 1))));

        assert(!Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).contains(PosInfInterval!Date(Date(1999, 5, 4))));

        assert(!Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).contains(NegInfInterval!Date(Date(1996, 5, 4))));
    }
}

//Test Interval's isBefore(time point).
unittest
{
    version(testStdDateTime)
    {
        auto interval = Interval!Date(Date(2010, 7, 4), Date(2012, 1, 7));

        assertThrown!DateTimeException(Interval!Date(Date(2010, 7, 4), dur!"days"(0)).isBefore(Date(2010, 7, 4)));

        assert(!interval.isBefore(Date(2009, 7, 3)));
        assert(!interval.isBefore(Date(2010, 7, 3)));
        assert(!interval.isBefore(Date(2010, 7, 4)));
        assert(!interval.isBefore(Date(2010, 7, 5)));
        assert(!interval.isBefore(Date(2011, 7, 1)));
        assert(!interval.isBefore(Date(2012, 1, 6)));
        assert(interval.isBefore(Date(2012, 1, 7)));
        assert(interval.isBefore(Date(2012, 1, 8)));
        assert(interval.isBefore(Date(2013, 1, 7)));

        const cdate = Date(2010, 7, 6);
        const cInterval = Interval!Date(Date(2010, 7, 4), Date(2012, 1, 7));
        immutable iInterval = Interval!Date(Date(2010, 7, 4), Date(2012, 1, 7));
        static assert(__traits(compiles, interval.isBefore(cdate)));
        static assert(__traits(compiles, cInterval.isBefore(cdate)));
        static assert(__traits(compiles, iInterval.isBefore(cdate)));

        //Verify Examples.
        assert(!Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).isBefore(Date(1994, 12, 24)));
        assert(!Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).isBefore(Date(2000, 1, 5)));
        assert(Interval!Date(Date(1996, 1, 2), Date(2012, 3, 1)).isBefore(Date(2012, 3, 1)));
    }
}

//Test Interval's isBefore(Interval).
unittest
{
    version(testStdDateTime)
    {
        auto interval = Interval!Date(Date(2010, 7, 4), Date(2012, 1, 7));

        assertThrown!DateTimeException(interval.isBefore(Interval!Date(Date(2010, 7, 4), dur!"days"(0))));
        assertThrown!DateTimeException(Interval!Date(Date(2010, 7, 4), dur!"days"(0)).isBefore(interval));
        assertThrown!DateTimeException(Interval!Date(Date(2010, 7, 4), dur!"days"(0)).isBefore(Interval!Date(Date(2010, 7, 4), dur!"days"(0))));

        assert(!interval.isBefore(interval));
        assert(!interval.isBefore(Interval!Date(Date(2010, 7, 1), Date(2010, 7, 3))));
        assert(!interval.isBefore(Interval!Date(Date(2010, 7, 1), Date(2013, 7, 3))));
        assert(!interval.isBefore(Interval!Date(Date(2010, 7, 3), Date(2010, 7, 4))));
        assert(!interval.isBefore(Interval!Date(Date(2010, 7, 3), Date(2010, 7, 5))));
        assert(!interval.isBefore(Interval!Date(Date(2010, 7, 3), Date(2012, 1, 7))));
        assert(!interval.isBefore(Interval!Date(Date(2010, 7, 3), Date(2012, 1, 8))));
        assert(!interval.isBefore(Interval!Date(Date(2010, 7, 5), Date(2012, 1, 6))));
        assert(!interval.isBefore(Interval!Date(Date(2010, 7, 5), Date(2012, 1, 7))));
        assert(!interval.isBefore(Interval!Date(Date(2012, 1, 6), Date(2012, 1, 7))));
        assert(!interval.isBefore(Interval!Date(Date(2012, 1, 6), Date(2012, 1, 8))));
        assert(interval.isBefore(Interval!Date(Date(2012, 1, 7), Date(2012, 1, 8))));
        assert(interval.isBefore(Interval!Date(Date(2012, 1, 8), Date(2012, 1, 9))));

        assert(Interval!Date(Date(2010, 7, 1), Date(2010, 7, 3)).isBefore(interval));
        assert(!Interval!Date(Date(2010, 7, 1), Date(2013, 7, 3)).isBefore(interval));
        assert(Interval!Date(Date(2010, 7, 3), Date(2010, 7, 4)).isBefore(interval));
        assert(!Interval!Date(Date(2010, 7, 3), Date(2010, 7, 5)).isBefore(interval));
        assert(!Interval!Date(Date(2010, 7, 3), Date(2012, 1, 7)).isBefore(interval));
        assert(!Interval!Date(Date(2010, 7, 3), Date(2012, 1, 8)).isBefore(interval));
        assert(!Interval!Date(Date(2010, 7, 5), Date(2012, 1, 6)).isBefore(interval));
        assert(!Interval!Date(Date(2010, 7, 5), Date(2012, 1, 7)).isBefore(interval));
        assert(!Interval!Date(Date(2012, 1, 6), Date(2012, 1, 7)).isBefore(interval));
        asse