/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/* 
 * Linux Desktop Testing Project http://ldtp.freedesktop.org
 *
 * Author:
 *    S. Aishwariya <aishwariyabhavan@yahoo.com >
 *    K. Sree Kamakshi <poorvakrishna@yahoo.com >
 *    Nagappan Alagappan <nagappan@gmail.com>
 *
 * Copyright 2004 - 2006 Novell, Inc.
 * Copyright 2007 - 2008 Nagappan Alagappan
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110, USA.
 */

#include "ldtp.h"
#include "ldtp-gui.h"
#include "ldtp-utils.h"
#include "ldtp-error.h"
#include "ldtp-logger.h"
#include "ldtp-command.h"
#include "ldtp-gui-comp.h"

static gboolean
is_object_slider (Accessible *object, FILE *log_fp)
{
	if (wait_till_object_state_contains (object, SLIDER, log_fp) != 0) {
		LDTPErrorCode error;
		error = LDTP_ERROR_INVALID_OBJECT_STATE;
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
		return FALSE;
	}
	return TRUE;
}

static gboolean
is_slider_horizontal (Accessible *object)
{
	AccessibleStateSet *state;
	state = Accessible_getStateSet (object);
	/*
	  Check if the slider is horizontal or vertical
	*/
	if (AccessibleStateSet_contains (state, SPI_STATE_HORIZONTAL))
		return TRUE;  //Horizontal State
	else
		return FALSE; //Vertical State
}

static gboolean
is_slider_vertical (Accessible *object)
{
	AccessibleStateSet *state;
	state = Accessible_getStateSet (object);
	/*
	  Check if the slider is horizontal or vertical
	*/
	if (AccessibleStateSet_contains (state, SPI_STATE_VERTICAL))
		return TRUE;  //Vertical State
	else
		return FALSE; //Not vertical State
}

static LDTPErrorCode
set_value (Accessible *object, long numvalue, FILE *log_fp)
{
	LDTPErrorCode error;
	SPIBoolean flag = FALSE;
	AccessibleValue *accessible_value;

	g_print ("Slider value to be set: %ld\n", numvalue);
	accessible_value = Accessible_getValue (object);
	if (accessible_value) {
		flag = AccessibleValue_setCurrentValue (accessible_value, numvalue);
		Accessible_unref (accessible_value);
	}
	if (flag)
		return LDTP_ERROR_SUCCESS;
	else {
		error = LDTP_ERROR_UNABLE_TO_SET_SLIDER_VALUE;
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
		return error;
	}
}

static LDTPErrorCode
set_max (Accessible *object, FILE *log_fp)
{ 
	double val;
	SPIBoolean flag = FALSE;
	LDTPErrorCode error;
	AccessibleValue *value = NULL;

	if (is_object_slider (object, log_fp) == TRUE) {
		value = Accessible_getValue (object);
	} else {
		error = LDTP_ERROR_SLIDER_SET_MAX_FAILED;
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
		return error;
	}

	/*
	  The slider is set to maximum value
	*/
	if (value) {
		val = AccessibleValue_getMaximumValue (value); 
		flag = AccessibleValue_setCurrentValue (value, val);
		Accessible_unref (value);
	}
	if (flag == TRUE)
		return LDTP_ERROR_SUCCESS;
	else {
		error = LDTP_ERROR_SLIDER_SET_MAX_FAILED;
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
		return error;
	}
}

static LDTPErrorCode
set_min (Accessible *object, FILE *log_fp)
{ 
	double val;
	AccessibleValue *value = NULL;
	LDTPErrorCode error;
	SPIBoolean flag;

	if (is_object_slider (object, log_fp) == FALSE) {
		error = LDTP_ERROR_SLIDER_SET_MIN_FAILED;
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
		return error;
	}
	value = Accessible_getValue(object);    
	val = AccessibleValue_getMinimumValue (value); 

	/*
	  The slider is set to minimum value
	*/
	flag = AccessibleValue_setCurrentValue (value, val);
	Accessible_unref (value);
	if (flag == TRUE)
		return LDTP_ERROR_SUCCESS;
	else {
		error = LDTP_ERROR_SLIDER_SET_MIN_FAILED;
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
		return error;
	}
}

static double
get_slider_value (Accessible *object, FILE *log_fp)
{
	double current_val = -1;
	if (is_object_slider (object, log_fp)) {
		AccessibleValue *value = NULL;
		value = Accessible_getValue (object);
		if (value) {
			current_val = AccessibleValue_getCurrentValue (value);
			Accessible_unref (value);
		}
	}
	return current_val;
}

static double
get_min_value (Accessible *object)
{
	double value = -1;
	AccessibleValue *accessible;
	if (!object)
		return -1;
	accessible = Accessible_getValue (object);
	if (accessible) {
		value = AccessibleValue_getMinimumValue (accessible);
		Accessible_unref (accessible);
	}
	return value;
}

static double
get_min_increment (Accessible *object)
{
	double value = -1;
	AccessibleValue *accessible;
	if (!object)
		return -1;
	accessible = Accessible_getValue (object);
	if (accessible) {
		value = AccessibleValue_getMinimumIncrement (accessible);
		Accessible_unref (accessible);
	}
	return value;
}

static double
get_max_value (Accessible *object)
{
	double value = -1;
	AccessibleValue *accessible;
	if (!object)
		return -1;
	accessible = Accessible_getValue (object);
	if (accessible) {
		value = AccessibleValue_getMaximumValue (accessible);
		Accessible_unref (accessible);
	}
	return value;
}

static LDTPErrorCode
decrease (Accessible *object, long num, FILE *log_fp)
{
	double val;
	long i = 1;
	AccessibleValue *value = NULL;
	SPIBoolean flag = FALSE;
	if (is_object_slider (object, log_fp)) {
		value = Accessible_getValue (object);
		if (!value)
			return LDTP_ERROR_UNABLE_TO_DECREASE_SLIDER_VALUE;
		while (i <= num) {    
			val = AccessibleValue_getCurrentValue (value) - 1;
			/*
			  Decrease the value of the slider
			*/
 
			flag = AccessibleValue_setCurrentValue (value, val);
			ldtp_nsleep (0, 15000);
			i++;
		}
	}
	Accessible_unref (value);
	if (flag == TRUE)
		return LDTP_ERROR_SUCCESS;
	else {
		LDTPErrorCode error;
		error = LDTP_ERROR_UNABLE_TO_DECREASE_SLIDER_VALUE;
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
		return error;
	}
}

static LDTPErrorCode
increase (Accessible *object, long num, FILE *log_fp)
{
	double val;
	long i = 1;
	AccessibleValue *value = NULL;
	SPIBoolean flag = FALSE;
	if (is_object_slider (object, log_fp)) {
		value = Accessible_getValue (object);
		if (!value)
			return LDTP_ERROR_UNABLE_TO_DECREASE_SLIDER_VALUE;
		while (i <= num) {    
			val = AccessibleValue_getCurrentValue (value) + 1;
			/*
			  Increase the value of the slider
			*/
 
			flag = AccessibleValue_setCurrentValue (value, val);
			ldtp_nsleep (0, 15000);
			i++;
		}
	}
	Accessible_unref (value);
	if (flag == TRUE)
		return LDTP_ERROR_SUCCESS;
	else {
		LDTPErrorCode error;
		error = LDTP_ERROR_UNABLE_TO_INCREASE_SLIDER_VALUE;
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
		return error;
	}
}

LDTPErrorCode
slider_main (LDTPClientContext* cctxt, int command)
{
	double value = -1;
	LDTPErrorCode error;
	switch (command) {
	case LDTP_CMD_SETMAX:
		error = set_max (cctxt->gui_handle->handle, cctxt->log_fp);
		break;
	case LDTP_CMD_SETMIN:
		error = set_min (cctxt->gui_handle->handle, cctxt->log_fp);
		break;
	case LDTP_CMD_SETVALUE:
		if (cctxt->req->arg_list && cctxt->req->arg_list->data)
			value = strtod (cctxt->req->arg_list->data, NULL);
		error = set_value (cctxt->gui_handle->handle, value,
				   cctxt->log_fp);
		break;
	case LDTP_CMD_GETSLIDERVALUE: {
		double current_val = -1;
		current_val = get_slider_value (cctxt->gui_handle->handle, cctxt->log_fp);
		if (current_val != -1) {
			cctxt->resp->data = g_strdup_printf ("%lf", current_val);
			cctxt->resp->data_len = g_utf8_strlen (cctxt->resp->data, -1);
			error = LDTP_ERROR_SUCCESS;
		}
		else {
			error = LDTP_ERROR_UNABLE_TO_GET_SLIDER_VALUE;
			log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), cctxt->log_fp);
		}
		break;
	}
	case LDTP_CMD_INCREASE: {
		long num = 0;
		if (cctxt->req->arg_list && cctxt->req->arg_list->data)
			num = atol (cctxt->req->arg_list->data);
		error = increase (cctxt->gui_handle->handle, num, cctxt->log_fp);
		break;
	}
	case LDTP_CMD_DECREASE: {
		long num = 0;
		if (cctxt->req->arg_list && cctxt->req->arg_list->data)
			num = atol (cctxt->req->arg_list->data);
		error = decrease(cctxt->gui_handle->handle, num, cctxt->log_fp);
		break;
	}
	case LDTP_CMD_GETMINVALUE:
		value = get_min_value (cctxt->gui_handle->handle);
		cctxt->resp->data = g_strdup_printf ("%lf", value);
		if (cctxt->resp->data)
			cctxt->resp->data_len = g_utf8_strlen (cctxt->resp->data, -1);
		if (value != -1)
			error = LDTP_ERROR_SUCCESS;
		else
			error = LDTP_ERROR_UNABLE_TO_GET_VALUE;
		break;
#ifdef ENABLE_NEWROLES
	// NOT supported by SLE10
	case LDTP_CMD_GETMININCREMENT:
		value = get_min_increment (cctxt->gui_handle->handle);
		cctxt->resp->data = g_strdup_printf ("%lf", value);
		if (cctxt->resp->data)
			cctxt->resp->data_len = g_utf8_strlen (cctxt->resp->data, -1);
		if (value != -1)
			error = LDTP_ERROR_SUCCESS;
		else
			error = LDTP_ERROR_UNABLE_TO_GET_VALUE;
		break;
#endif
	case LDTP_CMD_GETMAXVALUE:
		value = get_max_value (cctxt->gui_handle->handle);
		cctxt->resp->data = g_strdup_printf ("%lf", value);
		if (cctxt->resp->data)
			cctxt->resp->data_len = g_utf8_strlen (cctxt->resp->data, -1);
		if (value != -1)
			error = LDTP_ERROR_SUCCESS;
		else
			error = LDTP_ERROR_UNABLE_TO_GET_VALUE;
		break;
	case LDTP_CMD_GETOBJECTSIZE:
		cctxt->resp->data_len = 0;
		cctxt->resp->data = get_size (cctxt->gui_handle->handle, &error);
		if (cctxt->resp->data) {
			cctxt->resp->data_len = g_utf8_strlen (cctxt->resp->data, -1);
		}
		break;
	case LDTP_CMD_VERIFYSLIDER:
		if (is_object_slider (cctxt->gui_handle->handle, cctxt->log_fp))
			error = LDTP_ERROR_SUCCESS;
		else
			error = LDTP_ERROR_INVALID_OBJECT_STATE;
		break;
	case LDTP_CMD_VERIFYSLIDERVERTICAL:
		if (is_slider_vertical (cctxt->gui_handle->handle))
			error = LDTP_ERROR_SUCCESS;
		else
			error = LDTP_ERROR_NOT_VERTICAL_SLIDER;
		break;
	case LDTP_CMD_VERIFYSLIDERHORIZONTAL:
		if (is_slider_horizontal (cctxt->gui_handle->handle))
			error = LDTP_ERROR_SUCCESS;
		else
			error = LDTP_ERROR_NOT_HORIZONTAL_SLIDER;
		break;
	default:
		error = LDTP_ERROR_COMMAND_NOT_IMPLEMENTED;
	}
	return error;
}
