#!/usr/bin/python
# -*- coding: utf-8 -*-
### BEGIN LICENSE
# Copyright (C) 2009 Jono Bacon <jono@ubuntu.com>
# Copyright (C) 2010 Michael Budde <mbudde@gmail.com>
#
#This program is free software: you can redistribute it and/or modify it
#under the terms of the GNU General Public License version 3, as published
#by the Free Software Foundation.
#
#This program is distributed in the hope that it will be useful, but
#WITHOUT ANY WARRANTY; without even the implied warranties of
#MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
#PURPOSE.  See the GNU General Public License for more details.
#
#You should have received a copy of the GNU General Public License along
#with this program.  If not, see <http://www.gnu.org/licenses/>.
### END LICENSE


from desktopcouch.records.server import CouchDatabase
from desktopcouch.records.record import Record


class Preferences(object):

    _instance = None

    def __init__(self):
        self._db_name = 'lernid'
        self._database = CouchDatabase(self._db_name, create=True)
        self._record_type = "http://wiki.ubuntu.com/Quickly/RecordTypes/Lernid/Preferences"
        self._key = None
        self._load_preferences()

    def _load_preferences(self):
        self._preferences = {
            'record_type': self._record_type,
            'vertical': False,
            'show_appindicator': True,
            'main_window_maximize': False,
            'statusbar': True,
            'nick': '',
            'show_irc_time': False
        }
        results = self._database.get_records(record_type=self._record_type, create_view=True)
        if len(results.rows) == 0:
            #no preferences have ever been saved
            #save them before returning
            self._key = self._database.put_record(Record(self._preferences))
        else:
            prefs = results.rows[0].value
            self._preferences.update(prefs)
            self._key = prefs["_id"]
            self._database.update_fields(self._key, self._preferences)

    def _update(self, prefs):
        if '_id' in prefs: del prefs['_id']
        if '_rev' in prefs: del prefs['_rev']
        self._preferences.update(prefs)
        self._database.update_fields(self._key, prefs)

    @classmethod
    def get_instance(cls):
        if not cls._instance:
            cls._instance = cls()
        return cls._instance

    @classmethod
    def get(cls, name=None, default=None):
        """Return dict with preferences if @name is not given, else return
        the value of preference @name. If @name is not set, return @default
        which defaults to None.
        """
        klass = cls.get_instance()
        if name:
            return klass._preferences.get(name, default)
        return klass._preferences

    @classmethod
    def set(cls, *args, **kwargs):
        """set(dict) -> updates preferences with values from dict
        set(name, value) -> updates a single preference
        set(key=val, key=val, ...) -> updates preferences with key-value pairs
        """
        klass = cls.get_instance()
        if len(args) == 1:
            d = args[0]
        elif len(args) == 2:
            name, value = args
            d = {name: value}
        elif len(args) == 0 and kwargs:
            d = kwargs
        else:
            raise ValueError('set should be called as either set(key, val), '
                             'set(dict) or set(key=val, key=val, ...)')
        klass._update(d)

if __name__ == '__main__':
    print Preferences.get('width')
    Preferences.set('width', 100)
    print Preferences.get('width')
    Preferences.set(width=200, height=300)
    print Preferences.get()
    Preferences.set({'width': 500, 'height': 200})
    print Preferences.get()
