const rollup = require('rollup');
const typescript = require('rollup-plugin-typescript2');
const resolve = require('rollup-plugin-node-resolve');
const commonjs = require('rollup-plugin-commonjs');
const terser = require('rollup-plugin-terser').terser;
const banner = require('./banner');
const path = require('path');

const rootPath = path.join(__dirname, '..');

const args = require('minimist')(process.argv.slice(2));

let outDir = args.dist ? './dist' : './tmp';

async function buildBrowser() {
    let bundle = await rollup.rollup({
        input: './lib/less-browser/bootstrap.js',
        output: [
            {
                file: 'less.js',
                format: 'umd'
            },
            {
                file: 'less.min.js',
                format: 'umd'
            }
        ],
        plugins: [
            // resolve(),
            commonjs(),
            typescript({
                verbosity: 2,
                tsconfigDefaults: {
                    compilerOptions: {
                        allowJs: true,
                        sourceMap: true,
                        target: 'ES5'
                    }
                },
                include: [ '*.ts', '**/*.ts', '*.js', '**/*.js' ],
                exclude: ['node_modules'] // only transpile our source code
            }),
            terser({
                compress: true,
                include: [/^.+\.min\.js$/],
                output: {
                    comments: function(node, comment) {
                        if (comment.type == "comment2") {
                            // preserve banner
                            return /@license/i.test(comment.value);
                        }
                    }
                }
            })
        ]
    });

    if (!args.out || args.out.indexOf('less.js') > -1) {
        const file = args.out || `${outDir}/less.js`;
        console.log(`Writing ${file}...`);
        await bundle.write({
            file: path.join(rootPath, file),
            format: 'umd',
            name: 'less',
            banner
        }); 
    }

    if (!args.out || args.out.indexOf('less.min.js') > -1) {
        const file = args.out || `${outDir}/less.min.js`;
        console.log(`Writing ${file}...`);
        await bundle.write({
            file: path.join(rootPath, file),
            format: 'umd',
            name: 'less',
            sourcemap: true,
            banner
        });
    }
}

async function buildNode() {
    let bundle = await rollup.rollup({
        input: './lib/less-node/index.js',
        external(id) {
            return /^[^.]/.test(id)
        },
        plugins: [
            resolve(),
            commonjs(),
            typescript({
                verbosity: 1,
                tsconfigDefaults: {
                    compilerOptions: {
                        allowJs: true,
                        // checkJs: true,
                        sourceMap: true,
                        target: 'ES5'
                    },
                },
                include: [ '*.ts', '**/*.ts', '*.js', '**/*.js' ],
                exclude: ['node_modules'] // only transpile our source code
            })
        ]
    });

    const file = args.out || './dist/less.cjs.js';
    console.log(`Writing ${file}...`);

    await bundle.write({
        file: path.join(rootPath, file),
        format: 'cjs',
        interop: false
    });
}

async function buildLessC() {
    let bundle = await rollup.rollup({
        input: './lib/lessc.js',
        external(id) {
            return /^[^.]/.test(id)
        },
        plugins: [
            resolve(),
            commonjs(),
            typescript({
                verbosity: 1,
                tsconfigDefaults: {
                    compilerOptions: {
                        allowJs: true,
                        // checkJs: true,
                        sourceMap: true,
                        target: 'ES5'
                    },
                },
                include: [ '*.ts', '**/*.ts', '*.js', '**/*.js' ],
                exclude: ['node_modules'] // only transpile our source code
            })
        ]
    });

    const file = args.out || './bin/lessc';
    console.log(`Writing ${file}...`);

    await bundle.write({
        file: path.join(rootPath, file),
        banner: '#!/usr/bin/env node\n// AUTO-GENERATED BY ROLLUP\n',
        format: 'cjs',
        interop: false
    });
}

async function build() {
    if (args.dist || args.lessc) {
        await buildLessC();
    }
    if (args.dist || args.browser) {
        await buildBrowser();
    }
    if (args.dist || args.node) {
        await buildNode();
    }
}

build();