/***********************************************************************************

	Copyright (C) 2007-2011 Ahmet Öztürk (aoz_2@yahoo.com)

	This file is part of Lifeograph.

	Lifeograph is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Lifeograph is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "lifeobase.hpp"
#include "view_entry.hpp"
#include "printing.hpp"


using namespace LIFEO;


EntryView *Lifeobase::view_entry = NULL;


EntryView::EntryView( void )
:	m_tag_operation_cur( TO_NONE ), m_entry_sync( NULL ),
	m_flag_entrychanged( false )
{
	Entry::shower = this;

    Gtk::MenuItem *menuitem_print;

    Gtk::MenuItem *menuitem_hide( NULL );
    Gtk::MenuItem *menuitem_hide_before( NULL );
    Gtk::MenuItem *menuitem_hide_after( NULL );

	try
	{
		Lifeobase::builder->get_widget( "vbox_entry", m_vbox );
        Lifeobase::builder->get_widget_derived( "menu_entry_elem", m_menu );
        Lifeobase::builder->get_widget( "menuitem_entry_favorite", m_menuitem_toggle_favorite );
        Lifeobase::builder->get_widget( "menuseparator_entry_1", m_menuseparator_1 );
        Lifeobase::builder->get_widget( "menuitem_entry_print", menuitem_print );
        Lifeobase::builder->get_widget( "menuitem_entry_filter", menuitem_hide );
        Lifeobase::builder->get_widget( "menuitem_entry_filter_before", menuitem_hide_before );
        Lifeobase::builder->get_widget( "menuitem_entry_filter_after", menuitem_hide_after );
        Lifeobase::builder->get_widget( "menuitem_entry_reset_theme", m_menuitem_reset_theme );
        Lifeobase::builder->get_widget( "menuseparator_entry_3", m_menuseparator_2 );
        Lifeobase::builder->get_widget( "menuitem_entry_trash", m_menuitem_trash );
        Lifeobase::builder->get_widget( "menuitem_entry_restore", m_menuitem_restore );
        Lifeobase::builder->get_widget( "menuitem_entry_dismiss", m_menuitem_dismiss );
		Lifeobase::builder->get_widget( "button_entry_bold", m_button_bold );
		Lifeobase::builder->get_widget( "button_entry_italic", m_button_italic );
		Lifeobase::builder->get_widget( "button_entry_strikethrough", m_button_strikethrough );
		Lifeobase::builder->get_widget( "button_entry_highlight", m_button_highlight );
		Lifeobase::builder->get_widget( "label_entry_highlight", m_label_highlight );
		Lifeobase::builder->get_widget( "evntb_entry_highlight", m_eventbox_highlight );
		Lifeobase::builder->get_widget( "button_entry_add_bullet", m_button_add_bullet );
		Lifeobase::builder->get_widget( "button_entry_add_checkbox", m_button_add_checkbox );
        Lifeobase::builder->get_widget( "tgglbutton_entry_comments", m_button_toggle_comments );
		Lifeobase::builder->get_widget_derived( "textview_entry", m_textview );
		Lifeobase::builder->get_widget_derived( "entry_tag_filter", m_tag_widget );
		Lifeobase::builder->get_widget( "button_tag_operation", m_button_tag_operation );
		Lifeobase::builder->get_widget_derived( "drawingarea_entry_tags", m_widgettaglist );
		Lifeobase::builder->get_widget( "hbox_entry_toolbar_text", m_hbox_editing_toolbar );
        Lifeobase::builder->get_widget( "box_entry_tag_edit", m_box_tag_edit );
	}
	catch( ... ) {}

	// TEXT VIEW
	m_textview->drag_dest_get_target_list()->add( Lifeobase::base->drag_targets_tag );
	m_textview->drag_dest_get_target_list()->add( Lifeobase::base->drag_targets_entry );

    // TAGS WIDGET
    m_widgettaglist->drag_dest_set( Lifeobase::base->drag_targets_tag );

	// SIGNALS
    m_textview->m_buffer->signal_changed().connect(
			sigc::mem_fun( this, &LIFEO::EntryView::handle_textview_changed ) );

    m_menuitem_toggle_favorite->signal_activate().connect(
            sigc::mem_fun( this, &LIFEO::EntryView::toggle_favoredness ) );
    m_menuitem_reset_theme->signal_activate().connect(
            sigc::mem_fun( this, &LIFEO::EntryView::reset_theme ) );
    m_menuitem_trash->signal_activate().connect(
            sigc::mem_fun( this, &LIFEO::EntryView::trash_entry ) );
    m_menuitem_restore->signal_activate().connect(
            sigc::mem_fun( this, &LIFEO::EntryView::restore_entry ) );
    m_menuitem_dismiss->signal_activate().connect(
            sigc::mem_fun( this, &LIFEO::EntryView::dismiss_entry ) );
    menuitem_print->signal_activate().connect(
            sigc::mem_fun( this, &EntryView::print ) );
    menuitem_hide->signal_activate().connect( sigc::mem_fun( this, &EntryView::hide ) );
    menuitem_hide_before->signal_activate().connect(
            sigc::mem_fun( this, &EntryView::hide_before ) );
    menuitem_hide_after->signal_activate().connect(
            sigc::mem_fun( this, &EntryView::hide_after ) );

	m_button_add_bullet->signal_clicked().connect(
			sigc::mem_fun( m_textview->m_buffer, &TextbufferDiary::add_bullet ) );
	m_button_add_checkbox->signal_clicked().connect(
			sigc::mem_fun( m_textview->m_buffer, &TextbufferDiary::add_checkbox ) );

	m_textview->signal_drag_data_received().connect(
			sigc::mem_fun( this, &EntryView::handle_drag_data_received ) );

	m_tag_widget->signal_changed().connect(
			sigc::mem_fun( this, &EntryView::handle_entry_tag_changed ) );
	m_textview->signal_focus_in_event().connect_notify(
            sigc::mem_fun( this, &EntryView::handle_tag_entry_focus_out ) );
	m_tag_widget->signal_activate().connect(
			sigc::mem_fun( this, &EntryView::handle_button_tag_clicked ) );
	m_button_tag_operation->signal_clicked().connect(
			sigc::mem_fun( this, &EntryView::handle_button_tag_clicked ) );
    m_widgettaglist->signal_drag_data_received().connect(
            sigc::mem_fun( this, &EntryView::handle_drag_data_received ) );
    m_widgettaglist->signal_add_tag().connect(
            sigc::mem_fun( this, &EntryView::focus_tag_entry ) );
    m_widgettaglist->signal_tag_selected().connect(
            sigc::mem_fun( this, &EntryView::edit_tag_in_entry ) );

    // EDITOR ACTIONS
    Lifeobase::create_action(
            true, m_action_bold, "Bold", "", "", "",
            Gtk::AccelKey( GDK_KEY_b, Gdk::CONTROL_MASK ),
            sigc::mem_fun( m_textview->m_buffer, &TextbufferDiary::toggle_bold ),
            m_button_bold );
    Lifeobase::create_action(
            true, m_action_italic, "Italic", "", "", "",
            Gtk::AccelKey( GDK_KEY_i, Gdk::CONTROL_MASK ),
            sigc::mem_fun( m_textview->m_buffer, &TextbufferDiary::toggle_italic ),
            m_button_italic );
    Lifeobase::create_action(
            true, m_action_strikethrough, "Strikethrough", "", "", "",
            Gtk::AccelKey( GDK_KEY_s, Gdk::CONTROL_MASK ),
            sigc::mem_fun( m_textview->m_buffer, &TextbufferDiary::toggle_strikethrough ),
            m_button_strikethrough );
    Lifeobase::create_action(
            true, m_action_highlight, "Highlight", "", "", "",
            Gtk::AccelKey( GDK_KEY_h, Gdk::CONTROL_MASK ),
            sigc::mem_fun( m_textview->m_buffer, &TextbufferDiary::toggle_highlight ),
            m_button_highlight );

    Lifeobase::create_action(
            true, m_action_focus_tag, "FocusTag", "", "", "",
            Gtk::AccelKey( GDK_KEY_t, Gdk::CONTROL_MASK ),
            sigc::mem_fun( this, &EntryView::focus_tag_entry ) );

    Lifeobase::create_toggle_action(
            false, m_action_toggle_comments, "ToggleComments", "", "", "",
            Gtk::AccelKey( GDK_KEY_c, Gdk::MOD1_MASK ),
            sigc::mem_fun( this, &EntryView::toggle_comments ),
            m_button_toggle_comments );

    Lifeobase::create_action(
            true, m_action_indent, "Indent", "", "", "",
            Gtk::AccelKey( GDK_KEY_i, Gdk::MOD1_MASK ),
            sigc::mem_fun( m_textview->m_buffer, &TextbufferDiary::handle_indent ) );
    Lifeobase::create_action(
            true, m_action_unindent, "Unindent", "", "", "",
            Gtk::AccelKey( GDK_KEY_u, Gdk::MOD1_MASK ),
            sigc::mem_fun( m_textview->m_buffer, &TextbufferDiary::handle_unindent ) );

    Lifeobase::create_action(
            true, m_action_add_empty_line_above, "AddEmptyLineAbove", "", "", "",
            Gtk::AccelKey( GDK_KEY_l, Gdk::MOD1_MASK ),
            sigc::mem_fun( m_textview->m_buffer, &TextbufferDiary::add_empty_line_above ) );
    Lifeobase::create_action(
            true, m_action_remove_empty_line_above, "RemoveEmptyLineAbove", "", "", "",
            Gtk::AccelKey( GDK_KEY_l, Gdk::MOD1_MASK|Gdk::SHIFT_MASK ),
            sigc::mem_fun( m_textview->m_buffer, &TextbufferDiary::remove_empty_line_above ) );

    Lifeobase::create_action(
            true, m_action_move_line_up, "MoveLineUp", "", "", "",
            Gtk::AccelKey( GDK_KEY_Up, Gdk::MOD1_MASK ),
            sigc::mem_fun( m_textview->m_buffer, &TextbufferDiary::move_line_up ) );
    Lifeobase::create_action(
            true, m_action_move_line_down, "MoveLineDown", "", "", "",
            Gtk::AccelKey( GDK_KEY_Down, Gdk::MOD1_MASK ),
            sigc::mem_fun( m_textview->m_buffer, &TextbufferDiary::move_line_down ) );

    Lifeobase::create_action(
            true, m_action_time_stamp, "TimeStamp", "", "", "",
            Gtk::AccelKey( GDK_KEY_t, Gdk::MOD1_MASK ),
            sigc::mem_fun( m_textview->m_buffer, &TextbufferDiary::insert_time_stamp ) );
}

void
EntryView::sync( void )
{
	if( m_flag_entrychanged )
	{
		m_entry_sync->m_date_changed = time( NULL );
		m_entry_sync->m_text = m_textview->m_buffer->get_text();
		m_flag_entrychanged = false;
		PRINT_DEBUG( "entry synced" );
	}
}

void
EntryView::remove_selected_tags( void )
{
	/*std::vector< Gtk::TreeModel::Path > selected_items =
			m_iconview_tags->get_selected_items();
	for( std::vector< Gtk::TreeModel::Path >::iterator iter = selected_items.begin();
		 iter != selected_items.end();
		 ++iter )
	{
		Gtk::TreeRow row = *m_liststore_tags->get_iter( *iter );
		DiaryElement *element = row[ PanelExtra::colrec->ptr ];
		Tag *tag = dynamic_cast< Tag* >( element );

		m_ptr2elem->remove_tag( tag );
		m_liststore_tags->erase( row );
	}*/
}

void
EntryView::handle_login( void )
{
	m_textview->m_buffer->handle_login();

    // READ ONLY
    bool editable( ! Diary::d->is_read_only() );
    m_textview->set_editable( editable );
    m_widgettaglist->set_editable( editable );
    m_menuitem_toggle_favorite->set_visible( editable );
    m_menuseparator_1->set_visible( editable );
    m_menuitem_reset_theme->set_visible( editable );
    m_menuseparator_2->set_visible( editable );
    m_menuitem_trash->set_visible( editable );
    m_menuitem_restore->set_visible( editable );
    m_menuitem_dismiss->set_visible( editable );
    m_hbox_editing_toolbar->set_visible( Lifeobase::settings.show_formatting_toolbar &&
                                         editable );
}

void
EntryView::handle_textview_changed( void )
{
	if( Lifeobase::m_internaloperation ) return;

	m_flag_entrychanged = true;

	PRINT_DEBUG( "entry changed" );

	// TODO:
	//if( m_hbox_tagtools->is_visible() )
		//m_hbox_tagtools->hide();
}

void
EntryView::handle_drag_data_received(
				const Glib::RefPtr< Gdk::DragContext > &context,
				int x, int y,
				const Gtk::SelectionData &selection_data,
				uint info,
				guint time )
{
    if( Diary::d->is_read_only() )
        return;

	context->drag_finish( true, false, time );

	Glib::ustring name = selection_data.get_data_as_string();

	if( name.empty() )	// should never happen
		return;

	switch( info )
	{
		case DRAG_TARGET_TAG_INFO: // both tags and themes come from this source
		{
            if( selection_data.get_data_type() == TARGET_ENTRY_TAG )
            {
                Tag *tag = Diary::d->m_tags.find( name )->second;
                if( m_ptr2elem->add_tag( tag ) ) // if does not already have the tag
                    m_widgettaglist->update();
            }
            else
            {
                Theme *theme = Diary::d->m_themes.find( name )->second;
                m_ptr2elem->set_theme( theme );
                m_textview->m_buffer->set_theme( theme );
                m_menuitem_reset_theme->set_visible( true );
                update_highlight_button();
                m_widgettaglist->set_entry( m_ptr2elem );   // to update the theme
            }
            break;
		}
		case DRAG_TARGET_ENTRY_INFO:
		{
			DiaryElement *element = Lifeobase::panel_diary->get_dragged_elem();
			if( element != NULL )
                if( element->get_type() == DiaryElement::ET_ENTRY )
                    m_textview->m_buffer->insert_link( element );
			break;
		}
	}
}

void
EntryView::toggle_favoredness( void )
{
	if( Lifeobase::m_internaloperation )
		return;

	m_ptr2elem->m_option_favored = !m_ptr2elem->m_option_favored;
    Lifeobase::panel_main->set_icon( m_ptr2elem->get_icon32() );
	Gtk::TreeRow row = Lifeobase::base->get_element_row( m_ptr2elem );
	row[ ListData::colrec->icon ] = m_ptr2elem->get_icon();
}

void
EntryView::print( void )
{
    Lifeobase::view_entry->sync();

    Glib::RefPtr< PrintOpr > print = PrintOpr::create();

    try
    {
        Lifeobase::base->freeze_auto_logout();
        print->set_hide_comments( m_button_toggle_comments->get_active() );
        print->run( Gtk::PRINT_OPERATION_ACTION_PRINT_DIALOG, *Lifeobase::base );
        Lifeobase::base->unfreeze_auto_logout();
    }
    catch( const Gtk::PrintError &ex )
    {
        // See documentation for exact Gtk::PrintError error codes:
        print_error( "An error occurred while trying to run a print operation:" +
                ex.what() );
    }
}

void
EntryView::hide( void )
{
    m_ptr2elem->m_flag_filteredout = true;
    Diary::d->activate_individual_filter();
    Lifeobase::panel_diary->update_button_remove_filters();
    Lifeobase::panel_diary->update_calendar();
    Lifeobase::panel_diary->update_entry_list();
}

void
EntryView::hide_before( void )
{
    Diary::d->set_filter_date_begin( m_ptr2elem->get_date().m_date );
    Lifeobase::panel_diary->update_button_remove_filters();
    Lifeobase::panel_diary->update_calendar();
    Lifeobase::panel_diary->update_entry_list();
}

void
EntryView::hide_after( void )
{
    Diary::d->set_filter_date_end( m_ptr2elem->get_date().m_date );
    Lifeobase::panel_diary->update_button_remove_filters();
    Lifeobase::panel_diary->update_calendar();
    Lifeobase::panel_diary->update_entry_list();
}

void
EntryView::reset_theme( void )
{
	m_ptr2elem->unset_theme();
	m_textview->m_buffer->set_theme( Diary::d->get_default_theme() );
	m_menuitem_reset_theme->set_visible( false );
	update_highlight_button();
}

void
EntryView::trash_entry( void )
{
    // first make sure that entry to be deleted is completely out of business...
    // ...else dark forces may take action and cause SEGFAULT:
    Diary::d->show();

    Lifeobase::panel_main->remove_element_from_history( m_ptr2elem );
    m_ptr2elem->set_trashed( true );
    Diary::d->set_filtering_status_new();

    Lifeobase::m_internaloperation++;

    Lifeobase::panel_diary->update_calendar();
    Lifeobase::panel_diary->update_entry_list();

    Lifeobase::m_internaloperation--;
}

void
EntryView::restore_entry( void )
{
    Diary::d->show();

    m_ptr2elem->set_trashed( false );
    Diary::d->set_filtering_status_new();

    Lifeobase::m_internaloperation++;

    Lifeobase::panel_diary->update_calendar();
    Lifeobase::panel_diary->update_entry_list();

    Lifeobase::m_internaloperation--;
}

void
EntryView::dismiss_entry( void )
{
	if( ! Lifeobase::base->confirm_dismiss_element() )
		return;

    // first make sure that entry to be deleted is completely out of business...
    // ...else dark forces may take action and cause SEGFAULT:
    Diary::d->show();

	Lifeobase::panel_main->remove_element_from_history( m_ptr2elem );
	Diary::d->dismiss_entry( m_ptr2elem );

	Lifeobase::m_internaloperation++;

	Lifeobase::panel_diary->update_calendar();
	Lifeobase::panel_diary->update_entry_list();

	Lifeobase::m_internaloperation--;
}

void
EntryView::toggle_comments( void )
{
    if( Lifeobase::panel_main->get_cur_elem_type() == DiaryElement::ET_ENTRY )
        m_textview->m_buffer->set_comment_visibility(
                ! m_textview->m_buffer->get_comment_visibility() );
}

bool
EntryView::check_title_applicable( const Glib::ustring &str ) const
{
    try
    {
        Date date( str );
    }
    catch( LIFEO::Error &er )
    {
        return false;
    }

    return true;
}

bool
EntryView::apply_title( const Glib::ustring &str )
{
    try
    {
        Date date( str );
        date.reset_order_1();
        Diary::d->set_entry_date( m_ptr2elem, date );
        Lifeobase::panel_diary->update_entry_list();
        Lifeobase::panel_diary->select_date_in_calendar( m_ptr2elem->get_date() );
    }
    catch( LIFEO::Error &er )
    {
        return false;
    }

    return true;
}

void
EntryView::handle_entry_tag_changed( void )
{
	Glib::ustring filter = m_tag_widget->get_text();

	if( filter.empty() )
	{
		m_button_tag_operation->hide();
		m_tag_operation_cur = TO_NONE;
	}
	else
	{
		m_button_tag_operation->show();
		Tag *tag = Diary::d->get_tags()->get_tag( filter );
		if( tag == NULL )
		{
			m_button_tag_operation->set_label( _( "Create Tag" ) );
			m_tag_operation_cur = TO_CREATE_AND_ADD;
		}
		else
		{
			if( m_ptr2elem->get_tags().checkfor_member( tag ) )
			{
				m_button_tag_operation->set_label( _( "Remove Tag" ) );
				m_tag_operation_cur = TO_REMOVE;
			}
			else
			{
				m_button_tag_operation->set_label( _( "Add Tag" ) );
				m_tag_operation_cur = TO_ADD;
			}
		}
	}
}

void
EntryView::handle_button_tag_clicked( void )
{
	Tag *tag;
	Glib::ustring filter = m_tag_widget->get_text();

	switch( m_tag_operation_cur )
	{
		case TO_NONE:
			return;	// don't even clear
		case TO_REMOVE:
			tag = Diary::d->get_tags()->get_tag( filter );
			m_ptr2elem->remove_tag( tag );
			break;
		case TO_CREATE_AND_ADD:
			tag = Diary::d->create_tag( filter );
			m_ptr2elem->add_tag( tag );
			Lifeobase::panel_extra->populate();
            m_tag_widget->populate();
			break;
		case TO_ADD:
			tag = Diary::d->get_tags()->get_tag( filter );
			m_ptr2elem->add_tag( tag );
			break;
	}

    m_widgettaglist->update();
	m_tag_widget->set_text( "" );
}

void
EntryView::focus_tag_entry( void )
{
    if( Lifeobase::panel_main->get_cur_elem_type() == DiaryElement::ET_ENTRY )
    {
        m_box_tag_edit->show();
        m_tag_widget->set_text( "" );
        m_tag_widget->grab_focus();
    }
}

void
EntryView::handle_tag_entry_focus_out( GdkEventFocus* )
{
    m_box_tag_edit->set_visible( false );
}

void
EntryView::edit_tag_in_entry( const Tag *tag )
{
    if( Lifeobase::panel_main->get_cur_elem_type() == DiaryElement::ET_ENTRY &&
        !Diary::d->is_read_only() )
    {
        m_box_tag_edit->show();
        m_tag_widget->set_text( tag->get_name() );
        m_tag_widget->grab_focus();
    }
}

void
EntryView::show( Entry &entry )
{
	// do nothing if entry is already the current element:
	if( Lifeobase::panel_main->is_cur_elem( &entry ) )
		return;

	m_ptr2elem = &entry; // must be first thing

	m_box_tag_edit->set_visible( false );
	Lifeobase::panel_main->show( this );

    if( ! Diary::d->is_read_only() )
    {
        m_menuitem_trash->set_visible( !entry.is_trashed() );
        m_menuitem_restore->set_visible( entry.is_trashed() );
        m_menuitem_dismiss->set_visible( entry.is_trashed() );
        m_menuitem_reset_theme->set_visible( entry.get_theme_is_set() );
    }

	Lifeobase::m_internaloperation++;

	// TOOLBAR
	update_highlight_button();

	// BODY
	m_textview->set_richtext( &entry );

	Lifeobase::m_internaloperation--;

	m_widgettaglist->set_entry( &entry );
    m_tag_widget->populate();

    m_entry_sync = m_ptr2elem;

	PRINT_DEBUG( "entry shown" );
}

void
EntryView::prepare_for_hiding( Entry &entry )
{
	sync();
	m_tag_widget->set_text( "" );
}

void
EntryView::update_formatting_toolbar( void )
{
    m_hbox_editing_toolbar->set_visible( Lifeobase::settings.show_formatting_toolbar &&
                                         Diary::d->is_read_only() == false );
}

void
EntryView::update_highlight_button( void )
{
	const Theme *theme = m_ptr2elem->get_theme_is_set() ?
			m_ptr2elem->get_theme() : Diary::d->get_default_theme();
	m_label_highlight->override_color( theme->color_text, Gtk::STATE_FLAG_NORMAL );
	m_label_highlight->override_color( theme->color_text, Gtk::STATE_FLAG_PRELIGHT );
	m_label_highlight->override_color( theme->color_text, Gtk::STATE_FLAG_ACTIVE );
	m_eventbox_highlight->override_background_color(
	        theme->color_highlight, Gtk::STATE_FLAG_NORMAL );
	m_eventbox_highlight->override_background_color(
	        theme->color_highlight, Gtk::STATE_FLAG_PRELIGHT );
	m_eventbox_highlight->override_background_color(
	        theme->color_highlight, Gtk::STATE_FLAG_ACTIVE );
}
